/** \file libassogiate/mime-database.cc */
/*
 * This file is part of assoGiate,
 * an editor of the file types database for GNOME.
 *
 * Copyright (C) 2007 Kevin Daughtridge <kevin@kdau.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "private.hh"
#include "mime-database.hh"

#include <glib/gutils.h>
#include <glibmm/miscutils.h>

/******************************************************************************/
namespace assoGiate {
/******************************************************************************/

/******************************************************************************/
/* class MimeDatabase                                                         */
/******************************************************************************/

RefPtr<MimeDatabase>
MimeDatabase::load(bool user_db) throw(MimeDatabaseLoadError)
{	return RefPtr<MimeDatabase>(new MimeDatabase(user_db)); }

MimeDatabase::MimeDatabase(bool user_db) throw(MimeDatabaseLoadError)
:	Glib::ObjectBase(), m_mime_dirs(), m_mime_types(),
	m_target(NO_LOCATION), m_Override_pkg(NULL),
	s_changed()
{
	std::list<std::string> data_dirs;
	for (const gchar* const *i = g_get_system_data_dirs(); *i; ++i) // unwrapped
		data_dirs.push_back(*i);
	MimeDirectory *first_dir = NULL;

	if (user_db) {
		first_dir = MimeDirectory::create(g_get_user_data_dir(), USER_STANDARD);
		m_target = USER_OVERRIDE;
	} else /* system db */ if (!data_dirs.empty()) {
		first_dir = MimeDirectory::create(data_dirs.front(), SYSTEM_STANDARD);
		m_target = SYSTEM_OVERRIDE;
		data_dirs.pop_front();
	}
	
	if (!first_dir) throw MimeDatabaseLoadError("no data directories");

	first_dir->signal_changed().connect
		(sigc::mem_fun(*this, &MimeDatabase::rescan));
	m_mime_dirs.push_back(first_dir);

	m_Override_pkg = new WritableMimePackage(Glib::build_filename(first_dir
		->get_packages_dir(), "Override.xml"), *first_dir, m_target);
	m_Override_pkg->signal_reloaded().connect
		(sigc::mem_fun(*this, &MimeDatabase::rescan));

	FOREACH(std::list<std::string>, data_dirs, i) {
		MimeDirectory *dir = MimeDirectory::load(*i, SYSTEM_OVERRIDE);
		if (dir != NULL) {
			dir->signal_changed().connect
				(sigc::mem_fun(*this, &MimeDatabase::rescan));
			m_mime_dirs.push_back(dir);
		}
	}
	
	rescan();
}

MimeDatabase::~MimeDatabase()
{
	delete m_Override_pkg;

	FOREACH(std::list<MimeDirectory*>, m_mime_dirs, i)
		delete *i;
	m_mime_dirs.clear();

	clear_types();
}

MimeTypeMap&
MimeDatabase::get_types() throw()
{	return m_mime_types; }

const MimeTypeMap&
MimeDatabase::get_types() const throw()
{	return m_mime_types; }

Location
MimeDatabase::get_target() const throw()
{	return m_target; }

WritableMimePackage&
MimeDatabase::get_Override_pkg() throw()
{	return *m_Override_pkg; }

sigc::signal<void>
MimeDatabase::signal_changed() throw()
{	return s_changed; }

void
MimeDatabase::clear_types() throw()
{
	FOREACH(MimeTypeMap, m_mime_types, i)
		if (!i->second.second)
			delete i->second.first;
	m_mime_types.clear();
}

void
MimeDatabase::rescan() throw()
{
	clear_types();

	MimeNodeMap node_map;
	RFOREACH(std::list<MimeDirectory*>, m_mime_dirs, i)
		(*i)->extend_node_map(node_map);
	m_Override_pkg->extend_node_map(node_map);

	try {
		FOREACH(MimeNodeMap, node_map, i) {
			MimeTypeMap::iterator iter = m_mime_types.find(i->first);
			if (iter == m_mime_types.end()) {
				std::pair<ustring, ustring> name;
				try {
					name = MimeType::split_name(i->first);
				} catch (std::invalid_argument &e) {
					throw MimeDatabaseLoadError(e.what());
				}

				MimeType *type = new MimeType(name.first, name.second);
				iter = m_mime_types.insert(m_mime_types.end(),
					MimeTypeMap::value_type(i->first, std::pair<MimeType*, bool>
						(type, false)));
			}
			if (iter != m_mime_types.end()) {
				try {
					iter->second.first->accumulate
						(*i->second.first, i->second.second);
				} catch (xmlpp::exception& e) {
					throw MimeDatabaseLoadError(e.what());
				}
			}
		}
	} catch (MimeDatabaseLoadError& e) {
		e.unhandled();
	}

	FOREACH(MimeTypeMap, m_mime_types, i)
		FOREACH(std::list<ustring>, i->second.first->m_aliases, j)
			if (m_mime_types.find(*j) == m_mime_types.end())
				m_mime_types[*j] = std::pair<MimeType*, bool>
					(i->second.first, true);

	try {
		s_changed.emit();
	} catch (...) {}
}

} /* namespace assoGiate */
