/*
 * Asterisk -- An open source telephony toolkit.
 *
 * Copyright (C) 1999 - 2012, Digium, Inc.
 *
 * Mark Spencer <markster@digium.com>
 *
 * See http://www.asterisk.org for more information about
 * the Asterisk project. Please do not directly contact
 * any of the maintainers of this project for assistance;
 * the project provides a web site, mailing lists and IRC
 * channels for your use.
 *
 * This program is free software, distributed under the terms of
 * the GNU General Public License Version 2. See the LICENSE file
 * at the top of the source tree.
 */

/*!
 * \file
 * \brief Implementation of Session Initiation Protocol
 *
 * \author Mark Spencer <markster@digium.com>
 *
 * See Also:
 * \arg \ref AstCREDITS
 *
 * Implementation of RFC 3261 - without S/MIME, and experimental TCP and TLS support
 * Configuration file \link Config_sip sip.conf \endlink
 *
 * ********** IMPORTANT *
 * \note TCP/TLS support is EXPERIMENTAL and WILL CHANGE. This applies to configuration
 *	settings, dialplan commands and dialplans apps/functions
 * See \ref sip_tcp_tls
 *
 *
 * ******** General TODO:s
 * \todo Better support of forking
 * \todo VIA branch tag transaction checking
 * \todo Transaction support
 *
 * ******** Wishlist: Improvements
 * - Support of SIP domains for devices, so that we match on username@domain in the From: header
 * - Connect registrations with a specific device on the incoming call. It's not done
 *   automatically in Asterisk
 *
 * \ingroup channel_drivers
 *
 * \par Overview of the handling of SIP sessions
 * The SIP channel handles several types of SIP sessions, or dialogs,
 * not all of them being "telephone calls".
 * - Incoming calls that will be sent to the PBX core
 * - Outgoing calls, generated by the PBX
 * - SIP subscriptions and notifications of states and voicemail messages
 * - SIP registrations, both inbound and outbound
 * - SIP peer management (peerpoke, OPTIONS)
 * - SIP text messages
 *
 * In the SIP channel, there's a list of active SIP dialogs, which includes
 * all of these when they are active. "sip show channels" in the CLI will
 * show most of these, excluding subscriptions which are shown by
 * "sip show subscriptions"
 *
 * \par incoming packets
 * Incoming packets are received in the monitoring thread, then handled by
 * sipsock_read() for udp only. In tcp, packets are read by the tcp_helper thread.
 * sipsock_read() function parses the packet and matches an existing
 * dialog or starts a new SIP dialog.
 *
 * sipsock_read sends the packet to handle_incoming(), that parses a bit more.
 * If it is a response to an outbound request, the packet is sent to handle_response().
 * If it is a request, handle_incoming() sends it to one of a list of functions
 * depending on the request type - INVITE, OPTIONS, REFER, BYE, CANCEL etc
 * sipsock_read locks the ast_channel if it exists (an active call) and
 * unlocks it after we have processed the SIP message.
 *
 * A new INVITE is sent to handle_request_invite(), that will end up
 * starting a new channel in the PBX, the new channel after that executing
 * in a separate channel thread. This is an incoming "call".
 * When the call is answered, either by a bridged channel or the PBX itself
 * the sip_answer() function is called.
 *
 * The actual media - Video or Audio - is mostly handled by the RTP subsystem
 * in rtp.c
 *
 * \par Outbound calls
 * Outbound calls are set up by the PBX through the sip_request_call()
 * function. After that, they are activated by sip_call().
 *
 * \par Hanging up
 * The PBX issues a hangup on both incoming and outgoing calls through
 * the sip_hangup() function
 */

/*!
 * \page sip_tcp_tls SIP TCP and TLS support
 *
 * \par tcpfixes TCP implementation changes needed
 * \todo Fix TCP/TLS handling in dialplan, SRV records, transfers and much more
 * \todo Save TCP/TLS sessions in registry
 *	If someone registers a SIPS uri, this forces us to set up a TLS connection back.
 * \todo Add TCP/TLS information to function SIPPEER and SIPCHANINFO
 * \todo If tcpenable=yes, we must open a TCP socket on the same address as the IP for UDP.
 * 	The tcpbindaddr config option should only be used to open ADDITIONAL ports
 * 	So we should propably go back to
 *		bindaddr= the default address to bind to. If tcpenable=yes, then bind this to both udp and TCP
 *				if tlsenable=yes, open TLS port (provided we also have cert)
 *		tcpbindaddr = extra address for additional TCP connections
 *		tlsbindaddr = extra address for additional TCP/TLS connections
 *		udpbindaddr = extra address for additional UDP connections
 *			These three options should take multiple IP/port pairs
 *	Note: Since opening additional listen sockets is a *new* feature we do not have today
 *		the XXXbindaddr options needs to be disabled until we have support for it
 *
 * \todo re-evaluate the transport= setting in sip.conf. This is right now not well
 * 	thought of. If a device in sip.conf contacts us via TCP, we should not switch transport,
 *	even if udp is the configured first transport.
 *
 * \todo Be prepared for one outbound and another incoming socket per pvt. This applies
 *       specially to communication with other peers (proxies).
 * \todo We need to test TCP sessions with SIP proxies and in regards
 *       to the SIP outbound specs.
 * \todo ;transport=tls was deprecated in RFC3261 and should not be used at all. See section 26.2.2.
 *
 * \todo If the message is smaller than the given Content-length, the request should get a 400 Bad request
 *       message. If it's a response, it should be dropped. (RFC 3261, Section 18.3)
 * \todo Since we have had multidomain support in Asterisk for quite a while, we need to support
 *       multiple domains in our TLS implementation, meaning one socket and one cert per domain
 * \todo Selection of transport for a request needs to be done after we've parsed all route headers,
 *	 also considering outbound proxy options.
 *		First request: Outboundproxy, routes, (reg contact or URI. If URI doesn't have port:  DNS naptr, srv, AAA)
 *		Intermediate requests: Outboundproxy(only when forced), routes, contact/uri
 *	DNS naptr support is crucial. A SIP uri might lead to a TLS connection.
 *	Also note that due to outbound proxy settings, a SIPS uri might have to be sent on UDP (not to recommend though)
 * \todo Default transports are set to UDP, which cause the wrong behaviour when contacting remote
 *	devices directly from the dialplan. UDP is only a fallback if no other method works,
 *	in order to be compatible with RFC2543 (SIP/1.0) devices. For transactions that exceed the
 *	MTU (like INIVTE with video, audio and RTT)  TCP should be preferred.
 *
 *	When dialling unconfigured peers (with no port number)  or devices in external domains
 *	NAPTR records MUST be consulted to find configured transport. If they are not found,
 *	SRV records for both TCP and UDP should be checked. If there's a record for TCP, use that.
 *	If there's no record for TCP, then use UDP as a last resort. If there's no SRV records,
 *	\note this only applies if there's no outbound proxy configured for the session. If an outbound
 *	proxy is configured, these procedures might apply for locating the proxy and determining
 *	the transport to use for communication with the proxy.
 * \par Other bugs to fix ----
 * __set_address_from_contact(const char *fullcontact, struct sockaddr_in *sin, int tcp)
 *	- sets TLS port as default for all TCP connections, unless other port is given in contact.
 * parse_register_contact(struct sip_pvt *pvt, struct sip_peer *peer, struct sip_request *req)
 *	- assumes that the contact the UA registers is using the same transport as the REGISTER request, which is
 *	  a bad guess.
 *      - Does not save any information about TCP/TLS connected devices, which is a severe BUG, as discussed on the mailing list.
 * get_destination(struct sip_pvt *p, struct sip_request *oreq)
 *	- Doesn't store the information that we got an incoming SIPS request in the channel, so that
 *	  we can require a secure signalling path OUT of Asterisk (on SIP or IAX2). Possibly, the call should
 *	  fail on in-secure signalling paths if there's no override in our configuration. At least, provide a
 *	  channel variable in the dialplan.
 * get_refer_info(struct sip_pvt *transferer, struct sip_request *outgoing_req)
 *	- As above, if we have a SIPS: uri in the refer-to header
 * 	- Does not check transport in refer_to uri.
 */

/*** MODULEINFO
	<use>res_crypto</use>
	<depend>chan_local</depend>
	<support_level>core</support_level>
 ***/

/*!  \page sip_session_timers SIP Session Timers in Asterisk Chan_sip

	The SIP Session-Timers is an extension of the SIP protocol that allows end-points and proxies to
	refresh a session periodically. The sessions are kept alive by sending a RE-INVITE or UPDATE
	request at a negotiated interval. If a session refresh fails then all the entities that support Session-
	Timers clear their internal session state. In addition, UAs generate a BYE request in order to clear
	the state in the proxies and the remote UA (this is done for the benefit of SIP entities in the path
	that do not support Session-Timers).

	The Session-Timers can be configured on a system-wide, per-user, or per-peer basis. The peruser/
	per-peer settings override the global settings. The following new parameters have been
	added to the sip.conf file.
		session-timers=["accept", "originate", "refuse"]
		session-expires=[integer]
		session-minse=[integer]
		session-refresher=["uas", "uac"]

	The session-timers parameter in sip.conf defines the mode of operation of SIP session-timers feature in
	Asterisk. The Asterisk can be configured in one of the following three modes:

	1. Accept :: In the "accept" mode, the Asterisk server honors session-timers requests
		made by remote end-points. A remote end-point can request Asterisk to engage
		session-timers by either sending it an INVITE request with a "Supported: timer"
		header in it or by responding to Asterisk's INVITE with a 200 OK that contains
		Session-Expires: header in it. In this mode, the Asterisk server does not
		request session-timers from remote end-points. This is the default mode.
	2. Originate :: In the "originate" mode, the Asterisk server requests the remote
		end-points to activate session-timers in addition to honoring such requests
		made by the remote end-pints. In order to get as much protection as possible
		against hanging SIP channels due to network or end-point failures, Asterisk
		resends periodic re-INVITEs even if a remote end-point does not support
		the session-timers feature.
	3. Refuse :: In the "refuse" mode, Asterisk acts as if it does not support session-
		timers for inbound or outbound requests. If a remote end-point requests
		session-timers in a dialog, then Asterisk ignores that request unless it's
		noted as a requirement (Require: header), in which case the INVITE is
		rejected with a 420 Bad Extension response.

*/

#include "asterisk.h"

ASTERISK_FILE_VERSION(__FILE__, "$Revision: 369669 $")

#include <signal.h>
#include <sys/signal.h>
#include <regex.h>
#include <inttypes.h>

#include "asterisk/network.h"
#include "asterisk/paths.h"	/* need ast_config_AST_SYSTEM_NAME */
/*
   Uncomment the define below,  if you are having refcount related memory leaks.
   With this uncommented, this module will generate a file, /tmp/refs, which contains
   a history of the ao2_ref() calls. To be useful, all calls to ao2_* functions should
   be modified to ao2_t_* calls, and include a tag describing what is happening with
   enough detail, to make pairing up a reference count increment with its corresponding decrement.
   The refcounter program in utils/ can be invaluable in highlighting objects that are not
   balanced, along with the complete history for that object.
   In normal operation, the macros defined will throw away the tags, so they do not
   affect the speed of the program at all. They can be considered to be documentation.
*/
/* #define  REF_DEBUG 1 */
#include "asterisk/lock.h"
#include "asterisk/config.h"
#include "asterisk/module.h"
#include "asterisk/pbx.h"
#include "asterisk/sched.h"
#include "asterisk/io.h"
#include "asterisk/rtp_engine.h"
#include "asterisk/udptl.h"
#include "asterisk/acl.h"
#include "asterisk/manager.h"
#include "asterisk/callerid.h"
#include "asterisk/cli.h"
#include "asterisk/musiconhold.h"
#include "asterisk/dsp.h"
#include "asterisk/features.h"
#include "asterisk/srv.h"
#include "asterisk/astdb.h"
#include "asterisk/causes.h"
#include "asterisk/utils.h"
#include "asterisk/file.h"
#include "asterisk/astobj2.h"
#include "asterisk/dnsmgr.h"
#include "asterisk/devicestate.h"
#include "asterisk/monitor.h"
#include "asterisk/netsock2.h"
#include "asterisk/localtime.h"
#include "asterisk/abstract_jb.h"
#include "asterisk/threadstorage.h"
#include "asterisk/translate.h"
#include "asterisk/ast_version.h"
#include "asterisk/event.h"
#include "asterisk/cel.h"
#include "asterisk/data.h"
#include "asterisk/aoc.h"
#include "sip/include/sip.h"
#include "sip/include/globals.h"
#include "sip/include/config_parser.h"
#include "sip/include/reqresp_parser.h"
#include "sip/include/sip_utils.h"
#include "sip/include/srtp.h"
#include "sip/include/sdp_crypto.h"
#include "asterisk/ccss.h"
#include "asterisk/xml.h"
#include "sip/include/dialog.h"
#include "sip/include/dialplan_functions.h"


/*** DOCUMENTATION
	<application name="SIPDtmfMode" language="en_US">
		<synopsis>
			Change the dtmfmode for a SIP call.
		</synopsis>
		<syntax>
			<parameter name="mode" required="true">
				<enumlist>
					<enum name="inband" />
					<enum name="info" />
					<enum name="rfc2833" />
				</enumlist>
			</parameter>
		</syntax>
		<description>
			<para>Changes the dtmfmode for a SIP call.</para>
		</description>
	</application>
	<application name="SIPAddHeader" language="en_US">
		<synopsis>
			Add a SIP header to the outbound call.
		</synopsis>
		<syntax argsep=":">
			<parameter name="Header" required="true" />
			<parameter name="Content" required="true" />
		</syntax>
		<description>
			<para>Adds a header to a SIP call placed with DIAL.</para>
			<para>Remember to use the X-header if you are adding non-standard SIP
			headers, like <literal>X-Asterisk-Accountcode:</literal>. Use this with care.
			Adding the wrong headers may jeopardize the SIP dialog.</para>
			<para>Always returns <literal>0</literal>.</para>
		</description>
	</application>
	<application name="SIPRemoveHeader" language="en_US">
		<synopsis>
			Remove SIP headers previously added with SIPAddHeader
		</synopsis>
		<syntax>
			<parameter name="Header" required="false" />
		</syntax>
		<description>
			<para>SIPRemoveHeader() allows you to remove headers which were previously
			added with SIPAddHeader(). If no parameter is supplied, all previously added
			headers will be removed. If a parameter is supplied, only the matching headers
			will be removed.</para>
			<para>For example you have added these 2 headers:</para>
			<para>SIPAddHeader(P-Asserted-Identity: sip:foo@bar);</para>
			<para>SIPAddHeader(P-Preferred-Identity: sip:bar@foo);</para>
			<para></para>
			<para>// remove all headers</para>
			<para>SIPRemoveHeader();</para>
			<para>// remove all P- headers</para>
			<para>SIPRemoveHeader(P-);</para>
			<para>// remove only the PAI header (note the : at the end)</para>
			<para>SIPRemoveHeader(P-Asserted-Identity:);</para>
			<para></para>
			<para>Always returns <literal>0</literal>.</para>
		</description>
	</application>
	<function name="SIP_HEADER" language="en_US">
		<synopsis>
			Gets the specified SIP header from an incoming INVITE message.
		</synopsis>
		<syntax>
			<parameter name="name" required="true" />
			<parameter name="number">
				<para>If not specified, defaults to <literal>1</literal>.</para>
			</parameter>
		</syntax>
		<description>
			<para>Since there are several headers (such as Via) which can occur multiple
			times, SIP_HEADER takes an optional second argument to specify which header with
			that name to retrieve. Headers start at offset <literal>1</literal>.</para>
		</description>
	</function>
	<function name="SIPPEER" language="en_US">
		<synopsis>
			Gets SIP peer information.
		</synopsis>
		<syntax>
			<parameter name="peername" required="true" />
			<parameter name="item">
				<enumlist>
					<enum name="ip">
						<para>(default) The ip address.</para>
					</enum>
					<enum name="port">
						<para>The port number.</para>
					</enum>
					<enum name="mailbox">
						<para>The configured mailbox.</para>
					</enum>
					<enum name="context">
						<para>The configured context.</para>
					</enum>
					<enum name="expire">
						<para>The epoch time of the next expire.</para>
					</enum>
					<enum name="dynamic">
						<para>Is it dynamic? (yes/no).</para>
					</enum>
					<enum name="callerid_name">
						<para>The configured Caller ID name.</para>
					</enum>
					<enum name="callerid_num">
						<para>The configured Caller ID number.</para>
					</enum>
					<enum name="callgroup">
						<para>The configured Callgroup.</para>
					</enum>
					<enum name="pickupgroup">
						<para>The configured Pickupgroup.</para>
					</enum>
					<enum name="codecs">
						<para>The configured codecs.</para>
					</enum>
					<enum name="status">
						<para>Status (if qualify=yes).</para>
					</enum>
					<enum name="regexten">
						<para>Registration extension.</para>
					</enum>
					<enum name="limit">
						<para>Call limit (call-limit).</para>
					</enum>
					<enum name="busylevel">
						<para>Configured call level for signalling busy.</para>
					</enum>
					<enum name="curcalls">
						<para>Current amount of calls. Only available if call-limit is set.</para>
					</enum>
					<enum name="language">
						<para>Default language for peer.</para>
					</enum>
					<enum name="accountcode">
						<para>Account code for this peer.</para>
					</enum>
					<enum name="useragent">
						<para>Current user agent id for peer.</para>
					</enum>
					<enum name="maxforwards">
						<para>The value used for SIP loop prevention in outbound requests</para>
					</enum>
					<enum name="chanvar[name]">
						<para>A channel variable configured with setvar for this peer.</para>
					</enum>
					<enum name="codec[x]">
						<para>Preferred codec index number <replaceable>x</replaceable> (beginning with zero).</para>
					</enum>
				</enumlist>
			</parameter>
		</syntax>
		<description></description>
	</function>
	<function name="SIPCHANINFO" language="en_US">
		<synopsis>
			Gets the specified SIP parameter from the current channel.
		</synopsis>
		<syntax>
			<parameter name="item" required="true">
				<enumlist>
					<enum name="peerip">
						<para>The IP address of the peer.</para>
					</enum>
					<enum name="recvip">
						<para>The source IP address of the peer.</para>
					</enum>
					<enum name="from">
						<para>The URI from the <literal>From:</literal> header.</para>
					</enum>
					<enum name="uri">
						<para>The URI from the <literal>Contact:</literal> header.</para>
					</enum>
					<enum name="useragent">
						<para>The useragent.</para>
					</enum>
					<enum name="peername">
						<para>The name of the peer.</para>
					</enum>
					<enum name="t38passthrough">
						<para><literal>1</literal> if T38 is offered or enabled in this channel,
						otherwise <literal>0</literal>.</para>
					</enum>
				</enumlist>
			</parameter>
		</syntax>
		<description></description>
	</function>
	<function name="CHECKSIPDOMAIN" language="en_US">
		<synopsis>
			Checks if domain is a local domain.
		</synopsis>
		<syntax>
			<parameter name="domain" required="true" />
		</syntax>
		<description>
			<para>This function checks if the <replaceable>domain</replaceable> in the argument is configured
			as a local SIP domain that this Asterisk server is configured to handle.
			Returns the domain name if it is locally handled, otherwise an empty string.
			Check the <literal>domain=</literal> configuration in <filename>sip.conf</filename>.</para>
		</description>
	</function>
	<manager name="SIPpeers" language="en_US">
		<synopsis>
			List SIP peers (text format).
		</synopsis>
		<syntax>
			<xi:include xpointer="xpointer(/docs/manager[@name='Login']/syntax/parameter[@name='ActionID'])" />
		</syntax>
		<description>
			<para>Lists SIP peers in text format with details on current status.
			Peerlist will follow as separate events, followed by a final event called
			PeerlistComplete.</para>
		</description>
	</manager>
	<manager name="SIPshowpeer" language="en_US">
		<synopsis>
			show SIP peer (text format).
		</synopsis>
		<syntax>
			<xi:include xpointer="xpointer(/docs/manager[@name='Login']/syntax/parameter[@name='ActionID'])" />
			<parameter name="Peer" required="true">
				<para>The peer name you want to check.</para>
			</parameter>
		</syntax>
		<description>
			<para>Show one SIP peer with details on current status.</para>
		</description>
	</manager>
	<manager name="SIPqualifypeer" language="en_US">
		<synopsis>
			Qualify SIP peers.
		</synopsis>
		<syntax>
			<xi:include xpointer="xpointer(/docs/manager[@name='Login']/syntax/parameter[@name='ActionID'])" />
			<parameter name="Peer" required="true">
				<para>The peer name you want to qualify.</para>
			</parameter>
		</syntax>
		<description>
			<para>Qualify a SIP peer.</para>
		</description>
	</manager>
	<manager name="SIPshowregistry" language="en_US">
		<synopsis>
			Show SIP registrations (text format).
		</synopsis>
		<syntax>
			<xi:include xpointer="xpointer(/docs/manager[@name='Login']/syntax/parameter[@name='ActionID'])" />
		</syntax>
		<description>
			<para>Lists all registration requests and status. Registrations will follow as separate
			events. followed by a final event called RegistrationsComplete.</para>
		</description>
	</manager>
	<manager name="SIPnotify" language="en_US">
		<synopsis>
			Send a SIP notify.
		</synopsis>
		<syntax>
			<xi:include xpointer="xpointer(/docs/manager[@name='Login']/syntax/parameter[@name='ActionID'])" />
			<parameter name="Channel" required="true">
				<para>Peer to receive the notify.</para>
			</parameter>
			<parameter name="Variable" required="true">
				<para>At least one variable pair must be specified.
				<replaceable>name</replaceable>=<replaceable>value</replaceable></para>
			</parameter>
		</syntax>
		<description>
			<para>Sends a SIP Notify event.</para>
			<para>All parameters for this event must be specified in the body of this request
			via multiple Variable: name=value sequences.</para>
		</description>
	</manager>
 ***/

static int min_expiry = DEFAULT_MIN_EXPIRY;        /*!< Minimum accepted registration time */
static int max_expiry = DEFAULT_MAX_EXPIRY;        /*!< Maximum accepted registration time */
static int default_expiry = DEFAULT_DEFAULT_EXPIRY;
static int mwi_expiry = DEFAULT_MWI_EXPIRY;

static int unauth_sessions = 0;
static int authlimit = DEFAULT_AUTHLIMIT;
static int authtimeout = DEFAULT_AUTHTIMEOUT;

/*! \brief Global jitterbuffer configuration - by default, jb is disabled
 *  \note Values shown here match the defaults shown in sip.conf.sample */
static struct ast_jb_conf default_jbconf =
{
	.flags = 0,
	.max_size = 200,
	.resync_threshold = 1000,
	.impl = "fixed",
	.target_extra = 40,
};
static struct ast_jb_conf global_jbconf;                /*!< Global jitterbuffer configuration */

static const char config[] = "sip.conf";                /*!< Main configuration file */
static const char notify_config[] = "sip_notify.conf";  /*!< Configuration file for sending Notify with CLI commands to reconfigure or reboot phones */

/*! \brief Readable descriptions of device states.
 *  \note Should be aligned to above table as index */
static const struct invstate2stringtable {
	const enum invitestates state;
	const char *desc;
} invitestate2string[] = {
	{INV_NONE,              "None"  },
	{INV_CALLING,           "Calling (Trying)"},
	{INV_PROCEEDING,        "Proceeding "},
	{INV_EARLY_MEDIA,       "Early media"},
	{INV_COMPLETED,         "Completed (done)"},
	{INV_CONFIRMED,         "Confirmed (up)"},
	{INV_TERMINATED,        "Done"},
	{INV_CANCELLED,         "Cancelled"}
};

/*! \brief Subscription types that we support. We support
 * - dialoginfo updates (really device status, not dialog info as was the original intent of the standard)
 * - SIMPLE presence used for device status
 * - Voicemail notification subscriptions
 */
static const struct cfsubscription_types {
	enum subscriptiontype type;
	const char * const event;
	const char * const mediatype;
	const char * const text;
} subscription_types[] = {
	{ NONE,		   "-",        "unknown",	             "unknown" },
	/* RFC 4235: SIP Dialog event package */
	{ DIALOG_INFO_XML, "dialog",   "application/dialog-info+xml", "dialog-info+xml" },
	{ CPIM_PIDF_XML,   "presence", "application/cpim-pidf+xml",   "cpim-pidf+xml" },  /* RFC 3863 */
	{ PIDF_XML,        "presence", "application/pidf+xml",        "pidf+xml" },       /* RFC 3863 */
	{ XPIDF_XML,       "presence", "application/xpidf+xml",       "xpidf+xml" },       /* Pre-RFC 3863 with MS additions */
	{ MWI_NOTIFICATION,	"message-summary", "application/simple-message-summary", "mwi" } /* RFC 3842: Mailbox notification */
};

/*! \brief The core structure to setup dialogs. We parse incoming messages by using
 *  structure and then route the messages according to the type.
 *
 *  \note Note that sip_methods[i].id == i must hold or the code breaks
 */
static const struct  cfsip_methods {
	enum sipmethod id;
	int need_rtp;		/*!< when this is the 'primary' use for a pvt structure, does it need RTP? */
	char * const text;
	enum can_create_dialog can_create;
} sip_methods[] = {
	{ SIP_UNKNOWN,   RTP,    "-UNKNOWN-",CAN_CREATE_DIALOG },
	{ SIP_RESPONSE,  NO_RTP, "SIP/2.0",  CAN_NOT_CREATE_DIALOG },
	{ SIP_REGISTER,  NO_RTP, "REGISTER", CAN_CREATE_DIALOG },
	{ SIP_OPTIONS,   NO_RTP, "OPTIONS",  CAN_CREATE_DIALOG },
	{ SIP_NOTIFY,    NO_RTP, "NOTIFY",   CAN_CREATE_DIALOG },
	{ SIP_INVITE,    RTP,    "INVITE",   CAN_CREATE_DIALOG },
	{ SIP_ACK,       NO_RTP, "ACK",      CAN_NOT_CREATE_DIALOG },
	{ SIP_PRACK,     NO_RTP, "PRACK",    CAN_NOT_CREATE_DIALOG },
	{ SIP_BYE,       NO_RTP, "BYE",      CAN_NOT_CREATE_DIALOG },
	{ SIP_REFER,     NO_RTP, "REFER",    CAN_CREATE_DIALOG },
	{ SIP_SUBSCRIBE, NO_RTP, "SUBSCRIBE",CAN_CREATE_DIALOG },
	{ SIP_MESSAGE,   NO_RTP, "MESSAGE",  CAN_CREATE_DIALOG },
	{ SIP_UPDATE,    NO_RTP, "UPDATE",   CAN_NOT_CREATE_DIALOG },
	{ SIP_INFO,      NO_RTP, "INFO",     CAN_NOT_CREATE_DIALOG },
	{ SIP_CANCEL,    NO_RTP, "CANCEL",   CAN_NOT_CREATE_DIALOG },
	{ SIP_PUBLISH,   NO_RTP, "PUBLISH",  CAN_CREATE_DIALOG },
	{ SIP_PING,      NO_RTP, "PING",     CAN_CREATE_DIALOG_UNSUPPORTED_METHOD }
};

/*! \brief Diversion header reasons
 *
 * The core defines a bunch of constants used to define
 * redirecting reasons. This provides a translation table
 * between those and the strings which may be present in
 * a SIP Diversion header
 */
static const struct sip_reasons {
	enum AST_REDIRECTING_REASON code;
	char * const text;
} sip_reason_table[] = {
	{ AST_REDIRECTING_REASON_UNKNOWN, "unknown" },
	{ AST_REDIRECTING_REASON_USER_BUSY, "user-busy" },
	{ AST_REDIRECTING_REASON_NO_ANSWER, "no-answer" },
	{ AST_REDIRECTING_REASON_UNAVAILABLE, "unavailable" },
	{ AST_REDIRECTING_REASON_UNCONDITIONAL, "unconditional" },
	{ AST_REDIRECTING_REASON_TIME_OF_DAY, "time-of-day" },
	{ AST_REDIRECTING_REASON_DO_NOT_DISTURB, "do-not-disturb" },
	{ AST_REDIRECTING_REASON_DEFLECTION, "deflection" },
	{ AST_REDIRECTING_REASON_FOLLOW_ME, "follow-me" },
	{ AST_REDIRECTING_REASON_OUT_OF_ORDER, "out-of-service" },
	{ AST_REDIRECTING_REASON_AWAY, "away" },
	{ AST_REDIRECTING_REASON_CALL_FWD_DTE, "unknown"}
};


/*! \name DefaultSettings
	Default setttings are used as a channel setting and as a default when
	configuring devices
*/
/*@{*/
static char default_language[MAX_LANGUAGE];      /*!< Default language setting for new channels */
static char default_callerid[AST_MAX_EXTENSION]; /*!< Default caller ID for sip messages */
static char default_mwi_from[80];                /*!< Default caller ID for MWI updates */
static char default_fromdomain[AST_MAX_EXTENSION]; /*!< Default domain on outound messages */
static int default_fromdomainport;                 /*!< Default domain port on outbound messages */
static char default_notifymime[AST_MAX_EXTENSION]; /*!< Default MIME media type for MWI notify messages */
static char default_vmexten[AST_MAX_EXTENSION];    /*!< Default From Username on MWI updates */
static int default_qualify;                        /*!< Default Qualify= setting */
static char default_mohinterpret[MAX_MUSICCLASS];  /*!< Global setting for moh class to use when put on hold */
static char default_mohsuggest[MAX_MUSICCLASS];    /*!< Global setting for moh class to suggest when putting
                                                    *   a bridged channel on hold */
static char default_parkinglot[AST_MAX_CONTEXT];   /*!< Parkinglot */
static char default_engine[256];                   /*!< Default RTP engine */
static int default_maxcallbitrate;                 /*!< Maximum bitrate for call */
static struct ast_codec_pref default_prefs;        /*!< Default codec prefs */
static unsigned int default_transports;            /*!< Default Transports (enum sip_transport) that are acceptable */
static unsigned int default_primary_transport;     /*!< Default primary Transport (enum sip_transport) for outbound connections to devices */
/*@}*/

static struct sip_settings sip_cfg;		/*!< SIP configuration data.
					\note in the future we could have multiple of these (per domain, per device group etc) */

/*!< use this macro when ast_uri_decode is dependent on pedantic checking to be on. */
#define SIP_PEDANTIC_DECODE(str)	\
	if (sip_cfg.pedanticsipchecking && !ast_strlen_zero(str)) {	\
		ast_uri_decode(str);	\
	}	\

static unsigned int chan_idx;       /*!< used in naming sip channel */
static int global_match_auth_username;    /*!< Match auth username if available instead of From: Default off. */

static int global_relaxdtmf;        /*!< Relax DTMF */
static int global_prematuremediafilter;   /*!< Enable/disable premature frames in a call (causing 183 early media) */
static int global_rtptimeout;       /*!< Time out call if no RTP */
static int global_rtpholdtimeout;   /*!< Time out call if no RTP during hold */
static int global_rtpkeepalive;     /*!< Send RTP keepalives */
static int global_reg_timeout;      /*!< Global time between attempts for outbound registrations */
static int global_regattempts_max;  /*!< Registration attempts before giving up */
static int global_shrinkcallerid;   /*!< enable or disable shrinking of caller id  */
static int global_callcounter;      /*!< Enable call counters for all devices. This is currently enabled by setting the peer
                                     *   call-limit to INT_MAX. When we remove the call-limit from the code, we can make it
                                     *   with just a boolean flag in the device structure */
static unsigned int global_tos_sip;      /*!< IP type of service for SIP packets */
static unsigned int global_tos_audio;    /*!< IP type of service for audio RTP packets */
static unsigned int global_tos_video;    /*!< IP type of service for video RTP packets */
static unsigned int global_tos_text;     /*!< IP type of service for text RTP packets */
static unsigned int global_cos_sip;      /*!< 802.1p class of service for SIP packets */
static unsigned int global_cos_audio;    /*!< 802.1p class of service for audio RTP packets */
static unsigned int global_cos_video;    /*!< 802.1p class of service for video RTP packets */
static unsigned int global_cos_text;     /*!< 802.1p class of service for text RTP packets */
static unsigned int recordhistory;       /*!< Record SIP history. Off by default */
static unsigned int dumphistory;         /*!< Dump history to verbose before destroying SIP dialog */
static char global_useragent[AST_MAX_EXTENSION];    /*!< Useragent for the SIP channel */
static char global_sdpsession[AST_MAX_EXTENSION];   /*!< SDP session name for the SIP channel */
static char global_sdpowner[AST_MAX_EXTENSION];     /*!< SDP owner name for the SIP channel */
static int global_authfailureevents;     /*!< Whether we send authentication failure manager events or not. Default no. */
static int global_t1;           /*!< T1 time */
static int global_t1min;        /*!< T1 roundtrip time minimum */
static int global_timer_b;      /*!< Timer B - RFC 3261 Section 17.1.1.2 */
static unsigned int global_autoframing; /*!< Turn autoframing on or off. */
static int global_qualifyfreq;          /*!< Qualify frequency */
static int global_qualify_gap;          /*!< Time between our group of peer pokes */
static int global_qualify_peers;        /*!< Number of peers to poke at a given time */

static enum st_mode global_st_mode;           /*!< Mode of operation for Session-Timers           */
static enum st_refresher global_st_refresher; /*!< Session-Timer refresher                        */
static int global_min_se;                     /*!< Lowest threshold for session refresh interval  */
static int global_max_se;                     /*!< Highest threshold for session refresh interval */

static int global_store_sip_cause;    /*!< Whether the MASTER_CHANNEL(HASH(SIP_CAUSE,[chan_name])) var should be set */

static int global_dynamic_exclude_static = 0; /*!< Exclude static peers from contact registrations */
/*@}*/

/*!
 * We use libxml2 in order to parse XML that may appear in the body of a SIP message. Currently,
 * the only usage is for parsing PIDF bodies of incoming PUBLISH requests in the call-completion
 * event package. This variable is set at module load time and may be checked at runtime to determine
 * if XML parsing support was found.
 */
static int can_parse_xml;

/*! \name Object counters @{
 *  \bug These counters are not handled in a thread-safe way ast_atomic_fetchadd_int()
 *  should be used to modify these values. */
static int speerobjs = 0;     /*!< Static peers */
static int rpeerobjs = 0;     /*!< Realtime peers */
static int apeerobjs = 0;     /*!< Autocreated peer objects */
static int regobjs = 0;       /*!< Registry objects */
/* }@ */

static struct ast_flags global_flags[3] = {{0}};  /*!< global SIP_ flags */
static int global_t38_maxdatagram;                /*!< global T.38 FaxMaxDatagram override */

static struct ast_event_sub *network_change_event_subscription; /*!< subscription id for network change events */
static int network_change_event_sched_id = -1;

static char used_context[AST_MAX_CONTEXT];        /*!< name of automatically created context for unloading */

AST_MUTEX_DEFINE_STATIC(netlock);

/*! \brief Protect the monitoring thread, so only one process can kill or start it, and not
   when it's doing something critical. */
AST_MUTEX_DEFINE_STATIC(monlock);

AST_MUTEX_DEFINE_STATIC(sip_reload_lock);

/*! \brief This is the thread for the monitor which checks for input on the channels
   which are not currently in use.  */
static pthread_t monitor_thread = AST_PTHREADT_NULL;

static int sip_reloading = FALSE;                       /*!< Flag for avoiding multiple reloads at the same time */
static enum channelreloadreason sip_reloadreason;       /*!< Reason for last reload/load of configuration */

struct sched_context *sched;     /*!< The scheduling context */
static struct io_context *io;           /*!< The IO context */
static int *sipsock_read_id;            /*!< ID of IO entry for sipsock FD */
struct sip_pkt;
static AST_LIST_HEAD_STATIC(domain_list, domain);    /*!< The SIP domain list */

AST_LIST_HEAD_NOLOCK(sip_history_head, sip_history); /*!< history list, entry in sip_pvt */

static enum sip_debug_e sipdebug;

/*! \brief extra debugging for 'text' related events.
 *  At the moment this is set together with sip_debug_console.
 *  \note It should either go away or be implemented properly.
 */
static int sipdebug_text;

static const struct _map_x_s referstatusstrings[] = {
	{ REFER_IDLE,      "<none>" },
	{ REFER_SENT,      "Request sent" },
	{ REFER_RECEIVED,  "Request received" },
	{ REFER_CONFIRMED, "Confirmed" },
	{ REFER_ACCEPTED,  "Accepted" },
	{ REFER_RINGING,   "Target ringing" },
	{ REFER_200OK,     "Done" },
	{ REFER_FAILED,    "Failed" },
	{ REFER_NOAUTH,    "Failed - auth failure" },
	{ -1,               NULL} /* terminator */
};

/* --- Hash tables of various objects --------*/
#ifdef LOW_MEMORY
static const int HASH_PEER_SIZE = 17;
static const int HASH_DIALOG_SIZE = 17;
#else
static const int HASH_PEER_SIZE = 563;	/*!< Size of peer hash table, prime number preferred! */
static const int HASH_DIALOG_SIZE = 563;
#endif

static const struct {
	enum ast_cc_service_type service;
	const char *service_string;
} sip_cc_service_map [] = {
	[AST_CC_NONE] = { AST_CC_NONE, "" },
	[AST_CC_CCBS] = { AST_CC_CCBS, "BS" },
	[AST_CC_CCNR] = { AST_CC_CCNR, "NR" },
	[AST_CC_CCNL] = { AST_CC_CCNL, "NL" },
};

static enum ast_cc_service_type service_string_to_service_type(const char * const service_string)
{
	enum ast_cc_service_type service;
	for (service = AST_CC_CCBS; service <= AST_CC_CCNL; ++service) {
		if (!strcasecmp(service_string, sip_cc_service_map[service].service_string)) {
			return service;
		}
	}
	return AST_CC_NONE;
}

static const struct {
	enum sip_cc_notify_state state;
	const char *state_string;
} sip_cc_notify_state_map [] = {
	[CC_QUEUED] = {CC_QUEUED, "cc-state: queued"},
	[CC_READY] = {CC_READY, "cc-state: ready"},
};

AST_LIST_HEAD_STATIC(epa_static_data_list, epa_backend);

static int sip_epa_register(const struct epa_static_data *static_data)
{
	struct epa_backend *backend = ast_calloc(1, sizeof(*backend));

	if (!backend) {
		return -1;
	}

	backend->static_data = static_data;

	AST_LIST_LOCK(&epa_static_data_list);
	AST_LIST_INSERT_TAIL(&epa_static_data_list, backend, next);
	AST_LIST_UNLOCK(&epa_static_data_list);
	return 0;
}

static void sip_epa_unregister_all(void)
{
	struct epa_backend *backend;

	AST_LIST_LOCK(&epa_static_data_list);
	while ((backend = AST_LIST_REMOVE_HEAD(&epa_static_data_list, next))) {
		ast_free(backend);
	}
	AST_LIST_UNLOCK(&epa_static_data_list);
}

static void cc_handle_publish_error(struct sip_pvt *pvt, const int resp, struct sip_request *req, struct sip_epa_entry *epa_entry);

static void cc_epa_destructor(void *data)
{
	struct sip_epa_entry *epa_entry = data;
	struct cc_epa_entry *cc_entry = epa_entry->instance_data;
	ast_free(cc_entry);
}

static const struct epa_static_data cc_epa_static_data  = {
	.event = CALL_COMPLETION,
	.name = "call-completion",
	.handle_error = cc_handle_publish_error,
	.destructor = cc_epa_destructor,
};

static const struct epa_static_data *find_static_data(const char * const event_package)
{
	const struct epa_backend *backend = NULL;

	AST_LIST_LOCK(&epa_static_data_list);
	AST_LIST_TRAVERSE(&epa_static_data_list, backend, next) {
		if (!strcmp(backend->static_data->name, event_package)) {
			break;
		}
	}
	AST_LIST_UNLOCK(&epa_static_data_list);
	return backend ? backend->static_data : NULL;
}

static struct sip_epa_entry *create_epa_entry (const char * const event_package, const char * const destination)
{
	struct sip_epa_entry *epa_entry;
	const struct epa_static_data *static_data;

	if (!(static_data = find_static_data(event_package))) {
		return NULL;
	}

	if (!(epa_entry = ao2_t_alloc(sizeof(*epa_entry), static_data->destructor, "Allocate new EPA entry"))) {
		return NULL;
	}

	epa_entry->static_data = static_data;
	ast_copy_string(epa_entry->destination, destination, sizeof(epa_entry->destination));
	return epa_entry;
}

/*!
 * Used to create new entity IDs by ESCs.
 */
static int esc_etag_counter;
static const int DEFAULT_PUBLISH_EXPIRES = 3600;

#ifdef HAVE_LIBXML2
static int cc_esc_publish_handler(struct sip_pvt *pvt, struct sip_request *req, struct event_state_compositor *esc, struct sip_esc_entry *esc_entry);

static const struct sip_esc_publish_callbacks cc_esc_publish_callbacks = {
	.initial_handler = cc_esc_publish_handler,
	.modify_handler = cc_esc_publish_handler,
};
#endif

/*!
 * \brief The Event State Compositors
 *
 * An Event State Compositor is an entity which
 * accepts PUBLISH requests and acts appropriately
 * based on these requests.
 *
 * The actual event_state_compositor structure is simply
 * an ao2_container of sip_esc_entrys. When an incoming
 * PUBLISH is received, we can match the appropriate sip_esc_entry
 * using the entity ID of the incoming PUBLISH.
 */
static struct event_state_compositor {
	enum subscriptiontype event;
	const char * name;
	const struct sip_esc_publish_callbacks *callbacks;
	struct ao2_container *compositor;
} event_state_compositors [] = {
#ifdef HAVE_LIBXML2
	{CALL_COMPLETION, "call-completion", &cc_esc_publish_callbacks},
#endif
};

static const int ESC_MAX_BUCKETS = 37;

static void esc_entry_destructor(void *obj)
{
	struct sip_esc_entry *esc_entry = obj;
	if (esc_entry->sched_id > -1) {
		AST_SCHED_DEL(sched, esc_entry->sched_id);
	}
}

static int esc_hash_fn(const void *obj, const int flags)
{
	const struct sip_esc_entry *entry = obj;
	return ast_str_hash(entry->entity_tag);
}

static int esc_cmp_fn(void *obj, void *arg, int flags)
{
	struct sip_esc_entry *entry1 = obj;
	struct sip_esc_entry *entry2 = arg;

	return (!strcmp(entry1->entity_tag, entry2->entity_tag)) ? (CMP_MATCH | CMP_STOP) : 0;
}

static struct event_state_compositor *get_esc(const char * const event_package) {
	int i;
	for (i = 0; i < ARRAY_LEN(event_state_compositors); i++) {
		if (!strcasecmp(event_package, event_state_compositors[i].name)) {
			return &event_state_compositors[i];
		}
	}
	return NULL;
}

static struct sip_esc_entry *get_esc_entry(const char * entity_tag, struct event_state_compositor *esc) {
	struct sip_esc_entry *entry;
	struct sip_esc_entry finder;

	ast_copy_string(finder.entity_tag, entity_tag, sizeof(finder.entity_tag));

	entry = ao2_find(esc->compositor, &finder, OBJ_POINTER);

	return entry;
}

static int publish_expire(const void *data)
{
	struct sip_esc_entry *esc_entry = (struct sip_esc_entry *) data;
	struct event_state_compositor *esc = get_esc(esc_entry->event);

	ast_assert(esc != NULL);

	ao2_unlink(esc->compositor, esc_entry);
	ao2_ref(esc_entry, -1);
	return 0;
}

static void create_new_sip_etag(struct sip_esc_entry *esc_entry, int is_linked)
{
	int new_etag = ast_atomic_fetchadd_int(&esc_etag_counter, +1);
	struct event_state_compositor *esc = get_esc(esc_entry->event);

	ast_assert(esc != NULL);
	if (is_linked) {
		ao2_unlink(esc->compositor, esc_entry);
	}
	snprintf(esc_entry->entity_tag, sizeof(esc_entry->entity_tag), "%d", new_etag);
	ao2_link(esc->compositor, esc_entry);
}

static struct sip_esc_entry *create_esc_entry(struct event_state_compositor *esc, struct sip_request *req, const int expires)
{
	struct sip_esc_entry *esc_entry;
	int expires_ms;

	if (!(esc_entry = ao2_alloc(sizeof(*esc_entry), esc_entry_destructor))) {
		return NULL;
	}

	esc_entry->event = esc->name;

	expires_ms = expires * 1000;
	/* Bump refcount for scheduler */
	ao2_ref(esc_entry, +1);
	esc_entry->sched_id = ast_sched_add(sched, expires_ms, publish_expire, esc_entry);

	/* Note: This links the esc_entry into the ESC properly */
	create_new_sip_etag(esc_entry, 0);

	return esc_entry;
}

static int initialize_escs(void)
{
	int i, res = 0;
	for (i = 0; i < ARRAY_LEN(event_state_compositors); i++) {
		if (!((event_state_compositors[i].compositor) =
					ao2_container_alloc(ESC_MAX_BUCKETS, esc_hash_fn, esc_cmp_fn))) {
			res = -1;
		}
	}
	return res;
}

static void destroy_escs(void)
{
	int i;
	for (i = 0; i < ARRAY_LEN(event_state_compositors); i++) {
		ao2_ref(event_state_compositors[i].compositor, -1);
	}
}

/*!
 * \details
 * This container holds the dialogs that will be destroyed immediately.
 */
struct ao2_container *dialogs_to_destroy;

/*! \brief
 * Here we implement the container for dialogs (sip_pvt), defining
 * generic wrapper functions to ease the transition from the current
 * implementation (a single linked list) to a different container.
 * In addition to a reference to the container, we need functions to lock/unlock
 * the container and individual items, and functions to add/remove
 * references to the individual items.
 */
static struct ao2_container *dialogs;
#define sip_pvt_lock(x) ao2_lock(x)
#define sip_pvt_trylock(x) ao2_trylock(x)
#define sip_pvt_unlock(x) ao2_unlock(x)

/*! \brief  The table of TCP threads */
static struct ao2_container *threadt;

/*! \brief  The peer list: Users, Peers and Friends */
static struct ao2_container *peers;
static struct ao2_container *peers_by_ip;

/*! \brief  The register list: Other SIP proxies we register with and receive calls from */
static struct ast_register_list {
	ASTOBJ_CONTAINER_COMPONENTS(struct sip_registry);
	int recheck;
} regl;

/*! \brief  The MWI subscription list */
static struct ast_subscription_mwi_list {
	ASTOBJ_CONTAINER_COMPONENTS(struct sip_subscription_mwi);
} submwil;
static int temp_pvt_init(void *);
static void temp_pvt_cleanup(void *);

/*! \brief A per-thread temporary pvt structure */
AST_THREADSTORAGE_CUSTOM(ts_temp_pvt, temp_pvt_init, temp_pvt_cleanup);

/*! \brief Authentication container for realm authentication */
static struct sip_auth_container *authl = NULL;
/*! \brief Global authentication container protection while adjusting the references. */
AST_MUTEX_DEFINE_STATIC(authl_lock);

/* --- Sockets and networking --------------*/

/*! \brief Main socket for UDP SIP communication.
 *
 * sipsock is shared between the SIP manager thread (which handles reload
 * requests), the udp io handler (sipsock_read()) and the user routines that
 * issue udp writes (using __sip_xmit()).
 * The socket is -1 only when opening fails (this is a permanent condition),
 * or when we are handling a reload() that changes its address (this is
 * a transient situation during which we might have a harmless race, see
 * below). Because the conditions for the race to be possible are extremely
 * rare, we don't want to pay the cost of locking on every I/O.
 * Rather, we remember that when the race may occur, communication is
 * bound to fail anyways, so we just live with this event and let
 * the protocol handle this above us.
 */
static int sipsock  = -1;

struct ast_sockaddr bindaddr;	/*!< UDP: The address we bind to */

/*! \brief our (internal) default address/port to put in SIP/SDP messages
 *  internip is initialized picking a suitable address from one of the
 * interfaces, and the same port number we bind to. It is used as the
 * default address/port in SIP messages, and as the default address
 * (but not port) in SDP messages.
 */
static struct ast_sockaddr internip;

/*! \brief our external IP address/port for SIP sessions.
 * externaddr.sin_addr is only set when we know we might be behind
 * a NAT, and this is done using a variety of (mutually exclusive)
 * ways from the config file:
 *
 * + with "externaddr = host[:port]" we specify the address/port explicitly.
 *   The address is looked up only once when (re)loading the config file;
 *
 * + with "externhost = host[:port]" we do a similar thing, but the
 *   hostname is stored in externhost, and the hostname->IP mapping
 *   is refreshed every 'externrefresh' seconds;
 *
 * Other variables (externhost, externexpire, externrefresh) are used
 * to support the above functions.
 */
static struct ast_sockaddr externaddr;      /*!< External IP address if we are behind NAT */
static struct ast_sockaddr media_address; /*!< External RTP IP address if we are behind NAT */

static char externhost[MAXHOSTNAMELEN];   /*!< External host name */
static time_t externexpire;             /*!< Expiration counter for re-resolving external host name in dynamic DNS */
static int externrefresh = 10;          /*!< Refresh timer for DNS-based external address (dyndns) */
static uint16_t externtcpport;          /*!< external tcp port */ 
static uint16_t externtlsport;          /*!< external tls port */

/*! \brief  List of local networks
 * We store "localnet" addresses from the config file into an access list,
 * marked as 'DENY', so the call to ast_apply_ha() will return
 * AST_SENSE_DENY for 'local' addresses, and AST_SENSE_ALLOW for 'non local'
 * (i.e. presumably public) addresses.
 */
static struct ast_ha *localaddr;    /*!< List of local networks, on the same side of NAT as this Asterisk */

static int ourport_tcp;             /*!< The port used for TCP connections */
static int ourport_tls;             /*!< The port used for TCP/TLS connections */
static struct ast_sockaddr debugaddr;

static struct ast_config *notify_types = NULL;    /*!< The list of manual NOTIFY types we know how to send */

/*! some list management macros. */

#define UNLINK(element, head, prev) do {	\
	if (prev)				\
		(prev)->next = (element)->next;	\
	else					\
		(head) = (element)->next;	\
	} while (0)

/*---------------------------- Forward declarations of functions in chan_sip.c */
/* Note: This is added to help splitting up chan_sip.c into several files
	in coming releases. */

/*--- PBX interface functions */
static struct ast_channel *sip_request_call(const char *type, format_t format, const struct ast_channel *requestor, void *data, int *cause);
static int sip_devicestate(void *data);
static int sip_sendtext(struct ast_channel *ast, const char *text);
static int sip_call(struct ast_channel *ast, char *dest, int timeout);
static int sip_sendhtml(struct ast_channel *chan, int subclass, const char *data, int datalen);
static int sip_hangup(struct ast_channel *ast);
static int sip_answer(struct ast_channel *ast);
static struct ast_frame *sip_read(struct ast_channel *ast);
static int sip_write(struct ast_channel *ast, struct ast_frame *frame);
static int sip_indicate(struct ast_channel *ast, int condition, const void *data, size_t datalen);
static int sip_transfer(struct ast_channel *ast, const char *dest);
static int sip_fixup(struct ast_channel *oldchan, struct ast_channel *newchan);
static int sip_senddigit_begin(struct ast_channel *ast, char digit);
static int sip_senddigit_end(struct ast_channel *ast, char digit, unsigned int duration);
static int sip_setoption(struct ast_channel *chan, int option, void *data, int datalen);
static int sip_queryoption(struct ast_channel *chan, int option, void *data, int *datalen);
static const char *sip_get_callid(struct ast_channel *chan);

static int handle_request_do(struct sip_request *req, struct ast_sockaddr *addr);
static int sip_standard_port(enum sip_transport type, int port);
static int sip_prepare_socket(struct sip_pvt *p);
static int get_address_family_filter(const struct ast_sockaddr *addr);

/*--- Transmitting responses and requests */
static int sipsock_read(int *id, int fd, short events, void *ignore);
static int __sip_xmit(struct sip_pvt *p, struct ast_str *data);
static int __sip_reliable_xmit(struct sip_pvt *p, uint32_t seqno, int resp, struct ast_str *data, int fatal, int sipmethod);
static void add_cc_call_info_to_response(struct sip_pvt *p, struct sip_request *resp);
static int __transmit_response(struct sip_pvt *p, const char *msg, const struct sip_request *req, enum xmittype reliable);
static int retrans_pkt(const void *data);
static int transmit_response_using_temp(ast_string_field callid, struct ast_sockaddr *addr, int useglobal_nat, const int intended_method, const struct sip_request *req, const char *msg);
static int transmit_response(struct sip_pvt *p, const char *msg, const struct sip_request *req);
static int transmit_response_reliable(struct sip_pvt *p, const char *msg, const struct sip_request *req);
static int transmit_response_with_date(struct sip_pvt *p, const char *msg, const struct sip_request *req);
static int transmit_response_with_sdp(struct sip_pvt *p, const char *msg, const struct sip_request *req, enum xmittype reliable, int oldsdp, int rpid);
static int transmit_response_with_unsupported(struct sip_pvt *p, const char *msg, const struct sip_request *req, const char *unsupported);
static int transmit_response_with_auth(struct sip_pvt *p, const char *msg, const struct sip_request *req, const char *rand, enum xmittype reliable, const char *header, int stale);
static int transmit_provisional_response(struct sip_pvt *p, const char *msg, const struct sip_request *req, int with_sdp);
static int transmit_response_with_allow(struct sip_pvt *p, const char *msg, const struct sip_request *req, enum xmittype reliable);
static void transmit_fake_auth_response(struct sip_pvt *p, int sipmethod, struct sip_request *req, enum xmittype reliable);
static int transmit_request(struct sip_pvt *p, int sipmethod, uint32_t seqno, enum xmittype reliable, int newbranch);
static int transmit_request_with_auth(struct sip_pvt *p, int sipmethod, uint32_t seqno, enum xmittype reliable, int newbranch);
static int transmit_publish(struct sip_epa_entry *epa_entry, enum sip_publish_type publish_type, const char * const explicit_uri);
static int transmit_invite(struct sip_pvt *p, int sipmethod, int sdp, int init, const char * const explicit_uri);
static int transmit_reinvite_with_sdp(struct sip_pvt *p, int t38version, int oldsdp);
static int transmit_info_with_aoc(struct sip_pvt *p, struct ast_aoc_decoded *decoded);
static int transmit_info_with_digit(struct sip_pvt *p, const char digit, unsigned int duration);
static int transmit_info_with_vidupdate(struct sip_pvt *p);
static int transmit_message_with_text(struct sip_pvt *p, const char *text);
static int transmit_refer(struct sip_pvt *p, const char *dest);
static int transmit_notify_with_mwi(struct sip_pvt *p, int newmsgs, int oldmsgs, const char *vmexten);
static int transmit_notify_with_sipfrag(struct sip_pvt *p, int cseq, char *message, int terminate);
static int transmit_cc_notify(struct ast_cc_agent *agent, struct sip_pvt *subscription, enum sip_cc_notify_state state);
static int transmit_register(struct sip_registry *r, int sipmethod, const char *auth, const char *authheader);
static int send_response(struct sip_pvt *p, struct sip_request *req, enum xmittype reliable, uint32_t seqno);
static int send_request(struct sip_pvt *p, struct sip_request *req, enum xmittype reliable, uint32_t seqno);
static void copy_request(struct sip_request *dst, const struct sip_request *src);
static void receive_message(struct sip_pvt *p, struct sip_request *req);
static void parse_moved_contact(struct sip_pvt *p, struct sip_request *req, char **name, char **number, int set_call_forward);
static int sip_send_mwi_to_peer(struct sip_peer *peer, int cache_only);

/* Misc dialog routines */
static int __sip_autodestruct(const void *data);
static void *registry_unref(struct sip_registry *reg, char *tag);
static int update_call_counter(struct sip_pvt *fup, int event);
static int auto_congest(const void *arg);
static struct sip_pvt *find_call(struct sip_request *req, struct ast_sockaddr *addr, const int intended_method);
static void free_old_route(struct sip_route *route);
static void list_route(struct sip_route *route);
static void build_route(struct sip_pvt *p, struct sip_request *req, int backwards, int resp);
static enum check_auth_result register_verify(struct sip_pvt *p, struct ast_sockaddr *addr,
					      struct sip_request *req, const char *uri);
static struct sip_pvt *get_sip_pvt_byid_locked(const char *callid, const char *totag, const char *fromtag);
static void check_pendings(struct sip_pvt *p);
static void *sip_park_thread(void *stuff);
static int sip_park(struct ast_channel *chan1, struct ast_channel *chan2, struct sip_request *req, uint32_t seqno, const char *park_exten, const char *park_context);

static void *sip_pickup_thread(void *stuff);
static int sip_pickup(struct ast_channel *chan);

static int sip_sipredirect(struct sip_pvt *p, const char *dest);
static int is_method_allowed(unsigned int *allowed_methods, enum sipmethod method);

/*--- Codec handling / SDP */
static void try_suggested_sip_codec(struct sip_pvt *p);
static const char *get_sdp_iterate(int* start, struct sip_request *req, const char *name);
static char get_sdp_line(int *start, int stop, struct sip_request *req, const char **value);
static int find_sdp(struct sip_request *req);
static int process_sdp(struct sip_pvt *p, struct sip_request *req, int t38action);
static int process_sdp_o(const char *o, struct sip_pvt *p);
static int process_sdp_c(const char *c, struct ast_sockaddr *addr);
static int process_sdp_a_sendonly(const char *a, int *sendonly);
static int process_sdp_a_audio(const char *a, struct sip_pvt *p, struct ast_rtp_codecs *newaudiortp, int *last_rtpmap_codec);
static int process_sdp_a_video(const char *a, struct sip_pvt *p, struct ast_rtp_codecs *newvideortp, int *last_rtpmap_codec);
static int process_sdp_a_text(const char *a, struct sip_pvt *p, struct ast_rtp_codecs *newtextrtp, char *red_fmtp, int *red_num_gen, int *red_data_pt, int *last_rtpmap_codec);
static int process_sdp_a_image(const char *a, struct sip_pvt *p);
static void add_codec_to_sdp(const struct sip_pvt *p, format_t codec,
			     struct ast_str **m_buf, struct ast_str **a_buf,
			     int debug, int *min_packet_size);
static void add_noncodec_to_sdp(const struct sip_pvt *p, int format,
				struct ast_str **m_buf, struct ast_str **a_buf,
				int debug);
static enum sip_result add_sdp(struct sip_request *resp, struct sip_pvt *p, int oldsdp, int add_audio, int add_t38);
static void do_setnat(struct sip_pvt *p);
static void stop_media_flows(struct sip_pvt *p);

/*--- Authentication stuff */
static int reply_digest(struct sip_pvt *p, struct sip_request *req, char *header, int sipmethod, char *digest, int digest_len);
static int build_reply_digest(struct sip_pvt *p, int method, char *digest, int digest_len);
static enum check_auth_result check_auth(struct sip_pvt *p, struct sip_request *req, const char *username,
					 const char *secret, const char *md5secret, int sipmethod,
					 const char *uri, enum xmittype reliable, int ignore);
static enum check_auth_result check_user_full(struct sip_pvt *p, struct sip_request *req,
					      int sipmethod, const char *uri, enum xmittype reliable,
					      struct ast_sockaddr *addr, struct sip_peer **authpeer);
static int check_user(struct sip_pvt *p, struct sip_request *req, int sipmethod, const char *uri, enum xmittype reliable, struct ast_sockaddr *addr);

/*--- Domain handling */
static int check_sip_domain(const char *domain, char *context, size_t len); /* Check if domain is one of our local domains */
static int add_sip_domain(const char *domain, const enum domain_mode mode, const char *context);
static void clear_sip_domains(void);

/*--- SIP realm authentication */
static void add_realm_authentication(struct sip_auth_container **credentials, const char *configuration, int lineno);
static struct sip_auth *find_realm_authentication(struct sip_auth_container *credentials, const char *realm);

/*--- Misc functions */
static void check_rtp_timeout(struct sip_pvt *dialog, time_t t);
static int reload_config(enum channelreloadreason reason);
static void add_diversion_header(struct sip_request *req, struct sip_pvt *pvt);
static int expire_register(const void *data);
static void *do_monitor(void *data);
static int restart_monitor(void);
static void peer_mailboxes_to_str(struct ast_str **mailbox_str, struct sip_peer *peer);
static struct ast_variable *copy_vars(struct ast_variable *src);
static int dialog_find_multiple(void *obj, void *arg, int flags);
static struct ast_channel *sip_pvt_lock_full(struct sip_pvt *pvt);
/* static int sip_addrcmp(char *name, struct sockaddr_in *sin);	Support for peer matching */
static int sip_refer_allocate(struct sip_pvt *p);
static int sip_notify_allocate(struct sip_pvt *p);
static void ast_quiet_chan(struct ast_channel *chan);
static int attempt_transfer(struct sip_dual *transferer, struct sip_dual *target);
static int do_magic_pickup(struct ast_channel *channel, const char *extension, const char *context);

/*--- Device monitoring and Device/extension state/event handling */
static int cb_extensionstate(char *context, char* exten, int state, void *data);
static int sip_devicestate(void *data);
static int sip_poke_noanswer(const void *data);
static int sip_poke_peer(struct sip_peer *peer, int force);
static void sip_poke_all_peers(void);
static void sip_peer_hold(struct sip_pvt *p, int hold);
static void mwi_event_cb(const struct ast_event *, void *);
static void network_change_event_cb(const struct ast_event *, void *);

/*--- Applications, functions, CLI and manager command helpers */
static const char *sip_nat_mode(const struct sip_pvt *p);
static char *sip_show_inuse(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a);
static char *transfermode2str(enum transfermodes mode) attribute_const;
static int peer_status(struct sip_peer *peer, char *status, int statuslen);
static char *sip_show_sched(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a);
static char * _sip_show_peers(int fd, int *total, struct mansession *s, const struct message *m, int argc, const char *argv[]);
static char *sip_show_peers(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a);
static char *sip_show_objects(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a);
static void  print_group(int fd, ast_group_t group, int crlf);
static const char *dtmfmode2str(int mode) attribute_const;
static int str2dtmfmode(const char *str) attribute_unused;
static const char *insecure2str(int mode) attribute_const;
static const char *allowoverlap2str(int mode) attribute_const;
static void cleanup_stale_contexts(char *new, char *old);
static void print_codec_to_cli(int fd, struct ast_codec_pref *pref);
static const char *domain_mode_to_text(const enum domain_mode mode);
static char *sip_show_domains(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a);
static char *_sip_show_peer(int type, int fd, struct mansession *s, const struct message *m, int argc, const char *argv[]);
static char *sip_show_peer(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a);
static char *_sip_qualify_peer(int type, int fd, struct mansession *s, const struct message *m, int argc, const char *argv[]);
static char *sip_qualify_peer(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a);
static char *sip_show_registry(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a);
static char *sip_unregister(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a);
static char *sip_show_settings(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a);
static char *sip_show_mwi(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a);
static const char *subscription_type2str(enum subscriptiontype subtype) attribute_pure;
static const struct cfsubscription_types *find_subscription_type(enum subscriptiontype subtype);
static char *complete_sip_peer(const char *word, int state, int flags2);
static char *complete_sip_registered_peer(const char *word, int state, int flags2);
static char *complete_sip_show_history(const char *line, const char *word, int pos, int state);
static char *complete_sip_show_peer(const char *line, const char *word, int pos, int state);
static char *complete_sip_unregister(const char *line, const char *word, int pos, int state);
static char *complete_sipnotify(const char *line, const char *word, int pos, int state);
static char *sip_show_channel(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a);
static char *sip_show_channelstats(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a);
static char *sip_show_history(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a);
static char *sip_do_debug_ip(int fd, const char *arg);
static char *sip_do_debug_peer(int fd, const char *arg);
static char *sip_do_debug(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a);
static char *sip_cli_notify(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a);
static char *sip_set_history(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a);
static int sip_dtmfmode(struct ast_channel *chan, const char *data);
static int sip_addheader(struct ast_channel *chan, const char *data);
static int sip_do_reload(enum channelreloadreason reason);
static char *sip_reload(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a);
static int ast_sockaddr_resolve_first_af(struct ast_sockaddr *addr,
				      const char *name, int flag, int family);
static int ast_sockaddr_resolve_first(struct ast_sockaddr *addr,
				      const char *name, int flag);

/*--- Debugging
	Functions for enabling debug per IP or fully, or enabling history logging for
	a SIP dialog
*/
static void sip_dump_history(struct sip_pvt *dialog);	/* Dump history to debuglog at end of dialog, before destroying data */
static inline int sip_debug_test_addr(const struct ast_sockaddr *addr);
static inline int sip_debug_test_pvt(struct sip_pvt *p);
static void append_history_full(struct sip_pvt *p, const char *fmt, ...);
static void sip_dump_history(struct sip_pvt *dialog);

/*--- Device object handling */
static struct sip_peer *build_peer(const char *name, struct ast_variable *v, struct ast_variable *alt, int realtime, int devstate_only);
static int update_call_counter(struct sip_pvt *fup, int event);
static void sip_destroy_peer(struct sip_peer *peer);
static void sip_destroy_peer_fn(void *peer);
static void set_peer_defaults(struct sip_peer *peer);
static struct sip_peer *temp_peer(const char *name);
static void register_peer_exten(struct sip_peer *peer, int onoff);
static struct sip_peer *find_peer(const char *peer, struct ast_sockaddr *addr, int realtime, int forcenamematch, int devstate_only, int transport);
static int sip_poke_peer_s(const void *data);
static enum parse_register_result parse_register_contact(struct sip_pvt *pvt, struct sip_peer *p, struct sip_request *req);
static void reg_source_db(struct sip_peer *peer);
static void destroy_association(struct sip_peer *peer);
static void set_insecure_flags(struct ast_flags *flags, const char *value, int lineno);
static int handle_common_options(struct ast_flags *flags, struct ast_flags *mask, struct ast_variable *v);
static void set_socket_transport(struct sip_socket *socket, int transport);

/* Realtime device support */
static void realtime_update_peer(const char *peername, struct ast_sockaddr *addr, const char *username, const char *fullcontact, const char *useragent, int expirey, unsigned short deprecated_username, int lastms);
static void update_peer(struct sip_peer *p, int expire);
static struct ast_variable *get_insecure_variable_from_config(struct ast_config *config);
static const char *get_name_from_variable(const struct ast_variable *var);
static struct sip_peer *realtime_peer(const char *peername, struct ast_sockaddr *sin, int devstate_only, int which_objects);
static char *sip_prune_realtime(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a);

/*--- Internal UA client handling (outbound registrations) */
static void ast_sip_ouraddrfor(const struct ast_sockaddr *them, struct ast_sockaddr *us, struct sip_pvt *p);
static void sip_registry_destroy(struct sip_registry *reg);
static int sip_register(const char *value, int lineno);
static const char *regstate2str(enum sipregistrystate regstate) attribute_const;
static int sip_reregister(const void *data);
static int __sip_do_register(struct sip_registry *r);
static int sip_reg_timeout(const void *data);
static void sip_send_all_registers(void);
static int sip_reinvite_retry(const void *data);

/*--- Parsing SIP requests and responses */
static void append_date(struct sip_request *req);	/* Append date to SIP packet */
static int determine_firstline_parts(struct sip_request *req);
static const struct cfsubscription_types *find_subscription_type(enum subscriptiontype subtype);
static const char *gettag(const struct sip_request *req, const char *header, char *tagbuf, int tagbufsize);
static int find_sip_method(const char *msg);
static unsigned int parse_allowed_methods(struct sip_request *req);
static unsigned int set_pvt_allowed_methods(struct sip_pvt *pvt, struct sip_request *req);
static int parse_request(struct sip_request *req);
static const char *get_header(const struct sip_request *req, const char *name);
static const char *referstatus2str(enum referstatus rstatus) attribute_pure;
static int method_match(enum sipmethod id, const char *name);
static void parse_copy(struct sip_request *dst, const struct sip_request *src);
static const char *find_alias(const char *name, const char *_default);
static const char *__get_header(const struct sip_request *req, const char *name, int *start);
static void lws2sws(struct ast_str *msgbuf);
static void extract_uri(struct sip_pvt *p, struct sip_request *req);
static char *remove_uri_parameters(char *uri);
static int get_refer_info(struct sip_pvt *transferer, struct sip_request *outgoing_req);
static int get_also_info(struct sip_pvt *p, struct sip_request *oreq);
static int parse_ok_contact(struct sip_pvt *pvt, struct sip_request *req);
static int set_address_from_contact(struct sip_pvt *pvt);
static void check_via(struct sip_pvt *p, struct sip_request *req);
static int get_rpid(struct sip_pvt *p, struct sip_request *oreq);
static int get_rdnis(struct sip_pvt *p, struct sip_request *oreq, char **name, char **number, int *reason);
static enum sip_get_dest_result get_destination(struct sip_pvt *p, struct sip_request *oreq, int *cc_recall_core_id);
static int get_msg_text(char *buf, int len, struct sip_request *req);
static int transmit_state_notify(struct sip_pvt *p, int state, int full, int timeout);
static void update_connectedline(struct sip_pvt *p, const void *data, size_t datalen);
static void update_redirecting(struct sip_pvt *p, const void *data, size_t datalen);
static int get_domain(const char *str, char *domain, int len);
static void get_realm(struct sip_pvt *p, const struct sip_request *req);

/*-- TCP connection handling ---*/
static void *_sip_tcp_helper_thread(struct ast_tcptls_session_instance *tcptls_session);
static void *sip_tcp_worker_fn(void *);

/*--- Constructing requests and responses */
static void initialize_initreq(struct sip_pvt *p, struct sip_request *req);
static int init_req(struct sip_request *req, int sipmethod, const char *recip);
static void deinit_req(struct sip_request *req);
static int reqprep(struct sip_request *req, struct sip_pvt *p, int sipmethod, uint32_t seqno, int newbranch);
static void initreqprep(struct sip_request *req, struct sip_pvt *p, int sipmethod, const char * const explicit_uri);
static int init_resp(struct sip_request *resp, const char *msg);
static inline int resp_needs_contact(const char *msg, enum sipmethod method);
static int respprep(struct sip_request *resp, struct sip_pvt *p, const char *msg, const struct sip_request *req);
static const struct ast_sockaddr *sip_real_dst(const struct sip_pvt *p);
static void build_via(struct sip_pvt *p);
static int create_addr_from_peer(struct sip_pvt *r, struct sip_peer *peer);
static int create_addr(struct sip_pvt *dialog, const char *opeer, struct ast_sockaddr *addr, int newdialog, struct ast_sockaddr *remote_address);
static char *generate_random_string(char *buf, size_t size);
static void build_callid_pvt(struct sip_pvt *pvt);
static void change_callid_pvt(struct sip_pvt *pvt, const char *callid);
static void build_callid_registry(struct sip_registry *reg, const struct ast_sockaddr *ourip, const char *fromdomain);
static void make_our_tag(char *tagbuf, size_t len);
static int add_header(struct sip_request *req, const char *var, const char *value);
static int add_header_max_forwards(struct sip_pvt *dialog, struct sip_request *req);
static int add_content(struct sip_request *req, const char *line);
static int finalize_content(struct sip_request *req);
static int add_text(struct sip_request *req, const char *text);
static int add_digit(struct sip_request *req, char digit, unsigned int duration, int mode);
static int add_rpid(struct sip_request *req, struct sip_pvt *p);
static int add_vidupdate(struct sip_request *req);
static void add_route(struct sip_request *req, struct sip_route *route);
static int copy_header(struct sip_request *req, const struct sip_request *orig, const char *field);
static int copy_all_header(struct sip_request *req, const struct sip_request *orig, const char *field);
static int copy_via_headers(struct sip_pvt *p, struct sip_request *req, const struct sip_request *orig, const char *field);
static void set_destination(struct sip_pvt *p, char *uri);
static void append_date(struct sip_request *req);
static void build_contact(struct sip_pvt *p);

/*------Request handling functions */
static int handle_incoming(struct sip_pvt *p, struct sip_request *req, struct ast_sockaddr *addr, int *recount, int *nounlock);
static int handle_request_update(struct sip_pvt *p, struct sip_request *req);
static int handle_request_invite(struct sip_pvt *p, struct sip_request *req, int debug, uint32_t seqno, struct ast_sockaddr *addr, int *recount, const char *e, int *nounlock);
static int handle_request_refer(struct sip_pvt *p, struct sip_request *req, int debug, uint32_t seqno, int *nounlock);
static int handle_request_bye(struct sip_pvt *p, struct sip_request *req);
static int handle_request_register(struct sip_pvt *p, struct sip_request *req, struct ast_sockaddr *sin, const char *e);
static int handle_request_cancel(struct sip_pvt *p, struct sip_request *req);
static int handle_request_message(struct sip_pvt *p, struct sip_request *req);
static int handle_request_subscribe(struct sip_pvt *p, struct sip_request *req, struct ast_sockaddr *addr, uint32_t seqno, const char *e);
static void handle_request_info(struct sip_pvt *p, struct sip_request *req);
static int handle_request_options(struct sip_pvt *p, struct sip_request *req, struct ast_sockaddr *addr, const char *e);
static int handle_invite_replaces(struct sip_pvt *p, struct sip_request *req, int debug, uint32_t seqno, struct ast_sockaddr *addr, int *nounlock);
static int handle_request_notify(struct sip_pvt *p, struct sip_request *req, struct ast_sockaddr *addr, uint32_t seqno, const char *e);
static int local_attended_transfer(struct sip_pvt *transferer, struct sip_dual *current, struct sip_request *req, uint32_t seqno, int *nounlock);

/*------Response handling functions */
static void handle_response_publish(struct sip_pvt *p, int resp, const char *rest, struct sip_request *req, uint32_t seqno);
static void handle_response_invite(struct sip_pvt *p, int resp, const char *rest, struct sip_request *req, uint32_t seqno);
static void handle_response_notify(struct sip_pvt *p, int resp, const char *rest, struct sip_request *req, uint32_t seqno);
static void handle_response_refer(struct sip_pvt *p, int resp, const char *rest, struct sip_request *req, uint32_t seqno);
static void handle_response_subscribe(struct sip_pvt *p, int resp, const char *rest, struct sip_request *req, uint32_t seqno);
static int handle_response_register(struct sip_pvt *p, int resp, const char *rest, struct sip_request *req, uint32_t seqno);
static void handle_response(struct sip_pvt *p, int resp, const char *rest, struct sip_request *req, uint32_t seqno);

/*------ SRTP Support -------- */
static int setup_srtp(struct sip_srtp **srtp);
static int process_crypto(struct sip_pvt *p, struct ast_rtp_instance *rtp, struct sip_srtp **srtp, const char *a);

/*------ T38 Support --------- */
static int transmit_response_with_t38_sdp(struct sip_pvt *p, char *msg, struct sip_request *req, int retrans);
static struct ast_udptl *sip_get_udptl_peer(struct ast_channel *chan);
static int sip_set_udptl_peer(struct ast_channel *chan, struct ast_udptl *udptl);
static void change_t38_state(struct sip_pvt *p, int state);

/*------ Session-Timers functions --------- */
static void proc_422_rsp(struct sip_pvt *p, struct sip_request *rsp);
static int  proc_session_timer(const void *vp);
static void stop_session_timer(struct sip_pvt *p);
static void start_session_timer(struct sip_pvt *p);
static void restart_session_timer(struct sip_pvt *p);
static const char *strefresher2str(enum st_refresher r);
static int parse_session_expires(const char *p_hdrval, int *const p_interval, enum st_refresher *const p_ref);
static int parse_minse(const char *p_hdrval, int *const p_interval);
static int st_get_se(struct sip_pvt *, int max);
static enum st_refresher st_get_refresher(struct sip_pvt *);
static enum st_mode st_get_mode(struct sip_pvt *, int no_cached);
static struct sip_st_dlg* sip_st_alloc(struct sip_pvt *const p);

/*------- RTP Glue functions -------- */
static int sip_set_rtp_peer(struct ast_channel *chan, struct ast_rtp_instance *instance, struct ast_rtp_instance *vinstance, struct ast_rtp_instance *tinstance, format_t codecs, int nat_active);

/*!--- SIP MWI Subscription support */
static int sip_subscribe_mwi(const char *value, int lineno);
static void sip_subscribe_mwi_destroy(struct sip_subscription_mwi *mwi);
static void sip_send_all_mwi_subscriptions(void);
static int sip_subscribe_mwi_do(const void *data);
static int __sip_subscribe_mwi_do(struct sip_subscription_mwi *mwi);

/*! \brief Definition of this channel for PBX channel registration */
const struct ast_channel_tech sip_tech = {
	.type = "SIP",
	.description = "Session Initiation Protocol (SIP)",
	.capabilities = AST_FORMAT_AUDIO_MASK,	/* all audio formats */
	.properties = AST_CHAN_TP_WANTSJITTER | AST_CHAN_TP_CREATESJITTER,
	.requester = sip_request_call,			/* called with chan unlocked */
	.devicestate = sip_devicestate,			/* called with chan unlocked (not chan-specific) */
	.call = sip_call,			/* called with chan locked */
	.send_html = sip_sendhtml,
	.hangup = sip_hangup,			/* called with chan locked */
	.answer = sip_answer,			/* called with chan locked */
	.read = sip_read,			/* called with chan locked */
	.write = sip_write,			/* called with chan locked */
	.write_video = sip_write,		/* called with chan locked */
	.write_text = sip_write,
	.indicate = sip_indicate,		/* called with chan locked */
	.transfer = sip_transfer,		/* called with chan locked */
	.fixup = sip_fixup,			/* called with chan locked */
	.send_digit_begin = sip_senddigit_begin,	/* called with chan unlocked */
	.send_digit_end = sip_senddigit_end,
	.bridge = ast_rtp_instance_bridge,			/* XXX chan unlocked ? */
	.early_bridge = ast_rtp_instance_early_bridge,
	.send_text = sip_sendtext,		/* called with chan locked */
	.func_channel_read = sip_acf_channel_read,
	.setoption = sip_setoption,
	.queryoption = sip_queryoption,
	.get_pvt_uniqueid = sip_get_callid,
};

/*! \brief This version of the sip channel tech has no send_digit_begin
 * callback so that the core knows that the channel does not want
 * DTMF BEGIN frames.
 * The struct is initialized just before registering the channel driver,
 * and is for use with channels using SIP INFO DTMF.
 */
struct ast_channel_tech sip_tech_info;

static int sip_cc_agent_init(struct ast_cc_agent *agent, struct ast_channel *chan);
static int sip_cc_agent_start_offer_timer(struct ast_cc_agent *agent);
static int sip_cc_agent_stop_offer_timer(struct ast_cc_agent *agent);
static void sip_cc_agent_respond(struct ast_cc_agent *agent, enum ast_cc_agent_response_reason reason);
static int sip_cc_agent_status_request(struct ast_cc_agent *agent);
static int sip_cc_agent_start_monitoring(struct ast_cc_agent *agent);
static int sip_cc_agent_recall(struct ast_cc_agent *agent);
static void sip_cc_agent_destructor(struct ast_cc_agent *agent);

static struct ast_cc_agent_callbacks sip_cc_agent_callbacks = {
	.type = "SIP",
	.init = sip_cc_agent_init,
	.start_offer_timer = sip_cc_agent_start_offer_timer,
	.stop_offer_timer = sip_cc_agent_stop_offer_timer,
	.respond = sip_cc_agent_respond,
	.status_request = sip_cc_agent_status_request,
	.start_monitoring = sip_cc_agent_start_monitoring,
	.callee_available = sip_cc_agent_recall,
	.destructor = sip_cc_agent_destructor,
};

static int find_by_notify_uri_helper(void *obj, void *arg, int flags)
{
	struct ast_cc_agent *agent = obj;
	struct sip_cc_agent_pvt *agent_pvt = agent->private_data;
	const char *uri = arg;

	return !sip_uri_cmp(agent_pvt->notify_uri, uri) ? CMP_MATCH | CMP_STOP : 0;
}

static struct ast_cc_agent *find_sip_cc_agent_by_notify_uri(const char * const uri)
{
	struct ast_cc_agent *agent = ast_cc_agent_callback(0, find_by_notify_uri_helper, (char *)uri, "SIP");
	return agent;
}

static int find_by_subscribe_uri_helper(void *obj, void *arg, int flags)
{
	struct ast_cc_agent *agent = obj;
	struct sip_cc_agent_pvt *agent_pvt = agent->private_data;
	const char *uri = arg;

	return !sip_uri_cmp(agent_pvt->subscribe_uri, uri) ? CMP_MATCH | CMP_STOP : 0;
}

static struct ast_cc_agent *find_sip_cc_agent_by_subscribe_uri(const char * const uri)
{
	struct ast_cc_agent *agent = ast_cc_agent_callback(0, find_by_subscribe_uri_helper, (char *)uri, "SIP");
	return agent;
}

static int find_by_callid_helper(void *obj, void *arg, int flags)
{
	struct ast_cc_agent *agent = obj;
	struct sip_cc_agent_pvt *agent_pvt = agent->private_data;
	struct sip_pvt *call_pvt = arg;

	return !strcmp(agent_pvt->original_callid, call_pvt->callid) ? CMP_MATCH | CMP_STOP : 0;
}

static struct ast_cc_agent *find_sip_cc_agent_by_original_callid(struct sip_pvt *pvt)
{
	struct ast_cc_agent *agent = ast_cc_agent_callback(0, find_by_callid_helper, pvt, "SIP");
	return agent;
}

static int sip_cc_agent_init(struct ast_cc_agent *agent, struct ast_channel *chan)
{
	struct sip_cc_agent_pvt *agent_pvt = ast_calloc(1, sizeof(*agent_pvt));
	struct sip_pvt *call_pvt = chan->tech_pvt;

	if (!agent_pvt) {
		return -1;
	}

	ast_assert(!strcmp(chan->tech->type, "SIP"));

	ast_copy_string(agent_pvt->original_callid, call_pvt->callid, sizeof(agent_pvt->original_callid));
	ast_copy_string(agent_pvt->original_exten, call_pvt->exten, sizeof(agent_pvt->original_exten));
	agent_pvt->offer_timer_id = -1;
	agent->private_data = agent_pvt;
	sip_pvt_lock(call_pvt);
	ast_set_flag(&call_pvt->flags[0], SIP_OFFER_CC);
	sip_pvt_unlock(call_pvt);
	return 0;
}

static int sip_offer_timer_expire(const void *data)
{
	struct ast_cc_agent *agent = (struct ast_cc_agent *) data;
	struct sip_cc_agent_pvt *agent_pvt = agent->private_data;

	agent_pvt->offer_timer_id = -1;

	return ast_cc_failed(agent->core_id, "SIP agent %s's offer timer expired", agent->device_name);
}

static int sip_cc_agent_start_offer_timer(struct ast_cc_agent *agent)
{
	struct sip_cc_agent_pvt *agent_pvt = agent->private_data;
	int when;

	when = ast_get_cc_offer_timer(agent->cc_params) * 1000;
	agent_pvt->offer_timer_id = ast_sched_add(sched, when, sip_offer_timer_expire, agent);
	return 0;
}

static int sip_cc_agent_stop_offer_timer(struct ast_cc_agent *agent)
{
	struct sip_cc_agent_pvt *agent_pvt = agent->private_data;

	AST_SCHED_DEL(sched, agent_pvt->offer_timer_id);
	return 0;
}

static void sip_cc_agent_respond(struct ast_cc_agent *agent, enum ast_cc_agent_response_reason reason)
{
	struct sip_cc_agent_pvt *agent_pvt = agent->private_data;

	sip_pvt_lock(agent_pvt->subscribe_pvt);
	ast_set_flag(&agent_pvt->subscribe_pvt->flags[1], SIP_PAGE2_DIALOG_ESTABLISHED);
	if (reason == AST_CC_AGENT_RESPONSE_SUCCESS || !ast_strlen_zero(agent_pvt->notify_uri)) {
		/* The second half of this if statement may be a bit hard to grasp,
		 * so here's an explanation. When a subscription comes into
		 * chan_sip, as long as it is not malformed, it will be passed
		 * to the CC core. If the core senses an out-of-order state transition,
		 * then the core will call this callback with the "reason" set to a
		 * failure condition.
		 * However, an out-of-order state transition will occur during a resubscription
		 * for CC. In such a case, we can see that we have already generated a notify_uri
		 * and so we can detect that this isn't a *real* failure. Rather, it is just
		 * something the core doesn't recognize as a legitimate SIP state transition.
		 * Thus we respond with happiness and flowers.
		 */
		transmit_response(agent_pvt->subscribe_pvt, "200 OK", &agent_pvt->subscribe_pvt->initreq);
		transmit_cc_notify(agent, agent_pvt->subscribe_pvt, CC_QUEUED);
	} else {
		transmit_response(agent_pvt->subscribe_pvt, "500 Internal Error", &agent_pvt->subscribe_pvt->initreq);
	}
	sip_pvt_unlock(agent_pvt->subscribe_pvt);
	agent_pvt->is_available = TRUE;
}

static int sip_cc_agent_status_request(struct ast_cc_agent *agent)
{
	struct sip_cc_agent_pvt *agent_pvt = agent->private_data;
	enum ast_device_state state = agent_pvt->is_available ? AST_DEVICE_NOT_INUSE : AST_DEVICE_INUSE;
	return ast_cc_agent_status_response(agent->core_id, state);
}

static int sip_cc_agent_start_monitoring(struct ast_cc_agent *agent)
{
	/* To start monitoring just means to wait for an incoming PUBLISH
	 * to tell us that the caller has become available again. No special
	 * action is needed
	 */
	return 0;
}

static int sip_cc_agent_recall(struct ast_cc_agent *agent)
{
	struct sip_cc_agent_pvt *agent_pvt = agent->private_data;
	/* If we have received a PUBLISH beforehand stating that the caller in question
	 * is not available, we can save ourself a bit of effort here and just report
	 * the caller as busy
	 */
	if (!agent_pvt->is_available) {
		return ast_cc_agent_caller_busy(agent->core_id, "Caller %s is busy, reporting to the core",
				agent->device_name);
	}
	/* Otherwise, we transmit a NOTIFY to the caller and await either
	 * a PUBLISH or an INVITE
	 */
	sip_pvt_lock(agent_pvt->subscribe_pvt);
	transmit_cc_notify(agent, agent_pvt->subscribe_pvt, CC_READY);
	sip_pvt_unlock(agent_pvt->subscribe_pvt);
	return 0;
}

static void sip_cc_agent_destructor(struct ast_cc_agent *agent)
{
	struct sip_cc_agent_pvt *agent_pvt = agent->private_data;

	if (!agent_pvt) {
		/* The agent constructor probably failed. */
		return;
	}

	sip_cc_agent_stop_offer_timer(agent);
	if (agent_pvt->subscribe_pvt) {
		sip_pvt_lock(agent_pvt->subscribe_pvt);
		if (!ast_test_flag(&agent_pvt->subscribe_pvt->flags[1], SIP_PAGE2_DIALOG_ESTABLISHED)) {
			/* If we haven't sent a 200 OK for the SUBSCRIBE dialog yet, then we need to send a response letting
			 * the subscriber know something went wrong
			 */
			transmit_response(agent_pvt->subscribe_pvt, "500 Internal Server Error", &agent_pvt->subscribe_pvt->initreq);
		}
		sip_pvt_unlock(agent_pvt->subscribe_pvt);
		agent_pvt->subscribe_pvt = dialog_unref(agent_pvt->subscribe_pvt, "SIP CC agent destructor: Remove ref to subscription");
	}
	ast_free(agent_pvt);
}

struct ao2_container *sip_monitor_instances;

static int sip_monitor_instance_hash_fn(const void *obj, const int flags)
{
	const struct sip_monitor_instance *monitor_instance = obj;
	return monitor_instance->core_id;
}

static int sip_monitor_instance_cmp_fn(void *obj, void *arg, int flags)
{
	struct sip_monitor_instance *monitor_instance1 = obj;
	struct sip_monitor_instance *monitor_instance2 = arg;

	return monitor_instance1->core_id == monitor_instance2->core_id ? CMP_MATCH | CMP_STOP : 0;
}

static void sip_monitor_instance_destructor(void *data)
{
	struct sip_monitor_instance *monitor_instance = data;
	if (monitor_instance->subscription_pvt) {
		sip_pvt_lock(monitor_instance->subscription_pvt);
		monitor_instance->subscription_pvt->expiry = 0;
		transmit_invite(monitor_instance->subscription_pvt, SIP_SUBSCRIBE, FALSE, 0, monitor_instance->subscribe_uri);
		sip_pvt_unlock(monitor_instance->subscription_pvt);
		dialog_unref(monitor_instance->subscription_pvt, "Unref monitor instance ref of subscription pvt");
	}
	if (monitor_instance->suspension_entry) {
		monitor_instance->suspension_entry->body[0] = '\0';
		transmit_publish(monitor_instance->suspension_entry, SIP_PUBLISH_REMOVE ,monitor_instance->notify_uri);
		ao2_t_ref(monitor_instance->suspension_entry, -1, "Decrementing suspension entry refcount in sip_monitor_instance_destructor");
	}
	ast_string_field_free_memory(monitor_instance);
}

static struct sip_monitor_instance *sip_monitor_instance_init(int core_id, const char * const subscribe_uri, const char * const peername, const char * const device_name)
{
	struct sip_monitor_instance *monitor_instance = ao2_alloc(sizeof(*monitor_instance), sip_monitor_instance_destructor);

	if (!monitor_instance) {
		return NULL;
	}

	if (ast_string_field_init(monitor_instance, 256)) {
		ao2_ref(monitor_instance, -1);
		return NULL;
	}

	ast_string_field_set(monitor_instance, subscribe_uri, subscribe_uri);
	ast_string_field_set(monitor_instance, peername, peername);
	ast_string_field_set(monitor_instance, device_name, device_name);
	monitor_instance->core_id = core_id;
	ao2_link(sip_monitor_instances, monitor_instance);
	return monitor_instance;
}

static int find_sip_monitor_instance_by_subscription_pvt(void *obj, void *arg, int flags)
{
	struct sip_monitor_instance *monitor_instance = obj;
	return monitor_instance->subscription_pvt == arg ? CMP_MATCH | CMP_STOP : 0;
}

static int find_sip_monitor_instance_by_suspension_entry(void *obj, void *arg, int flags)
{
	struct sip_monitor_instance *monitor_instance = obj;
	return monitor_instance->suspension_entry == arg ? CMP_MATCH | CMP_STOP : 0;
}

static int sip_cc_monitor_request_cc(struct ast_cc_monitor *monitor, int *available_timer_id);
static int sip_cc_monitor_suspend(struct ast_cc_monitor *monitor);
static int sip_cc_monitor_unsuspend(struct ast_cc_monitor *monitor);
static int sip_cc_monitor_cancel_available_timer(struct ast_cc_monitor *monitor, int *sched_id);
static void sip_cc_monitor_destructor(void *private_data);

static struct ast_cc_monitor_callbacks sip_cc_monitor_callbacks = {
	.type = "SIP",
	.request_cc = sip_cc_monitor_request_cc,
	.suspend = sip_cc_monitor_suspend,
	.unsuspend = sip_cc_monitor_unsuspend,
	.cancel_available_timer = sip_cc_monitor_cancel_available_timer,
	.destructor = sip_cc_monitor_destructor,
};

static int sip_cc_monitor_request_cc(struct ast_cc_monitor *monitor, int *available_timer_id)
{
	struct sip_monitor_instance *monitor_instance = monitor->private_data;
	enum ast_cc_service_type service = monitor->service_offered;
	int when;

	if (!monitor_instance) {
		return -1;
	}

	if (!(monitor_instance->subscription_pvt = sip_alloc(NULL, NULL, 0, SIP_SUBSCRIBE, NULL))) {
		return -1;
	}

	when = service == AST_CC_CCBS ? ast_get_ccbs_available_timer(monitor->interface->config_params) :
		ast_get_ccnr_available_timer(monitor->interface->config_params);

	sip_pvt_lock(monitor_instance->subscription_pvt);
	ast_set_flag(&monitor_instance->subscription_pvt->flags[0], SIP_OUTGOING);
	create_addr(monitor_instance->subscription_pvt, monitor_instance->peername, 0, 1, NULL);
	ast_sip_ouraddrfor(&monitor_instance->subscription_pvt->sa, &monitor_instance->subscription_pvt->ourip, monitor_instance->subscription_pvt);
	monitor_instance->subscription_pvt->subscribed = CALL_COMPLETION;
	monitor_instance->subscription_pvt->expiry = when;

	transmit_invite(monitor_instance->subscription_pvt, SIP_SUBSCRIBE, FALSE, 2, monitor_instance->subscribe_uri);
	sip_pvt_unlock(monitor_instance->subscription_pvt);

	ao2_t_ref(monitor, +1, "Adding a ref to the monitor for the scheduler");
	*available_timer_id = ast_sched_add(sched, when * 1000, ast_cc_available_timer_expire, monitor);
	return 0;
}

static int construct_pidf_body(enum sip_cc_publish_state state, char *pidf_body, size_t size, const char *presentity)
{
	struct ast_str *body = ast_str_alloca(size);
	char tuple_id[32];

	generate_random_string(tuple_id, sizeof(tuple_id));

	/* We'll make this a bare-bones pidf body. In state_notify_build_xml, the PIDF
	 * body gets a lot more extra junk that isn't necessary, so we'll leave it out here.
	 */
	ast_str_append(&body, 0, "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
	/* XXX The entity attribute is currently set to the peer name associated with the
	 * dialog. This is because we currently only call this function for call-completion
	 * PUBLISH bodies. In such cases, the entity is completely disregarded. For other
	 * event packages, it may be crucial to have a proper URI as the presentity so this
	 * should be revisited as support is expanded.
	 */
	ast_str_append(&body, 0, "<presence xmlns=\"urn:ietf:params:xml:ns:pidf\" entity=\"%s\">\n", presentity);
	ast_str_append(&body, 0, "<tuple id=\"%s\">\n", tuple_id);
	ast_str_append(&body, 0, "<status><basic>%s</basic></status>\n", state == CC_OPEN ? "open" : "closed");
	ast_str_append(&body, 0, "</tuple>\n");
	ast_str_append(&body, 0, "</presence>\n");
	ast_copy_string(pidf_body, ast_str_buffer(body), size);
	return 0;
}

static int sip_cc_monitor_suspend(struct ast_cc_monitor *monitor)
{
	struct sip_monitor_instance *monitor_instance = monitor->private_data;
	enum sip_publish_type publish_type;
	struct cc_epa_entry *cc_entry;

	if (!monitor_instance) {
		return -1;
	}

	if (!monitor_instance->suspension_entry) {
		/* We haven't yet allocated the suspension entry, so let's give it a shot */
		if (!(monitor_instance->suspension_entry = create_epa_entry("call-completion", monitor_instance->peername))) {
			ast_log(LOG_WARNING, "Unable to allocate sip EPA entry for call-completion\n");
			ao2_ref(monitor_instance, -1);
			return -1;
		}
		if (!(cc_entry = ast_calloc(1, sizeof(*cc_entry)))) {
			ast_log(LOG_WARNING, "Unable to allocate space for instance data of EPA entry for call-completion\n");
			ao2_ref(monitor_instance, -1);
			return -1;
		}
		cc_entry->core_id = monitor->core_id;
		monitor_instance->suspension_entry->instance_data = cc_entry;
		publish_type = SIP_PUBLISH_INITIAL;
	} else {
		publish_type = SIP_PUBLISH_MODIFY;
		cc_entry = monitor_instance->suspension_entry->instance_data;
	}

	cc_entry->current_state = CC_CLOSED;

	if (ast_strlen_zero(monitor_instance->notify_uri)) {
		/* If we have no set notify_uri, then what this means is that we have
		 * not received a NOTIFY from this destination stating that he is
		 * currently available.
		 *
		 * This situation can arise when the core calls the suspend callbacks
		 * of multiple destinations. If one of the other destinations aside
		 * from this one notified Asterisk that he is available, then there
		 * is no reason to take any suspension action on this device. Rather,
		 * we should return now and if we receive a NOTIFY while monitoring
		 * is still "suspended" then we can immediately respond with the
		 * proper PUBLISH to let this endpoint know what is going on.
		 */
		return 0;
	}
	construct_pidf_body(CC_CLOSED, monitor_instance->suspension_entry->body, sizeof(monitor_instance->suspension_entry->body), monitor_instance->peername);
	return transmit_publish(monitor_instance->suspension_entry, publish_type, monitor_instance->notify_uri);
}

static int sip_cc_monitor_unsuspend(struct ast_cc_monitor *monitor)
{
	struct sip_monitor_instance *monitor_instance = monitor->private_data;
	struct cc_epa_entry *cc_entry;

	if (!monitor_instance) {
		return -1;
	}

	ast_assert(monitor_instance->suspension_entry != NULL);

	cc_entry = monitor_instance->suspension_entry->instance_data;
	cc_entry->current_state = CC_OPEN;
	if (ast_strlen_zero(monitor_instance->notify_uri)) {
		/* This means we are being asked to unsuspend a call leg we never
		 * sent a PUBLISH on. As such, there is no reason to send another
		 * PUBLISH at this point either. We can just return instead.
		 */
		return 0;
	}
	construct_pidf_body(CC_OPEN, monitor_instance->suspension_entry->body, sizeof(monitor_instance->suspension_entry->body), monitor_instance->peername);
	return transmit_publish(monitor_instance->suspension_entry, SIP_PUBLISH_MODIFY, monitor_instance->notify_uri);
}

static int sip_cc_monitor_cancel_available_timer(struct ast_cc_monitor *monitor, int *sched_id)
{
	if (*sched_id != -1) {
		AST_SCHED_DEL(sched, *sched_id);
		ao2_t_ref(monitor, -1, "Removing scheduler's reference to the monitor");
	}
	return 0;
}

static void sip_cc_monitor_destructor(void *private_data)
{
	struct sip_monitor_instance *monitor_instance = private_data;
	ao2_unlink(sip_monitor_instances, monitor_instance);
	ast_module_unref(ast_module_info->self);
}

static int sip_get_cc_information(struct sip_request *req, char *subscribe_uri, size_t size, enum ast_cc_service_type *service)
{
	char *call_info = ast_strdupa(get_header(req, "Call-Info"));
	char *uri;
	char *purpose;
	char *service_str;
	static const char cc_purpose[] = "purpose=call-completion";
	static const int cc_purpose_len = sizeof(cc_purpose) - 1;

	if (ast_strlen_zero(call_info)) {
		/* No Call-Info present. Definitely no CC offer */
		return -1;
	}

	uri = strsep(&call_info, ";");

	while ((purpose = strsep(&call_info, ";"))) {
		if (!strncmp(purpose, cc_purpose, cc_purpose_len)) {
			break;
		}
	}
	if (!purpose) {
		/* We didn't find the appropriate purpose= parameter. Oh well */
		return -1;
	}

	/* Okay, call-completion has been offered. Let's figure out what type of service this is */
	while ((service_str = strsep(&call_info, ";"))) {
		if (!strncmp(service_str, "m=", 2)) {
			break;
		}
	}
	if (!service_str) {
		/* So they didn't offer a particular service, We'll just go with CCBS since it really
		 * doesn't matter anyway
		 */
		service_str = "BS";
	} else {
		/* We already determined that there is an "m=" so no need to check
		 * the result of this strsep
		 */
		strsep(&service_str, "=");
	}

	if ((*service = service_string_to_service_type(service_str)) == AST_CC_NONE) {
		/* Invalid service offered */
		return -1;
	}

	ast_copy_string(subscribe_uri, get_in_brackets(uri), size);

	return 0;
}

/*
 * \brief Determine what, if any, CC has been offered and queue a CC frame if possible
 *
 * After taking care of some formalities to be sure that this call is eligible for CC,
 * we first try to see if we can make use of native CC. We grab the information from
 * the passed-in sip_request (which is always a response to an INVITE). If we can
 * use native CC monitoring for the call, then so be it.
 *
 * If native cc monitoring is not possible or not supported, then we will instead attempt
 * to use generic monitoring. Falling back to generic from a failed attempt at using native
 * monitoring will only work if the monitor policy of the endpoint is "always"
 *
 * \param pvt The current dialog. Contains CC parameters for the endpoint
 * \param req The response to the INVITE we want to inspect
 * \param service The service to use if generic monitoring is to be used. For native
 * monitoring, we get the service from the SIP response itself
 */
static void sip_handle_cc(struct sip_pvt *pvt, struct sip_request *req, enum ast_cc_service_type service)
{
	enum ast_cc_monitor_policies monitor_policy = ast_get_cc_monitor_policy(pvt->cc_params);
	int core_id;
	char interface_name[AST_CHANNEL_NAME];

	if (monitor_policy == AST_CC_MONITOR_NEVER) {
		/* Don't bother, just return */
		return;
	}

	if ((core_id = ast_cc_get_current_core_id(pvt->owner)) == -1) {
		/* For some reason, CC is invalid, so don't try it! */
		return;
	}

	ast_channel_get_device_name(pvt->owner, interface_name, sizeof(interface_name));

	if (monitor_policy == AST_CC_MONITOR_ALWAYS || monitor_policy == AST_CC_MONITOR_NATIVE) {
		char subscribe_uri[SIPBUFSIZE];
		char device_name[AST_CHANNEL_NAME];
		enum ast_cc_service_type offered_service;
		struct sip_monitor_instance *monitor_instance;
		if (sip_get_cc_information(req, subscribe_uri, sizeof(subscribe_uri), &offered_service)) {
			/* If CC isn't being offered to us, or for some reason the CC offer is
			 * not formatted correctly, then it may still be possible to use generic
			 * call completion since the monitor policy may be "always"
			 */
			goto generic;
		}
		ast_channel_get_device_name(pvt->owner, device_name, sizeof(device_name));
		if (!(monitor_instance = sip_monitor_instance_init(core_id, subscribe_uri, pvt->peername, device_name))) {
			/* Same deal. We can try using generic still */
			goto generic;
		}
		/* We bump the refcount of chan_sip because once we queue this frame, the CC core
		 * will have a reference to callbacks in this module. We decrement the module
		 * refcount once the monitor destructor is called
		 */
		ast_module_ref(ast_module_info->self);
		ast_queue_cc_frame(pvt->owner, "SIP", pvt->dialstring, offered_service, monitor_instance);
		ao2_ref(monitor_instance, -1);
		return;
	}

generic:
	if (monitor_policy == AST_CC_MONITOR_GENERIC || monitor_policy == AST_CC_MONITOR_ALWAYS) {
		ast_queue_cc_frame(pvt->owner, AST_CC_GENERIC_MONITOR_TYPE, interface_name, service, NULL);
	}
}

/*! \brief Working TLS connection configuration */
static struct ast_tls_config sip_tls_cfg;

/*! \brief Default TLS connection configuration */
static struct ast_tls_config default_tls_cfg;

/*! \brief The TCP server definition */
static struct ast_tcptls_session_args sip_tcp_desc = {
	.accept_fd = -1,
	.master = AST_PTHREADT_NULL,
	.tls_cfg = NULL,
	.poll_timeout = -1,
	.name = "SIP TCP server",
	.accept_fn = ast_tcptls_server_root,
	.worker_fn = sip_tcp_worker_fn,
};

/*! \brief The TCP/TLS server definition */
static struct ast_tcptls_session_args sip_tls_desc = {
	.accept_fd = -1,
	.master = AST_PTHREADT_NULL,
	.tls_cfg = &sip_tls_cfg,
	.poll_timeout = -1,
	.name = "SIP TLS server",
	.accept_fn = ast_tcptls_server_root,
	.worker_fn = sip_tcp_worker_fn,
};

/*! \brief Append to SIP dialog history
	\return Always returns 0 */
#define append_history(p, event, fmt , args... )	append_history_full(p, "%-15s " fmt, event, ## args)

struct sip_pvt *dialog_ref_debug(struct sip_pvt *p, char *tag, char *file, int line, const char *func)
{
	if (p)
#ifdef REF_DEBUG
		__ao2_ref_debug(p, 1, tag, file, line, func);
#else
		ao2_ref(p, 1);
#endif
	else
		ast_log(LOG_ERROR, "Attempt to Ref a null pointer\n");
	return p;
}

struct sip_pvt *dialog_unref_debug(struct sip_pvt *p, char *tag, char *file, int line, const char *func)
{
	if (p)
#ifdef REF_DEBUG
		__ao2_ref_debug(p, -1, tag, file, line, func);
#else
		ao2_ref(p, -1);
#endif
	return NULL;
}

/*! \brief map from an integer value to a string.
 * If no match is found, return errorstring
 */
static const char *map_x_s(const struct _map_x_s *table, int x, const char *errorstring)
{
	const struct _map_x_s *cur;

	for (cur = table; cur->s; cur++)
		if (cur->x == x)
			return cur->s;
	return errorstring;
}

/*! \brief map from a string to an integer value, case insensitive.
 * If no match is found, return errorvalue.
 */
static int map_s_x(const struct _map_x_s *table, const char *s, int errorvalue)
{
	const struct _map_x_s *cur;

	for (cur = table; cur->s; cur++)
		if (!strcasecmp(cur->s, s))
			return cur->x;
	return errorvalue;
}

static enum AST_REDIRECTING_REASON sip_reason_str_to_code(const char *text)
{
	enum AST_REDIRECTING_REASON ast = AST_REDIRECTING_REASON_UNKNOWN;
	int i;

	for (i = 0; i < ARRAY_LEN(sip_reason_table); ++i) {
		if (!strcasecmp(text, sip_reason_table[i].text)) {
			ast = sip_reason_table[i].code;
			break;
		}
	}

	return ast;
}

static const char *sip_reason_code_to_str(enum AST_REDIRECTING_REASON code)
{
	if (code >= 0 && code < ARRAY_LEN(sip_reason_table)) {
		return sip_reason_table[code].text;
	}

	return "unknown";
}

/*!
 * \brief generic function for determining if a correct transport is being
 * used to contact a peer
 *
 * this is done as a macro so that the "tmpl" var can be passed either a
 * sip_request or a sip_peer
 */
#define check_request_transport(peer, tmpl) ({ \
	int ret = 0; \
	if (peer->socket.type == tmpl->socket.type) \
		; \
	else if (!(peer->transports & tmpl->socket.type)) {\
		ast_log(LOG_ERROR, \
			"'%s' is not a valid transport for '%s'. we only use '%s'! ending call.\n", \
			get_transport(tmpl->socket.type), peer->name, get_transport_list(peer->transports) \
			); \
		ret = 1; \
	} else if (peer->socket.type & SIP_TRANSPORT_TLS) { \
		ast_log(LOG_WARNING, \
			"peer '%s' HAS NOT USED (OR SWITCHED TO) TLS in favor of '%s' (but this was allowed in sip.conf)!\n", \
			peer->name, get_transport(tmpl->socket.type) \
		); \
	} else { \
		ast_debug(1, \
			"peer '%s' has contacted us over %s even though we prefer %s.\n", \
			peer->name, get_transport(tmpl->socket.type), get_transport(peer->socket.type) \
		); \
	}\
	(ret); \
})

/*! \brief
 * duplicate a list of channel variables, \return the copy.
 */
static struct ast_variable *copy_vars(struct ast_variable *src)
{
	struct ast_variable *res = NULL, *tmp, *v = NULL;

	for (v = src ; v ; v = v->next) {
		if ((tmp = ast_variable_new(v->name, v->value, v->file))) {
			tmp->next = res;
			res = tmp;
		}
	}
	return res;
}

static void tcptls_packet_destructor(void *obj)
{
	struct tcptls_packet *packet = obj;

	ast_free(packet->data);
}

static void sip_tcptls_client_args_destructor(void *obj)
{
	struct ast_tcptls_session_args *args = obj;
	if (args->tls_cfg) {
		ast_free(args->tls_cfg->certfile);
		ast_free(args->tls_cfg->pvtfile);
		ast_free(args->tls_cfg->cipher);
		ast_free(args->tls_cfg->cafile);
		ast_free(args->tls_cfg->capath);
	}
	ast_free(args->tls_cfg);
	ast_free((char *) args->name);
}

static void sip_threadinfo_destructor(void *obj)
{
	struct sip_threadinfo *th = obj;
	struct tcptls_packet *packet;
	if (th->alert_pipe[1] > -1) {
		close(th->alert_pipe[0]);
	}
	if (th->alert_pipe[1] > -1) {
		close(th->alert_pipe[1]);
	}
	th->alert_pipe[0] = th->alert_pipe[1] = -1;

	while ((packet = AST_LIST_REMOVE_HEAD(&th->packet_q, entry))) {
		ao2_t_ref(packet, -1, "thread destruction, removing packet from frame queue");
	}

	if (th->tcptls_session) {
		ao2_t_ref(th->tcptls_session, -1, "remove tcptls_session for sip_threadinfo object");
	}
}

/*! \brief creates a sip_threadinfo object and links it into the threadt table. */
static struct sip_threadinfo *sip_threadinfo_create(struct ast_tcptls_session_instance *tcptls_session, int transport)
{
	struct sip_threadinfo *th;

	if (!tcptls_session || !(th = ao2_alloc(sizeof(*th), sip_threadinfo_destructor))) {
		return NULL;
	}

	th->alert_pipe[0] = th->alert_pipe[1] = -1;

	if (pipe(th->alert_pipe) == -1) {
		ao2_t_ref(th, -1, "Failed to open alert pipe on sip_threadinfo");
		ast_log(LOG_ERROR, "Could not create sip alert pipe in tcptls thread, error %s\n", strerror(errno));
		return NULL;
	}
	ao2_t_ref(tcptls_session, +1, "tcptls_session ref for sip_threadinfo object");
	th->tcptls_session = tcptls_session;
	th->type = transport ? transport : (tcptls_session->ssl ? SIP_TRANSPORT_TLS: SIP_TRANSPORT_TCP);
	ao2_t_link(threadt, th, "Adding new tcptls helper thread");
	ao2_t_ref(th, -1, "Decrementing threadinfo ref from alloc, only table ref remains");
	return th;
}

/*! \brief used to indicate to a tcptls thread that data is ready to be written */
static int sip_tcptls_write(struct ast_tcptls_session_instance *tcptls_session, const void *buf, size_t len)
{
	int res = len;
	struct sip_threadinfo *th = NULL;
	struct tcptls_packet *packet = NULL;
	struct sip_threadinfo tmp = {
		.tcptls_session = tcptls_session,
	};
	enum sip_tcptls_alert alert = TCPTLS_ALERT_DATA;

	if (!tcptls_session) {
		return XMIT_ERROR;
	}

	ast_mutex_lock(&tcptls_session->lock);

	if ((tcptls_session->fd == -1) ||
		!(th = ao2_t_find(threadt, &tmp, OBJ_POINTER, "ao2_find, getting sip_threadinfo in tcp helper thread")) ||
		!(packet = ao2_alloc(sizeof(*packet), tcptls_packet_destructor)) ||
		!(packet->data = ast_str_create(len))) {
		goto tcptls_write_setup_error;
	}

	/* goto tcptls_write_error should _NOT_ be used beyond this point */
	ast_str_set(&packet->data, 0, "%s", (char *) buf);
	packet->len = len;

	/* alert tcptls thread handler that there is a packet to be sent.
	 * must lock the thread info object to guarantee control of the
	 * packet queue */
	ao2_lock(th);
	if (write(th->alert_pipe[1], &alert, sizeof(alert)) == -1) {
		ast_log(LOG_ERROR, "write() to alert pipe failed: %s\n", strerror(errno));
		ao2_t_ref(packet, -1, "could not write to alert pipe, remove packet");
		packet = NULL;
		res = XMIT_ERROR;
	} else { /* it is safe to queue the frame after issuing the alert when we hold the threadinfo lock */
		AST_LIST_INSERT_TAIL(&th->packet_q, packet, entry);
	}
	ao2_unlock(th);

	ast_mutex_unlock(&tcptls_session->lock);
	ao2_t_ref(th, -1, "In sip_tcptls_write, unref threadinfo object after finding it");
	return res;

tcptls_write_setup_error:
	if (th) {
		ao2_t_ref(th, -1, "In sip_tcptls_write, unref threadinfo obj, could not create packet");
	}
	if (packet) {
		ao2_t_ref(packet, -1, "could not allocate packet's data");
	}
	ast_mutex_unlock(&tcptls_session->lock);

	return XMIT_ERROR;
}

/*! \brief SIP TCP connection handler */
static void *sip_tcp_worker_fn(void *data)
{
	struct ast_tcptls_session_instance *tcptls_session = data;

	return _sip_tcp_helper_thread(tcptls_session);
}

/*! \brief Check if the authtimeout has expired.
 * \param start the time when the session started
 *
 * \retval 0 the timeout has expired
 * \retval -1 error
 * \return the number of milliseconds until the timeout will expire
 */
static int sip_check_authtimeout(time_t start)
{
	int timeout;
	time_t now;
	if(time(&now) == -1) {
		ast_log(LOG_ERROR, "error executing time(): %s\n", strerror(errno));
		return -1;
	}

	timeout = (authtimeout - (now - start)) * 1000;
	if (timeout < 0) {
		/* we have timed out */
		return 0;
	}

	return timeout;
}

/*! \brief SIP TCP thread management function
	This function reads from the socket, parses the packet into a request
*/
static void *_sip_tcp_helper_thread(struct ast_tcptls_session_instance *tcptls_session)
{
	int res, cl, timeout = -1, authenticated = 0, flags, after_poll = 0, need_poll = 1;
	time_t start;
	struct sip_request req = { 0, } , reqcpy = { 0, };
	struct sip_threadinfo *me = NULL;
	char buf[1024] = "";
	struct pollfd fds[2] = { { 0 }, { 0 }, };
	struct ast_tcptls_session_args *ca = NULL;

	/* If this is a server session, then the connection has already been
	 * setup. Check if the authlimit has been reached and if not create the
	 * threadinfo object so we can access this thread for writing.
	 *
	 * if this is a client connection more work must be done.
	 * 1. We own the parent session args for a client connection.  This pointer needs
	 *    to be held on to so we can decrement it's ref count on thread destruction.
	 * 2. The threadinfo object was created before this thread was launched, however
	 *    it must be found within the threadt table.
	 * 3. Last, the tcptls_session must be started.
	 */
	if (!tcptls_session->client) {
		if (ast_atomic_fetchadd_int(&unauth_sessions, +1) >= authlimit) {
			/* unauth_sessions is decremented in the cleanup code */
			goto cleanup;
		}

		if ((flags = fcntl(tcptls_session->fd, F_GETFL)) == -1) {
			ast_log(LOG_ERROR, "error setting socket to non blocking mode, fcntl() failed: %s\n", strerror(errno));
			goto cleanup;
		}

		flags |= O_NONBLOCK;
		if (fcntl(tcptls_session->fd, F_SETFL, flags) == -1) {
			ast_log(LOG_ERROR, "error setting socket to non blocking mode, fcntl() failed: %s\n", strerror(errno));
			goto cleanup;
		}

		if (!(me = sip_threadinfo_create(tcptls_session, tcptls_session->ssl ? SIP_TRANSPORT_TLS : SIP_TRANSPORT_TCP))) {
			goto cleanup;
		}
		ao2_t_ref(me, +1, "Adding threadinfo ref for tcp_helper_thread");
	} else {
		struct sip_threadinfo tmp = {
			.tcptls_session = tcptls_session,
		};

		if ((!(ca = tcptls_session->parent)) ||
			(!(me = ao2_t_find(threadt, &tmp, OBJ_POINTER, "ao2_find, getting sip_threadinfo in tcp helper thread"))) ||
			(!(tcptls_session = ast_tcptls_client_start(tcptls_session)))) {
			goto cleanup;
		}
	}

	flags = 1;
	if (setsockopt(tcptls_session->fd, SOL_SOCKET, SO_KEEPALIVE, &flags, sizeof(flags))) {
		ast_log(LOG_ERROR, "error enabling TCP keep-alives on sip socket: %s\n", strerror(errno));
		goto cleanup;
	}

	me->threadid = pthread_self();
	ast_debug(2, "Starting thread for %s server\n", tcptls_session->ssl ? "SSL" : "TCP");

	/* set up pollfd to watch for reads on both the socket and the alert_pipe */
	fds[0].fd = tcptls_session->fd;
	fds[1].fd = me->alert_pipe[0];
	fds[0].events = fds[1].events = POLLIN | POLLPRI;

	if (!(req.data = ast_str_create(SIP_MIN_PACKET))) {
		goto cleanup;
	}
	if (!(reqcpy.data = ast_str_create(SIP_MIN_PACKET))) {
		goto cleanup;
	}

	if(time(&start) == -1) {
		ast_log(LOG_ERROR, "error executing time(): %s\n", strerror(errno));
		goto cleanup;
	}

	for (;;) {
		struct ast_str *str_save;

		if (!tcptls_session->client && req.authenticated && !authenticated) {
			authenticated = 1;
			ast_atomic_fetchadd_int(&unauth_sessions, -1);
		}

		/* calculate the timeout for unauthenticated server sessions */
		if (!tcptls_session->client && !authenticated ) {
			if ((timeout = sip_check_authtimeout(start)) < 0) {
				goto cleanup;
			}

			if (timeout == 0) {
				ast_debug(2, "SIP %s server timed out\n", tcptls_session->ssl ? "SSL": "TCP");
				goto cleanup;
			}
		} else {
			timeout = -1;
		}

		res = ast_poll(fds, 2, timeout); /* polls for both socket and alert_pipe */
		if (res < 0) {
			ast_debug(2, "SIP %s server :: ast_wait_for_input returned %d\n", tcptls_session->ssl ? "SSL": "TCP", res);
			goto cleanup;
		} else if (res == 0) {
			/* timeout */
			ast_debug(2, "SIP %s server timed out\n", tcptls_session->ssl ? "SSL": "TCP");
			goto cleanup;
		}

		/* handle the socket event, check for both reads from the socket fd,
		 * and writes from alert_pipe fd */
		if (fds[0].revents) { /* there is data on the socket to be read */
			after_poll = 1;

			fds[0].revents = 0;

			/* clear request structure */
			str_save = req.data;
			memset(&req, 0, sizeof(req));
			req.data = str_save;
			ast_str_reset(req.data);

			str_save = reqcpy.data;
			memset(&reqcpy, 0, sizeof(reqcpy));
			reqcpy.data = str_save;
			ast_str_reset(reqcpy.data);

			memset(buf, 0, sizeof(buf));

			if (tcptls_session->ssl) {
				set_socket_transport(&req.socket, SIP_TRANSPORT_TLS);
				req.socket.port = htons(ourport_tls);
			} else {
				set_socket_transport(&req.socket, SIP_TRANSPORT_TCP);
				req.socket.port = htons(ourport_tcp);
			}
			req.socket.fd = tcptls_session->fd;

			/* Read in headers one line at a time */
			while (ast_str_strlen(req.data) < 4 || strncmp(REQ_OFFSET_TO_STR(&req, data->used - 4), "\r\n\r\n", 4)) {
				if (!tcptls_session->client && !authenticated ) {
					if ((timeout = sip_check_authtimeout(start)) < 0) {
						goto cleanup;
					}

					if (timeout == 0) {
						ast_debug(2, "SIP %s server timed out\n", tcptls_session->ssl ? "SSL": "TCP");
						goto cleanup;
					}
				} else {
					timeout = -1;
				}

				/* special polling behavior is required for TLS
				 * sockets because of the buffering done in the
				 * TLS layer */
				if (!tcptls_session->ssl || need_poll) {
					need_poll = 0;
					after_poll = 1;
					res = ast_wait_for_input(tcptls_session->fd, timeout);
					if (res < 0) {
						ast_debug(2, "SIP TCP server :: ast_wait_for_input returned %d\n", res);
						goto cleanup;
					} else if (res == 0) {
						/* timeout */
						ast_debug(2, "SIP TCP server timed out\n");
						goto cleanup;
					}
				}

				ast_mutex_lock(&tcptls_session->lock);
				if (!fgets(buf, sizeof(buf), tcptls_session->f)) {
					ast_mutex_unlock(&tcptls_session->lock);
					if (after_poll) {
						goto cleanup;
					} else {
						need_poll = 1;
						continue;
					}
				}
				ast_mutex_unlock(&tcptls_session->lock);
				after_poll = 0;
				if (me->stop) {
					 goto cleanup;
				}
				ast_str_append(&req.data, 0, "%s", buf);
			}
			copy_request(&reqcpy, &req);
			parse_request(&reqcpy);
			/* In order to know how much to read, we need the content-length header */
			if (sscanf(get_header(&reqcpy, "Content-Length"), "%30d", &cl)) {
				while (cl > 0) {
					size_t bytes_read;
					if (!tcptls_session->client && !authenticated ) {
						if ((timeout = sip_check_authtimeout(start)) < 0) {
							goto cleanup;
						}

						if (timeout == 0) {
							ast_debug(2, "SIP %s server timed out\n", tcptls_session->ssl ? "SSL": "TCP");
							goto cleanup;
						}
					} else {
						timeout = -1;
					}

					if (!tcptls_session->ssl || need_poll) {
						need_poll = 0;
						after_poll = 1;
						res = ast_wait_for_input(tcptls_session->fd, timeout);
						if (res < 0) {
							ast_debug(2, "SIP TCP server :: ast_wait_for_input returned %d\n", res);
							goto cleanup;
						} else if (res == 0) {
							/* timeout */
							ast_debug(2, "SIP TCP server timed out\n");
							goto cleanup;
						}
					}

					ast_mutex_lock(&tcptls_session->lock);
					if (!(bytes_read = fread(buf, 1, MIN(sizeof(buf) - 1, cl), tcptls_session->f))) {
						ast_mutex_unlock(&tcptls_session->lock);
						if (after_poll) {
							goto cleanup;
						} else {
							need_poll = 1;
							continue;
						}
					}
					buf[bytes_read] = '\0';
					ast_mutex_unlock(&tcptls_session->lock);
					after_poll = 0;
					if (me->stop) {
						goto cleanup;
					}
					cl -= strlen(buf);
					ast_str_append(&req.data, 0, "%s", buf);
				}
			}
			/*! \todo XXX If there's no Content-Length or if the content-length and what
					we receive is not the same - we should generate an error */

			req.socket.tcptls_session = tcptls_session;
			handle_request_do(&req, &tcptls_session->remote_address);
		}

		if (fds[1].revents) { /* alert_pipe indicates there is data in the send queue to be sent */
			enum sip_tcptls_alert alert;
			struct tcptls_packet *packet;

			fds[1].revents = 0;

			if (read(me->alert_pipe[0], &alert, sizeof(alert)) == -1) {
				ast_log(LOG_ERROR, "read() failed: %s\n", strerror(errno));
				continue;
			}

			switch (alert) {
			case TCPTLS_ALERT_STOP:
				goto cleanup;
			case TCPTLS_ALERT_DATA:
				ao2_lock(me);
				if (!(packet = AST_LIST_REMOVE_HEAD(&me->packet_q, entry))) {
					ast_log(LOG_WARNING, "TCPTLS thread alert_pipe indicated packet should be sent, but frame_q is empty\n");
				}
				ao2_unlock(me);

				if (packet) {
					if (ast_tcptls_server_write(tcptls_session, ast_str_buffer(packet->data), packet->len) == -1) {
						ast_log(LOG_WARNING, "Failure to write to tcp/tls socket\n");
					}
					ao2_t_ref(packet, -1, "tcptls packet sent, this is no longer needed");
				}
				break;
			default:
				ast_log(LOG_ERROR, "Unknown tcptls thread alert '%d'\n", alert);
			}
		}
	}

	ast_debug(2, "Shutting down thread for %s server\n", tcptls_session->ssl ? "SSL" : "TCP");

cleanup:
	if (tcptls_session && !tcptls_session->client && !authenticated) {
		ast_atomic_fetchadd_int(&unauth_sessions, -1);
	}

	if (me) {
		ao2_t_unlink(threadt, me, "Removing tcptls helper thread, thread is closing");
		ao2_t_ref(me, -1, "Removing tcp_helper_threads threadinfo ref");
	}
	deinit_req(&reqcpy);
	deinit_req(&req);

	/* if client, we own the parent session arguments and must decrement ref */
	if (ca) {
		ao2_t_ref(ca, -1, "closing tcptls thread, getting rid of client tcptls_session arguments");
	}

	if (tcptls_session) {
		ast_mutex_lock(&tcptls_session->lock);
		ast_tcptls_close_session_file(tcptls_session);
		tcptls_session->parent = NULL;
		ast_mutex_unlock(&tcptls_session->lock);

		ao2_ref(tcptls_session, -1);
		tcptls_session = NULL;
	}
	return NULL;
}

#ifdef REF_DEBUG
#define ref_peer(arg1,arg2) _ref_peer((arg1),(arg2), __FILE__, __LINE__, __PRETTY_FUNCTION__)
#define unref_peer(arg1,arg2) _unref_peer((arg1),(arg2), __FILE__, __LINE__, __PRETTY_FUNCTION__)
static struct sip_peer *_ref_peer(struct sip_peer *peer, char *tag, char *file, int line, const char *func)
{
	if (peer)
		__ao2_ref_debug(peer, 1, tag, file, line, func);
	else
		ast_log(LOG_ERROR, "Attempt to Ref a null peer pointer\n");
	return peer;
}

static struct sip_peer *_unref_peer(struct sip_peer *peer, char *tag, char *file, int line, const char *func)
{
	if (peer)
		__ao2_ref_debug(peer, -1, tag, file, line, func);
	return NULL;
}
#else
/*!
 * helper functions to unreference various types of objects.
 * By handling them this way, we don't have to declare the
 * destructor on each call, which removes the chance of errors.
 */
static void *unref_peer(struct sip_peer *peer, char *tag)
{
	ao2_t_ref(peer, -1, tag);
	return NULL;
}

static struct sip_peer *ref_peer(struct sip_peer *peer, char *tag)
{
	ao2_t_ref(peer, 1, tag);
	return peer;
}
#endif /* REF_DEBUG */

static void peer_sched_cleanup(struct sip_peer *peer)
{
	if (peer->pokeexpire != -1) {
		AST_SCHED_DEL_UNREF(sched, peer->pokeexpire,
				unref_peer(peer, "removing poke peer ref"));
	}
	if (peer->expire != -1) {
		AST_SCHED_DEL_UNREF(sched, peer->expire,
				unref_peer(peer, "remove register expire ref"));
	}
}

typedef enum {
	SIP_PEERS_MARKED,
	SIP_PEERS_ALL,
} peer_unlink_flag_t;

/* this func is used with ao2_callback to unlink/delete all marked or linked
   peers, depending on arg */
static int match_and_cleanup_peer_sched(void *peerobj, void *arg, int flags)
{
	struct sip_peer *peer = peerobj;
	peer_unlink_flag_t which = *(peer_unlink_flag_t *)arg;

	if (which == SIP_PEERS_ALL || peer->the_mark) {
		peer_sched_cleanup(peer);
		if (peer->dnsmgr) {
			ast_dnsmgr_release(peer->dnsmgr);
			peer->dnsmgr = NULL;
			unref_peer(peer, "Release peer from dnsmgr");
		}
		return CMP_MATCH;
	}
	return 0;
}

static void unlink_peers_from_tables(peer_unlink_flag_t flag)
{
	ao2_t_callback(peers, OBJ_NODATA | OBJ_UNLINK | OBJ_MULTIPLE,
		match_and_cleanup_peer_sched, &flag, "initiating callback to remove marked peers");
	ao2_t_callback(peers_by_ip, OBJ_NODATA | OBJ_UNLINK | OBJ_MULTIPLE,
		match_and_cleanup_peer_sched, &flag, "initiating callback to remove marked peers");
}

/* \brief Unlink all marked peers from ao2 containers */
static void unlink_marked_peers_from_tables(void)
{
	unlink_peers_from_tables(SIP_PEERS_MARKED);
}

static void unlink_all_peers_from_tables(void)
{
	unlink_peers_from_tables(SIP_PEERS_ALL);
}

/* \brief Unlink single peer from all ao2 containers */
static void unlink_peer_from_tables(struct sip_peer *peer)
{
	ao2_t_unlink(peers, peer, "ao2_unlink of peer from peers table");
	if (!ast_sockaddr_isnull(&peer->addr)) {
		ao2_t_unlink(peers_by_ip, peer, "ao2_unlink of peer from peers_by_ip table");
	}
}

/*! \brief maintain proper refcounts for a sip_pvt's outboundproxy
 *
 * This function sets pvt's outboundproxy pointer to the one referenced
 * by the proxy parameter. Because proxy may be a refcounted object, and
 * because pvt's old outboundproxy may also be a refcounted object, we need
 * to maintain the proper refcounts.
 *
 * \param pvt The sip_pvt for which we wish to set the outboundproxy
 * \param proxy The sip_proxy which we will point pvt towards.
 * \return Returns void
 */
static void ref_proxy(struct sip_pvt *pvt, struct sip_proxy *proxy)
{
	struct sip_proxy *old_obproxy = pvt->outboundproxy;
	/* The sip_cfg.outboundproxy is statically allocated, and so
	 * we don't ever need to adjust refcounts for it
	 */
	if (proxy && proxy != &sip_cfg.outboundproxy) {
		ao2_ref(proxy, +1);
	}
	pvt->outboundproxy = proxy;
	if (old_obproxy && old_obproxy != &sip_cfg.outboundproxy) {
		ao2_ref(old_obproxy, -1);
	}
}

/*!
 * \brief Unlink a dialog from the dialogs container, as well as any other places
 * that it may be currently stored.
 *
 * \note A reference to the dialog must be held before calling this function, and this
 * function does not release that reference.
 */
void dialog_unlink_all(struct sip_pvt *dialog)
{
	struct sip_pkt *cp;
	struct ast_channel *owner;

	dialog_ref(dialog, "Let's bump the count in the unlink so it doesn't accidentally become dead before we are done");

	ao2_t_unlink(dialogs, dialog, "unlinking dialog via ao2_unlink");

	/* Unlink us from the owner (channel) if we have one */
	owner = sip_pvt_lock_full(dialog);
	if (owner) {
		ast_debug(1, "Detaching from channel %s\n", owner->name);
		owner->tech_pvt = dialog_unref(owner->tech_pvt, "resetting channel dialog ptr in unlink_all");
		ast_channel_unlock(owner);
		ast_channel_unref(owner);
		dialog->owner = NULL;
	}
	sip_pvt_unlock(dialog);

	if (dialog->registry) {
		if (dialog->registry->call == dialog) {
			dialog->registry->call = dialog_unref(dialog->registry->call, "nulling out the registry's call dialog field in unlink_all");
		}
		dialog->registry = registry_unref(dialog->registry, "delete dialog->registry");
	}
	if (dialog->stateid != -1) {
		ast_extension_state_del(dialog->stateid, cb_extensionstate);
		dialog->stateid = -1;
	}
	/* Remove link from peer to subscription of MWI */
	if (dialog->relatedpeer && dialog->relatedpeer->mwipvt == dialog) {
		dialog->relatedpeer->mwipvt = dialog_unref(dialog->relatedpeer->mwipvt, "delete ->relatedpeer->mwipvt");
	}
	if (dialog->relatedpeer && dialog->relatedpeer->call == dialog) {
		dialog->relatedpeer->call = dialog_unref(dialog->relatedpeer->call, "unset the relatedpeer->call field in tandem with relatedpeer field itself");
	}

	/* remove all current packets in this dialog */
	while((cp = dialog->packets)) {
		dialog->packets = dialog->packets->next;
		AST_SCHED_DEL(sched, cp->retransid);
		dialog_unref(cp->owner, "remove all current packets in this dialog, and the pointer to the dialog too as part of __sip_destroy");
		if (cp->data) {
			ast_free(cp->data);
		}
		ast_free(cp);
	}

	AST_SCHED_DEL_UNREF(sched, dialog->waitid, dialog_unref(dialog, "when you delete the waitid sched, you should dec the refcount for the stored dialog ptr"));

	AST_SCHED_DEL_UNREF(sched, dialog->initid, dialog_unref(dialog, "when you delete the initid sched, you should dec the refcount for the stored dialog ptr"));
	
	if (dialog->autokillid > -1) {
		AST_SCHED_DEL_UNREF(sched, dialog->autokillid, dialog_unref(dialog, "when you delete the autokillid sched, you should dec the refcount for the stored dialog ptr"));
	}

	if (dialog->request_queue_sched_id > -1) {
		AST_SCHED_DEL_UNREF(sched, dialog->request_queue_sched_id, dialog_unref(dialog, "when you delete the request_queue_sched_id sched, you should dec the refcount for the stored dialog ptr"));
	}

	AST_SCHED_DEL_UNREF(sched, dialog->provisional_keepalive_sched_id, dialog_unref(dialog, "when you delete the provisional_keepalive_sched_id, you should dec the refcount for the stored dialog ptr"));

	if (dialog->t38id > -1) {
		AST_SCHED_DEL_UNREF(sched, dialog->t38id, dialog_unref(dialog, "when you delete the t38id sched, you should dec the refcount for the stored dialog ptr"));
	}

	if (dialog->stimer) {
		stop_session_timer(dialog);
	}

	dialog_unref(dialog, "Let's unbump the count in the unlink so the poor pvt can disappear if it is time");
}

void *registry_unref(struct sip_registry *reg, char *tag)
{
	ast_debug(3, "SIP Registry %s: refcount now %d\n", reg->hostname, reg->refcount - 1);
	ASTOBJ_UNREF(reg, sip_registry_destroy);
	return NULL;
}

/*! \brief Add object reference to SIP registry */
static struct sip_registry *registry_addref(struct sip_registry *reg, char *tag)
{
	ast_debug(3, "SIP Registry %s: refcount now %d\n", reg->hostname, reg->refcount + 1);
	return ASTOBJ_REF(reg);	/* Add pointer to registry in packet */
}

/*! \brief Interface structure with callbacks used to connect to UDPTL module*/
static struct ast_udptl_protocol sip_udptl = {
	.type = "SIP",
	.get_udptl_info = sip_get_udptl_peer,
	.set_udptl_peer = sip_set_udptl_peer,
};

static void append_history_full(struct sip_pvt *p, const char *fmt, ...)
	__attribute__((format(printf, 2, 3)));


/*! \brief Convert transfer status to string */
static const char *referstatus2str(enum referstatus rstatus)
{
	return map_x_s(referstatusstrings, rstatus, "");
}

static inline void pvt_set_needdestroy(struct sip_pvt *pvt, const char *reason)
{
	if (pvt->final_destruction_scheduled) {
		return; /* This is already scheduled for final destruction, let the scheduler take care of it. */
	}
	append_history(pvt, "NeedDestroy", "Setting needdestroy because %s", reason);
	pvt->needdestroy = 1;
}

/*! \brief Initialize the initital request packet in the pvt structure.
 	This packet is used for creating replies and future requests in
	a dialog */
static void initialize_initreq(struct sip_pvt *p, struct sip_request *req)
{
	if (p->initreq.headers) {
		ast_debug(1, "Initializing already initialized SIP dialog %s (presumably reinvite)\n", p->callid);
	} else {
		ast_debug(1, "Initializing initreq for method %s - callid %s\n", sip_methods[req->method].text, p->callid);
	}
	/* Use this as the basis */
	copy_request(&p->initreq, req);
	parse_request(&p->initreq);
	if (req->debug) {
		ast_verbose("Initreq: %d headers, %d lines\n", p->initreq.headers, p->initreq.lines);
	}
}

/*! \brief Encapsulate setting of SIP_ALREADYGONE to be able to trace it with debugging */
static void sip_alreadygone(struct sip_pvt *dialog)
{
	ast_debug(3, "Setting SIP_ALREADYGONE on dialog %s\n", dialog->callid);
	dialog->alreadygone = 1;
}

/*! Resolve DNS srv name or host name in a sip_proxy structure */
static int proxy_update(struct sip_proxy *proxy)
{
	/* if it's actually an IP address and not a name,
           there's no need for a managed lookup */
	if (!ast_sockaddr_parse(&proxy->ip, proxy->name, 0)) {
		/* Ok, not an IP address, then let's check if it's a domain or host */
		/* XXX Todo - if we have proxy port, don't do SRV */
		proxy->ip.ss.ss_family = get_address_family_filter(&bindaddr); /* Filter address family */
		if (ast_get_ip_or_srv(&proxy->ip, proxy->name, sip_cfg.srvlookup ? "_sip._udp" : NULL) < 0) {
				ast_log(LOG_WARNING, "Unable to locate host '%s'\n", proxy->name);
				return FALSE;
		}

	}

	ast_sockaddr_set_port(&proxy->ip, proxy->port);

	proxy->last_dnsupdate = time(NULL);
	return TRUE;
}

/*! \brief converts ascii port to int representation. If no
 *  pt buffer is provided or the pt has errors when being converted
 *  to an int value, the port provided as the standard is used.
 */
unsigned int port_str2int(const char *pt, unsigned int standard)
{
	int port = standard;
	if (ast_strlen_zero(pt) || (sscanf(pt, "%30d", &port) != 1) || (port < 1) || (port > 65535)) {
		port = standard;
	}

	return port;
}

/*! \brief Get default outbound proxy or global proxy */
static struct sip_proxy *obproxy_get(struct sip_pvt *dialog, struct sip_peer *peer)
{
	if (peer && peer->outboundproxy) {
		if (sipdebug) {
			ast_debug(1, "OBPROXY: Applying peer OBproxy to this call\n");
		}
		append_history(dialog, "OBproxy", "Using peer obproxy %s", peer->outboundproxy->name);
		return peer->outboundproxy;
	}
	if (sip_cfg.outboundproxy.name[0]) {
		if (sipdebug) {
			ast_debug(1, "OBPROXY: Applying global OBproxy to this call\n");
		}
		append_history(dialog, "OBproxy", "Using global obproxy %s", sip_cfg.outboundproxy.name);
		return &sip_cfg.outboundproxy;
	}
	if (sipdebug) {
		ast_debug(1, "OBPROXY: Not applying OBproxy to this call\n");
	}
	return NULL;
}

/*! \brief returns true if 'name' (with optional trailing whitespace)
 * matches the sip method 'id'.
 * Strictly speaking, SIP methods are case SENSITIVE, but we do
 * a case-insensitive comparison to be more tolerant.
 * following Jon Postel's rule: Be gentle in what you accept, strict with what you send
 */
static int method_match(enum sipmethod id, const char *name)
{
	int len = strlen(sip_methods[id].text);
	int l_name = name ? strlen(name) : 0;
	/* true if the string is long enough, and ends with whitespace, and matches */
	return (l_name >= len && name[len] < 33 &&
		!strncasecmp(sip_methods[id].text, name, len));
}

/*! \brief  find_sip_method: Find SIP method from header */
static int find_sip_method(const char *msg)
{
	int i, res = 0;
	
	if (ast_strlen_zero(msg)) {
		return 0;
	}
	for (i = 1; i < ARRAY_LEN(sip_methods) && !res; i++) {
		if (method_match(i, msg)) {
			res = sip_methods[i].id;
		}
	}
	return res;
}

/*! \brief See if we pass debug IP filter */
static inline int sip_debug_test_addr(const struct ast_sockaddr *addr)
{
	/* Can't debug if sipdebug is not enabled */
	if (!sipdebug) {
		return 0;
	}

	/* A null debug_addr means we'll debug any address */
	if (ast_sockaddr_isnull(&debugaddr)) {
		return 1;
	}

	/* If no port was specified for a debug address, just compare the
	 * addresses, otherwise compare the address and port
	 */
	if (ast_sockaddr_port(&debugaddr)) {
		return !ast_sockaddr_cmp(&debugaddr, addr);
	} else {
		return !ast_sockaddr_cmp_addr(&debugaddr, addr);
	}
}

/*! \brief The real destination address for a write */
static const struct ast_sockaddr *sip_real_dst(const struct sip_pvt *p)
{
	if (p->outboundproxy) {
		return &p->outboundproxy->ip;
	}

	return ast_test_flag(&p->flags[0], SIP_NAT_FORCE_RPORT) || ast_test_flag(&p->flags[0], SIP_NAT_RPORT_PRESENT) ? &p->recv : &p->sa;
}

/*! \brief Display SIP nat mode */
static const char *sip_nat_mode(const struct sip_pvt *p)
{
	return ast_test_flag(&p->flags[0], SIP_NAT_FORCE_RPORT) ? "NAT" : "no NAT";
}

/*! \brief Test PVT for debugging output */
static inline int sip_debug_test_pvt(struct sip_pvt *p)
{
	if (!sipdebug) {
		return 0;
	}
	return sip_debug_test_addr(sip_real_dst(p));
}

/*! \brief Return int representing a bit field of transport types found in const char *transport */
static int get_transport_str2enum(const char *transport)
{
	int res = 0;

	if (ast_strlen_zero(transport)) {
		return res;
	}

	if (!strcasecmp(transport, "udp")) {
		res |= SIP_TRANSPORT_UDP;
	}
	if (!strcasecmp(transport, "tcp")) {
		res |= SIP_TRANSPORT_TCP;
	}
	if (!strcasecmp(transport, "tls")) {
		res |= SIP_TRANSPORT_TLS;
	}

	return res;
}

/*! \brief Return configuration of transports for a device */
static inline const char *get_transport_list(unsigned int transports) {
	switch (transports) {
		case SIP_TRANSPORT_UDP:
			return "UDP";
		case SIP_TRANSPORT_TCP:
			return "TCP";
		case SIP_TRANSPORT_TLS:
			return "TLS";
		case SIP_TRANSPORT_UDP | SIP_TRANSPORT_TCP:
			return "TCP,UDP";
		case SIP_TRANSPORT_UDP | SIP_TRANSPORT_TLS:
			return "TLS,UDP";
		case SIP_TRANSPORT_TCP | SIP_TRANSPORT_TLS:
			return "TLS,TCP";
		default:
			return transports ?
				"TLS,TCP,UDP" : "UNKNOWN";	
	}
}

/*! \brief Return transport as string */
static inline const char *get_transport(enum sip_transport t)
{
	switch (t) {
	case SIP_TRANSPORT_UDP:
		return "UDP";
	case SIP_TRANSPORT_TCP:
		return "TCP";
	case SIP_TRANSPORT_TLS:
		return "TLS";
	}

	return "UNKNOWN";
}

/*! \brief Return protocol string for srv dns query */
static inline const char *get_srv_protocol(enum sip_transport t)
{
	switch (t) {
	case SIP_TRANSPORT_UDP:
		return "udp";
	case SIP_TRANSPORT_TLS:
	case SIP_TRANSPORT_TCP:
		return "tcp";
	}

	return "udp";
}

/*! \brief Return service string for srv dns query */
static inline const char *get_srv_service(enum sip_transport t)
{
	switch (t) {
	case SIP_TRANSPORT_TCP:
	case SIP_TRANSPORT_UDP:
		return "sip";
	case SIP_TRANSPORT_TLS:
		return "sips";
	}
	return "sip";
}

/*! \brief Return transport of dialog.
	\note this is based on a false assumption. We don't always use the
	outbound proxy for all requests in a dialog. It depends on the
	"force" parameter. The FIRST request is always sent to the ob proxy.
	\todo Fix this function to work correctly
*/
static inline const char *get_transport_pvt(struct sip_pvt *p)
{
	if (p->outboundproxy && p->outboundproxy->transport) {
		set_socket_transport(&p->socket, p->outboundproxy->transport);
	}

	return get_transport(p->socket.type);
}

/*!
 * \internal
 * \brief Transmit SIP message
 *
 * \details
 * Sends a SIP request or response on a given socket (in the pvt)
 * \note
 * Called by retrans_pkt, send_request, send_response and __sip_reliable_xmit
 *
 * \return length of transmitted message, XMIT_ERROR on known network failures -1 on other failures.
 */
static int __sip_xmit(struct sip_pvt *p, struct ast_str *data)
{
	int res = 0;
	const struct ast_sockaddr *dst = sip_real_dst(p);

	ast_debug(2, "Trying to put '%.11s' onto %s socket destined for %s\n", data->str, get_transport_pvt(p), ast_sockaddr_stringify(dst));

	if (sip_prepare_socket(p) < 0) {
		return XMIT_ERROR;
	}

	if (p->socket.type == SIP_TRANSPORT_UDP) {
		res = ast_sendto(p->socket.fd, data->str, ast_str_strlen(data), 0, dst);
	} else if (p->socket.tcptls_session) {
		res = sip_tcptls_write(p->socket.tcptls_session, data->str, ast_str_strlen(data));
	} else {
		ast_debug(2, "Socket type is TCP but no tcptls_session is present to write to\n");
		return XMIT_ERROR;
	}

	if (res == -1) {
		switch (errno) {
		case EBADF: 		/* Bad file descriptor - seems like this is generated when the host exist, but doesn't accept the UDP packet */
		case EHOSTUNREACH: 	/* Host can't be reached */
		case ENETDOWN: 		/* Interface down */
		case ENETUNREACH:	/* Network failure */
		case ECONNREFUSED:      /* ICMP port unreachable */
			res = XMIT_ERROR;	/* Don't bother with trying to transmit again */
		}
	}
	if (res != ast_str_strlen(data)) {
		ast_log(LOG_WARNING, "sip_xmit of %p (len %zu) to %s returned %d: %s\n", data, ast_str_strlen(data), ast_sockaddr_stringify(dst), res, strerror(errno));
	}

	return res;
}

/*! \brief Build a Via header for a request */
static void build_via(struct sip_pvt *p)
{
	/* Work around buggy UNIDEN UIP200 firmware */
	const char *rport = (ast_test_flag(&p->flags[0], SIP_NAT_FORCE_RPORT) || ast_test_flag(&p->flags[0], SIP_NAT_RPORT_PRESENT)) ? ";rport" : "";

	/* z9hG4bK is a magic cookie.  See RFC 3261 section 8.1.1.7 */
	snprintf(p->via, sizeof(p->via), "SIP/2.0/%s %s;branch=z9hG4bK%08x%s",
		 get_transport_pvt(p),
		 ast_sockaddr_stringify_remote(&p->ourip),
		 (int) p->branch, rport);
}

/*! \brief NAT fix - decide which IP address to use for Asterisk server?
 *
 * Using the localaddr structure built up with localnet statements in sip.conf
 * apply it to their address to see if we need to substitute our
 * externaddr or can get away with our internal bindaddr
 * 'us' is always overwritten.
 */
static void ast_sip_ouraddrfor(const struct ast_sockaddr *them, struct ast_sockaddr *us, struct sip_pvt *p)
{
	struct ast_sockaddr theirs;

	/* Set want_remap to non-zero if we want to remap 'us' to an externally
	 * reachable IP address and port. This is done if:
	 * 1. we have a localaddr list (containing 'internal' addresses marked
	 *    as 'deny', so ast_apply_ha() will return AST_SENSE_DENY on them,
	 *    and AST_SENSE_ALLOW on 'external' ones);
	 * 2. externaddr is set, so we know what to use as the
	 *    externally visible address;
	 * 3. the remote address, 'them', is external;
	 * 4. the address returned by ast_ouraddrfor() is 'internal' (AST_SENSE_DENY
	 *    when passed to ast_apply_ha() so it does need to be remapped.
	 *    This fourth condition is checked later.
	 */
	int want_remap = 0;

	ast_sockaddr_copy(us, &internip); /* starting guess for the internal address */
	/* now ask the system what would it use to talk to 'them' */
	ast_ouraddrfor(them, us);
	ast_sockaddr_copy(&theirs, them);

	if (ast_sockaddr_is_ipv6(&theirs)) {
		if (localaddr && !ast_sockaddr_isnull(&externaddr) && !ast_sockaddr_is_any(&bindaddr)) {
			ast_log(LOG_WARNING, "Address remapping activated in sip.conf "
				"but we're using IPv6, which doesn't need it. Please "
				"remove \"localnet\" and/or \"externaddr\" settings.\n");
		}
	} else {
		want_remap = localaddr &&
			!ast_sockaddr_isnull(&externaddr) &&
			ast_apply_ha(localaddr, &theirs) == AST_SENSE_ALLOW ;
	}

	if (want_remap &&
	    (!sip_cfg.matchexternaddrlocally || !ast_apply_ha(localaddr, us)) ) {
		/* if we used externhost, see if it is time to refresh the info */
		if (externexpire && time(NULL) >= externexpire) {
			if (ast_sockaddr_resolve_first(&externaddr, externhost, 0)) {
				ast_log(LOG_NOTICE, "Warning: Re-lookup of '%s' failed!\n", externhost);
			}
			externexpire = time(NULL) + externrefresh;
		}
		if (!ast_sockaddr_isnull(&externaddr)) {
			ast_sockaddr_copy(us, &externaddr);
			switch (p->socket.type) {
			case SIP_TRANSPORT_TCP:
				if (!externtcpport && ast_sockaddr_port(&externaddr)) {
					/* for consistency, default to the externaddr port */
					externtcpport = ast_sockaddr_port(&externaddr);
				}
				ast_sockaddr_set_port(us, externtcpport);
				break;
			case SIP_TRANSPORT_TLS:
				ast_sockaddr_set_port(us, externtlsport);
				break;
			case SIP_TRANSPORT_UDP:
				if (!ast_sockaddr_port(&externaddr)) {
					ast_sockaddr_set_port(us, ast_sockaddr_port(&bindaddr));
				}
				break;
			default:
				break;
			}
		}
		ast_debug(1, "Target address %s is not local, substituting externaddr\n",
			  ast_sockaddr_stringify(them));
	} else if (p) {
		/* no remapping, but we bind to a specific address, so use it. */
		switch (p->socket.type) {
		case SIP_TRANSPORT_TCP:
			if (!ast_sockaddr_is_any(&sip_tcp_desc.local_address)) {
				ast_sockaddr_copy(us,
						  &sip_tcp_desc.local_address);
			} else {
				ast_sockaddr_set_port(us,
						      ast_sockaddr_port(&sip_tcp_desc.local_address));
			}
			break;
		case SIP_TRANSPORT_TLS:
			if (!ast_sockaddr_is_any(&sip_tls_desc.local_address)) {
				ast_sockaddr_copy(us,
						  &sip_tls_desc.local_address);
			} else {
				ast_sockaddr_set_port(us,
						      ast_sockaddr_port(&sip_tls_desc.local_address));
			}
			break;
		case SIP_TRANSPORT_UDP:
			/* fall through on purpose */
		default:
			if (!ast_sockaddr_is_any(&bindaddr)) {
				ast_sockaddr_copy(us, &bindaddr);
			}
			if (!ast_sockaddr_port(us)) {
				ast_sockaddr_set_port(us, ast_sockaddr_port(&bindaddr));
			}
		}
	} else if (!ast_sockaddr_is_any(&bindaddr)) {
		ast_sockaddr_copy(us, &bindaddr);
	}
	ast_debug(3, "Setting SIP_TRANSPORT_%s with address %s\n", get_transport(p->socket.type), ast_sockaddr_stringify(us));
}

/*! \brief Append to SIP dialog history with arg list  */
static __attribute__((format(printf, 2, 0))) void append_history_va(struct sip_pvt *p, const char *fmt, va_list ap)
{
	char buf[80], *c = buf; /* max history length */
	struct sip_history *hist;
	int l;

	vsnprintf(buf, sizeof(buf), fmt, ap);
	strsep(&c, "\r\n"); /* Trim up everything after \r or \n */
	l = strlen(buf) + 1;
	if (!(hist = ast_calloc(1, sizeof(*hist) + l))) {
		return;
	}
	if (!p->history && !(p->history = ast_calloc(1, sizeof(*p->history)))) {
		ast_free(hist);
		return;
	}
	memcpy(hist->event, buf, l);
	if (p->history_entries == MAX_HISTORY_ENTRIES) {
		struct sip_history *oldest;
		oldest = AST_LIST_REMOVE_HEAD(p->history, list);
		p->history_entries--;
		ast_free(oldest);
	}
	AST_LIST_INSERT_TAIL(p->history, hist, list);
	p->history_entries++;
}

/*! \brief Append to SIP dialog history with arg list  */
static void append_history_full(struct sip_pvt *p, const char *fmt, ...)
{
	va_list ap;

	if (!p) {
		return;
	}

	if (!p->do_history && !recordhistory && !dumphistory) {
		return;
	}

	va_start(ap, fmt);
	append_history_va(p, fmt, ap);
	va_end(ap);

	return;
}

/*! \brief Retransmit SIP message if no answer (Called from scheduler) */
static int retrans_pkt(const void *data)
{
	struct sip_pkt *pkt = (struct sip_pkt *)data, *prev, *cur = NULL;
	int reschedule = DEFAULT_RETRANS;
	int xmitres = 0;
	/* how many ms until retrans timeout is reached */
	int64_t diff = pkt->retrans_stop_time - ast_tvdiff_ms(ast_tvnow(), pkt->time_sent);

	/* Do not retransmit if time out is reached. This will be negative if the time between
	 * the first transmission and now is larger than our timeout period. This is a fail safe
	 * check in case the scheduler gets behind or the clock is changed. */
	if ((diff <= 0) || (diff > pkt->retrans_stop_time)) {
		pkt->retrans_stop = 1;
	}

	/* Lock channel PVT */
	sip_pvt_lock(pkt->owner);

	if (!pkt->retrans_stop) {
		pkt->retrans++;
		if (!pkt->timer_t1) {	/* Re-schedule using timer_a and timer_t1 */
			if (sipdebug) {
				ast_debug(4, "SIP TIMER: Not rescheduling id #%d:%s (Method %d) (No timer T1)\n",
					pkt->retransid,
					sip_methods[pkt->method].text,
					pkt->method);
			}
		} else {
			int siptimer_a;

			if (sipdebug) {
				ast_debug(4, "SIP TIMER: Rescheduling retransmission #%d (%d) %s - %d\n",
					pkt->retransid,
					pkt->retrans,
					sip_methods[pkt->method].text,
					pkt->method);
			}
			if (!pkt->timer_a) {
				pkt->timer_a = 2 ;
			} else {
				pkt->timer_a = 2 * pkt->timer_a;
			}

			/* For non-invites, a maximum of 4 secs */
			siptimer_a = pkt->timer_t1 * pkt->timer_a;	/* Double each time */
			if (pkt->method != SIP_INVITE && siptimer_a > 4000) {
				siptimer_a = 4000;
			}

			/* Reschedule re-transmit */
			reschedule = siptimer_a;
			ast_debug(4, "** SIP timers: Rescheduling retransmission %d to %d ms (t1 %d ms (Retrans id #%d)) \n",
				pkt->retrans + 1,
				siptimer_a,
				pkt->timer_t1,
				pkt->retransid);
		}

		if (sip_debug_test_pvt(pkt->owner)) {
			const struct ast_sockaddr *dst = sip_real_dst(pkt->owner);
			ast_verbose("Retransmitting #%d (%s) to %s:\n%s\n---\n",
				pkt->retrans, sip_nat_mode(pkt->owner),
				ast_sockaddr_stringify(dst),
				pkt->data->str);
		}

		append_history(pkt->owner, "ReTx", "%d %s", reschedule, pkt->data->str);
		xmitres = __sip_xmit(pkt->owner, pkt->data);

		/* If there was no error during the network transmission, schedule the next retransmission,
		 * but if the next retransmission is going to be beyond our timeout period, mark the packet's
		 * stop_retrans value and set the next retransmit to be the exact time of timeout.  This will
		 * allow any responses to the packet to be processed before the packet is destroyed on the next
		 * call to this function by the scheduler. */
		if (xmitres != XMIT_ERROR) {
			if (reschedule >= diff) {
				pkt->retrans_stop = 1;
				reschedule = diff;
			}
			sip_pvt_unlock(pkt->owner);
			return  reschedule;
		}
	}

	/* At this point, either the packet's retransmission timed out, or there was a
	 * transmission error, either way destroy the scheduler item and this packet. */

	pkt->retransid = -1; /* Kill this scheduler item */

	if (pkt->owner && pkt->method != SIP_OPTIONS && xmitres == 0) {
		if (pkt->is_fatal || sipdebug) { /* Tell us if it's critical or if we're debugging */
			ast_log(LOG_WARNING, "Retransmission timeout reached on transmission %s for seqno %u (%s %s) -- See https://wiki.asterisk.org/wiki/display/AST/SIP+Retransmissions\n"
				"Packet timed out after %dms with no response\n",
				pkt->owner->callid,
				pkt->seqno,
				pkt->is_fatal ? "Critical" : "Non-critical",
				pkt->is_resp ? "Response" : "Request",
				(int) ast_tvdiff_ms(ast_tvnow(), pkt->time_sent));
		}
	} else if (pkt->method == SIP_OPTIONS && sipdebug) {
		ast_log(LOG_WARNING, "Cancelling retransmit of OPTIONs (call id %s)  -- See https://wiki.asterisk.org/wiki/display/AST/SIP+Retransmissions\n", pkt->owner->callid);
	}

	if (xmitres == XMIT_ERROR) {
		ast_log(LOG_WARNING, "Transmit error :: Cancelling transmission on Call ID %s\n", pkt->owner->callid);
		append_history(pkt->owner, "XmitErr", "%s", pkt->is_fatal ? "(Critical)" : "(Non-critical)");
	} else {
		append_history(pkt->owner, "MaxRetries", "%s", pkt->is_fatal ? "(Critical)" : "(Non-critical)");
	}

	if (pkt->is_fatal) {
		while(pkt->owner->owner && ast_channel_trylock(pkt->owner->owner)) {
			sip_pvt_unlock(pkt->owner);	/* SIP_PVT, not channel */
			usleep(1);
			sip_pvt_lock(pkt->owner);
		}
		if (pkt->owner->owner && !pkt->owner->owner->hangupcause) {
			pkt->owner->owner->hangupcause = AST_CAUSE_NO_USER_RESPONSE;
		}
		if (pkt->owner->owner) {
			ast_log(LOG_WARNING, "Hanging up call %s - no reply to our critical packet (see https://wiki.asterisk.org/wiki/display/AST/SIP+Retransmissions).\n", pkt->owner->callid);

			if (pkt->is_resp &&
				(pkt->response_code >= 200) &&
				(pkt->response_code < 300) &&
				pkt->owner->pendinginvite &&
				ast_test_flag(&pkt->owner->flags[1], SIP_PAGE2_DIALOG_ESTABLISHED)) {
				/* This is a timeout of the 2XX response to a pending INVITE.  In this case terminate the INVITE
				 * transaction just as if we received the ACK, but immediately hangup with a BYE (sip_hangup
				 * will send the BYE as long as the dialog is not set as "alreadygone")
				 * RFC 3261 section 13.3.1.4.
				 * "If the server retransmits the 2xx response for 64*T1 seconds without receiving
				 * an ACK, the dialog is confirmed, but the session SHOULD be terminated.  This is
				 * accomplished with a BYE, as described in Section 15." */
				pkt->owner->invitestate = INV_TERMINATED;
				pkt->owner->pendinginvite = 0;
			} else {
				/* there is nothing left to do, mark the dialog as gone */
				sip_alreadygone(pkt->owner);
			}
			ast_queue_hangup_with_cause(pkt->owner->owner, AST_CAUSE_PROTOCOL_ERROR);
			ast_channel_unlock(pkt->owner->owner);
		} else {
			/* If no channel owner, destroy now */

			/* Let the peerpoke system expire packets when the timer expires for poke_noanswer */
			if (pkt->method != SIP_OPTIONS && pkt->method != SIP_REGISTER) {
				pvt_set_needdestroy(pkt->owner, "no response to critical packet");
				sip_alreadygone(pkt->owner);
				append_history(pkt->owner, "DialogKill", "Killing this failed dialog immediately");
			}
		}
	}

	if (pkt->method == SIP_BYE) {
		/* We're not getting answers on SIP BYE's.  Tear down the call anyway. */
		sip_alreadygone(pkt->owner);
		if (pkt->owner->owner) {
			ast_channel_unlock(pkt->owner->owner);
		}
		append_history(pkt->owner, "ByeFailure", "Remote peer doesn't respond to bye. Destroying call anyway.");
		pvt_set_needdestroy(pkt->owner, "no response to BYE");
	}

	/* Remove the packet */
	for (prev = NULL, cur = pkt->owner->packets; cur; prev = cur, cur = cur->next) {
		if (cur == pkt) {
			UNLINK(cur, pkt->owner->packets, prev);
			sip_pvt_unlock(pkt->owner);
			if (pkt->owner) {
				pkt->owner = dialog_unref(pkt->owner,"pkt is being freed, its dialog ref is dead now");
			}
			if (pkt->data) {
				ast_free(pkt->data);
			}
			pkt->data = NULL;
			ast_free(pkt);
			return 0;
		}
	}
	/* error case */
	ast_log(LOG_WARNING, "Weird, couldn't find packet owner!\n");
	sip_pvt_unlock(pkt->owner);
	return 0;
}

/*!
 * \internal
 * \brief Transmit packet with retransmits
 * \return 0 on success, -1 on failure to allocate packet
 */
static enum sip_result __sip_reliable_xmit(struct sip_pvt *p, uint32_t seqno, int resp, struct ast_str *data, int fatal, int sipmethod)
{
	struct sip_pkt *pkt = NULL;
	int siptimer_a = DEFAULT_RETRANS;
	int xmitres = 0;
	int respid;

	if (sipmethod == SIP_INVITE) {
		/* Note this is a pending invite */
		p->pendinginvite = seqno;
	}

	/* If the transport is something reliable (TCP or TLS) then don't really send this reliably */
	/* I removed the code from retrans_pkt that does the same thing so it doesn't get loaded into the scheduler */
	/*! \todo According to the RFC some packets need to be retransmitted even if its TCP, so this needs to get revisited */
	if (!(p->socket.type & SIP_TRANSPORT_UDP)) {
		xmitres = __sip_xmit(p, data);	/* Send packet */
		if (xmitres == XMIT_ERROR) {	/* Serious network trouble, no need to try again */
			append_history(p, "XmitErr", "%s", fatal ? "(Critical)" : "(Non-critical)");
			return AST_FAILURE;
		} else {
			return AST_SUCCESS;
		}
	}

	if (!(pkt = ast_calloc(1, sizeof(*pkt)))) {
		return AST_FAILURE;
	}
	/* copy data, add a terminator and save length */
	if (!(pkt->data = ast_str_create(ast_str_strlen(data)))) {
		ast_free(pkt);
		return AST_FAILURE;
	}
	ast_str_set(&pkt->data, 0, "%s%s", data->str, "\0");
	/* copy other parameters from the caller */
	pkt->method = sipmethod;
	pkt->seqno = seqno;
	pkt->is_resp = resp;
	pkt->is_fatal = fatal;
	pkt->owner = dialog_ref(p, "__sip_reliable_xmit: setting pkt->owner");
	pkt->next = p->packets;
	p->packets = pkt;	/* Add it to the queue */
	if (resp) {
		/* Parse out the response code */
		if (sscanf(ast_str_buffer(pkt->data), "SIP/2.0 %30u", &respid) == 1) {
			pkt->response_code = respid;
		}
	}
	pkt->timer_t1 = p->timer_t1;	/* Set SIP timer T1 */
	pkt->retransid = -1;
	if (pkt->timer_t1) {
		siptimer_a = pkt->timer_t1;
	}

	pkt->time_sent = ast_tvnow(); /* time packet was sent */
	pkt->retrans_stop_time = 64 * (pkt->timer_t1 ? pkt->timer_t1 : DEFAULT_TIMER_T1); /* time in ms after pkt->time_sent to stop retransmission */

	/* Schedule retransmission */
	AST_SCHED_REPLACE_VARIABLE(pkt->retransid, sched, siptimer_a, retrans_pkt, pkt, 1);
	if (sipdebug) {
		ast_debug(4, "*** SIP TIMER: Initializing retransmit timer on packet: Id  #%d\n", pkt->retransid);
	}

	xmitres = __sip_xmit(pkt->owner, pkt->data);	/* Send packet */

	if (xmitres == XMIT_ERROR) {	/* Serious network trouble, no need to try again */
		append_history(pkt->owner, "XmitErr", "%s", pkt->is_fatal ? "(Critical)" : "(Non-critical)");
		ast_log(LOG_ERROR, "Serious Network Trouble; __sip_xmit returns error for pkt data\n");
		AST_SCHED_DEL(sched, pkt->retransid);
		p->packets = pkt->next;
		pkt->owner = dialog_unref(pkt->owner,"pkt is being freed, its dialog ref is dead now");
		ast_free(pkt->data);
		ast_free(pkt);
		return AST_FAILURE;
	} else {
		/* This is odd, but since the retrans timer starts at 500ms and the do_monitor thread
		 * only wakes up every 1000ms by default, we have to poke the thread here to make
		 * sure it successfully detects this must be retransmitted in less time than
		 * it usually sleeps for. Otherwise it might not retransmit this packet for 1000ms. */
		if (monitor_thread != AST_PTHREADT_NULL) {
			pthread_kill(monitor_thread, SIGURG);
		}
		return AST_SUCCESS;
	}
}

/*! \brief Kill a SIP dialog (called only by the scheduler)
 * The scheduler has a reference to this dialog when p->autokillid != -1,
 * and we are called using that reference. So if the event is not
 * rescheduled, we need to call dialog_unref().
 */
static int __sip_autodestruct(const void *data)
{
	struct sip_pvt *p = (struct sip_pvt *)data;
	struct ast_channel *owner;

	/* If this is a subscription, tell the phone that we got a timeout */
	if (p->subscribed && p->subscribed != MWI_NOTIFICATION && p->subscribed != CALL_COMPLETION) {
		transmit_state_notify(p, AST_EXTENSION_DEACTIVATED, 1, TRUE);	/* Send last notification */
		p->subscribed = NONE;
		append_history(p, "Subscribestatus", "timeout");
		ast_debug(3, "Re-scheduled destruction of SIP subscription %s\n", p->callid ? p->callid : "<unknown>");
		return 10000;	/* Reschedule this destruction so that we know that it's gone */
	}

	/* If there are packets still waiting for delivery, delay the destruction */
	if (p->packets) {
		if (!p->needdestroy) {
			char method_str[31];
			ast_debug(3, "Re-scheduled destruction of SIP call %s\n", p->callid ? p->callid : "<unknown>");
			append_history(p, "ReliableXmit", "timeout");
			if (sscanf(p->lastmsg, "Tx: %30s", method_str) == 1 || sscanf(p->lastmsg, "Rx: %30s", method_str) == 1) {
				if (p->ongoing_reinvite || method_match(SIP_CANCEL, method_str) || method_match(SIP_BYE, method_str)) {
					pvt_set_needdestroy(p, "autodestruct");
				}
			}
			return 10000;
		} else {
			/* They've had their chance to respond. Time to bail */
			__sip_pretend_ack(p);
		}
	}

	/* Reset schedule ID */
	p->autokillid = -1;


	/*
	 * Lock both the pvt and the channel safely so that we can queue up a frame.
	 */
	owner = sip_pvt_lock_full(p);
	if (owner) {
		ast_log(LOG_WARNING, "Autodestruct on dialog '%s' with owner in place (Method: %s)\n", p->callid, sip_methods[p->method].text);
		ast_queue_hangup_with_cause(owner, AST_CAUSE_PROTOCOL_ERROR);
		ast_channel_unlock(owner);
		ast_channel_unref(owner);
	} else if (p->refer && !p->alreadygone) {
		ast_debug(3, "Finally hanging up channel after transfer: %s\n", p->callid);
		stop_media_flows(p);
		transmit_request_with_auth(p, SIP_BYE, 0, XMIT_RELIABLE, 1);
		append_history(p, "ReferBYE", "Sending BYE on transferer call leg %s", p->callid);
		sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
	} else {
		append_history(p, "AutoDestroy", "%s", p->callid);
		ast_debug(3, "Auto destroying SIP dialog '%s'\n", p->callid);
		sip_pvt_unlock(p);
		dialog_unlink_all(p); /* once it's unlinked and unrefd everywhere, it'll be freed automagically */
		sip_pvt_lock(p);
		/* dialog_unref(p, "unref dialog-- no other matching conditions"); -- unlink all now should finish off the dialog's references and free it. */
		/* sip_destroy(p); */		/* Go ahead and destroy dialog. All attempts to recover is done */
		/* sip_destroy also absorbs the reference */
	}

	sip_pvt_unlock(p);

	dialog_unref(p, "The ref to a dialog passed to this sched callback is going out of scope; unref it.");

	return 0;
}

/*! \brief Schedule final destruction of SIP dialog.  This can not be canceled.
 *  This function is used to keep a dialog around for a period of time in order
 *  to properly respond to any retransmits. */
void sip_scheddestroy_final(struct sip_pvt *p, int ms)
{
	if (p->final_destruction_scheduled) {
		return; /* already set final destruction */
	}

	sip_scheddestroy(p, ms);
	if (p->autokillid != -1) {
		p->final_destruction_scheduled = 1;
	}
}

/*! \brief Schedule destruction of SIP dialog */
void sip_scheddestroy(struct sip_pvt *p, int ms)
{
	if (p->final_destruction_scheduled) {
		return; /* already set final destruction */
	}

	if (ms < 0) {
		if (p->timer_t1 == 0) {
			p->timer_t1 = global_t1;	/* Set timer T1 if not set (RFC 3261) */
		}
		if (p->timer_b == 0) {
			p->timer_b = global_timer_b;  /* Set timer B if not set (RFC 3261) */
		}
		ms = p->timer_t1 * 64;
	}
	if (sip_debug_test_pvt(p)) {
		ast_verbose("Scheduling destruction of SIP dialog '%s' in %d ms (Method: %s)\n", p->callid, ms, sip_methods[p->method].text);
	}
	if (sip_cancel_destroy(p)) {
		ast_log(LOG_WARNING, "Unable to cancel SIP destruction.  Expect bad things.\n");
	}

	if (p->do_history) {
		append_history(p, "SchedDestroy", "%d ms", ms);
	}
	p->autokillid = ast_sched_add(sched, ms, __sip_autodestruct, dialog_ref(p, "setting ref as passing into ast_sched_add for __sip_autodestruct"));

	if (p->stimer && p->stimer->st_active == TRUE && p->stimer->st_schedid > 0) {
		stop_session_timer(p);
	}
}

/*! \brief Cancel destruction of SIP dialog.
 * Be careful as this also absorbs the reference - if you call it
 * from within the scheduler, this might be the last reference.
 */
int sip_cancel_destroy(struct sip_pvt *p)
{
	if (p->final_destruction_scheduled) {
		return 0;
	}

	if (p->autokillid > -1) {
		append_history(p, "CancelDestroy", "");
		AST_SCHED_DEL_UNREF(sched, p->autokillid, dialog_unref(p, "remove ref for autokillid"));
	}
	return 0;
}

/*! \brief Acknowledges receipt of a packet and stops retransmission
 * called with p locked*/
int __sip_ack(struct sip_pvt *p, uint32_t seqno, int resp, int sipmethod)
{
	struct sip_pkt *cur, *prev = NULL;
	const char *msg = "Not Found";	/* used only for debugging */
	int res = FALSE;

	/* If we have an outbound proxy for this dialog, then delete it now since
	  the rest of the requests in this dialog needs to follow the routing.
	  If obforcing is set, we will keep the outbound proxy during the whole
	  dialog, regardless of what the SIP rfc says
	*/
	if (p->outboundproxy && !p->outboundproxy->force){
		ref_proxy(p, NULL);
	}

	for (cur = p->packets; cur; prev = cur, cur = cur->next) {
		if (cur->seqno != seqno || cur->is_resp != resp) {
			continue;
		}
		if (cur->is_resp || cur->method == sipmethod) {
			res = TRUE;
			msg = "Found";
			if (!resp && (seqno == p->pendinginvite)) {
				ast_debug(1, "Acked pending invite %u\n", p->pendinginvite);
				p->pendinginvite = 0;
			}
			if (cur->retransid > -1) {
				if (sipdebug)
					ast_debug(4, "** SIP TIMER: Cancelling retransmit of packet (reply received) Retransid #%d\n", cur->retransid);
			}
			/* This odd section is designed to thwart a
			 * race condition in the packet scheduler. There are
			 * two conditions under which deleting the packet from the
			 * scheduler can fail.
			 *
			 * 1. The packet has been removed from the scheduler because retransmission
			 * is being attempted. The problem is that if the packet is currently attempting
			 * retransmission and we are at this point in the code, then that MUST mean
			 * that retrans_pkt is waiting on p's lock. Therefore we will relinquish the
			 * lock temporarily to allow retransmission.
			 *
			 * 2. The packet has reached its maximum number of retransmissions and has
			 * been permanently removed from the packet scheduler. If this is the case, then
			 * the packet's retransid will be set to -1. The atomicity of the setting and checking
			 * of the retransid to -1 is ensured since in both cases p's lock is held.
			 */
			while (cur->retransid > -1 && ast_sched_del(sched, cur->retransid)) {
				sip_pvt_unlock(p);
				usleep(1);
				sip_pvt_lock(p);
			}
			UNLINK(cur, p->packets, prev);
			dialog_unref(cur->owner, "unref pkt cur->owner dialog from sip ack before freeing pkt");
			if (cur->data) {
				ast_free(cur->data);
			}
			ast_free(cur);
			break;
		}
	}
	ast_debug(1, "Stopping retransmission on '%s' of %s %u: Match %s\n",
		p->callid, resp ? "Response" : "Request", seqno, msg);
	return res;
}

/*! \brief Pretend to ack all packets
 * called with p locked */
void __sip_pretend_ack(struct sip_pvt *p)
{
	struct sip_pkt *cur = NULL;

	while (p->packets) {
		int method;
		if (cur == p->packets) {
			ast_log(LOG_WARNING, "Have a packet that doesn't want to give up! %s\n", sip_methods[cur->method].text);
			return;
		}
		cur = p->packets;
		method = (cur->method) ? cur->method : find_sip_method(cur->data->str);
		__sip_ack(p, cur->seqno, cur->is_resp, method);
	}
}

/*! \brief Acks receipt of packet, keep it around (used for provisional responses) */
int __sip_semi_ack(struct sip_pvt *p, uint32_t seqno, int resp, int sipmethod)
{
	struct sip_pkt *cur;
	int res = FALSE;

	for (cur = p->packets; cur; cur = cur->next) {
		if (cur->seqno == seqno && cur->is_resp == resp &&
			(cur->is_resp || method_match(sipmethod, cur->data->str))) {
			/* this is our baby */
			if (cur->retransid > -1) {
				if (sipdebug)
					ast_debug(4, "*** SIP TIMER: Cancelling retransmission #%d - %s (got response)\n", cur->retransid, sip_methods[sipmethod].text);
			}
			AST_SCHED_DEL(sched, cur->retransid);
			res = TRUE;
			break;
		}
	}
	ast_debug(1, "(Provisional) Stopping retransmission (but retaining packet) on '%s' %s %u: %s\n", p->callid, resp ? "Response" : "Request", seqno, res == -1 ? "Not Found" : "Found");
	return res;
}


/*! \brief Copy SIP request, parse it */
static void parse_copy(struct sip_request *dst, const struct sip_request *src)
{
	copy_request(dst, src);
	parse_request(dst);
}

/*! \brief add a blank line if no body */
static void add_blank(struct sip_request *req)
{
	if (!req->lines) {
		/* Add extra empty return. add_header() reserves 4 bytes so cannot be truncated */
		ast_str_append(&req->data, 0, "\r\n");
	}
}

static int send_provisional_keepalive_full(struct sip_pvt *pvt, int with_sdp)
{
	const char *msg = NULL;
	struct ast_channel *chan;
	int res = 0;

	chan = sip_pvt_lock_full(pvt);

	if (!pvt->last_provisional || !strncasecmp(pvt->last_provisional, "100", 3)) {
		msg = "183 Session Progress";
	}

	if (pvt->invitestate < INV_COMPLETED) {
		if (with_sdp) {
			transmit_response_with_sdp(pvt, S_OR(msg, pvt->last_provisional), &pvt->initreq, XMIT_UNRELIABLE, FALSE, FALSE);
		} else {
			transmit_response(pvt, S_OR(msg, pvt->last_provisional), &pvt->initreq);
		}
		res = PROVIS_KEEPALIVE_TIMEOUT;
	}

	if (chan) {
		ast_channel_unlock(chan);
		chan = ast_channel_unref(chan);
	}

	if (!res) {
		pvt->provisional_keepalive_sched_id = -1;
	}

	sip_pvt_unlock(pvt);

#if 0
	/*
	 * XXX BUG TODO
	 *
	 * Without this code, it appears as if this function is leaking its
	 * reference to the sip_pvt.  However, adding it introduces a crash.
	 * This points to some sort of reference count imbalance elsewhere,
	 * but I'm not sure where ...
	 */
	if (!res) {
		dialog_unref(pvt, "dialog ref for provisional keepalive");
	}
#endif

	return res;
}

static int send_provisional_keepalive(const void *data) {
	struct sip_pvt *pvt = (struct sip_pvt *) data;

	return send_provisional_keepalive_full(pvt, 0);
}

static int send_provisional_keepalive_with_sdp(const void *data) {
	struct sip_pvt *pvt = (void *)data;

	return send_provisional_keepalive_full(pvt, 1);
}

static void update_provisional_keepalive(struct sip_pvt *pvt, int with_sdp)
{
	AST_SCHED_DEL_UNREF(sched, pvt->provisional_keepalive_sched_id, dialog_unref(pvt, "when you delete the provisional_keepalive_sched_id, you should dec the refcount for the stored dialog ptr"));

	pvt->provisional_keepalive_sched_id = ast_sched_add(sched, PROVIS_KEEPALIVE_TIMEOUT,
		with_sdp ? send_provisional_keepalive_with_sdp : send_provisional_keepalive, dialog_ref(pvt, "Increment refcount to pass dialog pointer to sched callback"));
}

/*! \brief Transmit response on SIP request*/
static int send_response(struct sip_pvt *p, struct sip_request *req, enum xmittype reliable, uint32_t seqno)
{
	int res;

	finalize_content(req);
	add_blank(req);
	if (sip_debug_test_pvt(p)) {
		const struct ast_sockaddr *dst = sip_real_dst(p);

		ast_verbose("\n<--- %sTransmitting (%s) to %s --->\n%s\n<------------>\n",
			reliable ? "Reliably " : "", sip_nat_mode(p),
			ast_sockaddr_stringify(dst),
			req->data->str);
	}
	if (p->do_history) {
		struct sip_request tmp = { .rlPart1 = 0, };
		parse_copy(&tmp, req);
		append_history(p, reliable ? "TxRespRel" : "TxResp", "%s / %s - %s", tmp.data->str, get_header(&tmp, "CSeq"),
			(tmp.method == SIP_RESPONSE || tmp.method == SIP_UNKNOWN) ? REQ_OFFSET_TO_STR(&tmp, rlPart2) : sip_methods[tmp.method].text);
		deinit_req(&tmp);
	}

	/* If we are sending a final response to an INVITE, stop retransmitting provisional responses */
	if (p->initreq.method == SIP_INVITE && reliable == XMIT_CRITICAL) {
		AST_SCHED_DEL_UNREF(sched, p->provisional_keepalive_sched_id, dialog_unref(p, "when you delete the provisional_keepalive_sched_id, you should dec the refcount for the stored dialog ptr"));
	}

	res = (reliable) ?
		 __sip_reliable_xmit(p, seqno, 1, req->data, (reliable == XMIT_CRITICAL), req->method) :
		__sip_xmit(p, req->data);
	deinit_req(req);
	if (res > 0) {
		return 0;
	}
	return res;
}

/*!
 * \internal
 * \brief Send SIP Request to the other part of the dialogue
 * \return see \ref __sip_xmit
 */
static int send_request(struct sip_pvt *p, struct sip_request *req, enum xmittype reliable, uint32_t seqno)
{
	int res;

	/* If we have an outbound proxy, reset peer address
		Only do this once.
	*/
	if (p->outboundproxy) {
		p->sa = p->outboundproxy->ip;
	}

	finalize_content(req);
	add_blank(req);
	if (sip_debug_test_pvt(p)) {
		if (ast_test_flag(&p->flags[0], SIP_NAT_FORCE_RPORT)) {
			ast_verbose("%sTransmitting (NAT) to %s:\n%s\n---\n", reliable ? "Reliably " : "", ast_sockaddr_stringify(&p->recv), req->data->str);
		} else {
			ast_verbose("%sTransmitting (no NAT) to %s:\n%s\n---\n", reliable ? "Reliably " : "", ast_sockaddr_stringify(&p->sa), req->data->str);
		}
	}
	if (p->do_history) {
		struct sip_request tmp = { .rlPart1 = 0, };
		parse_copy(&tmp, req);
		append_history(p, reliable ? "TxReqRel" : "TxReq", "%s / %s - %s", tmp.data->str, get_header(&tmp, "CSeq"), sip_methods[tmp.method].text);
		deinit_req(&tmp);
	}
	res = (reliable) ?
		__sip_reliable_xmit(p, seqno, 0, req->data, (reliable == XMIT_CRITICAL), req->method) :
		__sip_xmit(p, req->data);
	deinit_req(req);
	return res;
}

static void enable_dsp_detect(struct sip_pvt *p)
{
	int features = 0;

	if (p->dsp) {
		return;
	}

	if ((ast_test_flag(&p->flags[0], SIP_DTMF) == SIP_DTMF_INBAND) ||
	    (ast_test_flag(&p->flags[0], SIP_DTMF) == SIP_DTMF_AUTO)) {
		if (p->rtp) {
			ast_rtp_instance_dtmf_mode_set(p->rtp, AST_RTP_DTMF_MODE_INBAND);
		}
		features |= DSP_FEATURE_DIGIT_DETECT;
	}

	if (ast_test_flag(&p->flags[1], SIP_PAGE2_FAX_DETECT_CNG)) {
		features |= DSP_FEATURE_FAX_DETECT;
	}

	if (!features) {
		return;
	}

	if (!(p->dsp = ast_dsp_new())) {
		return;
	}

	ast_dsp_set_features(p->dsp, features);
	if (global_relaxdtmf) {
		ast_dsp_set_digitmode(p->dsp, DSP_DIGITMODE_DTMF | DSP_DIGITMODE_RELAXDTMF);
	}
}

static void disable_dsp_detect(struct sip_pvt *p)
{
	if (p->dsp) {
		ast_dsp_free(p->dsp);
		p->dsp = NULL;
	}
}

/*! \brief Set an option on a SIP dialog */
static int sip_setoption(struct ast_channel *chan, int option, void *data, int datalen)
{
	int res = -1;
	struct sip_pvt *p = chan->tech_pvt;

        if (!p) {
        	ast_log(LOG_ERROR, "Attempt to Ref a null pointer.  sip private structure is gone!\n");
        	return -1;
        }

	sip_pvt_lock(p);

	switch (option) {
	case AST_OPTION_FORMAT_READ:
		if (p->rtp) {
			res = ast_rtp_instance_set_read_format(p->rtp, *(int *) data);
		}
		break;
	case AST_OPTION_FORMAT_WRITE:
		if (p->rtp) {
			res = ast_rtp_instance_set_write_format(p->rtp, *(int *) data);
		}
		break;
	case AST_OPTION_MAKE_COMPATIBLE:
		if (p->rtp) {
			res = ast_rtp_instance_make_compatible(chan, p->rtp, (struct ast_channel *) data);
		}
		break;
	case AST_OPTION_DIGIT_DETECT:
		if ((ast_test_flag(&p->flags[0], SIP_DTMF) == SIP_DTMF_INBAND) ||
		    (ast_test_flag(&p->flags[0], SIP_DTMF) == SIP_DTMF_AUTO)) {
			char *cp = (char *) data;

			ast_debug(1, "%sabling digit detection on %s\n", *cp ? "En" : "Dis", chan->name);
			if (*cp) {
				enable_dsp_detect(p);
			} else {
				disable_dsp_detect(p);
			}
			res = 0;
		}
		break;
	case AST_OPTION_SECURE_SIGNALING:
		p->req_secure_signaling = *(unsigned int *) data;
		res = 0;
		break;
	case AST_OPTION_SECURE_MEDIA:
		ast_set2_flag(&p->flags[1], *(unsigned int *) data, SIP_PAGE2_USE_SRTP);
		res = 0;
		break;
	default:
		break;
	}

	sip_pvt_unlock(p);

	return res;
}

/*! \brief Query an option on a SIP dialog */
static int sip_queryoption(struct ast_channel *chan, int option, void *data, int *datalen)
{
	int res = -1;
	enum ast_t38_state state = T38_STATE_UNAVAILABLE;
	struct sip_pvt *p = (struct sip_pvt *) chan->tech_pvt;
	char *cp;

	sip_pvt_lock(p);

	switch (option) {
	case AST_OPTION_T38_STATE:
		/* Make sure we got an ast_t38_state enum passed in */
		if (*datalen != sizeof(enum ast_t38_state)) {
			ast_log(LOG_ERROR, "Invalid datalen for AST_OPTION_T38_STATE option. Expected %d, got %d\n", (int)sizeof(enum ast_t38_state), *datalen);
			break;
		}

		/* Now if T38 support is enabled we need to look and see what the current state is to get what we want to report back */
		if (ast_test_flag(&p->flags[1], SIP_PAGE2_T38SUPPORT)) {
			switch (p->t38.state) {
			case T38_LOCAL_REINVITE:
			case T38_PEER_REINVITE:
				state = T38_STATE_NEGOTIATING;
				break;
			case T38_ENABLED:
				state = T38_STATE_NEGOTIATED;
				break;
			default:
				state = T38_STATE_UNKNOWN;
			}
		}

		*((enum ast_t38_state *) data) = state;
		res = 0;

		break;
	case AST_OPTION_DIGIT_DETECT:
		cp = (char *) data;
		*cp = p->dsp ? 1 : 0;
		ast_debug(1, "Reporting digit detection %sabled on %s\n", *cp ? "en" : "dis", chan->name);
		break;
	case AST_OPTION_SECURE_SIGNALING:
		*((unsigned int *) data) = p->req_secure_signaling;
		res = 0;
		break;
	case AST_OPTION_SECURE_MEDIA:
		*((unsigned int *) data) = ast_test_flag(&p->flags[1], SIP_PAGE2_USE_SRTP) ? 1 : 0;
		res = 0;
		break;
	case AST_OPTION_DEVICE_NAME:
		if (p && p->outgoing_call) {
			cp = (char *) data;
			ast_copy_string(cp, p->dialstring, *datalen);
			res = 0;
		}
		/* We purposely break with a return of -1 in the
		 * implied else case here
		 */
		break;
	default:
		break;
	}

	sip_pvt_unlock(p);

	return res;
}

/*! \brief Locate closing quote in a string, skipping escaped quotes.
 * optionally with a limit on the search.
 * start must be past the first quote.
 */
const char *find_closing_quote(const char *start, const char *lim)
{
	char last_char = '\0';
	const char *s;
	for (s = start; *s && s != lim; last_char = *s++) {
		if (*s == '"' && last_char != '\\')
			break;
	}
	return s;
}

/*! \brief Send message with Access-URL header, if this is an HTML URL only! */
static int sip_sendhtml(struct ast_channel *chan, int subclass, const char *data, int datalen)
{
	struct sip_pvt *p = chan->tech_pvt;

	if (subclass != AST_HTML_URL)
		return -1;

	ast_string_field_build(p, url, "<%s>;mode=active", data);

	if (sip_debug_test_pvt(p))
		ast_debug(1, "Send URL %s, state = %d!\n", data, chan->_state);

	switch (chan->_state) {
	case AST_STATE_RING:
		transmit_response(p, "100 Trying", &p->initreq);
		break;
	case AST_STATE_RINGING:
		transmit_response(p, "180 Ringing", &p->initreq);
		break;
	case AST_STATE_UP:
		if (!p->pendinginvite) {		/* We are up, and have no outstanding invite */
			transmit_reinvite_with_sdp(p, FALSE, FALSE);
		} else if (!ast_test_flag(&p->flags[0], SIP_PENDINGBYE)) {
			ast_set_flag(&p->flags[0], SIP_NEEDREINVITE);	
		}	
		break;
	default:
		ast_log(LOG_WARNING, "Don't know how to send URI when state is %d!\n", chan->_state);
	}

	return 0;
}

/*! \brief Deliver SIP call ID for the call */
static const char *sip_get_callid(struct ast_channel *chan)
{
	return chan->tech_pvt ? ((struct sip_pvt *) chan->tech_pvt)->callid : "";
}

/*!
 * \internal
 * \brief Send SIP MESSAGE text within a call
 * \note Called from PBX core sendtext() application
 */
static int sip_sendtext(struct ast_channel *ast, const char *text)
{
	struct sip_pvt *dialog = ast->tech_pvt;
	int debug;

	if (!dialog) {
		return -1;
	}
	/* NOT ast_strlen_zero, because a zero-length message is specifically
	 * allowed by RFC 3428 (See section 10, Examples) */
	if (!text) {
		return 0;
	}
	if(!is_method_allowed(&dialog->allowed_methods, SIP_MESSAGE)) {
		ast_debug(2, "Trying to send MESSAGE to device that does not support it.\n");
		return(0);
	}

	debug = sip_debug_test_pvt(dialog);
	if (debug) {
		ast_verbose("Sending text %s on %s\n", text, ast->name);
	}

	transmit_message_with_text(dialog, text);
	return 0;	
}

/*! \brief Update peer object in realtime storage
	If the Asterisk system name is set in asterisk.conf, we will use
	that name and store that in the "regserver" field in the sippeers
	table to facilitate multi-server setups.
*/
static void realtime_update_peer(const char *peername, struct ast_sockaddr *addr, const char *defaultuser, const char *fullcontact, const char *useragent, int expirey, unsigned short deprecated_username, int lastms)
{
	char port[10];
	char ipaddr[INET6_ADDRSTRLEN];
	char regseconds[20];
	char *tablename = NULL;
	char str_lastms[20];

	const char *sysname = ast_config_AST_SYSTEM_NAME;
	char *syslabel = NULL;

	time_t nowtime = time(NULL) + expirey;
	const char *fc = fullcontact ? "fullcontact" : NULL;

	int realtimeregs = ast_check_realtime("sipregs");

	tablename = realtimeregs ? "sipregs" : "sippeers";
	

	snprintf(str_lastms, sizeof(str_lastms), "%d", lastms);
	snprintf(regseconds, sizeof(regseconds), "%d", (int)nowtime);	/* Expiration time */
	ast_copy_string(ipaddr, ast_sockaddr_isnull(addr) ? "" : ast_sockaddr_stringify_addr(addr), sizeof(ipaddr));
	ast_copy_string(port, ast_sockaddr_port(addr) ? ast_sockaddr_stringify_port(addr) : "", sizeof(port));

	if (ast_strlen_zero(sysname))	/* No system name, disable this */
		sysname = NULL;
	else if (sip_cfg.rtsave_sysname)
		syslabel = "regserver";

	/* XXX IMPORTANT: Anytime you add a new parameter to be updated, you
         *  must also add it to contrib/scripts/asterisk.ldap-schema,
         *  contrib/scripts/asterisk.ldif,
         *  and to configs/res_ldap.conf.sample as described in
         *  bugs 15156 and 15895 
         */
	if (fc) {
		ast_update_realtime(tablename, "name", peername, "ipaddr", ipaddr,
			"port", port, "regseconds", regseconds,
			deprecated_username ? "username" : "defaultuser", defaultuser,
			"useragent", useragent, "lastms", str_lastms,
			fc, fullcontact, syslabel, sysname, SENTINEL); /* note fc and syslabel _can_ be NULL */
	} else {
		ast_update_realtime(tablename, "name", peername, "ipaddr", ipaddr,
			"port", port, "regseconds", regseconds,
			"useragent", useragent, "lastms", str_lastms,
			deprecated_username ? "username" : "defaultuser", defaultuser,
			syslabel, sysname, SENTINEL); /* note syslabel _can_ be NULL */
	}
}

/*! \brief Automatically add peer extension to dial plan */
static void register_peer_exten(struct sip_peer *peer, int onoff)
{
	char multi[256];
	char *stringp, *ext, *context;
	struct pbx_find_info q = { .stacklen = 0 };

	/* XXX note that sip_cfg.regcontext is both a global 'enable' flag and
	 * the name of the global regexten context, if not specified
	 * individually.
	 */
	if (ast_strlen_zero(sip_cfg.regcontext))
		return;

	ast_copy_string(multi, S_OR(peer->regexten, peer->name), sizeof(multi));
	stringp = multi;
	while ((ext = strsep(&stringp, "&"))) {
		if ((context = strchr(ext, '@'))) {
			*context++ = '\0';	/* split ext@context */
			if (!ast_context_find(context)) {
				ast_log(LOG_WARNING, "Context %s must exist in regcontext= in sip.conf!\n", context);
				continue;
			}
		} else {
			context = sip_cfg.regcontext;
		}
		if (onoff) {
			if (!ast_exists_extension(NULL, context, ext, 1, NULL)) {
				ast_add_extension(context, 1, ext, 1, NULL, NULL, "Noop",
					 ast_strdup(peer->name), ast_free_ptr, "SIP");
			}
		} else if (pbx_find_extension(NULL, NULL, &q, context, ext, 1, NULL, "", E_MATCH)) {
			ast_context_remove_extension(context, ext, 1, NULL);
		}
	}
}

/*! Destroy mailbox subscriptions */
static void destroy_mailbox(struct sip_mailbox *mailbox)
{
	if (mailbox->event_sub)
		ast_event_unsubscribe(mailbox->event_sub);
	ast_free(mailbox);
}

/*! Destroy all peer-related mailbox subscriptions */
static void clear_peer_mailboxes(struct sip_peer *peer)
{
	struct sip_mailbox *mailbox;

	while ((mailbox = AST_LIST_REMOVE_HEAD(&peer->mailboxes, entry)))
		destroy_mailbox(mailbox);
}

static void sip_destroy_peer_fn(void *peer)
{
	sip_destroy_peer(peer);
}

/*! \brief Destroy peer object from memory */
static void sip_destroy_peer(struct sip_peer *peer)
{
	ast_debug(3, "Destroying SIP peer %s\n", peer->name);

	/*
	 * Remove any mailbox event subscriptions for this peer before
	 * we destroy anything.  An event subscription callback may be
	 * happening right now.
	 */
	clear_peer_mailboxes(peer);

	if (peer->outboundproxy) {
		ao2_ref(peer->outboundproxy, -1);
		peer->outboundproxy = NULL;
	}

	/* Delete it, it needs to disappear */
	if (peer->call) {
		dialog_unlink_all(peer->call);
		peer->call = dialog_unref(peer->call, "peer->call is being unset");
	}

	if (peer->mwipvt) {	/* We have an active subscription, delete it */
		dialog_unlink_all(peer->mwipvt);
		peer->mwipvt = dialog_unref(peer->mwipvt, "unreffing peer->mwipvt");
	}
	
	if (peer->chanvars) {
		ast_variables_destroy(peer->chanvars);
		peer->chanvars = NULL;
	}
	
	register_peer_exten(peer, FALSE);
	ast_free_ha(peer->ha);
	ast_free_ha(peer->directmediaha);
	if (peer->selfdestruct)
		ast_atomic_fetchadd_int(&apeerobjs, -1);
	else if (!ast_test_flag(&global_flags[1], SIP_PAGE2_RTCACHEFRIENDS) && peer->is_realtime) {
		ast_atomic_fetchadd_int(&rpeerobjs, -1);
		ast_debug(3, "-REALTIME- peer Destroyed. Name: %s. Realtime Peer objects: %d\n", peer->name, rpeerobjs);
	} else
		ast_atomic_fetchadd_int(&speerobjs, -1);
	if (peer->auth) {
		ao2_t_ref(peer->auth, -1, "Removing peer authentication");
		peer->auth = NULL;
	}

	if (peer->socket.tcptls_session) {
		ao2_ref(peer->socket.tcptls_session, -1);
		peer->socket.tcptls_session = NULL;
	}

	ast_cc_config_params_destroy(peer->cc_params);

	ast_string_field_free_memory(peer);
}

/*! \brief Update peer data in database (if used) */
static void update_peer(struct sip_peer *p, int expire)
{
	int rtcachefriends = ast_test_flag(&p->flags[1], SIP_PAGE2_RTCACHEFRIENDS);
	if (sip_cfg.peer_rtupdate &&
	    (p->is_realtime || rtcachefriends)) {
		realtime_update_peer(p->name, &p->addr, p->username, p->fullcontact, p->useragent, expire, p->deprecated_username, p->lastms);
	}
}

static struct ast_variable *get_insecure_variable_from_config(struct ast_config *cfg)
{
	struct ast_variable *var = NULL;
	struct ast_flags flags = {0};
	char *cat = NULL;
	const char *insecure;
	while ((cat = ast_category_browse(cfg, cat))) {
		insecure = ast_variable_retrieve(cfg, cat, "insecure");
		set_insecure_flags(&flags, insecure, -1);
		if (ast_test_flag(&flags, SIP_INSECURE_PORT)) {
			var = ast_category_root(cfg, cat);
			break;
		}
	}
	return var;
}

static struct ast_variable *get_insecure_variable_from_sippeers(const char *column, const char *value)
{
	struct ast_config *peerlist;
	struct ast_variable *var = NULL;
	if ((peerlist = ast_load_realtime_multientry("sippeers", column, value, "insecure LIKE", "%port%", SENTINEL))) {
		if ((var = get_insecure_variable_from_config(peerlist))) {
			/* Must clone, because var will get freed along with
			 * peerlist. */
			var = ast_variables_dup(var);
		}
		ast_config_destroy(peerlist);
	}
	return var;
}

/* Yes.. the only column that makes sense to pass is "ipaddr", but for
 * consistency's sake, we require the column name to be passed. As extra
 * argument, we take a pointer to var. We already got the info, so we better
 * return it and save the caller a query. If return value is nonzero, then *var
 * is nonzero too (and the other way around). */
static struct ast_variable *get_insecure_variable_from_sipregs(const char *column, const char *value, struct ast_variable **var)
{
	struct ast_variable *varregs = NULL;
	struct ast_config *regs, *peers;
	char *regscat;
	const char *regname;

	if (!(regs = ast_load_realtime_multientry("sipregs", column, value, SENTINEL))) {
		return NULL;
	}

	/* Load *all* peers that are probably insecure=port */
	if (!(peers = ast_load_realtime_multientry("sippeers", "insecure LIKE", "%port%", SENTINEL))) {
		ast_config_destroy(regs);
		return NULL;
	}

	/* Loop over the sipregs that match IP address and attempt to find an
	 * insecure=port match to it in sippeers. */
	regscat = NULL;
	while ((regscat = ast_category_browse(regs, regscat)) && (regname = ast_variable_retrieve(regs, regscat, "name"))) {
		char *peerscat;
		const char *peername;

		peerscat = NULL;
		while ((peerscat = ast_category_browse(peers, peerscat)) && (peername = ast_variable_retrieve(peers, peerscat, "name"))) {
			if (!strcasecmp(regname, peername)) {
				/* Ensure that it really is insecure=port and
				 * not something else. */
				const char *insecure = ast_variable_retrieve(peers, peerscat, "insecure");
				struct ast_flags flags = {0};
				set_insecure_flags(&flags, insecure, -1);
				if (ast_test_flag(&flags, SIP_INSECURE_PORT)) {
					/* ENOMEM checks till the bitter end. */
					if ((varregs = ast_variables_dup(ast_category_root(regs, regscat)))) {
						if (!(*var = ast_variables_dup(ast_category_root(peers, peerscat)))) {
							ast_variables_destroy(varregs);
							varregs = NULL;
						}
					}
					goto done;
				}
			}
		}
	}

done:
	ast_config_destroy(regs);
	ast_config_destroy(peers);
	return varregs;
}

static const char *get_name_from_variable(const struct ast_variable *var)
{
	/* Don't expect this to return non-NULL. Both NULL and empty
	 * values can cause the option to get removed from the variable
	 * list. This is called on ast_variables gotten from both
	 * ast_load_realtime and ast_load_realtime_multientry.
	 * - ast_load_realtime removes options with empty values
	 * - ast_load_realtime_multientry does not!
	 * For consistent behaviour, we check for the empty name and
	 * return NULL instead. */
	const struct ast_variable *tmp;
	for (tmp = var; tmp; tmp = tmp->next) {
		if (!strcasecmp(tmp->name, "name")) {
			if (!ast_strlen_zero(tmp->value)) {
				return tmp->value;
			}
			break;
		}
	}
	return NULL;
}

/* If varregs is NULL, we don't use sipregs.
 * Using empty if-bodies instead of goto's while avoiding unnecessary indents */
static int realtime_peer_by_name(const char *const *name, struct ast_sockaddr *addr, const char *ipaddr, struct ast_variable **var, struct ast_variable **varregs)
{
	/* Peer by name and host=dynamic */
	if ((*var = ast_load_realtime("sippeers", "name", *name, "host", "dynamic", SENTINEL))) {
		;
	/* Peer by name and host=IP */
	} else if (addr && !(*var = ast_load_realtime("sippeers", "name", *name, "host", ipaddr, SENTINEL))) {
		;
	/* Peer by name and host=HOSTNAME */
	} else if ((*var = ast_load_realtime("sippeers", "name", *name, SENTINEL))) {
		/*!\note
		 * If this one loaded something, then we need to ensure that the host
		 * field matched.  The only reason why we can't have this as a criteria
		 * is because we only have the IP address and the host field might be
		 * set as a name (and the reverse PTR might not match).
		 */
		if (addr) {
			struct ast_variable *tmp;
			for (tmp = *var; tmp; tmp = tmp->next) {
				if (!strcasecmp(tmp->name, "host")) {
					struct ast_sockaddr *addrs = NULL;

					if (ast_sockaddr_resolve(&addrs,
								 tmp->value,
								 PARSE_PORT_FORBID,
								 get_address_family_filter(&bindaddr)) <= 0 ||
								 ast_sockaddr_cmp(&addrs[0], addr)) {
						/* No match */
						ast_variables_destroy(*var);
						*var = NULL;
					}
					ast_free(addrs);
					break;
				}
			}
		}
	}

	/* Did we find anything? */
	if (*var) {
		if (varregs) {
			*varregs = ast_load_realtime("sipregs", "name", *name, SENTINEL);
		}
		return 1;
	}
	return 0;
}

/* Another little helper function for backwards compatibility: this
 * checks/fetches the sippeer that belongs to the sipreg. If none is
 * found, we free the sipreg and return false. This way we can do the
 * check inside the if-condition below. In the old code, not finding
 * the sippeer also had it continue look for another match, so we do
 * the same. */
static struct ast_variable *realtime_peer_get_sippeer_helper(const char **name, struct ast_variable **varregs) {
	struct ast_variable *var = NULL;
	const char *old_name = *name;
	*name = get_name_from_variable(*varregs);
	if (!*name || !(var = ast_load_realtime("sippeers", "name", *name, SENTINEL))) {
		if (!*name) {
			ast_log(LOG_WARNING, "Found sipreg but it has no name\n");
		}
		ast_variables_destroy(*varregs);
		*varregs = NULL;
		*name = old_name;
	}
	return var;
}

/* If varregs is NULL, we don't use sipregs. If we return true, then *name is
 * set. Using empty if-bodies instead of goto's while avoiding unnecessary
 * indents. */
static int realtime_peer_by_addr(const char **name, struct ast_sockaddr *addr, const char *ipaddr, struct ast_variable **var, struct ast_variable **varregs)
{
	char portstring[6]; /* up to 5 digits plus null terminator */
	ast_copy_string(portstring, ast_sockaddr_stringify_port(addr), sizeof(portstring));

	/* We're not finding this peer by this name anymore. Reset it. */
	*name = NULL;

	/* First check for fixed IP hosts */
	if ((*var = ast_load_realtime("sippeers", "host", ipaddr, "port", portstring, SENTINEL))) {
		;
	/* Check for registered hosts (in sipregs) */
	} else if (varregs && (*varregs = ast_load_realtime("sipregs", "ipaddr", ipaddr, "port", portstring, SENTINEL)) &&
			(*var = realtime_peer_get_sippeer_helper(name, varregs))) {
		;
	/* Check for registered hosts (in sippeers) */
	} else if (!varregs && (*var = ast_load_realtime("sippeers", "ipaddr", ipaddr, "port", portstring, SENTINEL))) {
		;
	/* We couldn't match on ipaddress and port, so we need to check if port is insecure */
	} else if ((*var = get_insecure_variable_from_sippeers("host", ipaddr))) {
		;
	/* Same as above, but try the IP address field (in sipregs)
	 * Observe that it fetches the name/var at the same time, without the
	 * realtime_peer_get_sippeer_helper. Also note that it is quite inefficient.
	 * Avoid sipregs if possible. */
	} else if (varregs && (*varregs = get_insecure_variable_from_sipregs("ipaddr", ipaddr, var))) {
		;
	/* Same as above, but try the IP address field (in sippeers) */
	} else if (!varregs && (*var = get_insecure_variable_from_sippeers("ipaddr", ipaddr))) {
		;
	}

	/* Nothing found? */
	if (!*var) {
		return 0;
	}

	/* Check peer name. It must not be empty. There may exist a
	 * different match that does have a name, but it's too late for
	 * that now. */
	if (!*name && !(*name = get_name_from_variable(*var))) {
		ast_log(LOG_WARNING, "Found peer for IP %s but it has no name\n", ipaddr);
		ast_variables_destroy(*var);
		*var = NULL;
		if (varregs && *varregs) {
			ast_variables_destroy(*varregs);
			*varregs = NULL;
		}
		return 0;
	}

	/* Make sure varregs is populated if var is. The inverse,
	 * ensuring that var is set when varregs is, is taken
	 * care of by realtime_peer_get_sippeer_helper(). */
	if (varregs && !*varregs) {
		*varregs = ast_load_realtime("sipregs", "name", *name, SENTINEL);
	}
	return 1;
}

/*! \brief  realtime_peer: Get peer from realtime storage
 * Checks the "sippeers" realtime family from extconfig.conf
 * Checks the "sipregs" realtime family from extconfig.conf if it's configured.
 * This returns a pointer to a peer and because we use build_peer, we can rest
 * assured that the refcount is bumped.
 * 
 * \note This is never called with both newpeername and addr at the same time.
 * If you do, be prepared to get a peer with a different name than newpeername.
 */
static struct sip_peer *realtime_peer(const char *newpeername, struct ast_sockaddr *addr, int devstate_only, int which_objects)
{
	struct sip_peer *peer = NULL;
	struct ast_variable *var = NULL;
	struct ast_variable *varregs = NULL;
	char ipaddr[INET6_ADDRSTRLEN];
	int realtimeregs = ast_check_realtime("sipregs");

	if (addr) {
		ast_copy_string(ipaddr, ast_sockaddr_stringify_addr(addr), sizeof(ipaddr));
	} else {
		ipaddr[0] = '\0';
	}

	if (newpeername && realtime_peer_by_name(&newpeername, addr, ipaddr, &var, realtimeregs ? &varregs : NULL)) {
		;
	} else if (addr && realtime_peer_by_addr(&newpeername, addr, ipaddr, &var, realtimeregs ? &varregs : NULL)) {
		;
	} else {
		return NULL;
	}

	/* If we're looking for users, don't return peers (although this check
	 * should probably be done in realtime_peer_by_* instead...) */
	if (which_objects == FINDUSERS) {
		struct ast_variable *tmp;
		for (tmp = var; tmp; tmp = tmp->next) {
			if (!strcasecmp(tmp->name, "type") && (!strcasecmp(tmp->value, "peer"))) {
				goto cleanup;
			}
		}
	}

	/* Peer found in realtime, now build it in memory */
	peer = build_peer(newpeername, var, varregs, TRUE, devstate_only);
	if (!peer) {
		goto cleanup;
	}

	ast_debug(3, "-REALTIME- loading peer from database to memory. Name: %s. Peer objects: %d\n", peer->name, rpeerobjs);

	if (ast_test_flag(&global_flags[1], SIP_PAGE2_RTCACHEFRIENDS) && !devstate_only) {
		/* Cache peer */
		ast_copy_flags(&peer->flags[1], &global_flags[1], SIP_PAGE2_RTAUTOCLEAR|SIP_PAGE2_RTCACHEFRIENDS);
		if (ast_test_flag(&global_flags[1], SIP_PAGE2_RTAUTOCLEAR)) {
			AST_SCHED_REPLACE_UNREF(peer->expire, sched, sip_cfg.rtautoclear * 1000, expire_register, peer,
					unref_peer(_data, "remove registration ref"),
					unref_peer(peer, "remove registration ref"),
					ref_peer(peer, "add registration ref"));
		}
		ao2_t_link(peers, peer, "link peer into peers table");
		if (!ast_sockaddr_isnull(&peer->addr)) {
			ao2_t_link(peers_by_ip, peer, "link peer into peers_by_ip table");
		}
	}
	peer->is_realtime = 1;

cleanup:
	ast_variables_destroy(var);
	ast_variables_destroy(varregs);
	return peer;
}

/* Function to assist finding peers by name only */
static int find_by_name(void *obj, void *arg, void *data, int flags)
{
	struct sip_peer *search = obj, *match = arg;
	int *which_objects = data;

	/* Usernames in SIP uri's are case sensitive. Domains are not */
	if (strcmp(search->name, match->name)) {
		return 0;
	}

	switch (*which_objects) {
	case FINDUSERS:
		if (!(search->type & SIP_TYPE_USER)) {
			return 0;
		}
		break;
	case FINDPEERS:
		if (!(search->type & SIP_TYPE_PEER)) {
			return 0;
		}
		break;
	case FINDALLDEVICES:
		break;
	}

	return CMP_MATCH | CMP_STOP;
}

/*!
 * \brief Locate device by name or ip address
 * \param peer, sin, realtime, devstate_only, transport
 * \param which_objects Define which objects should be matched when doing a lookup
 *        by name.  Valid options are FINDUSERS, FINDPEERS, or FINDALLDEVICES.
 *        Note that this option is not used at all when doing a lookup by IP.
 *
 *	This is used on find matching device on name or ip/port.
 * If the device was declared as type=peer, we don't match on peer name on incoming INVITEs.
 *
 * \note Avoid using this function in new functions if there is a way to avoid it,
 * since it might cause a database lookup.
 */
static struct sip_peer *find_peer(const char *peer, struct ast_sockaddr *addr, int realtime, int which_objects, int devstate_only, int transport)
{
	struct sip_peer *p = NULL;
	struct sip_peer tmp_peer;

	if (peer) {
		ast_copy_string(tmp_peer.name, peer, sizeof(tmp_peer.name));
		p = ao2_t_callback_data(peers, OBJ_POINTER, find_by_name, &tmp_peer, &which_objects, "ao2_find in peers table");
	} else if (addr) { /* search by addr? */
		ast_sockaddr_copy(&tmp_peer.addr, addr);
		tmp_peer.flags[0].flags = 0;
		tmp_peer.transports = transport;
		p = ao2_t_find(peers_by_ip, &tmp_peer, OBJ_POINTER, "ao2_find in peers_by_ip table"); /* WAS:  p = ASTOBJ_CONTAINER_FIND_FULL(&peerl, sin, name, sip_addr_hashfunc, 1, sip_addrcmp); */
		if (!p) {
			ast_set_flag(&tmp_peer.flags[0], SIP_INSECURE_PORT);
			p = ao2_t_find(peers_by_ip, &tmp_peer, OBJ_POINTER, "ao2_find in peers_by_ip table 2"); /* WAS:  p = ASTOBJ_CONTAINER_FIND_FULL(&peerl, sin, name, sip_addr_hashfunc, 1, sip_addrcmp); */
			if (p) {
				return p;
			}
		}
	}

	if (!p && (realtime || devstate_only)) {
		p = realtime_peer(peer, addr, devstate_only, which_objects);
		if (p) {
			switch (which_objects) {
			case FINDUSERS:
				if (!(p->type & SIP_TYPE_USER)) {
					unref_peer(p, "Wrong type of realtime SIP endpoint");
					return NULL;
				}
				break;
			case FINDPEERS:
				if (!(p->type & SIP_TYPE_PEER)) {
					unref_peer(p, "Wrong type of realtime SIP endpoint");
					return NULL;
				}
				break;
			case FINDALLDEVICES:
				break;
			}
		}
	}

	return p;
}

/*! \brief Set nat mode on the various data sockets */
static void do_setnat(struct sip_pvt *p)
{
	const char *mode;
	int natflags;

	natflags = ast_test_flag(&p->flags[1], SIP_PAGE2_SYMMETRICRTP);
	mode = natflags ? "On" : "Off";

	if (p->rtp) {
		ast_debug(1, "Setting NAT on RTP to %s\n", mode);
		ast_rtp_instance_set_prop(p->rtp, AST_RTP_PROPERTY_NAT, natflags);
	}
	if (p->vrtp) {
		ast_debug(1, "Setting NAT on VRTP to %s\n", mode);
		ast_rtp_instance_set_prop(p->vrtp, AST_RTP_PROPERTY_NAT, natflags);
	}
	if (p->udptl) {
		ast_debug(1, "Setting NAT on UDPTL to %s\n", mode);
		ast_udptl_setnat(p->udptl, natflags);
	}
	if (p->trtp) {
		ast_debug(1, "Setting NAT on TRTP to %s\n", mode);
		ast_rtp_instance_set_prop(p->trtp, AST_RTP_PROPERTY_NAT, natflags);
	}
}

/*! \brief Change the T38 state on a SIP dialog */
static void change_t38_state(struct sip_pvt *p, int state)
{
	int old = p->t38.state;
	struct ast_channel *chan = p->owner;
	struct ast_control_t38_parameters parameters = { .request_response = 0 };

	/* Don't bother changing if we are already in the state wanted */
	if (old == state)
		return;

	p->t38.state = state;
	ast_debug(2, "T38 state changed to %d on channel %s\n", p->t38.state, chan ? chan->name : "<none>");

	/* If no channel was provided we can't send off a control frame */
	if (!chan)
		return;

	/* Given the state requested and old state determine what control frame we want to queue up */
	switch (state) {
	case T38_PEER_REINVITE:
		parameters = p->t38.their_parms;
		parameters.max_ifp = ast_udptl_get_far_max_ifp(p->udptl);
		parameters.request_response = AST_T38_REQUEST_NEGOTIATE;
		ast_udptl_set_tag(p->udptl, "%s", chan->name);
		break;
	case T38_ENABLED:
		parameters = p->t38.their_parms;
		parameters.max_ifp = ast_udptl_get_far_max_ifp(p->udptl);
		parameters.request_response = AST_T38_NEGOTIATED;
		ast_udptl_set_tag(p->udptl, "%s", chan->name);
		break;
	case T38_DISABLED:
		if (old == T38_ENABLED) {
			parameters.request_response = AST_T38_TERMINATED;
		} else if (old == T38_LOCAL_REINVITE) {
			parameters.request_response = AST_T38_REFUSED;
		}
		break;
	case T38_LOCAL_REINVITE:
		/* wait until we get a peer response before responding to local reinvite */
		break;
	}

	/* Woot we got a message, create a control frame and send it on! */
	if (parameters.request_response)
		ast_queue_control_data(chan, AST_CONTROL_T38_PARAMETERS, &parameters, sizeof(parameters));
}

/*! \brief Set the global T38 capabilities on a SIP dialog structure */
static void set_t38_capabilities(struct sip_pvt *p)
{
	if (p->udptl) {
		if (ast_test_flag(&p->flags[1], SIP_PAGE2_T38SUPPORT) == SIP_PAGE2_T38SUPPORT_UDPTL_REDUNDANCY) {
                        ast_udptl_set_error_correction_scheme(p->udptl, UDPTL_ERROR_CORRECTION_REDUNDANCY);
		} else if (ast_test_flag(&p->flags[1], SIP_PAGE2_T38SUPPORT) == SIP_PAGE2_T38SUPPORT_UDPTL_FEC) {
			ast_udptl_set_error_correction_scheme(p->udptl, UDPTL_ERROR_CORRECTION_FEC);
		} else if (ast_test_flag(&p->flags[1], SIP_PAGE2_T38SUPPORT) == SIP_PAGE2_T38SUPPORT_UDPTL) {
			ast_udptl_set_error_correction_scheme(p->udptl, UDPTL_ERROR_CORRECTION_NONE);
		}
	}
}

static void copy_socket_data(struct sip_socket *to_sock, const struct sip_socket *from_sock)
{
	if (to_sock->tcptls_session) {
		ao2_ref(to_sock->tcptls_session, -1);
		to_sock->tcptls_session = NULL;
	}

	if (from_sock->tcptls_session) {
		ao2_ref(from_sock->tcptls_session, +1);
	}

	*to_sock = *from_sock;
}

/*! \brief Initialize RTP portion of a dialog
 * \return -1 on failure, 0 on success
 */
static int dialog_initialize_rtp(struct sip_pvt *dialog)
{
	struct ast_sockaddr bindaddr_tmp;

	if (!sip_methods[dialog->method].need_rtp) {
		return 0;
	}

	ast_sockaddr_copy(&bindaddr_tmp, &bindaddr);
	if (!(dialog->rtp = ast_rtp_instance_new(dialog->engine, sched, &bindaddr_tmp, NULL))) {
		return -1;
	}

	if (ast_test_flag(&dialog->flags[1], SIP_PAGE2_VIDEOSUPPORT_ALWAYS) ||
			(ast_test_flag(&dialog->flags[1], SIP_PAGE2_VIDEOSUPPORT) && (dialog->capability & AST_FORMAT_VIDEO_MASK))) {
		if (!(dialog->vrtp = ast_rtp_instance_new(dialog->engine, sched, &bindaddr_tmp, NULL))) {
			return -1;
		}
		ast_rtp_instance_set_timeout(dialog->vrtp, dialog->rtptimeout);
		ast_rtp_instance_set_hold_timeout(dialog->vrtp, dialog->rtpholdtimeout);
		ast_rtp_instance_set_keepalive(dialog->vrtp, dialog->rtpkeepalive);

		ast_rtp_instance_set_prop(dialog->vrtp, AST_RTP_PROPERTY_RTCP, 1);
	}

	if (ast_test_flag(&dialog->flags[1], SIP_PAGE2_TEXTSUPPORT)) {
		if (!(dialog->trtp = ast_rtp_instance_new(dialog->engine, sched, &bindaddr_tmp, NULL))) {
			return -1;
		}
		/* Do not timeout text as its not constant*/
		ast_rtp_instance_set_keepalive(dialog->trtp, dialog->rtpkeepalive);

		ast_rtp_instance_set_prop(dialog->trtp, AST_RTP_PROPERTY_RTCP, 1);
	}

	ast_rtp_instance_set_timeout(dialog->rtp, dialog->rtptimeout);
	ast_rtp_instance_set_hold_timeout(dialog->rtp, dialog->rtpholdtimeout);
	ast_rtp_instance_set_keepalive(dialog->rtp, dialog->rtpkeepalive);

	ast_rtp_instance_set_prop(dialog->rtp, AST_RTP_PROPERTY_RTCP, 1);
	ast_rtp_instance_set_prop(dialog->rtp, AST_RTP_PROPERTY_DTMF, ast_test_flag(&dialog->flags[0], SIP_DTMF) == SIP_DTMF_RFC2833);
	ast_rtp_instance_set_prop(dialog->rtp, AST_RTP_PROPERTY_DTMF_COMPENSATE, ast_test_flag(&dialog->flags[1], SIP_PAGE2_RFC2833_COMPENSATE));

	ast_rtp_instance_set_qos(dialog->rtp, global_tos_audio, global_cos_audio, "SIP RTP");

	do_setnat(dialog);

	return 0;
}

/*! \brief Create address structure from peer reference.
 *	This function copies data from peer to the dialog, so we don't have to look up the peer
 *	again from memory or database during the life time of the dialog.
 *
 * \return -1 on error, 0 on success.
 *
 */
static int create_addr_from_peer(struct sip_pvt *dialog, struct sip_peer *peer)
{
	struct sip_auth_container *credentials;

	/* this checks that the dialog is contacting the peer on a valid
	 * transport type based on the peers transport configuration,
	 * otherwise, this function bails out */
	if (dialog->socket.type && check_request_transport(peer, dialog))
		return -1;
	copy_socket_data(&dialog->socket, &peer->socket);

	if (!(ast_sockaddr_isnull(&peer->addr) && ast_sockaddr_isnull(&peer->defaddr)) &&
	    (!peer->maxms || ((peer->lastms >= 0)  && (peer->lastms <= peer->maxms)))) {
		dialog->sa = ast_sockaddr_isnull(&peer->addr) ? peer->defaddr : peer->addr;
		dialog->recv = dialog->sa;
	} else
		return -1;

	/* XXX TODO: get flags directly from peer only as they are needed using dialog->relatedpeer */
	ast_copy_flags(&dialog->flags[0], &peer->flags[0], SIP_FLAGS_TO_COPY);
	ast_copy_flags(&dialog->flags[1], &peer->flags[1], SIP_PAGE2_FLAGS_TO_COPY);
	ast_copy_flags(&dialog->flags[2], &peer->flags[2], SIP_PAGE3_FLAGS_TO_COPY);
	dialog->capability = peer->capability;
	dialog->prefs = peer->prefs;
	dialog->amaflags = peer->amaflags;

	ast_string_field_set(dialog, engine, peer->engine);

	dialog->rtptimeout = peer->rtptimeout;
	dialog->rtpholdtimeout = peer->rtpholdtimeout;
	dialog->rtpkeepalive = peer->rtpkeepalive;
	if (dialog_initialize_rtp(dialog)) {
		return -1;
	}

	if (dialog->rtp) { /* Audio */
		ast_rtp_instance_set_prop(dialog->rtp, AST_RTP_PROPERTY_DTMF, ast_test_flag(&dialog->flags[0], SIP_DTMF) == SIP_DTMF_RFC2833);
		ast_rtp_instance_set_prop(dialog->rtp, AST_RTP_PROPERTY_DTMF_COMPENSATE, ast_test_flag(&dialog->flags[1], SIP_PAGE2_RFC2833_COMPENSATE));
		/* Set Frame packetization */
		ast_rtp_codecs_packetization_set(ast_rtp_instance_get_codecs(dialog->rtp), dialog->rtp, &dialog->prefs);
		dialog->autoframing = peer->autoframing;
	}

	/* XXX TODO: get fields directly from peer only as they are needed using dialog->relatedpeer */
	ast_string_field_set(dialog, peername, peer->name);
	ast_string_field_set(dialog, authname, peer->username);
	ast_string_field_set(dialog, username, peer->username);
	ast_string_field_set(dialog, peersecret, peer->secret);
	ast_string_field_set(dialog, peermd5secret, peer->md5secret);
	ast_string_field_set(dialog, mohsuggest, peer->mohsuggest);
	ast_string_field_set(dialog, mohinterpret, peer->mohinterpret);
	ast_string_field_set(dialog, tohost, peer->tohost);
	ast_string_field_set(dialog, fullcontact, peer->fullcontact);
	ast_string_field_set(dialog, accountcode, peer->accountcode);
	ast_string_field_set(dialog, context, peer->context);
	ast_string_field_set(dialog, cid_num, peer->cid_num);
	ast_string_field_set(dialog, cid_name, peer->cid_name);
	ast_string_field_set(dialog, cid_tag, peer->cid_tag);
	ast_string_field_set(dialog, mwi_from, peer->mwi_from);
	if (!ast_strlen_zero(peer->parkinglot)) {
		ast_string_field_set(dialog, parkinglot, peer->parkinglot);
	}
	ast_string_field_set(dialog, engine, peer->engine);
	ref_proxy(dialog, obproxy_get(dialog, peer));
	dialog->callgroup = peer->callgroup;
	dialog->pickupgroup = peer->pickupgroup;
	dialog->allowtransfer = peer->allowtransfer;
	dialog->jointnoncodeccapability = dialog->noncodeccapability;

	/* Update dialog authorization credentials */
	ao2_lock(peer);
	credentials = peer->auth;
	if (credentials) {
		ao2_t_ref(credentials, +1, "Ref peer auth for dialog");
	}
	ao2_unlock(peer);
	ao2_lock(dialog);
	if (dialog->peerauth) {
		ao2_t_ref(dialog->peerauth, -1, "Unref old dialog peer auth");
	}
	dialog->peerauth = credentials;
	ao2_unlock(dialog);

	dialog->maxcallbitrate = peer->maxcallbitrate;
	dialog->disallowed_methods = peer->disallowed_methods;
	ast_cc_copy_config_params(dialog->cc_params, peer->cc_params);
	if (ast_strlen_zero(dialog->tohost))
		ast_string_field_set(dialog, tohost, ast_sockaddr_stringify_host_remote(&dialog->sa));
	if (!ast_strlen_zero(peer->fromdomain)) {
		ast_string_field_set(dialog, fromdomain, peer->fromdomain);
		if (!dialog->initreq.headers) {
			char *new_callid;
			char *tmpcall = ast_strdupa(dialog->callid);
			/* this sure looks to me like we are going to change the callid on this dialog!! */
			new_callid = strchr(tmpcall, '@');
			if (new_callid) {
				int callid_size;

				*new_callid = '\0';

				/* Change the dialog callid. */
				callid_size = strlen(tmpcall) + strlen(peer->fromdomain) + 2;
				new_callid = alloca(callid_size);
				snprintf(new_callid, callid_size, "%s@%s", tmpcall, peer->fromdomain);
				change_callid_pvt(dialog, new_callid);
			}
		}
	}
	if (!ast_strlen_zero(peer->fromuser))
		ast_string_field_set(dialog, fromuser, peer->fromuser);
	if (!ast_strlen_zero(peer->language))
		ast_string_field_set(dialog, language, peer->language);
	/* Set timer T1 to RTT for this peer (if known by qualify=) */
	/* Minimum is settable or default to 100 ms */
	/* If there is a maxms and lastms from a qualify use that over a manual T1
	   value. Otherwise, use the peer's T1 value. */
	if (peer->maxms && peer->lastms)
		dialog->timer_t1 = peer->lastms < global_t1min ? global_t1min : peer->lastms;
	else
		dialog->timer_t1 = peer->timer_t1;

	/* Set timer B to control transaction timeouts, the peer setting is the default and overrides
	   the known timer */
	if (peer->timer_b)
		dialog->timer_b = peer->timer_b;
	else
		dialog->timer_b = 64 * dialog->timer_t1;

	if ((ast_test_flag(&dialog->flags[0], SIP_DTMF) == SIP_DTMF_RFC2833) ||
	    (ast_test_flag(&dialog->flags[0], SIP_DTMF) == SIP_DTMF_AUTO))
		dialog->noncodeccapability |= AST_RTP_DTMF;
	else
		dialog->noncodeccapability &= ~AST_RTP_DTMF;
	dialog->directmediaha = ast_duplicate_ha_list(peer->directmediaha);
	if (peer->call_limit)
		ast_set_flag(&dialog->flags[0], SIP_CALL_LIMIT);
	if (!dialog->portinuri)
		dialog->portinuri = peer->portinuri;
	dialog->chanvars = copy_vars(peer->chanvars);
	if (peer->fromdomainport)
		dialog->fromdomainport = peer->fromdomainport;

	return 0;
}

/*! \brief The default sip port for the given transport */
static inline int default_sip_port(enum sip_transport type)
{
	return type == SIP_TRANSPORT_TLS ? STANDARD_TLS_PORT : STANDARD_SIP_PORT;
}

/*! \brief create address structure from device name
 *      Or, if peer not found, find it in the global DNS
 *      returns TRUE (-1) on failure, FALSE on success */
static int create_addr(struct sip_pvt *dialog, const char *opeer, struct ast_sockaddr *addr, int newdialog, struct ast_sockaddr *remote_address)
{
	struct sip_peer *peer;
	char *peername, *peername2, *hostn;
	char host[MAXHOSTNAMELEN];
	char service[MAXHOSTNAMELEN];
	int srv_ret = 0;
	int tportno;

	AST_DECLARE_APP_ARGS(hostport,
		AST_APP_ARG(host);
		AST_APP_ARG(port);
	);

	peername = ast_strdupa(opeer);
	peername2 = ast_strdupa(opeer);
	AST_NONSTANDARD_RAW_ARGS(hostport, peername2, ':');

	if (hostport.port)
		dialog->portinuri = 1;

	dialog->timer_t1 = global_t1; /* Default SIP retransmission timer T1 (RFC 3261) */
	dialog->timer_b = global_timer_b; /* Default SIP transaction timer B (RFC 3261) */
	peer = find_peer(peername, NULL, TRUE, FINDPEERS, FALSE, 0);

	if (peer) {
		int res;
		if (newdialog) {
			set_socket_transport(&dialog->socket, 0);
		}
		res = create_addr_from_peer(dialog, peer);
		if (!ast_sockaddr_isnull(remote_address)) {
			ast_sockaddr_copy(&dialog->sa, remote_address);
		}
		dialog->relatedpeer = ref_peer(peer, "create_addr: setting dialog's relatedpeer pointer");
		unref_peer(peer, "create_addr: unref peer from find_peer hashtab lookup");
		return res;
	} else if (ast_check_digits(peername)) {
		/* Although an IPv4 hostname *could* be represented as a 32-bit integer, it is uncommon and
		 * it makes dialing SIP/${EXTEN} for a peer that isn't defined resolve to an IP that is
		 * almost certainly not intended. It is much better to just reject purely numeric hostnames */
		ast_log(LOG_WARNING, "Purely numeric hostname (%s), and not a peer--rejecting!\n", peername);
		return -1;
	} else {
		dialog->rtptimeout = global_rtptimeout;
		dialog->rtpholdtimeout = global_rtpholdtimeout;
		dialog->rtpkeepalive = global_rtpkeepalive;
		if (dialog_initialize_rtp(dialog)) {
			return -1;
		}
	}

	ast_string_field_set(dialog, tohost, hostport.host);
	dialog->allowed_methods &= ~sip_cfg.disallowed_methods;

	/* Get the outbound proxy information */
	ref_proxy(dialog, obproxy_get(dialog, NULL));

	if (addr) {
		/* This address should be updated using dnsmgr */
		ast_sockaddr_copy(&dialog->sa, addr);
	} else {

		/* Let's see if we can find the host in DNS. First try DNS SRV records,
		   then hostname lookup */
		/*! \todo Fix this function. When we ask for SRV, we should check all transports
			  In the future, we should first check NAPTR to find out transport preference
		 */
		hostn = peername;
 		/* Section 4.2 of RFC 3263 specifies that if a port number is specified, then
		 * an A record lookup should be used instead of SRV.
		 */
		if (!hostport.port && sip_cfg.srvlookup) {
			snprintf(service, sizeof(service), "_%s._%s.%s", 
				 get_srv_service(dialog->socket.type),
				 get_srv_protocol(dialog->socket.type), peername);
			if ((srv_ret = ast_get_srv(NULL, host, sizeof(host), &tportno,
						   service)) > 0) {
				hostn = host;
			}
		}

		if (ast_sockaddr_resolve_first(&dialog->sa, hostn, 0)) {
			ast_log(LOG_WARNING, "No such host: %s\n", peername);
			return -1;
		}

		if (srv_ret > 0) {
			ast_sockaddr_set_port(&dialog->sa, tportno);
		}
	}

	if (!dialog->socket.type)
		set_socket_transport(&dialog->socket, SIP_TRANSPORT_UDP);
	if (!dialog->socket.port) {
		dialog->socket.port = htons(ast_sockaddr_port(&bindaddr));
	}

	if (!ast_sockaddr_port(&dialog->sa)) {
		ast_sockaddr_set_port(&dialog->sa, default_sip_port(dialog->socket.type));
	}
	ast_sockaddr_copy(&dialog->recv, &dialog->sa);
	return 0;
}

/*! \brief Scheduled congestion on a call.
 * Only called by the scheduler, must return the reference when done.
 */
static int auto_congest(const void *arg)
{
	struct sip_pvt *p = (struct sip_pvt *)arg;

	sip_pvt_lock(p);
	p->initid = -1;	/* event gone, will not be rescheduled */
	if (p->owner) {
		/* XXX fails on possible deadlock */
		if (!ast_channel_trylock(p->owner)) {
			append_history(p, "Cong", "Auto-congesting (timer)");
			ast_queue_control(p->owner, AST_CONTROL_CONGESTION);
			ast_channel_unlock(p->owner);
		}

		/* Give the channel a chance to act before we proceed with destruction */
		sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
	}
	sip_pvt_unlock(p);
	dialog_unref(p, "unreffing arg passed into auto_congest callback (p->initid)");
	return 0;
}


/*! \brief Initiate SIP call from PBX
 *      used from the dial() application      */
static int sip_call(struct ast_channel *ast, char *dest, int timeout)
{
	int res;
	struct sip_pvt *p = ast->tech_pvt;	/* chan is locked, so the reference cannot go away */
	struct varshead *headp;
	struct ast_var_t *current;
	const char *referer = NULL;   /* SIP referrer */
	int cc_core_id;
	char uri[SIPBUFSIZE] = "";

	if ((ast->_state != AST_STATE_DOWN) && (ast->_state != AST_STATE_RESERVED)) {
		ast_log(LOG_WARNING, "sip_call called on %s, neither down nor reserved\n", ast->name);
		return -1;
	}

	if (ast_cc_is_recall(ast, &cc_core_id, "SIP")) {
		char device_name[AST_CHANNEL_NAME];
		struct ast_cc_monitor *recall_monitor;
		struct sip_monitor_instance *monitor_instance;
		ast_channel_get_device_name(ast, device_name, sizeof(device_name));
		if ((recall_monitor = ast_cc_get_monitor_by_recall_core_id(cc_core_id, device_name))) {
			monitor_instance = recall_monitor->private_data;
			ast_copy_string(uri, monitor_instance->notify_uri, sizeof(uri));
			ao2_t_ref(recall_monitor, -1, "Got the URI we need so unreffing monitor");
		}
	}

	/* Check whether there is vxml_url, distinctive ring variables */
	headp=&ast->varshead;
	AST_LIST_TRAVERSE(headp, current, entries) {
		/* Check whether there is a VXML_URL variable */
		if (!p->options->vxml_url && !strcasecmp(ast_var_name(current), "VXML_URL")) {
			p->options->vxml_url = ast_var_value(current);
		} else if (!p->options->uri_options && !strcasecmp(ast_var_name(current), "SIP_URI_OPTIONS")) {
			p->options->uri_options = ast_var_value(current);
		} else if (!p->options->addsipheaders && !strncasecmp(ast_var_name(current), "SIPADDHEADER", strlen("SIPADDHEADER"))) {
			/* Check whether there is a variable with a name starting with SIPADDHEADER */
			p->options->addsipheaders = 1;
		} else if (!strcasecmp(ast_var_name(current), "SIPFROMDOMAIN")) {
			ast_string_field_set(p, fromdomain, ast_var_value(current));
		} else if (!strcasecmp(ast_var_name(current), "SIPTRANSFER")) {
			/* This is a transfered call */
			p->options->transfer = 1;
		} else if (!strcasecmp(ast_var_name(current), "SIPTRANSFER_REFERER")) {
			/* This is the referrer */
			referer = ast_var_value(current);
		} else if (!strcasecmp(ast_var_name(current), "SIPTRANSFER_REPLACES")) {
			/* We're replacing a call. */
			p->options->replaces = ast_var_value(current);
		} else if (!strcasecmp(ast_var_name(current), "SIP_MAX_FORWARDS")) {
			if (sscanf(ast_var_value(current), "%d", &(p->maxforwards)) != 1) {
				ast_log(LOG_WARNING, "The SIP_MAX_FORWARDS channel variable is not a valid integer.\n");
			}
		}
	}

	/* Check to see if we should try to force encryption */
	if (p->req_secure_signaling && p->socket.type != SIP_TRANSPORT_TLS) {
	   ast_log(LOG_WARNING, "Encrypted signaling is required\n");
	   ast->hangupcause = AST_CAUSE_BEARERCAPABILITY_NOTAVAIL;
	   return -1;
	}

	if (ast_test_flag(&p->flags[1], SIP_PAGE2_USE_SRTP)) {
		if (ast_test_flag(&p->flags[0], SIP_REINVITE)) {
			ast_debug(1, "Direct media not possible when using SRTP, ignoring canreinvite setting\n");
			ast_clear_flag(&p->flags[0], SIP_REINVITE);
		}

		if (p->rtp && !p->srtp && setup_srtp(&p->srtp) < 0) {
			ast_log(LOG_WARNING, "SRTP audio setup failed\n");
			return -1;
		}

		if (p->vrtp && !p->vsrtp && setup_srtp(&p->vsrtp) < 0) {
			ast_log(LOG_WARNING, "SRTP video setup failed\n");
			return -1;
		}

		if (p->trtp && !p->tsrtp && setup_srtp(&p->tsrtp) < 0) {
			ast_log(LOG_WARNING, "SRTP text setup failed\n");
			return -1;
		}
	}

	res = 0;
	ast_set_flag(&p->flags[0], SIP_OUTGOING);

	/* T.38 re-INVITE FAX detection should never be done for outgoing calls,
	 * so ensure it is disabled.
	 */
	ast_clear_flag(&p->flags[1], SIP_PAGE2_FAX_DETECT_T38);

	if (p->options->transfer) {
		char buf[SIPBUFSIZE/2];

		if (referer) {
			if (sipdebug)
				ast_debug(3, "Call for %s transfered by %s\n", p->username, referer);
			snprintf(buf, sizeof(buf)-1, "-> %s (via %s)", p->cid_name, referer);
		} else
			snprintf(buf, sizeof(buf)-1, "-> %s", p->cid_name);
		ast_string_field_set(p, cid_name, buf);
	}
	ast_debug(1, "Outgoing Call for %s\n", p->username);

	res = update_call_counter(p, INC_CALL_RINGING);

	if (res == -1) {
		ast->hangupcause = AST_CAUSE_USER_BUSY;
		return res;
	}
	p->callingpres = ast_party_id_presentation(&ast->caller.id);
	p->jointcapability = ast_rtp_instance_available_formats(p->rtp, p->capability, p->prefcodec);
	p->jointnoncodeccapability = p->noncodeccapability;

	/* If there are no audio formats left to offer, punt */
	if (!(p->jointcapability & AST_FORMAT_AUDIO_MASK)) {
		ast_log(LOG_WARNING, "No audio format found to offer. Cancelling call to %s\n", p->username);
		res = -1;
	} else {
		int xmitres;
		struct ast_party_connected_line connected;
		struct ast_set_party_connected_line update_connected;

		sip_pvt_lock(p);

		/* Supply initial connected line information if available. */
		memset(&update_connected, 0, sizeof(update_connected));
		ast_party_connected_line_init(&connected);
		if (!ast_strlen_zero(p->cid_num)
			|| (p->callingpres & AST_PRES_RESTRICTION) != AST_PRES_ALLOWED) {
			update_connected.id.number = 1;
			connected.id.number.valid = 1;
			connected.id.number.str = (char *) p->cid_num;
			connected.id.number.presentation = p->callingpres;
		}
		if (!ast_strlen_zero(p->cid_name)
			|| (p->callingpres & AST_PRES_RESTRICTION) != AST_PRES_ALLOWED) {
			update_connected.id.name = 1;
			connected.id.name.valid = 1;
			connected.id.name.str = (char *) p->cid_name;
			connected.id.name.presentation = p->callingpres;
		}
		if (update_connected.id.number || update_connected.id.name) {
			connected.id.tag = (char *) p->cid_tag;
			connected.source = AST_CONNECTED_LINE_UPDATE_SOURCE_ANSWER;
			ast_channel_queue_connected_line_update(ast, &connected, &update_connected);
		}

		xmitres = transmit_invite(p, SIP_INVITE, 1, 2, uri);
		sip_pvt_unlock(p);
		if (xmitres == XMIT_ERROR)
			return -1;
		p->invitestate = INV_CALLING;

		/* Initialize auto-congest time */
		AST_SCHED_REPLACE_UNREF(p->initid, sched, p->timer_b, auto_congest, p,
								dialog_unref(_data, "dialog ptr dec when SCHED_REPLACE del op succeeded"),
								dialog_unref(p, "dialog ptr dec when SCHED_REPLACE add failed"),
								dialog_ref(p, "dialog ptr inc when SCHED_REPLACE add succeeded") );
	}
	return res;
}

/*! \brief Destroy registry object
	Objects created with the register= statement in static configuration */
static void sip_registry_destroy(struct sip_registry *reg)
{
	/* Really delete */
	ast_debug(3, "Destroying registry entry for %s@%s\n", reg->username, reg->hostname);

	if (reg->call) {
		/* Clear registry before destroying to ensure
		   we don't get reentered trying to grab the registry lock */
		reg->call->registry = registry_unref(reg->call->registry, "destroy reg->call->registry");
		ast_debug(3, "Destroying active SIP dialog for registry %s@%s\n", reg->username, reg->hostname);
		dialog_unlink_all(reg->call);
		reg->call = dialog_unref(reg->call, "unref reg->call");
		/* reg->call = sip_destroy(reg->call); */
	}
	AST_SCHED_DEL(sched, reg->expire);
	AST_SCHED_DEL(sched, reg->timeout);

	ast_string_field_free_memory(reg);
	ast_atomic_fetchadd_int(&regobjs, -1);
	ast_free(reg);
}

/*! \brief Destroy MWI subscription object */
static void sip_subscribe_mwi_destroy(struct sip_subscription_mwi *mwi)
{
	if (mwi->call) {
		mwi->call->mwi = NULL;
		sip_destroy(mwi->call);
	}
	
	AST_SCHED_DEL(sched, mwi->resub);
	ast_string_field_free_memory(mwi);
	ast_free(mwi);
}

/*! \brief Execute destruction of SIP dialog structure, release memory */
void __sip_destroy(struct sip_pvt *p, int lockowner, int lockdialoglist)
{
	struct sip_request *req;

	/* Destroy Session-Timers if allocated */
 	if (p->stimer) {
		p->stimer->quit_flag = 1;
		stop_session_timer(p);
		ast_free(p->stimer);
		p->stimer = NULL;
	}

	if (sip_debug_test_pvt(p))
		ast_verbose("Really destroying SIP dialog '%s' Method: %s\n", p->callid, sip_methods[p->method].text);

	if (ast_test_flag(&p->flags[0], SIP_INC_COUNT) || ast_test_flag(&p->flags[1], SIP_PAGE2_CALL_ONHOLD)) {
		update_call_counter(p, DEC_CALL_LIMIT);
		ast_debug(2, "This call did not properly clean up call limits. Call ID %s\n", p->callid);
	}

	/* Unlink us from the owner if we have one */
	if (p->owner) {
		if (lockowner)
			ast_channel_lock(p->owner);
		ast_debug(1, "Detaching from %s\n", p->owner->name);
		p->owner->tech_pvt = NULL;
		/* Make sure that the channel knows its backend is going away */
		p->owner->_softhangup |= AST_SOFTHANGUP_DEV;
		if (lockowner)
			ast_channel_unlock(p->owner);
		/* Give the channel a chance to react before deallocation */
		usleep(1);
	}

	/* Remove link from peer to subscription of MWI */
	if (p->relatedpeer && p->relatedpeer->mwipvt)
		p->relatedpeer->mwipvt = dialog_unref(p->relatedpeer->mwipvt, "delete ->relatedpeer->mwipvt");
	if (p->relatedpeer && p->relatedpeer->call == p)
		p->relatedpeer->call = dialog_unref(p->relatedpeer->call, "unset the relatedpeer->call field in tandem with relatedpeer field itself");
	
	if (p->relatedpeer)
		p->relatedpeer = unref_peer(p->relatedpeer,"unsetting a dialog relatedpeer field in sip_destroy");
	
	if (p->registry) {
		if (p->registry->call == p)
			p->registry->call = dialog_unref(p->registry->call, "nulling out the registry's call dialog field in unlink_all");
		p->registry = registry_unref(p->registry, "delete p->registry");
	}
	
	if (p->mwi) {
		p->mwi->call = NULL;
	}

	if (dumphistory)
		sip_dump_history(p);

	if (p->options)
		ast_free(p->options);

	if (p->notify) {
		ast_variables_destroy(p->notify->headers);
		ast_free(p->notify->content);
		ast_free(p->notify);
	}
	if (p->rtp) {
		ast_rtp_instance_destroy(p->rtp);
	}
	if (p->vrtp) {
		ast_rtp_instance_destroy(p->vrtp);
	}
	if (p->trtp) {
		ast_rtp_instance_destroy(p->trtp);
	}
	if (p->udptl)
		ast_udptl_destroy(p->udptl);
	if (p->refer) {
		if (p->refer->refer_call) {
			p->refer->refer_call = dialog_unref(p->refer->refer_call, "unref dialog p->refer->refer_call");
		}
		ast_free(p->refer);
	}
	if (p->route) {
		free_old_route(p->route);
		p->route = NULL;
	}
	deinit_req(&p->initreq);

	/* Clear history */
	if (p->history) {
		struct sip_history *hist;
		while ( (hist = AST_LIST_REMOVE_HEAD(p->history, list)) ) {
			ast_free(hist);
			p->history_entries--;
		}
		ast_free(p->history);
		p->history = NULL;
	}

	while ((req = AST_LIST_REMOVE_HEAD(&p->request_queue, next))) {
		ast_free(req);
	}

	if (p->chanvars) {
		ast_variables_destroy(p->chanvars);
		p->chanvars = NULL;
	}

	if (p->srtp) {
		sip_srtp_destroy(p->srtp);
		p->srtp = NULL;
	}

	if (p->vsrtp) {
		sip_srtp_destroy(p->vsrtp);
		p->vsrtp = NULL;
	}

	if (p->tsrtp) {
		sip_srtp_destroy(p->tsrtp);
		p->tsrtp = NULL;
	}

	if (p->directmediaha) {
		ast_free_ha(p->directmediaha);
		p->directmediaha = NULL;
	}

	ast_string_field_free_memory(p);

	ast_cc_config_params_destroy(p->cc_params);

	if (p->epa_entry) {
		ao2_ref(p->epa_entry, -1);
		p->epa_entry = NULL;
	}

	if (p->socket.tcptls_session) {
		ao2_ref(p->socket.tcptls_session, -1);
		p->socket.tcptls_session = NULL;
	}

	if (p->peerauth) {
		ao2_t_ref(p->peerauth, -1, "Removing active peer authentication");
		p->peerauth = NULL;
	}
}

/*! \brief  update_call_counter: Handle call_limit for SIP devices
 * Setting a call-limit will cause calls above the limit not to be accepted.
 *
 * Remember that for a type=friend, there's one limit for the user and
 * another for the peer, not a combined call limit.
 * This will cause unexpected behaviour in subscriptions, since a "friend"
 * is *two* devices in Asterisk, not one.
 *
 * Thought: For realtime, we should probably update storage with inuse counter...
 *
 * \return 0 if call is ok (no call limit, below threshold)
 *	-1 on rejection of call
 *
 */
static int update_call_counter(struct sip_pvt *fup, int event)
{
	char name[256];
	int *inuse = NULL, *call_limit = NULL, *inringing = NULL;
	int outgoing = fup->outgoing_call;
	struct sip_peer *p = NULL;

	ast_debug(3, "Updating call counter for %s call\n", outgoing ? "outgoing" : "incoming");


	/* Test if we need to check call limits, in order to avoid
	   realtime lookups if we do not need it */
	if (!ast_test_flag(&fup->flags[0], SIP_CALL_LIMIT) && !ast_test_flag(&fup->flags[1], SIP_PAGE2_CALL_ONHOLD))
		return 0;

	ast_copy_string(name, fup->username, sizeof(name));

	/* Check the list of devices */
	if (fup->relatedpeer) {
		p = ref_peer(fup->relatedpeer, "ref related peer for update_call_counter");
		inuse = &p->inUse;
		call_limit = &p->call_limit;
		inringing = &p->inRinging;
		ast_copy_string(name, fup->peername, sizeof(name));
	}
	if (!p) {
		ast_debug(2, "%s is not a local device, no call limit\n", name);
		return 0;
	}

	switch(event) {
	/* incoming and outgoing affects the inUse counter */
	case DEC_CALL_LIMIT:
		/* Decrement inuse count if applicable */
		if (inuse) {
			sip_pvt_lock(fup);
			ao2_lock(p);
			if (*inuse > 0) {
				if (ast_test_flag(&fup->flags[0], SIP_INC_COUNT)) {
					(*inuse)--;
					ast_clear_flag(&fup->flags[0], SIP_INC_COUNT);
				}
			} else {
				*inuse = 0;
			}
			ao2_unlock(p);
			sip_pvt_unlock(fup);
		}

		/* Decrement ringing count if applicable */
		if (inringing) {
			sip_pvt_lock(fup);
			ao2_lock(p);
			if (*inringing > 0) {
				if (ast_test_flag(&fup->flags[0], SIP_INC_RINGING)) {
					(*inringing)--;
					ast_clear_flag(&fup->flags[0], SIP_INC_RINGING);
				}
			} else {
			   *inringing = 0;
			}
			ao2_unlock(p);
			sip_pvt_unlock(fup);
		}

		/* Decrement onhold count if applicable */
		sip_pvt_lock(fup);
		ao2_lock(p);
		if (ast_test_flag(&fup->flags[1], SIP_PAGE2_CALL_ONHOLD) && sip_cfg.notifyhold) {
			ast_clear_flag(&fup->flags[1], SIP_PAGE2_CALL_ONHOLD);
			ao2_unlock(p);
			sip_pvt_unlock(fup);
			sip_peer_hold(fup, FALSE);
		} else {
			ao2_unlock(p);
			sip_pvt_unlock(fup);
		}
		if (sipdebug)
			ast_debug(2, "Call %s %s '%s' removed from call limit %d\n", outgoing ? "to" : "from", "peer", name, *call_limit);
		break;

	case INC_CALL_RINGING:
	case INC_CALL_LIMIT:
		/* If call limit is active and we have reached the limit, reject the call */
		if (*call_limit > 0 ) {
			if (*inuse >= *call_limit) {
				ast_log(LOG_NOTICE, "Call %s %s '%s' rejected due to usage limit of %d\n", outgoing ? "to" : "from", "peer", name, *call_limit);
				unref_peer(p, "update_call_counter: unref peer p, call limit exceeded");
				return -1;
			}
		}
		if (inringing && (event == INC_CALL_RINGING)) {
			sip_pvt_lock(fup);
			ao2_lock(p);
			if (!ast_test_flag(&fup->flags[0], SIP_INC_RINGING)) {
				(*inringing)++;
				ast_set_flag(&fup->flags[0], SIP_INC_RINGING);
			}
			ao2_unlock(p);
			sip_pvt_unlock(fup);
		}
		if (inuse) {
			sip_pvt_lock(fup);
			ao2_lock(p);
			if (!ast_test_flag(&fup->flags[0], SIP_INC_COUNT)) {
				(*inuse)++;
				ast_set_flag(&fup->flags[0], SIP_INC_COUNT);
			}
			ao2_unlock(p);
			sip_pvt_unlock(fup);
		}
		if (sipdebug) {
			ast_debug(2, "Call %s %s '%s' is %d out of %d\n", outgoing ? "to" : "from", "peer", name, *inuse, *call_limit);
		}
		break;

	case DEC_CALL_RINGING:
		if (inringing) {
			sip_pvt_lock(fup);
			ao2_lock(p);
			if (ast_test_flag(&fup->flags[0], SIP_INC_RINGING)) {
				if (*inringing > 0) {
					(*inringing)--;
				}
				ast_clear_flag(&fup->flags[0], SIP_INC_RINGING);
			}
			ao2_unlock(p);
			sip_pvt_unlock(fup);
		}
		break;

	default:
		ast_log(LOG_ERROR, "update_call_counter(%s, %d) called with no event!\n", name, event);
	}

	if (p) {
		ast_devstate_changed(AST_DEVICE_UNKNOWN, "SIP/%s", p->name);
		unref_peer(p, "update_call_counter: unref_peer from call counter");
	}
	return 0;
}


static void sip_destroy_fn(void *p)
{
	sip_destroy(p);
}

/*! \brief Destroy SIP call structure.
 * Make it return NULL so the caller can do things like
 *	foo = sip_destroy(foo);
 * and reduce the chance of bugs due to dangling pointers.
 */
struct sip_pvt *sip_destroy(struct sip_pvt *p)
{
	ast_debug(3, "Destroying SIP dialog %s\n", p->callid);
	__sip_destroy(p, TRUE, TRUE);
	return NULL;
}

/*! \brief Convert SIP hangup causes to Asterisk hangup causes */
int hangup_sip2cause(int cause)
{
	/* Possible values taken from causes.h */

	switch(cause) {
		case 401:	/* Unauthorized */
			return AST_CAUSE_CALL_REJECTED;
		case 403:	/* Not found */
			return AST_CAUSE_CALL_REJECTED;
		case 404:	/* Not found */
			return AST_CAUSE_UNALLOCATED;
		case 405:	/* Method not allowed */
			return AST_CAUSE_INTERWORKING;
		case 407:	/* Proxy authentication required */
			return AST_CAUSE_CALL_REJECTED;
		case 408:	/* No reaction */
			return AST_CAUSE_NO_USER_RESPONSE;
		case 409:	/* Conflict */
			return AST_CAUSE_NORMAL_TEMPORARY_FAILURE;
		case 410:	/* Gone */
			return AST_CAUSE_NUMBER_CHANGED;
		case 411:	/* Length required */
			return AST_CAUSE_INTERWORKING;
		case 413:	/* Request entity too large */
			return AST_CAUSE_INTERWORKING;
		case 414:	/* Request URI too large */
			return AST_CAUSE_INTERWORKING;
		case 415:	/* Unsupported media type */
			return AST_CAUSE_INTERWORKING;
		case 420:	/* Bad extension */
			return AST_CAUSE_NO_ROUTE_DESTINATION;
		case 480:	/* No answer */
			return AST_CAUSE_NO_ANSWER;
		case 481:	/* No answer */
			return AST_CAUSE_INTERWORKING;
		case 482:	/* Loop detected */
			return AST_CAUSE_INTERWORKING;
		case 483:	/* Too many hops */
			return AST_CAUSE_NO_ANSWER;
		case 484:	/* Address incomplete */
			return AST_CAUSE_INVALID_NUMBER_FORMAT;
		case 485:	/* Ambiguous */
			return AST_CAUSE_UNALLOCATED;
		case 486:	/* Busy everywhere */
			return AST_CAUSE_BUSY;
		case 487:	/* Request terminated */
			return AST_CAUSE_INTERWORKING;
		case 488:	/* No codecs approved */
			return AST_CAUSE_BEARERCAPABILITY_NOTAVAIL;
		case 491:	/* Request pending */
			return AST_CAUSE_INTERWORKING;
		case 493:	/* Undecipherable */
			return AST_CAUSE_INTERWORKING;
		case 500:	/* Server internal failure */
			return AST_CAUSE_FAILURE;
		case 501:	/* Call rejected */
			return AST_CAUSE_FACILITY_REJECTED;
		case 502:
			return AST_CAUSE_DESTINATION_OUT_OF_ORDER;
		case 503:	/* Service unavailable */
			return AST_CAUSE_CONGESTION;
		case 504:	/* Gateway timeout */
			return AST_CAUSE_RECOVERY_ON_TIMER_EXPIRE;
		case 505:	/* SIP version not supported */
			return AST_CAUSE_INTERWORKING;
		case 600:	/* Busy everywhere */
			return AST_CAUSE_USER_BUSY;
		case 603:	/* Decline */
			return AST_CAUSE_CALL_REJECTED;
		case 604:	/* Does not exist anywhere */
			return AST_CAUSE_UNALLOCATED;
		case 606:	/* Not acceptable */
			return AST_CAUSE_BEARERCAPABILITY_NOTAVAIL;
		default:
			if (cause < 500 && cause >= 400) {
				/* 4xx class error that is unknown - someting wrong with our request */
				return AST_CAUSE_INTERWORKING;
			} else if (cause < 600 && cause >= 500) {
				/* 5xx class error - problem in the remote end */
				return AST_CAUSE_CONGESTION;
			} else if (cause < 700 && cause >= 600) {
				/* 6xx - global errors in the 4xx class */
				return AST_CAUSE_INTERWORKING;
			}
			return AST_CAUSE_NORMAL;
	}
	/* Never reached */
	return 0;
}

/*! \brief Convert Asterisk hangup causes to SIP codes
\verbatim
 Possible values from causes.h
        AST_CAUSE_NOTDEFINED    AST_CAUSE_NORMAL        AST_CAUSE_BUSY
        AST_CAUSE_FAILURE       AST_CAUSE_CONGESTION    AST_CAUSE_UNALLOCATED

	In addition to these, a lot of PRI codes is defined in causes.h
	...should we take care of them too ?

	Quote RFC 3398

   ISUP Cause value                        SIP response
   ----------------                        ------------
   1  unallocated number                   404 Not Found
   2  no route to network                  404 Not found
   3  no route to destination              404 Not found
   16 normal call clearing                 --- (*)
   17 user busy                            486 Busy here
   18 no user responding                   408 Request Timeout
   19 no answer from the user              480 Temporarily unavailable
   20 subscriber absent                    480 Temporarily unavailable
   21 call rejected                        403 Forbidden (+)
   22 number changed (w/o diagnostic)      410 Gone
   22 number changed (w/ diagnostic)       301 Moved Permanently
   23 redirection to new destination       410 Gone
   26 non-selected user clearing           404 Not Found (=)
   27 destination out of order             502 Bad Gateway
   28 address incomplete                   484 Address incomplete
   29 facility rejected                    501 Not implemented
   31 normal unspecified                   480 Temporarily unavailable
\endverbatim
*/
const char *hangup_cause2sip(int cause)
{
	switch (cause) {
		case AST_CAUSE_UNALLOCATED:		/* 1 */
		case AST_CAUSE_NO_ROUTE_DESTINATION:	/* 3 IAX2: Can't find extension in context */
		case AST_CAUSE_NO_ROUTE_TRANSIT_NET:	/* 2 */
			return "404 Not Found";
		case AST_CAUSE_CONGESTION:		/* 34 */
		case AST_CAUSE_SWITCH_CONGESTION:	/* 42 */
			return "503 Service Unavailable";
		case AST_CAUSE_NO_USER_RESPONSE:	/* 18 */
			return "408 Request Timeout";
		case AST_CAUSE_NO_ANSWER:		/* 19 */
		case AST_CAUSE_UNREGISTERED:        /* 20 */
			return "480 Temporarily unavailable";
		case AST_CAUSE_CALL_REJECTED:		/* 21 */
			return "403 Forbidden";
		case AST_CAUSE_NUMBER_CHANGED:		/* 22 */
			return "410 Gone";
		case AST_CAUSE_NORMAL_UNSPECIFIED:	/* 31 */
			return "480 Temporarily unavailable";
		case AST_CAUSE_INVALID_NUMBER_FORMAT:
			return "484 Address incomplete";
		case AST_CAUSE_USER_BUSY:
			return "486 Busy here";
		case AST_CAUSE_FAILURE:
			return "500 Server internal failure";
		case AST_CAUSE_FACILITY_REJECTED:	/* 29 */
			return "501 Not Implemented";
		case AST_CAUSE_CHAN_NOT_IMPLEMENTED:
			return "503 Service Unavailable";
		/* Used in chan_iax2 */
		case AST_CAUSE_DESTINATION_OUT_OF_ORDER:
			return "502 Bad Gateway";
		case AST_CAUSE_BEARERCAPABILITY_NOTAVAIL:	/* Can't find codec to connect to host */
			return "488 Not Acceptable Here";
			
		case AST_CAUSE_NOTDEFINED:
		default:
			ast_debug(1, "AST hangup cause %d (no match found in SIP)\n", cause);
			return NULL;
	}

	/* Never reached */
	return 0;
}

static int reinvite_timeout(const void *data)
{
	struct sip_pvt *dialog = (struct sip_pvt *) data;
	struct ast_channel *owner = sip_pvt_lock_full(dialog);
	dialog->reinviteid = -1;
	check_pendings(dialog);
	if (owner) {
		ast_channel_unlock(owner);
		ast_channel_unref(owner);
	}
	ao2_unlock(dialog);
	dialog_unref(dialog, "unref for reinvite timeout");
	return 0;
}

/*! \brief  sip_hangup: Hangup SIP call
 * Part of PBX interface, called from ast_hangup */
static int sip_hangup(struct ast_channel *ast)
{
	struct sip_pvt *p = ast->tech_pvt;
	int needcancel = FALSE;
	int needdestroy = 0;
	struct ast_channel *oldowner = ast;

	if (!p) {
		ast_debug(1, "Asked to hangup channel that was not connected\n");
		return 0;
	}
	if (ast_test_flag(ast, AST_FLAG_ANSWERED_ELSEWHERE) || ast->hangupcause == AST_CAUSE_ANSWERED_ELSEWHERE) {
		ast_debug(1, "This call was answered elsewhere\n");
		if (ast->hangupcause == AST_CAUSE_ANSWERED_ELSEWHERE) {
			ast_debug(1, "####### It's the cause code, buddy. The cause code!!!\n");
		}
		append_history(p, "Cancel", "Call answered elsewhere");
		p->answered_elsewhere = TRUE;
	}

	/* Store hangupcause locally in PVT so we still have it before disconnect */
	if (p->owner)
		p->hangupcause = p->owner->hangupcause;

	if (ast_test_flag(&p->flags[0], SIP_DEFER_BYE_ON_TRANSFER)) {
		if (ast_test_flag(&p->flags[0], SIP_INC_COUNT) || ast_test_flag(&p->flags[1], SIP_PAGE2_CALL_ONHOLD)) {
			if (sipdebug)
				ast_debug(1, "update_call_counter(%s) - decrement call limit counter on hangup\n", p->username);
			update_call_counter(p, DEC_CALL_LIMIT);
		}
		ast_debug(4, "SIP Transfer: Not hanging up right now... Rescheduling hangup for %s.\n", p->callid);
		sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
		ast_clear_flag(&p->flags[0], SIP_DEFER_BYE_ON_TRANSFER);	/* Really hang up next time */
		p->needdestroy = 0;
		p->owner->tech_pvt = dialog_unref(p->owner->tech_pvt, "unref p->owner->tech_pvt");
		sip_pvt_lock(p);
		p->owner = NULL;  /* Owner will be gone after we return, so take it away */
		sip_pvt_unlock(p);
		ast_module_unref(ast_module_info->self);
		return 0;
	}

	if (ast_test_flag(ast, AST_FLAG_ZOMBIE)) {
		if (p->refer)
			ast_debug(1, "SIP Transfer: Hanging up Zombie channel %s after transfer ... Call-ID: %s\n", ast->name, p->callid);
		else
			ast_debug(1, "Hanging up zombie call. Be scared.\n");
	} else
		ast_debug(1, "Hangup call %s, SIP callid %s\n", ast->name, p->callid);

	sip_pvt_lock(p);
	if (ast_test_flag(&p->flags[0], SIP_INC_COUNT) || ast_test_flag(&p->flags[1], SIP_PAGE2_CALL_ONHOLD)) {
		if (sipdebug)
			ast_debug(1, "update_call_counter(%s) - decrement call limit counter on hangup\n", p->username);
		update_call_counter(p, DEC_CALL_LIMIT);
	}

	/* Determine how to disconnect */
	if (p->owner != ast) {
		ast_log(LOG_WARNING, "Huh?  We aren't the owner? Can't hangup call.\n");
		sip_pvt_unlock(p);
		return 0;
	}
	/* If the call is not UP, we need to send CANCEL instead of BYE */
	/* In case of re-invites, the call might be UP even though we have an incomplete invite transaction */
	if (p->invitestate < INV_COMPLETED && p->owner->_state != AST_STATE_UP) {
		needcancel = TRUE;
		ast_debug(4, "Hanging up channel in state %s (not UP)\n", ast_state2str(ast->_state));
	}

	stop_media_flows(p); /* Immediately stop RTP, VRTP and UDPTL as applicable */

	append_history(p, needcancel ? "Cancel" : "Hangup", "Cause %s", p->owner ? ast_cause2str(p->hangupcause) : "Unknown");

	/* Disconnect */
	disable_dsp_detect(p);

	p->owner = NULL;
	ast->tech_pvt = dialog_unref(ast->tech_pvt, "unref ast->tech_pvt");

	ast_module_unref(ast_module_info->self);
	/* Do not destroy this pvt until we have timeout or
	   get an answer to the BYE or INVITE/CANCEL
	   If we get no answer during retransmit period, drop the call anyway.
	   (Sorry, mother-in-law, you can't deny a hangup by sending
	   603 declined to BYE...)
	*/
	if (p->alreadygone)
		needdestroy = 1;	/* Set destroy flag at end of this function */
	else if (p->invitestate != INV_CALLING)
		sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);

	/* Start the process if it's not already started */
	if (!p->alreadygone && p->initreq.data && !ast_strlen_zero(p->initreq.data->str)) {
		if (needcancel) {	/* Outgoing call, not up */
			if (ast_test_flag(&p->flags[0], SIP_OUTGOING)) {
				/* if we can't send right now, mark it pending */
				if (p->invitestate == INV_CALLING) {
					/* We can't send anything in CALLING state */
					ast_set_flag(&p->flags[0], SIP_PENDINGBYE);
					/* Do we need a timer here if we don't hear from them at all? Yes we do or else we will get hung dialogs and those are no fun. */
					sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
					append_history(p, "DELAY", "Not sending cancel, waiting for timeout");
				} else {
					struct sip_pkt *cur;

					for (cur = p->packets; cur; cur = cur->next) {
						__sip_semi_ack(p, cur->seqno, cur->is_resp, cur->method ? cur->method : find_sip_method(cur->data->str));
					}
					p->invitestate = INV_CANCELLED;
					/* Send a new request: CANCEL */
					transmit_request(p, SIP_CANCEL, p->lastinvite, XMIT_RELIABLE, FALSE);
					/* Actually don't destroy us yet, wait for the 487 on our original
					   INVITE, but do set an autodestruct just in case we never get it. */
					needdestroy = 0;
					sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
				}
			} else {	/* Incoming call, not up */
				const char *res;
				AST_SCHED_DEL_UNREF(sched, p->provisional_keepalive_sched_id, dialog_unref(p, "when you delete the provisional_keepalive_sched_id, you should dec the refcount for the stored dialog ptr"));
				if (p->hangupcause && (res = hangup_cause2sip(p->hangupcause)))
					transmit_response_reliable(p, res, &p->initreq);
				else
					transmit_response_reliable(p, "603 Declined", &p->initreq);
				p->invitestate = INV_TERMINATED;
			}
		} else {	/* Call is in UP state, send BYE */
			if (p->stimer->st_active == TRUE) {
				stop_session_timer(p);
			}

			if (!p->pendinginvite) {
				struct ast_channel *bridge = ast_bridged_channel(oldowner);
				char quality_buf[AST_MAX_USER_FIELD], *quality;

				/* We need to get the lock on bridge because ast_rtp_instance_set_stats_vars will attempt
				 * to lock the bridge. This may get hairy...
				 */
				while (bridge && ast_channel_trylock(bridge)) {
					sip_pvt_unlock(p);
					do {
						CHANNEL_DEADLOCK_AVOIDANCE(oldowner);
					} while (sip_pvt_trylock(p));
					bridge = ast_bridged_channel(oldowner);
				}

				if (p->rtp) {
					ast_rtp_instance_set_stats_vars(oldowner, p->rtp);
				}

				if (bridge) {
					struct sip_pvt *q = bridge->tech_pvt;

					if (IS_SIP_TECH(bridge->tech) && q && q->rtp) {
						ast_rtp_instance_set_stats_vars(bridge, q->rtp);
					}
					ast_channel_unlock(bridge);
				}

				if (p->do_history || oldowner) {
					if (p->rtp && (quality = ast_rtp_instance_get_quality(p->rtp, AST_RTP_INSTANCE_STAT_FIELD_QUALITY, quality_buf, sizeof(quality_buf)))) {
						if (p->do_history) {
							append_history(p, "RTCPaudio", "Quality:%s", quality);
						}
						if (oldowner) {
							pbx_builtin_setvar_helper(oldowner, "RTPAUDIOQOS", quality);
						}
					}
					if (p->vrtp && (quality = ast_rtp_instance_get_quality(p->vrtp, AST_RTP_INSTANCE_STAT_FIELD_QUALITY, quality_buf, sizeof(quality_buf)))) {
						if (p->do_history) {
							append_history(p, "RTCPvideo", "Quality:%s", quality);
						}
						if (oldowner) {
							pbx_builtin_setvar_helper(oldowner, "RTPVIDEOQOS", quality);
						}
					}
					if (p->trtp && (quality = ast_rtp_instance_get_quality(p->trtp, AST_RTP_INSTANCE_STAT_FIELD_QUALITY, quality_buf, sizeof(quality_buf)))) {
						if (p->do_history) {
							append_history(p, "RTCPtext", "Quality:%s", quality);
						}
						if (oldowner) {
							pbx_builtin_setvar_helper(oldowner, "RTPTEXTQOS", quality);
						}
					}
				}

				/* Send a hangup */
				if (oldowner->_state == AST_STATE_UP) {
					transmit_request_with_auth(p, SIP_BYE, 0, XMIT_RELIABLE, 1);
				}

			} else {
				/* Note we will need a BYE when this all settles out
				   but we can't send one while we have "INVITE" outstanding. */
				ast_set_flag(&p->flags[0], SIP_PENDINGBYE);	
				ast_clear_flag(&p->flags[0], SIP_NEEDREINVITE);	
				AST_SCHED_DEL_UNREF(sched, p->waitid, dialog_unref(p, "when you delete the waitid sched, you should dec the refcount for the stored dialog ptr"));
				if (sip_cancel_destroy(p)) {
					ast_log(LOG_WARNING, "Unable to cancel SIP destruction.  Expect bad things.\n");
				}
				/* If we have an ongoing reinvite, there is a chance that we have gotten a provisional
				 * response, but something weird has happened and we will never receive a final response.
				 * So, just in case, check for pending actions after a bit of time to trigger the pending
				 * bye that we are setting above */
				if (p->ongoing_reinvite && p->reinviteid < 0) {
					p->reinviteid = ast_sched_add(sched, 32 * p->timer_t1, reinvite_timeout, dialog_ref(p, "ref for reinvite_timeout"));
				}
			}
		}
	}
	if (needdestroy) {
		pvt_set_needdestroy(p, "hangup");
	}
	sip_pvt_unlock(p);
	return 0;
}

/*! \brief Try setting codec suggested by the SIP_CODEC channel variable */
static void try_suggested_sip_codec(struct sip_pvt *p)
{
	format_t fmt;
	const char *codec;

	if (p->outgoing_call) {
		codec = pbx_builtin_getvar_helper(p->owner, "SIP_CODEC_OUTBOUND");
	} else if (!(codec = pbx_builtin_getvar_helper(p->owner, "SIP_CODEC_INBOUND"))) {
		codec = pbx_builtin_getvar_helper(p->owner, "SIP_CODEC");
	}

	if (!codec) 
		return;

	fmt = ast_getformatbyname(codec);
	if (fmt) {
		ast_log(LOG_NOTICE, "Changing codec to '%s' for this call because of ${SIP_CODEC} variable\n", codec);
		if (p->jointcapability & fmt) {
			p->jointcapability &= fmt;
			p->capability &= fmt;
		} else
			ast_log(LOG_NOTICE, "Ignoring ${SIP_CODEC} variable because it is not shared by both ends.\n");
	} else
		ast_log(LOG_NOTICE, "Ignoring ${SIP_CODEC} variable because of unrecognized/not configured codec (check allow/disallow in sip.conf): %s\n", codec);
	return;	
}

/*! \brief  sip_answer: Answer SIP call , send 200 OK on Invite
 * Part of PBX interface */
static int sip_answer(struct ast_channel *ast)
{
	int res = 0;
	struct sip_pvt *p = ast->tech_pvt;

	sip_pvt_lock(p);
	if (ast->_state != AST_STATE_UP) {
		try_suggested_sip_codec(p);	

		ast_setstate(ast, AST_STATE_UP);
		ast_debug(1, "SIP answering channel: %s\n", ast->name);
		ast_rtp_instance_update_source(p->rtp);
		res = transmit_response_with_sdp(p, "200 OK", &p->initreq, XMIT_CRITICAL, FALSE, TRUE);
		ast_set_flag(&p->flags[1], SIP_PAGE2_DIALOG_ESTABLISHED);
	}
	sip_pvt_unlock(p);
	return res;
}

/*! \brief Send frame to media channel (rtp) */
static int sip_write(struct ast_channel *ast, struct ast_frame *frame)
{
	struct sip_pvt *p = ast->tech_pvt;
	int res = 0;

	switch (frame->frametype) {
	case AST_FRAME_VOICE:
		if (!(frame->subclass.codec & ast->nativeformats)) {
			char s1[512], s2[512], s3[512];
			ast_log(LOG_WARNING, "Asked to transmit frame type %s, while native formats is %s read/write = %s/%s\n",
				ast_getformatname(frame->subclass.codec),
				ast_getformatname_multiple(s1, sizeof(s1), ast->nativeformats & AST_FORMAT_AUDIO_MASK),
				ast_getformatname_multiple(s2, sizeof(s2), ast->readformat),
				ast_getformatname_multiple(s3, sizeof(s3), ast->writeformat));
			return 0;
		}
		if (p) {
			sip_pvt_lock(p);
			if (p->t38.state == T38_ENABLED) {
				/* drop frame, can't sent VOICE frames while in T.38 mode */
				sip_pvt_unlock(p);
				break;
			} else if (p->rtp) {
				/* If channel is not up, activate early media session */
				if ((ast->_state != AST_STATE_UP) &&
				    !ast_test_flag(&p->flags[0], SIP_PROGRESS_SENT) &&
				    !ast_test_flag(&p->flags[0], SIP_OUTGOING)) {
					ast_rtp_instance_update_source(p->rtp);
					if (!global_prematuremediafilter) {
						p->invitestate = INV_EARLY_MEDIA;
						transmit_provisional_response(p, "183 Session Progress", &p->initreq, TRUE);
						ast_set_flag(&p->flags[0], SIP_PROGRESS_SENT);
					}
				}
				p->lastrtptx = time(NULL);
				res = ast_rtp_instance_write(p->rtp, frame);
			}
			sip_pvt_unlock(p);
		}
		break;
	case AST_FRAME_VIDEO:
		if (p) {
			sip_pvt_lock(p);
			if (p->vrtp) {
				/* Activate video early media */
				if ((ast->_state != AST_STATE_UP) &&
				    !ast_test_flag(&p->flags[0], SIP_PROGRESS_SENT) &&
				    !ast_test_flag(&p->flags[0], SIP_OUTGOING)) {
					p->invitestate = INV_EARLY_MEDIA;
					transmit_provisional_response(p, "183 Session Progress", &p->initreq, TRUE);
					ast_set_flag(&p->flags[0], SIP_PROGRESS_SENT);
				}
				p->lastrtptx = time(NULL);
				res = ast_rtp_instance_write(p->vrtp, frame);
			}
			sip_pvt_unlock(p);
		}
		break;
	case AST_FRAME_TEXT:
		if (p) {
			sip_pvt_lock(p);
			if (p->red) {
				ast_rtp_red_buffer(p->trtp, frame);
			} else {
				if (p->trtp) {
					/* Activate text early media */
					if ((ast->_state != AST_STATE_UP) &&
					    !ast_test_flag(&p->flags[0], SIP_PROGRESS_SENT) &&
					    !ast_test_flag(&p->flags[0], SIP_OUTGOING)) {
						p->invitestate = INV_EARLY_MEDIA;
						transmit_provisional_response(p, "183 Session Progress", &p->initreq, TRUE);
						ast_set_flag(&p->flags[0], SIP_PROGRESS_SENT);
					}
					p->lastrtptx = time(NULL);
					res = ast_rtp_instance_write(p->trtp, frame);
				}
			}
			sip_pvt_unlock(p);
		}
		break;
	case AST_FRAME_IMAGE:
		return 0;
		break;
	case AST_FRAME_MODEM:
		if (p) {
			sip_pvt_lock(p);
			/* UDPTL requires two-way communication, so early media is not needed here.
				we simply forget the frames if we get modem frames before the bridge is up.
				Fax will re-transmit.
			*/
			if ((ast->_state == AST_STATE_UP) &&
			    p->udptl &&
			    (p->t38.state == T38_ENABLED)) {
				res = ast_udptl_write(p->udptl, frame);
			}
			sip_pvt_unlock(p);
		}
		break;
	default:
		ast_log(LOG_WARNING, "Can't send %d type frames with SIP write\n", frame->frametype);
		return 0;
	}

	return res;
}

/*! \brief  sip_fixup: Fix up a channel:  If a channel is consumed, this is called.
        Basically update any ->owner links */
static int sip_fixup(struct ast_channel *oldchan, struct ast_channel *newchan)
{
	int ret = -1;
	struct sip_pvt *p;

	if (newchan && ast_test_flag(newchan, AST_FLAG_ZOMBIE))
		ast_debug(1, "New channel is zombie\n");
	if (oldchan && ast_test_flag(oldchan, AST_FLAG_ZOMBIE))
		ast_debug(1, "Old channel is zombie\n");

	if (!newchan || !newchan->tech_pvt) {
		if (!newchan)
			ast_log(LOG_WARNING, "No new channel! Fixup of %s failed.\n", oldchan->name);
		else
			ast_log(LOG_WARNING, "No SIP tech_pvt! Fixup of %s failed.\n", oldchan->name);
		return -1;
	}
	p = newchan->tech_pvt;

	sip_pvt_lock(p);
	append_history(p, "Masq", "Old channel: %s\n", oldchan->name);
	append_history(p, "Masq (cont)", "...new owner: %s\n", newchan->name);
	if (p->owner != oldchan)
		ast_log(LOG_WARNING, "old channel wasn't %p but was %p\n", oldchan, p->owner);
	else {
		p->owner = newchan;
		/* Re-invite RTP back to Asterisk. Needed if channel is masqueraded out of a native
		   RTP bridge (i.e., RTP not going through Asterisk): RTP bridge code might not be
		   able to do this if the masquerade happens before the bridge breaks (e.g., AMI
		   redirect of both channels). Note that a channel can not be masqueraded *into*
		   a native bridge. So there is no danger that this breaks a native bridge that
		   should stay up. */
		sip_set_rtp_peer(newchan, NULL, NULL, 0, 0, 0);
		ret = 0;
	}
	ast_debug(3, "SIP Fixup: New owner for dialogue %s: %s (Old parent: %s)\n", p->callid, p->owner->name, oldchan->name);

	sip_pvt_unlock(p);
	return ret;
}

static int sip_senddigit_begin(struct ast_channel *ast, char digit)
{
	struct sip_pvt *p = ast->tech_pvt;
	int res = 0;

	sip_pvt_lock(p);
	switch (ast_test_flag(&p->flags[0], SIP_DTMF)) {
	case SIP_DTMF_INBAND:
		res = -1; /* Tell Asterisk to generate inband indications */
		break;
	case SIP_DTMF_RFC2833:
		if (p->rtp)
			ast_rtp_instance_dtmf_begin(p->rtp, digit);
		break;
	default:
		break;
	}
	sip_pvt_unlock(p);

	return res;
}

/*! \brief Send DTMF character on SIP channel
	within one call, we're able to transmit in many methods simultaneously */
static int sip_senddigit_end(struct ast_channel *ast, char digit, unsigned int duration)
{
	struct sip_pvt *p = ast->tech_pvt;
	int res = 0;

	sip_pvt_lock(p);
	switch (ast_test_flag(&p->flags[0], SIP_DTMF)) {
	case SIP_DTMF_INFO:
	case SIP_DTMF_SHORTINFO:
		transmit_info_with_digit(p, digit, duration);
		break;
	case SIP_DTMF_RFC2833:
		if (p->rtp)
			ast_rtp_instance_dtmf_end_with_duration(p->rtp, digit, duration);
		break;
	case SIP_DTMF_INBAND:
		res = -1; /* Tell Asterisk to stop inband indications */
		break;
	}
	sip_pvt_unlock(p);

	return res;
}

/*! \brief Transfer SIP call */
static int sip_transfer(struct ast_channel *ast, const char *dest)
{
	struct sip_pvt *p = ast->tech_pvt;
	int res;

	if (dest == NULL)	/* functions below do not take a NULL */
		dest = "";
	sip_pvt_lock(p);
	if (ast->_state == AST_STATE_RING)
		res = sip_sipredirect(p, dest);
	else
		res = transmit_refer(p, dest);
	sip_pvt_unlock(p);
	return res;
}

/*! \brief Helper function which updates T.38 capability information and triggers a reinvite */
static int interpret_t38_parameters(struct sip_pvt *p, const struct ast_control_t38_parameters *parameters)
{
	int res = 0;

	if (!ast_test_flag(&p->flags[1], SIP_PAGE2_T38SUPPORT) || !p->udptl) {
		return -1;
	}
	switch (parameters->request_response) {
	case AST_T38_NEGOTIATED:
	case AST_T38_REQUEST_NEGOTIATE:         /* Request T38 */
		/* Negotiation can not take place without a valid max_ifp value. */
		if (!parameters->max_ifp) {
			change_t38_state(p, T38_DISABLED);
			if (p->t38.state == T38_PEER_REINVITE) {
				AST_SCHED_DEL_UNREF(sched, p->t38id, dialog_unref(p, "when you delete the t38id sched, you should dec the refcount for the stored dialog ptr"));
				transmit_response_reliable(p, "488 Not acceptable here", &p->initreq);
			}
			break;
		} else if (p->t38.state == T38_PEER_REINVITE) {
			AST_SCHED_DEL_UNREF(sched, p->t38id, dialog_unref(p, "when you delete the t38id sched, you should dec the refcount for the stored dialog ptr"));
			p->t38.our_parms = *parameters;
			/* modify our parameters to conform to the peer's parameters,
			 * based on the rules in the ITU T.38 recommendation
			 */
			if (!p->t38.their_parms.fill_bit_removal) {
				p->t38.our_parms.fill_bit_removal = FALSE;
			}
			if (!p->t38.their_parms.transcoding_mmr) {
				p->t38.our_parms.transcoding_mmr = FALSE;
			}
			if (!p->t38.their_parms.transcoding_jbig) {
				p->t38.our_parms.transcoding_jbig = FALSE;
			}
			p->t38.our_parms.version = MIN(p->t38.our_parms.version, p->t38.their_parms.version);
			p->t38.our_parms.rate_management = p->t38.their_parms.rate_management;
			ast_udptl_set_local_max_ifp(p->udptl, p->t38.our_parms.max_ifp);
			change_t38_state(p, T38_ENABLED);
			transmit_response_with_t38_sdp(p, "200 OK", &p->initreq, XMIT_CRITICAL);
		} else if (p->t38.state != T38_ENABLED) {
			p->t38.our_parms = *parameters;
			ast_udptl_set_local_max_ifp(p->udptl, p->t38.our_parms.max_ifp);
			change_t38_state(p, T38_LOCAL_REINVITE);
			if (!p->pendinginvite) {
				transmit_reinvite_with_sdp(p, TRUE, FALSE);
			} else if (!ast_test_flag(&p->flags[0], SIP_PENDINGBYE)) {
				ast_set_flag(&p->flags[0], SIP_NEEDREINVITE);
			}
		}
		break;
	case AST_T38_TERMINATED:
	case AST_T38_REFUSED:
	case AST_T38_REQUEST_TERMINATE:         /* Shutdown T38 */
		if (p->t38.state == T38_PEER_REINVITE) {
			AST_SCHED_DEL_UNREF(sched, p->t38id, dialog_unref(p, "when you delete the t38id sched, you should dec the refcount for the stored dialog ptr"));
			change_t38_state(p, T38_DISABLED);
			transmit_response_reliable(p, "488 Not acceptable here", &p->initreq);
		} else if (p->t38.state == T38_ENABLED)
			transmit_reinvite_with_sdp(p, FALSE, FALSE);
		break;
	case AST_T38_REQUEST_PARMS: {		/* Application wants remote's parameters re-sent */
		struct ast_control_t38_parameters parameters = p->t38.their_parms;

		if (p->t38.state == T38_PEER_REINVITE) {
			AST_SCHED_DEL_UNREF(sched, p->t38id, dialog_unref(p, "when you delete the t38id sched, you should dec the refcount for the stored dialog ptr"));
			parameters.max_ifp = ast_udptl_get_far_max_ifp(p->udptl);
			parameters.request_response = AST_T38_REQUEST_NEGOTIATE;
			ast_queue_control_data(p->owner, AST_CONTROL_T38_PARAMETERS, &parameters, sizeof(parameters));
			/* we need to return a positive value here, so that applications that
			 * send this request can determine conclusively whether it was accepted or not...
			 * older versions of chan_sip would just silently accept it and return zero.
			 */
			res = AST_T38_REQUEST_PARMS;
		}
		break;
	}
	default:
		res = -1;
		break;
	}

	return res;
}

/*! \internal \brief Create and initialize UDPTL for the specified dialog
 * \param p SIP private structure to create UDPTL object for
 * \pre p is locked
 * \pre p->owner is locked
 *
 * \note In the case of failure, SIP_PAGE2_T38SUPPORT is cleared on p
 *
 * \return 0 on success, any other value on failure
 */
static int initialize_udptl(struct sip_pvt *p)
{
	int natflags = ast_test_flag(&p->flags[1], SIP_PAGE2_SYMMETRICRTP);

	if (!ast_test_flag(&p->flags[1], SIP_PAGE2_T38SUPPORT)) {
		return 1;
	}

	/* If we've already initialized T38, don't take any further action */
	if (p->udptl) {
		return 0;
	}

	/* T38 can be supported by this dialog, create it and set the derived properties */
	if ((p->udptl = ast_udptl_new_with_bindaddr(sched, io, 0, &bindaddr))) {
		if (p->owner) {
			ast_channel_set_fd(p->owner, 5, ast_udptl_fd(p->udptl));
		}

		ast_udptl_setqos(p->udptl, global_tos_audio, global_cos_audio);
		p->t38_maxdatagram = p->relatedpeer ? p->relatedpeer->t38_maxdatagram : global_t38_maxdatagram;
		set_t38_capabilities(p);

		ast_debug(1, "Setting NAT on UDPTL to %s\n", natflags ? "On" : "Off");
		ast_udptl_setnat(p->udptl, natflags);
	} else {
		ast_log(AST_LOG_WARNING, "UDPTL creation failed - disabling T38 for this dialog\n");
		ast_clear_flag(&p->flags[1], SIP_PAGE2_T38SUPPORT);
		return 1;
	}

	return 0;
}

/*! \brief Play indication to user
 * With SIP a lot of indications is sent as messages, letting the device play
   the indication - busy signal, congestion etc
   \return -1 to force ast_indicate to send indication in audio, 0 if SIP can handle the indication by sending a message
*/
static int sip_indicate(struct ast_channel *ast, int condition, const void *data, size_t datalen)
{
	struct sip_pvt *p = ast->tech_pvt;
	int res = 0;

	sip_pvt_lock(p);
	switch(condition) {
	case AST_CONTROL_RINGING:
		if (ast->_state == AST_STATE_RING) {
			p->invitestate = INV_EARLY_MEDIA;
			if (!ast_test_flag(&p->flags[0], SIP_PROGRESS_SENT) ||
			    (ast_test_flag(&p->flags[0], SIP_PROG_INBAND) == SIP_PROG_INBAND_NEVER)) {				
				/* Send 180 ringing if out-of-band seems reasonable */
				transmit_provisional_response(p, "180 Ringing", &p->initreq, 0);
				ast_set_flag(&p->flags[0], SIP_RINGING);
				if (ast_test_flag(&p->flags[0], SIP_PROG_INBAND) != SIP_PROG_INBAND_YES)
					break;
			} else {
				/* Well, if it's not reasonable, just send in-band */
			}
		}
		res = -1;
		break;
	case AST_CONTROL_BUSY:
		if (ast->_state != AST_STATE_UP) {
			transmit_response_reliable(p, "486 Busy Here", &p->initreq);
			p->invitestate = INV_COMPLETED;
			sip_alreadygone(p);
			ast_softhangup_nolock(ast, AST_SOFTHANGUP_DEV);
			break;
		}
		res = -1;
		break;
	case AST_CONTROL_CONGESTION:
		if (ast->_state != AST_STATE_UP) {
			transmit_response_reliable(p, "503 Service Unavailable", &p->initreq);
			p->invitestate = INV_COMPLETED;
			sip_alreadygone(p);
			ast_softhangup_nolock(ast, AST_SOFTHANGUP_DEV);
			break;
		}
		res = -1;
		break;
	case AST_CONTROL_INCOMPLETE:
		if (ast->_state != AST_STATE_UP) {
			switch (ast_test_flag(&p->flags[1], SIP_PAGE2_ALLOWOVERLAP)) {
			case SIP_PAGE2_ALLOWOVERLAP_YES:
				transmit_response_reliable(p, "484 Address Incomplete", &p->initreq);
				p->invitestate = INV_COMPLETED;
				sip_alreadygone(p);
				ast_softhangup_nolock(ast, AST_SOFTHANGUP_DEV);
				break;
			case SIP_PAGE2_ALLOWOVERLAP_DTMF:
				/* Just wait for inband DTMF digits */
				break;
			default:
				/* it actually means no support for overlap */
				transmit_response_reliable(p, "404 Not Found", &p->initreq);
				p->invitestate = INV_COMPLETED;
				sip_alreadygone(p);
				ast_softhangup_nolock(ast, AST_SOFTHANGUP_DEV);
				break;
			}
		}
		break;
	case AST_CONTROL_PROCEEDING:
		if ((ast->_state != AST_STATE_UP) &&
		    !ast_test_flag(&p->flags[0], SIP_PROGRESS_SENT) &&
		    !ast_test_flag(&p->flags[0], SIP_OUTGOING)) {
			transmit_response(p, "100 Trying", &p->initreq);
			p->invitestate = INV_PROCEEDING;
			break;
		}
		res = -1;
		break;
	case AST_CONTROL_PROGRESS:
		if ((ast->_state != AST_STATE_UP) &&
		    !ast_test_flag(&p->flags[0], SIP_PROGRESS_SENT) &&
		    !ast_test_flag(&p->flags[0], SIP_OUTGOING)) {
			p->invitestate = INV_EARLY_MEDIA;
			transmit_provisional_response(p, "183 Session Progress", &p->initreq, TRUE);
			ast_set_flag(&p->flags[0], SIP_PROGRESS_SENT);
			break;
		}
		res = -1;
		break;
	case AST_CONTROL_HOLD:
		ast_rtp_instance_update_source(p->rtp);
		ast_moh_start(ast, data, p->mohinterpret);
		break;
	case AST_CONTROL_UNHOLD:
		ast_rtp_instance_update_source(p->rtp);
		ast_moh_stop(ast);
		break;
	case AST_CONTROL_VIDUPDATE:	/* Request a video frame update */
		if (p->vrtp && !p->novideo) {
			transmit_info_with_vidupdate(p);
			/* ast_rtcp_send_h261fur(p->vrtp); */
		} else
			res = -1;
		break;
	case AST_CONTROL_T38_PARAMETERS:
		res = -1;
		if (datalen != sizeof(struct ast_control_t38_parameters)) {
			ast_log(LOG_ERROR, "Invalid datalen for AST_CONTROL_T38_PARAMETERS. Expected %d, got %d\n", (int) sizeof(struct ast_control_t38_parameters), (int) datalen);
		} else {
			const struct ast_control_t38_parameters *parameters = data;
			if (!initialize_udptl(p)) {
				res = interpret_t38_parameters(p, parameters);
			}
		}
		break;
	case AST_CONTROL_SRCUPDATE:
		ast_rtp_instance_update_source(p->rtp);
		break;
	case AST_CONTROL_SRCCHANGE:
		ast_rtp_instance_change_source(p->rtp);
		break;
	case AST_CONTROL_CONNECTED_LINE:
		update_connectedline(p, data, datalen);
		break;
	case AST_CONTROL_REDIRECTING:
		update_redirecting(p, data, datalen);
		break;
	case AST_CONTROL_AOC:
		{
			struct ast_aoc_decoded *decoded = ast_aoc_decode((struct ast_aoc_encoded *) data, datalen, ast);
			if (!decoded) {
				ast_log(LOG_ERROR, "Error decoding indicated AOC data\n");
				res = -1;
				break;
			}
			switch (ast_aoc_get_msg_type(decoded)) {
			case AST_AOC_REQUEST:
				if (ast_aoc_get_termination_request(decoded)) {
					/* TODO, once there is a way to get AOC-E on hangup, attempt that here
					 * before hanging up the channel.*/

					/* The other side has already initiated the hangup. This frame
					 * just says they are waiting to get AOC-E before completely tearing
					 * the call down.  Since SIP does not support this at the moment go
					 * ahead and terminate the call here to avoid an unnecessary timeout. */
					ast_debug(1, "AOC-E termination request received on %s. This is not yet supported on sip. Continue with hangup \n", p->owner->name);
					ast_softhangup_nolock(p->owner, AST_SOFTHANGUP_DEV);
				}
				break;
			case AST_AOC_D:
			case AST_AOC_E:
				if (ast_test_flag(&p->flags[2], SIP_PAGE3_SNOM_AOC)) {
					transmit_info_with_aoc(p, decoded);
				}
				break;
			case AST_AOC_S: /* S not supported yet */
			default:
				break;
			}
			ast_aoc_destroy_decoded(decoded);
		}
		break;
	case AST_CONTROL_UPDATE_RTP_PEER: /* Absorb this since it is handled by the bridge */
		break;
	case -1:
		res = -1;
		break;
	default:
		ast_log(LOG_WARNING, "Don't know how to indicate condition %d\n", condition);
		res = -1;
		break;
	}
	sip_pvt_unlock(p);
	return res;
}

/*!
 * \brief Initiate a call in the SIP channel
 *
 * \note called from sip_request_call (calls from the pbx ) for
 * outbound channels and from handle_request_invite for inbound
 * channels
 *
 * \pre i is locked
 *
 * \return New ast_channel locked.
 */
static struct ast_channel *sip_new(struct sip_pvt *i, int state, const char *title, const char *linkedid)
{
	struct ast_channel *tmp;
	struct ast_variable *v = NULL;
	format_t fmt;
	format_t what;
	format_t video;
	format_t text;
	format_t needvideo = 0;
	int needtext = 0;
	char buf[SIPBUFSIZE];
	char *exten;

	{
		const char *my_name;	/* pick a good name */
	
		if (title) {
			my_name = title;
		} else {
			my_name = ast_strdupa(i->fromdomain);
		}

		sip_pvt_unlock(i);
		/* Don't hold a sip pvt lock while we allocate a channel */
		tmp = ast_channel_alloc(1, state, i->cid_num, i->cid_name, i->accountcode, i->exten, i->context, linkedid, i->amaflags, "SIP/%s-%08x", my_name, ast_atomic_fetchadd_int((int *)&chan_idx, +1));
	}
	if (!tmp) {
		ast_log(LOG_WARNING, "Unable to allocate AST channel structure for SIP channel\n");
		sip_pvt_lock(i);
		return NULL;
	}
	ast_channel_lock(tmp);
	sip_pvt_lock(i);
	ast_channel_cc_params_init(tmp, i->cc_params);
	tmp->caller.id.tag = ast_strdup(i->cid_tag);

	tmp->tech = ( ast_test_flag(&i->flags[0], SIP_DTMF) == SIP_DTMF_INFO || ast_test_flag(&i->flags[0], SIP_DTMF) == SIP_DTMF_SHORTINFO) ?  &sip_tech_info : &sip_tech;

	/* Select our native format based on codec preference until we receive
	   something from another device to the contrary. */
	if (i->jointcapability) { 	/* The joint capabilities of us and peer */
		what = i->jointcapability;
		video = i->jointcapability & AST_FORMAT_VIDEO_MASK;
		text = i->jointcapability & AST_FORMAT_TEXT_MASK;
	} else if (i->capability) {		/* Our configured capability for this peer */
		what = i->capability;
		video = i->capability & AST_FORMAT_VIDEO_MASK;
		text = i->capability & AST_FORMAT_TEXT_MASK;
	} else {
		what = sip_cfg.capability;	/* Global codec support */
		video = sip_cfg.capability & AST_FORMAT_VIDEO_MASK;
		text = sip_cfg.capability & AST_FORMAT_TEXT_MASK;
	}

	/* Set the native formats for audio  and merge in video */
	tmp->nativeformats = ast_codec_choose(&i->prefs, what, 1) | video | text;
	ast_debug(3, "*** Our native formats are %s \n", ast_getformatname_multiple(buf, SIPBUFSIZE, tmp->nativeformats));
	ast_debug(3, "*** Joint capabilities are %s \n", ast_getformatname_multiple(buf, SIPBUFSIZE, i->jointcapability));
	ast_debug(3, "*** Our capabilities are %s \n", ast_getformatname_multiple(buf, SIPBUFSIZE, i->capability));
	ast_debug(3, "*** AST_CODEC_CHOOSE formats are %s \n", ast_getformatname_multiple(buf, SIPBUFSIZE, ast_codec_choose(&i->prefs, what, 1)));
	if (i->prefcodec)
		ast_debug(3, "*** Our preferred formats from the incoming channel are %s \n", ast_getformatname_multiple(buf, SIPBUFSIZE, i->prefcodec));

	/* XXX Why are we choosing a codec from the native formats?? */
	fmt = ast_best_codec(tmp->nativeformats);

	/* If we have a prefcodec setting, we have an inbound channel that set a
	   preferred format for this call. Otherwise, we check the jointcapability
	   We also check for vrtp. If it's not there, we are not allowed do any video anyway.
	 */
	if (i->vrtp) {
		if (ast_test_flag(&i->flags[1], SIP_PAGE2_VIDEOSUPPORT))
			needvideo = AST_FORMAT_VIDEO_MASK;
		else if (i->prefcodec)
			needvideo = i->prefcodec & AST_FORMAT_VIDEO_MASK;	/* Outbound call */
 		else
			needvideo = i->jointcapability & AST_FORMAT_VIDEO_MASK;	/* Inbound call */
	}

	if (i->trtp) {
		if (i->prefcodec)
			needtext = i->prefcodec & AST_FORMAT_TEXT_MASK;	/* Outbound call */
 		else
			needtext = i->jointcapability & AST_FORMAT_TEXT_MASK;	/* Inbound call */
	}

	if (needvideo)
		ast_debug(3, "This channel can handle video! HOLLYWOOD next!\n");
	else
		ast_debug(3, "This channel will not be able to handle video.\n");

	enable_dsp_detect(i);

	if ((ast_test_flag(&i->flags[0], SIP_DTMF) == SIP_DTMF_INBAND) ||
	    (ast_test_flag(&i->flags[0], SIP_DTMF) == SIP_DTMF_AUTO)) {
		if (i->rtp) {
			ast_rtp_instance_dtmf_mode_set(i->rtp, AST_RTP_DTMF_MODE_INBAND);
		}
	} else if (ast_test_flag(&i->flags[0], SIP_DTMF) == SIP_DTMF_RFC2833) {
		if (i->rtp) {
			ast_rtp_instance_dtmf_mode_set(i->rtp, AST_RTP_DTMF_MODE_RFC2833);
		}
	}

	/* Set file descriptors for audio, video, and realtime text.  Since
	 * UDPTL is created as needed in the lifetime of a dialog, its file
	 * descriptor is set in initialize_udptl */
	if (i->rtp) {
		ast_channel_set_fd(tmp, 0, ast_rtp_instance_fd(i->rtp, 0));
		ast_channel_set_fd(tmp, 1, ast_rtp_instance_fd(i->rtp, 1));
	}
	if (needvideo && i->vrtp) {
		ast_channel_set_fd(tmp, 2, ast_rtp_instance_fd(i->vrtp, 0));
		ast_channel_set_fd(tmp, 3, ast_rtp_instance_fd(i->vrtp, 1));
	}
	if (needtext && i->trtp) {
		ast_channel_set_fd(tmp, 4, ast_rtp_instance_fd(i->trtp, 0));
	}
	if (i->udptl) {
		ast_channel_set_fd(tmp, 5, ast_udptl_fd(i->udptl));
	}

	if (state == AST_STATE_RING)
		tmp->rings = 1;
	tmp->adsicpe = AST_ADSI_UNAVAILABLE;

	tmp->writeformat = fmt;
	tmp->rawwriteformat = fmt;
	ast_rtp_instance_set_write_format(i->rtp, fmt);

	tmp->readformat = fmt;
	tmp->rawreadformat = fmt;
	ast_rtp_instance_set_read_format(i->rtp, fmt);

	tmp->tech_pvt = dialog_ref(i, "sip_new: set chan->tech_pvt to i");

	tmp->callgroup = i->callgroup;
	tmp->pickupgroup = i->pickupgroup;
	tmp->caller.id.name.presentation = i->callingpres;
	tmp->caller.id.number.presentation = i->callingpres;
	if (!ast_strlen_zero(i->parkinglot))
		ast_string_field_set(tmp, parkinglot, i->parkinglot);
	if (!ast_strlen_zero(i->accountcode))
		ast_string_field_set(tmp, accountcode, i->accountcode);
	if (i->amaflags)
		tmp->amaflags = i->amaflags;
	if (!ast_strlen_zero(i->language))
		ast_string_field_set(tmp, language, i->language);
	i->owner = tmp;
	ast_module_ref(ast_module_info->self);
	ast_copy_string(tmp->context, i->context, sizeof(tmp->context));
	/*Since it is valid to have extensions in the dialplan that have unescaped characters in them
	 * we should decode the uri before storing it in the channel, but leave it encoded in the sip_pvt
	 * structure so that there aren't issues when forming URI's
	 */
	exten = ast_strdupa(i->exten);
	sip_pvt_unlock(i);
	ast_channel_unlock(tmp);
	if (!ast_exists_extension(NULL, i->context, i->exten, 1, i->cid_num)) {
		ast_uri_decode(exten);
	}
	ast_channel_lock(tmp);
	sip_pvt_lock(i);
	ast_copy_string(tmp->exten, exten, sizeof(tmp->exten));

	/* Don't use ast_set_callerid() here because it will
	 * generate an unnecessary NewCallerID event  */
	if (!ast_strlen_zero(i->cid_num)) {
		tmp->caller.ani.number.valid = 1;
		tmp->caller.ani.number.str = ast_strdup(i->cid_num);
	}
	if (!ast_strlen_zero(i->rdnis)) {
		tmp->redirecting.from.number.valid = 1;
		tmp->redirecting.from.number.str = ast_strdup(i->rdnis);
	}

	if (!ast_strlen_zero(i->exten) && strcmp(i->exten, "s")) {
		tmp->dialed.number.str = ast_strdup(i->exten);
	}

	tmp->priority = 1;
	if (!ast_strlen_zero(i->uri))
		pbx_builtin_setvar_helper(tmp, "SIPURI", i->uri);
	if (!ast_strlen_zero(i->domain))
		pbx_builtin_setvar_helper(tmp, "SIPDOMAIN", i->domain);
	if (!ast_strlen_zero(i->callid))
		pbx_builtin_setvar_helper(tmp, "SIPCALLID", i->callid);
	if (i->rtp)
		ast_jb_configure(tmp, &global_jbconf);

	/* Set channel variables for this call from configuration */
	for (v = i->chanvars ; v ; v = v->next) {
		char valuebuf[1024];
		pbx_builtin_setvar_helper(tmp, v->name, ast_get_encoded_str(v->value, valuebuf, sizeof(valuebuf)));
	}

	if (i->do_history)
		append_history(i, "NewChan", "Channel %s - from %s", tmp->name, i->callid);

	/* Inform manager user about new channel and their SIP call ID */
	if (sip_cfg.callevents)
		manager_event(EVENT_FLAG_SYSTEM, "ChannelUpdate",
			"Channel: %s\r\nUniqueid: %s\r\nChanneltype: %s\r\nSIPcallid: %s\r\nSIPfullcontact: %s\r\n",
			tmp->name, tmp->uniqueid, "SIP", i->callid, i->fullcontact);

	return tmp;
}

/*! \brief Reads one line of SIP message body */
static char *get_body_by_line(const char *line, const char *name, int nameLen, char delimiter)
{
	if (!strncasecmp(line, name, nameLen) && line[nameLen] == delimiter)
		return ast_skip_blanks(line + nameLen + 1);

	return "";
}

/*! \brief Lookup 'name' in the SDP starting
 * at the 'start' line. Returns the matching line, and 'start'
 * is updated with the next line number.
 */
static const char *get_sdp_iterate(int *start, struct sip_request *req, const char *name)
{
	int len = strlen(name);

	while (*start < (req->sdp_start + req->sdp_count)) {
		const char *r = get_body_by_line(REQ_OFFSET_TO_STR(req, line[(*start)++]), name, len, '=');
		if (r[0] != '\0')
			return r;
	}

	/* if the line was not found, ensure that *start points past the SDP */
	(*start)++;

	return "";
}

/*! \brief Fetches the next valid SDP line between the 'start' line
 * (inclusive) and the 'stop' line (exclusive). Returns the type
 * ('a', 'c', ...) and matching line in reference 'start' is updated
 * with the next line number.
 */
static char get_sdp_line(int *start, int stop, struct sip_request *req, const char **value)
{
	char type = '\0';
	const char *line = NULL;

	if (stop > (req->sdp_start + req->sdp_count)) {
		stop = req->sdp_start + req->sdp_count;
	}

	while (*start < stop) {
		line = REQ_OFFSET_TO_STR(req, line[(*start)++]);
		if (line[1] == '=') {
			type = line[0];
			*value = ast_skip_blanks(line + 2);
			break;
		}
	}

	return type;
}

/*! \brief Get a specific line from the message body */
static char *get_body(struct sip_request *req, char *name, char delimiter)
{
	int x;
	int len = strlen(name);
	char *r;

	for (x = 0; x < req->lines; x++) {
		r = get_body_by_line(REQ_OFFSET_TO_STR(req, line[x]), name, len, delimiter);
		if (r[0] != '\0')
			return r;
	}

	return "";
}

/*! \brief Find compressed SIP alias */
static const char *find_alias(const char *name, const char *_default)
{
	/*! \brief Structure for conversion between compressed SIP and "normal" SIP */
	static const struct cfalias {
		char * const fullname;
		char * const shortname;
	} aliases[] = {
		{ "Content-Type",	 "c" },
		{ "Content-Encoding",	 "e" },
		{ "From",		 "f" },
		{ "Call-ID",		 "i" },
		{ "Contact",		 "m" },
		{ "Content-Length",	 "l" },
		{ "Subject",		 "s" },
		{ "To",			 "t" },
		{ "Supported",		 "k" },
		{ "Refer-To",		 "r" },
		{ "Referred-By",	 "b" },
		{ "Allow-Events",	 "u" },
		{ "Event",		 "o" },
		{ "Via",		 "v" },
		{ "Accept-Contact",      "a" },
		{ "Reject-Contact",      "j" },
		{ "Request-Disposition", "d" },
		{ "Session-Expires",     "x" },
		{ "Identity",            "y" },
		{ "Identity-Info",       "n" },
	};
	int x;

	for (x = 0; x < ARRAY_LEN(aliases); x++) {
		if (!strcasecmp(aliases[x].fullname, name))
			return aliases[x].shortname;
	}

	return _default;
}

static const char *__get_header(const struct sip_request *req, const char *name, int *start)
{
	/*
	 * Technically you can place arbitrary whitespace both before and after the ':' in
	 * a header, although RFC3261 clearly says you shouldn't before, and place just
	 * one afterwards.  If you shouldn't do it, what absolute idiot decided it was
	 * a good idea to say you can do it, and if you can do it, why in the hell would.
	 * you say you shouldn't.
	 * Anyways, pedanticsipchecking controls whether we allow spaces before ':',
	 * and we always allow spaces after that for compatibility.
	 */
	const char *sname = find_alias(name, NULL);
	int x, len = strlen(name), slen = (sname ? 1 : 0);
	for (x = *start; x < req->headers; x++) {
		const char *header = REQ_OFFSET_TO_STR(req, header[x]);
		int smatch = 0, match = !strncasecmp(header, name, len);
		if (slen) {
			smatch = !strncasecmp(header, sname, slen);
		}
		if (match || smatch) {
			/* skip name */
			const char *r = header + (match ? len : slen );
			if (sip_cfg.pedanticsipchecking) {
				r = ast_skip_blanks(r);
			}

			if (*r == ':') {
				*start = x+1;
				return ast_skip_blanks(r+1);
			}
		}
	}

	/* Don't return NULL, so get_header is always a valid pointer */
	return "";
}

/*! \brief Get header from SIP request
	\return Always return something, so don't check for NULL because it won't happen :-)
*/
static const char *get_header(const struct sip_request *req, const char *name)
{
	int start = 0;
	return __get_header(req, name, &start);
}

/*! \brief Read RTP from network */
static struct ast_frame *sip_rtp_read(struct ast_channel *ast, struct sip_pvt *p, int *faxdetect)
{
	/* Retrieve audio/etc from channel.  Assumes p->lock is already held. */
	struct ast_frame *f;
	
	if (!p->rtp) {
		/* We have no RTP allocated for this channel */
		return &ast_null_frame;
	}

	switch(ast->fdno) {
	case 0:
		f = ast_rtp_instance_read(p->rtp, 0);	/* RTP Audio */
		break;
	case 1:
		f = ast_rtp_instance_read(p->rtp, 1);	/* RTCP Control Channel */
		break;
	case 2:
		f = ast_rtp_instance_read(p->vrtp, 0);	/* RTP Video */
		break;
	case 3:
		f = ast_rtp_instance_read(p->vrtp, 1);	/* RTCP Control Channel for video */
		break;
	case 4:
		f = ast_rtp_instance_read(p->trtp, 0);	/* RTP Text */
		if (sipdebug_text) {
			int i;
			unsigned char* arr = f->data.ptr;
			for (i=0; i < f->datalen; i++)
				ast_verbose("%c", (arr[i] > ' ' && arr[i] < '}') ? arr[i] : '.');
			ast_verbose(" -> ");
			for (i=0; i < f->datalen; i++)
				ast_verbose("%02X ", arr[i]);
			ast_verbose("\n");
		}
		break;
	case 5:
		f = ast_udptl_read(p->udptl);	/* UDPTL for T.38 */
		break;
	default:
		f = &ast_null_frame;
	}
	/* Don't forward RFC2833 if we're not supposed to */
	if (f && (f->frametype == AST_FRAME_DTMF_BEGIN || f->frametype == AST_FRAME_DTMF_END) &&
	    (ast_test_flag(&p->flags[0], SIP_DTMF) != SIP_DTMF_RFC2833)) {
		ast_debug(1, "Ignoring DTMF (%c) RTP frame because dtmfmode is not RFC2833\n", f->subclass.integer);
		return &ast_null_frame;
	}

	/* We already hold the channel lock */
	if (!p->owner || (f && f->frametype != AST_FRAME_VOICE))
		return f;

	if (f && f->subclass.codec != (p->owner->nativeformats & AST_FORMAT_AUDIO_MASK)) {
		if (!(f->subclass.codec & p->jointcapability)) {
			ast_debug(1, "Bogus frame of format '%s' received from '%s'!\n",
				ast_getformatname(f->subclass.codec), p->owner->name);
			return &ast_null_frame;
		}
		ast_debug(1, "Oooh, format changed to %s\n",
			ast_getformatname(f->subclass.codec));
		p->owner->nativeformats = (p->owner->nativeformats & (AST_FORMAT_VIDEO_MASK | AST_FORMAT_TEXT_MASK)) | f->subclass.codec;
		ast_set_read_format(p->owner, p->owner->readformat);
		ast_set_write_format(p->owner, p->owner->writeformat);
	}

	if (f && p->dsp) {
		f = ast_dsp_process(p->owner, p->dsp, f);
		if (f && f->frametype == AST_FRAME_DTMF) {
			if (f->subclass.integer == 'f') {
				ast_debug(1, "Fax CNG detected on %s\n", ast->name);
				*faxdetect = 1;
				/* If we only needed this DSP for fax detection purposes we can just drop it now */
				if (ast_test_flag(&p->flags[0], SIP_DTMF) == SIP_DTMF_INBAND) {
					ast_dsp_set_features(p->dsp, DSP_FEATURE_DIGIT_DETECT);
				} else {
					ast_dsp_free(p->dsp);
					p->dsp = NULL;
				}
			} else {
				ast_debug(1, "* Detected inband DTMF '%c'\n", f->subclass.integer);
			}
		}
	}
	
	return f;
}

/*! \brief Read SIP RTP from channel */
static struct ast_frame *sip_read(struct ast_channel *ast)
{
	struct ast_frame *fr;
	struct sip_pvt *p = ast->tech_pvt;
	int faxdetected = FALSE;

	sip_pvt_lock(p);
	fr = sip_rtp_read(ast, p, &faxdetected);
	p->lastrtprx = time(NULL);

	/* If we detect a CNG tone and fax detection is enabled then send us off to the fax extension */
	if (faxdetected && ast_test_flag(&p->flags[1], SIP_PAGE2_FAX_DETECT_CNG)) {
		if (strcmp(ast->exten, "fax")) {
			const char *target_context = S_OR(ast->macrocontext, ast->context);
			/* We need to unlock 'ast' here because
			 * ast_exists_extension has the potential to start and
			 * stop an autoservice on the channel. Such action is
			 * prone to deadlock if the channel is locked.
			 */
			sip_pvt_unlock(p);
			ast_channel_unlock(ast);
			if (ast_exists_extension(ast, target_context, "fax", 1,
				S_COR(ast->caller.id.number.valid, ast->caller.id.number.str, NULL))) {
				ast_channel_lock(ast);
				sip_pvt_lock(p);
				ast_verbose(VERBOSE_PREFIX_2 "Redirecting '%s' to fax extension due to CNG detection\n", ast->name);
				pbx_builtin_setvar_helper(ast, "FAXEXTEN", ast->exten);
				if (ast_async_goto(ast, target_context, "fax", 1)) {
					ast_log(LOG_NOTICE, "Failed to async goto '%s' into fax of '%s'\n", ast->name, target_context);
				}
				fr = &ast_null_frame;
			} else {
				ast_channel_lock(ast);
				sip_pvt_lock(p);
				ast_log(LOG_NOTICE, "FAX CNG detected but no fax extension\n");
			}
		}
	}

	/* Only allow audio through if they sent progress with SDP, or if the channel is actually answered */
	if (fr && fr->frametype == AST_FRAME_VOICE && p->invitestate != INV_EARLY_MEDIA && ast->_state != AST_STATE_UP) {
		fr = &ast_null_frame;
	}

	sip_pvt_unlock(p);

	return fr;
}


/*! \brief Generate 32 byte random string for callid's etc */
static char *generate_random_string(char *buf, size_t size)
{
	long val[4];
	int x;

	for (x=0; x<4; x++)
		val[x] = ast_random();
	snprintf(buf, size, "%08lx%08lx%08lx%08lx", val[0], val[1], val[2], val[3]);

	return buf;
}

static char *generate_uri(struct sip_pvt *pvt, char *buf, size_t size)
{
	struct ast_str *uri = ast_str_alloca(size);
	ast_str_set(&uri, 0, "%s", pvt->socket.type == SIP_TRANSPORT_TLS ? "sips:" : "sip:");
	/* Here would be a great place to generate a UUID, but for now we'll
	 * use the handy random string generation function we already have
	 */
	ast_str_append(&uri, 0, "%s", generate_random_string(buf, size));
	ast_str_append(&uri, 0, "@%s", ast_sockaddr_stringify_remote(&pvt->ourip));
	ast_copy_string(buf, ast_str_buffer(uri), size);
	return buf;
}

/*!
 * \brief Build SIP Call-ID value for a non-REGISTER transaction
 *
 * \note The passed in pvt must not be in a dialogs container
 * since this function changes the hash key used by the
 * container.
 */
static void build_callid_pvt(struct sip_pvt *pvt)
{
	char buf[33];
	const char *host = S_OR(pvt->fromdomain, ast_sockaddr_stringify_remote(&pvt->ourip));

	ast_string_field_build(pvt, callid, "%s@%s", generate_random_string(buf, sizeof(buf)), host);
}

/*! \brief Unlink the given object from the container and return TRUE if it was in the container. */
#define CONTAINER_UNLINK(container, obj, tag)								\
	({																		\
		int found = 0;														\
		typeof((obj)) __removed_obj;										\
		__removed_obj = ao2_t_callback((container),							\
			OBJ_UNLINK | OBJ_POINTER, ao2_match_by_addr, (obj), (tag));		\
		if (__removed_obj) {												\
			ao2_ref(__removed_obj, -1);										\
			found = 1;														\
		}																	\
		found;																\
	})

/*!
 * \internal
 * \brief Safely change the callid of the given SIP dialog.
 *
 * \param pvt SIP private structure to change callid
 * \param callid Specified new callid to use.  NULL if generate new callid.
 *
 * \return Nothing
 */
static void change_callid_pvt(struct sip_pvt *pvt, const char *callid)
{
	int in_dialog_container;

	ao2_lock(dialogs);
	in_dialog_container = CONTAINER_UNLINK(dialogs, pvt,
		"About to change the callid -- remove the old name");
	if (callid) {
		ast_string_field_set(pvt, callid, callid);
	} else {
		build_callid_pvt(pvt);
	}
	if (in_dialog_container) {
		ao2_t_link(dialogs, pvt, "New dialog callid -- inserted back into table");
	}
	ao2_unlock(dialogs);
}

/*! \brief Build SIP Call-ID value for a REGISTER transaction */
static void build_callid_registry(struct sip_registry *reg, const struct ast_sockaddr *ourip, const char *fromdomain)
{
	char buf[33];

	const char *host = S_OR(fromdomain, ast_sockaddr_stringify_host_remote(ourip));

	ast_string_field_build(reg, callid, "%s@%s", generate_random_string(buf, sizeof(buf)), host);
}

/*! \brief Make our SIP dialog tag */
static void make_our_tag(char *tagbuf, size_t len)
{
	snprintf(tagbuf, len, "as%08lx", ast_random());
}

/*! \brief Allocate Session-Timers struct w/in dialog */
static struct sip_st_dlg* sip_st_alloc(struct sip_pvt *const p)
{
	struct sip_st_dlg *stp;

	if (p->stimer) {
		ast_log(LOG_ERROR, "Session-Timer struct already allocated\n");
		return p->stimer;
	}

	if (!(stp = ast_calloc(1, sizeof(struct sip_st_dlg))))
		return NULL;

	p->stimer = stp;

	stp->st_schedid = -1;           /* Session-Timers ast_sched scheduler id */

	return p->stimer;
}

/*! \brief Allocate sip_pvt structure, set defaults and link in the container.
 * Returns a reference to the object so whoever uses it later must
 * remember to release the reference.
 */
struct sip_pvt *sip_alloc(ast_string_field callid, struct ast_sockaddr *addr,
				 int useglobal_nat, const int intended_method, struct sip_request *req)
{
	struct sip_pvt *p;

	if (!(p = ao2_t_alloc(sizeof(*p), sip_destroy_fn, "allocate a dialog(pvt) struct")))
		return NULL;

	if (ast_string_field_init(p, 512)) {
		ao2_t_ref(p, -1, "failed to string_field_init, drop p");
		return NULL;
	}

	if (!(p->cc_params = ast_cc_config_params_init())) {
		ao2_t_ref(p, -1, "Yuck, couldn't allocate cc_params struct. Get rid o' p");
		return NULL;
	}

	/* If this dialog is created as the result of an incoming Request. Lets store
	 * some information about that request */
	if (req) {
		struct sip_via *via;
		const char *cseq = get_header(req, "Cseq");
		uint32_t seqno;

		/* get branch parameter from initial Request that started this dialog */
		via = parse_via(get_header(req, "Via"));
		if (via) {
			/* only store the branch if it begins with the magic prefix "z9hG4bK", otherwise
			 * it is not useful to us to have it */
			if (!ast_strlen_zero(via->branch) && !strncasecmp(via->branch, "z9hG4bK", 7)) {
				ast_string_field_set(p, initviabranch, via->branch);
				ast_string_field_set(p, initviasentby, via->sent_by);
			}
			free_via(via);
		}

		/* Store initial incoming cseq. An error in sscanf here is ignored.  There is no approperiate
		 * except not storing the number.  CSeq validation must take place before dialog creation in find_call */
		if (!ast_strlen_zero(cseq) && (sscanf(cseq, "%30u", &seqno) == 1)) {
			p->init_icseq = seqno;
		}
		/* Later in ast_sip_ouraddrfor we need this to choose the right ip and port for the specific transport */
		set_socket_transport(&p->socket, req->socket.type);
	} else {
		set_socket_transport(&p->socket, SIP_TRANSPORT_UDP);
	}

	p->socket.fd = -1;
	p->method = intended_method;
	p->initid = -1;
	p->waitid = -1;
	p->reinviteid = -1;
	p->autokillid = -1;
	p->request_queue_sched_id = -1;
	p->provisional_keepalive_sched_id = -1;
	p->t38id = -1;
	p->subscribed = NONE;
	p->stateid = -1;
	p->sessionversion_remote = -1;
	p->session_modify = TRUE;
	p->stimer = NULL;
	p->prefs = default_prefs;		/* Set default codecs for this call */
	p->maxforwards = sip_cfg.default_max_forwards;

	if (intended_method != SIP_OPTIONS) {	/* Peerpoke has it's own system */
		p->timer_t1 = global_t1;	/* Default SIP retransmission timer T1 (RFC 3261) */
		p->timer_b = global_timer_b;	/* Default SIP transaction timer B (RFC 3261) */
	}

	if (!addr) {
		p->ourip = internip;
	} else {
		ast_sockaddr_copy(&p->sa, addr);
		ast_sip_ouraddrfor(&p->sa, &p->ourip, p);
	}

	/* Copy global flags to this PVT at setup. */
	ast_copy_flags(&p->flags[0], &global_flags[0], SIP_FLAGS_TO_COPY);
	ast_copy_flags(&p->flags[1], &global_flags[1], SIP_PAGE2_FLAGS_TO_COPY);
	ast_copy_flags(&p->flags[2], &global_flags[2], SIP_PAGE3_FLAGS_TO_COPY);

	p->do_history = recordhistory;

	p->branch = ast_random();	
	make_our_tag(p->tag, sizeof(p->tag));
	p->ocseq = INITIAL_CSEQ;
	p->allowed_methods = UINT_MAX;

	if (sip_methods[intended_method].need_rtp) {
		p->maxcallbitrate = default_maxcallbitrate;
		p->autoframing = global_autoframing;
	}

	if (useglobal_nat && addr) {
		/* Setup NAT structure according to global settings if we have an address */
		ast_copy_flags(&p->flags[0], &global_flags[0], SIP_NAT_FORCE_RPORT);
		ast_sockaddr_copy(&p->recv, addr);

		do_setnat(p);
	}

	if (p->method != SIP_REGISTER) {
		ast_string_field_set(p, fromdomain, default_fromdomain);
		p->fromdomainport = default_fromdomainport;
	}
	build_via(p);
	if (!callid)
		build_callid_pvt(p);
	else
		ast_string_field_set(p, callid, callid);
	/* Assign default music on hold class */
	ast_string_field_set(p, mohinterpret, default_mohinterpret);
	ast_string_field_set(p, mohsuggest, default_mohsuggest);
	p->capability = sip_cfg.capability;
	p->allowtransfer = sip_cfg.allowtransfer;
	if ((ast_test_flag(&p->flags[0], SIP_DTMF) == SIP_DTMF_RFC2833) ||
	    (ast_test_flag(&p->flags[0], SIP_DTMF) == SIP_DTMF_AUTO)) {
		p->noncodeccapability |= AST_RTP_DTMF;
	}
	ast_string_field_set(p, context, sip_cfg.default_context);
	ast_string_field_set(p, parkinglot, default_parkinglot);
	ast_string_field_set(p, engine, default_engine);

	AST_LIST_HEAD_INIT_NOLOCK(&p->request_queue);

	/* Add to active dialog list */

	ao2_t_link(dialogs, p, "link pvt into dialogs table");
	
	ast_debug(1, "Allocating new SIP dialog for %s - %s (%s)\n", callid ? callid : p->callid, sip_methods[intended_method].text, p->rtp ? "With RTP" : "No RTP");
	return p;
}

/*!
 * \brief Check if an ip is an multicast IP.
 * \parm addr the address to check
 *
 * This function checks if an address is in the 224.0.0.0/4 network block.
 * \return non-zero if this is a multicast address
 */
static int addr_is_multicast(const struct ast_sockaddr *addr)
{
	return ((ast_sockaddr_ipv4(addr) & 0xf0000000) == 0xe0000000);
}

/*!
 * \brief Process the Via header according to RFC 3261 section 18.2.2.
 * \param p a sip_pvt structure that will be modified according to the received
 * header
 * \param req a sip request with a Via header to process
 *
 * This function will update the destination of the response according to the
 * Via header in the request and RFC 3261 section 18.2.2. We do not have a
 * transport layer so we ignore certain values like the 'received' param (we
 * set the destination address to the addres the request came from in the
 * respprep() function).
 *
 * \retval -1 error
 * \retval 0 success
 */
static int process_via(struct sip_pvt *p, const struct sip_request *req)
{
	struct sip_via *via = parse_via(get_header(req, "Via"));

	if (!via) {
		ast_log(LOG_ERROR, "error processing via header\n");
		return -1;
	}

	if (via->maddr) {
		if (ast_sockaddr_resolve_first(&p->sa, via->maddr, PARSE_PORT_FORBID)) {
			ast_log(LOG_WARNING, "Can't find address for maddr '%s'\n", via->maddr);
			ast_log(LOG_ERROR, "error processing via header\n");
			free_via(via);
			return -1;
		}

		if (addr_is_multicast(&p->sa)) {
			setsockopt(sipsock, IPPROTO_IP, IP_MULTICAST_TTL, &via->ttl, sizeof(via->ttl));
		}
	}

	ast_sockaddr_set_port(&p->sa, via->port ? via->port : STANDARD_SIP_PORT);

	free_via(via);
	return 0;
}

/* \brief arguments used for Request/Response to matching */
struct match_req_args {
	int method;
	const char *callid;
	const char *totag;
	const char *fromtag;
	uint32_t seqno;

	/* Set if the method is a Request */
	const char *ruri;
	const char *viabranch;
	const char *viasentby;

	/* Set this if the Authentication header is present in the Request. */
	int authentication_present;
};

enum match_req_res {
	SIP_REQ_MATCH,
	SIP_REQ_NOT_MATCH,
	SIP_REQ_LOOP_DETECTED,
};

/*
 * \brief Match a incoming Request/Response to a dialog
 *
 * \retval enum match_req_res indicating if the dialog matches the arg
 */
static enum match_req_res match_req_to_dialog(struct sip_pvt *sip_pvt_ptr, struct match_req_args *arg)
{
	const char *init_ruri = NULL;
	if (sip_pvt_ptr->initreq.headers) {
		init_ruri = REQ_OFFSET_TO_STR(&sip_pvt_ptr->initreq, rlPart2);
	}

	/*
	 * Match Tags and call-id to Dialog
	 */
	if (!ast_strlen_zero(arg->callid) && strcmp(sip_pvt_ptr->callid, arg->callid)) {
		/* call-id does not match. */
		return SIP_REQ_NOT_MATCH;
	}
	if (arg->method == SIP_RESPONSE) {
		/* Verify totag if we have one stored for this dialog, but never be strict about this for
		 * a response until the dialog is established */
		if (!ast_strlen_zero(sip_pvt_ptr->theirtag) && ast_test_flag(&sip_pvt_ptr->flags[1], SIP_PAGE2_DIALOG_ESTABLISHED)) {
			if (ast_strlen_zero(arg->totag)) {
				/* missing totag when they already gave us one earlier */
				return SIP_REQ_NOT_MATCH;
			}
			if (strcmp(arg->totag, sip_pvt_ptr->theirtag)) {
				/* The totag of the response does not match the one we have stored */
				return SIP_REQ_NOT_MATCH;
			}
		}
		/* Verify fromtag of response matches the tag we gave them. */
		if (strcmp(arg->fromtag, sip_pvt_ptr->tag)) {
			/* fromtag from response does not match our tag */
			return SIP_REQ_NOT_MATCH;
		}
	} else {
		/* Verify the fromtag of Request matches the tag they provided earlier.
		 * If this is a Request with authentication credentials, forget their old
		 * tag as it is not valid after the 401 or 407 response. */
		if (!arg->authentication_present && strcmp(arg->fromtag, sip_pvt_ptr->theirtag)) {
			/* their tag does not match the one was have stored for them */
			return SIP_REQ_NOT_MATCH;
		}
		/* Verify if totag is present in Request, that it matches what we gave them as our tag earlier */
		if (!ast_strlen_zero(arg->totag) && (strcmp(arg->totag, sip_pvt_ptr->tag))) {
			/* totag from Request does not match our tag */
			return SIP_REQ_NOT_MATCH;
		}
	}

	/*
	 * Compare incoming request against initial transaction.
	 * 
	 * This is a best effort attempt at distinguishing forked requests from
	 * our initial transaction.  If all the elements are NOT in place to evaluate
	 * this, this block is ignored and the dialog match is made regardless.
	 * Once the totag is established after the dialog is confirmed, this is not necessary.
	 *
	 * CRITERIA required for initial transaction matching.
	 * 
	 * 1. Is a Request
	 * 2. Callid and theirtag match (this is done in the dialog matching block)
	 * 3. totag is NOT present
	 * 4. CSeq matchs our initial transaction's cseq number
	 * 5. pvt has init via branch parameter stored
	 */
	if ((arg->method != SIP_RESPONSE) &&                 /* must be a Request */
		ast_strlen_zero(arg->totag) &&                   /* must not have a totag */
		(sip_pvt_ptr->init_icseq == arg->seqno) &&       /* the cseq must be the same as this dialogs initial cseq */
		!ast_strlen_zero(sip_pvt_ptr->initviabranch) &&  /* The dialog must have started with a RFC3261 compliant branch tag */
		init_ruri) {                                     /* the dialog must have an initial request uri associated with it */
		/* This Request matches all the criteria required for Loop/Merge detection.
		 * Now we must go down the path of comparing VIA's and RURIs. */
		if (ast_strlen_zero(arg->viabranch) ||
			strcmp(arg->viabranch, sip_pvt_ptr->initviabranch) ||
			ast_strlen_zero(arg->viasentby) ||
			strcmp(arg->viasentby, sip_pvt_ptr->initviasentby)) {
			/* At this point, this request does not match this Dialog.*/

			/* if methods are different this is just a mismatch */
			if ((sip_pvt_ptr->method != arg->method)) {
				return SIP_REQ_NOT_MATCH;
			}

			/* If RUIs are different, this is a forked request to a separate URI.
			 * Returning a mismatch allows this Request to be processed separately. */
			if (sip_uri_cmp(init_ruri, arg->ruri)) {
				/* not a match, request uris are different */
				return SIP_REQ_NOT_MATCH;
			}

			/* Loop/Merge Detected
			 *
			 * ---Current Matches to Initial Request---
			 * request uri
			 * Call-id
			 * their-tag
			 * no totag present
			 * method
			 * cseq
			 *
			 * --- Does not Match Initial Request ---
			 * Top Via
			 *
			 * Without the same Via, this can not match our initial transaction for this dialog,
			 * but given that this Request matches everything else associated with that initial
			 * Request this is most certainly a Forked request in which we have already received
			 * part of the fork.
			 */
			return SIP_REQ_LOOP_DETECTED;
		}
	} /* end of Request Via check */

	/* Match Authentication Request.
	 *
	 * A Request with an Authentication header must come back with the
	 * same Request URI.  Otherwise it is not a match.
	 */
	if ((arg->method != SIP_RESPONSE) &&      /* Must be a Request type to even begin checking this */
		ast_strlen_zero(arg->totag) &&        /* no totag is present to match */
		arg->authentication_present &&        /* Authentication header is present in Request */
		sip_uri_cmp(init_ruri, arg->ruri)) {  /* Compare the Request URI of both the last Request and this new one */

		/* Authentication was provided, but the Request URI did not match the last one on this dialog. */
		return SIP_REQ_NOT_MATCH;
	}

	return SIP_REQ_MATCH;
}

/*! \internal
 *
 * \brief Locks both pvt and pvt owner if owner is present.
 *
 * \note This function gives a ref to pvt->owner if it is present and locked.
 *       This reference must be decremented after pvt->owner is unlocked.
 *
 * \note This function will never give you up,
 * \note This function will never let you down.
 * \note This function will run around and desert you.
 *
 * \pre pvt is not locked
 * \post pvt is locked
 * \post pvt->owner is locked and its reference count is increased (if pvt->owner is not NULL)
 *
 * \returns a pointer to the locked and reffed pvt->owner channel if it exists.
 */
static struct ast_channel *sip_pvt_lock_full(struct sip_pvt *pvt)
{
	struct ast_channel *chan;

	/* Locking is simple when it is done right.  If you see a deadlock resulting
	 * in this function, it is not this function's fault, Your problem exists elsewhere.
	 * This function is perfect... seriously. */
	for (;;) {
		/* First, get the channel and grab a reference to it */
		sip_pvt_lock(pvt);
		chan = pvt->owner;
		if (chan) {
			/* The channel can not go away while we hold the pvt lock.
			 * Give the channel a ref so it will not go away after we let
			 * the pvt lock go. */
			ast_channel_ref(chan);
		} else {
			/* no channel, return pvt locked */
			return NULL;
		}

		/* We had to hold the pvt lock while getting a ref to the owner channel
		 * but now we have to let this lock go in order to preserve proper
		 * locking order when grabbing the channel lock */
		sip_pvt_unlock(pvt);

		/* Look, no deadlock avoidance, hooray! */
		ast_channel_lock(chan);
		sip_pvt_lock(pvt);

		if (pvt->owner == chan) {
			/* done */
			break;
		}

		/* If the owner changed while everything was unlocked, no problem,
		 * just start over and everthing will work.  This is rare, do not be
		 * confused by this loop and think this it is an expensive operation.
		 * The majority of the calls to this function will never involve multiple
		 * executions of this loop. */
		ast_channel_unlock(chan);
		ast_channel_unref(chan);
		sip_pvt_unlock(pvt);
	}

	/* If owner exists, it is locked and reffed */
	return pvt->owner;
}

/*! \brief find or create a dialog structure for an incoming SIP message.
 * Connect incoming SIP message to current dialog or create new dialog structure
 * Returns a reference to the sip_pvt object, remember to give it back once done.
 *     Called by handle_request_do
 */
static struct sip_pvt *find_call(struct sip_request *req, struct ast_sockaddr *addr, const int intended_method)
{
	char totag[128];
	char fromtag[128];
	const char *callid = get_header(req, "Call-ID");
	const char *from = get_header(req, "From");
	const char *to = get_header(req, "To");
	const char *cseq = get_header(req, "Cseq");
	struct sip_pvt *sip_pvt_ptr;
	uint32_t seqno;
	/* Call-ID, to, from and Cseq are required by RFC 3261. (Max-forwards and via too - ignored now) */
	/* get_header always returns non-NULL so we must use ast_strlen_zero() */
	if (ast_strlen_zero(callid) || ast_strlen_zero(to) ||
			ast_strlen_zero(from) || ast_strlen_zero(cseq) ||
			(sscanf(cseq, "%30u", &seqno) != 1)) {

		/* RFC 3261 section 24.4.1.   Send a 400 Bad Request if the request is malformed. */
		if (intended_method != SIP_RESPONSE && intended_method != SIP_ACK) {
			transmit_response_using_temp(callid, addr, 1, intended_method,
						     req, "400 Bad Request");
		}
		return NULL;	/* Invalid packet */
	}

	if (sip_cfg.pedanticsipchecking) {
		/* In principle Call-ID's uniquely identify a call, but with a forking SIP proxy
		   we need more to identify a branch - so we have to check branch, from
		   and to tags to identify a call leg.
		   For Asterisk to behave correctly, you need to turn on pedanticsipchecking
		   in sip.conf
		   */
		if (gettag(req, "To", totag, sizeof(totag)))
			req->has_to_tag = 1;	/* Used in handle_request/response */
		gettag(req, "From", fromtag, sizeof(fromtag));

		ast_debug(5, "= Looking for  Call ID: %s (Checking %s) --From tag %s --To-tag %s  \n", callid, req->method==SIP_RESPONSE ? "To" : "From", fromtag, totag);

		/* All messages must always have From: tag */
		if (ast_strlen_zero(fromtag)) {
			ast_debug(5, "%s request has no from tag, dropping callid: %s from: %s\n", sip_methods[req->method].text , callid, from );
			return NULL;
		}
		/* reject requests that must always have a To: tag */
		if (ast_strlen_zero(totag) && (req->method == SIP_ACK || req->method == SIP_BYE || req->method == SIP_INFO )) {
			ast_debug(5, "%s must have a to tag. dropping callid: %s from: %s\n", sip_methods[req->method].text , callid, from );
			return NULL;
		}
	}

	if (!sip_cfg.pedanticsipchecking) {
		struct sip_pvt tmp_dialog = {
			.callid = callid,
		};
		sip_pvt_ptr = ao2_t_find(dialogs, &tmp_dialog, OBJ_POINTER, "ao2_find in dialogs");
		if (sip_pvt_ptr) {  /* well, if we don't find it-- what IS in there? */
			/* Found the call */
			return sip_pvt_ptr;
		}
	} else { /* in pedantic mode! -- do the fancy search */
		struct sip_pvt tmp_dialog = {
			.callid = callid,
		};
		struct match_req_args args = { 0, };
		int found;
		struct ao2_iterator *iterator = ao2_t_callback(dialogs,
			OBJ_POINTER | OBJ_MULTIPLE,
			dialog_find_multiple,
			&tmp_dialog,
			"pedantic ao2_find in dialogs");
		struct sip_via *via = NULL;

		args.method = req->method;
		args.callid = NULL; /* we already matched this. */
		args.totag = totag;
		args.fromtag = fromtag;
		args.seqno = seqno;

		/* If this is a Request, set the Via and Authorization header arguments */
		if (req->method != SIP_RESPONSE) {
			args.ruri = REQ_OFFSET_TO_STR(req, rlPart2);
			via = parse_via(get_header(req, "Via"));
			if (via) {
				args.viasentby = via->sent_by;
				args.viabranch = via->branch;
			}
			if (!ast_strlen_zero(get_header(req, "Authorization")) ||
				!ast_strlen_zero(get_header(req, "Proxy-Authorization"))) {
				args.authentication_present = 1;
			}
		}

		/* Iterate a list of dialogs already matched by Call-id */
		while (iterator && (sip_pvt_ptr = ao2_iterator_next(iterator))) {
			found = match_req_to_dialog(sip_pvt_ptr, &args);

			switch (found) {
			case SIP_REQ_MATCH:
				ao2_iterator_destroy(iterator);
				free_via(via);
				return sip_pvt_ptr; /* return pvt with ref */
			case SIP_REQ_LOOP_DETECTED:
				/* This is likely a forked Request that somehow resulted in us receiving multiple parts of the fork.
			 	* RFC 3261 section 8.2.2.2, Indicate that we want to merge requests by sending a 482 response. */
				transmit_response_using_temp(callid, addr, 1, intended_method, req, "482 (Loop Detected)");
				dialog_unref(sip_pvt_ptr, "pvt did not match incoming SIP msg, unref from search.");
				ao2_iterator_destroy(iterator);
				free_via(via);
				return NULL;
			case SIP_REQ_NOT_MATCH:
			default:
				dialog_unref(sip_pvt_ptr, "pvt did not match incoming SIP msg, unref from search");
			}
		}
		if (iterator) {
			ao2_iterator_destroy(iterator);
		}

		free_via(via);
	} /* end of pedantic mode Request/Reponse to Dialog matching */

	/* See if the method is capable of creating a dialog */
	if (sip_methods[intended_method].can_create == CAN_CREATE_DIALOG) {
		struct sip_pvt *p = NULL;

		if (intended_method == SIP_REFER) {
			/* We do support REFER, but not outside of a dialog yet */
			transmit_response_using_temp(callid, addr, 1, intended_method, req, "603 Declined (no dialog)");
	
		/* Ok, time to create a new SIP dialog object, a pvt */
		} else if (!(p = sip_alloc(callid, addr, 1, intended_method, req)))  {
			/* We have a memory or file/socket error (can't allocate RTP sockets or something) so we're not
				getting a dialog from sip_alloc.

				Without a dialog we can't retransmit and handle ACKs and all that, but at least
				send an error message.

				Sorry, we apologize for the inconvienience
			*/
			transmit_response_using_temp(callid, addr, 1, intended_method, req, "500 Server internal error");
			ast_debug(4, "Failed allocating SIP dialog, sending 500 Server internal error and giving up\n");
		}
		return p; /* can be NULL */
	} else if( sip_methods[intended_method].can_create == CAN_CREATE_DIALOG_UNSUPPORTED_METHOD) {
		/* A method we do not support, let's take it on the volley */
		transmit_response_using_temp(callid, addr, 1, intended_method, req, "501 Method Not Implemented");
		ast_debug(2, "Got a request with unsupported SIP method.\n");
	} else if (intended_method != SIP_RESPONSE && intended_method != SIP_ACK) {
		/* This is a request outside of a dialog that we don't know about */
		transmit_response_using_temp(callid, addr, 1, intended_method, req, "481 Call leg/transaction does not exist");
		ast_debug(2, "That's odd...  Got a request in unknown dialog. Callid %s\n", callid ? callid : "<unknown>");
	}
	/* We do not respond to responses for dialogs that we don't know about, we just drop
	   the session quickly */
	if (intended_method == SIP_RESPONSE)
		ast_debug(2, "That's odd...  Got a response on a call we don't know about. Callid %s\n", callid ? callid : "<unknown>");

	return NULL;
}

/*! \brief create sip_registry object from register=> line in sip.conf and link into reg container */
static int sip_register(const char *value, int lineno)
{
	struct sip_registry *reg;

	if (!(reg = ast_calloc_with_stringfields(1, struct sip_registry, 256))) {
		ast_log(LOG_ERROR, "Out of memory. Can't allocate SIP registry entry\n");
		return -1;
	}

	ast_atomic_fetchadd_int(&regobjs, 1);
	ASTOBJ_INIT(reg);

	if (sip_parse_register_line(reg, default_expiry, value, lineno)) {
		registry_unref(reg, "failure to parse, unref the reg pointer");
		return -1;
	}

	/* set default expiry if necessary */
	if (reg->refresh && !reg->expiry && !reg->configured_expiry) {
		reg->refresh = reg->expiry = reg->configured_expiry = default_expiry;
	}

	/* Add the new registry entry to the list */
	ASTOBJ_CONTAINER_LINK(&regl, reg);

	/* release the reference given by ASTOBJ_INIT. The container has another reference */
	registry_unref(reg, "unref the reg pointer");

	return 0;
}

/*! \brief Parse mwi=> line in sip.conf and add to list */
static int sip_subscribe_mwi(const char *value, int lineno)
{
	struct sip_subscription_mwi *mwi;
	int portnum = 0;
	enum sip_transport transport = SIP_TRANSPORT_UDP;
	char buf[256] = "";
	char *username = NULL, *hostname = NULL, *secret = NULL, *authuser = NULL, *porta = NULL, *mailbox = NULL, *at = NULL;

	if (!value) {
		return -1;
	}

	ast_copy_string(buf, value, sizeof(buf));

	if (!(at = strstr(buf, "@"))) {
		return -1;
	}

	if ((hostname = strrchr(buf, '@'))) {
		*hostname++ = '\0';
		username = buf;
	}

	if ((secret = strchr(username, ':'))) {
		*secret++ = '\0';
		if ((authuser = strchr(secret, ':'))) {
			*authuser++ = '\0';
		}
	}

	if ((mailbox = strchr(hostname, '/'))) {
		*mailbox++ = '\0';
	}

	if (ast_strlen_zero(username) || ast_strlen_zero(hostname) || ast_strlen_zero(mailbox)) {
		ast_log(LOG_WARNING, "Format for MWI subscription is user[:secret[:authuser]]@host[:port]/mailbox at line %d\n", lineno);
		return -1;
	}

	if ((porta = strchr(hostname, ':'))) {
		*porta++ = '\0';
		if (!(portnum = atoi(porta))) {
			ast_log(LOG_WARNING, "%s is not a valid port number at line %d\n", porta, lineno);
			return -1;
		}
	}

	if (!(mwi = ast_calloc_with_stringfields(1, struct sip_subscription_mwi, 256))) {
		return -1;
	}

	ASTOBJ_INIT(mwi);
	ast_string_field_set(mwi, username, username);
	if (secret) {
		ast_string_field_set(mwi, secret, secret);
	}
	if (authuser) {
		ast_string_field_set(mwi, authuser, authuser);
	}
	ast_string_field_set(mwi, hostname, hostname);
	ast_string_field_set(mwi, mailbox, mailbox);
	mwi->resub = -1;
	mwi->portno = portnum;
	mwi->transport = transport;

	ASTOBJ_CONTAINER_LINK(&submwil, mwi);
	ASTOBJ_UNREF(mwi, sip_subscribe_mwi_destroy);

	return 0;
}

static void mark_method_allowed(unsigned int *allowed_methods, enum sipmethod method)
{
	(*allowed_methods) |= (1 << method);
}

static void mark_method_unallowed(unsigned int *allowed_methods, enum sipmethod method)
{
	(*allowed_methods) &= ~(1 << method);
}

/*! \brief Check if method is allowed for a device or a dialog */
static int is_method_allowed(unsigned int *allowed_methods, enum sipmethod method)
{
	return ((*allowed_methods) >> method) & 1;
}

static void mark_parsed_methods(unsigned int *methods, char *methods_str)
{
	char *method;
	for (method = strsep(&methods_str, ","); !ast_strlen_zero(method); method = strsep(&methods_str, ",")) {
		int id = find_sip_method(ast_skip_blanks(method));
		if (id == SIP_UNKNOWN) {
			continue;
		}
		mark_method_allowed(methods, id);
	}
}
/*!
 * \brief parse the Allow header to see what methods the endpoint we
 * are communicating with allows.
 *
 * We parse the allow header on incoming Registrations and save the
 * result to the SIP peer that is registering. When the registration
 * expires, we clear what we know about the peer's allowed methods.
 * When the peer re-registers, we once again parse to see if the
 * list of allowed methods has changed.
 *
 * For peers that do not register, we parse the first message we receive
 * during a call to see what is allowed, and save the information
 * for the duration of the call.
 * \param req The SIP request we are parsing
 * \retval The methods allowed
 */
static unsigned int parse_allowed_methods(struct sip_request *req)
{
	char *allow = ast_strdupa(get_header(req, "Allow"));
	unsigned int allowed_methods = SIP_UNKNOWN;

	if (ast_strlen_zero(allow)) {
		/* I have witnessed that REGISTER requests from Polycom phones do not
		 * place the phone's allowed methods in an Allow header. Instead, they place the
		 * allowed methods in a methods= parameter in the Contact header.
		 */
		char *contact = ast_strdupa(get_header(req, "Contact"));
		char *methods = strstr(contact, ";methods=");

		if (ast_strlen_zero(methods)) {
			/* RFC 3261 states:
			 *
			 * "The absence of an Allow header field MUST NOT be
			 * interpreted to mean that the UA sending the message supports no
			 * methods.   Rather, it implies that the UA is not providing any
			 * information on what methods it supports."
			 *
			 * For simplicity, we'll assume that the peer allows all known
			 * SIP methods if they have no Allow header. We can then clear out the necessary
			 * bits if the peer lets us know that we have sent an unsupported method.
			 */
			return UINT_MAX;
		}
		allow = ast_strip_quoted(methods + 9, "\"", "\"");
	}
	mark_parsed_methods(&allowed_methods, allow);
	return allowed_methods;
}

/*! A wrapper for parse_allowed_methods geared toward sip_pvts
 *
 * This function, in addition to setting the allowed methods for a sip_pvt
 * also will take into account the setting of the SIP_PAGE2_RPID_UPDATE flag.
 *
 * \param pvt The sip_pvt we are setting the allowed_methods for
 * \param req The request which we are parsing
 * \retval The methods alloweded by the sip_pvt
 */
static unsigned int set_pvt_allowed_methods(struct sip_pvt *pvt, struct sip_request *req)
{
	pvt->allowed_methods = parse_allowed_methods(req);
	
	if (ast_test_flag(&pvt->flags[1], SIP_PAGE2_RPID_UPDATE)) {
		mark_method_allowed(&pvt->allowed_methods, SIP_UPDATE);
	}
	pvt->allowed_methods &= ~(pvt->disallowed_methods);

	return pvt->allowed_methods;
}

/*! \brief  Parse multiline SIP headers into one header
	This is enabled if pedanticsipchecking is enabled */
static void lws2sws(struct ast_str *data)
{
	char *msgbuf = data->str;
	int len = ast_str_strlen(data);
	int h = 0, t = 0;
	int lws = 0;

	for (; h < len;) {
		/* Eliminate all CRs */
		if (msgbuf[h] == '\r') {
			h++;
			continue;
		}
		/* Check for end-of-line */
		if (msgbuf[h] == '\n') {
			/* Check for end-of-message */
			if (h + 1 == len)
				break;
			/* Check for a continuation line */
			if (msgbuf[h + 1] == ' ' || msgbuf[h + 1] == '\t') {
				/* Merge continuation line */
				h++;
				continue;
			}
			/* Propagate LF and start new line */
			msgbuf[t++] = msgbuf[h++];
			lws = 0;
			continue;
		}
		if (msgbuf[h] == ' ' || msgbuf[h] == '\t') {
			if (lws) {
				h++;
				continue;
			}
			msgbuf[t++] = msgbuf[h++];
			lws = 1;
			continue;
		}
		msgbuf[t++] = msgbuf[h++];
		if (lws)
			lws = 0;
	}
	msgbuf[t] = '\0';
	data->used = t;
}

/*! \brief Parse a SIP message
	\note this function is used both on incoming and outgoing packets
*/
static int parse_request(struct sip_request *req)
{
	char *c = req->data->str;
	ptrdiff_t *dst = req->header;
	int i = 0, lim = SIP_MAX_HEADERS - 1;
	unsigned int skipping_headers = 0;
	ptrdiff_t current_header_offset = 0;
	char *previous_header = "";

	req->header[0] = 0;
	req->headers = -1;	/* mark that we are working on the header */
	for (; *c; c++) {
		if (*c == '\r') {		/* remove \r */
			*c = '\0';
		} else if (*c == '\n') { 	/* end of this line */
			*c = '\0';
			current_header_offset = (c + 1) - req->data->str;
			previous_header = req->data->str + dst[i];
			if (skipping_headers) {
				/* check to see if this line is blank; if so, turn off
				   the skipping flag, so the next line will be processed
				   as a body line */
				if (ast_strlen_zero(previous_header)) {
					skipping_headers = 0;
				}
				dst[i] = current_header_offset; /* record start of next line */
				continue;
			}
			if (sipdebug) {
				ast_debug(4, "%7s %2d [%3d]: %s\n",
					  req->headers < 0 ? "Header" : "Body",
					  i, (int) strlen(previous_header), previous_header);
			}
			if (ast_strlen_zero(previous_header) && req->headers < 0) {
				req->headers = i;	/* record number of header lines */
				dst = req->line;	/* start working on the body */
				i = 0;
				lim = SIP_MAX_LINES - 1;
			} else {	/* move to next line, check for overflows */
				if (i++ == lim) {
					/* if we're processing headers, then skip any remaining
					   headers and move on to processing the body, otherwise
					   we're done */
					if (req->headers != -1) {
						break;
					} else {
						req->headers = i;
						dst = req->line;
						i = 0;
						lim = SIP_MAX_LINES - 1;
						skipping_headers = 1;
					}
				}
			}
			dst[i] = current_header_offset; /* record start of next line */
		}
	}

	/* Check for last header or body line without CRLF. The RFC for SDP requires CRLF,
	   but since some devices send without, we'll be generous in what we accept. However,
	   if we've already reached the maximum number of lines for portion of the message
	   we were parsing, we can't accept any more, so just ignore it.
	*/
	previous_header = req->data->str + dst[i];
	if ((i < lim) && !ast_strlen_zero(previous_header)) {
		if (sipdebug) {
			ast_debug(4, "%7s %2d [%3d]: %s\n",
				  req->headers < 0 ? "Header" : "Body",
				  i, (int) strlen(previous_header), previous_header );
		}
		i++;
	}

	/* update count of header or body lines */
	if (req->headers >= 0) {	/* we are in the body */
		req->lines = i;
	} else {			/* no body */
		req->headers = i;
		req->lines = 0;
		/* req->data->used will be a NULL byte */
		req->line[0] = ast_str_strlen(req->data);
	}

	if (*c) {
		ast_log(LOG_WARNING, "Too many lines, skipping <%s>\n", c);
	}

	/* Split up the first line parts */
	return determine_firstline_parts(req);
}

/*!
  \brief Determine whether a SIP message contains an SDP in its body
  \param req the SIP request to process
  \return 1 if SDP found, 0 if not found

  Also updates req->sdp_start and req->sdp_count to indicate where the SDP
  lives in the message body.
*/
static int find_sdp(struct sip_request *req)
{
	const char *content_type;
	const char *content_length;
	const char *search;
	char *boundary;
	unsigned int x;
	int boundaryisquoted = FALSE;
	int found_application_sdp = FALSE;
	int found_end_of_headers = FALSE;

	content_length = get_header(req, "Content-Length");

	if (!ast_strlen_zero(content_length)) {
		if (sscanf(content_length, "%30u", &x) != 1) {
			ast_log(LOG_WARNING, "Invalid Content-Length: %s\n", content_length);
			return 0;
		}

		/* Content-Length of zero means there can't possibly be an
		   SDP here, even if the Content-Type says there is */
		if (x == 0)
			return 0;
	}

	content_type = get_header(req, "Content-Type");

	/* if the body contains only SDP, this is easy */
	if (!strncasecmp(content_type, "application/sdp", 15)) {
		req->sdp_start = 0;
		req->sdp_count = req->lines;
		return req->lines ? 1 : 0;
	}

	/* if it's not multipart/mixed, there cannot be an SDP */
	if (strncasecmp(content_type, "multipart/mixed", 15))
		return 0;

	/* if there is no boundary marker, it's invalid */
	if ((search = strcasestr(content_type, ";boundary=")))
		search += 10;
	else if ((search = strcasestr(content_type, "; boundary=")))
		search += 11;
	else
		return 0;

	if (ast_strlen_zero(search))
		return 0;

	/* If the boundary is quoted with ", remove quote */
	if (*search == '\"')  {
		search++;
		boundaryisquoted = TRUE;
	}

	/* make a duplicate of the string, with two extra characters
	   at the beginning */
	boundary = ast_strdupa(search - 2);
	boundary[0] = boundary[1] = '-';
	/* Remove final quote */
	if (boundaryisquoted)
		boundary[strlen(boundary) - 1] = '\0';

	/* search for the boundary marker, the empty line delimiting headers from
	   sdp part and the end boundry if it exists */

	for (x = 0; x < (req->lines); x++) {
		const char *line = REQ_OFFSET_TO_STR(req, line[x]);
		if (!strncasecmp(line, boundary, strlen(boundary))){
			if (found_application_sdp && found_end_of_headers) {
				req->sdp_count = (x - 1) - req->sdp_start;
				return 1;
			}
			found_application_sdp = FALSE;
		}
		if (!strcasecmp(line, "Content-Type: application/sdp"))
			found_application_sdp = TRUE;
		
		if (ast_strlen_zero(line)) {
			if (found_application_sdp && !found_end_of_headers){
				req->sdp_start = x;
				found_end_of_headers = TRUE;
			}
		}
	}
	if (found_application_sdp && found_end_of_headers) {
		req->sdp_count = x - req->sdp_start;
		return TRUE;
	}
	return FALSE;
}

/*! \brief Change hold state for a call */
static void change_hold_state(struct sip_pvt *dialog, struct sip_request *req, int holdstate, int sendonly)
{
	if (sip_cfg.notifyhold && (!holdstate || !ast_test_flag(&dialog->flags[1], SIP_PAGE2_CALL_ONHOLD)))
		sip_peer_hold(dialog, holdstate);
	if (sip_cfg.callevents)
		manager_event(EVENT_FLAG_CALL, "Hold",
			      "Status: %s\r\n"
			      "Channel: %s\r\n"
			      "Uniqueid: %s\r\n",
			      holdstate ? "On" : "Off",
			      dialog->owner->name,
			      dialog->owner->uniqueid);
	append_history(dialog, holdstate ? "Hold" : "Unhold", "%s", req->data->str);
	if (!holdstate) {	/* Put off remote hold */
		ast_clear_flag(&dialog->flags[1], SIP_PAGE2_CALL_ONHOLD);	/* Clear both flags */
		return;
	}
	/* No address for RTP, we're on hold */

	if (sendonly == 1)	/* One directional hold (sendonly/recvonly) */
		ast_set_flag(&dialog->flags[1], SIP_PAGE2_CALL_ONHOLD_ONEDIR);
	else if (sendonly == 2)	/* Inactive stream */
		ast_set_flag(&dialog->flags[1], SIP_PAGE2_CALL_ONHOLD_INACTIVE);
	else
		ast_set_flag(&dialog->flags[1], SIP_PAGE2_CALL_ONHOLD_ACTIVE);
	return;
}


static int get_ip_and_port_from_sdp(struct sip_request *req, const enum media_type media, struct ast_sockaddr *addr)
{
	const char *m;
	const char *c;
	int miterator = req->sdp_start;
	int citerator = req->sdp_start;
	int x = 0;
	int numberofports;
	int len;
	int af;
	char proto[4], host[258] = ""; /*Initialize to empty so we will know if we have any input */

	c = get_sdp_iterate(&citerator, req, "c");
	if (sscanf(c, "IN %3s %256s", proto, host) != 2) {
			ast_log(LOG_WARNING, "Invalid host in c= line, '%s'\n", c);
			/* Continue since there may be a valid host in a c= line specific to the audio stream */
	}
	/* We only want the m and c lines for audio */
	for (m = get_sdp_iterate(&miterator, req, "m"); !ast_strlen_zero(m); m = get_sdp_iterate(&miterator, req, "m")) {
		if ((media == SDP_AUDIO && ((sscanf(m, "audio %30u/%30u RTP/AVP %n", &x, &numberofports, &len) == 2 && len > 0) ||
		    (sscanf(m, "audio %30u RTP/AVP %n", &x, &len) == 1 && len > 0))) ||
			(media == SDP_VIDEO && ((sscanf(m, "video %30u/%30u RTP/AVP %n", &x, &numberofports, &len) == 2 && len > 0) ||
		    (sscanf(m, "video %30u RTP/AVP %n", &x, &len) == 1 && len > 0)))) {
			/* See if there's a c= line for this media stream.
			 * XXX There is no guarantee that we'll be grabbing the c= line for this
			 * particular media stream here. However, this is the same logic used in process_sdp.
			 */
			c = get_sdp_iterate(&citerator, req, "c");
			if (!ast_strlen_zero(c)) {
				sscanf(c, "IN %3s %256s", proto, host);
			}
			break;
		}
	}

	if (!strcmp("IP4", proto)) {
		af = AF_INET;
	} else if (!strcmp("IP6", proto)) {
		af = AF_INET6;
	} else {
		ast_log(LOG_WARNING, "Unknown protocol '%s'.\n", proto);
		return -1;
	}

	if (ast_strlen_zero(host) || x == 0) {
		ast_log(LOG_WARNING, "Failed to read an alternate host or port in SDP. Expect %s problems\n", media == SDP_AUDIO ? "audio" : "video");
		return -1;
	}

	if (ast_sockaddr_resolve_first_af(addr, host, 0, af)) {
		ast_log(LOG_WARNING, "Could not look up IP address of alternate hostname. Expect %s problems\n", media == SDP_AUDIO? "audio" : "video");
		return -1;
	}

	return 0;
}

/*! \internal
 * \brief Returns whether or not the address is null or ANY / unspecified (0.0.0.0 or ::)
 * \retval TRUE if the address is null or any
 * \retval FALSE if the address it not null or any
 * \note In some circumstances, calls should be placed on hold if either of these conditions exist.
 */
static int sockaddr_is_null_or_any(const struct ast_sockaddr *addr)
{
	return ast_sockaddr_isnull(addr) || ast_sockaddr_is_any(addr);
}

/*! \brief Process SIP SDP offer, select formats and activate RTP channels
	If offer is rejected, we will not change any properties of the call
 	Return 0 on success, a negative value on errors.
	Must be called after find_sdp().
*/
static int process_sdp(struct sip_pvt *p, struct sip_request *req, int t38action)
{
	/* Iterators for SDP parsing */
	int start = req->sdp_start;
	int next = start;
	int iterator = start;

	/* Temporary vars for SDP parsing */
	char type = '\0';
	const char *value = NULL;
	const char *m = NULL;           /* SDP media offer */
	const char *nextm = NULL;
	int len = -1;

	/* Host information */
	struct ast_sockaddr sessionsa;
	struct ast_sockaddr audiosa;
	struct ast_sockaddr videosa;
	struct ast_sockaddr textsa;
	struct ast_sockaddr imagesa;
	struct ast_sockaddr *sa = NULL;	/*!< RTP Audio host IP */
	struct ast_sockaddr *vsa = NULL;	/*!< RTP video host IP */
	struct ast_sockaddr *tsa = NULL;	/*!< RTP text host IP */
	struct ast_sockaddr *isa = NULL;     /*!< UDPTL host ip */
 	int portno = -1;		/*!< RTP Audio port number */
 	int vportno = -1;		/*!< RTP Video port number */
	int tportno = -1;		/*!< RTP Text port number */
	int udptlportno = -1;		/*!< UDPTL Image port number */

	/* Peer capability is the capability in the SDP, non codec is RFC2833 DTMF (101) */	
	format_t peercapability = 0, vpeercapability = 0, tpeercapability = 0;
	int peernoncodeccapability = 0, vpeernoncodeccapability = 0, tpeernoncodeccapability = 0;

	struct ast_rtp_codecs newaudiortp, newvideortp, newtextrtp;
	format_t newjointcapability;				/* Negotiated capability */
	format_t newpeercapability;
	int newnoncodeccapability;

	const char *codecs;
	int codec;

	/* SRTP */
	int secure_audio = FALSE;
	int secure_video = FALSE;

	/* Others */
	int sendonly = -1;
	int vsendonly = -1;
	int numberofports;
	int numberofmediastreams = 0;
	int last_rtpmap_codec = 0;
	int red_data_pt[10];		/* For T.140 red */
	int red_num_gen = 0;		/* For T.140 red */
	char red_fmtp[100] = "empty";	/* For T.140 red */
	int debug = sip_debug_test_pvt(p);

	/* START UNKNOWN */
	char buf[SIPBUFSIZE];
	/* END UNKNOWN */

	/* Initial check */
	if (!p->rtp) {
		ast_log(LOG_ERROR, "Got SDP but have no RTP session allocated.\n");
		return -1;
	}

	/* Make sure that the codec structures are all cleared out */
	ast_rtp_codecs_payloads_clear(&newaudiortp, NULL);
	ast_rtp_codecs_payloads_clear(&newvideortp, NULL);
	ast_rtp_codecs_payloads_clear(&newtextrtp, NULL);

	/* Update our last rtprx when we receive an SDP, too */
	p->lastrtprx = p->lastrtptx = time(NULL); /* XXX why both ? */

	memset(p->offered_media, 0, sizeof(p->offered_media));


	/* default: novideo and notext set */
	p->novideo = TRUE;
	p->notext = TRUE;

	if (p->vrtp) {
		ast_rtp_codecs_payloads_clear(&newvideortp, NULL);
	}

	if (p->trtp) {
		ast_rtp_codecs_payloads_clear(&newtextrtp, NULL);
	}

	/* Scan for the first media stream (m=) line to limit scanning of globals */
	nextm = get_sdp_iterate(&next, req, "m");
	if (ast_strlen_zero(nextm)) {
		ast_log(LOG_WARNING, "Insufficient information for SDP (m= not found)\n");
 		return -1;
 	}

	/* Scan session level SDP parameters (lines before first media stream) */
	while ((type = get_sdp_line(&iterator, next - 1, req, &value)) != '\0') {
		int processed = FALSE;
		switch (type) {
		case 'o':
			/* If we end up receiving SDP that doesn't actually modify the session we don't want to treat this as a fatal
			 * error. We just want to ignore the SDP and let the rest of the packet be handled as normal.
			 */
			if (!process_sdp_o(value, p))
				return (p->session_modify == FALSE) ? 0 : -1;
			break;
		case 'c':
			if (process_sdp_c(value, &sessionsa)) {
				processed = TRUE;
				sa = &sessionsa;
				vsa = sa;
				tsa = sa;
				isa = sa;
			}
			break;
		case 'a':
			if (process_sdp_a_sendonly(value, &sendonly)) {
				processed = TRUE;
				vsendonly = sendonly;
			}
			else if (process_sdp_a_audio(value, p, &newaudiortp, &last_rtpmap_codec))
				processed = TRUE;
			else if (process_sdp_a_video(value, p, &newvideortp, &last_rtpmap_codec))
				processed = TRUE;
			else if (process_sdp_a_text(value, p, &newtextrtp, red_fmtp, &red_num_gen, red_data_pt, &last_rtpmap_codec))
				processed = TRUE;
			else if (process_sdp_a_image(value, p))
				processed = TRUE;
			break;
		}

		ast_debug(3, "Processing session-level SDP %c=%s... %s\n", type, value, (processed == TRUE)? "OK." : "UNSUPPORTED.");
 	}



	/* Scan media stream (m=) specific parameters loop */
	while (!ast_strlen_zero(nextm)) {
		int audio = FALSE;
		int video = FALSE;
		int image = FALSE;
		int text = FALSE;
		int processed_crypto = FALSE;
		char protocol[5] = {0,};
		int x;

		numberofports = 1;
		len = -1;
		start = next;
		m = nextm;
		iterator = next;
		nextm = get_sdp_iterate(&next, req, "m");

		/* Search for audio media definition */
		if ((sscanf(m, "audio %30u/%30u RTP/%4s %n", &x, &numberofports, protocol, &len) == 3 && len > 0) ||
		    (sscanf(m, "audio %30u RTP/%4s %n", &x, protocol, &len) == 2 && len > 0)) {
			if (x == 0) {
				ast_log(LOG_WARNING, "ignoring 'audio' media offer because port number is zero\n");
				continue;
			}
			if (!strcmp(protocol, "SAVP")) {
				secure_audio = 1;
			} else if (strcmp(protocol, "AVP")) {
				ast_log(LOG_WARNING, "unknown SDP media protocol in offer: %s\n", protocol);
				continue;
			}
			if (p->offered_media[SDP_AUDIO].order_offered) {
				ast_log(LOG_WARNING, "Multiple audio streams are not supported\n");
				return -3;
			}
			audio = TRUE;
			p->offered_media[SDP_AUDIO].order_offered = ++numberofmediastreams;
			portno = x;

			/* Scan through the RTP payload types specified in a "m=" line: */
			codecs = m + len;
			ast_copy_string(p->offered_media[SDP_AUDIO].codecs, codecs, sizeof(p->offered_media[SDP_AUDIO].codecs));
			for (; !ast_strlen_zero(codecs); codecs = ast_skip_blanks(codecs + len)) {
				if (sscanf(codecs, "%30u%n", &codec, &len) != 1) {
					ast_log(LOG_WARNING, "Error in codec string '%s'\n", codecs);
					return -1;
				}
				if (debug)
					ast_verbose("Found RTP audio format %d\n", codec);

				ast_rtp_codecs_payloads_set_m_type(&newaudiortp, NULL, codec);
			}
		/* Search for video media definition */
		} else if ((sscanf(m, "video %30u/%30u RTP/%4s %n", &x, &numberofports, protocol, &len) == 3 && len > 0) ||
			   (sscanf(m, "video %30u RTP/%4s %n", &x, protocol, &len) == 2 && len > 0)) {
			if (x == 0) {
				ast_log(LOG_WARNING, "ignoring 'video' media offer because port number is zero\n");
				continue;
			}
			if (!strcmp(protocol, "SAVP")) {
				secure_video = 1;
			} else if (strcmp(protocol, "AVP")) {
				ast_log(LOG_WARNING, "unknown SDP media protocol in offer: %s\n", protocol);
				continue;
			}
			if (p->offered_media[SDP_VIDEO].order_offered) {
				ast_log(LOG_WARNING, "Multiple video streams are not supported\n");
				return -3;
			}
			video = TRUE;
			p->novideo = FALSE;
			p->offered_media[SDP_VIDEO].order_offered = ++numberofmediastreams;
			vportno = x;

			/* Scan through the RTP payload types specified in a "m=" line: */
			codecs = m + len;
			ast_copy_string(p->offered_media[SDP_VIDEO].codecs, codecs, sizeof(p->offered_media[SDP_VIDEO].codecs));
			for (; !ast_strlen_zero(codecs); codecs = ast_skip_blanks(codecs + len)) {
				if (sscanf(codecs, "%30u%n", &codec, &len) != 1) {
					ast_log(LOG_WARNING, "Error in codec string '%s'\n", codecs);
					return -1;
				}
				if (debug)
					ast_verbose("Found RTP video format %d\n", codec);
				ast_rtp_codecs_payloads_set_m_type(&newvideortp, NULL, codec);
			}
		/* Search for text media definition */
		} else if ((sscanf(m, "text %30u/%30u RTP/AVP %n", &x, &numberofports, &len) == 2 && len > 0) ||
			   (sscanf(m, "text %30u RTP/AVP %n", &x, &len) == 1 && len > 0)) {
			if (x == 0) {
				ast_log(LOG_WARNING, "ignoring 'text' media offer because port number is zero\n");
				continue;
			}
			if (p->offered_media[SDP_TEXT].order_offered) {
				ast_log(LOG_WARNING, "Multiple text streams are not supported\n");
				return -3;
			}
			text = TRUE;
			p->notext = FALSE;
			p->offered_media[SDP_TEXT].order_offered = ++numberofmediastreams;
			tportno = x;

			/* Scan through the RTP payload types specified in a "m=" line: */
			codecs = m + len;
			ast_copy_string(p->offered_media[SDP_TEXT].codecs, codecs, sizeof(p->offered_media[SDP_TEXT].codecs));
			for (; !ast_strlen_zero(codecs); codecs = ast_skip_blanks(codecs + len)) {
				if (sscanf(codecs, "%30u%n", &codec, &len) != 1) {
					ast_log(LOG_WARNING, "Error in codec string '%s'\n", codecs);
					return -1;
				}
				if (debug)
					ast_verbose("Found RTP text format %d\n", codec);
				ast_rtp_codecs_payloads_set_m_type(&newtextrtp, NULL, codec);
			}
		/* Search for image media definition */
		} else if (((sscanf(m, "image %30u udptl t38%n", &x, &len) == 1 && len > 0) ||
			    (sscanf(m, "image %30u UDPTL t38%n", &x, &len) == 1 && len > 0))) {
			if (x == 0) {
				ast_log(LOG_WARNING, "ignoring 'image' media offer because port number is zero\n");
				continue;
			}
			if (initialize_udptl(p)) {
				continue;
			}

			if (p->offered_media[SDP_IMAGE].order_offered) {
				ast_log(LOG_WARNING, "Multiple T.38 streams are not supported\n");
				return -3;
			}
			image = TRUE;
			if (debug)
				ast_verbose("Got T.38 offer in SDP in dialog %s\n", p->callid);
			p->offered_media[SDP_IMAGE].order_offered = ++numberofmediastreams;
			udptlportno = x;

			if (p->t38.state != T38_ENABLED) {
				memset(&p->t38.their_parms, 0, sizeof(p->t38.their_parms));

				/* default EC to none, the remote end should
				 * respond with the EC they want to use */
				ast_udptl_set_error_correction_scheme(p->udptl, UDPTL_ERROR_CORRECTION_NONE);
			}
		} else {
			ast_log(LOG_WARNING, "Unsupported SDP media type in offer: %s\n", m);
			continue;
		}

		/* Check for number of ports */
		if (numberofports > 1)
			ast_log(LOG_WARNING, "SDP offered %d ports for media, not supported by Asterisk. Will try anyway...\n", numberofports);
		
		/* Media stream specific parameters */
		while ((type = get_sdp_line(&iterator, next - 1, req, &value)) != '\0') {
			int processed = FALSE;

			switch (type) {
			case 'c':
				if (audio) {
					if (process_sdp_c(value, &audiosa)) {
						processed = TRUE;
						sa = &audiosa;
					}
				} else if (video) {
					if (process_sdp_c(value, &videosa)) {
						processed = TRUE;
						vsa = &videosa;
					}
				} else if (text) {
					if (process_sdp_c(value, &textsa)) {
						processed = TRUE;
						tsa = &textsa;
					}
				} else if (image) {
					if (process_sdp_c(value, &imagesa)) {
						processed = TRUE;
						isa = &imagesa;
					}
				}
				break;
			case 'a':
				/* Audio specific scanning */
				if (audio) {
					if (process_sdp_a_sendonly(value, &sendonly)) {
						processed = TRUE;
					} else if (!processed_crypto && process_crypto(p, p->rtp, &p->srtp, value)) {
						processed_crypto = TRUE;
						processed = TRUE;
					} else if (process_sdp_a_audio(value, p, &newaudiortp, &last_rtpmap_codec)) {
						processed = TRUE;
					}
				}
				/* Video specific scanning */
				else if (video) {
					if (process_sdp_a_sendonly(value, &vsendonly)) {
						processed = TRUE;
					} else if (!processed_crypto && process_crypto(p, p->vrtp, &p->vsrtp, value)) {
						processed_crypto = TRUE;
						processed = TRUE;
					} else if (process_sdp_a_video(value, p, &newvideortp, &last_rtpmap_codec)) {
						processed = TRUE;
					}
				}
				/* Text (T.140) specific scanning */
				else if (text) {
					if (process_sdp_a_text(value, p, &newtextrtp, red_fmtp, &red_num_gen, red_data_pt, &last_rtpmap_codec)) {
						processed = TRUE;
					} else if (!processed_crypto && process_crypto(p, p->trtp, &p->tsrtp, value)) {
						processed_crypto = TRUE;
						processed = TRUE;
					}
				}
				/* Image (T.38 FAX) specific scanning */
				else if (image) {
					if (process_sdp_a_image(value, p))
						processed = TRUE;
				}
				break;
			}

			ast_debug(3, "Processing media-level (%s) SDP %c=%s... %s\n",
					(audio == TRUE)? "audio" : (video == TRUE)? "video" : "image",
					type, value,
					(processed == TRUE)? "OK." : "UNSUPPORTED.");
		}
	}


	/* Sanity checks */
	if (!sa && !vsa && !tsa && !isa) {
		ast_log(LOG_WARNING, "Insufficient information in SDP (c=)...\n");
		return -1;
	}

	if (portno == -1 && vportno == -1 && udptlportno == -1  && tportno == -1) {
		/* No acceptable offer found in SDP  - we have no ports */
		/* Do not change RTP or VRTP if this is a re-invite */
		ast_log(LOG_WARNING, "Failing due to no acceptable offer found\n");
		return -2;
	}

	if (secure_audio && !(p->srtp && (ast_test_flag(p->srtp, SRTP_CRYPTO_OFFER_OK)))) {
		ast_log(LOG_WARNING, "Can't provide secure audio requested in SDP offer\n");
		return -4;
	}

	if (!secure_audio && p->srtp) {
		ast_log(LOG_WARNING, "We are requesting SRTP, but they responded without it!\n");
		return -4;
	}

	if (secure_video && !(p->vsrtp && (ast_test_flag(p->vsrtp, SRTP_CRYPTO_OFFER_OK)))) {
		ast_log(LOG_WARNING, "Can't provide secure video requested in SDP offer\n");
		return -4;
	}

	if (!p->novideo && !secure_video && p->vsrtp) {
		ast_log(LOG_WARNING, "We are requesting SRTP, but they responded without it!\n");
		return -4;
	}

	if (!(secure_audio || secure_video) && ast_test_flag(&p->flags[1], SIP_PAGE2_USE_SRTP)) {
		ast_log(LOG_WARNING, "Matched device setup to use SRTP, but request was not!\n");
		return -4;
	}

	if (udptlportno == -1) {
		change_t38_state(p, T38_DISABLED);
	}

	/* Now gather all of the codecs that we are asked for: */
	ast_rtp_codecs_payload_formats(&newaudiortp, &peercapability, &peernoncodeccapability);
	ast_rtp_codecs_payload_formats(&newvideortp, &vpeercapability, &vpeernoncodeccapability);
	ast_rtp_codecs_payload_formats(&newtextrtp, &tpeercapability, &tpeernoncodeccapability);

	newjointcapability = p->capability & (peercapability | vpeercapability | tpeercapability);
	newpeercapability = (peercapability | vpeercapability | tpeercapability);
	newnoncodeccapability = p->noncodeccapability & peernoncodeccapability;

	if (debug) {
		/* shame on whoever coded this.... */
		char s1[SIPBUFSIZE], s2[SIPBUFSIZE], s3[SIPBUFSIZE], s4[SIPBUFSIZE], s5[SIPBUFSIZE];

		ast_verbose("Capabilities: us - %s, peer - audio=%s/video=%s/text=%s, combined - %s\n",
			    ast_getformatname_multiple(s1, SIPBUFSIZE, p->capability),
			    ast_getformatname_multiple(s2, SIPBUFSIZE, peercapability),
			    ast_getformatname_multiple(s3, SIPBUFSIZE, vpeercapability),
			    ast_getformatname_multiple(s4, SIPBUFSIZE, tpeercapability),
			    ast_getformatname_multiple(s5, SIPBUFSIZE, newjointcapability));
	}
	if (debug) {
		struct ast_str *s1 = ast_str_alloca(SIPBUFSIZE);
		struct ast_str *s2 = ast_str_alloca(SIPBUFSIZE);
		struct ast_str *s3 = ast_str_alloca(SIPBUFSIZE);

		ast_verbose("Non-codec capabilities (dtmf): us - %s, peer - %s, combined - %s\n",
			    ast_rtp_lookup_mime_multiple2(s1, p->noncodeccapability, 0, 0),
			    ast_rtp_lookup_mime_multiple2(s2, peernoncodeccapability, 0, 0),
			    ast_rtp_lookup_mime_multiple2(s3, newnoncodeccapability, 0, 0));
	}
	if (!newjointcapability && udptlportno == -1) {
		ast_log(LOG_NOTICE, "No compatible codecs, not accepting this offer!\n");
		/* Do NOT Change current setting */
		return -1;
	}

	if (portno != -1 || vportno != -1 || tportno != -1) {
		/* We are now ready to change the sip session and p->rtp and p->vrtp with the offered codecs, since
		   they are acceptable */
		p->jointcapability = newjointcapability;                /* Our joint codec profile for this call */
		p->peercapability = newpeercapability;                  /* The other sides capability in latest offer */
		p->jointnoncodeccapability = newnoncodeccapability;     /* DTMF capabilities */
	
		/* respond with single most preferred joint codec, limiting the other side's choice */
		if (ast_test_flag(&p->flags[1], SIP_PAGE2_PREFERRED_CODEC)) {
			p->jointcapability = ast_codec_choose(&p->prefs, p->jointcapability, 1);
		}
	}

	/* Setup audio address and port */
	if (p->rtp) {
		if (portno > 0) {
			ast_sockaddr_set_port(sa, portno);
			ast_rtp_instance_set_remote_address(p->rtp, sa);
			if (debug) {
				ast_verbose("Peer audio RTP is at port %s\n",
					    ast_sockaddr_stringify(sa));
			}

			ast_rtp_codecs_payloads_copy(&newaudiortp, ast_rtp_instance_get_codecs(p->rtp), p->rtp);
			/* Ensure RTCP is enabled since it may be inactive
			   if we're coming back from a T.38 session */
			ast_rtp_instance_set_prop(p->rtp, AST_RTP_PROPERTY_RTCP, 1);
			/* Ensure audio RTCP reads are enabled */
			if (p->owner) {
				ast_channel_set_fd(p->owner, 1, ast_rtp_instance_fd(p->rtp, 1));
			}

			if (ast_test_flag(&p->flags[0], SIP_DTMF) == SIP_DTMF_AUTO) {
				ast_clear_flag(&p->flags[0], SIP_DTMF);
				if (newnoncodeccapability & AST_RTP_DTMF) {
					/* XXX Would it be reasonable to drop the DSP at this point? XXX */
					ast_set_flag(&p->flags[0], SIP_DTMF_RFC2833);
					/* Since RFC2833 is now negotiated we need to change some properties of the RTP stream */
					ast_rtp_instance_set_prop(p->rtp, AST_RTP_PROPERTY_DTMF, 1);
					ast_rtp_instance_set_prop(p->rtp, AST_RTP_PROPERTY_DTMF_COMPENSATE, ast_test_flag(&p->flags[1], SIP_PAGE2_RFC2833_COMPENSATE));
				} else {
					ast_set_flag(&p->flags[0], SIP_DTMF_INBAND);
				}
			}
		} else if (udptlportno > 0) {
			if (debug)
				ast_verbose("Got T.38 Re-invite without audio. Keeping RTP active during T.38 session.\n");
			/* Prevent audio RTCP reads */
			if (p->owner) {
				ast_channel_set_fd(p->owner, 1, -1);
			}
			/* Silence RTCP while audio RTP is inactive */
			ast_rtp_instance_set_prop(p->rtp, AST_RTP_PROPERTY_RTCP, 0);
		} else {
			ast_rtp_instance_stop(p->rtp);
			if (debug)
				ast_verbose("Peer doesn't provide audio\n");
		}
	}

	/* Setup video address and port */
	if (p->vrtp) {
		if (vportno > 0) {
			ast_sockaddr_set_port(vsa, vportno);
			ast_rtp_instance_set_remote_address(p->vrtp, vsa);
			if (debug) {
				ast_verbose("Peer video RTP is at port %s\n",
					    ast_sockaddr_stringify(vsa));
			}
			ast_rtp_codecs_payloads_copy(&newvideortp, ast_rtp_instance_get_codecs(p->vrtp), p->vrtp);
		} else {
			ast_rtp_instance_stop(p->vrtp);
			if (debug)
				ast_verbose("Peer doesn't provide video\n");
		}
	}

	/* Setup text address and port */
	if (p->trtp) {
		if (tportno > 0) {
			ast_sockaddr_set_port(tsa, tportno);
			ast_rtp_instance_set_remote_address(p->trtp, tsa);
			if (debug) {
				ast_verbose("Peer T.140 RTP is at port %s\n",
					    ast_sockaddr_stringify(tsa));
			}
			if ((p->jointcapability & AST_FORMAT_T140RED)) {
				p->red = 1;
				ast_rtp_red_init(p->trtp, 300, red_data_pt, 2);
			} else {
				p->red = 0;
			}
			ast_rtp_codecs_payloads_copy(&newtextrtp, ast_rtp_instance_get_codecs(p->trtp), p->trtp);
		} else {
			ast_rtp_instance_stop(p->trtp);
			if (debug)
				ast_verbose("Peer doesn't provide T.140\n");
		}
	}
	/* Setup image address and port */
	if (p->udptl) {
		if (udptlportno > 0) {
			if (ast_test_flag(&p->flags[1], SIP_PAGE2_SYMMETRICRTP) && ast_test_flag(&p->flags[1], SIP_PAGE2_UDPTL_DESTINATION)) {
				ast_rtp_instance_get_remote_address(p->rtp, isa);
				if (!ast_sockaddr_isnull(isa) && debug) {
					ast_debug(1, "Peer T.38 UDPTL is set behind NAT and with destination, destination address now %s\n", ast_sockaddr_stringify(isa));
				}
			}
			ast_sockaddr_set_port(isa, udptlportno);
			ast_udptl_set_peer(p->udptl, isa);
			if (debug)
				ast_debug(1,"Peer T.38 UDPTL is at port %s\n", ast_sockaddr_stringify(isa));

			/* verify the far max ifp can be calculated. this requires far max datagram to be set. */
			if (!ast_udptl_get_far_max_datagram(p->udptl)) {
				/* setting to zero will force a default if none was provided by the SDP */
				ast_udptl_set_far_max_datagram(p->udptl, 0);
			}

			/* Remote party offers T38, we need to update state */
			if ((t38action == SDP_T38_ACCEPT) &&
			    (p->t38.state == T38_LOCAL_REINVITE)) {
				change_t38_state(p, T38_ENABLED);
			} else if ((t38action == SDP_T38_INITIATE) &&
				   p->owner && p->lastinvite) {
				change_t38_state(p, T38_PEER_REINVITE); /* T38 Offered in re-invite from remote party */
				/* If fax detection is enabled then send us off to the fax extension */
				if (ast_test_flag(&p->flags[1], SIP_PAGE2_FAX_DETECT_T38)) {
					ast_channel_lock(p->owner);
					if (strcmp(p->owner->exten, "fax")) {
						const char *target_context = S_OR(p->owner->macrocontext, p->owner->context);
						ast_channel_unlock(p->owner);
						if (ast_exists_extension(p->owner, target_context, "fax", 1,
							S_COR(p->owner->caller.id.number.valid, p->owner->caller.id.number.str, NULL))) {
							ast_verbose(VERBOSE_PREFIX_2 "Redirecting '%s' to fax extension due to peer T.38 re-INVITE\n", p->owner->name);
							pbx_builtin_setvar_helper(p->owner, "FAXEXTEN", p->owner->exten);
							if (ast_async_goto(p->owner, target_context, "fax", 1)) {
								ast_log(LOG_NOTICE, "Failed to async goto '%s' into fax of '%s'\n", p->owner->name, target_context);
							}
						} else {
							ast_log(LOG_NOTICE, "T.38 re-INVITE detected but no fax extension\n");
						}
					} else {
						ast_channel_unlock(p->owner);
					}
				}
			}
		} else {
			change_t38_state(p, T38_DISABLED);
			ast_udptl_stop(p->udptl);
			if (debug)
				ast_debug(1, "Peer doesn't provide T.38 UDPTL\n");
		}
	}

	if ((portno == -1) && (p->t38.state != T38_DISABLED)) {
		ast_debug(3, "Have T.38 but no audio, accepting offer anyway\n");
		return 0;
        }

	/* Ok, we're going with this offer */
	ast_debug(2, "We're settling with these formats: %s\n", ast_getformatname_multiple(buf, SIPBUFSIZE, p->jointcapability));

	if (!p->owner) 	/* There's no open channel owning us so we can return here. For a re-invite or so, we proceed */
		return 0;

	ast_debug(4, "We have an owner, now see if we need to change this call\n");

	if (!(p->owner->nativeformats & p->jointcapability) && (p->jointcapability & AST_FORMAT_AUDIO_MASK)) {
		if (debug) {
			char s1[SIPBUFSIZE], s2[SIPBUFSIZE];
			ast_debug(1, "Oooh, we need to change our audio formats since our peer supports only %s and not %s\n",
				ast_getformatname_multiple(s1, SIPBUFSIZE, p->jointcapability),
				ast_getformatname_multiple(s2, SIPBUFSIZE, p->owner->nativeformats));
		}
		p->owner->nativeformats = ast_codec_choose(&p->prefs, p->jointcapability, 1) | (p->capability & vpeercapability) | (p->capability & tpeercapability);
		ast_set_read_format(p->owner, p->owner->readformat);
		ast_set_write_format(p->owner, p->owner->writeformat);
	}
	
	if (ast_test_flag(&p->flags[1], SIP_PAGE2_CALL_ONHOLD) && (!ast_sockaddr_isnull(sa) || !ast_sockaddr_isnull(vsa) || !ast_sockaddr_isnull(tsa) || !ast_sockaddr_isnull(isa)) && (!sendonly || sendonly == -1)) {
		ast_queue_control(p->owner, AST_CONTROL_UNHOLD);
		/* Activate a re-invite */
		ast_queue_frame(p->owner, &ast_null_frame);
		change_hold_state(p, req, FALSE, sendonly);
	} else if ((sockaddr_is_null_or_any(sa) && sockaddr_is_null_or_any(vsa) && sockaddr_is_null_or_any(tsa) && sockaddr_is_null_or_any(isa)) || (sendonly && sendonly != -1)) {
		ast_queue_control_data(p->owner, AST_CONTROL_HOLD,
				       S_OR(p->mohsuggest, NULL),
				       !ast_strlen_zero(p->mohsuggest) ? strlen(p->mohsuggest) + 1 : 0);
		if (sendonly)
			ast_rtp_instance_stop(p->rtp);
		/* RTCP needs to go ahead, even if we're on hold!!! */
		/* Activate a re-invite */
		ast_queue_frame(p->owner, &ast_null_frame);
		change_hold_state(p, req, TRUE, sendonly);
	}
	
	return 0;
}

static int process_sdp_o(const char *o, struct sip_pvt *p)
{
	char *o_copy;
	char *token;
	int64_t rua_version;

	/* Store the SDP version number of remote UA. This will allow us to
	distinguish between session modifications and session refreshes. If
	the remote UA does not send an incremented SDP version number in a
	subsequent RE-INVITE then that means its not changing media session.
	The RE-INVITE may have been sent to update connected party, remote
	target or to refresh the session (Session-Timers).  Asterisk must not
	change media session and increment its own version number in answer
	SDP in this case. */

	p->session_modify = TRUE;

	if (ast_strlen_zero(o)) {
		ast_log(LOG_WARNING, "SDP syntax error. SDP without an o= line\n");
		return FALSE;
	}

	o_copy = ast_strdupa(o);
	token = strsep(&o_copy, " ");  /* Skip username   */
	if (!o_copy) {
		ast_log(LOG_WARNING, "SDP syntax error in o= line username\n");
		return FALSE;
	}
	token = strsep(&o_copy, " ");  /* Skip session-id */
	if (!o_copy) {
		ast_log(LOG_WARNING, "SDP syntax error in o= line session-id\n");
		return FALSE;
	}
	token = strsep(&o_copy, " ");  /* Version         */
	if (!o_copy) {
		ast_log(LOG_WARNING, "SDP syntax error in o= line\n");
		return FALSE;
	}
	if (!sscanf(token, "%30" SCNd64, &rua_version)) {
		ast_log(LOG_WARNING, "SDP syntax error in o= line version\n");
		return FALSE;
	}

	/* we need to check the SDP version number the other end sent us;
	 * our rules for deciding what to accept are a bit complex.
	 *
	 * 1) if 'ignoresdpversion' has been set for this dialog, then
	 *    we will just accept whatever they sent and assume it is
	 *    a modification of the session, even if it is not
	 * 2) otherwise, if this is the first SDP we've seen from them
	 *    we accept it
	 * 3) otherwise, if the new SDP version number is higher than the
	 *    old one, we accept it
	 * 4) otherwise, if this SDP is in response to us requesting a switch
	 *    to T.38, we accept the SDP, but also generate a warning message
	 *    that this peer should have the 'ignoresdpversion' option set,
	 *    because it is not following the SDP offer/answer RFC; if we did
	 *    not request a switch to T.38, then we stop parsing the SDP, as it
	 *    has not changed from the previous version
	 */

	if (ast_test_flag(&p->flags[1], SIP_PAGE2_IGNORESDPVERSION) ||
	    (p->sessionversion_remote < 0) ||
	    (p->sessionversion_remote < rua_version)) {
		p->sessionversion_remote = rua_version;
	} else {
		if (p->t38.state == T38_LOCAL_REINVITE) {
			p->sessionversion_remote = rua_version;
			ast_log(LOG_WARNING, "Call %s responded to our T.38 reinvite without changing SDP version; 'ignoresdpversion' should be set for this peer.\n", p->callid);
		} else {
			p->session_modify = FALSE;
			ast_debug(2, "Call %s responded to our reinvite without changing SDP version; ignoring SDP.\n", p->callid);
			return FALSE;
		}
	}

	return TRUE;
}

static int process_sdp_c(const char *c, struct ast_sockaddr *addr)
{
	char proto[4], host[258];
	int af;

	/* Check for Media-description-level-address */
	if (sscanf(c, "IN %3s %255s", proto, host) == 2) {
		if (!strcmp("IP4", proto)) {
			af = AF_INET;
		} else if (!strcmp("IP6", proto)) {
			af = AF_INET6;
		} else {
			ast_log(LOG_WARNING, "Unknown protocol '%s'.\n", proto);
			return FALSE;
		}
		if (ast_sockaddr_resolve_first_af(addr, host, 0, af)) {
			ast_log(LOG_WARNING, "Unable to lookup RTP Audio host in c= line, '%s'\n", c);
			return FALSE;
		}
		return TRUE;
	} else {
		ast_log(LOG_WARNING, "Invalid host in c= line, '%s'\n", c);
		return FALSE;
	}
	return FALSE;
}

static int process_sdp_a_sendonly(const char *a, int *sendonly)
{
	int found = FALSE;

	if (!strcasecmp(a, "sendonly")) {
		if (*sendonly == -1)
			*sendonly = 1;
		found = TRUE;
	} else if (!strcasecmp(a, "inactive")) {
		if (*sendonly == -1)
			*sendonly = 2;
		found = TRUE;
	}  else if (!strcasecmp(a, "sendrecv")) {
		if (*sendonly == -1)
			*sendonly = 0;
		found = TRUE;
	}
	return found;
}

static int process_sdp_a_audio(const char *a, struct sip_pvt *p, struct ast_rtp_codecs *newaudiortp, int *last_rtpmap_codec)
{
	int found = FALSE;
	int codec;
	char mimeSubtype[128];
	char fmtp_string[64];
	unsigned int sample_rate;
	int debug = sip_debug_test_pvt(p);

	if (!strncasecmp(a, "ptime", 5)) {
		char *tmp = strrchr(a, ':');
		long int framing = 0;
		if (tmp) {
			tmp++;
			framing = strtol(tmp, NULL, 10);
			if (framing == LONG_MIN || framing == LONG_MAX) {
				framing = 0;
				ast_debug(1, "Can't read framing from SDP: %s\n", a);
			}
		}
		if (framing && p->autoframing) {
			struct ast_codec_pref *pref = &ast_rtp_instance_get_codecs(p->rtp)->pref;
			int codec_n;
			for (codec_n = 0; codec_n < AST_RTP_MAX_PT; codec_n++) {
				struct ast_rtp_payload_type format = ast_rtp_codecs_payload_lookup(ast_rtp_instance_get_codecs(p->rtp), codec_n);
				if (!format.asterisk_format || !format.code)	/* non-codec or not found */
					continue;
				ast_debug(1, "Setting framing for %s to %ld\n", ast_getformatname(format.code), framing);
				ast_codec_pref_setsize(pref, format.code, framing);
			}
			ast_rtp_codecs_packetization_set(ast_rtp_instance_get_codecs(p->rtp), p->rtp, pref);
		}
		found = TRUE;
	} else if (sscanf(a, "rtpmap: %30u %127[^/]/%30u", &codec, mimeSubtype, &sample_rate) == 3) {
		/* We have a rtpmap to handle */
		if (*last_rtpmap_codec < SDP_MAX_RTPMAP_CODECS) {
			if (!(ast_rtp_codecs_payloads_set_rtpmap_type_rate(newaudiortp, NULL, codec, "audio", mimeSubtype,
			    ast_test_flag(&p->flags[0], SIP_G726_NONSTANDARD) ? AST_RTP_OPT_G726_NONSTANDARD : 0, sample_rate))) {
				if (debug)
					ast_verbose("Found audio description format %s for ID %d\n", mimeSubtype, codec);
				//found_rtpmap_codecs[last_rtpmap_codec] = codec;
				(*last_rtpmap_codec)++;
				found = TRUE;
			} else {
				ast_rtp_codecs_payloads_unset(newaudiortp, NULL, codec);
				if (debug)
					ast_verbose("Found unknown media description format %s for ID %d\n", mimeSubtype, codec);
			}
		} else {
			if (debug)
				ast_verbose("Discarded description format %s for ID %d\n", mimeSubtype, codec);
		}
	} else if (sscanf(a, "fmtp: %30u %63s", &codec, fmtp_string) == 2) {
		struct ast_rtp_payload_type payload;

		payload = ast_rtp_codecs_payload_lookup(newaudiortp, codec);
		if (payload.code && payload.asterisk_format) {
			unsigned int bit_rate;

			switch (payload.code) {
			case AST_FORMAT_SIREN7:
				if (sscanf(fmtp_string, "bitrate=%30u", &bit_rate) == 1) {
					if (bit_rate != 32000) {
						ast_log(LOG_WARNING, "Got Siren7 offer at %d bps, but only 32000 bps supported; ignoring.\n", bit_rate);
						ast_rtp_codecs_payloads_unset(newaudiortp, NULL, codec);
					} else {
						found = TRUE;
					}
				}
				break;
			case AST_FORMAT_SIREN14:
				if (sscanf(fmtp_string, "bitrate=%30u", &bit_rate) == 1) {
					if (bit_rate != 48000) {
						ast_log(LOG_WARNING, "Got Siren14 offer at %d bps, but only 48000 bps supported; ignoring.\n", bit_rate);
						ast_rtp_codecs_payloads_unset(newaudiortp, NULL, codec);
					} else {
						found = TRUE;
					}
				}
				break;
			case AST_FORMAT_G719:
				if (sscanf(fmtp_string, "bitrate=%30u", &bit_rate) == 1) {
					if (bit_rate != 64000) {
						ast_log(LOG_WARNING, "Got G.719 offer at %d bps, but only 64000 bps supported; ignoring.\n", bit_rate);
						ast_rtp_codecs_payloads_unset(newaudiortp, NULL, codec);
					} else {
						found = TRUE;
					}
				}
			}
		}
	}

	return found;
}

static int process_sdp_a_video(const char *a, struct sip_pvt *p, struct ast_rtp_codecs *newvideortp, int *last_rtpmap_codec)
{
	int found = FALSE;
	int codec;
	char mimeSubtype[128];
	unsigned int sample_rate;
	int debug = sip_debug_test_pvt(p);

	if (sscanf(a, "rtpmap: %30u %127[^/]/%30u", &codec, mimeSubtype, &sample_rate) == 3) {
		/* We have a rtpmap to handle */
		if (*last_rtpmap_codec < SDP_MAX_RTPMAP_CODECS) {
			/* Note: should really look at the '#chans' params too */
			if (!strncasecmp(mimeSubtype, "H26", 3) || !strncasecmp(mimeSubtype, "MP4", 3)) {
				if (!(ast_rtp_codecs_payloads_set_rtpmap_type_rate(newvideortp, NULL, codec, "video", mimeSubtype, 0, sample_rate))) {
					if (debug)
						ast_verbose("Found video description format %s for ID %d\n", mimeSubtype, codec);
					//found_rtpmap_codecs[last_rtpmap_codec] = codec;
					(*last_rtpmap_codec)++;
					found = TRUE;
				} else {
					ast_rtp_codecs_payloads_unset(newvideortp, NULL, codec);
					if (debug)
						ast_verbose("Found unknown media description format %s for ID %d\n", mimeSubtype, codec);
				}
			}
		} else {
			if (debug)
				ast_verbose("Discarded description format %s for ID %d\n", mimeSubtype, codec);
		}
	}

	return found;
}

static int process_sdp_a_text(const char *a, struct sip_pvt *p, struct ast_rtp_codecs *newtextrtp, char *red_fmtp, int *red_num_gen, int *red_data_pt, int *last_rtpmap_codec)
{
	int found = FALSE;
	int codec;
	char mimeSubtype[128];
	unsigned int sample_rate;
	char *red_cp;
	int debug = sip_debug_test_pvt(p);

	if (sscanf(a, "rtpmap: %30u %127[^/]/%30u", &codec, mimeSubtype, &sample_rate) == 3) {
		/* We have a rtpmap to handle */
		if (*last_rtpmap_codec < SDP_MAX_RTPMAP_CODECS) {
			if (!strncasecmp(mimeSubtype, "T140", 4)) { /* Text */
				if (p->trtp) {
					/* ast_verbose("Adding t140 mimeSubtype to textrtp struct\n"); */
					ast_rtp_codecs_payloads_set_rtpmap_type_rate(newtextrtp, NULL, codec, "text", mimeSubtype, 0, sample_rate);
					found = TRUE;
				}
			} else if (!strncasecmp(mimeSubtype, "RED", 3)) { /* Text with Redudancy */
				if (p->trtp) {
					ast_rtp_codecs_payloads_set_rtpmap_type_rate(newtextrtp, NULL, codec, "text", mimeSubtype, 0, sample_rate);
					sprintf(red_fmtp, "fmtp:%d ", codec);
					if (debug)
						ast_verbose("RED submimetype has payload type: %d\n", codec);
					found = TRUE;
				}
			}
		} else {
			if (debug)
				ast_verbose("Discarded description format %s for ID %d\n", mimeSubtype, codec);
		}
	} else if (!strncmp(a, red_fmtp, strlen(red_fmtp))) {
		/* count numbers of generations in fmtp */
		red_cp = &red_fmtp[strlen(red_fmtp)];
		strncpy(red_fmtp, a, 100);

		sscanf(red_cp, "%30u", &red_data_pt[*red_num_gen]);
		red_cp = strtok(red_cp, "/");
		while (red_cp && (*red_num_gen)++ < AST_RED_MAX_GENERATION) {
			sscanf(red_cp, "%30u", &red_data_pt[*red_num_gen]);
			red_cp = strtok(NULL, "/");
		}
		red_cp = red_fmtp;
		found = TRUE;
	}

	return found;
}

static int process_sdp_a_image(const char *a, struct sip_pvt *p)
{
	int found = FALSE;
	char s[256];
	unsigned int x;

	if (initialize_udptl(p)) {
		return found;
	}

	if ((sscanf(a, "T38FaxMaxBuffer:%30u", &x) == 1)) {
		ast_debug(3, "MaxBufferSize:%d\n", x);
		found = TRUE;
	} else if ((sscanf(a, "T38MaxBitRate:%30u", &x) == 1) || (sscanf(a, "T38FaxMaxRate:%30u", &x) == 1)) {
		ast_debug(3, "T38MaxBitRate: %d\n", x);
		switch (x) {
		case 14400:
			p->t38.their_parms.rate = AST_T38_RATE_14400;
			break;
		case 12000:
			p->t38.their_parms.rate = AST_T38_RATE_12000;
			break;
		case 9600:
			p->t38.their_parms.rate = AST_T38_RATE_9600;
			break;
		case 7200:
			p->t38.their_parms.rate = AST_T38_RATE_7200;
			break;
		case 4800:
			p->t38.their_parms.rate = AST_T38_RATE_4800;
			break;
		case 2400:
			p->t38.their_parms.rate = AST_T38_RATE_2400;
			break;
		}
		found = TRUE;
	} else if ((sscanf(a, "T38FaxVersion:%30u", &x) == 1)) {
		ast_debug(3, "FaxVersion: %u\n", x);
		p->t38.their_parms.version = x;
		found = TRUE;
	} else if ((sscanf(a, "T38FaxMaxDatagram:%30u", &x) == 1) || (sscanf(a, "T38MaxDatagram:%30u", &x) == 1)) {
		/* override the supplied value if the configuration requests it */
		if (((signed int) p->t38_maxdatagram >= 0) && ((unsigned int) p->t38_maxdatagram > x)) {
			ast_debug(1, "Overriding T38FaxMaxDatagram '%d' with '%d'\n", x, p->t38_maxdatagram);
			x = p->t38_maxdatagram;
		}
		ast_debug(3, "FaxMaxDatagram: %u\n", x);
		ast_udptl_set_far_max_datagram(p->udptl, x);
		found = TRUE;
	} else if ((strncmp(a, "T38FaxFillBitRemoval", 20) == 0)) {
		if (sscanf(a, "T38FaxFillBitRemoval:%30u", &x) == 1) {
			ast_debug(3, "FillBitRemoval: %d\n", x);
			if (x == 1) {
				p->t38.their_parms.fill_bit_removal = TRUE;
			}
		} else {
			ast_debug(3, "FillBitRemoval\n");
			p->t38.their_parms.fill_bit_removal = TRUE;
		}
		found = TRUE;
	} else if ((strncmp(a, "T38FaxTranscodingMMR", 20) == 0)) {
		if (sscanf(a, "T38FaxTranscodingMMR:%30u", &x) == 1) {
			ast_debug(3, "Transcoding MMR: %d\n", x);
			if (x == 1) {
				p->t38.their_parms.transcoding_mmr = TRUE;
			}
		} else {
			ast_debug(3, "Transcoding MMR\n");
			p->t38.their_parms.transcoding_mmr = TRUE;
		}
		found = TRUE;
	} else if ((strncmp(a, "T38FaxTranscodingJBIG", 21) == 0)) {
		if (sscanf(a, "T38FaxTranscodingJBIG:%30u", &x) == 1) {
			ast_debug(3, "Transcoding JBIG: %d\n", x);
			if (x == 1) {
				p->t38.their_parms.transcoding_jbig = TRUE;
			}
		} else {
			ast_debug(3, "Transcoding JBIG\n");
			p->t38.their_parms.transcoding_jbig = TRUE;
		}
		found = TRUE;
	} else if ((sscanf(a, "T38FaxRateManagement:%255s", s) == 1)) {
		ast_debug(3, "RateManagement: %s\n", s);
		if (!strcasecmp(s, "localTCF"))
			p->t38.their_parms.rate_management = AST_T38_RATE_MANAGEMENT_LOCAL_TCF;
		else if (!strcasecmp(s, "transferredTCF"))
			p->t38.their_parms.rate_management = AST_T38_RATE_MANAGEMENT_TRANSFERRED_TCF;
		found = TRUE;
	} else if ((sscanf(a, "T38FaxUdpEC:%255s", s) == 1)) {
		ast_debug(3, "UDP EC: %s\n", s);
		if (!strcasecmp(s, "t38UDPRedundancy")) {
			ast_udptl_set_error_correction_scheme(p->udptl, UDPTL_ERROR_CORRECTION_REDUNDANCY);
		} else if (!strcasecmp(s, "t38UDPFEC")) {
			ast_udptl_set_error_correction_scheme(p->udptl, UDPTL_ERROR_CORRECTION_FEC);
		} else {
			ast_udptl_set_error_correction_scheme(p->udptl, UDPTL_ERROR_CORRECTION_NONE);
		}
		found = TRUE;
	}

	return found;
}

/*! \brief Add "Supported" header to sip message.  Since some options may
 *  be disabled in the config, the sip_pvt must be inspected to determine what
 *  is supported for this dialog. */
static int add_supported_header(struct sip_pvt *pvt, struct sip_request *req)
{
	int res;
	if (st_get_mode(pvt, 0) != SESSION_TIMER_MODE_REFUSE) {
		res = add_header(req, "Supported", "replaces, timer");
	} else {
		res = add_header(req, "Supported", "replaces");
	}
	return res;
}

/*! \brief Add header to SIP message */
static int add_header(struct sip_request *req, const char *var, const char *value)
{
	if (req->headers == SIP_MAX_HEADERS) {
		ast_log(LOG_WARNING, "Out of SIP header space\n");
		return -1;
	}

	if (req->lines) {
		ast_log(LOG_WARNING, "Can't add more headers when lines have been added\n");
		return -1;
	}

	if (sip_cfg.compactheaders) {
		var = find_alias(var, var);
	}

	ast_str_append(&req->data, 0, "%s: %s\r\n", var, value);
	req->header[req->headers] = ast_str_strlen(req->data);

	req->headers++;

	return 0;	
}

/*! 
 * \pre dialog is assumed to be locked while calling this function
 * \brief Add 'Max-Forwards' header to SIP message 
 */
static int add_header_max_forwards(struct sip_pvt *dialog, struct sip_request *req)
{
	char clen[10];

	snprintf(clen, sizeof(clen), "%d", dialog->maxforwards);

	return add_header(req, "Max-Forwards", clen);
}

/*! \brief Add 'Content-Length' header and content to SIP message */
static int finalize_content(struct sip_request *req)
{
	char clen[10];

	if (req->lines) {
		ast_log(LOG_WARNING, "finalize_content() called on a message that has already been finalized\n");
		return -1;
	}

	snprintf(clen, sizeof(clen), "%zd", ast_str_strlen(req->content));
	add_header(req, "Content-Length", clen);

	if (ast_str_strlen(req->content)) {
		ast_str_append(&req->data, 0, "\r\n%s", ast_str_buffer(req->content));
	}
	req->lines = ast_str_strlen(req->content) ? 1 : 0;
	return 0;
}

/*! \brief Add content (not header) to SIP message */
static int add_content(struct sip_request *req, const char *line)
{
	if (req->lines) {
		ast_log(LOG_WARNING, "Can't add more content when the content has been finalized\n");
		return -1;
	}

	ast_str_append(&req->content, 0, "%s", line);
	return 0;
}

/*! \brief Copy one header field from one request to another */
static int copy_header(struct sip_request *req, const struct sip_request *orig, const char *field)
{
	const char *tmp = get_header(orig, field);

	if (!ast_strlen_zero(tmp)) /* Add what we're responding to */
		return add_header(req, field, tmp);
	ast_log(LOG_NOTICE, "No field '%s' present to copy\n", field);
	return -1;
}

/*! \brief Copy all headers from one request to another */
static int copy_all_header(struct sip_request *req, const struct sip_request *orig, const char *field)
{
	int start = 0;
	int copied = 0;
	for (;;) {
		const char *tmp = __get_header(orig, field, &start);

		if (ast_strlen_zero(tmp))
			break;
		/* Add what we're responding to */
		add_header(req, field, tmp);
		copied++;
	}
	return copied ? 0 : -1;
}

/*! \brief Copy SIP VIA Headers from the request to the response
\note	If the client indicates that it wishes to know the port we received from,
	it adds ;rport without an argument to the topmost via header. We need to
	add the port number (from our point of view) to that parameter.
\verbatim
	We always add ;received=<ip address> to the topmost via header.
\endverbatim
	Received: RFC 3261, rport RFC 3581 */
static int copy_via_headers(struct sip_pvt *p, struct sip_request *req, const struct sip_request *orig, const char *field)
{
	int copied = 0;
	int start = 0;

	for (;;) {
		char new[512];
		const char *oh = __get_header(orig, field, &start);

		if (ast_strlen_zero(oh))
			break;

		if (!copied) {	/* Only check for empty rport in topmost via header */
			char leftmost[512], *others, *rport;

			/* Only work on leftmost value */
			ast_copy_string(leftmost, oh, sizeof(leftmost));
			others = strchr(leftmost, ',');
			if (others)
			    *others++ = '\0';

			/* Find ;rport;  (empty request) */
			rport = strstr(leftmost, ";rport");
			if (rport && *(rport+6) == '=')
				rport = NULL;		/* We already have a parameter to rport */

			if (((ast_test_flag(&p->flags[0], SIP_NAT_FORCE_RPORT)) || (rport && ast_test_flag(&p->flags[0], SIP_NAT_RPORT_PRESENT)))) {
				/* We need to add received port - rport */
				char *end;

				rport = strstr(leftmost, ";rport");

				if (rport) {
					end = strchr(rport + 1, ';');
					if (end)
						memmove(rport, end, strlen(end) + 1);
					else
						*rport = '\0';
				}

				/* Add rport to first VIA header if requested */
				snprintf(new, sizeof(new), "%s;received=%s;rport=%d%s%s",
					leftmost, ast_sockaddr_stringify_addr_remote(&p->recv),
					ast_sockaddr_port(&p->recv),
					others ? "," : "", others ? others : "");
			} else {
				/* We should *always* add a received to the topmost via */
				snprintf(new, sizeof(new), "%s;received=%s%s%s",
					leftmost, ast_sockaddr_stringify_addr_remote(&p->recv),
					others ? "," : "", others ? others : "");
			}
			oh = new;	/* the header to copy */
		}  /* else add the following via headers untouched */
		add_header(req, field, oh);
		copied++;
	}
	if (!copied) {
		ast_log(LOG_NOTICE, "No header field '%s' present to copy\n", field);
		return -1;
	}
	return 0;
}

/*! \brief Add route header into request per learned route */
static void add_route(struct sip_request *req, struct sip_route *route)
{
	char r[SIPBUFSIZE*2], *p;
	int n, rem = sizeof(r);

	if (!route)
		return;

	p = r;
	for (;route ; route = route->next) {
		n = strlen(route->hop);
		if (rem < n+3) /* we need room for ",<route>" */
			break;
		if (p != r) {	/* add a separator after fist route */
			*p++ = ',';
			--rem;
		}
		*p++ = '<';
		ast_copy_string(p, route->hop, rem); /* cannot fail */
		p += n;
		*p++ = '>';
		rem -= (n+2);
	}
	*p = '\0';
	add_header(req, "Route", r);
}

/*! \brief Set destination from SIP URI
 *
 * Parse uri to h (host) and port - uri is already just the part inside the <>
 * general form we are expecting is sip[s]:username[:password][;parameter]@host[:port][;...]
 * If there's a port given, turn NAPTR/SRV off. NAPTR might indicate SIPS preference even
 * for SIP: uri's
 *
 * If there's a sips: uri scheme, TLS will be required.
 */
static void set_destination(struct sip_pvt *p, char *uri)
{
	char *h, *maddr, hostname[256];
	int hn;
	int debug=sip_debug_test_pvt(p);
	int tls_on = FALSE;

	if (debug)
		ast_verbose("set_destination: Parsing <%s> for address/port to send to\n", uri);

	/* Find and parse hostname */
	h = strchr(uri, '@');
	if (h)
		++h;
	else {
		h = uri;
		if (!strncasecmp(h, "sip:", 4)) {
			h += 4;
		} else if (!strncasecmp(h, "sips:", 5)) {
			h += 5;
			tls_on = TRUE;
		}
	}
	hn = strcspn(h, ";>") + 1;
	if (hn > sizeof(hostname))
		hn = sizeof(hostname);
	ast_copy_string(hostname, h, hn);
	/* XXX bug here if string has been trimmed to sizeof(hostname) */
	h += hn - 1;

	/*! \todo XXX If we have sip_cfg.srvlookup on, then look for NAPTR/SRV,
	 * otherwise, just look for A records */
	if (ast_sockaddr_resolve_first(&p->sa, hostname, 0)) {
		ast_log(LOG_WARNING, "Can't find address for host '%s'\n", hostname);
		return;
	}

	/* Got the hostname - but maybe there's a "maddr=" to override address? */
	maddr = strstr(h, "maddr=");
	if (maddr) {
		int port;

		maddr += 6;
		hn = strspn(maddr, "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ"
			           "0123456789-.:[]") + 1;
		if (hn > sizeof(hostname))
			hn = sizeof(hostname);
		ast_copy_string(hostname, maddr, hn);

		port = ast_sockaddr_port(&p->sa);

		/*! \todo XXX If we have sip_cfg.srvlookup on, then look for
		 * NAPTR/SRV, otherwise, just look for A records */
		if (ast_sockaddr_resolve_first(&p->sa, hostname, PARSE_PORT_FORBID)) {
			ast_log(LOG_WARNING, "Can't find address for host '%s'\n", hostname);
			return;
		}

		ast_sockaddr_set_port(&p->sa, port);
	}

	if (!ast_sockaddr_port(&p->sa)) {
		ast_sockaddr_set_port(&p->sa, tls_on ?
				      STANDARD_TLS_PORT : STANDARD_SIP_PORT);
	}

	if (debug) {
		ast_verbose("set_destination: set destination to %s\n",
			    ast_sockaddr_stringify(&p->sa));
	}
}

/*! \brief Initialize SIP response, based on SIP request */
static int init_resp(struct sip_request *resp, const char *msg)
{
	/* Initialize a response */
	memset(resp, 0, sizeof(*resp));
	resp->method = SIP_RESPONSE;
	if (!(resp->data = ast_str_create(SIP_MIN_PACKET)))
		goto e_return;
	if (!(resp->content = ast_str_create(SIP_MIN_PACKET)))
		goto e_free_data;
	resp->header[0] = 0;
	ast_str_set(&resp->data, 0, "SIP/2.0 %s\r\n", msg);
	resp->headers++;
	return 0;

e_free_data:
	ast_free(resp->data);
	resp->data = NULL;
e_return:
	return -1;
}

/*! \brief Initialize SIP request */
static int init_req(struct sip_request *req, int sipmethod, const char *recip)
{
	/* Initialize a request */
	memset(req, 0, sizeof(*req));
	if (!(req->data = ast_str_create(SIP_MIN_PACKET)))
		goto e_return;
	if (!(req->content = ast_str_create(SIP_MIN_PACKET)))
		goto e_free_data;
	req->method = sipmethod;
	req->header[0] = 0;
	ast_str_set(&req->data, 0, "%s %s SIP/2.0\r\n", sip_methods[sipmethod].text, recip);
	req->headers++;
	return 0;

e_free_data:
	ast_free(req->data);
	req->data = NULL;
e_return:
	return -1;
}

/*! \brief Deinitialize SIP response/request */
static void deinit_req(struct sip_request *req)
{
	if (req->data) {
		ast_free(req->data);
		req->data = NULL;
	}
	if (req->content) {
		ast_free(req->content);
		req->content = NULL;
	}
}


/*! \brief Test if this response needs a contact header */
static inline int resp_needs_contact(const char *msg, enum sipmethod method) {
	/* Requirements for Contact header inclusion in responses generated
	 * from the header tables found in the following RFCs.  Where the
	 * Contact header was marked mandatory (m) or optional (o) this
	 * function returns 1.
	 *
	 * - RFC 3261 (ACK, BYE, CANCEL, INVITE, OPTIONS, REGISTER)
	 * - RFC 2976 (INFO)
	 * - RFC 3262 (PRACK)
	 * - RFC 3265 (SUBSCRIBE, NOTIFY)
	 * - RFC 3311 (UPDATE)
	 * - RFC 3428 (MESSAGE)
	 * - RFC 3515 (REFER)
	 * - RFC 3903 (PUBLISH)
	 */

	switch (method) {
		/* 1xx, 2xx, 3xx, 485 */
		case SIP_INVITE:
		case SIP_UPDATE:
		case SIP_SUBSCRIBE:
		case SIP_NOTIFY:
			if ((msg[0] >= '1' && msg[0] <= '3') || !strncmp(msg, "485", 3))
				return 1;
			break;

		/* 2xx, 3xx, 485 */
		case SIP_REGISTER:
		case SIP_OPTIONS:
			if (msg[0] == '2' || msg[0] == '3' || !strncmp(msg, "485", 3))
				return 1;
			break;

		/* 3xx, 485 */
		case SIP_BYE:
		case SIP_PRACK:
		case SIP_MESSAGE:
		case SIP_PUBLISH:
			if (msg[0] == '3' || !strncmp(msg, "485", 3))
				return 1;
			break;

		/* 2xx, 3xx, 4xx, 5xx, 6xx */
		case SIP_REFER:
			if (msg[0] >= '2' && msg[0] <= '6')
				return 1;
			break;

		/* contact will not be included for everything else */
		case SIP_ACK:
		case SIP_CANCEL:
		case SIP_INFO:
		case SIP_PING:
		default:
			return 0;
	}
	return 0;
}

/*! \brief Prepare SIP response packet */
static int respprep(struct sip_request *resp, struct sip_pvt *p, const char *msg, const struct sip_request *req)
{
	char newto[256];
	const char *ot;

	init_resp(resp, msg);
	copy_via_headers(p, resp, req, "Via");
	if (msg[0] == '1' || msg[0] == '2')
		copy_all_header(resp, req, "Record-Route");
	copy_header(resp, req, "From");
	ot = get_header(req, "To");
	if (!strcasestr(ot, "tag=") && strncmp(msg, "100", 3)) {
		/* Add the proper tag if we don't have it already.  If they have specified
		   their tag, use it.  Otherwise, use our own tag */
		if (!ast_strlen_zero(p->theirtag) && ast_test_flag(&p->flags[0], SIP_OUTGOING))
			snprintf(newto, sizeof(newto), "%s;tag=%s", ot, p->theirtag);
		else if (p->tag && !ast_test_flag(&p->flags[0], SIP_OUTGOING))
			snprintf(newto, sizeof(newto), "%s;tag=%s", ot, p->tag);
		else
			ast_copy_string(newto, ot, sizeof(newto));
		ot = newto;
	}
	add_header(resp, "To", ot);
	copy_header(resp, req, "Call-ID");
	copy_header(resp, req, "CSeq");
	if (!ast_strlen_zero(global_useragent))
		add_header(resp, "Server", global_useragent);
	add_header(resp, "Allow", ALLOWED_METHODS);
	add_supported_header(p, resp);

	/* If this is an invite, add Session-Timers related headers if the feature is active for this session */
	if (p->method == SIP_INVITE && p->stimer && p->stimer->st_active == TRUE && p->stimer->st_active_peer_ua == TRUE) {
		char se_hdr[256];
		snprintf(se_hdr, sizeof(se_hdr), "%d;refresher=%s", p->stimer->st_interval,
			strefresher2str(p->stimer->st_ref));
		add_header(resp, "Session-Expires", se_hdr);
	}

	if (msg[0] == '2' && (p->method == SIP_SUBSCRIBE || p->method == SIP_REGISTER)) {
		/* For registration responses, we also need expiry and
		   contact info */
		char tmp[256];

		snprintf(tmp, sizeof(tmp), "%d", p->expiry);
		add_header(resp, "Expires", tmp);
		if (p->expiry) {	/* Only add contact if we have an expiry time */
			char contact[SIPBUFSIZE];
			const char *contact_uri = p->method == SIP_SUBSCRIBE ? p->our_contact : p->fullcontact;
			char *brackets = strchr(contact_uri, '<');
			snprintf(contact, sizeof(contact), "%s%s%s;expires=%d", brackets ? "" : "<", contact_uri, brackets ? "" : ">", p->expiry);
			add_header(resp, "Contact", contact);	/* Not when we unregister */
		}
	} else if (!ast_strlen_zero(p->our_contact) && resp_needs_contact(msg, p->method)) {
		add_header(resp, "Contact", p->our_contact);
	}

	if (!ast_strlen_zero(p->url)) {
		add_header(resp, "Access-URL", p->url);
		ast_string_field_set(p, url, NULL);
	}

	/* default to routing the response to the address where the request
	 * came from.  Since we don't have a transport layer, we do this here.
	 * The process_via() function will update the port to either the port
	 * specified in the via header or the default port later on (per RFC
	 * 3261 section 18.2.2).
	 */
	p->sa = p->recv;

	if (process_via(p, req)) {
		ast_log(LOG_WARNING, "error processing via header, will send response to originating address\n");
	}

	return 0;
}

/*! \brief Initialize a SIP request message (not the initial one in a dialog) */
static int reqprep(struct sip_request *req, struct sip_pvt *p, int sipmethod, uint32_t seqno, int newbranch)
{
	struct sip_request *orig = &p->initreq;
	char stripped[80];
	char tmp[80];
	char newto[256];
	const char *c;
	const char *ot, *of;
	int is_strict = FALSE;		/*!< Strict routing flag */
	int is_outbound = ast_test_flag(&p->flags[0], SIP_OUTGOING);	/* Session direction */

	memset(req, 0, sizeof(struct sip_request));
	
	snprintf(p->lastmsg, sizeof(p->lastmsg), "Tx: %s", sip_methods[sipmethod].text);
	
	if (!seqno) {
		p->ocseq++;
		seqno = p->ocseq;
	}
	
	/* A CANCEL must have the same branch as the INVITE that it is canceling. */
	if (sipmethod == SIP_CANCEL) {
		p->branch = p->invite_branch;
		build_via(p);
	} else if (newbranch && (sipmethod == SIP_INVITE)) {
		p->branch ^= ast_random();
		p->invite_branch = p->branch;
		build_via(p);
	} else if (newbranch) {
		p->branch ^= ast_random();
		build_via(p);
	}

	/* Check for strict or loose router */
	if (p->route && !ast_strlen_zero(p->route->hop) && strstr(p->route->hop, ";lr") == NULL) {
		is_strict = TRUE;
		if (sipdebug)
			ast_debug(1, "Strict routing enforced for session %s\n", p->callid);
	}
	
	if (sipmethod == SIP_CANCEL)
		c = REQ_OFFSET_TO_STR(&p->initreq, rlPart2);	/* Use original URI */
	else if (sipmethod == SIP_ACK) {
		/* Use URI from Contact: in 200 OK (if INVITE)
		(we only have the contacturi on INVITEs) */
		if (!ast_strlen_zero(p->okcontacturi))
			c = is_strict ? p->route->hop : p->okcontacturi;
 		else
 			c = REQ_OFFSET_TO_STR(&p->initreq, rlPart2);
	} else if (!ast_strlen_zero(p->okcontacturi))
		c = is_strict ? p->route->hop : p->okcontacturi; /* Use for BYE or REINVITE */
	else if (!ast_strlen_zero(p->uri))
		c = p->uri;
	else {
		char *n;
		/* We have no URI, use To: or From:  header as URI (depending on direction) */
		ast_copy_string(stripped, get_header(orig, is_outbound ? "To" : "From"),
				sizeof(stripped));
		n = get_in_brackets(stripped);
		c = remove_uri_parameters(n);
	}	
	init_req(req, sipmethod, c);

	snprintf(tmp, sizeof(tmp), "%u %s", seqno, sip_methods[sipmethod].text);

	add_header(req, "Via", p->via);
	/*
	 * Use the learned route set unless this is a CANCEL on an ACK for a non-2xx
	 * final response. For a CANCEL or ACK, we have to send to the same destination
	 * as the original INVITE.
	 */
	if (sipmethod == SIP_CANCEL ||
			(sipmethod == SIP_ACK && (p->invitestate == INV_COMPLETED || p->invitestate == INV_CANCELLED))) {
		set_destination(p, ast_strdupa(p->uri));
	} else if (p->route) {
		set_destination(p, p->route->hop);
		add_route(req, is_strict ? p->route->next : p->route);
	}
	add_header_max_forwards(p, req);

	ot = get_header(orig, "To");
	of = get_header(orig, "From");

	/* Add tag *unless* this is a CANCEL, in which case we need to send it exactly
	   as our original request, including tag (or presumably lack thereof) */
	if (!strcasestr(ot, "tag=") && sipmethod != SIP_CANCEL) {
		/* Add the proper tag if we don't have it already.  If they have specified
		   their tag, use it.  Otherwise, use our own tag */
		if (is_outbound && !ast_strlen_zero(p->theirtag))
			snprintf(newto, sizeof(newto), "%s;tag=%s", ot, p->theirtag);
		else if (!is_outbound)
			snprintf(newto, sizeof(newto), "%s;tag=%s", ot, p->tag);
		else
			snprintf(newto, sizeof(newto), "%s", ot);
		ot = newto;
	}

	if (is_outbound) {
		add_header(req, "From", of);
		add_header(req, "To", ot);
	} else {
		add_header(req, "From", ot);
		add_header(req, "To", of);
	}
	/* Do not add Contact for MESSAGE, BYE and Cancel requests */
	if (sipmethod != SIP_BYE && sipmethod != SIP_CANCEL && sipmethod != SIP_MESSAGE)
		add_header(req, "Contact", p->our_contact);

	copy_header(req, orig, "Call-ID");
	add_header(req, "CSeq", tmp);

	if (!ast_strlen_zero(global_useragent))
		add_header(req, "User-Agent", global_useragent);

	if (!ast_strlen_zero(p->url)) {
		add_header(req, "Access-URL", p->url);
		ast_string_field_set(p, url, NULL);
	}

	/* Add Session-Timers related headers if the feature is active for this session.
	   An exception to this behavior is the ACK request. Since Asterisk never requires
	   session-timers support from a remote end-point (UAS) in an INVITE, it must
	   not send 'Require: timer' header in the ACK request.
	   This should only be added in the INVITE transactions, not MESSAGE or REFER or other
	   in-dialog messages.
	*/
	if (p->stimer && p->stimer->st_active == TRUE && p->stimer->st_active_peer_ua == TRUE
	    && sipmethod == SIP_INVITE) {
		char se_hdr[256];
		snprintf(se_hdr, sizeof(se_hdr), "%d;refresher=%s", p->stimer->st_interval,
			strefresher2str(p->stimer->st_ref));
		add_header(req, "Session-Expires", se_hdr);
		snprintf(se_hdr, sizeof(se_hdr), "%d", st_get_se(p, FALSE));
		add_header(req, "Min-SE", se_hdr);
	}

	return 0;
}

/*! \brief Base transmit response function */
static int __transmit_response(struct sip_pvt *p, const char *msg, const struct sip_request *req, enum xmittype reliable)
{
	struct sip_request resp;
	uint32_t seqno = 0;

	if (reliable && (sscanf(get_header(req, "CSeq"), "%30u ", &seqno) != 1)) {
		ast_log(LOG_WARNING, "Unable to determine sequence number from '%s'\n", get_header(req, "CSeq"));
		return -1;
	}
	respprep(&resp, p, msg, req);

	if (ast_test_flag(&p->flags[0], SIP_SENDRPID)
			&& ast_test_flag(&p->flags[1], SIP_PAGE2_CONNECTLINEUPDATE_PEND)
			&& (!strncmp(msg, "180", 3) || !strncmp(msg, "183", 3))) {
		ast_clear_flag(&p->flags[1], SIP_PAGE2_CONNECTLINEUPDATE_PEND);
		add_rpid(&resp, p);
	}
	if (ast_test_flag(&p->flags[0], SIP_OFFER_CC)) {
		add_cc_call_info_to_response(p, &resp);
	}

	/* If we are sending a 302 Redirect we can add a diversion header if the redirect information is set */
	if (!strncmp(msg, "302", 3)) {
		add_diversion_header(&resp, p);
	}

	/* If we are cancelling an incoming invite for some reason, add information
		about the reason why we are doing this in clear text */
	if (p->method == SIP_INVITE && msg[0] != '1') {
		char buf[20];

		if (ast_test_flag(&p->flags[1], SIP_PAGE2_Q850_REASON)) {
			int hangupcause = 0;

			if (p->owner && p->owner->hangupcause) {
				hangupcause = p->owner->hangupcause;
			} else if (p->hangupcause) {
				hangupcause = p->hangupcause;
			} else {
				int respcode;
				if (sscanf(msg, "%30d ", &respcode))
					hangupcause = hangup_sip2cause(respcode);
			}

			if (hangupcause) {
				sprintf(buf, "Q.850;cause=%i", hangupcause & 0x7f);
				add_header(&resp, "Reason", buf);
			}
		}

		if (p->owner && p->owner->hangupcause) {
			add_header(&resp, "X-Asterisk-HangupCause", ast_cause2str(p->owner->hangupcause));
			snprintf(buf, sizeof(buf), "%d", p->owner->hangupcause);
			add_header(&resp, "X-Asterisk-HangupCauseCode", buf);
		}
	}
	return send_response(p, &resp, reliable, seqno);
}

static int transmit_response_with_sip_etag(struct sip_pvt *p, const char *msg, const struct sip_request *req, struct sip_esc_entry *esc_entry, int need_new_etag)
{
	struct sip_request resp;

	if (need_new_etag) {
		create_new_sip_etag(esc_entry, 1);
	}
	respprep(&resp, p, msg, req);
	add_header(&resp, "SIP-ETag", esc_entry->entity_tag);

	return send_response(p, &resp, 0, 0);
}

static int temp_pvt_init(void *data)
{
	struct sip_pvt *p = data;

	p->do_history = 0;	/* XXX do we need it ? isn't already all 0 ? */
	return ast_string_field_init(p, 512);
}

static void temp_pvt_cleanup(void *data)
{
	struct sip_pvt *p = data;

	ast_string_field_free_memory(p);

	ast_free(data);
}

/*! \brief Transmit response, no retransmits, using a temporary pvt structure */
static int transmit_response_using_temp(ast_string_field callid, struct ast_sockaddr *addr, int useglobal_nat, const int intended_method, const struct sip_request *req, const char *msg)
{
	struct sip_pvt *p = NULL;

	if (!(p = ast_threadstorage_get(&ts_temp_pvt, sizeof(*p)))) {
		ast_log(LOG_ERROR, "Failed to get temporary pvt\n");
		return -1;
	}

	/* XXX the structure may be dirty from previous usage.
	 * Here we should state clearly how we should reinitialize it
	 * before using it.
	 * E.g. certainly the threadstorage should be left alone,
	 * but other thihngs such as flags etc. maybe need cleanup ?
	 */

	/* Initialize the bare minimum */
	p->method = intended_method;

	if (!addr) {
		ast_sockaddr_copy(&p->ourip, &internip);
	} else {
		ast_sockaddr_copy(&p->sa, addr);
		ast_sip_ouraddrfor(&p->sa, &p->ourip, p);
	}

	p->branch = ast_random();
	make_our_tag(p->tag, sizeof(p->tag));
	p->ocseq = INITIAL_CSEQ;

	if (useglobal_nat && addr) {
		ast_copy_flags(&p->flags[0], &global_flags[0], SIP_NAT_FORCE_RPORT);
		ast_sockaddr_copy(&p->recv, addr);
		do_setnat(p);
	}

	ast_string_field_set(p, fromdomain, default_fromdomain);
	p->fromdomainport = default_fromdomainport;
	build_via(p);
	ast_string_field_set(p, callid, callid);

	copy_socket_data(&p->socket, &req->socket);

	/* Use this temporary pvt structure to send the message */
	__transmit_response(p, msg, req, XMIT_UNRELIABLE);

	/* Free the string fields, but not the pool space */
	ast_string_field_init(p, 0);

	return 0;
}

/*! \brief Transmit response, no retransmits */
static int transmit_response(struct sip_pvt *p, const char *msg, const struct sip_request *req)
{
	return __transmit_response(p, msg, req, XMIT_UNRELIABLE);
}

/*! \brief Transmit response, no retransmits */
static int transmit_response_with_unsupported(struct sip_pvt *p, const char *msg, const struct sip_request *req, const char *unsupported)
{
	struct sip_request resp;
	respprep(&resp, p, msg, req);
	append_date(&resp);
	add_header(&resp, "Unsupported", unsupported);
	return send_response(p, &resp, XMIT_UNRELIABLE, 0);
}

/*! \brief Transmit 422 response with Min-SE header (Session-Timers)  */
static int transmit_response_with_minse(struct sip_pvt *p, const char *msg, const struct sip_request *req, int minse_int)
{
	struct sip_request resp;
	char minse_str[20];

	respprep(&resp, p, msg, req);
	append_date(&resp);

	snprintf(minse_str, sizeof(minse_str), "%d", minse_int);
	add_header(&resp, "Min-SE", minse_str);
	return send_response(p, &resp, XMIT_UNRELIABLE, 0);
}


/*! \brief Transmit response, Make sure you get an ACK
	This is only used for responses to INVITEs, where we need to make sure we get an ACK
*/
static int transmit_response_reliable(struct sip_pvt *p, const char *msg, const struct sip_request *req)
{
	return __transmit_response(p, msg, req, req->ignore ? XMIT_UNRELIABLE : XMIT_CRITICAL);
}

/*! \brief Append date to SIP message */
static void append_date(struct sip_request *req)
{
	char tmpdat[256];
	struct tm tm;
	time_t t = time(NULL);

	gmtime_r(&t, &tm);
	strftime(tmpdat, sizeof(tmpdat), "%a, %d %b %Y %T GMT", &tm);
	add_header(req, "Date", tmpdat);
}

/*! \brief Append Retry-After header field when transmitting response */
static int transmit_response_with_retry_after(struct sip_pvt *p, const char *msg, const struct sip_request *req, const char *seconds)
{
	struct sip_request resp;
	respprep(&resp, p, msg, req);
	add_header(&resp, "Retry-After", seconds);
	return send_response(p, &resp, XMIT_UNRELIABLE, 0);
}

/*! \brief Append date and content length before transmitting response */
static int transmit_response_with_date(struct sip_pvt *p, const char *msg, const struct sip_request *req)
{
	struct sip_request resp;
	respprep(&resp, p, msg, req);
	append_date(&resp);
	return send_response(p, &resp, XMIT_UNRELIABLE, 0);
}

/*! \brief Append Accept header, content length before transmitting response */
static int transmit_response_with_allow(struct sip_pvt *p, const char *msg, const struct sip_request *req, enum xmittype reliable)
{
	struct sip_request resp;
	respprep(&resp, p, msg, req);
	add_header(&resp, "Accept", "application/sdp");
	return send_response(p, &resp, reliable, 0);
}

/*! \brief Append Min-Expires header, content length before transmitting response */
static int transmit_response_with_minexpires(struct sip_pvt *p, const char *msg, const struct sip_request *req)
{
	struct sip_request resp;
	char tmp[32];

	snprintf(tmp, sizeof(tmp), "%d", min_expiry);
	respprep(&resp, p, msg, req);
	add_header(&resp, "Min-Expires", tmp);
	return send_response(p, &resp, XMIT_UNRELIABLE, 0);
}

/*! \brief Respond with authorization request */
static int transmit_response_with_auth(struct sip_pvt *p, const char *msg, const struct sip_request *req, const char *randdata, enum xmittype reliable, const char *header, int stale)
{
	struct sip_request resp;
	char tmp[512];
	uint32_t seqno = 0;

	if (reliable && (sscanf(get_header(req, "CSeq"), "%30u ", &seqno) != 1)) {
		ast_log(LOG_WARNING, "Unable to determine sequence number from '%s'\n", get_header(req, "CSeq"));
		return -1;
	}
	/* Choose Realm */
	get_realm(p, req);

	/* Stale means that they sent us correct authentication, but
	   based it on an old challenge (nonce) */
	snprintf(tmp, sizeof(tmp), "Digest algorithm=MD5, realm=\"%s\", nonce=\"%s\"%s", p->realm, randdata, stale ? ", stale=true" : "");
	respprep(&resp, p, msg, req);
	add_header(&resp, header, tmp);
	append_history(p, "AuthChal", "Auth challenge sent for %s - nc %d", p->username, p->noncecount);
	return send_response(p, &resp, reliable, seqno);
}

/*!
 \brief Extract domain from SIP To/From header
 \return -1 on error, 1 if domain string is empty, 0 if domain was properly extracted
 \note TODO: Such code is all over SIP channel, there is a sense to organize
      this patern in one function
*/
static int get_domain(const char *str, char *domain, int len)
{
	char tmpf[256];
	char *a, *from;

	*domain = '\0';
	ast_copy_string(tmpf, str, sizeof(tmpf));
	from = get_in_brackets(tmpf);
	if (!ast_strlen_zero(from)) {
		if (strncasecmp(from, "sip:", 4)) {
			ast_log(LOG_WARNING, "Huh?  Not a SIP header (%s)?\n", from);
			return -1;
		}
		from += 4;
	} else
		from = NULL;

	if (from) {
		int bracket = 0;

		/* Strip any params or options from user */
		if ((a = strchr(from, ';')))
			*a = '\0';
		/* Strip port from domain if present */
		for (a = from; *a != '\0'; ++a) {
			if (*a == ':' && bracket == 0) {
				*a = '\0';
				break;
			} else if (*a == '[') {
				++bracket;
			} else if (*a == ']') {
				--bracket;
			}
		}
		if ((a = strchr(from, '@'))) {
			*a = '\0';
			ast_copy_string(domain, a + 1, len);
		} else
			ast_copy_string(domain, from, len);
	}

	return ast_strlen_zero(domain);
}

/*!
  \brief Choose realm based on From header and then To header or use globaly configured realm.
  Realm from From/To header should be listed among served domains in config file: domain=...
*/
static void get_realm(struct sip_pvt *p, const struct sip_request *req)
{
	char domain[MAXHOSTNAMELEN];

	if (!ast_strlen_zero(p->realm))
		return;

	if (sip_cfg.domainsasrealm &&
	    !AST_LIST_EMPTY(&domain_list))
	{
		/* Check From header first */
		if (!get_domain(get_header(req, "From"), domain, sizeof(domain))) {
			if (check_sip_domain(domain, NULL, 0)) {
				ast_string_field_set(p, realm, domain);
				return;
			}
		}
		/* Check To header */
		if (!get_domain(get_header(req, "To"), domain, sizeof(domain))) {
			if (check_sip_domain(domain, NULL, 0)) {
				ast_string_field_set(p, realm, domain);
				return;
			}
		}
	}
	
	/* Use default realm from config file */
	ast_string_field_set(p, realm, sip_cfg.realm);
}

/*!
 * \internal
 *
 * \arg msg Only use a string constant for the msg, here, it is shallow copied
 *
 * \note assumes the sip_pvt is locked.
 */
static int transmit_provisional_response(struct sip_pvt *p, const char *msg, const struct sip_request *req, int with_sdp)
{
	int res;

	if (!(res = with_sdp ? transmit_response_with_sdp(p, msg, req, XMIT_UNRELIABLE, FALSE, FALSE) : transmit_response(p, msg, req))) {
		p->last_provisional = msg;
		update_provisional_keepalive(p, with_sdp);
	}

	return res;
}

/*! \brief Add text body to SIP message */
static int add_text(struct sip_request *req, const char *text)
{
	/* XXX Convert \n's to \r\n's XXX */
	add_header(req, "Content-Type", "text/plain;charset=UTF-8");
	add_content(req, text);
	return 0;
}

/*! \brief Add DTMF INFO tone to sip message
	Mode = 	0 for application/dtmf-relay (Cisco)
		1 for application/dtmf
*/
static int add_digit(struct sip_request *req, char digit, unsigned int duration, int mode)
{
	char tmp[256];
	int event;
	if (mode) {
		/* Application/dtmf short version used by some implementations */
		if ('0' <= digit && digit <= '9') {
			event = digit - '0';
		} else if (digit == '*') {
			event = 10;
		} else if (digit == '#') {
			event = 11;
		} else if ('A' <= digit && digit <= 'D') {
			event = 12 + digit - 'A';
		} else if ('a' <= digit && digit <= 'd') {
			event = 12 + digit - 'a';
		} else {
			/* Unknown digit */
			event = 0;
		}
		snprintf(tmp, sizeof(tmp), "%d\r\n", event);
		add_header(req, "Content-Type", "application/dtmf");
		add_content(req, tmp);
	} else {
		/* Application/dtmf-relay as documented by Cisco */
		snprintf(tmp, sizeof(tmp), "Signal=%c\r\nDuration=%u\r\n", digit, duration);
		add_header(req, "Content-Type", "application/dtmf-relay");
		add_content(req, tmp);
	}
	return 0;
}

/*!
 * \pre if p->owner exists, it must be locked
 * \brief Add Remote-Party-ID header to SIP message
 */
static int add_rpid(struct sip_request *req, struct sip_pvt *p)
{
	struct ast_str *tmp = ast_str_alloca(256);
	char tmp2[256];
	char *lid_num = NULL;
	char *lid_name = NULL;
	int lid_pres;
	const char *fromdomain;
	const char *privacy = NULL;
	const char *screen = NULL;
	const char *anonymous_string = "\"Anonymous\" <sip:anonymous@anonymous.invalid>";

	if (!ast_test_flag(&p->flags[0], SIP_SENDRPID)) {
		return 0;
	}

	if (p->owner && p->owner->connected.id.number.valid
		&& p->owner->connected.id.number.str) {
		lid_num = p->owner->connected.id.number.str;
	}
	if (p->owner && p->owner->connected.id.name.valid
		&& p->owner->connected.id.name.str) {
		lid_name = p->owner->connected.id.name.str;
	}
	lid_pres = (p->owner) ? ast_party_id_presentation(&p->owner->connected.id) : AST_PRES_NUMBER_NOT_AVAILABLE;

	if (ast_strlen_zero(lid_num))
		return 0;
	if (ast_strlen_zero(lid_name))
		lid_name = lid_num;
	fromdomain = S_OR(p->fromdomain, ast_sockaddr_stringify_host_remote(&p->ourip));

	lid_num = ast_uri_encode(lid_num, tmp2, sizeof(tmp2), 0);

	if (ast_test_flag(&p->flags[0], SIP_SENDRPID_PAI)) {
		if ((lid_pres & AST_PRES_RESTRICTION) != AST_PRES_ALLOWED) {
			ast_str_set(&tmp, -1, "%s", anonymous_string);
		} else {
			ast_str_set(&tmp, -1, "\"%s\" <sip:%s@%s>", lid_name, lid_num, fromdomain);
		}
		add_header(req, "P-Asserted-Identity", ast_str_buffer(tmp));
	} else {
		ast_str_set(&tmp, -1, "\"%s\" <sip:%s@%s>;party=%s", lid_name, lid_num, fromdomain, ast_test_flag(&p->flags[0], SIP_OUTGOING) ? "calling" : "called");

		switch (lid_pres) {
		case AST_PRES_ALLOWED_USER_NUMBER_NOT_SCREENED:
		case AST_PRES_ALLOWED_USER_NUMBER_FAILED_SCREEN:
			privacy = "off";
			screen = "no";
			break;
		case AST_PRES_ALLOWED_USER_NUMBER_PASSED_SCREEN:
		case AST_PRES_ALLOWED_NETWORK_NUMBER:
			privacy = "off";
			screen = "yes";
			break;
		case AST_PRES_PROHIB_USER_NUMBER_NOT_SCREENED:
		case AST_PRES_PROHIB_USER_NUMBER_FAILED_SCREEN:
			privacy = "full";
			screen = "no";
			break;
		case AST_PRES_PROHIB_USER_NUMBER_PASSED_SCREEN:
		case AST_PRES_PROHIB_NETWORK_NUMBER:
			privacy = "full";
			screen = "yes";
			break;
		case AST_PRES_NUMBER_NOT_AVAILABLE:
			break;
		default:
			if ((lid_pres & AST_PRES_RESTRICTION) != AST_PRES_ALLOWED) {
				privacy = "full";
			}
			else
				privacy = "off";
			screen = "no";
			break;
		}

		if (!ast_strlen_zero(privacy) && !ast_strlen_zero(screen)) {
			ast_str_append(&tmp, -1, ";privacy=%s;screen=%s", privacy, screen);
		}

		add_header(req, "Remote-Party-ID", ast_str_buffer(tmp));
	}
	return 0;
}

/*! \brief add XML encoded media control with update
	\note XML: The only way to turn 0 bits of information into a few hundred. (markster) */
static int add_vidupdate(struct sip_request *req)
{
	const char *xml_is_a_huge_waste_of_space =
		"<?xml version=\"1.0\" encoding=\"utf-8\" ?>\r\n"
		" <media_control>\r\n"
		"  <vc_primitive>\r\n"
		"   <to_encoder>\r\n"
		"    <picture_fast_update>\r\n"
		"    </picture_fast_update>\r\n"
		"   </to_encoder>\r\n"
		"  </vc_primitive>\r\n"
		" </media_control>\r\n";
	add_header(req, "Content-Type", "application/media_control+xml");
	add_content(req, xml_is_a_huge_waste_of_space);
	return 0;
}

/*! \brief Add codec offer to SDP offer/answer body in INVITE or 200 OK */
static void add_codec_to_sdp(const struct sip_pvt *p, format_t codec,
			     struct ast_str **m_buf, struct ast_str **a_buf,
			     int debug, int *min_packet_size)
{
	int rtp_code;
	struct ast_format_list fmt;


	if (debug)
		ast_verbose("Adding codec 0x%" PRIx64 " (%s) to SDP\n", codec, ast_getformatname(codec));
	if ((rtp_code = ast_rtp_codecs_payload_code(ast_rtp_instance_get_codecs(p->rtp), 1, codec)) == -1)
		return;

	if (p->rtp) {
		struct ast_codec_pref *pref = &ast_rtp_instance_get_codecs(p->rtp)->pref;
		fmt = ast_codec_pref_getsize(pref, codec);
	} else /* I don't see how you couldn't have p->rtp, but good to check for and error out if not there like earlier code */
		return;
	ast_str_append(m_buf, 0, " %d", rtp_code);
	ast_str_append(a_buf, 0, "a=rtpmap:%d %s/%d\r\n", rtp_code,
		       ast_rtp_lookup_mime_subtype2(1, codec,
						   ast_test_flag(&p->flags[0], SIP_G726_NONSTANDARD) ? AST_RTP_OPT_G726_NONSTANDARD : 0),
		       ast_rtp_lookup_sample_rate2(1, codec));

	switch (codec) {
	case AST_FORMAT_G729A:
		/* Indicate that we don't support VAD (G.729 annex B) */
		ast_str_append(a_buf, 0, "a=fmtp:%d annexb=no\r\n", rtp_code);
		break;
	case AST_FORMAT_G723_1:
		/* Indicate that we don't support VAD (G.723.1 annex A) */
		ast_str_append(a_buf, 0, "a=fmtp:%d annexa=no\r\n", rtp_code);
		break;
	case AST_FORMAT_ILBC:
		/* Add information about us using only 20/30 ms packetization */
		ast_str_append(a_buf, 0, "a=fmtp:%d mode=%d\r\n", rtp_code, fmt.cur_ms);
		break;
	case AST_FORMAT_SIREN7:
		/* Indicate that we only expect 32Kbps */
		ast_str_append(a_buf, 0, "a=fmtp:%d bitrate=32000\r\n", rtp_code);
		break;
	case AST_FORMAT_SIREN14:
		/* Indicate that we only expect 48Kbps */
		ast_str_append(a_buf, 0, "a=fmtp:%d bitrate=48000\r\n", rtp_code);
		break;
	case AST_FORMAT_G719:
		/* Indicate that we only expect 64Kbps */
		ast_str_append(a_buf, 0, "a=fmtp:%d bitrate=64000\r\n", rtp_code);
		break;
	}

	if (fmt.cur_ms && (fmt.cur_ms < *min_packet_size))
		*min_packet_size = fmt.cur_ms;

	/* Our first codec packetization processed cannot be zero */
	if ((*min_packet_size)==0 && fmt.cur_ms)
		*min_packet_size = fmt.cur_ms;
}

/*! \brief Add video codec offer to SDP offer/answer body in INVITE or 200 OK */
/* This is different to the audio one now so we can add more caps later */
static void add_vcodec_to_sdp(const struct sip_pvt *p, format_t codec,
			     struct ast_str **m_buf, struct ast_str **a_buf,
			     int debug, int *min_packet_size)
{
	int rtp_code;

	if (!p->vrtp)
		return;

	if (debug)
		ast_verbose("Adding video codec 0x%" PRIx64 " (%s) to SDP\n", codec, ast_getformatname(codec));

	if ((rtp_code = ast_rtp_codecs_payload_code(ast_rtp_instance_get_codecs(p->vrtp), 1, codec)) == -1)
		return;

	ast_str_append(m_buf, 0, " %d", rtp_code);
	ast_str_append(a_buf, 0, "a=rtpmap:%d %s/%d\r\n", rtp_code,
		       ast_rtp_lookup_mime_subtype2(1, codec, 0),
		       ast_rtp_lookup_sample_rate2(1, codec));
	/* Add fmtp code here */
}

/*! \brief Add text codec offer to SDP offer/answer body in INVITE or 200 OK */
static void add_tcodec_to_sdp(const struct sip_pvt *p, int codec,
			     struct ast_str **m_buf, struct ast_str **a_buf,
			     int debug, int *min_packet_size)
{
	int rtp_code;

	if (!p->trtp)
		return;

	if (debug)
		ast_verbose("Adding text codec 0x%x (%s) to SDP\n", codec, ast_getformatname(codec));

	if ((rtp_code = ast_rtp_codecs_payload_code(ast_rtp_instance_get_codecs(p->trtp), 1, codec)) == -1)
		return;

	ast_str_append(m_buf, 0, " %d", rtp_code);
	ast_str_append(a_buf, 0, "a=rtpmap:%d %s/%d\r\n", rtp_code,
		       ast_rtp_lookup_mime_subtype2(1, codec, 0),
		       ast_rtp_lookup_sample_rate2(1, codec));
	/* Add fmtp code here */

	if (codec == AST_FORMAT_T140RED) {
		int t140code = ast_rtp_codecs_payload_code(ast_rtp_instance_get_codecs(p->trtp), 1, AST_FORMAT_T140);
		ast_str_append(a_buf, 0, "a=fmtp:%d %d/%d/%d\r\n", rtp_code,
			 t140code,
			 t140code,
			 t140code);

	}
}


/*! \brief Get Max T.38 Transmission rate from T38 capabilities */
static unsigned int t38_get_rate(enum ast_control_t38_rate rate)
{
	switch (rate) {
	case AST_T38_RATE_2400:
		return 2400;
	case AST_T38_RATE_4800:
		return 4800;
	case AST_T38_RATE_7200:
		return 7200;
	case AST_T38_RATE_9600:
		return 9600;
	case AST_T38_RATE_12000:
		return 12000;
	case AST_T38_RATE_14400:
		return 14400;
	default:
		return 0;
	}
}

/*! \brief Add RFC 2833 DTMF offer to SDP */
static void add_noncodec_to_sdp(const struct sip_pvt *p, int format,
				struct ast_str **m_buf, struct ast_str **a_buf,
				int debug)
{
	int rtp_code;

	if (debug)
		ast_verbose("Adding non-codec 0x%x (%s) to SDP\n", format, ast_rtp_lookup_mime_subtype2(0, format, 0));
	if ((rtp_code = ast_rtp_codecs_payload_code(ast_rtp_instance_get_codecs(p->rtp), 0, format)) == -1)
		return;

	ast_str_append(m_buf, 0, " %d", rtp_code);
	ast_str_append(a_buf, 0, "a=rtpmap:%d %s/%d\r\n", rtp_code,
		       ast_rtp_lookup_mime_subtype2(0, format, 0),
		       ast_rtp_lookup_sample_rate2(0, format));
	if (format == AST_RTP_DTMF)	/* Indicate we support DTMF and FLASH... */
		ast_str_append(a_buf, 0, "a=fmtp:%d 0-16\r\n", rtp_code);
}

/*! \brief Set all IP media addresses for this call
	\note called from add_sdp()
*/
static void get_our_media_address(struct sip_pvt *p, int needvideo, int needtext,
				  struct ast_sockaddr *addr, struct ast_sockaddr *vaddr,
				  struct ast_sockaddr *taddr, struct ast_sockaddr *dest,
				  struct ast_sockaddr *vdest, struct ast_sockaddr *tdest)
{
	int use_externip = 0;

	/* First, get our address */
	ast_rtp_instance_get_local_address(p->rtp, addr);
	if (p->vrtp) {
		ast_rtp_instance_get_local_address(p->vrtp, vaddr);
	}
	if (p->trtp) {
		ast_rtp_instance_get_local_address(p->trtp, taddr);
	}

	/* If our real IP differs from the local address returned by the RTP engine, use it. */
	/* The premise is that if we are already using that IP to communicate with the client, */
	/* we should be using it for RTP too. */
        use_externip = ast_sockaddr_cmp_addr(&p->ourip, addr);

	/* Now, try to figure out where we want them to send data */
	/* Is this a re-invite to move the media out, then use the original offer from caller  */
	if (!ast_sockaddr_isnull(&p->redirip)) {	/* If we have a redirection IP, use it */
		ast_sockaddr_copy(dest, &p->redirip);
	} else {
		/*
		 * Audio Destination IP:
		 *
		 * 1. Specifically configured media address.
		 * 2. Local address as specified by the RTP engine.
		 * 3. The local IP as defined by chan_sip.
		 *
		 * Audio Destination Port:
		 *
		 * 1. Provided by the RTP engine.
		 */
		ast_sockaddr_copy(dest,
				  !ast_sockaddr_isnull(&media_address) ? &media_address :
				  !ast_sockaddr_is_any(addr) && !use_externip ? addr    :
				  &p->ourip);
		ast_sockaddr_set_port(dest, ast_sockaddr_port(addr));
	}

	if (needvideo) {
		/* Determine video destination */
		if (!ast_sockaddr_isnull(&p->vredirip)) {
			ast_sockaddr_copy(vdest, &p->vredirip);
		} else {
			/*
			 * Video Destination IP:
			 *
			 * 1. Specifically configured media address.
			 * 2. Local address as specified by the RTP engine.
			 * 3. The local IP as defined by chan_sip.
			 *
			 * Video Destination Port:
			 *
			 * 1. Provided by the RTP engine.
			 */
			ast_sockaddr_copy(vdest,
					  !ast_sockaddr_isnull(&media_address) ? &media_address :
					  !ast_sockaddr_is_any(vaddr) && !use_externip ? vaddr  :
					  &p->ourip);
			ast_sockaddr_set_port(vdest, ast_sockaddr_port(vaddr));
		}
	}

	if (needtext) {
		/* Determine text destination */
		if (!ast_sockaddr_isnull(&p->tredirip)) {
			ast_sockaddr_copy(tdest, &p->tredirip);
		} else {
			/*
			 * Text Destination IP:
			 *
			 * 1. Specifically configured media address.
			 * 2. Local address as specified by the RTP engine.
			 * 3. The local IP as defined by chan_sip.
			 *
			 * Text Destination Port:
			 *
			 * 1. Provided by the RTP engine.
			 */
			ast_sockaddr_copy(tdest,
					  !ast_sockaddr_isnull(&media_address) ? &media_address  :
					  !ast_sockaddr_is_any(taddr) && !use_externip ? taddr   :
					  &p->ourip);
			ast_sockaddr_set_port(tdest, ast_sockaddr_port(taddr));
		}
	}
}

static void get_crypto_attrib(struct sip_srtp *srtp, const char **a_crypto)
{
	/* Set encryption properties */
	if (srtp) {
		if (!srtp->crypto) {
			srtp->crypto = sdp_crypto_setup();
		}
		if (srtp->crypto && (sdp_crypto_offer(srtp->crypto) >= 0)) {
			*a_crypto = sdp_crypto_attrib(srtp->crypto);
		}

		if (!*a_crypto) {
			ast_log(LOG_WARNING, "No SRTP key management enabled\n");
		}
	}
}

/*! \brief Add Session Description Protocol message

    If oldsdp is TRUE, then the SDP version number is not incremented. This mechanism
    is used in Session-Timers where RE-INVITEs are used for refreshing SIP sessions
    without modifying the media session in any way.
*/
static enum sip_result add_sdp(struct sip_request *resp, struct sip_pvt *p, int oldsdp, int add_audio, int add_t38)
{
	format_t alreadysent = 0;
	int doing_directmedia = FALSE;

	struct ast_sockaddr addr = { {0,} };
	struct ast_sockaddr vaddr = { {0,} };
	struct ast_sockaddr taddr = { {0,} };
	struct ast_sockaddr udptladdr = { {0,} };
	struct ast_sockaddr dest = { {0,} };
	struct ast_sockaddr vdest = { {0,} };
	struct ast_sockaddr tdest = { {0,} };
	struct ast_sockaddr udptldest = { {0,} };

	/* SDP fields */
	char *version = 	"v=0\r\n";		/* Protocol version */
	char subject[256];				/* Subject of the session */
	char owner[256];				/* Session owner/creator */
	char connection[256];				/* Connection data */
	char *session_time = "t=0 0\r\n"; 			/* Time the session is active */
	char bandwidth[256] = "";			/* Max bitrate */
	char *hold = "";
	struct ast_str *m_audio = ast_str_alloca(256);  /* Media declaration line for audio */
	struct ast_str *m_video = ast_str_alloca(256);  /* Media declaration line for video */
	struct ast_str *m_text = ast_str_alloca(256);   /* Media declaration line for text */
	struct ast_str *m_modem = ast_str_alloca(256);  /* Media declaration line for modem */
	struct ast_str *a_audio = ast_str_alloca(1024); /* Attributes for audio */
	struct ast_str *a_video = ast_str_alloca(1024); /* Attributes for video */
	struct ast_str *a_text = ast_str_alloca(1024);  /* Attributes for text */
	struct ast_str *a_modem = ast_str_alloca(1024); /* Attributes for modem */
	const char *a_crypto = NULL;
	const char *v_a_crypto = NULL;
	const char *t_a_crypto = NULL;

	format_t x;
	format_t capability = 0;
	int needaudio = FALSE;
	int needvideo = FALSE;
	int needtext = FALSE;
	int debug = sip_debug_test_pvt(p);
	int min_audio_packet_size = 0;
	int min_video_packet_size = 0;
	int min_text_packet_size = 0;

	char codecbuf[SIPBUFSIZE];
	char buf[SIPBUFSIZE];
	char dummy_answer[256];

	/* Set the SDP session name */
	snprintf(subject, sizeof(subject), "s=%s\r\n", ast_strlen_zero(global_sdpsession) ? "-" : global_sdpsession);

	if (!p->rtp) {
		ast_log(LOG_WARNING, "No way to add SDP without an RTP structure\n");
		return AST_FAILURE;
	}
	/* XXX We should not change properties in the SIP dialog until
		we have acceptance of the offer if this is a re-invite */

	/* Set RTP Session ID and version */
	if (!p->sessionid) {
		p->sessionid = (int)ast_random();
		p->sessionversion = p->sessionid;
	} else {
		if (oldsdp == FALSE)
			p->sessionversion++;
	}

	if (add_audio) {
		doing_directmedia = (!ast_sockaddr_isnull(&p->redirip) && p->redircodecs) ? TRUE : FALSE;
		/* Check if we need video in this call */
		if ((p->jointcapability & AST_FORMAT_VIDEO_MASK) && !p->novideo) {
			if (doing_directmedia && !(p->jointcapability & AST_FORMAT_VIDEO_MASK & p->redircodecs)) {
				ast_debug(2, "This call needs video offers, but caller probably did not offer it!\n");
			} else if (p->vrtp) {
				needvideo = TRUE;
				ast_debug(2, "This call needs video offers!\n");
			} else {
				ast_debug(2, "This call needs video offers, but there's no video support enabled!\n");
			}
		}
		/* Check if we need text in this call */
		if ((p->jointcapability & AST_FORMAT_TEXT_MASK) && !p->notext) {
			if (sipdebug_text)
				ast_verbose("We think we can do text\n");
			if (p->trtp) {
				if (sipdebug_text) {
					ast_verbose("And we have a text rtp object\n");
				}
				needtext = TRUE;
				ast_debug(2, "This call needs text offers! \n");
			} else {
				ast_debug(2, "This call needs text offers, but there's no text support enabled ! \n");
			}
		}
	}

	get_our_media_address(p, needvideo, needtext, &addr, &vaddr, &taddr, &dest, &vdest, &tdest);

	snprintf(owner, sizeof(owner), "o=%s %d %d IN %s %s\r\n",
		 ast_strlen_zero(global_sdpowner) ? "-" : global_sdpowner,
		 p->sessionid, p->sessionversion,
		 (ast_sockaddr_is_ipv6(&dest) && !ast_sockaddr_is_ipv4_mapped(&dest)) ?
			"IP6" : "IP4",
		 ast_sockaddr_stringify_addr_remote(&dest));

	snprintf(connection, sizeof(connection), "c=IN %s %s\r\n",
		 (ast_sockaddr_is_ipv6(&dest) && !ast_sockaddr_is_ipv4_mapped(&dest)) ?
			"IP6" : "IP4",
		 ast_sockaddr_stringify_addr_remote(&dest));

	if (add_audio) {
		if (ast_test_flag(&p->flags[1], SIP_PAGE2_CALL_ONHOLD) == SIP_PAGE2_CALL_ONHOLD_ONEDIR) {
			hold = "a=recvonly\r\n";
			doing_directmedia = FALSE;
		} else if (ast_test_flag(&p->flags[1], SIP_PAGE2_CALL_ONHOLD) == SIP_PAGE2_CALL_ONHOLD_INACTIVE) {
			hold = "a=inactive\r\n";
			doing_directmedia = FALSE;
		} else {
			hold = "a=sendrecv\r\n";
		}

		capability = p->jointcapability;

		/* XXX note, Video and Text are negated - 'true' means 'no' */
		ast_debug(1, "** Our capability: %s Video flag: %s Text flag: %s\n", ast_getformatname_multiple(codecbuf, sizeof(codecbuf), capability),
			  p->novideo ? "True" : "False", p->notext ? "True" : "False");
		ast_debug(1, "** Our prefcodec: %s \n", ast_getformatname_multiple(codecbuf, sizeof(codecbuf), p->prefcodec));

		if (doing_directmedia) {
			capability &= p->redircodecs;
			ast_debug(1, "** Our native-bridge filtered capablity: %s\n", ast_getformatname_multiple(codecbuf, sizeof(codecbuf), capability));
		}

		/* Check if we need audio */
		if (capability & AST_FORMAT_AUDIO_MASK)
			needaudio = TRUE;

		if (debug) {
			ast_verbose("Audio is at %s\n", ast_sockaddr_stringify_port(&addr));
		}

		/* Ok, we need video. Let's add what we need for video and set codecs.
		   Video is handled differently than audio since we can not transcode. */
		if (needvideo) {
			get_crypto_attrib(p->vsrtp, &v_a_crypto);
			ast_str_append(&m_video, 0, "m=video %d RTP/%s", ast_sockaddr_port(&vdest),
				v_a_crypto ? "SAVP" : "AVP");

			/* Build max bitrate string */
			if (p->maxcallbitrate)
				snprintf(bandwidth, sizeof(bandwidth), "b=CT:%d\r\n", p->maxcallbitrate);
			if (debug) {
				ast_verbose("Video is at %s\n", ast_sockaddr_stringify(&vdest));
			}
		}

		/* Ok, we need text. Let's add what we need for text and set codecs.
		   Text is handled differently than audio since we can not transcode. */
		if (needtext) {
			if (sipdebug_text)
				ast_verbose("Lets set up the text sdp\n");
			get_crypto_attrib(p->tsrtp, &t_a_crypto);
			ast_str_append(&m_text, 0, "m=text %d RTP/%s", ast_sockaddr_port(&tdest),
				t_a_crypto ? "SAVP" : "AVP");
			if (debug) {  /* XXX should I use tdest below ? */
				ast_verbose("Text is at %s\n", ast_sockaddr_stringify(&taddr));
			}
		}

		/* Start building generic SDP headers */

		/* We break with the "recommendation" and send our IP, in order that our
		   peer doesn't have to ast_gethostbyname() us */

		get_crypto_attrib(p->srtp, &a_crypto);
		ast_str_append(&m_audio, 0, "m=audio %d RTP/%s", ast_sockaddr_port(&dest),
			a_crypto ? "SAVP" : "AVP");

		/* Now, start adding audio codecs. These are added in this order:
		   - First what was requested by the calling channel
		   - Then preferences in order from sip.conf device config for this peer/user
		   - Then other codecs in capabilities, including video
		*/

		/* Prefer the audio codec we were requested to use, first, no matter what
		   Note that p->prefcodec can include video codecs, so mask them out
		*/
		if (capability & p->prefcodec) {
			format_t codec = p->prefcodec & AST_FORMAT_AUDIO_MASK;

			add_codec_to_sdp(p, codec, &m_audio, &a_audio, debug, &min_audio_packet_size);
			alreadysent |= codec;
		}

		/* Start by sending our preferred audio/video codecs */
		for (x = 0; x < 64; x++) {
			format_t codec;

			if (!(codec = ast_codec_pref_index(&p->prefs, x)))
				break;

			if (!(capability & codec))
				continue;

			if (alreadysent & codec)
				continue;

			add_codec_to_sdp(p, codec, &m_audio, &a_audio, debug, &min_audio_packet_size);
			alreadysent |= codec;
		}

		/* Now send any other common audio and video codecs, and non-codec formats: */
		for (x = 1ULL; x <= (needtext ? AST_FORMAT_TEXT_MASK : (needvideo ? AST_FORMAT_VIDEO_MASK : AST_FORMAT_AUDIO_MASK)); x <<= 1) {
			if (!(capability & x))	/* Codec not requested */
				continue;

			if (alreadysent & x)	/* Already added to SDP */
				continue;

			if (x & AST_FORMAT_AUDIO_MASK)
				add_codec_to_sdp(p, x, &m_audio, &a_audio, debug, &min_audio_packet_size);
			else if (x & AST_FORMAT_VIDEO_MASK)
				add_vcodec_to_sdp(p, x, &m_video, &a_video, debug, &min_video_packet_size);
			else if (x & AST_FORMAT_TEXT_MASK)
				add_tcodec_to_sdp(p, x, &m_text, &a_text, debug, &min_text_packet_size);
		}

		/* Now add DTMF RFC2833 telephony-event as a codec */
		for (x = 1LL; x <= AST_RTP_MAX; x <<= 1) {
			if (!(p->jointnoncodeccapability & x))
				continue;

			add_noncodec_to_sdp(p, x, &m_audio, &a_audio, debug);
		}

		ast_debug(3, "-- Done with adding codecs to SDP\n");

		if (!p->owner || !ast_internal_timing_enabled(p->owner))
			ast_str_append(&a_audio, 0, "a=silenceSupp:off - - - -\r\n");

		if (min_audio_packet_size)
			ast_str_append(&a_audio, 0, "a=ptime:%d\r\n", min_audio_packet_size);

		/* XXX don't think you can have ptime for video */
		if (min_video_packet_size)
			ast_str_append(&a_video, 0, "a=ptime:%d\r\n", min_video_packet_size);

		/* XXX don't think you can have ptime for text */
		if (min_text_packet_size)
			ast_str_append(&a_text, 0, "a=ptime:%d\r\n", min_text_packet_size);

		if (m_audio->len - m_audio->used < 2 || m_video->len - m_video->used < 2 ||
		    m_text->len - m_text->used < 2 || a_text->len - a_text->used < 2 ||
		    a_audio->len - a_audio->used < 2 || a_video->len - a_video->used < 2)
			ast_log(LOG_WARNING, "SIP SDP may be truncated due to undersized buffer!!\n");
	}

	if (add_t38) {
		/* Our T.38 end is */
		ast_udptl_get_us(p->udptl, &udptladdr);

		/* Determine T.38 UDPTL destination */
		if (!ast_sockaddr_isnull(&p->udptlredirip)) {
			ast_sockaddr_copy(&udptldest, &p->udptlredirip);
		} else {
			ast_sockaddr_copy(&udptldest, &p->ourip);
			ast_sockaddr_set_port(&udptldest, ast_sockaddr_port(&udptladdr));
		}

		if (debug) {
			ast_debug(1, "T.38 UDPTL is at %s port %d\n", ast_sockaddr_stringify_addr(&p->ourip), ast_sockaddr_port(&udptladdr));
		}

		/* We break with the "recommendation" and send our IP, in order that our
		   peer doesn't have to ast_gethostbyname() us */

		ast_str_append(&m_modem, 0, "m=image %d udptl t38\r\n", ast_sockaddr_port(&udptldest));

		if (!ast_sockaddr_cmp(&udptldest, &dest)) {
			ast_str_append(&m_modem, 0, "c=IN %s %s\r\n",
					(ast_sockaddr_is_ipv6(&dest) && !ast_sockaddr_is_ipv4_mapped(&dest)) ?
					"IP6" : "IP4", ast_sockaddr_stringify_addr_remote(&udptldest));
		}

		ast_str_append(&a_modem, 0, "a=T38FaxVersion:%d\r\n", p->t38.our_parms.version);
		ast_str_append(&a_modem, 0, "a=T38MaxBitRate:%d\r\n", t38_get_rate(p->t38.our_parms.rate));
		if (p->t38.our_parms.fill_bit_removal) {
			ast_str_append(&a_modem, 0, "a=T38FaxFillBitRemoval\r\n");
		}
		if (p->t38.our_parms.transcoding_mmr) {
			ast_str_append(&a_modem, 0, "a=T38FaxTranscodingMMR\r\n");
		}
		if (p->t38.our_parms.transcoding_jbig) {
			ast_str_append(&a_modem, 0, "a=T38FaxTranscodingJBIG\r\n");
		}
		switch (p->t38.our_parms.rate_management) {
		case AST_T38_RATE_MANAGEMENT_TRANSFERRED_TCF:
			ast_str_append(&a_modem, 0, "a=T38FaxRateManagement:transferredTCF\r\n");
			break;
		case AST_T38_RATE_MANAGEMENT_LOCAL_TCF:
			ast_str_append(&a_modem, 0, "a=T38FaxRateManagement:localTCF\r\n");
			break;
		}
		ast_str_append(&a_modem, 0, "a=T38FaxMaxDatagram:%u\r\n", ast_udptl_get_local_max_datagram(p->udptl));
		switch (ast_udptl_get_error_correction_scheme(p->udptl)) {
		case UDPTL_ERROR_CORRECTION_NONE:
			break;
		case UDPTL_ERROR_CORRECTION_FEC:
			ast_str_append(&a_modem, 0, "a=T38FaxUdpEC:t38UDPFEC\r\n");
			break;
		case UDPTL_ERROR_CORRECTION_REDUNDANCY:
			ast_str_append(&a_modem, 0, "a=T38FaxUdpEC:t38UDPRedundancy\r\n");
			break;
		}
	}

	if (needaudio)
 		ast_str_append(&m_audio, 0, "\r\n");
 	if (needvideo)
 		ast_str_append(&m_video, 0, "\r\n");
 	if (needtext)
 		ast_str_append(&m_text, 0, "\r\n");

	add_header(resp, "Content-Type", "application/sdp");
	add_content(resp, version);
	add_content(resp, owner);
	add_content(resp, subject);
	add_content(resp, connection);
	/* only if video response is appropriate */
	if (needvideo) {
		add_content(resp, bandwidth);
	}
	add_content(resp, session_time);
	/* if this is a response to an invite, order our offers properly */
	if (p->offered_media[SDP_AUDIO].order_offered ||
		p->offered_media[SDP_VIDEO].order_offered ||
		p->offered_media[SDP_TEXT].order_offered ||
		p->offered_media[SDP_IMAGE].order_offered) {
		int i;
		/* we have up to 3 streams as limited by process_sdp */
		for (i = 1; i <= 3; i++) {
			if (p->offered_media[SDP_AUDIO].order_offered == i) {
				if (needaudio) {
					add_content(resp, m_audio->str);
					add_content(resp, a_audio->str);
					add_content(resp, hold);
					if (a_crypto) {
						add_content(resp, a_crypto);
					}
				} else {
					snprintf(dummy_answer, sizeof(dummy_answer), "m=audio 0 RTP/AVP %s\r\n", p->offered_media[SDP_AUDIO].codecs);
					add_content(resp, dummy_answer);
				}
			} else if (p->offered_media[SDP_VIDEO].order_offered == i) {
				if (needvideo) { /* only if video response is appropriate */
					add_content(resp, m_video->str);
					add_content(resp, a_video->str);
					add_content(resp, hold);	/* Repeat hold for the video stream */
					if (v_a_crypto) {
						add_content(resp, v_a_crypto);
					}
				} else {
					snprintf(dummy_answer, sizeof(dummy_answer), "m=video 0 RTP/AVP %s\r\n", p->offered_media[SDP_VIDEO].codecs);
					add_content(resp, dummy_answer);
				}
			} else if (p->offered_media[SDP_TEXT].order_offered == i) {
				if (needtext) { /* only if text response is appropriate */
					add_content(resp, m_text->str);
					add_content(resp, a_text->str);
					add_content(resp, hold);	/* Repeat hold for the text stream */
					if (t_a_crypto) {
						add_content(resp, t_a_crypto);
					}
				} else {
					snprintf(dummy_answer, sizeof(dummy_answer), "m=text 0 RTP/AVP %s\r\n", p->offered_media[SDP_TEXT].codecs);
					add_content(resp, dummy_answer);
				}
			} else if (p->offered_media[SDP_IMAGE].order_offered == i) {
				if (add_t38) {
					add_content(resp, m_modem->str);
					add_content(resp, a_modem->str);
				} else {
					add_content(resp, "m=image 0 udptl t38\r\n");
				}
			}
		}
	} else {
		/* generate new SDP from scratch, no offers */
		if (needaudio) {
			add_content(resp, m_audio->str);
			add_content(resp, a_audio->str);
			add_content(resp, hold);
			if (a_crypto) {
				add_content(resp, a_crypto);
			}
		}
		if (needvideo) { /* only if video response is appropriate */
			add_content(resp, m_video->str);
			add_content(resp, a_video->str);
			add_content(resp, hold);	/* Repeat hold for the video stream */
			if (v_a_crypto) {
				add_content(resp, v_a_crypto);
			}
		}
		if (needtext) { /* only if text response is appropriate */
			add_content(resp, m_text->str);
			add_content(resp, a_text->str);
			add_content(resp, hold);	/* Repeat hold for the text stream */
			if (t_a_crypto) {
				add_content(resp, t_a_crypto);
			}
		}
		if (add_t38) {
			add_content(resp, m_modem->str);
			add_content(resp, a_modem->str);
		}
	}

	/* Update lastrtprx when we send our SDP */
	p->lastrtprx = p->lastrtptx = time(NULL); /* XXX why both ? */

	ast_debug(3, "Done building SDP. Settling with this capability: %s\n", ast_getformatname_multiple(buf, SIPBUFSIZE, capability));

	return AST_SUCCESS;
}

/*! \brief Used for 200 OK and 183 early media */
static int transmit_response_with_t38_sdp(struct sip_pvt *p, char *msg, struct sip_request *req, int retrans)
{
	struct sip_request resp;
	uint32_t seqno;
	
	if (sscanf(get_header(req, "CSeq"), "%30u ", &seqno) != 1) {
		ast_log(LOG_WARNING, "Unable to get seqno from '%s'\n", get_header(req, "CSeq"));
		return -1;
	}
	respprep(&resp, p, msg, req);
	if (p->udptl) {
		add_sdp(&resp, p, 0, 0, 1);
	} else
		ast_log(LOG_ERROR, "Can't add SDP to response, since we have no UDPTL session allocated. Call-ID %s\n", p->callid);
	if (retrans && !p->pendinginvite)
		p->pendinginvite = seqno;		/* Buggy clients sends ACK on RINGING too */
	return send_response(p, &resp, retrans, seqno);
}

/*! \brief copy SIP request (mostly used to save request for responses) */
static void copy_request(struct sip_request *dst, const struct sip_request *src)
{
	/* XXX this function can encounter memory allocation errors, perhaps it
	 * should return a value */

	struct ast_str *duplicate = dst->data;
	struct ast_str *duplicate_content = dst->content;

	/* copy the entire request then restore the original data and content
	 * members from the dst request */
	memcpy(dst, src, sizeof(*dst));
	dst->data = duplicate;
	dst->content = duplicate_content;

	/* copy the data into the dst request */
	if (!dst->data && !(dst->data = ast_str_create(ast_str_strlen(src->data) + 1)))
		return;
	ast_str_copy_string(&dst->data, src->data);

	/* copy the content into the dst request (if it exists) */
	if (src->content) {
		if (!dst->content && !(dst->content = ast_str_create(ast_str_strlen(src->content) + 1)))
			return;
		ast_str_copy_string(&dst->content, src->content);
	}
}

static void add_cc_call_info_to_response(struct sip_pvt *p, struct sip_request *resp)
{
	char uri[SIPBUFSIZE];
	struct ast_str *header = ast_str_alloca(SIPBUFSIZE);
	struct ast_cc_agent *agent = find_sip_cc_agent_by_original_callid(p);
	struct sip_cc_agent_pvt *agent_pvt;

	if (!agent) {
		/* Um, what? How could the SIP_OFFER_CC flag be set but there not be an
		 * agent? Oh well, we'll just warn and return without adding the header.
		 */
		ast_log(LOG_WARNING, "Can't find SIP CC agent for call '%s' even though OFFER_CC flag was set?\n", p->callid);
		return;
	}

	agent_pvt = agent->private_data;

	if (!ast_strlen_zero(agent_pvt->subscribe_uri)) {
		ast_copy_string(uri, agent_pvt->subscribe_uri, sizeof(uri));
	} else {
		generate_uri(p, uri, sizeof(uri));
		ast_copy_string(agent_pvt->subscribe_uri, uri, sizeof(agent_pvt->subscribe_uri));
	}
	/* XXX Hardcode "NR" as the m reason for now. This should perhaps be changed
	 * to be more accurate. This parameter has no bearing on the actual operation
	 * of the feature; it's just there for informational purposes.
	 */
	ast_str_set(&header, 0, "<%s>;purpose=call-completion;m=%s", uri, "NR");
	add_header(resp, "Call-Info", ast_str_buffer(header));
	ao2_ref(agent, -1);
}

/*! \brief Used for 200 OK and 183 early media
	\return Will return XMIT_ERROR for network errors.
*/
static int transmit_response_with_sdp(struct sip_pvt *p, const char *msg, const struct sip_request *req, enum xmittype reliable, int oldsdp, int rpid)
{
	struct sip_request resp;
	uint32_t seqno;
	if (sscanf(get_header(req, "CSeq"), "%30u ", &seqno) != 1) {
		ast_log(LOG_WARNING, "Unable to get seqno from '%s'\n", get_header(req, "CSeq"));
		return -1;
	}
	respprep(&resp, p, msg, req);
	if (rpid == TRUE) {
		add_rpid(&resp, p);
	}
	if (ast_test_flag(&p->flags[0], SIP_OFFER_CC)) {
		add_cc_call_info_to_response(p, &resp);
	}
	if (p->rtp) {
		if (!p->autoframing && !ast_test_flag(&p->flags[0], SIP_OUTGOING)) {
			ast_debug(1, "Setting framing from config on incoming call\n");
			ast_rtp_codecs_packetization_set(ast_rtp_instance_get_codecs(p->rtp), p->rtp, &p->prefs);
		}
		ast_rtp_instance_activate(p->rtp);
		try_suggested_sip_codec(p);
		if (p->t38.state == T38_ENABLED) {
			add_sdp(&resp, p, oldsdp, TRUE, TRUE);
		} else {
			add_sdp(&resp, p, oldsdp, TRUE, FALSE);
		}
	} else
		ast_log(LOG_ERROR, "Can't add SDP to response, since we have no RTP session allocated. Call-ID %s\n", p->callid);
	if (reliable && !p->pendinginvite)
		p->pendinginvite = seqno;		/* Buggy clients sends ACK on RINGING too */
	return send_response(p, &resp, reliable, seqno);
}

/*! \brief Parse first line of incoming SIP request */
static int determine_firstline_parts(struct sip_request *req)
{
	char *e = ast_skip_blanks(req->data->str);	/* there shouldn't be any */
	char *local_rlPart1;

	if (!*e)
		return -1;
	req->rlPart1 = e - req->data->str;	/* method or protocol */
	local_rlPart1 = e;
	e = ast_skip_nonblanks(e);
	if (*e)
		*e++ = '\0';
	/* Get URI or status code */
	e = ast_skip_blanks(e);
	if ( !*e )
		return -1;
	ast_trim_blanks(e);

	if (!strcasecmp(local_rlPart1, "SIP/2.0") ) { /* We have a response */
		if (strlen(e) < 3)	/* status code is 3 digits */
			return -1;
		req->rlPart2 = e - req->data->str;
	} else { /* We have a request */
		if ( *e == '<' ) { /* XXX the spec says it must not be in <> ! */
			ast_debug(3, "Oops. Bogus uri in <> %s\n", e);
			e++;
			if (!*e)
				return -1;
		}
		req->rlPart2 = e - req->data->str;	/* URI */
		e = ast_skip_nonblanks(e);
		if (*e)
			*e++ = '\0';
		e = ast_skip_blanks(e);
		if (strcasecmp(e, "SIP/2.0") ) {
			ast_debug(3, "Skipping packet - Bad request protocol %s\n", e);
			return -1;
		}
	}
	return 1;
}

/*! \brief Transmit reinvite with SDP
\note 	A re-invite is basically a new INVITE with the same CALL-ID and TAG as the
	INVITE that opened the SIP dialogue
	We reinvite so that the audio stream (RTP) go directly between
	the SIP UAs. SIP Signalling stays with * in the path.
	
	If t38version is TRUE, we send T38 SDP for re-invite from audio/video to
	T38 UDPTL transmission on the channel

    If oldsdp is TRUE then the SDP version number is not incremented. This
    is needed for Session-Timers so we can send a re-invite to refresh the
    SIP session without modifying the media session.
*/
static int transmit_reinvite_with_sdp(struct sip_pvt *p, int t38version, int oldsdp)
{
	struct sip_request req;
	
	reqprep(&req, p, ast_test_flag(&p->flags[0], SIP_REINVITE_UPDATE) ?  SIP_UPDATE : SIP_INVITE, 0, 1);

	add_header(&req, "Allow", ALLOWED_METHODS);
	add_supported_header(p, &req);
	if (sipdebug) {
		if (oldsdp == TRUE)
			add_header(&req, "X-asterisk-Info", "SIP re-invite (Session-Timers)");
		else
			add_header(&req, "X-asterisk-Info", "SIP re-invite (External RTP bridge)");
	}

	if (ast_test_flag(&p->flags[0], SIP_SENDRPID))
		add_rpid(&req, p);

	if (p->do_history) {
		append_history(p, "ReInv", "Re-invite sent");
	}
	memset(p->offered_media, 0, sizeof(p->offered_media));

	try_suggested_sip_codec(p);
	if (t38version) {
		add_sdp(&req, p, oldsdp, FALSE, TRUE);
	} else {
		add_sdp(&req, p, oldsdp, TRUE, FALSE);
	}

	/* Use this as the basis */
	initialize_initreq(p, &req);
	p->lastinvite = p->ocseq;
	ast_set_flag(&p->flags[0], SIP_OUTGOING);       /* Change direction of this dialog */
	p->ongoing_reinvite = 1;
	return send_request(p, &req, XMIT_CRITICAL, p->ocseq);
}

/* \brief Remove URI parameters at end of URI, not in username part though */
static char *remove_uri_parameters(char *uri)
{
	char *atsign;
	atsign = strchr(uri, '@');	/* First, locate the at sign */
	if (!atsign) {
		atsign = uri;	/* Ok hostname only, let's stick with the rest */
	}
	atsign = strchr(atsign, ';');	/* Locate semi colon */
	if (atsign)
		*atsign = '\0';	/* Kill at the semi colon */
	return uri;
}

/*! \brief Check Contact: URI of SIP message */
static void extract_uri(struct sip_pvt *p, struct sip_request *req)
{
	char stripped[SIPBUFSIZE];
	char *c;

	ast_copy_string(stripped, get_header(req, "Contact"), sizeof(stripped));
	c = get_in_brackets(stripped);
	/* Cut the URI at the at sign after the @, not in the username part */
	c = remove_uri_parameters(c);
	if (!ast_strlen_zero(c)) {
		ast_string_field_set(p, uri, c);
	}

}

/*! \brief Build contact header - the contact header we send out */
static void build_contact(struct sip_pvt *p)
{
	char tmp[SIPBUFSIZE];
	char *user = ast_uri_encode(p->exten, tmp, sizeof(tmp), 0);

	if (p->socket.type == SIP_TRANSPORT_UDP) {
		ast_string_field_build(p, our_contact, "<sip:%s%s%s>", user,
			ast_strlen_zero(user) ? "" : "@", ast_sockaddr_stringify_remote(&p->ourip));
	} else {
		ast_string_field_build(p, our_contact, "<sip:%s%s%s;transport=%s>", user,
			ast_strlen_zero(user) ? "" : "@", ast_sockaddr_stringify_remote(&p->ourip),
			get_transport(p->socket.type));
	}
}

/*! \brief Initiate new SIP request to peer/user */
static void initreqprep(struct sip_request *req, struct sip_pvt *p, int sipmethod, const char * const explicit_uri)
{
	struct ast_str *invite = ast_str_alloca(256);
	char from[256];
	char to[256];
	char tmp_n[SIPBUFSIZE/2];	/* build a local copy of 'n' if needed */
	char tmp_l[SIPBUFSIZE/2];	/* build a local copy of 'l' if needed */
	const char *l = NULL;	/* XXX what is this, exactly ? */
	const char *n = NULL;	/* XXX what is this, exactly ? */
	const char *d = NULL;	/* domain in from header */
	const char *urioptions = "";
	int ourport;

	if (ast_test_flag(&p->flags[0], SIP_USEREQPHONE)) {
	 	const char *s = p->username;	/* being a string field, cannot be NULL */

		/* Test p->username against allowed characters in AST_DIGIT_ANY
			If it matches the allowed characters list, then sipuser = ";user=phone"
			If not, then sipuser = ""
		*/
		/* + is allowed in first position in a tel: uri */
		if (*s == '+')
			s++;
		for (; *s; s++) {
			if (!strchr(AST_DIGIT_ANYNUM, *s) )
				break;
		}
		/* If we have only digits, add ;user=phone to the uri */
		if (!*s)
			urioptions = ";user=phone";
	}


	snprintf(p->lastmsg, sizeof(p->lastmsg), "Init: %s", sip_methods[sipmethod].text);

	d = S_OR(p->fromdomain, ast_sockaddr_stringify_host_remote(&p->ourip));
	if (p->owner) {
		if ((ast_party_id_presentation(&p->owner->connected.id) & AST_PRES_RESTRICTION) == AST_PRES_ALLOWED) {
			l = p->owner->connected.id.number.valid ? p->owner->connected.id.number.str : NULL;
			n = p->owner->connected.id.name.valid ? p->owner->connected.id.name.str : NULL;
		} else {
			/* Even if we are using RPID, we shouldn't leak information in the From if the user wants
			 * their callerid restricted */
			l = CALLERID_UNKNOWN;
			n = l;
			d = FROMDOMAIN_INVALID;
		}
	}

	/* Hey, it's a NOTIFY! See if they've configured a mwi_from.
	 * XXX Right now, this logic works because the only place that mwi_from
	 * is set on the sip_pvt is in sip_send_mwi_to_peer. If things changed, then
	 * we might end up putting the mwi_from setting into other types of NOTIFY
	 * messages as well.
	 */
	if (sipmethod == SIP_NOTIFY && !ast_strlen_zero(p->mwi_from)) {
		l = p->mwi_from;
	}

	if (ast_strlen_zero(l))
		l = default_callerid;
	if (ast_strlen_zero(n))
		n = l;
	/* Allow user to be overridden */
	if (!ast_strlen_zero(p->fromuser))
		l = p->fromuser;
	else /* Save for any further attempts */
		ast_string_field_set(p, fromuser, l);

	/* Allow user to be overridden */
	if (!ast_strlen_zero(p->fromname))
		n = p->fromname;
	else /* Save for any further attempts */
		ast_string_field_set(p, fromname, n);

	if (sip_cfg.pedanticsipchecking) {
		ast_escape_quoted(n, tmp_n, sizeof(tmp_n));
		n = tmp_n;
		ast_uri_encode(l, tmp_l, sizeof(tmp_l), 0);
		l = tmp_l;
	}

	ourport = (p->fromdomainport) ? p->fromdomainport : ast_sockaddr_port(&p->ourip);
	if (!sip_standard_port(p->socket.type, ourport)) {
		snprintf(from, sizeof(from), "\"%s\" <sip:%s@%s:%d>;tag=%s", n, l, d, ourport, p->tag);
	} else {
		snprintf(from, sizeof(from), "\"%s\" <sip:%s@%s>;tag=%s", n, l, d, p->tag);
	}

	if (!ast_strlen_zero(explicit_uri)) {
		ast_str_set(&invite, 0, "%s", explicit_uri);
	} else {
		/* If we're calling a registered SIP peer, use the fullcontact to dial to the peer */
		if (!ast_strlen_zero(p->fullcontact)) {
			/* If we have full contact, trust it */
			ast_str_append(&invite, 0, "%s", p->fullcontact);
		} else {
			/* Otherwise, use the username while waiting for registration */
			ast_str_append(&invite, 0, "sip:");
			if (!ast_strlen_zero(p->username)) {
				n = p->username;
				if (sip_cfg.pedanticsipchecking) {
					ast_uri_encode(n, tmp_n, sizeof(tmp_n), 0);
					n = tmp_n;
				}
				ast_str_append(&invite, 0, "%s@", n);
			}
			ast_str_append(&invite, 0, "%s", p->tohost);
			if (p->portinuri) {
				ast_str_append(&invite, 0, ":%d", ast_sockaddr_port(&p->sa));
			}
			ast_str_append(&invite, 0, "%s", urioptions);
		}
	}

	/* If custom URI options have been provided, append them */
	if (p->options && !ast_strlen_zero(p->options->uri_options))
		ast_str_append(&invite, 0, ";%s", p->options->uri_options);
	
 	/* This is the request URI, which is the next hop of the call
 		which may or may not be the destination of the call
 	*/
	ast_string_field_set(p, uri, invite->str);

 	if (!ast_strlen_zero(p->todnid)) {
 		/*! \todo Need to add back the VXML URL here at some point, possibly use build_string for all this junk */
 		if (!strchr(p->todnid, '@')) {
 			/* We have no domain in the dnid */
 			snprintf(to, sizeof(to), "<sip:%s@%s>%s%s", p->todnid, p->tohost, ast_strlen_zero(p->theirtag) ? "" : ";tag=", p->theirtag);
 		} else {
 			snprintf(to, sizeof(to), "<sip:%s>%s%s", p->todnid, ast_strlen_zero(p->theirtag) ? "" : ";tag=", p->theirtag);
 		}
 	} else {
 		if (sipmethod == SIP_NOTIFY && !ast_strlen_zero(p->theirtag)) {
 			/* If this is a NOTIFY, use the From: tag in the subscribe (RFC 3265) */
			snprintf(to, sizeof(to), "<%s%s>;tag=%s", (strncasecmp(p->uri, "sip:", 4) ? "sip:" : ""), p->uri, p->theirtag);
 		} else if (p->options && p->options->vxml_url) {
 			/* If there is a VXML URL append it to the SIP URL */
 			snprintf(to, sizeof(to), "<%s>;%s", p->uri, p->options->vxml_url);
 		} else {
 			snprintf(to, sizeof(to), "<%s>", p->uri);
		}
 	}

	init_req(req, sipmethod, p->uri);
	/* now tmp_n is available so reuse it to build the CSeq */
	snprintf(tmp_n, sizeof(tmp_n), "%u %s", ++p->ocseq, sip_methods[sipmethod].text);

	add_header(req, "Via", p->via);
	add_header_max_forwards(p, req);
	/* This will be a no-op most of the time. However, under certain circumstances,
	 * NOTIFY messages will use this function for preparing the request and should
	 * have Route headers present.
	 */
	add_route(req, p->route);

	add_header(req, "From", from);
	add_header(req, "To", to);
	ast_string_field_set(p, exten, l);
	build_contact(p);
	add_header(req, "Contact", p->our_contact);
	add_header(req, "Call-ID", p->callid);
	add_header(req, "CSeq", tmp_n);
	if (!ast_strlen_zero(global_useragent)) {
		add_header(req, "User-Agent", global_useragent);
	}
}

/*! \brief Add "Diversion" header to outgoing message
 *
 * We need to add a Diversion header if the owner channel of
 * this dialog has redirecting information associated with it.
 *
 * \param req The request/response to which we will add the header
 * \param pvt The sip_pvt which represents the call-leg
 */
static void add_diversion_header(struct sip_request *req, struct sip_pvt *pvt)
{
	const char *diverting_number;
	const char *diverting_name;
	const char *reason;
	char header_text[256];

	if (!pvt->owner) {
		return;
	}

	diverting_number = pvt->owner->redirecting.from.number.str;
	if (!pvt->owner->redirecting.from.number.valid
		|| ast_strlen_zero(diverting_number)) {
		return;
	}

	reason = sip_reason_code_to_str(pvt->owner->redirecting.reason);

	/* We at least have a number to place in the Diversion header, which is enough */
	diverting_name = pvt->owner->redirecting.from.name.str;
	if (!pvt->owner->redirecting.from.name.valid
		|| ast_strlen_zero(diverting_name)) {
		snprintf(header_text, sizeof(header_text), "<sip:%s@%s>;reason=%s", diverting_number,
				ast_sockaddr_stringify_host_remote(&pvt->ourip), reason);
	} else {
		snprintf(header_text, sizeof(header_text), "\"%s\" <sip:%s@%s>;reason=%s",
				diverting_name, diverting_number,
				ast_sockaddr_stringify_host_remote(&pvt->ourip), reason);
	}

	add_header(req, "Diversion", header_text);
}

static int transmit_publish(struct sip_epa_entry *epa_entry, enum sip_publish_type publish_type, const char * const explicit_uri)
{
	struct sip_pvt *pvt;
	int expires;

	epa_entry->publish_type = publish_type;

	if (!(pvt = sip_alloc(NULL, NULL, 0, SIP_PUBLISH, NULL))) {
		return -1;
	}

	sip_pvt_lock(pvt);

	if (create_addr(pvt, epa_entry->destination, NULL, TRUE, NULL)) {
		sip_pvt_unlock(pvt);
		dialog_unlink_all(pvt);
		dialog_unref(pvt, "create_addr failed in transmit_publish. Unref dialog");
		return -1;
	}
	ast_sip_ouraddrfor(&pvt->sa, &pvt->ourip, pvt);
	ast_set_flag(&pvt->flags[0], SIP_OUTGOING);
	expires = (publish_type == SIP_PUBLISH_REMOVE) ? 0 : DEFAULT_PUBLISH_EXPIRES;
	pvt->expiry = expires;

	/* Bump refcount for sip_pvt's reference */
	ao2_ref(epa_entry, +1);
	pvt->epa_entry = epa_entry;

	transmit_invite(pvt, SIP_PUBLISH, FALSE, 2, explicit_uri);
	sip_pvt_unlock(pvt);
	sip_scheddestroy(pvt, DEFAULT_TRANS_TIMEOUT);
	dialog_unref(pvt, "Done with the sip_pvt allocated for transmitting PUBLISH");
	return 0;
}

/*! 
 * \brief Build REFER/INVITE/OPTIONS/SUBSCRIBE message and transmit it
 * \param p sip_pvt structure
 * \param sipmethod
 * \param sdp unknown
 * \param init 0 = Prepare request within dialog, 1= prepare request, new branch,
 *  2= prepare new request and new dialog. do_proxy_auth calls this with init!=2
 * \param explicit_uri
*/
static int transmit_invite(struct sip_pvt *p, int sipmethod, int sdp, int init, const char * const explicit_uri)
{
	struct sip_request req;
	struct ast_variable *var;
	
	req.method = sipmethod;
	if (init) {/* Bump branch even on initial requests */
		p->branch ^= ast_random();
		p->invite_branch = p->branch;
		build_via(p);
	}
	if (init > 1) {
		initreqprep(&req, p, sipmethod, explicit_uri);
	} else {
		/* If init=1, we should not generate a new branch. If it's 0, we need a new branch. */
		reqprep(&req, p, sipmethod, 0, init ? 0 : 1);
	}

	if (p->options && p->options->auth) {
		add_header(&req, p->options->authheader, p->options->auth);
	}
	append_date(&req);
	if (sipmethod == SIP_REFER) {	/* Call transfer */
		if (p->refer) {
			char buf[SIPBUFSIZE];
			if (!ast_strlen_zero(p->refer->refer_to)) {
				add_header(&req, "Refer-To", p->refer->refer_to);
			}
			if (!ast_strlen_zero(p->refer->referred_by)) {
				snprintf(buf, sizeof(buf), "%s <%s>", p->refer->referred_by_name, p->refer->referred_by);
				add_header(&req, "Referred-By", buf);
			}
		}
	} else if (sipmethod == SIP_SUBSCRIBE) {
		char buf[SIPBUFSIZE];
		if (p->subscribed == MWI_NOTIFICATION) {
			add_header(&req, "Event", "message-summary");
			add_header(&req, "Accept", "application/simple-message-summary");
		} else if (p->subscribed == CALL_COMPLETION) {
			add_header(&req, "Event", "call-completion");
			add_header(&req, "Accept", "application/call-completion");
		}
		snprintf(buf, sizeof(buf), "%d", p->expiry);
		add_header(&req, "Expires", buf);
	}

	/* This new INVITE is part of an attended transfer. Make sure that the
	other end knows and replace the current call with this new call */
	if (p->options && !ast_strlen_zero(p->options->replaces)) {
		add_header(&req, "Replaces", p->options->replaces);
		add_header(&req, "Require", "replaces");
	}

	/* Add Session-Timers related headers */
	if (st_get_mode(p, 0) == SESSION_TIMER_MODE_ORIGINATE) {
		char i2astr[10];

		if (!p->stimer->st_interval) {
			p->stimer->st_interval = st_get_se(p, TRUE);
		}

		p->stimer->st_active = TRUE;
		
		snprintf(i2astr, sizeof(i2astr), "%d", p->stimer->st_interval);
		add_header(&req, "Session-Expires", i2astr);
		snprintf(i2astr, sizeof(i2astr), "%d", st_get_se(p, FALSE));
		add_header(&req, "Min-SE", i2astr);
	}

	add_header(&req, "Allow", ALLOWED_METHODS);
	add_supported_header(p, &req);

	if (p->options && p->options->addsipheaders && p->owner) {
		struct ast_channel *chan = p->owner; /* The owner channel */
		struct varshead *headp;
	
		ast_channel_lock(chan);

		headp = &chan->varshead;

		if (!headp) {
			ast_log(LOG_WARNING, "No Headp for the channel...ooops!\n");
		} else {
			const struct ast_var_t *current;
			AST_LIST_TRAVERSE(headp, current, entries) {
				/* SIPADDHEADER: Add SIP header to outgoing call */
				if (!strncasecmp(ast_var_name(current), "SIPADDHEADER", strlen("SIPADDHEADER"))) {
					char *content, *end;
					const char *header = ast_var_value(current);
					char *headdup = ast_strdupa(header);

					/* Strip of the starting " (if it's there) */
					if (*headdup == '"') {
				 		headdup++;
					}
					if ((content = strchr(headdup, ':'))) {
						*content++ = '\0';
						content = ast_skip_blanks(content); /* Skip white space */
						/* Strip the ending " (if it's there) */
				 		end = content + strlen(content) -1;	
						if (*end == '"') {
							*end = '\0';
						}
					
						add_header(&req, headdup, content);
						if (sipdebug) {
							ast_debug(1, "Adding SIP Header \"%s\" with content :%s: \n", headdup, content);
						}
					}
				}
			}
		}

		ast_channel_unlock(chan);
	}
	if ((sipmethod == SIP_INVITE || sipmethod == SIP_UPDATE) && ast_test_flag(&p->flags[0], SIP_SENDRPID))
		add_rpid(&req, p);
	if (sipmethod == SIP_INVITE) {
		add_diversion_header(&req, p);
	}
	if (sdp) {
		memset(p->offered_media, 0, sizeof(p->offered_media));
		if (p->udptl && p->t38.state == T38_LOCAL_REINVITE) {
			ast_debug(1, "T38 is in state %d on channel %s\n", p->t38.state, p->owner ? p->owner->name : "<none>");
			add_sdp(&req, p, FALSE, FALSE, TRUE);
		} else if (p->rtp) {
			try_suggested_sip_codec(p);
			add_sdp(&req, p, FALSE, TRUE, FALSE);
		}
	} else if (p->notify) {
		for (var = p->notify->headers; var; var = var->next) {
			add_header(&req, var->name, var->value);
		}
		if (ast_str_strlen(p->notify->content)) {
			add_content(&req, ast_str_buffer(p->notify->content));
		}
	} else if (sipmethod == SIP_PUBLISH) {
		char expires[SIPBUFSIZE];

		switch (p->epa_entry->static_data->event) {
		case CALL_COMPLETION:
			snprintf(expires, sizeof(expires), "%d", p->expiry);
			add_header(&req, "Event", "call-completion");
			add_header(&req, "Expires", expires);
			if (p->epa_entry->publish_type != SIP_PUBLISH_INITIAL) {
				add_header(&req, "SIP-If-Match", p->epa_entry->entity_tag);
			}

			if (!ast_strlen_zero(p->epa_entry->body)) {
				add_header(&req, "Content-Type", "application/pidf+xml");
				add_content(&req, p->epa_entry->body);
			}
		default:
			break;
		}
	}

	if (!p->initreq.headers || init > 2) {
		initialize_initreq(p, &req);
	}
	if (sipmethod == SIP_INVITE || sipmethod == SIP_SUBSCRIBE) {
		p->lastinvite = p->ocseq;
	}
	return send_request(p, &req, init ? XMIT_CRITICAL : XMIT_RELIABLE, p->ocseq);
}

/*! \brief Send a subscription or resubscription for MWI */
static int sip_subscribe_mwi_do(const void *data)
{
	struct sip_subscription_mwi *mwi = (struct sip_subscription_mwi*)data;
	
	if (!mwi) {
		return -1;
	}
	
	mwi->resub = -1;
	__sip_subscribe_mwi_do(mwi);
	ASTOBJ_UNREF(mwi, sip_subscribe_mwi_destroy);
	
	return 0;
}

static void on_dns_update_registry(struct ast_sockaddr *old, struct ast_sockaddr *new, void *data)
{
	struct sip_registry *reg = data;
	const char *old_str;

	/* This shouldn't happen, but just in case */
	if (ast_sockaddr_isnull(new)) {
		ast_debug(1, "Empty sockaddr change...ignoring!\n");
		return;
	}

	if (!ast_sockaddr_port(new)) {
		ast_sockaddr_set_port(new, reg->portno);
	}

	old_str = ast_strdupa(ast_sockaddr_stringify(old));

	ast_debug(1, "Changing registry %s from %s to %s\n", S_OR(reg->peername, reg->hostname), old_str, ast_sockaddr_stringify(new));
	ast_sockaddr_copy(&reg->us, new);
}

static void on_dns_update_peer(struct ast_sockaddr *old, struct ast_sockaddr *new, void *data)
{
	struct sip_peer *peer = data;
	const char *old_str;

	/* This shouldn't happen, but just in case */
	if (ast_sockaddr_isnull(new)) {
		ast_debug(1, "Empty sockaddr change...ignoring!\n");
		return;
	}

	if (!ast_sockaddr_isnull(&peer->addr)) {
		ao2_unlink(peers_by_ip, peer);
	}

	if (!ast_sockaddr_port(new)) {
		ast_sockaddr_set_port(new, default_sip_port(peer->socket.type));
	}

	old_str = ast_strdupa(ast_sockaddr_stringify(old));
	ast_debug(1, "Changing peer %s address from %s to %s\n", peer->name, old_str, ast_sockaddr_stringify(new));

	ao2_lock(peer);
	ast_sockaddr_copy(&peer->addr, new);
	ao2_unlock(peer);

	ao2_link(peers_by_ip, peer);
}

static void on_dns_update_mwi(struct ast_sockaddr *old, struct ast_sockaddr *new, void *data)
{
	struct sip_subscription_mwi *mwi = data;
	const char *old_str;

	/* This shouldn't happen, but just in case */
	if (ast_sockaddr_isnull(new)) {
		ast_debug(1, "Empty sockaddr change...ignoring!\n");
		return;
	}

	old_str = ast_strdupa(ast_sockaddr_stringify(old));
	ast_debug(1, "Changing mwi %s from %s to %s\n", mwi->hostname, old_str, ast_sockaddr_stringify(new));
	ast_sockaddr_copy(&mwi->us, new);
}

/*! \brief Actually setup an MWI subscription or resubscribe */
static int __sip_subscribe_mwi_do(struct sip_subscription_mwi *mwi)
{
	/* If we have no DNS manager let's do a lookup */
	if (!mwi->dnsmgr) {
		char transport[MAXHOSTNAMELEN];
		struct sip_subscription_mwi *saved;
		snprintf(transport, sizeof(transport), "_%s._%s", get_srv_service(mwi->transport), get_srv_protocol(mwi->transport));

		mwi->us.ss.ss_family = get_address_family_filter(&bindaddr); /* Filter address family */
		saved = ASTOBJ_REF(mwi);
		ast_dnsmgr_lookup_cb(mwi->hostname, &mwi->us, &mwi->dnsmgr, sip_cfg.srvlookup ? transport : NULL, on_dns_update_mwi, saved);
		if (!mwi->dnsmgr) {
			ASTOBJ_UNREF(saved, sip_subscribe_mwi_destroy); /* dnsmgr disabled, remove reference */
		}
	}

	/* If we already have a subscription up simply send a resubscription */
	if (mwi->call) {
		transmit_invite(mwi->call, SIP_SUBSCRIBE, 0, 0, NULL);
		return 0;
	}
	
	/* Create a dialog that we will use for the subscription */
	if (!(mwi->call = sip_alloc(NULL, NULL, 0, SIP_SUBSCRIBE, NULL))) {
		return -1;
	}

	ref_proxy(mwi->call, obproxy_get(mwi->call, NULL));

	if (!ast_sockaddr_port(&mwi->us) && mwi->portno) {
		ast_sockaddr_set_port(&mwi->us, mwi->portno);
	}
	
	/* Setup the destination of our subscription */
	if (create_addr(mwi->call, mwi->hostname, &mwi->us, 0, NULL)) {
		dialog_unlink_all(mwi->call);
		mwi->call = dialog_unref(mwi->call, "unref dialog after unlink_all");
		return 0;
	}

	mwi->call->expiry = mwi_expiry;
	
	if (!mwi->dnsmgr && mwi->portno) {
		ast_sockaddr_set_port(&mwi->call->sa, mwi->portno);
		ast_sockaddr_set_port(&mwi->call->recv, mwi->portno);
	} else {
		mwi->portno = ast_sockaddr_port(&mwi->call->sa);
	}
	
	/* Set various other information */
	if (!ast_strlen_zero(mwi->authuser)) {
		ast_string_field_set(mwi->call, peername, mwi->authuser);
		ast_string_field_set(mwi->call, authname, mwi->authuser);
		ast_string_field_set(mwi->call, fromuser, mwi->authuser);
	} else {
		ast_string_field_set(mwi->call, peername, mwi->username);
		ast_string_field_set(mwi->call, authname, mwi->username);
		ast_string_field_set(mwi->call, fromuser, mwi->username);
	}
	ast_string_field_set(mwi->call, username, mwi->username);
	if (!ast_strlen_zero(mwi->secret)) {
		ast_string_field_set(mwi->call, peersecret, mwi->secret);
	}
	set_socket_transport(&mwi->call->socket, mwi->transport);
	mwi->call->socket.port = htons(mwi->portno);
	ast_sip_ouraddrfor(&mwi->call->sa, &mwi->call->ourip, mwi->call);
	build_contact(mwi->call);
	build_via(mwi->call);

	/* Change the dialog callid. */
	change_callid_pvt(mwi->call, NULL);

	ast_set_flag(&mwi->call->flags[0], SIP_OUTGOING);
	
	/* Associate the call with us */
	mwi->call->mwi = ASTOBJ_REF(mwi);

	mwi->call->subscribed = MWI_NOTIFICATION;

	/* Actually send the packet */
	transmit_invite(mwi->call, SIP_SUBSCRIBE, 0, 2, NULL);

	return 0;
}

/*! \brief Find the channel that is causing the RINGING update */
static int find_calling_channel(void *obj, void *arg, void *data, int flags)
{
	struct ast_channel *c = obj;
	struct sip_pvt *p = data;
	int res;

	ast_channel_lock(c);

	res = (c->pbx &&
			(!strcasecmp(c->macroexten, p->exten) || !strcasecmp(c->exten, p->exten)) &&
			(sip_cfg.notifycid == IGNORE_CONTEXT || !strcasecmp(c->context, p->context)));

	ast_channel_unlock(c);

	return res ? CMP_MATCH | CMP_STOP : 0;
}

/*! \brief Builds XML portion of NOTIFY messages for presence or dialog updates */
static void state_notify_build_xml(int state, int full, const char *exten, const char *context, struct ast_str **tmp, struct sip_pvt *p, int subscribed, const char *mfrom, const char *mto)
{
	enum state { NOTIFY_OPEN, NOTIFY_INUSE, NOTIFY_CLOSED } local_state = NOTIFY_OPEN;
	const char *statestring = "terminated";
	const char *pidfstate = "--";
	const char *pidfnote= "Ready";
	char hint[AST_MAX_EXTENSION];

	switch (state) {
	case (AST_EXTENSION_RINGING | AST_EXTENSION_INUSE):
		statestring = (sip_cfg.notifyringing) ? "early" : "confirmed";
		local_state = NOTIFY_INUSE;
		pidfstate = "busy";
		pidfnote = "Ringing";
		break;
	case AST_EXTENSION_RINGING:
		statestring = "early";
		local_state = NOTIFY_INUSE;
		pidfstate = "busy";
		pidfnote = "Ringing";
		break;
	case AST_EXTENSION_INUSE:
		statestring = "confirmed";
		local_state = NOTIFY_INUSE;
		pidfstate = "busy";
		pidfnote = "On the phone";
		break;
	case AST_EXTENSION_BUSY:
		statestring = "confirmed";
		local_state = NOTIFY_CLOSED;
		pidfstate = "busy";
		pidfnote = "On the phone";
		break;
	case AST_EXTENSION_UNAVAILABLE:
		statestring = "terminated";
		local_state = NOTIFY_CLOSED;
		pidfstate = "away";
		pidfnote = "Unavailable";
		break;
	case AST_EXTENSION_ONHOLD:
		statestring = "confirmed";
		local_state = NOTIFY_CLOSED;
		pidfstate = "busy";
		pidfnote = "On hold";
		break;
	case AST_EXTENSION_NOT_INUSE:
	default:
		/* Default setting */
		break;
	}

	/* Check which device/devices we are watching  and if they are registered */
	if (ast_get_hint(hint, sizeof(hint), NULL, 0, NULL, context, exten)) {
		char *hint2 = hint, *individual_hint = NULL;
		int hint_count = 0, unavailable_count = 0;

		while ((individual_hint = strsep(&hint2, "&"))) {
			hint_count++;

			if (ast_device_state(individual_hint) == AST_DEVICE_UNAVAILABLE)
				unavailable_count++;
		}

		/* If none of the hinted devices are registered, we will
		 * override notification and show no availability.
		 */
		if (hint_count > 0 && hint_count == unavailable_count) {
			local_state = NOTIFY_CLOSED;
			pidfstate = "away";
			pidfnote = "Not online";
		}
	}

	switch (subscribed) {
	case XPIDF_XML:
	case CPIM_PIDF_XML:
		ast_str_append(tmp, 0,
			"<?xml version=\"1.0\"?>\n"
			"<!DOCTYPE presence PUBLIC \"-//IETF//DTD RFCxxxx XPIDF 1.0//EN\" \"xpidf.dtd\">\n"
			"<presence>\n");
		ast_str_append(tmp, 0, "<presentity uri=\"%s;method=SUBSCRIBE\" />\n", mfrom);
		ast_str_append(tmp, 0, "<atom id=\"%s\">\n", exten);
		ast_str_append(tmp, 0, "<address uri=\"%s;user=ip\" priority=\"0.800000\">\n", mto);
		ast_str_append(tmp, 0, "<status status=\"%s\" />\n", (local_state ==  NOTIFY_OPEN) ? "open" : (local_state == NOTIFY_INUSE) ? "inuse" : "closed");
		ast_str_append(tmp, 0, "<msnsubstatus substatus=\"%s\" />\n", (local_state == NOTIFY_OPEN) ? "online" : (local_state == NOTIFY_INUSE) ? "onthephone" : "offline");
		ast_str_append(tmp, 0, "</address>\n</atom>\n</presence>\n");
		break;
	case PIDF_XML: /* Eyebeam supports this format */
		ast_str_append(tmp, 0,
			"<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>\n"
			"<presence xmlns=\"urn:ietf:params:xml:ns:pidf\" \nxmlns:pp=\"urn:ietf:params:xml:ns:pidf:person\"\nxmlns:es=\"urn:ietf:params:xml:ns:pidf:rpid:status:rpid-status\"\nxmlns:ep=\"urn:ietf:params:xml:ns:pidf:rpid:rpid-person\"\nentity=\"%s\">\n", mfrom);
		ast_str_append(tmp, 0, "<pp:person><status>\n");
		if (pidfstate[0] != '-') {
			ast_str_append(tmp, 0, "<ep:activities><ep:%s/></ep:activities>\n", pidfstate);
		}
		ast_str_append(tmp, 0, "</status></pp:person>\n");
		ast_str_append(tmp, 0, "<note>%s</note>\n", pidfnote); /* Note */
		ast_str_append(tmp, 0, "<tuple id=\"%s\">\n", exten); /* Tuple start */
		ast_str_append(tmp, 0, "<contact priority=\"1\">%s</contact>\n", mto);
		if (pidfstate[0] == 'b') /* Busy? Still open ... */
			ast_str_append(tmp, 0, "<status><basic>open</basic></status>\n");
		else
			ast_str_append(tmp, 0, "<status><basic>%s</basic></status>\n", (local_state != NOTIFY_CLOSED) ? "open" : "closed");
		ast_str_append(tmp, 0, "</tuple>\n</presence>\n");
		break;
	case DIALOG_INFO_XML: /* SNOM subscribes in this format */
		ast_str_append(tmp, 0, "<?xml version=\"1.0\"?>\n");
		ast_str_append(tmp, 0, "<dialog-info xmlns=\"urn:ietf:params:xml:ns:dialog-info\" version=\"%u\" state=\"%s\" entity=\"%s\">\n", p->dialogver, full ? "full" : "partial", mto);
		if ((state & AST_EXTENSION_RINGING) && sip_cfg.notifyringing) {
			const char *local_display = exten;
			char *local_target = ast_strdupa(mto);

			/* There are some limitations to how this works.  The primary one is that the
			   callee must be dialing the same extension that is being monitored.  Simply dialing
			   the hint'd device is not sufficient. */
			if (sip_cfg.notifycid) {
				struct ast_channel *caller;

				if ((caller = ast_channel_callback(find_calling_channel, NULL, p, 0))) {
					char *cid_num;
					int need;

					ast_channel_lock(caller);
					cid_num = S_COR(caller->caller.id.number.valid,
						caller->caller.id.number.str, "");
					need = strlen(cid_num) + strlen(p->fromdomain) + sizeof("sip:@");
					local_target = alloca(need);
					snprintf(local_target, need, "sip:%s@%s", cid_num, p->fromdomain);
					local_display = ast_strdupa(S_COR(caller->caller.id.name.valid,
						caller->caller.id.name.str, ""));
					ast_channel_unlock(caller);
					caller = ast_channel_unref(caller);
				}

				/* We create a fake call-id which the phone will send back in an INVITE
				   Replaces header which we can grab and do some magic with. */
				if (sip_cfg.pedanticsipchecking) {
					ast_str_append(tmp, 0, "<dialog id=\"%s\" call-id=\"pickup-%s\" local-tag=\"%s\" remote-tag=\"%s\" direction=\"recipient\">\n",
						exten, p->callid, p->theirtag, p->tag);
				} else {
					ast_str_append(tmp, 0, "<dialog id=\"%s\" call-id=\"pickup-%s\" direction=\"recipient\">\n",
						exten, p->callid);
				}
				ast_str_append(tmp, 0,
						"<remote>\n"
						/* See the limitations of this above.  Luckily the phone seems to still be
						   happy when these values are not correct. */
						"<identity display=\"%s\">%s</identity>\n"
						"<target uri=\"%s\"/>\n"
						"</remote>\n"
						"<local>\n"
						"<identity>%s</identity>\n"
						"<target uri=\"%s\"/>\n"
						"</local>\n",
						local_display, local_target, local_target, mto, mto);
			} else {
				ast_str_append(tmp, 0, "<dialog id=\"%s\" direction=\"recipient\">\n", exten);
			}

		} else {
			ast_str_append(tmp, 0, "<dialog id=\"%s\">\n", exten);
		}
		ast_str_append(tmp, 0, "<state>%s</state>\n", statestring);
		if (state == AST_EXTENSION_ONHOLD) {
				ast_str_append(tmp, 0, "<local>\n<target uri=\"%s\">\n"
			                                    "<param pname=\"+sip.rendering\" pvalue=\"no\"/>\n"
			                                    "</target>\n</local>\n", mto);
		}
		ast_str_append(tmp, 0, "</dialog>\n</dialog-info>\n");
		break;
	case NONE:
	default:
		break;
	}
}

static int transmit_cc_notify(struct ast_cc_agent *agent, struct sip_pvt *subscription, enum sip_cc_notify_state state)
{
	struct sip_request req;
	struct sip_cc_agent_pvt *agent_pvt = agent->private_data;
	char uri[SIPBUFSIZE];
	char state_str[64];
	char subscription_state_hdr[64];

	if (state < CC_QUEUED || state > CC_READY) {
		ast_log(LOG_WARNING, "Invalid state provided for transmit_cc_notify (%d)\n", state);
		return -1;
	}

	reqprep(&req, subscription, SIP_NOTIFY, 0, TRUE);
	snprintf(state_str, sizeof(state_str), "%s\r\n", sip_cc_notify_state_map[state].state_string);
	add_header(&req, "Event", "call-completion");
	add_header(&req, "Content-Type", "application/call-completion");
	snprintf(subscription_state_hdr, sizeof(subscription_state_hdr), "active;expires=%d", subscription->expiry);
	add_header(&req, "Subscription-State", subscription_state_hdr);
	if (state == CC_READY) {
		generate_uri(subscription, agent_pvt->notify_uri, sizeof(agent_pvt->notify_uri));
		snprintf(uri, sizeof(uri) - 1, "cc-URI: %s\r\n", agent_pvt->notify_uri);
	}
	add_content(&req, state_str);
	if (state == CC_READY) {
		add_content(&req, uri);
	}
	return send_request(subscription, &req, XMIT_RELIABLE, subscription->ocseq);
}

/*! \brief Used in the SUBSCRIBE notification subsystem (RFC3265) */
static int transmit_state_notify(struct sip_pvt *p, int state, int full, int timeout)
{
	struct ast_str *tmp = ast_str_alloca(4000);
	char from[256], to[256];
	char *c, *mfrom, *mto;
	struct sip_request req;
	const struct cfsubscription_types *subscriptiontype;

	memset(from, 0, sizeof(from));
	memset(to, 0, sizeof(to));

	subscriptiontype = find_subscription_type(p->subscribed);

	ast_copy_string(from, get_header(&p->initreq, "From"), sizeof(from));
	c = get_in_brackets(from);
	if (strncasecmp(c, "sip:", 4) && strncasecmp(c, "sips:", 5)) {
		ast_log(LOG_WARNING, "Huh?  Not a SIP header (%s)?\n", c);
		return -1;
	}

	mfrom = remove_uri_parameters(c);

	ast_copy_string(to, get_header(&p->initreq, "To"), sizeof(to));
	c = get_in_brackets(to);
	if (strncasecmp(c, "sip:", 4) && strncasecmp(c, "sips:", 5)) {
		ast_log(LOG_WARNING, "Huh?  Not a SIP header (%s)?\n", c);
		return -1;
	}
	mto = remove_uri_parameters(c);

	reqprep(&req, p, SIP_NOTIFY, 0, 1);

	switch(state) {
	case AST_EXTENSION_DEACTIVATED:
		if (timeout)
			add_header(&req, "Subscription-State", "terminated;reason=timeout");
		else {
			add_header(&req, "Subscription-State", "terminated;reason=probation");
			add_header(&req, "Retry-After", "60");
		}
		break;
	case AST_EXTENSION_REMOVED:
		add_header(&req, "Subscription-State", "terminated;reason=noresource");
		break;
	default:
		if (p->expiry)
			add_header(&req, "Subscription-State", "active");
		else	/* Expired */
			add_header(&req, "Subscription-State", "terminated;reason=timeout");
	}

	switch (p->subscribed) {
	case XPIDF_XML:
	case CPIM_PIDF_XML:
		add_header(&req, "Event", subscriptiontype->event);
		state_notify_build_xml(state, full, p->exten, p->context, &tmp, p, p->subscribed, mfrom, mto);
		add_header(&req, "Content-Type", subscriptiontype->mediatype);
		p->dialogver++;
		break;
	case PIDF_XML: /* Eyebeam supports this format */
		add_header(&req, "Event", subscriptiontype->event);
		state_notify_build_xml(state, full, p->exten, p->context, &tmp, p, p->subscribed, mfrom, mto);
		add_header(&req, "Content-Type", subscriptiontype->mediatype);
		p->dialogver++;
		break;
	case DIALOG_INFO_XML: /* SNOM subscribes in this format */
		add_header(&req, "Event", subscriptiontype->event);
		state_notify_build_xml(state, full, p->exten, p->context, &tmp, p, p->subscribed, mfrom, mto);
		add_header(&req, "Content-Type", subscriptiontype->mediatype);
		p->dialogver++;
		break;
	case NONE:
	default:
		break;
	}

	add_content(&req, tmp->str);

	p->pendinginvite = p->ocseq;	/* Remember that we have a pending NOTIFY in order not to confuse the NOTIFY subsystem */

	return send_request(p, &req, XMIT_RELIABLE, p->ocseq);
}

/*! \brief Notify user of messages waiting in voicemail (RFC3842)
\note	- Notification only works for registered peers with mailbox= definitions
	in sip.conf
	- We use the SIP Event package message-summary
	 MIME type defaults to  "application/simple-message-summary";
 */
static int transmit_notify_with_mwi(struct sip_pvt *p, int newmsgs, int oldmsgs, const char *vmexten)
{
	struct sip_request req;
	struct ast_str *out = ast_str_alloca(500);
	int ourport = (p->fromdomainport) ? p->fromdomainport : ast_sockaddr_port(&p->ourip);
	const char *domain;
	const char *exten = S_OR(vmexten, default_vmexten);

	initreqprep(&req, p, SIP_NOTIFY, NULL);
	add_header(&req, "Event", "message-summary");
	add_header(&req, "Content-Type", default_notifymime);
	ast_str_append(&out, 0, "Messages-Waiting: %s\r\n", newmsgs ? "yes" : "no");

	/* domain initialization occurs here because initreqprep changes ast_sockaddr_stringify string. */
	domain = S_OR(p->fromdomain, ast_sockaddr_stringify_host_remote(&p->ourip));

	if (!sip_standard_port(p->socket.type, ourport)) {
		if (p->socket.type == SIP_TRANSPORT_UDP) {
			ast_str_append(&out, 0, "Message-Account: sip:%s@%s:%d\r\n", exten, domain, ourport);
		} else {
			ast_str_append(&out, 0, "Message-Account: sip:%s@%s:%d;transport=%s\r\n", exten, domain, ourport, get_transport(p->socket.type));
		}
	} else {
		if (p->socket.type == SIP_TRANSPORT_UDP) {
			ast_str_append(&out, 0, "Message-Account: sip:%s@%s\r\n", exten, domain);
		} else {
			ast_str_append(&out, 0, "Message-Account: sip:%s@%s;transport=%s\r\n", exten, domain, get_transport(p->socket.type));
		}
	}
	/* Cisco has a bug in the SIP stack where it can't accept the
		(0/0) notification. This can temporarily be disabled in
		sip.conf with the "buggymwi" option */
	ast_str_append(&out, 0, "Voice-Message: %d/%d%s\r\n",
		newmsgs, oldmsgs, (ast_test_flag(&p->flags[1], SIP_PAGE2_BUGGY_MWI) ? "" : " (0/0)"));

	if (p->subscribed) {
		if (p->expiry) {
			add_header(&req, "Subscription-State", "active");
		} else {	/* Expired */
			add_header(&req, "Subscription-State", "terminated;reason=timeout");
		}
	}

	add_content(&req, out->str);

	if (!p->initreq.headers) {
		initialize_initreq(p, &req);
	}
	return send_request(p, &req, XMIT_RELIABLE, p->ocseq);
}

/*! \brief Notify a transferring party of the status of transfer (RFC3515) */
static int transmit_notify_with_sipfrag(struct sip_pvt *p, int cseq, char *message, int terminate)
{
	struct sip_request req;
	char tmp[SIPBUFSIZE/2];
	
	reqprep(&req, p, SIP_NOTIFY, 0, 1);
	snprintf(tmp, sizeof(tmp), "refer;id=%d", cseq);
	add_header(&req, "Event", tmp);
	add_header(&req, "Subscription-state", terminate ? "terminated;reason=noresource" : "active");
	add_header(&req, "Content-Type", "message/sipfrag;version=2.0");
	add_header(&req, "Allow", ALLOWED_METHODS);
	add_supported_header(p, &req);

	snprintf(tmp, sizeof(tmp), "SIP/2.0 %s\r\n", message);
	add_content(&req, tmp);

	if (!p->initreq.headers) {
		initialize_initreq(p, &req);
	}

	return send_request(p, &req, XMIT_RELIABLE, p->ocseq);
}

static int manager_sipnotify(struct mansession *s, const struct message *m)
{
	const char *channame = astman_get_header(m, "Channel");
	struct ast_variable *vars = astman_get_variables(m);
	struct sip_pvt *p;
	struct ast_variable *header, *var;

	if (ast_strlen_zero(channame)) {
		astman_send_error(s, m, "SIPNotify requires a channel name");
		return 0;
	}

	if (!strncasecmp(channame, "sip/", 4)) {
		channame += 4;
	}

	if (!(p = sip_alloc(NULL, NULL, 0, SIP_NOTIFY, NULL))) {
		astman_send_error(s, m, "Unable to build sip pvt data for notify (memory/socket error)");
		return 0;
	}

	if (create_addr(p, channame, NULL, 0, NULL)) {
		/* Maybe they're not registered, etc. */
		dialog_unlink_all(p);
		dialog_unref(p, "unref dialog inside for loop" );
		/* sip_destroy(p); */
		astman_send_error(s, m, "Could not create address");
		return 0;
	}

	/* Notify is outgoing call */
	ast_set_flag(&p->flags[0], SIP_OUTGOING);
	sip_notify_allocate(p);

	p->notify->headers = header = ast_variable_new("Subscription-State", "terminated", "");

	for (var = vars; var; var = var->next) {
		if (!strcasecmp(var->name, "Content")) {
			if (ast_str_strlen(p->notify->content))
				ast_str_append(&p->notify->content, 0, "\r\n");
			ast_str_append(&p->notify->content, 0, "%s", var->value);
		} else if (!strcasecmp(var->name, "Content-Length")) {
			ast_log(LOG_WARNING, "it is not necessary to specify Content-Length, ignoring\n");
		} else {
			header->next = ast_variable_new(var->name, var->value, "");
			header = header->next;
		}
	}

	sip_scheddestroy(p, SIP_TRANS_TIMEOUT);
	transmit_invite(p, SIP_NOTIFY, 0, 2, NULL);
	dialog_unref(p, "bump down the count of p since we're done with it.");

	astman_send_ack(s, m, "Notify Sent");
	ast_variables_destroy(vars);
	return 0;
}

/*! \brief Send a provisional response indicating that a call was redirected
 */
static void update_redirecting(struct sip_pvt *p, const void *data, size_t datalen)
{
	struct sip_request resp;

	if (p->owner->_state == AST_STATE_UP || ast_test_flag(&p->flags[0], SIP_OUTGOING)) {
		return;
	}

	respprep(&resp, p, "181 Call is being forwarded", &p->initreq);
	add_diversion_header(&resp, p);
	send_response(p, &resp, XMIT_UNRELIABLE, 0);
}

/*! \brief Notify peer that the connected line has changed */
static void update_connectedline(struct sip_pvt *p, const void *data, size_t datalen)
{

	if (!ast_test_flag(&p->flags[0], SIP_SENDRPID)) {
		return;
	}
	if (!p->owner->connected.id.number.valid
		|| ast_strlen_zero(p->owner->connected.id.number.str)) {
		return;
	}

	append_history(p, "ConnectedLine", "%s party is now %s <%s>",
		ast_test_flag(&p->flags[0], SIP_OUTGOING) ? "Calling" : "Called",
		S_COR(p->owner->connected.id.name.valid, p->owner->connected.id.name.str, ""),
		S_COR(p->owner->connected.id.number.valid, p->owner->connected.id.number.str, ""));

	if (p->owner->_state == AST_STATE_UP || ast_test_flag(&p->flags[0], SIP_OUTGOING)) {
		struct sip_request req;

		if (!p->pendinginvite && (p->invitestate == INV_CONFIRMED || p->invitestate == INV_TERMINATED)) {
			reqprep(&req, p, ast_test_flag(&p->flags[0], SIP_REINVITE_UPDATE) ? SIP_UPDATE : SIP_INVITE, 0, 1);

			add_header(&req, "Allow", ALLOWED_METHODS);
			add_supported_header(p, &req);
			add_rpid(&req, p);
			add_sdp(&req, p, FALSE, TRUE, FALSE);

			initialize_initreq(p, &req);
			p->lastinvite = p->ocseq;
			ast_set_flag(&p->flags[0], SIP_OUTGOING);
			p->invitestate = INV_CALLING;
			send_request(p, &req, XMIT_CRITICAL, p->ocseq);
		} else if ((is_method_allowed(&p->allowed_methods, SIP_UPDATE)) && (!ast_strlen_zero(p->okcontacturi))) { 
			reqprep(&req, p, SIP_UPDATE, 0, 1);
			add_rpid(&req, p);
			add_header(&req, "X-Asterisk-rpid-update", "Yes");
			send_request(p, &req, XMIT_CRITICAL, p->ocseq);
		} else {
			/* We cannot send the update yet, so we have to wait until we can */
			ast_set_flag(&p->flags[0], SIP_NEEDREINVITE);
		}
	} else {
		ast_set_flag(&p->flags[1], SIP_PAGE2_CONNECTLINEUPDATE_PEND);
		if (ast_test_flag(&p->flags[1], SIP_PAGE2_RPID_IMMEDIATE)) {
			struct sip_request resp;

			if ((p->owner->_state == AST_STATE_RING) && !ast_test_flag(&p->flags[0], SIP_PROGRESS_SENT)) {
				ast_clear_flag(&p->flags[1], SIP_PAGE2_CONNECTLINEUPDATE_PEND);
				respprep(&resp, p, "180 Ringing", &p->initreq);
				add_rpid(&resp, p);
				send_response(p, &resp, XMIT_UNRELIABLE, 0);
				ast_set_flag(&p->flags[0], SIP_RINGING);
			} else if (p->owner->_state == AST_STATE_RINGING) {
				ast_clear_flag(&p->flags[1], SIP_PAGE2_CONNECTLINEUPDATE_PEND);
				respprep(&resp, p, "183 Session Progress", &p->initreq);
				add_rpid(&resp, p);
				send_response(p, &resp, XMIT_UNRELIABLE, 0);
				ast_set_flag(&p->flags[0], SIP_PROGRESS_SENT);
			} else {
				ast_debug(1, "Unable able to send update to '%s' in state '%s'\n", p->owner->name, ast_state2str(p->owner->_state));
			}
		}
	}
}

static const struct _map_x_s regstatestrings[] = {
	{ REG_STATE_FAILED,     "Failed" },
	{ REG_STATE_UNREGISTERED, "Unregistered"},
	{ REG_STATE_REGSENT, "Request Sent"},
	{ REG_STATE_AUTHSENT, "Auth. Sent"},
	{ REG_STATE_REGISTERED, "Registered"},
	{ REG_STATE_REJECTED, "Rejected"},
	{ REG_STATE_TIMEOUT, "Timeout"},
	{ REG_STATE_NOAUTH, "No Authentication"},
	{ -1, NULL } /* terminator */
};

/*! \brief Convert registration state status to string */
static const char *regstate2str(enum sipregistrystate regstate)
{
	return map_x_s(regstatestrings, regstate, "Unknown");
}

/*! \brief Update registration with SIP Proxy.
 * Called from the scheduler when the previous registration expires,
 * so we don't have to cancel the pending event.
 * We assume the reference so the sip_registry is valid, since it
 * is stored in the scheduled event anyways.
 */
static int sip_reregister(const void *data)
{
	/* if we are here, we know that we need to reregister. */
	struct sip_registry *r = (struct sip_registry *) data;

	/* if we couldn't get a reference to the registry object, punt */
	if (!r) {
		return 0;
	}

	if (r->call && r->call->do_history) {
		append_history(r->call, "RegistryRenew", "Account: %s@%s", r->username, r->hostname);
	}
	/* Since registry's are only added/removed by the the monitor thread, this
	   may be overkill to reference/dereference at all here */
	if (sipdebug) {
		ast_log(LOG_NOTICE, "   -- Re-registration for  %s@%s\n", r->username, r->hostname);
	}

	r->expire = -1;
	r->expiry = r->configured_expiry;
	__sip_do_register(r);
	registry_unref(r, "unref the re-register scheduled event");
	return 0;
}

/*! \brief Register with SIP proxy 
	\return see \ref __sip_xmit 
*/
static int __sip_do_register(struct sip_registry *r)
{
	int res;

	res = transmit_register(r, SIP_REGISTER, NULL, NULL);
	return res;
}

/*! \brief Registration timeout, register again
 * Registered as a timeout handler during transmit_register(),
 * to retransmit the packet if a reply does not come back.
 * This is called by the scheduler so the event is not pending anymore when
 * we are called.
 */
static int sip_reg_timeout(const void *data)
{

	/* if we are here, our registration timed out, so we'll just do it over */
	struct sip_registry *r = (struct sip_registry *)data; /* the ref count should have been bumped when the sched item was added */
	struct sip_pvt *p;

	/* if we couldn't get a reference to the registry object, punt */
	if (!r) {
		return 0;
	}

	if (r->dnsmgr) {
		/* If the registration has timed out, maybe the IP changed.  Force a refresh. */
		ast_dnsmgr_refresh(r->dnsmgr);
	}

	/* If the initial tranmission failed, we may not have an existing dialog,
	 * so it is possible that r->call == NULL.
	 * Otherwise destroy it, as we have a timeout so we don't want it.
	 */
	if (r->call) {
		/* Unlink us, destroy old call.  Locking is not relevant here because all this happens
		   in the single SIP manager thread. */
		p = r->call;
		sip_pvt_lock(p);
		pvt_set_needdestroy(p, "registration timeout");
		/* Pretend to ACK anything just in case */
		__sip_pretend_ack(p);
		sip_pvt_unlock(p);

		/* decouple the two objects */
		/* p->registry == r, so r has 2 refs, and the unref won't take the object away */
		if (p->registry) {
			p->registry = registry_unref(p->registry, "p->registry unreffed");
		}
		r->call = dialog_unref(r->call, "unrefing r->call");
	}
	/* If we have a limit, stop registration and give up */
	r->timeout = -1;
	if (global_regattempts_max && r->regattempts >= global_regattempts_max) {
		/* Ok, enough is enough. Don't try any more */
		/* We could add an external notification here...
			steal it from app_voicemail :-) */
		ast_log(LOG_NOTICE, "   -- Last Registration Attempt #%d failed, Giving up forever trying to register '%s@%s'\n", r->regattempts, r->username, r->hostname);
		r->regstate = REG_STATE_FAILED;
	} else {
		r->regstate = REG_STATE_UNREGISTERED;
		transmit_register(r, SIP_REGISTER, NULL, NULL);
		ast_log(LOG_NOTICE, "   -- Registration for '%s@%s' timed out, trying again (Attempt #%d)\n", r->username, r->hostname, r->regattempts);
	}
	manager_event(EVENT_FLAG_SYSTEM, "Registry", "ChannelType: SIP\r\nUsername: %s\r\nDomain: %s\r\nStatus: %s\r\n", r->username, r->hostname, regstate2str(r->regstate));
	registry_unref(r, "unreffing registry_unref r");
	return 0;
}

static const char *sip_sanitized_host(const char *host)
{
	struct ast_sockaddr addr = { { 0, 0, }, };

	/* peer/sip_pvt->tohost and sip_registry->hostname should never have a port
	 * in them, so we use PARSE_PORT_FORBID here. If this lookup fails, we return
	 * the original host which is most likely a host name and not an IP. */
	if (!ast_sockaddr_parse(&addr, host, PARSE_PORT_FORBID)) {
		return host;
	}
	return ast_sockaddr_stringify_host_remote(&addr);
}

/*! \brief Transmit register to SIP proxy or UA
 * auth = NULL on the initial registration (from sip_reregister())
 */
static int transmit_register(struct sip_registry *r, int sipmethod, const char *auth, const char *authheader)
{
	struct sip_request req;
	char from[256];
	char to[256];
	char tmp[80];
	char addr[80];
	struct sip_pvt *p;
	struct sip_peer *peer = NULL;
	int res;
	int portno = 0;

	/* exit if we are already in process with this registrar ?*/
	if (r == NULL || ((auth == NULL) && (r->regstate == REG_STATE_REGSENT || r->regstate == REG_STATE_AUTHSENT))) {
		if (r) {
			ast_log(LOG_NOTICE, "Strange, trying to register %s@%s when registration already pending\n", r->username, r->hostname);
		}
		return 0;
	}

	if (r->dnsmgr == NULL) {
		char transport[MAXHOSTNAMELEN];
		peer = find_peer(r->hostname, NULL, TRUE, FINDPEERS, FALSE, 0);
		snprintf(transport, sizeof(transport), "_%s._%s",get_srv_service(r->transport), get_srv_protocol(r->transport)); /* have to use static get_transport function */
		r->us.ss.ss_family = get_address_family_filter(&bindaddr); /* Filter address family */

		/* No point in doing a DNS lookup of the register hostname if we're just going to
		 * end up using an outbound proxy. obproxy_get is safe to call with either of r->call
		 * or peer NULL. Since we're only concerned with its existence, we're not going to
		 * bother getting a ref to the proxy*/
		if (!obproxy_get(r->call, peer)) {
			registry_addref(r, "add reg ref for dnsmgr");
			ast_dnsmgr_lookup_cb(peer ? peer->tohost : r->hostname, &r->us, &r->dnsmgr, sip_cfg.srvlookup ? transport : NULL, on_dns_update_registry, r);
			if (!r->dnsmgr) {
				/*dnsmgr refresh disabled, no reference added! */
				registry_unref(r, "remove reg ref, dnsmgr disabled");
			}
		}
		if (peer) {
			peer = unref_peer(peer, "removing peer ref for dnsmgr_lookup");
		}
	}

	if (r->call) {	/* We have a registration */
		if (!auth) {
			ast_log(LOG_WARNING, "Already have a REGISTER going on to %s@%s?? \n", r->username, r->hostname);
			return 0;
		} else {
			p = dialog_ref(r->call, "getting a copy of the r->call dialog in transmit_register");
			make_our_tag(p->tag, sizeof(p->tag));	/* create a new local tag for every register attempt */
			ast_string_field_set(p, theirtag, NULL);	/* forget their old tag, so we don't match tags when getting response */
		}
	} else {
		/* Build callid for registration if we haven't registered before */
		if (!r->callid_valid) {
			build_callid_registry(r, &internip, default_fromdomain);
			r->callid_valid = TRUE;
		}
		/* Allocate SIP dialog for registration */
		if (!(p = sip_alloc( r->callid, NULL, 0, SIP_REGISTER, NULL))) {
			ast_log(LOG_WARNING, "Unable to allocate registration transaction (memory or socket error)\n");
			return 0;
		}
		
		if (p->do_history) {
			append_history(p, "RegistryInit", "Account: %s@%s", r->username, r->hostname);
		}

		/* Use port number specified if no SRV record was found */
		if (!ast_sockaddr_isnull(&r->us)) {
			if (!ast_sockaddr_port(&r->us) && r->portno) {
				ast_sockaddr_set_port(&r->us, r->portno);
			}

			/* It is possible that DNS was unavailable at the time the peer was created.
			 * Here, if we've updated the address in the registry via manually calling
			 * ast_dnsmgr_lookup_cb() above, then we call the same function that dnsmgr would
			 * call if it was updating a peer's address */
			if ((peer = find_peer(S_OR(r->peername, r->hostname), NULL, TRUE, FINDPEERS, FALSE, 0))) {
				if (ast_sockaddr_cmp(&peer->addr, &r->us)) {
					on_dns_update_peer(&peer->addr, &r->us, peer);
				}
				peer = unref_peer(peer, "unref after find_peer");
			}
		}

		/* Find address to hostname */
		if (create_addr(p, S_OR(r->peername, r->hostname), &r->us, 0, NULL)) {
			/* we have what we hope is a temporary network error,
			 * probably DNS.  We need to reschedule a registration try */
			dialog_unlink_all(p);
			p = dialog_unref(p, "unref dialog after unlink_all");
			if (r->timeout > -1) {
				AST_SCHED_REPLACE_UNREF(r->timeout, sched, global_reg_timeout * 1000, sip_reg_timeout, r,
										registry_unref(_data, "del for REPLACE of registry ptr"),
										registry_unref(r, "object ptr dec when SCHED_REPLACE add failed"),
										registry_addref(r,"add for REPLACE registry ptr"));
				ast_log(LOG_WARNING, "Still have a registration timeout for %s@%s (create_addr() error), %d\n", r->username, r->hostname, r->timeout);
			} else {
				r->timeout = ast_sched_add(sched, global_reg_timeout * 1000, sip_reg_timeout, registry_addref(r, "add for REPLACE registry ptr"));
				ast_log(LOG_WARNING, "Probably a DNS error for registration to %s@%s, trying REGISTER again (after %d seconds)\n", r->username, r->hostname, global_reg_timeout);
			}
			r->regattempts++;
			return 0;
		}

		/* Copy back Call-ID in case create_addr changed it */
		ast_string_field_set(r, callid, p->callid);

		if (!r->dnsmgr && r->portno) {
			ast_sockaddr_set_port(&p->sa, r->portno);
			ast_sockaddr_set_port(&p->recv, r->portno);
		}
		if (!ast_strlen_zero(p->fromdomain)) {
			portno = (p->fromdomainport) ? p->fromdomainport : STANDARD_SIP_PORT;
		} else if (!ast_strlen_zero(r->regdomain)) {
			portno = (r->regdomainport) ? r->regdomainport : STANDARD_SIP_PORT;
		} else {
			portno = ast_sockaddr_port(&p->sa);
		}

		ast_set_flag(&p->flags[0], SIP_OUTGOING);	/* Registration is outgoing call */
		r->call = dialog_ref(p, "copying dialog into registry r->call");		/* Save pointer to SIP dialog */
		p->registry = registry_addref(r, "transmit_register: addref to p->registry in transmit_register");	/* Add pointer to registry in packet */
		if (!ast_strlen_zero(r->secret)) {	/* Secret (password) */
			ast_string_field_set(p, peersecret, r->secret);
		}
		if (!ast_strlen_zero(r->md5secret))
			ast_string_field_set(p, peermd5secret, r->md5secret);
		/* User name in this realm
		- if authuser is set, use that, otherwise use username */
		if (!ast_strlen_zero(r->authuser)) {
			ast_string_field_set(p, peername, r->authuser);
			ast_string_field_set(p, authname, r->authuser);
		} else if (!ast_strlen_zero(r->username)) {
			ast_string_field_set(p, peername, r->username);
			ast_string_field_set(p, authname, r->username);
			ast_string_field_set(p, fromuser, r->username);
		}
		if (!ast_strlen_zero(r->username)) {
			ast_string_field_set(p, username, r->username);
		}
		/* Save extension in packet */
		if (!ast_strlen_zero(r->callback)) {
			ast_string_field_set(p, exten, r->callback);
		}

		/* Set transport and port so the correct contact is built */
		set_socket_transport(&p->socket, r->transport);
		if (r->transport == SIP_TRANSPORT_TLS || r->transport == SIP_TRANSPORT_TCP) {
			p->socket.port =
			    htons(ast_sockaddr_port(&sip_tcp_desc.local_address));
		}

		/*
		  check which address we should use in our contact header
		  based on whether the remote host is on the external or
		  internal network so we can register through nat
		 */
		ast_sip_ouraddrfor(&p->sa, &p->ourip, p);
		build_contact(p);
	}

	/* set up a timeout */
	if (auth == NULL)  {
		if (r->timeout > -1) {
			ast_log(LOG_WARNING, "Still have a registration timeout, #%d - deleting it\n", r->timeout);
		}
		AST_SCHED_REPLACE_UNREF(r->timeout, sched, global_reg_timeout * 1000, sip_reg_timeout, r,
								registry_unref(_data,"reg ptr unrefed from del in SCHED_REPLACE"),
								registry_unref(r,"reg ptr unrefed from add failure in SCHED_REPLACE"),
								registry_addref(r,"reg ptr reffed from add in SCHED_REPLACE"));
		ast_debug(1, "Scheduled a registration timeout for %s id  #%d \n", r->hostname, r->timeout);
	}

	snprintf(from, sizeof(from), "<sip:%s@%s>;tag=%s", r->username, S_OR(r->regdomain, sip_sanitized_host(p->tohost)), p->tag);
	if (!ast_strlen_zero(p->theirtag)) {
		snprintf(to, sizeof(to), "<sip:%s@%s>;tag=%s", r->username, S_OR(r->regdomain, sip_sanitized_host(p->tohost)), p->theirtag);
	} else {
		snprintf(to, sizeof(to), "<sip:%s@%s>", r->username, S_OR(r->regdomain, sip_sanitized_host(p->tohost)));
	}

	/* Fromdomain is what we are registering to, regardless of actual
  	   host name from SRV */
	if (portno && portno != STANDARD_SIP_PORT) {
		snprintf(addr, sizeof(addr), "sip:%s:%d", S_OR(p->fromdomain,S_OR(r->regdomain, sip_sanitized_host(r->hostname))), portno);
	} else {
		snprintf(addr, sizeof(addr), "sip:%s", S_OR(p->fromdomain,S_OR(r->regdomain, sip_sanitized_host(r->hostname))));
	}

	ast_string_field_set(p, uri, addr);

	p->branch ^= ast_random();

	init_req(&req, sipmethod, addr);

	/* Add to CSEQ */
	snprintf(tmp, sizeof(tmp), "%u %s", ++r->ocseq, sip_methods[sipmethod].text);
	p->ocseq = r->ocseq;

	build_via(p);
	add_header(&req, "Via", p->via);
	add_header_max_forwards(p, &req);
	add_header(&req, "From", from);
	add_header(&req, "To", to);
	add_header(&req, "Call-ID", p->callid);
	add_header(&req, "CSeq", tmp);
	if (!ast_strlen_zero(global_useragent))
		add_header(&req, "User-Agent", global_useragent);

	if (auth) { 	/* Add auth header */
		add_header(&req, authheader, auth);
	} else if (!ast_strlen_zero(r->nonce)) {
		char digest[1024];

		/* We have auth data to reuse, build a digest header.
		 * Note, this is not always useful because some parties do not
		 * like nonces to be reused (for good reasons!) so they will
		 * challenge us anyways.
		 */
		if (sipdebug) {
			ast_debug(1, "   >>> Re-using Auth data for %s@%s\n", r->username, r->hostname);
		}
		ast_string_field_set(p, realm, r->realm);
		ast_string_field_set(p, nonce, r->nonce);
		ast_string_field_set(p, domain, r->authdomain);
		ast_string_field_set(p, opaque, r->opaque);
		ast_string_field_set(p, qop, r->qop);
		p->noncecount = ++r->noncecount;

		memset(digest, 0, sizeof(digest));
		if(!build_reply_digest(p, sipmethod, digest, sizeof(digest))) {
			add_header(&req, "Authorization", digest);
		} else {
			ast_log(LOG_NOTICE, "No authorization available for authentication of registration to %s@%s\n", r->username, r->hostname);
		}
	}

	snprintf(tmp, sizeof(tmp), "%d", r->expiry);
	add_header(&req, "Expires", tmp);
	add_header(&req, "Contact", p->our_contact);

	initialize_initreq(p, &req);
	if (sip_debug_test_pvt(p)) {
		ast_verbose("REGISTER %d headers, %d lines\n", p->initreq.headers, p->initreq.lines);
	}
	r->regstate = auth ? REG_STATE_AUTHSENT : REG_STATE_REGSENT;
	r->regattempts++;	/* Another attempt */
	ast_debug(4, "REGISTER attempt %d to %s@%s\n", r->regattempts, r->username, r->hostname);
	res = send_request(p, &req, XMIT_CRITICAL, p->ocseq);
	dialog_unref(p, "p is finished here at the end of transmit_register");
	return res;
}

/*! \brief Transmit text with SIP MESSAGE method */
static int transmit_message_with_text(struct sip_pvt *p, const char *text)
{
	struct sip_request req;
	
	reqprep(&req, p, SIP_MESSAGE, 0, 1);
	add_text(&req, text);
	return send_request(p, &req, XMIT_RELIABLE, p->ocseq);
}

/*! \brief Allocate SIP refer structure */
static int sip_refer_allocate(struct sip_pvt *p)
{
	p->refer = ast_calloc(1, sizeof(struct sip_refer));
	return p->refer ? 1 : 0;
}

/*! \brief Allocate SIP refer structure */
static int sip_notify_allocate(struct sip_pvt *p)
{
	p->notify = ast_calloc(1, sizeof(struct sip_notify));
	if (p->notify) {
		p->notify->content = ast_str_create(128);
	}
	return p->notify ? 1 : 0;
}

/*! \brief Transmit SIP REFER message (initiated by the transfer() dialplan application
	\note this is currently broken as we have no way of telling the dialplan
	engine whether a transfer succeeds or fails.
	\todo Fix the transfer() dialplan function so that a transfer may fail
*/
static int transmit_refer(struct sip_pvt *p, const char *dest)
{
	struct sip_request req = {
		.headers = 0,	
	};
	char from[256];
	const char *of;
	char *c;
	char referto[256];
	int	use_tls=FALSE;

	if (sipdebug) {
		ast_debug(1, "SIP transfer of %s to %s\n", p->callid, dest);
	}

	/* Are we transfering an inbound or outbound call ? */
	if (ast_test_flag(&p->flags[0], SIP_OUTGOING))  {
		of = get_header(&p->initreq, "To");
	} else {
		of = get_header(&p->initreq, "From");
	}

	ast_copy_string(from, of, sizeof(from));
	of = get_in_brackets(from);
	ast_string_field_set(p, from, of);
	if (!strncasecmp(of, "sip:", 4)) {
		of += 4;
	} else if (!strncasecmp(of, "sips:", 5)) {
		of += 5;
		use_tls = TRUE;
	} else {
		ast_log(LOG_NOTICE, "From address missing 'sip(s):', assuming sip:\n");
	}
	/* Get just the username part */
	if ((c = strchr(dest, '@'))) {
		c = NULL;
	} else if ((c = strchr(of, '@'))) {
		*c++ = '\0';
	}
	if (c) {
		snprintf(referto, sizeof(referto), "<sip%s:%s@%s>", use_tls ? "s" : "", dest, c);
	} else {
		snprintf(referto, sizeof(referto), "<sip%s:%s>", use_tls ? "s" : "", dest);
	}

	/* save in case we get 407 challenge */
	sip_refer_allocate(p);
	ast_copy_string(p->refer->refer_to, referto, sizeof(p->refer->refer_to));
	ast_copy_string(p->refer->referred_by, p->our_contact, sizeof(p->refer->referred_by));
	p->refer->status = REFER_SENT;   /* Set refer status */

	reqprep(&req, p, SIP_REFER, 0, 1);

	add_header(&req, "Refer-To", referto);
	add_header(&req, "Allow", ALLOWED_METHODS);
	add_supported_header(p, &req);
	if (!ast_strlen_zero(p->our_contact)) {
		add_header(&req, "Referred-By", p->our_contact);
	}

	return send_request(p, &req, XMIT_RELIABLE, p->ocseq);

	/* We should propably wait for a NOTIFY here until we ack the transfer */
	/* Maybe fork a new thread and wait for a STATUS of REFER_200OK on the refer status before returning to app_transfer */

	/*! \todo In theory, we should hang around and wait for a reply, before
	returning to the dial plan here. Don't know really how that would
	affect the transfer() app or the pbx, but, well, to make this
	useful we should have a STATUS code on transfer().
	*/
}

/*! \brief Send SIP INFO advice of charge message */
static int transmit_info_with_aoc(struct sip_pvt *p, struct ast_aoc_decoded *decoded)
{
	struct sip_request req;
	struct ast_str *str = ast_str_alloca(512);
	const struct ast_aoc_unit_entry *unit_entry = ast_aoc_get_unit_info(decoded, 0);
	enum ast_aoc_charge_type charging = ast_aoc_get_charge_type(decoded);

	reqprep(&req, p, SIP_INFO, 0, 1);

	if (ast_aoc_get_msg_type(decoded) == AST_AOC_D) {
		ast_str_append(&str, 0, "type=active;");
	} else if (ast_aoc_get_msg_type(decoded) == AST_AOC_E) {
		ast_str_append(&str, 0, "type=terminated;");
	} else {
		/* unsupported message type */
		return -1;
	}

	switch (charging) {
	case AST_AOC_CHARGE_FREE:
		ast_str_append(&str, 0, "free-of-charge;");
		break;
	case AST_AOC_CHARGE_CURRENCY:
		ast_str_append(&str, 0, "charging;");
		ast_str_append(&str, 0, "charging-info=currency;");
		ast_str_append(&str, 0, "amount=%u;", ast_aoc_get_currency_amount(decoded));
		ast_str_append(&str, 0, "multiplier=%s;", ast_aoc_get_currency_multiplier_decimal(decoded));
		if (!ast_strlen_zero(ast_aoc_get_currency_name(decoded))) {
			ast_str_append(&str, 0, "currency=%s;", ast_aoc_get_currency_name(decoded));
		}
		break;
	case AST_AOC_CHARGE_UNIT:
		ast_str_append(&str, 0, "charging;");
		ast_str_append(&str, 0, "charging-info=pulse;");
		if (unit_entry) {
			ast_str_append(&str, 0, "recorded-units=%u;", unit_entry->amount);
		}
		break;
	default:
		ast_str_append(&str, 0, "not-available;");
	};

	add_header(&req, "AOC", ast_str_buffer(str));

	return send_request(p, &req, XMIT_RELIABLE, p->ocseq);
}

/*! \brief Send SIP INFO dtmf message, see Cisco documentation on cisco.com */
static int transmit_info_with_digit(struct sip_pvt *p, const char digit, unsigned int duration)
{
	struct sip_request req;
	
	reqprep(&req, p, SIP_INFO, 0, 1);
	add_digit(&req, digit, duration, (ast_test_flag(&p->flags[0], SIP_DTMF) == SIP_DTMF_SHORTINFO));
	return send_request(p, &req, XMIT_RELIABLE, p->ocseq);
}

/*! \brief Send SIP INFO with video update request */
static int transmit_info_with_vidupdate(struct sip_pvt *p)
{
	struct sip_request req;
	
	reqprep(&req, p, SIP_INFO, 0, 1);
	add_vidupdate(&req);
	return send_request(p, &req, XMIT_RELIABLE, p->ocseq);
}

/*! \brief Transmit generic SIP request
	returns XMIT_ERROR if transmit failed with a critical error (don't retry)
*/
static int transmit_request(struct sip_pvt *p, int sipmethod, uint32_t seqno, enum xmittype reliable, int newbranch)
{
	struct sip_request resp;
	
	reqprep(&resp, p, sipmethod, seqno, newbranch);
	if (sipmethod == SIP_CANCEL && p->answered_elsewhere) {
		add_header(&resp, "Reason", "SIP;cause=200;text=\"Call completed elsewhere\"");
	}

	if (sipmethod == SIP_ACK) {
		p->invitestate = INV_CONFIRMED;
	}

	return send_request(p, &resp, reliable, seqno ? seqno : p->ocseq);
}

/*! \brief return the request and response header for a 401 or 407 code */
static void auth_headers(enum sip_auth_type code, char **header, char **respheader)
{
	if (code == WWW_AUTH) {			/* 401 */
		*header = "WWW-Authenticate";
		*respheader = "Authorization";
	} else if (code == PROXY_AUTH) {	/* 407 */
		*header = "Proxy-Authenticate";
		*respheader = "Proxy-Authorization";
	} else {
		ast_verbose("-- wrong response code %d\n", code);
		*header = *respheader = "Invalid";
	}
}

/*! \brief Transmit SIP request, auth added */
static int transmit_request_with_auth(struct sip_pvt *p, int sipmethod, uint32_t seqno, enum xmittype reliable, int newbranch)
{
	struct sip_request resp;
	
	reqprep(&resp, p, sipmethod, seqno, newbranch);
	if (!ast_strlen_zero(p->realm)) {
		char digest[1024];

		memset(digest, 0, sizeof(digest));
		if(!build_reply_digest(p, sipmethod, digest, sizeof(digest))) {
			char *dummy, *response;
			enum sip_auth_type code = p->options ? p->options->auth_type : PROXY_AUTH; /* XXX force 407 if unknown */
			auth_headers(code, &dummy, &response);
			add_header(&resp, response, digest);
		} else {
			ast_log(LOG_WARNING, "No authentication available for call %s\n", p->callid);
		}
	}
	/* If we are hanging up and know a cause for that, send it in clear text to make
		debugging easier. */
	if (sipmethod == SIP_BYE)	{
		char buf[20];

		if (ast_test_flag(&p->flags[1], SIP_PAGE2_Q850_REASON) && p->hangupcause) {
			sprintf(buf, "Q.850;cause=%i", p->hangupcause & 0x7f);
			add_header(&resp, "Reason", buf);
		}

		add_header(&resp, "X-Asterisk-HangupCause", ast_cause2str(p->hangupcause));
		snprintf(buf, sizeof(buf), "%d", p->hangupcause);
		add_header(&resp, "X-Asterisk-HangupCauseCode", buf);
	}

	return send_request(p, &resp, reliable, seqno ? seqno : p->ocseq);	
}

/*! \brief Remove registration data from realtime database or AST/DB when registration expires */
static void destroy_association(struct sip_peer *peer)
{
	int realtimeregs = ast_check_realtime("sipregs");
	char *tablename = (realtimeregs) ? "sipregs" : "sippeers";

	if (!sip_cfg.ignore_regexpire) {
		if (peer->rt_fromcontact && sip_cfg.peer_rtupdate) {
			ast_update_realtime(tablename, "name", peer->name, "fullcontact", "", "ipaddr", "", "port", "", "regseconds", "0", "regserver", "", "useragent", "", "lastms", "0", SENTINEL);
		} else {
			ast_db_del("SIP/Registry", peer->name);
			ast_db_del("SIP/PeerMethods", peer->name);
		}
	}
}

static void set_socket_transport(struct sip_socket *socket, int transport)
{
	/* if the transport type changes, clear all socket data */
	if (socket->type != transport) {
		socket->fd = -1;
		socket->type = transport;
		if (socket->tcptls_session) {
			ao2_ref(socket->tcptls_session, -1);
			socket->tcptls_session = NULL;
		}
	}
}

/*! \brief Expire registration of SIP peer */
static int expire_register(const void *data)
{
	struct sip_peer *peer = (struct sip_peer *)data;

	if (!peer) {		/* Hmmm. We have no peer. Weird. */
		return 0;
	}

	peer->expire = -1;
	peer->portinuri = 0;

	destroy_association(peer);	/* remove registration data from storage */
	set_socket_transport(&peer->socket, peer->default_outbound_transport);

	if (peer->socket.tcptls_session) {
		ao2_ref(peer->socket.tcptls_session, -1);
		peer->socket.tcptls_session = NULL;
	}

	manager_event(EVENT_FLAG_SYSTEM, "PeerStatus", "ChannelType: SIP\r\nPeer: SIP/%s\r\nPeerStatus: Unregistered\r\nCause: Expired\r\n", peer->name);
	register_peer_exten(peer, FALSE);	/* Remove regexten */
	ast_devstate_changed(AST_DEVICE_UNKNOWN, "SIP/%s", peer->name);

	/* Do we need to release this peer from memory?
		Only for realtime peers and autocreated peers
	*/
	if (peer->is_realtime) {
		ast_debug(3, "-REALTIME- peer expired registration. Name: %s. Realtime peer objects now %d\n", peer->name, rpeerobjs);
	}

	if (peer->selfdestruct ||
	    ast_test_flag(&peer->flags[1], SIP_PAGE2_RTAUTOCLEAR)) {
		unlink_peer_from_tables(peer);
	} else if (!ast_sockaddr_isnull(&peer->addr)) {
		/* If we aren't self-destructing a temp_peer, we still need to unlink the peer
		 * from the peers_by_ip table, otherwise we end up with multiple copies hanging
		 * around each time a registration expires and the peer re-registers. */
		ao2_t_unlink(peers_by_ip, peer, "ao2_unlink of peer from peers_by_ip table");
	}

	/* Only clear the addr after we check for destruction.  The addr must remain
	 * in order to unlink from the peers_by_ip container correctly */
	memset(&peer->addr, 0, sizeof(peer->addr));

	unref_peer(peer, "removing peer ref for expire_register");

	return 0;
}

/*! \brief Poke peer (send qualify to check if peer is alive and well) */
static int sip_poke_peer_s(const void *data)
{
	struct sip_peer *peer = (struct sip_peer *)data;
	struct sip_peer *foundpeer;

	peer->pokeexpire = -1;

	foundpeer = ao2_find(peers, peer, OBJ_POINTER);
	if (!foundpeer) {
		unref_peer(peer, "removing poke peer ref");
		return 0;
	} else if (foundpeer->name != peer->name) {
		unref_peer(foundpeer, "removing above peer ref");
		unref_peer(peer, "removing poke peer ref");
		return 0;
	}

	unref_peer(foundpeer, "removing above peer ref");
	sip_poke_peer(peer, 0);
	unref_peer(peer, "removing poke peer ref");

	return 0;
}

/*! \brief Get registration details from Asterisk DB */
static void reg_source_db(struct sip_peer *peer)
{
	char data[256];
	struct ast_sockaddr sa;
	int expire;
	char full_addr[128];
	AST_DECLARE_APP_ARGS(args,
		AST_APP_ARG(addr);
		AST_APP_ARG(port);
		AST_APP_ARG(expiry_str);
		AST_APP_ARG(username);
		AST_APP_ARG(contact);
	);

	/* If read-only RT backend, then refresh from local DB cache */
	if (peer->rt_fromcontact && sip_cfg.peer_rtupdate) {
		return;
	}
	if (ast_db_get("SIP/Registry", peer->name, data, sizeof(data))) {
		return;
	}

	AST_NONSTANDARD_RAW_ARGS(args, data, ':');

	snprintf(full_addr, sizeof(full_addr), "%s:%s", args.addr, args.port);

	if (!ast_sockaddr_parse(&sa, full_addr, 0)) {
		return;
	}

	if (args.expiry_str) {
		expire = atoi(args.expiry_str);
	} else {
		return;
	}

	if (args.username) {
		ast_string_field_set(peer, username, args.username);
	}
	if (args.contact) {
		ast_string_field_set(peer, fullcontact, args.contact);
	}

	ast_debug(2, "SIP Seeding peer from astdb: '%s' at %s@%s for %d\n",
	    peer->name, peer->username, ast_sockaddr_stringify_host(&sa), expire);

	ast_sockaddr_copy(&peer->addr, &sa);
	if (sipsock < 0) {
		/* SIP isn't up yet, so schedule a poke only, pretty soon */
		AST_SCHED_REPLACE_UNREF(peer->pokeexpire, sched, ast_random() % 5000 + 1, sip_poke_peer_s, peer,
				unref_peer(_data, "removing poke peer ref"),
				unref_peer(peer, "removing poke peer ref"),
				ref_peer(peer, "adding poke peer ref"));
	} else {
		sip_poke_peer(peer, 0);
	}
	AST_SCHED_REPLACE_UNREF(peer->expire, sched, (expire + 10) * 1000, expire_register, peer,
			unref_peer(_data, "remove registration ref"),
			unref_peer(peer, "remove registration ref"),
			ref_peer(peer, "add registration ref"));
	register_peer_exten(peer, TRUE);
}

/*! \brief Save contact header for 200 OK on INVITE */
static int parse_ok_contact(struct sip_pvt *pvt, struct sip_request *req)
{
	char contact[SIPBUFSIZE];
	char *c;

	/* Look for brackets */
	ast_copy_string(contact, get_header(req, "Contact"), sizeof(contact));
	c = get_in_brackets(contact);

	/* Save full contact to call pvt for later bye or re-invite */
	ast_string_field_set(pvt, fullcontact, c);

	/* Save URI for later ACKs, BYE or RE-invites */
	ast_string_field_set(pvt, okcontacturi, c);

	/* We should return false for URI:s we can't handle,
		like tel:, mailto:,ldap: etc */
	return TRUE;		
}

/*! \brief parse uri in a way that allows semicolon stripping if legacy mode is enabled
 *
 * \note This calls parse_uri which has the unexpected property that passing more
 *       arguments results in more splitting. Most common is to leave out the pass
 *       argument, causing user to contain user:pass if available.
 */
static int parse_uri_legacy_check(char *uri, const char *scheme, char **user, char **pass, char **hostport, char **transport)
{
	int ret = parse_uri(uri, scheme, user, pass, hostport, transport);
	if (sip_cfg.legacy_useroption_parsing) { /* if legacy mode is active, strip semis from the user field */
		char *p;
		if ((p = strchr(uri, (int)';'))) {
			*p = '\0';
		}
	}
	return ret;
}

static int __set_address_from_contact(const char *fullcontact, struct ast_sockaddr *addr, int tcp)
{
	char *hostport, *transport;
	char contact_buf[256];
	char *contact;

	/* Work on a copy */
	ast_copy_string(contact_buf, fullcontact, sizeof(contact_buf));
	contact = contact_buf;

	/* 
	 * We have only the part in <brackets> here so we just need to parse a SIP URI.
	 *
	 * Note: The outbound proxy could be using UDP between the proxy and Asterisk.
	 * We still need to be able to send to the remote agent through the proxy.
	 */

	if (parse_uri_legacy_check(contact, "sip:,sips:", &contact, NULL, &hostport,
		      &transport)) {
		ast_log(LOG_WARNING, "Invalid contact uri %s (missing sip: or sips:), attempting to use anyway\n", fullcontact);
	}

	/* XXX This could block for a long time XXX */
	/* We should only do this if it's a name, not an IP */
	/* \todo - if there's no PORT number in contact - we are required to check NAPTR/SRV records
		to find transport, port address and hostname. If there's a port number, we have to
		assume that the hostport part is a host name and only look for an A/AAAA record in DNS.
	*/

	/* If we took in an invalid URI, hostport may not have been initialized */
	/* ast_sockaddr_resolve requires an initialized hostport string. */
	if (ast_strlen_zero(hostport)) {
		ast_log(LOG_WARNING, "Invalid URI: parse_uri failed to acquire hostport\n");
		return -1;
	}

	if (ast_sockaddr_resolve_first(addr, hostport, 0)) {
		ast_log(LOG_WARNING, "Invalid host name in Contact: (can't "
			"resolve in DNS) : '%s'\n", hostport);
		return -1;
	}

	/* set port */
	if (!ast_sockaddr_port(addr)) {
		ast_sockaddr_set_port(addr,
				      (get_transport_str2enum(transport) ==
				       SIP_TRANSPORT_TLS ||
				       !strncasecmp(fullcontact, "sips", 4)) ?
				      STANDARD_TLS_PORT : STANDARD_SIP_PORT);
	}

	return 0;
}

/*! \brief Change the other partys IP address based on given contact */
static int set_address_from_contact(struct sip_pvt *pvt)
{
	if (ast_test_flag(&pvt->flags[0], SIP_NAT_FORCE_RPORT)) {
		/* NAT: Don't trust the contact field.  Just use what they came to us
		   with. */
		/*! \todo We need to save the TRANSPORT here too */
		pvt->sa = pvt->recv;
		return 0;
	}

	return __set_address_from_contact(pvt->fullcontact, &pvt->sa, pvt->socket.type == SIP_TRANSPORT_TLS ? 1 : 0);
}

/*! \brief Parse contact header and save registration (peer registration) */
static enum parse_register_result parse_register_contact(struct sip_pvt *pvt, struct sip_peer *peer, struct sip_request *req)
{
	char contact[SIPBUFSIZE];
	char data[SIPBUFSIZE];
	const char *expires = get_header(req, "Expires");
	int expire = atoi(expires);
	char *curi = NULL, *hostport = NULL, *transport = NULL;
	int transport_type;
	const char *useragent;
	struct ast_sockaddr oldsin, testsa;
	char *firstcuri = NULL;
	int start = 0;
	int wildcard_found = 0;
	int single_binding_found = 0;

	ast_copy_string(contact, __get_header(req, "Contact", &start), sizeof(contact));

	if (ast_strlen_zero(expires)) {	/* No expires header, try look in Contact: */
		char *s = strcasestr(contact, ";expires=");
		if (s) {
			expires = strsep(&s, ";"); /* trim ; and beyond */
			if (sscanf(expires + 9, "%30d", &expire) != 1) {
				expire = default_expiry;
			}
		} else {
			/* Nothing has been specified */
			expire = default_expiry;
		}
	}

	copy_socket_data(&pvt->socket, &req->socket);

	do {
		/* Look for brackets */
		curi = contact;
		if (strchr(contact, '<') == NULL)	/* No <, check for ; and strip it */
			strsep(&curi, ";");	/* This is Header options, not URI options */
		curi = get_in_brackets(contact);
		if (!firstcuri) {
			firstcuri = ast_strdupa(curi);
		}

		if (!strcasecmp(curi, "*")) {
			wildcard_found = 1;
		} else {
			single_binding_found = 1;
		}

		if (wildcard_found && (ast_strlen_zero(expires) || expire != 0 || single_binding_found)) {
			/* Contact header parameter "*" detected, so punt if: Expires header is missing,
			 * Expires value is not zero, or another Contact header is present. */
			return PARSE_REGISTER_FAILED;
		}

		ast_copy_string(contact, __get_header(req, "Contact", &start), sizeof(contact));
	} while (!ast_strlen_zero(contact));
	curi = firstcuri;

	/* if they did not specify Contact: or Expires:, they are querying
	   what we currently have stored as their contact address, so return
	   it
	*/
	if (ast_strlen_zero(curi) && ast_strlen_zero(expires)) {
		/* If we have an active registration, tell them when the registration is going to expire */
		if (peer->expire > -1 && !ast_strlen_zero(peer->fullcontact)) {
			pvt->expiry = ast_sched_when(sched, peer->expire);
		}
		return PARSE_REGISTER_QUERY;
	} else if (!strcasecmp(curi, "*") || !expire) {	/* Unregister this peer */
		/* This means remove all registrations and return OK */
		AST_SCHED_DEL_UNREF(sched, peer->expire,
				unref_peer(peer, "remove register expire ref"));
		ast_verb(3, "Unregistered SIP '%s'\n", peer->name);
		expire_register(ref_peer(peer,"add ref for explicit expire_register"));
		return PARSE_REGISTER_UPDATE;
	}

	/* Store whatever we got as a contact from the client */
	ast_string_field_set(peer, fullcontact, curi);

	/* For the 200 OK, we should use the received contact */
	ast_string_field_build(pvt, our_contact, "<%s>", curi);

	/* Make sure it's a SIP URL */
	if (ast_strlen_zero(curi) || parse_uri_legacy_check(curi, "sip:,sips:", &curi, NULL, &hostport, &transport)) {
		ast_log(LOG_NOTICE, "Not a valid SIP contact (missing sip:/sips:) trying to use anyway\n");
	}

	/* handle the transport type specified in Contact header. */
	if (!(transport_type = get_transport_str2enum(transport))) {
		transport_type = pvt->socket.type;
	}

	/* if the peer's socket type is different than the Registration
	 * transport type, change it.  If it got this far, it is a
	 * supported type, but check just in case */
	if ((peer->socket.type != transport_type) && (peer->transports & transport_type)) {
		set_socket_transport(&peer->socket, transport_type);
	}

	oldsin = peer->addr;

	/* If we were already linked into the peers_by_ip container unlink ourselves so nobody can find us */
	if (!ast_sockaddr_isnull(&peer->addr) && (!peer->is_realtime || ast_test_flag(&global_flags[1], SIP_PAGE2_RTCACHEFRIENDS))) {
		ao2_t_unlink(peers_by_ip, peer, "ao2_unlink of peer from peers_by_ip table");
	}

	if (!ast_test_flag(&peer->flags[0], SIP_NAT_FORCE_RPORT) && !ast_test_flag(&peer->flags[0], SIP_NAT_RPORT_PRESENT)) {
		 /* use the data provided in the Contact header for call routing */
		ast_debug(1, "Store REGISTER's Contact header for call routing.\n");
		/* XXX This could block for a long time XXX */
		/*! \todo Check NAPTR/SRV if we have not got a port in the URI */
		if (ast_sockaddr_resolve_first(&testsa, hostport, 0)) {
			ast_log(LOG_WARNING, "Invalid hostport '%s'\n", hostport);
			ast_string_field_set(peer, fullcontact, "");
			ast_string_field_set(pvt, our_contact, "");
			return PARSE_REGISTER_FAILED;
		}

		/* If we have a port number in the given URI, make sure we do remember to not check for NAPTR/SRV records.
		   The hostport part is actually a host. */
		peer->portinuri = ast_sockaddr_port(&testsa) ? TRUE : FALSE;

		if (!ast_sockaddr_port(&testsa)) {
			ast_sockaddr_set_port(&testsa, default_sip_port(transport_type));
		}

		ast_sockaddr_copy(&peer->addr, &testsa);
	} else {
		/* Don't trust the contact field.  Just use what they came to us
		   with */
		ast_debug(1, "Store REGISTER's src-IP:port for call routing.\n");
		peer->addr = pvt->recv;
	}

	/* Check that they're allowed to register at this IP */
	if (ast_apply_ha(sip_cfg.contact_ha, &peer->addr) != AST_SENSE_ALLOW ||
			ast_apply_ha(peer->contactha, &peer->addr) != AST_SENSE_ALLOW) {
		ast_log(LOG_WARNING, "Domain '%s' disallowed by contact ACL (violating IP %s)\n", hostport,
			ast_sockaddr_stringify_addr(&testsa));
		ast_string_field_set(peer, fullcontact, "");
		ast_string_field_set(pvt, our_contact, "");
		return PARSE_REGISTER_DENIED;
	}

	/* if the Contact header information copied into peer->addr matches the
	 * received address, and the transport types are the same, then copy socket
	 * data into the peer struct */
	if ((peer->socket.type == pvt->socket.type) &&
		!ast_sockaddr_cmp(&peer->addr, &pvt->recv)) {
		copy_socket_data(&peer->socket, &pvt->socket);
	}

	/* Now that our address has been updated put ourselves back into the container for lookups */
	if (!peer->is_realtime || ast_test_flag(&peer->flags[1], SIP_PAGE2_RTCACHEFRIENDS)) {
		ao2_t_link(peers_by_ip, peer, "ao2_link into peers_by_ip table");
	}

	/* Save SIP options profile */
	peer->sipoptions = pvt->sipoptions;

	if (!ast_strlen_zero(curi) && ast_strlen_zero(peer->username)) {
		ast_string_field_set(peer, username, curi);
	}

	AST_SCHED_DEL_UNREF(sched, peer->expire,
			unref_peer(peer, "remove register expire ref"));

	if (expire > max_expiry) {
		expire = max_expiry;
	}
	if (expire < min_expiry) {
		expire = min_expiry;
	}
	if (peer->is_realtime && !ast_test_flag(&peer->flags[1], SIP_PAGE2_RTCACHEFRIENDS)) {
		peer->expire = -1;
	} else {
		peer->expire = ast_sched_add(sched, (expire + 10) * 1000, expire_register,
				ref_peer(peer, "add registration ref"));
		if (peer->expire == -1) {
			unref_peer(peer, "remote registration ref");
		}
	}
	pvt->expiry = expire;
	snprintf(data, sizeof(data), "%s:%d:%s:%s", ast_sockaddr_stringify(&peer->addr),
		 expire, peer->username, peer->fullcontact);
	/* We might not immediately be able to reconnect via TCP, but try caching it anyhow */
	if (!peer->rt_fromcontact || !sip_cfg.peer_rtupdate)
		ast_db_put("SIP/Registry", peer->name, data);
	manager_event(EVENT_FLAG_SYSTEM, "PeerStatus", "ChannelType: SIP\r\nPeer: SIP/%s\r\nPeerStatus: Registered\r\nAddress: %s\r\n", peer->name,  ast_sockaddr_stringify(&peer->addr));

	/* Is this a new IP address for us? */
	if (VERBOSITY_ATLEAST(2) && ast_sockaddr_cmp(&peer->addr, &oldsin)) {
		ast_verbose(VERBOSE_PREFIX_3 "Registered SIP '%s' at %s\n", peer->name,
				ast_sockaddr_stringify(&peer->addr));
	}
	sip_poke_peer(peer, 0);
	register_peer_exten(peer, 1);
	
	/* Save User agent */
	useragent = get_header(req, "User-Agent");
	if (strcasecmp(useragent, peer->useragent)) {
		ast_string_field_set(peer, useragent, useragent);
		ast_verb(4, "Saved useragent \"%s\" for peer %s\n", peer->useragent, peer->name);
	}
	return PARSE_REGISTER_UPDATE;
}

/*! \brief Remove route from route list */
static void free_old_route(struct sip_route *route)
{
	struct sip_route *next;

	while (route) {
		next = route->next;
		ast_free(route);
		route = next;
	}
}

/*! \brief List all routes - mostly for debugging */
static void list_route(struct sip_route *route)
{
	if (!route) {
		ast_verbose("list_route: no route\n");
	} else {
		for (;route; route = route->next)
			ast_verbose("list_route: hop: <%s>\n", route->hop);
	}
}

/*! \brief Build route list from Record-Route header 
    \param resp the SIP response code or 0 for a request */
static void build_route(struct sip_pvt *p, struct sip_request *req, int backwards, int resp)
{
	struct sip_route *thishop, *head, *tail;
	int start = 0;
	int len;
	const char *rr, *c;

	/* Once a persistent route is set, don't fool with it */
	if (p->route && p->route_persistent) {
		ast_debug(1, "build_route: Retaining previous route: <%s>\n", p->route->hop);
		return;
	}

	if (p->route) {
		free_old_route(p->route);
		p->route = NULL;
	}

	/* We only want to create the route set the first time this is called except
	   it is called from a provisional response.*/
	if ((resp < 100) || (resp > 199)) {
		p->route_persistent = 1;
	}

	/* Build a tailq, then assign it to p->route when done.
	 * If backwards, we add entries from the head so they end up
	 * in reverse order. However, we do need to maintain a correct
	 * tail pointer because the contact is always at the end.
	 */
	head = NULL;
	tail = head;
	/* 1st we pass through all the hops in any Record-Route headers */
	for (;;) {
		/* Each Record-Route header */
		char rr_copy[256];
		char *rr_copy_ptr;
		char *rr_iter;
		rr = __get_header(req, "Record-Route", &start);
		if (*rr == '\0') {
			break;
		}
		ast_copy_string(rr_copy, rr, sizeof(rr_copy));
		rr_copy_ptr = rr_copy;
		while ((rr_iter = strsep(&rr_copy_ptr, ","))) { /* Each route entry */
			char *uri = get_in_brackets(rr_iter);
			len = strlen(uri) + 1;
			/* Make a struct route */
			if ((thishop = ast_malloc(sizeof(*thishop) + len))) {
				/* ast_calloc is not needed because all fields are initialized in this block */
				ast_copy_string(thishop->hop, uri, len);
				ast_debug(2, "build_route: Record-Route hop: <%s>\n", thishop->hop);
				/* Link in */
				if (backwards) {
					/* Link in at head so they end up in reverse order */
					thishop->next = head;
					head = thishop;
					/* If this was the first then it'll be the tail */
					if (!tail) {
						tail = thishop;
					}
				} else {
					thishop->next = NULL;
					/* Link in at the end */
					if (tail) {
						tail->next = thishop;
					} else {
						head = thishop;
					}
					tail = thishop;
				}
			}
		}
	}

	/* Only append the contact if we are dealing with a strict router */
	if (!head || (!ast_strlen_zero(head->hop) && strstr(head->hop, ";lr") == NULL) ) {
		/* 2nd append the Contact: if there is one */
		/* Can be multiple Contact headers, comma separated values - we just take the first */
		char *contact = ast_strdupa(get_header(req, "Contact"));
		if (!ast_strlen_zero(contact)) {
			ast_debug(2, "build_route: Contact hop: %s\n", contact);
			/* Look for <: delimited address */
			c = get_in_brackets(contact);
			len = strlen(c) + 1;
			if ((thishop = ast_malloc(sizeof(*thishop) + len))) {
				/* ast_calloc is not needed because all fields are initialized in this block */
				ast_copy_string(thishop->hop, c, len);
				thishop->next = NULL;
				/* Goes at the end */
				if (tail) {
					tail->next = thishop;
				} else {
					head = thishop;
				}
			}
		}
	}

	/* Store as new route */
	p->route = head;

	/* For debugging dump what we ended up with */
	if (sip_debug_test_pvt(p)) {
		list_route(p->route);
	}
}

/*! \brief builds the sip_pvt's randdata field which is used for the nonce
 *  challenge.  When forceupdate is not set, the nonce is only updated if
 *  the current one is stale.  In this case, a stalenonce is one which
 *  has already received a response, if a nonce has not received a response
 *  it is not always necessary or beneficial to create a new one. */

static void set_nonce_randdata(struct sip_pvt *p, int forceupdate)
{
	if (p->stalenonce || forceupdate || ast_strlen_zero(p->randdata)) {
		ast_string_field_build(p, randdata, "%08lx", ast_random());	/* Create nonce for challenge */
		p->stalenonce = 0;
	}
}

AST_THREADSTORAGE(check_auth_buf);
#define CHECK_AUTH_BUF_INITLEN   256

/*! \brief  Check user authorization from peer definition
	Some actions, like REGISTER and INVITEs from peers require
	authentication (if peer have secret set)
    \return 0 on success, non-zero on error
*/
static enum check_auth_result check_auth(struct sip_pvt *p, struct sip_request *req, const char *username,
					 const char *secret, const char *md5secret, int sipmethod,
					 const char *uri, enum xmittype reliable, int ignore)
{
	const char *response;
	char *reqheader, *respheader;
	const char *authtoken;
	char a1_hash[256];
	char resp_hash[256]="";
	char *c;
	int  wrongnonce = FALSE;
	int  good_response;
	const char *usednonce = p->randdata;
	struct ast_str *buf;
	int res;

	/* table of recognised keywords, and their value in the digest */
	enum keys { K_RESP, K_URI, K_USER, K_NONCE, K_LAST };
	struct x {
		const char *key;
		const char *s;
	} *i, keys[] = {
		[K_RESP] = { "response=", "" },
		[K_URI] = { "uri=", "" },
		[K_USER] = { "username=", "" },
		[K_NONCE] = { "nonce=", "" },
		[K_LAST] = { NULL, NULL}
	};

	/* Always OK if no secret */
	if (ast_strlen_zero(secret) && ast_strlen_zero(md5secret))
		return AUTH_SUCCESSFUL;

	/* Always auth with WWW-auth since we're NOT a proxy */
	/* Using proxy-auth in a B2BUA may block proxy authorization in the same transaction */
	response = "401 Unauthorized";

	/*
	 * Note the apparent swap of arguments below, compared to other
	 * usages of auth_headers().
	 */
	auth_headers(WWW_AUTH, &respheader, &reqheader);

	authtoken =  get_header(req, reqheader);	
	if (ignore && !ast_strlen_zero(p->randdata) && ast_strlen_zero(authtoken)) {
		/* This is a retransmitted invite/register/etc, don't reconstruct authentication
		   information */
		if (!reliable) {
			/* Resend message if this was NOT a reliable delivery.   Otherwise the
			   retransmission should get it */
			transmit_response_with_auth(p, response, req, p->randdata, reliable, respheader, 0);
			/* Schedule auto destroy in 32 seconds (according to RFC 3261) */
			sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
		}
		return AUTH_CHALLENGE_SENT;
	} else if (ast_strlen_zero(p->randdata) || ast_strlen_zero(authtoken)) {
		/* We have no auth, so issue challenge and request authentication */
		set_nonce_randdata(p, 1); /* Create nonce for challenge */
		transmit_response_with_auth(p, response, req, p->randdata, reliable, respheader, 0);
		/* Schedule auto destroy in 32 seconds */
		sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
		return AUTH_CHALLENGE_SENT;
	}

	/* --- We have auth, so check it */

	/* Whoever came up with the authentication section of SIP can suck my %&#$&* for not putting
	   an example in the spec of just what it is you're doing a hash on. */

	if (!(buf = ast_str_thread_get(&check_auth_buf, CHECK_AUTH_BUF_INITLEN))) {
		return AUTH_SECRET_FAILED; /*! XXX \todo need a better return code here */
	}

	/* Make a copy of the response and parse it */
	res = ast_str_set(&buf, 0, "%s", authtoken);

	if (res == AST_DYNSTR_BUILD_FAILED) {
		return AUTH_SECRET_FAILED; /*! XXX \todo need a better return code here */
	}

	c = buf->str;

	while(c && *(c = ast_skip_blanks(c)) ) { /* lookup for keys */
		for (i = keys; i->key != NULL; i++) {
			const char *separator = ",";	/* default */

			if (strncasecmp(c, i->key, strlen(i->key)) != 0) {
				continue;
			}
			/* Found. Skip keyword, take text in quotes or up to the separator. */
			c += strlen(i->key);
			if (*c == '"') { /* in quotes. Skip first and look for last */
				c++;
				separator = "\"";
			}
			i->s = c;
			strsep(&c, separator);
			break;
		}
		if (i->key == NULL) { /* not found, jump after space or comma */
			strsep(&c, " ,");
		}
	}

	/* Verify that digest username matches  the username we auth as */
	if (strcmp(username, keys[K_USER].s)) {
		ast_log(LOG_WARNING, "username mismatch, have <%s>, digest has <%s>\n",
			username, keys[K_USER].s);
		/* Oops, we're trying something here */
		return AUTH_USERNAME_MISMATCH;
	}

	/* Verify nonce from request matches our nonce, and the nonce has not already been responded to.
	 * If this check fails, send 401 with new nonce */
	if (strcasecmp(p->randdata, keys[K_NONCE].s) || p->stalenonce) { /* XXX it was 'n'casecmp ? */
		wrongnonce = TRUE;
		usednonce = keys[K_NONCE].s;
	} else {
		p->stalenonce = 1; /* now, since the nonce has a response, mark it as stale so it can't be sent or responded to again */
	}

	if (!ast_strlen_zero(md5secret)) {
		ast_copy_string(a1_hash, md5secret, sizeof(a1_hash));
	} else {
		char a1[256];

		snprintf(a1, sizeof(a1), "%s:%s:%s", username, p->realm, secret);
		ast_md5_hash(a1_hash, a1);
	}

	/* compute the expected response to compare with what we received */
	{
		char a2[256];
		char a2_hash[256];
		char resp[256];

		snprintf(a2, sizeof(a2), "%s:%s", sip_methods[sipmethod].text,
				S_OR(keys[K_URI].s, uri));
		ast_md5_hash(a2_hash, a2);
		snprintf(resp, sizeof(resp), "%s:%s:%s", a1_hash, usednonce, a2_hash);
		ast_md5_hash(resp_hash, resp);
	}

	good_response = keys[K_RESP].s &&
			!strncasecmp(keys[K_RESP].s, resp_hash, strlen(resp_hash));
	if (wrongnonce) {
		if (good_response) {
			if (sipdebug)
				ast_log(LOG_NOTICE, "Correct auth, but based on stale nonce received from '%s'\n", get_header(req, "From"));
			/* We got working auth token, based on stale nonce . */
			set_nonce_randdata(p, 0);
			transmit_response_with_auth(p, response, req, p->randdata, reliable, respheader, TRUE);
		} else {
			/* Everything was wrong, so give the device one more try with a new challenge */
			if (!req->ignore) {
				if (sipdebug) {
					ast_log(LOG_NOTICE, "Bad authentication received from '%s'\n", get_header(req, "To"));
				}
				set_nonce_randdata(p, 1);
			} else {
				if (sipdebug) {
					ast_log(LOG_NOTICE, "Duplicate authentication received from '%s'\n", get_header(req, "To"));
				}
			}
			transmit_response_with_auth(p, response, req, p->randdata, reliable, respheader, FALSE);
		}

		/* Schedule auto destroy in 32 seconds */
		sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
		return AUTH_CHALLENGE_SENT;
	}
	if (good_response) {
		append_history(p, "AuthOK", "Auth challenge successful for %s", username);
		return AUTH_SUCCESSFUL;
	}

	/* Ok, we have a bad username/secret pair */
	/* Tell the UAS not to re-send this authentication data, because
	   it will continue to fail
	*/

	return AUTH_SECRET_FAILED;
}

/*! \brief Change onhold state of a peer using a pvt structure */
static void sip_peer_hold(struct sip_pvt *p, int hold)
{
	if (!p->relatedpeer) {
		return;
	}

	/* If they put someone on hold, increment the value... otherwise decrement it */
	ast_atomic_fetchadd_int(&p->relatedpeer->onHold, (hold ? +1 : -1));

	/* Request device state update */
	ast_devstate_changed(AST_DEVICE_UNKNOWN, "SIP/%s", p->relatedpeer->name);
	
	return;
}

/*! \brief Receive MWI events that we have subscribed to */
static void mwi_event_cb(const struct ast_event *event, void *userdata)
{
	struct sip_peer *peer = userdata;

	sip_send_mwi_to_peer(peer, 0);
}

static void network_change_event_subscribe(void)
{
	if (!network_change_event_subscription) {
		network_change_event_subscription = ast_event_subscribe(AST_EVENT_NETWORK_CHANGE,
			network_change_event_cb, "SIP Network Change", NULL, AST_EVENT_IE_END);
	}
}

static void network_change_event_unsubscribe(void)
{
	if (network_change_event_subscription) {
		network_change_event_subscription = ast_event_unsubscribe(network_change_event_subscription);
	}
}

static int network_change_event_sched_cb(const void *data)
{
	network_change_event_sched_id = -1;
	sip_send_all_registers();
	sip_send_all_mwi_subscriptions();
	return 0;
}

static void network_change_event_cb(const struct ast_event *event, void *userdata)
{
	ast_debug(1, "SIP, got a network change event, renewing all SIP registrations.\n");
	if (network_change_event_sched_id == -1) {
		network_change_event_sched_id = ast_sched_add(sched, 1000, network_change_event_sched_cb, NULL);
	}
}

static void cb_extensionstate_destroy(int id, void *data)
{
	struct sip_pvt *p = data;

	dialog_unref(p, "the extensionstate containing this dialog ptr was destroyed");
}

/*! \brief Callback for the devicestate notification (SUBSCRIBE) support subsystem
\note	If you add an "hint" priority to the extension in the dial plan,
	you will get notifications on device state changes */
static int cb_extensionstate(char *context, char* exten, int state, void *data)
{
	struct sip_pvt *p = data;

	sip_pvt_lock(p);

	switch(state) {
	case AST_EXTENSION_DEACTIVATED:	/* Retry after a while */
	case AST_EXTENSION_REMOVED:	/* Extension is gone */
		sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);	/* Delete subscription in 32 secs */
		ast_verb(2, "Extension state: Watcher for hint %s %s. Notify User %s\n", exten, state == AST_EXTENSION_DEACTIVATED ? "deactivated" : "removed", p->username);
		p->subscribed = NONE;
		append_history(p, "Subscribestatus", "%s", state == AST_EXTENSION_REMOVED ? "HintRemoved" : "Deactivated");
		break;
	default:	/* Tell user */
		p->laststate = state;
		break;
	}
	if (p->subscribed != NONE) {	/* Only send state NOTIFY if we know the format */
		if (!p->pendinginvite) {
			transmit_state_notify(p, state, 1, FALSE);
		} else {
			/* We already have a NOTIFY sent that is not answered. Queue the state up.
			   if many state changes happen meanwhile, we will only send a notification of the last one */
			ast_set_flag(&p->flags[1], SIP_PAGE2_STATECHANGEQUEUE);
		}
	}
	ast_verb(2, "Extension Changed %s[%s] new state %s for Notify User %s %s\n", exten, context, ast_extension_state2str(state), p->username,
			ast_test_flag(&p->flags[1], SIP_PAGE2_STATECHANGEQUEUE) ? "(queued)" : "");

	sip_pvt_unlock(p);

	return 0;
}

/*! \brief Send a fake 401 Unauthorized response when the administrator
  wants to hide the names of local devices  from fishers
 */
static void transmit_fake_auth_response(struct sip_pvt *p, int sipmethod, struct sip_request *req, enum xmittype reliable)
{
	/* We have to emulate EXACTLY what we'd get with a good peer
	 * and a bad password, or else we leak information. */
	const char *response = "401 Unauthorized";
	const char *reqheader = "Authorization";
	const char *respheader = "WWW-Authenticate";
	const char *authtoken;
	struct ast_str *buf;
	char *c;

	/* table of recognised keywords, and their value in the digest */
	enum keys { K_NONCE, K_LAST };
	struct x {
		const char *key;
		const char *s;
	} *i, keys[] = {
		[K_NONCE] = { "nonce=", "" },
		[K_LAST] = { NULL, NULL}
	};

	authtoken = get_header(req, reqheader);
	if (req->ignore && !ast_strlen_zero(p->randdata) && ast_strlen_zero(authtoken)) {
		/* This is a retransmitted invite/register/etc, don't reconstruct authentication
		 * information */
		transmit_response_with_auth(p, response, req, p->randdata, reliable, respheader, 0);
		/* Schedule auto destroy in 32 seconds (according to RFC 3261) */
		sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
		return;
	} else if (ast_strlen_zero(p->randdata) || ast_strlen_zero(authtoken)) {
		/* We have no auth, so issue challenge and request authentication */
		set_nonce_randdata(p, 1);
		transmit_response_with_auth(p, response, req, p->randdata, reliable, respheader, 0);
		/* Schedule auto destroy in 32 seconds */
		sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
		return;
	}

	if (!(buf = ast_str_thread_get(&check_auth_buf, CHECK_AUTH_BUF_INITLEN))) {
		transmit_response(p, "403 Forbidden (Bad auth)", &p->initreq);
		return;
	}

	/* Make a copy of the response and parse it */
	if (ast_str_set(&buf, 0, "%s", authtoken) == AST_DYNSTR_BUILD_FAILED) {
		transmit_response(p, "403 Forbidden (Bad auth)", &p->initreq);
		return;
	}

	c = buf->str;

	while (c && *(c = ast_skip_blanks(c))) { /* lookup for keys */
		for (i = keys; i->key != NULL; i++) {
			const char *separator = ",";	/* default */

			if (strncasecmp(c, i->key, strlen(i->key)) != 0) {
				continue;
			}
			/* Found. Skip keyword, take text in quotes or up to the separator. */
			c += strlen(i->key);
			if (*c == '"') { /* in quotes. Skip first and look for last */
				c++;
				separator = "\"";
			}
			i->s = c;
			strsep(&c, separator);
			break;
		}
		if (i->key == NULL) { /* not found, jump after space or comma */
			strsep(&c, " ,");
		}
	}

	/* Verify nonce from request matches our nonce.  If not, send 401 with new nonce */
	if (strcasecmp(p->randdata, keys[K_NONCE].s)) {
		if (!req->ignore) {
			set_nonce_randdata(p, 1);
		}
		transmit_response_with_auth(p, response, req, p->randdata, reliable, respheader, FALSE);

		/* Schedule auto destroy in 32 seconds */
		sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
	} else {
		transmit_response(p, "403 Forbidden (Bad auth)", &p->initreq);
	}
}

/*!
 * Terminate the uri at the first ';' or space.
 * Technically we should ignore escaped space per RFC3261 (19.1.1 etc)
 * but don't do it for the time being. Remember the uri format is:
 * (User-parameters was added after RFC 3261)
 *\verbatim
 *
 *	sip:user:password;user-parameters@host:port;uri-parameters?headers
 *	sips:user:password;user-parameters@host:port;uri-parameters?headers
 *
 *\endverbatim
 * \todo As this function does not support user-parameters, it's considered broken
 *	and needs fixing.
 */
static char *terminate_uri(char *uri)
{
	char *t = uri;
	while (*t && *t > ' ' && *t != ';') {
		t++;
	}
	*t = '\0';
	return uri;
}

/*! \brief Terminate a host:port at the ':'
 * \param hostport The address of the hostport string
 *
 * \note In the case of a bracket-enclosed IPv6 address, the hostport variable
 * will contain the non-bracketed host as a result of calling this function.
 */
static void extract_host_from_hostport(char **hostport)
{
	char *dont_care;
	ast_sockaddr_split_hostport(*hostport, hostport, &dont_care, PARSE_PORT_IGNORE);
}

/*! \brief Verify registration of user
	- Registration is done in several steps, first a REGISTER without auth
	  to get a challenge (nonce) then a second one with auth
	- Registration requests are only matched with peers that are marked as "dynamic"
 */
static enum check_auth_result register_verify(struct sip_pvt *p, struct ast_sockaddr *addr,
					      struct sip_request *req, const char *uri)
{
	enum check_auth_result res = AUTH_NOT_FOUND;
	struct sip_peer *peer;
	char tmp[256];
	char *c, *name, *unused_password, *domain;
	char *uri2 = ast_strdupa(uri);

	terminate_uri(uri2);

	ast_copy_string(tmp, get_header(req, "To"), sizeof(tmp));

	c = get_in_brackets(tmp);
	c = remove_uri_parameters(c);

	if (parse_uri_legacy_check(c, "sip:,sips:", &name, &unused_password, &domain, NULL)) {
		ast_log(LOG_NOTICE, "Invalid to address: '%s' from %s (missing sip:) trying to use anyway...\n", c, ast_sockaddr_stringify_addr(addr));
		return -1;
	}

	SIP_PEDANTIC_DECODE(name);
	SIP_PEDANTIC_DECODE(domain);

	extract_host_from_hostport(&domain);

	if (ast_strlen_zero(domain)) {
		/* <sip:name@[EMPTY]>, never good */
		transmit_response(p, "404 Not found", &p->initreq);
		return AUTH_UNKNOWN_DOMAIN;
	}

	if (ast_strlen_zero(name)) {
		/* <sip:[EMPTY][@]hostport>, unsure whether valid for
		 * registration. RFC 3261, 10.2 states:
		 * "The To header field and the Request-URI field typically
		 * differ, as the former contains a user name."
		 * But, Asterisk has always treated the domain-only uri as a
		 * username: we allow admins to create accounts described by
		 * domain name. */
		name = domain;
	}

	/* This here differs from 1.4 and 1.6: the domain matching ACLs were
	 * skipped if it was a domain-only URI (used as username). Here we treat
	 * <sip:hostport> as <sip:host@hostport> and won't forget to test the
	 * domain ACLs against host. */
	if (!AST_LIST_EMPTY(&domain_list)) {
		if (!check_sip_domain(domain, NULL, 0)) {
			if (sip_cfg.alwaysauthreject) {
				transmit_fake_auth_response(p, SIP_REGISTER, &p->initreq, XMIT_UNRELIABLE);
			} else {
				transmit_response(p, "404 Not found (unknown domain)", &p->initreq);
			}
			return AUTH_UNKNOWN_DOMAIN;
		}
	}

	ast_string_field_set(p, exten, name);
	build_contact(p);
	if (req->ignore) {
		/* Expires is a special case, where we only want to load the peer if this isn't a deregistration attempt */
		const char *expires = get_header(req, "Expires");
		int expire = atoi(expires);

		if (ast_strlen_zero(expires)) { /* No expires header; look in Contact */
			if ((expires = strcasestr(get_header(req, "Contact"), ";expires="))) {
				expire = atoi(expires + 9);
			}
		}
		if (!ast_strlen_zero(expires) && expire == 0) {
			transmit_response_with_date(p, "200 OK", req);
			return 0;
		}
	}
	peer = find_peer(name, NULL, TRUE, FINDPEERS, FALSE, 0);

	if (!(peer && ast_apply_ha(peer->ha, addr))) {
		/* Peer fails ACL check */
		if (peer) {
			unref_peer(peer, "register_verify: unref_peer: from find_peer operation");
			peer = NULL;
			res = AUTH_ACL_FAILED;
		} else {
			res = AUTH_NOT_FOUND;
		}
	}

	if (peer) {
		ao2_lock(peer);
		if (!peer->host_dynamic) {
			ast_log(LOG_ERROR, "Peer '%s' is trying to register, but not configured as host=dynamic\n", peer->name);
			res = AUTH_PEER_NOT_DYNAMIC;
		} else {
			ast_copy_flags(&p->flags[0], &peer->flags[0], SIP_NAT_FORCE_RPORT);
			if (!(res = check_auth(p, req, peer->name, peer->secret, peer->md5secret, SIP_REGISTER, uri2, XMIT_UNRELIABLE, req->ignore))) {
				if (sip_cancel_destroy(p))
					ast_log(LOG_WARNING, "Unable to cancel SIP destruction.  Expect bad things.\n");

				if (check_request_transport(peer, req)) {
					ast_set_flag(&p->flags[0], SIP_PENDINGBYE);
					transmit_response_with_date(p, "403 Forbidden", req);
					res = AUTH_BAD_TRANSPORT;
				} else {

					/* We have a successful registration attempt with proper authentication,
				   	now, update the peer */
					switch (parse_register_contact(p, peer, req)) {
					case PARSE_REGISTER_DENIED:
						ast_log(LOG_WARNING, "Registration denied because of contact ACL\n");
						transmit_response_with_date(p, "603 Denied", req);
						peer->lastmsgssent = -1;
						res = 0;
						break;
					case PARSE_REGISTER_FAILED:
						ast_log(LOG_WARNING, "Failed to parse contact info\n");
						transmit_response_with_date(p, "400 Bad Request", req);
						peer->lastmsgssent = -1;
						res = 0;
						break;
					case PARSE_REGISTER_QUERY:
						ast_string_field_set(p, fullcontact, peer->fullcontact);
						transmit_response_with_date(p, "200 OK", req);
						peer->lastmsgssent = -1;
						res = 0;
						break;
					case PARSE_REGISTER_UPDATE:
						ast_string_field_set(p, fullcontact, peer->fullcontact);
						update_peer(peer, p->expiry);
						/* Say OK and ask subsystem to retransmit msg counter */
						transmit_response_with_date(p, "200 OK", req);
						if (!ast_test_flag((&peer->flags[1]), SIP_PAGE2_SUBSCRIBEMWIONLY))
							peer->lastmsgssent = -1;
						res = 0;
						break;
					}
				}

			}
		}
		ao2_unlock(peer);
	}
	if (!peer && sip_cfg.autocreatepeer) {
		/* Create peer if we have autocreate mode enabled */
		peer = temp_peer(name);
		if (peer) {
			ao2_t_link(peers, peer, "link peer into peer table");
			if (!ast_sockaddr_isnull(&peer->addr)) {
				ao2_t_link(peers_by_ip, peer, "link peer into peers-by-ip table");
			}
			ao2_lock(peer);
			if (sip_cancel_destroy(p))
				ast_log(LOG_WARNING, "Unable to cancel SIP destruction.  Expect bad things.\n");
			switch (parse_register_contact(p, peer, req)) {
			case PARSE_REGISTER_DENIED:
				ast_log(LOG_WARNING, "Registration denied because of contact ACL\n");
				transmit_response_with_date(p, "403 Forbidden (ACL)", req);
				peer->lastmsgssent = -1;
				res = 0;
				break;
			case PARSE_REGISTER_FAILED:
				ast_log(LOG_WARNING, "Failed to parse contact info\n");
				transmit_response_with_date(p, "400 Bad Request", req);
				peer->lastmsgssent = -1;
				res = 0;
				break;
			case PARSE_REGISTER_QUERY:
				ast_string_field_set(p, fullcontact, peer->fullcontact);
				transmit_response_with_date(p, "200 OK", req);
				peer->lastmsgssent = -1;
				res = 0;
				break;
			case PARSE_REGISTER_UPDATE:
				ast_string_field_set(p, fullcontact, peer->fullcontact);
				/* Say OK and ask subsystem to retransmit msg counter */
				transmit_response_with_date(p, "200 OK", req);
				manager_event(EVENT_FLAG_SYSTEM, "PeerStatus", "ChannelType: SIP\r\nPeer: SIP/%s\r\nPeerStatus: Registered\r\nAddress: %s\r\n", peer->name, ast_sockaddr_stringify(addr));
				peer->lastmsgssent = -1;
				res = 0;
				break;
			}
			ao2_unlock(peer);
		}
	}
	if (!res) {
		ast_devstate_changed(AST_DEVICE_UNKNOWN, "SIP/%s", peer->name);
	}
	if (res < 0) {
		switch (res) {
		case AUTH_SECRET_FAILED:
			/* Wrong password in authentication. Go away, don't try again until you fixed it */
			transmit_response(p, "403 Forbidden (Bad auth)", &p->initreq);
			if (global_authfailureevents) {
				const char *peer_addr = ast_strdupa(ast_sockaddr_stringify_addr(addr));
				const char *peer_port = ast_strdupa(ast_sockaddr_stringify_port(addr));
				manager_event(EVENT_FLAG_SYSTEM, "PeerStatus",
					      "ChannelType: SIP\r\n"
					      "Peer: SIP/%s\r\n"
					      "PeerStatus: Rejected\r\n"
					      "Cause: AUTH_SECRET_FAILED\r\n"
					      "Address: %s\r\n"
					      "Port: %s\r\n",
					      name, peer_addr, peer_port);
			}
			break;
		case AUTH_USERNAME_MISMATCH:
			/* Username and digest username does not match.
			   Asterisk uses the From: username for authentication. We need the
			   devices to use the same authentication user name until we support
			   proper authentication by digest auth name */
		case AUTH_NOT_FOUND:
		case AUTH_PEER_NOT_DYNAMIC:
		case AUTH_ACL_FAILED:
			if (sip_cfg.alwaysauthreject) {
				transmit_fake_auth_response(p, SIP_REGISTER, &p->initreq, XMIT_UNRELIABLE);
				if (global_authfailureevents) {
					const char *peer_addr = ast_strdupa(ast_sockaddr_stringify_addr(addr));
					const char *peer_port = ast_strdupa(ast_sockaddr_stringify_port(addr));
					manager_event(EVENT_FLAG_SYSTEM, "PeerStatus",
						      "ChannelType: SIP\r\n"
						      "Peer: SIP/%s\r\n"
						      "PeerStatus: Rejected\r\n"
						      "Cause: %s\r\n"
						      "Address: %s\r\n"
						      "Port: %s\r\n",
						      name,
						      res == AUTH_PEER_NOT_DYNAMIC ? "AUTH_PEER_NOT_DYNAMIC" : "URI_NOT_FOUND",
						      peer_addr, peer_port);
				}
			} else {
				/* URI not found */
				if (res == AUTH_PEER_NOT_DYNAMIC) {
					transmit_response(p, "403 Forbidden", &p->initreq);
					if (global_authfailureevents) {
						const char *peer_addr = ast_strdupa(ast_sockaddr_stringify_addr(addr));
						const char *peer_port = ast_strdupa(ast_sockaddr_stringify_port(addr));
						manager_event(EVENT_FLAG_SYSTEM, "PeerStatus",
							"ChannelType: SIP\r\n"
							"Peer: SIP/%s\r\n"
							"PeerStatus: Rejected\r\n"
							"Cause: AUTH_PEER_NOT_DYNAMIC\r\n"
							"Address: %s\r\n"
							"Port: %s\r\n",
							name, peer_addr, peer_port);
					}
				} else {
					transmit_response(p, "404 Not found", &p->initreq);
					if (global_authfailureevents) {
						const char *peer_addr = ast_strdupa(ast_sockaddr_stringify_addr(addr));
						const char *peer_port = ast_strdupa(ast_sockaddr_stringify_port(addr));
						manager_event(EVENT_FLAG_SYSTEM, "PeerStatus",
							"ChannelType: SIP\r\n"
							"Peer: SIP/%s\r\n"
							"PeerStatus: Rejected\r\n"
							"Cause: %s\r\n"
							"Address: %s\r\n"
							"Port: %s\r\n",
							name,
							(res == AUTH_USERNAME_MISMATCH) ? "AUTH_USERNAME_MISMATCH" : "URI_NOT_FOUND",
							peer_addr, peer_port);
					}
				}
			}
			break;
		case AUTH_BAD_TRANSPORT:
		default:
			break;
		}
	}
	if (peer) {
		unref_peer(peer, "register_verify: unref_peer: tossing stack peer pointer at end of func");
	}

	return res;
}

/*! \brief Translate referring cause */
static void sip_set_redirstr(struct sip_pvt *p, char *reason) {

	if (!strcmp(reason, "unknown")) {
		ast_string_field_set(p, redircause, "UNKNOWN");
	} else if (!strcmp(reason, "user-busy")) {
		ast_string_field_set(p, redircause, "BUSY");
	} else if (!strcmp(reason, "no-answer")) {
		ast_string_field_set(p, redircause, "NOANSWER");
	} else if (!strcmp(reason, "unavailable")) {
		ast_string_field_set(p, redircause, "UNREACHABLE");
	} else if (!strcmp(reason, "unconditional")) {
		ast_string_field_set(p, redircause, "UNCONDITIONAL");
	} else if (!strcmp(reason, "time-of-day")) {
		ast_string_field_set(p, redircause, "UNKNOWN");
	} else if (!strcmp(reason, "do-not-disturb")) {
		ast_string_field_set(p, redircause, "UNKNOWN");
	} else if (!strcmp(reason, "deflection")) {
		ast_string_field_set(p, redircause, "UNKNOWN");
	} else if (!strcmp(reason, "follow-me")) {
		ast_string_field_set(p, redircause, "UNKNOWN");
	} else if (!strcmp(reason, "out-of-service")) {
		ast_string_field_set(p, redircause, "UNREACHABLE");
	} else if (!strcmp(reason, "away")) {
		ast_string_field_set(p, redircause, "UNREACHABLE");
	} else {
		ast_string_field_set(p, redircause, "UNKNOWN");
	}
}

/*! \brief Parse the parts of the P-Asserted-Identity header
 * on an incoming packet. Returns 1 if a valid header is found
 * and it is different from the current caller id.
 */
static int get_pai(struct sip_pvt *p, struct sip_request *req)
{
	char pai[256];
	char privacy[64];
	char *cid_num = NULL;
	char *cid_name = NULL;
	char emptyname[1] = "";
	int callingpres = AST_PRES_ALLOWED_USER_NUMBER_NOT_SCREENED;
	char *uri = NULL;
	int is_anonymous = 0, do_update = 1, no_name = 0;

	ast_copy_string(pai, get_header(req, "P-Asserted-Identity"), sizeof(pai));

	if (ast_strlen_zero(pai)) {
		return 0;
	}

	/* use the reqresp_parser function get_name_and_number*/
	if (get_name_and_number(pai, &cid_name, &cid_num)) {
		return 0;
	}

	if (global_shrinkcallerid && ast_is_shrinkable_phonenumber(cid_num)) {
		ast_shrink_phone_number(cid_num);
	}

	uri = get_in_brackets(pai);
	if (!strncasecmp(uri, "sip:anonymous@anonymous.invalid", 31)) {
		callingpres = AST_PRES_PROHIB_USER_NUMBER_NOT_SCREENED;
		/*XXX Assume no change in cid_num. Perhaps it should be
		 * blanked?
		 */
		ast_free(cid_num);
		is_anonymous = 1;
		cid_num = (char *)p->cid_num;
	}

	ast_copy_string(privacy, get_header(req, "Privacy"), sizeof(privacy));
	if (!ast_strlen_zero(privacy) && !strncmp(privacy, "id", 2)) {
		callingpres = AST_PRES_PROHIB_USER_NUMBER_NOT_SCREENED;
	}
	if (!cid_name) {
		no_name = 1;
		cid_name = (char *)emptyname;
	}	
	/* Only return true if the supplied caller id is different */
	if (!strcasecmp(p->cid_num, cid_num) && !strcasecmp(p->cid_name, cid_name) && p->callingpres == callingpres) {
		do_update = 0;
	} else {

		ast_string_field_set(p, cid_num, cid_num);
		ast_string_field_set(p, cid_name, cid_name);
		p->callingpres = callingpres;

		if (p->owner) {
			ast_set_callerid(p->owner, cid_num, cid_name, NULL);
			p->owner->caller.id.name.presentation = callingpres;
			p->owner->caller.id.number.presentation = callingpres;
		}
	}

	/* get_name_and_number allocates memory for cid_num and cid_name so we have to free it */
	if (!is_anonymous) {
		ast_free(cid_num);
	}
	if (!no_name) {
		ast_free(cid_name);
	}

	return do_update;
}

/*! \brief Get name, number and presentation from remote party id header,
 *  returns true if a valid header was found and it was different from the
 *  current caller id.
 */
static int get_rpid(struct sip_pvt *p, struct sip_request *oreq)
{
	char tmp[256];
	struct sip_request *req;
	char *cid_num = "";
	char *cid_name = "";
	int callingpres = AST_PRES_ALLOWED_USER_NUMBER_NOT_SCREENED;
	char *privacy = "";
	char *screen = "";
	char *start, *end;

	if (!ast_test_flag(&p->flags[0], SIP_TRUSTRPID))
		return 0;
	req = oreq;
	if (!req)
		req = &p->initreq;
	ast_copy_string(tmp, get_header(req, "Remote-Party-ID"), sizeof(tmp));
	if (ast_strlen_zero(tmp)) {
		return get_pai(p, req);
	}

	start = tmp;
	if (*start == '"') {
		*start++ = '\0';
		end = strchr(start, '"');
		if (!end)
			return 0;
		*end++ = '\0';
		cid_name = start;
		start = ast_skip_blanks(end);
	}

	if (*start != '<')
		return 0;
	*start++ = '\0';
	end = strchr(start, '@');
	if (!end)
		return 0;
	*end++ = '\0';
	if (strncasecmp(start, "sip:", 4))
		return 0;
	cid_num = start + 4;
	if (global_shrinkcallerid && ast_is_shrinkable_phonenumber(cid_num))
		ast_shrink_phone_number(cid_num);
	start = end;

	end = strchr(start, '>');
	if (!end)
		return 0;
	*end++ = '\0';
	if (*end) {
		start = end;
		if (*start != ';')
			return 0;
		*start++ = '\0';
		while (!ast_strlen_zero(start)) {
			end = strchr(start, ';');
			if (end)
				*end++ = '\0';
			if (!strncasecmp(start, "privacy=", 8))
				privacy = start + 8;
			else if (!strncasecmp(start, "screen=", 7))
				screen = start + 7;
			start = end;
		}

		if (!strcasecmp(privacy, "full")) {
			if (!strcasecmp(screen, "yes"))
				callingpres = AST_PRES_PROHIB_USER_NUMBER_PASSED_SCREEN;
			else if (!strcasecmp(screen, "no"))
				callingpres = AST_PRES_PROHIB_USER_NUMBER_NOT_SCREENED;
		} else {
			if (!strcasecmp(screen, "yes"))
				callingpres = AST_PRES_ALLOWED_USER_NUMBER_PASSED_SCREEN;
			else if (!strcasecmp(screen, "no"))
				callingpres = AST_PRES_ALLOWED_USER_NUMBER_NOT_SCREENED;
		}
	}

	/* Only return true if the supplied caller id is different */
	if (!strcasecmp(p->cid_num, cid_num) && !strcasecmp(p->cid_name, cid_name) && p->callingpres == callingpres)
		return 0;

	ast_string_field_set(p, cid_num, cid_num);
	ast_string_field_set(p, cid_name, cid_name);
	p->callingpres = callingpres;

	if (p->owner) {
		ast_set_callerid(p->owner, cid_num, cid_name, NULL);
		p->owner->caller.id.name.presentation = callingpres;
		p->owner->caller.id.number.presentation = callingpres;
	}

	return 1;
}

/*! \brief Get referring dnis */
static int get_rdnis(struct sip_pvt *p, struct sip_request *oreq, char **name, char **number, int *reason)
{
	char tmp[256], *exten, *rexten, *rdomain, *rname = NULL;
	char *params, *reason_param = NULL;
	struct sip_request *req;

	req = oreq ? oreq : &p->initreq;

	ast_copy_string(tmp, get_header(req, "Diversion"), sizeof(tmp));
	if (ast_strlen_zero(tmp))
		return -1;

	if ((params = strchr(tmp, '>'))) {
		params = strchr(params, ';');
	}

	exten = get_in_brackets(tmp);
	if (!strncasecmp(exten, "sip:", 4)) {
		exten += 4;
	} else if (!strncasecmp(exten, "sips:", 5)) {
		exten += 5;
	} else {
		ast_log(LOG_WARNING, "Huh?  Not an RDNIS SIP header (%s)?\n", exten);
		return -1;
	}

	/* Get diversion-reason param if present */
	if (params) {
		*params = '\0';	/* Cut off parameters  */
		params++;
		while (*params == ';' || *params == ' ')
			params++;
		/* Check if we have a reason parameter */
		if ((reason_param = strcasestr(params, "reason="))) {
			char *end;
			reason_param+=7;
			if ((end = strchr(reason_param, ';'))) {
				*end = '\0';
			}
			/* Remove enclosing double-quotes */
			if (*reason_param == '"')
				ast_strip_quoted(reason_param, "\"", "\"");
			if (!ast_strlen_zero(reason_param)) {
				sip_set_redirstr(p, reason_param);
				if (p->owner) {
					pbx_builtin_setvar_helper(p->owner, "__PRIREDIRECTREASON", p->redircause);
					pbx_builtin_setvar_helper(p->owner, "__SIPREDIRECTREASON", reason_param);
				}
			}
		}
	}

	rdomain = exten;
	rexten = strsep(&rdomain, "@");	/* trim anything after @ */
	if (p->owner)
		pbx_builtin_setvar_helper(p->owner, "__SIPRDNISDOMAIN", rdomain);

	if (sip_debug_test_pvt(p))
		ast_verbose("RDNIS for this call is %s (reason %s)\n", exten, reason ? reason_param : "");

	/*ast_string_field_set(p, rdnis, rexten);*/

	if (*tmp == '\"') {
		char *end_quote;
		rname = tmp + 1;
		end_quote = strchr(rname, '\"');
		*end_quote = '\0';
	}

	if (number) {
		*number = ast_strdup(rexten);
	}

	if (name && rname) {
		*name = ast_strdup(rname);
	}

	if (reason && !ast_strlen_zero(reason_param)) {
		*reason = sip_reason_str_to_code(reason_param);
	}

	return 0;
}

/*!
 * \brief Find out who the call is for.
 *
 * \details
 * We use the request uri as a destination.
 * This code assumes authentication has been done, so that the
 * device (peer/user) context is already set.
 *
 * \return 0 on success (found a matching extension), non-zero on failure
 *
 * \note If the incoming uri is a SIPS: uri, we are required to carry this across
 * the dialplan, so that the outbound call also is a sips: call or encrypted
 * IAX2 call. If that's not available, the call should FAIL.
 */
static enum sip_get_dest_result get_destination(struct sip_pvt *p, struct sip_request *oreq, int *cc_recall_core_id)
{
	char tmp[256] = "", *uri, *unused_password, *domain;
	char tmpf[256] = "", *from = NULL;
	struct sip_request *req;
	char *decoded_uri;

	req = oreq;
	if (!req) {
		req = &p->initreq;
	}

	/* Find the request URI */
	if (req->rlPart2)
		ast_copy_string(tmp, REQ_OFFSET_TO_STR(req, rlPart2), sizeof(tmp));
	
	uri = ast_strdupa(get_in_brackets(tmp));

	if (parse_uri_legacy_check(uri, "sip:,sips:", &uri, &unused_password, &domain, NULL)) {
		ast_log(LOG_WARNING, "Not a SIP header (%s)?\n", uri);
		return SIP_GET_DEST_INVALID_URI;
	}

	SIP_PEDANTIC_DECODE(domain);
	SIP_PEDANTIC_DECODE(uri);

	extract_host_from_hostport(&domain);

	if (ast_strlen_zero(uri)) {
		/*
		 * Either there really was no extension found or the request
		 * URI had encoded nulls that made the string "empty".  Use "s"
		 * as the extension.
		 */
		uri = "s";
	}

	ast_string_field_set(p, domain, domain);

	/* Now find the From: caller ID and name */
	/* XXX Why is this done in get_destination? Isn't it already done?
	   Needs to be checked
        */
	ast_copy_string(tmpf, get_header(req, "From"), sizeof(tmpf));
	if (!ast_strlen_zero(tmpf)) {
		from = get_in_brackets(tmpf);
		if (parse_uri_legacy_check(from, "sip:,sips:", &from, NULL, &domain, NULL)) {
			ast_log(LOG_WARNING, "Not a SIP header (%s)?\n", from);
			return SIP_GET_DEST_INVALID_URI;
		}

		SIP_PEDANTIC_DECODE(from);
		SIP_PEDANTIC_DECODE(domain);

		extract_host_from_hostport(&domain);

		ast_string_field_set(p, fromdomain, domain);
	}

	if (!AST_LIST_EMPTY(&domain_list)) {
		char domain_context[AST_MAX_EXTENSION];

		domain_context[0] = '\0';
		if (!check_sip_domain(p->domain, domain_context, sizeof(domain_context))) {
			if (!sip_cfg.allow_external_domains && (req->method == SIP_INVITE || req->method == SIP_REFER)) {
				ast_debug(1, "Got SIP %s to non-local domain '%s'; refusing request.\n", sip_methods[req->method].text, p->domain);
				return SIP_GET_DEST_REFUSED;
			}
		}
		/* If we don't have a peer (i.e. we're a guest call),
		 * overwrite the original context */
		if (!ast_test_flag(&p->flags[1], SIP_PAGE2_HAVEPEERCONTEXT) && !ast_strlen_zero(domain_context)) {
			ast_string_field_set(p, context, domain_context);
		}
	}

	/* If the request coming in is a subscription and subscribecontext has been specified use it */
	if (req->method == SIP_SUBSCRIBE && !ast_strlen_zero(p->subscribecontext)) {
		ast_string_field_set(p, context, p->subscribecontext);
	}

	if (sip_debug_test_pvt(p)) {
		ast_verbose("Looking for %s in %s (domain %s)\n", uri, p->context, p->domain);
	}

	/* Since extensions.conf can have unescaped characters, try matching a
	 * decoded uri in addition to the non-decoded uri. */
	decoded_uri = ast_strdupa(uri);
	ast_uri_decode(decoded_uri);

	/* If this is a subscription we actually just need to see if a hint exists for the extension */
	if (req->method == SIP_SUBSCRIBE) {
		char hint[AST_MAX_EXTENSION];
		int which = 0;
		if (ast_get_hint(hint, sizeof(hint), NULL, 0, NULL, p->context, uri) ||
		    (ast_get_hint(hint, sizeof(hint), NULL, 0, NULL, p->context, decoded_uri) && (which = 1))) {
			if (!oreq) {
				ast_string_field_set(p, exten, which ? decoded_uri : uri);
			}
			return SIP_GET_DEST_EXTEN_FOUND;
		} else {
			return SIP_GET_DEST_EXTEN_NOT_FOUND;
		}
	} else {
		struct ast_cc_agent *agent;
		/* Check the dialplan for the username part of the request URI,
		   the domain will be stored in the SIPDOMAIN variable
		   Return 0 if we have a matching extension */
		if (ast_exists_extension(NULL, p->context, uri, 1, S_OR(p->cid_num, from))) {
			if (!oreq) {
				ast_string_field_set(p, exten, uri);
			}
			return SIP_GET_DEST_EXTEN_FOUND;
		}
		if (ast_exists_extension(NULL, p->context, decoded_uri, 1, S_OR(p->cid_num, from))
			|| !strcmp(decoded_uri, ast_pickup_ext())) {
			if (!oreq) {
				ast_string_field_set(p, exten, decoded_uri);
			}
			return SIP_GET_DEST_EXTEN_FOUND;
		}
		if ((agent = find_sip_cc_agent_by_notify_uri(tmp))) {
			struct sip_cc_agent_pvt *agent_pvt = agent->private_data;
			/* This is a CC recall. We can set p's extension to the exten from
			 * the original INVITE
			 */
			ast_string_field_set(p, exten, agent_pvt->original_exten);
			/* And we need to let the CC core know that the caller is attempting
			 * his recall
			 */
			ast_cc_agent_recalling(agent->core_id, "SIP caller %s is attempting recall",
					agent->device_name);
			if (cc_recall_core_id) {
				*cc_recall_core_id = agent->core_id;
			}
			ao2_ref(agent, -1);
			return SIP_GET_DEST_EXTEN_FOUND;
		}
	}

	if (ast_test_flag(&global_flags[1], SIP_PAGE2_ALLOWOVERLAP)
		&& (ast_canmatch_extension(NULL, p->context, uri, 1, S_OR(p->cid_num, from))
			|| ast_canmatch_extension(NULL, p->context, decoded_uri, 1, S_OR(p->cid_num, from))
			|| !strncmp(decoded_uri, ast_pickup_ext(), strlen(decoded_uri)))) {
		/* Overlap dialing is enabled and we need more digits to match an extension. */
		return SIP_GET_DEST_EXTEN_MATCHMORE;
	}

	return SIP_GET_DEST_EXTEN_NOT_FOUND;
}

/*! \brief Lock dialog lock and find matching pvt lock
	\return a reference, remember to release it when done
*/
static struct sip_pvt *get_sip_pvt_byid_locked(const char *callid, const char *totag, const char *fromtag)
{
	struct sip_pvt *sip_pvt_ptr;
	struct sip_pvt tmp_dialog = {
		.callid = callid,
	};

	if (totag) {
		ast_debug(4, "Looking for callid %s (fromtag %s totag %s)\n", callid, fromtag ? fromtag : "<no fromtag>", totag ? totag : "<no totag>");
	}

	/* Search dialogs and find the match */
	
	sip_pvt_ptr = ao2_t_find(dialogs, &tmp_dialog, OBJ_POINTER, "ao2_find of dialog in dialogs table");
	if (sip_pvt_ptr) {
		/* Go ahead and lock it (and its owner) before returning */
		sip_pvt_lock(sip_pvt_ptr);
		if (sip_cfg.pedanticsipchecking) {
			unsigned char frommismatch = 0, tomismatch = 0;

			if (ast_strlen_zero(fromtag)) {
				sip_pvt_unlock(sip_pvt_ptr);
				ast_debug(4, "Matched %s call for callid=%s - no from tag specified, pedantic check fails\n",
					  sip_pvt_ptr->outgoing_call == TRUE ? "OUTGOING": "INCOMING", sip_pvt_ptr->callid);
				return NULL;
			}

			if (ast_strlen_zero(totag)) {
				sip_pvt_unlock(sip_pvt_ptr);
				ast_debug(4, "Matched %s call for callid=%s - no to tag specified, pedantic check fails\n",
					  sip_pvt_ptr->outgoing_call == TRUE ? "OUTGOING": "INCOMING", sip_pvt_ptr->callid);
				return NULL;
			}
			/* RFC 3891
			 * > 3.  User Agent Server Behavior: Receiving a Replaces Header
			 * > The Replaces header contains information used to match an existing
			 * > SIP dialog (call-id, to-tag, and from-tag).  Upon receiving an INVITE
			 * > with a Replaces header, the User Agent (UA) attempts to match this
			 * > information with a confirmed or early dialog.  The User Agent Server
			 * > (UAS) matches the to-tag and from-tag parameters as if they were tags
			 * > present in an incoming request.  In other words, the to-tag parameter
			 * > is compared to the local tag, and the from-tag parameter is compared
			 * > to the remote tag.
			 *
			 * Thus, the totag is always compared to the local tag, regardless if
			 * this our call is an incoming or outgoing call.
			 */
			frommismatch = !!strcmp(fromtag, sip_pvt_ptr->theirtag);
			tomismatch = !!strcmp(totag, sip_pvt_ptr->tag);

			if (frommismatch || tomismatch) {
				sip_pvt_unlock(sip_pvt_ptr);
				if (frommismatch) {
					ast_debug(4, "Matched %s call for callid=%s - pedantic from tag check fails; their tag is %s our tag is %s\n",
						  sip_pvt_ptr->outgoing_call == TRUE ? "OUTGOING": "INCOMING", sip_pvt_ptr->callid,
						  fromtag, sip_pvt_ptr->theirtag);
				}
				if (tomismatch) {
					ast_debug(4, "Matched %s call for callid=%s - pedantic to tag check fails; their tag is %s our tag is %s\n",
						  sip_pvt_ptr->outgoing_call == TRUE ? "OUTGOING": "INCOMING", sip_pvt_ptr->callid,
						  totag, sip_pvt_ptr->tag);
				}
				return NULL;
			}
		}
		
		if (totag)
			ast_debug(4, "Matched %s call - their tag is %s Our tag is %s\n",
					  sip_pvt_ptr->outgoing_call == TRUE ? "OUTGOING": "INCOMING",
					  sip_pvt_ptr->theirtag, sip_pvt_ptr->tag);

		/* deadlock avoidance... */
		while (sip_pvt_ptr->owner && ast_channel_trylock(sip_pvt_ptr->owner)) {
			sip_pvt_unlock(sip_pvt_ptr);
			usleep(1);
			sip_pvt_lock(sip_pvt_ptr);
		}
	}
	
	return sip_pvt_ptr;
}

/*! \brief Call transfer support (the REFER method)
 * 	Extracts Refer headers into pvt dialog structure
 *
 * \note If we get a SIPS uri in the refer-to header, we're required to set up a secure signalling path
 *	to that extension. As a minimum, this needs to be added to a channel variable, if not a channel
 *	flag.
 */
static int get_refer_info(struct sip_pvt *transferer, struct sip_request *outgoing_req)
{

	const char *p_referred_by = NULL;
	char *h_refer_to = NULL;
	char *h_referred_by = NULL;
	char *refer_to;
	const char *p_refer_to;
	char *referred_by_uri = NULL;
	char *ptr;
	struct sip_request *req = NULL;
	const char *transfer_context = NULL;
	struct sip_refer *referdata;


	req = outgoing_req;
	referdata = transferer->refer;

	if (!req) {
		req = &transferer->initreq;
	}

	p_refer_to = get_header(req, "Refer-To");
	if (ast_strlen_zero(p_refer_to)) {
		ast_log(LOG_WARNING, "Refer-To Header missing. Skipping transfer.\n");
		return -2;	/* Syntax error */
	}
	h_refer_to = ast_strdupa(p_refer_to);
	refer_to = get_in_brackets(h_refer_to);
	if (!strncasecmp(refer_to, "sip:", 4)) {
		refer_to += 4;			/* Skip sip: */
	} else if (!strncasecmp(refer_to, "sips:", 5)) {
		refer_to += 5;
	} else {
		ast_log(LOG_WARNING, "Can't transfer to non-sip: URI.  (Refer-to: %s)?\n", refer_to);
		return -3;
	}

	/* Get referred by header if it exists */
	p_referred_by = get_header(req, "Referred-By");

	/* Give useful transfer information to the dialplan */
	if (transferer->owner) {
		struct ast_channel *peer = ast_bridged_channel(transferer->owner);
		if (peer) {
			pbx_builtin_setvar_helper(peer, "SIPREFERRINGCONTEXT", transferer->context);
			pbx_builtin_setvar_helper(peer, "SIPREFERREDBYHDR", p_referred_by);
		}
	}

	if (!ast_strlen_zero(p_referred_by)) {
		char *lessthan;
		h_referred_by = ast_strdupa(p_referred_by);

		/* Store referrer's caller ID name */
		ast_copy_string(referdata->referred_by_name, h_referred_by, sizeof(referdata->referred_by_name));
		if ((lessthan = strchr(referdata->referred_by_name, '<'))) {
			*(lessthan - 1) = '\0';	/* Space */
		}

		referred_by_uri = get_in_brackets(h_referred_by);

		if (!strncasecmp(referred_by_uri, "sip:", 4)) {
			referred_by_uri += 4;		/* Skip sip: */
		} else if (!strncasecmp(referred_by_uri, "sips:", 5)) {
			referred_by_uri += 5;		/* Skip sips: */
		} else {
			ast_log(LOG_WARNING, "Huh?  Not a sip: header (Referred-by: %s). Skipping.\n", referred_by_uri);
			referred_by_uri = NULL;
		}
	}

	/* Check for arguments in the refer_to header */
	if ((ptr = strcasestr(refer_to, "replaces="))) {
		char *to = NULL, *from = NULL;
		
		/* This is an attended transfer */
		referdata->attendedtransfer = 1;
		ast_copy_string(referdata->replaces_callid, ptr+9, sizeof(referdata->replaces_callid));
		ast_uri_decode(referdata->replaces_callid);
		if ((ptr = strchr(referdata->replaces_callid, ';'))) 	/* Find options */ {
			*ptr++ = '\0';
		}
		
		if (ptr) {
			/* Find the different tags before we destroy the string */
			to = strcasestr(ptr, "to-tag=");
			from = strcasestr(ptr, "from-tag=");
		}
		
		/* Grab the to header */
		if (to) {
			ptr = to + 7;
			if ((to = strchr(ptr, '&'))) {
				*to = '\0';
			}
			if ((to = strchr(ptr, ';'))) {
				*to = '\0';
			}
			ast_copy_string(referdata->replaces_callid_totag, ptr, sizeof(referdata->replaces_callid_totag));
		}
		
		if (from) {
			ptr = from + 9;
			if ((to = strchr(ptr, '&'))) {
				*to = '\0';
			}
			if ((to = strchr(ptr, ';'))) {
				*to = '\0';
			}
			ast_copy_string(referdata->replaces_callid_fromtag, ptr, sizeof(referdata->replaces_callid_fromtag));
		}

		if (!strcmp(referdata->replaces_callid, transferer->callid) &&
			(!sip_cfg.pedanticsipchecking ||
			(!strcmp(referdata->replaces_callid_fromtag, transferer->theirtag) &&
			!strcmp(referdata->replaces_callid_totag, transferer->tag)))) {
				ast_log(LOG_WARNING, "Got an attempt to replace own Call-ID on %s\n", transferer->callid);
				return -4;
		}

		if (!sip_cfg.pedanticsipchecking) {
			ast_debug(2, "Attended transfer: Will use Replace-Call-ID : %s (No check of from/to tags)\n", referdata->replaces_callid );
		} else {
			ast_debug(2, "Attended transfer: Will use Replace-Call-ID : %s F-tag: %s T-tag: %s\n", referdata->replaces_callid, referdata->replaces_callid_fromtag ? referdata->replaces_callid_fromtag : "<none>", referdata->replaces_callid_totag ? referdata->replaces_callid_totag : "<none>" );
		}
	}
	
	if ((ptr = strchr(refer_to, '@'))) {	/* Separate domain */
		char *urioption = NULL, *domain;
		int bracket = 0;
		*ptr++ = '\0';

		if ((urioption = strchr(ptr, ';'))) { /* Separate urioptions */
			*urioption++ = '\0';
		}

		domain = ptr;

		/* Remove :port */
		for (; *ptr != '\0'; ++ptr) {
			if (*ptr == ':' && bracket == 0) {
				*ptr = '\0';
				break;
			} else if (*ptr == '[') {
				++bracket;
			} else if (*ptr == ']') {
				--bracket;
			}
		}

		SIP_PEDANTIC_DECODE(domain);
		SIP_PEDANTIC_DECODE(urioption);

		/* Save the domain for the dial plan */
		ast_copy_string(referdata->refer_to_domain, domain, sizeof(referdata->refer_to_domain));
		if (urioption) {
			ast_copy_string(referdata->refer_to_urioption, urioption, sizeof(referdata->refer_to_urioption));
		}
	}

	if ((ptr = strchr(refer_to, ';'))) 	/* Remove options */
		*ptr = '\0';

	SIP_PEDANTIC_DECODE(refer_to);
	ast_copy_string(referdata->refer_to, refer_to, sizeof(referdata->refer_to));
	
	if (referred_by_uri) {
		if ((ptr = strchr(referred_by_uri, ';'))) 	/* Remove options */
			*ptr = '\0';
		SIP_PEDANTIC_DECODE(referred_by_uri);
		ast_copy_string(referdata->referred_by, referred_by_uri, sizeof(referdata->referred_by));
	} else {
		referdata->referred_by[0] = '\0';
	}

	/* Determine transfer context */
	if (transferer->owner)	/* Mimic behaviour in res_features.c */
		transfer_context = pbx_builtin_getvar_helper(transferer->owner, "TRANSFER_CONTEXT");

	/* By default, use the context in the channel sending the REFER */
	if (ast_strlen_zero(transfer_context)) {
		transfer_context = S_OR(transferer->owner->macrocontext,
					S_OR(transferer->context, sip_cfg.default_context));
	}

	ast_copy_string(referdata->refer_to_context, transfer_context, sizeof(referdata->refer_to_context));
	
	/* Either an existing extension or the parking extension */
	if (referdata->attendedtransfer || ast_exists_extension(NULL, transfer_context, refer_to, 1, NULL) ) {
		if (sip_debug_test_pvt(transferer)) {
			ast_verbose("SIP transfer to extension %s@%s by %s\n", refer_to, transfer_context, referred_by_uri);
		}
		/* We are ready to transfer to the extension */
		return 0;
	}
	if (sip_debug_test_pvt(transferer))
		ast_verbose("Failed SIP Transfer to non-existing extension %s in context %s\n n", refer_to, transfer_context);

	/* Failure, we can't find this extension */
	return -1;
}


/*! \brief Call transfer support (old way, deprecated by the IETF)
 *	\note does not account for SIPS: uri requirements, nor check transport
 */
static int get_also_info(struct sip_pvt *p, struct sip_request *oreq)
{
	char tmp[256] = "", *c, *a;
	struct sip_request *req = oreq ? oreq : &p->initreq;
	struct sip_refer *referdata = NULL;
	const char *transfer_context = NULL;
	
	if (!p->refer && !sip_refer_allocate(p))
		return -1;

	referdata = p->refer;

	ast_copy_string(tmp, get_header(req, "Also"), sizeof(tmp));
	c = get_in_brackets(tmp);

	if (parse_uri_legacy_check(c, "sip:,sips:", &c, NULL, &a, NULL)) {
		ast_log(LOG_WARNING, "Huh?  Not a SIP header in Also: transfer (%s)?\n", c);
		return -1;
	}
	
	SIP_PEDANTIC_DECODE(c);
	SIP_PEDANTIC_DECODE(a);

	if (!ast_strlen_zero(a)) {
		ast_copy_string(referdata->refer_to_domain, a, sizeof(referdata->refer_to_domain));
	}

	if (sip_debug_test_pvt(p))
		ast_verbose("Looking for %s in %s\n", c, p->context);

	if (p->owner)	/* Mimic behaviour in res_features.c */
		transfer_context = pbx_builtin_getvar_helper(p->owner, "TRANSFER_CONTEXT");

	/* By default, use the context in the channel sending the REFER */
	if (ast_strlen_zero(transfer_context)) {
		transfer_context = S_OR(p->owner->macrocontext,
					S_OR(p->context, sip_cfg.default_context));
	}
	if (ast_exists_extension(NULL, transfer_context, c, 1, NULL)) {
		/* This is a blind transfer */
		ast_debug(1, "SIP Bye-also transfer to Extension %s@%s \n", c, transfer_context);
		ast_copy_string(referdata->refer_to, c, sizeof(referdata->refer_to));
		ast_copy_string(referdata->referred_by, "", sizeof(referdata->referred_by));
		ast_copy_string(referdata->refer_contact, "", sizeof(referdata->refer_contact));
		/* Set new context */
		ast_string_field_set(p, context, transfer_context);
		return 0;
	} else if (ast_canmatch_extension(NULL, p->context, c, 1, NULL)) {
		return 1;
	}

	return -1;
}

/*! \brief check received= and rport= in a SIP response.
 * If we get a response with received= and/or rport= in the Via:
 * line, use them as 'p->ourip' (see RFC 3581 for rport,
 * and RFC 3261 for received).
 * Using these two fields SIP can produce the correct
 * address and port in the SIP headers without the need for STUN.
 * The address part is also reused for the media sessions.
 * Note that ast_sip_ouraddrfor() still rewrites p->ourip
 * if you specify externaddr/seternaddr/.
 */
static attribute_unused void check_via_response(struct sip_pvt *p, struct sip_request *req)
{
	char via[256];
	char *cur, *opts;

	ast_copy_string(via, get_header(req, "Via"), sizeof(via));

	/* Work on the leftmost value of the topmost Via header */
	opts = strchr(via, ',');
	if (opts)
		*opts = '\0';

	/* parse all relevant options */
	opts = strchr(via, ';');
	if (!opts)
		return;	/* no options to parse */
	*opts++ = '\0';
	while ( (cur = strsep(&opts, ";")) ) {
		if (!strncmp(cur, "rport=", 6)) {
			int port = strtol(cur+6, NULL, 10);
			/* XXX add error checking */
			ast_sockaddr_set_port(&p->ourip, port);
		} else if (!strncmp(cur, "received=", 9)) {
			if (ast_parse_arg(cur + 9, PARSE_ADDR, &p->ourip))
				;	/* XXX add error checking */
		}
	}
}

/*! \brief check Via: header for hostname, port and rport request/answer */
static void check_via(struct sip_pvt *p, struct sip_request *req)
{
	char via[512];
	char *c, *maddr;
	struct ast_sockaddr tmp = { { 0, } };
	uint16_t port;

	ast_copy_string(via, get_header(req, "Via"), sizeof(via));

	/* Work on the leftmost value of the topmost Via header */
	c = strchr(via, ',');
	if (c)
		*c = '\0';

	/* Check for rport */
	c = strstr(via, ";rport");
	if (c && (c[6] != '='))	{ /* rport query, not answer */
		ast_set_flag(&p->flags[1], SIP_PAGE2_RPORT_PRESENT);
		ast_set_flag(&p->flags[0], SIP_NAT_RPORT_PRESENT);
	}

	/* Check for maddr */
	maddr = strstr(via, "maddr=");
	if (maddr) {
		maddr += 6;
		c = maddr + strspn(maddr, "abcdefghijklmnopqrstuvwxyz"
				          "ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789-.:[]");
		*c = '\0';
	}

	c = strchr(via, ';');
	if (c)
		*c = '\0';

	c = strchr(via, ' ');
	if (c) {
		*c = '\0';
		c = ast_skip_blanks(c+1);
		if (strcasecmp(via, "SIP/2.0/UDP") && strcasecmp(via, "SIP/2.0/TCP") && strcasecmp(via, "SIP/2.0/TLS")) {
			ast_log(LOG_WARNING, "Don't know how to respond via '%s'\n", via);
			return;
		}

		if (maddr && ast_sockaddr_resolve_first(&p->sa, maddr, 0)) {
			p->sa = p->recv;
		}

		ast_sockaddr_resolve_first(&tmp, c, 0);
		port = ast_sockaddr_port(&tmp);
		ast_sockaddr_set_port(&p->sa,
				      port != 0 ? port : STANDARD_SIP_PORT);

		if (sip_debug_test_pvt(p)) {
			ast_verbose("Sending to %s (%s)\n",
				    ast_sockaddr_stringify(sip_real_dst(p)),
				    sip_nat_mode(p));
		}
	}
}

/*! \brief Validate device authentication */
static enum check_auth_result check_peer_ok(struct sip_pvt *p, char *of,
	struct sip_request *req, int sipmethod, struct ast_sockaddr *addr,
	struct sip_peer **authpeer,
	enum xmittype reliable, char *calleridname, char *uri2)
{
	enum check_auth_result res;
	int debug = sip_debug_test_addr(addr);
	struct sip_peer *peer;

	if (sipmethod == SIP_SUBSCRIBE) {
		/* For subscribes, match on device name only; for other methods,
	 	* match on IP address-port of the incoming request.
	 	*/
		peer = find_peer(of, NULL, TRUE, FINDALLDEVICES, FALSE, 0);
	} else {
		/* First find devices based on username (avoid all type=peer's) */
		peer = find_peer(of, NULL, TRUE, FINDUSERS, FALSE, 0);

		/* Then find devices based on IP */
		if (!peer) {
			peer = find_peer(NULL, &p->recv, TRUE, FINDPEERS, FALSE, p->socket.type);
		}
	}

	if (!peer) {
		if (debug) {
			ast_verbose("No matching peer for '%s' from '%s'\n",
				of, ast_sockaddr_stringify(&p->recv));
		}
		return AUTH_DONT_KNOW;
	}

	if (!ast_apply_ha(peer->ha, addr)) {
		ast_debug(2, "Found peer '%s' for '%s', but fails host access\n", peer->name, of);
		unref_peer(peer, "unref_peer: check_peer_ok: from find_peer call, early return of AUTH_ACL_FAILED");
		return AUTH_ACL_FAILED;
	}
	if (debug)
		ast_verbose("Found peer '%s' for '%s' from %s\n",
			peer->name, of, ast_sockaddr_stringify(&p->recv));

	/* XXX what about p->prefs = peer->prefs; ? */
	/* Set Frame packetization */
	if (p->rtp) {
		ast_rtp_codecs_packetization_set(ast_rtp_instance_get_codecs(p->rtp), p->rtp, &peer->prefs);
		p->autoframing = peer->autoframing;
	}

	/* Take the peer */
	ast_copy_flags(&p->flags[0], &peer->flags[0], SIP_FLAGS_TO_COPY);
	ast_copy_flags(&p->flags[1], &peer->flags[1], SIP_PAGE2_FLAGS_TO_COPY);
	ast_copy_flags(&p->flags[2], &peer->flags[2], SIP_PAGE3_FLAGS_TO_COPY);

	if (ast_test_flag(&p->flags[1], SIP_PAGE2_T38SUPPORT) && p->udptl) {
		p->t38_maxdatagram = peer->t38_maxdatagram;
		set_t38_capabilities(p);
	}

	/* Copy SIP extensions profile to peer */
	/* XXX is this correct before a successful auth ? */
	if (p->sipoptions)
		peer->sipoptions = p->sipoptions;

	do_setnat(p);

	ast_string_field_set(p, peersecret, peer->secret);
	ast_string_field_set(p, peermd5secret, peer->md5secret);
	ast_string_field_set(p, subscribecontext, peer->subscribecontext);
	ast_string_field_set(p, mohinterpret, peer->mohinterpret);
	ast_string_field_set(p, mohsuggest, peer->mohsuggest);
	if (!ast_strlen_zero(peer->parkinglot)) {
		ast_string_field_set(p, parkinglot, peer->parkinglot);
	}
	ast_string_field_set(p, engine, peer->engine);
	p->disallowed_methods = peer->disallowed_methods;
	set_pvt_allowed_methods(p, req);
	ast_cc_copy_config_params(p->cc_params, peer->cc_params);
	if (peer->callingpres)	/* Peer calling pres setting will override RPID */
		p->callingpres = peer->callingpres;
	if (peer->maxms && peer->lastms)
		p->timer_t1 = peer->lastms < global_t1min ? global_t1min : peer->lastms;
 	else
 		p->timer_t1 = peer->timer_t1;

 	/* Set timer B to control transaction timeouts */
 	if (peer->timer_b)
 		p->timer_b = peer->timer_b;
 	else
 		p->timer_b = 64 * p->timer_t1;

	if (ast_test_flag(&peer->flags[0], SIP_INSECURE_INVITE)) {
		/* Pretend there is no required authentication */
		ast_string_field_set(p, peersecret, NULL);
		ast_string_field_set(p, peermd5secret, NULL);
	}
	if (!(res = check_auth(p, req, peer->name, p->peersecret, p->peermd5secret, sipmethod, uri2, reliable, req->ignore))) {
		ast_copy_flags(&p->flags[0], &peer->flags[0], SIP_FLAGS_TO_COPY);
		ast_copy_flags(&p->flags[1], &peer->flags[1], SIP_PAGE2_FLAGS_TO_COPY);
		ast_copy_flags(&p->flags[2], &peer->flags[2], SIP_PAGE3_FLAGS_TO_COPY);
		/* If we have a call limit, set flag */
		if (peer->call_limit)
			ast_set_flag(&p->flags[0], SIP_CALL_LIMIT);
		ast_string_field_set(p, peername, peer->name);
		ast_string_field_set(p, authname, peer->name);

		if (sipmethod == SIP_INVITE) {
			/* destroy old channel vars and copy in new ones. */
			ast_variables_destroy(p->chanvars);
			p->chanvars = copy_vars(peer->chanvars);
		}

		if (authpeer) {
			ao2_t_ref(peer, 1, "copy pointer into (*authpeer)");
			(*authpeer) = peer;	/* Add a ref to the object here, to keep it in memory a bit longer if it is realtime */
		}

		if (!ast_strlen_zero(peer->username)) {
			ast_string_field_set(p, username, peer->username);
			/* Use the default username for authentication on outbound calls */
			/* XXX this takes the name from the caller... can we override ? */
			ast_string_field_set(p, authname, peer->username);
		}
		if (!get_rpid(p, req)) {
			if (!ast_strlen_zero(peer->cid_num)) {
				char *tmp = ast_strdupa(peer->cid_num);
				if (global_shrinkcallerid && ast_is_shrinkable_phonenumber(tmp))
					ast_shrink_phone_number(tmp);
				ast_string_field_set(p, cid_num, tmp);
			}
			if (!ast_strlen_zero(peer->cid_name))
				ast_string_field_set(p, cid_name, peer->cid_name);
			if (!ast_strlen_zero(peer->cid_tag))
				ast_string_field_set(p, cid_tag, peer->cid_tag);
			if (peer->callingpres)
				p->callingpres = peer->callingpres;
		}
		ast_string_field_set(p, fullcontact, peer->fullcontact);

		if (!ast_strlen_zero(peer->context)) {
			ast_string_field_set(p, context, peer->context);
		}
		if (!ast_strlen_zero(peer->mwi_from)) {
			ast_string_field_set(p, mwi_from, peer->mwi_from);
		}

		ast_string_field_set(p, peersecret, peer->secret);
		ast_string_field_set(p, peermd5secret, peer->md5secret);
		ast_string_field_set(p, language, peer->language);
		ast_string_field_set(p, accountcode, peer->accountcode);
		p->amaflags = peer->amaflags;
		p->callgroup = peer->callgroup;
		p->pickupgroup = peer->pickupgroup;
		p->capability = peer->capability;
		p->prefs = peer->prefs;
		p->jointcapability = peer->capability;
 		if (peer->maxforwards > 0) {
			p->maxforwards = peer->maxforwards;
		}
		if (p->peercapability)
			p->jointcapability &= p->peercapability;
		p->maxcallbitrate = peer->maxcallbitrate;
		if ((ast_test_flag(&p->flags[0], SIP_DTMF) == SIP_DTMF_RFC2833) ||
		    (ast_test_flag(&p->flags[0], SIP_DTMF) == SIP_DTMF_AUTO))
			p->noncodeccapability |= AST_RTP_DTMF;
		else
			p->noncodeccapability &= ~AST_RTP_DTMF;
		p->jointnoncodeccapability = p->noncodeccapability;
		p->rtptimeout = peer->rtptimeout;
		p->rtpholdtimeout = peer->rtpholdtimeout;
		p->rtpkeepalive = peer->rtpkeepalive;
		if (!dialog_initialize_rtp(p)) {
			if (p->rtp) {
				ast_rtp_codecs_packetization_set(ast_rtp_instance_get_codecs(p->rtp), p->rtp, &peer->prefs);
				p->autoframing = peer->autoframing;
			}
		} else {
			res = AUTH_RTP_FAILED;
		}
	}
	unref_peer(peer, "check_peer_ok: unref_peer: tossing temp ptr to peer from find_peer");
	return res;
}


/*! \brief  Check if matching user or peer is defined
 	Match user on From: user name and peer on IP/port
	This is used on first invite (not re-invites) and subscribe requests
    \return 0 on success, non-zero on failure
*/
static enum check_auth_result check_user_full(struct sip_pvt *p, struct sip_request *req,
					      int sipmethod, const char *uri, enum xmittype reliable,
					      struct ast_sockaddr *addr, struct sip_peer **authpeer)
{
	char from[256] = "", *of, *name, *unused_password, *domain;
	enum check_auth_result res = AUTH_DONT_KNOW;
	char calleridname[50];
	char *uri2 = ast_strdupa(uri);

	terminate_uri(uri2);	/* trim extra stuff */

	ast_copy_string(from, get_header(req, "From"), sizeof(from));
	/* XXX here tries to map the username for invite things */

	/* strip the display-name portion off the beginning of the FROM header. */
	if (!(of = (char *) get_calleridname(from, calleridname, sizeof(calleridname)))) {
		ast_log(LOG_ERROR, "FROM header can not be parsed \n");
		return res;
	}

	if (calleridname[0]) {
		ast_string_field_set(p, cid_name, calleridname);
	}

	if (ast_strlen_zero(p->exten)) {
		char *t = uri2;
		if (!strncasecmp(t, "sip:", 4))
			t+= 4;
		else if (!strncasecmp(t, "sips:", 5))
			t += 5;
		ast_string_field_set(p, exten, t);
		t = strchr(p->exten, '@');
		if (t)
			*t = '\0';

		if (ast_strlen_zero(p->our_contact))
			build_contact(p);
	}

	of = get_in_brackets(of);

	/* save the URI part of the From header */
	ast_string_field_set(p, from, of);

	if (parse_uri_legacy_check(of, "sip:,sips:", &name, &unused_password, &domain, NULL)) {
		ast_log(LOG_NOTICE, "From address missing 'sip:', using it anyway\n");
	}

	SIP_PEDANTIC_DECODE(name);
	SIP_PEDANTIC_DECODE(domain);

	extract_host_from_hostport(&domain);

	if (ast_strlen_zero(domain)) {
		/* <sip:name@[EMPTY]>, never good */
		ast_log(LOG_ERROR, "Empty domain name in FROM header\n");
		return res;
	}

	if (ast_strlen_zero(name)) {
		/* <sip:[EMPTY][@]hostport>. Asterisk 1.4 and 1.6 have always
		 * treated that as a username, so we continue the tradition:
		 * uri is now <sip:host@hostport>. */
		name = domain;
	} else {
		/* Non-empty name, try to get caller id from it */
		char *tmp = ast_strdupa(name);
		/* We need to be able to handle from-headers looking like
			<sip:8164444422;phone-context=+1@1.2.3.4:5060;user=phone;tag=SDadkoa01-gK0c3bdb43>
		*/
		tmp = strsep(&tmp, ";");
		if (global_shrinkcallerid && ast_is_shrinkable_phonenumber(tmp)) {
			ast_shrink_phone_number(tmp);
		}
		ast_string_field_set(p, cid_num, tmp);
	}

	if (global_match_auth_username) {
		/*
		 * XXX This is experimental code to grab the search key from the
		 * Auth header's username instead of the 'From' name, if available.
		 * Do not enable this block unless you understand the side effects (if any!)
		 * Note, the search for "username" should be done in a more robust way.
		 * Note2, at the moment we check both fields, though maybe we should
		 * pick one or another depending on the request ? XXX
		 */
		const char *hdr = get_header(req, "Authorization");
		if (ast_strlen_zero(hdr)) {
			hdr = get_header(req, "Proxy-Authorization");
		}

		if (!ast_strlen_zero(hdr) && (hdr = strstr(hdr, "username=\""))) {
			ast_copy_string(from, hdr + strlen("username=\""), sizeof(from));
			name = from;
			name = strsep(&name, "\"");
		}
	}

	res = check_peer_ok(p, name, req, sipmethod, addr,
			authpeer, reliable, calleridname, uri2);
	if (res != AUTH_DONT_KNOW) {
		return res;
	}

	/* Finally, apply the guest policy */
	if (sip_cfg.allowguest) {
		get_rpid(p, req);
		p->rtptimeout = global_rtptimeout;
		p->rtpholdtimeout = global_rtpholdtimeout;
		p->rtpkeepalive = global_rtpkeepalive;
		if (!dialog_initialize_rtp(p)) {
			res = AUTH_SUCCESSFUL;
		} else {
			res = AUTH_RTP_FAILED;
		}
	} else if (sip_cfg.alwaysauthreject) {
		res = AUTH_FAKE_AUTH; /* reject with fake authorization request */
	} else {
		res = AUTH_SECRET_FAILED; /* we don't want any guests, authentication will fail */
	}

	if (ast_test_flag(&p->flags[1], SIP_PAGE2_RPORT_PRESENT)) {
		ast_set_flag(&p->flags[0], SIP_NAT_RPORT_PRESENT);
	}

	return res;
}

/*! \brief  Find user
	If we get a match, this will add a reference pointer to the user object in ASTOBJ, that needs to be unreferenced
*/
static int check_user(struct sip_pvt *p, struct sip_request *req, int sipmethod, const char *uri, enum xmittype reliable, struct ast_sockaddr *addr)
{
	return check_user_full(p, req, sipmethod, uri, reliable, addr, NULL);
}

/*! \brief Get message body from a SIP request
 * \param buf Destination buffer
 * \param len Destination buffer size
 * \param req The SIP request
 *
 * When parsing the request originally, the lines are split by LF or CRLF.
 * This function adds a single LF after every line.
 */
static int get_msg_text(char *buf, int len, struct sip_request *req)
{
	int x;
	int linelen;

	buf[0] = '\0';
	--len; /* reserve strncat null */
	for (x = 0; len && x < req->lines; ++x) {
		const char *line = REQ_OFFSET_TO_STR(req, line[x]);
		strncat(buf, line, len); /* safe */
		linelen = strlen(buf);
		buf += linelen;
		len -= linelen;
		if (len) {
			strcat(buf, "\n"); /* safe */
			++buf;
			--len;
		}
	}
	return 0;
}


/*! \brief  Receive SIP MESSAGE method messages
\note	We only handle messages within current calls currently
	Reference: RFC 3428 */
static void receive_message(struct sip_pvt *p, struct sip_request *req)
{
	char buf[1400];	
	char *bufp;
	struct ast_frame f;
	const char *content_type = get_header(req, "Content-Type");

	if (strncmp(content_type, "text/plain", strlen("text/plain"))) { /* No text/plain attachment */
		transmit_response(p, "415 Unsupported Media Type", req); /* Good enough, or? */
		if (!p->owner)
			sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
		return;
	}

	if (get_msg_text(buf, sizeof(buf), req)) {
		ast_log(LOG_WARNING, "Unable to retrieve text from %s\n", p->callid);
		transmit_response(p, "500 Internal Server Error", req);
		if (!p->owner) {
			sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
		}
		return;
	}

	/* Strip trailing line feeds from message body. (get_msg_text may add
	 * a trailing linefeed and we don't need any at the end) */
	bufp = buf + strlen(buf);
	while (--bufp >= buf && *bufp == '\n') {
		*bufp = '\0';
	}

	if (p->owner) {
		if (sip_debug_test_pvt(p))
			ast_verbose("SIP Text message received: '%s'\n", buf);
		memset(&f, 0, sizeof(f));
		f.frametype = AST_FRAME_TEXT;
		f.subclass.integer = 0;
		f.offset = 0;
		f.data.ptr = buf;
		f.datalen = strlen(buf) + 1;
		ast_queue_frame(p->owner, &f);
		transmit_response(p, "202 Accepted", req); /* We respond 202 accepted, since we relay the message */
		return;
	}

	/* Message outside of a call, we do not support that */
	ast_log(LOG_WARNING, "Received message to %s from %s, dropped it...\n  Content-Type:%s\n  Message: %s\n", get_header(req, "To"), get_header(req, "From"), content_type, buf);
	transmit_response(p, "405 Method Not Allowed", req);
	sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
	return;
}

/*! \brief  CLI Command to show calls within limits set by call_limit */
static char *sip_show_inuse(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
#define FORMAT "%-25.25s %-15.15s %-15.15s \n"
#define FORMAT2 "%-25.25s %-15.15s %-15.15s \n"
	char ilimits[40];
	char iused[40];
	int showall = FALSE;
	struct ao2_iterator i;
	struct sip_peer *peer;
	
	switch (cmd) {
	case CLI_INIT:
		e->command = "sip show inuse";
		e->usage =
			"Usage: sip show inuse [all]\n"
			"       List all SIP devices usage counters and limits.\n"
			"       Add option \"all\" to show all devices, not only those with a limit.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc < 3)
		return CLI_SHOWUSAGE;

	if (a->argc == 4 && !strcmp(a->argv[3], "all"))
		showall = TRUE;
	
	ast_cli(a->fd, FORMAT, "* Peer name", "In use", "Limit");

	i = ao2_iterator_init(peers, 0);
	while ((peer = ao2_t_iterator_next(&i, "iterate thru peer table"))) {
		ao2_lock(peer);
		if (peer->call_limit)
			snprintf(ilimits, sizeof(ilimits), "%d", peer->call_limit);
		else
			ast_copy_string(ilimits, "N/A", sizeof(ilimits));
		snprintf(iused, sizeof(iused), "%d/%d/%d", peer->inUse, peer->inRinging, peer->onHold);
		if (showall || peer->call_limit)
			ast_cli(a->fd, FORMAT2, peer->name, iused, ilimits);
		ao2_unlock(peer);
		unref_peer(peer, "toss iterator pointer");
	}
	ao2_iterator_destroy(&i);

	return CLI_SUCCESS;
#undef FORMAT
#undef FORMAT2
}


/*! \brief Convert transfer mode to text string */
static char *transfermode2str(enum transfermodes mode)
{
	if (mode == TRANSFER_OPENFORALL)
		return "open";
	else if (mode == TRANSFER_CLOSED)
		return "closed";
	return "strict";
}

/*! \brief  Report Peer status in character string
 *  \return 0 if peer is unreachable, 1 if peer is online, -1 if unmonitored
 */


/* Session-Timer Modes */
static const struct _map_x_s stmodes[] = {
        { SESSION_TIMER_MODE_ACCEPT,    "Accept"},
        { SESSION_TIMER_MODE_ORIGINATE, "Originate"},
        { SESSION_TIMER_MODE_REFUSE,    "Refuse"},
        { -1,                           NULL},
};

static const char *stmode2str(enum st_mode m)
{
	return map_x_s(stmodes, m, "Unknown");
}

static enum st_mode str2stmode(const char *s)
{
	return map_s_x(stmodes, s, -1);
}

/* Session-Timer Refreshers */
static const struct _map_x_s strefreshers[] = {
        { SESSION_TIMER_REFRESHER_AUTO,     "auto"},
        { SESSION_TIMER_REFRESHER_UAC,      "uac"},
        { SESSION_TIMER_REFRESHER_UAS,      "uas"},
        { -1,                               NULL},
};

static const char *strefresher2str(enum st_refresher r)
{
	return map_x_s(strefreshers, r, "Unknown");
}

static enum st_refresher str2strefresher(const char *s)
{
	return map_s_x(strefreshers, s, -1);
}


static int peer_status(struct sip_peer *peer, char *status, int statuslen)
{
	int res = 0;
	if (peer->maxms) {
		if (peer->lastms < 0) {
			ast_copy_string(status, "UNREACHABLE", statuslen);
		} else if (peer->lastms > peer->maxms) {
			snprintf(status, statuslen, "LAGGED (%d ms)", peer->lastms);
			res = 1;
		} else if (peer->lastms) {
			snprintf(status, statuslen, "OK (%d ms)", peer->lastms);
			res = 1;
		} else {
			ast_copy_string(status, "UNKNOWN", statuslen);
		}
	} else {
		ast_copy_string(status, "Unmonitored", statuslen);
		/* Checking if port is 0 */
		res = -1;
	}
	return res;
}

/*! \brief  Show active TCP connections */
static char *sip_show_tcp(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	struct sip_threadinfo *th;
	struct ao2_iterator i;

#define FORMAT2 "%-47.47s %9.9s %6.6s\n"
#define FORMAT  "%-47.47s %-9.9s %-6.6s\n"

	switch (cmd) {
	case CLI_INIT:
		e->command = "sip show tcp";
		e->usage =
			"Usage: sip show tcp\n"
			"       Lists all active TCP/TLS sessions.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc != 3)
		return CLI_SHOWUSAGE;

	ast_cli(a->fd, FORMAT2, "Address", "Transport", "Type");

	i = ao2_iterator_init(threadt, 0);
	while ((th = ao2_t_iterator_next(&i, "iterate through tcp threads for 'sip show tcp'"))) {
		ast_cli(a->fd, FORMAT,
			ast_sockaddr_stringify(&th->tcptls_session->remote_address),
			get_transport(th->type),
			(th->tcptls_session->client ? "Client" : "Server"));
		ao2_t_ref(th, -1, "decrement ref from iterator");
	}
	ao2_iterator_destroy(&i);

	return CLI_SUCCESS;
#undef FORMAT
#undef FORMAT2
}

/*! \brief  CLI Command 'SIP Show Users' */
static char *sip_show_users(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	regex_t regexbuf;
	int havepattern = FALSE;
	struct ao2_iterator user_iter;
	struct sip_peer *user;

#define FORMAT  "%-25.25s  %-15.15s  %-15.15s  %-15.15s  %-5.5s%-10.10s\n"

	switch (cmd) {
	case CLI_INIT:
		e->command = "sip show users";
		e->usage =
			"Usage: sip show users [like <pattern>]\n"
			"       Lists all known SIP users.\n"
			"       Optional regular expression pattern is used to filter the user list.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	switch (a->argc) {
	case 5:
		if (!strcasecmp(a->argv[3], "like")) {
			if (regcomp(&regexbuf, a->argv[4], REG_EXTENDED | REG_NOSUB))
				return CLI_SHOWUSAGE;
			havepattern = TRUE;
		} else
			return CLI_SHOWUSAGE;
	case 3:
		break;
	default:
		return CLI_SHOWUSAGE;
	}

	ast_cli(a->fd, FORMAT, "Username", "Secret", "Accountcode", "Def.Context", "ACL", "ForcerPort");

	user_iter = ao2_iterator_init(peers, 0);
	while ((user = ao2_t_iterator_next(&user_iter, "iterate thru peers table"))) {
		ao2_lock(user);
		if (!(user->type & SIP_TYPE_USER)) {
			ao2_unlock(user);
			unref_peer(user, "sip show users");
			continue;
		}

		if (havepattern && regexec(&regexbuf, user->name, 0, NULL, 0)) {
			ao2_unlock(user);
			unref_peer(user, "sip show users");
			continue;
		}

		ast_cli(a->fd, FORMAT, user->name,
			user->secret,
			user->accountcode,
			user->context,
			AST_CLI_YESNO(user->ha != NULL),
			AST_CLI_YESNO(ast_test_flag(&user->flags[0], SIP_NAT_FORCE_RPORT)));
		ao2_unlock(user);
		unref_peer(user, "sip show users");
	}
	ao2_iterator_destroy(&user_iter);

	if (havepattern)
		regfree(&regexbuf);

	return CLI_SUCCESS;
#undef FORMAT
}

/*! \brief Show SIP registrations in the manager API */
static int manager_show_registry(struct mansession *s, const struct message *m)
{
	const char *id = astman_get_header(m, "ActionID");
	char idtext[256] = "";
	int total = 0;

	if (!ast_strlen_zero(id))
		snprintf(idtext, sizeof(idtext), "ActionID: %s\r\n", id);

	astman_send_listack(s, m, "Registrations will follow", "start");

	ASTOBJ_CONTAINER_TRAVERSE(&regl, 1, do {
		ASTOBJ_RDLOCK(iterator);
		astman_append(s,
			"Event: RegistryEntry\r\n"
			"%s"
			"Host: %s\r\n"
			"Port: %d\r\n"
			"Username: %s\r\n"
			"Domain: %s\r\n"
			"DomainPort: %d\r\n"
			"Refresh: %d\r\n"
			"State: %s\r\n"
			"RegistrationTime: %ld\r\n"
			"\r\n",
			idtext,
			iterator->hostname,
			iterator->portno ? iterator->portno : STANDARD_SIP_PORT,
			iterator->username,
			S_OR(iterator->regdomain,iterator->hostname),
			iterator->regdomainport ? iterator->regdomainport : STANDARD_SIP_PORT,
			iterator->refresh,
			regstate2str(iterator->regstate),
			(long) iterator->regtime.tv_sec);
		ASTOBJ_UNLOCK(iterator);
		total++;
	} while(0));

	astman_append(s,
		"Event: RegistrationsComplete\r\n"
		"EventList: Complete\r\n"
		"ListItems: %d\r\n"
		"%s"
		"\r\n", total, idtext);
	
	return 0;
}

/*! \brief  Show SIP peers in the manager API */
/*    Inspired from chan_iax2 */
static int manager_sip_show_peers(struct mansession *s, const struct message *m)
{
	const char *id = astman_get_header(m, "ActionID");
	const char *a[] = {"sip", "show", "peers"};
	char idtext[256] = "";
	int total = 0;

	if (!ast_strlen_zero(id))
		snprintf(idtext, sizeof(idtext), "ActionID: %s\r\n", id);

	astman_send_listack(s, m, "Peer status list will follow", "start");
	/* List the peers in separate manager events */
	_sip_show_peers(-1, &total, s, m, 3, a);
	/* Send final confirmation */
	astman_append(s,
	"Event: PeerlistComplete\r\n"
	"EventList: Complete\r\n"
	"ListItems: %d\r\n"
	"%s"
	"\r\n", total, idtext);
	return 0;
}

/*! \brief  CLI Show Peers command */
static char *sip_show_peers(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	switch (cmd) {
	case CLI_INIT:
		e->command = "sip show peers";
		e->usage =
			"Usage: sip show peers [like <pattern>]\n"
			"       Lists all known SIP peers.\n"
			"       Optional regular expression pattern is used to filter the peer list.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	return _sip_show_peers(a->fd, NULL, NULL, NULL, a->argc, (const char **) a->argv);
}

int peercomparefunc(const void *a, const void *b);

int peercomparefunc(const void *a, const void *b)
{
	struct sip_peer **ap = (struct sip_peer **)a;
	struct sip_peer **bp = (struct sip_peer **)b;
	return strcmp((*ap)->name, (*bp)->name);
}


/*! \brief Execute sip show peers command */
static char *_sip_show_peers(int fd, int *total, struct mansession *s, const struct message *m, int argc, const char *argv[])
{
	regex_t regexbuf;
	int havepattern = FALSE;
	struct sip_peer *peer;
	struct ao2_iterator* it_peers;

/* the last argument is left-aligned, so we don't need a size anyways */
#define FORMAT2 "%-25.25s  %-39.39s %-3.3s %-10.10s %-3.3s %-8s %-10s %s\n"

	char name[256];
	int total_peers = 0;
	int peers_mon_online = 0;
	int peers_mon_offline = 0;
	int peers_unmon_offline = 0;
	int peers_unmon_online = 0;
	const char *id;
	char idtext[256] = "";
	int realtimepeers;
	struct sip_peer **peerarray;
	int k;

	realtimepeers = ast_check_realtime("sippeers");

	if (s) {	/* Manager - get ActionID */
		id = astman_get_header(m, "ActionID");
		if (!ast_strlen_zero(id))
			snprintf(idtext, sizeof(idtext), "ActionID: %s\r\n", id);
	}

	switch (argc) {
	case 5:
		if (!strcasecmp(argv[3], "like")) {
			if (regcomp(&regexbuf, argv[4], REG_EXTENDED | REG_NOSUB))
				return CLI_SHOWUSAGE;
			havepattern = TRUE;
		} else
			return CLI_SHOWUSAGE;
	case 3:
		break;
	default:
		return CLI_SHOWUSAGE;
	}

	if (!s) {
		/* Normal list */
		ast_cli(fd, FORMAT2, "Name/username", "Host", "Dyn", "Forcerport", "ACL", "Port", "Status", (realtimepeers ? "Realtime" : ""));
	}

	ao2_lock(peers);
	if (!(it_peers = ao2_callback(peers, OBJ_MULTIPLE, NULL, NULL))) {
		ast_log(AST_LOG_ERROR, "Unable to create iterator for peers container for sip show peers\n");
		ao2_unlock(peers);
		return CLI_FAILURE;
	}
	if (!(peerarray = ast_calloc(sizeof(struct sip_peer *), ao2_container_count(peers)))) {
		ast_log(AST_LOG_ERROR, "Unable to allocate peer array for sip show peers\n");
		ao2_iterator_destroy(it_peers);
		ao2_unlock(peers);
		return CLI_FAILURE;
	}
	ao2_unlock(peers);

	while ((peer = ao2_t_iterator_next(it_peers, "iterate thru peers table"))) {
		ao2_lock(peer);

		if (!(peer->type & SIP_TYPE_PEER)) {
			ao2_unlock(peer);
			unref_peer(peer, "unref peer because it's actually a user");
			continue;
		}

		if (havepattern && regexec(&regexbuf, peer->name, 0, NULL, 0)) {
			ao2_unlock(peer);
			unref_peer(peer, "toss iterator peer ptr before continue");
			continue;
		}

		peerarray[total_peers++] = peer;
		ao2_unlock(peer);
	}
	ao2_iterator_destroy(it_peers);

	qsort(peerarray, total_peers, sizeof(struct sip_peer *), peercomparefunc);

	for(k=0; k < total_peers; k++) {
		char status[20] = "";
		char srch[2000];
		char pstatus;

		/*
		 * tmp_port and tmp_host store copies of ast_sockaddr_stringify strings since the
		 * string pointers for that function aren't valid between subsequent calls to
		 * ast_sockaddr_stringify functions
		 */
		char *tmp_port;
		char *tmp_host;

		peer = peerarray[k];

		tmp_port = ast_sockaddr_isnull(&peer->addr) ?
			"0" : ast_strdupa(ast_sockaddr_stringify_port(&peer->addr));

		tmp_host = ast_sockaddr_isnull(&peer->addr) ?
			"(Unspecified)" : ast_strdupa(ast_sockaddr_stringify_addr(&peer->addr));

		ao2_lock(peer);
		if (havepattern && regexec(&regexbuf, peer->name, 0, NULL, 0)) {
			ao2_unlock(peer);
			peer = peerarray[k] = unref_peer(peer, "toss iterator peer ptr before continue");
			continue;
		}

		if (!ast_strlen_zero(peer->username) && !s)
			snprintf(name, sizeof(name), "%s/%s", peer->name, peer->username);
		else
			ast_copy_string(name, peer->name, sizeof(name));

		pstatus = peer_status(peer, status, sizeof(status));
		if (pstatus == 1)
			peers_mon_online++;
		else if (pstatus == 0)
			peers_mon_offline++;
		else {
			if (ast_sockaddr_isnull(&peer->addr) ||
			    !ast_sockaddr_port(&peer->addr)) {
				peers_unmon_offline++;
			} else {
				peers_unmon_online++;
			}
		}

		snprintf(srch, sizeof(srch), FORMAT2, name,
			tmp_host,
			peer->host_dynamic ? " D " : "   ",	/* Dynamic or not? */
			ast_test_flag(&peer->flags[0], SIP_NAT_FORCE_RPORT) ? " N " : "   ",	/* NAT=yes? */
			peer->ha ? " A " : "   ",	/* permit/deny */
			tmp_port, status,
			realtimepeers ? (peer->is_realtime ? "Cached RT":"") : "");

		if (!s)  {/* Normal CLI list */
			ast_cli(fd, FORMAT2, name,
			tmp_host,
			peer->host_dynamic ? " D " : "   ",	/* Dynamic or not? */
			ast_test_flag(&peer->flags[0], SIP_NAT_FORCE_RPORT) ? " N " : "   ",	/* NAT=yes? */
			peer->ha ? " A " : "   ",       /* permit/deny */
			tmp_port, status,
			realtimepeers ? (peer->is_realtime ? "Cached RT":"") : "");
		} else {	/* Manager format */
			/* The names here need to be the same as other channels */
			astman_append(s,
			"Event: PeerEntry\r\n%s"
			"Channeltype: SIP\r\n"
			"ObjectName: %s\r\n"
			"ChanObjectType: peer\r\n"	/* "peer" or "user" */
			"IPaddress: %s\r\n"
			"IPport: %s\r\n"
			"Dynamic: %s\r\n"
			"Forcerport: %s\r\n"
			"VideoSupport: %s\r\n"
			"TextSupport: %s\r\n"
			"ACL: %s\r\n"
			"Status: %s\r\n"
			"RealtimeDevice: %s\r\n\r\n",
			idtext,
			peer->name,
			ast_sockaddr_isnull(&peer->addr) ? "-none-" : tmp_host,
			ast_sockaddr_isnull(&peer->addr) ? "0" : tmp_port,
			peer->host_dynamic ? "yes" : "no",	/* Dynamic or not? */
			ast_test_flag(&peer->flags[0], SIP_NAT_FORCE_RPORT) ? "yes" : "no",	/* NAT=yes? */
			ast_test_flag(&peer->flags[1], SIP_PAGE2_VIDEOSUPPORT) ? "yes" : "no",	/* VIDEOSUPPORT=yes? */
			ast_test_flag(&peer->flags[1], SIP_PAGE2_TEXTSUPPORT) ? "yes" : "no",	/* TEXTSUPPORT=yes? */
			peer->ha ? "yes" : "no",       /* permit/deny */
			status,
			realtimepeers ? (peer->is_realtime ? "yes":"no") : "no");
		}
		ao2_unlock(peer);
		peer = peerarray[k] = unref_peer(peer, "toss iterator peer ptr");
	}

	if (!s)
		ast_cli(fd, "%d sip peers [Monitored: %d online, %d offline Unmonitored: %d online, %d offline]\n",
		        total_peers, peers_mon_online, peers_mon_offline, peers_unmon_online, peers_unmon_offline);

	if (havepattern)
		regfree(&regexbuf);

	if (total)
		*total = total_peers;

	ast_free(peerarray);

	return CLI_SUCCESS;
#undef FORMAT2
}

static int peer_dump_func(void *userobj, void *arg, int flags)
{
	struct sip_peer *peer = userobj;
	int refc = ao2_t_ref(userobj, 0, "");
	struct ast_cli_args *a = (struct ast_cli_args *) arg;
	
	ast_cli(a->fd, "name: %s\ntype: peer\nobjflags: %d\nrefcount: %d\n\n",
		peer->name, 0, refc);
	return 0;
}

static int dialog_dump_func(void *userobj, void *arg, int flags)
{
	struct sip_pvt *pvt = userobj;
	int refc = ao2_t_ref(userobj, 0, "");
	struct ast_cli_args *a = (struct ast_cli_args *) arg;
	
	ast_cli(a->fd, "name: %s\ntype: dialog\nobjflags: %d\nrefcount: %d\n\n",
		pvt->callid, 0, refc);
	return 0;
}


/*! \brief List all allocated SIP Objects (realtime or static) */
static char *sip_show_objects(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	char tmp[256];
	
	switch (cmd) {
	case CLI_INIT:
		e->command = "sip show objects";
		e->usage =
			"Usage: sip show objects\n"
			"       Lists status of known SIP objects\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}	

	if (a->argc != 3)
		return CLI_SHOWUSAGE;
	ast_cli(a->fd, "-= Peer objects: %d static, %d realtime, %d autocreate =-\n\n", speerobjs, rpeerobjs, apeerobjs);
	ao2_t_callback(peers, OBJ_NODATA, peer_dump_func, a, "initiate ao2_callback to dump peers");
	ast_cli(a->fd, "-= Peer objects by IP =-\n\n"); 
	ao2_t_callback(peers_by_ip, OBJ_NODATA, peer_dump_func, a, "initiate ao2_callback to dump peers_by_ip");
	ast_cli(a->fd, "-= Registry objects: %d =-\n\n", regobjs);
	ASTOBJ_CONTAINER_DUMP(a->fd, tmp, sizeof(tmp), &regl);
	ast_cli(a->fd, "-= Dialog objects:\n\n");
	ao2_t_callback(dialogs, OBJ_NODATA, dialog_dump_func, a, "initiate ao2_callback to dump dialogs");
	return CLI_SUCCESS;
}
/*! \brief Print call group and pickup group */
static void print_group(int fd, ast_group_t group, int crlf)
{
	char buf[256];
	ast_cli(fd, crlf ? "%s\r\n" : "%s\n", ast_print_group(buf, sizeof(buf), group) );
}

/*! \brief mapping between dtmf flags and strings */
static const struct _map_x_s dtmfstr[] = {
	{ SIP_DTMF_RFC2833,     "rfc2833" },
	{ SIP_DTMF_INFO,        "info" },
	{ SIP_DTMF_SHORTINFO,   "shortinfo" },
	{ SIP_DTMF_INBAND,      "inband" },
	{ SIP_DTMF_AUTO,        "auto" },
	{ -1,                   NULL }, /* terminator */
};

/*! \brief Convert DTMF mode to printable string */
static const char *dtmfmode2str(int mode)
{
	return map_x_s(dtmfstr, mode, "<error>");
}

/*! \brief maps a string to dtmfmode, returns -1 on error */
static int str2dtmfmode(const char *str)
{
	return map_s_x(dtmfstr, str, -1);
}

static const struct _map_x_s insecurestr[] = {
	{ SIP_INSECURE_PORT,    "port" },
	{ SIP_INSECURE_INVITE,  "invite" },
	{ SIP_INSECURE_PORT | SIP_INSECURE_INVITE, "port,invite" },
	{ 0,                    "no" },
	{ -1,                   NULL }, /* terminator */
};

/*! \brief Convert Insecure setting to printable string */
static const char *insecure2str(int mode)
{
	return map_x_s(insecurestr, mode, "<error>");
}

static const struct _map_x_s allowoverlapstr[] = {
	{ SIP_PAGE2_ALLOWOVERLAP_YES,   "Yes" },
	{ SIP_PAGE2_ALLOWOVERLAP_DTMF,  "DTMF" },
	{ SIP_PAGE2_ALLOWOVERLAP_NO,    "No" },
	{ -1,                           NULL }, /* terminator */
};

/*! \brief Convert AllowOverlap setting to printable string */
static const char *allowoverlap2str(int mode)
{
	return map_x_s(allowoverlapstr, mode, "<error>");
}

/*! \brief Destroy disused contexts between reloads
	Only used in reload_config so the code for regcontext doesn't get ugly
*/
static void cleanup_stale_contexts(char *new, char *old)
{
	char *oldcontext, *newcontext, *stalecontext, *stringp, newlist[AST_MAX_CONTEXT];

	while ((oldcontext = strsep(&old, "&"))) {
		stalecontext = '\0';
		ast_copy_string(newlist, new, sizeof(newlist));
		stringp = newlist;
		while ((newcontext = strsep(&stringp, "&"))) {
			if (!strcmp(newcontext, oldcontext)) {
				/* This is not the context you're looking for */
				stalecontext = '\0';
				break;
			} else if (strcmp(newcontext, oldcontext)) {
				stalecontext = oldcontext;
			}
			
		}
		if (stalecontext)
			ast_context_destroy(ast_context_find(stalecontext), "SIP");
	}
}

/*!
 * \brief Match dialogs that need to be destroyed
 *
 * \details This is used with ao2_callback to unlink/delete all dialogs that
 * are marked needdestroy.
 *
 * \todo Re-work this to improve efficiency.  Currently, this function is called
 * on _every_ dialog after processing _every_ incoming SIP/UDP packet, or
 * potentially even more often when the scheduler has entries to run.
 */
static int dialog_needdestroy(void *dialogobj, void *arg, int flags)
{
	struct sip_pvt *dialog = dialogobj;
	time_t *t = arg;

	if (sip_pvt_trylock(dialog)) {
		/* Don't block the monitor thread.  This function is called often enough
		 * that we can wait for the next time around. */
		return 0;
	}

	/* Check RTP timeouts and kill calls if we have a timeout set and do not get RTP */
	check_rtp_timeout(dialog, *t);

	/* We absolutely cannot destroy the rtp struct while a bridge is active or we WILL crash */
	if (dialog->rtp && ast_rtp_instance_get_bridged(dialog->rtp)) {
		ast_debug(2, "Bridge still active.  Delaying destroy of SIP dialog '%s' Method: %s\n", dialog->callid, sip_methods[dialog->method].text);
		sip_pvt_unlock(dialog);
		return 0;
	}

	if (dialog->vrtp && ast_rtp_instance_get_bridged(dialog->vrtp)) {
		ast_debug(2, "Bridge still active.  Delaying destroy of SIP dialog '%s' Method: %s\n", dialog->callid, sip_methods[dialog->method].text);
		sip_pvt_unlock(dialog);
		return 0;
	}

	/* If we have sessions that needs to be destroyed, do it now */
	/* Check if we have outstanding requests not responsed to or an active call
	   - if that's the case, wait with destruction */
	if (dialog->needdestroy && !dialog->packets && !dialog->owner) {
		/* We absolutely cannot destroy the rtp struct while a bridge is active or we WILL crash */
		if (dialog->rtp && ast_rtp_instance_get_bridged(dialog->rtp)) {
			ast_debug(2, "Bridge still active.  Delaying destruction of SIP dialog '%s' Method: %s\n", dialog->callid, sip_methods[dialog->method].text);
			sip_pvt_unlock(dialog);
			return 0;
		}
		
		if (dialog->vrtp && ast_rtp_instance_get_bridged(dialog->vrtp)) {
			ast_debug(2, "Bridge still active.  Delaying destroy of SIP dialog '%s' Method: %s\n", dialog->callid, sip_methods[dialog->method].text);
			sip_pvt_unlock(dialog);
			return 0;
		}

		sip_pvt_unlock(dialog);

		/* This dialog needs to be destroyed. */
		ao2_t_link(dialogs_to_destroy, dialog, "Link dialog for destruction");
		return 0;
	}

	sip_pvt_unlock(dialog);

	return 0;
}

/*!
 * \internal
 * \brief ao2_callback to unlink the specified dialog object.
 *
 * \param obj Ptr to dialog to unlink.
 * \param arg Don't care.
 * \param flags Don't care.
 *
 * \retval CMP_MATCH
 */
static int dialog_unlink_callback(void *obj, void *arg, int flags)
{
	struct sip_pvt *dialog = obj;

	dialog_unlink_all(dialog);

	return CMP_MATCH;
}

/*! \brief Remove temporary realtime objects from memory (CLI) */
/*! \todo XXXX Propably needs an overhaul after removal of the devices */
static char *sip_prune_realtime(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	struct sip_peer *peer, *pi;
	int prunepeer = FALSE;
	int multi = FALSE;
	const char *name = NULL;
	regex_t regexbuf;
	int havepattern = 0;
	struct ao2_iterator i;
	static const char * const choices[] = { "all", "like", NULL };
	char *cmplt;
	
	if (cmd == CLI_INIT) {
		e->command = "sip prune realtime [peer|all]";
		e->usage =
			"Usage: sip prune realtime [peer [<name>|all|like <pattern>]|all]\n"
			"       Prunes object(s) from the cache.\n"
			"       Optional regular expression pattern is used to filter the objects.\n";
		return NULL;
	} else if (cmd == CLI_GENERATE) {
		if (a->pos == 4 && !strcasecmp(a->argv[3], "peer")) {
			cmplt = ast_cli_complete(a->word, choices, a->n);
			if (!cmplt)
				cmplt = complete_sip_peer(a->word, a->n - sizeof(choices), SIP_PAGE2_RTCACHEFRIENDS);
			return cmplt;
		}
		if (a->pos == 5 && !strcasecmp(a->argv[4], "like"))
			return complete_sip_peer(a->word, a->n, SIP_PAGE2_RTCACHEFRIENDS);
		return NULL;
	}
	switch (a->argc) {
	case 4:
		name = a->argv[3];
		/* we accept a name in position 3, but keywords are not good. */
		if (!strcasecmp(name, "peer") || !strcasecmp(name, "like"))
			return CLI_SHOWUSAGE;
		prunepeer = TRUE;
		if (!strcasecmp(name, "all")) {
			multi = TRUE;
			name = NULL;
		}
		/* else a single name, already set */
		break;
	case 5:
		/* sip prune realtime {peer|like} name */
		name = a->argv[4];
		if (!strcasecmp(a->argv[3], "peer"))
			prunepeer = TRUE;
		else if (!strcasecmp(a->argv[3], "like")) {
			prunepeer = TRUE;
			multi = TRUE;
		} else
			return CLI_SHOWUSAGE;
		if (!strcasecmp(name, "like"))
			return CLI_SHOWUSAGE;
		if (!multi && !strcasecmp(name, "all")) {
			multi = TRUE;
			name = NULL;
		}
		break;
	case 6:
		name = a->argv[5];
		multi = TRUE;
		/* sip prune realtime {peer} like name */
		if (strcasecmp(a->argv[4], "like"))
			return CLI_SHOWUSAGE;
		if (!strcasecmp(a->argv[3], "peer")) {
			prunepeer = TRUE;
		} else
			return CLI_SHOWUSAGE;
		break;
	default:
		return CLI_SHOWUSAGE;
	}

	if (multi && name) {
		if (regcomp(&regexbuf, name, REG_EXTENDED | REG_NOSUB)) {
			return CLI_SHOWUSAGE;
		}
		havepattern = 1;
	}

	if (multi) {
		if (prunepeer) {
			int pruned = 0;
			
			i = ao2_iterator_init(peers, 0);
			while ((pi = ao2_t_iterator_next(&i, "iterate thru peers table"))) {
				ao2_lock(pi);
				if (name && regexec(&regexbuf, pi->name, 0, NULL, 0)) {
					ao2_unlock(pi);
					unref_peer(pi, "toss iterator peer ptr before continue");
					continue;
				};
				if (ast_test_flag(&pi->flags[1], SIP_PAGE2_RTCACHEFRIENDS)) {
					pi->the_mark = 1;
					pruned++;
				}
				ao2_unlock(pi);
				unref_peer(pi, "toss iterator peer ptr");
			}
			ao2_iterator_destroy(&i);
			if (pruned) {
				unlink_marked_peers_from_tables();
				ast_cli(a->fd, "%d peers pruned.\n", pruned);
			} else
				ast_cli(a->fd, "No peers found to prune.\n");
		}
	} else {
		if (prunepeer) {
			struct sip_peer tmp;
			ast_copy_string(tmp.name, name, sizeof(tmp.name));
			if ((peer = ao2_t_find(peers, &tmp, OBJ_POINTER | OBJ_UNLINK, "finding to unlink from peers"))) {
				if (!ast_sockaddr_isnull(&peer->addr)) {
					ao2_t_unlink(peers_by_ip, peer, "unlinking peer from peers_by_ip also");
				}
				if (!ast_test_flag(&peer->flags[1], SIP_PAGE2_RTCACHEFRIENDS)) {
					ast_cli(a->fd, "Peer '%s' is not a Realtime peer, cannot be pruned.\n", name);
					/* put it back! */
					ao2_t_link(peers, peer, "link peer into peer table");
					if (!ast_sockaddr_isnull(&peer->addr)) {
						ao2_t_link(peers_by_ip, peer, "link peer into peers_by_ip table");
					}
				} else
					ast_cli(a->fd, "Peer '%s' pruned.\n", name);
				unref_peer(peer, "sip_prune_realtime: unref_peer: tossing temp peer ptr");
			} else
				ast_cli(a->fd, "Peer '%s' not found.\n", name);
		}
	}

	if (havepattern) {
		regfree(&regexbuf);
	}

	return CLI_SUCCESS;
}

/*! \brief Print codec list from preference to CLI/manager */
static void print_codec_to_cli(int fd, struct ast_codec_pref *pref)
{
	int x;
	format_t codec;

	for(x = 0; x < 64 ; x++) {
		codec = ast_codec_pref_index(pref, x);
		if (!codec)
			break;
		ast_cli(fd, "%s", ast_getformatname(codec));
		ast_cli(fd, ":%d", pref->framing[x]);
		if (x < 31 && ast_codec_pref_index(pref, x + 1))
			ast_cli(fd, ",");
	}
	if (!x)
		ast_cli(fd, "none");
}

/*! \brief Print domain mode to cli */
static const char *domain_mode_to_text(const enum domain_mode mode)
{
	switch (mode) {
	case SIP_DOMAIN_AUTO:
		return "[Automatic]";
	case SIP_DOMAIN_CONFIG:
		return "[Configured]";
	}

	return "";
}

/*! \brief CLI command to list local domains */
static char *sip_show_domains(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	struct domain *d;
#define FORMAT "%-40.40s %-20.20s %-16.16s\n"

	switch (cmd) {
	case CLI_INIT:
		e->command = "sip show domains";
		e->usage =
			"Usage: sip show domains\n"
			"       Lists all configured SIP local domains.\n"
			"       Asterisk only responds to SIP messages to local domains.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (AST_LIST_EMPTY(&domain_list)) {
		ast_cli(a->fd, "SIP Domain support not enabled.\n\n");
		return CLI_SUCCESS;
	} else {
		ast_cli(a->fd, FORMAT, "Our local SIP domains:", "Context", "Set by");
		AST_LIST_LOCK(&domain_list);
		AST_LIST_TRAVERSE(&domain_list, d, list)
			ast_cli(a->fd, FORMAT, d->domain, S_OR(d->context, "(default)"),
				domain_mode_to_text(d->mode));
		AST_LIST_UNLOCK(&domain_list);
		ast_cli(a->fd, "\n");
		return CLI_SUCCESS;
	}
}
#undef FORMAT

/*! \brief Show SIP peers in the manager API  */
static int manager_sip_show_peer(struct mansession *s, const struct message *m)
{
	const char *a[4];
	const char *peer;

	peer = astman_get_header(m, "Peer");
	if (ast_strlen_zero(peer)) {
		astman_send_error(s, m, "Peer: <name> missing.");
		return 0;
	}
	a[0] = "sip";
	a[1] = "show";
	a[2] = "peer";
	a[3] = peer;

	_sip_show_peer(1, -1, s, m, 4, a);
	astman_append(s, "\r\n" );
	return 0;
}

/*! \brief Show one peer in detail */
static char *sip_show_peer(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	switch (cmd) {
	case CLI_INIT:
		e->command = "sip show peer";
		e->usage =
			"Usage: sip show peer <name> [load]\n"
			"       Shows all details on one SIP peer and the current status.\n"
			"       Option \"load\" forces lookup of peer in realtime storage.\n";
		return NULL;
	case CLI_GENERATE:
		return complete_sip_show_peer(a->line, a->word, a->pos, a->n);
	}
	return _sip_show_peer(0, a->fd, NULL, NULL, a->argc, (const char **) a->argv);
}

/*! \brief Send qualify message to peer from cli or manager. Mostly for debugging. */
static char *_sip_qualify_peer(int type, int fd, struct mansession *s, const struct message *m, int argc, const char *argv[])
{
	struct sip_peer *peer;
	int load_realtime;

	if (argc < 4)
		return CLI_SHOWUSAGE;

	load_realtime = (argc == 5 && !strcmp(argv[4], "load")) ? TRUE : FALSE;
	if ((peer = find_peer(argv[3], NULL, load_realtime, FINDPEERS, FALSE, 0))) {
		sip_poke_peer(peer, 1);
		unref_peer(peer, "qualify: done with peer");
	} else if (type == 0) {
		ast_cli(fd, "Peer '%s' not found\n", argv[3]);
	} else {
		astman_send_error(s, m, "Peer not found");
	}
	return CLI_SUCCESS;
}

/*! \brief Qualify SIP peers in the manager API  */
static int manager_sip_qualify_peer(struct mansession *s, const struct message *m)
{
	const char *a[4];
	const char *peer;

	peer = astman_get_header(m, "Peer");
	if (ast_strlen_zero(peer)) {
		astman_send_error(s, m, "Peer: <name> missing.");
		return 0;
	}
	a[0] = "sip";
	a[1] = "qualify";
	a[2] = "peer";
	a[3] = peer;

	_sip_qualify_peer(1, -1, s, m, 4, a);
	astman_append(s, "\r\n\r\n" );
	return 0;
}

/*! \brief Send an OPTIONS packet to a SIP peer */
static char *sip_qualify_peer(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	switch (cmd) {
	case CLI_INIT:
		e->command = "sip qualify peer";
		e->usage =
			"Usage: sip qualify peer <name> [load]\n"
			"       Requests a response from one SIP peer and the current status.\n"
			"       Option \"load\" forces lookup of peer in realtime storage.\n";
		return NULL;
	case CLI_GENERATE:
		return complete_sip_show_peer(a->line, a->word, a->pos, a->n);
	}
	return _sip_qualify_peer(0, a->fd, NULL, NULL, a->argc, (const char **) a->argv);
}

/*! \brief list peer mailboxes to CLI */
static void peer_mailboxes_to_str(struct ast_str **mailbox_str, struct sip_peer *peer)
{
	struct sip_mailbox *mailbox;

	AST_LIST_TRAVERSE(&peer->mailboxes, mailbox, entry) {
		ast_str_append(mailbox_str, 0, "%s%s%s%s",
			mailbox->mailbox,
			ast_strlen_zero(mailbox->context) ? "" : "@",
			S_OR(mailbox->context, ""),
			AST_LIST_NEXT(mailbox, entry) ? "," : "");
	}
}

static struct _map_x_s faxecmodes[] = {
	{ SIP_PAGE2_T38SUPPORT_UDPTL,			"None"},
	{ SIP_PAGE2_T38SUPPORT_UDPTL_FEC,		"FEC"},
	{ SIP_PAGE2_T38SUPPORT_UDPTL_REDUNDANCY,	"Redundancy"},
	{ -1,						NULL},
};

static const char *faxec2str(int faxec)
{
	return map_x_s(faxecmodes, faxec, "Unknown");
}

/*! \brief Show one peer in detail (main function) */
static char *_sip_show_peer(int type, int fd, struct mansession *s, const struct message *m, int argc, const char *argv[])
{
	char status[30] = "";
	char cbuf[256];
	struct sip_peer *peer;
	char codec_buf[512];
	struct ast_codec_pref *pref;
	struct ast_variable *v;
	int x = 0, load_realtime;
	format_t codec = 0;
	int realtimepeers;

	realtimepeers = ast_check_realtime("sippeers");

	if (argc < 4)
		return CLI_SHOWUSAGE;

	load_realtime = (argc == 5 && !strcmp(argv[4], "load")) ? TRUE : FALSE;
	peer = find_peer(argv[3], NULL, load_realtime, FINDPEERS, FALSE, 0);

	if (s) { 	/* Manager */
		if (peer) {
			const char *id = astman_get_header(m, "ActionID");

			astman_append(s, "Response: Success\r\n");
			if (!ast_strlen_zero(id))
				astman_append(s, "ActionID: %s\r\n", id);
		} else {
			snprintf (cbuf, sizeof(cbuf), "Peer %s not found.", argv[3]);
			astman_send_error(s, m, cbuf);
			return CLI_SUCCESS;
		}
	}
	if (peer && type==0 ) { /* Normal listing */
		struct ast_str *mailbox_str = ast_str_alloca(512);
		struct sip_auth_container *credentials;

		ao2_lock(peer);
		credentials = peer->auth;
		if (credentials) {
			ao2_t_ref(credentials, +1, "Ref peer auth for show");
		}
		ao2_unlock(peer);

		ast_cli(fd, "\n\n");
		ast_cli(fd, "  * Name       : %s\n", peer->name);
		if (realtimepeers) {	/* Realtime is enabled */
			ast_cli(fd, "  Realtime peer: %s\n", peer->is_realtime ? "Yes, cached" : "No");
		}
		ast_cli(fd, "  Secret       : %s\n", ast_strlen_zero(peer->secret)?"<Not set>":"<Set>");
		ast_cli(fd, "  MD5Secret    : %s\n", ast_strlen_zero(peer->md5secret)?"<Not set>":"<Set>");
		ast_cli(fd, "  Remote Secret: %s\n", ast_strlen_zero(peer->remotesecret)?"<Not set>":"<Set>");
		if (credentials) {
			struct sip_auth *auth;

			AST_LIST_TRAVERSE(&credentials->list, auth, node) {
				ast_cli(fd, "  Realm-auth   : Realm %-15.15s User %-10.20s %s\n",
					auth->realm,
					auth->username,
					!ast_strlen_zero(auth->secret)
						? "<Secret set>"
						: (!ast_strlen_zero(auth->md5secret)
							? "<MD5secret set>" : "<Not set>"));
			}
			ao2_t_ref(credentials, -1, "Unref peer auth for show");
		}
		ast_cli(fd, "  Context      : %s\n", peer->context);
		ast_cli(fd, "  Subscr.Cont. : %s\n", S_OR(peer->subscribecontext, "<Not set>") );
		ast_cli(fd, "  Language     : %s\n", peer->language);
		if (!ast_strlen_zero(peer->accountcode))
			ast_cli(fd, "  Accountcode  : %s\n", peer->accountcode);
		ast_cli(fd, "  AMA flags    : %s\n", ast_cdr_flags2str(peer->amaflags));
		ast_cli(fd, "  Transfer mode: %s\n", transfermode2str(peer->allowtransfer));
		ast_cli(fd, "  CallingPres  : %s\n", ast_describe_caller_presentation(peer->callingpres));
		if (!ast_strlen_zero(peer->fromuser))
			ast_cli(fd, "  FromUser     : %s\n", peer->fromuser);
		if (!ast_strlen_zero(peer->fromdomain))
			ast_cli(fd, "  FromDomain   : %s Port %d\n", peer->fromdomain, (peer->fromdomainport) ? peer->fromdomainport : STANDARD_SIP_PORT);
		ast_cli(fd, "  Callgroup    : ");
		print_group(fd, peer->callgroup, 0);
		ast_cli(fd, "  Pickupgroup  : ");
		print_group(fd, peer->pickupgroup, 0);
		peer_mailboxes_to_str(&mailbox_str, peer);
		ast_cli(fd, "  MOH Suggest  : %s\n", peer->mohsuggest);
		ast_cli(fd, "  Mailbox      : %s\n", mailbox_str->str);
		ast_cli(fd, "  VM Extension : %s\n", peer->vmexten);
		ast_cli(fd, "  LastMsgsSent : %d/%d\n", (peer->lastmsgssent & 0x7fff0000) >> 16, peer->lastmsgssent & 0xffff);
		ast_cli(fd, "  Call limit   : %d\n", peer->call_limit);
		ast_cli(fd, "  Max forwards : %d\n", peer->maxforwards);
		if (peer->busy_level)
			ast_cli(fd, "  Busy level   : %d\n", peer->busy_level);
		ast_cli(fd, "  Dynamic      : %s\n", AST_CLI_YESNO(peer->host_dynamic));
		ast_cli(fd, "  Callerid     : %s\n", ast_callerid_merge(cbuf, sizeof(cbuf), peer->cid_name, peer->cid_num, "<unspecified>"));
		ast_cli(fd, "  MaxCallBR    : %d kbps\n", peer->maxcallbitrate);
		ast_cli(fd, "  Expire       : %ld\n", ast_sched_when(sched, peer->expire));
		ast_cli(fd, "  Insecure     : %s\n", insecure2str(ast_test_flag(&peer->flags[0], SIP_INSECURE)));
		ast_cli(fd, "  Force rport  : %s\n", AST_CLI_YESNO(ast_test_flag(&peer->flags[0], SIP_NAT_FORCE_RPORT)));
		ast_cli(fd, "  ACL          : %s\n", AST_CLI_YESNO(peer->ha != NULL));
		ast_cli(fd, "  DirectMedACL : %s\n", AST_CLI_YESNO(peer->directmediaha != NULL));
		ast_cli(fd, "  T.38 support : %s\n", AST_CLI_YESNO(ast_test_flag(&peer->flags[1], SIP_PAGE2_T38SUPPORT)));
		ast_cli(fd, "  T.38 EC mode : %s\n", faxec2str(ast_test_flag(&peer->flags[1], SIP_PAGE2_T38SUPPORT)));
		ast_cli(fd, "  T.38 MaxDtgrm: %d\n", peer->t38_maxdatagram);
		ast_cli(fd, "  DirectMedia  : %s\n", AST_CLI_YESNO(ast_test_flag(&peer->flags[0], SIP_DIRECT_MEDIA)));
		ast_cli(fd, "  PromiscRedir : %s\n", AST_CLI_YESNO(ast_test_flag(&peer->flags[0], SIP_PROMISCREDIR)));
		ast_cli(fd, "  User=Phone   : %s\n", AST_CLI_YESNO(ast_test_flag(&peer->flags[0], SIP_USEREQPHONE)));
		ast_cli(fd, "  Video Support: %s\n", AST_CLI_YESNO(ast_test_flag(&peer->flags[1], SIP_PAGE2_VIDEOSUPPORT) || ast_test_flag(&peer->flags[1], SIP_PAGE2_VIDEOSUPPORT_ALWAYS)));
		ast_cli(fd, "  Text Support : %s\n", AST_CLI_YESNO(ast_test_flag(&peer->flags[1], SIP_PAGE2_TEXTSUPPORT)));
		ast_cli(fd, "  Ign SDP ver  : %s\n", AST_CLI_YESNO(ast_test_flag(&peer->flags[1], SIP_PAGE2_IGNORESDPVERSION)));
		ast_cli(fd, "  Trust RPID   : %s\n", AST_CLI_YESNO(ast_test_flag(&peer->flags[0], SIP_TRUSTRPID)));
		ast_cli(fd, "  Send RPID    : %s\n", AST_CLI_YESNO(ast_test_flag(&peer->flags[0], SIP_SENDRPID)));
		ast_cli(fd, "  Subscriptions: %s\n", AST_CLI_YESNO(ast_test_flag(&peer->flags[1], SIP_PAGE2_ALLOWSUBSCRIBE)));
		ast_cli(fd, "  Overlap dial : %s\n", allowoverlap2str(ast_test_flag(&peer->flags[1], SIP_PAGE2_ALLOWOVERLAP)));
		if (peer->outboundproxy)
			ast_cli(fd, "  Outb. proxy  : %s %s\n", ast_strlen_zero(peer->outboundproxy->name) ? "<not set>" : peer->outboundproxy->name,
							peer->outboundproxy->force ? "(forced)" : "");

		/* - is enumerated */
		ast_cli(fd, "  DTMFmode     : %s\n", dtmfmode2str(ast_test_flag(&peer->flags[0], SIP_DTMF)));
		ast_cli(fd, "  Timer T1     : %d\n", peer->timer_t1);
		ast_cli(fd, "  Timer B      : %d\n", peer->timer_b);
		ast_cli(fd, "  ToHost       : %s\n", peer->tohost);
		ast_cli(fd, "  Addr->IP     : %s\n", ast_sockaddr_stringify(&peer->addr));
		ast_cli(fd, "  Defaddr->IP  : %s\n", ast_sockaddr_stringify(&peer->defaddr));
		ast_cli(fd, "  Prim.Transp. : %s\n", get_transport(peer->socket.type));
		ast_cli(fd, "  Allowed.Trsp : %s\n", get_transport_list(peer->transports));
		if (!ast_strlen_zero(sip_cfg.regcontext))
			ast_cli(fd, "  Reg. exten   : %s\n", peer->regexten);
		ast_cli(fd, "  Def. Username: %s\n", peer->username);
		ast_cli(fd, "  SIP Options  : ");
		if (peer->sipoptions) {
			int lastoption = -1;
			for (x = 0 ; x < ARRAY_LEN(sip_options); x++) {
				if (sip_options[x].id != lastoption) {
					if (peer->sipoptions & sip_options[x].id)
						ast_cli(fd, "%s ", sip_options[x].text);
					lastoption = x;
				}
			}
		} else
			ast_cli(fd, "(none)");

		ast_cli(fd, "\n");
		ast_cli(fd, "  Codecs       : ");
		ast_getformatname_multiple(codec_buf, sizeof(codec_buf) -1, peer->capability);
		ast_cli(fd, "%s\n", codec_buf);
		ast_cli(fd, "  Codec Order  : (");
		print_codec_to_cli(fd, &peer->prefs);
		ast_cli(fd, ")\n");

		ast_cli(fd, "  Auto-Framing :  %s \n", AST_CLI_YESNO(peer->autoframing));
		ast_cli(fd, "  Status       : ");
		peer_status(peer, status, sizeof(status));
		ast_cli(fd, "%s\n", status);
		ast_cli(fd, "  Useragent    : %s\n", peer->useragent);
		ast_cli(fd, "  Reg. Contact : %s\n", peer->fullcontact);
		ast_cli(fd, "  Qualify Freq : %d ms\n", peer->qualifyfreq);
		if (peer->chanvars) {
			ast_cli(fd, "  Variables    :\n");
			for (v = peer->chanvars ; v ; v = v->next)
				ast_cli(fd, "                 %s = %s\n", v->name, v->value);
		}

		ast_cli(fd, "  Sess-Timers  : %s\n", stmode2str(peer->stimer.st_mode_oper));
		ast_cli(fd, "  Sess-Refresh : %s\n", strefresher2str(peer->stimer.st_ref));
		ast_cli(fd, "  Sess-Expires : %d secs\n", peer->stimer.st_max_se);
		ast_cli(fd, "  Min-Sess     : %d secs\n", peer->stimer.st_min_se);
		ast_cli(fd, "  RTP Engine   : %s\n", peer->engine);
		ast_cli(fd, "  Parkinglot   : %s\n", peer->parkinglot);
		ast_cli(fd, "  Use Reason   : %s\n", AST_CLI_YESNO(ast_test_flag(&peer->flags[1], SIP_PAGE2_Q850_REASON)));
		ast_cli(fd, "  Encryption   : %s\n", AST_CLI_YESNO(ast_test_flag(&peer->flags[1], SIP_PAGE2_USE_SRTP)));
		ast_cli(fd, "\n");
		peer = unref_peer(peer, "sip_show_peer: unref_peer: done with peer ptr");
	} else  if (peer && type == 1) { /* manager listing */
		char buffer[256];
		struct ast_str *mailbox_str = ast_str_alloca(512);
		astman_append(s, "Channeltype: SIP\r\n");
		astman_append(s, "ObjectName: %s\r\n", peer->name);
		astman_append(s, "ChanObjectType: peer\r\n");
		astman_append(s, "SecretExist: %s\r\n", ast_strlen_zero(peer->secret)?"N":"Y");
		astman_append(s, "RemoteSecretExist: %s\r\n", ast_strlen_zero(peer->remotesecret)?"N":"Y");
		astman_append(s, "MD5SecretExist: %s\r\n", ast_strlen_zero(peer->md5secret)?"N":"Y");
		astman_append(s, "Context: %s\r\n", peer->context);
		astman_append(s, "Language: %s\r\n", peer->language);
		if (!ast_strlen_zero(peer->accountcode))
			astman_append(s, "Accountcode: %s\r\n", peer->accountcode);
		astman_append(s, "AMAflags: %s\r\n", ast_cdr_flags2str(peer->amaflags));
		astman_append(s, "CID-CallingPres: %s\r\n", ast_describe_caller_presentation(peer->callingpres));
		if (!ast_strlen_zero(peer->fromuser))
			astman_append(s, "SIP-FromUser: %s\r\n", peer->fromuser);
		if (!ast_strlen_zero(peer->fromdomain))
			astman_append(s, "SIP-FromDomain: %s\r\nSip-FromDomain-Port: %d\r\n", peer->fromdomain, (peer->fromdomainport) ? peer->fromdomainport : STANDARD_SIP_PORT);
		astman_append(s, "Callgroup: ");
		astman_append(s, "%s\r\n", ast_print_group(buffer, sizeof(buffer), peer->callgroup));
		astman_append(s, "Pickupgroup: ");
		astman_append(s, "%s\r\n", ast_print_group(buffer, sizeof(buffer), peer->pickupgroup));
		astman_append(s, "MOHSuggest: %s\r\n", peer->mohsuggest);
		peer_mailboxes_to_str(&mailbox_str, peer);
		astman_append(s, "VoiceMailbox: %s\r\n", mailbox_str->str);
		astman_append(s, "TransferMode: %s\r\n", transfermode2str(peer->allowtransfer));
		astman_append(s, "LastMsgsSent: %d\r\n", peer->lastmsgssent);
		astman_append(s, "Maxforwards: %d\r\n", peer->maxforwards);
		astman_append(s, "Call-limit: %d\r\n", peer->call_limit);
		astman_append(s, "Busy-level: %d\r\n", peer->busy_level);
		astman_append(s, "MaxCallBR: %d kbps\r\n", peer->maxcallbitrate);
		astman_append(s, "Dynamic: %s\r\n", peer->host_dynamic?"Y":"N");
		astman_append(s, "Callerid: %s\r\n", ast_callerid_merge(cbuf, sizeof(cbuf), peer->cid_name, peer->cid_num, ""));
		astman_append(s, "RegExpire: %ld seconds\r\n", ast_sched_when(sched, peer->expire));
		astman_append(s, "SIP-AuthInsecure: %s\r\n", insecure2str(ast_test_flag(&peer->flags[0], SIP_INSECURE)));
		astman_append(s, "SIP-Forcerport: %s\r\n", (ast_test_flag(&peer->flags[0], SIP_NAT_FORCE_RPORT)?"Y":"N"));
		astman_append(s, "ACL: %s\r\n", (peer->ha?"Y":"N"));
		astman_append(s, "SIP-CanReinvite: %s\r\n", (ast_test_flag(&peer->flags[0], SIP_DIRECT_MEDIA)?"Y":"N"));
		astman_append(s, "SIP-DirectMedia: %s\r\n", (ast_test_flag(&peer->flags[0], SIP_DIRECT_MEDIA)?"Y":"N"));
		astman_append(s, "SIP-PromiscRedir: %s\r\n", (ast_test_flag(&peer->flags[0], SIP_PROMISCREDIR)?"Y":"N"));
		astman_append(s, "SIP-UserPhone: %s\r\n", (ast_test_flag(&peer->flags[0], SIP_USEREQPHONE)?"Y":"N"));
		astman_append(s, "SIP-VideoSupport: %s\r\n", (ast_test_flag(&peer->flags[1], SIP_PAGE2_VIDEOSUPPORT)?"Y":"N"));
		astman_append(s, "SIP-TextSupport: %s\r\n", (ast_test_flag(&peer->flags[1], SIP_PAGE2_TEXTSUPPORT)?"Y":"N"));
		astman_append(s, "SIP-T.38Support: %s\r\n", (ast_test_flag(&peer->flags[1], SIP_PAGE2_T38SUPPORT)?"Y":"N"));
		astman_append(s, "SIP-T.38EC: %s\r\n", faxec2str(ast_test_flag(&peer->flags[1], SIP_PAGE2_T38SUPPORT)));
		astman_append(s, "SIP-T.38MaxDtgrm: %d\r\n", peer->t38_maxdatagram);
		astman_append(s, "SIP-Sess-Timers: %s\r\n", stmode2str(peer->stimer.st_mode_oper));
		astman_append(s, "SIP-Sess-Refresh: %s\r\n", strefresher2str(peer->stimer.st_ref));
		astman_append(s, "SIP-Sess-Expires: %d\r\n", peer->stimer.st_max_se);
		astman_append(s, "SIP-Sess-Min: %d\r\n", peer->stimer.st_min_se);
		astman_append(s, "SIP-RTP-Engine: %s\r\n", peer->engine);
		astman_append(s, "SIP-Encryption: %s\r\n", ast_test_flag(&peer->flags[1], SIP_PAGE2_USE_SRTP) ? "Y" : "N");

		/* - is enumerated */
		astman_append(s, "SIP-DTMFmode: %s\r\n", dtmfmode2str(ast_test_flag(&peer->flags[0], SIP_DTMF)));
		astman_append(s, "ToHost: %s\r\n", peer->tohost);
		astman_append(s, "Address-IP: %s\r\nAddress-Port: %d\r\n", ast_sockaddr_stringify_addr(&peer->addr), ast_sockaddr_port(&peer->addr));
		astman_append(s, "Default-addr-IP: %s\r\nDefault-addr-port: %d\r\n", ast_sockaddr_stringify_addr(&peer->defaddr), ast_sockaddr_port(&peer->defaddr));
		astman_append(s, "Default-Username: %s\r\n", peer->username);
		if (!ast_strlen_zero(sip_cfg.regcontext))
			astman_append(s, "RegExtension: %s\r\n", peer->regexten);
		astman_append(s, "Codecs: ");
		ast_getformatname_multiple(codec_buf, sizeof(codec_buf) -1, peer->capability);
		astman_append(s, "%s\r\n", codec_buf);
		astman_append(s, "CodecOrder: ");
		pref = &peer->prefs;
		for(x = 0; x < 64 ; x++) {
			codec = ast_codec_pref_index(pref, x);
			if (!codec)
				break;
			astman_append(s, "%s", ast_getformatname(codec));
			if (x < 63 && ast_codec_pref_index(pref, x+1))
				astman_append(s, ",");
		}

		astman_append(s, "\r\n");
		astman_append(s, "Status: ");
		peer_status(peer, status, sizeof(status));
		astman_append(s, "%s\r\n", status);
		astman_append(s, "SIP-Useragent: %s\r\n", peer->useragent);
		astman_append(s, "Reg-Contact: %s\r\n", peer->fullcontact);
		astman_append(s, "QualifyFreq: %d ms\r\n", peer->qualifyfreq);
		astman_append(s, "Parkinglot: %s\r\n", peer->parkinglot);
		if (peer->chanvars) {
			for (v = peer->chanvars ; v ; v = v->next) {
				astman_append(s, "ChanVariable: %s=%s\r\n", v->name, v->value);
			}
		}
		astman_append(s, "SIP-Use-Reason-Header: %s\r\n", (ast_test_flag(&peer->flags[1], SIP_PAGE2_Q850_REASON)) ? "Y" : "N");

		peer = unref_peer(peer, "sip_show_peer: unref_peer: done with peer");

	} else {
		ast_cli(fd, "Peer %s not found.\n", argv[3]);
		ast_cli(fd, "\n");
	}

	return CLI_SUCCESS;
}

/*! \brief Do completion on user name */
static char *complete_sip_user(const char *word, int state)
{
	char *result = NULL;
	int wordlen = strlen(word);
	int which = 0;
	struct ao2_iterator user_iter;
	struct sip_peer *user;

	user_iter = ao2_iterator_init(peers, 0);
	while ((user = ao2_t_iterator_next(&user_iter, "iterate thru peers table"))) {
		ao2_lock(user);
		if (!(user->type & SIP_TYPE_USER)) {
			ao2_unlock(user);
			unref_peer(user, "complete sip user");
			continue;
		}
		/* locking of the object is not required because only the name and flags are being compared */
		if (!strncasecmp(word, user->name, wordlen) && ++which > state) {
			result = ast_strdup(user->name);
		}
		ao2_unlock(user);
		unref_peer(user, "complete sip user");
		if (result) {
			break;
		}
	}
	ao2_iterator_destroy(&user_iter);
	return result;
}
/*! \brief Support routine for 'sip show user' CLI */
static char *complete_sip_show_user(const char *line, const char *word, int pos, int state)
{
	if (pos == 3)
		return complete_sip_user(word, state);

	return NULL;
}

/*! \brief Show one user in detail */
static char *sip_show_user(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	char cbuf[256];
	struct sip_peer *user;
	struct ast_variable *v;
	int load_realtime;

	switch (cmd) {
	case CLI_INIT:
		e->command = "sip show user";
		e->usage =
			"Usage: sip show user <name> [load]\n"
			"       Shows all details on one SIP user and the current status.\n"
			"       Option \"load\" forces lookup of peer in realtime storage.\n";
		return NULL;
	case CLI_GENERATE:
		return complete_sip_show_user(a->line, a->word, a->pos, a->n);
	}

	if (a->argc < 4)
		return CLI_SHOWUSAGE;

	/* Load from realtime storage? */
	load_realtime = (a->argc == 5 && !strcmp(a->argv[4], "load")) ? TRUE : FALSE;

	if ((user = find_peer(a->argv[3], NULL, load_realtime, FINDUSERS, FALSE, 0))) {
		ao2_lock(user);
		ast_cli(a->fd, "\n\n");
		ast_cli(a->fd, "  * Name       : %s\n", user->name);
		ast_cli(a->fd, "  Secret       : %s\n", ast_strlen_zero(user->secret)?"<Not set>":"<Set>");
		ast_cli(a->fd, "  MD5Secret    : %s\n", ast_strlen_zero(user->md5secret)?"<Not set>":"<Set>");
		ast_cli(a->fd, "  Context      : %s\n", user->context);
		ast_cli(a->fd, "  Language     : %s\n", user->language);
		if (!ast_strlen_zero(user->accountcode))
			ast_cli(a->fd, "  Accountcode  : %s\n", user->accountcode);
		ast_cli(a->fd, "  AMA flags    : %s\n", ast_cdr_flags2str(user->amaflags));
		ast_cli(a->fd, "  Transfer mode: %s\n", transfermode2str(user->allowtransfer));
		ast_cli(a->fd, "  MaxCallBR    : %d kbps\n", user->maxcallbitrate);
		ast_cli(a->fd, "  CallingPres  : %s\n", ast_describe_caller_presentation(user->callingpres));
		ast_cli(a->fd, "  Call limit   : %d\n", user->call_limit);
		ast_cli(a->fd, "  Callgroup    : ");
		print_group(a->fd, user->callgroup, 0);
		ast_cli(a->fd, "  Pickupgroup  : ");
		print_group(a->fd, user->pickupgroup, 0);
		ast_cli(a->fd, "  Callerid     : %s\n", ast_callerid_merge(cbuf, sizeof(cbuf), user->cid_name, user->cid_num, "<unspecified>"));
		ast_cli(a->fd, "  ACL          : %s\n", AST_CLI_YESNO(user->ha != NULL));
 		ast_cli(a->fd, "  Sess-Timers  : %s\n", stmode2str(user->stimer.st_mode_oper));
 		ast_cli(a->fd, "  Sess-Refresh : %s\n", strefresher2str(user->stimer.st_ref));
 		ast_cli(a->fd, "  Sess-Expires : %d secs\n", user->stimer.st_max_se);
 		ast_cli(a->fd, "  Sess-Min-SE  : %d secs\n", user->stimer.st_min_se);
		ast_cli(a->fd, "  RTP Engine   : %s\n", user->engine);

		ast_cli(a->fd, "  Codec Order  : (");
		print_codec_to_cli(a->fd, &user->prefs);
		ast_cli(a->fd, ")\n");

		ast_cli(a->fd, "  Auto-Framing:  %s \n", AST_CLI_YESNO(user->autoframing));
		if (user->chanvars) {
 			ast_cli(a->fd, "  Variables    :\n");
			for (v = user->chanvars ; v ; v = v->next)
 				ast_cli(a->fd, "                 %s = %s\n", v->name, v->value);
		}

		ast_cli(a->fd, "\n");

		ao2_unlock(user);
		unref_peer(user, "sip show user");
	} else {
		ast_cli(a->fd, "User %s not found.\n", a->argv[3]);
		ast_cli(a->fd, "\n");
	}

	return CLI_SUCCESS;
}


static char *sip_show_sched(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	struct ast_str *cbuf;
	struct ast_cb_names cbnames = {9, { "retrans_pkt",
                                        "__sip_autodestruct",
                                        "expire_register",
                                        "auto_congest",
                                        "sip_reg_timeout",
                                        "sip_poke_peer_s",
                                        "sip_poke_noanswer",
                                        "sip_reregister",
                                        "sip_reinvite_retry"},
								   { retrans_pkt,
                                     __sip_autodestruct,
                                     expire_register,
                                     auto_congest,
                                     sip_reg_timeout,
                                     sip_poke_peer_s,
                                     sip_poke_noanswer,
                                     sip_reregister,
                                     sip_reinvite_retry}};
	
	switch (cmd) {
	case CLI_INIT:
		e->command = "sip show sched";
		e->usage =
			"Usage: sip show sched\n"
			"       Shows stats on what's in the sched queue at the moment\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	cbuf = ast_str_alloca(2048);

	ast_cli(a->fd, "\n");
	ast_sched_report(sched, &cbuf, &cbnames);
	ast_cli(a->fd, "%s", cbuf->str);

	return CLI_SUCCESS;
}

/*! \brief  Show SIP Registry (registrations with other SIP proxies */
static char *sip_show_registry(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
#define FORMAT2 "%-39.39s %-6.6s %-12.12s  %8.8s %-20.20s %-25.25s\n"
#define FORMAT  "%-39.39s %-6.6s %-12.12s  %8d %-20.20s %-25.25s\n"
	char host[80];
	char user[80];
	char tmpdat[256];
	struct ast_tm tm;
	int counter = 0;

	switch (cmd) {
	case CLI_INIT:
		e->command = "sip show registry";
		e->usage =
			"Usage: sip show registry\n"
			"       Lists all registration requests and status.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc != 3)
		return CLI_SHOWUSAGE;
	ast_cli(a->fd, FORMAT2, "Host", "dnsmgr", "Username", "Refresh", "State", "Reg.Time");
	
	ASTOBJ_CONTAINER_TRAVERSE(&regl, 1, do {
		ASTOBJ_RDLOCK(iterator);
		snprintf(host, sizeof(host), "%s:%d", iterator->hostname, iterator->portno ? iterator->portno : STANDARD_SIP_PORT);
		snprintf(user, sizeof(user), "%s", iterator->username);
		if (!ast_strlen_zero(iterator->regdomain)) {
			snprintf(tmpdat, sizeof(tmpdat), "%s", user);
			snprintf(user, sizeof(user), "%s@%s", tmpdat, iterator->regdomain);}
		if (iterator->regdomainport) {
			snprintf(tmpdat, sizeof(tmpdat), "%s", user);
			snprintf(user, sizeof(user), "%s:%d", tmpdat, iterator->regdomainport);}
		if (iterator->regtime.tv_sec) {
			ast_localtime(&iterator->regtime, &tm, NULL);
			ast_strftime(tmpdat, sizeof(tmpdat), "%a, %d %b %Y %T", &tm);
		} else
			tmpdat[0] = '\0';
		ast_cli(a->fd, FORMAT, host, (iterator->dnsmgr) ? "Y" : "N", user, iterator->refresh, regstate2str(iterator->regstate), tmpdat);
		ASTOBJ_UNLOCK(iterator);
		counter++;
	} while(0));
	ast_cli(a->fd, "%d SIP registrations.\n", counter);
	return CLI_SUCCESS;
#undef FORMAT
#undef FORMAT2
}

/*! \brief Unregister (force expiration) a SIP peer in the registry via CLI
	\note This function does not tell the SIP device what's going on,
	so use it with great care.
*/
static char *sip_unregister(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	struct sip_peer *peer;
	int load_realtime = 0;

	switch (cmd) {
	case CLI_INIT:
		e->command = "sip unregister";
		e->usage =
			"Usage: sip unregister <peer>\n"
			"       Unregister (force expiration) a SIP peer from the registry\n";
		return NULL;
	case CLI_GENERATE:
		return complete_sip_unregister(a->line, a->word, a->pos, a->n);
	}
	
	if (a->argc != 3)
		return CLI_SHOWUSAGE;
	
	if ((peer = find_peer(a->argv[2], NULL, load_realtime, FINDPEERS, TRUE, 0))) {
		if (peer->expire > 0) {
			AST_SCHED_DEL_UNREF(sched, peer->expire,
				unref_peer(peer, "remove register expire ref"));
			expire_register(ref_peer(peer, "ref for expire_register"));
			ast_cli(a->fd, "Unregistered peer \'%s\'\n\n", a->argv[2]);
		} else {
			ast_cli(a->fd, "Peer %s not registered\n", a->argv[2]);
		}
		unref_peer(peer, "sip_unregister: unref_peer via sip_unregister: done with peer from find_peer call");
	} else {
		ast_cli(a->fd, "Peer unknown: \'%s\'. Not unregistered.\n", a->argv[2]);
	}
	
	return CLI_SUCCESS;
}

/*! \brief Callback for show_chanstats */
static int show_chanstats_cb(void *__cur, void *__arg, int flags)
{
#define FORMAT2 "%-15.15s  %-11.11s  %-8.8s %-10.10s  %-10.10s (     %%) %-6.6s %-10.10s  %-10.10s (     %%) %-6.6s\n"
#define FORMAT  "%-15.15s  %-11.11s  %-8.8s %-10.10u%-1.1s %-10.10u (%5.2f%%) %-6.4lf %-10.10u%-1.1s %-10.10u (%5.2f%%) %-6.4lf\n"
	struct sip_pvt *cur = __cur;
	struct ast_rtp_instance_stats stats;
	char durbuf[10];
	int duration;
	int durh, durm, durs;
	struct ast_channel *c = cur->owner;
	struct __show_chan_arg *arg = __arg;
	int fd = arg->fd;


	if (cur->subscribed != NONE) /* Subscriptions */
		return 0;	/* don't care, we scan all channels */

	if (!cur->rtp) {
		if (sipdebug) {
			ast_cli(fd, "%-15.15s  %-11.11s (inv state: %s) -- %s\n",
				ast_sockaddr_stringify_addr(&cur->sa), cur->callid,
				invitestate2string[cur->invitestate].desc,
				"-- No RTP active");
		}
		return 0;	/* don't care, we scan all channels */
	}

	ast_rtp_instance_get_stats(cur->rtp, &stats, AST_RTP_INSTANCE_STAT_ALL);

	if (c && c->cdr && !ast_tvzero(c->cdr->start)) {
		duration = (int)(ast_tvdiff_ms(ast_tvnow(), c->cdr->start) / 1000);
		durh = duration / 3600;
		durm = (duration % 3600) / 60;
		durs = duration % 60;
		snprintf(durbuf, sizeof(durbuf), "%02d:%02d:%02d", durh, durm, durs);
	} else {
		durbuf[0] = '\0';
	}

	ast_cli(fd, FORMAT,
		ast_sockaddr_stringify_addr(&cur->sa),
		cur->callid,
		durbuf,
		stats.rxcount > (unsigned int) 100000 ? (unsigned int) (stats.rxcount)/(unsigned int) 1000 : stats.rxcount,
		stats.rxcount > (unsigned int) 100000 ? "K":" ",
		stats.rxploss,
		(stats.rxcount + stats.rxploss) > 0 ? (double) stats.rxploss / (stats.rxcount + stats.rxploss) * 100 : 0,
		stats.rxjitter,
		stats.txcount > (unsigned int) 100000 ? (unsigned int) (stats.txcount)/(unsigned int) 1000 : stats.txcount,
		stats.txcount > (unsigned int) 100000 ? "K":" ",
		stats.txploss,
		stats.txcount > 0 ? (double) stats.txploss / stats.txcount * 100 : 0,
		stats.txjitter
	);
	arg->numchans++;

	return 0;	/* don't care, we scan all channels */
}

/*! \brief SIP show channelstats CLI (main function) */
static char *sip_show_channelstats(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	struct __show_chan_arg arg = { .fd = a->fd, .numchans = 0 };

	switch (cmd) {
	case CLI_INIT:
		e->command = "sip show channelstats";
		e->usage =
			"Usage: sip show channelstats\n"
			"       Lists all currently active SIP channel's RTCP statistics.\n"
			"       Note that calls in the much optimized RTP P2P bridge mode will not show any packets here.";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc != 3)
		return CLI_SHOWUSAGE;

	ast_cli(a->fd, FORMAT2, "Peer", "Call ID", "Duration", "Recv: Pack", "Lost", "Jitter", "Send: Pack", "Lost", "Jitter");
	/* iterate on the container and invoke the callback on each item */
	ao2_t_callback(dialogs, OBJ_NODATA, show_chanstats_cb, &arg, "callback to sip show chanstats");
	ast_cli(a->fd, "%d active SIP channel%s\n", arg.numchans, (arg.numchans != 1) ? "s" : "");
	return CLI_SUCCESS;
}
#undef FORMAT
#undef FORMAT2

/*! \brief List global settings for the SIP channel */
static char *sip_show_settings(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int realtimepeers;
	int realtimeregs;
	char codec_buf[SIPBUFSIZE];
	const char *msg;	/* temporary msg pointer */
	struct sip_auth_container *credentials;

	switch (cmd) {
	case CLI_INIT:
		e->command = "sip show settings";
		e->usage =
			"Usage: sip show settings\n"
			"       Provides detailed list of the configuration of the SIP channel.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc != 3)
		return CLI_SHOWUSAGE;

	realtimepeers = ast_check_realtime("sippeers");
	realtimeregs = ast_check_realtime("sipregs");

	ast_mutex_lock(&authl_lock);
	credentials = authl;
	if (credentials) {
		ao2_t_ref(credentials, +1, "Ref global auth for show");
	}
	ast_mutex_unlock(&authl_lock);

	ast_cli(a->fd, "\n\nGlobal Settings:\n");
	ast_cli(a->fd, "----------------\n");
	ast_cli(a->fd, "  UDP Bindaddress:        %s\n", ast_sockaddr_stringify(&bindaddr));
	if (ast_sockaddr_is_ipv6(&bindaddr) && ast_sockaddr_is_any(&bindaddr)) {
		ast_cli(a->fd, "  ** Additional Info:\n");
		ast_cli(a->fd, "     [::] may include IPv4 in addition to IPv6, if such a feature is enabled in the OS.\n");
	}
	ast_cli(a->fd, "  TCP SIP Bindaddress:    %s\n",
		sip_cfg.tcp_enabled != FALSE ?
				ast_sockaddr_stringify(&sip_tcp_desc.local_address) :
				"Disabled");
	ast_cli(a->fd, "  TLS SIP Bindaddress:    %s\n",
		default_tls_cfg.enabled != FALSE ?
				ast_sockaddr_stringify(&sip_tls_desc.local_address) :
				"Disabled");
	ast_cli(a->fd, "  Videosupport:           %s\n", AST_CLI_YESNO(ast_test_flag(&global_flags[1], SIP_PAGE2_VIDEOSUPPORT)));
	ast_cli(a->fd, "  Textsupport:            %s\n", AST_CLI_YESNO(ast_test_flag(&global_flags[1], SIP_PAGE2_TEXTSUPPORT)));
	ast_cli(a->fd, "  Ignore SDP sess. ver.:  %s\n", AST_CLI_YESNO(ast_test_flag(&global_flags[1], SIP_PAGE2_IGNORESDPVERSION)));
	ast_cli(a->fd, "  AutoCreate Peer:        %s\n", AST_CLI_YESNO(sip_cfg.autocreatepeer));
	ast_cli(a->fd, "  Match Auth Username:    %s\n", AST_CLI_YESNO(global_match_auth_username));
	ast_cli(a->fd, "  Allow unknown access:   %s\n", AST_CLI_YESNO(sip_cfg.allowguest));
	ast_cli(a->fd, "  Allow subscriptions:    %s\n", AST_CLI_YESNO(ast_test_flag(&global_flags[1], SIP_PAGE2_ALLOWSUBSCRIBE)));
	ast_cli(a->fd, "  Allow overlap dialing:  %s\n", allowoverlap2str(ast_test_flag(&global_flags[1], SIP_PAGE2_ALLOWOVERLAP)));
	ast_cli(a->fd, "  Allow promisc. redir:   %s\n", AST_CLI_YESNO(ast_test_flag(&global_flags[0], SIP_PROMISCREDIR)));
	ast_cli(a->fd, "  Enable call counters:   %s\n", AST_CLI_YESNO(global_callcounter));
	ast_cli(a->fd, "  SIP domain support:     %s\n", AST_CLI_YESNO(!AST_LIST_EMPTY(&domain_list)));
	ast_cli(a->fd, "  Realm. auth:            %s\n", AST_CLI_YESNO(credentials != NULL));
	if (credentials) {
		struct sip_auth *auth;

		AST_LIST_TRAVERSE(&credentials->list, auth, node) {
			ast_cli(a->fd, "  Realm. auth entry:      Realm %-15.15s User %-10.20s %s\n",
				auth->realm,
				auth->username,
				!ast_strlen_zero(auth->secret)
					? "<Secret set>"
					: (!ast_strlen_zero(auth->md5secret)
						? "<MD5secret set>" : "<Not set>"));
		}
		ao2_t_ref(credentials, -1, "Unref global auth for show");
	}
	ast_cli(a->fd, "  Our auth realm          %s\n", sip_cfg.realm);
	ast_cli(a->fd, "  Use domains as realms:  %s\n", AST_CLI_YESNO(sip_cfg.domainsasrealm));
	ast_cli(a->fd, "  Call to non-local dom.: %s\n", AST_CLI_YESNO(sip_cfg.allow_external_domains));
	ast_cli(a->fd, "  URI user is phone no:   %s\n", AST_CLI_YESNO(ast_test_flag(&global_flags[0], SIP_USEREQPHONE)));
	ast_cli(a->fd, "  Always auth rejects:    %s\n", AST_CLI_YESNO(sip_cfg.alwaysauthreject));
	ast_cli(a->fd, "  Direct RTP setup:       %s\n", AST_CLI_YESNO(sip_cfg.directrtpsetup));
	ast_cli(a->fd, "  User Agent:             %s\n", global_useragent);
	ast_cli(a->fd, "  SDP Session Name:       %s\n", ast_strlen_zero(global_sdpsession) ? "-" : global_sdpsession);
	ast_cli(a->fd, "  SDP Owner Name:         %s\n", ast_strlen_zero(global_sdpowner) ? "-" : global_sdpowner);
	ast_cli(a->fd, "  Reg. context:           %s\n", S_OR(sip_cfg.regcontext, "(not set)"));
	ast_cli(a->fd, "  Regexten on Qualify:    %s\n", AST_CLI_YESNO(sip_cfg.regextenonqualify));
	ast_cli(a->fd, "  Legacy userfield parse: %s\n", AST_CLI_YESNO(sip_cfg.legacy_useroption_parsing));
	ast_cli(a->fd, "  Caller ID:              %s\n", default_callerid);
	if ((default_fromdomainport) && (default_fromdomainport != STANDARD_SIP_PORT)) {
		ast_cli(a->fd, "  From: Domain:           %s:%d\n", default_fromdomain, default_fromdomainport);
	} else {
		ast_cli(a->fd, "  From: Domain:           %s\n", default_fromdomain);
	}
	ast_cli(a->fd, "  Record SIP history:     %s\n", AST_CLI_ONOFF(recordhistory));
	ast_cli(a->fd, "  Call Events:            %s\n", AST_CLI_ONOFF(sip_cfg.callevents));
	ast_cli(a->fd, "  Auth. Failure Events:   %s\n", AST_CLI_ONOFF(global_authfailureevents));

	ast_cli(a->fd, "  T.38 support:           %s\n", AST_CLI_YESNO(ast_test_flag(&global_flags[1], SIP_PAGE2_T38SUPPORT)));
	ast_cli(a->fd, "  T.38 EC mode:           %s\n", faxec2str(ast_test_flag(&global_flags[1], SIP_PAGE2_T38SUPPORT)));
	ast_cli(a->fd, "  T.38 MaxDtgrm:          %d\n", global_t38_maxdatagram);
	if (!realtimepeers && !realtimeregs)
		ast_cli(a->fd, "  SIP realtime:           Disabled\n" );
	else
		ast_cli(a->fd, "  SIP realtime:           Enabled\n" );
	ast_cli(a->fd, "  Qualify Freq :          %d ms\n", global_qualifyfreq);
	ast_cli(a->fd, "  Q.850 Reason header:    %s\n", AST_CLI_YESNO(ast_test_flag(&global_flags[1], SIP_PAGE2_Q850_REASON)));
	ast_cli(a->fd, "  Store SIP_CAUSE:        %s\n", AST_CLI_YESNO(global_store_sip_cause));
	ast_cli(a->fd, "\nNetwork QoS Settings:\n");
	ast_cli(a->fd, "---------------------------\n");
	ast_cli(a->fd, "  IP ToS SIP:             %s\n", ast_tos2str(global_tos_sip));
	ast_cli(a->fd, "  IP ToS RTP audio:       %s\n", ast_tos2str(global_tos_audio));
	ast_cli(a->fd, "  IP ToS RTP video:       %s\n", ast_tos2str(global_tos_video));
	ast_cli(a->fd, "  IP ToS RTP text:        %s\n", ast_tos2str(global_tos_text));
	ast_cli(a->fd, "  802.1p CoS SIP:         %d\n", global_cos_sip);
	ast_cli(a->fd, "  802.1p CoS RTP audio:   %d\n", global_cos_audio);
	ast_cli(a->fd, "  802.1p CoS RTP video:   %d\n", global_cos_video);
	ast_cli(a->fd, "  802.1p CoS RTP text:    %d\n", global_cos_text);
	ast_cli(a->fd, "  Jitterbuffer enabled:   %s\n", AST_CLI_YESNO(ast_test_flag(&global_jbconf, AST_JB_ENABLED)));
	if (ast_test_flag(&global_jbconf, AST_JB_ENABLED)) {
		ast_cli(a->fd, "  Jitterbuffer forced:    %s\n", AST_CLI_YESNO(ast_test_flag(&global_jbconf, AST_JB_FORCED)));
		ast_cli(a->fd, "  Jitterbuffer max size:  %ld\n", global_jbconf.max_size);
		ast_cli(a->fd, "  Jitterbuffer resync:    %ld\n", global_jbconf.resync_threshold);
		ast_cli(a->fd, "  Jitterbuffer impl:      %s\n", global_jbconf.impl);
		if (!strcasecmp(global_jbconf.impl, "adaptive")) {
			ast_cli(a->fd, "  Jitterbuffer tgt extra: %ld\n", global_jbconf.target_extra);
		}
		ast_cli(a->fd, "  Jitterbuffer log:       %s\n", AST_CLI_YESNO(ast_test_flag(&global_jbconf, AST_JB_LOG)));
	}

	ast_cli(a->fd, "\nNetwork Settings:\n");
	ast_cli(a->fd, "---------------------------\n");
	/* determine if/how SIP address can be remapped */
	if (localaddr == NULL)
		msg = "Disabled, no localnet list";
	else if (ast_sockaddr_isnull(&externaddr))
		msg = "Disabled";
	else if (!ast_strlen_zero(externhost))
		msg = "Enabled using externhost";
	else
		msg = "Enabled using externaddr";
	ast_cli(a->fd, "  SIP address remapping:  %s\n", msg);
	ast_cli(a->fd, "  Externhost:             %s\n", S_OR(externhost, "<none>"));
	ast_cli(a->fd, "  Externaddr:             %s\n", ast_sockaddr_stringify(&externaddr));
	ast_cli(a->fd, "  Externrefresh:          %d\n", externrefresh);
	{
		struct ast_ha *d;
		const char *prefix = "Localnet:";

		for (d = localaddr; d ; prefix = "", d = d->next) {
			const char *addr = ast_strdupa(ast_sockaddr_stringify_addr(&d->addr));
			const char *mask = ast_strdupa(ast_sockaddr_stringify_addr(&d->netmask));
			ast_cli(a->fd, "  %-24s%s/%s\n", prefix, addr, mask);
		}
	}
	ast_cli(a->fd, "\nGlobal Signalling Settings:\n");
	ast_cli(a->fd, "---------------------------\n");
	ast_cli(a->fd, "  Codecs:                 ");
	ast_getformatname_multiple(codec_buf, sizeof(codec_buf) -1, sip_cfg.capability);
	ast_cli(a->fd, "%s\n", codec_buf);
	ast_cli(a->fd, "  Codec Order:            ");
	print_codec_to_cli(a->fd, &default_prefs);
	ast_cli(a->fd, "\n");
	ast_cli(a->fd, "  Relax DTMF:             %s\n", AST_CLI_YESNO(global_relaxdtmf));
	ast_cli(a->fd, "  RFC2833 Compensation:   %s\n", AST_CLI_YESNO(ast_test_flag(&global_flags[1], SIP_PAGE2_RFC2833_COMPENSATE)));
	ast_cli(a->fd, "  Symmetric RTP:          %s\n", AST_CLI_YESNO(ast_test_flag(&global_flags[1], SIP_PAGE2_SYMMETRICRTP)));
	ast_cli(a->fd, "  Compact SIP headers:    %s\n", AST_CLI_YESNO(sip_cfg.compactheaders));
	ast_cli(a->fd, "  RTP Keepalive:          %d %s\n", global_rtpkeepalive, global_rtpkeepalive ? "" : "(Disabled)" );
	ast_cli(a->fd, "  RTP Timeout:            %d %s\n", global_rtptimeout, global_rtptimeout ? "" : "(Disabled)" );
	ast_cli(a->fd, "  RTP Hold Timeout:       %d %s\n", global_rtpholdtimeout, global_rtpholdtimeout ? "" : "(Disabled)");
	ast_cli(a->fd, "  MWI NOTIFY mime type:   %s\n", default_notifymime);
	ast_cli(a->fd, "  DNS SRV lookup:         %s\n", AST_CLI_YESNO(sip_cfg.srvlookup));
	ast_cli(a->fd, "  Pedantic SIP support:   %s\n", AST_CLI_YESNO(sip_cfg.pedanticsipchecking));
	ast_cli(a->fd, "  Reg. min duration       %d secs\n", min_expiry);
	ast_cli(a->fd, "  Reg. max duration:      %d secs\n", max_expiry);
	ast_cli(a->fd, "  Reg. default duration:  %d secs\n", default_expiry);
	ast_cli(a->fd, "  Outbound reg. timeout:  %d secs\n", global_reg_timeout);
	ast_cli(a->fd, "  Outbound reg. attempts: %d\n", global_regattempts_max);
	ast_cli(a->fd, "  Notify ringing state:   %s\n", AST_CLI_YESNO(sip_cfg.notifyringing));
	if (sip_cfg.notifyringing) {
		ast_cli(a->fd, "    Include CID:          %s%s\n",
				AST_CLI_YESNO(sip_cfg.notifycid),
				sip_cfg.notifycid == IGNORE_CONTEXT ? " (Ignoring context)" : "");
	}
	ast_cli(a->fd, "  Notify hold state:      %s\n", AST_CLI_YESNO(sip_cfg.notifyhold));
	ast_cli(a->fd, "  SIP Transfer mode:      %s\n", transfermode2str(sip_cfg.allowtransfer));
	ast_cli(a->fd, "  Max Call Bitrate:       %d kbps\n", default_maxcallbitrate);
	ast_cli(a->fd, "  Auto-Framing:           %s\n", AST_CLI_YESNO(global_autoframing));
	ast_cli(a->fd, "  Outb. proxy:            %s %s\n", ast_strlen_zero(sip_cfg.outboundproxy.name) ? "<not set>" : sip_cfg.outboundproxy.name,
							sip_cfg.outboundproxy.force ? "(forced)" : "");
	ast_cli(a->fd, "  Session Timers:         %s\n", stmode2str(global_st_mode));
	ast_cli(a->fd, "  Session Refresher:      %s\n", strefresher2str (global_st_refresher));
	ast_cli(a->fd, "  Session Expires:        %d secs\n", global_max_se);
	ast_cli(a->fd, "  Session Min-SE:         %d secs\n", global_min_se);
 	ast_cli(a->fd, "  Timer T1:               %d\n", global_t1);
	ast_cli(a->fd, "  Timer T1 minimum:       %d\n", global_t1min);
 	ast_cli(a->fd, "  Timer B:                %d\n", global_timer_b);
	ast_cli(a->fd, "  No premature media:     %s\n", AST_CLI_YESNO(global_prematuremediafilter));
	ast_cli(a->fd, "  Max forwards:           %d\n", sip_cfg.default_max_forwards);

	ast_cli(a->fd, "\nDefault Settings:\n");
	ast_cli(a->fd, "-----------------\n");
	ast_cli(a->fd, "  Allowed transports:     %s\n", get_transport_list(default_transports));
	ast_cli(a->fd, "  Outbound transport:	  %s\n", get_transport(default_primary_transport));
	ast_cli(a->fd, "  Context:                %s\n", sip_cfg.default_context);
	ast_cli(a->fd, "  Force rport:            %s\n", AST_CLI_YESNO(ast_test_flag(&global_flags[0], SIP_NAT_FORCE_RPORT)));
	ast_cli(a->fd, "  DTMF:                   %s\n", dtmfmode2str(ast_test_flag(&global_flags[0], SIP_DTMF)));
	ast_cli(a->fd, "  Qualify:                %d\n", default_qualify);
	ast_cli(a->fd, "  Use ClientCode:         %s\n", AST_CLI_YESNO(ast_test_flag(&global_flags[0], SIP_USECLIENTCODE)));
	ast_cli(a->fd, "  Progress inband:        %s\n", (ast_test_flag(&global_flags[0], SIP_PROG_INBAND) == SIP_PROG_INBAND_NEVER) ? "Never" : (AST_CLI_YESNO(ast_test_flag(&global_flags[0], SIP_PROG_INBAND) != SIP_PROG_INBAND_NO)));
	ast_cli(a->fd, "  Language:               %s\n", default_language);
	ast_cli(a->fd, "  MOH Interpret:          %s\n", default_mohinterpret);
	ast_cli(a->fd, "  MOH Suggest:            %s\n", default_mohsuggest);
	ast_cli(a->fd, "  Voice Mail Extension:   %s\n", default_vmexten);

	
	if (realtimepeers || realtimeregs) {
		ast_cli(a->fd, "\nRealtime SIP Settings:\n");
		ast_cli(a->fd, "----------------------\n");
		ast_cli(a->fd, "  Realtime Peers:         %s\n", AST_CLI_YESNO(realtimepeers));
		ast_cli(a->fd, "  Realtime Regs:          %s\n", AST_CLI_YESNO(realtimeregs));
		ast_cli(a->fd, "  Cache Friends:          %s\n", AST_CLI_YESNO(ast_test_flag(&global_flags[1], SIP_PAGE2_RTCACHEFRIENDS)));
		ast_cli(a->fd, "  Update:                 %s\n", AST_CLI_YESNO(sip_cfg.peer_rtupdate));
		ast_cli(a->fd, "  Ignore Reg. Expire:     %s\n", AST_CLI_YESNO(sip_cfg.ignore_regexpire));
		ast_cli(a->fd, "  Save sys. name:         %s\n", AST_CLI_YESNO(sip_cfg.rtsave_sysname));
		ast_cli(a->fd, "  Auto Clear:             %d (%s)\n", sip_cfg.rtautoclear, ast_test_flag(&global_flags[1], SIP_PAGE2_RTAUTOCLEAR) ? "Enabled" : "Disabled");
	}
	ast_cli(a->fd, "\n----\n");
	return CLI_SUCCESS;
}

static char *sip_show_mwi(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
#define FORMAT  "%-30.30s  %-12.12s  %-10.10s  %-10.10s\n"
	char host[80];
	
	switch (cmd) {
	case CLI_INIT:
		e->command = "sip show mwi";
		e->usage =
			"Usage: sip show mwi\n"
			"       Provides a list of MWI subscriptions and status.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}
	
	ast_cli(a->fd, FORMAT, "Host", "Username", "Mailbox", "Subscribed");
	
	ASTOBJ_CONTAINER_TRAVERSE(&submwil, 1, do {
		ASTOBJ_RDLOCK(iterator);
		snprintf(host, sizeof(host), "%s:%d", iterator->hostname, iterator->portno ? iterator->portno : STANDARD_SIP_PORT);
		ast_cli(a->fd, FORMAT, host, iterator->username, iterator->mailbox, AST_CLI_YESNO(iterator->subscribed));
		ASTOBJ_UNLOCK(iterator);
	} while(0));

	return CLI_SUCCESS;
#undef FORMAT
}


/*! \brief Show subscription type in string format */
static const char *subscription_type2str(enum subscriptiontype subtype)
{
	int i;

	for (i = 1; i < ARRAY_LEN(subscription_types); i++) {
		if (subscription_types[i].type == subtype) {
			return subscription_types[i].text;
		}
	}
	return subscription_types[0].text;
}

/*! \brief Find subscription type in array */
static const struct cfsubscription_types *find_subscription_type(enum subscriptiontype subtype)
{
	int i;

	for (i = 1; i < ARRAY_LEN(subscription_types); i++) {
		if (subscription_types[i].type == subtype) {
			return &subscription_types[i];
		}
	}
	return &subscription_types[0];
}

/*
 * We try to structure all functions that loop on data structures as
 * a handler for individual entries, and a mainloop that iterates
 * on the main data structure. This way, moving the code to containers
 * that support iteration through callbacks will be a lot easier.
 */

#define FORMAT4 "%-15.15s  %-15.15s  %-15.15s  %-15.15s  %-13.13s  %-15.15s %-10.10s %-6.6d\n"
#define FORMAT3 "%-15.15s  %-15.15s  %-15.15s  %-15.15s  %-13.13s  %-15.15s %-10.10s %-6.6s\n"
#define FORMAT2 "%-15.15s  %-15.15s  %-15.15s  %-15.15s  %-7.7s  %-15.15s %-10.10s %-10.10s\n"
#define FORMAT  "%-15.15s  %-15.15s  %-15.15s  %-15.15s  %-3.3s %-3.3s  %-15.15s %-10.10s %-10.10s\n"

/*! \brief callback for show channel|subscription */
static int show_channels_cb(void *__cur, void *__arg, int flags)
{
	struct sip_pvt *cur = __cur;
	struct __show_chan_arg *arg = __arg;
	const struct ast_sockaddr *dst = sip_real_dst(cur);
	
	/* XXX indentation preserved to reduce diff. Will be fixed later */
	if (cur->subscribed == NONE && !arg->subscriptions) {
		/* set if SIP transfer in progress */
		const char *referstatus = cur->refer ? referstatus2str(cur->refer->status) : "";
		char formatbuf[SIPBUFSIZE/2];
		
		ast_cli(arg->fd, FORMAT, ast_sockaddr_stringify_addr(dst),
				S_OR(cur->username, S_OR(cur->cid_num, "(None)")),
				cur->callid,
				ast_getformatname_multiple(formatbuf, sizeof(formatbuf), cur->owner ? cur->owner->nativeformats : 0),
				AST_CLI_YESNO(ast_test_flag(&cur->flags[1], SIP_PAGE2_CALL_ONHOLD)),
				cur->needdestroy ? "(d)" : "",
				cur->lastmsg ,
				referstatus,
				cur->relatedpeer ? cur->relatedpeer->name : "<guest>"
			);
		arg->numchans++;
	}
	if (cur->subscribed != NONE && arg->subscriptions) {
		struct ast_str *mailbox_str = ast_str_alloca(512);
		if (cur->subscribed == MWI_NOTIFICATION && cur->relatedpeer)
			peer_mailboxes_to_str(&mailbox_str, cur->relatedpeer);
		ast_cli(arg->fd, FORMAT4, ast_sockaddr_stringify_addr(dst),
				S_OR(cur->username, S_OR(cur->cid_num, "(None)")),
			   	cur->callid,
				/* the 'complete' exten/context is hidden in the refer_to field for subscriptions */
				cur->subscribed == MWI_NOTIFICATION ? "--" : cur->subscribeuri,
				cur->subscribed == MWI_NOTIFICATION ? "<none>" : ast_extension_state2str(cur->laststate),
				subscription_type2str(cur->subscribed),
				cur->subscribed == MWI_NOTIFICATION ? S_OR(mailbox_str->str, "<none>") : "<none>",
				cur->expiry
			);
		arg->numchans++;
	}
	return 0;	/* don't care, we scan all channels */
}

/*! \brief CLI for show channels or subscriptions.
 * This is a new-style CLI handler so a single function contains
 * the prototype for the function, the 'generator' to produce multiple
 * entries in case it is required, and the actual handler for the command.
 */
static char *sip_show_channels(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	struct __show_chan_arg arg = { .fd = a->fd, .numchans = 0 };


	if (cmd == CLI_INIT) {
		e->command = "sip show {channels|subscriptions}";
		e->usage =
			"Usage: sip show channels\n"
			"       Lists all currently active SIP calls (dialogs).\n"
			"Usage: sip show subscriptions\n"
			"       Lists active SIP subscriptions.\n";
		return NULL;
	} else if (cmd == CLI_GENERATE)
		return NULL;

	if (a->argc != e->args)
		return CLI_SHOWUSAGE;
	arg.subscriptions = !strcasecmp(a->argv[e->args - 1], "subscriptions");
	if (!arg.subscriptions)
		ast_cli(arg.fd, FORMAT2, "Peer", "User/ANR", "Call ID", "Format", "Hold", "Last Message", "Expiry", "Peer");
	else
		ast_cli(arg.fd, FORMAT3, "Peer", "User", "Call ID", "Extension", "Last state", "Type", "Mailbox", "Expiry");

	/* iterate on the container and invoke the callback on each item */
	ao2_t_callback(dialogs, OBJ_NODATA, show_channels_cb, &arg, "callback to show channels");
	
	/* print summary information */
	ast_cli(arg.fd, "%d active SIP %s%s\n", arg.numchans,
		(arg.subscriptions ? "subscription" : "dialog"),
		ESS(arg.numchans));	/* ESS(n) returns an "s" if n>1 */
	return CLI_SUCCESS;
#undef FORMAT
#undef FORMAT2
#undef FORMAT3
}

/*! \brief Support routine for 'sip show channel' and 'sip show history' CLI
 * This is in charge of generating all strings that match a prefix in the
 * given position. As many functions of this kind, each invokation has
 * O(state) time complexity so be careful in using it.
 */
static char *complete_sipch(const char *line, const char *word, int pos, int state)
{
	int which=0;
	struct sip_pvt *cur;
	char *c = NULL;
	int wordlen = strlen(word);
	struct ao2_iterator i;

	if (pos != 3) {
		return NULL;
	}

	i = ao2_iterator_init(dialogs, 0);
	while ((cur = ao2_t_iterator_next(&i, "iterate thru dialogs"))) {
		sip_pvt_lock(cur);
		if (!strncasecmp(word, cur->callid, wordlen) && ++which > state) {
			c = ast_strdup(cur->callid);
			sip_pvt_unlock(cur);
			dialog_unref(cur, "drop ref in iterator loop break");
			break;
		}
		sip_pvt_unlock(cur);
		dialog_unref(cur, "drop ref in iterator loop");
	}
	ao2_iterator_destroy(&i);
	return c;
}


/*! \brief Do completion on peer name */
static char *complete_sip_peer(const char *word, int state, int flags2)
{
	char *result = NULL;
	int wordlen = strlen(word);
	int which = 0;
	struct ao2_iterator i = ao2_iterator_init(peers, 0);
	struct sip_peer *peer;

	while ((peer = ao2_t_iterator_next(&i, "iterate thru peers table"))) {
		/* locking of the object is not required because only the name and flags are being compared */
		if (!strncasecmp(word, peer->name, wordlen) &&
				(!flags2 || ast_test_flag(&peer->flags[1], flags2)) &&
				++which > state)
			result = ast_strdup(peer->name);
		unref_peer(peer, "toss iterator peer ptr before break");
		if (result) {
			break;
		}
	}
	ao2_iterator_destroy(&i);
	return result;
}

/*! \brief Do completion on registered peer name */
static char *complete_sip_registered_peer(const char *word, int state, int flags2)
{
       char *result = NULL;
       int wordlen = strlen(word);
       int which = 0;
       struct ao2_iterator i;
       struct sip_peer *peer;
       
       i = ao2_iterator_init(peers, 0);
       while ((peer = ao2_t_iterator_next(&i, "iterate thru peers table"))) {
	       if (!strncasecmp(word, peer->name, wordlen) &&
		   (!flags2 || ast_test_flag(&peer->flags[1], flags2)) &&
		   ++which > state && peer->expire > 0)
		       result = ast_strdup(peer->name);
	       if (result) {
		       unref_peer(peer, "toss iterator peer ptr before break");
		       break;
	       }
	       unref_peer(peer, "toss iterator peer ptr");
       }
       ao2_iterator_destroy(&i);
       return result;
}

/*! \brief Support routine for 'sip show history' CLI */
static char *complete_sip_show_history(const char *line, const char *word, int pos, int state)
{
	if (pos == 3)
		return complete_sipch(line, word, pos, state);

	return NULL;
}

/*! \brief Support routine for 'sip show peer' CLI */
static char *complete_sip_show_peer(const char *line, const char *word, int pos, int state)
{
	if (pos == 3) {
		return complete_sip_peer(word, state, 0);
	}

	return NULL;
}

/*! \brief Support routine for 'sip unregister' CLI */
static char *complete_sip_unregister(const char *line, const char *word, int pos, int state)
{
       if (pos == 2)
               return complete_sip_registered_peer(word, state, 0);

       return NULL;
}

/*! \brief Support routine for 'sip notify' CLI */
static char *complete_sipnotify(const char *line, const char *word, int pos, int state)
{
	char *c = NULL;

	if (pos == 2) {
		int which = 0;
		char *cat = NULL;
		int wordlen = strlen(word);

		/* do completion for notify type */

		if (!notify_types)
			return NULL;
		
		while ( (cat = ast_category_browse(notify_types, cat)) ) {
			if (!strncasecmp(word, cat, wordlen) && ++which > state) {
				c = ast_strdup(cat);
				break;
			}
		}
		return c;
	}

	if (pos > 2)
		return complete_sip_peer(word, state, 0);

	return NULL;
}

/*! \brief Show details of one active dialog */
static char *sip_show_channel(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	struct sip_pvt *cur;
	size_t len;
	int found = 0;
	struct ao2_iterator i;

	switch (cmd) {
	case CLI_INIT:
		e->command = "sip show channel";
		e->usage =
			"Usage: sip show channel <call-id>\n"
			"       Provides detailed status on a given SIP dialog (identified by SIP call-id).\n";
		return NULL;
	case CLI_GENERATE:
		return complete_sipch(a->line, a->word, a->pos, a->n);
	}

	if (a->argc != 4)
		return CLI_SHOWUSAGE;
	len = strlen(a->argv[3]);
	
	i = ao2_iterator_init(dialogs, 0);
	while ((cur = ao2_t_iterator_next(&i, "iterate thru dialogs"))) {
		sip_pvt_lock(cur);

		if (!strncasecmp(cur->callid, a->argv[3], len)) {
			char formatbuf[SIPBUFSIZE/2];
			ast_cli(a->fd, "\n");
			if (cur->subscribed != NONE)
				ast_cli(a->fd, "  * Subscription (type: %s)\n", subscription_type2str(cur->subscribed));
			else
				ast_cli(a->fd, "  * SIP Call\n");
			ast_cli(a->fd, "  Curr. trans. direction:  %s\n", ast_test_flag(&cur->flags[0], SIP_OUTGOING) ? "Outgoing" : "Incoming");
			ast_cli(a->fd, "  Call-ID:                %s\n", cur->callid);
			ast_cli(a->fd, "  Owner channel ID:       %s\n", cur->owner ? cur->owner->name : "<none>");
			ast_cli(a->fd, "  Our Codec Capability:   %s\n", ast_getformatname_multiple(formatbuf, sizeof(formatbuf), cur->capability));
			ast_cli(a->fd, "  Non-Codec Capability (DTMF):   %d\n", cur->noncodeccapability);
			ast_cli(a->fd, "  Their Codec Capability:   %s\n", ast_getformatname_multiple(formatbuf, sizeof(formatbuf), cur->peercapability));
			ast_cli(a->fd, "  Joint Codec Capability:   %s\n", ast_getformatname_multiple(formatbuf, sizeof(formatbuf), cur->jointcapability));
			ast_cli(a->fd, "  Format:                 %s\n", ast_getformatname_multiple(formatbuf, sizeof(formatbuf), cur->owner ? cur->owner->nativeformats : 0) );
			ast_cli(a->fd, "  T.38 support            %s\n", AST_CLI_YESNO(cur->udptl != NULL));
			ast_cli(a->fd, "  Video support           %s\n", AST_CLI_YESNO(cur->vrtp != NULL));
			ast_cli(a->fd, "  MaxCallBR:              %d kbps\n", cur->maxcallbitrate);
			ast_cli(a->fd, "  Theoretical Address:    %s\n", ast_sockaddr_stringify(&cur->sa));
			ast_cli(a->fd, "  Received Address:       %s\n", ast_sockaddr_stringify(&cur->recv));
			ast_cli(a->fd, "  SIP Transfer mode:      %s\n", transfermode2str(cur->allowtransfer));
			ast_cli(a->fd, "  Force rport:            %s\n", AST_CLI_YESNO(ast_test_flag(&cur->flags[0], SIP_NAT_FORCE_RPORT)));
			if (ast_sockaddr_isnull(&cur->redirip)) {
				ast_cli(a->fd,
					"  Audio IP:               %s (local)\n",
					ast_sockaddr_stringify_addr(&cur->ourip));
			} else {
				ast_cli(a->fd,
					"  Audio IP:               %s (Outside bridge)\n",
					ast_sockaddr_stringify_addr(&cur->redirip));
			}
			ast_cli(a->fd, "  Our Tag:                %s\n", cur->tag);
			ast_cli(a->fd, "  Their Tag:              %s\n", cur->theirtag);
			ast_cli(a->fd, "  SIP User agent:         %s\n", cur->useragent);
			if (!ast_strlen_zero(cur->username))
				ast_cli(a->fd, "  Username:               %s\n", cur->username);
			if (!ast_strlen_zero(cur->peername))
				ast_cli(a->fd, "  Peername:               %s\n", cur->peername);
			if (!ast_strlen_zero(cur->uri))
				ast_cli(a->fd, "  Original uri:           %s\n", cur->uri);
			if (!ast_strlen_zero(cur->cid_num))
				ast_cli(a->fd, "  Caller-ID:              %s\n", cur->cid_num);
			ast_cli(a->fd, "  Need Destroy:           %s\n", AST_CLI_YESNO(cur->needdestroy));
			ast_cli(a->fd, "  Last Message:           %s\n", cur->lastmsg);
			ast_cli(a->fd, "  Promiscuous Redir:      %s\n", AST_CLI_YESNO(ast_test_flag(&cur->flags[0], SIP_PROMISCREDIR)));
			ast_cli(a->fd, "  Route:                  %s\n", cur->route ? cur->route->hop : "N/A");
			ast_cli(a->fd, "  DTMF Mode:              %s\n", dtmfmode2str(ast_test_flag(&cur->flags[0], SIP_DTMF)));
			ast_cli(a->fd, "  SIP Options:            ");
			if (cur->sipoptions) {
				int x;
				for (x = 0 ; x < ARRAY_LEN(sip_options); x++) {
					if (cur->sipoptions & sip_options[x].id)
						ast_cli(a->fd, "%s ", sip_options[x].text);
				}
				ast_cli(a->fd, "\n");
			} else
				ast_cli(a->fd, "(none)\n");

			if (!cur->stimer)
 				ast_cli(a->fd, "  Session-Timer:          Uninitiallized\n");
			else {
 				ast_cli(a->fd, "  Session-Timer:          %s\n", cur->stimer->st_active ? "Active" : "Inactive");
 				if (cur->stimer->st_active == TRUE) {
 					ast_cli(a->fd, "  S-Timer Interval:       %d\n", cur->stimer->st_interval);
 					ast_cli(a->fd, "  S-Timer Refresher:      %s\n", strefresher2str(cur->stimer->st_ref));
 					ast_cli(a->fd, "  S-Timer Expirys:        %d\n", cur->stimer->st_expirys);
 					ast_cli(a->fd, "  S-Timer Sched Id:       %d\n", cur->stimer->st_schedid);
 					ast_cli(a->fd, "  S-Timer Peer Sts:       %s\n", cur->stimer->st_active_peer_ua ? "Active" : "Inactive");
 					ast_cli(a->fd, "  S-Timer Cached Min-SE:  %d\n", cur->stimer->st_cached_min_se);
 					ast_cli(a->fd, "  S-Timer Cached SE:      %d\n", cur->stimer->st_cached_max_se);
 					ast_cli(a->fd, "  S-Timer Cached Ref:     %s\n", strefresher2str(cur->stimer->st_cached_ref));
 					ast_cli(a->fd, "  S-Timer Cached Mode:    %s\n", stmode2str(cur->stimer->st_cached_mode));
 				}
			}

			ast_cli(a->fd, "\n\n");

			found++;
		}

		sip_pvt_unlock(cur);

		ao2_t_ref(cur, -1, "toss dialog ptr set by iterator_next");
	}
	ao2_iterator_destroy(&i);

	if (!found)
		ast_cli(a->fd, "No such SIP Call ID starting with '%s'\n", a->argv[3]);

	return CLI_SUCCESS;
}

/*! \brief Show history details of one dialog */
static char *sip_show_history(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	struct sip_pvt *cur;
	size_t len;
	int found = 0;
	struct ao2_iterator i;

	switch (cmd) {
	case CLI_INIT:
		e->command = "sip show history";
		e->usage =
			"Usage: sip show history <call-id>\n"
			"       Provides detailed dialog history on a given SIP call (specified by call-id).\n";
		return NULL;
	case CLI_GENERATE:
		return complete_sip_show_history(a->line, a->word, a->pos, a->n);
	}

	if (a->argc != 4)
		return CLI_SHOWUSAGE;

	if (!recordhistory)
		ast_cli(a->fd, "\n***Note: History recording is currently DISABLED.  Use 'sip set history on' to ENABLE.\n");

	len = strlen(a->argv[3]);

	i = ao2_iterator_init(dialogs, 0);
	while ((cur = ao2_t_iterator_next(&i, "iterate thru dialogs"))) {
		sip_pvt_lock(cur);
		if (!strncasecmp(cur->callid, a->argv[3], len)) {
			struct sip_history *hist;
			int x = 0;

			ast_cli(a->fd, "\n");
			if (cur->subscribed != NONE)
				ast_cli(a->fd, "  * Subscription\n");
			else
				ast_cli(a->fd, "  * SIP Call\n");
			if (cur->history)
				AST_LIST_TRAVERSE(cur->history, hist, list)
					ast_cli(a->fd, "%d. %s\n", ++x, hist->event);
			if (x == 0)
				ast_cli(a->fd, "Call '%s' has no history\n", cur->callid);
			found++;
		}
		sip_pvt_unlock(cur);
		ao2_t_ref(cur, -1, "toss dialog ptr from iterator_next");
	}
	ao2_iterator_destroy(&i);

	if (!found)
		ast_cli(a->fd, "No such SIP Call ID starting with '%s'\n", a->argv[3]);

	return CLI_SUCCESS;
}

/*! \brief Dump SIP history to debug log file at end of lifespan for SIP dialog */
static void sip_dump_history(struct sip_pvt *dialog)
{
	int x = 0;
	struct sip_history *hist;
	static int errmsg = 0;

	if (!dialog)
		return;

	if (!option_debug && !sipdebug) {
		if (!errmsg) {
			ast_log(LOG_NOTICE, "You must have debugging enabled (SIP or Asterisk) in order to dump SIP history.\n");
			errmsg = 1;
		}
		return;
	}

	ast_debug(1, "\n---------- SIP HISTORY for '%s' \n", dialog->callid);
	if (dialog->subscribed)
		ast_debug(1, "  * Subscription\n");
	else
		ast_debug(1, "  * SIP Call\n");
	if (dialog->history)
		AST_LIST_TRAVERSE(dialog->history, hist, list)
			ast_debug(1, "  %-3.3d. %s\n", ++x, hist->event);
	if (!x)
		ast_debug(1, "Call '%s' has no history\n", dialog->callid);
	ast_debug(1, "\n---------- END SIP HISTORY for '%s' \n", dialog->callid);
}


/*! \brief  Receive SIP INFO Message */
static void handle_request_info(struct sip_pvt *p, struct sip_request *req)
{
	char buf[1024] = "";
	unsigned int event;
	const char *c = get_header(req, "Content-Type");

	/* Need to check the media/type */
	if (!strcasecmp(c, "application/dtmf-relay") ||
	    !strcasecmp(c, "application/vnd.nortelnetworks.digits") ||
	    !strcasecmp(c, "application/dtmf")) {
		unsigned int duration = 0;

		if (!p->owner) {	/* not a PBX call */
			transmit_response(p, "481 Call leg/transaction does not exist", req);
			sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
			return;
		}

		/* If dtmf-relay or vnd.nortelnetworks.digits, parse the signal and duration;
		 * otherwise use the body as the signal */
		if (strcasecmp(c, "application/dtmf")) {
			const char *msg_body;

			if (   ast_strlen_zero(msg_body = get_body(req, "Signal", '='))
				&& ast_strlen_zero(msg_body = get_body(req, "d", '='))) {
				ast_log(LOG_WARNING, "Unable to retrieve DTMF signal for INFO message on "
						"call %s\n", p->callid);
				transmit_response(p, "200 OK", req);
				return;
			}
			ast_copy_string(buf, msg_body, sizeof(buf));

			if (!ast_strlen_zero((msg_body = get_body(req, "Duration", '=')))) {
				sscanf(msg_body, "%30u", &duration);
			}
		} else {
			/* Type is application/dtmf, simply use what's in the message body */
			get_msg_text(buf, sizeof(buf), req);
		}

		/* An empty message body requires us to send a 200 OK */
		if (ast_strlen_zero(buf)) {
			transmit_response(p, "200 OK", req);
			return;
		}

		if (!duration) {
			duration = 100; /* 100 ms */
		}

		if (buf[0] == '*') {
			event = 10;
		} else if (buf[0] == '#') {
			event = 11;
		} else if (buf[0] == '!') {
			event = 16;
		} else if ('A' <= buf[0] && buf[0] <= 'D') {
			event = 12 + buf[0] - 'A';
		} else if ('a' <= buf[0] && buf[0] <= 'd') {
			event = 12 + buf[0] - 'a';
		} else if ((sscanf(buf, "%30u", &event) != 1) || event > 16) {
			ast_log(AST_LOG_WARNING, "Unable to convert DTMF event signal code to a valid "
					"value for INFO message on call %s\n", p->callid);
			transmit_response(p, "200 OK", req);
			return;
		}

		if (event == 16) {
			/* send a FLASH event */
			struct ast_frame f = { AST_FRAME_CONTROL, { AST_CONTROL_FLASH, } };
			ast_queue_frame(p->owner, &f);
			if (sipdebug) {
				ast_verbose("* DTMF-relay event received: FLASH\n");
			}
		} else {
			/* send a DTMF event */
			struct ast_frame f = { AST_FRAME_DTMF, };
			if (event < 10) {
				f.subclass.integer = '0' + event;
			} else if (event == 10) {
				f.subclass.integer = '*';
			} else if (event == 11) {
				f.subclass.integer = '#';
			} else {
				f.subclass.integer = 'A' + (event - 12);
			}
			f.len = duration;
			ast_queue_frame(p->owner, &f);
			if (sipdebug) {
				ast_verbose("* DTMF-relay event received: %c\n", (int) f.subclass.integer);
			}
		}
		transmit_response(p, "200 OK", req);
		return;
	} else if (!strcasecmp(c, "application/media_control+xml")) {
		/* Eh, we'll just assume it's a fast picture update for now */
		if (p->owner)
			ast_queue_control(p->owner, AST_CONTROL_VIDUPDATE);
		transmit_response(p, "200 OK", req);
		return;
	} else if (!ast_strlen_zero(c = get_header(req, "X-ClientCode"))) {
		/* Client code (from SNOM phone) */
		if (ast_test_flag(&p->flags[0], SIP_USECLIENTCODE)) {
			if (p->owner && p->owner->cdr)
				ast_cdr_setuserfield(p->owner, c);
			if (p->owner && ast_bridged_channel(p->owner) && ast_bridged_channel(p->owner)->cdr)
				ast_cdr_setuserfield(ast_bridged_channel(p->owner), c);
			transmit_response(p, "200 OK", req);
		} else {
			transmit_response(p, "403 Forbidden", req);
		}
		return;
	} else if (!ast_strlen_zero(c = get_header(req, "Record"))) {
		/* INFO messages generated by some phones to start/stop recording
			on phone calls.
			OEJ: I think this should be something that is enabled/disabled
			per device. I don't want incoming callers to record calls in my
			pbx.
		*/
		
		struct ast_call_feature *feat;
		int j;
		struct ast_frame f = { AST_FRAME_DTMF, };

		if (!p->owner) {        /* not a PBX call */
			transmit_response(p, "481 Call leg/transaction does not exist", req);
			sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
			return;
		}

		/* first, get the feature string, if it exists */
		ast_rdlock_call_features();
		feat = ast_find_call_feature("automon");
		if (!feat || ast_strlen_zero(feat->exten)) {
			ast_log(LOG_WARNING, "Recording requested, but no One Touch Monitor registered. (See features.conf)\n");
			/* 403 means that we don't support this feature, so don't request it again */
			transmit_response(p, "403 Forbidden", req);
			ast_unlock_call_features();
			return;
		}
		/* Send the feature code to the PBX as DTMF, just like the handset had sent it */
		f.len = 100;
		for (j=0; j < strlen(feat->exten); j++) {
			f.subclass.integer = feat->exten[j];
			ast_queue_frame(p->owner, &f);
			if (sipdebug)
				ast_verbose("* DTMF-relay event faked: %c\n", f.subclass.integer);
		}
		ast_unlock_call_features();

		ast_debug(1, "Got a Request to Record the channel, state %s\n", c);
		transmit_response(p, "200 OK", req);
		return;
	} else if (ast_strlen_zero(c = get_header(req, "Content-Length")) || !strcasecmp(c, "0")) {
		/* This is probably just a packet making sure the signalling is still up, just send back a 200 OK */
		transmit_response(p, "200 OK", req);
		return;
	}

	/* Other type of INFO message, not really understood by Asterisk */
	/* if (get_msg_text(buf, sizeof(buf), req)) { */

	ast_log(LOG_WARNING, "Unable to parse INFO message from %s. Content %s\n", p->callid, buf);
	transmit_response(p, "415 Unsupported media type", req);
	return;
}

/*! \brief Enable SIP Debugging for a single IP */
static char *sip_do_debug_ip(int fd, const char *arg)
{
	if (ast_sockaddr_resolve_first(&debugaddr, arg, 0)) {
		return CLI_SHOWUSAGE;
	}

	ast_cli(fd, "SIP Debugging Enabled for IP: %s\n", ast_sockaddr_stringify_addr(&debugaddr));
	sipdebug |= sip_debug_console;

	return CLI_SUCCESS;
}

/*! \brief  Turn on SIP debugging for a given peer */
static char *sip_do_debug_peer(int fd, const char *arg)
{
	struct sip_peer *peer = find_peer(arg, NULL, TRUE, FINDPEERS, FALSE, 0);
	if (!peer)
		ast_cli(fd, "No such peer '%s'\n", arg);
	else if (ast_sockaddr_isnull(&peer->addr))
		ast_cli(fd, "Unable to get IP address of peer '%s'\n", arg);
	else {
		ast_sockaddr_copy(&debugaddr, &peer->addr);
		ast_cli(fd, "SIP Debugging Enabled for IP: %s\n", ast_sockaddr_stringify_addr(&debugaddr));
		sipdebug |= sip_debug_console;
	}
	if (peer)
		unref_peer(peer, "sip_do_debug_peer: unref_peer, from find_peer call");
	return CLI_SUCCESS;
}

/*! \brief Turn on SIP debugging (CLI command) */
static char *sip_do_debug(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int oldsipdebug = sipdebug & sip_debug_console;
	const char *what;

	if (cmd == CLI_INIT) {
		e->command = "sip set debug {on|off|ip|peer}";
		e->usage =
			"Usage: sip set debug {off|on|ip addr[:port]|peer peername}\n"
			"       Globally disables dumping of SIP packets,\n"
			"       or enables it either globally or for a (single)\n"
			"       IP address or registered peer.\n";
		return NULL;
	} else if (cmd == CLI_GENERATE) {
		if (a->pos == 4 && !strcasecmp(a->argv[3], "peer"))
			return complete_sip_peer(a->word, a->n, 0);
		return NULL;
        }

	what = a->argv[e->args-1];      /* guaranteed to exist */
	if (a->argc == e->args) {       /* on/off */
		if (!strcasecmp(what, "on")) {
			sipdebug |= sip_debug_console;
			sipdebug_text = 1;	/*! \note this can be a special debug command - "sip debug text" or something */
			memset(&debugaddr, 0, sizeof(debugaddr));
			ast_cli(a->fd, "SIP Debugging %senabled\n", oldsipdebug ? "re-" : "");
			return CLI_SUCCESS;
		} else if (!strcasecmp(what, "off")) {
			sipdebug &= ~sip_debug_console;
			sipdebug_text = 0;
			ast_cli(a->fd, "SIP Debugging Disabled\n");
			return CLI_SUCCESS;
		}
	} else if (a->argc == e->args +1) {/* ip/peer */
		if (!strcasecmp(what, "ip"))
			return sip_do_debug_ip(a->fd, a->argv[e->args]);
		else if (!strcasecmp(what, "peer"))
			return sip_do_debug_peer(a->fd, a->argv[e->args]);
	}
	return CLI_SHOWUSAGE;   /* default, failure */
}

/*! \brief Cli command to send SIP notify to peer */
static char *sip_cli_notify(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	struct ast_variable *varlist;
	int i;

	switch (cmd) {
	case CLI_INIT:
		e->command = "sip notify";
		e->usage =
			"Usage: sip notify <type> <peer> [<peer>...]\n"
			"       Send a NOTIFY message to a SIP peer or peers\n"
			"       Message types are defined in sip_notify.conf\n";
		return NULL;
	case CLI_GENERATE:
		return complete_sipnotify(a->line, a->word, a->pos, a->n);
	}

	if (a->argc < 4)
		return CLI_SHOWUSAGE;

	if (!notify_types) {
		ast_cli(a->fd, "No %s file found, or no types listed there\n", notify_config);
		return CLI_FAILURE;
	}

	varlist = ast_variable_browse(notify_types, a->argv[2]);

	if (!varlist) {
		ast_cli(a->fd, "Unable to find notify type '%s'\n", a->argv[2]);
		return CLI_FAILURE;
	}

	for (i = 3; i < a->argc; i++) {
		struct sip_pvt *p;
		char buf[512];
		struct ast_variable *header, *var;

		if (!(p = sip_alloc(NULL, NULL, 0, SIP_NOTIFY, NULL))) {
			ast_log(LOG_WARNING, "Unable to build sip pvt data for notify (memory/socket error)\n");
			return CLI_FAILURE;
		}

		if (create_addr(p, a->argv[i], NULL, 1, NULL)) {
			/* Maybe they're not registered, etc. */
			dialog_unlink_all(p);
			dialog_unref(p, "unref dialog inside for loop" );
			/* sip_destroy(p); */
			ast_cli(a->fd, "Could not create address for '%s'\n", a->argv[i]);
			continue;
		}

		/* Notify is outgoing call */
		ast_set_flag(&p->flags[0], SIP_OUTGOING);
		sip_notify_allocate(p);
		p->notify->headers = header = ast_variable_new("Subscription-State", "terminated", "");

		for (var = varlist; var; var = var->next) {
			ast_copy_string(buf, var->value, sizeof(buf));
			ast_unescape_semicolon(buf);

			if (!strcasecmp(var->name, "Content")) {
				if (ast_str_strlen(p->notify->content))
					ast_str_append(&p->notify->content, 0, "\r\n");
				ast_str_append(&p->notify->content, 0, "%s", buf);
			} else if (!strcasecmp(var->name, "Content-Length")) {
				ast_log(LOG_WARNING, "it is not necessary to specify Content-Length in sip_notify.conf, ignoring\n");
			} else {
				header->next = ast_variable_new(var->name, buf, "");
				header = header->next;
			}
		}

		/* Recalculate our side, and recalculate Call ID */
		ast_cli(a->fd, "Sending NOTIFY of type '%s' to '%s'\n", a->argv[2], a->argv[i]);
		sip_scheddestroy(p, SIP_TRANS_TIMEOUT);
		transmit_invite(p, SIP_NOTIFY, 0, 2, NULL);
		dialog_unref(p, "bump down the count of p since we're done with it.");
	}

	return CLI_SUCCESS;
}

/*! \brief Enable/Disable SIP History logging (CLI) */
static char *sip_set_history(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	switch (cmd) {
	case CLI_INIT:
		e->command = "sip set history {on|off}";
		e->usage =
			"Usage: sip set history {on|off}\n"
			"       Enables/Disables recording of SIP dialog history for debugging purposes.\n"
			"       Use 'sip show history' to view the history of a call number.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc != e->args)
		return CLI_SHOWUSAGE;

	if (!strncasecmp(a->argv[e->args - 1], "on", 2)) {
		recordhistory = TRUE;
		ast_cli(a->fd, "SIP History Recording Enabled (use 'sip show history')\n");
	} else if (!strncasecmp(a->argv[e->args - 1], "off", 3)) {
		recordhistory = FALSE;
		ast_cli(a->fd, "SIP History Recording Disabled\n");
	} else {
		return CLI_SHOWUSAGE;
	}
	return CLI_SUCCESS;
}

/*! \brief Authenticate for outbound registration */
static int do_register_auth(struct sip_pvt *p, struct sip_request *req, enum sip_auth_type code)
{
	char *header, *respheader;
	char digest[1024];

	p->authtries++;
	auth_headers(code, &header, &respheader);
	memset(digest, 0, sizeof(digest));
	if (reply_digest(p, req, header, SIP_REGISTER, digest, sizeof(digest))) {
		/* There's nothing to use for authentication */
 		/* No digest challenge in request */
 		if (sip_debug_test_pvt(p) && p->registry)
 			ast_verbose("No authentication challenge, sending blank registration to domain/host name %s\n", p->registry->hostname);
 			/* No old challenge */
		return -1;
	}
	if (p->do_history)
		append_history(p, "RegistryAuth", "Try: %d", p->authtries);
 	if (sip_debug_test_pvt(p) && p->registry)
 		ast_verbose("Responding to challenge, registration to domain/host name %s\n", p->registry->hostname);
	return transmit_register(p->registry, SIP_REGISTER, digest, respheader);
}

/*! \brief Add authentication on outbound SIP packet */
static int do_proxy_auth(struct sip_pvt *p, struct sip_request *req, enum sip_auth_type code, int sipmethod, int init)
{
	char *header, *respheader;
	char digest[1024];

	if (!p->options && !(p->options = ast_calloc(1, sizeof(*p->options))))
		return -2;

	p->authtries++;
	auth_headers(code, &header, &respheader);
	ast_debug(2, "Auth attempt %d on %s\n", p->authtries, sip_methods[sipmethod].text);
	memset(digest, 0, sizeof(digest));
	if (reply_digest(p, req, header, sipmethod, digest, sizeof(digest) )) {
		/* No way to authenticate */
		return -1;
	}
	/* Now we have a reply digest */
	p->options->auth = digest;
	p->options->authheader = respheader;
	return transmit_invite(p, sipmethod, sipmethod == SIP_INVITE, init, NULL);
}

/*! \brief  reply to authentication for outbound registrations
\return	Returns -1 if we have no auth
\note	This is used for register= servers in sip.conf, SIP proxies we register
	with  for receiving calls from.  */
static int reply_digest(struct sip_pvt *p, struct sip_request *req, char *header, int sipmethod,  char *digest, int digest_len)
{
	char tmp[512];
	char *c;
	char oldnonce[256];

	/* table of recognised keywords, and places where they should be copied */
	const struct x {
		const char *key;
		const ast_string_field *field;
	} *i, keys[] = {
		{ "realm=", &p->realm },
		{ "nonce=", &p->nonce },
		{ "opaque=", &p->opaque },
		{ "qop=", &p->qop },
		{ "domain=", &p->domain },
		{ NULL, 0 },
	};

	ast_copy_string(tmp, get_header(req, header), sizeof(tmp));
	if (ast_strlen_zero(tmp))
		return -1;
	if (strncasecmp(tmp, "Digest ", strlen("Digest "))) {
		ast_log(LOG_WARNING, "missing Digest.\n");
		return -1;
	}
	c = tmp + strlen("Digest ");
	ast_copy_string(oldnonce, p->nonce, sizeof(oldnonce));
	while (c && *(c = ast_skip_blanks(c))) {	/* lookup for keys */
		for (i = keys; i->key != NULL; i++) {
			char *src, *separator;
			if (strncasecmp(c, i->key, strlen(i->key)) != 0)
				continue;
			/* Found. Skip keyword, take text in quotes or up to the separator. */
			c += strlen(i->key);
			if (*c == '"') {
				src = ++c;
				separator = "\"";
			} else {
				src = c;
				separator = ",";
			}
			strsep(&c, separator); /* clear separator and move ptr */
			ast_string_field_ptr_set(p, i->field, src);
			break;
		}
		if (i->key == NULL) /* not found, try ',' */
			strsep(&c, ",");
	}
	/* Reset nonce count */
	if (strcmp(p->nonce, oldnonce))
		p->noncecount = 0;

	/* Save auth data for following registrations */
	if (p->registry) {
		struct sip_registry *r = p->registry;

		if (strcmp(r->nonce, p->nonce)) {
			ast_string_field_set(r, realm, p->realm);
			ast_string_field_set(r, nonce, p->nonce);
			ast_string_field_set(r, authdomain, p->domain);
			ast_string_field_set(r, opaque, p->opaque);
			ast_string_field_set(r, qop, p->qop);
			r->noncecount = 0;
		}
	}
	return build_reply_digest(p, sipmethod, digest, digest_len);
}

/*! \brief  Build reply digest
\return	Returns -1 if we have no auth
\note	Build digest challenge for authentication of registrations and calls
	Also used for authentication of BYE
*/
static int build_reply_digest(struct sip_pvt *p, int method, char* digest, int digest_len)
{
	char a1[256];
	char a2[256];
	char a1_hash[256];
	char a2_hash[256];
	char resp[256];
	char resp_hash[256];
	char uri[256];
	char opaque[256] = "";
	char cnonce[80];
	const char *username;
	const char *secret;
	const char *md5secret;
	struct sip_auth *auth;	/* Realm authentication credential */
	struct sip_auth_container *credentials;

	if (!ast_strlen_zero(p->domain))
		snprintf(uri, sizeof(uri), "%s:%s", p->socket.type == SIP_TRANSPORT_TLS ? "sips" : "sip", p->domain);
	else if (!ast_strlen_zero(p->uri))
		ast_copy_string(uri, p->uri, sizeof(uri));
	else
		snprintf(uri, sizeof(uri), "%s:%s@%s", p->socket.type == SIP_TRANSPORT_TLS ? "sips" : "sip", p->username, ast_sockaddr_stringify_host_remote(&p->sa));

	snprintf(cnonce, sizeof(cnonce), "%08lx", ast_random());

	/* Check if we have peer credentials */
	ao2_lock(p);
	credentials = p->peerauth;
	if (credentials) {
		ao2_t_ref(credentials, +1, "Ref peer auth for digest");
	}
	ao2_unlock(p);
	auth = find_realm_authentication(credentials, p->realm);
	if (!auth) {
		/* If not, check global credentials */
		if (credentials) {
			ao2_t_ref(credentials, -1, "Unref peer auth for digest");
		}
		ast_mutex_lock(&authl_lock);
		credentials = authl;
		if (credentials) {
			ao2_t_ref(credentials, +1, "Ref global auth for digest");
		}
		ast_mutex_unlock(&authl_lock);
		auth = find_realm_authentication(credentials, p->realm);
	}

 	if (auth) {
		ast_debug(3, "use realm [%s] from peer [%s][%s]\n", auth->username, p->peername, p->username);
 		username = auth->username;
 		secret = auth->secret;
 		md5secret = auth->md5secret;
		if (sipdebug)
 			ast_debug(1, "Using realm %s authentication for call %s\n", p->realm, p->callid);
 	} else {
 		/* No authentication, use peer or register= config */
 		username = p->authname;
 		secret = p->relatedpeer 
			&& !ast_strlen_zero(p->relatedpeer->remotesecret)
				? p->relatedpeer->remotesecret : p->peersecret;
 		md5secret = p->peermd5secret;
 	}
	if (ast_strlen_zero(username)) {
		/* We have no authentication */
		if (credentials) {
			ao2_t_ref(credentials, -1, "Unref auth for digest");
		}
		return -1;
	}

 	/* Calculate SIP digest response */
 	snprintf(a1, sizeof(a1), "%s:%s:%s", username, p->realm, secret);
	snprintf(a2, sizeof(a2), "%s:%s", sip_methods[method].text, uri);
	if (!ast_strlen_zero(md5secret))
		ast_copy_string(a1_hash, md5secret, sizeof(a1_hash));
	else
		ast_md5_hash(a1_hash, a1);
	ast_md5_hash(a2_hash, a2);

	p->noncecount++;
	if (!ast_strlen_zero(p->qop))
		snprintf(resp, sizeof(resp), "%s:%s:%08x:%s:%s:%s", a1_hash, p->nonce, p->noncecount, cnonce, "auth", a2_hash);
	else
		snprintf(resp, sizeof(resp), "%s:%s:%s", a1_hash, p->nonce, a2_hash);
	ast_md5_hash(resp_hash, resp);

	/* only include the opaque string if it's set */
	if (!ast_strlen_zero(p->opaque)) {
		snprintf(opaque, sizeof(opaque), ", opaque=\"%s\"", p->opaque);
	}

	/* XXX We hard code our qop to "auth" for now.  XXX */
	if (!ast_strlen_zero(p->qop))
		snprintf(digest, digest_len, "Digest username=\"%s\", realm=\"%s\", algorithm=MD5, uri=\"%s\", nonce=\"%s\", response=\"%s\"%s, qop=auth, cnonce=\"%s\", nc=%08x", username, p->realm, uri, p->nonce, resp_hash, opaque, cnonce, p->noncecount);
	else
		snprintf(digest, digest_len, "Digest username=\"%s\", realm=\"%s\", algorithm=MD5, uri=\"%s\", nonce=\"%s\", response=\"%s\"%s", username, p->realm, uri, p->nonce, resp_hash, opaque);

	append_history(p, "AuthResp", "Auth response sent for %s in realm %s - nc %d", username, p->realm, p->noncecount);

	if (credentials) {
		ao2_t_ref(credentials, -1, "Unref auth for digest");
	}
	return 0;
}
	
/*! \brief Read SIP header (dialplan function) */
static int func_header_read(struct ast_channel *chan, const char *function, char *data, char *buf, size_t len)
{
	struct sip_pvt *p;
	const char *content = NULL;
	AST_DECLARE_APP_ARGS(args,
		AST_APP_ARG(header);
		AST_APP_ARG(number);
	);
	int i, number, start = 0;

 	if (ast_strlen_zero(data)) {
		ast_log(LOG_WARNING, "This function requires a header name.\n");
		return -1;
	}

	ast_channel_lock(chan);
	if (!IS_SIP_TECH(chan->tech)) {
		ast_log(LOG_WARNING, "This function can only be used on SIP channels.\n");
		ast_channel_unlock(chan);
		return -1;
	}

	AST_STANDARD_APP_ARGS(args, data);
	if (!args.number) {
		number = 1;
	} else {
		sscanf(args.number, "%30d", &number);
		if (number < 1)
			number = 1;
	}

	p = chan->tech_pvt;

	/* If there is no private structure, this channel is no longer alive */
	if (!p) {
		ast_channel_unlock(chan);
		return -1;
	}

	for (i = 0; i < number; i++)
		content = __get_header(&p->initreq, args.header, &start);

	if (ast_strlen_zero(content)) {
		ast_channel_unlock(chan);
		return -1;
	}

	ast_copy_string(buf, content, len);
	ast_channel_unlock(chan);

	return 0;
}

static struct ast_custom_function sip_header_function = {
	.name = "SIP_HEADER",
	.read = func_header_read,
};

/*! \brief  Dial plan function to check if domain is local */
static int func_check_sipdomain(struct ast_channel *chan, const char *cmd, char *data, char *buf, size_t len)
{
	if (ast_strlen_zero(data)) {
		ast_log(LOG_WARNING, "CHECKSIPDOMAIN requires an argument - A domain name\n");
		return -1;
	}
	if (check_sip_domain(data, NULL, 0))
		ast_copy_string(buf, data, len);
	else
		buf[0] = '\0';
	return 0;
}

static struct ast_custom_function checksipdomain_function = {
	.name = "CHECKSIPDOMAIN",
	.read = func_check_sipdomain,
};

/*! \brief  ${SIPPEER()} Dialplan function - reads peer data */
static int function_sippeer(struct ast_channel *chan, const char *cmd, char *data, char *buf, size_t len)
{
	struct sip_peer *peer;
	char *colname;

	if ((colname = strchr(data, ':'))) {	/*! \todo Will be deprecated after 1.4 */
		static int deprecation_warning = 0;
		*colname++ = '\0';
		if (deprecation_warning++ % 10 == 0)
			ast_log(LOG_WARNING, "SIPPEER(): usage of ':' to separate arguments is deprecated.  Please use ',' instead.\n");
	} else if ((colname = strchr(data, ',')))
		*colname++ = '\0';
	else
		colname = "ip";

	if (!(peer = find_peer(data, NULL, TRUE, FINDPEERS, FALSE, 0)))
		return -1;

	if (!strcasecmp(colname, "ip")) {
		ast_copy_string(buf, ast_sockaddr_stringify_addr(&peer->addr), len);
	} else  if (!strcasecmp(colname, "port")) {
		snprintf(buf, len, "%d", ast_sockaddr_port(&peer->addr));
	} else  if (!strcasecmp(colname, "status")) {
		peer_status(peer, buf, len);
	} else  if (!strcasecmp(colname, "language")) {
		ast_copy_string(buf, peer->language, len);
	} else  if (!strcasecmp(colname, "regexten")) {
		ast_copy_string(buf, peer->regexten, len);
	} else  if (!strcasecmp(colname, "limit")) {
		snprintf(buf, len, "%d", peer->call_limit);
	} else  if (!strcasecmp(colname, "busylevel")) {
		snprintf(buf, len, "%d", peer->busy_level);
	} else  if (!strcasecmp(colname, "curcalls")) {
		snprintf(buf, len, "%d", peer->inUse);
	} else if (!strcasecmp(colname, "maxforwards")) {
		snprintf(buf, len, "%d", peer->maxforwards);
	} else  if (!strcasecmp(colname, "accountcode")) {
		ast_copy_string(buf, peer->accountcode, len);
	} else  if (!strcasecmp(colname, "callgroup")) {
		ast_print_group(buf, len, peer->callgroup);
	} else  if (!strcasecmp(colname, "pickupgroup")) {
		ast_print_group(buf, len, peer->pickupgroup);
	} else  if (!strcasecmp(colname, "useragent")) {
		ast_copy_string(buf, peer->useragent, len);
	} else  if (!strcasecmp(colname, "mailbox")) {
		struct ast_str *mailbox_str = ast_str_alloca(512);
		peer_mailboxes_to_str(&mailbox_str, peer);
		ast_copy_string(buf, mailbox_str->str, len);
	} else  if (!strcasecmp(colname, "context")) {
		ast_copy_string(buf, peer->context, len);
	} else  if (!strcasecmp(colname, "expire")) {
		snprintf(buf, len, "%d", peer->expire);
	} else  if (!strcasecmp(colname, "dynamic")) {
		ast_copy_string(buf, peer->host_dynamic ? "yes" : "no", len);
	} else  if (!strcasecmp(colname, "callerid_name")) {
		ast_copy_string(buf, peer->cid_name, len);
	} else  if (!strcasecmp(colname, "callerid_num")) {
		ast_copy_string(buf, peer->cid_num, len);
	} else  if (!strcasecmp(colname, "codecs")) {
		ast_getformatname_multiple(buf, len -1, peer->capability);
	} else if (!strcasecmp(colname, "encryption")) {
		snprintf(buf, len, "%d", ast_test_flag(&peer->flags[1], SIP_PAGE2_USE_SRTP));
	} else  if (!strncasecmp(colname, "chanvar[", 8)) {
		char *chanvar=colname + 8;
		struct ast_variable *v;
	
		chanvar = strsep(&chanvar, "]");
		for (v = peer->chanvars ; v ; v = v->next) {
			if (!strcasecmp(v->name, chanvar)) {
				ast_copy_string(buf, v->value, len);
			}
		}
	} else  if (!strncasecmp(colname, "codec[", 6)) {
		char *codecnum;
		format_t codec = 0;
		
		codecnum = colname + 6;	/* move past the '[' */
		codecnum = strsep(&codecnum, "]"); /* trim trailing ']' if any */
		if((codec = ast_codec_pref_index(&peer->prefs, atoi(codecnum)))) {
			ast_copy_string(buf, ast_getformatname(codec), len);
		} else {
			buf[0] = '\0';
		}
	} else {
		buf[0] = '\0';
	}

	unref_peer(peer, "unref_peer from function_sippeer, just before return");

	return 0;
}

/*! \brief Structure to declare a dialplan function: SIPPEER */
static struct ast_custom_function sippeer_function = {
	.name = "SIPPEER",
	.read = function_sippeer,
};

/*! \brief ${SIPCHANINFO()} Dialplan function - reads sip channel data */
static int function_sipchaninfo_read(struct ast_channel *chan, const char *cmd, char *data, char *buf, size_t len)
{
	struct sip_pvt *p;
	static int deprecated = 0;

	*buf = 0;
	
 	if (!data) {
		ast_log(LOG_WARNING, "This function requires a parameter name.\n");
		return -1;
	}

	ast_channel_lock(chan);
	if (!IS_SIP_TECH(chan->tech)) {
		ast_log(LOG_WARNING, "This function can only be used on SIP channels.\n");
		ast_channel_unlock(chan);
		return -1;
	}

	if (deprecated++ % 20 == 0) {
		/* Deprecated in 1.6.1 */
		ast_log(LOG_WARNING, "SIPCHANINFO() is deprecated.  Please transition to using CHANNEL().\n");
	}

	p = chan->tech_pvt;

	/* If there is no private structure, this channel is no longer alive */
	if (!p) {
		ast_channel_unlock(chan);
		return -1;
	}

	if (!strcasecmp(data, "peerip")) {
		ast_copy_string(buf, ast_sockaddr_stringify_addr(&p->sa), len);
	} else  if (!strcasecmp(data, "recvip")) {
		ast_copy_string(buf, ast_sockaddr_stringify_addr(&p->recv), len);
	} else  if (!strcasecmp(data, "from")) {
		ast_copy_string(buf, p->from, len);
	} else  if (!strcasecmp(data, "uri")) {
		ast_copy_string(buf, p->uri, len);
	} else  if (!strcasecmp(data, "useragent")) {
		ast_copy_string(buf, p->useragent, len);
	} else  if (!strcasecmp(data, "peername")) {
		ast_copy_string(buf, p->peername, len);
	} else if (!strcasecmp(data, "t38passthrough")) {
		if (p->t38.state == T38_DISABLED) {
			ast_copy_string(buf, "0", len);
		} else { /* T38 is offered or enabled in this call */
			ast_copy_string(buf, "1", len);
		}
	} else {
		ast_channel_unlock(chan);
		return -1;
	}
	ast_channel_unlock(chan);

	return 0;
}

/*! \brief Structure to declare a dialplan function: SIPCHANINFO */
static struct ast_custom_function sipchaninfo_function = {
	.name = "SIPCHANINFO",
	.read = function_sipchaninfo_read,
};

/*! \brief update redirecting information for a channel based on headers
 *
 */
static void change_redirecting_information(struct sip_pvt *p, struct sip_request *req,
	struct ast_party_redirecting *redirecting,
	struct ast_set_party_redirecting *update_redirecting, int set_call_forward)
{
	char *redirecting_from_name = NULL;
	char *redirecting_from_number = NULL;
	char *redirecting_to_name = NULL;
	char *redirecting_to_number = NULL;
	int reason = AST_REDIRECTING_REASON_UNCONDITIONAL;
	int is_response = req->method == SIP_RESPONSE;
	int res = 0;

	res = get_rdnis(p, req, &redirecting_from_name, &redirecting_from_number, &reason);
	if (res == -1) {
		if (is_response) {
			get_name_and_number(get_header(req, "TO"), &redirecting_from_name, &redirecting_from_number);
		} else {
			return;
		}
	}

	/* At this point, all redirecting "from" info should be filled in appropriately
	 * on to the "to" info
	 */

	if (is_response) {
		parse_moved_contact(p, req, &redirecting_to_name, &redirecting_to_number, set_call_forward);
	} else {
		get_name_and_number(get_header(req, "TO"), &redirecting_to_name, &redirecting_to_number);
	}

	if (!ast_strlen_zero(redirecting_from_number)) {
		ast_debug(3, "Got redirecting from number %s\n", redirecting_from_number);
		update_redirecting->from.number = 1;
		redirecting->from.number.valid = 1;
		ast_free(redirecting->from.number.str);
		redirecting->from.number.str = redirecting_from_number;
	}
	if (!ast_strlen_zero(redirecting_from_name)) {
		ast_debug(3, "Got redirecting from name %s\n", redirecting_from_name);
		update_redirecting->from.name = 1;
		redirecting->from.name.valid = 1;
		ast_free(redirecting->from.name.str);
		redirecting->from.name.str = redirecting_from_name;
	}
	if (!ast_strlen_zero(p->cid_tag)) {
		ast_free(redirecting->from.tag);
		redirecting->from.tag = ast_strdup(p->cid_tag);
		ast_free(redirecting->to.tag);
		redirecting->to.tag = ast_strdup(p->cid_tag);
	}
	if (!ast_strlen_zero(redirecting_to_number)) {
		ast_debug(3, "Got redirecting to number %s\n", redirecting_to_number);
		update_redirecting->to.number = 1;
		redirecting->to.number.valid = 1;
		ast_free(redirecting->to.number.str);
		redirecting->to.number.str = redirecting_to_number;
	}
	if (!ast_strlen_zero(redirecting_to_name)) {
		ast_debug(3, "Got redirecting to name %s\n", redirecting_from_number);
		update_redirecting->to.name = 1;
		redirecting->to.name.valid = 1;
		ast_free(redirecting->to.name.str);
		redirecting->to.name.str = redirecting_to_name;
	}
	redirecting->reason = reason;
}

/*! \brief Parse 302 Moved temporalily response
	\todo XXX Doesn't redirect over TLS on sips: uri's.
		If we get a redirect to a SIPS: uri, this needs to be going back to the
		dialplan (this is a request for a secure signalling path).
		Note that transport=tls is deprecated, but we need to support it on incoming requests.
*/
static void parse_moved_contact(struct sip_pvt *p, struct sip_request *req, char **name, char **number, int set_call_forward)
{
	char contact[SIPBUFSIZE];
	char *contact_name = NULL;
	char *contact_number = NULL;
	char *separator, *trans;
	char *domain;
	enum sip_transport transport = SIP_TRANSPORT_UDP;

	ast_copy_string(contact, get_header(req, "Contact"), sizeof(contact));
	if ((separator = strchr(contact, ',')))
		*separator = '\0';

	contact_number = get_in_brackets(contact);
	if ((trans = strcasestr(contact_number, ";transport="))) {
		trans += 11;

		if ((separator = strchr(trans, ';')))
			*separator = '\0';

		if (!strncasecmp(trans, "tcp", 3))
			transport = SIP_TRANSPORT_TCP;
		else if (!strncasecmp(trans, "tls", 3))
			transport = SIP_TRANSPORT_TLS;
		else {
			if (strncasecmp(trans, "udp", 3))
				ast_debug(1, "received contact with an invalid transport, '%s'\n", contact_number);
			/* This will assume UDP for all unknown transports */
			transport = SIP_TRANSPORT_UDP;
		}
	}
	contact_number = remove_uri_parameters(contact_number);

	if (p->socket.tcptls_session) {
		ao2_ref(p->socket.tcptls_session, -1);
		p->socket.tcptls_session = NULL;
	}

	set_socket_transport(&p->socket, transport);

	if (set_call_forward && ast_test_flag(&p->flags[0], SIP_PROMISCREDIR)) {
		char *host = NULL;
		if (!strncasecmp(contact_number, "sip:", 4))
			contact_number += 4;
		else if (!strncasecmp(contact_number, "sips:", 5))
			contact_number += 5;
		separator = strchr(contact_number, '/');
		if (separator)
			*separator = '\0';
		if ((host = strchr(contact_number, '@'))) {
			*host++ = '\0';
			ast_debug(2, "Found promiscuous redirection to 'SIP/%s::::%s@%s'\n", contact_number, get_transport(transport), host);
			if (p->owner)
				ast_string_field_build(p->owner, call_forward, "SIP/%s::::%s@%s", contact_number, get_transport(transport), host);
		} else {
			ast_debug(2, "Found promiscuous redirection to 'SIP/::::%s@%s'\n", get_transport(transport), contact_number);
			if (p->owner)
				ast_string_field_build(p->owner, call_forward, "SIP/::::%s@%s", get_transport(transport), contact_number);
		}
	} else {
		separator = strchr(contact, '@');
		if (separator) {
			*separator++ = '\0';
			domain = separator;
		} else {
			/* No username part */
			domain = contact;
		}
		separator = strchr(contact, '/');	/* WHEN do we hae a forward slash in the URI? */
		if (separator)
			*separator = '\0';

		if (!strncasecmp(contact_number, "sip:", 4))
			contact_number += 4;
		else if (!strncasecmp(contact_number, "sips:", 5))
			contact_number += 5;
		separator = strchr(contact_number, ';');	/* And username ; parameters? */
		if (separator)
			*separator = '\0';
		ast_uri_decode(contact_number);
		if (set_call_forward) {
			ast_debug(2, "Received 302 Redirect to extension '%s' (domain %s)\n", contact_number, domain);
			if (p->owner) {
				pbx_builtin_setvar_helper(p->owner, "SIPDOMAIN", domain);
				ast_string_field_set(p->owner, call_forward, contact_number);
			}
		}
	}

	/* We've gotten the number for the contact, now get the name */

	if (*contact == '\"') {
		contact_name = contact + 1;
		if (!(separator = (char *)find_closing_quote(contact_name, NULL))) {
			ast_log(LOG_NOTICE, "No closing quote on name in Contact header? %s\n", contact);
		}
		*separator = '\0';
	}

	if (name && !ast_strlen_zero(contact_name)) {
		*name = ast_strdup(contact_name);
	}
	if (number) {
		*number = ast_strdup(contact_number);
	}
}

/*! \brief Check pending actions on SIP call 
 *
 * \note both sip_pvt and sip_pvt's owner channel (if present)
 *  must be locked for this function.
 */
static void check_pendings(struct sip_pvt *p)
{
	if (ast_test_flag(&p->flags[0], SIP_PENDINGBYE)) {
		if (p->reinviteid > -1) {
			/* Outstanding p->reinviteid timeout, so wait... */
			return;
		} else if (p->invitestate == INV_PROCEEDING || p->invitestate == INV_EARLY_MEDIA) {
			/* if we can't BYE, then this is really a pending CANCEL */
			p->invitestate = INV_CANCELLED;
			transmit_request(p, SIP_CANCEL, p->lastinvite, XMIT_RELIABLE, FALSE);
			/* If the cancel occurred on an initial invite, cancel the pending BYE */
			if (!ast_test_flag(&p->flags[1], SIP_PAGE2_DIALOG_ESTABLISHED)) {
				ast_clear_flag(&p->flags[0], SIP_PENDINGBYE);
			}
			/* Actually don't destroy us yet, wait for the 487 on our original
			   INVITE, but do set an autodestruct just in case we never get it. */
		} else {
			/* We have a pending outbound invite, don't send something
			 * new in-transaction, unless it is a pending reinvite, then
			 * by the time we are called here, we should probably just hang up. */
			if (p->pendinginvite && !p->ongoing_reinvite)
				return;

			if (p->owner) {
				ast_softhangup_nolock(p->owner, AST_SOFTHANGUP_DEV);
			}
			/* Perhaps there is an SD change INVITE outstanding */
			transmit_request_with_auth(p, SIP_BYE, 0, XMIT_RELIABLE, TRUE);
			ast_clear_flag(&p->flags[0], SIP_PENDINGBYE);
		}
		sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
	} else if (ast_test_flag(&p->flags[0], SIP_NEEDREINVITE)) {
		/* if we can't REINVITE, hold it for later */
		if (p->pendinginvite || p->invitestate == INV_CALLING || p->invitestate == INV_PROCEEDING || p->invitestate == INV_EARLY_MEDIA || p->waitid > 0) {
			ast_debug(2, "NOT Sending pending reinvite (yet) on '%s'\n", p->callid);
		} else {
			ast_debug(2, "Sending pending reinvite on '%s'\n", p->callid);
			/* Didn't get to reinvite yet, so do it now */
			transmit_reinvite_with_sdp(p, (p->t38.state == T38_LOCAL_REINVITE ? TRUE : FALSE), FALSE);
			ast_clear_flag(&p->flags[0], SIP_NEEDREINVITE);	
		}
	}
}

/*! \brief Reset the NEEDREINVITE flag after waiting when we get 491 on a Re-invite
	to avoid race conditions between asterisk servers.
	Called from the scheduler.
*/
static int sip_reinvite_retry(const void *data)
{
	struct sip_pvt *p = (struct sip_pvt *) data;
	struct ast_channel *owner;

	sip_pvt_lock(p); /* called from schedule thread which requires a lock */
	while ((owner = p->owner) && ast_channel_trylock(owner)) {
		sip_pvt_unlock(p);
		usleep(1);
		sip_pvt_lock(p);
	}
	ast_set_flag(&p->flags[0], SIP_NEEDREINVITE);
	p->waitid = -1;
	check_pendings(p);
	sip_pvt_unlock(p);
	if (owner) {
		ast_channel_unlock(owner);
	}
	dialog_unref(p, "unref the dialog ptr from sip_reinvite_retry, because it held a dialog ptr");
	return 0;
}

/*!
 * \brief Handle authentication challenge for SIP UPDATE
 *
 * This function is only called upon the receipt of a 401/407 response to an UPDATE.
 */
static void handle_response_update(struct sip_pvt *p, int resp, const char *rest, struct sip_request *req, uint32_t seqno)
{
	if (p->options) {
		p->options->auth_type = (resp == 401 ? WWW_AUTH : PROXY_AUTH);
	}
	if ((p->authtries == MAX_AUTHTRIES) || do_proxy_auth(p, req, resp, SIP_UPDATE, 1)) {
		ast_log(LOG_NOTICE, "Failed to authenticate on UPDATE to '%s'\n", get_header(&p->initreq, "From"));
	}
}

static void cc_handle_publish_error(struct sip_pvt *pvt, const int resp, struct sip_request *req, struct sip_epa_entry *epa_entry)
{
	struct cc_epa_entry *cc_entry = epa_entry->instance_data;
	struct sip_monitor_instance *monitor_instance = ao2_callback(sip_monitor_instances, 0,
			find_sip_monitor_instance_by_suspension_entry, epa_entry);
	const char *min_expires;

	if (!monitor_instance) {
		ast_log(LOG_WARNING, "Can't find monitor_instance corresponding to epa_entry %p.\n", epa_entry);
		return;
	}

	if (resp != 423) {
		ast_cc_monitor_failed(cc_entry->core_id, monitor_instance->device_name,
				"Received error response to our PUBLISH");
		ao2_ref(monitor_instance, -1);
		return;
	}

	/* Allrighty, the other end doesn't like our Expires value. They think it's
	 * too small, so let's see if they've provided a more sensible value. If they
	 * haven't, then we'll just double our Expires value and see if they like that
	 * instead.
	 *
	 * XXX Ideally this logic could be placed into its own function so that SUBSCRIBE,
	 * PUBLISH, and REGISTER could all benefit from the same shared code.
	 */
	min_expires = get_header(req, "Min-Expires");
	if (ast_strlen_zero(min_expires)) {
		pvt->expiry *= 2;
		if (pvt->expiry < 0) {
			/* You dork! You overflowed! */
			ast_cc_monitor_failed(cc_entry->core_id, monitor_instance->device_name,
					"PUBLISH expiry overflowed");
			ao2_ref(monitor_instance, -1);
			return;
		}
	} else if (sscanf(min_expires, "%d", &pvt->expiry) != 1) {
		ast_cc_monitor_failed(cc_entry->core_id, monitor_instance->device_name,
				"Min-Expires has non-numeric value");
		ao2_ref(monitor_instance, -1);
		return;
	}
	/* At this point, we have most certainly changed pvt->expiry, so try transmitting
	 * again
	 */
	transmit_invite(pvt, SIP_PUBLISH, FALSE, 0, NULL);
	ao2_ref(monitor_instance, -1);
}

static void handle_response_publish(struct sip_pvt *p, int resp, const char *rest, struct sip_request *req, uint32_t seqno)
{
	struct sip_epa_entry *epa_entry = p->epa_entry;
	const char *etag = get_header(req, "Sip-ETag");

	ast_assert(epa_entry != NULL);

	if (resp == 401 || resp == 407) {
		ast_string_field_set(p, theirtag, NULL);
		if (p->options) {
			p->options->auth_type = (resp == 401 ? WWW_AUTH : PROXY_AUTH);
		}
		if ((p->authtries == MAX_AUTHTRIES) || do_proxy_auth(p, req, resp, SIP_PUBLISH, 0)) {
			ast_log(LOG_NOTICE, "Failed to authenticate on PUBLISH to '%s'\n", get_header(&p->initreq, "From"));
			pvt_set_needdestroy(p, "Failed to authenticate on PUBLISH");
			sip_alreadygone(p);
		}
		return;
	}

	if (resp == 501 || resp == 405) {
		mark_method_unallowed(&p->allowed_methods, SIP_PUBLISH);
	}

	if (resp == 200) {
		p->authtries = 0;
		/* If I've read section 6, item 6 of RFC 3903 correctly,
		 * an ESC will only generate a new etag when it sends a 200 OK
		 */
		if (!ast_strlen_zero(etag)) {
			ast_copy_string(epa_entry->entity_tag, etag, sizeof(epa_entry->entity_tag));
		}
		/* The nominal case. Everything went well. Everybody is happy.
		 * Each EPA will have a specific action to take as a result of this
		 * development, so ... callbacks!
		 */
		if (epa_entry->static_data->handle_ok) {
			epa_entry->static_data->handle_ok(p, req, epa_entry);
		}
	} else {
		/* Rather than try to make individual callbacks for each error
		 * type, there is just a single error callback. The callback
		 * can distinguish between error messages and do what it needs to
		 */
		if (epa_entry->static_data->handle_error) {
			epa_entry->static_data->handle_error(p, resp, req, epa_entry);
		}
	}
}

/*! \brief Handle SIP response to INVITE dialogue */
static void handle_response_invite(struct sip_pvt *p, int resp, const char *rest, struct sip_request *req, uint32_t seqno)
{
	int outgoing = ast_test_flag(&p->flags[0], SIP_OUTGOING);
	int res = 0;
	int xmitres = 0;
	int reinvite = ast_test_flag(&p->flags[1], SIP_PAGE2_DIALOG_ESTABLISHED);
	char *p_hdrval;
	int rtn;
	struct ast_party_connected_line connected;
	struct ast_set_party_connected_line update_connected;

	if (reinvite)
		ast_debug(4, "SIP response %d to RE-invite on %s call %s\n", resp, outgoing ? "outgoing" : "incoming", p->callid);
	else
		ast_debug(4, "SIP response %d to standard invite\n", resp);

	if (p->alreadygone) { /* This call is already gone */
		ast_debug(1, "Got response on call that is already terminated: %s (ignoring)\n", p->callid);
		return;
	}

	/* Acknowledge sequence number - This only happens on INVITE from SIP-call */
	/* Don't auto congest anymore since we've gotten something useful back */
	AST_SCHED_DEL_UNREF(sched, p->initid, dialog_unref(p, "when you delete the initid sched, you should dec the refcount for the stored dialog ptr"));

	/* RFC3261 says we must treat every 1xx response (but not 100)
	   that we don't recognize as if it was 183.
	*/
	if (resp > 100 && resp < 200 && resp!=101 && resp != 180 && resp != 181 && resp != 182 && resp != 183)
		resp = 183;

 	/* Any response between 100 and 199 is PROCEEDING */
 	if (resp >= 100 && resp < 200 && p->invitestate == INV_CALLING)
 		p->invitestate = INV_PROCEEDING;

 	/* Final response, not 200 ? */
 	if (resp >= 300 && (p->invitestate == INV_CALLING || p->invitestate == INV_PROCEEDING || p->invitestate == INV_EARLY_MEDIA ))
 		p->invitestate = INV_COMPLETED;
 	
	if ((resp >= 200 && reinvite)) {
		p->ongoing_reinvite = 0;
		if (p->reinviteid > -1) {
			AST_SCHED_DEL_UNREF(sched, p->reinviteid, dialog_unref(p, "unref dialog for reinvite timeout because of a final response"));
		}
	}

	/* Final response, clear out pending invite */
	if ((resp == 200 || resp >= 300) && p->pendinginvite && seqno == p->pendinginvite) {
		p->pendinginvite = 0;
	}

	/* If this is a response to our initial INVITE, we need to set what we can use
	 * for this peer.
	 */
	if (!reinvite) {
		set_pvt_allowed_methods(p, req);
	}

	switch (resp) {
	case 100:	/* Trying */
	case 101:	/* Dialog establishment */
		if (!req->ignore && p->invitestate != INV_CANCELLED && sip_cancel_destroy(p))
			ast_log(LOG_WARNING, "Unable to cancel SIP destruction.  Expect bad things.\n");
		check_pendings(p);
		break;

	case 180:	/* 180 Ringing */
	case 182:       /* 182 Queued */
		if (!req->ignore && p->invitestate != INV_CANCELLED && sip_cancel_destroy(p))
			ast_log(LOG_WARNING, "Unable to cancel SIP destruction.  Expect bad things.\n");
		/* Store Route-set from provisional SIP responses so
		 * early-dialog request can be routed properly
		 * */
		parse_ok_contact(p, req);
		if (!reinvite) {
			build_route(p, req, 1, resp);
		}
		if (!req->ignore && p->owner) {
			if (get_rpid(p, req)) {
				/* Queue a connected line update */
				ast_party_connected_line_init(&connected);
				memset(&update_connected, 0, sizeof(update_connected));

				update_connected.id.number = 1;
				connected.id.number.valid = 1;
				connected.id.number.str = (char *) p->cid_num;
				connected.id.number.presentation = p->callingpres;

				update_connected.id.name = 1;
				connected.id.name.valid = 1;
				connected.id.name.str = (char *) p->cid_name;
				connected.id.name.presentation = p->callingpres;

				connected.id.tag = (char *) p->cid_tag;
				connected.source = AST_CONNECTED_LINE_UPDATE_SOURCE_ANSWER;
				ast_channel_queue_connected_line_update(p->owner, &connected,
					&update_connected);
			}
			sip_handle_cc(p, req, AST_CC_CCNR);
			ast_queue_control(p->owner, AST_CONTROL_RINGING);
			if (p->owner->_state != AST_STATE_UP) {
				ast_setstate(p->owner, AST_STATE_RINGING);
			}
		}
		if (find_sdp(req)) {
			if (p->invitestate != INV_CANCELLED)
				p->invitestate = INV_EARLY_MEDIA;
			res = process_sdp(p, req, SDP_T38_NONE);
			if (!req->ignore && p->owner) {
				/* Queue a progress frame only if we have SDP in 180 or 182 */
				ast_queue_control(p->owner, AST_CONTROL_PROGRESS);
			}
			ast_rtp_instance_activate(p->rtp);
		}
		check_pendings(p);
		break;

	case 181:	/* Call Is Being Forwarded */
		if (!req->ignore && (p->invitestate != INV_CANCELLED) && sip_cancel_destroy(p))
			ast_log(LOG_WARNING, "Unable to cancel SIP destruction.  Expect bad things.\n");
		/* Store Route-set from provisional SIP responses so
		 * early-dialog request can be routed properly
		 * */
		parse_ok_contact(p, req);
		if (!reinvite) {
			build_route(p, req, 1, resp);
		}
		if (!req->ignore && p->owner) {
			struct ast_party_redirecting redirecting;
			struct ast_set_party_redirecting update_redirecting;

			ast_party_redirecting_init(&redirecting);
			memset(&update_redirecting, 0, sizeof(update_redirecting));
			change_redirecting_information(p, req, &redirecting, &update_redirecting,
				FALSE);
			ast_channel_queue_redirecting_update(p->owner, &redirecting,
				&update_redirecting);
			ast_party_redirecting_free(&redirecting);
			sip_handle_cc(p, req, AST_CC_CCNR);
		}
		check_pendings(p);
		break;

	case 183:	/* Session progress */
		if (!req->ignore && (p->invitestate != INV_CANCELLED) && sip_cancel_destroy(p))
			ast_log(LOG_WARNING, "Unable to cancel SIP destruction.  Expect bad things.\n");
		/* Store Route-set from provisional SIP responses so
		 * early-dialog request can be routed properly
		 * */
		parse_ok_contact(p, req);
		if (!reinvite) {
			build_route(p, req, 1, resp);
		}
		if (!req->ignore && p->owner) {
			if (get_rpid(p, req)) {
				/* Queue a connected line update */
				ast_party_connected_line_init(&connected);
				memset(&update_connected, 0, sizeof(update_connected));

				update_connected.id.number = 1;
				connected.id.number.valid = 1;
				connected.id.number.str = (char *) p->cid_num;
				connected.id.number.presentation = p->callingpres;

				update_connected.id.name = 1;
				connected.id.name.valid = 1;
				connected.id.name.str = (char *) p->cid_name;
				connected.id.name.presentation = p->callingpres;

				connected.id.tag = (char *) p->cid_tag;
				connected.source = AST_CONNECTED_LINE_UPDATE_SOURCE_ANSWER;
				ast_channel_queue_connected_line_update(p->owner, &connected,
					&update_connected);
			}
			sip_handle_cc(p, req, AST_CC_CCNR);
		}
		if (find_sdp(req)) {
			if (p->invitestate != INV_CANCELLED)
				p->invitestate = INV_EARLY_MEDIA;
			res = process_sdp(p, req, SDP_T38_NONE);
			if (!req->ignore && p->owner) {
				/* Queue a progress frame */
				ast_queue_control(p->owner, AST_CONTROL_PROGRESS);
			}
			ast_rtp_instance_activate(p->rtp);
		} else {
			/* Alcatel PBXs are known to send 183s with no SDP after sending
			 * a 100 Trying response. We're just going to treat this sort of thing
			 * the same as we would treat a 180 Ringing
			 */
			if (!req->ignore && p->owner) {
				ast_queue_control(p->owner, AST_CONTROL_RINGING);
			}
		}
		check_pendings(p);
		break;

	case 200:	/* 200 OK on invite - someone's answering our call */
		if (!req->ignore && (p->invitestate != INV_CANCELLED) && sip_cancel_destroy(p))
			ast_log(LOG_WARNING, "Unable to cancel SIP destruction.  Expect bad things.\n");
		p->authtries = 0;
		if (find_sdp(req)) {
			if ((res = process_sdp(p, req, SDP_T38_ACCEPT)) && !req->ignore)
				if (!reinvite) {
					/* This 200 OK's SDP is not acceptable, so we need to ack, then hangup */
					/* For re-invites, we try to recover */
					ast_set_flag(&p->flags[0], SIP_PENDINGBYE);
				}
			ast_rtp_instance_activate(p->rtp);
		}

		if (!req->ignore && p->owner) {
			int rpid_changed;

			rpid_changed = get_rpid(p, req);
			if (rpid_changed || !reinvite) {
				/* Queue a connected line update */
				ast_party_connected_line_init(&connected);
				memset(&update_connected, 0, sizeof(update_connected));
				if (rpid_changed
					|| !ast_strlen_zero(p->cid_num)
					|| (p->callingpres & AST_PRES_RESTRICTION) != AST_PRES_ALLOWED) {
					update_connected.id.number = 1;
					connected.id.number.valid = 1;
					connected.id.number.str = (char *) p->cid_num;
					connected.id.number.presentation = p->callingpres;
				}
				if (rpid_changed
					|| !ast_strlen_zero(p->cid_name)
					|| (p->callingpres & AST_PRES_RESTRICTION) != AST_PRES_ALLOWED) {
					update_connected.id.name = 1;
					connected.id.name.valid = 1;
					connected.id.name.str = (char *) p->cid_name;
					connected.id.name.presentation = p->callingpres;
				}
				if (update_connected.id.number || update_connected.id.name) {
					connected.id.tag = (char *) p->cid_tag;
					connected.source = AST_CONNECTED_LINE_UPDATE_SOURCE_ANSWER;
					ast_channel_queue_connected_line_update(p->owner, &connected,
						&update_connected);
				}
			}
		}

		/* Parse contact header for continued conversation */
		/* When we get 200 OK, we know which device (and IP) to contact for this call */
		/* This is important when we have a SIP proxy between us and the phone */
		if (outgoing) {
			update_call_counter(p, DEC_CALL_RINGING);
			parse_ok_contact(p, req);
			/* Save Record-Route for any later requests we make on this dialogue */
			if (!reinvite) {
				build_route(p, req, 1, resp);
			}
			if(set_address_from_contact(p)) {
				/* Bad contact - we don't know how to reach this device */
				/* We need to ACK, but then send a bye */
				if (!p->route && !req->ignore)
					ast_set_flag(&p->flags[0], SIP_PENDINGBYE);	
			}

		}

		if (!req->ignore && p->owner) {
			if (!reinvite) {
				ast_queue_control(p->owner, AST_CONTROL_ANSWER);
				if (sip_cfg.callevents)
					manager_event(EVENT_FLAG_SYSTEM, "ChannelUpdate",
						"Channel: %s\r\nChanneltype: %s\r\nUniqueid: %s\r\nSIPcallid: %s\r\nSIPfullcontact: %s\r\nPeername: %s\r\n",
						p->owner->name, "SIP", p->owner->uniqueid, p->callid, p->fullcontact, p->peername);
			} else {	/* RE-invite */
				if (p->t38.state == T38_DISABLED) {
					ast_queue_control(p->owner, AST_CONTROL_UPDATE_RTP_PEER);
				} else {
					ast_queue_frame(p->owner, &ast_null_frame);
				}
			}
		} else {
			 /* It's possible we're getting an 200 OK after we've tried to disconnect
				  by sending CANCEL */
			/* First send ACK, then send bye */
			if (!req->ignore)
				ast_set_flag(&p->flags[0], SIP_PENDINGBYE);	
		}

		/* Check for Session-Timers related headers */
		if (st_get_mode(p, 0) != SESSION_TIMER_MODE_REFUSE && p->outgoing_call == TRUE && !reinvite) {
			p_hdrval = (char*)get_header(req, "Session-Expires");
			if (!ast_strlen_zero(p_hdrval)) {
				/* UAS supports Session-Timers */
				enum st_refresher tmp_st_ref = SESSION_TIMER_REFRESHER_AUTO;
				int tmp_st_interval = 0;
				rtn = parse_session_expires(p_hdrval, &tmp_st_interval, &tmp_st_ref);
				if (rtn != 0) {
					ast_set_flag(&p->flags[0], SIP_PENDINGBYE);	
				}
				if (tmp_st_ref == SESSION_TIMER_REFRESHER_UAC ||
					tmp_st_ref == SESSION_TIMER_REFRESHER_UAS) {
					p->stimer->st_ref = tmp_st_ref;
				}
				if (tmp_st_interval) {
					p->stimer->st_interval = tmp_st_interval;
				}
				p->stimer->st_active = TRUE;
				p->stimer->st_active_peer_ua = TRUE;
				start_session_timer(p);
			} else {
				/* UAS doesn't support Session-Timers */
				if (st_get_mode(p, 0) == SESSION_TIMER_MODE_ORIGINATE) {
					p->stimer->st_ref = SESSION_TIMER_REFRESHER_UAC;
					p->stimer->st_active_peer_ua = FALSE;
					start_session_timer(p);
				}
			}
		}


		/* If I understand this right, the branch is different for a non-200 ACK only */
		p->invitestate = INV_TERMINATED;
		ast_set_flag(&p->flags[1], SIP_PAGE2_DIALOG_ESTABLISHED);
		xmitres = transmit_request(p, SIP_ACK, seqno, XMIT_UNRELIABLE, TRUE);
		check_pendings(p);
		break;

	case 407: /* Proxy authentication */
	case 401: /* Www auth */
		/* First we ACK */
		xmitres = transmit_request(p, SIP_ACK, seqno, XMIT_UNRELIABLE, FALSE);
		if (p->options)
			p->options->auth_type = resp;

		/* Then we AUTH */
		ast_string_field_set(p, theirtag, NULL);	/* forget their old tag, so we don't match tags when getting response */
		if (!req->ignore) {
			if (p->authtries < MAX_AUTHTRIES)
				p->invitestate = INV_CALLING;
			if (p->authtries == MAX_AUTHTRIES || do_proxy_auth(p, req, resp, SIP_INVITE, 1)) {
				ast_log(LOG_NOTICE, "Failed to authenticate on INVITE to '%s'\n", get_header(&p->initreq, "From"));
				pvt_set_needdestroy(p, "failed to authenticate on INVITE");
				sip_alreadygone(p);
				if (p->owner)
					ast_queue_control(p->owner, AST_CONTROL_CONGESTION);
			}
		}
		break;

	case 403: /* Forbidden */
		/* First we ACK */
		xmitres = transmit_request(p, SIP_ACK, seqno, XMIT_UNRELIABLE, FALSE);
		ast_log(LOG_WARNING, "Received response: \"Forbidden\" from '%s'\n", get_header(&p->initreq, "From"));
		if (!req->ignore && p->owner) {
			ast_set_hangupsource(p->owner, p->owner->name, 0);
			ast_queue_hangup_with_cause(p->owner, AST_CAUSE_CONGESTION);
		}
		break;

	case 404: /* Not found */
		xmitres = transmit_request(p, SIP_ACK, seqno, XMIT_UNRELIABLE, FALSE);
		if (p->owner && !req->ignore) {
			ast_set_hangupsource(p->owner, p->owner->name, 0);
			ast_queue_hangup_with_cause(p->owner, AST_CAUSE_CONGESTION);
		}
		break;

	case 408: /* Request timeout */
	case 481: /* Call leg does not exist */
		/* Could be REFER caused INVITE with replaces */
		ast_log(LOG_WARNING, "Re-invite to non-existing call leg on other UA. SIP dialog '%s'. Giving up.\n", p->callid);
		xmitres = transmit_request(p, SIP_ACK, seqno, XMIT_UNRELIABLE, FALSE);
		if (p->owner) {
			ast_queue_hangup_with_cause(p->owner, AST_CAUSE_CONGESTION);
		}
		break;

	case 422: /* Session-Timers: Session interval too small */
		xmitres = transmit_request(p, SIP_ACK, seqno, XMIT_UNRELIABLE, FALSE);
		ast_string_field_set(p, theirtag, NULL);
		proc_422_rsp(p, req);
		break;

	case 428: /* Use identity header - rfc 4474 - not supported by Asterisk yet */
		xmitres = transmit_request(p, SIP_ACK, seqno, XMIT_UNRELIABLE, FALSE);
		append_history(p, "Identity", "SIP identity is required. Not supported by Asterisk.");
		ast_log(LOG_WARNING, "SIP identity required by proxy. SIP dialog '%s'. Giving up.\n", p->callid);
		if (p->owner && !req->ignore) {
			ast_queue_hangup_with_cause(p->owner, AST_CAUSE_CONGESTION);
		}
		break;

	case 487: /* Cancelled transaction */
		/* We have sent CANCEL on an outbound INVITE
			This transaction is already scheduled to be killed by sip_hangup().
		*/
		xmitres = transmit_request(p, SIP_ACK, seqno, XMIT_UNRELIABLE, FALSE);
		if (p->owner && !req->ignore) {
			ast_queue_hangup_with_cause(p->owner, AST_CAUSE_NORMAL_CLEARING);
			append_history(p, "Hangup", "Got 487 on CANCEL request from us. Queued AST hangup request");
 		} else if (!req->ignore) {
			update_call_counter(p, DEC_CALL_LIMIT);
			append_history(p, "Hangup", "Got 487 on CANCEL request from us on call without owner. Killing this dialog.");
		}
		check_pendings(p);
		sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
		break;
	case 415: /* Unsupported media type */
	case 488: /* Not acceptable here */
	case 606: /* Not Acceptable */
		xmitres = transmit_request(p, SIP_ACK, seqno, XMIT_UNRELIABLE, FALSE);
		if (p->udptl && p->t38.state == T38_LOCAL_REINVITE) {
			change_t38_state(p, T38_DISABLED);
			/* Try to reset RTP timers */
			//ast_rtp_set_rtptimers_onhold(p->rtp);

			/* Trigger a reinvite back to audio */
			transmit_reinvite_with_sdp(p, FALSE, FALSE);
		} else {
			/* We can't set up this call, so give up */
			if (p->owner && !req->ignore) {
				ast_queue_hangup_with_cause(p->owner, AST_CAUSE_CONGESTION);
			}
		}
		break;
	case 491: /* Pending */
		xmitres = transmit_request(p, SIP_ACK, seqno, XMIT_UNRELIABLE, FALSE);
		if (p->owner && !req->ignore) {
			if (p->owner->_state != AST_STATE_UP) {
				ast_queue_hangup_with_cause(p->owner, AST_CAUSE_CONGESTION);
			} else {
				/* This is a re-invite that failed. */
				/* Reset the flag after a while
				 */
				int wait;
				/* RFC 3261, if owner of call, wait between 2.1 to 4 seconds,
				 * if not owner of call, wait 0 to 2 seconds */
				if (p->outgoing_call) {
					wait = 2100 + ast_random() % 2000;
				} else {
					wait = ast_random() % 2000;
				}
				p->waitid = ast_sched_add(sched, wait, sip_reinvite_retry, dialog_ref(p, "passing dialog ptr into sched structure based on waitid for sip_reinvite_retry."));
				ast_log(LOG_WARNING, "just did sched_add waitid(%d) for sip_reinvite_retry for dialog %s in handle_response_invite\n", p->waitid, p->callid);
				ast_debug(2, "Reinvite race. Waiting %d secs before retry\n", wait);
			}
		}
		break;

	case 405: /* Not allowed */
	case 501: /* Not implemented */
		xmitres = transmit_request(p, SIP_ACK, seqno, XMIT_UNRELIABLE, FALSE);
		if (p->owner) {
			ast_queue_hangup_with_cause(p->owner, AST_CAUSE_CONGESTION);
		}
		break;
	}
	if (xmitres == XMIT_ERROR)
		ast_log(LOG_WARNING, "Could not transmit message in dialog %s\n", p->callid);
}

/* \brief Handle SIP response in NOTIFY transaction
       We've sent a NOTIFY, now handle responses to it
  */
static void handle_response_notify(struct sip_pvt *p, int resp, const char *rest, struct sip_request *req, uint32_t seqno)
{
	switch (resp) {
	case 200:   /* Notify accepted */
		/* They got the notify, this is the end */
		if (p->owner) {
			if (p->refer) {
				ast_log(LOG_NOTICE, "Got OK on REFER Notify message\n");
			} else {
				ast_log(LOG_WARNING, "Notify answer on an owned channel? - %s\n", p->owner->name);
			}
		} else {
			if (p->subscribed == NONE && !p->refer) {
				ast_debug(4, "Got 200 accepted on NOTIFY %s\n", p->callid);
				pvt_set_needdestroy(p, "received 200 response");
			}
			if (ast_test_flag(&p->flags[1], SIP_PAGE2_STATECHANGEQUEUE)) {
				/* Ready to send the next state we have on queue */
				ast_clear_flag(&p->flags[1], SIP_PAGE2_STATECHANGEQUEUE);
				cb_extensionstate((char *)p->context, (char *)p->exten, p->laststate, (void *) p);
			}
		}
		break;
	case 401:   /* Not www-authorized on SIP method */
	case 407:   /* Proxy auth */
		if (!p->notify) {
			break; /* Only device notify can use NOTIFY auth */
		}
		ast_string_field_set(p, theirtag, NULL);
		if (ast_strlen_zero(p->authname)) {
			ast_log(LOG_WARNING, "Asked to authenticate NOTIFY to %s but we have no matching peer or realm auth!\n", ast_sockaddr_stringify(&p->recv));
			pvt_set_needdestroy(p, "unable to authenticate NOTIFY");
		}
		if (p->authtries > 1 || do_proxy_auth(p, req, resp, SIP_NOTIFY, 0)) {
			ast_log(LOG_NOTICE, "Failed to authenticate on NOTIFY to '%s'\n", get_header(&p->initreq, "From"));
			pvt_set_needdestroy(p, "failed to authenticate NOTIFY");
		}
		break;
	case 481: /* Call leg does not exist */
		pvt_set_needdestroy(p, "Received 481 response for NOTIFY");
		break;
	}
}

/* \brief Handle SIP response in SUBSCRIBE transaction */
static void handle_response_subscribe(struct sip_pvt *p, int resp, const char *rest, struct sip_request *req, uint32_t seqno)
{
	if (p->subscribed == CALL_COMPLETION) {
		struct sip_monitor_instance *monitor_instance;

		if (resp < 300) {
			return;
		}

		/* Final failure response received. */
		monitor_instance = ao2_callback(sip_monitor_instances, 0,
			find_sip_monitor_instance_by_subscription_pvt, p);
		if (monitor_instance) {
			ast_cc_monitor_failed(monitor_instance->core_id,
				monitor_instance->device_name,
				"Received error response to our SUBSCRIBE");
		}
		return;
	}

	if (p->subscribed != MWI_NOTIFICATION) {
		return;
	}
	if (!p->mwi) {
		return;
	}

	switch (resp) {
	case 200: /* Subscription accepted */
		ast_debug(3, "Got 200 OK on subscription for MWI\n");
		set_pvt_allowed_methods(p, req);
		if (p->options) {
			ast_free(p->options);
			p->options = NULL;
		}
		p->mwi->subscribed = 1;
		if ((p->mwi->resub = ast_sched_add(sched, mwi_expiry * 1000, sip_subscribe_mwi_do, ASTOBJ_REF(p->mwi))) < 0) {
			ASTOBJ_UNREF(p->mwi, sip_subscribe_mwi_destroy);
		}
		break;
	case 401:
	case 407:
		ast_string_field_set(p, theirtag, NULL);
		if (p->authtries > 1 || do_proxy_auth(p, req, resp, SIP_SUBSCRIBE, 0)) {
			ast_log(LOG_NOTICE, "Failed to authenticate on SUBSCRIBE to '%s'\n", get_header(&p->initreq, "From"));
			p->mwi->call = NULL;
			ASTOBJ_UNREF(p->mwi, sip_subscribe_mwi_destroy);
			pvt_set_needdestroy(p, "failed to authenticate SUBSCRIBE");
		}
		break;
	case 403:
		transmit_response_with_date(p, "200 OK", req);
		ast_log(LOG_WARNING, "Authentication failed while trying to subscribe for MWI.\n");
		p->mwi->call = NULL;
		ASTOBJ_UNREF(p->mwi, sip_subscribe_mwi_destroy);
		pvt_set_needdestroy(p, "received 403 response");
		sip_alreadygone(p);
		break;
	case 404:
		ast_log(LOG_WARNING, "Subscription failed for MWI. The remote side said that a mailbox may not have been configured.\n");
		p->mwi->call = NULL;
		ASTOBJ_UNREF(p->mwi, sip_subscribe_mwi_destroy);
		pvt_set_needdestroy(p, "received 404 response");
		break;
	case 481:
		ast_log(LOG_WARNING, "Subscription failed for MWI. The remote side said that our dialog did not exist.\n");
		p->mwi->call = NULL;
		ASTOBJ_UNREF(p->mwi, sip_subscribe_mwi_destroy);
		pvt_set_needdestroy(p, "received 481 response");
		break;
	case 500:
	case 501:
		ast_log(LOG_WARNING, "Subscription failed for MWI. The remote side may have suffered a heart attack.\n");
		p->mwi->call = NULL;
		ASTOBJ_UNREF(p->mwi, sip_subscribe_mwi_destroy);
		pvt_set_needdestroy(p, "received 500/501 response");
		break;
	}
}

/* \brief Handle SIP response in REFER transaction
	We've sent a REFER, now handle responses to it
  */
static void handle_response_refer(struct sip_pvt *p, int resp, const char *rest, struct sip_request *req, uint32_t seqno)
{
	enum ast_control_transfer message = AST_TRANSFER_FAILED;

	/* If no refer structure exists, then do nothing */
	if (!p->refer)
		return;

	switch (resp) {
	case 202:   /* Transfer accepted */
		/* We need  to do something here */
		/* The transferee is now sending INVITE to target */
		p->refer->status = REFER_ACCEPTED;
		/* Now wait for next message */
		ast_debug(3, "Got 202 accepted on transfer\n");
		/* We should hang along, waiting for NOTIFY's here */
		break;

	case 401:   /* Not www-authorized on SIP method */
	case 407:   /* Proxy auth */
		if (ast_strlen_zero(p->authname)) {
			ast_log(LOG_WARNING, "Asked to authenticate REFER to %s but we have no matching peer or realm auth!\n",
				ast_sockaddr_stringify(&p->recv));
			if (p->owner) {
				ast_queue_control_data(p->owner, AST_CONTROL_TRANSFER, &message, sizeof(message));
			}
			pvt_set_needdestroy(p, "unable to authenticate REFER");
		}
		if (p->authtries > 1 || do_proxy_auth(p, req, resp, SIP_REFER, 0)) {
			ast_log(LOG_NOTICE, "Failed to authenticate on REFER to '%s'\n", get_header(&p->initreq, "From"));
			p->refer->status = REFER_NOAUTH;
			if (p->owner) {
				ast_queue_control_data(p->owner, AST_CONTROL_TRANSFER, &message, sizeof(message));
			}
			pvt_set_needdestroy(p, "failed to authenticate REFER");
		}
		break;
	
	case 405:   /* Method not allowed */
		/* Return to the current call onhold */
		/* Status flag needed to be reset */
		ast_log(LOG_NOTICE, "SIP transfer to %s failed, REFER not allowed. \n", p->refer->refer_to);
		pvt_set_needdestroy(p, "received 405 response");
		p->refer->status = REFER_FAILED;
		if (p->owner) {
			ast_queue_control_data(p->owner, AST_CONTROL_TRANSFER, &message, sizeof(message));
		}
		break;

	case 481: /* Call leg does not exist */

		/* A transfer with Replaces did not work */
		/* OEJ: We should Set flag, cancel the REFER, go back
		to original call - but right now we can't */
		ast_log(LOG_WARNING, "Remote host can't match REFER request to call '%s'. Giving up.\n", p->callid);
		if (p->owner)
			ast_queue_control(p->owner, AST_CONTROL_CONGESTION);
		pvt_set_needdestroy(p, "received 481 response");
		break;

	case 500:   /* Server error */
	case 501:   /* Method not implemented */
		/* Return to the current call onhold */
		/* Status flag needed to be reset */
		ast_log(LOG_NOTICE, "SIP transfer to %s failed, call miserably fails. \n", p->refer->refer_to);
		pvt_set_needdestroy(p, "received 500/501 response");
		p->refer->status = REFER_FAILED;
		if (p->owner) {
			ast_queue_control_data(p->owner, AST_CONTROL_TRANSFER, &message, sizeof(message));
		}
		break;
	case 603:   /* Transfer declined */
		ast_log(LOG_NOTICE, "SIP transfer to %s declined, call miserably fails. \n", p->refer->refer_to);
		p->refer->status = REFER_FAILED;
		pvt_set_needdestroy(p, "received 603 response");
		if (p->owner) {
			ast_queue_control_data(p->owner, AST_CONTROL_TRANSFER, &message, sizeof(message));
		}
		break;
	default:
		/* We should treat unrecognized 9xx as 900.  400 is actually
		   specified as a possible response, but any 4-6xx is 
		   theoretically possible. */

		if (resp < 299) { /* 1xx cases don't get here */
			ast_log(LOG_WARNING, "SIP transfer to %s had unxpected 2xx response (%d), confusion is possible. \n", p->refer->refer_to, resp);
		} else {
			ast_log(LOG_WARNING, "SIP transfer to %s with response (%d). \n", p->refer->refer_to, resp);
		}

		p->refer->status = REFER_FAILED;
		pvt_set_needdestroy(p, "received failure response");
		if (p->owner) {
			ast_queue_control_data(p->owner, AST_CONTROL_TRANSFER, &message, sizeof(message));
		}
		break;
	}
}

/*! \brief Handle responses on REGISTER to services */
static int handle_response_register(struct sip_pvt *p, int resp, const char *rest, struct sip_request *req, uint32_t seqno)
{
	int expires, expires_ms;
	struct sip_registry *r;
	r=p->registry;
	
	switch (resp) {
	case 401:	/* Unauthorized */
		if (p->authtries == MAX_AUTHTRIES || do_register_auth(p, req, resp)) {
			ast_log(LOG_NOTICE, "Failed to authenticate on REGISTER to '%s@%s' (Tries %d)\n", p->registry->username, p->registry->hostname, p->authtries);
			pvt_set_needdestroy(p, "failed to authenticate REGISTER");
		}
		break;
	case 403:	/* Forbidden */
		ast_log(LOG_WARNING, "Forbidden - wrong password on authentication for REGISTER for '%s' to '%s'\n", p->registry->username, p->registry->hostname);
		AST_SCHED_DEL_UNREF(sched, r->timeout, registry_unref(r, "reg ptr unref from handle_response_register 403"));
		r->regstate = REG_STATE_NOAUTH;
		pvt_set_needdestroy(p, "received 403 response");
		break;
	case 404:	/* Not found */
		ast_log(LOG_WARNING, "Got 404 Not found on SIP register to service %s@%s, giving up\n", p->registry->username, p->registry->hostname);
		pvt_set_needdestroy(p, "received 404 response");
		if (r->call)
			r->call = dialog_unref(r->call, "unsetting registry->call pointer-- case 404");
		r->regstate = REG_STATE_REJECTED;
		AST_SCHED_DEL_UNREF(sched, r->timeout, registry_unref(r, "reg ptr unref from handle_response_register 404"));
		break;
	case 407:	/* Proxy auth */
		if (p->authtries == MAX_AUTHTRIES || do_register_auth(p, req, resp)) {
			ast_log(LOG_NOTICE, "Failed to authenticate on REGISTER to '%s' (tries '%d')\n", get_header(&p->initreq, "From"), p->authtries);
			pvt_set_needdestroy(p, "failed to authenticate REGISTER");
		}
		break;
	case 408:	/* Request timeout */
		/* Got a timeout response, so reset the counter of failed responses */
		if (r) {
			r->regattempts = 0;
		} else {
			ast_log(LOG_WARNING, "Got a 408 response to our REGISTER on call %s after we had destroyed the registry object\n", p->callid);
		}
		break;
	case 423:	/* Interval too brief */
		r->expiry = atoi(get_header(req, "Min-Expires"));
		ast_log(LOG_WARNING, "Got 423 Interval too brief for service %s@%s, minimum is %d seconds\n", p->registry->username, p->registry->hostname, r->expiry);
		AST_SCHED_DEL_UNREF(sched, r->timeout, registry_unref(r, "reg ptr unref from handle_response_register 423"));
		if (r->call) {
			r->call = dialog_unref(r->call, "unsetting registry->call pointer-- case 423");
			pvt_set_needdestroy(p, "received 423 response");
		}
		if (r->expiry > max_expiry) {
			ast_log(LOG_WARNING, "Required expiration time from %s@%s is too high, giving up\n", p->registry->username, p->registry->hostname);
			r->expiry = r->configured_expiry;
			r->regstate = REG_STATE_REJECTED;
		} else {
			r->regstate = REG_STATE_UNREGISTERED;
			transmit_register(r, SIP_REGISTER, NULL, NULL);
		}
		manager_event(EVENT_FLAG_SYSTEM, "Registry", "ChannelType: SIP\r\nUsername: %s\r\nDomain: %s\r\nStatus: %s\r\n", r->username, r->hostname, regstate2str(r->regstate));
		break;
	case 479:	/* SER: Not able to process the URI - address is wrong in register*/
		ast_log(LOG_WARNING, "Got error 479 on register to %s@%s, giving up (check config)\n", p->registry->username, p->registry->hostname);
		pvt_set_needdestroy(p, "received 479 response");
		if (r->call)
			r->call = dialog_unref(r->call, "unsetting registry->call pointer-- case 479");
		r->regstate = REG_STATE_REJECTED;
		AST_SCHED_DEL_UNREF(sched, r->timeout, registry_unref(r, "reg ptr unref from handle_response_register 479"));
		break;
	case 200:	/* 200 OK */
		if (!r) {
			ast_log(LOG_WARNING, "Got 200 OK on REGISTER, but there isn't a registry entry for '%s' (we probably already got the OK)\n", S_OR(p->peername, p->username));
			pvt_set_needdestroy(p, "received erroneous 200 response");
			return 0;
		}
		
		r->regstate = REG_STATE_REGISTERED;
		r->regtime = ast_tvnow();		/* Reset time of last successful registration */
		manager_event(EVENT_FLAG_SYSTEM, "Registry", "ChannelType: SIP\r\nDomain: %s\r\nStatus: %s\r\n", r->hostname, regstate2str(r->regstate));
		r->regattempts = 0;
		ast_debug(1, "Registration successful\n");
		if (r->timeout > -1) {
			ast_debug(1, "Cancelling timeout %d\n", r->timeout);
		}
		AST_SCHED_DEL_UNREF(sched, r->timeout, registry_unref(r, "reg ptr unref from handle_response_register 200"));
		if (r->call)
			r->call = dialog_unref(r->call, "unsetting registry->call pointer-- case 200");
		p->registry = registry_unref(p->registry, "unref registry entry p->registry");
		/* Let this one hang around until we have all the responses */
		sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
		/* p->needdestroy = 1; */
		
		/* set us up for re-registering
		 * figure out how long we got registered for
		 * according to section 6.13 of RFC, contact headers override
		 * expires headers, so check those first */
		expires = 0;

		/* XXX todo: try to save the extra call */
		if (!ast_strlen_zero(get_header(req, "Contact"))) {
			const char *contact = NULL;
			const char *tmptmp = NULL;
			int start = 0;
			for(;;) {
				contact = __get_header(req, "Contact", &start);
				/* this loop ensures we get a contact header about our register request */
				if(!ast_strlen_zero(contact)) {
					if( (tmptmp=strstr(contact, p->our_contact))) {
						contact=tmptmp;
						break;
					}
				} else
					break;
			}
			tmptmp = strcasestr(contact, "expires=");
			if (tmptmp) {
				if (sscanf(tmptmp + 8, "%30d;", &expires) != 1)
					expires = 0;
			}
			
		}
		if (!expires)
			expires=atoi(get_header(req, "expires"));
		if (!expires)
			expires=default_expiry;
		
		expires_ms = expires * 1000;
		if (expires <= EXPIRY_GUARD_LIMIT)
			expires_ms -= MAX((expires_ms * EXPIRY_GUARD_PCT), EXPIRY_GUARD_MIN);
		else
			expires_ms -= EXPIRY_GUARD_SECS * 1000;
		if (sipdebug)
			ast_log(LOG_NOTICE, "Outbound Registration: Expiry for %s is %d sec (Scheduling reregistration in %d s)\n", r->hostname, expires, expires_ms/1000);
		
		r->refresh= (int) expires_ms / 1000;
		
		/* Schedule re-registration before we expire */
		AST_SCHED_REPLACE_UNREF(r->expire, sched, expires_ms, sip_reregister, r,
								registry_unref(_data,"unref in REPLACE del fail"),
								registry_unref(r,"unref in REPLACE add fail"),
								registry_addref(r,"The Addition side of REPLACE"));
	}
	return 1;
}

/*! \brief Handle qualification responses (OPTIONS) */
static void handle_response_peerpoke(struct sip_pvt *p, int resp, struct sip_request *req)
{
	struct sip_peer *peer = /* ref_peer( */ p->relatedpeer /* , "bump refcount on p, as it is being used in this function(handle_response_peerpoke)")*/ ; /* hope this is already refcounted! */
	int statechanged, is_reachable, was_reachable;
	int pingtime = ast_tvdiff_ms(ast_tvnow(), peer->ps);

	/*
	 * Compute the response time to a ping (goes in peer->lastms.)
	 * -1 means did not respond, 0 means unknown,
	 * 1..maxms is a valid response, >maxms means late response.
	 */
	if (pingtime < 1) {	/* zero = unknown, so round up to 1 */
		pingtime = 1;
	}

	if (!peer->maxms) { /* this should never happens */
		pvt_set_needdestroy(p, "got OPTIONS response but qualify is not enabled");
		return;
	}

	/* Now determine new state and whether it has changed.
	 * Use some helper variables to simplify the writing
	 * of the expressions.
	 */
	was_reachable = peer->lastms > 0 && peer->lastms <= peer->maxms;
	is_reachable = pingtime <= peer->maxms;
	statechanged = peer->lastms == 0 /* yes, unknown before */
		|| was_reachable != is_reachable;

	peer->lastms = pingtime;
	peer->call = dialog_unref(peer->call, "unref dialog peer->call");
	if (statechanged) {
		const char *s = is_reachable ? "Reachable" : "Lagged";
		char str_lastms[20];
		snprintf(str_lastms, sizeof(str_lastms), "%d", pingtime);

		ast_log(LOG_NOTICE, "Peer '%s' is now %s. (%dms / %dms)\n",
			peer->name, s, pingtime, peer->maxms);
		ast_devstate_changed(AST_DEVICE_UNKNOWN, "SIP/%s", peer->name);
		if (sip_cfg.peer_rtupdate) {
			ast_update_realtime(ast_check_realtime("sipregs") ? "sipregs" : "sippeers", "name", peer->name, "lastms", str_lastms, SENTINEL);
		}
		manager_event(EVENT_FLAG_SYSTEM, "PeerStatus",
			"ChannelType: SIP\r\nPeer: SIP/%s\r\nPeerStatus: %s\r\nTime: %d\r\n",
			peer->name, s, pingtime);
		if (is_reachable && sip_cfg.regextenonqualify)
			register_peer_exten(peer, TRUE);
	}

	pvt_set_needdestroy(p, "got OPTIONS response");

	/* Try again eventually */
	AST_SCHED_REPLACE_UNREF(peer->pokeexpire, sched,
			is_reachable ? peer->qualifyfreq : DEFAULT_FREQ_NOTOK,
			sip_poke_peer_s, peer,
			unref_peer(_data, "removing poke peer ref"),
			unref_peer(peer, "removing poke peer ref"),
			ref_peer(peer, "adding poke peer ref"));
}

/*!
 * \internal
 * \brief Handle responses to INFO messages
 *
 * \note The INFO method MUST NOT change the state of calls or
 * related sessions (RFC 2976).
 */
static void handle_response_info(struct sip_pvt *p, int resp, const char *rest, struct sip_request *req, uint32_t seqno)
{
	int sipmethod = SIP_INFO;

	switch (resp) {
	case 401: /* Not www-authorized on SIP method */
	case 407: /* Proxy auth required */
		ast_log(LOG_WARNING, "Host '%s' requests authentication (%d) for '%s'\n",
			ast_sockaddr_stringify(&p->sa), resp, sip_methods[sipmethod].text);
		break;
	case 405: /* Method not allowed */
	case 501: /* Not Implemented */
		mark_method_unallowed(&p->allowed_methods, sipmethod);
		if (p->relatedpeer) {
			mark_method_allowed(&p->relatedpeer->disallowed_methods, sipmethod);
		}
		ast_log(LOG_WARNING, "Host '%s' does not implement '%s'\n",
			ast_sockaddr_stringify(&p->sa), sip_methods[sipmethod].text);
		break;
	default:
		if (300 <= resp && resp < 700) {
			ast_verb(3, "Got SIP %s response %d \"%s\" back from host '%s'\n",
				sip_methods[sipmethod].text, resp, rest, ast_sockaddr_stringify(&p->sa));
		}
		break;
	}
}

/*!
 * \internal
 * \brief Handle responses to MESSAGE messages
 *
 * \note The MESSAGE method should not change the state of calls
 * or related sessions if associated with a dialog. (Implied by
 * RFC 3428 Section 2).
 */
static void handle_response_message(struct sip_pvt *p, int resp, const char *rest, struct sip_request *req, uint32_t seqno)
{
	int sipmethod = SIP_MESSAGE;
	/* Out-of-dialog MESSAGE currently not supported. */
	//int in_dialog = ast_test_flag(&p->flags[1], SIP_PAGE2_DIALOG_ESTABLISHED);

	switch (resp) {
	case 401: /* Not www-authorized on SIP method */
	case 407: /* Proxy auth required */
		ast_log(LOG_WARNING, "Host '%s' requests authentication (%d) for '%s'\n",
			ast_sockaddr_stringify(&p->sa), resp, sip_methods[sipmethod].text);
		break;
	case 405: /* Method not allowed */
	case 501: /* Not Implemented */
		mark_method_unallowed(&p->allowed_methods, sipmethod);
		if (p->relatedpeer) {
			mark_method_allowed(&p->relatedpeer->disallowed_methods, sipmethod);
		}
		ast_log(LOG_WARNING, "Host '%s' does not implement '%s'\n",
			ast_sockaddr_stringify(&p->sa), sip_methods[sipmethod].text);
		break;
	default:
		if (100 <= resp && resp < 200) {
			/* Must allow provisional responses for out-of-dialog requests. */
		} else if (200 <= resp && resp < 300) {
			p->authtries = 0;	/* Reset authentication counter */
		} else if (300 <= resp && resp < 700) {
			ast_verb(3, "Got SIP %s response %d \"%s\" back from host '%s'\n",
				sip_methods[sipmethod].text, resp, rest, ast_sockaddr_stringify(&p->sa));
		}
		break;
	}
}

/*! \brief Immediately stop RTP, VRTP and UDPTL as applicable */
static void stop_media_flows(struct sip_pvt *p)
{
	/* Immediately stop RTP, VRTP and UDPTL as applicable */
	if (p->rtp)
		ast_rtp_instance_stop(p->rtp);
	if (p->vrtp)
		ast_rtp_instance_stop(p->vrtp);
	if (p->trtp)
		ast_rtp_instance_stop(p->trtp);
	if (p->udptl)
		ast_udptl_stop(p->udptl);
}

/*! \brief Handle SIP response in dialogue
	\note only called by handle_incoming */
static void handle_response(struct sip_pvt *p, int resp, const char *rest, struct sip_request *req, uint32_t seqno)
{
	struct ast_channel *owner;
	int sipmethod;
	const char *c = get_header(req, "Cseq");
	/* GCC 4.2 complains if I try to cast c as a char * when passing it to ast_skip_nonblanks, so make a copy of it */
	char *c_copy = ast_strdupa(c);
	/* Skip the Cseq and its subsequent spaces */
	const char *msg = ast_skip_blanks(ast_skip_nonblanks(c_copy));

	if (!msg)
		msg = "";

	sipmethod = find_sip_method(msg);

	owner = p->owner;
	if (owner) {
		const char *rp = NULL, *rh = NULL;

		owner->hangupcause = 0;
		if (ast_test_flag(&p->flags[1], SIP_PAGE2_Q850_REASON) && (rh = get_header(req, "Reason"))) {
			rh = ast_skip_blanks(rh);
			if (!strncasecmp(rh, "Q.850", 5)) {
				rp = strstr(rh, "cause=");
				if (rp && sscanf(rp + 6, "%30d", &owner->hangupcause) == 1) {
					owner->hangupcause &= 0x7f;
					if (req->debug)
						ast_verbose("Using Reason header for cause code: %d\n", owner->hangupcause);
				}
			}
		}

		if (!owner->hangupcause)
			owner->hangupcause = hangup_sip2cause(resp);
	}

	if (p->socket.type == SIP_TRANSPORT_UDP) {
		int ack_res = FALSE;

		/* Acknowledge whatever it is destined for */
		if ((resp >= 100) && (resp <= 199)) {
			/* NON-INVITE messages do not ack a 1XX response. RFC 3261 section 17.1.2.2 */
			if (sipmethod == SIP_INVITE) {
				ack_res = __sip_semi_ack(p, seqno, 0, sipmethod);
			}
		} else {
			ack_res = __sip_ack(p, seqno, 0, sipmethod);
		}

		if (ack_res == FALSE) {
			/* RFC 3261 13.2.2.4 and 17.1.1.2 - We must re-send ACKs to re-transmitted final responses */
			if (sipmethod == SIP_INVITE && resp >= 200) {
				transmit_request(p, SIP_ACK, seqno, XMIT_UNRELIABLE, resp < 300 ? TRUE: FALSE);
			}

			append_history(p, "Ignore", "Ignoring this retransmit\n");
			return;
		}
	}

	/* If this is a NOTIFY for a subscription clear the flag that indicates that we have a NOTIFY pending */
	if (!p->owner && sipmethod == SIP_NOTIFY && p->pendinginvite) {
		p->pendinginvite = 0;
	}

	/* Get their tag if we haven't already */
	if (ast_strlen_zero(p->theirtag) || (resp >= 200)) {
		char tag[128];

		gettag(req, "To", tag, sizeof(tag));
		ast_string_field_set(p, theirtag, tag);
	}
	/* This needs to be configurable on a channel/peer level,
	   not mandatory for all communication. Sadly enough, NAT implementations
	   are not so stable so we can always rely on these headers.
		Temporarily disabled, while waiting for fix.
	   Fix assigned to Rizzo :-)
	*/
	/* check_via_response(p, req); */

	/* RFC 3261 Section 15 specifies that if we receive a 408 or 481
	 * in response to a BYE, then we should end the current dialog
	 * and session.  It is known that at least one phone manufacturer
	 * potentially will send a 404 in response to a BYE, so we'll be
	 * liberal in what we accept and end the dialog and session if we
	 * receive any of those responses to a BYE.
	 */
	if ((resp == 404 || resp == 408 || resp == 481) && sipmethod == SIP_BYE) {
		pvt_set_needdestroy(p, "received 4XX response to a BYE");
		return;
	}

	if (p->relatedpeer && sipmethod == SIP_OPTIONS) {
		/* We don't really care what the response is, just that it replied back.
		   Well, as long as it's not a 100 response...  since we might
		   need to hang around for something more "definitive" */
		if (resp != 100)
			handle_response_peerpoke(p, resp, req);
	} else if (sipmethod == SIP_REFER && resp >= 200) {
		handle_response_refer(p, resp, rest, req, seqno);
	} else if (sipmethod == SIP_PUBLISH) {
		/* SIP PUBLISH transcends this morass of doodoo and instead
		 * we just always call the response handler. Good gravy!
		 */
		handle_response_publish(p, resp, rest, req, seqno);
	} else if (sipmethod == SIP_INFO) {
		/* More good gravy! */
		handle_response_info(p, resp, rest, req, seqno);
	} else if (sipmethod == SIP_MESSAGE) {
		/* More good gravy! */
		handle_response_message(p, resp, rest, req, seqno);
	} else if (sipmethod == SIP_NOTIFY) {
		/* The gravy train continues to roll */
		handle_response_notify(p, resp, rest, req, seqno);
	} else if (ast_test_flag(&p->flags[0], SIP_OUTGOING)) {
		switch(resp) {
		case 100:	/* 100 Trying */
		case 101:	/* 101 Dialog establishment */
		case 183:	/* 183 Session Progress */
		case 180:	/* 180 Ringing */
		case 182:	/* 182 Queued */
		case 181:	/* 181 Call Is Being Forwarded */
			if (sipmethod == SIP_INVITE)
				handle_response_invite(p, resp, rest, req, seqno);
			break;
		case 200:	/* 200 OK */
			p->authtries = 0;	/* Reset authentication counter */
			if (sipmethod == SIP_INVITE) {
				handle_response_invite(p, resp, rest, req, seqno);
			} else if (sipmethod == SIP_REGISTER) {
				handle_response_register(p, resp, rest, req, seqno);
			} else if (sipmethod == SIP_SUBSCRIBE) {
				ast_set_flag(&p->flags[1], SIP_PAGE2_DIALOG_ESTABLISHED);
				handle_response_subscribe(p, resp, rest, req, seqno);
			} else if (sipmethod == SIP_BYE) {		/* Ok, we're ready to go */
				pvt_set_needdestroy(p, "received 200 response");
				ast_clear_flag(&p->flags[1], SIP_PAGE2_DIALOG_ESTABLISHED);
			}
			break;
		case 401: /* Not www-authorized on SIP method */
		case 407: /* Proxy auth required */
			if (sipmethod == SIP_INVITE)
				handle_response_invite(p, resp, rest, req, seqno);
			else if (sipmethod == SIP_SUBSCRIBE)
				handle_response_subscribe(p, resp, rest, req, seqno);
			else if (p->registry && sipmethod == SIP_REGISTER)
				handle_response_register(p, resp, rest, req, seqno);
			else if (sipmethod == SIP_UPDATE) {
				handle_response_update(p, resp, rest, req, seqno);
			} else if (sipmethod == SIP_BYE) {
				if (p->options)
					p->options->auth_type = resp;
				if (ast_strlen_zero(p->authname)) {
					ast_log(LOG_WARNING, "Asked to authenticate %s, to %s but we have no matching peer!\n",
							msg, ast_sockaddr_stringify(&p->recv));
					pvt_set_needdestroy(p, "unable to authenticate BYE");
				} else if ((p->authtries == MAX_AUTHTRIES) || do_proxy_auth(p, req, resp,  sipmethod, 0)) {
					ast_log(LOG_NOTICE, "Failed to authenticate on %s to '%s'\n", msg, get_header(&p->initreq, "From"));
					pvt_set_needdestroy(p, "failed to authenticate BYE");
				}
			} else {
				ast_log(LOG_WARNING, "Got authentication request (%d) on %s to '%s'\n", resp, sip_methods[sipmethod].text, get_header(req, "To"));
				pvt_set_needdestroy(p, "received 407 response");
			}
			break;
		case 403: /* Forbidden - we failed authentication */
			if (sipmethod == SIP_INVITE)
				handle_response_invite(p, resp, rest, req, seqno);
			else if (sipmethod == SIP_SUBSCRIBE)
				handle_response_subscribe(p, resp, rest, req, seqno);
			else if (p->registry && sipmethod == SIP_REGISTER)
				handle_response_register(p, resp, rest, req, seqno);
			else {
				ast_log(LOG_WARNING, "Forbidden - maybe wrong password on authentication for %s\n", msg);
				pvt_set_needdestroy(p, "received 403 response");
			}
			break;
		case 404: /* Not found */
			if (p->registry && sipmethod == SIP_REGISTER)
				handle_response_register(p, resp, rest, req, seqno);
			else if (sipmethod == SIP_INVITE)
				handle_response_invite(p, resp, rest, req, seqno);
			else if (sipmethod == SIP_SUBSCRIBE)
				handle_response_subscribe(p, resp, rest, req, seqno);
			else if (owner)
				ast_queue_control(p->owner, AST_CONTROL_CONGESTION);
			break;
		case 423: /* Interval too brief */
			if (sipmethod == SIP_REGISTER)
				handle_response_register(p, resp, rest, req, seqno);
			break;
		case 408: /* Request timeout - terminate dialog */
			if (sipmethod == SIP_INVITE)
				handle_response_invite(p, resp, rest, req, seqno);
			else if (sipmethod == SIP_REGISTER)
				handle_response_register(p, resp, rest, req, seqno);
			else if (sipmethod == SIP_BYE) {
				pvt_set_needdestroy(p, "received 408 response");
				ast_debug(4, "Got timeout on bye. Thanks for the answer. Now, kill this call\n");
			} else {
				if (owner)
					ast_queue_control(p->owner, AST_CONTROL_CONGESTION);
				pvt_set_needdestroy(p, "received 408 response");
			}
			break;

		case 428:
		case 422: /* Session-Timers: Session Interval Too Small */
			if (sipmethod == SIP_INVITE) {
				handle_response_invite(p, resp, rest, req, seqno);
			}
			break;

		case 481: /* Call leg does not exist */
			if (sipmethod == SIP_INVITE) {
				handle_response_invite(p, resp, rest, req, seqno);
			} else if (sipmethod == SIP_SUBSCRIBE) {
				handle_response_subscribe(p, resp, rest, req, seqno);
			} else if (sipmethod == SIP_BYE) {
				/* The other side has no transaction to bye,
				just assume it's all right then */
				ast_log(LOG_WARNING, "Remote host can't match request %s to call '%s'. Giving up.\n", sip_methods[sipmethod].text, p->callid);
			} else if (sipmethod == SIP_CANCEL) {
				/* The other side has no transaction to cancel,
				just assume it's all right then */
				ast_log(LOG_WARNING, "Remote host can't match request %s to call '%s'. Giving up.\n", sip_methods[sipmethod].text, p->callid);
			} else {
				ast_log(LOG_WARNING, "Remote host can't match request %s to call '%s'. Giving up.\n", sip_methods[sipmethod].text, p->callid);
				/* Guessing that this is not an important request */
			}
			break;
		case 487:
			if (sipmethod == SIP_INVITE)
				handle_response_invite(p, resp, rest, req, seqno);
			break;
		case 415: /* Unsupported media type */
		case 488: /* Not acceptable here - codec error */
		case 606: /* Not Acceptable */
			if (sipmethod == SIP_INVITE)
				handle_response_invite(p, resp, rest, req, seqno);
			break;
		case 491: /* Pending */
			if (sipmethod == SIP_INVITE)
				handle_response_invite(p, resp, rest, req, seqno);
			else {
				ast_debug(1, "Got 491 on %s, unsupported. Call ID %s\n", sip_methods[sipmethod].text, p->callid);
				pvt_set_needdestroy(p, "received 491 response");
			}
			break;
		case 405: /* Method not allowed */
		case 501: /* Not Implemented */
			mark_method_unallowed(&p->allowed_methods, sipmethod);
			if (p->relatedpeer) {
				mark_method_allowed(&p->relatedpeer->disallowed_methods, sipmethod);
			}
			if (sipmethod == SIP_INVITE)
				handle_response_invite(p, resp, rest, req, seqno);
			else
				ast_log(LOG_WARNING, "Host '%s' does not implement '%s'\n", ast_sockaddr_stringify(&p->sa), msg);
			break;
		default:
			if ((resp >= 300) && (resp < 700)) {
				/* Fatal response */
				if ((resp != 487))
					ast_verb(3, "Got SIP response %d \"%s\" back from %s\n", resp, rest, ast_sockaddr_stringify(&p->sa));
	
				if (sipmethod == SIP_INVITE)
					stop_media_flows(p); /* Immediately stop RTP, VRTP and UDPTL as applicable */

				/* XXX Locking issues?? XXX */
				switch(resp) {
				case 300: /* Multiple Choices */
				case 301: /* Moved permanently */
				case 302: /* Moved temporarily */
				case 305: /* Use Proxy */
					if (p->owner) {
						struct ast_party_redirecting redirecting;
						struct ast_set_party_redirecting update_redirecting;

						ast_party_redirecting_init(&redirecting);
						memset(&update_redirecting, 0, sizeof(update_redirecting));
						change_redirecting_information(p, req, &redirecting,
							&update_redirecting, TRUE);
						ast_channel_set_redirecting(p->owner, &redirecting,
							&update_redirecting);
						ast_party_redirecting_free(&redirecting);
					}
					/* Fall through */
				case 486: /* Busy here */
				case 600: /* Busy everywhere */
				case 603: /* Decline */
					if (p->owner) {
						sip_handle_cc(p, req, AST_CC_CCBS);
						ast_queue_control(p->owner, AST_CONTROL_BUSY);
					}
					break;
				case 482: /* Loop Detected */
				case 480: /* Temporarily Unavailable */
				case 404: /* Not Found */
				case 410: /* Gone */
				case 400: /* Bad Request */
				case 500: /* Server error */
					if (sipmethod == SIP_SUBSCRIBE) {
						handle_response_subscribe(p, resp, rest, req, seqno);
						break;
					}
					/* Fall through */
				case 502: /* Bad gateway */
				case 503: /* Service Unavailable */
				case 504: /* Server Timeout */
					if (owner)
						ast_queue_control(p->owner, AST_CONTROL_CONGESTION);
					break;
				case 484: /* Address Incomplete */
					if (owner && sipmethod != SIP_BYE) {
						switch (ast_test_flag(&p->flags[1], SIP_PAGE2_ALLOWOVERLAP)) {
						case SIP_PAGE2_ALLOWOVERLAP_YES:
							ast_queue_hangup_with_cause(p->owner, hangup_sip2cause(resp));
							break;
						default:
							ast_queue_hangup_with_cause(p->owner, hangup_sip2cause(404));
							break;
						}
					}
					break;
				default:
					/* Send hangup */	
					if (owner && sipmethod != SIP_BYE)
						ast_queue_hangup_with_cause(p->owner, AST_CAUSE_PROTOCOL_ERROR);
					break;
				}
				/* ACK on invite */
				if (sipmethod == SIP_INVITE)
					transmit_request(p, SIP_ACK, seqno, XMIT_UNRELIABLE, FALSE);
				sip_alreadygone(p);
				if (!p->owner) {
					pvt_set_needdestroy(p, "transaction completed");
				}
			} else if ((resp >= 100) && (resp < 200)) {
				if (sipmethod == SIP_INVITE) {
					if (!req->ignore && sip_cancel_destroy(p))
						ast_log(LOG_WARNING, "Unable to cancel SIP destruction.  Expect bad things.\n");
					if (find_sdp(req))
						process_sdp(p, req, SDP_T38_NONE);
					if (p->owner) {
						/* Queue a progress frame */
						ast_queue_control(p->owner, AST_CONTROL_PROGRESS);
					}
				}
			} else
				ast_log(LOG_NOTICE, "Don't know how to handle a %d %s response from %s\n", resp, rest, p->owner ? p->owner->name : ast_sockaddr_stringify(&p->sa));
		}
	} else {	
		/* Responses to OUTGOING SIP requests on INCOMING calls
		   get handled here. As well as out-of-call message responses */
		if (req->debug)
			ast_verbose("SIP Response message for INCOMING dialog %s arrived\n", msg);

		if (sipmethod == SIP_INVITE && resp == 200) {
			/* Tags in early session is replaced by the tag in 200 OK, which is
		  	the final reply to our INVITE */
			char tag[128];

			gettag(req, "To", tag, sizeof(tag));
			ast_string_field_set(p, theirtag, tag);
		}

		switch(resp) {
		case 200:
			if (sipmethod == SIP_INVITE) {
				handle_response_invite(p, resp, rest, req, seqno);
			} else if (sipmethod == SIP_CANCEL) {
				ast_debug(1, "Got 200 OK on CANCEL\n");

				/* Wait for 487, then destroy */
			} else if (sipmethod == SIP_BYE) {
				pvt_set_needdestroy(p, "transaction completed");
			}
			break;
		case 401:	/* www-auth */
		case 407:
			if (sipmethod == SIP_INVITE)
				handle_response_invite(p, resp, rest, req, seqno);
			else if (sipmethod == SIP_BYE) {
				if (p->authtries == MAX_AUTHTRIES || do_proxy_auth(p, req, resp, sipmethod, 0)) {
					ast_log(LOG_NOTICE, "Failed to authenticate on %s to '%s'\n", msg, get_header(&p->initreq, "From"));
					pvt_set_needdestroy(p, "failed to authenticate BYE");
				}
			}
			break;
		case 481:	/* Call leg does not exist */
			if (sipmethod == SIP_INVITE) {
				/* Re-invite failed */
				handle_response_invite(p, resp, rest, req, seqno);
			} else if (sipmethod == SIP_BYE) {
				pvt_set_needdestroy(p, "received 481 response");
			} else if (sipdebug) {
				ast_debug(1, "Remote host can't match request %s to call '%s'. Giving up\n", sip_methods[sipmethod].text, p->callid);
			}
			break;
		case 501: /* Not Implemented */
			if (sipmethod == SIP_INVITE)
				handle_response_invite(p, resp, rest, req, seqno);
			break;
		default:	/* Errors without handlers */
			if ((resp >= 100) && (resp < 200)) {
				if (sipmethod == SIP_INVITE) { 	/* re-invite */
					if (!req->ignore && sip_cancel_destroy(p))
						ast_log(LOG_WARNING, "Unable to cancel SIP destruction.  Expect bad things.\n");
				}
			}
			if ((resp >= 300) && (resp < 700)) {
				if ((resp != 487))
					ast_verb(3, "Incoming call: Got SIP response %d \"%s\" back from %s\n", resp, rest, ast_sockaddr_stringify(&p->sa));
				switch(resp) {
				case 415: /* Unsupported media type */
				case 488: /* Not acceptable here - codec error */
				case 603: /* Decline */
				case 500: /* Server error */
				case 502: /* Bad gateway */
				case 503: /* Service Unavailable */
				case 504: /* Server timeout */

					/* re-invite failed */
					if (sipmethod == SIP_INVITE && sip_cancel_destroy(p))
						ast_log(LOG_WARNING, "Unable to cancel SIP destruction.  Expect bad things.\n");
					break;
				}
			}
			break;
		}
	}
}


/*! \brief Park SIP call support function
	Starts in a new thread, then parks the call
	XXX Should we add a wait period after streaming audio and before hangup?? Sometimes the
		audio can't be heard before hangup
*/
static void *sip_park_thread(void *stuff)
{
	struct ast_channel *transferee, *transferer;	/* Chan1: The transferee, Chan2: The transferer */
	struct sip_dual *d;
	int ext;
	int res;

	d = stuff;
	transferee = d->chan1;
	transferer = d->chan2;

	ast_debug(4, "SIP Park: Transferer channel %s, Transferee %s\n", transferer->name, transferee->name);

	res = ast_park_call_exten(transferee, transferer, d->park_exten, d->park_context, 0, &ext);

#ifdef WHEN_WE_KNOW_THAT_THE_CLIENT_SUPPORTS_MESSAGE
	if (res) {
		transmit_message_with_text(transferer->tech_pvt, "Unable to park call.\n");
	} else {
		/* Then tell the transferer what happened */
		sprintf(buf, "Call parked on extension '%d'", ext);
		transmit_message_with_text(transferer->tech_pvt, buf);
	}
#endif

	/* Any way back to the current call??? */
	/* Transmit response to the REFER request */
	if (!res)	{
		/* Transfer succeeded */
		append_history(transferer->tech_pvt, "SIPpark", "Parked call on %d", ext);
		transmit_notify_with_sipfrag(transferer->tech_pvt, d->seqno, "200 OK", TRUE);
		transferer->hangupcause = AST_CAUSE_NORMAL_CLEARING;
		ast_hangup(transferer); /* This will cause a BYE */
		ast_debug(1, "SIP Call parked on extension '%d'\n", ext);
	} else {
		transmit_notify_with_sipfrag(transferer->tech_pvt, d->seqno, "503 Service Unavailable", TRUE);
		append_history(transferer->tech_pvt, "SIPpark", "Parking failed\n");
		ast_debug(1, "SIP Call parked failed \n");
		/* Do not hangup call */
	}
	deinit_req(&d->req);
	ast_free(d->park_exten);
	ast_free(d->park_context);
	ast_free(d);
	return NULL;
}

/*! DO NOT hold any locks while calling sip_park */
static int sip_park(struct ast_channel *chan1, struct ast_channel *chan2, struct sip_request *req, uint32_t seqno, const char *park_exten, const char *park_context)
{
	struct sip_dual *d;
	struct ast_channel *transferee, *transferer;
	pthread_t th;

	transferee = ast_channel_alloc(0, AST_STATE_DOWN, 0, 0, chan1->accountcode, chan1->exten, chan1->context, chan1->linkedid, chan1->amaflags, "Parking/%s", chan1->name);
	transferer = ast_channel_alloc(0, AST_STATE_DOWN, 0, 0, chan2->accountcode, chan2->exten, chan2->context, chan2->linkedid, chan2->amaflags, "SIPPeer/%s", chan2->name);
	d = ast_calloc(1, sizeof(*d));
	if (!transferee || !transferer || !d) {
		if (transferee) {
			ast_hangup(transferee);
		}
		if (transferer) {
			ast_hangup(transferer);
		}
		ast_free(d);
		return -1;
	}
	d->park_exten = ast_strdup(park_exten);
	d->park_context = ast_strdup(park_context);
	if (!d->park_exten || !d->park_context) {
		ast_hangup(transferee);
		ast_hangup(transferer);
		ast_free(d->park_exten);
		ast_free(d->park_context);
		ast_free(d);
		return -1;
	}

	/* Make formats okay */
	transferee->readformat = chan1->readformat;
	transferee->writeformat = chan1->writeformat;

	/* Prepare for taking over the channel */
	if (ast_channel_masquerade(transferee, chan1)) {
		ast_hangup(transferee);
		ast_hangup(transferer);
		ast_free(d->park_exten);
		ast_free(d->park_context);
		ast_free(d);
		return -1;
	}

	/* Setup the extensions and such */
	ast_copy_string(transferee->context, chan1->context, sizeof(transferee->context));
	ast_copy_string(transferee->exten, chan1->exten, sizeof(transferee->exten));
	transferee->priority = chan1->priority;

	ast_do_masquerade(transferee);

	/* We make a clone of the peer channel too, so we can play
	   back the announcement */

	/* Make formats okay */
	transferer->readformat = chan2->readformat;
	transferer->writeformat = chan2->writeformat;
	ast_string_field_set(transferer, parkinglot, chan2->parkinglot);

	/* Prepare for taking over the channel */
	if (ast_channel_masquerade(transferer, chan2)) {
		ast_hangup(transferer);
		ast_free(d->park_exten);
		ast_free(d->park_context);
		ast_free(d);
		return -1;
	}

	/* Setup the extensions and such */
	ast_copy_string(transferer->context, chan2->context, sizeof(transferer->context));
	ast_copy_string(transferer->exten, chan2->exten, sizeof(transferer->exten));
	transferer->priority = chan2->priority;

	ast_do_masquerade(transferer);

	/* Save original request for followup */
	copy_request(&d->req, req);
	d->chan1 = transferee;	/* Transferee */
	d->chan2 = transferer;	/* Transferer */
	d->seqno = seqno;
	if (ast_pthread_create_detached_background(&th, NULL, sip_park_thread, d) < 0) {
		/* Could not start thread */
		deinit_req(&d->req);
		ast_free(d->park_exten);
		ast_free(d->park_context);
		ast_free(d);	/* We don't need it anymore. If thread is created, d will be free'd
				   by sip_park_thread() */
		return -1;
	}
	return 0;
}


/*! \brief SIP pickup support function
 *	Starts in a new thread, then pickup the call
 */
static void *sip_pickup_thread(void *stuff)
{
	struct ast_channel *chan;
	chan = stuff;

	if (ast_pickup_call(chan)) {
		chan->hangupcause = AST_CAUSE_CALL_REJECTED;
	} else {
		chan->hangupcause = AST_CAUSE_NORMAL_CLEARING;
	}
	ast_hangup(chan);
	ast_channel_unref(chan);
	chan = NULL;
	return NULL;
}

/*! \brief Pickup a call using the subsystem in features.c
 *	This is executed in a separate thread
 */
static int sip_pickup(struct ast_channel *chan)
{
	pthread_t threadid;

	ast_channel_ref(chan);

	if (ast_pthread_create_detached_background(&threadid, NULL, sip_pickup_thread, chan)) {
		ast_debug(1, "Unable to start Group pickup thread on channel %s\n", chan->name);
		ast_channel_unref(chan);
		return -1;
	}
	ast_debug(1, "Started Group pickup thread on channel %s\n", chan->name);
	return 0;
}


/*! \brief Turn off generator data
	XXX Does this function belong in the SIP channel?
*/
static void ast_quiet_chan(struct ast_channel *chan)
{
	if (chan && chan->_state == AST_STATE_UP) {
		if (ast_test_flag(chan, AST_FLAG_MOH))
			ast_moh_stop(chan);
		else if (chan->generatordata)
			ast_deactivate_generator(chan);
	}
}

/*! \brief Attempt transfer of SIP call
	This fix for attended transfers on a local PBX */
static int attempt_transfer(struct sip_dual *transferer, struct sip_dual *target)
{
	int res = 0;
	struct ast_channel *peera = NULL,	
		*peerb = NULL,
		*peerc = NULL,
		*peerd = NULL;


	/* We will try to connect the transferee with the target and hangup
	   all channels to the transferer */	
	ast_debug(4, "Sip transfer:--------------------\n");
	if (transferer->chan1)
		ast_debug(4, "-- Transferer to PBX channel: %s State %s\n", transferer->chan1->name, ast_state2str(transferer->chan1->_state));
	else
		ast_debug(4, "-- No transferer first channel - odd??? \n");
	if (target->chan1)
		ast_debug(4, "-- Transferer to PBX second channel (target): %s State %s\n", target->chan1->name, ast_state2str(target->chan1->_state));
	else
		ast_debug(4, "-- No target first channel ---\n");
	if (transferer->chan2)
		ast_debug(4, "-- Bridged call to transferee: %s State %s\n", transferer->chan2->name, ast_state2str(transferer->chan2->_state));
	else
		ast_debug(4, "-- No bridged call to transferee\n");
	if (target->chan2)
		ast_debug(4, "-- Bridged call to transfer target: %s State %s\n", target->chan2 ? target->chan2->name : "<none>", target->chan2 ? ast_state2str(target->chan2->_state) : "(none)");
	else
		ast_debug(4, "-- No target second channel ---\n");
	ast_debug(4, "-- END Sip transfer:--------------------\n");
	if (transferer->chan2) { /* We have a bridge on the transferer's channel */
		peera = transferer->chan1;	/* Transferer - PBX -> transferee channel * the one we hangup */
		peerb = target->chan1;		/* Transferer - PBX -> target channel - This will get lost in masq */
		peerc = transferer->chan2;	/* Asterisk to Transferee */
		peerd = target->chan2;		/* Asterisk to Target */
		ast_debug(3, "SIP transfer: Four channels to handle\n");
	} else if (target->chan2) {	/* Transferer has no bridge (IVR), but transferee */
		peera = target->chan1;		/* Transferer to PBX -> target channel */
		peerb = transferer->chan1;	/* Transferer to IVR*/
		peerc = target->chan2;		/* Asterisk to Target */
		peerd = transferer->chan2;	/* Nothing */
		ast_debug(3, "SIP transfer: Three channels to handle\n");
	}

	if (peera && peerb && peerc && (peerb != peerc)) {
		ast_quiet_chan(peera);		/* Stop generators */
		ast_quiet_chan(peerb);	
		ast_quiet_chan(peerc);
		if (peerd)
			ast_quiet_chan(peerd);

		ast_debug(4, "SIP transfer: trying to masquerade %s into %s\n", peerc->name, peerb->name);
		if (ast_channel_masquerade(peerb, peerc)) {
			ast_log(LOG_WARNING, "Failed to masquerade %s into %s\n", peerb->name, peerc->name);
			res = -1;
		} else
			ast_debug(4, "SIP transfer: Succeeded to masquerade channels.\n");
		return res;
	} else {
		ast_log(LOG_NOTICE, "SIP Transfer attempted with no appropriate bridged calls to transfer\n");
		if (transferer->chan1)
			ast_softhangup_nolock(transferer->chan1, AST_SOFTHANGUP_DEV);
		if (target->chan1)
			ast_softhangup_nolock(target->chan1, AST_SOFTHANGUP_DEV);
		return -1;
	}
	return 0;
}

/*! \brief Get tag from packet
 *
 * \return Returns the pointer to the provided tag buffer,
 *         or NULL if the tag was not found.
 */
static const char *gettag(const struct sip_request *req, const char *header, char *tagbuf, int tagbufsize)
{
	const char *thetag;

	if (!tagbuf)
		return NULL;
	tagbuf[0] = '\0'; 	/* reset the buffer */
	thetag = get_header(req, header);
	thetag = strcasestr(thetag, ";tag=");
	if (thetag) {
		thetag += 5;
		ast_copy_string(tagbuf, thetag, tagbufsize);
		return strsep(&tagbuf, ";");
	}
	return NULL;
}

static int handle_cc_notify(struct sip_pvt *pvt, struct sip_request *req)
{
	struct sip_monitor_instance *monitor_instance = ao2_callback(sip_monitor_instances, 0,
			find_sip_monitor_instance_by_subscription_pvt, pvt);
	const char *status = get_body(req, "cc-state", ':');
	struct cc_epa_entry *cc_entry;
	char *uri;

	if (!monitor_instance) {
		transmit_response(pvt, "400 Bad Request", req);
		return -1;
	}

	if (ast_strlen_zero(status)) {
		ao2_ref(monitor_instance, -1);
		transmit_response(pvt, "400 Bad Request", req);
		return -1;
	}

	if (!strcmp(status, "queued")) {
		/* We've been told that we're queued. This is the endpoint's way of telling
		 * us that it has accepted our CC request. We need to alert the core of this
		 * development
		 */
		ast_cc_monitor_request_acked(monitor_instance->core_id, "SIP endpoint %s accepted request", monitor_instance->device_name);
		transmit_response(pvt, "200 OK", req);
		ao2_ref(monitor_instance, -1);
		return 0;
	}

	/* It's open! Yay! */
	uri = get_body(req, "cc-URI", ':');
	if (ast_strlen_zero(uri)) {
		uri = get_in_brackets((char *)get_header(req, "From"));
	}

	ast_string_field_set(monitor_instance, notify_uri, uri);
	if (monitor_instance->suspension_entry) {
		cc_entry = monitor_instance->suspension_entry->instance_data;
		if (cc_entry->current_state == CC_CLOSED) {
			/* If we've created a suspension entry and the current state is closed, then that means
			 * we got a notice from the CC core earlier to suspend monitoring, but because this particular
			 * call leg had not yet notified us that it was ready for recall, it meant that we
			 * could not yet send a PUBLISH. Now, however, we can.
			 */
			construct_pidf_body(CC_CLOSED, monitor_instance->suspension_entry->body,
					sizeof(monitor_instance->suspension_entry->body), monitor_instance->peername);
			transmit_publish(monitor_instance->suspension_entry, SIP_PUBLISH_INITIAL, monitor_instance->notify_uri);
		} else {
			ast_cc_monitor_callee_available(monitor_instance->core_id, "SIP monitored callee has become available");
		}
	} else {
		ast_cc_monitor_callee_available(monitor_instance->core_id, "SIP monitored callee has become available");
	}
	ao2_ref(monitor_instance, -1);
	transmit_response(pvt, "200 OK", req);

	return 0;
}

/*! \brief Handle incoming notifications */
static int handle_request_notify(struct sip_pvt *p, struct sip_request *req, struct ast_sockaddr *addr, uint32_t seqno, const char *e)
{
	/* This is mostly a skeleton for future improvements */
	/* Mostly created to return proper answers on notifications on outbound REFER's */
	int res = 0;
	const char *event = get_header(req, "Event");
	char *sep;

	if( (sep = strchr(event, ';')) ) {	/* XXX bug here - overwriting string ? */
		*sep++ = '\0';
	}
	
	if (sipdebug)
		ast_debug(2, "Got NOTIFY Event: %s\n", event);

	if (!strcmp(event, "refer")) {
		/* Save nesting depth for now, since there might be other events we will
			support in the future */

		/* Handle REFER notifications */

		char buf[1024];
		char *cmd, *code;
		int respcode;
		int success = TRUE;

		/* EventID for each transfer... EventID is basically the REFER cseq

		 We are getting notifications on a call that we transfered
		 We should hangup when we are getting a 200 OK in a sipfrag
		 Check if we have an owner of this event */
		
		/* Check the content type */
		if (strncasecmp(get_header(req, "Content-Type"), "message/sipfrag", strlen("message/sipfrag"))) {
			/* We need a sipfrag */
			transmit_response(p, "400 Bad request", req);
			sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
			return -1;
		}

		/* Get the text of the attachment */
		if (get_msg_text(buf, sizeof(buf), req)) {
			ast_log(LOG_WARNING, "Unable to retrieve attachment from NOTIFY %s\n", p->callid);
			transmit_response(p, "400 Bad request", req);
			sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
			return -1;
		}

		/*
		From the RFC...
		A minimal, but complete, implementation can respond with a single
		NOTIFY containing either the body:
			SIP/2.0 100 Trying
		
		if the subscription is pending, the body:
			SIP/2.0 200 OK
		if the reference was successful, the body:
			SIP/2.0 503 Service Unavailable
		if the reference failed, or the body:
			SIP/2.0 603 Declined

		if the REFER request was accepted before approval to follow the
		reference could be obtained and that approval was subsequently denied
		(see Section 2.4.7).
		
		If there are several REFERs in the same dialog, we need to
		match the ID of the event header...
		*/
		ast_debug(3, "* SIP Transfer NOTIFY Attachment: \n---%s\n---\n", buf);
		cmd = ast_skip_blanks(buf);
		code = cmd;
		/* We are at SIP/2.0 */
		while(*code && (*code > 32)) {	/* Search white space */
			code++;
		}
		*code++ = '\0';
		code = ast_skip_blanks(code);
		sep = code;
		sep++;
		while(*sep && (*sep > 32)) {	/* Search white space */
			sep++;
		}
		*sep++ = '\0';			/* Response string */
		respcode = atoi(code);
		switch (respcode) {
		case 200:	/* OK: The new call is up, hangup this call */
			/* Hangup the call that we are replacing */
			break;
		case 301: /* Moved permenantly */
		case 302: /* Moved temporarily */
			/* Do we get the header in the packet in this case? */
			success = FALSE;
			break;
		case 503:	/* Service Unavailable: The new call failed */
		case 603:	/* Declined: Not accepted */
				/* Cancel transfer, continue the current call */
			success = FALSE;
			break;
		case 0:		/* Parse error */
				/* Cancel transfer, continue the current call */
			ast_log(LOG_NOTICE, "Error parsing sipfrag in NOTIFY in response to REFER.\n");
			success = FALSE;
			break;
		default:
			if (respcode < 200) {
				/* ignore provisional responses */
				success = -1;
			} else {
				ast_log(LOG_NOTICE, "Got unknown code '%d' in NOTIFY in response to REFER.\n", respcode);
				success = FALSE;
			}
			break;
		}
		if (success == FALSE) {
			ast_log(LOG_NOTICE, "Transfer failed. Sorry. Nothing further to do with this call\n");
		}

		if (p->owner && success != -1) {
			enum ast_control_transfer message = success ? AST_TRANSFER_SUCCESS : AST_TRANSFER_FAILED;
			ast_queue_control_data(p->owner, AST_CONTROL_TRANSFER, &message, sizeof(message));
		}
		/* Confirm that we received this packet */
		transmit_response(p, "200 OK", req);
	} else if (!strcmp(event, "message-summary")) {
		const char *mailbox = NULL;
		char *c = ast_strdupa(get_body(req, "Voice-Message", ':'));

		if (!p->mwi) {
			struct sip_peer *peer = find_peer(NULL, &p->recv, TRUE, FINDPEERS, FALSE, p->socket.type);

			if (peer) {
				mailbox = ast_strdupa(peer->unsolicited_mailbox);
				unref_peer(peer, "removing unsolicited mwi ref");
			}
		} else {
			mailbox = p->mwi->mailbox;
		}

		if (!ast_strlen_zero(mailbox) && !ast_strlen_zero(c)) {
			char *old = strsep(&c, " ");
			char *new = strsep(&old, "/");
			struct ast_event *event;

			if ((event = ast_event_new(AST_EVENT_MWI,
						   AST_EVENT_IE_MAILBOX, AST_EVENT_IE_PLTYPE_STR, mailbox,
						   AST_EVENT_IE_CONTEXT, AST_EVENT_IE_PLTYPE_STR, "SIP_Remote",
						   AST_EVENT_IE_NEWMSGS, AST_EVENT_IE_PLTYPE_UINT, atoi(new),
						   AST_EVENT_IE_OLDMSGS, AST_EVENT_IE_PLTYPE_UINT, atoi(old),
						   AST_EVENT_IE_END))) {
				ast_event_queue_and_cache(event);
			}
			transmit_response(p, "200 OK", req);
		} else {
			transmit_response(p, "489 Bad event", req);
			res = -1;
		}
	} else if (!strcmp(event, "keep-alive")) {
		 /* Used by Sipura/Linksys for NAT pinhole,
		  * just confirm that we received the packet. */
		transmit_response(p, "200 OK", req);
	} else if (!strcmp(event, "call-completion")) {
		res = handle_cc_notify(p, req);
	} else {
		/* We don't understand this event. */
		transmit_response(p, "489 Bad event", req);
		res = -1;
	}

	if (!p->lastinvite)
		sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);

	return res;
}

/*! \brief Handle incoming OPTIONS request
	An OPTIONS request should be answered like an INVITE from the same UA, including SDP
*/
static int handle_request_options(struct sip_pvt *p, struct sip_request *req, struct ast_sockaddr *addr, const char *e)
{
	const char *msg;
	enum sip_get_dest_result gotdest;
	int res;

	if (p->lastinvite) {
		/* if this is a request in an active dialog, just confirm that the dialog exists. */
		transmit_response_with_allow(p, "200 OK", req, 0);
		return 0;
	}

	if (sip_cfg.auth_options_requests) {
		/* Do authentication if this OPTIONS request began the dialog */
		copy_request(&p->initreq, req);
		set_pvt_allowed_methods(p, req);
		res = check_user(p, req, SIP_OPTIONS, e, XMIT_UNRELIABLE, addr);
		if (res == AUTH_CHALLENGE_SENT) {
			sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
			return 0;
		}
		if (res < 0) { /* Something failed in authentication */
			if (res == AUTH_FAKE_AUTH) {
				ast_log(LOG_NOTICE, "Sending fake auth rejection for device %s\n", get_header(req, "From"));
				transmit_fake_auth_response(p, SIP_OPTIONS, req, XMIT_UNRELIABLE);
			} else {
				ast_log(LOG_NOTICE, "Failed to authenticate device %s\n", get_header(req, "From"));
				transmit_response(p, "403 Forbidden", req);
			}
			sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
			return 0;
		}
	}

	/* must go through authentication before getting here */
	gotdest = get_destination(p, req, NULL);
	build_contact(p);

	if (ast_strlen_zero(p->context))
		ast_string_field_set(p, context, sip_cfg.default_context);

	if (ast_shutting_down()) {
		msg = "503 Unavailable";
	} else {
		msg = "404 Not Found";
		switch (gotdest) {
		case SIP_GET_DEST_INVALID_URI:
			msg = "416 Unsupported URI scheme";
			break;
		case SIP_GET_DEST_EXTEN_MATCHMORE:
		case SIP_GET_DEST_REFUSED:
		case SIP_GET_DEST_EXTEN_NOT_FOUND:
			//msg = "404 Not Found";
			break;
		case SIP_GET_DEST_EXTEN_FOUND:
			msg = "200 OK";
			break;
		}
	}
	transmit_response_with_allow(p, msg, req, 0);

	/* Destroy if this OPTIONS was the opening request, but not if
	   it's in the middle of a normal call flow. */
	sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);

	return 0;
}

/*! \brief Handle the transfer part of INVITE with a replaces: header,
    meaning a target pickup or an attended transfer.
    Used only once.
	XXX 'ignore' is unused.

	\note this function is called by handle_request_invite(). Four locks
	held at the beginning of this function, p, p->owner, p->refer->refer_call and
	p->refere->refer_call->owner.  only p's lock should remain at the end of this
	function.  p's lock as well as the channel p->owner's lock are held by
	handle_request_do(), we unlock p->owner before the masq.  By setting nounlock
	we are indicating to handle_request_do() that we have already unlocked the owner.
 */
static int handle_invite_replaces(struct sip_pvt *p, struct sip_request *req, int debug, uint32_t seqno, struct ast_sockaddr *addr, int *nounlock)
{
	int earlyreplace = 0;
	int oneleggedreplace = 0;		/* Call with no bridge, propably IVR or voice message */
	struct ast_channel *c = p->owner;	/* Our incoming call */
	struct ast_channel *replacecall = p->refer->refer_call->owner;	/* The channel we're about to take over */
	struct ast_channel *targetcall;		/* The bridge to the take-over target */

	/* Check if we're in ring state */
	if (replacecall->_state == AST_STATE_RING)
		earlyreplace = 1;

	/* Check if we have a bridge */
	if (!(targetcall = ast_bridged_channel(replacecall))) {
		/* We have no bridge */
		if (!earlyreplace) {
			ast_debug(2, "	Attended transfer attempted to replace call with no bridge (maybe ringing). Channel %s!\n", replacecall->name);
			oneleggedreplace = 1;
		}
	}
	if (targetcall && targetcall->_state == AST_STATE_RINGING)
		ast_debug(4, "SIP transfer: Target channel is in ringing state\n");

	if (targetcall)
		ast_debug(4, "SIP transfer: Invite Replace incoming channel should bridge to channel %s while hanging up channel %s\n", targetcall->name, replacecall->name);
	else
		ast_debug(4, "SIP transfer: Invite Replace incoming channel should replace and hang up channel %s (one call leg)\n", replacecall->name);

	if (req->ignore) {
		ast_log(LOG_NOTICE, "Ignoring this INVITE with replaces in a stupid way.\n");
		/* We should answer something here. If we are here, the
			call we are replacing exists, so an accepted
			can't harm */
		transmit_response_with_sdp(p, "200 OK", req, XMIT_RELIABLE, FALSE, FALSE);
		/* Do something more clever here */
		if (c) {
			*nounlock = 1;
			ast_channel_unlock(c);
		}
		ast_channel_unlock(replacecall);
		sip_pvt_unlock(p->refer->refer_call);
		return 1;
	}
	if (!c) {
		/* What to do if no channel ??? */
		ast_log(LOG_ERROR, "Unable to create new channel.  Invite/replace failed.\n");
		transmit_response_reliable(p, "503 Service Unavailable", req);
		append_history(p, "Xfer", "INVITE/Replace Failed. No new channel.");
		sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
		ast_channel_unlock(replacecall);
		sip_pvt_unlock(p->refer->refer_call);
		return 1;
	}
	append_history(p, "Xfer", "INVITE/Replace received");
	/* We have three channels to play with
		channel c: New incoming call
		targetcall: Call from PBX to target
		p->refer->refer_call: SIP pvt dialog from transferer to pbx.
		replacecall: The owner of the previous
		We need to masq C into refer_call to connect to
		targetcall;
		If we are talking to internal audio stream, target call is null.
	*/

	/* Fake call progress */
	transmit_response(p, "100 Trying", req);
	ast_setstate(c, AST_STATE_RING);

	/* Masquerade the new call into the referred call to connect to target call
	   Targetcall is not touched by the masq */

	/* Answer the incoming call and set channel to UP state */
	transmit_response_with_sdp(p, "200 OK", req, XMIT_RELIABLE, FALSE, FALSE);

	ast_setstate(c, AST_STATE_UP);

	/* Stop music on hold and other generators */
	ast_quiet_chan(replacecall);
	ast_quiet_chan(targetcall);
	ast_debug(4, "Invite/Replaces: preparing to masquerade %s into %s\n", c->name, replacecall->name);

	/* Make sure that the masq does not free our PVT for the old call */
	if (! earlyreplace && ! oneleggedreplace )
		ast_set_flag(&p->refer->refer_call->flags[0], SIP_DEFER_BYE_ON_TRANSFER);	/* Delay hangup */

	/* Prepare the masquerade - if this does not happen, we will be gone */
	if(ast_channel_masquerade(replacecall, c))
		ast_log(LOG_ERROR, "Failed to masquerade C into Replacecall\n");
	else
		ast_debug(4, "Invite/Replaces: Going to masquerade %s into %s\n", c->name, replacecall->name);

	/* C should now be in place of replacecall. all channel locks and pvt locks should be removed
	 * before issuing the masq.  Since we are unlocking both the pvt (p) and its owner channel (c)
	 * it is possible for channel c to be destroyed on us.  To prevent this, we must give c a reference
	 * before any unlocking takes place and remove it only once we are completely done with it */
	ast_channel_ref(c);
	ast_channel_unlock(replacecall);
	ast_channel_unlock(c);
	sip_pvt_unlock(p->refer->refer_call);
	sip_pvt_unlock(p);
	if (ast_do_masquerade(replacecall)) {
		ast_log(LOG_WARNING, "Failed to perform masquerade with INVITE replaces\n");
	}
	ast_channel_lock(c);
	if (earlyreplace || oneleggedreplace ) {
		c->hangupcause = AST_CAUSE_SWITCH_CONGESTION;
	}
	ast_setstate(c, AST_STATE_DOWN);
	ast_channel_unlock(c);

	/* The call should be down with no ast_channel, so hang it up */
	c->tech_pvt = dialog_unref(c->tech_pvt, "unref dialog c->tech_pvt");

	/* c and c's tech pvt must be unlocked at this point for ast_hangup */
	ast_hangup(c);
	/* this indicates to handle_request_do that the owner channel has already been unlocked */
	*nounlock = 1;
	/* lock PVT structure again after hangup */
	sip_pvt_lock(p);
	ast_channel_unref(c);
	return 0;
}

/*! \note No channel or pvt locks should be held while calling this function. */
static int do_magic_pickup(struct ast_channel *channel, const char *extension, const char *context)
{
	struct ast_str *str = ast_str_alloca(AST_MAX_EXTENSION + AST_MAX_CONTEXT + 2);
	struct ast_app *pickup = pbx_findapp("Pickup");

	if (!pickup) {
		ast_log(LOG_ERROR, "Unable to perform pickup: Application 'Pickup' not loaded (app_directed_pickup.so).\n");
		return -1;
	}

	ast_str_set(&str, 0, "%s@%s", extension, sip_cfg.notifycid == IGNORE_CONTEXT ? "PICKUPMARK" : context);

	ast_debug(2, "About to call Pickup(%s)\n", str->str);

	/* There is no point in capturing the return value since pickup_exec
	   doesn't return anything meaningful unless the passed data is an empty
	   string (which in our case it will not be) */
	pbx_exec(channel, pickup, str->str);

	return 0;
}

/*! \brief Called to deny a T38 reinvite if the core does not respond to our request */
static int sip_t38_abort(const void *data)
{
	struct sip_pvt *p = (struct sip_pvt *) data;

	sip_pvt_lock(p);
	/* an application may have taken ownership of the T.38 negotiation on this
	 * channel while we were waiting to grab the lock... if it did, the scheduler
	 * id will have been reset to -1, which is our indication that we do *not*
	 * want to abort the negotiation process
	 */
	if (p->t38id != -1) {
		change_t38_state(p, T38_DISABLED);
		transmit_response_reliable(p, "488 Not acceptable here", &p->initreq);
		p->t38id = -1;
		dialog_unref(p, "unref the dialog ptr from sip_t38_abort, because it held a dialog ptr");
	}
	sip_pvt_unlock(p);
	return 0;
}

/*!
 * \brief bare-bones support for SIP UPDATE
 *
 * XXX This is not even close to being RFC 3311-compliant. We don't advertise
 * that we support the UPDATE method, so no one should ever try sending us
 * an UPDATE anyway. However, Asterisk can send an UPDATE to change connected
 * line information, so we need to be prepared to handle this. The way we distinguish
 * such an UPDATE is through the X-Asterisk-rpid-update header.
 *
 * Actually updating the media session may be some future work.
 */
static int handle_request_update(struct sip_pvt *p, struct sip_request *req)
{
	if (ast_strlen_zero(get_header(req, "X-Asterisk-rpid-update"))) {
		transmit_response(p, "501 Method Not Implemented", req);
		return 0;
	}
	if (!p->owner) {
		transmit_response(p, "481 Call/Transaction Does Not Exist", req);
		return 0;
	}
	if (get_rpid(p, req)) {
		struct ast_party_connected_line connected;
		struct ast_set_party_connected_line update_connected;

		ast_party_connected_line_init(&connected);
		memset(&update_connected, 0, sizeof(update_connected));

		update_connected.id.number = 1;
		connected.id.number.valid = 1;
		connected.id.number.str = (char *) p->cid_num;
		connected.id.number.presentation = p->callingpres;

		update_connected.id.name = 1;
		connected.id.name.valid = 1;
		connected.id.name.str = (char *) p->cid_name;
		connected.id.name.presentation = p->callingpres;

		connected.id.tag = (char *) p->cid_tag;
		connected.source = AST_CONNECTED_LINE_UPDATE_SOURCE_TRANSFER;
		ast_channel_queue_connected_line_update(p->owner, &connected, &update_connected);
	}
	transmit_response(p, "200 OK", req);
	return 0;
}

/*!
 * \brief Handle incoming INVITE request
 * \note If the INVITE has a Replaces header, it is part of an
 *	attended transfer. If so, we do not go through the dial
 *	plan but try to find the active call and masquerade
 *	into it
 */
static int handle_request_invite(struct sip_pvt *p, struct sip_request *req, int debug, uint32_t seqno, struct ast_sockaddr *addr, int *recount, const char *e, int *nounlock)
{
	int res = 1;
	int gotdest;
	const char *p_replaces;
	char *replace_id = NULL;
	int refer_locked = 0;
	const char *required;
	unsigned int required_profile = 0;
	struct ast_channel *c = NULL;		/* New channel */
	struct sip_peer *authpeer = NULL;	/* Matching Peer */
	int reinvite = 0;
	int rtn;
	struct ast_party_redirecting redirecting;
	struct ast_set_party_redirecting update_redirecting;

	const char *p_uac_se_hdr;       /* UAC's Session-Expires header string                      */
	const char *p_uac_min_se;       /* UAC's requested Min-SE interval (char string)            */
	int uac_max_se = -1;            /* UAC's Session-Expires in integer format                  */
	int uac_min_se = -1;            /* UAC's Min-SE in integer format                           */
	int st_active = FALSE;          /* Session-Timer on/off boolean                             */
	int st_interval = 0;            /* Session-Timer negotiated refresh interval                */
	enum st_refresher st_ref;       /* Session-Timer session refresher                          */
	int dlg_min_se = -1;
	struct {
		char exten[AST_MAX_EXTENSION];
		char context[AST_MAX_CONTEXT];
	} pickup = {
			.exten = "",
	};
	st_ref = SESSION_TIMER_REFRESHER_AUTO;

	/* Find out what they support */
	if (!p->sipoptions) {
		const char *supported = get_header(req, "Supported");
		if (!ast_strlen_zero(supported)) {
			p->sipoptions = parse_sip_options(supported, NULL, 0);
		}
	}

	/* Find out what they require */
	required = get_header(req, "Require");
	if (!ast_strlen_zero(required)) {
		char unsupported[256] = { 0, };
		required_profile = parse_sip_options(required, unsupported, ARRAY_LEN(unsupported));

		/* If there are any options required that we do not support,
		 * then send a 420 with only those unsupported options listed */
		if (!ast_strlen_zero(unsupported)) {
			transmit_response_with_unsupported(p, "420 Bad extension (unsupported)", req, unsupported);
			ast_log(LOG_WARNING, "Received SIP INVITE with unsupported required extension: required:%s unsupported:%s\n", required, unsupported);
			p->invitestate = INV_COMPLETED;
			if (!p->lastinvite)
				sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
			res = -1;
			goto request_invite_cleanup;
		}
	}

	/* The option tags may be present in Supported: or Require: headers.
	Include the Require: option tags for further processing as well */
	p->sipoptions |= required_profile;
	p->reqsipoptions = required_profile;

	/* Check if this is a loop */
	if (ast_test_flag(&p->flags[0], SIP_OUTGOING) && p->owner && (p->invitestate != INV_TERMINATED && p->invitestate != INV_CONFIRMED) && p->owner->_state != AST_STATE_UP) {
		/* This is a call to ourself.  Send ourselves an error code and stop
	   	processing immediately, as SIP really has no good mechanism for
	   	being able to call yourself */
		/* If pedantic is on, we need to check the tags. If they're different, this is
	   	in fact a forked call through a SIP proxy somewhere. */
		int different;
		const char *initial_rlPart2 = REQ_OFFSET_TO_STR(&p->initreq, rlPart2);
		const char *this_rlPart2 = REQ_OFFSET_TO_STR(req, rlPart2);
		if (sip_cfg.pedanticsipchecking)
			different = sip_uri_cmp(initial_rlPart2, this_rlPart2);
		else
			different = strcmp(initial_rlPart2, this_rlPart2);
		if (!different) {
			transmit_response(p, "482 Loop Detected", req);
			p->invitestate = INV_COMPLETED;
			sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
			res = 0;
			goto request_invite_cleanup;
		} else {
			/*! This is a spiral. What we need to do is to just change the outgoing INVITE
			 * so that it now routes to the new Request URI. Since we created the INVITE ourselves
			 * that should be all we need to do.
			 *
 			 * \todo XXX This needs to be reviewed.  YOu don't change the request URI really, you route the packet
			 * correctly instead...
			 */
			char *uri = ast_strdupa(this_rlPart2);
			char *at = strchr(uri, '@');
			char *peerorhost;
			ast_debug(2, "Potential spiral detected. Original RURI was %s, new RURI is %s\n", initial_rlPart2, this_rlPart2);
			transmit_response(p, "100 Trying", req);
			if (at) {
				*at = '\0';
			}
			/* Parse out "sip:" */
			if ((peerorhost = strchr(uri, ':'))) {
				*peerorhost++ = '\0';
			}
			ast_string_field_set(p, theirtag, NULL);
			/* Treat this as if there were a call forward instead...
			 */
			ast_string_field_set(p->owner, call_forward, peerorhost);
			ast_queue_control(p->owner, AST_CONTROL_BUSY);
			res = 0;
			goto request_invite_cleanup;
		}
	}

	if (!req->ignore && p->pendinginvite) {
		if (!ast_test_flag(&p->flags[0], SIP_OUTGOING) && (p->invitestate == INV_COMPLETED || p->invitestate == INV_TERMINATED)) {
			/* What do these circumstances mean? We have received an INVITE for an "incoming" dialog for which we
			 * have sent a final response. We have not yet received an ACK, though (which is why p->pendinginvite is non-zero).
			 * We also know that the INVITE is not a retransmission, because otherwise the "ignore" flag would be set.
			 * This means that either we are receiving a reinvite for a terminated dialog, or we are receiving an INVITE with
			 * credentials based on one we challenged earlier.
			 *
			 * The action to take in either case is to treat the INVITE as though it contains an implicit ACK for the previous
			 * transaction. Calling __sip_ack will take care of this by clearing the p->pendinginvite and removing the response
			 * from the previous transaction from the list of outstanding packets.
			 */
			__sip_ack(p, p->pendinginvite, 1, 0);
		} else {
			/* We already have a pending invite. Sorry. You are on hold. */
			p->glareinvite = seqno;
			if (p->rtp && find_sdp(req)) {
				struct ast_sockaddr addr;
				if (get_ip_and_port_from_sdp(req, SDP_AUDIO, &addr)) {
					ast_log(LOG_WARNING, "Failed to set an alternate media source on glared reinvite. Audio may not work properly on this call.\n");
				} else {
					ast_rtp_instance_set_alt_remote_address(p->rtp, &addr);
				}
				if (p->vrtp) {
					if (get_ip_and_port_from_sdp(req, SDP_VIDEO, &addr)) {
						ast_log(LOG_WARNING, "Failed to set an alternate media source on glared reinvite. Video may not work properly on this call.\n");
					} else {
						ast_rtp_instance_set_alt_remote_address(p->vrtp, &addr);
					}
				}
			}
			transmit_response_reliable(p, "491 Request Pending", req);
			p->pendinginvite = seqno;
			check_via(p, req);
			copy_request(&p->initreq, req);
			ast_debug(1, "Got INVITE on call where we already have pending INVITE, deferring that - %s\n", p->callid);
			/* Don't destroy dialog here */
			res = 0;
			goto request_invite_cleanup;
		}
	}

	p_replaces = get_header(req, "Replaces");
	if (!ast_strlen_zero(p_replaces)) {
		/* We have a replaces header */
		char *ptr;
		char *fromtag = NULL;
		char *totag = NULL;
		char *start, *to;
		int error = 0;

		if (p->owner) {
			ast_debug(3, "INVITE w Replaces on existing call? Refusing action. [%s]\n", p->callid);
			transmit_response_reliable(p, "400 Bad request", req);	/* The best way to not not accept the transfer */
			p->pendinginvite = seqno;
			check_via(p, req);
			copy_request(&p->initreq, req);
			/* Do not destroy existing call */
			res = -1;
			goto request_invite_cleanup;
		}

		if (sipdebug)
			ast_debug(3, "INVITE part of call transfer. Replaces [%s]\n", p_replaces);
		/* Create a buffer we can manipulate */
		replace_id = ast_strdupa(p_replaces);
		ast_uri_decode(replace_id);

		if (!p->refer && !sip_refer_allocate(p)) {
			transmit_response_reliable(p, "500 Server Internal Error", req);
			append_history(p, "Xfer", "INVITE/Replace Failed. Out of memory.");
			sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
			p->invitestate = INV_COMPLETED;
			p->pendinginvite = seqno;
			check_via(p, req);
			copy_request(&p->initreq, req);
			res = -1;
			goto request_invite_cleanup;
		}

		/*  Todo: (When we find phones that support this)
			if the replaces header contains ";early-only"
			we can only replace the call in early
			stage, not after it's up.

			If it's not in early mode, 486 Busy.
		*/

		/* Skip leading whitespace */
		replace_id = ast_skip_blanks(replace_id);

		start = replace_id;
		while ( (ptr = strsep(&start, ";")) ) {
			ptr = ast_skip_blanks(ptr); /* XXX maybe unnecessary ? */
			if ( (to = strcasestr(ptr, "to-tag=") ) )
				totag = to + 7;	/* skip the keyword */
			else if ( (to = strcasestr(ptr, "from-tag=") ) ) {
				fromtag = to + 9;	/* skip the keyword */
				fromtag = strsep(&fromtag, "&"); /* trim what ? */
			}
		}

		if (sipdebug)
			ast_debug(4, "Invite/replaces: Will use Replace-Call-ID : %s Fromtag: %s Totag: %s\n",
					  replace_id,
					  fromtag ? fromtag : "<no from tag>",
					  totag ? totag : "<no to tag>");

		/* Try to find call that we are replacing.
		   If we have a Replaces header, we need to cancel that call if we succeed with this call.
		   First we cheat a little and look for a magic call-id from phones that support
		   dialog-info+xml so we can do technology independent pickup... */
		if (strncmp(replace_id, "pickup-", 7) == 0) {
			struct sip_pvt *subscription = NULL;
			replace_id += 7; /* Worst case we are looking at \0 */

			if ((subscription = get_sip_pvt_byid_locked(replace_id, totag, fromtag)) == NULL) {
				ast_log(LOG_NOTICE, "Unable to find subscription with call-id: %s\n", replace_id);
				transmit_response_reliable(p, "481 Call Leg Does Not Exist (Replaces)", req);
				error = 1;
			} else {
				ast_log(LOG_NOTICE, "Trying to pick up %s@%s\n", subscription->exten, subscription->context);
				ast_copy_string(pickup.exten, subscription->exten, sizeof(pickup.exten));
				ast_copy_string(pickup.context, subscription->context, sizeof(pickup.context));
				sip_pvt_unlock(subscription);
				if (subscription->owner) {
					ast_channel_unlock(subscription->owner);
				}
				subscription = dialog_unref(subscription, "unref dialog subscription");
			}
		}

		/* This locks both refer_call pvt and refer_call pvt's owner!!!*/
		if (!error && ast_strlen_zero(pickup.exten) && (p->refer->refer_call = get_sip_pvt_byid_locked(replace_id, totag, fromtag)) == NULL) {
			ast_log(LOG_NOTICE, "Supervised transfer attempted to replace non-existent call id (%s)!\n", replace_id);
			transmit_response_reliable(p, "481 Call Leg Does Not Exist (Replaces)", req);
			error = 1;
		} else {
			refer_locked = 1;
		}

		/* The matched call is the call from the transferer to Asterisk .
			We want to bridge the bridged part of the call to the
			incoming invite, thus taking over the refered call */

		if (p->refer->refer_call == p) {
			ast_log(LOG_NOTICE, "INVITE with replaces into it's own call id (%s == %s)!\n", replace_id, p->callid);
			transmit_response_reliable(p, "400 Bad request", req);	/* The best way to not not accept the transfer */
			error = 1;
		}

		if (!error && ast_strlen_zero(pickup.exten) && !p->refer->refer_call->owner) {
			/* Oops, someting wrong anyway, no owner, no call */
			ast_log(LOG_NOTICE, "Supervised transfer attempted to replace non-existing call id (%s)!\n", replace_id);
			/* Check for better return code */
			transmit_response_reliable(p, "481 Call Leg Does Not Exist (Replace)", req);
			error = 1;
		}

		if (!error && ast_strlen_zero(pickup.exten) && p->refer->refer_call->owner->_state != AST_STATE_RINGING && p->refer->refer_call->owner->_state != AST_STATE_RING && p->refer->refer_call->owner->_state != AST_STATE_UP) {
			ast_log(LOG_NOTICE, "Supervised transfer attempted to replace non-ringing or active call id (%s)!\n", replace_id);
			transmit_response_reliable(p, "603 Declined (Replaces)", req);
			error = 1;
		}

		if (error) {	/* Give up this dialog */
			append_history(p, "Xfer", "INVITE/Replace Failed.");
			sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
			sip_pvt_unlock(p);
			if (p->refer->refer_call) {
				sip_pvt_unlock(p->refer->refer_call);
				if (p->refer->refer_call->owner) {
					ast_channel_unlock(p->refer->refer_call->owner);
				}
				p->refer->refer_call = dialog_unref(p->refer->refer_call, "unref dialog p->refer->refer_call");
			}
			refer_locked = 0;
			p->invitestate = INV_COMPLETED;
			p->pendinginvite = seqno;
			check_via(p, req);
			copy_request(&p->initreq, req);
			res = -1;
			goto request_invite_cleanup;
		}
	}

	/* Check if this is an INVITE that sets up a new dialog or
	   a re-invite in an existing dialog */

	if (!req->ignore) {
		int newcall = (p->initreq.headers ? TRUE : FALSE);

		if (sip_cancel_destroy(p))
			ast_log(LOG_WARNING, "Unable to cancel SIP destruction.  Expect bad things.\n");
		/* This also counts as a pending invite */
		p->pendinginvite = seqno;
		check_via(p, req);

		copy_request(&p->initreq, req);		/* Save this INVITE as the transaction basis */
		if (sipdebug)
			ast_debug(1, "Initializing initreq for method %s - callid %s\n", sip_methods[req->method].text, p->callid);
		if (!p->owner) {	/* Not a re-invite */
			if (debug)
				ast_verbose("Using INVITE request as basis request - %s\n", p->callid);
			if (newcall)
				append_history(p, "Invite", "New call: %s", p->callid);
			parse_ok_contact(p, req);
		} else {	/* Re-invite on existing call */
			ast_clear_flag(&p->flags[0], SIP_OUTGOING);	/* This is now an inbound dialog */
			if (get_rpid(p, req)) {
				struct ast_party_connected_line connected;
				struct ast_set_party_connected_line update_connected;

				ast_party_connected_line_init(&connected);
				memset(&update_connected, 0, sizeof(update_connected));

				update_connected.id.number = 1;
				connected.id.number.valid = 1;
				connected.id.number.str = (char *) p->cid_num;
				connected.id.number.presentation = p->callingpres;

				update_connected.id.name = 1;
				connected.id.name.valid = 1;
				connected.id.name.str = (char *) p->cid_name;
				connected.id.name.presentation = p->callingpres;

				connected.id.tag = (char *) p->cid_tag;
				connected.source = AST_CONNECTED_LINE_UPDATE_SOURCE_TRANSFER;
				ast_channel_queue_connected_line_update(p->owner, &connected,
					&update_connected);
			}
			/* Handle SDP here if we already have an owner */
			if (find_sdp(req)) {
				if (process_sdp(p, req, SDP_T38_INITIATE)) {
					if (!ast_strlen_zero(get_header(req, "Content-Encoding"))) {
						/* Asterisk does not yet support any Content-Encoding methods.  Always
						 * attempt to process the sdp, but return a 415 if a Content-Encoding header
						 * was present after processing failed.  */
						transmit_response_reliable(p, "415 Unsupported Media type", req);
					} else {
						transmit_response_reliable(p, "488 Not acceptable here", req);
					}
					if (!p->lastinvite)
						sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
					res = -1;
					goto request_invite_cleanup;
				}
				ast_queue_control(p->owner, AST_CONTROL_SRCUPDATE);
			} else {
				p->jointcapability = p->capability;
				ast_debug(1, "Hm....  No sdp for the moment\n");
				/* Some devices signal they want to be put off hold by sending a re-invite
				   *without* an SDP, which is supposed to mean "Go back to your state"
				   and since they put os on remote hold, we go back to off hold */
				if (ast_test_flag(&p->flags[1], SIP_PAGE2_CALL_ONHOLD)) {
					ast_queue_control(p->owner, AST_CONTROL_UNHOLD);
					/* Activate a re-invite */
					ast_queue_frame(p->owner, &ast_null_frame);
					change_hold_state(p, req, FALSE, 0);
				}
			}
			if (p->do_history) /* This is a response, note what it was for */
				append_history(p, "ReInv", "Re-invite received");
		}
	} else if (debug)
		ast_verbose("Ignoring this INVITE request\n");

	if (!p->lastinvite && !req->ignore && !p->owner) {
		/* This is a new invite */
		/* Handle authentication if this is our first invite */
		int cc_recall_core_id = -1;
		set_pvt_allowed_methods(p, req);
		res = check_user_full(p, req, SIP_INVITE, e, XMIT_RELIABLE, addr, &authpeer);
		if (res == AUTH_CHALLENGE_SENT) {
			p->invitestate = INV_COMPLETED;		/* Needs to restart in another INVITE transaction */
			res = 0;
			goto request_invite_cleanup;
		}
		if (res < 0) { /* Something failed in authentication */
			if (res == AUTH_FAKE_AUTH) {
				ast_log(LOG_NOTICE, "Sending fake auth rejection for device %s\n", get_header(req, "From"));
				transmit_fake_auth_response(p, SIP_INVITE, req, XMIT_RELIABLE);
			} else {
				ast_log(LOG_NOTICE, "Failed to authenticate device %s\n", get_header(req, "From"));
				transmit_response_reliable(p, "403 Forbidden", req);
			}
			p->invitestate = INV_COMPLETED;
			sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
			res = 0;
			goto request_invite_cleanup;
		}

		/* Successful authentication and peer matching so record the peer related to this pvt (for easy access to peer settings) */
		if (p->relatedpeer) {
			p->relatedpeer = unref_peer(p->relatedpeer,"unsetting the relatedpeer field in the dialog, before it is set to something else.");
		}
		if (authpeer) {
			p->relatedpeer = ref_peer(authpeer, "setting dialog's relatedpeer pointer");
		}

		req->authenticated = 1;

		/* We have a successful authentication, process the SDP portion if there is one */
		if (find_sdp(req)) {
			if (process_sdp(p, req, SDP_T38_INITIATE)) {
				/* Asterisk does not yet support any Content-Encoding methods.  Always
				 * attempt to process the sdp, but return a 415 if a Content-Encoding header
				 * was present after processing fails. */
				if (!ast_strlen_zero(get_header(req, "Content-Encoding"))) {
					transmit_response_reliable(p, "415 Unsupported Media type", req);
				} else {
					/* Unacceptable codecs */
					transmit_response_reliable(p, "488 Not acceptable here", req);
				}
				p->invitestate = INV_COMPLETED;
				sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
				ast_debug(1, "No compatible codecs for this SIP call.\n");
				res = -1;
				goto request_invite_cleanup;
			}
		} else {	/* No SDP in invite, call control session */
			p->jointcapability = p->capability;
			ast_debug(2, "No SDP in Invite, third party call control\n");
		}

		/* Queue NULL frame to prod ast_rtp_bridge if appropriate */
		/* This seems redundant ... see !p-owner above */
		if (p->owner)
			ast_queue_frame(p->owner, &ast_null_frame);


		/* Initialize the context if it hasn't been already */
		if (ast_strlen_zero(p->context))
			ast_string_field_set(p, context, sip_cfg.default_context);


		/* Check number of concurrent calls -vs- incoming limit HERE */
		ast_debug(1, "Checking SIP call limits for device %s\n", p->username);
		if ((res = update_call_counter(p, INC_CALL_LIMIT))) {
			if (res < 0) {
				ast_log(LOG_NOTICE, "Failed to place call for device %s, too many calls\n", p->username);
				transmit_response_reliable(p, "480 Temporarily Unavailable (Call limit) ", req);
				sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
				p->invitestate = INV_COMPLETED;
			}
			res = 0;
			goto request_invite_cleanup;
		}
		gotdest = get_destination(p, NULL, &cc_recall_core_id);	/* Get destination right away */
		extract_uri(p, req);			/* Get the Contact URI */
		build_contact(p);			/* Build our contact header */

		if (p->rtp) {
			ast_rtp_instance_set_prop(p->rtp, AST_RTP_PROPERTY_DTMF, ast_test_flag(&p->flags[0], SIP_DTMF) == SIP_DTMF_RFC2833);
			ast_rtp_instance_set_prop(p->rtp, AST_RTP_PROPERTY_DTMF_COMPENSATE, ast_test_flag(&p->flags[1], SIP_PAGE2_RFC2833_COMPENSATE));
		}

		if (!replace_id && (gotdest != SIP_GET_DEST_EXTEN_FOUND)) {	/* No matching extension found */
			switch(gotdest) {
			case SIP_GET_DEST_INVALID_URI:
				transmit_response_reliable(p, "416 Unsupported URI scheme", req);
				break;
			case SIP_GET_DEST_EXTEN_MATCHMORE:
				if (ast_test_flag(&p->flags[1], SIP_PAGE2_ALLOWOVERLAP)
					== SIP_PAGE2_ALLOWOVERLAP_YES) {
					transmit_response_reliable(p, "484 Address Incomplete", req);
					break;
				}
				/*
				 * XXX We would have to implement collecting more digits in
				 * chan_sip for any other schemes of overlap dialing.
				 *
				 * For SIP_PAGE2_ALLOWOVERLAP_DTMF it is better to do this in
				 * the dialplan using the Incomplete application rather than
				 * having the channel driver do it.
				 */
				/* Fall through */
			case SIP_GET_DEST_EXTEN_NOT_FOUND:
			case SIP_GET_DEST_REFUSED:
			default:
				{
					char *decoded_exten = ast_strdupa(p->exten);
					transmit_response_reliable(p, "404 Not Found", req);
					ast_uri_decode(decoded_exten);
					ast_log(LOG_NOTICE, "Call from '%s' (%s) to extension"
						" '%s' rejected because extension not found in context '%s'.\n",
						S_OR(p->username, p->peername), ast_sockaddr_stringify(&p->recv), decoded_exten, p->context);
				}
			} /* end switch */

			p->invitestate = INV_COMPLETED;
			update_call_counter(p, DEC_CALL_LIMIT);
			sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
			res = 0;
			goto request_invite_cleanup;
		} else {

			/* If no extension was specified, use the s one */
			/* Basically for calling to IP/Host name only */
			if (ast_strlen_zero(p->exten))
				ast_string_field_set(p, exten, "s");
			/* Initialize our tag */

			make_our_tag(p->tag, sizeof(p->tag));
			/* First invitation - create the channel.  Allocation
			 * failures are handled below. */
			c = sip_new(p, AST_STATE_DOWN, S_OR(p->peername, NULL), NULL);
			if (cc_recall_core_id != -1) {
				ast_setup_cc_recall_datastore(c, cc_recall_core_id);
				ast_cc_agent_set_interfaces_chanvar(c);
			}
			*recount = 1;

			/* Save Record-Route for any later requests we make on this dialogue */
			build_route(p, req, 0, 0);

			if (c) {
				ast_party_redirecting_init(&redirecting);
				memset(&update_redirecting, 0, sizeof(update_redirecting));
				change_redirecting_information(p, req, &redirecting, &update_redirecting,
					FALSE); /*Will return immediately if no Diversion header is present */
				ast_channel_set_redirecting(c, &redirecting, &update_redirecting);
				ast_party_redirecting_free(&redirecting);
			}
		}
	} else {
		ast_party_redirecting_init(&redirecting);
		memset(&update_redirecting, 0, sizeof(update_redirecting));
		if (sipdebug) {
			if (!req->ignore)
				ast_debug(2, "Got a SIP re-invite for call %s\n", p->callid);
			else
				ast_debug(2, "Got a SIP re-transmit of INVITE for call %s\n", p->callid);
		}
		if (!req->ignore)
			reinvite = 1;
		c = p->owner;
		change_redirecting_information(p, req, &redirecting, &update_redirecting, FALSE); /*Will return immediately if no Diversion header is present */
		if (c) {
			ast_channel_set_redirecting(c, &redirecting, &update_redirecting);
		}
		ast_party_redirecting_free(&redirecting);
	}

	/* Session-Timers */
	if ((p->sipoptions & SIP_OPT_TIMER) && !ast_strlen_zero(get_header(req, "Session-Expires"))) {
		/* The UAC has requested session-timers for this session. Negotiate
		the session refresh interval and who will be the refresher */
		ast_debug(2, "Incoming INVITE with 'timer' option supported and \"Session-Expires\" header.\n");

		/* Allocate Session-Timers struct w/in the dialog */
		if (!p->stimer)
			sip_st_alloc(p);

		/* Parse the Session-Expires header */
		p_uac_se_hdr = get_header(req, "Session-Expires");
		rtn = parse_session_expires(p_uac_se_hdr, &uac_max_se, &st_ref);
		if (rtn != 0) {
			transmit_response_reliable(p, "400 Session-Expires Invalid Syntax", req);
			p->invitestate = INV_COMPLETED;
			if (!p->lastinvite) {
				sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
			}
			res = -1;
			goto request_invite_cleanup;
		}

		/* Parse the Min-SE header */
		p_uac_min_se = get_header(req, "Min-SE");
		if (!ast_strlen_zero(p_uac_min_se)) {
			rtn = parse_minse(p_uac_min_se, &uac_min_se);
			if (rtn != 0) {
				transmit_response_reliable(p, "400 Min-SE Invalid Syntax", req);
				p->invitestate = INV_COMPLETED;
				if (!p->lastinvite) {
					sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
				}
				res = -1;
				goto request_invite_cleanup;
			}
		}

		dlg_min_se = st_get_se(p, FALSE);
		switch (st_get_mode(p, 1)) {
		case SESSION_TIMER_MODE_ACCEPT:
		case SESSION_TIMER_MODE_ORIGINATE:
			if (uac_max_se > 0 && uac_max_se < dlg_min_se) {
				transmit_response_with_minse(p, "422 Session Interval Too Small", req, dlg_min_se);
				p->invitestate = INV_COMPLETED;
				if (!p->lastinvite) {
					sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
				}
				res = -1;
				goto request_invite_cleanup;
			}

			p->stimer->st_active_peer_ua = TRUE;
			st_active = TRUE;
			if (st_ref == SESSION_TIMER_REFRESHER_AUTO) {
				st_ref = st_get_refresher(p);
			}

			if (uac_max_se > 0) {
				int dlg_max_se = st_get_se(p, TRUE);
				if (dlg_max_se >= uac_min_se) {
					st_interval = (uac_max_se < dlg_max_se) ? uac_max_se : dlg_max_se;
				} else {
					st_interval = uac_max_se;
				}
			} else {
				/* Set to default max value */
				st_interval = global_max_se;
			}
			break;

		case SESSION_TIMER_MODE_REFUSE:
			if (p->reqsipoptions & SIP_OPT_TIMER) {
				transmit_response_with_unsupported(p, "420 Option Disabled", req, required);
				ast_log(LOG_WARNING, "Received SIP INVITE with supported but disabled option: %s\n", required);
				p->invitestate = INV_COMPLETED;
				if (!p->lastinvite) {
					sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
				}
				res = -1;
				goto request_invite_cleanup;
			}
			break;

		default:
			ast_log(LOG_ERROR, "Internal Error %d at %s:%d\n", st_get_mode(p, 1), __FILE__, __LINE__);
			break;
		}
	} else {
		/* The UAC did not request session-timers.  Asterisk (UAS), will now decide
		(based on session-timer-mode in sip.conf) whether to run session-timers for
		this session or not. */
		switch (st_get_mode(p, 1)) {
		case SESSION_TIMER_MODE_ORIGINATE:
			st_active = TRUE;
			st_interval = st_get_se(p, TRUE);
			st_ref = SESSION_TIMER_REFRESHER_UAS;
			p->stimer->st_active_peer_ua = FALSE;
			break;

		default:
			break;
		}
	}

	if (reinvite == 0) {
		/* Session-Timers: Start session refresh timer based on negotiation/config */
		if (st_active == TRUE) {
			p->stimer->st_active   = TRUE;
			p->stimer->st_interval = st_interval;
			p->stimer->st_ref      = st_ref;
			start_session_timer(p);
		}
	} else {
		if (p->stimer->st_active == TRUE) {
			/* Session-Timers:  A re-invite request sent within a dialog will serve as
			a refresh request, no matter whether the re-invite was sent for refreshing
			the session or modifying it.*/
			ast_debug (2, "Restarting session-timers on a refresh - %s\n", p->callid);

			/* The UAC may be adjusting the session-timers mid-session */
			if (st_interval > 0) {
				p->stimer->st_interval = st_interval;
				p->stimer->st_ref      = st_ref;
			}

			restart_session_timer(p);
			if (p->stimer->st_expirys > 0) {
				p->stimer->st_expirys--;
			}
		}
	}

	if (!req->ignore && p)
		p->lastinvite = seqno;

	if (c && replace_id) {	/* Attended transfer or call pickup - we're the target */
		if (!ast_strlen_zero(pickup.exten)) {
			append_history(p, "Xfer", "INVITE/Replace received");

			/* Let the caller know we're giving it a shot */
			transmit_response(p, "100 Trying", req);
			p->invitestate = INV_PROCEEDING;
			ast_setstate(c, AST_STATE_RING);

			/* Do the pickup itself */
			ast_channel_unlock(c);
			*nounlock = 1;

			/* since p->owner (c) is unlocked, we need to go ahead and unlock pvt for both
			 * magic pickup and ast_hangup.  Both of these functions will attempt to lock
			 * p->owner again, which can cause a deadlock if we already hold a lock on p.
			 * Locking order is, channel then pvt.  Dead lock avoidance must be used if
			 * called the other way around. */
			sip_pvt_unlock(p);
			do_magic_pickup(c, pickup.exten, pickup.context);
			/* Now we're either masqueraded or we failed to pickup, in either case we... */
			ast_hangup(c);
			sip_pvt_lock(p); /* pvt is expected to remain locked on return, so re-lock it */

			res = 0;
			goto request_invite_cleanup;
		} else {
			/* Go and take over the target call */
			if (sipdebug)
				ast_debug(4, "Sending this call to the invite/replcaes handler %s\n", p->callid);
			res = handle_invite_replaces(p, req, debug, seqno, addr, nounlock);
			refer_locked = 0;
			goto request_invite_cleanup;
		}
	}


	if (c) {	/* We have a call  -either a new call or an old one (RE-INVITE) */
		enum ast_channel_state c_state = c->_state;

		if (c_state != AST_STATE_UP && reinvite &&
			(p->invitestate == INV_TERMINATED || p->invitestate == INV_CONFIRMED)) {
			/* If these conditions are true, and the channel is still in the 'ringing'
			 * state, then this likely means that we have a situation where the initial
			 * INVITE transaction has completed *but* the channel's state has not yet been
			 * changed to UP. The reason this could happen is if the reinvite is received
			 * on the SIP socket prior to an application calling ast_read on this channel
			 * to read the answer frame we earlier queued on it. In this case, the reinvite
			 * is completely legitimate so we need to handle this the same as if the channel
			 * were already UP. Thus we are purposely falling through to the AST_STATE_UP case.
			 */
			c_state = AST_STATE_UP;
		}

		switch(c_state) {
		case AST_STATE_DOWN:
			ast_debug(2, "%s: New call is still down.... Trying... \n", c->name);
			transmit_provisional_response(p, "100 Trying", req, 0);
			p->invitestate = INV_PROCEEDING;
			ast_setstate(c, AST_STATE_RING);
			if (strcmp(p->exten, ast_pickup_ext())) {	/* Call to extension -start pbx on this call */
				enum ast_pbx_result result;

				result = ast_pbx_start(c);

				switch(result) {
				case AST_PBX_FAILED:
					ast_log(LOG_WARNING, "Failed to start PBX :(\n");
					p->invitestate = INV_COMPLETED;
					transmit_response_reliable(p, "503 Unavailable", req);
					break;
				case AST_PBX_CALL_LIMIT:
					ast_log(LOG_WARNING, "Failed to start PBX (call limit reached) \n");
					p->invitestate = INV_COMPLETED;
					transmit_response_reliable(p, "480 Temporarily Unavailable", req);
					break;
				case AST_PBX_SUCCESS:
					/* nothing to do */
					break;
				}

				if (result) {

					/* Unlock locks so ast_hangup can do its magic */
					ast_channel_unlock(c);
					*nounlock = 1;
					sip_pvt_unlock(p);
					ast_hangup(c);
					sip_pvt_lock(p);
					c = NULL;
				}
			} else {	/* Pickup call in call group */
				if (sip_pickup(c)) {
					ast_log(LOG_WARNING, "Failed to start Group pickup by %s\n", c->name);
					transmit_response_reliable(p, "480 Temporarily Unavailable", req);
					sip_alreadygone(p);
					c->hangupcause = AST_CAUSE_FAILURE;

					/* Unlock locks so ast_hangup can do its magic */
					ast_channel_unlock(c);
					*nounlock = 1;

					p->invitestate = INV_COMPLETED;
					sip_pvt_unlock(p);
					ast_hangup(c);
					sip_pvt_lock(p);
					c = NULL;
				}
			}
			break;
		case AST_STATE_RING:
			transmit_provisional_response(p, "100 Trying", req, 0);
			p->invitestate = INV_PROCEEDING;
			break;
		case AST_STATE_RINGING:
			transmit_provisional_response(p, "180 Ringing", req, 0);
			p->invitestate = INV_PROCEEDING;
			break;
		case AST_STATE_UP:
			ast_debug(2, "%s: This call is UP.... \n", c->name);

			transmit_response(p, "100 Trying", req);

			if (p->t38.state == T38_PEER_REINVITE) {
				if (p->t38id > -1) {
					/* reset t38 abort timer */
					AST_SCHED_DEL_UNREF(sched, p->t38id, dialog_unref(p, "remove ref for t38id"));
				}
				p->t38id = ast_sched_add(sched, 5000, sip_t38_abort, dialog_ref(p, "passing dialog ptr into sched structure based on t38id for sip_t38_abort."));
			} else if (p->t38.state == T38_ENABLED) {
				ast_set_flag(&p->flags[1], SIP_PAGE2_DIALOG_ESTABLISHED);
				transmit_response_with_t38_sdp(p, "200 OK", req, (reinvite ? XMIT_RELIABLE : (req->ignore ?  XMIT_UNRELIABLE : XMIT_CRITICAL)));
			} else if (p->t38.state == T38_DISABLED) {
				/* If this is not a re-invite or something to ignore - it's critical */
				if (p->srtp && !ast_test_flag(p->srtp, SRTP_CRYPTO_OFFER_OK)) {
					ast_log(LOG_WARNING, "Target does not support required crypto\n");
					transmit_response_reliable(p, "488 Not Acceptable Here (crypto)", req);
				} else {
					ast_set_flag(&p->flags[1], SIP_PAGE2_DIALOG_ESTABLISHED);
					transmit_response_with_sdp(p, "200 OK", req, (reinvite ? XMIT_RELIABLE : (req->ignore ?  XMIT_UNRELIABLE : XMIT_CRITICAL)), p->session_modify == TRUE ? FALSE : TRUE, FALSE);
					ast_queue_control(p->owner, AST_CONTROL_UPDATE_RTP_PEER);
				}
			}

			p->invitestate = INV_TERMINATED;
			break;
		default:
			ast_log(LOG_WARNING, "Don't know how to handle INVITE in state %d\n", c->_state);
			transmit_response(p, "100 Trying", req);
			break;
		}
	} else {
		if (p && (p->autokillid == -1)) {
			const char *msg;

			if (!p->jointcapability)
				msg = "488 Not Acceptable Here (codec error)";
			else {
				ast_log(LOG_NOTICE, "Unable to create/find SIP channel for this INVITE\n");
				msg = "503 Unavailable";
			}
			transmit_response_reliable(p, msg, req);
			p->invitestate = INV_COMPLETED;
			sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
		}
	}

request_invite_cleanup:

	if (refer_locked && p->refer && p->refer->refer_call) {
		sip_pvt_unlock(p->refer->refer_call);
		if (p->refer->refer_call->owner) {
			ast_channel_unlock(p->refer->refer_call->owner);
		}
		p->refer->refer_call = dialog_unref(p->refer->refer_call, "unref dialog p->refer->refer_call");
	}
	if (authpeer) {
		authpeer = unref_peer(authpeer, "unref_peer, from handle_request_invite authpeer");
	}

	return res;
}

/*! \brief  Find all call legs and bridge transferee with target
 *	called from handle_request_refer
 *
 *	\note this function assumes two locks to begin with, sip_pvt transferer and current.chan1 (the pvt's owner)... 
 *	2 additional locks are held at the beginning of the function, targetcall_pvt, and targetcall_pvt's owner
 *	channel (which is stored in target.chan1).  These 2 locks _MUST_ be let go by the end of the function.  Do
 *	not be confused into thinking a pvt's owner is the same thing as the channels locked at the beginning of
 *	this function, after the masquerade this may not be true.  Be consistent and unlock only the exact same
 *	pointers that were locked to begin with.
 *
 *	If this function is successful, only the transferer pvt lock will remain on return.  Setting nounlock indicates
 *	to handle_request_do() that the pvt's owner it locked does not require an unlock.
 */
static int local_attended_transfer(struct sip_pvt *transferer, struct sip_dual *current, struct sip_request *req, uint32_t seqno, int *nounlock)
{
	struct sip_dual target;		/* Chan 1: Call from tranferer to Asterisk */
					/* Chan 2: Call from Asterisk to target */
	int res = 0;
	struct sip_pvt *targetcall_pvt;
	struct ast_party_connected_line connected_to_transferee;
	struct ast_party_connected_line connected_to_target;
	char transferer_linkedid[32];
	struct ast_channel *chans[2];

	/* Check if the call ID of the replaces header does exist locally */
	if (!(targetcall_pvt = get_sip_pvt_byid_locked(transferer->refer->replaces_callid, transferer->refer->replaces_callid_totag,
		transferer->refer->replaces_callid_fromtag))) {
		if (transferer->refer->localtransfer) {
			/* We did not find the refered call. Sorry, can't accept then */
			/* Let's fake a response from someone else in order
		   	to follow the standard */
			transmit_notify_with_sipfrag(transferer, seqno, "481 Call leg/transaction does not exist", TRUE);
			append_history(transferer, "Xfer", "Refer failed");
			ast_clear_flag(&transferer->flags[0], SIP_GOTREFER);
			transferer->refer->status = REFER_FAILED;
			return -1;
		}
		/* Fall through for remote transfers that we did not find locally */
		ast_debug(3, "SIP attended transfer: Not our call - generating INVITE with replaces\n");
		return 0;
	}

	/* Ok, we can accept this transfer */
	append_history(transferer, "Xfer", "Refer accepted");
	if (!targetcall_pvt->owner) {	/* No active channel */
		ast_debug(4, "SIP attended transfer: Error: No owner of target call\n");
		/* Cancel transfer */
		transmit_notify_with_sipfrag(transferer, seqno, "503 Service Unavailable", TRUE);
		append_history(transferer, "Xfer", "Refer failed");
		ast_clear_flag(&transferer->flags[0], SIP_GOTREFER);
		transferer->refer->status = REFER_FAILED;
		sip_pvt_unlock(targetcall_pvt);
		if (targetcall_pvt)
			ao2_t_ref(targetcall_pvt, -1, "Drop targetcall_pvt pointer");
		return -1;
	}

	/* We have a channel, find the bridge */
	target.chan1 = targetcall_pvt->owner;				/* Transferer to Asterisk */
	target.chan2 = ast_bridged_channel(targetcall_pvt->owner);	/* Asterisk to target */

	if (!target.chan2 || !(target.chan2->_state == AST_STATE_UP || target.chan2->_state == AST_STATE_RINGING) ) {
		/* Wrong state of new channel */
		if (target.chan2)
			ast_debug(4, "SIP attended transfer: Error: Wrong state of target call: %s\n", ast_state2str(target.chan2->_state));
		else if (target.chan1->_state != AST_STATE_RING)
			ast_debug(4, "SIP attended transfer: Error: No target channel\n");
		else
			ast_debug(4, "SIP attended transfer: Attempting transfer in ringing state\n");
	}

	/* Transfer */
	if (sipdebug) {
		if (current->chan2)	/* We have two bridges */
			ast_debug(4, "SIP attended transfer: trying to bridge %s and %s\n", target.chan1->name, current->chan2->name);
		else			/* One bridge, propably transfer of IVR/voicemail etc */
			ast_debug(4, "SIP attended transfer: trying to make %s take over (masq) %s\n", target.chan1->name, current->chan1->name);
	}

	ast_set_flag(&transferer->flags[0], SIP_DEFER_BYE_ON_TRANSFER);	/* Delay hangup */

	ast_copy_string(transferer_linkedid, transferer->owner->linkedid, sizeof(transferer_linkedid));

	/* Perform the transfer */
	chans[0] = transferer->owner;
	chans[1] = target.chan1;
	ast_manager_event_multichan(EVENT_FLAG_CALL, "Transfer", 2, chans,
		"TransferMethod: SIP\r\n"
		"TransferType: Attended\r\n"
		"Channel: %s\r\n"
		"Uniqueid: %s\r\n"
		"SIP-Callid: %s\r\n"
		"TargetChannel: %s\r\n"
		"TargetUniqueid: %s\r\n",
		transferer->owner->name,
		transferer->owner->uniqueid,
		transferer->callid,
		target.chan1->name,
		target.chan1->uniqueid);
	ast_party_connected_line_init(&connected_to_transferee);
	ast_party_connected_line_init(&connected_to_target);
	/* No need to lock current->chan1 here since it was locked in sipsock_read */
	ast_party_connected_line_copy(&connected_to_transferee, &current->chan1->connected);
	/* No need to lock target.chan1 here since it was locked in get_sip_pvt_byid_locked */
	ast_party_connected_line_copy(&connected_to_target, &target.chan1->connected);
	connected_to_target.source = connected_to_transferee.source = AST_CONNECTED_LINE_UPDATE_SOURCE_TRANSFER;
	res = attempt_transfer(current, &target);
	if (res) {
		/* Failed transfer */
		transmit_notify_with_sipfrag(transferer, seqno, "486 Busy Here", TRUE);
		append_history(transferer, "Xfer", "Refer failed");
		ast_clear_flag(&transferer->flags[0], SIP_DEFER_BYE_ON_TRANSFER);
		/* if transfer failed, go ahead and unlock targetcall_pvt and it's owner channel */
		sip_pvt_unlock(targetcall_pvt);
		ast_channel_unlock(target.chan1);
	} else {
		/* Transfer succeeded! */
		const char *xfersound = pbx_builtin_getvar_helper(target.chan1, "ATTENDED_TRANSFER_COMPLETE_SOUND");

		/* target.chan1 was locked in get_sip_pvt_byid_locked, do not unlock target.chan1 before this */
		ast_cel_report_event(target.chan1, AST_CEL_ATTENDEDTRANSFER, NULL, transferer_linkedid, target.chan2);

		/* Tell transferer that we're done. */
		transmit_notify_with_sipfrag(transferer, seqno, "200 OK", TRUE);
		append_history(transferer, "Xfer", "Refer succeeded");
		transferer->refer->status = REFER_200OK;
		if (target.chan2 && !ast_strlen_zero(xfersound) && ast_streamfile(target.chan2, xfersound, target.chan2->language) >= 0) {
			ast_waitstream(target.chan2, "");
		}

		/* By forcing the masquerade, we know that target.chan1 and target.chan2 are bridged. We then
		 * can queue connected line updates where they need to go.
		 *
		 * before a masquerade, all channel and pvt locks must be unlocked.  Any recursive
		 * channel locks held before this function invalidates channel container locking order.
		 * Since we are unlocking both the pvt (transferer) and its owner channel (current.chan1)
		 * it is possible for current.chan1 to be destroyed in the pbx thread.  To prevent this
		 * we must give c a reference before any unlocking takes place.
		 */

		ast_channel_ref(current->chan1);
		ast_channel_unlock(current->chan1); /* current.chan1 is p->owner before the masq, it was locked by socket_read()*/
		ast_channel_unlock(target.chan1);
		*nounlock = 1;  /* we just unlocked the dialog's channel and have no plans of locking it again. */
		sip_pvt_unlock(targetcall_pvt);
		sip_pvt_unlock(transferer);

		ast_do_masquerade(target.chan1);

		ast_indicate(target.chan1, AST_CONTROL_UNHOLD);
		if (target.chan2) {
			ast_indicate(target.chan2, AST_CONTROL_UNHOLD);
		}

		if (current->chan2 && current->chan2->_state == AST_STATE_RING) {
			ast_indicate(target.chan1, AST_CONTROL_RINGING);
		}

		if (target.chan2) {
			ast_channel_queue_connected_line_update(target.chan1, &connected_to_transferee, NULL);
			ast_channel_queue_connected_line_update(target.chan2, &connected_to_target, NULL);
		} else {
			/* Since target.chan1 isn't actually connected to another channel, there is no way for us
			 * to queue a frame so that its connected line status will be updated.
			 *
			 * Instead, we use the somewhat hackish approach of using a special control frame type that
			 * instructs ast_read to perform a specific action. In this case, the frame we queue tells
			 * ast_read to call the connected line interception macro configured for target.chan1.
			 */
			struct ast_control_read_action_payload *frame_payload;
			int payload_size;
			int frame_size;
			unsigned char connected_line_data[1024];
			payload_size = ast_connected_line_build_data(connected_line_data,
				sizeof(connected_line_data), &connected_to_target, NULL);
			frame_size = payload_size + sizeof(*frame_payload);
			if (payload_size != -1 && (frame_payload = alloca(frame_size))) {
				frame_payload->payload_size = payload_size;
				memcpy(frame_payload->payload, connected_line_data, payload_size);
				frame_payload->action = AST_FRAME_READ_ACTION_CONNECTED_LINE_MACRO;
				ast_queue_control_data(target.chan1, AST_CONTROL_READ_ACTION, frame_payload, frame_size);
			}
			/* In addition to queueing the read action frame so that target.chan1's connected line info
			 * will be updated, we also are going to queue a plain old connected line update on target.chan1. This
			 * way, either Dial or Queue can apply this connected line update to the outgoing ringing channel.
			 */
			ast_channel_queue_connected_line_update(target.chan1, &connected_to_transferee, NULL);

		}
		sip_pvt_lock(transferer); /* the transferer pvt is expected to remain locked on return */

		ast_channel_unref(current->chan1);
	}

	/* at this point if the transfer is successful only the transferer pvt should be locked. */
	ast_party_connected_line_free(&connected_to_target);
	ast_party_connected_line_free(&connected_to_transferee);
	if (targetcall_pvt)
		ao2_t_ref(targetcall_pvt, -1, "drop targetcall_pvt");
	return 1;
}


/*! \brief Handle incoming REFER request */
/*! \page SIP_REFER SIP transfer Support (REFER)

	REFER is used for call transfer in SIP. We get a REFER
	to place a new call with an INVITE somwhere and then
	keep the transferor up-to-date of the transfer. If the
	transfer fails, get back on line with the orginal call.

	- REFER can be sent outside or inside of a dialog.
	  Asterisk only accepts REFER inside of a dialog.

	- If we get a replaces header, it is an attended transfer

	\par Blind transfers
	The transferor provides the transferee
	with the transfer targets contact. The signalling between
	transferer or transferee should not be cancelled, so the
	call is recoverable if the transfer target can not be reached
	by the transferee.

	In this case, Asterisk receives a TRANSFER from
	the transferor, thus is the transferee. We should
	try to set up a call to the contact provided
	and if that fails, re-connect the current session.
	If the new call is set up, we issue a hangup.
	In this scenario, we are following section 5.2
	in the SIP CC Transfer draft. (Transfer without
	a GRUU)

	\par Transfer with consultation hold
	In this case, the transferor
	talks to the transfer target before the transfer takes place.
	This is implemented with SIP hold and transfer.
	Note: The invite From: string could indicate a transfer.
	(Section 6. Transfer with consultation hold)
	The transferor places the transferee on hold, starts a call
	with the transfer target to alert them to the impending
	transfer, terminates the connection with the target, then
	proceeds with the transfer (as in Blind transfer above)

	\par Attended transfer
	The transferor places the transferee
	on hold, calls the transfer target to alert them,
	places the target on hold, then proceeds with the transfer
	using a Replaces header field in the Refer-to header. This
	will force the transfee to send an Invite to the target,
	with a replaces header that instructs the target to
	hangup the call between the transferor and the target.
	In this case, the Refer/to: uses the AOR address. (The same
	URI that the transferee used to establish the session with
	the transfer target (To: ). The Require: replaces header should
	be in the INVITE to avoid the wrong UA in a forked SIP proxy
	scenario to answer and have no call to replace with.

	The referred-by header is *NOT* required, but if we get it,
	can be copied into the INVITE to the transfer target to
	inform the target about the transferor

	"Any REFER request has to be appropriately authenticated.".
	
	We can't destroy dialogs, since we want the call to continue.
	
	*/
static int handle_request_refer(struct sip_pvt *p, struct sip_request *req, int debug, uint32_t seqno, int *nounlock)
{
	/*!
	 * Chan1: Call between asterisk and transferer
	 * Chan2: Call between asterisk and transferee
	 */
	struct sip_dual current = { 0, };
	struct ast_channel *chans[2] = { 0, };
	char *refer_to = NULL;
	char *refer_to_domain = NULL;
	char *refer_to_context = NULL;
	char *referred_by = NULL;
	char *callid = NULL;
	int localtransfer = 0;
	int attendedtransfer = 0;
	int res = 0;

	if (req->debug) {
		ast_verbose("Call %s got a SIP call transfer from %s: (REFER)!\n",
			p->callid,
			ast_test_flag(&p->flags[0], SIP_OUTGOING) ? "callee" : "caller");
	}

	if (!p->owner) {
		/* This is a REFER outside of an existing SIP dialog */
		/* We can't handle that, so decline it */
		ast_debug(3, "Call %s: Declined REFER, outside of dialog...\n", p->callid);
		transmit_response(p, "603 Declined (No dialog)", req);
		if (!req->ignore) {
			append_history(p, "Xfer", "Refer failed. Outside of dialog.");
			sip_alreadygone(p);
			pvt_set_needdestroy(p, "outside of dialog");
		}
		res = 0;
		goto handle_refer_cleanup;
	}

	/* Check if transfer is allowed from this device */
	if (p->allowtransfer == TRANSFER_CLOSED ) {
		/* Transfer not allowed, decline */
		transmit_response(p, "603 Declined (policy)", req);
		append_history(p, "Xfer", "Refer failed. Allowtransfer == closed.");
		/* Do not destroy SIP session */
		res = 0;
		goto handle_refer_cleanup;
	}

	if (!req->ignore && ast_test_flag(&p->flags[0], SIP_GOTREFER)) {
		/* Already have a pending REFER */
		transmit_response(p, "491 Request pending", req);
		append_history(p, "Xfer", "Refer failed. Request pending.");
		res = 0;
		goto handle_refer_cleanup;
	}

	/* Allocate memory for call transfer data */
	if (!p->refer && !sip_refer_allocate(p)) {
		transmit_response(p, "500 Internal Server Error", req);
		append_history(p, "Xfer", "Refer failed. Memory allocation error.");
		res = -3;
		goto handle_refer_cleanup;
	}

	res = get_refer_info(p, req);	/* Extract headers */

	p->refer->status = REFER_SENT;

	if (res != 0) {
		switch (res) {
		case -2:	/* Syntax error */
			transmit_response(p, "400 Bad Request (Refer-to missing)", req);
			append_history(p, "Xfer", "Refer failed. Refer-to missing.");
			if (req->debug) {
				ast_debug(1, "SIP transfer to black hole can't be handled (no refer-to: )\n");
			}
			break;
		case -3:
			transmit_response(p, "603 Declined (Non sip: uri)", req);
			append_history(p, "Xfer", "Refer failed. Non SIP uri");
			if (req->debug) {
				ast_debug(1, "SIP transfer to non-SIP uri denied\n");
			}
			break;
		default:
			/* Refer-to extension not found, fake a failed transfer */
			transmit_response(p, "202 Accepted", req);
			append_history(p, "Xfer", "Refer failed. Bad extension.");
			transmit_notify_with_sipfrag(p, seqno, "404 Not found", TRUE);
			ast_clear_flag(&p->flags[0], SIP_GOTREFER);	
			if (req->debug) {
				ast_debug(1, "SIP transfer to bad extension: %s\n", p->refer->refer_to);
			}
			break;
		}
		res = 0;
		goto handle_refer_cleanup;
	}
	if (ast_strlen_zero(p->context)) {
		ast_string_field_set(p, context, sip_cfg.default_context);
	}

	/* If we do not support SIP domains, all transfers are local */
	if (sip_cfg.allow_external_domains && check_sip_domain(p->refer->refer_to_domain, NULL, 0)) {
		p->refer->localtransfer = 1;
		if (sipdebug) {
			ast_debug(3, "This SIP transfer is local : %s\n", p->refer->refer_to_domain);
		}
	} else if (AST_LIST_EMPTY(&domain_list) || check_sip_domain(p->refer->refer_to_domain, NULL, 0)) {
		/* This PBX doesn't bother with SIP domains or domain is local, so this transfer is local */
		p->refer->localtransfer = 1;
	} else if (sipdebug) {
		ast_debug(3, "This SIP transfer is to a remote SIP extension (remote domain %s)\n", p->refer->refer_to_domain);
	}

	/* Is this a repeat of a current request? Ignore it */
	/* Don't know what else to do right now. */
	if (req->ignore) {
		goto handle_refer_cleanup;
	}

	/* If this is a blind transfer, we have the following
	channels to work with:
	- chan1, chan2: The current call between transferer and transferee (2 channels)
	- target_channel: A new call from the transferee to the target (1 channel)
	We need to stay tuned to what happens in order to be able
	to bring back the call to the transferer */

	/* If this is a attended transfer, we should have all call legs within reach:
	- chan1, chan2: The call between the transferer and transferee (2 channels)
	- target_channel, targetcall_pvt: The call between the transferer and the target (2 channels)
	We want to bridge chan2 with targetcall_pvt!
	
	The replaces call id in the refer message points
	to the call leg between Asterisk and the transferer.
	So we need to connect the target and the transferee channel
	and hangup the two other channels silently
	
	If the target is non-local, the call ID could be on a remote
	machine and we need to send an INVITE with replaces to the
	target. We basically handle this as a blind transfer
	and let the sip_call function catch that we need replaces
	header in the INVITE.
	*/

	/* Get the transferer's channel */
	chans[0] = current.chan1 = p->owner;

	/* Find the other part of the bridge (2) - transferee */
	chans[1] = current.chan2 = ast_bridged_channel(current.chan1);

	ast_channel_ref(current.chan1);
	if (current.chan2) {
		ast_channel_ref(current.chan2);
	}

	if (sipdebug) {
		ast_debug(3, "SIP %s transfer: Transferer channel %s, transferee channel %s\n",
			p->refer->attendedtransfer ? "attended" : "blind",
			current.chan1->name,
			current.chan2 ? current.chan2->name : "<none>");
	}

	if (!current.chan2 && !p->refer->attendedtransfer) {
		/* No bridged channel, propably IVR or echo or similar... */
		/* Guess we should masquerade or something here */
		/* Until we figure it out, refuse transfer of such calls */
		if (sipdebug) {
			ast_debug(3, "Refused SIP transfer on non-bridged channel.\n");
		}
		p->refer->status = REFER_FAILED;
		append_history(p, "Xfer", "Refer failed. Non-bridged channel.");
		transmit_response(p, "603 Declined", req);
		res = -1;
		goto handle_refer_cleanup;
	}

	if (current.chan2) {
		if (sipdebug) {
			ast_debug(4, "Got SIP transfer, applying to bridged peer '%s'\n", current.chan2->name);
		}
		ast_queue_control(current.chan1, AST_CONTROL_UNHOLD);
	}

	ast_set_flag(&p->flags[0], SIP_GOTREFER);

	/* From here on failures will be indicated with NOTIFY requests */
	transmit_response(p, "202 Accepted", req);

	/* Attended transfer: Find all call legs and bridge transferee with target*/
	if (p->refer->attendedtransfer) {
		/* both p and p->owner _MUST_ be locked while calling local_attended_transfer */
		if ((res = local_attended_transfer(p, &current, req, seqno, nounlock))) {
			goto handle_refer_cleanup; /* We're done with the transfer */
		}
		/* Fall through for remote transfers that we did not find locally */
		if (sipdebug) {
			ast_debug(4, "SIP attended transfer: Still not our call - generating INVITE with replaces\n");
		}
		/* Fallthrough if we can't find the call leg internally */
	}

	/* Copy data we can not safely access after letting the pvt lock go. */
	refer_to = ast_strdupa(p->refer->refer_to);
	refer_to_domain = ast_strdupa(p->refer->refer_to_domain);
	refer_to_context = ast_strdupa(p->refer->refer_to_context);
	referred_by = ast_strdupa(p->refer->referred_by);
	callid = ast_strdupa(p->callid);
	localtransfer = p->refer->localtransfer;
	attendedtransfer = p->refer->attendedtransfer;

	if (!*nounlock) {
		ast_channel_unlock(p->owner);
		*nounlock = 1;
	}
	sip_pvt_unlock(p);

	/* Parking a call.  DO NOT hold any locks while calling ast_parking_ext_valid() */
	if (localtransfer && ast_parking_ext_valid(refer_to, current.chan1, refer_to_context)) {
		sip_pvt_lock(p);
		ast_clear_flag(&p->flags[0], SIP_GOTREFER);
		p->refer->status = REFER_200OK;
		append_history(p, "Xfer", "REFER to call parking.");
		sip_pvt_unlock(p);

		ast_manager_event_multichan(EVENT_FLAG_CALL, "Transfer", 2, chans,
			"TransferMethod: SIP\r\n"
			"TransferType: Blind\r\n"
			"Channel: %s\r\n"
			"Uniqueid: %s\r\n"
			"SIP-Callid: %s\r\n"
			"TargetChannel: %s\r\n"
			"TargetUniqueid: %s\r\n"
			"TransferExten: %s\r\n"
			"Transfer2Parking: Yes\r\n",
			current.chan1->name,
			current.chan1->uniqueid,
			callid,
			current.chan2->name,
			current.chan2->uniqueid,
			refer_to);

		if (sipdebug) {
			ast_debug(4, "SIP transfer to parking: trying to park %s. Parked by %s\n", current.chan2->name, current.chan1->name);
		}

		/* DO NOT hold any locks while calling sip_park */
		if (sip_park(current.chan2, current.chan1, req, seqno, refer_to, refer_to_context)) {
			sip_pvt_lock(p);
			transmit_notify_with_sipfrag(p, seqno, "500 Internal Server Error", TRUE);
		} else {
			sip_pvt_lock(p);
		}
		goto handle_refer_cleanup;
	}

	/* Blind transfers and remote attended xfers.
	 * Locks should not be held while calling pbx_builtin_setvar_helper. This function
	 * locks the channel being passed into it.*/
	if (current.chan1 && current.chan2) {
		ast_debug(3, "chan1->name: %s\n", current.chan1->name);
		pbx_builtin_setvar_helper(current.chan1, "BLINDTRANSFER", current.chan2->name);
	}

	if (current.chan2) {
		pbx_builtin_setvar_helper(current.chan2, "BLINDTRANSFER", current.chan1->name);
		pbx_builtin_setvar_helper(current.chan2, "SIPDOMAIN", refer_to_domain);
		pbx_builtin_setvar_helper(current.chan2, "SIPTRANSFER", "yes");
		/* One for the new channel */
		pbx_builtin_setvar_helper(current.chan2, "_SIPTRANSFER", "yes");
		/* Attended transfer to remote host, prepare headers for the INVITE */
		if (!ast_strlen_zero(referred_by)) {
			pbx_builtin_setvar_helper(current.chan2, "_SIPTRANSFER_REFERER", referred_by);
		}
	}

	sip_pvt_lock(p);
	/* Generate a Replaces string to be used in the INVITE during attended transfer */
	if (!ast_strlen_zero(p->refer->replaces_callid)) {
		char tempheader[SIPBUFSIZE];
		snprintf(tempheader, sizeof(tempheader), "%s%s%s%s%s", p->refer->replaces_callid,
			p->refer->replaces_callid_totag ? ";to-tag=" : "",
			p->refer->replaces_callid_totag,
			p->refer->replaces_callid_fromtag ? ";from-tag=" : "",
			p->refer->replaces_callid_fromtag);

		if (current.chan2) {
			sip_pvt_unlock(p);
			pbx_builtin_setvar_helper(current.chan2, "_SIPTRANSFER_REPLACES", tempheader);
			sip_pvt_lock(p);
		}
	}

	/* Connect the call */

	/* FAKE ringing if not attended transfer */
	if (!p->refer->attendedtransfer) {
		transmit_notify_with_sipfrag(p, seqno, "180 Ringing", FALSE);
	}

	/* For blind transfer, this will lead to a new call */
	/* For attended transfer to remote host, this will lead to
	   a new SIP call with a replaces header, if the dial plan allows it
	*/
	if (!current.chan2) {
		/* We have no bridge, so we're talking with Asterisk somehow */
		/* We need to masquerade this call */
		/* What to do to fix this situation:
		   * Set up the new call in a new channel
		   * Let the new channel masq into this channel
		   Please add that code here :-)
		*/
		p->refer->status = REFER_FAILED;
		transmit_notify_with_sipfrag(p, seqno, "503 Service Unavailable (can't handle one-legged xfers)", TRUE);
		ast_clear_flag(&p->flags[0], SIP_GOTREFER);	
		append_history(p, "Xfer", "Refer failed (only bridged calls).");
		res = -1;
		goto handle_refer_cleanup;
	}
	ast_set_flag(&p->flags[0], SIP_DEFER_BYE_ON_TRANSFER);	/* Delay hangup */

	/* Do not hold the pvt lock during the indicate and async_goto. Those functions
	 * lock channels which will invalidate locking order if the pvt lock is held.*/
	/* For blind transfers, move the call to the new extensions. For attended transfers on multiple
	 * servers - generate an INVITE with Replaces. Either way, let the dial plan decided
	 * indicate before masquerade so the indication actually makes it to the real channel
	 * when using local channels with MOH passthru */
	sip_pvt_unlock(p);
	ast_indicate(current.chan2, AST_CONTROL_UNHOLD);
	res = ast_async_goto(current.chan2, refer_to_context, refer_to, 1);

	if (!res) {
		ast_manager_event_multichan(EVENT_FLAG_CALL, "Transfer", 2, chans,
			"TransferMethod: SIP\r\n"
			"TransferType: Blind\r\n"
			"Channel: %s\r\n"
			"Uniqueid: %s\r\n"
			"SIP-Callid: %s\r\n"
			"TargetChannel: %s\r\n"
			"TargetUniqueid: %s\r\n"
			"TransferExten: %s\r\n"
			"TransferContext: %s\r\n",
			current.chan1->name,
			current.chan1->uniqueid,
			callid,
			current.chan2->name,
			current.chan2->uniqueid,
			refer_to,
			refer_to_context);
		/* Success  - we have a new channel */
		ast_debug(3, "%s transfer succeeded. Telling transferer.\n", attendedtransfer? "Attended" : "Blind");

		/* XXX - what to we put in CEL 'extra' for attended transfers to external systems? NULL for now */
		ast_channel_lock(current.chan1);
		ast_cel_report_event(current.chan1, p->refer->attendedtransfer? AST_CEL_ATTENDEDTRANSFER : AST_CEL_BLINDTRANSFER, NULL, p->refer->attendedtransfer ? NULL : p->refer->refer_to, current.chan2);
		ast_channel_unlock(current.chan1);

		sip_pvt_lock(p);
		transmit_notify_with_sipfrag(p, seqno, "200 Ok", TRUE);
		if (p->refer->localtransfer) {
			p->refer->status = REFER_200OK;
		}
		if (p->owner) {
			p->owner->hangupcause = AST_CAUSE_NORMAL_CLEARING;
		}
		append_history(p, "Xfer", "Refer succeeded.");
		ast_clear_flag(&p->flags[0], SIP_GOTREFER);	
		/* Do not hangup call, the other side do that when we say 200 OK */
		/* We could possibly implement a timer here, auto congestion */
		res = 0;
	} else {
		sip_pvt_lock(p);
		ast_clear_flag(&p->flags[0], SIP_DEFER_BYE_ON_TRANSFER);	/* Don't delay hangup */
		ast_debug(3, "%s transfer failed. Resuming original call.\n", p->refer->attendedtransfer? "Attended" : "Blind");
		append_history(p, "Xfer", "Refer failed.");
		/* Failure of some kind */
		p->refer->status = REFER_FAILED;
		transmit_notify_with_sipfrag(p, seqno, "503 Service Unavailable", TRUE);
		ast_clear_flag(&p->flags[0], SIP_GOTREFER);	
		res = -1;
	}

handle_refer_cleanup:
	if (current.chan1) {
		ast_channel_unref(current.chan1);
	}
	if (current.chan2) {
		ast_channel_unref(current.chan2);
	}

	/* Make sure we exit with the pvt locked */
	return res;
}

/*! \brief Handle incoming CANCEL request */
static int handle_request_cancel(struct sip_pvt *p, struct sip_request *req)
{

	check_via(p, req);
	sip_alreadygone(p);

	if (p->owner && p->owner->_state == AST_STATE_UP) {
		/* This call is up, cancel is ignored, we need a bye */
		transmit_response(p, "200 OK", req);
		ast_debug(1, "Got CANCEL on an answered call. Ignoring... \n");
		return 0;
	}

	/* At this point, we could have cancelled the invite at the same time
	   as the other side sends a CANCEL. Our final reply with error code
	   might not have been received by the other side before the CANCEL
	   was sent, so let's just give up retransmissions and waiting for
	   ACK on our error code. The call is hanging up any way. */
	if (p->invitestate == INV_TERMINATED || p->invitestate == INV_COMPLETED) {
		__sip_pretend_ack(p);
	}
	if (p->invitestate != INV_TERMINATED)
		p->invitestate = INV_CANCELLED;

	if (ast_test_flag(&p->flags[0], SIP_INC_COUNT) || ast_test_flag(&p->flags[1], SIP_PAGE2_CALL_ONHOLD))
		update_call_counter(p, DEC_CALL_LIMIT);

	stop_media_flows(p); /* Immediately stop RTP, VRTP and UDPTL as applicable */
	if (p->owner) {
		ast_set_hangupsource(p->owner, p->owner->name, 0);
		ast_queue_hangup(p->owner);
	}
	else
		sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
	if (ast_str_strlen(p->initreq.data) > 0) {
		struct sip_pkt *pkt, *prev_pkt;
		/* If the CANCEL we are receiving is a retransmission, and we already have scheduled
		 * a reliable 487, then we don't want to schedule another one on top of the previous
		 * one.
		 *
		 * As odd as this may sound, we can't rely on the previously-transmitted "reliable"
		 * response in this situation. What if we've sent all of our reliable responses
		 * already and now all of a sudden, we get this second CANCEL?
		 *
		 * The only way to do this correctly is to cancel our previously-scheduled reliably-
		 * transmitted response and send a new one in its place.
		 */
		for (pkt = p->packets, prev_pkt = NULL; pkt; prev_pkt = pkt, pkt = pkt->next) {
			if (pkt->seqno == p->lastinvite && pkt->response_code == 487) {
				AST_SCHED_DEL(sched, pkt->retransid);
				UNLINK(pkt, p->packets, prev_pkt);
				dialog_unref(pkt->owner, "unref packet->owner from dialog");
				if (pkt->data) {
					ast_free(pkt->data);
				}
				ast_free(pkt);
				break;
			}
		}
		transmit_response_reliable(p, "487 Request Terminated", &p->initreq);
		transmit_response(p, "200 OK", req);
		return 1;
	} else {
		transmit_response(p, "481 Call Leg Does Not Exist", req);
		return 0;
	}
}

/*! \brief Handle incoming BYE request */
static int handle_request_bye(struct sip_pvt *p, struct sip_request *req)
{
	struct ast_channel *c=NULL;
	int res;
	struct ast_channel *bridged_to;
	const char *required;

	/* If we have an INCOMING invite that we haven't answered, terminate that transaction */
	if (p->pendinginvite && !ast_test_flag(&p->flags[0], SIP_OUTGOING) && !req->ignore) {
		transmit_response_reliable(p, "487 Request Terminated", &p->initreq);
	}

	__sip_pretend_ack(p);

	p->invitestate = INV_TERMINATED;

	copy_request(&p->initreq, req);
	if (sipdebug)
		ast_debug(1, "Initializing initreq for method %s - callid %s\n", sip_methods[req->method].text, p->callid);
	check_via(p, req);
	sip_alreadygone(p);

	/* Get RTCP quality before end of call */
	if (p->do_history || p->owner) {
		char quality_buf[AST_MAX_USER_FIELD], *quality;
		struct ast_channel *bridge = p->owner ? ast_bridged_channel(p->owner) : NULL;

		/* We need to get the lock on bridge because ast_rtp_instance_set_stats_vars will attempt
		 * to lock the bridge. This may get hairy...
		 */
		while (bridge && ast_channel_trylock(bridge)) {
			ast_channel_unlock(p->owner);
			do {
				/* Can't use DEADLOCK_AVOIDANCE since p is an ao2 object */
				sip_pvt_unlock(p);
				usleep(1);
				sip_pvt_lock(p);
			} while (p->owner && ast_channel_trylock(p->owner));
			bridge = p->owner ? ast_bridged_channel(p->owner) : NULL;
		}


		if (p->rtp && (quality = ast_rtp_instance_get_quality(p->rtp, AST_RTP_INSTANCE_STAT_FIELD_QUALITY, quality_buf, sizeof(quality_buf)))) {
			if (p->do_history) {
				append_history(p, "RTCPaudio", "Quality:%s", quality);

				if ((quality = ast_rtp_instance_get_quality(p->rtp, AST_RTP_INSTANCE_STAT_FIELD_QUALITY_JITTER, quality_buf, sizeof(quality_buf)))) {
					append_history(p, "RTCPaudioJitter", "Quality:%s", quality);
				}
				if ((quality = ast_rtp_instance_get_quality(p->rtp, AST_RTP_INSTANCE_STAT_FIELD_QUALITY_LOSS, quality_buf, sizeof(quality_buf)))) {
					append_history(p, "RTCPaudioLoss", "Quality:%s", quality);
				}
				if ((quality = ast_rtp_instance_get_quality(p->rtp, AST_RTP_INSTANCE_STAT_FIELD_QUALITY_RTT, quality_buf, sizeof(quality_buf)))) {
					append_history(p, "RTCPaudioRTT", "Quality:%s", quality);
				}
			}

			if (p->owner) {
				ast_rtp_instance_set_stats_vars(p->owner, p->rtp);
			}

		}

		if (bridge) {
			struct sip_pvt *q = bridge->tech_pvt;

			if (IS_SIP_TECH(bridge->tech) && q && q->rtp) {
				ast_rtp_instance_set_stats_vars(bridge, q->rtp);
			}
			ast_channel_unlock(bridge);
		}

		if (p->vrtp && (quality = ast_rtp_instance_get_quality(p->vrtp, AST_RTP_INSTANCE_STAT_FIELD_QUALITY, quality_buf, sizeof(quality_buf)))) {
			if (p->do_history) {
				append_history(p, "RTCPvideo", "Quality:%s", quality);
			}
			if (p->owner) {
				pbx_builtin_setvar_helper(p->owner, "RTPVIDEOQOS", quality);
			}
		}
		if (p->trtp && (quality = ast_rtp_instance_get_quality(p->trtp, AST_RTP_INSTANCE_STAT_FIELD_QUALITY, quality_buf, sizeof(quality_buf)))) {
			if (p->do_history) {
				append_history(p, "RTCPtext", "Quality:%s", quality);
			}
			if (p->owner) {
				pbx_builtin_setvar_helper(p->owner, "RTPTEXTQOS", quality);
			}
		}
	}

	stop_media_flows(p); /* Immediately stop RTP, VRTP and UDPTL as applicable */
	stop_session_timer(p); /* Stop Session-Timer */

	if (!ast_strlen_zero(get_header(req, "Also"))) {
		ast_log(LOG_NOTICE, "Client '%s' using deprecated BYE/Also transfer method.  Ask vendor to support REFER instead\n",
			ast_sockaddr_stringify(&p->recv));
		if (ast_strlen_zero(p->context))
			ast_string_field_set(p, context, sip_cfg.default_context);
		res = get_also_info(p, req);
		if (!res) {
			c = p->owner;
			if (c) {
				bridged_to = ast_bridged_channel(c);
				if (bridged_to) {
					/* Don't actually hangup here... */
					ast_queue_control(c, AST_CONTROL_UNHOLD);
					ast_channel_unlock(c);  /* async_goto can do a masquerade, no locks can be held during a masq */
					ast_async_goto(bridged_to, p->context, p->refer->refer_to, 1);
					ast_channel_lock(c);
				} else
					ast_queue_hangup(p->owner);
			}
		} else {
			ast_log(LOG_WARNING, "Invalid transfer information from '%s'\n", ast_sockaddr_stringify(&p->recv));
			if (p->owner)
				ast_queue_hangup_with_cause(p->owner, AST_CAUSE_PROTOCOL_ERROR);
		}
	} else if (p->owner) {
		ast_set_hangupsource(p->owner, p->owner->name, 0);
		ast_queue_hangup(p->owner);
		sip_scheddestroy_final(p, DEFAULT_TRANS_TIMEOUT);
		ast_debug(3, "Received bye, issuing owner hangup\n");
	} else {
		sip_scheddestroy_final(p, DEFAULT_TRANS_TIMEOUT);
		ast_debug(3, "Received bye, no owner, selfdestruct soon.\n");
	}
	ast_clear_flag(&p->flags[1], SIP_PAGE2_DIALOG_ESTABLISHED);

	/* Find out what they require */
	required = get_header(req, "Require");
	if (!ast_strlen_zero(required)) {
		char unsupported[256] = { 0, };
		parse_sip_options(required, unsupported, ARRAY_LEN(unsupported));
		/* If there are any options required that we do not support,
		 * then send a 420 with only those unsupported options listed */
		if (!ast_strlen_zero(unsupported)) {
			transmit_response_with_unsupported(p, "420 Bad extension (unsupported)", req, unsupported);
			ast_log(LOG_WARNING, "Received SIP BYE with unsupported required extension: required:%s unsupported:%s\n", required, unsupported);
		} else {
			transmit_response(p, "200 OK", req);
		}
	} else {
		transmit_response(p, "200 OK", req);
	}

	return 1;
}

/*! \brief Handle incoming MESSAGE request */
static int handle_request_message(struct sip_pvt *p, struct sip_request *req)
{
	if (!req->ignore) {
		if (req->debug)
			ast_verbose("Receiving message!\n");
		receive_message(p, req);
	} else
		transmit_response(p, "202 Accepted", req);
	return 1;
}

static enum sip_publish_type determine_sip_publish_type(struct sip_request *req, const char * const event, const char * const etag, const char * const expires, int *expires_int)
{
	int etag_present = !ast_strlen_zero(etag);
	int body_present = req->lines > 0;

	ast_assert(expires_int != NULL);

	if (ast_strlen_zero(expires)) {
		/* Section 6, item 4, second bullet point of RFC 3903 says to
		 * use a locally-configured default expiration if none is provided
		 * in the request
		 */
		*expires_int = DEFAULT_PUBLISH_EXPIRES;
	} else if (sscanf(expires, "%30d", expires_int) != 1) {
		return SIP_PUBLISH_UNKNOWN;
	}

	if (*expires_int == 0) {
		return SIP_PUBLISH_REMOVE;
	} else if (!etag_present && body_present) {
		return SIP_PUBLISH_INITIAL;
	} else if (etag_present && !body_present) {
		return SIP_PUBLISH_REFRESH;
	} else if (etag_present && body_present) {
		return SIP_PUBLISH_MODIFY;
	}

	return SIP_PUBLISH_UNKNOWN;
}

#ifdef HAVE_LIBXML2
static void get_pidf_body(struct sip_request *req, char *pidf_body, size_t size)
{
	int i;
	struct ast_str *str = ast_str_alloca(size);
	for (i = 0; i < req->lines; ++i) {
		ast_str_append(&str, 0, "%s", REQ_OFFSET_TO_STR(req, line[i]));
	}
	ast_copy_string(pidf_body, ast_str_buffer(str), size);
}

static int pidf_validate_tuple(struct ast_xml_node *tuple_node)
{
	const char *id;
	int status_found = FALSE;
	struct ast_xml_node *tuple_children;
	struct ast_xml_node *tuple_children_iterator;
	/* Tuples have to have an id attribute or they're invalid */
	if (!(id = ast_xml_get_attribute(tuple_node, "id"))) {
		ast_log(LOG_WARNING, "Tuple XML element has no attribute 'id'\n");
		return FALSE;
	}
	/* We don't care what it actually is, just that it's there */
	ast_xml_free_attr(id);
	/* This is a tuple. It must have a status element */
	if (!(tuple_children = ast_xml_node_get_children(tuple_node))) {
		/* The tuple has no children. It sucks */
		ast_log(LOG_WARNING, "Tuple XML element has no child elements\n");
		return FALSE;
	}
	for (tuple_children_iterator = tuple_children; tuple_children_iterator;
			tuple_children_iterator = ast_xml_node_get_next(tuple_children_iterator)) {
		/* Similar to the wording used regarding tuples, the status element should appear
		 * first. However, we will once again relax things and accept the status at any
		 * position. We will enforce that only a single status element can be present.
		 */
		if (strcmp(ast_xml_node_get_name(tuple_children_iterator), "status")) {
			/* Not the status, we don't care */
			continue;
		}
		if (status_found == TRUE) {
			/* THERE CAN BE ONLY ONE!!! */
			ast_log(LOG_WARNING, "Multiple status elements found in tuple. Only one allowed\n");
			return FALSE;
		}
		status_found = TRUE;
	}
	return status_found;
}


static int pidf_validate_presence(struct ast_xml_doc *doc)
{
	struct ast_xml_node *presence_node = ast_xml_get_root(doc);
	struct ast_xml_node *child_nodes;
	struct ast_xml_node *node_iterator;
	struct ast_xml_ns *ns;
	const char *entity;
	const char *namespace;
	const char presence_namespace[] = "urn:ietf:params:xml:ns:pidf";

	if (!presence_node) {
		ast_log(LOG_WARNING, "Unable to retrieve root node of the XML document\n");
		return FALSE;
	}
	/* Okay, we managed to open the document! YAY! Now, let's start making sure it's all PIDF-ified
	 * correctly.
	 */
	if (strcmp(ast_xml_node_get_name(presence_node), "presence")) {
		ast_log(LOG_WARNING, "Root node of PIDF document is not 'presence'. Invalid\n");
		return FALSE;
	}

	/* The presence element must have an entity attribute and an xmlns attribute. Furthermore
	 * the xmlns attribute must be "urn:ietf:params:xml:ns:pidf"
	 */
	if (!(entity = ast_xml_get_attribute(presence_node, "entity"))) {
		ast_log(LOG_WARNING, "Presence element of PIDF document has no 'entity' attribute\n");
		return FALSE;
	}
	/* We're not interested in what the entity is, just that it exists */
	ast_xml_free_attr(entity);

	if (!(ns = ast_xml_find_namespace(doc, presence_node, NULL))) {
		ast_log(LOG_WARNING, "Couldn't find default namespace...\n");
		return FALSE;
	}

	namespace = ast_xml_get_ns_href(ns);
	if (ast_strlen_zero(namespace) || strcmp(namespace, presence_namespace)) {
		ast_log(LOG_WARNING, "PIDF document has invalid namespace value %s\n", namespace);
		return FALSE;
	}

	if (!(child_nodes = ast_xml_node_get_children(presence_node))) {
		ast_log(LOG_WARNING, "PIDF document has no elements as children of 'presence'. Invalid\n");
		return FALSE;
	}

	/* Check for tuple elements. RFC 3863 says that PIDF documents can have any number of
	 * tuples, including 0. The big thing here is that if there are tuple elements present,
	 * they have to have a single status element within.
	 *
	 * The RFC is worded such that tuples should appear as the first elements as children of
	 * the presence element. However, we'll be accepting of documents which may place other elements
	 * before the tuple(s).
	 */
	for (node_iterator = child_nodes; node_iterator;
			node_iterator = ast_xml_node_get_next(node_iterator)) {
		if (strcmp(ast_xml_node_get_name(node_iterator), "tuple")) {
			/* Not a tuple. We don't give a rat's hind quarters */
			continue;
		}
		if (pidf_validate_tuple(node_iterator) == FALSE) {
			ast_log(LOG_WARNING, "Unable to validate tuple\n");
			return FALSE;
		}
	}

	return TRUE;
}

/*!
 * \brief Makes sure that body is properly formatted PIDF
 *
 * Specifically, we check that the document has a "presence" element
 * at the root and that within that, there is at least one "tuple" element
 * that contains a "status" element.
 *
 * XXX This function currently assumes a default namespace is used. Of course
 * if you're not using a default namespace, you're probably a stupid jerk anyway.
 *
 * \param req The SIP request to check
 * \param[out] pidf_doc The validated PIDF doc.
 * \retval FALSE The XML was malformed or the basic PIDF structure was marred
 * \retval TRUE The PIDF document is of a valid format
 */
static int sip_pidf_validate(struct sip_request *req, struct ast_xml_doc **pidf_doc)
{
	struct ast_xml_doc *doc;
	int content_length;
	const char *content_length_str = get_header(req, "Content-Length");
	const char *content_type = get_header(req, "Content-Type");
	char pidf_body[SIPBUFSIZE];
	int res;

	if (ast_strlen_zero(content_type) || strcmp(content_type, "application/pidf+xml")) {
		ast_log(LOG_WARNING, "Content type is not PIDF\n");
		return FALSE;
	}

	if (ast_strlen_zero(content_length_str)) {
		ast_log(LOG_WARNING, "No content length. Can't determine bounds of PIDF document\n");
		return FALSE;
	}

	if (sscanf(content_length_str, "%30d", &content_length) != 1) {
		ast_log(LOG_WARNING, "Invalid content length provided\n");
		return FALSE;
	}

	if (content_length > sizeof(pidf_body)) {
		ast_log(LOG_WARNING, "Content length of PIDF document truncated to %d bytes\n", (int) sizeof(pidf_body));
		content_length = sizeof(pidf_body);
	}

	get_pidf_body(req, pidf_body, content_length);

	if (!(doc = ast_xml_read_memory(pidf_body, content_length))) {
		ast_log(LOG_WARNING, "Unable to open XML PIDF document. Is it malformed?\n");
		return FALSE;
	}

	res = pidf_validate_presence(doc);
	if (res == TRUE) {
		*pidf_doc = doc;
	} else {
		ast_xml_close(doc);
	}
	return res;
}

static int cc_esc_publish_handler(struct sip_pvt *pvt, struct sip_request *req, struct event_state_compositor *esc, struct sip_esc_entry *esc_entry)
{
	const char *uri = REQ_OFFSET_TO_STR(req, rlPart2);
	struct ast_cc_agent *agent;
	struct sip_cc_agent_pvt *agent_pvt;
	struct ast_xml_doc *pidf_doc = NULL;
	const char *basic_status = NULL;
	struct ast_xml_node *presence_node;
	struct ast_xml_node *presence_children;
	struct ast_xml_node *tuple_node;
	struct ast_xml_node *tuple_children;
	struct ast_xml_node *status_node;
	struct ast_xml_node *status_children;
	struct ast_xml_node *basic_node;
	int res = 0;

	if (!((agent = find_sip_cc_agent_by_notify_uri(uri)) || (agent = find_sip_cc_agent_by_subscribe_uri(uri)))) {
		ast_log(LOG_WARNING, "Could not find agent using uri '%s'\n", uri);
		transmit_response(pvt, "412 Conditional Request Failed", req);
		return -1;
	}

	agent_pvt = agent->private_data;

	if (sip_pidf_validate(req, &pidf_doc) == FALSE) {
		res = -1;
		goto cc_publish_cleanup;
	}

	/* It's important to note that the PIDF validation routine has no knowledge
	 * of what we specifically want in this instance. A valid PIDF document could
	 * have no tuples, or it could have tuples whose status element has no basic
	 * element contained within. While not violating the PIDF spec, these are
	 * insufficient for our needs in this situation
	 */
	presence_node = ast_xml_get_root(pidf_doc);
	if (!(presence_children = ast_xml_node_get_children(presence_node))) {
		ast_log(LOG_WARNING, "No tuples within presence element.\n");
		res = -1;
		goto cc_publish_cleanup;
	}

	if (!(tuple_node = ast_xml_find_element(presence_children, "tuple", NULL, NULL))) {
		ast_log(LOG_NOTICE, "Couldn't find tuple node?\n");
		res = -1;
		goto cc_publish_cleanup;
	}

	/* We already made sure that the tuple has a status node when we validated the PIDF
	 * document earlier. So there's no need to enclose this operation in an if statement.
	 */
	tuple_children = ast_xml_node_get_children(tuple_node);
	status_node = ast_xml_find_element(tuple_children, "status", NULL, NULL);

	if (!(status_children = ast_xml_node_get_children(status_node))) {
		ast_log(LOG_WARNING, "No basic elements within status element.\n");
		res = -1;
		goto cc_publish_cleanup;
	}

	if (!(basic_node = ast_xml_find_element(status_children, "basic", NULL, NULL))) {
		ast_log(LOG_WARNING, "Couldn't find basic node?\n");
		res = -1;
		goto cc_publish_cleanup;
	}

	basic_status = ast_xml_get_text(basic_node);

	if (ast_strlen_zero(basic_status)) {
		ast_log(LOG_NOTICE, "NOthing in basic node?\n");
		res = -1;
		goto cc_publish_cleanup;
	}

	if (!strcmp(basic_status, "open")) {
		agent_pvt->is_available = TRUE;
		ast_cc_agent_caller_available(agent->core_id, "Received PUBLISH stating SIP caller %s is available",
				agent->device_name);
	} else if (!strcmp(basic_status, "closed")) {
		agent_pvt->is_available = FALSE;
		ast_cc_agent_caller_busy(agent->core_id, "Received PUBLISH stating SIP caller %s is busy",
				agent->device_name);
	} else {
		ast_log(LOG_NOTICE, "Invalid content in basic element: %s\n", basic_status);
	}

cc_publish_cleanup:
	if (basic_status) {
		ast_xml_free_text(basic_status);
	}
	if (pidf_doc) {
		ast_xml_close(pidf_doc);
	}
	ao2_ref(agent, -1);
	if (res) {
		transmit_response(pvt, "400 Bad Request", req);
	}
	return res;
}

#endif /* HAVE_LIBXML2 */

static int handle_sip_publish_initial(struct sip_pvt *p, struct sip_request *req, struct event_state_compositor *esc, const int expires)
{
	struct sip_esc_entry *esc_entry = create_esc_entry(esc, req, expires);
	int res = 0;

	if (!esc_entry) {
		transmit_response(p, "503 Internal Server Failure", req);
		return -1;
	}

	if (esc->callbacks->initial_handler) {
		res = esc->callbacks->initial_handler(p, req, esc, esc_entry);
	}

	if (!res) {
		transmit_response_with_sip_etag(p, "200 OK", req, esc_entry, 0);
	}

	ao2_ref(esc_entry, -1);
	return res;
}

static int handle_sip_publish_refresh(struct sip_pvt *p, struct sip_request *req, struct event_state_compositor *esc, const char * const etag, const int expires)
{
	struct sip_esc_entry *esc_entry = get_esc_entry(etag, esc);
	int expires_ms = expires * 1000;
	int res = 0;

	if (!esc_entry) {
		transmit_response(p, "412 Conditional Request Failed", req);
		return -1;
	}

	AST_SCHED_REPLACE_UNREF(esc_entry->sched_id, sched, expires_ms, publish_expire, esc_entry,
			ao2_ref(_data, -1),
			ao2_ref(esc_entry, -1),
			ao2_ref(esc_entry, +1));

	if (esc->callbacks->refresh_handler) {
		res = esc->callbacks->refresh_handler(p, req, esc, esc_entry);
	}

	if (!res) {
		transmit_response_with_sip_etag(p, "200 OK", req, esc_entry, 1);
	}

	ao2_ref(esc_entry, -1);
	return res;
}

static int handle_sip_publish_modify(struct sip_pvt *p, struct sip_request *req, struct event_state_compositor *esc, const char * const etag, const int expires)
{
	struct sip_esc_entry *esc_entry = get_esc_entry(etag, esc);
	int expires_ms = expires * 1000;
	int res = 0;

	if (!esc_entry) {
		transmit_response(p, "412 Conditional Request Failed", req);
		return -1;
	}

	AST_SCHED_REPLACE_UNREF(esc_entry->sched_id, sched, expires_ms, publish_expire, esc_entry,
			ao2_ref(_data, -1),
			ao2_ref(esc_entry, -1),
			ao2_ref(esc_entry, +1));

	if (esc->callbacks->modify_handler) {
		res = esc->callbacks->modify_handler(p, req, esc, esc_entry);
	}

	if (!res) {
		transmit_response_with_sip_etag(p, "200 OK", req, esc_entry, 1);
	}

	ao2_ref(esc_entry, -1);
	return res;
}

static int handle_sip_publish_remove(struct sip_pvt *p, struct sip_request *req, struct event_state_compositor *esc, const char * const etag)
{
	struct sip_esc_entry *esc_entry = get_esc_entry(etag, esc);
	int res = 0;

	if (!esc_entry) {
		transmit_response(p, "412 Conditional Request Failed", req);
		return -1;
	}

	AST_SCHED_DEL(sched, esc_entry->sched_id);
	/* Scheduler's ref of the esc_entry */
	ao2_ref(esc_entry, -1);

	if (esc->callbacks->remove_handler) {
		res = esc->callbacks->remove_handler(p, req, esc, esc_entry);
	}

	if (!res) {
		transmit_response_with_sip_etag(p, "200 OK", req, esc_entry, 1);
	}

	/* Ref from finding the esc_entry earlier in function */
	ao2_unlink(esc->compositor, esc_entry);
	ao2_ref(esc_entry, -1);
	return res;
}

static int handle_request_publish(struct sip_pvt *p, struct sip_request *req, struct ast_sockaddr *addr, const uint32_t seqno, const char *uri)
{
	const char *etag = get_header(req, "SIP-If-Match");
	const char *event = get_header(req, "Event");
	struct event_state_compositor *esc;
	enum sip_publish_type publish_type;
	const char *expires_str = get_header(req, "Expires");
	int expires_int;
	int auth_result;
	int handler_result = -1;

	if (ast_strlen_zero(event)) {
		transmit_response(p, "489 Bad Event", req);
		pvt_set_needdestroy(p, "missing Event: header");
		return -1;
	}

	if (!(esc = get_esc(event))) {
		transmit_response(p, "489 Bad Event", req);
		pvt_set_needdestroy(p, "unknown event package in publish");
		return -1;
	}

	auth_result = check_user(p, req, SIP_PUBLISH, uri, XMIT_RELIABLE, addr);
	if (auth_result == AUTH_CHALLENGE_SENT) {
		p->lastinvite = seqno;
		return 0;
	} else if (auth_result < 0) {
		if (auth_result == AUTH_FAKE_AUTH) {
			ast_log(LOG_NOTICE, "Sending fake auth rejection for device %s\n", get_header(req, "From"));
			transmit_fake_auth_response(p, SIP_INVITE, req, XMIT_RELIABLE);
		} else {
			ast_log(LOG_NOTICE, "Failed to authenticate device %s\n", get_header(req, "From"));
			transmit_response_reliable(p, "403 Forbidden", req);
		}
		sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
		ast_string_field_set(p, theirtag, NULL);
		return 0;
	} else if (auth_result == AUTH_SUCCESSFUL && p->lastinvite) {
		/* We need to stop retransmitting the 401 */
		__sip_ack(p, p->lastinvite, 1, 0);
	}

	publish_type = determine_sip_publish_type(req, event, etag, expires_str, &expires_int);

	if (expires_int > max_expiry) {
		expires_int = max_expiry;
	} else if (expires_int < min_expiry && expires_int > 0) {
		transmit_response_with_minexpires(p, "423 Interval too small", req);
		pvt_set_needdestroy(p, "Expires is less that the min expires allowed.");
		return 0;
	}
	p->expiry = expires_int;

	/* It is the responsibility of these handlers to formulate any response
	 * sent for a PUBLISH
	 */
	switch (publish_type) {
	case SIP_PUBLISH_UNKNOWN:
		transmit_response(p, "400 Bad Request", req);
		break;
	case SIP_PUBLISH_INITIAL:
		handler_result = handle_sip_publish_initial(p, req, esc, expires_int);
		break;
	case SIP_PUBLISH_REFRESH:
		handler_result = handle_sip_publish_refresh(p, req, esc, etag, expires_int);
		break;
	case SIP_PUBLISH_MODIFY:
		handler_result = handle_sip_publish_modify(p, req, esc, etag, expires_int);
		break;
	case SIP_PUBLISH_REMOVE:
		handler_result = handle_sip_publish_remove(p, req, esc, etag);
		break;
	default:
		transmit_response(p, "400 Impossible Condition", req);
		break;
	}
	if (!handler_result && p->expiry > 0) {
		sip_scheddestroy(p, (p->expiry + 10) * 1000);
	} else {
		pvt_set_needdestroy(p, "forcing expiration");
	}

	return handler_result;
}

/*! \internal \brief Subscribe to MWI events for the specified peer
 * \note The peer cannot be locked during this method.  sip_send_mwi_peer will
 * attempt to lock the peer after the event subscription lock is held; if the peer is locked during
 * this method then we will attempt to lock the event subscription lock but after the peer, creating
 * a locking inversion.
 */
static void add_peer_mwi_subs(struct sip_peer *peer)
{
	struct sip_mailbox *mailbox;

	AST_LIST_TRAVERSE(&peer->mailboxes, mailbox, entry) {
		if (mailbox->event_sub) {
			ast_event_unsubscribe(mailbox->event_sub);
		}

		mailbox->event_sub = ast_event_subscribe(AST_EVENT_MWI, mwi_event_cb, "SIP mbox event", peer,
			AST_EVENT_IE_MAILBOX, AST_EVENT_IE_PLTYPE_STR, mailbox->mailbox,
			AST_EVENT_IE_CONTEXT, AST_EVENT_IE_PLTYPE_STR, S_OR(mailbox->context, "default"),
			AST_EVENT_IE_END);
	}
}

static int handle_cc_subscribe(struct sip_pvt *p, struct sip_request *req)
{
	const char *uri = REQ_OFFSET_TO_STR(req, rlPart2);
	char *param_separator;
	struct ast_cc_agent *agent;
	struct sip_cc_agent_pvt *agent_pvt;
	const char *expires_str = get_header(req, "Expires");
	int expires = -1; /* Just need it to be non-zero */

	if (!ast_strlen_zero(expires_str)) {
		sscanf(expires_str, "%d", &expires);
	}

	if ((param_separator = strchr(uri, ';'))) {
		*param_separator = '\0';
	}

	p->subscribed = CALL_COMPLETION;

	if (!(agent = find_sip_cc_agent_by_subscribe_uri(uri))) {
		if (!expires) {
			/* Typically, if a 0 Expires reaches us and we can't find
			 * the corresponding agent, it means that the CC transaction
			 * has completed and so the calling side is just trying to
			 * clean up its subscription. We'll just respond with a
			 * 200 OK and be done with it
			 */
			transmit_response(p, "200 OK", req);
			return 0;
		}
		ast_log(LOG_WARNING, "Invalid URI '%s' in CC subscribe\n", uri);
		transmit_response(p, "404 Not Found", req);
		return -1;
	}

	agent_pvt = agent->private_data;

	if (!expires) {
		/* We got sent a SUBSCRIBE and found an agent. This means that CC
		 * is being canceled.
		 */
		ast_cc_failed(agent->core_id, "CC is being canceled by %s", agent->device_name);
		transmit_response(p, "200 OK", req);
		ao2_ref(agent, -1);
		return 0;
	}

	agent_pvt->subscribe_pvt = dialog_ref(p, "SIP CC agent gains reference to subscription dialog");
	ast_cc_agent_accept_request(agent->core_id, "SIP caller %s has requested CC via SUBSCRIBE",
			agent->device_name);

	/* We don't send a response here. That is done in the agent's ack callback or in the
	 * agent destructor, should a failure occur before we have responded
	 */
	ao2_ref(agent, -1);
	return 0;
}

/*! \brief  Handle incoming SUBSCRIBE request */
static int handle_request_subscribe(struct sip_pvt *p, struct sip_request *req, struct ast_sockaddr *addr, uint32_t seqno, const char *e)
{
	int gotdest = 0;
	int res = 0;
	int firststate;
	struct sip_peer *authpeer = NULL;
	const char *eventheader = get_header(req, "Event");	/* Get Event package name */
	int resubscribe = (p->subscribed != NONE) && !req->ignore;
	char *temp, *event;

	if (p->initreq.headers) {	
		/* We already have a dialog */
		if (p->initreq.method != SIP_SUBSCRIBE) {
			/* This is a SUBSCRIBE within another SIP dialog, which we do not support */
			/* For transfers, this could happen, but since we haven't seen it happening, let us just refuse this */
 			transmit_response(p, "403 Forbidden (within dialog)", req);
			/* Do not destroy session, since we will break the call if we do */
			ast_debug(1, "Got a subscription within the context of another call, can't handle that - %s (Method %s)\n", p->callid, sip_methods[p->initreq.method].text);
			return 0;
		} else if (req->debug) {
			if (resubscribe)
				ast_debug(1, "Got a re-subscribe on existing subscription %s\n", p->callid);
			else
				ast_debug(1, "Got a new subscription %s (possibly with auth) or retransmission\n", p->callid);
		}
	}

	/* Check if we have a global disallow setting on subscriptions.
		if so, we don't have to check peer settings after auth, which saves a lot of processing
	*/
	if (!sip_cfg.allowsubscribe) {
 		transmit_response(p, "403 Forbidden (policy)", req);
		pvt_set_needdestroy(p, "forbidden");
		return 0;
	}

	if (!req->ignore && !resubscribe) {	/* Set up dialog, new subscription */
		const char *to = get_header(req, "To");
		char totag[128];
		set_pvt_allowed_methods(p, req);

		/* Check to see if a tag was provided, if so this is actually a resubscription of a dialog we no longer know about */
		if (!ast_strlen_zero(to) && gettag(req, "To", totag, sizeof(totag))) {
			if (req->debug)
				ast_verbose("Received resubscription for a dialog we no longer know about. Telling remote side to subscribe again.\n");
			transmit_response(p, "481 Subscription does not exist", req);
			pvt_set_needdestroy(p, "subscription does not exist");
			return 0;
		}

		/* Use this as the basis */
		if (req->debug)
			ast_verbose("Creating new subscription\n");

		copy_request(&p->initreq, req);
		if (sipdebug)
			ast_debug(4, "Initializing initreq for method %s - callid %s\n", sip_methods[req->method].text, p->callid);
		check_via(p, req);
		build_route(p, req, 0, 0);
	} else if (req->debug && req->ignore)
		ast_verbose("Ignoring this SUBSCRIBE request\n");

	/* Find parameters to Event: header value and remove them for now */
	if (ast_strlen_zero(eventheader)) {
		transmit_response(p, "489 Bad Event", req);
		ast_debug(2, "Received SIP subscribe for unknown event package: <none>\n");
		pvt_set_needdestroy(p, "unknown event package in subscribe");
		return 0;
	}

	if ( (strchr(eventheader, ';'))) {
		event = ast_strdupa(eventheader);	/* Since eventheader is a const, we can't change it */
		temp = strchr(event, ';'); 		
		*temp = '\0';				/* Remove any options for now */
							/* We might need to use them later :-) */
	} else
		event = (char *) eventheader;		/* XXX is this legal ? */

	/* Handle authentication if we're new and not a retransmission. We can't just
	 * use if !req->ignore, because then we'll end up sending
	 * a 200 OK if someone retransmits without sending auth */
	if (p->subscribed == NONE || resubscribe) {
		res = check_user_full(p, req, SIP_SUBSCRIBE, e, 0, addr, &authpeer);

		/* if an authentication response was sent, we are done here */
		if (res == AUTH_CHALLENGE_SENT)	/* authpeer = NULL here */
			return 0;
		if (res != AUTH_SUCCESSFUL) {
			if (res == AUTH_FAKE_AUTH) {
				ast_log(LOG_NOTICE, "Sending fake auth rejection for device %s\n", get_header(req, "From"));
				transmit_fake_auth_response(p, SIP_SUBSCRIBE, req, XMIT_UNRELIABLE);
			} else {
				ast_log(LOG_NOTICE, "Failed to authenticate device %s for SUBSCRIBE\n", get_header(req, "From"));
				transmit_response_reliable(p, "403 Forbidden", req);
			}

			pvt_set_needdestroy(p, "authentication failed");
			return 0;
		}
	}

	/* At this point, we hold a reference to authpeer (if not NULL).  It
	 * must be released when done.
	 */

	/* Check if this device  is allowed to subscribe at all */
	if (!ast_test_flag(&p->flags[1], SIP_PAGE2_ALLOWSUBSCRIBE)) {
		transmit_response(p, "403 Forbidden (policy)", req);
		pvt_set_needdestroy(p, "subscription not allowed");
		if (authpeer) {
			unref_peer(authpeer, "unref_peer, from handle_request_subscribe (authpeer 1)");
		}
		return 0;
	}

	if (strcmp(event, "message-summary") && strcmp(event, "call-completion")) {
		/* Get destination right away */
		gotdest = get_destination(p, NULL, NULL);
	}

	/* Get full contact header - this needs to be used as a request URI in NOTIFY's */
	parse_ok_contact(p, req);

	build_contact(p);
	if (gotdest != SIP_GET_DEST_EXTEN_FOUND) {
		if (gotdest == SIP_GET_DEST_INVALID_URI) {
			transmit_response(p, "416 Unsupported URI scheme", req);
		} else {
			transmit_response(p, "404 Not Found", req);
		}
		pvt_set_needdestroy(p, "subscription target not found");
		if (authpeer) {
			unref_peer(authpeer, "unref_peer, from handle_request_subscribe (authpeer 2)");
		}
		return 0;
	}

	/* Initialize tag for new subscriptions */	
	if (ast_strlen_zero(p->tag))
		make_our_tag(p->tag, sizeof(p->tag));

	if (!strcmp(event, "presence") || !strcmp(event, "dialog")) { /* Presence, RFC 3842 */
		unsigned int pidf_xml;
		const char *accept;
		int start = 0;
		enum subscriptiontype subscribed = NONE;
		const char *unknown_acceptheader = NULL;

		/* Header from Xten Eye-beam Accept: multipart/related, application/rlmi+xml, application/pidf+xml, application/xpidf+xml */
		accept = __get_header(req, "Accept", &start);
		while ((subscribed == NONE) && !ast_strlen_zero(accept)) {
			pidf_xml = strstr(accept, "application/pidf+xml") ? 1 : 0;

			/* Older versions of Polycom firmware will claim pidf+xml, but really
			 * they only support xpidf+xml. */
			if (pidf_xml && strstr(p->useragent, "Polycom")) {
				subscribed = XPIDF_XML;
			} else if (pidf_xml) {
				subscribed = PIDF_XML;         /* RFC 3863 format */
			} else if (strstr(accept, "application/dialog-info+xml")) {
				subscribed = DIALOG_INFO_XML;
				/* IETF draft: draft-ietf-sipping-dialog-package-05.txt */
			} else if (strstr(accept, "application/cpim-pidf+xml")) {
				subscribed = CPIM_PIDF_XML;    /* RFC 3863 format */
			} else if (strstr(accept, "application/xpidf+xml")) {
				subscribed = XPIDF_XML;        /* Early pre-RFC 3863 format with MSN additions (Microsoft Messenger) */
			} else {
				unknown_acceptheader = accept;
			}
			/* check to see if there is another Accept header present */
			accept = __get_header(req, "Accept", &start);
		}

		if (!start) {
			if (p->subscribed == NONE) { /* if the subscribed field is not already set, and there is no accept header... */
				transmit_response(p, "489 Bad Event", req);
				ast_log(LOG_WARNING,"SUBSCRIBE failure: no Accept header: pvt: "
					"stateid: %d, laststate: %d, dialogver: %u, subscribecont: "
					"'%s', subscribeuri: '%s'\n",
					p->stateid,
					p->laststate,
					p->dialogver,
					p->subscribecontext,
					p->subscribeuri);
				pvt_set_needdestroy(p, "no Accept header");
				if (authpeer) {
					unref_peer(authpeer, "unref_peer, from handle_request_subscribe (authpeer 2)");
				}
				return 0;
			}
			/* if p->subscribed is non-zero, then accept is not obligatory; according to rfc 3265 section 3.1.3, at least.
			   so, we'll just let it ride, keeping the value from a previous subscription, and not abort the subscription */
		} else if (subscribed == NONE) {
			/* Can't find a format for events that we know about */
			char mybuf[200];
			if (!ast_strlen_zero(unknown_acceptheader)) {
				snprintf(mybuf, sizeof(mybuf), "489 Bad Event (format %s)", unknown_acceptheader);
			} else {
				snprintf(mybuf, sizeof(mybuf), "489 Bad Event");
			}
			transmit_response(p, mybuf, req);
			ast_log(LOG_WARNING,"SUBSCRIBE failure: unrecognized format:"
				"'%s' pvt: subscribed: %d, stateid: %d, laststate: %d,"
				"dialogver: %u, subscribecont: '%s', subscribeuri: '%s'\n",
				unknown_acceptheader,
				(int)p->subscribed,
				p->stateid,
				p->laststate,
				p->dialogver,
				p->subscribecontext,
				p->subscribeuri);
			pvt_set_needdestroy(p, "unrecognized format");
			if (authpeer) {
				unref_peer(authpeer, "unref_peer, from handle_request_subscribe (authpeer 2)");
			}
			return 0;
		} else {
			p->subscribed = subscribed;
		}
	} else if (!strcmp(event, "message-summary")) {
		int start = 0;
		int found_supported = 0;
		const char *acceptheader;

		acceptheader = __get_header(req, "Accept", &start);
		while (!found_supported && !ast_strlen_zero(acceptheader)) {
			found_supported = strcmp(acceptheader, "application/simple-message-summary") ? 0 : 1;
			if (!found_supported && (option_debug > 2)) {
				ast_log(LOG_DEBUG, "Received SIP mailbox subscription for unknown format: %s\n", acceptheader);
			}
			acceptheader = __get_header(req, "Accept", &start);
		}
		if (start && !found_supported) {
			/* Format requested that we do not support */
			transmit_response(p, "406 Not Acceptable", req);
			ast_debug(2, "Received SIP mailbox subscription for unknown format: %s\n", acceptheader);
			pvt_set_needdestroy(p, "unknown format");
			if (authpeer) {
				unref_peer(authpeer, "unref_peer, from handle_request_subscribe (authpeer 3)");
			}
			return 0;
		}
		/* Looks like they actually want a mailbox status
		  This version of Asterisk supports mailbox subscriptions
		  The subscribed URI needs to exist in the dial plan
		  In most devices, this is configurable to the voicemailmain extension you use
		*/
		if (!authpeer || AST_LIST_EMPTY(&authpeer->mailboxes)) {
			if (!authpeer) {
				transmit_response(p, "404 Not found", req);
			} else {
				transmit_response(p, "404 Not found (no mailbox)", req);
				ast_log(LOG_NOTICE, "Received SIP subscribe for peer without mailbox: %s\n", S_OR(authpeer->name, ""));
			}
			pvt_set_needdestroy(p, "received 404 response");
			if (authpeer) {
				unref_peer(authpeer, "unref_peer, from handle_request_subscribe (authpeer 3)");
			}
			return 0;
		}

		p->subscribed = MWI_NOTIFICATION;
		if (ast_test_flag(&authpeer->flags[1], SIP_PAGE2_SUBSCRIBEMWIONLY)) {
			ao2_unlock(p);
			add_peer_mwi_subs(authpeer);
			ao2_lock(p);
		}
		if (authpeer->mwipvt != p) {	/* Destroy old PVT if this is a new one */
			/* We only allow one subscription per peer */
			if (authpeer->mwipvt) {
				dialog_unlink_all(authpeer->mwipvt);
				authpeer->mwipvt = dialog_unref(authpeer->mwipvt, "unref dialog authpeer->mwipvt");
			}
			authpeer->mwipvt = dialog_ref(p, "setting peers' mwipvt to p");
		}
		if (p->relatedpeer != authpeer) {
			if (p->relatedpeer) {
				unref_peer(p->relatedpeer, "Unref previously stored relatedpeer ptr");
			}
			p->relatedpeer = ref_peer(authpeer, "setting dialog's relatedpeer pointer");
		}
		/* Do not release authpeer here */
	} else if (!strcmp(event, "call-completion")) {
		handle_cc_subscribe(p, req);
	} else { /* At this point, Asterisk does not understand the specified event */
		transmit_response(p, "489 Bad Event", req);
		ast_debug(2, "Received SIP subscribe for unknown event package: %s\n", event);
		pvt_set_needdestroy(p, "unknown event package");
		if (authpeer) {
			unref_peer(authpeer, "unref_peer, from handle_request_subscribe (authpeer 5)");
		}
		return 0;
	}

	/* Add subscription for extension state from the PBX core */
	if (p->subscribed != MWI_NOTIFICATION  && p->subscribed != CALL_COMPLETION && !resubscribe) {
		if (p->stateid != -1) {
			ast_extension_state_del(p->stateid, cb_extensionstate);
		}
		dialog_ref(p, "copying dialog ptr into extension state struct");
		p->stateid = ast_extension_state_add_destroy(p->context, p->exten,
			cb_extensionstate, cb_extensionstate_destroy, p);
		if (p->stateid == -1) {
			dialog_unref(p, "copying dialog ptr into extension state struct failed");
		}
	}

	if (!req->ignore && p)
		p->lastinvite = seqno;
	if (p && !p->needdestroy) {
		p->expiry = atoi(get_header(req, "Expires"));

		/* check if the requested expiry-time is within the approved limits from sip.conf */
		if (p->expiry > max_expiry) {
			p->expiry = max_expiry;
		} else if (p->expiry < min_expiry && p->expiry > 0) {
			transmit_response_with_minexpires(p, "423 Interval too small", req);
			ast_log(LOG_WARNING, "Received subscription for extension \"%s\" context \"%s\" "
				"with Expire header less that 'minexpire' limit. Received \"Expire: %d\" min is %d\n",
				p->exten, p->context, p->expiry, min_expiry);
			pvt_set_needdestroy(p, "Expires is less that the min expires allowed.");
			if (authpeer) {
				unref_peer(authpeer, "unref_peer, from handle_request_subscribe (authpeer 6)");
			}
			return 0;
		}

		if (sipdebug) {
			const char *action = p->expiry > 0 ? "Adding" : "Removing";
			if (p->subscribed == MWI_NOTIFICATION && p->relatedpeer) {
				ast_debug(2, "%s subscription for mailbox notification - peer %s\n",
						action, p->relatedpeer->name);
			} else if (p->subscribed == CALL_COMPLETION) {
				ast_debug(2, "%s CC subscription for peer %s\n", action, p->username);
			} else {
				ast_debug(2, "%s subscription for extension %s context %s for peer %s\n",
						action, p->exten, p->context, p->username);
			}
		}
		if (p->autokillid > -1 && sip_cancel_destroy(p))	/* Remove subscription expiry for renewals */
			ast_log(LOG_WARNING, "Unable to cancel SIP destruction.  Expect bad things.\n");
		if (p->expiry > 0)
			sip_scheddestroy(p, (p->expiry + 10) * 1000);	/* Set timer for destruction of call at expiration */

		if (p->subscribed == MWI_NOTIFICATION) {
			ast_set_flag(&p->flags[1], SIP_PAGE2_DIALOG_ESTABLISHED);
			transmit_response(p, "200 OK", req);
			if (p->relatedpeer) {	/* Send first notification */
				struct sip_peer *peer = p->relatedpeer;
				ref_peer(peer, "ensure a peer ref is held during MWI sending");
				ao2_unlock(p);
				sip_send_mwi_to_peer(peer, 0);
				ao2_lock(p);
				unref_peer(peer, "release a peer ref now that MWI is sent");
			}
		} else if (p->subscribed != CALL_COMPLETION) {
			if ((firststate = ast_extension_state(NULL, p->context, p->exten)) < 0) {

				ast_log(LOG_NOTICE, "Got SUBSCRIBE for extension %s@%s from %s, but there is no hint for that extension.\n", p->exten, p->context, ast_sockaddr_stringify(&p->sa));
				transmit_response(p, "404 Not found", req);
				pvt_set_needdestroy(p, "no extension for SUBSCRIBE");
				if (authpeer) {
					unref_peer(authpeer, "unref_peer, from handle_request_subscribe (authpeer 6)");
				}
				return 0;
			}
			ast_set_flag(&p->flags[1], SIP_PAGE2_DIALOG_ESTABLISHED);
			transmit_response(p, "200 OK", req);
			transmit_state_notify(p, firststate, 1, FALSE);	/* Send first notification */
			append_history(p, "Subscribestatus", "%s", ast_extension_state2str(firststate));
			/* hide the 'complete' exten/context in the refer_to field for later display */
			ast_string_field_build(p, subscribeuri, "%s@%s", p->exten, p->context);
			/* Deleted the slow iteration of all sip dialogs to find old subscribes from this peer for exten@context */

		}
		if (!p->expiry) {
			pvt_set_needdestroy(p, "forcing expiration");
		}
	}

	if (authpeer) {
		unref_peer(authpeer, "unref pointer into (*authpeer)");
	}
	return 1;
}

/*! \brief Handle incoming REGISTER request */
static int handle_request_register(struct sip_pvt *p, struct sip_request *req, struct ast_sockaddr *addr, const char *e)
{
	enum check_auth_result res;

	/* If this is not the intial request, and the initial request isn't
	 * a register, something screwy happened, so bail */
	if (p->initreq.headers && p->initreq.method != SIP_REGISTER) {
		ast_log(LOG_WARNING, "Ignoring spurious REGISTER with Call-ID: %s\n", p->callid);
		return -1;
	}

	/* Use this as the basis */
	copy_request(&p->initreq, req);
	if (sipdebug)
		ast_debug(4, "Initializing initreq for method %s - callid %s\n", sip_methods[req->method].text, p->callid);
	check_via(p, req);
	if ((res = register_verify(p, addr, req, e)) < 0) {
		const char *reason;

		switch (res) {
		case AUTH_SECRET_FAILED:
			reason = "Wrong password";
			break;
		case AUTH_USERNAME_MISMATCH:
			reason = "Username/auth name mismatch";
			break;
		case AUTH_NOT_FOUND:
			reason = "No matching peer found";
			break;
		case AUTH_UNKNOWN_DOMAIN:
			reason = "Not a local domain";
			break;
		case AUTH_PEER_NOT_DYNAMIC:
			reason = "Peer is not supposed to register";
			break;
		case AUTH_ACL_FAILED:
			reason = "Device does not match ACL";
			break;
		case AUTH_BAD_TRANSPORT:
			reason = "Device not configured to use this transport type";
			break;
		default:
			reason = "Unknown failure";
			break;
		}
		ast_log(LOG_NOTICE, "Registration from '%s' failed for '%s' - %s\n",
			get_header(req, "To"), ast_sockaddr_stringify(addr),
			reason);
		append_history(p, "RegRequest", "Failed : Account %s : %s", get_header(req, "To"), reason);
	} else {
		req->authenticated = 1;
		append_history(p, "RegRequest", "Succeeded : Account %s", get_header(req, "To"));
	}

	if (res < 1) {
		/* Destroy the session, but keep us around for just a bit in case they don't
		   get our 200 OK */
		sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
	}
	return res;
}

/*!
 * \brief Handle incoming SIP requests (methods)
 * \note
 * This is where all incoming requests go first.
 * \note
 * called with p and p->owner locked
 */
static int handle_incoming(struct sip_pvt *p, struct sip_request *req, struct ast_sockaddr *addr, int *recount, int *nounlock)
{
	/* Called with p->lock held, as well as p->owner->lock if appropriate, keeping things
	   relatively static */
	const char *cmd;
	const char *cseq;
	const char *useragent;
	const char *via;
	const char *callid;
	int via_pos = 0;
	uint32_t seqno;
	int len;
	int respid;
	int res = 0;
	int debug = sip_debug_test_pvt(p);
	const char *e;
	int error = 0;
	int oldmethod = p->method;
	int acked = 0;

	/* RFC 3261 - 8.1.1 A valid SIP request must contain To, From, CSeq, Call-ID and Via.
	 * 8.2.6.2 Response must have To, From, Call-ID CSeq, and Via related to the request,
	 * so we can check to make sure these fields exist for all requests and responses */
	cseq = get_header(req, "Cseq");
	cmd = REQ_OFFSET_TO_STR(req, header[0]);
	/* Save the via_pos so we can check later that responses only have 1 Via header */
	via = __get_header(req, "Via", &via_pos);
	/* This must exist already because we've called find_call by now */
	callid = get_header(req, "Call-ID");

	/* Must have Cseq */
	if (ast_strlen_zero(cmd) || ast_strlen_zero(cseq) || ast_strlen_zero(via)) {
		ast_log(LOG_ERROR, "Dropping this SIP message with Call-ID '%s', it's incomplete.\n", callid);
		error = 1;
	}
	if (!error && sscanf(cseq, "%30u%n", &seqno, &len) != 1) {
		ast_log(LOG_ERROR, "No seqno in '%s'. Dropping incomplete message.\n", cmd);
		error = 1;
	}
	if (error) {
		if (!p->initreq.headers) {	/* New call */
			pvt_set_needdestroy(p, "no headers");
		}
		return -1;
	}
	/* Get the command XXX */

	cmd = REQ_OFFSET_TO_STR(req, rlPart1);
	e = ast_skip_blanks(REQ_OFFSET_TO_STR(req, rlPart2));

	/* Save useragent of the client */
	useragent = get_header(req, "User-Agent");
	if (!ast_strlen_zero(useragent))
		ast_string_field_set(p, useragent, useragent);

	/* Find out SIP method for incoming request */
	if (req->method == SIP_RESPONSE) {	/* Response to our request */
		/* ignore means "don't do anything with it" but still have to
		 * respond appropriately.
		 * But in this case this is a response already, so we really
		 * have nothing to do with this message, and even setting the
		 * ignore flag is pointless.
		 */
		if (ast_strlen_zero(e)) {
			return 0;
		}
		if (sscanf(e, "%30d %n", &respid, &len) != 1) {
			ast_log(LOG_WARNING, "Invalid response: '%s'\n", e);
			return 0;
		}
		if (respid <= 0) {
			ast_log(LOG_WARNING, "Invalid SIP response code: '%d'\n", respid);
			return 0;
		}
		/* RFC 3261 - 8.1.3.3 If more than one Via header field value is present in a reponse
		 * the UAC SHOULD discard the message. This is not perfect, as it will not catch multiple
		 * headers joined with a comma. Fixing that would pretty much involve writing a new parser */
		if (!ast_strlen_zero(__get_header(req, "via", &via_pos))) {
			ast_log(LOG_WARNING, "Misrouted SIP response '%s' with Call-ID '%s', too many vias\n", e, callid);
			return 0;
		}
		if (p->ocseq && (p->ocseq < seqno)) {
			ast_debug(1, "Ignoring out of order response %u (expecting %u)\n", seqno, p->ocseq);
			return -1;
		} else {
			char causevar[256], causeval[256];

			if ((respid == 200) || ((respid >= 300) && (respid <= 399))) {
				extract_uri(p, req);
			}

			handle_response(p, respid, e + len, req, seqno);

			if (global_store_sip_cause && p->owner) {
				struct ast_channel *owner = p->owner;

				snprintf(causevar, sizeof(causevar), "MASTER_CHANNEL(HASH(SIP_CAUSE,%s))", owner->name);
				snprintf(causeval, sizeof(causeval), "SIP %s", REQ_OFFSET_TO_STR(req, rlPart2));

				ast_channel_ref(owner);
				sip_pvt_unlock(p);
				ast_channel_unlock(owner);
				*nounlock = 1;
				pbx_builtin_setvar_helper(owner, causevar, causeval);
				ast_channel_unref(owner);
				sip_pvt_lock(p);
			}
		}
		return 0;
	}

	/* New SIP request coming in
	   (could be new request in existing SIP dialog as well...)
	 */			
	
	p->method = req->method;	/* Find out which SIP method they are using */
	ast_debug(4, "**** Received %s (%d) - Command in SIP %s\n", sip_methods[p->method].text, sip_methods[p->method].id, cmd);

	if (p->icseq && (p->icseq > seqno) ) {
		if (p->pendinginvite && seqno == p->pendinginvite && (req->method == SIP_ACK || req->method == SIP_CANCEL)) {
			ast_debug(2, "Got CANCEL or ACK on INVITE with transactions in between.\n");
		} else {
			ast_debug(1, "Ignoring too old SIP packet packet %u (expecting >= %u)\n", seqno, p->icseq);
			if (req->method == SIP_INVITE) {
				unsigned int ran = (ast_random() % 10) + 1;
				char seconds[4];
				snprintf(seconds, sizeof(seconds), "%u", ran);
				transmit_response_with_retry_after(p, "500 Server error", req, seconds);	/* respond according to RFC 3261 14.2 with Retry-After betwewn 0 and 10 */
			} else if (req->method != SIP_ACK) {
				transmit_response(p, "500 Server error", req);	/* We must respond according to RFC 3261 sec 12.2 */
			}
			return -1;
		}
	} else if (p->icseq &&
		   p->icseq == seqno &&
		   req->method != SIP_ACK &&
		   (p->method != SIP_CANCEL || p->alreadygone)) {
		/* ignore means "don't do anything with it" but still have to
		   respond appropriately.  We do this if we receive a repeat of
		   the last sequence number  */
		req->ignore = 1;
		ast_debug(3, "Ignoring SIP message because of retransmit (%s Seqno %u, ours %u)\n", sip_methods[p->method].text, p->icseq, seqno);
	}

	/* RFC 3261 section 9. "CANCEL has no effect on a request to which a UAS has
	 * already given a final response." */
	if (!p->pendinginvite && (req->method == SIP_CANCEL)) {
		transmit_response(p, "481 Call/Transaction Does Not Exist", req);
		return res;
	}

	if (seqno >= p->icseq)
		/* Next should follow monotonically (but not necessarily
		   incrementally -- thanks again to the genius authors of SIP --
		   increasing */
		p->icseq = seqno;

	/* Find their tag if we haven't got it */
	if (ast_strlen_zero(p->theirtag)) {
		char tag[128];

		gettag(req, "From", tag, sizeof(tag));
		ast_string_field_set(p, theirtag, tag);
	}
	snprintf(p->lastmsg, sizeof(p->lastmsg), "Rx: %s", cmd);

	if (sip_cfg.pedanticsipchecking) {
		/* If this is a request packet without a from tag, it's not
			correct according to RFC 3261  */
		/* Check if this a new request in a new dialog with a totag already attached to it,
			RFC 3261 - section 12.2 - and we don't want to mess with recovery  */
		if (!p->initreq.headers && req->has_to_tag) {
			/* If this is a first request and it got a to-tag, it is not for us */
			if (!req->ignore && req->method == SIP_INVITE) {
				transmit_response_reliable(p, "481 Call/Transaction Does Not Exist", req);
				/* Will cease to exist after ACK */
			} else if (req->method != SIP_ACK) {
				transmit_response(p, "481 Call/Transaction Does Not Exist", req);
				sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
			} else {
				ast_debug(1, "Got ACK for unknown dialog... strange.\n");
			}
			return res;
		}
	}

	if (!e && (p->method == SIP_INVITE || p->method == SIP_SUBSCRIBE || p->method == SIP_REGISTER || p->method == SIP_NOTIFY || p->method == SIP_PUBLISH)) {
		transmit_response(p, "400 Bad request", req);
		sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
		return -1;
	}

	/* Handle various incoming SIP methods in requests */
	switch (p->method) {
	case SIP_OPTIONS:
		res = handle_request_options(p, req, addr, e);
		break;
	case SIP_INVITE:
		res = handle_request_invite(p, req, debug, seqno, addr, recount, e, nounlock);
		break;
	case SIP_REFER:
		res = handle_request_refer(p, req, debug, seqno, nounlock);
		break;
	case SIP_CANCEL:
		res = handle_request_cancel(p, req);
		break;
	case SIP_BYE:
		res = handle_request_bye(p, req);
		break;
	case SIP_MESSAGE:
		res = handle_request_message(p, req);
		break;
	case SIP_PUBLISH:
		res = handle_request_publish(p, req, addr, seqno, e);
		break;
	case SIP_SUBSCRIBE:
		res = handle_request_subscribe(p, req, addr, seqno, e);
		break;
	case SIP_REGISTER:
		res = handle_request_register(p, req, addr, e);
		break;
	case SIP_INFO:
		if (req->debug)
			ast_verbose("Receiving INFO!\n");
		if (!req->ignore)
			handle_request_info(p, req);
		else  /* if ignoring, transmit response */
			transmit_response(p, "200 OK", req);
		break;
	case SIP_NOTIFY:
		res = handle_request_notify(p, req, addr, seqno, e);
		break;
	case SIP_UPDATE:
		res = handle_request_update(p, req);
		break;
	case SIP_ACK:
		/* Make sure we don't ignore this */
		if (seqno == p->pendinginvite) {
			p->invitestate = INV_TERMINATED;
			p->pendinginvite = 0;
			acked = __sip_ack(p, seqno, 1 /* response */, 0);
			if (find_sdp(req)) {
				if (process_sdp(p, req, SDP_T38_NONE))
					return -1;
			}
			check_pendings(p);
		} else if (p->glareinvite == seqno) {
			/* handle ack for the 491 pending sent for glareinvite */
			p->glareinvite = 0;
			acked = __sip_ack(p, seqno, 1, 0);
		}
		if (!acked) {
			/* Got an ACK that did not match anything. Ignore
			 * silently and restore previous method */
			p->method = oldmethod;
		}
		if (!p->lastinvite && ast_strlen_zero(p->randdata)) {
			pvt_set_needdestroy(p, "unmatched ACK");
		}
		break;
	default:
		transmit_response_with_allow(p, "501 Method Not Implemented", req, 0);
		ast_log(LOG_NOTICE, "Unknown SIP command '%s' from '%s'\n",
			cmd, ast_sockaddr_stringify(&p->sa));
		/* If this is some new method, and we don't have a call, destroy it now */
		if (!p->initreq.headers) {
			pvt_set_needdestroy(p, "unimplemented method");
		}
		break;
	}
	return res;
}

/*! \brief Read data from SIP UDP socket
\note sipsock_read locks the owner channel while we are processing the SIP message
\return 1 on error, 0 on success
\note Successful messages is connected to SIP call and forwarded to handle_incoming()
*/
static int sipsock_read(int *id, int fd, short events, void *ignore)
{
	struct sip_request req;
	struct ast_sockaddr addr;
	int res;
	static char readbuf[65535];

	memset(&req, 0, sizeof(req));
	res = ast_recvfrom(fd, readbuf, sizeof(readbuf) - 1, 0, &addr);
	if (res < 0) {
#if !defined(__FreeBSD__)
		if (errno == EAGAIN)
			ast_log(LOG_NOTICE, "SIP: Received packet with bad UDP checksum\n");
		else
#endif
		if (errno != ECONNREFUSED)
			ast_log(LOG_WARNING, "Recv error: %s\n", strerror(errno));
		return 1;
	}

	readbuf[res] = '\0';

	if (!(req.data = ast_str_create(SIP_MIN_PACKET))) {
		return 1;
	}

	if (ast_str_set(&req.data, 0, "%s", readbuf) == AST_DYNSTR_BUILD_FAILED) {
		return -1;
	}

	req.socket.fd = sipsock;
	set_socket_transport(&req.socket, SIP_TRANSPORT_UDP);
	req.socket.tcptls_session	= NULL;
	req.socket.port = htons(ast_sockaddr_port(&bindaddr));

	handle_request_do(&req, &addr);
	deinit_req(&req);

	return 1;
}

/*! \brief Handle incoming SIP message - request or response

 	This is used for all transports (udp, tcp and tcp/tls)
*/
static int handle_request_do(struct sip_request *req, struct ast_sockaddr *addr)
{
	struct sip_pvt *p;
	struct ast_channel *owner_chan_ref = NULL;
	int recount = 0;
	int nounlock = 0;

	if (sip_debug_test_addr(addr))	/* Set the debug flag early on packet level */
		req->debug = 1;
	if (sip_cfg.pedanticsipchecking)
		lws2sws(req->data);	/* Fix multiline headers */
	if (req->debug) {
		ast_verbose("\n<--- SIP read from %s:%s --->\n%s\n<------------->\n",
			get_transport(req->socket.type), ast_sockaddr_stringify(addr), req->data->str);
	}

	if (parse_request(req) == -1) { /* Bad packet, can't parse */
		ast_str_reset(req->data); /* nulling this out is NOT a good idea here. */
		return 1;
	}
	req->method = find_sip_method(REQ_OFFSET_TO_STR(req, rlPart1));

	if (req->debug)
		ast_verbose("--- (%d headers %d lines)%s ---\n", req->headers, req->lines, (req->headers + req->lines == 0) ? " Nat keepalive" : "");

	if (req->headers < 2) {	/* Must have at least two headers */
		ast_str_reset(req->data); /* nulling this out is NOT a good idea here. */
		return 1;
	}

	/* Process request, with netlock held, and with usual deadlock avoidance */
	ast_mutex_lock(&netlock);

	/* Find the active SIP dialog or create a new one */
	p = find_call(req, addr, req->method);	/* returns p with a reference only. _NOT_ locked*/
	if (p == NULL) {
		ast_debug(1, "Invalid SIP message - rejected , no callid, len %zu\n", ast_str_strlen(req->data));
		ast_mutex_unlock(&netlock);
		return 1;
	}

	/* Lock both the pvt and the owner if owner is present.  This will
	 * not fail. */
	owner_chan_ref = sip_pvt_lock_full(p);

	copy_socket_data(&p->socket, &req->socket);
	ast_sockaddr_copy(&p->recv, addr);

	/* if we have an owner, then this request has been authenticated */
	if (p->owner) {
		req->authenticated = 1;
	}

	if (p->do_history) /* This is a request or response, note what it was for */
		append_history(p, "Rx", "%s / %s / %s", req->data->str, get_header(req, "CSeq"), REQ_OFFSET_TO_STR(req, rlPart2));

	if (handle_incoming(p, req, addr, &recount, &nounlock) == -1) {
		/* Request failed */
		ast_debug(1, "SIP message could not be handled, bad request: %-70.70s\n", p->callid[0] ? p->callid : "<no callid>");
	}

	if (recount) {
		ast_update_use_count();
	}

	if (p->owner && !nounlock) {
		ast_channel_unlock(p->owner);
	}
	if (owner_chan_ref) {
		ast_channel_unref(owner_chan_ref);
	}
	sip_pvt_unlock(p);
	ao2_t_ref(p, -1, "throw away dialog ptr from find_call at end of routine"); /* p is gone after the return */
	ast_mutex_unlock(&netlock);

	return 1;
}

/*! \brief Returns the port to use for this socket
 *
 * \param type The type of transport used
 * \param port Port we are checking to see if it's the standard port.
 * \note port is expected in host byte order
 */
static int sip_standard_port(enum sip_transport type, int port)
{
	if (type & SIP_TRANSPORT_TLS)
		return port == STANDARD_TLS_PORT;
	else
		return port == STANDARD_SIP_PORT;
}

static int threadinfo_locate_cb(void *obj, void *arg, int flags)
{
	struct sip_threadinfo *th = obj;
	struct ast_sockaddr *s = arg;

	if (!ast_sockaddr_cmp(s, &th->tcptls_session->remote_address)) {
		return CMP_MATCH | CMP_STOP;
	}

	return 0;
}

/*!
 * \brief Find thread for TCP/TLS session (based on IP/Port
 *
 * \note This function returns an astobj2 reference
 */
static struct ast_tcptls_session_instance *sip_tcp_locate(struct ast_sockaddr *s)
{
	struct sip_threadinfo *th;
	struct ast_tcptls_session_instance *tcptls_instance = NULL;

	if ((th = ao2_callback(threadt, 0, threadinfo_locate_cb, s))) {
		tcptls_instance = (ao2_ref(th->tcptls_session, +1), th->tcptls_session);
		ao2_t_ref(th, -1, "decrement ref from callback");
	}

	return tcptls_instance;
}

/*!
 * \brief Helper for dns resolution to filter by address family.
 *
 * \note return 0 if addr is [::] else it returns addr's family.
 */
int get_address_family_filter(const struct ast_sockaddr *addr)
{
	if (ast_sockaddr_is_ipv6(addr) && ast_sockaddr_is_any(addr)) {
		return 0;
	}

	return addr->ss.ss_family;
}

/*! \todo Get socket for dialog, prepare if needed, and return file handle  */
static int sip_prepare_socket(struct sip_pvt *p)
{
	struct sip_socket *s = &p->socket;
	static const char name[] = "SIP socket";
	struct sip_threadinfo *th = NULL;
	struct ast_tcptls_session_instance *tcptls_session;
	struct ast_tcptls_session_args *ca;
	struct ast_sockaddr sa_tmp;
	pthread_t launched;

	/* check to see if a socket is already active */
	if ((s->fd != -1) && (s->type == SIP_TRANSPORT_UDP)) {
		return s->fd;
	}
	if ((s->type & (SIP_TRANSPORT_TCP | SIP_TRANSPORT_TLS)) &&
			(s->tcptls_session) &&
			(s->tcptls_session->fd != -1)) {
		return s->tcptls_session->fd;
	}

	/*! \todo Check this... This might be wrong, depending on the proxy configuration
		If proxy is in "force" mode its correct.
	 */
	if (p->outboundproxy && p->outboundproxy->transport) {
		s->type = p->outboundproxy->transport;
	}

	if (s->type == SIP_TRANSPORT_UDP) {
		s->fd = sipsock;
		return s->fd;
	}

	/* At this point we are dealing with a TCP/TLS connection
	 * 1. We need to check to see if a connection thread exists
	 *    for this address, if so use that.
	 * 2. If a thread does not exist for this address, but the tcptls_session
	 *    exists on the socket, the connection was closed.
	 * 3. If no tcptls_session thread exists for the address, and no tcptls_session
	 *    already exists on the socket, create a new one and launch a new thread.
	 */

	/* 1.  check for existing threads */
	ast_sockaddr_copy(&sa_tmp, sip_real_dst(p));
	if ((tcptls_session = sip_tcp_locate(&sa_tmp))) {
		s->fd = tcptls_session->fd;
		if (s->tcptls_session) {
			ao2_ref(s->tcptls_session, -1);
			s->tcptls_session = NULL;
		}
		s->tcptls_session = tcptls_session;
		return s->fd;
	/* 2.  Thread not found, if tcptls_session already exists, it once had a thread and is now terminated */
	} else if (s->tcptls_session) {
		return s->fd; /* XXX whether reconnection is ever necessary here needs to be investigated further */
	}

	/* 3.  Create a new TCP/TLS client connection */
	/* create new session arguments for the client connection */
	if (!(ca = ao2_alloc(sizeof(*ca), sip_tcptls_client_args_destructor)) ||
		!(ca->name = ast_strdup(name))) {
		goto create_tcptls_session_fail;
	}
	ca->accept_fd = -1;
	ast_sockaddr_copy(&ca->remote_address,sip_real_dst(p));
	/* if type is TLS, we need to create a tls cfg for this session arg */
	if (s->type == SIP_TRANSPORT_TLS) {
		if (!(ca->tls_cfg = ast_calloc(1, sizeof(*ca->tls_cfg)))) {
			goto create_tcptls_session_fail;
		}
		memcpy(ca->tls_cfg, &default_tls_cfg, sizeof(*ca->tls_cfg));

		if (!(ca->tls_cfg->certfile = ast_strdup(default_tls_cfg.certfile)) ||
			!(ca->tls_cfg->pvtfile = ast_strdup(default_tls_cfg.pvtfile)) ||
			!(ca->tls_cfg->cipher = ast_strdup(default_tls_cfg.cipher)) ||
			!(ca->tls_cfg->cafile = ast_strdup(default_tls_cfg.cafile)) ||
			!(ca->tls_cfg->capath = ast_strdup(default_tls_cfg.capath))) {

			goto create_tcptls_session_fail;
		}

		/* this host is used as the common name in ssl/tls */
		if (!ast_strlen_zero(p->tohost)) {
			ast_copy_string(ca->hostname, p->tohost, sizeof(ca->hostname));
		}
	}

	/* Create a client connection for address, this does not start the connection, just sets it up. */
	if (!(s->tcptls_session = ast_tcptls_client_create(ca))) {
		goto create_tcptls_session_fail;
	}

	s->fd = s->tcptls_session->fd;

	/* client connections need to have the sip_threadinfo object created before
	 * the thread is detached.  This ensures the alert_pipe is up before it will
	 * be used.  Note that this function links the new threadinfo object into the
	 * threadt container. */
	if (!(th = sip_threadinfo_create(s->tcptls_session, s->type))) {
		goto create_tcptls_session_fail;
	}

	/* Give the new thread a reference to the tcptls_session */
	ao2_ref(s->tcptls_session, +1);

	if (ast_pthread_create_detached_background(&launched, NULL, sip_tcp_worker_fn, s->tcptls_session)) {
		ast_debug(1, "Unable to launch '%s'.", ca->name);
		ao2_ref(s->tcptls_session, -1); /* take away the thread ref we just gave it */
		goto create_tcptls_session_fail;
	}

	return s->fd;

create_tcptls_session_fail:
	if (ca) {
		ao2_t_ref(ca, -1, "failed to create client, getting rid of client tcptls_session arguments");
	}
	if (s->tcptls_session) {
		ast_tcptls_close_session_file(tcptls_session);
		s->fd = -1;
		ao2_ref(s->tcptls_session, -1);
		s->tcptls_session = NULL;
	}
	if (th) {
		ao2_t_unlink(threadt, th, "Removing tcptls thread info object, thread failed to open");
	}

	return -1;
}

/*!
 * \brief Get cached MWI info
 * \return TRUE if found MWI in cache
 */
static int get_cached_mwi(struct sip_peer *peer, int *new, int *old)
{
	struct sip_mailbox *mailbox;
	int in_cache;

	in_cache = 0;
	AST_LIST_TRAVERSE(&peer->mailboxes, mailbox, entry) {
		struct ast_event *event;
		event = ast_event_get_cached(AST_EVENT_MWI,
			AST_EVENT_IE_MAILBOX, AST_EVENT_IE_PLTYPE_STR, mailbox->mailbox,
			AST_EVENT_IE_CONTEXT, AST_EVENT_IE_PLTYPE_STR, S_OR(mailbox->context, "default"),
			AST_EVENT_IE_END);
		if (!event)
			continue;
		*new += ast_event_get_ie_uint(event, AST_EVENT_IE_NEWMSGS);
		*old += ast_event_get_ie_uint(event, AST_EVENT_IE_OLDMSGS);
		ast_event_destroy(event);
		in_cache = 1;
	}

	return in_cache;
}

/*! \brief Send message waiting indication to alert peer that they've got voicemail
 *  \note Both peer and associated sip_pvt must be unlocked prior to calling this function
*/
static int sip_send_mwi_to_peer(struct sip_peer *peer, int cache_only)
{
	/* Called with peer lock, but releases it */
	struct sip_pvt *p;
	int newmsgs = 0, oldmsgs = 0;
	const char *vmexten = NULL;

	ao2_lock(peer);

	if (peer->vmexten) {
		vmexten = ast_strdupa(peer->vmexten);
	}

	if (ast_test_flag((&peer->flags[1]), SIP_PAGE2_SUBSCRIBEMWIONLY) && !peer->mwipvt) {
		ao2_unlock(peer);
		return 0;
	}

	/* Do we have an IP address? If not, skip this peer */
	if (ast_sockaddr_isnull(&peer->addr) && ast_sockaddr_isnull(&peer->defaddr)) {
		ao2_unlock(peer);
		return 0;
	}

	/* Attempt to use cached mwi to get message counts. */
	if (!get_cached_mwi(peer, &newmsgs, &oldmsgs) && !cache_only) {
		/* Fall back to manually checking the mailbox if not cache_only and get_cached_mwi failed */
		struct ast_str *mailbox_str = ast_str_alloca(512);
		peer_mailboxes_to_str(&mailbox_str, peer);
		ao2_unlock(peer);
		ast_app_inboxcount(mailbox_str->str, &newmsgs, &oldmsgs);
		ao2_lock(peer);
	}

	if (peer->mwipvt) {
		/* Base message on subscription */
		p = dialog_ref(peer->mwipvt, "sip_send_mwi_to_peer: Setting dialog ptr p from peer->mwipvt");
		ao2_unlock(peer);
	} else {
		ao2_unlock(peer);
		/* Build temporary dialog for this message */
		if (!(p = sip_alloc(NULL, NULL, 0, SIP_NOTIFY, NULL))) {
			return -1;
		}

		/* If we don't set the socket type to 0, then create_addr_from_peer will fail immediately if the peer
		 * uses any transport other than UDP. We set the type to 0 here and then let create_addr_from_peer copy
		 * the peer's socket information to the sip_pvt we just allocated
		 */
		set_socket_transport(&p->socket, 0);
		if (create_addr_from_peer(p, peer)) {
			/* Maybe they're not registered, etc. */
			dialog_unlink_all(p);
			dialog_unref(p, "unref dialog p just created via sip_alloc");
			/* sip_destroy(p); */
			return 0;
		}
		/* Recalculate our side, and recalculate Call ID */
		ast_sip_ouraddrfor(&p->sa, &p->ourip, p);
		build_via(p);

		ao2_lock(peer);
		if (!ast_strlen_zero(peer->mwi_from)) {
			ast_string_field_set(p, mwi_from, peer->mwi_from);
		} else if (!ast_strlen_zero(default_mwi_from)) {
			ast_string_field_set(p, mwi_from, default_mwi_from);
		}
		ao2_unlock(peer);

		/* Change the dialog callid. */
		change_callid_pvt(p, NULL);

		/* Destroy this session after 32 secs */
		sip_scheddestroy(p, DEFAULT_TRANS_TIMEOUT);
	}

	/* We have multiple threads (mwi events and monitor retransmits) working with this PVT and as we modify the sip history if that's turned on,
	   we really need to have a lock on it */
	sip_pvt_lock(p);

	/* Send MWI */
	ast_set_flag(&p->flags[0], SIP_OUTGOING);
	/* the following will decrement the refcount on p as it finishes */
	transmit_notify_with_mwi(p, newmsgs, oldmsgs, vmexten);
	sip_pvt_unlock(p);
	dialog_unref(p, "unref dialog ptr p just before it goes out of scope at the end of sip_send_mwi_to_peer.");

	return 0;
}

/*! \brief helper function for the monitoring thread -- seems to be called with the assumption that the dialog is locked */
static void check_rtp_timeout(struct sip_pvt *dialog, time_t t)
{
	/* If we have no RTP or no active owner, no need to check timers */
	if (!dialog->rtp || !dialog->owner)
		return;
	/* If the call is not in UP state or redirected outside Asterisk, no need to check timers */

	if (dialog->owner->_state != AST_STATE_UP || !ast_sockaddr_isnull(&dialog->redirip))
		return;

	/* If the call is involved in a T38 fax session do not check RTP timeout */
	if (dialog->t38.state == T38_ENABLED)
		return;

	/* If we have no timers set, return now */
	if (!ast_rtp_instance_get_keepalive(dialog->rtp) && !ast_rtp_instance_get_timeout(dialog->rtp) && !ast_rtp_instance_get_hold_timeout(dialog->rtp)) {
		return;
	}

	/* Check AUDIO RTP keepalives */
	if (dialog->lastrtptx && ast_rtp_instance_get_keepalive(dialog->rtp) &&
		    (t > dialog->lastrtptx + ast_rtp_instance_get_keepalive(dialog->rtp))) {
		/* Need to send an empty RTP packet */
		dialog->lastrtptx = time(NULL);
		ast_rtp_instance_sendcng(dialog->rtp, 0);
	}

	/*! \todo Check video RTP keepalives

		Do we need to move the lastrtptx to the RTP structure to have one for audio and one
		for video? It really does belong to the RTP structure.
	*/

	/* Check AUDIO RTP timers */
	if (dialog->lastrtprx && (ast_rtp_instance_get_timeout(dialog->rtp) || ast_rtp_instance_get_hold_timeout(dialog->rtp)) && (t > dialog->lastrtprx + ast_rtp_instance_get_timeout(dialog->rtp))) {
		if (!ast_test_flag(&dialog->flags[1], SIP_PAGE2_CALL_ONHOLD) || (ast_rtp_instance_get_hold_timeout(dialog->rtp) && (t > dialog->lastrtprx + ast_rtp_instance_get_hold_timeout(dialog->rtp)))) {
			/* Needs a hangup */
			if (ast_rtp_instance_get_timeout(dialog->rtp)) {
				if (!dialog->owner || ast_channel_trylock(dialog->owner)) {
					/*
					 * Don't block, just try again later.
					 * If there was no owner, the call is dead already.
					 */
					return;
				}
				ast_log(LOG_NOTICE, "Disconnecting call '%s' for lack of RTP activity in %ld seconds\n",
					dialog->owner->name, (long) (t - dialog->lastrtprx));
				/* Issue a softhangup */
				ast_softhangup_nolock(dialog->owner, AST_SOFTHANGUP_DEV);
				ast_channel_unlock(dialog->owner);
				/* forget the timeouts for this call, since a hangup
				   has already been requested and we don't want to
				   repeatedly request hangups
				*/
				ast_rtp_instance_set_timeout(dialog->rtp, 0);
				ast_rtp_instance_set_hold_timeout(dialog->rtp, 0);
				if (dialog->vrtp) {
					ast_rtp_instance_set_timeout(dialog->vrtp, 0);
					ast_rtp_instance_set_hold_timeout(dialog->vrtp, 0);
				}
			}
		}
	}
}

/*! \brief The SIP monitoring thread
\note	This thread monitors all the SIP sessions and peers that needs notification of mwi
	(and thus do not have a separate thread) indefinitely
*/
static void *do_monitor(void *data)
{
	int res;
	time_t t;
	int reloading;

	/* Add an I/O event to our SIP UDP socket */
	if (sipsock > -1)
		sipsock_read_id = ast_io_add(io, sipsock, sipsock_read, AST_IO_IN, NULL);

	/* From here on out, we die whenever asked */
	for(;;) {
		/* Check for a reload request */
		ast_mutex_lock(&sip_reload_lock);
		reloading = sip_reloading;
		sip_reloading = FALSE;
		ast_mutex_unlock(&sip_reload_lock);
		if (reloading) {
			ast_verb(1, "Reloading SIP\n");
			sip_do_reload(sip_reloadreason);

			/* Change the I/O fd of our UDP socket */
			if (sipsock > -1) {
				if (sipsock_read_id)
					sipsock_read_id = ast_io_change(io, sipsock_read_id, sipsock, NULL, 0, NULL);
				else
					sipsock_read_id = ast_io_add(io, sipsock, sipsock_read, AST_IO_IN, NULL);
			} else if (sipsock_read_id) {
				ast_io_remove(io, sipsock_read_id);
				sipsock_read_id = NULL;
			}
		}

		/* Check for dialogs needing to be killed */
		t = time(NULL);
		/* don't scan the dialogs list if it hasn't been a reasonable period
		   of time since the last time we did it (when MWI is being sent, we can
		   get back to this point every millisecond or less)
		*/
		/*
		 * We cannot hold the dialogs container lock when we destroy a
		 * dialog because of potential deadlocks.  Instead we link the
		 * doomed dialog into dialogs_to_destroy and then iterate over
		 * that container destroying the dialogs.
		 */
		ao2_t_callback(dialogs, OBJ_NODATA | OBJ_MULTIPLE, dialog_needdestroy, &t,
			"callback to monitor dialog status");
		if (ao2_container_count(dialogs_to_destroy)) {
			/* Now destroy the found dialogs that need to be destroyed. */
			ao2_t_callback(dialogs_to_destroy, OBJ_UNLINK | OBJ_NODATA | OBJ_MULTIPLE,
				dialog_unlink_callback, NULL, "callback to dialog_unlink_all");
		}

		/* XXX TODO The scheduler usage in this module does not have sufficient
		 * synchronization being done between running the scheduler and places
		 * scheduling tasks.  As it is written, any scheduled item may not run
		 * any sooner than about  1 second, regardless of whether a sooner time
		 * was asked for. */

		pthread_testcancel();
		/* Wait for sched or io */
		res = ast_sched_wait(sched);
		if ((res < 0) || (res > 1000))
			res = 1000;
		res = ast_io_wait(io, res);
		if (res > 20)
			ast_debug(1, "chan_sip: ast_io_wait ran %d all at once\n", res);
		ast_mutex_lock(&monlock);
		res = ast_sched_runq(sched);
		if (res >= 20)
			ast_debug(1, "chan_sip: ast_sched_runq ran %d all at once\n", res);
		if (global_store_sip_cause && res >= 100)
			ast_log(LOG_WARNING, "scheduler delays detected, setting 'storesipcause' to 'no' in %s will improve performance\n", config);
		ast_mutex_unlock(&monlock);
	}

	/* Never reached */
	return NULL;
}

/*! \brief Start the channel monitor thread */
static int restart_monitor(void)
{
	/* If we're supposed to be stopped -- stay stopped */
	if (monitor_thread == AST_PTHREADT_STOP)
		return 0;
	ast_mutex_lock(&monlock);
	if (monitor_thread == pthread_self()) {
		ast_mutex_unlock(&monlock);
		ast_log(LOG_WARNING, "Cannot kill myself\n");
		return -1;
	}
	if (monitor_thread != AST_PTHREADT_NULL) {
		/* Wake up the thread */
		pthread_kill(monitor_thread, SIGURG);
	} else {
		/* Start a new monitor */
		if (ast_pthread_create_background(&monitor_thread, NULL, do_monitor, NULL) < 0) {
			ast_mutex_unlock(&monlock);
			ast_log(LOG_ERROR, "Unable to start monitor thread.\n");
			return -1;
		}
	}
	ast_mutex_unlock(&monlock);
	return 0;
}


/*! \brief Session-Timers: Restart session timer */
static void restart_session_timer(struct sip_pvt *p)
{
	if (!p->stimer) {
		ast_log(LOG_WARNING, "Null stimer in restart_session_timer - %s\n", p->callid);
		return;
	}

	if (p->stimer->st_active == TRUE) {
		ast_debug(2, "Session timer stopped: %d - %s\n", p->stimer->st_schedid, p->callid);
		AST_SCHED_DEL_UNREF(sched, p->stimer->st_schedid,
				dialog_unref(p, "Removing session timer ref"));
		start_session_timer(p);
	}
}


/*! \brief Session-Timers: Stop session timer */
static void stop_session_timer(struct sip_pvt *p)
{
	if (!p->stimer) {
		ast_log(LOG_WARNING, "Null stimer in stop_session_timer - %s\n", p->callid);
		return;
	}

	if (p->stimer->st_active == TRUE) {
		p->stimer->st_active = FALSE;
		ast_debug(2, "Session timer stopped: %d - %s\n", p->stimer->st_schedid, p->callid);
		AST_SCHED_DEL_UNREF(sched, p->stimer->st_schedid,
				dialog_unref(p, "removing session timer ref"));
	}
}


/*! \brief Session-Timers: Start session timer */
static void start_session_timer(struct sip_pvt *p)
{
	if (!p->stimer) {
		ast_log(LOG_WARNING, "Null stimer in start_session_timer - %s\n", p->callid);
		return;
	}

	if (p->stimer->st_schedid > -1) {
		/* in the event a timer is already going, stop it */
		ast_debug(2, "Session timer stopped: %d - %s\n", p->stimer->st_schedid, p->callid);
		AST_SCHED_DEL_UNREF(sched, p->stimer->st_schedid,
			dialog_unref(p, "unref stimer->st_schedid from dialog"));
	}

	p->stimer->st_schedid  = ast_sched_add(sched, p->stimer->st_interval * 1000 / 2, proc_session_timer, 
			dialog_ref(p, "adding session timer ref"));
	if (p->stimer->st_schedid < 0) {
		dialog_unref(p, "removing session timer ref");
		ast_log(LOG_ERROR, "ast_sched_add failed - %s\n", p->callid);
	} else {
		p->stimer->st_active = TRUE;
		ast_debug(2, "Session timer started: %d - %s\n", p->stimer->st_schedid, p->callid);
	}
}


/*! \brief Session-Timers: Process session refresh timeout event */
static int proc_session_timer(const void *vp)
{
	struct sip_pvt *p = (struct sip_pvt *) vp;
	int sendreinv = FALSE;
	int res = 0;

	if (!p->stimer) {
		ast_log(LOG_WARNING, "Null stimer in proc_session_timer - %s\n", p->callid);
		goto return_unref;
	}

	ast_debug(2, "Session timer expired: %d - %s\n", p->stimer->st_schedid, p->callid);

	if (!p->owner) {
		goto return_unref;
	}

	if ((p->stimer->st_active != TRUE) || (p->owner->_state != AST_STATE_UP)) {
		goto return_unref;
	}

	switch (p->stimer->st_ref) {
	case SESSION_TIMER_REFRESHER_UAC:
		if (p->outgoing_call == TRUE) {
	  		sendreinv = TRUE;
		}
		break;
	case SESSION_TIMER_REFRESHER_UAS:
		if (p->outgoing_call != TRUE) {
  			sendreinv = TRUE;
		}
		break;
	default:
		ast_log(LOG_ERROR, "Unknown session refresher %d\n", p->stimer->st_ref);
		goto return_unref;
	}

	if (sendreinv == TRUE) {
		res = 1;
		transmit_reinvite_with_sdp(p, FALSE, TRUE);
	} else {
		p->stimer->st_expirys++;
		if (p->stimer->st_expirys >= 2) {
			if (p->stimer->quit_flag) {
				goto return_unref;
			}
			ast_log(LOG_WARNING, "Session-Timer expired - %s\n", p->callid);
			sip_pvt_lock(p);
			while (p->owner && ast_channel_trylock(p->owner)) {
				sip_pvt_unlock(p);
				usleep(1);
				if (p->stimer && p->stimer->quit_flag) {
					goto return_unref;
				}
				sip_pvt_lock(p);
			}

			ast_softhangup_nolock(p->owner, AST_SOFTHANGUP_DEV);
			ast_channel_unlock(p->owner);
			sip_pvt_unlock(p);
		} else {
			res = 1;
		}
	}

return_unref:
	if (!res) {
		/* An error occurred.  Stop session timer processing */
		if (p->stimer) {
			ast_debug(2, "Session timer stopped: %d - %s\n", p->stimer->st_schedid, p->callid);
			p->stimer->st_schedid = -1;
			stop_session_timer(p);
		}

		/* If we are not asking to be rescheduled, then we need to release our
		 * reference to the dialog. */
		dialog_unref(p, "removing session timer ref");
	}

	return res;
}


/*! \brief Session-Timers: Function for parsing Min-SE header */
int parse_minse (const char *p_hdrval, int *const p_interval)
{
	if (ast_strlen_zero(p_hdrval)) {
		ast_log(LOG_WARNING, "Null Min-SE header\n");
		return -1;
	}

	*p_interval = 0;
	p_hdrval = ast_skip_blanks(p_hdrval);
	if (!sscanf(p_hdrval, "%30d", p_interval)) {
		ast_log(LOG_WARNING, "Parsing of Min-SE header failed %s\n", p_hdrval);
		return -1;
	}

	ast_debug(2, "Received Min-SE: %d\n", *p_interval);
	return 0;
}


/*! \brief Session-Timers: Function for parsing Session-Expires header */
int parse_session_expires(const char *p_hdrval, int *const p_interval, enum st_refresher *const p_ref)
{
	char *p_token;
	int  ref_idx;
	char *p_se_hdr;

	if (ast_strlen_zero(p_hdrval)) {
		ast_log(LOG_WARNING, "Null Session-Expires header\n");
		return -1;
	}

	*p_ref = SESSION_TIMER_REFRESHER_AUTO;
	*p_interval = 0;

	p_se_hdr = ast_strdupa(p_hdrval);
	p_se_hdr = ast_skip_blanks(p_se_hdr);

	while ((p_token = strsep(&p_se_hdr, ";"))) {
		p_token = ast_skip_blanks(p_token);
		if (!sscanf(p_token, "%30d", p_interval)) {
			ast_log(LOG_WARNING, "Parsing of Session-Expires failed\n");
			return -1;
		}

		ast_debug(2, "Session-Expires: %d\n", *p_interval);

		if (!p_se_hdr)
			continue;

		p_se_hdr = ast_skip_blanks(p_se_hdr);
		ref_idx = strlen("refresher=");
		if (!strncasecmp(p_se_hdr, "refresher=", ref_idx)) {
			p_se_hdr += ref_idx;
			p_se_hdr = ast_skip_blanks(p_se_hdr);

			if (!strncasecmp(p_se_hdr, "uac", strlen("uac"))) {
				*p_ref = SESSION_TIMER_REFRESHER_UAC;
				ast_debug(2, "Refresher: UAC\n");
			} else if (!strncasecmp(p_se_hdr, "uas", strlen("uas"))) {
				*p_ref = SESSION_TIMER_REFRESHER_UAS;
				ast_debug(2, "Refresher: UAS\n");
			} else {
				ast_log(LOG_WARNING, "Invalid refresher value %s\n", p_se_hdr);
				return -1;
			}
			break;
		}
	}
	return 0;
}


/*! \brief Handle 422 response to INVITE with session-timer requested

   Session-Timers:   An INVITE originated by Asterisk that asks for session-timers support
   from the UAS can result into a 422 response. This is how a UAS or an intermediary proxy
   server tells Asterisk that the session refresh interval offered by Asterisk is too low
   for them.  The proc_422_rsp() function handles a 422 response.  It extracts the Min-SE
   header that comes back in 422 and sends a new INVITE accordingly. */
static void proc_422_rsp(struct sip_pvt *p, struct sip_request *rsp)
{
	int rtn;
	const char *p_hdrval;
	int minse;

	p_hdrval = get_header(rsp, "Min-SE");
	if (ast_strlen_zero(p_hdrval)) {
		ast_log(LOG_WARNING, "422 response without a Min-SE header %s\n", p_hdrval);
		return;
	}
	rtn = parse_minse(p_hdrval, &minse);
	if (rtn != 0) {
		ast_log(LOG_WARNING, "Parsing of Min-SE header failed %s\n", p_hdrval);
		return;
	}
	p->stimer->st_interval = minse;
	transmit_invite(p, SIP_INVITE, 1, 2, NULL);
}


/*! \brief Get Max or Min SE (session timer expiry)
 * \param p pointer to the SIP dialog
 * \param max if true, get max se, otherwise min se
*/
int st_get_se(struct sip_pvt *p, int max)
{
	if (max == TRUE) {
		if (p->stimer->st_cached_max_se) {
			return  p->stimer->st_cached_max_se;
		}
		if (p->relatedpeer) {
			p->stimer->st_cached_max_se = p->relatedpeer->stimer.st_max_se;
			return (p->stimer->st_cached_max_se);
		}
		p->stimer->st_cached_max_se = global_max_se;
		return (p->stimer->st_cached_max_se);
	} 
	/* Find Min SE timer */
	if (p->stimer->st_cached_min_se) {
		return p->stimer->st_cached_min_se;
	} 
	if (p->relatedpeer) {
		p->stimer->st_cached_min_se = p->relatedpeer->stimer.st_min_se;
		return (p->stimer->st_cached_min_se);
	}
	p->stimer->st_cached_min_se = global_min_se;
	return (p->stimer->st_cached_min_se);
}


/*! \brief Get the entity (UAC or UAS) that's acting as the session-timer refresher
 * \param p pointer to the SIP dialog
*/
enum st_refresher st_get_refresher(struct sip_pvt *p)
{
	if (p->stimer->st_cached_ref != SESSION_TIMER_REFRESHER_AUTO)
		return p->stimer->st_cached_ref;

	if (p->relatedpeer) {
		p->stimer->st_cached_ref = p->relatedpeer->stimer.st_ref;
		return p->stimer->st_cached_ref;
	}
	
	p->stimer->st_cached_ref = global_st_refresher;
	return global_st_refresher;
}


/*!
 * \brief Get the session-timer mode 
 * \param p pointer to the SIP dialog 
 * \param no_cached, set this to true in order to force a peername lookup on
 *        the session timer mode.
*/
enum st_mode st_get_mode(struct sip_pvt *p, int no_cached)
{
	if (!p->stimer)
		sip_st_alloc(p);

	if (!no_cached && p->stimer->st_cached_mode != SESSION_TIMER_MODE_INVALID)
		return p->stimer->st_cached_mode;

	if (p->relatedpeer) {
		p->stimer->st_cached_mode = p->relatedpeer->stimer.st_mode_oper;
		return p->stimer->st_cached_mode;
	}

	p->stimer->st_cached_mode = global_st_mode;
	return global_st_mode;
}


/*! \brief React to lack of answer to Qualify poke */
static int sip_poke_noanswer(const void *data)
{
	struct sip_peer *peer = (struct sip_peer *)data;

	peer->pokeexpire = -1;

	if (peer->lastms > -1) {
		ast_log(LOG_NOTICE, "Peer '%s' is now UNREACHABLE!  Last qualify: %d\n", peer->name, peer->lastms);
		if (sip_cfg.peer_rtupdate) {
			ast_update_realtime(ast_check_realtime("sipregs") ? "sipregs" : "sippeers", "name", peer->name, "lastms", "-1", SENTINEL);
		}
		manager_event(EVENT_FLAG_SYSTEM, "PeerStatus", "ChannelType: SIP\r\nPeer: SIP/%s\r\nPeerStatus: Unreachable\r\nTime: %d\r\n", peer->name, -1);
		if (sip_cfg.regextenonqualify) {
			register_peer_exten(peer, FALSE);
		}
	}

	if (peer->call) {
		dialog_unlink_all(peer->call);
		peer->call = dialog_unref(peer->call, "unref dialog peer->call");
		/* peer->call = sip_destroy(peer->call);*/
	}

	/* Don't send a devstate change if nothing changed. */
	if (peer->lastms > -1) {
		peer->lastms = -1;
		ast_devstate_changed(AST_DEVICE_UNKNOWN, "SIP/%s", peer->name);
	}

	/* Try again quickly */
	AST_SCHED_REPLACE_UNREF(peer->pokeexpire, sched,
			DEFAULT_FREQ_NOTOK, sip_poke_peer_s, peer,
			unref_peer(_data, "removing poke peer ref"),
			unref_peer(peer, "removing poke peer ref"),
			ref_peer(peer, "adding poke peer ref"));

	/* Release the ref held by the running scheduler entry */
	unref_peer(peer, "release peer poke noanswer ref");

	return 0;
}

/*! \brief Check availability of peer, also keep NAT open
\note	This is done with 60 seconds between each ping,
	unless forced by cli or manager. If peer is unreachable,
	we check every 10th second by default.
*/
static int sip_poke_peer(struct sip_peer *peer, int force)
{
	struct sip_pvt *p;
	int xmitres = 0;
	
	if ((!peer->maxms && !force) || ast_sockaddr_isnull(&peer->addr)) {
		/* IF we have no IP, or this isn't to be monitored, return
		  immediately after clearing things out */
		AST_SCHED_DEL_UNREF(sched, peer->pokeexpire,
				unref_peer(peer, "removing poke peer ref"));
		
		peer->lastms = 0;
		if (peer->call) {
			peer->call = dialog_unref(peer->call, "unref dialog peer->call");
		}
		return 0;
	}
	if (peer->call) {
		if (sipdebug) {
			ast_log(LOG_NOTICE, "Still have a QUALIFY dialog active, deleting\n");
		}
		dialog_unlink_all(peer->call);
		peer->call = dialog_unref(peer->call, "unref dialog peer->call");
		/* peer->call = sip_destroy(peer->call); */
	}
	if (!(p = sip_alloc(NULL, NULL, 0, SIP_OPTIONS, NULL))) {
		return -1;
	}
	peer->call = dialog_ref(p, "copy sip alloc from p to peer->call");

	p->sa = peer->addr;
	p->recv = peer->addr;
	copy_socket_data(&p->socket, &peer->socket);
	ast_copy_flags(&p->flags[0], &peer->flags[0], SIP_FLAGS_TO_COPY);
	ast_copy_flags(&p->flags[1], &peer->flags[1], SIP_PAGE2_FLAGS_TO_COPY);
	ast_copy_flags(&p->flags[2], &peer->flags[2], SIP_PAGE3_FLAGS_TO_COPY);

	/* Send OPTIONs to peer's fullcontact */
	if (!ast_strlen_zero(peer->fullcontact)) {
		ast_string_field_set(p, fullcontact, peer->fullcontact);
	}

	if (!ast_strlen_zero(peer->fromuser)) {
		ast_string_field_set(p, fromuser, peer->fromuser);
	}

	if (!ast_strlen_zero(peer->tohost)) {
		ast_string_field_set(p, tohost, peer->tohost);
	} else {
		ast_string_field_set(p, tohost, ast_sockaddr_stringify_host_remote(&peer->addr));
	}

	/* Recalculate our side, and recalculate Call ID */
	ast_sip_ouraddrfor(&p->sa, &p->ourip, p);
	build_via(p);

	/* Change the dialog callid. */
	change_callid_pvt(p, NULL);

	AST_SCHED_DEL_UNREF(sched, peer->pokeexpire,
			unref_peer(peer, "removing poke peer ref"));
	
	if (p->relatedpeer)
		p->relatedpeer = unref_peer(p->relatedpeer,"unsetting the relatedpeer field in the dialog, before it is set to something else.");
	p->relatedpeer = ref_peer(peer, "setting the relatedpeer field in the dialog");
	ast_set_flag(&p->flags[0], SIP_OUTGOING);
#ifdef VOCAL_DATA_HACK
	ast_copy_string(p->username, "__VOCAL_DATA_SHOULD_READ_THE_SIP_SPEC__", sizeof(p->username));
	xmitres = transmit_invite(p, SIP_INVITE, 0, 2, NULL); /* sinks the p refcount */
#else
	xmitres = transmit_invite(p, SIP_OPTIONS, 0, 2, NULL); /* sinks the p refcount */
#endif
	peer->ps = ast_tvnow();
	if (xmitres == XMIT_ERROR) {
		/* Immediately unreachable, network problems */
		sip_poke_noanswer(ref_peer(peer, "add ref for peerexpire (fake, for sip_poke_noanswer to remove)"));
	} else if (!force) {
		AST_SCHED_REPLACE_UNREF(peer->pokeexpire, sched, peer->maxms * 2, sip_poke_noanswer, peer,
				unref_peer(_data, "removing poke peer ref"),
				unref_peer(peer, "removing poke peer ref"),
				ref_peer(peer, "adding poke peer ref"));
	}
	dialog_unref(p, "unref dialog at end of sip_poke_peer, obtained from sip_alloc, just before it goes out of scope");
	return 0;
}

/*! \brief Part of PBX channel interface
\note
\par	Return values:---

	If we have qualify on and the device is not reachable, regardless of registration
	state we return AST_DEVICE_UNAVAILABLE

	For peers with call limit:
		- not registered			AST_DEVICE_UNAVAILABLE
		- registered, no call			AST_DEVICE_NOT_INUSE
		- registered, active calls		AST_DEVICE_INUSE
		- registered, call limit reached	AST_DEVICE_BUSY
		- registered, onhold			AST_DEVICE_ONHOLD
		- registered, ringing			AST_DEVICE_RINGING

	For peers without call limit:
		- not registered			AST_DEVICE_UNAVAILABLE
		- registered				AST_DEVICE_NOT_INUSE
		- fixed IP (!dynamic)			AST_DEVICE_NOT_INUSE
	
	Peers that does not have a known call and can't be reached by OPTIONS
		- unreachable				AST_DEVICE_UNAVAILABLE

	If we return AST_DEVICE_UNKNOWN, the device state engine will try to find
	out a state by walking the channel list.

	The queue system (\ref app_queue.c) treats a member as "active"
	if devicestate is != AST_DEVICE_UNAVAILBALE && != AST_DEVICE_INVALID

	When placing a call to the queue member, queue system sets a member to busy if
	!= AST_DEVICE_NOT_INUSE and != AST_DEVICE_UNKNOWN

*/
static int sip_devicestate(void *data)
{
	char *host;
	char *tmp;
	struct sip_peer *p;

	int res = AST_DEVICE_INVALID;

	/* make sure data is not null. Maybe unnecessary, but better be safe */
	host = ast_strdupa(data ? data : "");
	if ((tmp = strchr(host, '@')))
		host = tmp + 1;

	ast_debug(3, "Checking device state for peer %s\n", host);

	/* If find_peer asks for a realtime peer, then this breaks rtautoclear.  This
	 * is because when a peer tries to autoexpire, the last thing it does is to
	 * queue up an event telling the system that the devicestate has changed
	 * (presumably to unavailable).  If we ask for a realtime peer here, this would
	 * load it BACK into memory, thus defeating the point of trying to clear dead
	 * hosts out of memory.
	 */
	if ((p = find_peer(host, NULL, FALSE, FINDALLDEVICES, TRUE, 0))) {
		if (!(ast_sockaddr_isnull(&p->addr) && ast_sockaddr_isnull(&p->defaddr))) {
			/* we have an address for the peer */
		
			/* Check status in this order
				- Hold
				- Ringing
				- Busy (enforced only by call limit)
				- Inuse (we have a call)
				- Unreachable (qualify)
			   If we don't find any of these state, report AST_DEVICE_NOT_INUSE
			   for registered devices */

			if (p->onHold)
				/* First check for hold or ring states */
				res = AST_DEVICE_ONHOLD;
			else if (p->inRinging) {
				if (p->inRinging == p->inUse)
					res = AST_DEVICE_RINGING;
				else
					res = AST_DEVICE_RINGINUSE;
			} else if (p->call_limit && (p->inUse == p->call_limit))
				/* check call limit */
				res = AST_DEVICE_BUSY;
			else if (p->call_limit && p->busy_level && p->inUse >= p->busy_level)
				/* We're forcing busy before we've reached the call limit */
				res = AST_DEVICE_BUSY;
			else if (p->call_limit && p->inUse)
				/* Not busy, but we do have a call */
				res = AST_DEVICE_INUSE;
			else if (p->maxms && ((p->lastms > p->maxms) || (p->lastms < 0)))
				/* We don't have a call. Are we reachable at all? Requires qualify= */
				res = AST_DEVICE_UNAVAILABLE;
			else	/* Default reply if we're registered and have no other data */
				res = AST_DEVICE_NOT_INUSE;
		} else {
			/* there is no address, it's unavailable */
			res = AST_DEVICE_UNAVAILABLE;
		}
		unref_peer(p, "unref_peer, from sip_devicestate, release ref from find_peer");
	}

	return res;
}

/*! \brief PBX interface function -build SIP pvt structure
 *	SIP calls initiated by the PBX arrive here.
 *
 * \verbatim	
 * 	SIP Dial string syntax
 *		SIP/exten@host!dnid
 *	or	SIP/host/exten!dnid
 *	or	SIP/host!dnid
 * \endverbatim
*/
static struct ast_channel *sip_request_call(const char *type, format_t format, const struct ast_channel *requestor, void *data, int *cause)
{
	struct sip_pvt *p;
	struct ast_channel *tmpc = NULL;
	char *ext = NULL, *host;
	char tmp[256];
	char *dest = data;
	char *dnid;
	char *secret = NULL;
	char *md5secret = NULL;
	char *authname = NULL;
	char *trans = NULL;
	char dialstring[256];
	char *remote_address;
	enum sip_transport transport = 0;
	struct ast_sockaddr remote_address_sa = { {0,} };
	format_t oldformat = format;
	AST_DECLARE_APP_ARGS(args,
		AST_APP_ARG(peerorhost);
		AST_APP_ARG(exten);
		AST_APP_ARG(remote_address);
	);

	/* mask request with some set of allowed formats.
	 * XXX this needs to be fixed.
	 * The original code uses AST_FORMAT_AUDIO_MASK, but it is
	 * unclear what to use here. We have global_capabilities, which is
	 * configured from sip.conf, and sip_tech.capabilities, which is
	 * hardwired to all audio formats.
	 */
	format &= AST_FORMAT_AUDIO_MASK;
	if (!format) {
		ast_log(LOG_NOTICE, "Asked to get a channel of unsupported format %s while capability is %s\n", ast_getformatname(oldformat), ast_getformatname(sip_cfg.capability));
		*cause = AST_CAUSE_BEARERCAPABILITY_NOTAVAIL;	/* Can't find codec to connect to host */
		return NULL;
	}
	ast_debug(1, "Asked to create a SIP channel with formats: %s\n", ast_getformatname_multiple(tmp, sizeof(tmp), oldformat));

	if (ast_strlen_zero(dest)) {
		ast_log(LOG_ERROR, "Unable to create channel with empty destination.\n");
		*cause = AST_CAUSE_CHANNEL_UNACCEPTABLE;
		return NULL;
	}

	if (!(p = sip_alloc(NULL, NULL, 0, SIP_INVITE, NULL))) {
		ast_log(LOG_ERROR, "Unable to build sip pvt data for '%s' (Out of memory or socket error)\n", dest);
		*cause = AST_CAUSE_SWITCH_CONGESTION;
		return NULL;
	}

	p->outgoing_call = TRUE;

	snprintf(dialstring, sizeof(dialstring), "%s/%s", type, dest);
	ast_string_field_set(p, dialstring, dialstring);

	if (!(p->options = ast_calloc(1, sizeof(*p->options)))) {
		dialog_unlink_all(p);
		dialog_unref(p, "unref dialog p from mem fail");
		/* sip_destroy(p); */
		ast_log(LOG_ERROR, "Unable to build option SIP data structure - Out of memory\n");
		*cause = AST_CAUSE_SWITCH_CONGESTION;
		return NULL;
	}

	/* Save the destination, the SIP dial string */
	ast_copy_string(tmp, dest, sizeof(tmp));

	/* Find DNID and take it away */
	dnid = strchr(tmp, '!');
	if (dnid != NULL) {
		*dnid++ = '\0';
		ast_string_field_set(p, todnid, dnid);
	}

	/* Divvy up the items separated by slashes */
	AST_NONSTANDARD_APP_ARGS(args, tmp, '/');

	/* Find at sign - @ */
	host = strchr(args.peerorhost, '@');
	if (host) {
		*host++ = '\0';
		ext = args.peerorhost;
		secret = strchr(ext, ':');
	}
	if (secret) {
		*secret++ = '\0';
		md5secret = strchr(secret, ':');
	}
	if (md5secret) {
		*md5secret++ = '\0';
		authname = strchr(md5secret, ':');
	}
	if (authname) {
		*authname++ = '\0';
		trans = strchr(authname, ':');
	}
	if (trans) {
		*trans++ = '\0';
		if (!strcasecmp(trans, "tcp"))
			transport = SIP_TRANSPORT_TCP;
		else if (!strcasecmp(trans, "tls"))
			transport = SIP_TRANSPORT_TLS;
		else {
			if (strcasecmp(trans, "udp"))
				ast_log(LOG_WARNING, "'%s' is not a valid transport option to Dial() for SIP calls, using udp by default.\n", trans);
			transport = SIP_TRANSPORT_UDP;
		}
	} else { /* use default */
		transport = SIP_TRANSPORT_UDP;
	}

	if (!host) {
		ext = args.exten;
		host = args.peerorhost;
		remote_address = args.remote_address;
	} else {
		remote_address = args.remote_address;
		if (!ast_strlen_zero(args.exten)) {
			ast_log(LOG_NOTICE, "Conflicting extension values given. Using '%s' and not '%s'\n", ext, args.exten);
		}
	}

	if (!ast_strlen_zero(remote_address)) {
		if (ast_sockaddr_resolve_first(&remote_address_sa, remote_address, 0)) {
			ast_log(LOG_WARNING, "Unable to find IP address for host %s. We will not use this remote IP address\n", remote_address);
		} else {
			if (!ast_sockaddr_port(&remote_address_sa)) {
				ast_sockaddr_set_port(&remote_address_sa,
						      transport & SIP_TRANSPORT_TLS ?
						      STANDARD_TLS_PORT :
						      STANDARD_SIP_PORT);
			}
		}
	}

	set_socket_transport(&p->socket, transport);

	/* We now have
		host = peer name, DNS host name or DNS domain (for SRV)
		ext = extension (user part of URI)
		dnid = destination of the call (applies to the To: header)
	*/
	if (create_addr(p, host, NULL, 1, &remote_address_sa)) {
		*cause = AST_CAUSE_UNREGISTERED;
		ast_debug(3, "Cant create SIP call - target device not registered\n");
		dialog_unlink_all(p);
		dialog_unref(p, "unref dialog p UNREGISTERED");
		/* sip_destroy(p); */
		return NULL;
	}
	if (ast_strlen_zero(p->peername) && ext)
		ast_string_field_set(p, peername, ext);
	/* Recalculate our side, and recalculate Call ID */
	ast_sip_ouraddrfor(&p->sa, &p->ourip, p);
	build_via(p);

	/* Change the dialog callid. */
	change_callid_pvt(p, NULL);

	/* We have an extension to call, don't use the full contact here */
	/* This to enable dialing registered peers with extension dialling,
	   like SIP/peername/extension 	
	   SIP/peername will still use the full contact
	 */
	if (ext) {
		ast_string_field_set(p, username, ext);
		ast_string_field_set(p, fullcontact, NULL);
	}
	if (secret && !ast_strlen_zero(secret))
		ast_string_field_set(p, peersecret, secret);

	if (md5secret && !ast_strlen_zero(md5secret))
		ast_string_field_set(p, peermd5secret, md5secret);

	if (authname && !ast_strlen_zero(authname))
		ast_string_field_set(p, authname, authname);
#if 0
	printf("Setting up to call extension '%s' at '%s'\n", ext ? ext : "<none>", host);
#endif
	p->prefcodec = oldformat;				/* Format for this call */
	p->jointcapability = oldformat & p->capability;
	sip_pvt_lock(p);
	tmpc = sip_new(p, AST_STATE_DOWN, host, requestor ? requestor->linkedid : NULL);	/* Place the call */
	if (sip_cfg.callevents)
		manager_event(EVENT_FLAG_SYSTEM, "ChannelUpdate",
			"Channel: %s\r\nChanneltype: %s\r\nSIPcallid: %s\r\nSIPfullcontact: %s\r\nPeername: %s\r\n",
			p->owner? p->owner->name : "", "SIP", p->callid, p->fullcontact, p->peername);
	sip_pvt_unlock(p);
	if (!tmpc) {
		dialog_unlink_all(p);
		/* sip_destroy(p); */
	} else {
		ast_channel_unlock(tmpc);
	}
	dialog_unref(p, "toss pvt ptr at end of sip_request_call");
	ast_update_use_count();
	restart_monitor();
	return tmpc;
}

/*! \brief Parse insecure= setting in sip.conf and set flags according to setting */
static void set_insecure_flags (struct ast_flags *flags, const char *value, int lineno)
{
	if (ast_strlen_zero(value))
		return;

	if (!ast_false(value)) {
		char buf[64];
		char *word, *next;

		ast_copy_string(buf, value, sizeof(buf));
		next = buf;
		while ((word = strsep(&next, ","))) {
			if (!strcasecmp(word, "port"))
				ast_set_flag(&flags[0], SIP_INSECURE_PORT);
			else if (!strcasecmp(word, "invite"))
				ast_set_flag(&flags[0], SIP_INSECURE_INVITE);
			else
				ast_log(LOG_WARNING, "Unknown insecure mode '%s' on line %d\n", value, lineno);
		}
	}
}

/*!
  \brief Handle T.38 configuration options common to users and peers
  \returns non-zero if any config options were handled, zero otherwise
*/
static int handle_t38_options(struct ast_flags *flags, struct ast_flags *mask, struct ast_variable *v,
			      int *maxdatagram)
{
	int res = 1;

	if (!strcasecmp(v->name, "t38pt_udptl")) {
		char *buf = ast_strdupa(v->value);
		char *word, *next = buf;

		ast_set_flag(&mask[1], SIP_PAGE2_T38SUPPORT);

		while ((word = strsep(&next, ","))) {
			if (ast_true(word) || !strcasecmp(word, "fec")) {
				ast_clear_flag(&flags[1], SIP_PAGE2_T38SUPPORT);
				ast_set_flag(&flags[1], SIP_PAGE2_T38SUPPORT_UDPTL_FEC);
			} else if (!strcasecmp(word, "redundancy")) {
				ast_clear_flag(&flags[1], SIP_PAGE2_T38SUPPORT);
				ast_set_flag(&flags[1], SIP_PAGE2_T38SUPPORT_UDPTL_REDUNDANCY);
			} else if (!strcasecmp(word, "none")) {
				ast_clear_flag(&flags[1], SIP_PAGE2_T38SUPPORT);
				ast_set_flag(&flags[1], SIP_PAGE2_T38SUPPORT_UDPTL);
			} else if (!strncasecmp(word, "maxdatagram=", 12)) {
				if (sscanf(&word[12], "%30u", maxdatagram) != 1) {
					ast_log(LOG_WARNING, "Invalid maxdatagram '%s' at line %d of %s\n", v->value, v->lineno, config);
					*maxdatagram = global_t38_maxdatagram;
				}
			}
		}
	} else if (!strcasecmp(v->name, "t38pt_usertpsource")) {
		ast_set_flag(&mask[1], SIP_PAGE2_UDPTL_DESTINATION);
		ast_set2_flag(&flags[1], ast_true(v->value), SIP_PAGE2_UDPTL_DESTINATION);
	} else {
		res = 0;
	}

	return res;
}

/*!
  \brief Handle flag-type options common to configuration of devices - peers
  \param flags array of two struct ast_flags
  \param mask array of two struct ast_flags
  \param v linked list of config variables to process
  \returns non-zero if any config options were handled, zero otherwise
*/
static int handle_common_options(struct ast_flags *flags, struct ast_flags *mask, struct ast_variable *v)
{
	int res = 1;

	if (!strcasecmp(v->name, "trustrpid")) {
		ast_set_flag(&mask[0], SIP_TRUSTRPID);
		ast_set2_flag(&flags[0], ast_true(v->value), SIP_TRUSTRPID);
	} else if (!strcasecmp(v->name, "sendrpid")) {
		ast_set_flag(&mask[0], SIP_SENDRPID);
		if (!strcasecmp(v->value, "pai")) {
			ast_set_flag(&flags[0], SIP_SENDRPID_PAI);
		} else if (!strcasecmp(v->value, "rpid")) {
			ast_set_flag(&flags[0], SIP_SENDRPID_RPID);
		} else if (ast_true(v->value)) {
			ast_set_flag(&flags[0], SIP_SENDRPID_RPID);
		}
	} else if (!strcasecmp(v->name, "rpid_update")) {
		ast_set_flag(&mask[1], SIP_PAGE2_RPID_UPDATE);
		ast_set2_flag(&flags[1], ast_true(v->value), SIP_PAGE2_RPID_UPDATE);
	} else if (!strcasecmp(v->name, "rpid_immediate")) {
		ast_set_flag(&mask[1], SIP_PAGE2_RPID_IMMEDIATE);
		ast_set2_flag(&flags[1], ast_true(v->value), SIP_PAGE2_RPID_IMMEDIATE);
	} else if (!strcasecmp(v->name, "g726nonstandard")) {
		ast_set_flag(&mask[0], SIP_G726_NONSTANDARD);
		ast_set2_flag(&flags[0], ast_true(v->value), SIP_G726_NONSTANDARD);
	} else if (!strcasecmp(v->name, "useclientcode")) {
		ast_set_flag(&mask[0], SIP_USECLIENTCODE);
		ast_set2_flag(&flags[0], ast_true(v->value), SIP_USECLIENTCODE);
	} else if (!strcasecmp(v->name, "dtmfmode")) {
		ast_set_flag(&mask[0], SIP_DTMF);
		ast_clear_flag(&flags[0], SIP_DTMF);
		if (!strcasecmp(v->value, "inband"))
			ast_set_flag(&flags[0], SIP_DTMF_INBAND);
		else if (!strcasecmp(v->value, "rfc2833"))
			ast_set_flag(&flags[0], SIP_DTMF_RFC2833);
		else if (!strcasecmp(v->value, "info"))
			ast_set_flag(&flags[0], SIP_DTMF_INFO);
		else if (!strcasecmp(v->value, "shortinfo"))
			ast_set_flag(&flags[0], SIP_DTMF_SHORTINFO);
		else if (!strcasecmp(v->value, "auto"))
			ast_set_flag(&flags[0], SIP_DTMF_AUTO);
		else {
			ast_log(LOG_WARNING, "Unknown dtmf mode '%s' on line %d, using rfc2833\n", v->value, v->lineno);
			ast_set_flag(&flags[0], SIP_DTMF_RFC2833);
		}
	} else if (!strcasecmp(v->name, "nat")) {
		ast_set_flag(&mask[0], SIP_NAT_FORCE_RPORT);
		ast_set_flag(&flags[0], SIP_NAT_FORCE_RPORT); /* Default to "force_rport" */
		if (!strcasecmp(v->value, "no")) {
			ast_clear_flag(&flags[0], SIP_NAT_FORCE_RPORT);
		} else if (!strcasecmp(v->value, "yes")) {
			/* We've already defaulted to force_rport */
			ast_set_flag(&mask[1], SIP_PAGE2_SYMMETRICRTP);
			ast_set_flag(&flags[1], SIP_PAGE2_SYMMETRICRTP);
		} else if (!strcasecmp(v->value, "comedia")) {
			ast_clear_flag(&flags[0], SIP_NAT_FORCE_RPORT);
			ast_set_flag(&mask[1], SIP_PAGE2_SYMMETRICRTP);
			ast_set_flag(&flags[1], SIP_PAGE2_SYMMETRICRTP);
		}
	} else if (!strcasecmp(v->name, "directmedia") || !strcasecmp(v->name, "canreinvite")) {
		ast_set_flag(&mask[0], SIP_REINVITE);
		ast_clear_flag(&flags[0], SIP_REINVITE);
		if (ast_true(v->value)) {
			ast_set_flag(&flags[0], SIP_DIRECT_MEDIA | SIP_DIRECT_MEDIA_NAT);
		} else if (!ast_false(v->value)) {
			char buf[64];
			char *word, *next = buf;

			ast_copy_string(buf, v->value, sizeof(buf));
			while ((word = strsep(&next, ","))) {
				if (!strcasecmp(word, "update")) {
					ast_set_flag(&flags[0], SIP_REINVITE_UPDATE | SIP_DIRECT_MEDIA);
				} else if (!strcasecmp(word, "nonat")) {
					ast_set_flag(&flags[0], SIP_DIRECT_MEDIA);
					ast_clear_flag(&flags[0], SIP_DIRECT_MEDIA_NAT);
				} else {
					ast_log(LOG_WARNING, "Unknown directmedia mode '%s' on line %d\n", v->value, v->lineno);
				}
			}
		}
	} else if (!strcasecmp(v->name, "insecure")) {
		ast_set_flag(&mask[0], SIP_INSECURE);
		ast_clear_flag(&flags[0], SIP_INSECURE);
		set_insecure_flags(&flags[0], v->value, v->lineno);	
	} else if (!strcasecmp(v->name, "progressinband")) {
		ast_set_flag(&mask[0], SIP_PROG_INBAND);
		ast_clear_flag(&flags[0], SIP_PROG_INBAND);
		if (ast_true(v->value))
			ast_set_flag(&flags[0], SIP_PROG_INBAND_YES);
		else if (strcasecmp(v->value, "never"))
			ast_set_flag(&flags[0], SIP_PROG_INBAND_NO);
	} else if (!strcasecmp(v->name, "promiscredir")) {
		ast_set_flag(&mask[0], SIP_PROMISCREDIR);
		ast_set2_flag(&flags[0], ast_true(v->value), SIP_PROMISCREDIR);
	} else if (!strcasecmp(v->name, "videosupport")) {
		if (!strcasecmp(v->value, "always")) {
			ast_set_flag(&mask[1], SIP_PAGE2_VIDEOSUPPORT_ALWAYS);
			ast_set_flag(&flags[1], SIP_PAGE2_VIDEOSUPPORT_ALWAYS);
		} else {
			ast_set_flag(&mask[1], SIP_PAGE2_VIDEOSUPPORT);
			ast_set2_flag(&flags[1], ast_true(v->value), SIP_PAGE2_VIDEOSUPPORT);
		}
	} else if (!strcasecmp(v->name, "textsupport")) {
		ast_set_flag(&mask[1], SIP_PAGE2_TEXTSUPPORT);
		ast_set2_flag(&flags[1], ast_true(v->value), SIP_PAGE2_TEXTSUPPORT);
		res = 1;
	} else if (!strcasecmp(v->name, "allowoverlap")) {
		ast_set_flag(&mask[1], SIP_PAGE2_ALLOWOVERLAP);
		ast_clear_flag(&flags[1], SIP_PAGE2_ALLOWOVERLAP);
		if (ast_true(v->value)) {
			ast_set_flag(&flags[1], SIP_PAGE2_ALLOWOVERLAP_YES);
		} else if (!strcasecmp(v->value, "dtmf")){
			ast_set_flag(&flags[1], SIP_PAGE2_ALLOWOVERLAP_DTMF);
		}
	} else if (!strcasecmp(v->name, "allowsubscribe")) {
		ast_set_flag(&mask[1], SIP_PAGE2_ALLOWSUBSCRIBE);
		ast_set2_flag(&flags[1], ast_true(v->value), SIP_PAGE2_ALLOWSUBSCRIBE);
	} else if (!strcasecmp(v->name, "ignoresdpversion")) {
		ast_set_flag(&mask[1], SIP_PAGE2_IGNORESDPVERSION);
		ast_set2_flag(&flags[1], ast_true(v->value), SIP_PAGE2_IGNORESDPVERSION);
	} else if (!strcasecmp(v->name, "faxdetect")) {
		ast_set_flag(&mask[1], SIP_PAGE2_FAX_DETECT);
		if (ast_true(v->value)) {
			ast_set_flag(&flags[1], SIP_PAGE2_FAX_DETECT_BOTH);
		} else if (ast_false(v->value)) {
			ast_clear_flag(&flags[1], SIP_PAGE2_FAX_DETECT_BOTH);
		} else {
			char *buf = ast_strdupa(v->value);
			char *word, *next = buf;

			while ((word = strsep(&next, ","))) {
				if (!strcasecmp(word, "cng")) {
					ast_set_flag(&flags[1], SIP_PAGE2_FAX_DETECT_CNG);
				} else if (!strcasecmp(word, "t38")) {
					ast_set_flag(&flags[1], SIP_PAGE2_FAX_DETECT_T38);
				} else {
					ast_log(LOG_WARNING, "Unknown faxdetect mode '%s' on line %d.\n", word, v->lineno);
				}
			}
		}
	} else if (!strcasecmp(v->name, "rfc2833compensate")) {
		ast_set_flag(&mask[1], SIP_PAGE2_RFC2833_COMPENSATE);
		ast_set2_flag(&flags[1], ast_true(v->value), SIP_PAGE2_RFC2833_COMPENSATE);
	} else if (!strcasecmp(v->name, "buggymwi")) {
		ast_set_flag(&mask[1], SIP_PAGE2_BUGGY_MWI);
		ast_set2_flag(&flags[1], ast_true(v->value), SIP_PAGE2_BUGGY_MWI);
	} else
		res = 0;

	return res;
}

/*! \brief Add SIP domain to list of domains we are responsible for */
static int add_sip_domain(const char *domain, const enum domain_mode mode, const char *context)
{
	struct domain *d;

	if (ast_strlen_zero(domain)) {
		ast_log(LOG_WARNING, "Zero length domain.\n");
		return 1;
	}

	if (!(d = ast_calloc(1, sizeof(*d))))
		return 0;

	ast_copy_string(d->domain, domain, sizeof(d->domain));

	if (!ast_strlen_zero(context))
		ast_copy_string(d->context, context, sizeof(d->context));

	d->mode = mode;

	AST_LIST_LOCK(&domain_list);
	AST_LIST_INSERT_TAIL(&domain_list, d, list);
	AST_LIST_UNLOCK(&domain_list);

 	if (sipdebug)	
		ast_debug(1, "Added local SIP domain '%s'\n", domain);

	return 1;
}

/*! \brief  check_sip_domain: Check if domain part of uri is local to our server */
static int check_sip_domain(const char *domain, char *context, size_t len)
{
	struct domain *d;
	int result = 0;

	AST_LIST_LOCK(&domain_list);
	AST_LIST_TRAVERSE(&domain_list, d, list) {
		if (strcasecmp(d->domain, domain)) {
			continue;
		}

		if (len && !ast_strlen_zero(d->context))
			ast_copy_string(context, d->context, len);
		
		result = 1;
		break;
	}
	AST_LIST_UNLOCK(&domain_list);

	return result;
}

/*! \brief Clear our domain list (at reload) */
static void clear_sip_domains(void)
{
	struct domain *d;

	AST_LIST_LOCK(&domain_list);
	while ((d = AST_LIST_REMOVE_HEAD(&domain_list, list)))
		ast_free(d);
	AST_LIST_UNLOCK(&domain_list);
}

/*!
 * \internal
 * \brief Realm authentication container destructor.
 *
 * \param obj Container object to destroy.
 *
 * \return Nothing
 */
static void destroy_realm_authentication(void *obj)
{
	struct sip_auth_container *credentials = obj;
	struct sip_auth *auth;

	while ((auth = AST_LIST_REMOVE_HEAD(&credentials->list, node))) {
		ast_free(auth);
	}
}

/*!
 * \internal
 * \brief Add realm authentication to credentials.
 *
 * \param credentials Realm authentication container to create/add authentication credentials.
 * \param configuration Credential configuration value.
 * \param lineno Line number in config file.
 *
 * \return Nothing
 */
static void add_realm_authentication(struct sip_auth_container **credentials, const char *configuration, int lineno)
{
	char *authcopy;
	char *username=NULL, *realm=NULL, *secret=NULL, *md5secret=NULL;
	struct sip_auth *auth;

	if (ast_strlen_zero(configuration)) {
		/* Nothing to add */
		return;
	}

	ast_debug(1, "Auth config ::  %s\n", configuration);

	authcopy = ast_strdupa(configuration);
	username = authcopy;

	/* split user[:secret] and relm */
	realm = strrchr(username, '@');
	if (realm)
		*realm++ = '\0';
	if (ast_strlen_zero(username) || ast_strlen_zero(realm)) {
		ast_log(LOG_WARNING, "Format for authentication entry is user[:secret]@realm at line %d\n", lineno);
		return;
	}

	/* parse username at ':' for secret, or '#" for md5secret */
	if ((secret = strchr(username, ':'))) {
		*secret++ = '\0';
	} else if ((md5secret = strchr(username, '#'))) {
		*md5secret++ = '\0';
	}

	/* Create the continer if needed. */
	if (!*credentials) {
		*credentials = ao2_t_alloc(sizeof(**credentials), destroy_realm_authentication,
			"Create realm auth container.");
		if (!*credentials) {
			/* Failed to create the credentials container. */
			return;
		}
	}

	/* Create the authentication credential entry. */
	auth = ast_calloc(1, sizeof(*auth));
	if (!auth) {
		return;
	}
	ast_copy_string(auth->realm, realm, sizeof(auth->realm));
	ast_copy_string(auth->username, username, sizeof(auth->username));
	if (secret)
		ast_copy_string(auth->secret, secret, sizeof(auth->secret));
	if (md5secret)
		ast_copy_string(auth->md5secret, md5secret, sizeof(auth->md5secret));

	/* Add credential to container list. */
	AST_LIST_INSERT_TAIL(&(*credentials)->list, auth, node);

	ast_verb(3, "Added authentication for realm %s\n", realm);
}

/*!
 * \internal
 * \brief Find authentication for a specific realm.
 *
 * \param credentials Realm authentication container to search.
 * \param realm Authentication realm to find.
 *
 * \return Found authentication credential or NULL.
 */
static struct sip_auth *find_realm_authentication(struct sip_auth_container *credentials, const char *realm)
{
	struct sip_auth *auth;

	if (credentials) {
		AST_LIST_TRAVERSE(&credentials->list, auth, node) {
			if (!strcasecmp(auth->realm, realm)) {
				break;
			}
		}
	} else {
		auth = NULL;
	}

	return auth;
}

/*! \brief
 * implement the setvar config line
 */
static struct ast_variable *add_var(const char *buf, struct ast_variable *list)
{
	struct ast_variable *tmpvar = NULL;
	char *varname = ast_strdupa(buf), *varval = NULL;
	
	if ((varval = strchr(varname, '='))) {
		*varval++ = '\0';
		if ((tmpvar = ast_variable_new(varname, varval, ""))) {
			tmpvar->next = list;
			list = tmpvar;
		}
	}
	return list;
}

/*! \brief Set peer defaults before configuring specific configurations */
static void set_peer_defaults(struct sip_peer *peer)
{
	if (peer->expire == 0) {
		/* Don't reset expire or port time during reload
		   if we have an active registration
		*/
		peer->expire = -1;
		peer->pokeexpire = -1;
		set_socket_transport(&peer->socket, SIP_TRANSPORT_UDP);
	}
	peer->type = SIP_TYPE_PEER;
	ast_copy_flags(&peer->flags[0], &global_flags[0], SIP_FLAGS_TO_COPY);
	ast_copy_flags(&peer->flags[1], &global_flags[1], SIP_PAGE2_FLAGS_TO_COPY);
	ast_copy_flags(&peer->flags[2], &global_flags[2], SIP_PAGE3_FLAGS_TO_COPY);
	ast_string_field_set(peer, context, sip_cfg.default_context);
	ast_string_field_set(peer, subscribecontext, sip_cfg.default_subscribecontext);
	ast_string_field_set(peer, language, default_language);
	ast_string_field_set(peer, mohinterpret, default_mohinterpret);
	ast_string_field_set(peer, mohsuggest, default_mohsuggest);
	ast_string_field_set(peer, engine, default_engine);
	ast_sockaddr_setnull(&peer->addr);
	ast_sockaddr_setnull(&peer->defaddr);
	peer->capability = sip_cfg.capability;
	peer->maxcallbitrate = default_maxcallbitrate;
	peer->rtptimeout = global_rtptimeout;
	peer->rtpholdtimeout = global_rtpholdtimeout;
	peer->rtpkeepalive = global_rtpkeepalive;
	peer->allowtransfer = sip_cfg.allowtransfer;
	peer->autoframing = global_autoframing;
	peer->t38_maxdatagram = global_t38_maxdatagram;
	peer->qualifyfreq = global_qualifyfreq;
	if (global_callcounter)
		peer->call_limit=INT_MAX;
	ast_string_field_set(peer, vmexten, default_vmexten);
	ast_string_field_set(peer, secret, "");
	ast_string_field_set(peer, remotesecret, "");
	ast_string_field_set(peer, md5secret, "");
	ast_string_field_set(peer, cid_num, "");
	ast_string_field_set(peer, cid_name, "");
	ast_string_field_set(peer, cid_tag, "");
	ast_string_field_set(peer, fromdomain, "");
	ast_string_field_set(peer, fromuser, "");
	ast_string_field_set(peer, regexten, "");
	peer->callgroup = 0;
	peer->pickupgroup = 0;
	peer->maxms = default_qualify;
	peer->prefs = default_prefs;
	peer->stimer.st_mode_oper = global_st_mode;	/* Session-Timers */
	peer->stimer.st_ref = global_st_refresher;
	peer->stimer.st_min_se = global_min_se;
	peer->stimer.st_max_se = global_max_se;
	peer->timer_t1 = global_t1;
	peer->timer_b = global_timer_b;
	clear_peer_mailboxes(peer);
	peer->disallowed_methods = sip_cfg.disallowed_methods;
	peer->transports = default_transports;
	peer->default_outbound_transport = default_primary_transport;
}

/*! \brief Create temporary peer (used in autocreatepeer mode) */
static struct sip_peer *temp_peer(const char *name)
{
	struct sip_peer *peer;

	if (!(peer = ao2_t_alloc(sizeof(*peer), sip_destroy_peer_fn, "allocate a peer struct")))
		return NULL;

	if (ast_string_field_init(peer, 512)) {
		ao2_t_ref(peer, -1, "failed to string_field_init, drop peer");
		return NULL;
	}
	
	if (!(peer->cc_params = ast_cc_config_params_init())) {
		ao2_t_ref(peer, -1, "failed to allocate cc_params for peer");
		return NULL;
	}

	ast_atomic_fetchadd_int(&apeerobjs, 1);
	set_peer_defaults(peer);

	ast_copy_string(peer->name, name, sizeof(peer->name));

	peer->selfdestruct = TRUE;
	peer->host_dynamic = TRUE;
	peer->prefs = default_prefs;
	reg_source_db(peer);

	return peer;
}

/*! \todo document this function */
static void add_peer_mailboxes(struct sip_peer *peer, const char *value)
{
	char *next, *mbox, *context;

	next = ast_strdupa(value);

	while ((mbox = context = strsep(&next, ","))) {
		struct sip_mailbox *mailbox;
		int duplicate = 0;
		/* remove leading/trailing whitespace from mailbox string */
		mbox = ast_strip(mbox);
		strsep(&context, "@");

		if (ast_strlen_zero(mbox)) {
			continue;
		}

		/* Check whether the mailbox is already in the list */
		AST_LIST_TRAVERSE(&peer->mailboxes, mailbox, entry) {
			if (!strcmp(mailbox->mailbox, mbox) && !strcmp(S_OR(mailbox->context, ""), S_OR(context, ""))) {
				duplicate = 1;
				break;
			}
		}
		if (duplicate) {
			continue;
		}

		if (!(mailbox = ast_calloc(1, sizeof(*mailbox) + strlen(mbox) + strlen(S_OR(context, ""))))) {
			continue;
		}

		if (!ast_strlen_zero(context)) {
			mailbox->context = mailbox->mailbox + strlen(mbox) + 1;
			strcpy(mailbox->context, context); /* SAFE */
		}
		strcpy(mailbox->mailbox, mbox); /* SAFE */

		AST_LIST_INSERT_TAIL(&peer->mailboxes, mailbox, entry);
	}
}

/*! \brief Build peer from configuration (file or realtime static/dynamic) */
static struct sip_peer *build_peer(const char *name, struct ast_variable *v, struct ast_variable *alt, int realtime, int devstate_only)
{
	struct sip_peer *peer = NULL;
	struct ast_ha *oldha = NULL;
	struct ast_ha *olddirectmediaha = NULL;
	int found = 0;
	int firstpass = 1;
	uint16_t port = 0;
	int format = 0;		/* Ama flags */
	int timerb_set = 0, timert1_set = 0;
	time_t regseconds = 0;
	struct ast_flags peerflags[3] = {{(0)}};
	struct ast_flags mask[3] = {{(0)}};
	char callback[256] = "";
	struct sip_peer tmp_peer;
	const char *srvlookup = NULL;
	static int deprecation_warning = 1;
	int alt_fullcontact = alt ? 1 : 0, headercount = 0;
	struct ast_str *fullcontact = ast_str_alloca(512);

	if (!realtime || ast_test_flag(&global_flags[1], SIP_PAGE2_RTCACHEFRIENDS)) {
		/* Note we do NOT use find_peer here, to avoid realtime recursion */
		/* We also use a case-sensitive comparison (unlike find_peer) so
		   that case changes made to the peer name will be properly handled
		   during reload
		*/
		ast_copy_string(tmp_peer.name, name, sizeof(tmp_peer.name));
		peer = ao2_t_find(peers, &tmp_peer, OBJ_POINTER | OBJ_UNLINK, "find and unlink peer from peers table");
	}

	if (peer) {
		/* Already in the list, remove it and it will be added back (or FREE'd)  */
		found++;
		/* we've unlinked the peer from the peers container but not unlinked from the peers_by_ip container yet
		  this leads to a wrong refcounter and the peer object is never destroyed */
		if (!ast_sockaddr_isnull(&peer->addr)) {
			ao2_t_unlink(peers_by_ip, peer, "ao2_unlink peer from peers_by_ip table");
		}
		if (!(peer->the_mark))
			firstpass = 0;
	} else {
		if (!(peer = ao2_t_alloc(sizeof(*peer), sip_destroy_peer_fn, "allocate a peer struct")))
			return NULL;

		if (ast_string_field_init(peer, 512)) {
			ao2_t_ref(peer, -1, "failed to string_field_init, drop peer");
			return NULL;
		}

		if (!(peer->cc_params = ast_cc_config_params_init())) {
			ao2_t_ref(peer, -1, "failed to allocate cc_params for peer");
			return NULL;
		}

		if (realtime && !ast_test_flag(&global_flags[1], SIP_PAGE2_RTCACHEFRIENDS)) {
			ast_atomic_fetchadd_int(&rpeerobjs, 1);
			ast_debug(3, "-REALTIME- peer built. Name: %s. Peer objects: %d\n", name, rpeerobjs);
		} else
			ast_atomic_fetchadd_int(&speerobjs, 1);
	}

	/* Note that our peer HAS had its reference count increased */
	if (firstpass) {
		peer->lastmsgssent = -1;
		oldha = peer->ha;
		peer->ha = NULL;
		olddirectmediaha = peer->directmediaha;
		peer->directmediaha = NULL;
		set_peer_defaults(peer);	/* Set peer defaults */
		peer->type = 0;
	}

	/* in case the case of the peer name has changed, update the name */
	ast_copy_string(peer->name, name, sizeof(peer->name));

	/* If we have channel variables, remove them (reload) */
	if (peer->chanvars) {
		ast_variables_destroy(peer->chanvars);
		peer->chanvars = NULL;
		/* XXX should unregister ? */
	}

	if (found)
		peer->portinuri = 0;

	/* If we have realm authentication information, remove them (reload) */
	ao2_lock(peer);
	if (peer->auth) {
		ao2_t_ref(peer->auth, -1, "Removing old peer authentication");
		peer->auth = NULL;
	}
	ao2_unlock(peer);

	/* clear the transport information.  We will detect if a default value is required after parsing the config */
	peer->default_outbound_transport = 0;
	peer->transports = 0;

	if (!devstate_only) {
		struct sip_mailbox *mailbox;
		AST_LIST_TRAVERSE(&peer->mailboxes, mailbox, entry) {
			mailbox->delme = 1;
		}
	}

	for (; v || ((v = alt) && !(alt=NULL)); v = v->next) {
		if (!devstate_only) {
			if (handle_common_options(&peerflags[0], &mask[0], v)) {
				continue;
			}
			if (handle_t38_options(&peerflags[0], &mask[0], v, &peer->t38_maxdatagram)) {
				continue;
			}
			if (!strcasecmp(v->name, "transport")) {
				char *val = ast_strdupa(v->value);
				char *trans;

				peer->transports = peer->default_outbound_transport = 0;
				while ((trans = strsep(&val, ","))) {
					trans = ast_skip_blanks(trans);

					if (!strncasecmp(trans, "udp", 3)) {
						peer->transports |= SIP_TRANSPORT_UDP;
					} else if (sip_cfg.tcp_enabled && !strncasecmp(trans, "tcp", 3)) {
						peer->transports |= SIP_TRANSPORT_TCP;
					} else if (default_tls_cfg.enabled && !strncasecmp(trans, "tls", 3)) {
						peer->transports |= SIP_TRANSPORT_TLS;
					} else if (!strncasecmp(trans, "tcp", 3) || !strncasecmp(trans, "tls", 3)) {
						ast_log(LOG_WARNING, "'%.3s' is not a valid transport type when %.3senable=no. If no other is specified, the defaults from general will be used.\n", trans, trans);
					} else {
						ast_log(LOG_NOTICE, "'%s' is not a valid transport type. if no other is specified, the defaults from general will be used.\n", trans);
					}

					if (!peer->default_outbound_transport) { /*!< The first transport listed should be default outbound */
						peer->default_outbound_transport = peer->transports;
					}
				}
			} else if (realtime && !strcasecmp(v->name, "regseconds")) {
				ast_get_time_t(v->value, &regseconds, 0, NULL);
			} else if (realtime && !strcasecmp(v->name, "name")) {
				ast_copy_string(peer->name, v->value, sizeof(peer->name));
			} else if (realtime && !strcasecmp(v->name, "useragent")) {
				ast_string_field_set(peer, useragent, v->value);
			} else if (!strcasecmp(v->name, "type")) {
				if (!strcasecmp(v->value, "peer")) {
					peer->type |= SIP_TYPE_PEER;
				} else if (!strcasecmp(v->value, "user")) {
					peer->type |= SIP_TYPE_USER;
				} else if (!strcasecmp(v->value, "friend")) {
					peer->type = SIP_TYPE_USER | SIP_TYPE_PEER;
				}
			} else if (!strcasecmp(v->name, "remotesecret")) {
				ast_string_field_set(peer, remotesecret, v->value);
			} else if (!strcasecmp(v->name, "secret")) {
				ast_string_field_set(peer, secret, v->value);
			} else if (!strcasecmp(v->name, "md5secret")) {
				ast_string_field_set(peer, md5secret, v->value);
			} else if (!strcasecmp(v->name, "auth")) {
				add_realm_authentication(&peer->auth, v->value, v->lineno);
			} else if (!strcasecmp(v->name, "callerid")) {
				char cid_name[80] = { '\0' }, cid_num[80] = { '\0' };

				ast_callerid_split(v->value, cid_name, sizeof(cid_name), cid_num, sizeof(cid_num));
				ast_string_field_set(peer, cid_name, cid_name);
				ast_string_field_set(peer, cid_num, cid_num);
			} else if (!strcasecmp(v->name, "mwi_from")) {
				ast_string_field_set(peer, mwi_from, v->value);
			} else if (!strcasecmp(v->name, "fullname")) {
				ast_string_field_set(peer, cid_name, v->value);
			} else if (!strcasecmp(v->name, "trunkname")) {
				/* This is actually for a trunk, so we don't want to override callerid */
				ast_string_field_set(peer, cid_name, "");
			} else if (!strcasecmp(v->name, "cid_number")) {
				ast_string_field_set(peer, cid_num, v->value);
			} else if (!strcasecmp(v->name, "cid_tag")) {
				ast_string_field_set(peer, cid_tag, v->value);
			} else if (!strcasecmp(v->name, "context")) {
				ast_string_field_set(peer, context, v->value);
				ast_set_flag(&peer->flags[1], SIP_PAGE2_HAVEPEERCONTEXT);
			} else if (!strcasecmp(v->name, "subscribecontext")) {
				ast_string_field_set(peer, subscribecontext, v->value);
			} else if (!strcasecmp(v->name, "fromdomain")) {
				char *fromdomainport;
				ast_string_field_set(peer, fromdomain, v->value);
				if ((fromdomainport = strchr(peer->fromdomain, ':'))) {
					*fromdomainport++ = '\0';
					if (!(peer->fromdomainport = port_str2int(fromdomainport, 0))) {
						ast_log(LOG_NOTICE, "'%s' is not a valid port number for fromdomain.\n",fromdomainport);
					}
				} else {
					peer->fromdomainport = STANDARD_SIP_PORT;
				}
			} else if (!strcasecmp(v->name, "usereqphone")) {
				ast_set2_flag(&peer->flags[0], ast_true(v->value), SIP_USEREQPHONE);
			} else if (!strcasecmp(v->name, "fromuser")) {
				ast_string_field_set(peer, fromuser, v->value);
			} else if (!strcasecmp(v->name, "outboundproxy")) {
				char *tok, *proxyname;

				if (ast_strlen_zero(v->value)) {
					ast_log(LOG_WARNING, "no value given for outbound proxy on line %d of sip.conf.", v->lineno);
					continue;
				}

				peer->outboundproxy =
				    ao2_alloc(sizeof(*peer->outboundproxy), NULL);

				tok = ast_skip_blanks(strtok(ast_strdupa(v->value), ","));

				sip_parse_host(tok, v->lineno, &proxyname,
					       &peer->outboundproxy->port,
					       &peer->outboundproxy->transport);

				tok = ast_skip_blanks(strtok(ast_strdupa(v->value), ","));

				if ((tok = strtok(NULL, ","))) {
					peer->outboundproxy->force = !strncasecmp(ast_skip_blanks(tok), "force", 5);
				} else {
					peer->outboundproxy->force = FALSE;
				}

				if (ast_strlen_zero(proxyname)) {
					ast_log(LOG_WARNING, "you must specify a name for the outboundproxy on line %d of sip.conf.", v->lineno);
					sip_cfg.outboundproxy.name[0] = '\0';
					continue;
				}

				ast_copy_string(peer->outboundproxy->name, proxyname, sizeof(peer->outboundproxy->name));

				proxy_update(peer->outboundproxy);
			} else if (!strcasecmp(v->name, "host")) {
				if (!strcasecmp(v->value, "dynamic")) {
					/* They'll register with us */
					if (!found || !peer->host_dynamic) {
						/* Initialize stuff if this is a new peer, or if it used to
						 * not be dynamic before the reload. */
						ast_sockaddr_setnull(&peer->addr);
					}
					peer->host_dynamic = TRUE;
				} else {
					/* Non-dynamic.  Make sure we become that way if we're not */
					AST_SCHED_DEL_UNREF(sched, peer->expire,
							unref_peer(peer, "removing register expire ref"));
					peer->host_dynamic = FALSE;
					srvlookup = v->value;
				}
			} else if (!strcasecmp(v->name, "defaultip")) {
				if (!ast_strlen_zero(v->value) && ast_get_ip(&peer->defaddr, v->value)) {
					unref_peer(peer, "unref_peer: from build_peer defaultip");
					return NULL;
				}
			} else if (!strcasecmp(v->name, "permit") || !strcasecmp(v->name, "deny")) {
				int ha_error = 0;
				if (!ast_strlen_zero(v->value)) {
					peer->ha = ast_append_ha(v->name, v->value, peer->ha, &ha_error);
				}
				if (ha_error) {
					ast_log(LOG_ERROR, "Bad ACL entry in configuration line %d : %s\n", v->lineno, v->value);
				}
			} else if (!strcasecmp(v->name, "contactpermit") || !strcasecmp(v->name, "contactdeny")) {
				int ha_error = 0;
				if (!ast_strlen_zero(v->value)) {
					peer->contactha = ast_append_ha(v->name + 7, v->value, peer->contactha, &ha_error);
				}
				if (ha_error) {
					ast_log(LOG_ERROR, "Bad ACL entry in configuration line %d : %s\n", v->lineno, v->value);
				}
			} else if (!strcasecmp(v->name, "directmediapermit") || !strcasecmp(v->name, "directmediadeny")) {
				int ha_error = 0;
				peer->directmediaha = ast_append_ha(v->name + 11, v->value, peer->directmediaha, &ha_error);
				if (ha_error) {
					ast_log(LOG_ERROR, "Bad directmedia ACL entry in configuration line %d : %s\n", v->lineno, v->value);
				}
			} else if (!strcasecmp(v->name, "port")) {
				peer->portinuri = 1;
				if (!(port = port_str2int(v->value, 0))) {
					if (realtime) {
						/* If stored as integer, could be 0 for some DBs (notably MySQL) */
						peer->portinuri = 0;
					} else {
						ast_log(LOG_WARNING, "Invalid peer port configuration at line %d : %s\n", v->lineno, v->value);
					}
				}
			} else if (!strcasecmp(v->name, "callingpres")) {
				peer->callingpres = ast_parse_caller_presentation(v->value);
				if (peer->callingpres == -1) {
					peer->callingpres = atoi(v->value);
				}
			} else if (!strcasecmp(v->name, "username") || !strcmp(v->name, "defaultuser")) {	/* "username" is deprecated */
				ast_string_field_set(peer, username, v->value);
				if (!strcasecmp(v->name, "username")) {
					if (deprecation_warning) {
						ast_log(LOG_NOTICE, "The 'username' field for sip peers has been deprecated in favor of the term 'defaultuser'\n");
						deprecation_warning = 0;
					}
					peer->deprecated_username = 1;
				}
			} else if (!strcasecmp(v->name, "language")) {
				ast_string_field_set(peer, language, v->value);
			} else if (!strcasecmp(v->name, "regexten")) {
				ast_string_field_set(peer, regexten, v->value);
			} else if (!strcasecmp(v->name, "callbackextension")) {
				ast_copy_string(callback, v->value, sizeof(callback));
			} else if (!strcasecmp(v->name, "amaflags")) {
				format = ast_cdr_amaflags2int(v->value);
				if (format < 0) {
					ast_log(LOG_WARNING, "Invalid AMA Flags for peer: %s at line %d\n", v->value, v->lineno);
				} else {
					peer->amaflags = format;
				}
			} else if (!strcasecmp(v->name, "maxforwards")) {
				if (sscanf(v->value, "%30d", &peer->maxforwards) != 1
					|| peer->maxforwards < 1 || 255 < peer->maxforwards) {
					ast_log(LOG_WARNING, "'%s' is not a valid maxforwards value at line %d.  Using default.\n", v->value, v->lineno);
					peer->maxforwards = sip_cfg.default_max_forwards;
				}
			} else if (!strcasecmp(v->name, "accountcode")) {
				ast_string_field_set(peer, accountcode, v->value);
			} else if (!strcasecmp(v->name, "mohinterpret")) {
				ast_string_field_set(peer, mohinterpret, v->value);
			} else if (!strcasecmp(v->name, "mohsuggest")) {
				ast_string_field_set(peer, mohsuggest, v->value);
			} else if (!strcasecmp(v->name, "parkinglot")) {
				ast_string_field_set(peer, parkinglot, v->value);
			} else if (!strcasecmp(v->name, "rtp_engine")) {
				ast_string_field_set(peer, engine, v->value);
			} else if (!strcasecmp(v->name, "mailbox")) {
				add_peer_mailboxes(peer, v->value);
			} else if (!strcasecmp(v->name, "hasvoicemail")) {
				/* People expect that if 'hasvoicemail' is set, that the mailbox will
				 * be also set, even if not explicitly specified. */
				if (ast_true(v->value) && AST_LIST_EMPTY(&peer->mailboxes)) {
					add_peer_mailboxes(peer, name);
				}
			} else if (!strcasecmp(v->name, "subscribemwi")) {
				ast_set2_flag(&peer->flags[1], ast_true(v->value), SIP_PAGE2_SUBSCRIBEMWIONLY);
			} else if (!strcasecmp(v->name, "vmexten")) {
				ast_string_field_set(peer, vmexten, v->value);
			} else if (!strcasecmp(v->name, "callgroup")) {
				peer->callgroup = ast_get_group(v->value);
			} else if (!strcasecmp(v->name, "allowtransfer")) {
				peer->allowtransfer = ast_true(v->value) ? TRANSFER_OPENFORALL : TRANSFER_CLOSED;
			} else if (!strcasecmp(v->name, "pickupgroup")) {
				peer->pickupgroup = ast_get_group(v->value);
			} else if (!strcasecmp(v->name, "allow")) {
				int error =  ast_parse_allow_disallow(&peer->prefs, &peer->capability, v->value, TRUE);
				if (error) {
					ast_log(LOG_WARNING, "Codec configuration errors found in line %d : %s = %s\n", v->lineno, v->name, v->value);
				}
			} else if (!strcasecmp(v->name, "disallow")) {
				int error =  ast_parse_allow_disallow(&peer->prefs, &peer->capability, v->value, FALSE);
				if (error) {
					ast_log(LOG_WARNING, "Codec configuration errors found in line %d : %s = %s\n", v->lineno, v->name, v->value);
				}
			} else if (!strcasecmp(v->name, "preferred_codec_only")) {
				ast_set2_flag(&peer->flags[1], ast_true(v->value), SIP_PAGE2_PREFERRED_CODEC);
			} else if (!strcasecmp(v->name, "autoframing")) {
				peer->autoframing = ast_true(v->value);
			} else if (!strcasecmp(v->name, "rtptimeout")) {
				if ((sscanf(v->value, "%30d", &peer->rtptimeout) != 1) || (peer->rtptimeout < 0)) {
					ast_log(LOG_WARNING, "'%s' is not a valid RTP hold time at line %d.  Using default.\n", v->value, v->lineno);
					peer->rtptimeout = global_rtptimeout;
				}
			} else if (!strcasecmp(v->name, "rtpholdtimeout")) {
				if ((sscanf(v->value, "%30d", &peer->rtpholdtimeout) != 1) || (peer->rtpholdtimeout < 0)) {
					ast_log(LOG_WARNING, "'%s' is not a valid RTP hold time at line %d.  Using default.\n", v->value, v->lineno);
					peer->rtpholdtimeout = global_rtpholdtimeout;
				}
			} else if (!strcasecmp(v->name, "rtpkeepalive")) {
				if ((sscanf(v->value, "%30d", &peer->rtpkeepalive) != 1) || (peer->rtpkeepalive < 0)) {
					ast_log(LOG_WARNING, "'%s' is not a valid RTP keepalive time at line %d.  Using default.\n", v->value, v->lineno);
					peer->rtpkeepalive = global_rtpkeepalive;
				}
			} else if (!strcasecmp(v->name, "timert1")) {
				if ((sscanf(v->value, "%30d", &peer->timer_t1) != 1) || (peer->timer_t1 < 200) || (peer->timer_t1 < global_t1min)) {
					ast_log(LOG_WARNING, "'%s' is not a valid T1 time at line %d.  Using default.\n", v->value, v->lineno);
					peer->timer_t1 = global_t1min;
				}
				timert1_set = 1;
			} else if (!strcasecmp(v->name, "timerb")) {
				if ((sscanf(v->value, "%30d", &peer->timer_b) != 1) || (peer->timer_b < 200)) {
					ast_log(LOG_WARNING, "'%s' is not a valid Timer B time at line %d.  Using default.\n", v->value, v->lineno);
					peer->timer_b = global_timer_b;
				}
				timerb_set = 1;
			} else if (!strcasecmp(v->name, "setvar")) {
				peer->chanvars = add_var(v->value, peer->chanvars);
			} else if (!strcasecmp(v->name, "header")) {
				char tmp[4096];
				snprintf(tmp, sizeof(tmp), "__SIPADDHEADERpre%2d=%s", ++headercount, v->value);
				peer->chanvars = add_var(tmp, peer->chanvars);
			} else if (!strcasecmp(v->name, "qualifyfreq")) {
				int i;
				if (sscanf(v->value, "%30d", &i) == 1) {
					peer->qualifyfreq = i * 1000;
				} else {
					ast_log(LOG_WARNING, "Invalid qualifyfreq number '%s' at line %d of %s\n", v->value, v->lineno, config);
					peer->qualifyfreq = global_qualifyfreq;
				}
			} else if (!strcasecmp(v->name, "maxcallbitrate")) {
				peer->maxcallbitrate = atoi(v->value);
				if (peer->maxcallbitrate < 0) {
					peer->maxcallbitrate = default_maxcallbitrate;
				}
			} else if (!strcasecmp(v->name, "session-timers")) {
				int i = (int) str2stmode(v->value);
				if (i < 0) {
					ast_log(LOG_WARNING, "Invalid session-timers '%s' at line %d of %s\n", v->value, v->lineno, config);
					peer->stimer.st_mode_oper = global_st_mode;
				} else {
					peer->stimer.st_mode_oper = i;
				}
			} else if (!strcasecmp(v->name, "session-expires")) {
				if (sscanf(v->value, "%30d", &peer->stimer.st_max_se) != 1) {
					ast_log(LOG_WARNING, "Invalid session-expires '%s' at line %d of %s\n", v->value, v->lineno, config);
					peer->stimer.st_max_se = global_max_se;
				}
			} else if (!strcasecmp(v->name, "session-minse")) {
				if (sscanf(v->value, "%30d", &peer->stimer.st_min_se) != 1) {
					ast_log(LOG_WARNING, "Invalid session-minse '%s' at line %d of %s\n", v->value, v->lineno, config);
					peer->stimer.st_min_se = global_min_se;
				}
				if (peer->stimer.st_min_se < 90) {
					ast_log(LOG_WARNING, "session-minse '%s' at line %d of %s is not allowed to be < 90 secs\n", v->value, v->lineno, config);
					peer->stimer.st_min_se = global_min_se;
				}
			} else if (!strcasecmp(v->name, "session-refresher")) {
				int i = (int) str2strefresher(v->value);
				if (i < 0) {
					ast_log(LOG_WARNING, "Invalid session-refresher '%s' at line %d of %s\n", v->value, v->lineno, config);
					peer->stimer.st_ref = global_st_refresher;
				} else {
					peer->stimer.st_ref = i;
				}
			} else if (!strcasecmp(v->name, "disallowed_methods")) {
				char *disallow = ast_strdupa(v->value);
				mark_parsed_methods(&peer->disallowed_methods, disallow);
			} else if (!strcasecmp(v->name, "unsolicited_mailbox")) {
				ast_string_field_set(peer, unsolicited_mailbox, v->value);
			} else if (!strcasecmp(v->name, "use_q850_reason")) {
				ast_set2_flag(&peer->flags[1], ast_true(v->value), SIP_PAGE2_Q850_REASON);
			} else if (!strcasecmp(v->name, "encryption")) {
				ast_set2_flag(&peer->flags[1], ast_true(v->value), SIP_PAGE2_USE_SRTP);
			} else if (!strcasecmp(v->name, "snom_aoc_enabled")) {
				ast_set2_flag(&peer->flags[2], ast_true(v->value), SIP_PAGE3_SNOM_AOC);
			}
		}

		/* These apply to devstate lookups */
		if (realtime && !strcasecmp(v->name, "lastms")) {
			sscanf(v->value, "%30d", &peer->lastms);
		} else if (realtime && !strcasecmp(v->name, "ipaddr") && !ast_strlen_zero(v->value) ) {
			ast_sockaddr_parse(&peer->addr, v->value, PARSE_PORT_FORBID);
		} else if (realtime && !strcasecmp(v->name, "fullcontact")) {
			if (alt_fullcontact && !alt) {
				/* Reset, because the alternate also has a fullcontact and we
				 * do NOT want the field value to be doubled. It might be
				 * tempting to skip this, but the first table might not have
				 * fullcontact and since we're here, we know that the alternate
				 * absolutely does. */
				alt_fullcontact = 0;
				ast_str_reset(fullcontact);
			}
			/* Reconstruct field, because realtime separates our value at the ';' */
			if (fullcontact->used > 0) {
				ast_str_append(&fullcontact, 0, ";%s", v->value);
			} else {
				ast_str_set(&fullcontact, 0, "%s", v->value);
			}
		} else if (!strcasecmp(v->name, "qualify")) {
			if (!strcasecmp(v->value, "no")) {
				peer->maxms = 0;
			} else if (!strcasecmp(v->value, "yes")) {
				peer->maxms = default_qualify ? default_qualify : DEFAULT_MAXMS;
			} else if (sscanf(v->value, "%30d", &peer->maxms) != 1) {
				ast_log(LOG_WARNING, "Qualification of peer '%s' should be 'yes', 'no', or a number of milliseconds at line %d of sip.conf\n", peer->name, v->lineno);
				peer->maxms = 0;
			}
			if (realtime && !ast_test_flag(&global_flags[1], SIP_PAGE2_RTCACHEFRIENDS) && peer->maxms > 0) {
				/* This would otherwise cause a network storm, where the
				 * qualify response refreshes the peer from the database,
				 * which in turn causes another qualify to be sent, ad
				 * infinitum. */
				ast_log(LOG_WARNING, "Qualify is incompatible with dynamic uncached realtime.  Please either turn rtcachefriends on or turn qualify off on peer '%s'\n", peer->name);
				peer->maxms = 0;
			}
		} else if (!strcasecmp(v->name, "callcounter")) {
			peer->call_limit = ast_true(v->value) ? INT_MAX : 0;
		} else if (!strcasecmp(v->name, "call-limit")) {
			peer->call_limit = atoi(v->value);
			if (peer->call_limit < 0) {
				peer->call_limit = 0;
			}
		} else if (!strcasecmp(v->name, "busylevel")) {
			peer->busy_level = atoi(v->value);
			if (peer->busy_level < 0) {
				peer->busy_level = 0;
			}
		} else if (ast_cc_is_config_param(v->name)) {
			ast_cc_set_param(peer->cc_params, v->name, v->value);
		}
	}

	if (!devstate_only) {
		struct sip_mailbox *mailbox;
		AST_LIST_TRAVERSE_SAFE_BEGIN(&peer->mailboxes, mailbox, entry) {
			if (mailbox->delme) {
				AST_LIST_REMOVE_CURRENT(entry);
				destroy_mailbox(mailbox);
			}
		}
		AST_LIST_TRAVERSE_SAFE_END;
	}

	if (!can_parse_xml && (ast_get_cc_agent_policy(peer->cc_params) == AST_CC_AGENT_NATIVE)) {
		ast_log(LOG_WARNING, "Peer %s has a cc_agent_policy of 'native' but required libxml2 dependency is not installed. Changing policy to 'never'\n", peer->name);
		ast_set_cc_agent_policy(peer->cc_params, AST_CC_AGENT_NEVER);
	}

	/* Note that Timer B is dependent upon T1 and MUST NOT be lower
	 * than T1 * 64, according to RFC 3261, Section 17.1.1.2 */
	if (peer->timer_b < peer->timer_t1 * 64) {
		if (timerb_set && timert1_set) {
			ast_log(LOG_WARNING, "Timer B has been set lower than recommended for peer %s (%d < 64 * Timer-T1=%d)\n", peer->name, peer->timer_b, peer->timer_t1);
		} else if (timerb_set) {
			if ((peer->timer_t1 = peer->timer_b / 64) < global_t1min) {
				ast_log(LOG_WARNING, "Timer B has been set lower than recommended (%d < 64 * timert1=%d). (RFC 3261, 17.1.1.2)\n", peer->timer_b, peer->timer_t1);
				peer->timer_t1 = global_t1min;
				peer->timer_b = peer->timer_t1 * 64;
			}
			peer->timer_t1 = peer->timer_b / 64;
		} else {
			peer->timer_b = peer->timer_t1 * 64;
		}
	}

	if (!peer->default_outbound_transport) {
		/* Set default set of transports */
		peer->transports = default_transports;
		/* Set default primary transport */
		peer->default_outbound_transport = default_primary_transport;
	}

	/* The default transport type set during build_peer should only replace the socket.type when...
	 * 1. Registration is not present and the socket.type and default transport types are different.
	 * 2. The socket.type is not an acceptable transport type after rebuilding peer.
	 * 3. The socket.type is not set yet. */
	if (((peer->socket.type != peer->default_outbound_transport) && (peer->expire == -1)) ||
		!(peer->socket.type & peer->transports) || !(peer->socket.type)) {

		set_socket_transport(&peer->socket, peer->default_outbound_transport);
	}

	if (ast_str_strlen(fullcontact)) {
		ast_string_field_set(peer, fullcontact, ast_str_buffer(fullcontact));
		peer->rt_fromcontact = TRUE;
		/* We have a hostname in the fullcontact, but if we don't have an
		 * address listed on the entry (or if it's 'dynamic'), then we need to
		 * parse the entry to obtain the IP address, so a dynamic host can be
		 * contacted immediately after reload (as opposed to waiting for it to
		 * register once again). But if we have an address for this peer and NAT was
		 * specified, use that address instead. */
		/* XXX May need to revisit the final argument; does the realtime DB store whether
		 * the original contact was over TLS or not? XXX */
		if (!ast_test_flag(&peer->flags[0], SIP_NAT_FORCE_RPORT) || ast_sockaddr_isnull(&peer->addr)) {
			__set_address_from_contact(fullcontact->str, &peer->addr, 0);
		}
	}

	if (srvlookup && peer->dnsmgr == NULL) {
		char transport[MAXHOSTNAMELEN];
		char _srvlookup[MAXHOSTNAMELEN];
		char *params;

		ast_copy_string(_srvlookup, srvlookup, sizeof(_srvlookup));
		if ((params = strchr(_srvlookup, ';'))) {
			*params++ = '\0';
		}

		snprintf(transport, sizeof(transport), "_%s._%s", get_srv_service(peer->socket.type), get_srv_protocol(peer->socket.type));

		peer->addr.ss.ss_family = get_address_family_filter(&bindaddr); /* Filter address family */
		if (ast_dnsmgr_lookup_cb(_srvlookup, &peer->addr, &peer->dnsmgr, sip_cfg.srvlookup && !peer->portinuri ? transport : NULL,
					on_dns_update_peer, ref_peer(peer, "Store peer on dnsmgr"))) {
			ast_log(LOG_ERROR, "srvlookup failed for host: %s, on peer %s, removing peer\n", _srvlookup, peer->name);
			unref_peer(peer, "dnsmgr lookup failed, getting rid of peer dnsmgr ref");
			unref_peer(peer, "getting rid of a peer pointer");
			return NULL;
		}
		if (!peer->dnsmgr) {
			/* dnsmgr refresh disabeld, release reference */
			unref_peer(peer, "dnsmgr disabled, unref peer");
		}

		ast_string_field_set(peer, tohost, srvlookup);

		if (global_dynamic_exclude_static && !ast_sockaddr_isnull(&peer->addr)) {
			int ha_error = 0;
			sip_cfg.contact_ha = ast_append_ha("deny", ast_sockaddr_stringify_addr(&peer->addr), 
							sip_cfg.contact_ha, &ha_error);
			if (ha_error) {
				ast_log(LOG_ERROR, "Bad or unresolved host/IP entry in configuration for peer %s, cannot add to contact ACL\n", peer->name);
			}
		}
	} else if (peer->dnsmgr && !peer->host_dynamic) {
		/* force a refresh here on reload if dnsmgr already exists and host is set. */
		ast_dnsmgr_refresh(peer->dnsmgr);
	}

	if (port && !realtime && peer->host_dynamic) {
		ast_sockaddr_set_port(&peer->defaddr, port);
	} else if (port) {
		ast_sockaddr_set_port(&peer->addr, port);
	}

	if (ast_sockaddr_port(&peer->addr) == 0) {
		ast_sockaddr_set_port(&peer->addr,
				      (peer->socket.type & SIP_TRANSPORT_TLS) ?
				      STANDARD_TLS_PORT : STANDARD_SIP_PORT);
	}
	if (ast_sockaddr_port(&peer->defaddr) == 0) {
		ast_sockaddr_set_port(&peer->defaddr,
				      (peer->socket.type & SIP_TRANSPORT_TLS) ?
				      STANDARD_TLS_PORT : STANDARD_SIP_PORT);
	}
	if (!peer->socket.port) {
		peer->socket.port = htons(((peer->socket.type & SIP_TRANSPORT_TLS) ? STANDARD_TLS_PORT : STANDARD_SIP_PORT));
	}

	if (!sip_cfg.ignore_regexpire && peer->host_dynamic && realtime) {
		time_t nowtime = time(NULL);

		if ((nowtime - regseconds) > 0) {
			destroy_association(peer);
			memset(&peer->addr, 0, sizeof(peer->addr));
			peer->lastms = -1;
			ast_debug(1, "Bah, we're expired (%d/%d/%d)!\n", (int)(nowtime - regseconds), (int)regseconds, (int)nowtime);
		}
	}

	/* Startup regular pokes */
	if (!devstate_only && realtime && peer->lastms > 0) {
		ref_peer(peer, "schedule qualify");
		sip_poke_peer(peer, 0);
	}

	ast_copy_flags(&peer->flags[0], &peerflags[0], mask[0].flags);
	ast_copy_flags(&peer->flags[1], &peerflags[1], mask[1].flags);
	ast_copy_flags(&peer->flags[2], &peerflags[2], mask[2].flags);
	if (ast_test_flag(&peer->flags[1], SIP_PAGE2_ALLOWSUBSCRIBE)) {
		sip_cfg.allowsubscribe = TRUE;	/* No global ban any more */
	}
	/* If read-only RT backend, then refresh from local DB cache */
	if (peer->host_dynamic && (!peer->is_realtime || !sip_cfg.peer_rtupdate)) {
		reg_source_db(peer);
	}

	/* If they didn't request that MWI is sent *only* on subscribe, go ahead and
	 * subscribe to it now. */
	if (!devstate_only && !ast_test_flag(&peer->flags[1], SIP_PAGE2_SUBSCRIBEMWIONLY) &&
		!AST_LIST_EMPTY(&peer->mailboxes)) {
		add_peer_mwi_subs(peer);
		/* Send MWI from the event cache only.  This is so we can send initial
		 * MWI if app_voicemail got loaded before chan_sip.  If it is the other
		 * way, then we will get events when app_voicemail gets loaded. */
		sip_send_mwi_to_peer(peer, 1);
	}

	peer->the_mark = 0;

	ast_free_ha(oldha);
	ast_free_ha(olddirectmediaha);
	if (!ast_strlen_zero(callback)) { /* build string from peer info */
		char *reg_string;
		if (asprintf(&reg_string, "%s?%s:%s@%s/%s", peer->name, peer->username, !ast_strlen_zero(peer->remotesecret) ? peer->remotesecret : peer->secret, peer->tohost, callback) < 0) {
			ast_log(LOG_WARNING, "asprintf() failed: %s\n", strerror(errno));
		} else if (reg_string) {
			sip_register(reg_string, 0); /* XXX TODO: count in registry_count */
			ast_free(reg_string);
		}
	}
	return peer;
}

static int peer_markall_func(void *device, void *arg, int flags)
{
	struct sip_peer *peer = device;
	peer->the_mark = 1;
	return 0;
}

static void display_nat_warning(const char *cat, int reason, struct ast_flags *flags) {
	int global_nat, specific_nat;

	if (reason == CHANNEL_MODULE_LOAD && (specific_nat = ast_test_flag(&flags[0], SIP_NAT_FORCE_RPORT)) != (global_nat = ast_test_flag(&global_flags[0], SIP_NAT_FORCE_RPORT))) {
		ast_log(LOG_WARNING, "!!! PLEASE NOTE: Setting 'nat' for a peer/user that differs from the  global setting can make\n");
		ast_log(LOG_WARNING, "!!! the name of that peer/user discoverable by an attacker. Replies for non-existent peers/users\n");
		ast_log(LOG_WARNING, "!!! will be sent to a different port than replies for an existing peer/user. If at all possible,\n");
		ast_log(LOG_WARNING, "!!! use the global 'nat' setting and do not set 'nat' per peer/user.\n");
		ast_log(LOG_WARNING, "!!! (config category='%s' global force_rport='%s' peer/user force_rport='%s')\n", cat, AST_CLI_YESNO(global_nat), AST_CLI_YESNO(specific_nat));
	}
}

static void cleanup_all_regs(void)
{
		/* First, destroy all outstanding registry calls */
		/* This is needed, since otherwise active registry entries will not be destroyed */
		ASTOBJ_CONTAINER_TRAVERSE(&regl, 1, do {  /* regl is locked */
				ASTOBJ_WRLOCK(iterator); /* now regl is locked, and the object is also locked */
				if (iterator->call) {
					ast_debug(3, "Destroying active SIP dialog for registry %s@%s\n", iterator->username, iterator->hostname);
					/* This will also remove references to the registry */
					dialog_unlink_all(iterator->call);
					iterator->call = dialog_unref(iterator->call, "remove iterator->call from registry traversal");
				}
				if (iterator->expire > -1) {
					AST_SCHED_DEL_UNREF(sched, iterator->expire, registry_unref(iterator, "reg ptr unref from reload config"));
				}
				if (iterator->timeout > -1) {
					AST_SCHED_DEL_UNREF(sched, iterator->timeout, registry_unref(iterator, "reg ptr unref from reload config"));
				}
				if (iterator->dnsmgr) {
					ast_dnsmgr_release(iterator->dnsmgr);
					iterator->dnsmgr = NULL;
					registry_unref(iterator, "reg ptr unref from dnsmgr");
				}
				ASTOBJ_UNLOCK(iterator);
		} while(0));
}

/*! \brief Re-read SIP.conf config file
\note	This function reloads all config data, except for
	active peers (with registrations). They will only
	change configuration data at restart, not at reload.
	SIP debug and recordhistory state will not change
 */
static int reload_config(enum channelreloadreason reason)
{
	struct ast_config *cfg, *ucfg;
	struct ast_variable *v;
	struct sip_peer *peer;
	char *cat, *stringp, *context, *oldregcontext;
	char newcontexts[AST_MAX_CONTEXT], oldcontexts[AST_MAX_CONTEXT];
	struct ast_flags dummy[2];
	struct ast_flags config_flags = { reason == CHANNEL_MODULE_LOAD ? 0 : ast_test_flag(&global_flags[1], SIP_PAGE2_RTCACHEFRIENDS) ? 0 : CONFIG_FLAG_FILEUNCHANGED };
	int auto_sip_domains = FALSE;
	struct ast_sockaddr old_bindaddr = bindaddr;
	int registry_count = 0, peer_count = 0, timerb_set = 0, timert1_set = 0;
	int subscribe_network_change = 1;
	time_t run_start, run_end;
	int bindport = 0;

	run_start = time(0);
	ast_unload_realtime("sipregs");
	ast_unload_realtime("sippeers");
	cfg = ast_config_load(config, config_flags);

	/* We *must* have a config file otherwise stop immediately */
	if (!cfg) {
		ast_log(LOG_NOTICE, "Unable to load config %s\n", config);
		return -1;
	} else if (cfg == CONFIG_STATUS_FILEUNCHANGED) {
		ucfg = ast_config_load("users.conf", config_flags);
		if (ucfg == CONFIG_STATUS_FILEUNCHANGED) {
			return 1;
		} else if (ucfg == CONFIG_STATUS_FILEINVALID) {
			ast_log(LOG_ERROR, "Contents of users.conf are invalid and cannot be parsed\n");
			return 1;
		}
		/* Must reread both files, because one changed */
		ast_clear_flag(&config_flags, CONFIG_FLAG_FILEUNCHANGED);
		if ((cfg = ast_config_load(config, config_flags)) == CONFIG_STATUS_FILEINVALID) {
			ast_log(LOG_ERROR, "Contents of %s are invalid and cannot be parsed\n", config);
			ast_config_destroy(ucfg);
			return 1;
		}
	} else if (cfg == CONFIG_STATUS_FILEINVALID) {
		ast_log(LOG_ERROR, "Contents of %s are invalid and cannot be parsed\n", config);
		return 1;
	} else {
		ast_clear_flag(&config_flags, CONFIG_FLAG_FILEUNCHANGED);
		if ((ucfg = ast_config_load("users.conf", config_flags)) == CONFIG_STATUS_FILEINVALID) {
			ast_log(LOG_ERROR, "Contents of users.conf are invalid and cannot be parsed\n");
			ast_config_destroy(cfg);
			return 1;
		}
	}

	ast_free_ha(sip_cfg.contact_ha);
	sip_cfg.contact_ha = NULL;

	default_tls_cfg.enabled = FALSE;		/* Default: Disable TLS */

	if (reason != CHANNEL_MODULE_LOAD) {
		ast_debug(4, "--------------- SIP reload started\n");

		clear_sip_domains();
		ast_mutex_lock(&authl_lock);
		if (authl) {
			ao2_t_ref(authl, -1, "Removing old global authentication");
			authl = NULL;
		}
		ast_mutex_unlock(&authl_lock);


		cleanup_all_regs();
		/* Then, actually destroy users and registry */
		ASTOBJ_CONTAINER_DESTROYALL(&regl, sip_registry_destroy);
		ast_debug(4, "--------------- Done destroying registry list\n");
		ao2_t_callback(peers, OBJ_NODATA, peer_markall_func, NULL, "callback to mark all peers");
	}

	/* Reset certificate handling for TLS sessions */
	if (reason != CHANNEL_MODULE_LOAD) {
		ast_free(default_tls_cfg.certfile);
		ast_free(default_tls_cfg.pvtfile);
		ast_free(default_tls_cfg.cipher);
		ast_free(default_tls_cfg.cafile);
		ast_free(default_tls_cfg.capath);
	}
	default_tls_cfg.certfile = ast_strdup(AST_CERTFILE); /*XXX Not sure if this is useful */
	default_tls_cfg.pvtfile = ast_strdup("");
	default_tls_cfg.cipher = ast_strdup("");
	default_tls_cfg.cafile = ast_strdup("");
	default_tls_cfg.capath = ast_strdup("");

	/* Initialize copy of current sip_cfg.regcontext for later use in removing stale contexts */
	ast_copy_string(oldcontexts, sip_cfg.regcontext, sizeof(oldcontexts));
	oldregcontext = oldcontexts;

	/* Clear all flags before setting default values */
	/* Preserve debugging settings for console */
	sipdebug &= sip_debug_console;
	ast_clear_flag(&global_flags[0], AST_FLAGS_ALL);
	ast_clear_flag(&global_flags[1], AST_FLAGS_ALL);
	ast_clear_flag(&global_flags[2], AST_FLAGS_ALL);

	/* Reset IP addresses  */
	ast_sockaddr_parse(&bindaddr, "0.0.0.0:0", 0);
	memset(&internip, 0, sizeof(internip));

	/* Free memory for local network address mask */
	ast_free_ha(localaddr);
	memset(&localaddr, 0, sizeof(localaddr));
	memset(&externaddr, 0, sizeof(externaddr));
	memset(&media_address, 0, sizeof(media_address));
	memset(&default_prefs, 0 , sizeof(default_prefs));
	memset(&sip_cfg.outboundproxy, 0, sizeof(struct sip_proxy));
	sip_cfg.outboundproxy.force = FALSE;		/*!< Don't force proxy usage, use route: headers */
	default_transports = SIP_TRANSPORT_UDP;
	default_primary_transport = SIP_TRANSPORT_UDP;
	ourport_tcp = STANDARD_SIP_PORT;
	ourport_tls = STANDARD_TLS_PORT;
	externtcpport = STANDARD_SIP_PORT;
	externtlsport = STANDARD_TLS_PORT;
	sip_cfg.srvlookup = DEFAULT_SRVLOOKUP;
	global_tos_sip = DEFAULT_TOS_SIP;
	global_tos_audio = DEFAULT_TOS_AUDIO;
	global_tos_video = DEFAULT_TOS_VIDEO;
	global_tos_text = DEFAULT_TOS_TEXT;
	global_cos_sip = DEFAULT_COS_SIP;
	global_cos_audio = DEFAULT_COS_AUDIO;
	global_cos_video = DEFAULT_COS_VIDEO;
	global_cos_text = DEFAULT_COS_TEXT;

	externhost[0] = '\0';			/* External host name (for behind NAT DynDNS support) */
	externexpire = 0;			/* Expiration for DNS re-issuing */
	externrefresh = 10;

	/* Reset channel settings to default before re-configuring */
	sip_cfg.allow_external_domains = DEFAULT_ALLOW_EXT_DOM;				/* Allow external invites */
	sip_cfg.regcontext[0] = '\0';
	sip_cfg.capability = DEFAULT_CAPABILITY;
	sip_cfg.regextenonqualify = DEFAULT_REGEXTENONQUALIFY;
	sip_cfg.legacy_useroption_parsing = DEFAULT_LEGACY_USEROPTION_PARSING;
	sip_cfg.notifyringing = DEFAULT_NOTIFYRINGING;
	sip_cfg.notifycid = DEFAULT_NOTIFYCID;
	sip_cfg.notifyhold = FALSE;		/*!< Keep track of hold status for a peer */
	sip_cfg.directrtpsetup = FALSE;		/* Experimental feature, disabled by default */
	sip_cfg.alwaysauthreject = DEFAULT_ALWAYSAUTHREJECT;
	sip_cfg.auth_options_requests = DEFAULT_AUTH_OPTIONS;
	sip_cfg.allowsubscribe = FALSE;
	sip_cfg.disallowed_methods = SIP_UNKNOWN;
	sip_cfg.contact_ha = NULL;		/* Reset the contact ACL */
	snprintf(global_useragent, sizeof(global_useragent), "%s %s", DEFAULT_USERAGENT, ast_get_version());
	snprintf(global_sdpsession, sizeof(global_sdpsession), "%s %s", DEFAULT_SDPSESSION, ast_get_version());
	snprintf(global_sdpowner, sizeof(global_sdpowner), "%s", DEFAULT_SDPOWNER);
	global_prematuremediafilter = TRUE;
	ast_copy_string(default_notifymime, DEFAULT_NOTIFYMIME, sizeof(default_notifymime));
	ast_copy_string(sip_cfg.realm, S_OR(ast_config_AST_SYSTEM_NAME, DEFAULT_REALM), sizeof(sip_cfg.realm));
	sip_cfg.domainsasrealm = DEFAULT_DOMAINSASREALM;
	ast_copy_string(default_callerid, DEFAULT_CALLERID, sizeof(default_callerid));
	ast_copy_string(default_mwi_from, DEFAULT_MWI_FROM, sizeof(default_mwi_from));
	sip_cfg.compactheaders = DEFAULT_COMPACTHEADERS;
	global_reg_timeout = DEFAULT_REGISTRATION_TIMEOUT;
	global_regattempts_max = 0;
	sip_cfg.pedanticsipchecking = DEFAULT_PEDANTIC;
	sip_cfg.autocreatepeer = DEFAULT_AUTOCREATEPEER;
	global_autoframing = 0;
	sip_cfg.allowguest = DEFAULT_ALLOWGUEST;
	global_callcounter = DEFAULT_CALLCOUNTER;
	global_match_auth_username = FALSE;		/*!< Match auth username if available instead of From: Default off. */
	global_rtptimeout = 0;
	global_rtpholdtimeout = 0;
	global_rtpkeepalive = DEFAULT_RTPKEEPALIVE;
	sip_cfg.allowtransfer = TRANSFER_OPENFORALL;	/* Merrily accept all transfers by default */
	sip_cfg.rtautoclear = 120;
	ast_set_flag(&global_flags[1], SIP_PAGE2_ALLOWSUBSCRIBE);	/* Default for all devices: TRUE */
	ast_set_flag(&global_flags[1], SIP_PAGE2_ALLOWOVERLAP_YES);	/* Default for all devices: Yes */
	sip_cfg.peer_rtupdate = TRUE;
	global_dynamic_exclude_static = 0;	/* Exclude static peers */
	sip_cfg.tcp_enabled = FALSE;

	/* Session-Timers */
	global_st_mode = SESSION_TIMER_MODE_ACCEPT;
	global_st_refresher = SESSION_TIMER_REFRESHER_UAS;
	global_min_se  = DEFAULT_MIN_SE;
	global_max_se  = DEFAULT_MAX_SE;

	/* Peer poking settings */
	global_qualify_gap = DEFAULT_QUALIFY_GAP;
	global_qualify_peers = DEFAULT_QUALIFY_PEERS;

	/* Initialize some reasonable defaults at SIP reload (used both for channel and as default for devices */
	ast_copy_string(sip_cfg.default_context, DEFAULT_CONTEXT, sizeof(sip_cfg.default_context));
	sip_cfg.default_subscribecontext[0] = '\0';
	sip_cfg.default_max_forwards = DEFAULT_MAX_FORWARDS;
	default_language[0] = '\0';
	default_fromdomain[0] = '\0';
	default_fromdomainport = 0;
	default_qualify = DEFAULT_QUALIFY;
	default_maxcallbitrate = DEFAULT_MAX_CALL_BITRATE;
	ast_copy_string(default_mohinterpret, DEFAULT_MOHINTERPRET, sizeof(default_mohinterpret));
	ast_copy_string(default_mohsuggest, DEFAULT_MOHSUGGEST, sizeof(default_mohsuggest));
	ast_copy_string(default_vmexten, DEFAULT_VMEXTEN, sizeof(default_vmexten)