/*
 * atanks - obliterate each other with oversize weapons
 * Copyright (C) 2003  Thomas Hudson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * */


#include "environment.h"
#include "globaldata.h"
#include "player.h"
#include "tank.h"
#include "menu.h"
#include "lineseq.h"
#include "files.h"

// When defined draws AI 'planning'
//#define AI_PLANNING_DEBUG	1

PLAYER::PLAYER (GLOBALDATA *global, ENVIRONMENT *env)
{
        int my_colour;

	_global = global;
	_env = env;
	money = 15000;
	score = 0;
	played = 0;
	won = 0;
	tank = NULL;
        team = TEAM_NEUTRAL;
	_turnStage = 0;
	selected = FALSE;
        changed_weapon = false;


        tank_bitmap = 0.0;

	nm[0] = 99;
	for (int count = 1; count < WEAPONS; count++)
		nm[count] = 0;

	for (int count = 0; count < ITEMS; count++)
		ni[count] = 0;

	strncpy (_name, "New", NAME_LENGTH);
	type = HUMAN_PLAYER;

	// 10% of time set to perplay weapon preferences
	preftype = (rand () % 10)?ALWAYS_PREF:PERPLAY_PREF;
	generatePreferences ();
	vengeful = rand () % 100;
	vengeanceThreshold = (double)(rand () % 1000) / 1000.0;
	revenge = NULL;
	selfPreservation = (double)(rand () % 3000) / 1000;
	painSensitivity = (double)(rand () % 3000) / 1000;
	defensive = (double)((rand () % 10000) - 5000) / 5000.0;

	// color = rand () % WHITE;
        my_colour = rand() % 4;
        switch (my_colour)
        {
           case 0: color = makecol(200 + (rand() % 56), rand() % 25, rand() % 25); break;
           case 1: color = makecol( rand() % 25, 200 + (rand() % 56), rand() % 25); break;
           case 2: color = makecol( rand() % 25, rand() % 25, 200 + (rand() % 56) ); break;
           case 3: color = makecol( 200 + (rand() % 56), rand() % 25, 200 + (rand() % 56)); break;
        }
	typeText[0] = "Human";
	typeText[1] = "Useless";
	typeText[2] = "Guesser";
	typeText[3] = "Range";
	typeText[4] = "Targetter";
	typeText[5] = "Deadly";
	preftypeText[0] = "Per Game";
	preftypeText[1] = "Only Once";
        tank_type[0] = "Normal";
        tank_type[1] = "Classic";
        teamText[0] = "Sith";
        teamText[1] = "Neutral";
        teamText[2] = "Jedi";

	initMenuDesc ();
}

PLAYER::PLAYER (GLOBALDATA *global, ENVIRONMENT *env, FILE *file, bool text_file)
{
	_global = global;
	_env = env;
	money = 15000;
	score = 0;
	tank = NULL;
	_turnStage = 0;
	selected = FALSE;
        
        tank_bitmap = 0.0;
        team = TEAM_NEUTRAL;


	nm[0] = 99;
	for (int count = 1; count < WEAPONS; count++)
		nm[count] = 0;

	for (int count = 0; count < ITEMS; count++)
		ni[count] = 0;

        // generate fresh weapon preferences because I goofed
        // a while back and did not save them all. So to
        // be backward compatible, generate the preferences.
        // If they are saved, then the saved ones will
        // over-write these ones
        generatePreferences();

	if (text_file)
           loadFromFile_Text (file);

	type = (int)type;

	typeText[0] = "Human";
	typeText[1] = "Useless";
	typeText[2] = "Guesser";
	typeText[3] = "Ranger";
	typeText[4] = "Targetter";
	typeText[5] = "Deadly";
	preftypeText[0] = "Per Game";
	preftypeText[1] = "Only Once";
        tank_type[0] = "Normal";
        tank_type[1] = "Classic";
        teamText[0] = "Sith";
        teamText[1] = "Neutral";
        teamText[2] = "Jedi";
	initMenuDesc ();
}

void PLAYER::setComputerValues ()
{
	switch ((int)type) {
		case HUMAN_PLAYER:
			break;
		case USELESS_PLAYER:
			rangeFindAttempts = 1;
			retargetAttempts = 0;
			focusRate = 2.0;
			errorMultiplier = 2.0;
			break;
		case GUESSER_PLAYER:
			rangeFindAttempts = 1;
			retargetAttempts = 0;
			focusRate = 0.6;
			errorMultiplier = 1.0;
			break;
		case RANGEFINDER_PLAYER:
			rangeFindAttempts = 10;
			retargetAttempts = 2;
			focusRate = 0.1;
			errorMultiplier = 0.3;
			break;
		case TARGETTER_PLAYER:
			rangeFindAttempts = 20;
			retargetAttempts = 5;
			focusRate = 1.0;
			errorMultiplier = 0.1;
			break;
		case DEADLY_PLAYER:
                case PART_TIME_BOT:
			rangeFindAttempts = 40;
			retargetAttempts = 10;
			focusRate = 1.0;
			errorMultiplier = 0.02;
			break;
		default:
			rangeFindAttempts = 0;
			retargetAttempts = 0;
			focusRate = 0;
			errorMultiplier = 1.0;
			break;
	}
}

int displayPlayerName (GLOBALDATA *global, ENVIRONMENT *env, int x, int y, void *data);
void PLAYER::initMenuDesc ()
{

	int destroyPlayer (GLOBALDATA *global, ENVIRONMENT *env, void *data);
        int i = 0;
        int height = -68;

	menudesc = new MENUDESC;
	if (!menudesc) {
		perror ("player.cc: Failed allocating memory for menudesc in PLAYER::initMenuDesc");
		exit (1);
	}
	menudesc->title = _name;

	// Name,Color
	menudesc->numEntries = 10;
	menudesc->okayButton = TRUE;
	menudesc->quitButton = FALSE;

	menuopts = new MENUENTRY[menudesc->numEntries];
	if (!menuopts) {
		perror ("player.cc: Failed allocating memory for menuopts in PLAYER::initMenuDesc");
		exit (1);
	}

        //init memory
        memset(menuopts, 0, menudesc->numEntries * sizeof(MENUENTRY));

	// Player name
	menuopts[i].name = "Name";
	menuopts[i].displayFunc = NULL;
	menuopts[i].color = color;
	menuopts[i].value = (double*)_name;
	menuopts[i].specialOpts = NULL;
	menuopts[i].type = OPTION_TEXTTYPE;
	menuopts[i].viewonly = FALSE;
	menuopts[i].x = _global->halfWidth - 3;
	menuopts[i].y = _global->halfHeight + height;
        i++; height += 20;

	// Player colour
	menuopts[i].name = "Colour";
	menuopts[i].displayFunc = NULL;
	menuopts[i].color = WHITE;
	menuopts[i].value = (double*)&color;
	menuopts[i].specialOpts = NULL;
	menuopts[i].type = OPTION_COLORTYPE;
	menuopts[i].viewonly = FALSE;
	menuopts[i].x = _global->halfWidth - 3;
	menuopts[i].y = _global->halfHeight + height;
        i++; height += 20;

	// Player type (human, computer)
	menuopts[i].name = "Type";
	menuopts[i].displayFunc = NULL;
	menuopts[i].color = WHITE;
	menuopts[i].value = (double*)&type;
	menuopts[i].min = 0;
	menuopts[i].max = LAST_PLAYER_TYPE - 1;
	menuopts[i].increment = 1;
	menuopts[i].defaultv = 0;
	menuopts[i].format = "%s";
	menuopts[i].specialOpts = typeText;
	menuopts[i].type = OPTION_SPECIALTYPE;
	menuopts[i].viewonly = FALSE;
	menuopts[i].x = _global->halfWidth - 3;
	menuopts[i].y = _global->halfHeight + height;
	i++; height += 20;

        menuopts[i].name = "Team";
        menuopts[i].displayFunc = NULL;
        menuopts[i].color = WHITE;
        menuopts[i].value = (double *)&team;
        menuopts[i].min = 0;
        menuopts[i].max = TEAM_JEDI;
        menuopts[i].increment = 1;
        menuopts[i].defaultv = TEAM_NEUTRAL;
        menuopts[i].format = "%s";
        menuopts[i].specialOpts = teamText;
        menuopts[i].type = OPTION_SPECIALTYPE;
        menuopts[i].viewonly = FALSE;
        menuopts[i].x = _global->halfWidth - 3;
        menuopts[i].y = _global->halfHeight + height;
        i++; height += 20;

	// Player preftype (human, computer)
	menuopts[i].name = "Generate Pref.";
	menuopts[i].displayFunc = NULL;
	menuopts[i].color = WHITE;
	menuopts[i].value = (double*)&preftype;
	menuopts[i].min = 0;
	menuopts[i].max = ALWAYS_PREF;
	menuopts[i].increment = 1;
	menuopts[i].defaultv = 0;
	menuopts[i].format = "%s";
	menuopts[i].specialOpts = preftypeText;
	menuopts[i].type = OPTION_SPECIALTYPE;
	menuopts[i].viewonly = FALSE;
	menuopts[i].x = _global->halfWidth - 3;
	menuopts[i].y = _global->halfHeight + height;
        i++; height += 20;

	menuopts[i].name = "Played";
	menuopts[i].displayFunc = NULL;
	menuopts[i].color = WHITE;
	menuopts[i].value = (double*)&played;
	menuopts[i].format = "%.0f";
	menuopts[i].specialOpts = NULL;
	menuopts[i].type = OPTION_DOUBLETYPE;
	menuopts[i].viewonly = TRUE;
	menuopts[i].x = _global->halfWidth - 3;
	menuopts[i].y = _global->halfHeight + height;
        i++; height += 20;

	menuopts[i].name = "Won";
	menuopts[i].displayFunc = NULL;
	menuopts[i].color = WHITE;
	menuopts[i].value = (double*)&won;
	menuopts[i].format = "%.0f";
	menuopts[i].specialOpts = NULL;
	menuopts[i].type = OPTION_DOUBLETYPE;
	menuopts[i].viewonly = TRUE;
	menuopts[i].x = _global->halfWidth - 3;
	menuopts[i].y = _global->halfHeight + height;
        i++; height += 20;

        menuopts[i].name = "Tank Type";
        menuopts[i].displayFunc = NULL;
        menuopts[i].color = WHITE;
        menuopts[i].value = &tank_bitmap;
        menuopts[i].min = 0;
        menuopts[i].max = 1;
        menuopts[i].increment = 1;
        menuopts[i].defaultv = 0;
        menuopts[i].format = "%1.0f";
        menuopts[i].specialOpts = tank_type;
        menuopts[i].type = OPTION_SPECIALTYPE;
        menuopts[i].viewonly = FALSE;
        menuopts[i].x = _global->halfWidth - 3;
        menuopts[i].y = _global->halfHeight + height;
        i++; height += 20;

	menuopts[i].name = "Delete This Player";
	menuopts[i].displayFunc = NULL;
	menuopts[i].color = WHITE;
	menuopts[i].value = (double*)destroyPlayer;
	menuopts[i].data = (void*)this;
	menuopts[i].type = OPTION_ACTIONTYPE;
	menuopts[i].viewonly = FALSE;
	menuopts[i].x = _global->halfWidth - 3;
	menuopts[i].y = _global->halfHeight + height;
        i++; height += 20;

	menudesc->entries = menuopts;
}

PLAYER::~PLAYER ()
{
	if (tank)
		delete tank;

	delete menuopts;
	delete menudesc;
}




/*
Save the player settings in text form. Fields are
formateed as
[name]=[value]\n

Function returns true on success and false on failure.
*/
int PLAYER::saveToFile_Text (FILE *file)
{
   int count;

   if (! file) return FALSE;
   // start section with "*PLAYER*"
   fprintf(file, "*PLAYER*\n");
   fprintf(file, "NAME=%s\n", _name);
   fprintf(file, "VENGEFUL=%d\n", vengeful);
   fprintf(file, "VENGEANCETHRESHOLD=%lf\n", vengeanceThreshold);
   fprintf(file, "TYPE=%lf\n", type);
   fprintf(file, "TYPESAVED=%lf\n", type_saved);
   fprintf(file, "COLOR=%d\n", color);
   fprintf(file, "COLOR2=%d\n", color2);
   for (count = 0; count < THINGS; count++)
      fprintf(file, "WEAPONPREFERENCES=%d %d\n", count, _weaponPreference[count]);

   fprintf(file, "PLAYED=%lf\n", played);
   fprintf(file, "WON=%lf\n", won);
   fprintf(file, "PREFTYPE=%lf\n", preftype);
   fprintf(file, "SELFPRESERVATION=%lf\n", selfPreservation);
   fprintf(file, "PAINSENSITIVITY=%lf\n", painSensitivity);
   fprintf(file, "DEFENSIVE=%lf\n", defensive);
   fprintf(file, "TANK_BITMAP=%lf\n", tank_bitmap);
   fprintf(file, "TEAM=%lf\n", team);
   fprintf(file, "***\n");
   return TRUE;
}


/*
This function tries to load player data from a text file.
Each line is parsed for "field=value", except WEAPONPREFERENCES
which is parsed "field=index value".
If all goes well TRUE is returned, on error the function returns FALSE.
-- Jesse
*/
int PLAYER::loadFromFile_Text (FILE *file)
{
    char line[MAX_CONFIG_LINE];
    int equal_position, line_length;
    int index, wp_value;
    char field[MAX_CONFIG_LINE], value[MAX_CONFIG_LINE];
    char *result = NULL;
    bool done = false;

    if (! file) return FALSE;

    // read until we hit line "*PLAYER*" or "***" or EOF
    do
    {
      result = fgets(line, MAX_CONFIG_LINE, file);
      if (! result)     // eof
         return FALSE;
      // if (! strncmp(line, "***", 3) )     // end of record
      //   return FALSE;
    } while ( strncmp(line, "*PLAYER*", 8) );     // read until we hit new record

    while ( (result) && (!done) )
    {
       // read a line
       memset(line, '\0', MAX_CONFIG_LINE);
       result = fgets(line, MAX_CONFIG_LINE, file);
       if (result)
       {
          // if we hit end of the record, stop
          if (! strncmp(line, "***", 3) ) return TRUE;
          // find equal sign
          line_length = strlen(line);
          // strip newline character
          if ( line[line_length - 1] == '\n')
          {
               line[line_length - 1] = '\0';
               line_length--;
          }
          equal_position = 1;
          while ( ( equal_position < line_length) && (line[equal_position] != '=') )
              equal_position++;
          // make sure we have valid equal sign
          if (equal_position <= line_length)
          {
             // seperate field from value
             memset(field, '\0', MAX_CONFIG_LINE);
             memset(value, '\0', MAX_CONFIG_LINE);
             strncpy(field, line, equal_position);
             strcpy(value, & (line[equal_position + 1]));
             // check which field we have and process value
             if (! strcasecmp(field, "name") )
                 strncpy(_name, value, NAME_LENGTH);
             else if (! strcasecmp(field, "vengeful") )
                 sscanf(value, "%d", &vengeful);
             else if (! strcasecmp(field, "vengeancethreshold") )
                 sscanf(value, "%lf", &vengeanceThreshold);
             else if (! strcasecmp(field, "type") )
                 sscanf(value, "%lf", &type);
             else if (! strcasecmp(field, "typesaved") )
             {
                 sscanf(value, "%lf", &type_saved);
                 if (type_saved > HUMAN_PLAYER)
                    type = type_saved;
             }
             else if (! strcasecmp(field, "color") )
                 sscanf(value, "%d", &color);
             else if (! strcasecmp(field, "color2") )
                 sscanf(value, "%d", &color2);
             else if (! strcasecmp(field, "played") )
                 sscanf(value, "%lf", &played);
             else if (! strcasecmp(field, "won") )
                 sscanf(value, "%lf", &won);
             else if (! strcasecmp(field, "preftype") )
                 sscanf(value, "%lf", &preftype);
             else if (! strcasecmp(field, "selfpreservation") )
                 sscanf(value, "%lf", &selfPreservation);
             else if (! strcasecmp(field, "painsensititveity") )
                 sscanf(value, "%lf", &painSensitivity);
             else if (! strcasecmp(field, "defensive") )
                 sscanf(value, "%lf", &defensive);
             else if (! strcasecmp(field, "tank_bitmap") )
                 sscanf(value, "%lf", &tank_bitmap);
             else if (! strcasecmp(field, "team") )
                 sscanf(value, "%lf", &team);
             else if (! strcasecmp(field, "weaponpreferences") )
             {
                 sscanf(value, "%d %d", &index, &wp_value);
                 if ( (index < THINGS) && (index >= 0) )
                    _weaponPreference[index] = wp_value;
             }

          }    // end of valid data line
       }      // end of if we read a line properly
    }   // end of while not done

     // make sure previous human players are restored as humans
    if (type == PART_TIME_BOT)
           type = HUMAN_PLAYER;

    return TRUE;
}




void PLAYER::exitShop ()
{
	double tmpDM = 0;

	damageMultiplier = 1.0;
	tmpDM += ni[ITEM_INTENSITY_AMP] * item[ITEM_INTENSITY_AMP].vals[0];
	tmpDM += ni[ITEM_VIOLENT_FORCE] * item[ITEM_VIOLENT_FORCE].vals[0];
	if (tmpDM > 0)
		damageMultiplier += pow (tmpDM, 0.6);
}


// run this at the begining of each turn
void PLAYER::newRound ()
{
        // if the player is under computer control, give it back to the player
        if ( type == PART_TIME_BOT )
           type = HUMAN_PLAYER;

	setComputerValues ();
	if (!tank) {
		tank = new TANK(_global, _env);
		if (!tank) {
			perror ("player.cc: Failed allocating memory for tank in PLAYER::newRound");
			exit (1);
		}
		tank->player = this;
	}
	tank->newRound ();

        changed_weapon = false;
        // if we are playing in a campaign, raise the AI level
        if (_global->campaign_mode)
        {
            if ( (type > HUMAN_PLAYER) && (type < DEADLY_PLAYER) )
              type += 1.0;
        }
        
}


void PLAYER::initialise ()
{
	long int totalPrefs;
	int rouletteCount;

	nm[0] = 99;
	for (int count = 1; count < WEAPONS; count++)
		nm[count] = 0;

	for (int count = 0; count < ITEMS; count++)
		ni[count] = 0;

	totalPrefs = 0;
	for (int weapCount = 0; weapCount < THINGS; weapCount++)
		totalPrefs += _weaponPreference[weapCount];

	rouletteCount = 0;
	for (int weapCount = 0; weapCount < THINGS; weapCount++) {
		int weapRSpace = (int)((double)_weaponPreference[weapCount] / totalPrefs * NUM_ROULETTE_SLOTS);
		int weapRCount = 0;

		if (weapRSpace < 1)
			weapRSpace = 1;
		while (weapRCount < weapRSpace && rouletteCount + weapRCount < NUM_ROULETTE_SLOTS) {
			_rouletteWheel[rouletteCount + weapRCount] = weapCount;
			weapRCount++;
		}
		rouletteCount += weapRSpace;
	}
	while (rouletteCount < NUM_ROULETTE_SLOTS)
		_rouletteWheel[rouletteCount++] = rand () % THINGS;

        kills = killed = 0;
}

void PLAYER::generatePreferences ()
{
	for (int weapCount = 0; weapCount < THINGS; weapCount++) {
		_weaponPreference[weapCount] = (rand () % MAX_WEAP_PROBABILITY);
	}
	// Add greater chance of buying parachutes
	_weaponPreference[WEAPONS + ITEM_PARACHUTE] += MAX_WEAP_PROBABILITY;
}

int PLAYER::selectRandomItem ()
{
	return (_rouletteWheel[rand () % NUM_ROULETTE_SLOTS]);
}

void PLAYER::setName (char *name)
{
        // initalize name
        memset(_name, '\0', NAME_LENGTH);
	strncpy (_name, name, NAME_LENGTH - 1);
}

int PLAYER::controlTank (TANK *ctank)
{
	if (key[KEY_F1])
		save_bmp ("scrnshot.bmp", _env->db, NULL);

	if (key_shifts & KB_CTRL_FLAG && ctrlUsedUp) {
		if (key[KEY_LEFT] || key[KEY_RIGHT] ||
			key[KEY_UP] || key[KEY_DOWN] ||
                        key[KEY_PGUP] || key[KEY_PGDN])
			ctrlUsedUp = TRUE;
		else
			ctrlUsedUp = FALSE;
	} else {
		ctrlUsedUp = FALSE;
	}

	if (_global->computerPlayersOnly &&
		((int)_global->skipComputerPlay >= SKIP_HUMANS_DEAD)) {
		if (_env->stage == STAGE_ENDGAME)
			return (-1);
	}

	k = 0;
	if (keypressed () && !fi) {
		k = readkey ();

		if ((_env->stage == STAGE_ENDGAME) &&
			(k >> 8 == KEY_ENTER ||
			 k >> 8 == KEY_ESC ||
			 k >> 8 == KEY_SPACE))
			return (-1);
		if (k >> 8 == KEY_ESC) {
			void clockadd ();
			install_int_ex (clockadd, SECS_TO_TIMER(6000));
			int mm = _env->ingamemenu ();
                        install_int_ex(clockadd, BPS_TO_TIMER(_global->frames_per_second));
			_env->make_update (0, 0, _global->screenWidth, _global->screenHeight);
			_env->make_bgupdate (0, 0, _global->screenWidth, _global->screenHeight);
			
			//Main Menu
			if (mm == 1) 
			{
				_global->command = GLOBAL_COMMAND_MENU;
				return (-1);
			}
			else if (mm == 2)  //Quit game
			{
				_global->command = GLOBAL_COMMAND_QUIT;
				return (-1);
			}
		}
                // check for number key being pressed
                if ( (k >> 8 >= KEY_0) && (k >> 8 <= KEY_9) )
                {
                    int value = (k >> 8) - KEY_0;
                   
                    // make sure the value is within range
                    if (value < _global->numPlayers)
                    {
                        if ( _global->players[value] )
                        {
                            TANK *my_tank = _global->players[value]->tank;
                            if (my_tank)
                            { 
                                sprintf(_global->tank_status, "%s: %d + %d -- Team: ", _global->players[value]->_name,
                                        my_tank->l, my_tank->sh);
                                strcat(_global->tank_status, _global->players[value]->Get_Team_Name());
                                _global->tank_status_colour = _global->players[value]->color;
                                _global->updateMenu = 1;
                            }
                            else
                              _global->tank_status[0] = 0;
                        }
                    }
                       
                }    // end of check status keys
	}
	if ((int)type == HUMAN_PLAYER || !ctank) {
		return (humanControls (ctank));
	} else if (_env->stage == STAGE_AIM) {
		return (computerControls (ctank));
	}
	return (0);
}

int PLAYER::chooseItemToBuy ()
{
	int currItem = selectRandomItem ();
	int itemNum = currItem - WEAPONS;
	int cumulative;
	int nextMod,curramt,newamt;

        /* if the AI is intelligent and there is gravity,
           they buy parachutes
        */
        if ( ( (type >= RANGEFINDER_PLAYER) && (type <= DEADLY_PLAYER) ) &&
             ( _env->landSlideType > LANDSLIDE_NONE))
        {
            if ( ni[ITEM_PARACHUTE] < 10)
            { 
               // do we have the funds?
               if (money >= item[ITEM_PARACHUTE].cost )
               {
                     money -= item[ITEM_PARACHUTE].cost;
                     ni[ITEM_PARACHUTE] += item[ITEM_PARACHUTE].amt;
                     return ITEM_PARACHUTE;
               }
            }
        }

	//loop through all weapon preferences
	for (int count = 0; count < THINGS; count++) {

		//is item available for purchase or selected first item
		//first item is free
		if ( (!_env->isItemAvailable (currItem)) || (currItem==0))
			continue;

		//determine the likelyhood of purchasing this selection
		//less likely the more of the item is owned		
		//if have zero of item, it is a fifty/fifty chance of purchase	
		if(currItem < WEAPONS) {
		  curramt = nm[currItem];
		  newamt = weapon[currItem].amt;
		} else {
		  curramt = ni[currItem - WEAPONS];
		  newamt = item[currItem - WEAPONS].amt;
		}

		if ((itemNum >= ITEM_INTENSITY_AMP &&
			itemNum <= ITEM_VIOLENT_FORCE) ||
                        (itemNum == ITEM_REPAIRKIT) ||
			(itemNum >= ITEM_ARMOUR &&
			 itemNum <= ITEM_PLASTEEL)) {
			cumulative = TRUE;
		} else {
			cumulative = FALSE;
		}
		  
		if (!cumulative) {
			nextMod = curramt / newamt;
		} else {
			nextMod = 1;
		}
		if (nextMod <= 0) {
			nextMod = 1;
		}

		if (rand () % nextMod) 
			continue;
			
		//weapon
		if (currItem < WEAPONS)
		{
			//don't buy if already maxed out
			if(nm[currItem] >=99) continue;
			//purchase the item
			if (money >= weapon[currItem].cost)
			{
			  money -= weapon[currItem].cost;
			  nm[currItem] += weapon[currItem].amt;
			  //don't allow more than 99
			  if(nm[currItem] > 99) nm[currItem] = 99;
			  return currItem;
			}
		}
		else  //item
		{
			//don't buy if already maxed out
			if(ni[currItem-WEAPONS] >=99) continue;
			//purchase the item
			if (money >= item[currItem - WEAPONS].cost) {
				money -= item[currItem - WEAPONS].cost;
				ni[currItem - WEAPONS] += item[currItem - WEAPONS].amt;
				//don't allow more than 99
				if(ni[currItem - WEAPONS] > 99) ni[currItem - WEAPONS] = 99;
				return (currItem);
			}
		}
	}
	return (-1);
}

char *PLAYER::selectRevengePhrase (double scale)
{
        char *filename;
        char *line;

        filename = (char *) calloc( strlen(DATA_DIR) + 64, sizeof(char) );
        if (! filename) return NULL;

        strcpy(filename, DATA_DIR);
        if (_global->language == LANGUAGE_ENGLISH)
           strcat(filename, "/revenge.txt");
        else if (_global->language == LANGUAGE_PORTUGUESE)
           strcat(filename, "/revenge.pt_BR.txt");
        else if (_global->language == LANGUAGE_FRENCH)
           strcat(filename, "/revenge_fr.txt");

        // static const LINESEQ says( filename, LINESEQ::skip_blanks );
        line = Random_Line(filename);
        free(filename);
        return line;
	// return says.random( );
}

char *PLAYER::selectGloatPhrase (double scale)
{
        char *filename;
        char *line;

        filename = (char *) calloc( strlen(DATA_DIR) + 64, sizeof(char) );
        if (! filename) return NULL;

        strcpy(filename, DATA_DIR);
	if (_global->language == LANGUAGE_ENGLISH)
           strcat(filename, "/gloat.txt");
        else if (_global->language == LANGUAGE_PORTUGUESE)
           strcat(filename, "/gloat.pt_BR.txt");
        else if (_global->language == LANGUAGE_FRENCH)
           strcat(filename, "/gloat_fr.txt");

        // static const LINESEQ says( filename, LINESEQ::skip_blanks );
        line = Random_Line(filename);
        free(filename); 
        return line;
	// return says.random( );
}

char *PLAYER::selectSuicidePhrase (double scale)
{
        char *filename;
        char *line;

        filename = (char *) calloc( strlen(DATA_DIR) + 64, sizeof(char) );
        if (! filename) return NULL;

        strcpy(filename, DATA_DIR);
        if (_global->language == LANGUAGE_ENGLISH)
           strcat(filename, "/suicide.txt");
        else if (_global->language == LANGUAGE_PORTUGUESE)
           strcat(filename, "/suicide.pt_BR.txt");
        else if (_global->language == LANGUAGE_FRENCH)
           strcat(filename, "/suicide_fr.txt");

        // static const LINESEQ says( filename, LINESEQ::skip_blanks );
        line = Random_Line(filename);       
        free(filename);
        return line;
        // return says.random();
}


int PLAYER::traceShellTrajectory (TANK *ctank, int *estimateX, int *estimateY, double targetX, double targetY, 
                                  double x, double y, double xv, double yv)
{
	TANK *ltank;
	int hitSomething = 0;
	double distance;
	int direction = (char)copysign (1, xv);
	double drag;
	int ticks = 0;
	int maxTicks = 500;
	double startX, startY;

	drag = weapon[ctank->cw].drag;
	if (ni[ITEM_DIMPLEP]) {
		drag *= item[ITEM_DIMPLEP].vals[0];
	} else if (ni[ITEM_SLICKP]) {
		drag *= item[ITEM_SLICKP].vals[0];
	}

	while ((!hitSomething) && (ticks < maxTicks) && (y < MENU ||
		 (getpixel (_env->terrain, (int)x, (int)y) == PINK))) {
		for (int objCount = 0; (ltank = (TANK*)_env->getNextOfClass (TANK_CLASS, &objCount)) && ltank; objCount++) {
			if (ltank->player != this) {
				double xaccel = 0, yaccel = 0;
				ltank->repulse (x, y, &xaccel, &yaccel);
				xv += xaccel;
				yv += yaccel;
			}
		}

		startX = x;
		startY = y;
                // hit side of the screen
		if (x + xv < 1 || x + xv > (_global->screenWidth-1)) {
                      // bounce if wall is rubber
                      if ( _env->current_wallType == WALL_RUBBER )
                      {
		  	  xv = -xv;	//bounce on the border
			  direction = -direction;
                      }
                      else if ( _env->current_wallType == WALL_SPRING )
                      {
                          xv = -xv * SPRING_CHANGE;
                          direction = -direction; 
                      }
                      else if ( _env->current_wallType == WALL_STEEL )  // wall is steel
                      {
                          hitSomething = 1;
                          break;
                      }
                      else if ( _env->current_wallType == WALL_WRAP )
                      {
                          if (xv < 0)
                             x = _global->screenWidth - 1;
                          else 
                             x = 1;
                      }
		} else {
			// motion - wind affected
			double accel = (_env->wind - xv) / weapon[ctank->cw].mass * drag * _env->viscosity;
			xv += accel;
			x += xv;
		}
		if (y + yv >= _global->screenHeight) 
		{
                    if ( _env->current_wallType == WALL_RUBBER )
                    {
			yv = -yv * 0.5;
			xv *= 0.95;
                    }
                    else if ( _env->current_wallType == WALL_SPRING )
                       yv = -yv * SPRING_CHANGE;
                    else     // steel or wrap
                       hitSomething = 1;
            
			if (fabs(xv) + fabs(yv) < 0.8)
				hitSomething = 1;
		}

		y += yv;
		yv += _env->gravity * (100.0 / _global->frames_per_second);

		if (	(y > _global->screenHeight + 100) ||
			(x > _global->screenWidth + 100) ||
			(x < 0 - 100)) {
			hitSomething = 1;
			break;
		}
		if (checkPixelsBetweenTwoPoints (_global, _env, &startX, &startY, x, y)) {
			x = startX;
			y = startY;
			hitSomething = 1;
			break;
		}
#ifdef	AI_PLANNING_DEBUG
		// Plot trajectories for debugging purposes
		circlefill (screen, (int)x, (int)y, 2, color);
#endif
		ticks++;
	}
#ifdef	AI_PLANNING_DEBUG
	// Targetting circle for debugging purposes
	circlefill (screen, (int)x, (int)y, 10, color);
	circlefill (screen, (int)targetX, (int)targetY, 20, color);
#endif

	*estimateX = (int)x;
	*estimateY = (int)y;
	distance = vector_length_f ((float)(x - targetX), (float)(y - targetY), 0.0);
	if (ticks > maxTicks)
		_targetPower /= 2;
	if ((targetX - x < 0) && (direction < 0))
		distance = -distance;
	else if ((targetX - x > 0) && (direction > 0))
		distance = -distance;

	return ((int)distance);
}

int PLAYER::rangeFind (TANK *ctank, int *estimateX, int *estimateY, int maxAttempts)
{
	int findCount;
	int spread = weapon[ctank->cw].spread;
	int overshoot = 1;
	int z;

	findCount = 0;
	while (overshoot && (findCount < maxAttempts)) {
		spread = 1;
		for (z = 0; z < spread; z++) {
			double mxv,myv;
			int ca;

			ca = _targetAngle + ((SPREAD * z) - (SPREAD * (spread - 1) / 2));
			mxv = _global->slope[ca][0] * _targetPower * (100.0 / _global->frames_per_second) / 100;
			myv = _global->slope[ca][1] * _targetPower * (100.0 / _global->frames_per_second) / 100;

			overshoot = (int)(traceShellTrajectory (
				ctank,
				estimateX, estimateY,
				_targetX, _targetY,
				ctank->x + (_global->slope[ca][0] * GUNLENGTH) - mxv,
				ctank->y + (_global->slope[ca][1] * GUNLENGTH) - myv,
					mxv, myv));
			if (overshoot < 0) {
				_targetPower += (int)((rand () % (abs (overshoot) + 10)) * focusRate);
				if (_targetPower > MAX_POWER)
					_targetPower = 2000;
			} else {
				_targetPower -= (int)((rand () % (abs (overshoot) + 10)) * focusRate);
				if (_targetPower < 0)
					_targetPower = 0;
			}
			if (abs (overshoot) < abs (ctank->smallestOvershoot)) {
				ctank->smallestOvershoot = overshoot;
				ctank->bestPower = _targetPower;
				ctank->bestAngle = _targetAngle;
			}
			findCount++;
		}
	}
	return (overshoot);
}

int PLAYER::calculateDirectAngle (int dx, int dy)
{
	double angle;

	angle = atan2 ((double)dx, (double)dy) / PI * 180;
	angle += (rand () % 40 - 20) * errorMultiplier;

	if (angle < 0)
		angle = angle + 360;
	if (angle < 90)
		angle = 90;
	else if (angle > 270)
		angle = 270;

	return ((int)angle);
}

int PLAYER::calculateAngle (int dx, int dy)
{
	double angle;

	angle = atan2 ((double)dx, (double)dy - abs (dx)) / PI * 180;
	angle += (rand () % 40 - 20) * errorMultiplier;

	if (angle < 0)
		angle = angle + 360;
	if (angle < 90)
		angle = 135;
	else if (angle > 270)
		angle = 225;

	return ((int)angle);
}

int PLAYER::calculatePowerGivenAngle (int dx, int dy, int angle)
{
	double airTime, dxTime;
	double requiredPower;
	double slopeX, slopeY;
	double powerMult;
	double minYv, minV;

        if (tank)
        {
            if ( (tank->cw >= WEAPONS) || 
               ( (tank->cw >= SML_LAZER) && (tank->cw <= LRG_LAZER) ) )
               return _targetPower;
        }

	slopeX = _global->slope[angle][1];
	slopeY = _global->slope[angle][0];

	/*
		The greater dx, the longer the missile will be in the air.
		Steeper angle, longer air time.
		The greater dy, the shorter air time.
		Double the gravity, double the power.
		Following wind = less power needed.
	*/


	// calculate minimum V to reach height dy
	if (dy > 0)
		minYv = sqrt (dy * _env->gravity * 2 * (100.0 / _global->frames_per_second));
	else
		minYv = 0;
	if ((slopeY != 0) && (minYv != 0))
		minV = fabs (minYv / slopeY);
	else
		minV = 0;
	if (slopeX != 0) {
		dxTime = (dx/fabs(slopeX));
	} else {
		// entirely down to the elements now
		dxTime = (dx / 0.000001);
	}
	//windAccel = (_env->wind / 20 * 0.2 * _env->viscosity);
	//dxTime = dxTime - (0.5 * windAccel * (dxTime*dxTime));
	if (dx < 0) {
		if (dxTime > 0)
			return (-1); // Won't ever get there
	} else {
		if (dxTime < 0)
			return (-1); // Won't ever get there
	}
	// Low target, less power
	// xdistance proportional to sqrt(dy)
	airTime = fabs(dxTime) + ((dy * slopeY) * _env->gravity * (100.0 / _global->frames_per_second)) * 2;

	// Less airTime doesn't necessarily mean less power
	// Horizontal firing means more power needed even though
	//   airTime is minimised
	powerMult = 1;
//	powerMult = (slopeX * slopeY);
//	if (powerMult == 0)
//		powerMult = 0.01;
	requiredPower = (sqrt (airTime * _env->gravity * (100.0 / _global->frames_per_second)) / powerMult) * 100;
	requiredPower += (int)((rand () % 100 - 50) * errorMultiplier);
	return ((int)requiredPower);
}

int PLAYER::adjustAngleGivenClearance (TANK *ctank, int xdist, int ydist, int clearance)
{
	while (!ctank->shootClearance (_targetAngle, clearance)) {
		if (xdist > 0) {
			if (_targetAngle < 270) {
				_targetAngle++;
			} else {
				_targetAngle += (int)((double)(rand () % 40 - 20) / (rangeFindAttempts + 1));
				break;
			}
		} else {
			if (_targetAngle > 90) {
				_targetAngle--;
			} else {
				_targetAngle += (int)((double)(rand () % 40 - 20) / (rangeFindAttempts + 1));
				break;
			}
		}
	}
	if (_targetAngle > 270)
		_targetAngle = 270 - (_targetAngle - 270);
	else if (_targetAngle < 90)
		_targetAngle = 90 + (90 - _targetAngle);

	return (_targetAngle);
}




double PLAYER::selectTarget ()
{
	int targetXCoord, targetYCoord;

	// return (selectTarget (&targetXCoord, &targetYCoord));
        return ( Select_Target( &targetXCoord, &targetYCoord) );
}



int PLAYER::computerSelectItem (TANK *ctank)
{
	int current_weapon = 0; // Current Weapon
	int targetXCoord, targetYCoord;

	if (ctank->howBuried () > BURIED_LEVEL) {
		if (nm[RIOT_BLAST] > 0)
			current_weapon = RIOT_BLAST;
		else if (nm[RIOT_CHARGE] > 0)
			current_weapon = RIOT_CHARGE;
		else if (nm[RIOT_BOMB] > 0)
			current_weapon = RIOT_BOMB;
		else if (nm[HVY_RIOT_BOMB] > 0)
			current_weapon = HVY_RIOT_BOMB;
		else if (ni[ITEM_TELEPORT] > 0)
			current_weapon = WEAPONS + ITEM_TELEPORT;
		else if (nm[SML_LAZER] > 0)
			current_weapon = SML_LAZER;
		else if (nm[MED_LAZER] > 0)
			current_weapon = MED_LAZER;
		else if (nm[LRG_LAZER] > 0)
			current_weapon = LRG_LAZER;
	}

        current_weapon = Select_Random_Weapon();
        // do not use riot items when not buried
        if ( (current_weapon == RIOT_CHARGE) || (current_weapon == RIOT_BLAST) )
        {
           if ( ctank->howBuried() < BURIED_LEVEL )
             current_weapon = 0;
        }

        if (! current_weapon)
           current_weapon = Select_Random_Item();

	if ((current_weapon < WEAPONS && !nm[current_weapon]) || 
            (current_weapon >= THINGS) || (!ni[current_weapon - WEAPONS]))
		current_weapon = 0;

        ctank->cw = current_weapon;
        // calcDamageMatrix (ctank, current_weapon);
        // selectTarget (&targetXCoord, &targetYCoord);
        Select_Target(& targetXCoord, &targetYCoord);
	_targetX = targetXCoord;
	_targetY = targetYCoord;

	_global->updateMenu = 1;

        // avoid trying to shoot below the tank's level with lasers
        if ( (_targetY > ctank->y) &&
             ( (current_weapon >= SML_LAZER) && (current_weapon <= LRG_LAZER) ) )
            ctank->cw = current_weapon = 0;

	return (current_weapon);
}

int PLAYER::computerControls (TANK *ctank)
{
	// At the most basic: select target, select weapon, aim and fire
	ctank->requireUpdate ();
	if (_turnStage == SELECT_WEAPON) {
		computerSelectItem (ctank);
		_turnStage++;

	} else if (_turnStage == SELECT_TARGET) {
                // selectTarget ();
		_turnStage++;

        } else if ( (_turnStage == CALCULATE_ATTACK) && ( ctank->cw >= WEAPONS ) ) {
               _turnStage++;
       
	} else if (_turnStage == CALCULATE_ATTACK) {
		int overshoot;
		int lastOvershoot = 0;    // avoid compiler warning
		int estimateX, estimateY;
		int xdist = (int)(_targetX - ctank->x);
		int ydist = (int)(_targetY - ctank->y);

		ctank->smallestOvershoot = _global->screenWidth;
		if (ctank->howBuried () > BURIED_LEVEL) {
			_targetAngle = 180;
			_targetPower = MAX_POWER / 10;
		} else {
			if (ctank->cw >= SML_LAZER && ctank->cw <= LRG_LAZER) {
				_targetAngle = calculateDirectAngle (xdist, ydist);
			} else {
				_targetAngle = calculateAngle (xdist, ydist);
				_targetAngle = adjustAngleGivenClearance (ctank, xdist, ydist, 150);
				_targetPower = calculatePowerGivenAngle (xdist, ydist, _targetAngle);
				if (rangeFindAttempts > 0)
					overshoot = rangeFind (ctank, &estimateX, &estimateY, rangeFindAttempts);
				else
					overshoot = 0;
				if (abs (overshoot) > weapon[ctank->cw].radius) {
					for (int retargetCount = 0; (retargetCount < retargetAttempts) && (abs (overshoot) > weapon[ctank->cw].radius); retargetCount++) {
						if (overshoot != 0 && (float)abs (overshoot - lastOvershoot) / abs (overshoot) < 0.01) {
							// There could be a
							//   more useful test?
							_targetAngle = (rand () % 180) + 90;
                                                        _targetPower = MAX_POWER / 2;
						} else {
							_targetAngle = calculateAngle (xdist, ydist);
							_targetAngle = adjustAngleGivenClearance (ctank, xdist, ydist, 150);
							_targetPower = calculatePowerGivenAngle (xdist, ydist, _targetAngle);
						}
						if (rangeFindAttempts > 0)
							overshoot = rangeFind (ctank, &estimateX, &estimateY, rangeFindAttempts);
						else
							overshoot = 0;
						lastOvershoot = overshoot;
					}
				}
				if (abs (ctank->smallestOvershoot) > weapon[ctank->cw].radius * 3) {
					_targetAngle = calculateAngle (xdist, ydist);
					_targetAngle = adjustAngleGivenClearance (ctank, xdist, ydist, (int)(weapon[ctank->cw].radius * 1.5));
					_targetPower = (MAX_POWER / 2);
				} else {
					_targetAngle = ctank->bestAngle;
					_targetPower = ctank->bestPower;
				}
			}
		}
		_turnStage++;

	} else if (_turnStage == AIM_WEAPON) {
		if (_targetAngle > ctank->a && ctank->a < 270) {
			// Left
			ctank->a++;
			_global->updateMenu = 1;
		} else if (_targetAngle < ctank->a && ctank->a > 90) {
			// Right
			ctank->a--;
			_global->updateMenu = 1;
		} else if (_targetPower < (ctank->p - 3) && ctank->p > 0) {
			// Reduce power
			ctank->p -= 5;
			_global->updateMenu = 1;
		} else if (_targetPower > (ctank->p + 3) && ctank->p < MAX_POWER) {
			// Increase Power
			ctank->p += 5;
			_global->updateMenu = 1;
		} else {
			_turnStage++;
		}

	} else if (fi) {
		// if (fi) don't do any of the following
	} else if (_turnStage == FIRE_WEAPON) {
		// ctank->activateCurrentSelection ();
                ctank->simActivateCurrentSelection();
                gloating = false;
		_turnStage = 0;
	}
	return (0);
}

int PLAYER::humanControls (TANK *ctank)
{
     int moved = 0;

	if (ctank) {
		if (!_env->mouseclock && mouse_b & 1 && mouse_x >= 250
			&& mouse_x < 378 && mouse_y >= 11 && mouse_y < 19) {
			_global->updateMenu = 1;
			if (ctank->fs) {
				ctank->sht++;
			}
			ctank->fs = 1;
			if (ctank->sht > SHIELDS - 1) {
				ctank->sht = 0;
			}
		}
		if (!_env->mouseclock && mouse_b & 1 && mouse_x >= 250
			&& mouse_x < 378 && mouse_y >= 21
			&& mouse_y < 29 && ctank->player->ni[ctank->sht] > 0 && (ctank->fs || ctank->sh > 0)) {
			_global->updateMenu = 1;
			ctank->ds = ctank->sht;
			ctank->player->ni[ctank->sht]--;
			ctank->sh = (int)item[ctank->sht].vals[SHIELD_ENERGY];
		}
	
		ctank->requireUpdate ();
	}
	
	//Keyboard control
	if ( _env->stage == STAGE_AIM)    
        {
		if (ctank) {
			if (key[KEY_LEFT] && !ctrlUsedUp && ctank->a < 270) {
				ctank->a++;
				_global->updateMenu = 1;
				if (key_shifts & KB_CTRL_FLAG)
					ctrlUsedUp = TRUE;
			}
			if (key[KEY_RIGHT] && !ctrlUsedUp && ctank->a > 90) {
				ctank->a--;
				_global->updateMenu = 1;
				if (key_shifts & KB_CTRL_FLAG)
					ctrlUsedUp = TRUE;
			}
			if (key[KEY_DOWN] && !ctrlUsedUp && ctank->p > 0) {
				ctank->p -= 5;
				_global->updateMenu = 1;
				if (key_shifts & KB_CTRL_FLAG)
					ctrlUsedUp = TRUE;
			}
			if (key[KEY_UP] && !ctrlUsedUp && ctank->p < MAX_POWER) {
				ctank->p += 5;
				_global->updateMenu = 1;
				if (key_shifts & KB_CTRL_FLAG)
					ctrlUsedUp = TRUE;
			}
                        if ( key[KEY_PGUP] && !ctrlUsedUp && ctank->p < MAX_POWER )
                        {
                            ctank->p += 100;
                            if (ctank->p > MAX_POWER)
                               ctank->p = MAX_POWER;
                            _global->updateMenu = 1;
			    if (key_shifts & KB_CTRL_FLAG)
				ctrlUsedUp = TRUE;
                        }
                        if ( key[KEY_PGDN] && !ctrlUsedUp && ctank->p > 0)
                        {
                            ctank->p -= 100;
                            if (ctank->p < 0)
                               ctank->p = 0;
                            _global->updateMenu = 1;
		            if (key_shifts & KB_CTRL_FLAG)
			      ctrlUsedUp = TRUE;
                        }
		}
	}
	if (k && !fi) {
		if ( _env->stage == STAGE_AIM) {
			if (ctank) {
				if (k >> 8 == KEY_N) {
					ctank->a = 180;
					_global->updateMenu = 1;
				}
				if (k >> 8 == KEY_TAB) {
					_global->updateMenu = 1;
					while (1) {
						ctank->cw++;
						if (ctank->cw >= THINGS)
							ctank->cw = 0;
						if (ctank->cw < WEAPONS) {
							if (ctank->player->nm[ctank->cw])
								break;
						} else {
							if (item[ctank->cw - WEAPONS].selectable && ctank->player->ni[ctank->cw - WEAPONS])
								break;

						}
					}
					//calcDamageMatrix (ctank, ctank->cw);
					//selectTarget ();
                                        changed_weapon = false;
				}
                                if ( k >> 8 == KEY_BACKSPACE) {
                                        _global->updateMenu = 1;
                                        while (1) {
                                                ctank->cw--;
                                                if (ctank->cw < 0)
                                                   ctank->cw = THINGS - 1;
 
                                                if (ctank->cw < WEAPONS) {
                                                        if (ctank->player->nm[ctank->cw])
                                                                break;
                                                } else {
                                                        if (item[ctank->cw - WEAPONS].selectable && ctank->player->ni[ctank->cw - WEAPONS])
                                                                break;

                                                }
                                        }
                                        changed_weapon = false;
                                }

                                // put the tank under computer control
                                if ((k >> 8 == KEY_F10) )
                                {
                                    type = PART_TIME_BOT;
                                    setComputerValues();
                                    return (computerControls (ctank));
                                }

                                // move the tank
                                if ((k >> 8 == KEY_COMMA) )
                                   moved = ctank->Move_Tank(DIR_LEFT);
                                if ((k >> 8 == KEY_STOP) )
                                   moved = ctank->Move_Tank(DIR_RIGHT);
                                if (moved)
                                   _global->updateMenu = 1;
 
				if ((k >> 8 == KEY_SPACE) &&
					(((ctank->cw < WEAPONS) && (ctank->player->nm[ctank->cw] > 0)) ||
						((ctank->cw < THINGS) && (ctank->player->ni[ctank->cw - WEAPONS] > 0)))) {
				//	ctank->activateCurrentSelection ();
                                ctank->simActivateCurrentSelection();
                                gloating = false;
				}
			}
		}
		if ((_env->stage == STAGE_ENDGAME) && (k >> 8 == KEY_ENTER || k >> 8 == KEY_ESC || k >> 8 == KEY_SPACE))
			return (-1);
	}
	return (0);
}



// returns a static string to the player's team name
char *PLAYER::Get_Team_Name()
{
    char *team_name = "";

    switch ( (int) team)
    {
       case TEAM_JEDI: team_name = "Jedi"; break;
       case TEAM_NEUTRAL: team_name = "Neutral"; break;
       case TEAM_SITH: team_name = "Sith"; break;
    }

    return team_name;
}



// Pick a weapon to fire at random.
// The weapon number is returned. If no other
// weapon is in inventory, then 0 - small missile is
// used.
int PLAYER::Select_Random_Weapon()
{
     int index;
     int num_weapons = 0;
     int random_weapon;

     // count number of different weapons we have
     for (index = 1; index < WEAPONS; index++)
     {
         if ( nm[index] )
           num_weapons++;
     }

    // we have no weapons, use default
    if ( num_weapons == 0 )
       return 0;

    // pick a random offset from the bottom of the list
    random_weapon = (rand() % num_weapons) + 1;

    index = WEAPONS - 1;
    while ( (index) && (random_weapon) )
    {
        if ( nm[index] )
            random_weapon--;
        if (random_weapon)
            index--; 
    }

    return index;
}



// This function selects a random item for
// the AI to use.
// This item to use is returned. If no
// item can be found, then the function
// returns 0
int PLAYER::Select_Random_Item()
{
   int index, item_num;
   int num_items = 0;
   int random_item;
   
   // count the items we have
   for (index = WEAPONS; index < THINGS; index++)
   {
       item_num = index - WEAPONS;
       if ( (ni[item_num] ) && ( item[item_num].selectable) )
           num_items++;
   }    // end of counting items

   if (! num_items)
      return 0;

    // if we have an item, there is still a 75% chance
    // that we may not use it
    if (! (rand() % 4) )
         return 0;

    // pick a random offset from the bottom of the list
    random_item = (rand() % num_items) + 1;

    index = THINGS - 1;
    while ( (index) && (random_item) )
    {
        if ( ( ni[index - WEAPONS] ) && ( item[index - WEAPONS].selectable) )
            random_item--;
        if (random_item)
            index--;
    }

    // do not use teleport without a parachute
    item_num = index - WEAPONS;
    if ( ( item_num == ITEM_TELEPORT) && (! ni[ITEM_PARACHUTE] ) )
       index = 0;

    // do not self destruct
    if ( (item_num == ITEM_VENGEANCE) || (item_num == ITEM_DYING_WRATH) ||
         (item_num ==  ITEM_FATAL_FURY) )
       index = 0;

    return index;
}




// This function selects which X and Y position to shoot
// at. The following factors are taken into consideration.
//
// 1. Are we on the same team?
// 2. Is the target stronger than I am?
// 
double PLAYER::Select_Target(int *target_X, int *target_Y)
{
    int random_target;
    int attempts = 0, max_attempts;
    TANK *best_target = NULL;
    int current_score, best_score = 0;
    int tank_count, object_count;
    TANK *current_tank;

    // find out how many tries we have to find a good target
    switch ( (int) type )
    {
      case USELESS_PLAYER: max_attempts = 5; break;
      case GUESSER_PLAYER: max_attempts = 10; break;
      case RANGEFINDER_PLAYER: max_attempts = 20; break;
      case TARGETTER_PLAYER: max_attempts = 30; break;
      case DEADLY_PLAYER:
      case PART_TIME_BOT:
      default:    max_attempts = 50; break;
    }


    // who do we want to shoot at?
    while ( attempts < max_attempts )
    {
        // select random tank for target
        random_target = rand() % _global->numPlayers;

        tank_count = 0;
        object_count = 0;
        current_tank = (TANK*) _env->getNextOfClass (TANK_CLASS, &object_count);
        while ( tank_count < random_target )
        {
            tank_count++;
            object_count++;
            current_tank = (TANK*) _env->getNextOfClass (TANK_CLASS, &object_count);
        }
    
        if (current_tank)
        {    
          // assign score to target
          // is the target alive
          if (current_tank->l > 0)
            current_score = 500;
          else
            current_score -= 500;

          // is the target weaker than we are
          if ( (tank->l + tank->sh) > (current_tank->l + current_tank->sh) )
             current_score += 200;

          // check to see if we are on the same team
          if ( ( team == TEAM_JEDI) || (team == TEAM_SITH ) )
          {
               if ( team == current_tank->player->team )
                  current_score -= 100;
               else
                  current_score += 100;
          }
      
          // try not to shoot ourselves 
          if ( current_tank->player == this )
             current_score -= 100;

          // do we have a grudge against the target
          if ( current_tank->player == revenge )
             current_score += 200;

          // prefer targets further away when violent death is on
          if (_global->violent_death)
          {
              int distance;
              distance = (int) fabs(tank->x - current_tank->x);
              if (distance > _global->halfWidth)
                 current_score += 100;
          }

          // decide if this target is better than others
          if (current_score > best_score)
          {
              best_score = current_score;
              best_target = current_tank;
          }
        }     // end of if we have a valid tank
          attempts++;
    }

    // assign target values
    if (best_target)
    {
        *target_X = (int) best_target->x;
        *target_Y = (int) best_target->y;
    }
    else       // no target, pick a random one
    {
        *target_X = rand() % _global->screenWidth + 1;
        *target_Y = _env->surface[ *target_X ];
    }

    return best_score;
}

