/*
 * operator for ALSA sequencer
 *
 * Virtual Tiny Keyboard
 *
 * Copyright (c) 1997-1999 by Takashi Iwai
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "vkb.h"
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <linux/asequencer.h>


/*
 * functions
 */
static int seq_open(Tcl_Interp *ip, void **private_return);
static void seq_close(void *private);
static void note_on(void *private, int note, int vel);
static void note_off(void *private, int note, int vel);
static void control(void *private, int type, int val);
static void program(void *private, int bank, int type);
static void bender(void *private, int bend);
static void call_timer(int cmd);
static void send_event(int do_flush);
static void putev(snd_seq_event_t *e);


/*
 * definition of device information
 */

static vkb_oper_t alsa_oper = {
	seq_open,
	seq_close,
	program,
	note_on,
	note_off,
	control,
	bender,
	NULL, /* chorus_mode */
	NULL, /* reverb_mode */
};

vkb_devinfo_t alsa_devinfo = {
	"alsa",
	"ALSA sequencer",
	&alsa_oper,
};


/*
 */

#define SEQUENCER_DEV	"/dev/snd/seq"

static int seqfd;
static int my_client, my_port;
static int seq_client = 21;
static int seq_port = 0;
static int seq_queue = 0;
static int chan_no = 0;


/*
 */

#define SEQBUF_SIZE	32	/* enough size for short events */
static snd_seq_event_t seqbuf[SEQBUF_SIZE];
static int num_bufs;

inline static void
flushev(void)
{
	if (write(seqfd, seqbuf, num_bufs * sizeof(snd_seq_event_t)) < 0)
		fprintf(stderr, "error writing event\n");
	num_bufs = 0;
}

static void
putev(snd_seq_event_t *e)
{
	seqbuf[num_bufs++] = *e;
	if (num_bufs >= SEQBUF_SIZE)
		flushev();
}


/*
 * open device
 */

static int
seq_open(Tcl_Interp *ip, void **private_return)
{
	char *devfile = SEQUENCER_DEV;
	char *var;
	snd_seq_port_subscribe_t subs;
	snd_seq_port_info_t port;
	snd_seq_queue_client_t qinfo;

	if ((var = Tcl_GetVar(ip, "devicefile", 0)) != NULL)
		devfile = var;

	if ((seqfd = open(devfile, O_RDWR)) < 0) {
		vkb_error(ip, "can't open sequencer device '%s'", devfile);
		return 0;
	}

	/* get my client id */
	if (ioctl(seqfd, SND_SEQ_IOCTL_CLIENT_ID, &my_client) < 0) {
		vkb_error(ip, "can't get client id\n");
		close(seqfd);
		return 0;
	}

	/* create port */
	memset(&port, 0, sizeof(port));
	port.client = my_client;
	port.capability = SND_SEQ_PORT_CAP_OUT;
	port.type = SND_SEQ_PORT_TYPE_APPLICATION;
	port.kernel = NULL;
	if (ioctl(seqfd, SND_SEQ_IOCTL_CREATE_PORT, &port) < 0) {
		vkb_error(ip, "can't create port\n");
		close(seqfd);
		return 0;
	}
	my_port = port.port;

	/* copy from Tcl variables */
	if ((var = Tcl_GetVar(ip, "alsaclient", 0)) != NULL)
		seq_client = atoi(var);
	if ((var = Tcl_GetVar(ip, "alsaport", 0)) != NULL)
		seq_port = atoi(var);
	if ((var = Tcl_GetVar(ip, "alsaqueue", 0)) != NULL)
		seq_queue = atoi(var);

	/* allocate queue */
	memset(&qinfo, 0, sizeof(qinfo));
	qinfo.queue = seq_queue;
	qinfo.client = my_client;
	qinfo.used = 1;
	qinfo.low = 1;
	qinfo.high = 500;
	if (ioctl(seqfd, SND_SEQ_IOCTL_SET_QUEUE_CLIENT, &qinfo) < 0) {
		vkb_error(ip, "can't set queue info\n");
		close(seqfd);
		return 0;
	}

	/* subscribe to MIDI port */
	memset(&subs, 0, sizeof(subs));
	subs.sender.client = my_client;
	subs.sender.port = my_port;
	subs.sender.queue = 0;
	subs.dest.client = seq_client;
	subs.dest.port = seq_port;
	subs.dest.queue = seq_queue;
	subs.exclusive = 0;
	subs.realtime = 0;

	if (ioctl(seqfd, SND_SEQ_IOCTL_SUBSCRIBE_PORT, &subs) < 0) {
		vkb_error(ip, "can't subscribe to MIDI port\n");
		close(seqfd);
		return 0;
	}

	/* clear buffer */
	num_bufs = 0;

	call_timer(SND_SEQ_EVENT_START);

	return 1;
}


static void
seq_close(void *private)
{
	call_timer(SND_SEQ_EVENT_STOP);
	close(seqfd);
}


/*
 * start / stop timer
 */

static void
call_timer(int cmd)
{
	snd_seq_event_t e;
	memset(&e, 0, sizeof(e));
	e.type = cmd;
	e.dest.client = SND_SEQ_CLIENT_SYSTEM;
	e.dest.port = SND_SEQ_PORT_SYSTEM_TIMER;
	putev(&e);
	flushev();
	//usleep(0.1E6);
}


/*
 */

static snd_seq_event_t ev;

static void
send_event(int do_flush)
{
	ev.flags = SND_SEQ_TIME_MODE_REL;
	ev.time.tick = 0;
	ev.dest.queue = seq_queue;
	ev.dest.client = seq_client;
	ev.dest.port = seq_port;
	ev.dest.channel = chan_no;
	ev.source.client = my_client;
	ev.source.port = my_port;

	putev(&ev);
	if (do_flush)
		flushev();
}

static void
note_on(void *private, int note, int vel)
{
	ev.type = SND_SEQ_EVENT_NOTEON;
	ev.data.note.note = note;
	ev.data.note.velocity = vel;
	send_event(1);
}

static void
note_off(void *private, int note, int vel)
{
	ev.type = SND_SEQ_EVENT_NOTEOFF;
	ev.data.note.note = note;
	ev.data.note.velocity = vel;
	send_event(1);
}

static void
control(void *private, int type, int val)
{
	ev.type = SND_SEQ_EVENT_CONTROLLER;
	ev.data.control.param = type;
	ev.data.control.value = val;
	send_event(1);
}

static void
program(void *private, int bank, int preset)
{
	if (bank == 128)
		chan_no = 9;
	else {
		chan_no = 0;
		ev.type = SND_SEQ_EVENT_CONTROLLER;
		ev.data.control.param = 0;
		ev.data.control.value = bank;
		send_event(0);
	}

	ev.type = SND_SEQ_EVENT_PGMCHANGE;
	ev.data.control.value = preset;
	send_event(0);
}

static void
bender(void *private, int bend)
{
	ev.type = SND_SEQ_EVENT_PITCHBEND;
	ev.data.control.value = bend;
	send_event(1);
}
