# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package AXP::Command::revlib::cleanup;
use base 'AXP::Command::revlib';

use Arch::Util qw(load_file);

sub infoline ($) {
	"remove casual incomplete revlib directories";
}

sub options ($) {
	(
		test    => { sh => 't', desc => "do not actually remove anything" },
		verbose => { sh => 'v', desc => "be more verbose" },
		quiet   => { sh => 'q', desc => "be quiet" },
	);
}

sub helptext ($) {
	q{
		tla creates temporary ,,new-revision.* directories when adds
		revisions to the library.  If tla is killed for some reason
		these directories will stay forever.  (This is possible, for
		example, if tla is called from the web script that is killed
		by the web server for exceeding the CPU/time quota or is
		stopped by a web visitor.)

		This command may be run hourly or daily from a cronjob, like:
			14 0/3 * * * /path/to/axp revlib prune -q -n 2000
			14 1/3 * * * /path/to/axp revlib prune -q -s 5GB
			14 2/3 * * * /path/to/axp revlib cleanup -q

		The exit status is 0 if any directories to be cleaned up are
		found, and 1 otherwise.
	};
}

sub execute ($) {
	my $self = shift;
	my %opt = %{$self->{options}};

	my @revlib_location_files = glob("$ENV{HOME}/.arch-params/=revision-library*");
	print "** loading config files:", (map { "\n\t$_" } @revlib_location_files), "\n"
		if $opt{verbose};

	my $lines = [];
	my @revlib_locations = map { load_file($_, $lines); @$lines; } @revlib_location_files;
	print "** searching in revlib locations:", (map { "\n\t$_" } @revlib_locations), "\n"
		if $opt{verbose};

	my @dirs = map { glob("$_/*/*/*/*/,,new-revision.*") } @revlib_locations;
	# get only stalled temporary dirs (older than 15 minutes)
	@dirs = grep { -d $_ && (-M $_) * 24 * 60 > 15
		or $opt{verbose} && print("** skipping recent $_\n") && 0 } @dirs;

	print "* test, no real directories are removed\n"
		if $opt{test} && !$opt{quiet};
	if (!@dirs) {
		print "* no incomplete temporary directories to cleanup found\n"
			unless $opt{quiet};
		exit 1;
	} else {
		print "* going to cleanup ", scalar(@dirs), " incomplete temporary directories\n"
			unless $opt{quiet};
		foreach (@dirs) {
			print "** removing $_\n" if $opt{verbose};
			system("/bin/rm", "-rf", $_) unless $opt{test};
		}
		print "* done\n"
			unless $opt{quiet};
	}
}

1;
