<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    clock-defs.php                                          */
/* Author:      Paul Waite                                              */
/* Description: Definitions for displaying a javascript clock.          */
/*              This clock is a digital read-out in a layer or div.     */
/*              The time and date are taken from the server rather than */
/*              relying on possibly innaccurate user browser.           */
/*                                                                      */
/* ******************************************************************** */
/** @package datetime */

// The following are the defined display properties. These
// are all properties which can be set for the clock.
/** Clock display hours (am/pm) */
define("DISPLAY_24_HOURS",    0);
/** Display to seconds or mins */
define("DISPLAY_SECONDS",     1);
/** Show date or not */
define("DISPLAY_DATE",        2);
/** Show day name or not */
define("DISPLAY_DAYNAME",     3);
/** Abbreviate day name or not */
define("ABBREVIATE_DAYNAME",  4);
/** Show the date first */
define("DISPLAY_DATE_FIRST",  5);
/** Show date and time on one line */
define("DISPLAY_INLINE",      6);
/** Clock overall clock table attributes (eg. style, width etc.) */
define("CLOCK_ATTRS",         7);
/** Clock time string style and/or class */
define("CLOCK_TIME_CSS",      8);
/** Clock date string style and/or class */
define("CLOCK_DATE_CSS",      9);
/** Clock date format M=month no, m=month name, D=day no,
    CC=century (20), YY=year (99) */
define("CLOCK_DATE_FORMAT",  10);

// ----------------------------------------------------------------------
/**
* The digital_clock class. This is implemented as a 'layer' or 'div'
* and rendered using Javascript. UNless you specify the initial time
* yourself, the time will be taken from the server, rather than from
* the user's browser, which may be innaccurate. Timezone settings are
* then used to adjust it to display the appropriate date & time.
* @package datetime
*/
class digital_clock extends HTMLObject {
  /** Unique ID for this clock widget */
  var $clockid = "";
  /** Initial time to set the clock to */
  var $initial_ts = "";
  /** The timezone offset in +/- hours from GMT.
      If this is left unset, then the displayed time will be
      offset to local time according to the client browser
      timezone settings. */
  var $tz;
  /** X-offset from left */
  var $leftx;
  /** Y-offset from top */
  var $topy;
  /** Positioning mode */
  var $position = "";
  /** Visibility setting */
  var $visibility = "visible";
  /** Clock display settings. An array of booleans. */
  var $display_properties = array();
  /** Style or class for time digits */
  var $css_time;
  /** Style or class for date display */
  var $css_date;
  /** Date format eg. 'M d, ccyy' for 'April 4, 2005' */
  var $date_format = "M d, ccyy";
  // Private
  /** Interval between clock updates in millisecs */
  var $update_interval_ms = 1000;
  // .....................................................................
  /**
  * Constructor for the clock
  * @param integer $initial_ts Optional initial time - a Unix timestamp
  */
  function digital_clock($initial_ts="") {
    // Set a unique ID for this widget..
    $this->clockid = "clock_" . md5(uniqid(rand(), true));

    // Initial starting time for this clock..
    if ($initial_ts != "") {
      $this->initial_ts = $initial_ts;
    }
    else {
      $this->initial_ts = time();
    }
    // Set default styles..
    $this->css_time = new StylableObject("font:bold 12pt arial,helvetica,sans-serif");
    $this->css_date = new StylableObject("font:bold 8pt arial,helvetica,sans-serif");

    // Set the default boolean properties..
    $this->display_properties[DISPLAY_24_HOURS] = false;
    $this->display_properties[DISPLAY_SECONDS] = false;
    $this->display_properties[DISPLAY_DATE] = false;
    $this->display_properties[DISPLAY_DAYNAME] = false;
    $this->display_properties[ABBREVIATE_DAYNAME] = false;
    $this->display_properties[DISPLAY_DATE_FIRST] = false;
    $this->display_properties[DISPLAY_INLINE] = false;

  } // digital_clock
  // .....................................................................
  /** Set the position and the positioning directive for the clock.
  * @param integer $leftx Pixels from left margin
  * @param integer $topy Pixels from top margin
  * @param string $pos Positioning mode "absolute" or "relative"
  */
  function set_position($leftx="", $topy="", $pos="absolute") {
    $this->leftx = $leftx;
    $this->topy  = $topy;
    $this->position = $pos;
  } // set_position
  // .....................................................................
  /** Set the starting visibility for the clock. In case you want to
  * reveal/hide it with some javascript.
  * @param string $vis Visibility: "visible" or "hidden"
  */
  function set_visibility($vis="visible") {
    $this->visibility = ($vis === "visible") ? "visible" : "hidden";
  } // set_visibility
  // .....................................................................
  /** Set the timezone to use for this clock. The argument should be
  * an integer (or decimal) in the range +12 to -12, in hours offset from GMT.
  * @param mixed $tz From +12 to -12 hours offset from GMT.
  */
  function set_timezone($tz) {
    if ($tz === "") {
      if (isset($this->tz)) {
        unset($this->tz);
      }
    }
    else {
      $this->tz = $tz;
    }
  } // set_timezone
  // .....................................................................
  /** Special set timezone case - setting to GMT (0 hours offset). */
  function set_to_gmt() {
    $this->set_timezone(0);
  } // set_to_gmt
  // .....................................................................
  /** Set the number of seconds to wait before updating the clock each time.
  * The default value is every second, but if you aren't showing seconds then
  * you might want to offload the browser a little and make it every 30 secs
  * or even longer.
  * @param integer $secs Number of seconds between updates
  */
  function set_update_interval_secs($secs) {
    $this->update_interval_ms = ceil($secs * 1000);
    if ($this->update_interval_ms <= 0) {
      $this->update_interval_ms = 1000;
    }
  } // set_update_interval_secs
  // .....................................................................
  /** Set a display property for the clock. Display property IDs are defined
  * at the top of this script, and can be any type of content. NB: Boolean
  * properties are so arranged that the default is always false.
  * @param integer $propertyid ID of the property to set
  * @param mixed $value Value for the given display property
  */
  function set_display_property($propertyid, $value) {
    if (isset($this->display_properties[$propertyid])) {
      $this->display_properties[$propertyid] = $value;
    }
  } // set_display_property

  // Some nicer wrappers for the above..
  function show_24hrs()         { $this->set_display_property(DISPLAY_24_HOURS, true);   }
  function show_seconds()       { $this->set_display_property(DISPLAY_SECONDS, true);    }
  function show_date()          { $this->set_display_property(DISPLAY_DATE, true);       }
  function show_dayname()       { $this->set_display_property(DISPLAY_DAYNAME, true);    }
  function abbreviate_dayname() { $this->set_display_property(ABBREVIATE_DAYNAME, true); }
  function show_date_first()    { $this->set_display_property(DISPLAY_DATE_FIRST, true); }
  function show_inline()        { $this->set_display_property(DISPLAY_INLINE, true);     }

  // .....................................................................
  /** Set the style of class for the time display
  * @param $css string Style or class name for time display
  */
  function set_css_time($css) {
    $this->css_time->setcss($css);
  } // set_css_time
  // .....................................................................
  /** Set the style of class for the date display
  * @param $css string Style or class name for date display
  */
  function set_css_date($css) {
    $this->css_date->setcss($css);
  } // set_css_date
  // .....................................................................
  /** Set the format for the date display. This is just done by replacement
  * in a format string where d=month date, M=month name, m=month no, cc=century,
  * and yy=year (99). Eg. 'm d, ccyy' would give 'April 4, 2005'. Another
  * example would be 'd/m/yy' which would render '4/4/05'.
  * @param $css string Style or class name for date display
  */
  function set_date_format($fmt) {
    $this->date_format = $fmt;
  } // set_date_format
  // .....................................................................
  /**
  * Render the clock.
  */
  function html() {
    debug_trace($this);
    global $RESPONSE;

    // Set the style and class settings..
    $this->display_properties[CLOCK_ATTRS] = $this->attributes();
    $this->display_properties[CLOCK_TIME_CSS] = $this->css_time->style_attributes();
    $this->display_properties[CLOCK_DATE_CSS] = $this->css_date->style_attributes();
    $this->display_properties[CLOCK_DATE_FORMAT] = $this->date_format;

    // Only insert this stuff into the webpage once..
    static $done_once = false;
    if (!$done_once) {
      $RESPONSE->add_named_script(
          "var daynames=new Array('Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday');\n"
        . "var monthnames=new Array('January', 'February', 'March', 'April', 'May', 'June',"
        . "'July', 'August', 'September', 'October', 'November', 'December');\n"
        . "function writeLayer(clockID,htmlcontent) {\n"
        . " if (document.layers){\n"
        . "  lyr=document.layers[''+clockID+''].document;\n"
        . "  lyr.open();\n"
        . "  lyr.write(htmlcontent);\n"
        . "  lyr.close();\n"
        . " }\n"
        . " else if (document.all){\n"
        . "  document.all[''+clockID+''].innerHTML=htmlcontent;\n"
        . " }\n"
        . " else if (document.getElementById){\n"
        . "  range=document.createRange();\n"
        . "  element=document.getElementById(''+clockID+'');\n"
        . "  range.setStartBefore(element);\n"
        . "  content=range.createContextualFragment(htmlcontent)\n"
        . "  while(element.hasChildNodes()) element.removeChild(element.lastChild);\n"
        . "  element.appendChild(content);\n"
        . " }\n"
        . "}\n",
        "clock"
        );
      // Assmeble the generic formatTime function..
      $js = "function formatTime(clock_dt,clock_props){\n"
          . " var timeStr = '';\n"
          . " var monthno = clock_dt.getMonth();\n"
          . " var monthname = monthnames[monthno];\n"
          . " var mthday = String(clock_dt.getDate());\n"
          . " var year = String(clock_dt.getFullYear());\n"
          . " var shortyear = year.substr(2,2);\n"
          . " var century = year.substr(0,2);\n"
          . " var dayno = clock_dt.getDay();\n"
          . " var dayname = daynames[dayno];\n"
          . " var dtfmt = clock_props[" . CLOCK_DATE_FORMAT . "];\n"
          . " if(clock_props[" . ABBREVIATE_DAYNAME . "]) dayname = dayname.substring(0,3);\n"
          . " var hrs  = clock_dt.getHours();\n"
          . " var mins = clock_dt.getMinutes();\n"
          . " var mm  = String(mins);\n"
          . " var secs = clock_dt.getSeconds();\n"
          . " var ss  = String(secs);\n"
          . " var ampm = '';\n"
          . " if (!clock_props[" . DISPLAY_24_HOURS . "]) {\n"
          . "  if (hrs > 12) {\n"
          . "   hrs = hrs - 12;\n"
          . "   ampm = 'pm';\n"
          . "  }\n"
          . "  else if (hrs == 12) {\n"
          . "   ampm = 'pm';\n"
          . "  }\n"
          . "  else if (hrs == 0) {\n"
          . "   hrs = 12;\n"
          . "   ampm = 'am';\n"
          . "  }\n"
          . "  else {\n"
          . "   ampm = 'am';\n"
          . "  }\n"
          . " }\n"
          . " var hh = String(hrs);\n"
          . " var Str = dtStr = tiStr = '';\n"
          . " tiStr += '<td' + clock_props[" . CLOCK_TIME_CSS . "] + '>';\n"
          . " tiStr += '<span' + clock_props[" . CLOCK_TIME_CSS . "] + '>';\n"
          . " tiStr += ((hrs < 10) ? ' ' : '') + hh + ((mins < 10) ? ':0' : ':') + mm;\n"
          . " if(clock_props[" . DISPLAY_SECONDS . "]) tiStr += ((secs < 10) ? ':0' : ':') + ss;\n"
          . " if(!clock_props[" . DISPLAY_24_HOURS . "]) tiStr += ampm;\n"
          . " tiStr += '</span></td>';\n"
          . " Str += '<table border=\"0\" cellspacing=\"0\" cellpadding=\"0\"' + clock_props[" . CLOCK_ATTRS . "] + '><tr>';\n"
          . " if(clock_props[" . DISPLAY_DATE . "]) {\n"
          . "  dtStr += '<td' + clock_props[" . CLOCK_DATE_CSS . "] + '><span' + clock_props[" . CLOCK_DATE_CSS . "] + '>';"
          . "  if(clock_props[" . DISPLAY_DAYNAME . "]) {dtStr += dayname + ' '}\n"
          . "  var dt = dtfmt.replace(/d/, mthday);\n"
          . "  dt = dt.replace(/M/, monthname);\n"
          . "  dt = dt.replace(/m/, String(monthno+1));\n"
          . "  dt = dt.replace(/yy/, shortyear);\n"
          . "  dt = dt.replace(/cc/, century);\n"
          . "  dtStr += dt + '</span></td>';\n"
          . "  if(clock_props[" . DISPLAY_DATE_FIRST . "]) {Str += dtStr;} else {Str += tiStr;}\n"
          . "  if(!clock_props[" . DISPLAY_INLINE . "]) {Str += '</tr><tr>';}\n"
          . "  if(clock_props[" . DISPLAY_DATE_FIRST . "]) {Str += tiStr;} else {Str += dtStr;}\n"
          . " }\n"
          . " else {Str += tiStr;}\n"
          . " Str += '</tr></table>';\n"
          . " return Str;\n"
          . "}\n"
          ;
      // Finally add the assmebled script..
      $RESPONSE->add_named_script($js, "clock");

      // Only want it inserted once..
      $done_once = true;
    }

    // Initial time derived from server, which allows us to
    // determine the accuracy of the clock shown to user..
    $gmdate = gmdate("Y,m-1,d,H,i,s", $this->initial_ts);

    // Define names..
    $clock_var = "refer_" . $this->clockid;
    $local_var = "local_" . $this->clockid;
    $props_var = "props_" . $this->clockid;
    $fn_showclock = "display_" . $this->clockid;
    if (isset($this->tz)) {
      $tz_offset_ms = "($this->tz * 60 * 60 * 1000)";
    }
    else {
      $tz_offset_ms = "($clock_var.getTimezoneOffset() * 60 * 1000)";
    }
    // Add main vars..
    $RESPONSE->add_named_script(
        "var $clock_var=new Date($gmdate);\n"
      . "var $local_var=new Date();\n"
      . "$clock_var.setTime($clock_var.getTime() + $tz_offset_ms);\n"
      . "var $props_var = new Array();\n"
      ,
      "clock"
      );
    $ix = 0;
    foreach ($this->display_properties as $propval) {
      if (is_bool($propval)) {
        $assignment = $props_var . "[" . $ix . "] = " . ($propval === true ? "true" : "false") . ";\n";
      }
      else {
        $assignment = $props_var . "[" . $ix . "] = '$propval';\n";
      }
      $RESPONSE->add_named_script($assignment, "clock");
      $ix += 1;
    }

    $RESPONSE->add_named_script(
        "function $fn_showclock(){\n"
      . "setTimeout('$fn_showclock()'," . $this->update_interval_ms . ");\n"
      . "var now_dt = new Date();\n"
      . " $clock_var.setTime($clock_var.getTime() + (now_dt.getTime() - $local_var.getTime()));\n"
      . " $local_var.setTime(now_dt.getTime());\n"
      . "var s = formatTime($clock_var,$props_var);\n"
      . "writeLayer('$this->clockid', s);\n"
      . "}\n"
      ,
      "clock"
      );
    $div = new HTMLObject();
    $div->inherit_attributes($this);
    $div->setstyle("visibility:$this->visibility");
    $div->setstyle("z-index:1");
    if ($this->position != "") {
      $div->setstyle("position:$this->position");
      if (isset($this->leftx) && $this->leftx != "") {
        $div->setstyle("left:$this->leftx");
      }
      if (isset($this->topy) && $this->topy != "") {
        $div->setstyle("top:$this->topy");
      }
    }
    $s = "<div id=\"$this->clockid\"" . $div->attributes() . "></div>";
    $RESPONSE->set_onload("$fn_showclock();");
    return $s;
    debug_trace();
  } // html
} // digital_clock class

// ----------------------------------------------------------------------
?>