/* Bezerk
 * Copyright (C) 1998 Tony Gale.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */
#include <gtk/gtk.h>
#include <glib.h>
#include <string.h>
#include <sys/time.h>
#include <unistd.h>

#include "reply.h"
#include "servers.h"
#include "message.h"
#include "irc.h"
#include "debug.h"
#include "util.h"
#include "ch_utils.h"
#include "msg_utils.h"
#include "ctcp.h"
#include "bezerk.h"
#include "dialogs.h"

#ifdef HAVE_TCL
#include <tcl.h>
#include "tcl_script.h"

extern Tcl_Interp *interp;
#endif

extern messageData irc_message;

extern GdkColormap *cmap;
extern GdkColor colour_red;
extern GdkColor colour_yellow;
extern GdkColor colour_blue;
extern GdkColor colour_grey;
extern GdkColor colour_green;
extern GdkColor colour_white;

void recv_ping(Connection *connection)
{

  bs_function_enter();

  if (!connection) {
    bs_function_leave();
    return;
  }

  irc_send_format(connection->sd, "PONG %s",  (irc_message.args ? irc_message.args[0] : NULL) );

  bs_function_leave();
  return;
}

void recv_pong(Connection *connection)
{

  bs_function_enter();

  bs_function_leave();
  return;
}

/* TODO: i don't think this can happen but, check for list of groups on join message (and others) */

void recv_privmsg(Connection *connection)
{
  Message *message;
  ChannelInfo *target_channel;

  bs_function_enter();

  if (!connection) {
    bs_function_leave();
    return;
  }

  if ( !irc_message.nick || !irc_message.args || !irc_message.args[0] || !irc_message.args[1] ) {
    message = message_new(MT_CHANNEL, connection, NULL);
    message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_red, NULL, NULL,
						     "Malformed PRIVMSG from server"));
    bs_function_leave();
    return;
  }

  if ( (irc_message.args[1][0] == '\001' ) && 
       (irc_message.args[1][strlen(irc_message.args[1])-1]  == '\001') ) {
    process_ctcp_request(connection);
    bs_function_leave();
    return;
  }

  if ( !strcasecmp(connection->nick, irc_message.args[0])) {

    /* It's a message directed at me */
    message = message_new(MT_NICK, connection, irc_message.nick);
    if ( find_message_entry(connection->messages, irc_message.nick) < 0 ) {
      message->parts = g_slist_append(message->parts, message_part_new(&colour_red, NULL, NULL, "msg"));
      message->parts = g_slist_append(message->parts, message_part_new(&colour_white, NULL, NULL, "("));
      message->parts = g_slist_append(message->parts,
				      message_part_new(&colour_white, NULL, NULL, irc_message.nick));
      message->parts = g_slist_append(message->parts, message_part_new(&colour_white, NULL, NULL, "): "));
      message->parts = message_part_parse_new(message->parts, &colour_white, 
					      irc_message.args[1]);
    } else {
      message->parts = g_slist_append(message->parts, message_part_new(&colour_white, NULL, NULL, "<"));
      message->parts = g_slist_append(message->parts, 
				      message_part_new(&colour_white, NULL, NULL, irc_message.nick));
      message->parts = g_slist_append(message->parts, message_part_new(&colour_white, NULL, NULL, "> "));
      message->parts = message_part_parse_new(message->parts, &colour_white, 
					      irc_message.args[1]);
    }      

#ifdef HAVE_TCL
    /* davidw TCL */
    do_hook (PRIVMSGME, 3, irc_message.nick,  irc_message.address, irc_message.args[1]);
    /* -----  */
#endif

    bs_function_leave();
    return;
  } else if ( (target_channel = find_channel(connection->channels, irc_message.args[0])) != NULL ) {
    message = message_new(MT_CHANNEL, connection, irc_message.args[0]);
    message->parts = g_slist_append(message->parts, message_part_new(&colour_white, NULL, NULL, "<"));
    if (!strncasecmp(irc_message.args[1], connection->nick, strlen(connection->nick))) {
      /* Message starts with my nick, so highlight senders nick */
      message->parts = g_slist_append(message->parts, 
				     message_part_new(&colour_yellow, NULL, NULL, irc_message.nick));
    } else {
      message->parts = g_slist_append(message->parts, 
				     message_part_new(&colour_white, NULL, NULL, irc_message.nick));
    }
    if ( target_channel != target_channel->window->current_channel ) {
      message->parts = g_slist_append(message->parts, message_part_new(&colour_white, NULL, NULL, " ["));
      message->parts = g_slist_append(message->parts, message_part_new(&colour_yellow, NULL, NULL, 
								       irc_message.args[0]));
      message->parts = g_slist_append(message->parts, message_part_new(&colour_white, NULL, NULL, "]"));
    }
    message->parts = g_slist_append(message->parts, message_part_new(&colour_white, NULL, NULL, "> "));
    message->parts = message_part_parse_new(message->parts, &colour_white, 

					   irc_message.args[1]);
#ifdef HAVE_TCL
    /* davidw TCL */
    do_hook (PRIVMSGCHAN, 4, irc_message.nick,  irc_message.address,
            irc_message.args[0], irc_message.args[1]);
#endif

    bs_function_leave();
    return;
  }

  bs_function_leave();
  return;
}

void  recv_notice(Connection *connection)
{
  Message *message;
  ChannelInfo *target_channel;

  bs_function_enter();

  if (!connection) {
    bs_function_leave();
    return;
  }

  if ( !irc_message.args || !irc_message.args[0] || !irc_message.args[1] ) {
    message = message_new(MT_CHANNEL, connection, NULL);
    message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_red, NULL, NULL, "Malformed PRIVMSG from server"));
    bs_function_leave();
    return;
  }

  if ( (irc_message.args[1][0] == '\001' ) && 
       (irc_message.args[1][strlen(irc_message.args[1])-1]  == '\001') ) {
    process_ctcp_response(connection);
    bs_function_leave();
    return;
  }

  if ( !irc_message.nick ) {
    message = message_new(MT_CHANNEL, connection, NULL);
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_yellow, NULL, NULL, "notice: "));
    message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_white, NULL, NULL, irc_message.args[1]));
    bs_function_leave();
    return;
  }

  if ( !strcasecmp(connection->nick, irc_message.nick) && !strncasecmp("LAG ",  irc_message.args[1], 4)) {
    /* It's a lag test message from myself */
    char *time_ptr;

    time_ptr = get_time_diff(&irc_message.args[1][4]);
    strncpy(connection->lag, time_ptr, LAG_LEN);
    g_free(time_ptr);
    set_user_status(connection);
    bs_function_leave();
    return;
  }

  if (!strcasecmp(connection->nick, irc_message.args[0])) {

    /* It's a notice directed at me */
    message = message_new(MT_NICK, connection, irc_message.nick);
    if ( find_message_entry(connection->messages, irc_message.nick) < 0 ) {
      message->parts = g_slist_append(message->parts, 
				      message_part_new(&colour_yellow, NULL, NULL, "notice"));
      message->parts = g_slist_append(message->parts, message_part_new(&colour_white, NULL, NULL, "("));
      message->parts = g_slist_append(message->parts, 
				      message_part_new(&colour_white, NULL, NULL, irc_message.nick));
      message->parts = g_slist_append(message->parts, message_part_new(&colour_white, NULL, NULL, "): "));
      message->parts = message_part_parse_new(message->parts, &colour_white, 
					      irc_message.args[1]);
    } else {
      message->parts = g_slist_append(message->parts, message_part_new(&colour_white, NULL, NULL, "<"));
      message->parts = g_slist_append(message->parts, 
				      message_part_new(&colour_white, NULL, NULL, irc_message.nick));
      message->parts = g_slist_append(message->parts, message_part_new(&colour_white, NULL, NULL, "> "));
      message->parts = message_part_parse_new(message->parts, &colour_white, 
					      irc_message.args[1]);
    }

#ifdef HAVE_TCL
    /* davidw TCL */
    do_hook (NOTICEME, 3, irc_message.nick,  irc_message.address, irc_message.args[1]);
#endif
    
    bs_function_leave();
    return;
  } else if ( (target_channel = find_channel(connection->channels, irc_message.args[0])) != NULL ) {
    message = message_new(MT_CHANNEL, connection, irc_message.args[0]);
    message->parts = g_slist_append(message->parts, message_part_new(&colour_white, NULL, NULL, "<"));
    if (!strncasecmp(irc_message.args[1], connection->nick, strlen(connection->nick))) {
      /* Message starts with my nick, so highlight senders nick */
      message->parts = g_slist_append(message->parts, 
				     message_part_new(&colour_yellow, NULL, NULL, irc_message.nick));
    } else {
      message->parts = g_slist_append(message->parts, 
				     message_part_new(&colour_white, NULL, NULL, irc_message.nick));
    }
    if ( target_channel != target_channel->window->current_channel ) {
      message->parts = g_slist_append(message->parts, 
				     message_part_new(&colour_white, NULL, NULL, " ["));
      message->parts = g_slist_append(message->parts, 
				  message_part_new(&colour_yellow, NULL, NULL, irc_message.args[0]));
      message->parts = g_slist_append(message->parts, message_part_new(&colour_white, NULL, NULL, "]"));
    }
    message->parts = g_slist_append(message->parts, message_part_new(&colour_white, NULL, NULL, "> "));
    message->parts = message_part_parse_new(message->parts, &colour_white, 
					   irc_message.args[1]);

#ifdef HAVE_TCL
    /* davidw TCL */
    do_hook (NOTICECHAN, 4, irc_message.nick,  irc_message.address,
	     irc_message.args[0], irc_message.args[1]);
#endif
  }    

  bs_function_leave();
  return;
}

void  recv_join(Connection *connection)
{
  Message *message;
  ChannelInfo *channel;

  bs_function_enter();

  if (!connection) {
    bs_function_leave();
    return;
  }

  if ( !irc_message.nick || !irc_message.args || !irc_message.args[0]) {
    message = message_new(MT_CHANNEL, connection, NULL);
    message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_red, NULL, NULL, "Malformed JOIN from server"));
    bs_function_leave();
    return;
  }

  if (strcasecmp(irc_message.nick, connection->nick)) {
    /* A new user has joined a channel */
    channel = find_channel(connection->channels, irc_message.args[0]);
    if (channel)
      add_channel_member(channel, irc_message.nick);
    message = message_new(MT_CHANNEL, connection, irc_message.args[0]);
    message->parts = g_slist_append(message->parts, 
				   message_part_new(&colour_yellow, NULL, NULL, irc_message.nick));
    message->parts = g_slist_append(message->parts, 
				   message_part_new(&colour_white, NULL, NULL, " has joined "));
    message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_yellow, NULL, NULL, irc_message.args[0]));
    if (channel == channel->window->current_channel) {
      set_members_list( channel );
    }
  }

#ifdef HAVE_TCL
  /* davidw TCL */
  do_hook (JOIN, 4, irc_message.nick,  irc_message.address, irc_message.args[0]);
#endif

  bs_function_leave();
  return;
}

void recv_part(Connection *connection)
{
  Message *message;
  ChannelInfo *target_channel;
  int position;

  bs_function_enter();

  if (!connection) {
    bs_function_leave();
    return;
  }

  if ( !irc_message.nick || !irc_message.args || !irc_message.args[0] ) {
    message = message_new(MT_CHANNEL, connection, NULL);
    message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_red, NULL, NULL, "Malformed PART from server"));
    bs_function_leave();
    return;
  }

  if ( !(target_channel = find_channel(connection->channels, irc_message.args[0])) ) {
    /* We're not on this channel, which shouldn't happen, or it's us who left the channel */
    bs_function_leave();
    return;
  }

  position = remove_channel_member(target_channel, irc_message.nick);
  if (target_channel == target_channel->window->current_channel) {
    set_members_list(target_channel);
  }

  message = message_new(MT_CHANNEL, connection, irc_message.args[0]);
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_yellow, NULL, NULL, irc_message.nick));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_white, NULL, NULL, " has left "));
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_yellow, NULL, NULL, irc_message.args[0]));

#ifdef HAVE_TCL
  do_hook (PART, 3, irc_message.nick,  irc_message.address,
	   irc_message.args[0]);
#endif

  bs_function_leave();
  return;
}

void recv_mode(Connection *connection)
{
  /* TODO need some good ideas regarding
     what to pass to the tcl function  - davidw */

  Message *message = NULL;
  ChannelInfo *target_channel;
  char *arg;
  char *current;
  int arg_num=2;
  int action=FALSE;
#ifdef HAVE_TCL
  char plusminus='\0';
#endif

  if (!connection) {
    bs_function_leave();
    return;
  }

  bs_function_enter();

  if ( !irc_message.nick || !irc_message.args || !irc_message.args[0] || !irc_message.args[1]) {
    message = message_new(MT_CHANNEL, connection, NULL);
    message->parts = g_slist_append(message->parts,
				   message_part_new(&colour_red, NULL, NULL, "Malformed MODE from server"));
    bs_function_leave();
    return;
  }

  target_channel = find_channel(connection->channels, irc_message.args[0]);
  arg = irc_message.args[arg_num];
  current = irc_message.args[1];

  if (target_channel) {
    message = message_new(MT_CHANNEL, connection, irc_message.args[0]);
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_yellow, NULL, NULL, "MODE("));
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_white, NULL, NULL, irc_message.args[0]));
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_yellow, NULL, NULL, ") "));
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_white, NULL, NULL, irc_message.args[1]));
    while (irc_message.args[arg_num]) {
      message->parts = g_slist_append(message->parts,
				      message_part_new(&colour_white, NULL, NULL, " "));
 
      message->parts = g_slist_append(message->parts,
				      message_part_new(&colour_white, NULL, NULL, irc_message.args[arg_num]));
      arg_num++;
    }
    arg_num=2;
    message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_white, NULL, NULL, " by "));
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_white, NULL, NULL, irc_message.nick));
    while(strlen(current)) {
#ifdef HAVE_TCL
      char md[3];
      md[2] = '\0';
#endif
      switch(*current) {
      case '+':
	action = TRUE;
#ifdef HAVE_TCL
	plusminus = *current;
#endif
	break;
      case '-':
	action = FALSE;
#ifdef HAVE_TCL
	plusminus = *current;
#endif
	break;
      case 'o':
	if (!arg) {
	  break;
	}
	if (action) {
	  set_member_status(target_channel, arg, USEROPER);
	} else {
	  set_member_status(target_channel, arg, USERBASIC);
	}
	arg = irc_message.args[++arg_num];
	if (target_channel->window && (target_channel == target_channel->window->current_channel) ) {
	  set_members_list(target_channel);
	}
#ifdef HAVE_TCL
	/* davidw TCL */
	md[0] = plusminus;
	md[1] = *current;
	do_hook (MODE, 5, irc_message.nick,  irc_message.address,
		 target_channel->name, md, arg);
#endif
 	break;
      case 'v':
	if (!arg) {
	  break;
	}
	if (action) {
	  set_member_status(target_channel, arg, USERVOICE);
	} else {
	  set_member_status(target_channel, arg, USERBASIC);
	}
	arg = irc_message.args[++arg_num];
	if (target_channel->window && (target_channel == target_channel->window->current_channel) ) {
	  set_members_list(target_channel);
	}
#ifdef HAVE_TCL
	/* davidw TCL */
	md[0] = plusminus;
	md[1] = *current;
	do_hook (MODE, 5, irc_message.nick,  irc_message.address,
		 target_channel->name, md, arg);
#endif
	break;
      case 'n':
      case 'p':
      case 's':
      case 'i':
      case 't':
      case 'm':
	set_channel_mode(target_channel, *current, action);
#ifdef HAVE_TCL
	/* davidw TCL */
	md[0] = plusminus;
	md[1] = *current;
	do_hook (CHANMODE, 4, irc_message.nick,  irc_message.address,
		 target_channel->name, md);

#endif
	break;
      }
      current++;
    }
    if (target_channel->window && (target_channel == target_channel->window->current_channel) ) {
      set_channel_status(target_channel);
    }
  } else {
    message = message_new(MT_CHANNEL, connection, NULL);
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_yellow, NULL, NULL, "MODE("));
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_white, NULL, NULL, irc_message.args[0]));
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_yellow, NULL, NULL, ") "));
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_white, NULL, NULL, irc_message.args[1]));
    message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_white, NULL, NULL, " by "));
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_white, NULL, NULL, irc_message.nick));
    while(strlen(current)) {
      switch(*current) {
      case '+':
	action = TRUE;
	break;
      case '-':
	action = FALSE;
	break;
      case 'i':
      case 's':
      case 'w':
      case 'o':
	set_user_mode(connection, *current, action);
	break;
      }
      current++;
    }
    set_user_status(connection);
  }

  bs_function_leave();
  return;
}

void recv_quit(Connection *connection)
{
  Message *message=NULL;
  ChannelInfo *target_channel;
  GSList *glist_entry;
  int member_number;

  bs_function_enter();

  if (!connection) {
    bs_function_leave();
    return;
  }

  if ( !irc_message.nick || !irc_message.args || !irc_message.args[0] ) {
    message = message_new(MT_CHANNEL, connection, NULL);
    message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_red, NULL, NULL, "Malformed QUIT from server"));
    bs_function_leave();
    return;
  }

  glist_entry = connection->channels;

  while(glist_entry)  {
    target_channel = (ChannelInfo *) glist_entry->data;
    member_number = remove_channel_member(target_channel, irc_message.nick);
    if ( member_number >= 0 ) {
      message = message_new(MT_CHANNEL, connection, target_channel->name);
      message->parts = g_slist_append(message->parts, 
				      message_part_new(&colour_yellow, NULL, NULL, irc_message.nick));
      message->parts = g_slist_append(message->parts,
				      message_part_new(&colour_yellow, NULL, NULL, " has QUIT: "));
      message->parts = message_part_parse_new(message->parts, &colour_white, 
					      irc_message.args[0]);
    }     
    if ( (member_number >= 0) && (target_channel == target_channel->window->current_channel) ) {
      set_members_list(target_channel);
    }
    glist_entry = g_slist_next(glist_entry);
  }

  /* TODO: need a quit message for each channel the person was on. */
  /*       This requires a list of messages */
/*   message = message_new(MT_CHANNEL, connection, NULL); */
/*   message->parts = g_slist_append(message->parts,  */
/* 				 message_part_new(&colour_yellow, NULL, NULL, irc_message.nick)); */
/*   message->parts = g_slist_append(message->parts, */
/* 				 message_part_new(&colour_yellow, NULL, NULL, " has QUIT: ")); */
/*   message->parts = message_part_parse_new(message->parts, &colour_white,  */
/* 					 irc_message.args[0]); */

#ifdef HAVE_TCL
  /* davidw TCL */
  sprintf(cl, "onquit %s %s { %s }", irc_message.nick,  irc_message.address, irc_message.args[0]);
  Tcl_Eval(interp, cl);
  /* fprintf(stderr, "%s\n", interp->result); */
#endif

  bs_function_leave();
  return;
}

void recv_nick(Connection *connection)
{
  Message *message=NULL;
  ChannelInfo *target_channel;
  GSList *glist_entry;
  int member_number;

  bs_function_enter();

  if (!connection) {
    bs_function_leave();
    return;
  }

  if ( !irc_message.nick || !irc_message.args || !irc_message.args[0] ) {
    message = message_new(MT_CONSOLE, connection, NULL);
    message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_red, NULL, NULL, "Malformed NICK from server"));
    bs_function_leave();
    return;
  }

  glist_entry = connection->channels;

  if (!strcasecmp(irc_message.nick, connection->nick)) {
    /* User has changed his nick */
    if (connection->oldnick) {
      g_free(connection->oldnick);
    }
    connection->oldnick = connection->nick;
    connection->nick = g_strdup(irc_message.args[0]);
    set_user_status(connection);
  }

  while(glist_entry)  {
    target_channel = (ChannelInfo *) glist_entry->data;
    member_number = change_member_nick(target_channel, 
				       irc_message.nick, irc_message.args[0]);
    if ( member_number >=0 ) {
      message = message_new(MT_CHANNEL, connection, target_channel->name);
      message->parts = g_slist_append(message->parts, 
				      message_part_new(&colour_yellow, NULL, NULL, irc_message.nick));
      message->parts = g_slist_append(message->parts,
				      message_part_new(&colour_white, NULL, NULL, " is now known as "));
      message->parts = g_slist_append(message->parts,
				      message_part_new(&colour_yellow, NULL, NULL, irc_message.args[0]));
    }
    if ( (member_number >= 0) && (target_channel == target_channel->window->current_channel) ) {
      set_members_list(target_channel);
    }
    glist_entry = g_slist_next(glist_entry);
  }

  /* TODO: will need a nick change message for each channel the person was on */
  /*       when we have multiple windows. */
/*   message = message_new(MT_CONSOLE, connection, irc_message.args[0]); */
/*   message->parts = g_slist_append(message->parts,  */
/* 				 message_part_new(&colour_yellow, NULL, NULL, irc_message.nick)); */
/*   message->parts = g_slist_append(message->parts, */
/* 				 message_part_new(&colour_white, NULL, NULL, " is now known as ")); */
/*   message->parts = g_slist_append(message->parts, */
/* 				 message_part_new(&colour_yellow, NULL, NULL, irc_message.args[0])); */

#ifdef HAVE_TCL
  /* davidw TCL */
  sprintf(cl, "onnick %s %s %s", irc_message.nick,
	  irc_message.address, irc_message.args[0]);
  Tcl_Eval(interp, cl);
  /* fprintf(stderr, "%s\n", interp->result); */
#endif

  bs_function_leave();
  return;
}

void recv_topic(Connection *connection)
{
  Message *message;
  char buff[BUFFLEN+20];
  ChannelInfo *target_channel;

  bs_function_enter();

  if (!connection) {
    bs_function_leave();
    return;
  }

  if ( !irc_message.args || !irc_message.args[0] || !irc_message.args[1] ) {
    message = message_new(MT_CHANNEL, connection, NULL);
    message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_red, NULL, NULL, "Malformed TOPIC from server"));
    bs_function_leave();
    return;
  }

  if ( !(target_channel = find_channel(connection->channels, irc_message.args[0]))) {
    bs_function_leave();
    return;
  }

  if ( !target_channel->topic ) {
    target_channel->topic = g_strdup(irc_message.args[1]);
    if (target_channel == target_channel->window->current_channel) {
      g_snprintf(buff, BUFFLEN+20, "Bezerk: %s - %s", target_channel->name, target_channel->topic);
      gtk_window_set_title(GTK_WINDOW(target_channel->window->window), buff);
    }
  } else if ( strcmp(target_channel->topic, irc_message.args[1]) ) {
    g_free(target_channel->topic);
    target_channel->topic = g_strdup(irc_message.args[1]);
    if (target_channel == target_channel->window->current_channel) {
      g_snprintf(buff, BUFFLEN+20, "Bezerk: %s - %s", target_channel->name, target_channel->topic);
      gtk_window_set_title(GTK_WINDOW(target_channel->window->window), buff);
    }
  }

  message = message_new(MT_CHANNEL, connection, irc_message.args[0]);
  message->parts = g_slist_append(message->parts, 
				  message_part_new(&colour_blue, NULL, NULL, "Topic on "));
  message->parts = g_slist_append(message->parts, 
				  message_part_new(&colour_blue, NULL, NULL, irc_message.args[0]));
  message->parts = g_slist_append(message->parts,
				  message_part_new(&colour_blue, NULL, NULL, ": "));
  message->parts = message_part_parse_new(message->parts, &colour_white, 
					  irc_message.args[1]);

#ifdef HAVE_TCL
  sprintf(cl, "ontopic %s %s %s { %s }", irc_message.nick,  irc_message.address,
	  irc_message.args[0], irc_message.args[1]);
  Tcl_Eval(interp, cl);
  /* fprintf(stderr, "%s\n", interp->result); */
#endif
 
  bs_function_leave();
  return;
}

void recv_kick(Connection *connection)
{
  Message *message;
  ChannelInfo *target_channel;
  int position;

  bs_function_enter();

  if (!connection) {
    bs_function_leave();
    return;
  }

  if ( !irc_message.args || !irc_message.args[0] || !irc_message.args[1]) {
    message = message_new(MT_CHANNEL, connection, NULL);
    message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_red, NULL, NULL, "Malformed KICK from server"));
    bs_function_leave();
    return;
  }

  if ( !(target_channel = find_channel(connection->channels, irc_message.args[0])) ) {
    /* We're not on this channel - shouldn't happen */
    message = message_new(MT_CHANNEL, connection, NULL);
    message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_red, NULL, NULL, "Malformed KICK from server"));    
    bs_function_leave();
    return;
  }

  message = message_new(MT_CHANNEL, connection, NULL);
  if (!strcasecmp(connection->nick, irc_message.args[1])) {
    /* User has been kicked from channel */
    /* TODO: optionally rejoin channel */
    connection->channels = remove_channel(connection->channels, irc_message.args[0]);
    message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_yellow, NULL, NULL, "You were kicked from "));
  } else {
    position = remove_channel_member(target_channel, irc_message.args[1]);
    if (target_channel == target_channel->window->current_channel) {
      set_members_list(target_channel);
    }
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_yellow, NULL, NULL, irc_message.args[1]));
    message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_white, NULL, NULL, " was kicked from "));
  }

  message->parts = g_slist_append(message->parts, 
				  message_part_new(&colour_yellow,  NULL, NULL, irc_message.args[0]));
  message->parts = g_slist_append(message->parts,
				  message_part_new(&colour_yellow, NULL, NULL, " by "));
  message->parts = g_slist_append(message->parts, 
				  message_part_new(&colour_yellow,  NULL, NULL, irc_message.nick));
  if (irc_message.args[2]) {
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_white,  NULL, NULL, ": "));
    message->parts = message_part_parse_new(message->parts, &colour_white,  
					    irc_message.args[2]);
  }

#ifdef HAVE_TCL
  /* davidw TCL */
  sprintf(cl, "onkick %s %s %s %s { %s }", irc_message.nick,  irc_message.address,
	  irc_message.args[0], irc_message.args[1], irc_message.args[2]);
  Tcl_Eval(interp, cl);
  /* fprintf(stderr, "%s\n", interp->result); */
#endif

  bs_function_leave();
  return;
}

DIALOG_CALLBACK(invite_send_join)
{
  char *channel_name = data;

  bs_function_enter();

  send_join("JOIN", channel_name, NULL); 
  g_free(channel_name);

  bs_function_leave();
  return;
}

void recv_invite(Connection *connection)
{
  char buff[BUFFLEN];

  bs_function_enter();

  if (!connection) {
    bs_function_leave();
    return;
  }

  if ( !irc_message.nick || !irc_message.args || !irc_message.args[0] || !irc_message.args[1] ) {
    Message *message;
    message = message_new(MT_CHANNEL, connection, NULL);
    message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_red, NULL, NULL, "Malformed INVITE from server"));
    bs_function_leave();
    return;
  }

  if (find_channel(connection->channels, irc_message.args[1])) {
    /* Already joined or joining channel */
    bs_function_leave();
    return;
  }

  g_snprintf(buff, BUFFLEN, "%s has invited you to join channel %s", irc_message.nick, irc_message.args[1]);

  choice_dialog(g_strdup(irc_message.args[1]),
		"Channel Invite",
		buff,
		"Join",
		"Don't Join",
		invite_send_join,
		NULL);

#ifdef HAVE_TCL
  /* davidw TCL */
  sprintf(cl, "oninvite %s %s %s", irc_message.nick,  irc_message.address, irc_message.args[1]);
  Tcl_Eval(interp, cl);
  /* fprintf(stderr, "%s\n", interp->result); */
#endif

  bs_function_leave();
  return;
}

void recv_wallops(Connection *connection)
{
  Message *message;

  bs_function_enter();

  if (!connection) {
    bs_function_leave();
    return;
  }

  if ( !irc_message.args || !irc_message.args[0] ) {
    message = message_new(MT_CHANNEL, connection, NULL);
    message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_red, NULL, NULL, "Malformed WALLOPS from server"));
    bs_function_leave();
    return;
  }

  message = message_new(MT_CONSOLE, connection, irc_message.nick);
  message->parts = g_slist_append(message->parts, 
				  message_part_new(&colour_red, NULL, NULL, "WALLOPS("));
  if (irc_message.nick) {
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_white, NULL, NULL, irc_message.nick));
  }
  message->parts = g_slist_append(message->parts,
				  message_part_new(&colour_red, NULL, NULL, "): "));
  message->parts = message_part_parse_new(message->parts, &colour_white, 
					 irc_message.args[0]);

  bs_function_leave();
  return;
}

void recv_kill(Connection *connection)
{
  Message *message;

  bs_function_enter();

  if (!connection) {
    bs_function_leave();
    return;
  }

  if ( !irc_message.args || !irc_message.args[0] || !irc_message.args[1] ) {
    message = message_new(MT_CHANNEL, connection, NULL);
    message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_red, NULL, NULL, "Malformed KILL from server"));
    bs_function_leave();
    return;
  }

  message = message_new(MT_CONSOLE, connection, irc_message.args[0]);
  message->parts = g_slist_append(message->parts, 
				  message_part_new(&colour_red, NULL, NULL, "KILL: "));
  message->parts = g_slist_append(message->parts, 
				  message_part_new(&colour_white, NULL, NULL,  irc_message.args[1]));

#ifdef HAVE_TCL
  /* davidw TCL */
  sprintf(cl, "onkill %s %s %s { %s }", irc_message.nick,  irc_message.address,
	  irc_message.args[0], irc_message.args[1]);
  Tcl_Eval(interp, cl);
  fprintf(stderr, "%s\n", interp->result);
#endif

  bs_function_leave();
  return;
}

void recv_error(Connection *connection)
{
  Message *message;

  bs_function_enter();

  if (!connection) {
    bs_function_leave();
    return;
  }

  if ( !irc_message.args || !irc_message.args[0] ) {
    message = message_new(MT_CHANNEL, connection, NULL);
    message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_red, NULL, NULL, "Malformed ERROR from server"));
    bs_function_leave();
    return;
  }

  message = message_new(MT_CONSOLE, connection, irc_message.args[0]);
  message->parts = g_slist_append(message->parts, 
				  message_part_new(&colour_red, NULL, NULL, "ERROR: "));
  message->parts = g_slist_append(message->parts, 
				  message_part_new(&colour_white,  NULL, NULL, irc_message.args[0]));

  bs_function_leave();
  return;
}

void process_command(Connection *connection)
{

  bs_function_enter();


  /* remove when not testing */
#ifdef DEBUG_ECHO
  if (irc_message.nick == NULL) {
    bs_function_leave();
    return;
  }
#endif

  if ( !connection ) {
    bs_function_leave();
    return;
  }
    
  if ( !irc_message.command || !irc_message.command[0]) {
    Message *message;
    message = message_new(MT_CHANNEL, connection, NULL);
    message->parts = g_slist_append(message->parts,
			       message_part_new(&colour_red, NULL, NULL, "Malformed command from server"));
    bs_function_leave();
    return;
  }

  /* Check command */
  if (!strcmp("PRIVMSG", irc_message.command))       recv_privmsg(connection);
  else if (!strcmp("JOIN", irc_message.command))     recv_join(connection);
  else if (!strcmp("PART", irc_message.command))     recv_part(connection);
  else if (!strcmp("MODE", irc_message.command))     recv_mode(connection);
  else if (!strcmp("QUIT", irc_message.command))     recv_quit(connection);
  else if (!strcmp("NOTICE", irc_message.command))   recv_notice(connection);
  else if (!strcmp("NICK", irc_message.command))     recv_nick(connection);
  else if (!strcmp("TOPIC", irc_message.command))    recv_topic(connection);
  else if (!strcmp("PING", irc_message.command))     recv_ping(connection);
  else if (!strcmp("PONG", irc_message.command))     recv_pong(connection);
  else if (!strcmp("KICK", irc_message.command))     recv_kick(connection);
  else if (!strcmp("INVITE", irc_message.command))   recv_invite(connection);
  else if (!strcmp("WALLOPS", irc_message.command))  recv_wallops(connection);
  else if (!strcmp("KILL", irc_message.command))     recv_kill(connection);
  else if (!strcmp("ERROR", irc_message.command))    recv_error(connection);

  bs_function_leave();
  return;
}
