/*
 *  dmachinemon / a distributed machine monitor by dancer.
 *  Copyright (C) 2001 Junichi Uekawa
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * libp2papplication - application in P2P, doing things.
 *
 *$Id: dmachinemon-libp2papplication.c,v 1.18 2002/01/10 08:36:52 dancer Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include "dmachinemon/dmachinemon.h"

static int nodeport;
static int clientport;

#define malloc_with_error(A,BYTES,ERROR) if (NULL==((A)=malloc(BYTES)))\
{fprintf(stderr, "out of memory in allocating %i bytes\n", BYTES);return ERROR ;}

#define LOCALHOST "localhost"

/**@name P2P application framework
   This is a group of functions that can be used for building 
   P2P applications using dmachinemon framework.
   These functions are there to facilitate development of such programs.
 */
//@{

/**
   Initialization for dmachinemon P2P interface.
   Must be called on application startup.
   The argc, and argv arguments passed to 
   main function should be passed on to here.

   @return 1 on error, 0 on success.
 */
int dm_Init (int * ac /// &argc
	     , char *** av /// &argv
	     )
{
  int fd;
  dm_commandoption p2pcdat;
  
  dmachinemon_parse_options(*ac, *av, &p2pcdat);

  nodeport = atoi(p2pcdat.port_nodes);
  clientport = atoi(p2pcdat.port_client);
  
  fd = dm_connect_to_host(LOCALHOST, nodeport);
  if (fd == -1)
    {
      fprintf (stderr, "dm_Init-error: error ... cannot link to %s.\n", LOCALHOST);
      return 1;
    }

  dm_obtain_route (fd, 0, NULL);
  close (fd);

  /* debug. */
  if (0)
    {
      char *s;
      s = dm_get_route();
      fprintf(stderr, "DEBUG: Route is :[%s] \n", s);
      if (s) free (s);
    }

  return 0;
}

/**
   Deinitialization for dmachinemon P2P interface.
   Must be called on application termination.
   It is currently a dummy application.

   @return 1 on error, 0 on success.
 */
int dm_Finalize(void)
{
  return 0;
}

/**
   Function to send information to the immediate uplink,
   to be gathered using \Ref{dm_gatherinfo}.

   Other nodes will be able to see the information 
   sent with the TAG, for nodes which are connected to the 
   same uplink, and all the children of the nodes which are
   connected to the same uplink.

   It takes some time before the information is sent to uplink,
   namely the --sleep parameter given to dmachinemon-servent
   determines it.

   Also this function should fail if dmachinemon-servent
   has terminated for some reason, maybe with --Dieonload
   option.

   Unless --layers is specified when invoking dmachinemon-servent,
   the information that has been given with sendinfo will be available on
   all of the uplinks, with gatherinfo.


   @return 1 on error, 0 on success.
 */
int dm_sendinfo (const char * TAG, /// TAG to identify information
		 const char * string /// The information string.
		 )
{
  int fd = dm_connect_to_host(LOCALHOST, nodeport);
  FILE*f ;
  char * this_hostname = dm_gethostname_versatile ();
  if (fd == -1)
    {
      fprintf (stderr, "Error: error ... cannot link to %s.\n", LOCALHOST);
      return 1;
    }

  dm_obtain_route (fd, 0, NULL);
  f = fdopen (fd, "w");
  if (!f)
    {
      fprintf (stderr, "Error: cannot fdopen to [%s] although open succeeded.\n", LOCALHOST);
      return 1;
    }
 
  fprintf (f, 
	   ":%s:\n"
	   SEEN_BY ": localhost\n"
	   "%s: %s\n", this_hostname, TAG, string);
  fclose (f);

  return 0;
}


dm_machinelist_information myinfo = {NULL, PTHREAD_MUTEX_INITIALIZER};

/**
   Gather information with the specified tag from the immediate uplink.
   Other nodes have already sent this information using 
   \Ref{dm_sendinfo}.
   Information obtained with this function should be freed with
   \Ref{dm_freeinfo}.

   There is a notable delay between sendinfo and actually receiving
   information via gatherinfo.
   
   @see dm_freeinfo
   @see dm_sendinfo
   @return array of character pointers, or NULL on failure.
 */
char ** dm_gatherinfo (const char * TAG /// Name of tag to search for
		       ) 
{
  char ** buf = NULL;
  char * grandparentlink = dm_obtain_grandparent_host();
  int fd;
  FILE * f;  
  int membercount = 0 ;
  int currentcount = 0;
  char * curinfo ;
  
  dm_machinelist * thismachine = NULL;

  if (!grandparentlink)
    {
      fprintf(stderr, "DEBUG: grandparent link has not been established yet\n");
      return NULL;
    }
  
  fd = dm_connect_to_host (grandparentlink, clientport);
  f= (fd!=-1)?fdopen(fd,"r"):NULL;

  if (!f)
    {
      fprintf (stderr, "error: cannot link to grandparent [%s].\n", grandparentlink);
      if (fd == -1)
	fprintf (stderr, "diagnosis: open failed.\n" );
      return NULL;
    }

  dm_process_nodes_input (f, &myinfo, NULL, NULL);
  thismachine = myinfo.machinedb;
  membercount = dm_count_nodes_number (&myinfo);
  malloc_with_error (buf, sizeof (char*) * membercount, NULL); 
  
  while (thismachine)
    {
				/* gather the info... */
      if (0!=(curinfo=dm_get_value_text (thismachine, TAG)))
	{
	  (*(buf + currentcount++)) = strdup(curinfo);
	}
      thismachine = thismachine -> next;
    }
  *(buf + currentcount) = 0;
  fclose(f);
  return buf;
}

/**
   Free the information obtained with \Ref{dm_gatherinfo}.

   @see dm_gatherinfo
 */
void
dm_freeinfo(char ** m)
{
  int i= 0;
  while (m[i])
      free (m[i++]);
  free (m);
}

//@}


