/* glppfi.h */

/*----------------------------------------------------------------------
-- Copyright (C) 2000, 2001, 2002 Andrew Makhorin <mao@mai2.rcnet.ru>,
--               Department for Applied Informatics, Moscow Aviation
--               Institute, Moscow, Russia. All rights reserved.
--
-- This file is a part of GLPK (GNU Linear Programming Kit).
--
-- GLPK is free software; you can redistribute it and/or modify it
-- under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2, or (at your option)
-- any later version.
--
-- GLPK is distributed in the hope that it will be useful, but WITHOUT
-- ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
-- or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public
-- License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with GLPK; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
-- 02111-1307, USA.
----------------------------------------------------------------------*/

#ifndef _GLPPFI_H
#define _GLPPFI_H

#include "glpeta.h"
#include "glpgel.h"

#define create_pfi            _glp_create_pfi
#define build_pfi             _glp_build_pfi
#define pfi_ftran             _glp_pfi_ftran
#define pfi_btran             _glp_pfi_btran
#define update_pfi            _glp_update_pfi
#define delete_pfi            _glp_delete_pfi

/* The structure PFI defines Product Form of the Inverse which is the
-- following representation of the current basis matrix B:
--
--    B = B0 * H,
--
-- where B0 is the initial basis matrix, H is the eta-matrix.
--
-- The initial basis matrix B0 has the form of LU-factorization:
--
--    B0 = P * L * U * Q,
--
-- where P and Q are permutation matrices, L is a lower triangular
-- matrix with unit diagonal, U is an upper triangular matrix.
--
-- The eta-matrix H has the form of eta-file:
--
--    H = H(1) * H(2) * ... * H(t),
--
-- where H(1), H(2), ..., H(t) are elementary matrix multipliers or
-- eta-terms, t is the total number of terms (size of eta-file). */

typedef struct PFI PFI;

struct PFI
{     /* product form of the inverse (PFI) */
      int m;
      /* order of basis matrix */
      LU *lu;
      /* initial basis matrix B0 in the form of LU-factorization */
      HFILE *eta;
      /* eta-matrix H in the form of eta-file */
      double *col;
      /* transformed column corresponding to that non-basis variable
         which has been chosen to enter the basis */
      int flag;
      /* transformed column flag:
         0 - column is not prepared yet
         1 - column is ready for updating */
      double *work; /* double work[1+m]; */
      /* auxiliary array */
};

extern PFI *create_pfi(int m);
/* create PFI */

extern int build_pfi(PFI *pfi, MAT *A, int indb[]);
/* build PFI for given basis matrix */

extern double *pfi_ftran(PFI *pfi, double z[], int save);
/* perform forward transformation (FTRAN) using PFI */

extern double *pfi_btran(PFI *pfi, double z[]);
/* perform backward transformation (BTRAN) using PFI */

extern int update_pfi(PFI *pfi, int p);
/* update PFI for adjacent basis matrix */

extern void delete_pfi(PFI *pfi);
/* delete PFI */

#endif

/* eof */
