/* -*- Mode: C; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/**
 * ggv-postscript-view.c
 *
 * Author:  Jaka Mocnik  <jaka@gnu.org>
 *
 * Copyright (c) 2001 Free Software Foundation
 */

#include <config.h>
#include <stdio.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkmarshal.h>
#include <gtk/gtktypeutils.h>
#include <gconf/gconf-client.h>

#include <gnome.h>

#include <ggv-postscript-view.h>
#include <ggv-control.h>

#include <gtkgs.h>
#include <gsdefaults.h>
#include <ggvutils.h>

#include <bonobo.h>

struct _GgvPostscriptViewPrivate {
	GtkWidget *gs;
	GtkObject *hadj, *vadj;

	GNOME_GGV_Size def_size;
	GNOME_GGV_Orientation def_orientation;

	gint magstep;
	gboolean pan;
	gdouble prev_x, prev_y;
	gchar *tmp_name;

	BonoboPropertyBag     *property_bag;
	BonoboPropertyControl *property_control;

	BonoboUIComponent     *uic;

	gboolean               zoom_fit;
};

struct _GgvPostscriptViewClassPrivate {
	BonoboUINode   *size_menus;
	gchar          **size_paths;
};

enum {
	LAST_SIGNAL
};

enum {
	PROP_ANTIALIASING,
	PROP_DEFAULT_ORIENTATION,
	PROP_OVERRIDE_ORIENTATION,
	PROP_DEFAULT_SIZE,
	PROP_OVERRIDE_SIZE,
	PROP_RESPECT_EOF,
	PROP_WATCH_FILE
};

enum {
	PROP_CONTROL_TITLE
};

static GdkCursor *pan_cursor = NULL;

static guint ggv_postscript_view_signals [LAST_SIGNAL];

#define PARENT_TYPE BONOBO_X_OBJECT_TYPE

static BonoboObjectClass *ggv_postscript_view_parent_class;

static gchar *orientation_paths[] = {
	"/commands/OverrideOrientation",
	"/commands/OrientationPortrait",
	"/commands/OrientationLandscape",
	"/commands/OrientationUpsideDown",
	"/commands/OrientationSeascape",
	NULL
};

/*
 * Loads an postscript document from a Bonobo_Stream
 */
static void
load_ps_from_stream (BonoboPersistStream *ps,
					 Bonobo_Stream stream,
					 Bonobo_Persist_ContentType type,
					 void *data,
					 CORBA_Environment *ev)
{
	GgvPostscriptView *ps_view;
	Bonobo_Stream_iobuf  *buffer;
	CORBA_long bytes_read, bytes_written;

	g_return_if_fail (data != NULL);
	g_return_if_fail (GGV_IS_POSTSCRIPT_VIEW (data));

	ps_view = GGV_POSTSCRIPT_VIEW(data);

	/* copy stream to a tmp file */
	if(ps_view->priv->tmp_name != NULL) {
		unlink(ps_view->priv->tmp_name);
	}
	else {
		ps_view->priv->tmp_name = g_malloc(256);
	}
	tmpnam(ps_view->priv->tmp_name);
	buffer = Bonobo_Stream_iobuf__alloc();
	Bonobo_Stream_copyTo(stream, ps_view->priv->tmp_name,
						 -1, &bytes_read, &bytes_written, ev);
	gtk_gs_load(GTK_GS(ps_view->priv->gs), ps_view->priv->tmp_name);
	gtk_gs_set_page_size(GTK_GS(ps_view->priv->gs), -1, 0);
	gtk_gs_goto_page(GTK_GS(ps_view->priv->gs), 0);
}

/* 
 * handlers for mouse actions
 */
static void
view_button_press_cb(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
	GgvPostscriptView *ps_view;

	g_return_if_fail(data != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(data));

	ps_view = GGV_POSTSCRIPT_VIEW(data);

	if(event->button == 1 && !ps_view->priv->pan) {
		gint wx = 0, wy = 0;
			
		gdk_window_get_pointer(widget->window, &wx, &wy, NULL);
			
		gtk_gs_start_scroll(GTK_GS(ps_view->priv->gs));

		ps_view->priv->pan = TRUE;
		if(pan_cursor == NULL)
			pan_cursor = gdk_cursor_new(GDK_FLEUR);
			
		gtk_grab_add(widget);
		gdk_pointer_grab(widget->window, FALSE,
						 GDK_POINTER_MOTION_MASK |
						 GDK_BUTTON_RELEASE_MASK, NULL,
						 pan_cursor, GDK_CURRENT_TIME);
		ps_view->priv->prev_x = wx;
		ps_view->priv->prev_y = wy;
	}
}

static void
view_button_release_cb(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
	GgvPostscriptView *ps_view;

	g_return_if_fail(data != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(data));

	ps_view = GGV_POSTSCRIPT_VIEW(data);

	if(event->button == 1 && ps_view->priv->pan) {
		ps_view->priv->pan = FALSE;
		gdk_pointer_ungrab(GDK_CURRENT_TIME);
		gtk_grab_remove(widget);
		gtk_gs_end_scroll(GTK_GS(ps_view->priv->gs));
	}
}

static void
view_motion_cb(GtkWidget *widget, GdkEventMotion *event, gpointer data)
{
	GgvPostscriptView *ps_view;

	g_return_if_fail(data != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(data));

	ps_view = GGV_POSTSCRIPT_VIEW(data);

	if(ps_view->priv->pan) {
		gtk_gs_scroll(GTK_GS(ps_view->priv->gs),
					  -event->x + ps_view->priv->prev_x,
					  -event->y + ps_view->priv->prev_y);
		ps_view->priv->prev_x = event->x;
		ps_view->priv->prev_y = event->y;
	}
}


/*
 * Loads an postscript document from a Bonobo_File
 */
static gint
load_ps_from_file (BonoboPersistFile *pf, const CORBA_char *filename,
				   CORBA_Environment *ev, void *data)
{
	GgvPostscriptView *ps_view;

	g_return_val_if_fail (data != NULL, -1);
	g_return_val_if_fail (GGV_IS_POSTSCRIPT_VIEW (data), -1);

	ps_view = GGV_POSTSCRIPT_VIEW (data);

	gtk_gs_load(GTK_GS(ps_view->priv->gs), filename);

	return 0;
}

static void
listener_Orientation_cb (BonoboUIComponent *uic, const char *path,
						 Bonobo_UIComponent_EventType type, const char *state,
						 gpointer user_data)
{
	BonoboArg *arg;
	GgvPostscriptView *ps_view;
	GNOME_GGV_Orientation orientation;

	g_return_if_fail(user_data != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW (user_data));

	if(type != Bonobo_UIComponent_STATE_CHANGED)
		return;

	if(!state || !atoi(state))
		return;

	ps_view = GGV_POSTSCRIPT_VIEW(user_data);

	if(!strcmp(path, "OrientationPortrait"))
		orientation = GNOME_GGV_ORIENTATION_PORTRAIT;
	else if(!strcmp(path, "OrientationLandscape"))
		orientation = GNOME_GGV_ORIENTATION_LANDSCAPE;
	else if(!strcmp(path, "OrientationUpsideDown"))
		orientation = GNOME_GGV_ORIENTATION_UPSIDE_DOWN;
	else if(!strcmp(path, "OrientationSeascape"))
		orientation = GNOME_GGV_ORIENTATION_SEASCAPE;
	else {
		g_warning("Unknown orientation `%s'", path);
		return;
	}

	arg = bonobo_arg_new(TC_GNOME_GGV_Orientation);
	BONOBO_ARG_SET_GENERAL(arg, orientation, TC_GNOME_GGV_Orientation,
						   GNOME_GGV_Orientation, NULL);

	bonobo_property_bag_set_value(ps_view->priv->property_bag,
								   "orientation", arg, NULL);

	bonobo_arg_release(arg);
}

static void
listener_Toggle_cb (BonoboUIComponent *uic, const char *path,
					Bonobo_UIComponent_EventType type, const char *state,
					gpointer user_data)
{
	BonoboArg *arg;
	GgvPostscriptView *ps_view;
	CORBA_boolean flag;
	gchar *prop;
	int i;

	static gchar *properties[][2] = {
		{ "OverrideOrientation", "override_orientation" },
		{ "OverrideSize", "override_size" },
		{ "RespectEOF", "respect_eof" },
		{ "Antialiasing", "antialiasing" },
		{ "WatchFile", "watch_file" },
		{ NULL, NULL }
	};

	g_return_if_fail(user_data != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW (user_data));

	if(type != Bonobo_UIComponent_STATE_CHANGED)
		return;

	if(!state)
		return;
	
	ps_view = GGV_POSTSCRIPT_VIEW(user_data);

	if(atoi(state))
		flag = CORBA_TRUE;
	else
		flag = CORBA_FALSE;

	prop = NULL;
	for(i = 0; properties[i][0]; i++) {
		if(!strcmp(path, properties[i][0])) {
			prop = properties[i][1];
			break;
		}
	}

	g_assert(prop != NULL);

	arg = bonobo_arg_new(TC_CORBA_boolean);
	BONOBO_ARG_SET_GENERAL(arg, flag, TC_CORBA_boolean,
						   CORBA_boolean, NULL);

	bonobo_property_bag_set_value(ps_view->priv->property_bag,
								  prop, arg, NULL);

	bonobo_arg_release(arg);
}

	
static void
listener_Size_cb(BonoboUIComponent *uic, const char *path,
				  Bonobo_UIComponent_EventType type, const char *state,
				  gpointer user_data)
{
	GgvPostscriptView *ps_view;
	GNOME_GGV_Size size;
	BonoboArg *arg;

	g_return_if_fail(user_data != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(user_data));

	if(type != Bonobo_UIComponent_STATE_CHANGED)
		return;

	if(!state || !atoi(state))
		return;

	ps_view = GGV_POSTSCRIPT_VIEW(user_data);

	size = CORBA_string_dup(path);

	arg = bonobo_arg_new(TC_GNOME_GGV_Size);
	BONOBO_ARG_SET_GENERAL(arg, size, TC_GNOME_GGV_Size,
						   GNOME_GGV_Size, NULL);

	bonobo_property_bag_set_value(ps_view->priv->property_bag,
								  "size", arg, NULL);

	bonobo_arg_release(arg);
}

static void
set_page_item_sensitivity(GgvPostscriptView *ps_view)
{
	GtkGS *gs = GTK_GS(ps_view->priv->gs);
	gchar *prop_val;

	prop_val = (gtk_gs_get_current_page(gs) >= gtk_gs_get_page_count(gs) - 1)?
		"0":"1";
	bonobo_ui_component_set_prop(ps_view->priv->uic, "/commands/NextPage",
								 "sensitive", prop_val, NULL);
	prop_val = (gtk_gs_get_current_page(gs) <= 0 || !gs->structured_doc)?
		"0":"1";
	bonobo_ui_component_set_prop(ps_view->priv->uic, "/commands/PrevPage",
								 "sensitive", prop_val, NULL);
}


static void
verb_NextPage_cb(BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
	GgvPostscriptView *ps_view;
	GtkGS *gs;

	g_return_if_fail (user_data != NULL);
	g_return_if_fail (GGV_IS_POSTSCRIPT_VIEW (user_data));

	ps_view = GGV_POSTSCRIPT_VIEW (user_data);
	gs = GTK_GS(ps_view->priv->gs);

	gtk_gs_goto_page(gs, gtk_gs_get_current_page(gs) + 1);

	set_page_item_sensitivity(ps_view);
}

static void
verb_PrevPage_cb(BonoboUIComponent *uic, gpointer user_data, const char *cname)
{
	GgvPostscriptView *ps_view;

	g_return_if_fail (user_data != NULL);
	g_return_if_fail (GGV_IS_POSTSCRIPT_VIEW (user_data));

	ps_view = GGV_POSTSCRIPT_VIEW (user_data);

	gtk_gs_goto_page(GTK_GS(ps_view->priv->gs),
					 GTK_GS(ps_view->priv->gs)->current_page - 1);

	set_page_item_sensitivity(ps_view);
}

static void
sync_orientation_items(GgvPostscriptView *ps_view)
{
	CORBA_Environment ev;
	gint i;
	gboolean val, orient_state;

	CORBA_exception_init(&ev);

	val = gtk_gs_get_override_orientation(GTK_GS(ps_view->priv->gs));
	bonobo_ui_component_set_prop(ps_view->priv->uic, orientation_paths[0],
								 "state", val?"1":"0", NULL);
	for(i = 1; orientation_paths[i] != NULL; i++) {
		bonobo_ui_component_set_prop(ps_view->priv->uic, orientation_paths[i],
									 "sensitive", val?"1":"0", NULL);
		orient_state = i - 1 == gtk_gs_get_default_orientation(GTK_GS(ps_view->priv->gs));
		bonobo_ui_component_set_prop(ps_view->priv->uic, orientation_paths[i],
									 "state", orient_state?"1":"0", NULL);
	}
}

static void
sync_size_items(GgvPostscriptView *ps_view)
{
	CORBA_Environment ev;
	gint i;
	gboolean val, size_state;
	gchar **paths;

	CORBA_exception_init(&ev);

	paths = GGV_POSTSCRIPT_VIEW_CLASS(GTK_OBJECT(ps_view)->klass)->priv->size_paths;
	val = gtk_gs_get_override_size(GTK_GS(ps_view->priv->gs));
	bonobo_ui_component_set_prop(ps_view->priv->uic, paths[0],
								 "state", val?"1":"0", NULL);
	for(i = 1; paths[i] != NULL; i++) {
		bonobo_ui_component_set_prop(ps_view->priv->uic, paths[i],
									 "sensitive", val?"1":"0", NULL);
		size_state = i - 1 == gtk_gs_get_default_size(GTK_GS(ps_view->priv->gs));
		bonobo_ui_component_set_prop(ps_view->priv->uic, paths[i],
									 "state", size_state?"1":"0", NULL);
	}
}

static void
ggv_postscript_view_create_ui(GgvPostscriptView *ps_view)
{
	static BonoboUIVerb ggv_postscript_view_verbs[] = {
		BONOBO_UI_VERB ("NextPage", verb_NextPage_cb),
		BONOBO_UI_VERB ("PrevPage", verb_PrevPage_cb),
		BONOBO_UI_VERB_END
	};
	BonoboUINode *new_size_menus;
	BonoboUINode *ui_root;
	GgvPostscriptViewClass *klass = GGV_POSTSCRIPT_VIEW_CLASS(GTK_OBJECT(ps_view)->klass);
	gint i;

	g_return_if_fail(ps_view != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view));

	/* Set up the UI from an XML file. */
#if 1
	ui_root = bonobo_ui_util_new_ui(ps_view->priv->uic, 
									DATADIR "/gnome/ui/ggv-postscript-view-ui.xml",
									DATADIR,
									"GgvPostscriptView");
	bonobo_ui_util_translate_ui(ui_root);
	bonobo_ui_component_set_tree(ps_view->priv->uic, "/", ui_root, NULL);
#else
	bonobo_ui_util_set_ui(ps_view->priv->uic, DATADIR,
						  "ggv-postscript-view-ui.xml", 
						  "GgvPostscriptView");
#endif

	new_size_menus = bonobo_ui_node_copy(klass->priv->size_menus, TRUE);
	if(new_size_menus)
		bonobo_ui_component_set_tree(ps_view->priv->uic, "/menu/View/Size",
									 new_size_menus, NULL);

	bonobo_ui_component_add_listener(ps_view->priv->uic, "OrientationPortrait",
									 listener_Orientation_cb, ps_view);
	bonobo_ui_component_add_listener(ps_view->priv->uic, "OrientationLandscape",
									 listener_Orientation_cb, ps_view);
	bonobo_ui_component_add_listener(ps_view->priv->uic, "OrientationUpsideDown",
									 listener_Orientation_cb, ps_view);
	bonobo_ui_component_add_listener(ps_view->priv->uic, "OrientationSeascape",
									 listener_Orientation_cb, ps_view);
	bonobo_ui_component_add_listener(ps_view->priv->uic, "OverrideOrientation",
									 listener_Toggle_cb, ps_view);
	for(i = 1; klass->priv->size_paths[i] != NULL; i++) {
		bonobo_ui_component_add_listener(ps_view->priv->uic,
										 klass->priv->size_paths[i] + strlen("/commands/"),
										 listener_Size_cb, ps_view);
	}
	bonobo_ui_component_add_listener(ps_view->priv->uic, "OverrideSize",
									 listener_Toggle_cb, ps_view);

	bonobo_ui_component_add_verb_list_with_data(ps_view->priv->uic,
												ggv_postscript_view_verbs,
												ps_view);

	set_page_item_sensitivity(ps_view);
	sync_orientation_items(ps_view);
	sync_size_items(ps_view);
}

static void
ggv_postscript_view_get_prop(BonoboPropertyBag *bag,
							 BonoboArg         *arg,
							 guint              arg_id,
							 CORBA_Environment *ev,
							 gpointer           user_data)
{
	GgvPostscriptView *ps_view;
	gchar *size;

	g_return_if_fail(user_data != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(user_data));

	ps_view = GGV_POSTSCRIPT_VIEW(user_data);

	switch(arg_id) {
	case PROP_DEFAULT_ORIENTATION: {
		g_assert(arg->_type == TC_GNOME_GGV_Orientation);

		*(GNOME_GGV_Orientation *)arg->_value = ps_view->priv->def_orientation;
		break;
	}
	case PROP_OVERRIDE_ORIENTATION: {
		g_assert(arg->_type == TC_CORBA_boolean);

		*(CORBA_boolean *)arg->_value =
			gtk_gs_get_override_orientation(GTK_GS(ps_view->priv->gs))?
			CORBA_TRUE:CORBA_FALSE;
		break;
	}
	case PROP_DEFAULT_SIZE: {
		g_assert(arg->_type == TC_GNOME_GGV_Size);

		size = gtk_gs_defaults_get_paper_sizes()
			[gtk_gs_get_default_size(GTK_GS(ps_view->priv->gs))].name;

		*(GNOME_GGV_Size *)arg->_value =
			CORBA_string_dup(size);
		break;
	}
	case PROP_OVERRIDE_SIZE: {
		g_assert(arg->_type == TC_CORBA_boolean);

		*(CORBA_boolean *)arg->_value =
			gtk_gs_get_override_size(GTK_GS(ps_view->priv->gs))?
			CORBA_TRUE:CORBA_FALSE;
		break;
	}
	case PROP_RESPECT_EOF: {
		g_assert(arg->_type == TC_CORBA_boolean);

		*(CORBA_boolean *)arg->_value =
			gtk_gs_get_respect_eof(GTK_GS(ps_view->priv->gs))?
			CORBA_TRUE:CORBA_FALSE;
		break;
	}
	case PROP_ANTIALIASING: {
		g_assert(arg->_type == TC_CORBA_boolean);

		*(CORBA_boolean *)arg->_value =
			gtk_gs_get_antialiasing(GTK_GS(ps_view->priv->gs))?
			CORBA_TRUE:CORBA_FALSE;
		break;
	}
	case PROP_WATCH_FILE: {
		g_assert(arg->_type == TC_CORBA_boolean);

		*(CORBA_boolean *)arg->_value =
			gtk_gs_get_watch_file(GTK_GS(ps_view->priv->gs))?
			CORBA_TRUE:CORBA_FALSE;
		break;
	}
	default:
		g_assert_not_reached();
	}
}

static void
ggv_postscript_view_set_prop(BonoboPropertyBag *bag,
							 const BonoboArg   *arg,
							 guint              arg_id,
							 CORBA_Environment *ev,
							 gpointer           user_data)
{
	GgvPostscriptView *ps_view;
	GgvPostscriptViewClass *klass;

	g_return_if_fail(user_data != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(user_data));

	ps_view = GGV_POSTSCRIPT_VIEW(user_data);

	klass = GGV_POSTSCRIPT_VIEW_CLASS(GTK_OBJECT(ps_view)->klass);

	switch(arg_id) {
	case PROP_DEFAULT_ORIENTATION: {
		GtkGSOrientation orient = GTK_GS_ORIENTATION_PORTRAIT;

		g_assert(arg->_type == TC_GNOME_GGV_Orientation);

		ps_view->priv->def_orientation = *(GNOME_GGV_Orientation *) arg->_value;

		switch(ps_view->priv->def_orientation) {
		case GNOME_GGV_ORIENTATION_PORTRAIT:
			orient = GTK_GS_ORIENTATION_PORTRAIT;
			break;
		case GNOME_GGV_ORIENTATION_LANDSCAPE:
			orient = GTK_GS_ORIENTATION_LANDSCAPE;
			break;
		case GNOME_GGV_ORIENTATION_UPSIDE_DOWN:
			orient = GTK_GS_ORIENTATION_UPSIDEDOWN;
			break;
		case GNOME_GGV_ORIENTATION_SEASCAPE:
			orient = GTK_GS_ORIENTATION_SEASCAPE;
			break;
		default:
			g_assert_not_reached();
		}

		gtk_gs_set_default_orientation(GTK_GS(ps_view->priv->gs), orient);
		sync_orientation_items(ps_view);

		break;
	}
	case PROP_OVERRIDE_ORIENTATION: {
		 gtk_gs_set_override_orientation(GTK_GS(ps_view->priv->gs),
										 *(CORBA_boolean *)arg->_value);
		 sync_orientation_items(ps_view);
		 break;
	}
	case PROP_DEFAULT_SIZE: {
		gint size;
		gchar *size_name;

		g_assert(arg->_type == TC_GNOME_GGV_Size);
		
		size_name = *(GNOME_GGV_Size *)arg->_value;

		size = gtk_gs_get_size_index(size_name,
									   gtk_gs_defaults_get_paper_sizes());

		gtk_gs_set_default_size(GTK_GS(ps_view->priv->gs), size);
		sync_size_items(ps_view);

		break;
	}
	case PROP_OVERRIDE_SIZE: {
		 gtk_gs_set_override_size(GTK_GS(ps_view->priv->gs),
								   *(CORBA_boolean *)arg->_value);
		 sync_size_items(ps_view);
		 break;
	}
	case PROP_ANTIALIASING: {
		 gtk_gs_set_override_size(GTK_GS(ps_view->priv->gs),
								   *(CORBA_boolean *)arg->_value);
		 break;
	}
	case PROP_RESPECT_EOF: {
		 gtk_gs_set_respect_eof(GTK_GS(ps_view->priv->gs),
								*(CORBA_boolean *)arg->_value);
		 break;
	}
	case PROP_WATCH_FILE: {
		 gtk_gs_set_watch_file(GTK_GS(ps_view->priv->gs),
							   *(CORBA_boolean *)arg->_value);
		 break;
	}
	default:
		g_assert_not_reached();
	}
}
	
static Bonobo_Unknown
ggv_postscript_view_get_object(BonoboItemContainer *item_container,
							   CORBA_char          *item_name,
							   CORBA_boolean       only_if_exists,
							   CORBA_Environment   *ev,
							   GgvPostscriptView   *ps_view)
{
	Bonobo_Unknown corba_object;
	BonoboObject *object = NULL;
	GSList *params, *c;

	g_return_val_if_fail(ps_view != NULL, CORBA_OBJECT_NIL);
	g_return_val_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view), CORBA_OBJECT_NIL);

	g_message ("ggv_postscript_view_get_object: %d - %s",
			   only_if_exists, item_name);

	params = ggv_split_string (item_name, "!");
	for (c = params; c; c = c->next) {
		gchar *name = c->data;

		if ((!strcmp (name, "control") || !strcmp (name, "embeddable"))
		    && (object != NULL)) {
			g_message ("ggv_postscript_view_get_object: "
					   "can only return one kind of an Object");
			continue;
		}

		if (!strcmp (name, "control"))
			object = (BonoboObject *) ggv_control_new (ps_view);
#if 0
		else if (!strcmp (item_name, "embeddable"))
			object = (BonoboObject *) ggv_embeddable_new (image);
#endif
		else
			g_message ("ggv_postscript_view_get_object: "
					   "unknown parameter `%s'",
					   name);
	}

	g_slist_foreach (params, (GFunc) g_free, NULL);
	g_slist_free (params);

	if (object == NULL)
		return NULL;

	corba_object = bonobo_object_corba_objref (object);

	return bonobo_object_dup_ref (corba_object, ev);
}

BonoboObject *
ggv_postscript_view_add_interfaces (GgvPostscriptView *ps_view,
									BonoboObject *to_aggregate)
{
	BonoboPersistFile   *file;
	BonoboPersistStream *stream;
	BonoboItemContainer *item_container;
	
	g_return_val_if_fail (GGV_IS_POSTSCRIPT_VIEW (ps_view), NULL);
	g_return_val_if_fail (BONOBO_IS_OBJECT (to_aggregate), NULL);

	/* Interface Bonobo::PersistStream */
	stream = bonobo_persist_stream_new (load_ps_from_stream, 
										NULL, NULL, NULL, ps_view);
	if (!stream) {
		bonobo_object_unref (BONOBO_OBJECT (to_aggregate));
		return NULL;
	}

	bonobo_object_add_interface (BONOBO_OBJECT (to_aggregate),
								 BONOBO_OBJECT (stream));

	/* Interface Bonobo::PersistFile */
	file = bonobo_persist_file_new (load_ps_from_file, NULL, ps_view);
	if (!file) {
		bonobo_object_unref (BONOBO_OBJECT (to_aggregate));
		return NULL;
	}

	bonobo_object_add_interface (BONOBO_OBJECT (to_aggregate),
								 BONOBO_OBJECT (file));

	/* BonoboItemContainer */
	item_container = bonobo_item_container_new ();

	gtk_signal_connect (GTK_OBJECT (item_container),
						"get_object",
						GTK_SIGNAL_FUNC (ggv_postscript_view_get_object),
						ps_view);

	bonobo_object_add_interface (BONOBO_OBJECT (to_aggregate),
								 BONOBO_OBJECT (item_container));

	return to_aggregate;
}

BonoboPropertyBag *
ggv_postscript_view_get_property_bag(GgvPostscriptView *ps_view)
{
	g_return_val_if_fail(ps_view != NULL, NULL);
	g_return_val_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view), NULL);

	bonobo_object_ref(BONOBO_OBJECT(ps_view->priv->property_bag));

	return ps_view->priv->property_bag;
}

BonoboPropertyControl *
ggv_postscript_view_get_property_control(GgvPostscriptView *ps_view)
{
	g_return_val_if_fail(ps_view != NULL, NULL);
	g_return_val_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view), NULL);

	bonobo_object_ref(BONOBO_OBJECT(ps_view->priv->property_control));

	return ps_view->priv->property_control;
}

void
ggv_postscript_view_set_ui_container(GgvPostscriptView *ps_view,
				 Bonobo_UIContainer ui_container)
{
	g_return_if_fail(ps_view != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view));
	g_return_if_fail(ui_container != CORBA_OBJECT_NIL);

	bonobo_ui_component_set_container(ps_view->priv->uic, ui_container);

	ggv_postscript_view_create_ui(ps_view);
}

void
ggv_postscript_view_unset_ui_container(GgvPostscriptView *ps_view)
{
	g_return_if_fail(ps_view != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view));

	bonobo_ui_component_unset_container(ps_view->priv->uic);
}

GtkWidget *
ggv_postscript_view_get_widget(GgvPostscriptView *ps_view)
{
	g_return_val_if_fail(ps_view != NULL, NULL);
	g_return_val_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view), NULL);

	gtk_widget_ref(ps_view->priv->gs);

	return ps_view->priv->gs;
}

float
ggv_postscript_view_get_zoom_factor(GgvPostscriptView *ps_view)
{
	g_return_val_if_fail(ps_view != NULL, 0.0);
	g_return_val_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view), 0.0);

	return gtk_gs_get_zoom(GTK_GS(ps_view->priv->gs));
}

void
ggv_postscript_view_set_zoom_factor(GgvPostscriptView *ps_view,
									float zoom_factor)
{
	g_return_if_fail(ps_view != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view));
	g_return_if_fail(zoom_factor > 0.0);

	gtk_gs_set_zoom(GTK_GS(ps_view->priv->gs), zoom_factor);
}

void
ggv_postscript_view_set_zoom(GgvPostscriptView *ps_view,
							 double zoomx, double zoomy)
{
	g_return_if_fail(zoomx > 0.0);
	g_return_if_fail(zoomy > 0.0);
	g_return_if_fail(ps_view != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view));

	gtk_gs_set_zoom(GTK_GS(ps_view->priv->gs), zoomx);
}

gfloat
ggv_postscript_view_zoom_to_fit(GgvPostscriptView *ps_view,
								gboolean fit_width)
{
	g_return_val_if_fail(ps_view != NULL, 1.0);
	g_return_val_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view), 0.0);

	/* we don't know how not to keep the aspect ratio */
	return  gtk_gs_zoom_to_fit(GTK_GS(ps_view->priv->gs), fit_width);
}

void
ggv_postscript_view_set_orientation(GgvPostscriptView *ps_view,
									GNOME_GGV_Orientation orientation)
{
	BonoboArg *arg;

	g_return_if_fail(ps_view != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view));

	arg = bonobo_arg_new(TC_GNOME_GGV_Orientation);
	BONOBO_ARG_SET_GENERAL(arg, orientation, TC_GNOME_GGV_Orientation,
						   GNOME_GGV_Orientation, NULL);

	bonobo_property_bag_set_value(ps_view->priv->property_bag,
								  "orientation", arg, NULL);
}

GNOME_GGV_Orientation
ggv_postscript_view_get_orientation(GgvPostscriptView *ps_view)
{
	BonoboArg *arg;

	g_return_val_if_fail(ps_view != NULL, 0);
	g_return_val_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view), 0);

	arg = bonobo_property_bag_get_value(ps_view->priv->property_bag,
										"orientation", NULL);
	if(arg == NULL)
		return 0;

	g_assert(arg->_type == TC_GNOME_GGV_Orientation);
	return *(GNOME_GGV_Orientation *) arg->_value;
}

static void
ggv_postscript_view_destroy(GtkObject *object)
{
	GgvPostscriptView *ps_view;

	g_return_if_fail(object != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(object));

	ps_view = GGV_POSTSCRIPT_VIEW(object);

	if(ps_view->priv->property_bag)
		bonobo_object_unref(BONOBO_OBJECT(ps_view->priv->property_bag));
	ps_view->priv->property_bag = NULL;

	if(ps_view->priv->property_control)
		bonobo_object_unref(BONOBO_OBJECT(ps_view->priv->property_control));
	ps_view->priv->property_control = NULL;

	if(ps_view->priv->gs)
		gtk_widget_destroy(ps_view->priv->gs);
	ps_view->priv->gs = NULL;

	if(ps_view->priv->uic)
		bonobo_object_unref(BONOBO_OBJECT(ps_view->priv->uic));
	ps_view->priv->uic = NULL;

	GTK_OBJECT_CLASS(ggv_postscript_view_parent_class)->destroy(object);
}

static void
ggv_postscript_view_finalize(GtkObject *object)
{
	GgvPostscriptView *ps_view;

	g_return_if_fail(object != NULL);
	g_return_if_fail(GGV_IS_POSTSCRIPT_VIEW(object));

	ps_view = GGV_POSTSCRIPT_VIEW(object);

	g_free(ps_view->priv);

	GTK_OBJECT_CLASS(ggv_postscript_view_parent_class)->finalize(object);
}

static void
ggv_postscript_view_class_init(GgvPostscriptViewClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass *)klass;
	gint n = gtk_gs_defaults_get_paper_count(), i;
	BonoboUINode *placeholder;
	BonoboUINode *item;
	gchar *enc_str;
	GtkGSPaperSize *papersizes;

	klass->priv = g_new0(GgvPostscriptViewClassPrivate, 1);

#if 0
	POA_GNOME_Ggv_PostscriptView__epv *epv;
#endif

	ggv_postscript_view_parent_class = gtk_type_class(PARENT_TYPE);

	/* init size menu template */
	placeholder = bonobo_ui_node_new("placeholder");
	bonobo_ui_node_set_attr(placeholder, "name", "SizeList");
	bonobo_ui_node_set_attr(placeholder, "delimit", "top");
	papersizes = gtk_gs_defaults_get_paper_sizes();
	klass->priv->size_paths = g_new0(gchar *, n + 2);
	for(i = 0; i < n; i++) {
		item = bonobo_ui_node_new("menuitem");
		klass->priv->size_paths[i + 1] =
			g_strdup_printf("/commands/%s", papersizes[i].name);
		bonobo_ui_node_set_attr(item, "name", papersizes[i].name);
		enc_str = bonobo_ui_util_encode_str(_(papersizes[i].name));
		bonobo_ui_node_set_attr(item, "label", enc_str);
		g_free(enc_str);
		bonobo_ui_node_set_attr(item, "type", "radio");
		bonobo_ui_node_set_attr(item, "group", "Size");
		bonobo_ui_node_set_attr(item, "verb", "");
		bonobo_ui_node_add_child(placeholder, item);
	}
	klass->priv->size_paths[0] = "/commands/OverrideSize";
	klass->priv->size_paths[n + 1] = NULL;
	klass->priv->size_menus = placeholder;

	gtk_object_class_add_signals(object_class, ggv_postscript_view_signals, LAST_SIGNAL);

	object_class->destroy = ggv_postscript_view_destroy;
	object_class->finalize = ggv_postscript_view_finalize;
	
#if 0
	epv = &klass->epv;

	epv->getImage = impl_GNOME_Ggv_PostscriptView_getImage;
#endif
}

static void
ggv_postscript_view_init(GgvPostscriptView *ps_view)
{
	ps_view->priv = g_new0(GgvPostscriptViewPrivate, 1);
}

BONOBO_X_TYPE_FUNC_FULL(GgvPostscriptView, 
						GNOME_GGV_PostscriptView,
						PARENT_TYPE,
						ggv_postscript_view);

static void
property_control_get_prop(BonoboPropertyBag *bag,
						  BonoboArg         *arg,
						  guint              arg_id,
						  CORBA_Environment *ev,
						  gpointer           user_data)
{
	switch(arg_id) {
	case PROP_CONTROL_TITLE:
		g_assert(arg->_type == BONOBO_ARG_STRING);
		BONOBO_ARG_SET_STRING(arg, _("Display"));
		break;
	default:
		g_assert_not_reached();
	}
}

static BonoboControl *
property_control_get_cb(BonoboPropertyControl *property_control,
						int page_number, void *closure)
{
	GgvPostscriptView *ps_view;
#if 0
	GtkWidget *container;
	BonoboControl *control;
	BonoboPropertyBag *property_bag;
#endif

	g_return_val_if_fail(closure != NULL, NULL);
	g_return_val_if_fail(GGV_IS_POSTSCRIPT_VIEW(closure), NULL);
	g_return_val_if_fail(page_number == 0, NULL);

	ps_view = GGV_POSTSCRIPT_VIEW(closure);

#if 0
	container = eog_create_preferences_page(ps_view, page_number);

	gtk_widget_show_all(container);

	control = bonobo_control_new(container);

	/* Property Bag */
	property_bag = bonobo_property_bag_new(property_control_get_prop,
										   NULL, control);

	bonobo_property_bag_add(property_bag, "bonobo:title",
							PROP_CONTROL_TITLE, BONOBO_ARG_STRING,
							NULL, NULL, BONOBO_PROPERTY_READABLE);

	bonobo_object_add_interface(BONOBO_OBJECT(control),
								BONOBO_OBJECT(property_bag));

	return control;
#else
	return NULL;
#endif
}

GgvPostscriptView *
ggv_postscript_view_construct(GgvPostscriptView *ps_view,
							  GtkGS *gs, gboolean zoom_fit)
{
	g_return_val_if_fail(ps_view != NULL, NULL);
	g_return_val_if_fail(GGV_IS_POSTSCRIPT_VIEW(ps_view), NULL);
	g_return_val_if_fail(gs != NULL, NULL);
	g_return_val_if_fail(GTK_IS_GS(gs), NULL);
	g_return_val_if_fail(!GTK_WIDGET_REALIZED(gs), NULL);

	/* Make sure GConf is initialized */
	if(!gconf_is_initialized())
		gconf_init(0, NULL, NULL);
	
	ps_view->priv->gs = GTK_WIDGET(gs);
	gtk_widget_set_events(GTK_WIDGET(gs), 
						  GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
						  GDK_POINTER_MOTION_MASK);
	gtk_signal_connect(GTK_OBJECT(gs), "button_press_event",
					   GTK_SIGNAL_FUNC(view_button_press_cb), ps_view);
	gtk_signal_connect(GTK_OBJECT(gs), "button_release_event",
					   GTK_SIGNAL_FUNC(view_button_release_cb), ps_view);
	gtk_signal_connect(GTK_OBJECT(gs), "motion_notify_event",
					   GTK_SIGNAL_FUNC(view_motion_cb), ps_view);
	gtk_widget_show(GTK_WIDGET(gs));
	ps_view->priv->zoom_fit = zoom_fit;

	/* Property Bag */
	ps_view->priv->property_bag = bonobo_property_bag_new(ggv_postscript_view_get_prop,
														  ggv_postscript_view_set_prop,
														  ps_view);

	bonobo_property_bag_add(ps_view->priv->property_bag, "orientation",
							PROP_DEFAULT_ORIENTATION, TC_GNOME_GGV_Orientation, NULL,
							_("Default orientation"), 
							BONOBO_PROPERTY_READABLE | BONOBO_PROPERTY_WRITEABLE);
	bonobo_property_bag_add(ps_view->priv->property_bag, "size",
							PROP_DEFAULT_SIZE, TC_GNOME_GGV_Size, NULL,
							_("Default size"), 
							BONOBO_PROPERTY_READABLE | BONOBO_PROPERTY_WRITEABLE);
	bonobo_property_bag_add(ps_view->priv->property_bag, "override_orientation",
							PROP_OVERRIDE_ORIENTATION, TC_CORBA_boolean, NULL,
							_("Override document orientation"), 
							BONOBO_PROPERTY_READABLE | BONOBO_PROPERTY_WRITEABLE);
	bonobo_property_bag_add(ps_view->priv->property_bag, "override_size",
							PROP_OVERRIDE_SIZE, TC_CORBA_boolean, NULL,
							_("Override document size"), 
							BONOBO_PROPERTY_READABLE | BONOBO_PROPERTY_WRITEABLE);
	bonobo_property_bag_add(ps_view->priv->property_bag, "respect_eof",
							PROP_RESPECT_EOF, TC_CORBA_boolean, NULL,
							_("Respect EOF comment"), 
							BONOBO_PROPERTY_READABLE | BONOBO_PROPERTY_WRITEABLE);
	bonobo_property_bag_add(ps_view->priv->property_bag, "watch_file",
							PROP_WATCH_FILE, TC_CORBA_boolean, NULL,
							_("Watch displayed file for changes"), 
							BONOBO_PROPERTY_READABLE | BONOBO_PROPERTY_WRITEABLE);
	bonobo_property_bag_add(ps_view->priv->property_bag, "antialiasing",
							PROP_ANTIALIASING, TC_CORBA_boolean, NULL,
							_("Antialiasing"), 
							BONOBO_PROPERTY_READABLE | BONOBO_PROPERTY_WRITEABLE);

	/* Property Control */
	ps_view->priv->property_control = bonobo_property_control_new
		(property_control_get_cb, 1, ps_view);

	/* UI Component */
	ps_view->priv->uic = bonobo_ui_component_new("GgvPostscriptView");

	ggv_postscript_view_add_interfaces (ps_view, BONOBO_OBJECT (ps_view));

	return ps_view;
}

GgvPostscriptView *
ggv_postscript_view_new(GtkGS *gs, gboolean zoom_fit)
{
	GgvPostscriptView *ps_view;
	
	g_return_val_if_fail(gs != NULL, NULL);
	g_return_val_if_fail(GTK_IS_GS(gs), NULL);

	ps_view = gtk_type_new(ggv_postscript_view_get_type());

	return ggv_postscript_view_construct(ps_view, gs, zoom_fit);
}
