/* Josh Pieper, (c) 2000 */

/* This file is distributed under the GPL, see file COPYING for details */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>

#include "gnut.h" /* 0.4.28.c01 */
#include "hash.h"
#include "lib.h"
#include "queue.h"
#include "route.h"

/* 0.4.28.c01 Moved mutex to gnut.c */

Gnut_Hash *route_hash=0;
Gnut_Queue *route_queue=0;
lru_hash qreplies;

void fre_re(route_entry **x, int bugnum)
{
  yfre((void **) x, bugnum);
}

/* This is useful for analyzing the GUIDs to see how well (or poorly) they are
   distributed. */
void dump_guid(uchar *guid)
{
  int i;
  uchar c;

  for(i=0; i<16; i++) {
    c = guid[i];
    if (i) {
      printf(".");
    }
    printf("%02x", c);
  }
  printf(" '");
  for(i=0; i<16; i++) {
    c = guid[i];
    if ((c >= 32) && (c <= 126)) { /* tagok */
      printf("%c", c);
    } else {
      printf(" ");
    }
  }
  printf("'\n");
}

/* The routing table uses a hash and a list, both store 16-byte GUID's.
 * The list is used to keep track of which GUID's are the oldest, and the
 * hash is used to quickly locate a GUID (to see if it's in the table
 * or not) */

/* The compare func compares all 16 bytes of GUID data */
int route_compare_func(void *a, void *b)
{
  gnut_mprobe(a);
  gnut_mprobe(b);
  if (memcmp(a, b, 16) == 0)
	return 0;
  return -1;
}

/* GUID is hashed by adding its 16 bytes together */
uchar route_hash_func(void *a)
{
  route_entry *re;
  uint32 crc;

  re = a;
  crc = crc32_block(&(re->guid[0]), 16);
  crc ^= (crc >> 16);
  crc ^= (crc >> 8);
  return (crc & 0xff);
}

int route_init()
{
  route_hash = gnut_hash_new(route_hash_func, route_compare_func);
  route_queue = gnut_queue_new(514);
  lru_init(&qreplies, 1000);
  return 0;
}

/* int route_guid_add(uchar guid[16], gcs *gc)
 *
 * adds the guid (which came over connection gc) into the routing table
 *
 * When there are more than ROUTE_MAX entries in the table, the oldest
 * is deleted */
int route_guid_add(uchar guid[16], gcs *gc)
{
  route_entry *re;

  GD_S(3, "route_guid_add entering gc="); GD_P(3, gc); GD_S(3, "\n");

  /* dump_guid(guid); */

  /* al.locate and copy the data */
  re = (route_entry *) ymaloc(sizeof(route_entry), 296);
  memcpy(re->guid, guid, 16);
  re->p = gc;

  /* Store in both queue and hash. Neither copies the data block, they
   * just point to it */
  pthread_mutex_lock(&route_mutex);

  /* printf("route_guid_add: "); */
  /* for (i=0;i<16;i++) printf("%02X",re->guid[i]); */
  /* printf("\n\n");   */
  gnut_hash_insert(route_hash, re, 498);
  gnut_queue_insert(route_queue, re, 503);

  if (route_queue->size > ROUTE_MAX) {
    GD_S(5,"hit max routing table size\n");
    re = gnut_queue_remove(route_queue);
    gnut_hash_remove(route_hash, re);
    fre_re(&re, 372);
  }

  pthread_mutex_unlock(&route_mutex);
  
  GD_S(3, "route_guid_add returning success\n");
  return 0;
}

/* gcs * route.guid_find(uchar guid[16])
 *
 * looks through the routing table, and if an entry is found,
 * the corresponding connection is returned, otherwise
 * 0 is returned */
gcs * route_guid_find(uchar * guid, route_entry **the_re)
{
  route_entry *re,*re2;
  
  GD_S(3,"route_guid_find entering\n");
  
  re = (route_entry *) ymaloc(sizeof(route_entry), 297);
  gnut_mprobe(re);

  memcpy(re->guid, guid, 16);

  pthread_mutex_lock(&route_mutex);
  GD_S(3, "route_guid_find locked mutex\n");
  
  dqi(0x015a);
  re2 = gnut_hash_find(route_hash,re);
    
  pthread_mutex_unlock(&route_mutex);
  
  gnut_mprobe(re);
  
  fre_re(&re, 373);
  if (re2) {
    *the_re = re2;
    return re2->p;
  }
  GD_S(3, "route_guid_find not found!\n");
  *the_re = 0;
  return 0;
}
      
/* int route_guid_clear(gcs *gc)
 *
 * turns all occurances of gc in routing table into 0
 * so that they won't be matched again */
int route_guid_clear(gcs *gc)
{
  Gnut_List *tmp;
  route_entry *re;
  GD_S(4, "route_guid_clear entering gc="); GD_P(4, gc); GD_S(4, "\n");
  
  pthread_mutex_lock(&route_mutex);
  GD_S(4, "route_guid_clear locked mutex\n");
  
  /* since the hash contains pointers to the
   * same data, I can do this here, and save a lot
   * of time... */
  for (tmp=gnut_queue_list(route_queue); tmp; tmp=tmp->next) {
    gnut_mprobe(tmp);
    re=tmp->data;
    if (re->p==gc)
      re->p=0;
  }

  pthread_mutex_unlock(&route_mutex);

  GD_S(4,"route_guid_clear returning success\n");
  return 0;
}
