/*  GTK-Imonc - imond client for linux/unix
 *  Copyright (C) 2001-2003 Stefan Strigler <steve@zeank.in-berlin.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>

#include <gtk/gtk.h>

#include "interface.h"
#include "support.h"
#include "imonc.h"
#include "callbacks.h"
#include "timer.h"

void show_usage(char *argv[])
{
	fprintf (stderr, "usage: %s [options] hostname [port]\n", argv[0]);
	fprintf (stderr, "  options:\n");
	fprintf (stderr, "  -n\t\tdon't display news panel\n");
	fprintf (stderr, "  -t [command]\tuse call notification - [command] can be a custom command which is executed on incoming call instead of doing a message pop-up\n");
	fprintf (stderr, "  -i\t\tuse systemtray icon\n");
	fprintf (stderr, "  -m\t\tstart in minimode\n");
	fprintf (stderr, "  -p pass\tuse pass as password\n");
	fprintf (stderr, "  -h\t\tprint this help message\n");
	fprintf (stderr, "  -v\t\tshow version and exit\n");
	exit(1);
}

void show_version(char *argv[])
{
	printf("%s version %s\n",argv[0],VERSION);
	exit(0);
}

int get_bool(char *var) {
	if (!strcmp(var,"yes"))
		return TRUE;
	if(!strcmp(var,"1"))
		return TRUE;
	return FALSE;
}

void parse_config(char *configfile) {
	FILE *fp;
  GIOChannel *conf_gioc;
  GError *err = NULL;
  GError *err2 = NULL;
	char *line = NULL;
	gchar var[255], val[255];
  gsize len;


  if (!(conf_gioc = g_io_channel_new_file(configfile,"r",&err))) {
    g_print("config not found: %s no such file\n", configfile);
    return;
  }

  while (g_io_channel_read_line(conf_gioc, &line, &len, NULL, &err2 ) == G_IO_STATUS_NORMAL) {
    
		while(--len)
			if (line[(int) len] == '#')
				line[(int) len] = '\0'; /* terminate line at comment */
		if (sscanf(line,"%s = %s",var,val)==2) {
			if (!strcmp(var,"hostname"))
				host_name = strdup(val);
			else if (!strcmp(var,"port"))
				port = atoi(val);
			else if (!strcmp(var,"get_news"))
				use_news_tab = get_bool(val);
			else if (!strcmp(var,"mini_mode"))
				is_mini = start_mini = get_bool(val);
			else if (!strcmp(var,"call_notification"))
				telmond_popup = get_bool(val);
			else if (!strcmp(var,"call_cmd"))
				call_notification = strdup(val);
			else if (!strcmp(var,"use_trayicon"))
				use_trayicon = get_bool(val);
			else
				fprintf(stderr,"unknown option: %s\n",var);
		}
    if (line)
      g_free(line);
	}
}

int
main (int argc, char *argv[])
{
	gchar *answer;
	gchar buf[256];
	struct sigaction act;

	/* getopt vars */
	int opt;
	extern int optind, opterr, optopt;

	siginterrupt (SIGALRM, 1);

	/* Assign sig_pipe as SIGPIPE handler */
	act.sa_handler = fli4l_reconnect;

	/* We don't want to block any other signals */
	sigemptyset(&act.sa_mask);

	if (sigaction(SIGPIPE, &act, NULL) < 0) {
		fprintf(stderr, "sigaction failed\n");
		return 1;
	}

	signal (SIGALRM, fli4l_reconnect);

#ifdef ENABLE_NLS
  bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);
#endif

  gtk_set_locale ();
	gtk_init (&argc, &argv);

  add_pixmap_directory (PACKAGE_DATA_DIR "/" PACKAGE "/pixmaps");

/* 	add_pixmap_directory (PACKAGE_DATA_DIR "/pixmaps"); */
/* 	add_pixmap_directory (PACKAGE_SOURCE_DIR "/pixmaps"); */


	/* parse command line arguments
	 * proposed format:
	 * $ gtk-imonc [-n] -m [-p pass] hostname [port]
	 */
	port = IMOND_PORT;
	use_news_tab = TRUE;
	is_mini = FALSE;
	telmond_popup = FALSE;
	use_trayicon = FALSE;
	memset(pass,'\0',64);

	/* read config */
	snprintf(buf,255,"%s/.imoncrc",getenv("HOME"));
	parse_config(buf);

	while ((opt=getopt(argc,argv,"vhnmit::p:"))>0) {
		switch(opt) {
		case ':':
		case '?':
		case 'h': show_usage(argv); break;
		case 'v': show_version(argv); break;
		case 'n': use_news_tab = FALSE; break;
		case 'm': is_mini = TRUE; break;
		case 'i': use_trayicon = TRUE; break;
		case 't': 
			telmond_popup = TRUE; 
			if ((optind < argc-1) && *argv[optind] != '-')
				call_notification = argv[optind++];
			else
				call_notification = NULL; // overwrite settings from config file
			break;
		case 'p':
			strncpy(pass,optarg,63);
			memset(optarg,'x',strlen(optarg));
			break;
		}
	}

	if (argc > optind)		
		host_name = argv[optind++];
	
	if (argc > optind)
		port = atoi(argv[optind]);

	/* initialize phonebook */
	phonebook = g_hash_table_new (g_str_hash, g_str_equal);
	fill_phonebook (phonebook, "/etc/phonebook");
	snprintf(buf, 256, "%s/.phonebook",getenv("HOME"));
	fill_phonebook (phonebook, buf);

	gtk_imonc = create_gtk_imonc ();
	gtk_miniimonc = create_gtk_miniimonc();
	rightclick_menu = create_rightclick_menu();	

	if (!host_name) {
    host_name = "";
		goto PREF_DIALOG;
		//show_usage(argv);
	}

	/* connect */
	if ((fd = service_connect (host_name, port)) < 0) {
	PREF_DIALOG: on_preferences_activate(NULL,NULL); // create prefs dialog
	gtk_main();
	return 0;
		//exit (1);
	}

	/* check for salt */
	send_command (fd, "salt");
	answer = get_answer (fd);

	if (strlen(answer))
		salt = strdup(answer);
	else
		salt = NULL;

	/* check password */
	if (strlen(pass)>0) {
		if (salt) {
			gchar tmp[256];
			snprintf(tmp,256,"%s%s",pass,salt);
			sprintf (buf, "md5pass %s", hexmd5(tmp));
		} else 
			sprintf (buf, "pass %s", pass);
		send_command (fd, buf);
	} else {
		send_command (fd, "pass");
	}

	is_admin = FALSE; /* remember if we are in admin mode */
	answer = get_answer (fd);
			
	/* 	From Nico Wallmeier:
	 *		1: Userpasswort nötig
	 * 		2: Adminpasswort nötig
	 * 		3: User+ Adminpasswort nötig
	 * 		+4 im Adminmodus 
	 */

	if (!strlen(answer) || /* supplied password was wrong */
			(!strlen(pass) && (atoi(answer) == 1 || atoi(answer) == 3))) { /* no password given and auth needed */
		/* need to check password */
		GtkWidget *auth_dialog;
		
		auth_dialog = create_auth_dialog ();
		gtk_widget_show (auth_dialog);
	} else {
		/* check if we are in admin mode */
		if (atoi (answer)>=4)
			is_admin = TRUE; /* we are already in admin mode */

		gtk_widget_show (gtk_imonc);
		imonc_init (gtk_imonc); /* initialize imonc functionality */
		imonc_update (gtk_imonc); /* init with current values */
		
		if (is_mini) {
			GtkWidget *mini_view;

			is_mini = FALSE;
			mini_view = lookup_widget (rightclick_menu,"mini_view");
			gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mini_view),TRUE);
		}
		timeout_tag = gtk_timeout_add (TIMEOUT_MSEC, imonc_update, gtk_imonc);
	}
	
	gtk_main ();
	
	service_disconnect (fd);
	return 0;
}
