/* IMSpector - Instant Messenger Transparent Proxy Service
 * http://www.imspector.org/
 * (c) Lawrence Manning <lawrence@aslak.net>, 2006
 * 
 * Contributions from:
 *     Ryan Wagoner <ryan@wgnrs.dynu.com>, 2006
 *          
 * Released under the GPL v2. */

#include "imspector.h"

/* Remove the first newline in a string and truncate. */
void stripnewline(char *buffer)
{
	char *t;
	
	t = strchr(buffer, '\r'); if (t) *t = '\0';
	t = strchr(buffer, '\n'); if (t) *t = '\0';
}

/* snprintf for std::string.  There is a max size of BUFFER_SIZE, above which
 * the string will get truncated. */
std::string stringprintf(const char *fmt, ...)
{
	char buffer[BUFFER_SIZE];
	va_list argp;

	memset(buffer, 0, BUFFER_SIZE);
	
	va_start(argp, fmt);
	vsnprintf(buffer, BUFFER_SIZE - 1, fmt, argp);
	va_end(argp);

	return (std::string)(buffer);
}

/* Chops the heading first line into its command and arg list. Returns the
 * start of the following lines. */
char *chopline(char* buffer, std::string &command, std::vector<std::string> &args, int &argc)
{
	char *s = buffer;
	
	/* Copy the command, while the character looks valid. */
	for (; *s && *s != ' ' && *s != '\r' && *s != '\n'; s++)
		command.push_back(*s);
	s++;
	
	/* Same for the argument list. */
	argc = 0;
	while (*s && *s != '\r' && *s != '\n')
	{
		std::string arg;
		for (; *s && *s != ' ' && *s != '\r' && *s != '\n'; s++)
			arg.push_back(*s);
		s++;
		
		args.push_back(arg); argc++;
	}
	
	/* Need to advance s to the start of the next line, skipping over the eol chars. */
	for (; *s && (*s == '\r' || *s == '\n'); s++);
	
	return s;
}

/* Prints debug. This is a simple wrapper around syslog to prevent unecessary
 * calls when not in debug mode. */
void debugprint(bool debugflag, const char *fmt, ...)
{
	if (!debugflag) return;

	va_list argp;

	va_start(argp, fmt);
	vsyslog(LOG_DEBUG, fmt, argp);
	va_end(argp);
}

int decodebase64(std::string line, uint8_t *buffer, int bufferlen)
{
	uint32_t quartet = 0;
	uint8_t d;
	int c = 0; int len = line.length() - 4;
	
	for (int i = 0; i < len && c < bufferlen - 3; i += 4)
	{
		quartet = 0;
		d = decodebase64char(line[i + 0]);
		quartet = quartet | d;
		d = decodebase64char(line[i + 1]);
		quartet = (quartet << 6) | d;
		d = decodebase64char(line[i + 2]);
		quartet = (quartet << 6) | d;
		d = decodebase64char(line[i + 3]);
		quartet = (quartet << 6) | d;
		d = (quartet & 0xFF0000) >> 16;
		buffer[c++] = (uint8_t) d;
		d = (quartet & 0xFF00) >> 8;
		buffer[c++] = (uint8_t) d;
		d = quartet & 0xFF;
		buffer[c++] = (uint8_t) d;
	}
	
	return c;
}

uint8_t decodebase64char(char c)
{
	uint8_t i = '\0';
	
	switch (c)
	{
		case '+':
			i = 62;
			break;
			
		case '/':
			i = 63;
			break;
			
		case '=':
			i = 0;
			break;
			
		default:
			/* Must be alphanumeric. */
			i = '9' - c;
			if (i > 0x3F)
			{
				/* It is under 9 */
				i = 'Z' - c;
				if (i > 0x3F)
				{
					/* It is after Z. */
					i = 'z' - c;
					if (i > 0x3F)
						i = 0x80;
					else
						/* It is a-z. */
						i = c - 71;
				}
				else
					/* A-Z */
					i = c - 65;
			}
			else 
				i = c + 4;
				
			break;
	}
	
	return i;
}

/* This little function is used to generate trace packets. */
void tracepacket(const char *protocol, int packetcount, char *buffer, int bufferlength)
{
	std::string filename = stringprintf(TRACE_DIR "/%s.%d.%d", protocol,
		getpid(), packetcount);

	int fd = -1;
	if ((fd = creat(filename.c_str(), 0600)) > 0)
	{
		write(fd, buffer, bufferlength);
		close(fd);
	}
}

/* DJBs hash function. */
unsigned long hash(const char *str)
{
	unsigned long hash = 5381;
	int c;
	
	while ((c = *str++))
		hash = ((hash << 5) + hash) + c; /* hash * 33 + c */
	
	return hash;
}

/* Our tag parser. */
char *parsexmltag(bool debugmode, char* buffer, std::string &payload, int &payloadlength, 
	std::string &tag, bool &closing, std::map<std::string, std::string> &params)
{
	closing = false;

	char *s = buffer;

	/* Copy the payload, while the character looks valid. */
	for (; *s && *s != '<'; s++)
		payload.push_back(*s);
	
	removenewlines(payload);
	
	payloadlength = s - buffer;
	
	debugprint(debugmode, "XML Parse: Payload: %s", payload.c_str());
	
	/* Skip the <. */
	if (*s) s++;
	
	/* Copy the command, while the character looks valid. */
	for (; *s && *s != ' ' && *s != '>'; s++)
		tag.push_back(*s);
		
	removenewlines(tag);

	debugprint(debugmode, "XML Parse: Tag: %s", tag.c_str());

	/* Skip the spaces. */
	for (; *s && (*s == ' ' || *s == '\n' || *s == '\r'); s++);

	while (*s && *s != '>' && *s != '/')
	{
		std::string key; std::string value;

		for (; *s && *s != ' ' && *s != '='; s++)
			key.push_back(*s);
		/* Skip the = */
		if (*s) s++;
			
		/* Skip the quote */
		if (*s) s++;
		for (; *s && *s != '\'' && *s != '\"'; s++)
			value.push_back(*s);
		/* Skip the quote */
		if (*s) s++;
		
		/* Skip the spaces. */
		for (; *s && (*s == ' ' || *s == '\n' || *s == '\r'); s++);

		/* Allow an empty value as it's valid. */
		if (!key.empty()) params[key] = value;
		
		debugprint(debugmode, "XML Parse: Key: %s Value: %s", key.c_str(), value.c_str());
	}
	
	/* Look for the closing flag "/>" at the end of a tag. This means there is no enclosing
	 * tags. */
	if (*s == '/')
	{
		closing = true;
		debugprint(debugmode, "XML Parse: Closing tag");
	}
	
	s++;
	
	return s;
}

/* Truncate a string to the first /, if any. Probably better written using std::string methods. */
void stripslash(std::string &in)
{
	std::string result;
	
	for (char *s = (char *) in.c_str(); *s && *s != '/'; s++)
		result.push_back(*s);
		
	in = result;
}

/* Remove all newlines from a string. Ditto about using std::string functions. */
void removenewlines(std::string &in)
{
	std::string result;
	
	for (char *s = (char *) in.c_str(); *s; s++)
	{
		if (*s != '\r' && *s != '\n')
			result.push_back(*s);
	}
	
	in = result;
}
