#include "sqlite_inter.h"
#include <cstring>

void* sqlite_connect(wchar_t* info)
{
    sqlite3* db;
    int rc = sqlite3_open(wctostr(info), &db);
    SQLiteConnection* con = new SQLiteConnection();
    con->db = db;
    con->ok = (rc==0);
    return (void*)con;
}

int sqlite_ok(void* con)
{
    return ((SQLiteConnection*)con)->ok;
}

wchar_t* sqlite_getError(void* con)
{
    sqlite3* db = ((SQLiteConnection*)con)->db;
    return strtowc(sqlite3_errmsg(db));
}

void sqlite_close(void* con)
{
    sqlite3* db = ((SQLiteConnection*)con)->db;
    sqlite3_close(db);
}

void sqlite_buildRow(sqlite3_stmt* stmt, SQLiteResult* sqlr)
{
    int cols = sqlite3_column_count(stmt);
    int getnames = 0;
    if (KayaArraySize(sqlr->col_names)==0) {
	getnames = 1;
    }

    KValue entry;
    int intval;
    double dblval;
    const char* txt;

    KArray row = newKayaArray(cols);
    sqlr->cols = cols;

    for(int x = 0; x<cols; ++x) {
	switch(sqlite3_column_type(stmt, x)) {
	case SQLITE_INTEGER:
	    intval = sqlite3_column_int(stmt,x);
	    entry = KayaUnion(1,1);
	    KayaUnionSetArg(entry, 0, KayaInt(intval));
	    KayaArrayPush(row, entry);
	    break;
	case SQLITE_FLOAT:
	    dblval = sqlite3_column_double(stmt,x);
	    entry = KayaUnion(2,1);
	    KayaUnionSetArg(entry, 0, KayaFloat(dblval));
	    KayaArrayPush(row, entry);
	    break;
	default:
	    txt = (const char*)(sqlite3_column_text(stmt, x));
	    entry = KayaUnion(0,1);
	    KayaUnionSetArg(entry, 0, KayaString(KSTRING(txt)));
	    KayaArrayPush(row, entry);
	}
	if (getnames==1) {
	    txt = (sqlite3_column_name(stmt, x));
	    KayaArrayPush(sqlr->col_names, KayaString(KSTRING(txt)));
	}
    }
    KayaArrayPush(sqlr->res_table, KayaArrayVal(row));
}

KArray sqlite_get(sqlite3_stmt* stmt, SQLiteResult* sqlr)
{
    int cols = sqlite3_column_count(stmt);
    int getnames = 0;
    if (KayaArraySize(sqlr->col_names)==0) {
	getnames = 1;
    }

    KValue entry;
    int intval;
    double dblval;
    const char* txt;

    KArray row = newKayaArray(cols);
    sqlr->cols = cols;

    for(int x = 0; x<cols; ++x) {
	switch(sqlite3_column_type(stmt, x)) {
	case SQLITE_INTEGER:
	    intval = sqlite3_column_int(stmt,x);
	    entry = KayaUnion(1,1);
	    KayaUnionSetArg(entry, 0, KayaInt(intval));
	    KayaArrayPush(row, entry);
	    break;
	case SQLITE_FLOAT:
	    dblval = sqlite3_column_double(stmt,x);
	    entry = KayaUnion(2,1);
	    KayaUnionSetArg(entry, 0, KayaFloat(dblval));
	    KayaArrayPush(row, entry);
	    break;
	default:
	    txt = (const char*)(sqlite3_column_text(stmt, x));
	    entry = KayaUnion(0,1);
	    KayaUnionSetArg(entry, 0, KayaString(KSTRING(txt)));
	    KayaArrayPush(row, entry);
	}
	if (getnames==1) {
	    txt = (sqlite3_column_name(stmt, x));
	    KayaArrayPush(sqlr->col_names, KayaString(KSTRING(txt)));
	}
    }
    return row;
}


void* sqlite_incexec(void* con, wchar_t* query)
{
    // result array had better be empty here!

    sqlite3_stmt* stmt; // statement handle
    const char* tail; // unused, from sqlite3_prepare

    sqlite3* db = ((SQLiteConnection*)con)->db;
    SQLiteResult* result = new SQLiteResult();
    result->res = 0;
    result->res_table = newKayaArray(0);
    result->col_names = newKayaArray(0);
    result->rows = 0;
    result->cols = 0;

    int rc = sqlite3_prepare(db, wctostr(query), -1, &stmt, &tail);
    result->resptr = stmt;

    if (rc!=0) {
	result->res = 2;
    }
    return (void*)result;
}

KArray sqlite_getrow(void* vmptr, void* resptr) {
  SQLiteResult* result = (SQLiteResult*) resptr;
  VMState* vm = (VMState*) vmptr;

  sqlite3_stmt* stmt = result->resptr;
  while(true) {
    int res = sqlite3_step(stmt);
    switch(res) {
    case SQLITE_BUSY: 
      result->res = 1;
      vm->kaya_internalError(1);
      break;
    case SQLITE_ERROR:
    case SQLITE_MISUSE:
      result->res = 2;
      vm->kaya_internalError(2);
      break;
    case SQLITE_ROW:
      result->rows++;
      return sqlite_get(stmt, result);
    case SQLITE_DONE:
      vm->kaya_internalError(0); // no more rows
      break;
    default:
      result->res = 3;
      //      vm->kaya_internalError(3);
    }
  }
  return newKayaArray(1);
}

void* sqlite_exec(void* con, wchar_t* query)
{
    // result array had better be empty here!

    sqlite3_stmt* stmt; // statement handle
    const char* tail; // unused, from sqlite3_prepare

    sqlite3* db = ((SQLiteConnection*)con)->db;
    SQLiteResult* result = new SQLiteResult();
    result->res = 0;
    result->res_table = newKayaArray(0);
    result->col_names = newKayaArray(0);
    result->rows = 0;
    result->cols = 0;

    int rc = sqlite3_prepare(db, wctostr(query), -1, &stmt, &tail);
    if (rc!=0) {
	result->res = 2;
	return result;
    }

    int res;
    do {
	res = sqlite3_step(stmt);
	switch(res) {
	case SQLITE_BUSY: 
	    result->res = 1;
	    break;
	case SQLITE_ERROR:
	case SQLITE_MISUSE:
	    result->res = 2;
	    break;
	case SQLITE_ROW:
	    result->rows++;
	    sqlite_buildRow(stmt, result);
	case SQLITE_DONE:
	    break;
	default:
	    result->res = 3;
	}
    } while (res!=SQLITE_DONE);
    sqlite3_finalize(stmt);
    return (void*)result;
}

void sqlite_discard(void* sptr) {
  sqlite3_finalize((sqlite3_stmt*)sptr);
}

KArray sql_getcolnames(void* res)
{
    return ((SQLiteResult*)res)->col_names;
}

KArray sql_getrestable(void* res)
{
    return ((SQLiteResult*)res)->res_table;
}

int sql_numrows(void* res)
{
    return ((SQLiteResult*)res)->rows;
}

int sql_numcols(void* res)
{
    return ((SQLiteResult*)res)->cols;
}

int sql_resrc(void* res)
{
    return ((SQLiteResult*)res)->res;
}

void* sqlite_prepare(void* vmptr, void* con, wchar_t* query) 
{
    sqlite3_stmt* stmt; // statement handle
    const char* tail; // unused, from sqlite3_prepare

    sqlite3* db = ((SQLiteConnection*)con)->db;

    int rc = sqlite3_prepare(db, wctostr(query), -1, &stmt, &tail);
    if (rc!=0) {
      VMState* vm = (VMState*) vmptr;
      vm->kaya_internalError(1);
    }
    SQLiteStatement* statement = new SQLiteStatement(stmt);   
    return (void*) statement;
}

void* sqlite_execp(void* stptr, KArray params) {
    SQLiteStatement* statement = (SQLiteStatement*) stptr;
    sqlite3_stmt* stmt = statement->stmt;
    SQLiteResult* result = new SQLiteResult();
    result->res = 0;
    result->res_table = newKayaArray(0);
    result->col_names = newKayaArray(0);
    result->rows = 0;
    result->cols = 0;
    sqlite3_reset(stmt);
    int sz = KayaArraySize(params);
    for (int i=0;i<sz;i++) {
      KValue val = KayaArrayLookup(params,i);
      if (KayaUnionGetTag(val) == 0) {
	sqlite3_bind_null(stmt,i+1);
      } else {
	char* param = CSTRING(KayaGetString(KayaUnionGetArg(val,0)));
	sqlite3_bind_text(stmt,i+1,param,strlen(param),SQLITE_STATIC);
      }
    }

    int res;
    do {
	res = sqlite3_step(stmt);
	switch(res) {
	case SQLITE_BUSY: 
	    result->res = 1;
	    break;
	case SQLITE_ERROR:
	    result->res = 2;
	    break;
	case SQLITE_ROW:
	    result->rows++;
	    sqlite_buildRow(stmt, result);
	case SQLITE_DONE:
	    break;
	default:
	    result->res = 3;
	}
    } while (res!=SQLITE_DONE);
    return (void*)result; 
}
