/*
Magpie - reference librarian for Debian systems
Copyright (C) 2000  Bear Giles <bgiles@coyotesong.com>

This program is free software; you may redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the license, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char rcsid[] = "$Id$";

/*****
This module (mostly) encapsulates the functions that read the 
Debian-specific database.
*****/
#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>
#include <stdlib.h>
#include <ctype.h>
#include <errno.h>
#include <locale.h>
#include <dirent.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <pwd.h>
#include <grp.h>
#include "magpie.h"

/*+
This file contains the manifest constants used by the parser.
They always have the form T_XXX.
+*/
#include "parser.h"

extern void free_info (struct package_info *);
extern void free_flex (void);



/*+
This is an arbitrary number that should be large enough for the
next few years.  We can make this number fairly large since it
only allocates a pointer - the parser allocates all necessary
buffers itself.
+*/
#define MAX_PACKAGES	(16 * 1024)

/*+
We use a simple array to access the structures allocated by the
parser.  This allows us to easily sort the contents.
+*/
struct package_info *cache[MAX_PACKAGES];
int cachecnt = 0;

/*+
This array is always kept sorted, and used for binary searches
based on the package name.
+*/
static struct package_info *scache[MAX_PACKAGES];

/*+
Some globals used by the bison/flex parser.
+*/
extern FILE *yyin;
extern int yydebug;
extern int yyparse (void);
extern int lineno;

/*+
arrays containing the number of packages in each category.  This
allows us to avoid creating empty files.
+*/
int matrixcnt_s[CNT_SECTIONS];
int matrixcnt_sc[CNT_SECTIONS][CNT_CATEGORIES];
int matrixcnt_scp[CNT_SECTIONS][CNT_CATEGORIES][CNT_PRIORITIES];
int matrixcnt_sp[CNT_SECTIONS][CNT_PRIORITIES];
int matrixcnt_p[CNT_PRIORITIES];


/*+
Comparison function for sorting by package name, version (descending),
section, category.  It is used to sort the array so nodes can be found
by name.
+*/
static int cmp_package_name (const void *p, const void *q)
{
	struct package_info *pp = *((struct package_info **) p);
	struct package_info *qq = *((struct package_info **) q);
	int r;
	
	assert (pp->name);
	assert (qq->name);

	r = strcoll (pp->name, qq->name);
	if (r)
		return r;

	r = strcoll (pp->version, qq->version);
	if (r)
		return -r;

	r = pp->section - qq->section;
	if (r)
		return r;

	return pp->category - qq->category;
}


/*+
Find a package given only its name.  If there are duplicate names,
the first matching name is selected.  Returns NULL if no match is
found.
+*/
struct package_info *mp_lookup (const char *name)
{
	int lo, hi, mid;
	struct package_info *plo, *phi, *p;
	int r;

	lo = 0;
	hi = cachecnt-1;
	plo = scache[lo];
	phi = scache[hi];
	if (strcoll (plo->name, name) == 0)
		return plo;
	if (strcoll (phi->name, name) == 0)
		return phi;
	while (lo <= hi) {
		mid = (lo + hi) / 2;
		p = scache[mid];
		r = strcoll (name, p->name);
		if (r < 0)
			hi = mid-1;
		else if (r > 0)
			lo = mid+1;
		else {
			mid--;
			for (mid--; mid > 0; mid--) {
				if (strcoll (scache[mid]->name, name) != 0)
					break;
				p = scache[mid];
			}
			return p;
		}
	}
	return NULL;
}


/*+
Release the pacakge database and buffers used in flex.  This allows
us to check for memory leaks with a profiler or 'purify'.
+*/
static int debian_cleanup (void)
{
	int i;
	struct package_info *p;

	for (i = 0; i < cachecnt; i++) {
		p = cache[i];
		free_info (p);
	}

	free_flex ();
	return 0;
}

/*+
Ingest a single Debian 'packages' file.  All of the heavy lifting
is handled by the bison parser.
+*/
static int process (
	const char *filename, 	/*+ the name of the packages or status file +*/
	int flag)				/*+ 0 = packages file, 1 = status file +*/
{
	struct package_info *p, *q;
	int i, j;

	yyin = fopen (filename, "r");
	if (yyin == NULL)
		return -1;

	lineno = 0;
	while ((p = (struct package_info *) yyparse ()) > 0) {
		if (p == 1) {
			printf ("eek!\n");
			continue;
		}

	  	/* some quick sanity checks */
		assert (0 <= p->section && p->section < CNT_SECTIONS);
		assert (0 <= p->priority && p->priority < CNT_PRIORITIES);
		assert (0 <= p->category && p->category < CNT_CATEGORIES);

		assert (cachecnt < MAX_PACKAGES);

		/* when reading 'status', we only care about
		 * installed and unpacked packages */
		if (flag) {
			if (p->status[0] && strcmp (p->status[0], "install") != 0) {
				free_info (p);
				goto nextpackage;
			}

			p->installed = !strcmp (p->status[2], "installed");
			p->unpacked = !strcmp (p->status[2], "unpacked");
			if (!p->installed && !p->unpacked) {
				free_info (p);
				goto nextpackage;
			}

			/* if the package is already installed, merge the records. */
			/* a binary or hashed search would be *much* faster */
			for (i = 0; i < cachecnt; i++) {
				q = cache[i];
				if (strcoll (p->name, q->name) == 0 &&
					strcoll (p->version, q->version) == 0) {
					q->status[0] = p->status[0];
					q->status[1] = p->status[1];
					q->status[2] = p->status[2];

					p->status[0] = 0;
					p->status[1] = 0;
					p->status[2] = 0;

					q->cfcnt = p->cfcnt;
					for (j = 0; j < p->cfcnt; j++) {
						q->conffiles[j] = p->conffiles[j];
						p->conffiles[j] = 0;
					}

					q->installed = p->installed;
					q->unpacked =  q->unpacked;

					free_info (p);

					goto nextpackage;
				}
			}
		}

		cache[cachecnt] = p;
		scache[cachecnt] = p;
		cachecnt++;

		/* update our counters */
		matrixcnt_s[p->section]++;
		matrixcnt_sc[p->section][p->category]++;
		matrixcnt_scp[p->section][p->category][p->priority]++;
		matrixcnt_sp[p->section][p->priority]++;
		matrixcnt_p[p->priority]++;

		/* to save our sanity, we check for some null pointers */
		/* this can happen when loading information on purged packages. */
		if (!p->summary)
			p->summary = strdup ("(unknown)");
		if (!p->maintainer)
			p->maintainer = strdup ("(unknown)");
nextpackage:
	}

	fclose (yyin);
	return 0;
}


/*+
Ingest all 'packages' files in the apt cache, plus the dpkg status file.
We don't perform any locking, although we really should!
+*/
int debian_database (void)
{
  	DIR *dirp;
	struct dirent *dp;
	const char prefix[] = "/var/state/apt/lists";
	char pathname[256];
	int n;

	memset (matrixcnt_s,   0, sizeof matrixcnt_s);
	memset (matrixcnt_sc,  0, sizeof matrixcnt_sc);
	memset (matrixcnt_scp, 0, sizeof matrixcnt_scp);
	memset (matrixcnt_sp,  0, sizeof matrixcnt_sp);
	memset (matrixcnt_p,   0, sizeof matrixcnt_p);

	/*
	 * If the parser gives us fits, we can set this value to '1'.
	 */
	yydebug = 0;
	
	/*
	 * Process anything matching "/var/state/apt/lists/ *_Packages".
	 */
	dirp = opendir (prefix);
	if (dirp != NULL) {
		while ((dp = readdir (dirp)) != NULL) {
		  	if (dp->d_name[0] == '.')
			  	continue;
		  	n = strlen (dp->d_name);
			if (n < 10)
			  	continue;
			if (strcmp (&dp->d_name[n-9], "_Packages") != 0)
			  	continue;

			sprintf (pathname, "%s/%s", prefix, dp->d_name);
			fprintf (stderr, "processing '%s'\n", dp->d_name);
			process (pathname, 0);
		}
		closedir (dirp);
	}

	/*
	 * We also process the status file.
	 */
	process ("/var/lib/dpkg/status", 1);

	qsort (scache, cachecnt, sizeof scache[0], cmp_package_name);

	return 0;
}


struct magpie_module mod_debian = { 
	version           : MAGPIE_VERSION,
	description       : "Debian database module",
	database          : debian_database,
	init              : NULL,
	cleanup           : debian_cleanup,
	annotated_index   : NULL,
	unannotated_index : NULL,
	miscellaneous     : NULL
};
