/*
mime.c - MessageWall MIME handling definitions
Copyright (C) 2002 Ian Gulliver

This program is free software; you can redistribute it and/or modify
it under the terms of version 2 of the GNU General Public License as
published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <firestring.h>
#include "messagewall.h"
#include "rfc822.h"
#include "dnsdcc.h"
#include "smtp.h"
#include "virus.h"
#include "mime.h"

static const char tagstring[] = "$Id: mime.c,v 1.32 2002/07/11 03:10:38 ian Exp $";

unsigned char mime_base64_table[256];
unsigned char mime_qp_table[256];
struct firestring_estr_t mime_buffer;

int mime_parse(int client, int depth, int *part, int parent, struct firestring_estr_t *message, struct messagewall_profile_t *profile) {
	struct firestring_estr_t *value, *value2;
	struct messagewall_estr_ll_t *estr_ll;
	struct messagewall_estr_score_ll_t *estr_score_ll;
	int i,j,k,end;
	int ourpart;

	/*
	 * check our depth
	 */
	if (depth >= max_depth) {
		smtp_reject(client,"MIME","too deep",SMTP_MIMEDEEP,1,1,NULL,NULL);
		return 1;
	}

	/*
	 * check for # of parts
	 */
	(*part)++;
	if (*part >= max_parts) {
		smtp_reject(client,"MIME","too many parts",SMTP_MIMEPARTS,1,1,NULL,NULL);
		return 1;
	}

	ourpart = *part;
	clients[client].parts[ourpart].parent = parent;

	/*
	 * split message
	 */
	if (rfc822_split_message(message,&clients[client].parts[ourpart].message) != 0) {
		smtp_reject(client,"MIME","failed to split message",SMTP_BADMIME,1,1,NULL,NULL);
		return 1;
	}

	/*
	 * determine content type of this part
	 */
	value = rfc822_header_value(&clients[client].parts[ourpart].message,"Content-Type:");
	if (value == NULL) {
		clients[client].parts[ourpart].multipart = 0;
		if (parent != -1 && firestring_estr_strcasecmp(&clients[client].parts[parent].content_type,"multipart/digest") == 0)
			firestring_estr_strcpy(&clients[client].parts[ourpart].content_type,"message/rfc822");
		else
			firestring_estr_strcpy(&clients[client].parts[ourpart].content_type,"text/plain");
	} else {
		if (value->l > RFC822_VALUE_MAXLEN) {
			smtp_reject(client,"MIME","header value too long",SMTP_BADMIME,1,1,NULL,NULL);
			return 1;
		}
		firestring_estr_estrcpy(&clients[client].parts[ourpart].content_type,value,0);
		i = firestring_estr_strchr(&clients[client].parts[ourpart].content_type,';',0);
		if (firestring_estr_starts(&clients[client].parts[ourpart].content_type,"multipart/") == 0) {
			clients[client].parts[ourpart].multipart = 1;

			value = rfc822_parameter_value(&clients[client].parts[ourpart].content_type,"boundary");
			if (value == NULL) {
				/*
				 * multipart without boundary? i think not
				 */
				smtp_reject(client,"MIME","invalid message structure",SMTP_BADMIME,1,1,NULL,NULL);
				return 1;
			}
			clients[client].parts[ourpart].boundary.s = value->s;
			clients[client].parts[ourpart].boundary.l = value->l;
			clients[client].parts[ourpart].boundary.a = value->a;

			clients[client].parts[ourpart].content_type.l = i;
		} else {
			clients[client].parts[ourpart].multipart = 0;

			if (i != -1) {
				/*
			 	* extended parameters
			 	*/
				value2 = rfc822_parameter_value(value,"name");
				if (value2 != NULL) {
					clients[client].parts[ourpart].attachment = 1;
					clients[client].parts[ourpart].filename.s = value2->s;
					clients[client].parts[ourpart].filename.l = value2->l;
					clients[client].parts[ourpart].filename.a = value2->a;

					estr_score_ll = profile->filename_reject;
					while (estr_score_ll != NULL) {
						if (firestring_estr_estristr(&clients[client].parts[ourpart].filename,&estr_score_ll->string,0) != -1)
							if (smtp_reject(client,"MIME","illegal attachment filename: '%e' contains '%e'",SMTP_ATTACHMENT,estr_score_ll->score,0,&clients[client].parts[ourpart].filename,&estr_score_ll->string) == 1)
								return 1;
						estr_score_ll = estr_score_ll->next;
					}
				}
				clients[client].parts[ourpart].content_type.l = i;
			}
		}
	}

	estr_score_ll = profile->mime_reject;
	while (estr_score_ll != NULL) {
		if (firestring_estr_estrcasecmp(&clients[client].parts[ourpart].content_type,&estr_score_ll->string,0) == 0)
			if (smtp_reject(client,"MIME","illegal part: %e",SMTP_MIMEREJECT,estr_score_ll->score,0,&estr_score_ll->string,NULL) == 1)
				return 1;
		estr_score_ll = estr_score_ll->next;
	}

	value = rfc822_header_value(&clients[client].parts[ourpart].message,"Content-Transfer-Encoding:");
	if (value == NULL)
		firestring_estr_strcpy(&clients[client].parts[ourpart].content_transfer_encoding,"7bit");
	else {
		i = firestring_estr_strchr(value,';',0);
		if (i != -1)
			value->l = i;
		firestring_estr_estrcpy(&clients[client].parts[ourpart].content_transfer_encoding,value,0);
	}

	value = rfc822_header_value(&clients[client].parts[ourpart].message,"Content-Disposition:");
	if (value == NULL || firestring_estr_starts(value,"inline") == 0)
		clients[client].parts[ourpart].attachment = 0;
	else if (firestring_estr_starts(value,"attachment") == 0) {
		value2 = rfc822_parameter_value(value,"filename");
		if (value2 == NULL)
			clients[client].parts[ourpart].attachment = 0;
		else {
			clients[client].parts[ourpart].attachment = 1;
			clients[client].parts[ourpart].filename.s = value2->s;
			clients[client].parts[ourpart].filename.l = value2->l;
			clients[client].parts[ourpart].filename.a = value2->a;

			estr_score_ll = profile->filename_reject;
			while (estr_score_ll != NULL) {
				if (firestring_estr_estristr(&clients[client].parts[ourpart].filename,&estr_score_ll->string,0) != -1)
					if (smtp_reject(client,"MIME","illegal attachment filename: '%e' contains '%e'",SMTP_ATTACHMENT,estr_score_ll->score,0,&clients[client].parts[ourpart].filename,&estr_score_ll->string) == 1)
						return 1;

				estr_score_ll = estr_score_ll->next;
			}
		}
	} else {
		smtp_reject(client,"MIME","invalid content disposition: %e",SMTP_BADMIME,1,1,value,NULL);
		return 1;
	}

	if (firestring_estr_strcasecmp(&clients[client].parts[ourpart].content_transfer_encoding,"7bit") == 0
			|| firestring_estr_strcasecmp(&clients[client].parts[ourpart].content_transfer_encoding,"8bit") == 0
			|| firestring_estr_strcasecmp(&clients[client].parts[ourpart].content_transfer_encoding,"binary") == 0) {
		/*
		 * identity, do nothing
		 */
		if (clients[client].parts[ourpart].multipart == 0) {
			if (mime_checks(client,ourpart,&clients[client].parts[ourpart].message.body) == 1)
				return 1;
		}
	} else if (firestring_estr_strcasecmp(&clients[client].parts[ourpart].content_transfer_encoding,"base64") == 0) {
		/*
		 * check for illegal multipart encoding
		 */
		if (clients[client].parts[ourpart].multipart == 1) {
			smtp_reject(client,"MIME","illegal multipart encoding: base64",SMTP_BADMIME,1,1,NULL,NULL);
			return 1;
		}

		if (mime_base64(&clients[client].parts[ourpart].message.body,&mime_buffer) != 0) {
			smtp_reject(client,"MIME","invalid base64 encoding",SMTP_BADMIME,1,1,NULL,NULL);
			return 1;
		}

		if (mime_checks(client,ourpart,&mime_buffer) == 1)
			return 1;
	} else if (firestring_estr_strcasecmp(&clients[client].parts[ourpart].content_transfer_encoding,"quoted-printable") == 0) {
		/*
		 * check for illegal multipart encoding
		 */
		if (clients[client].parts[ourpart].multipart == 1) {
			smtp_reject(client,"MIME","illegal multipart encoding: quoted-printable",SMTP_BADMIME,1,1,NULL,NULL);
			return 1;
		}

		if (mime_qp(&clients[client].parts[ourpart].message.body) != 0) {
			smtp_reject(client,"MIME","invalid quoted-printable encoding",SMTP_BADMIME,1,1,NULL,NULL);
			return 1;
		}

		if (mime_checks(client,ourpart,&mime_buffer) == 1)
			return 1;
	} else {
		/*
		 * unknown encoding
		 */
		smtp_reject(client,"MIME","unknown encoding: %e",SMTP_BADMIME,1,1,&clients[client].parts[ourpart].content_transfer_encoding,NULL);
		return 1;
	}


	if (clients[client].parts[ourpart].multipart == 1) {
		struct firestring_estr_t childmessage;
		/*
		 * time to break out the parts
		 */
		j = -1;
		i = 0;
		while (1) {
multipart_again:
			i = firestring_estr_estrstr(&clients[client].parts[ourpart].message.body,&clients[client].parts[ourpart].boundary,i);

			if (i == -1) {
				smtp_reject(client,"MIME","unable to find boundary",SMTP_BADMIME,1,1,NULL,NULL);
				return 1;
			}

			/*
			 * boundary must be followed by linear whitespace and CRLF
			 */
			k = i + clients[client].parts[ourpart].boundary.l;
			if (clients[client].parts[ourpart].message.body.l > k + 1 && clients[client].parts[ourpart].message.body.s[k] == '-' && clients[client].parts[ourpart].message.body.s[k + 1] == '-') {
				end = 1;
				k += 2;
			} else
				end = 0;
			for (; k < clients[client].parts[ourpart].message.body.l - 1; k++) {
				if (clients[client].parts[ourpart].message.body.s[k] == '\r' && clients[client].parts[ourpart].message.body.s[k + 1] == '\n')
					break;
				if (strchr(RFC822_WHITESPACE,clients[client].parts[ourpart].message.body.s[k]) == NULL) {
					/*
					 * not a boundary
					 */
					i++;
					goto multipart_again;
				}
			}

			if (k == clients[client].parts[ourpart].message.body.l - 1) {
				smtp_reject(client,"MIME","cannot find end of boundary line",SMTP_BADMIME,1,1,NULL,NULL);
				return 1;
			}

			if (i < 2 ||
				   clients[client].parts[ourpart].message.body.s[i-1] != '-' ||
					 clients[client].parts[ourpart].message.body.s[i-2] != '-' ||
					 clients[client].parts[ourpart].message.body.s[i-3] != '\n' ||
					 clients[client].parts[ourpart].message.body.s[i-4] != '\r') {
				i++;
				continue;
			}

			if (j != -1) {
				/*
				 * there was a previous part, and we now know where it ends
				 */
				childmessage.s = &clients[client].parts[ourpart].message.body.s[j];
				childmessage.a = childmessage.l = i - j - 4;
				if (mime_parse(client, depth + 1, part, ourpart, &childmessage, profile) == 1)
					return 1;
			}

			i += clients[client].parts[ourpart].boundary.l;
			if (end == 1)
				/* 
				 * end of multipart
				 */
				break;

			i = k + 2;

			j = i;
		}
		/*
		 * we're only visible if one or more of our children is
		 */
		clients[client].parts[ourpart].visible = 0;
		for (i = ourpart + 1; i <= *part; i++)
			if (clients[client].parts[i].visible == 1) {
				clients[client].parts[ourpart].visible = 1;
				break;
			}
	} else {
		if (profile->mime_allow == NULL) {
			/*
			 * visible by default, check mime_strip
			 */
			clients[client].parts[ourpart].visible = 1;
			estr_ll = profile->mime_strip;
			while (estr_ll != NULL) {
				if (firestring_estr_estrcasecmp(&estr_ll->string,&clients[client].parts[ourpart].content_type,0) == 0) {
					clients[client].parts[ourpart].visible = 0;
					break;
				}
				estr_ll = estr_ll->next;
			}
		} else {
			/*
			 * invisible by default, check mime_allow
			 */
			estr_ll = profile->mime_allow;
			while (estr_ll != NULL) {
				if (firestring_estr_estrcasecmp(&estr_ll->string,&clients[client].parts[ourpart].content_type,0) == 0) {
					clients[client].parts[ourpart].visible = 1;
					break;
				}
				estr_ll = estr_ll->next;
			}
		}
	}

	return 0;
}

void mime_startup() {
	int i,j;

	i = 0;

	memset(mime_base64_table,66,256);

	for (j = 'A'; j <= 'Z'; j++)
		mime_base64_table[j] = i++;

	for (j = 'a'; j <= 'z'; j++)
		mime_base64_table[j] = i++;

	for (j = '0'; j <= '9'; j++)
		mime_base64_table[j] = i++;

	mime_base64_table['+'] = i++;
	mime_base64_table['/'] = i++;
	mime_base64_table['='] = 0;


	i = 0;
	
	memset(mime_qp_table,66,256);

	for (j = '0'; j <= '9'; j++)
		mime_qp_table[j] = i++;

	for (j = 'A'; j <= 'F'; j++)
		mime_qp_table[j] = i++;

	firestring_estr_alloc(&mime_buffer,max_message_size);
}

int mime_base64(struct firestring_estr_t *chunk, struct firestring_estr_t *destination) {
	int i,g;
	unsigned char group[4];
	unsigned char pregroup[4];

	destination->l = 0;

	for (i = 0, g = 0; i < chunk->l; i++) {
		if (mime_base64_table[(unsigned int) chunk->s[i]] != 66) {
			pregroup[g] = chunk->s[i];
			group[g++] = mime_base64_table[(unsigned char) chunk->s[i]];
			if (g == 4) {
				/*
				 * got a complete group
				 */
				destination->s[destination->l++] = (group[0] << 2 | group[1] >> 4);
				destination->s[destination->l++] = (group[1] << 4 | group[2] >> 2);
				destination->s[destination->l++] = (group[2] << 6 | group[3]);
				if (pregroup[0] == '=' || pregroup[1] == '=')
					return 1;
				if (pregroup[2] == '=') {
					destination->l -= 2;
					return 0;
				}
				if (pregroup[3] == '=') {
					destination->l -= 1;
					return 0;
				}
				g = 0;
			}
		}
	}
	return 0;
}

int mime_qp(struct firestring_estr_t *chunk) {
	int i,j,escape;

	mime_buffer.l = 0;
	escape = 0;

	for (i = 0; i < chunk->l; i++) {
		if (strchr(RFC822_WHITESPACE,chunk->s[i]) != NULL) {
			/*
			 * whitespace
			 * if it ends in CRLF, we can skip ahead
			 */
			for (j = i + 1; j < chunk->l - 1; j++) {
				if (chunk->s[j] == '\r' && chunk->s[j+1] == '\n') {
					i = j;
					break;
				}
				if (strchr(RFC822_WHITESPACE,chunk->s[j]) == NULL)
					break;
			}
		}
		if (chunk->s[i] == '=') {
			/*
			 * escape sequence
			 */
			escape = 1;
			continue;
		}
		if (escape == 1) {
			if (chunk->l <= i + 1)
				return 1;
			if (chunk->s[i] == '\r' && chunk->s[i+1] == '\n') {
				/* 
				 * soft line break, just ignore it
				 */
			} else {
				/*
				 * real escape sequence
				 */
				if (mime_qp_table[(int) chunk->s[i]] == 66 || mime_qp_table[(int) chunk->s[i+1]] == 66)
					return 1;
				mime_buffer.s[mime_buffer.l++] = (char) mime_qp_table[(int) chunk->s[i]] * 16 + mime_qp_table[(int) chunk->s[i+1]];
			}
			i++;
			escape = 0;
		} else
			mime_buffer.s[mime_buffer.l++] = chunk->s[i];
	}
	return 0;
}

int mime_build_message(int client, int backend, int *part) {
	int ourpart;
	int i;
	/*
	 * create the message in the backend buffer
	 * using the mime tree
	 */

	ourpart = *part;

	/*
	 * if we're first in, add the warning header if its there
	 */
	if (ourpart == 0) {
		firestring_estr_sprintf(&backends[backend].message,"X-MessageWall-Score: %d (%s)\r\n",clients[client].score,domain);
		for (i = 0; i < clients[client].num_warnings; i++) {
			firestring_estr_strcat(&backends[backend].message,"X-MessageWall-Warning: ");
			firestring_estr_estrcat(&backends[backend].message,&clients[client].warnings[i],0);
		}
	}


	/*
	 * header goes in either way
	 */
	for (i = 0; i < clients[client].parts[ourpart].message.header.l; i++) {
		if (clients[client].parts[ourpart].message.header.s[i] == '.') {
			/* 
			 * optimize this check for lack of bare newlines
			 */
			if (i == 0 || clients[client].parts[ourpart].message.header.s[i-1] == '\n')
				backends[backend].message.s[backends[backend].message.l++] = '.';
			backends[backend].message.s[backends[backend].message.l++] = clients[client].parts[ourpart].message.header.s[i];
		} else
			backends[backend].message.s[backends[backend].message.l++] = clients[client].parts[ourpart].message.header.s[i];
	}

	if (clients[client].parts[ourpart].multipart == 1) {
		backends[backend].message.s[backends[backend].message.l++] = '\r';
		backends[backend].message.s[backends[backend].message.l++] = '\n';
		while (*part < clients[client].num_parts && clients[client].parts[(*part) + 1].parent == ourpart) {
			(*part)++;
			if (clients[client].parts[*part].visible == 0)
				continue;
			backends[backend].message.s[backends[backend].message.l++] = '\r';
			backends[backend].message.s[backends[backend].message.l++] = '\n';
			backends[backend].message.s[backends[backend].message.l++] = '-';
			backends[backend].message.s[backends[backend].message.l++] = '-';
			for (i = 0; i < clients[client].parts[ourpart].boundary.l; i++)
				backends[backend].message.s[backends[backend].message.l++] = clients[client].parts[ourpart].boundary.s[i];
			backends[backend].message.s[backends[backend].message.l++] = '\r';
			backends[backend].message.s[backends[backend].message.l++] = '\n';
			mime_build_message(client,backend,part);
		}
		backends[backend].message.s[backends[backend].message.l++] = '\r';
		backends[backend].message.s[backends[backend].message.l++] = '\n';
		backends[backend].message.s[backends[backend].message.l++] = '-';
		backends[backend].message.s[backends[backend].message.l++] = '-';
		for (i = 0; i < clients[client].parts[ourpart].boundary.l; i++)
			backends[backend].message.s[backends[backend].message.l++] = clients[client].parts[ourpart].boundary.s[i];
		backends[backend].message.s[backends[backend].message.l++] = '-';
		backends[backend].message.s[backends[backend].message.l++] = '-';
		backends[backend].message.s[backends[backend].message.l++] = '\r';
		backends[backend].message.s[backends[backend].message.l++] = '\n';
	} else {
		if (clients[client].parts[ourpart].message.body.l > 0) {
			backends[backend].message.s[backends[backend].message.l++] = '\r';
			backends[backend].message.s[backends[backend].message.l++] = '\n';
		}
		/*
		 * copy body into output buffer, smtp quoting as we go
		 */
		for (i = 0; i < clients[client].parts[ourpart].message.body.l; i++) {
			if (clients[client].parts[ourpart].message.body.s[i] == '.') {
				/* 
				 * optimize this check for lack of bare newlines
				 */
				if (i == 0 || clients[client].parts[ourpart].message.body.s[i-1] == '\n')
					backends[backend].message.s[backends[backend].message.l++] = '.';
				backends[backend].message.s[backends[backend].message.l++] = clients[client].parts[ourpart].message.body.s[i];
			} else
				backends[backend].message.s[backends[backend].message.l++] = clients[client].parts[ourpart].message.body.s[i];
		}
	}

	if (ourpart == 0) {
		/*
		 * end of message
		 */
		backends[backend].message.s[backends[backend].message.l++] = '.';
		backends[backend].message.s[backends[backend].message.l++] = '\r';
		backends[backend].message.s[backends[backend].message.l++] = '\n';
	}
	return 0;
}

int mime_checks(int client, int ourpart, struct firestring_estr_t *body) {
	dnsdcc_send_queries(client,ourpart,dnsdcc_checksum_aa,body);
	if (firestring_estr_strcasecmp(&clients[client].parts[ourpart].content_type,"text/plain") == 0)
		dnsdcc_send_queries(client,ourpart,dnsdcc_checksum_ba,body);

	if (rfc822_body_reject_check(client,body) == 1 || rfc822_body_rejecti_check(client,body) == 1)
		return 1;

	if (virus_check(client,body) == 1)
		return 1;
	return 0;
}
