/*
**  POP3Folder.m
**
**  Copyright (c) 2001, 2002
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**  
**  This library is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
**  Lesser General Public License for more details.
**  
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
#import <Pantomime/POP3Folder.h>

#import <Pantomime/Constants.h>
#import <Pantomime/LocalFolder.h>
#import <Pantomime/Message.h>
#import <Pantomime/POP3CacheManager.h>
#import <Pantomime/POP3CacheObject.h>
#import <Pantomime/POP3Message.h>
#import <Pantomime/POP3Store.h>
#import <Pantomime/TCPConnection.h>
#import <Pantomime/NSDataExtensions.h>

@implementation POP3Folder

- (id) initWithName: (NSString *) theName
{
  self = [super initWithName: theName];

  [self setLeaveOnServer: YES];
  [self setRetainPeriod: 0];

  return self;
}


//
//
//
- (void) dealloc
{
  RELEASE(pop3CacheManager);

  [super dealloc];
}


//
//
//
- (void) deleteMessageAtIndex: (int) theIndex
{
  NS_DURING
    {
      POP3Store *aStore;
      
      aStore = (POP3Store *)[self store];
      
      [[aStore tcpConnection] writeLine: [NSString stringWithFormat: @"DELE %i", theIndex]];
      
      if (! [aStore responseFromServerIsValid: NULL] ) 
	{
	  NSLog(@"Unable to delete the message on the POP3 server...");
	}
    }
  NS_HANDLER
    {
      NSLog(@"POP3Folder: An error occured while deleting the message (%d) from the POP3 server.", theIndex);
    }
  NS_ENDHANDLER
}

//
// This method is used to fetch a message at a specified index
// in the POP3 Inbox folder.
//
- (NSData *) prefetchMessageAtIndex: (int) theIndex
{
  POP3CacheObject *aPOP3CacheObject;
  NSString *anUIDString;
  POP3Store *aStore;
  
  NSAutoreleasePool *pool;

  // We create our local autorelease pool
  pool = [[NSAutoreleasePool alloc] init];

  // We obtain the pointer to our store
  aStore = (POP3Store *)[self store];

  // We ask for the UIDL of the message and we read it
  [[aStore tcpConnection] writeLine: [NSString stringWithFormat: @"UIDL %i", theIndex]];
  anUIDString = [self readUID];
  
  // Next we verify if we really need to transfer the message
  if ( [self pop3CacheManager] )
    {
      aPOP3CacheObject = [[self pop3CacheManager] findPOP3CacheObject: anUIDString];
    }
  else
    {
      aPOP3CacheObject = nil;
    }
  
  if (! aPOP3CacheObject )
    {
      NSMutableData *aMutableData;
 
      BOOL endOfHeaders, seenStatusHeader;
      int length;
      
      // The UID hasn't been found so we first cache it
      if ( [self pop3CacheManager] )
	{
	  aPOP3CacheObject = [[POP3CacheObject alloc] initWithUID: anUIDString
						      date: [NSCalendarDate calendarDate]];
	  [[self pop3CacheManager] addObject: aPOP3CacheObject];
	}
      
      // Then we transfer our messsage by getting the length of the message first
      [[aStore tcpConnection] writeLine: [NSString stringWithFormat: @"LIST %i", theIndex]];
      length = [self readLength];

      // We then ask for the complete message
      [[aStore tcpConnection] writeLine: [NSString stringWithFormat: @"RETR %i", theIndex]];
      
      if (! [aStore responseFromServerIsValid: NULL] )
	{
	  NSLog( @"Invalid request for message index = %d", theIndex );
	  RELEASE(pool);
	  return nil;
	}

      aMutableData = [[NSMutableData alloc] initWithCapacity: length];
      endOfHeaders = NO;
      seenStatusHeader = NO;
      
      while ( YES )
	{
	  NSString *theLine;
	  
	  theLine = [[aStore tcpConnection] readLine];
	  
	  if ( [theLine hasSuffix: @"\r\n"] )
	    {
	      // We verify if we're at the end of the message
	      if ( ([theLine length] > 2) && [theLine hasPrefix: @".\r\n"] )
		{
		  break;
		}
	      // We verify if we are at the end of the headers, if so we add our UID
	      else if (!endOfHeaders && [theLine isEqualToString:@"\r\n"] )
		{  
		  // If we are at the end of the headers but we haven't seen the
		  // Status: header, let's add it.
		  if (! seenStatusHeader)
		    {
		      theLine = [NSString stringWithFormat: @"Status:  \n"];
		    }
		  else
		    {
		      theLine = @"";
		    }

		  theLine =  [NSString stringWithFormat: @"%@X-UID: %@\n\n", 
				       theLine,
				       anUIDString];

	      	  endOfHeaders = YES;
	      	}
	      
	      // If not, replace \r\n (CRLF) by \n (LF)
	      else
		{
		  NSString *theChoppedLine;
		  
		  theChoppedLine = [theLine substringToIndex: ([theLine length] - 2)];
		  theLine = [NSString stringWithFormat: @"%@\n",
				      theChoppedLine];

		  if ( !endOfHeaders && [theLine hasPrefix: @"Status: "] )
		    {
		      seenStatusHeader = YES;
		    }
		}
	    }
	  
	  // We append the line in our message
	  [aMutableData appendCString: [theLine cString]];
	  
	} // while ( YES ) ...
	 
      
      // We synchronize our POP3UIDManager and we return the message that has been read
      [[self pop3CacheManager] synchronize];
     
      // We release our local pool
      RELEASE(pool);

      return AUTORELEASE(aMutableData);
    } // if (! aPOP3CacheObject )
  
  RELEASE(pool);
  return nil;
}

//
// This method is used to cache the messages from the POP3 server
// locally (in memory).
//
- (BOOL) prefetch
{
  BOOL didTransferMessages;
  NSData *aData;
  POP3Message *aMessage;
  int i, count;

  didTransferMessages = NO;
  count = [self count];

  for (i = 1; i <= count; i++)
    {
      aData = [self prefetchMessageAtIndex: i];
      
      if ( aData )
	{
	  // The data read is valid, we can now add the message to our folder
	  aMessage = [[POP3Message alloc] initWithData: aData];
	  
	  // We set some initial properties to our message
	  [aMessage setInitialized: YES];
	  [aMessage setMessageNumber: i];
	  [aMessage setFolder: self];
	  [aMessage setSize: [aData length]];
	  
	  [self appendMessage: aMessage];
	  RELEASE(aMessage);

	  didTransferMessages = YES;
	}
    } // for
  
  // We mark it as deleted if we need to
  if (! [self leaveOnServer] )
    {
      for (i = 1; i <= count; i++)
	{
	  [self deleteMessageAtIndex: i];
	}
    }
  else if ( [self retainPeriod] > 0 )
    {

      for (i = 1; i <= count; i++)
	{
	  POP3CacheObject *pop3CacheObject;
	  NSString *anUIDString;
	  
	  // We ask for the UIDL of the message and we read it
	  [[(POP3Store *)[self store] tcpConnection] writeLine: [NSString stringWithFormat: @"UIDL %i", i]];
	  anUIDString = [self readUID];
	  
	  // We get our POP3 cached object
	  pop3CacheObject = [[self pop3CacheManager] findPOP3CacheObject: anUIDString];

	  if ( pop3CacheObject )
	    {
	      NSCalendarDate *date;
	      int days;

	      // We get the days interval between our two dates
	      date = [NSCalendarDate calendarDate];
	      [date years: NULL
		    months: NULL
		    days: &days
		    hours: NULL
		    minutes: NULL
		    seconds: NULL
		    sinceDate: [pop3CacheObject date]];
	      
	      if ( days >= [self retainPeriod] )
		{
		  NSLog(@"Deleting message with UID %@ since it's %d days old", anUIDString, days);
		  [self deleteMessageAtIndex: i];
		}
	    }
	}
    }
  
  return didTransferMessages;
}


//
// This method returns the number of message in this POP3 folder.
// It re-implements the method of the support class.
// 
- (int) count
{
  NS_DURING
    {
      POP3Store *aStore;
      NSString *aString;
      int n, size;
      
      aStore = (POP3Store *)[self store];
      n = size = 0;

      // We sent the STAT command to our POP3 server
      [[aStore tcpConnection] writeLine: @"STAT"];
  
      // We read back the response and we put it in our buffer buf
      aString = [[aStore tcpConnection] readLine];
      
      if (! aString )
	{
	  NSLog(@"POP3Folder: An error occured while STATing the POP3 folder.");
	  NS_VALUERETURN(0, int);
	}
      
      // We get the number of messages in the mailbox by scanning our buffer
      sscanf([aString cString],"+OK %i %i\r\n", &n, &size);
      
      NS_VALUERETURN(n, int);
    }
  NS_HANDLER
    {
      NSLog(@"POP3Folder: An error occured while obtaining the message count from the POP3 server.");
    }
  NS_ENDHANDLER

  return 0;
}


//
// This method does nothing.
//
- (void) close
{
  // We do nothing.
}


//
// This method is used to read the UID (obtained from the command UIDL)
// of a message.
//
- (NSString *) readUID
{
  NS_DURING
    {
      NSString *aString;
      int messageNumber;
      char uniqueID[71];
      
      aString = [[(POP3Store *)[self store] tcpConnection] readLine];

      bzero(uniqueID, 71);
      sscanf([aString cString],"+OK %i %s\r\n", &messageNumber, uniqueID);
      
      NS_VALUERETURN([NSString stringWithCString: uniqueID], NSString *);
    }
  NS_HANDLER
    {
      NSLog(@"POP3Folder: An error occured while reading the UID of a message from the POP3 server.");
    }
  NS_ENDHANDLER

  return nil;
}

//
// This command is used to read the length of a message
// obtained by the command LIST
//
- (int) readLength
{
  NS_DURING
    {
      int messageNumber, messageLength;
      NSString *aString;
      
      aString = [[(POP3Store *)[self store] tcpConnection] readLine];
      
      sscanf([aString cString],"+OK %i %i\r\n", &messageNumber, &messageLength);
      
      NS_VALUERETURN(messageLength, int);
    }
  NS_HANDLER
    {
      NSLog(@"POP3Folder: An error occured while reading the length of a message from the POP3 server.");
    }
  NS_ENDHANDLER

  return 0;
}

//
// Return YES/NO depending if we leave or not the messages on the POP3 Server
//
- (BOOL) leaveOnServer
{
  return leaveOnServer;
}

//
// This method is used to set the flag to leave or not the messages on the POP3
// server after fetching them.
//
- (void) setLeaveOnServer: (BOOL) aBOOL
{
  leaveOnServer = aBOOL;
}


//
//
//
- (int) retainPeriod
{
  return retainPeriod;
}

- (void) setRetainPeriod: (int) theRetainPeriod
{
  retainPeriod = theRetainPeriod;
}

//
// This method transfer all the messages (while respecting leaveOnServer)
// to the specified folder. It returns the number of messages transferred
// or -1 if an error occured.
//
- (int) transferMessagesToFolder: (Folder *) theFolder
{
  NSData *aData;
  int i, count, transferredMessageCount;
  
  if (! theFolder )
    {
      return -1;
    }
  
  transferredMessageCount = 0;
  count = [self count];
  
  for (i = 1; i <= count; i++)
    {
      aData = [self prefetchMessageAtIndex: i];
      
      if ( aData )
	{
	/* LocalFolder will add "From -\n" itself it it is necessary */
	  [(LocalFolder *) theFolder appendMessageFromRawSource: aData];
	  transferredMessageCount++;
	}
    } /* for */
  
  // We mark the messages as deleted if we need to
  if (! [self leaveOnServer] )
    {
      for (i = 1; i <= count; i++)
	{
	  [self deleteMessageAtIndex: i];
	}
    }

  return transferredMessageCount;
}


//
//
//
- (POP3CacheManager *) pop3CacheManager
{
  return pop3CacheManager;
}


//
//
//
- (void) setPOP3CacheManager: (POP3CacheManager *) thePOP3CacheManager
{
  RETAIN(thePOP3CacheManager);
  RELEASE(pop3CacheManager);
  pop3CacheManager = thePOP3CacheManager;
}


//
// This method does nothing except returning an empty array.
// Expunging a POP3 folder doesn't make a lot of sense.
//
- (NSArray *) expunge: (BOOL) returnDeletedMessages
{
  return [NSArray array];
}

@end
