/*******************************************************************************
 *                                                                             *
 * Program:   paul                                                             *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                  *
 *            (L)aserbildern                                                   *
 * Uses:      GTK+ 1.2, Imlib                                                  *
 * Modul:     picuti.c                                                         *
 *            Miscellaneous funktions to handle paul structures and image data *
 * Author:    Andreas Tille                                                    *
 * Date:      18.02.1998                                                       *
 * Copyright: Andreas Tille, 1999; GNU Public License                          *
 *                                                                             *
 *******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/param.h>

#include <gdk_imlib.h>
#include <gdk/gdk.h>

#include "paul.h"
#include "names.h"

PICTURE *_BILD(GList *l, char *file, int line) 
/* extracts PICTURE-data from list
 */
{
  if ( !l ) return NULL;
  if ( IS_PICTURE((l->data)) ) return (PICTURE *)(l->data);
  g_warning(_("No PICTURE structure in list (file %s, line %i)!"), file, line);
  return NULL;
}

int _CheckPicList(GList *pl, char *file, int line) 
/* Check if picture list is not empty and contains at least one valid image
 */
{
  if ( !pl ) {
    g_warning(_("Empty picture list (file %s, line %i)!"), file, line);
    return 0;
  }
  return ( _BILD(pl, file, line) ? 1 : 0 ) ;
}

int CheckOperating(PAUL *p) 
/* Check if paul has a valid operating image
 */
{
  if ( !IS_PAUL(p) ) return 0;
  if ( !IS_PICTURE(p->op) ) {
    g_warning(_("Please select an %s by clicking a mouse button\nwhile holding down the <Ctrl>-key."), operating);
    return 0;
  }
  return 1;
}

int NBILDER(GList *l)
/* Get number of pictures in list
 */
{
  register int n = 0;
   
  if ( !l ) return 0;
  do {
    if ( BILD(l) ) n++;
  } while ( (l = l->next) );
   
  return n;
}

void Backup(const char *name)
/* create backup <name>~
 * --- Parameter: ---
 * char *name     : filename to backup
 */
{
  char  *bname;

  if ( !CheckFile(name) ) return;   
  bname = g_strconcat(name, "~", NULL);
  rename(name, bname);
  FREE(bname);
}

GdkImlibImage *ChangedRGBData(GdkImlibImage *im, unsigned char **data, int w, int h)
/* since ImLib 1.8.2 the changing of the rgb_data is not allowed so we have to
 * build a new ImlibStructure
 * --- Parameter: ---
 * GdkImlibImage *im               : old ImLib structure
 * char         **data             : the RGB Data to store in the new ImLib structure
 * int            w                : possibly new width
 * int            h                : possibly new height
 * --- Return: ---
 * GdkImlibImage *ChangedRGBData() : new ImLib structure
 * char         **data             : my own copy of the rgb_data pointer
 */
{
  GdkImlibImage *new;

  g_return_val_if_fail ( *data, NULL );
  g_return_val_if_fail ( (w || h) , NULL );
  if ( im ) gdk_imlib_kill_image(im);
  
  g_return_val_if_fail ( (new = gdk_imlib_create_image_from_data(*data, NULL, w, h)), NULL );
  FREE(*data);
  *data = new->rgb_data;
  return new;
}

char *CheckFile(const char *file)
/* Check, whether it is an existing file but no directory
 * --- Parameter: ---
 * char *file        : filename to check
 * --- Return: ---
 * char *CheckFile() : file if OK, else NULL
 */
{
  struct stat stat_buf;

  g_return_val_if_fail ( file, NULL );
  if ( stat(file, &stat_buf) ) return NULL;
  if ( S_ISDIR(stat_buf.st_mode) ) return NULL;
  return (char *)file;
}

int CreateShowPixmaps(GList *piclist)
/* create Imlib structures to given image data or tells imlib that data changed
 * --- Parameter ---
 * GList *piclist : list of images
 * --- Return ---
 * GList *piclist : list of images with valid Imlib data
 */
{
  PICTURE *bild;
  GList   *pl;

  g_return_val_if_fail ( bild = BILD(pl = piclist), RET_ERR );
  
  for ( ; pl; bild = BILD(pl = pl->next) ) {
    if ( !bild->im ) 
      g_return_val_if_fail ((bild->im = at_imlib_set_image_from_data(&(bild->DATA), bild->W, bild->H)),
                            RET_ERR);
    else
      gdk_imlib_changed_image(bild->im);
  }
  return RET_OK;
}

void FreeBild(PICTURE *bild)
/* free all allocated memory from a single PICTURE strukture
 * --- Parameter: ---
 * PICTURE *bild : structure to free
 */
{
  g_return_if_fail ( IS_PICTURE(bild) ) ;
  FreeSpecs(bild->spec);
  if ( bild->im ) { 
    gdk_imlib_kill_image(bild->im);
    bild->im   = NULL;
    bild->DATA = NULL;
  } else {
    FREE(bild->DATA);
  }
  FREE(bild->file); 
  FREE(bild->ext);
  FREE(bild->dir);
  FREE(bild->his);
#ifdef HAVE_LIBFFTW
  FREE(bild->fftprof);
#endif
  if ( bild->n_gamma ) FREE(bild->gamma);
    /* if bild->label is destroyed, which would make sense in case of using it as     *
     * window of operating image, the functions which relay on (bild->label)->parent  *
     * as list_item widget will fail!!!  Leave that out here                          *
     * if ( bild->label   ) gtk_widget_destroy(bild->label);                          */
  FREE(bild);
}

void FreePiclist(GList *piclist)
/* free memory from a whole list of PICTURE structures as usual is used in paul
 * --- Parameter: ---
 * GList *piclist     : list of PICTURES to free
 */
{
  PICTURE *bild;
  GList   *pl;

  if ( !piclist ) return;
  
  for ( bild = BILD(pl = piclist); pl; bild = BILD(pl = pl->next) ) 
    FreeBild(bild);
}

void FreePaul(PAUL *p)
/* Try to free all allocated memory
 */
{
  g_return_if_fail ( IS_PAUL(p) );
   
  FreePiclist(p->piclist);
  G_LIST_FREE(p->piclist);
  if ( p->opt ) {  /* should be true in any case, but who knows :) */
    FREE(p->opt->bright);
    FREE(p->opt->html);
    FREE(p->opt->link);
    FREE(p->opt->cut);
  }
  if ( p->op ) FreeBild(p->op);
#ifdef PHYSICAL_PAUL
  FreePhysical(&PG);
#endif
}

void FreeSpecs(CHUNK *spec)
/* Free chunks for spezifikation
 * --- Parameter: ---
 * CHUNK *spec: chunks to make free
 */
{
  CHUNK *cp;
   
  if ( spec ) {
    for ( cp = spec; cp->key != NULL; cp++ ) 
      if ( cp->typ == CT_LTEXT || cp->typ == CT_STRING ) 
        FREE(cp->val.s);
    FREE(spec);
  }
}

int GetListPosition(GList *list, GList *elem)
/* calculates position from elem in list
 */
{
  register int n = 0;
  register GList *l = list;
   
  if ( !l ) return -1;
  while ( l && l != elem ) {
    l = l->next;
    n++;
  }
  if ( !l ) return -1;
  return n;
}

gchar *ImgFileName(PICTURE *bild)
/* return complete filename with extension if it is set
 */
{
  gchar *name;
  
  g_return_val_if_fail ( IS_PICTURE(bild), NULL);
  g_return_val_if_fail ( bild->file, NULL );
  
  if ( bild->ext ) name = g_strdup_printf("%s.%s", bild->file, bild->ext);
  else             name = g_strdup(bild->file);
  return name;
}

void ImgChunksUpdate(PICTURE *bild, char *type, char *desc, char *app, long flag)
/* Do the necessary things after an operation
 * --- Parameter: ---
 * PICTURE *bild : image to update
 * char    *type : string to insert as type chunk
 * char    *desc : string to insert as Description chunk
 * char    *app  : appendix for filename
 * char     flag : flag to set
 */
{
  g_return_if_fail ( IS_PICTURE(bild) ) ;
   
  if ( type ) {
    C_VALUE cv;
   
    cv.s = type;
    CopySpec(bild->spec, ChunkNameTyp, cv);
  }
  if ( desc ) CreateDescription(bild, desc);
  if ( app  ) NewFileName(bild, app);
  bild->flag |= flag;
}

int NewImage(PICTURE *bild, int w_new, int h_new, unsigned char *data)
/* Set new image data
 * --- Parameter: ---
 * PICTURE      *bild       : picture structur
 * int           w_new      : new width
 * int           h_new      : new height
 * unsigned char data       : new image data
 * --- Return: ---
 * PICTURE      *bild       : new picture structur
 * int           NewImage() : RET_ERR or RET_OK
 */
{
  g_return_val_if_fail ( IS_PICTURE(bild), RET_ERR );

  if ( bild->im ) {
    gdk_imlib_kill_image(bild->im);
    bild->im   = NULL;  /* make sure, that bild->im is created later */
  } else 
    FREE(bild->DATA);
  bild->DATA = data;
  bild->W    = w_new;
  bild->H    = h_new;
  return RET_OK;
}

BOX *ReadBoxFromString(char *string)
/* read BOX coordinates from string, if neccessary sort to (left,top)-(right,bottom)
 * --- Parameter: ---
 * char *string             : string to read
 * --- Return: ---
 * BOX  *ReadBoxFromString(): BOX, NULL if an error occured
 */
{
  int i = 0;
  BOX *cut;
   
  g_return_val_if_fail ( string, NULL ) ;
  cut = g_new(BOX, 1);
   
  if ( strchr(string, '_' ) )
    i = sscanf(string, "%i,%i_%i,%i", &(cut->x1), &(cut->y1), &(cut->x2), &(cut->y2));
  else if ( strchr(string, '+' ) ) {
    i = sscanf(string, "%ix%i+%i+%i", &(cut->x2), &(cut->y2), &(cut->x1), &(cut->y1));
    cut->x2 += cut->x1;
    cut->y2 += cut->y1;
  } 
  if ( i != 4 ) {
    g_warning(_("Invalid box format! Use: \"x0,y0_x1,y1\" or \"wxh+x0+y0\""));
    FREE(cut);
    return NULL;
  }
  if ( cut->x1 < 0 || cut->y1 < 0 || cut->x2 < 0 || cut->y2 < 0 )  {
    g_warning(_("Box coordinates out of range"));
    FREE(cut);
    return NULL;
  }
  if ( cut->x1 == cut->x2 && cut->y1 == cut->y2 ) {
    g_warning(_("(left,top) == (right,buttom) ... zero boxes are not supported"));
    FREE(cut);
    return NULL;
  }
  return cut;
}

char *RemoveUnderscore(char *string)
/* Remove underscores ('_') from a string (usefull to find menu entries)
 * --- Parameter: ---
 * char *string            : string possibly containing underscores
 * --- Return: ---
 * char *RemoveUnderscore(): new allocated string, without underscores
 */
{
  register char *sp, *np;
  char          *new;
  
  np = (new = g_new0(char, strlen(string))) - 1;
  sp = string - 1;
  while ( *++sp ) 
    if ( *sp != '_' ) *++np = *sp;
  return new;
}

static int IsMonochrome(PICTURE *bild)
/* test, whether all three byte of one pixel are the same
 * --- Parameter: ---
 * PICTURE *bild     : possibly monochrome image
 * --- Return: ---
 * int IsMonochrome(): 1 if monochrome, else 3 (samples per pixel!)
 */
{
  register unsigned char *ap, *fip;
   
  g_return_val_if_fail ( IS_PICTURE(bild), 0 );
  for ( fip = (ap = bild->DATA) + bild->size; ap < fip; ap += 3 )
    if ( (*ap != *(ap+1) || *ap != *(ap+2)) &&
         (*ap || *(ap+2)) ) /* test, whether it is a paul specific green image */
      return 3;

  return 1;
}
   
static int Gray2Green(unsigned char *data, unsigned int n)
/* make green image from monochrome by setting RGB -> 0X0 
 * paul uses green images to use the other colours for different purpose
 * THERE IS NO TEST IF THE SOURCE IMAGE IS MONOCHROME OR NOT
 * --- Parameter: ---
 * unsigned char *data: RGB data of monochrome source image
 * unsigned int   n   : number of RGB pixels
 * --- Return: ---
 * unsigned char *data: RGB data of the created green image where R=0 and B=0
 * int Gray2Green()   : RET_ERR or RET_OK
 */
{
  unsigned char *ap, *fip;
   
  g_return_val_if_fail ( data, RET_ERR ) ;

  for ( fip = (ap = data) + 3*n; ap < fip; ap += 3 ) 
    *ap = *(ap+2) = 0;
      
  return RET_OK;
}

char *NewFileName(PICTURE *bild, char *appendix)
/* Creates a new file name from the old one and an appendix
 * deletes old extension if neccessary
 * --- Parameter: ---
 * PICTURE *bild         : picture structure with old name
 * char    *appendix     : Appendix
 * --- Return: ---
 * PICTURE *bild         : picture structure with new name
 * char    *NewFileName(): new name
 */
{
  char *filename;

  g_return_val_if_fail ( IS_PICTURE(bild), NULL );
  g_return_val_if_fail ( (bild->file), NULL );
  if ( !appendix   ) appendix = "";
   
  filename = g_strdup_printf("%s%s", bild->file, appendix);
  FREE(bild->file);
  FREE(bild->ext);
  return bild->file = filename;
}

GList *ReadPic(char *file, OPTIONS *opt)
/* Read image file of any format and initialize PICTURE structure
 * --- Parameter: ---
 * char    *file       : filename of image file
 * OPTIONS *opt        : flag : what to do 
 *                     : shr  : level of shrinkage
 *                     : opt == NULL for images to insert
 * --- Return: ---
 * GList *ReadPic()    : list with new PICTURE structures containing
 *                       contents of file or NULL
 */
{
  PICTURE      *bild;
  char         *buf;
  long          f;
  GList        *pl, *loadlist;
  register  int flag;
   
  if ( !CheckFile(file) )  return NULL;
  g_return_val_if_fail ( (bild = InitSpec()), NULL );

  if ( opt ) f = opt->f;
  else       f = 0;
  
  if ( (loadlist = LoadPicture(bild, file, f)) == NULL ) {
    g_warning(_("Unable to load %s."), file);
    return NULL;
  }
  for ( bild = BILD(pl = loadlist), flag = 0; pl; bild = BILD(pl = pl->next), flag++ ) {
    printf("%s\n", buf = ImgFileName(bild));
    FREE(buf);
    bild->im       = NULL;
    if ( !opt && !flag ) continue;  /* images to insert doesn't need this stuff */
  
    if ( IsAnimation(opt) ) bild->mark |= SEL;
#ifdef PHYSICAL_PAUL
    if ( (buf = GetCharSpec(bild->spec, ChunkPhysicalDataFile)) ) {
      LoadPhysicalParameters(buf);
    }
#endif
    if ( OnlyInfo(opt->f) ) continue;

    if ( bild->spp != 1 ) {
      if ( !NotMonoChrom(opt->f) && (bild->spp = IsMonochrome(bild)) == 1 ) {
        bild->spp = 1;
        if ( !DisplayGray(opt->f) ) Gray2Green(bild->DATA, bild->size);
      }
    }
    if ( Shrink(opt->f) ) ShrinkPicture(bild, opt->shr);
  }
  return loadlist;
}

GList *ReadPicFiles(GList *piclist, int nfiles, char **efiles, OPTIONS *opt)
/* Read a list of image files
 * --- Parameter: ---
 * int       nfiles      : number of filenames
 * char    **efiles      : filenames
 * OPTIONS  *opt         : f   : DelBorder and other for ReadPic()
 *                         cut : box to cut from image if given at command line
 *                         shr : level of shrinkage      
 * --- Return  : ---
 * GList     ReadPicFiles(): list of images, NULL in case of error
 */
{
  PICTURE  *bild;
  char    **fp, **fip;
  int       i = 0;
  GList    *pl;

  for ( fip = (fp = efiles) + nfiles; fp < fip; fp++ ) {
    if ( !(pl = ReadPic(*fp, opt)) ) continue;
    for ( bild = BILD(pl); pl; bild = BILD(pl = pl->next) )
      piclist = g_list_append(piclist, bild);
    if ( opt->cut && !SaveExtrema(opt->f ) ) {
      if ( DelBorder(opt->f) ) 
	g_warning(_("No cutting (-c) of images if option -r (delete border) was given!"));
      else
        if ( CutAreas(g_list_last(piclist), opt) ) {
          FreePiclist(piclist);
          return NULL;
	}
    }
    bild++;
    i++;
  }
  if ( Shrink(opt->f) ) {
    opt->shr = 0;
    opt->f  &= ~SHRINK;
  }
  if ( !SaveExtrema(opt->f) ) FREE(opt->cut);
  return piclist;
}


char *time_t2string(time_t zeit)
{
  char *p, *ptr = asctime(localtime(&zeit));

  if ( (p = strchr(ptr, '\n')) ) *p = 0;
  return ptr;
}

