#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: stokes.c,v 1.5 2000/02/01 17:02:51 knepley Exp $";
#endif

static char help[] = "This is the Stokes problem with P2/P1 elements.\n\n";

int STOKES_COOKIE;
int STOKES_ComputeField;

#include "stokes.h"

/*
  Here we are solving the Stokes equation:

    / -\nu \Delta u - \nabla p \ = / f(x)\
    \           \nabla \cdot u /   \  0  /

  Thus we now let

    VelocitySolutionFunction() --> u_S (which we prescribe)
    PressureSolutionFunction() --> p_S (which we prescribe)
    VelocityRhsFunction()      --> f
*/
#undef  __FUNCT__
#define __FUNCT__ "main"
int main(int argc, char **args) {
  StokesContext ctx; /* Holds problem specific information */
  int           ierr;

  PetscFunctionBegin;
  ierr = PetscInitialize(&argc, &args, 0, help);                                    CHKERRABORT(PETSC_COMM_WORLD, ierr);

  ierr = StokesInitializePackage(PETSC_NULL);                                       CHKERRABORT(PETSC_COMM_WORLD, ierr);
  ierr = StokesContextCreate(PETSC_COMM_WORLD, &ctx);                               CHKERRABORT(PETSC_COMM_WORLD, ierr);
  ierr = StokesComputeFlowField(ctx);                                               CHKERRABORT(PETSC_COMM_WORLD, ierr);
  ierr = StokesContextDestroy(ctx);                                                 CHKERRABORT(PETSC_COMM_WORLD, ierr);

  CHKMEMQ;
  PetscFinalize();
  PetscFunctionReturn(0);
}

#undef __FUNCT__  
#define __FUNCT__ "StokesInitializePackage"
/*@C
  StokesInitializePackage - This function initializes everything in the Stokes package.

  Input Parameter:
. path - The dynamic library path, or PETSC_NULL

  Level: developer

.keywords: Stokes, initialize, package
.seealso: PetscInitialize()
@*/
int StokesInitializePackage(char *path) {
  static PetscTruth initialized = PETSC_FALSE;
  char              logList[256];
  char             *className;
  PetscTruth        opt;
  int               ierr;

  PetscFunctionBegin;
  if (initialized == PETSC_TRUE) PetscFunctionReturn(0);
  initialized = PETSC_TRUE;
  /* Register Classes */
  ierr = PetscLogClassRegister(&STOKES_COOKIE, "Stokes");                                                 CHKERRQ(ierr);
  /* Register Constructors and Serializers */
  /* Register Events */
  ierr = PetscLogEventRegister(&STOKES_ComputeField, "StokesComputeField", STOKES_COOKIE);                CHKERRQ(ierr);
  /* Process info exclusions */
  ierr = PetscOptionsGetString(PETSC_NULL, "-log_info_exclude", logList, 256, &opt);                      CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    ierr = PetscStrstr(logList, "StokesContext", &className);                                             CHKERRQ(ierr);
    if (className) {
      ierr = PetscLogInfoDeactivateClass(STOKES_COOKIE);                                                  CHKERRQ(ierr);
    }
  }
  /* Process summary exclusions */
  ierr = PetscOptionsGetString(PETSC_NULL, "-log_summary_exclude", logList, 256, &opt);                   CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    ierr = PetscStrstr(logList, "StokesContext", &className);                                             CHKERRQ(ierr);
    if (className) {
      ierr = PetscLogEventDeactivateClass(STOKES_COOKIE);                                                 CHKERRQ(ierr);
    }
  }
  PetscFunctionReturn(0);
}

/*-------------------------------------------- StokesContext Creation ------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "StokesContextCreate"
/*@
  StokesContextCreate - This function initializes the Stokes context.

  Collective on MPI_Comm

  Input Parameter:
. comm - The communicator

  Output Parameter:
. sCtx - The StokesContext

  Level: beginner

.keywords: Stokes, context, create
.seealso: StokesContextDestroy(), StokesContextPrint(), StokesContextSetup()
@*/
int StokesContextCreate(MPI_Comm comm, StokesContext *sCtx) {
  StokesContext ctx;

  PetscFunctionBegin;
  /* Setup context */
  PetscHeaderCreate(ctx, _StokesContext, int, STOKES_COOKIE, -1, "context", comm, 0, 0);
  PetscLogObjectCreate(ctx);
  PetscLogObjectMemory(ctx, sizeof(struct _StokesContext));

  /* Initialize subobjects */
  ctx->grid       = PETSC_NULL;
  ctx->sles       = PETSC_NULL;
  ctx->A          = PETSC_NULL;
  ctx->u          = PETSC_NULL;
  ctx->p          = PETSC_NULL;
  ctx->f          = PETSC_NULL;
  ctx->origF      = PETSC_NULL;
  ctx->uExact     = PETSC_NULL;
  ctx->pExact     = PETSC_NULL;
  ctx->RofS       = -1;
  /* Setup domain */
  ctx->geometryCtx.size[0]  = 2.0;
  ctx->geometryCtx.size[1]  = 2.0;
  ctx->geometryCtx.start[0] = 0.0;
  ctx->geometryCtx.start[1] = 0.0;
  /* Setup refinement */
  ctx->geometryCtx.maxArea  = 0.5;
  ctx->geometryCtx.areaFunc = PointFunctionConstant;
  ctx->geometryCtx.areaCtx  = PETSC_NULL;
  /* Initialize physical information */
  ctx->physicalCtx.nu      = -1.0;
  ctx->physicalCtx.fluidSG = 1.0;
  /* Initialize preconditioning suites */
  ctx->useML      = PETSC_FALSE;
  ctx->useSchur   = PETSC_FALSE;
  /* Initialize problem loop */
  ctx->dim        = 2;
  ctx->linear     = PETSC_FALSE;
  ctx->refineStep = 0;
  ctx->numLoops   = 0;

  *sCtx = ctx;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "StokesContextDestroy"
/*@
  StokesContextDestroy - This function destroys the Stokes context.

  Collective on StokesContext

  Input Parameter:
. ctx - The StokesContext

  Level: beginner

.keywords: Stokes, context, destroy
.seealso: StokesContextCreate(), StokesContextSetup()
@*/
int StokesContextDestroy(StokesContext ctx) {
  Grid grid = ctx->grid;
  int  ierr;

  PetscFunctionBegin;
  if (--ctx->refct > 0) SETERRQ(PETSC_ERR_PLIB, "Stokes context should not be referenced more than once");
  ierr = GridFinalizeBC(grid);                                                                            CHKERRQ(ierr);
  ierr = GridDestroy(grid);                                                                               CHKERRQ(ierr);
  PetscLogObjectDestroy(ctx);
  PetscHeaderDestroy(ctx);
  PetscFunctionReturn(0);
}

/*-------------------------------------------- StokesContext Creation ------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "StokesContextSetup"
/*@
  StokesContextSetup - This function configures the context from user options.

  Collective on StokesContext

  Input Parameter:
. ctx - A StokesContext

  Options Database Keys:
+ dim <num>               - The problem dimension
. linear                  - Using a linear discretization
. num_systems <num>       - The number of systems to solve
. refine_step <num>       - The step to start refining the mesh
. physical_nu <num>       - The fluid kinematic viscosity
. physical_fluid_sg <num> - The fluid specific gravity
. mesh_max_area <num>     - The maximum area of a triangle in the refined mesh
. pc_ml                   - The ML preconditioning suite
- pc_schur                - The Schur preconditioning suite

  Level: beginner

.seealso StokesRefineGrid(), StokesDestroyGrid()
@*/
int StokesContextSetup(StokesContext ctx) {
  PetscTruth opt;
  int        ierr;

  PetscFunctionBegin;
  /* Determine the problem dimension */
  ierr = PetscOptionsGetInt(PETSC_NULL, "-dim", &ctx->dim, &opt);                                         CHKERRQ(ierr);
  /* Determine the element type */
  ierr = PetscOptionsHasName(PETSC_NULL, "-linear", &ctx->linear);                                        CHKERRQ(ierr);
  /* The first iteration at which to refine the mesh */
  ierr = PetscOptionsGetInt(PETSC_NULL, "-refine_step", &ctx->refineStep, &opt);                          CHKERRQ(ierr);
  /* Determine how many systems to solve */
  ierr = PetscOptionsGetInt(PETSC_NULL, "-num_systems", &ctx->numLoops, &opt);                            CHKERRQ(ierr);
  /* Setup refinement */
  ierr = PetscOptionsGetReal("mesh", "-max_area", &ctx->geometryCtx.maxArea, &opt);                       CHKERRQ(ierr);
  /* Get physical information */
  ierr = PetscOptionsGetReal("physical_", "-nu",       &ctx->physicalCtx.nu,      &opt);                  CHKERRQ(ierr);
  ierr = PetscOptionsGetReal("physical_", "-fluid_sg", &ctx->physicalCtx.fluidSG, &opt);                  CHKERRQ(ierr);
  /* Setup preconditioning suites */
  ierr = PetscOptionsHasName(PETSC_NULL, "-pc_ml",    &ctx->useML);                                       CHKERRQ(ierr);
  ierr = PetscOptionsHasName(PETSC_NULL, "-pc_schur", &ctx->useSchur);                                    CHKERRQ(ierr);

  /* Create main problem */
  ierr = StokesContextCreateGrid(ctx);                                                                    CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

/*------------------------------------------------ Grid Creation -----------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "StokesContextCreateMeshBoundary"
/*@
  StokesContextCreateMeshBoundary - This function creates a mesh boundary for the main problem.

  Collective on StokesContext

  Input Parameter:
. ctx - The StokesContext with problem specific information

  Level: beginner

.seealso StokesContextDestroyMeshBoundary(), StokesContextCreateMesh()
@*/
int StokesContextCreateMeshBoundary(StokesContext ctx) {
  MPI_Comm             comm;
  MeshGeometryContext *geomCtx = &ctx->geometryCtx;
  int                  ierr;

  PetscFunctionBegin;
  ierr = PetscObjectGetComm((PetscObject) ctx, &comm);                                                    CHKERRQ(ierr);
  switch(ctx->dim) {
  case 1:
    ierr = MeshBoundary1DCreateSimple(comm, geomCtx, &ctx->boundaryCtx);                                  CHKERRQ(ierr);
    break;
  case 2:
    ierr = MeshBoundary2DCreateSimple(comm, geomCtx, &ctx->boundaryCtx);                                  CHKERRQ(ierr);
    break;
  default:
    SETERRQ1(PETSC_ERR_SUP, "Do not support meshes of dimension %d", ctx->dim);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "StokesContextDestroyMeshBoundary"
/*@
  StokesContextDestroyMeshBoundary - This function destroys a mesh boundary for the main problem.

  Collective on StokesContext

  Input Parameter:
. ctx - The StokesContext with problem specific information

  Level: beginner

.seealso StokesContextCreateMeshBoundary(), StokesContextDestroyMesh()
@*/
int StokesContextDestroyMeshBoundary(StokesContext ctx) {
  MPI_Comm comm;
  int      ierr;

  PetscFunctionBegin;
  ierr = PetscObjectGetComm((PetscObject) ctx, &comm);                                                    CHKERRQ(ierr);
  switch(ctx->dim) {
  case 1:
    ierr = MeshBoundary1DDestroy(comm, &ctx->boundaryCtx);                                                CHKERRQ(ierr);
    break;
  case 2:
    ierr = MeshBoundary2DDestroy(comm, &ctx->boundaryCtx);                                                CHKERRQ(ierr);
    break;
  default:
    SETERRQ1(PETSC_ERR_SUP, "Do not support meshes of dimension %d", ctx->dim);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "StokesContextCreateMesh"
/*@
  StokesContextCreateMesh - This function creates a mesh for the main problem.

  Collective on StokesContext

  Input Parameter:
. ctx - A StokesContext with problem specific information

  Output Parameter:
. m   - The Mesh

  Options Database Keys:
. mesh_refine   - Refines the mesh based on area criteria
. mesh_max_area - The maximum area of an element

  Level: beginner

.seealso StokesRefineGrid(), StokesDestroyGrid()
@*/
int StokesContextCreateMesh(StokesContext ctx, Mesh *m) {
  MeshGeometryContext *geomCtx = &ctx->geometryCtx;
  MPI_Comm             comm;
  Mesh                 mesh;
  Partition            part;
  int                  totalElements, totalNodes, totalEdges;
  char                 name[1024];
  int                  d;
  int                  ierr;

  PetscFunctionBegin;
  ierr = PetscObjectGetComm((PetscObject) ctx, &comm);                                                    CHKERRQ(ierr);
  ierr = StokesContextCreateMeshBoundary(ctx);                                                            CHKERRQ(ierr);
  ierr = MeshCreate(comm, &mesh);                                                                         CHKERRQ(ierr);
  ierr = MeshSetDimension(mesh, ctx->dim);                                                                CHKERRQ(ierr);
  for(d = 0; d < ctx->dim; d++) {
    ierr = MeshSetPeriodicDimension(mesh, d, geomCtx->isPeriodic[d]);                                     CHKERRQ(ierr);
  }
  switch(ctx->dim) {
  case 1:
    if (ctx->linear == PETSC_TRUE) {
      ierr = MeshSetNumCorners(mesh, 2);                                                                  CHKERRQ(ierr);
    } else {
      ierr = MeshSetNumCorners(mesh, 3);                                                                  CHKERRQ(ierr);
    }
    break;
  case 2:
    if (ctx->linear == PETSC_TRUE) {
      ierr = MeshSetNumCorners(mesh, 3);                                                                  CHKERRQ(ierr);
    } else {
      ierr = MeshSetNumCorners(mesh, 6);                                                                  CHKERRQ(ierr);
    }
    break;
  default:
    SETERRQ1(PETSC_ERR_SUP, "Do not support meshes of dimension %d", ctx->dim);
  }
  ierr = MeshSetBoundary(mesh, &ctx->boundaryCtx);                                                        CHKERRQ(ierr);
  sprintf(name, "mesh.r%.4g", geomCtx->maxArea);
  ierr = PetscObjectSetName((PetscObject) mesh, name);                                                    CHKERRQ(ierr);
  ierr = MeshSetFromOptions(mesh);                                                                        CHKERRQ(ierr);
  ierr = StokesContextDestroyMeshBoundary(ctx);                                                           CHKERRQ(ierr);
  /* Setup refinement */
  if (ctx->geometryCtx.areaCtx != PETSC_NULL) {
    ierr = MeshSetUserContext(mesh, geomCtx->areaCtx);                                                    CHKERRQ(ierr);
  } else {
    ierr = MeshSetUserContext(mesh, &geomCtx->maxArea);                                                   CHKERRQ(ierr);
  }
  /* Report on mesh */
  ierr = MeshGetPartition(mesh, &part);                                                                   CHKERRQ(ierr);
  switch(ctx->dim) {
  case 1:
    ierr = PartitionGetTotalElements(part, &totalElements);                                               CHKERRQ(ierr);
    ierr = PartitionGetTotalNodes(part, &totalNodes);                                                     CHKERRQ(ierr);
    PetscPrintf(ctx->comm, "Elements: %d Nodes: %d\n", totalElements, totalNodes);
    break;
  case 2:
    ierr = PartitionGetTotalElements(part, &totalElements);                                               CHKERRQ(ierr);
    ierr = PartitionGetTotalNodes(part, &totalNodes);                                                     CHKERRQ(ierr);
    ierr = PartitionGetTotalEdges(part, &totalEdges);                                                     CHKERRQ(ierr);
    PetscPrintf(ctx->comm, "Elements: %d Nodes: %d Edges: %d\n", totalElements, totalNodes, totalEdges);
    break;
  default:
    SETERRQ1(PETSC_ERR_SUP, "Do not support meshes of dimension %d", ctx->dim);
  }

  *m = mesh;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "StokesContextCreateGrid"
/*@
  StokesContextCreateGrid - This function creates a grid for the main problem.

  Collective on StokesContext

  Input Parameter:
. ctx     - A StokesContext with problem specific information

  Level: beginner

.seealso StokesRefineGrid(), StokesDestroyGrid()
@*/
int StokesContextCreateGrid(StokesContext ctx) {
  Mesh mesh;
  Grid grid;
  int  ierr;

  PetscFunctionBegin;
  /* Construct the mesh */
  ierr = StokesContextCreateMesh(ctx, &mesh);                                                             CHKERRQ(ierr);
  /* Construct the grid */
  ierr = GridCreate(mesh, &grid);                                                                         CHKERRQ(ierr);
  ierr = MeshDestroy(mesh);                                                                               CHKERRQ(ierr);

  switch(ctx->dim) {
  case 1:
    if (ctx->linear == PETSC_TRUE) {
      ierr = GridAddField(grid, "Velocity", DISCRETIZATION_TRIANGULAR_1D_LINEAR,   2, PETSC_NULL);        CHKERRQ(ierr);
      ierr = GridAddField(grid, "Pressure", DISCRETIZATION_TRIANGULAR_1D_CONSTANT, 1, PETSC_NULL);        CHKERRQ(ierr);
    } else {
      ierr = GridAddField(grid, "Velocity", DISCRETIZATION_TRIANGULAR_1D_QUADRATIC, 2, PETSC_NULL);       CHKERRQ(ierr);
      ierr = GridAddField(grid, "Pressure", DISCRETIZATION_TRIANGULAR_1D_LINEAR,    1, PETSC_NULL);       CHKERRQ(ierr);
    }
    break;
  case 2:
    if (ctx->linear == PETSC_TRUE) {
      ierr = GridAddField(grid, "Velocity", DISCRETIZATION_TRIANGULAR_2D_LINEAR,   2, PETSC_NULL);        CHKERRQ(ierr);
      ierr = GridAddField(grid, "Pressure", DISCRETIZATION_TRIANGULAR_2D_CONSTANT, 1, PETSC_NULL);        CHKERRQ(ierr);
    } else {
      ierr = GridAddField(grid, "Velocity", DISCRETIZATION_TRIANGULAR_2D_QUADRATIC, 2, PETSC_NULL);       CHKERRQ(ierr);
      ierr = GridAddField(grid, "Pressure", DISCRETIZATION_TRIANGULAR_2D_LINEAR,    1, PETSC_NULL);       CHKERRQ(ierr);
    }
    break;
  default:
    SETERRQ1(PETSC_ERR_SUP, "Do not support meshes of dimension %d", ctx->dim);
  }
  ierr = GridSetFromOptions(grid);                                                                        CHKERRQ(ierr);

  ctx->grid = grid;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "StokesRefineGrid"
/*@
  StokesRefineGrid - This function refines the mesh for the main grid.

  Collective on StokesContext

  Input Parameters:
+ ctx - A StokesContext

  Options Database Keys:
. mesh_max_area - The maximum area an element may have

  Level: beginner

.seealso StokesCreateGrid(), StokesDestroyGrid()
@*/
int StokesRefineGrid(StokesContext ctx) {
  Grid                 oldGrid = ctx->grid;
  Grid                 grid;
  Mesh                 mesh;
  Partition            part;
  MeshGeometryContext *geomCtx = &ctx->geometryCtx;
  char                 name[1024];
  int                  totalElements, totalNodes, totalEdges;
  int                  ierr;

  PetscFunctionBegin;
  /* Construct a refined mesh */
  if (geomCtx->areaCtx != PETSC_NULL) {
    ierr = GridRefineMesh(oldGrid, geomCtx->areaFunc, &grid);                                             CHKERRQ(ierr);
    ierr = GridGetMesh(grid, &mesh);                                                                      CHKERRQ(ierr);
    ierr = MeshSetUserContext(mesh, geomCtx->areaCtx);                                                    CHKERRQ(ierr);
  } else {
    geomCtx->maxArea *= 0.5;
    ierr = GridRefineMesh(oldGrid, geomCtx->areaFunc, &grid);                                             CHKERRQ(ierr);
    ierr = GridGetMesh(grid, &mesh);                                                                      CHKERRQ(ierr);
    ierr = MeshSetUserContext(mesh, &geomCtx->maxArea);                                                   CHKERRQ(ierr);
  }
  sprintf(name, "mesh.r%.4g", geomCtx->maxArea);
  ierr = PetscObjectSetName((PetscObject) mesh, name);                                                    CHKERRQ(ierr);
  ierr = MeshSetOptionsPrefix(mesh, "ref_");                                                              CHKERRQ(ierr);
  ierr = GridSetOptionsPrefix(grid, "ref_");                                                              CHKERRQ(ierr);
  ierr = GridSetFromOptions(grid);                                                                        CHKERRQ(ierr);

  ierr = MeshGetPartition(mesh, &part);                                                                   CHKERRQ(ierr);
  ierr = PartitionGetTotalElements(part, &totalElements);                                                 CHKERRQ(ierr);
  ierr = PartitionGetTotalNodes(part, &totalNodes);                                                       CHKERRQ(ierr);
  ierr = PartitionGetTotalEdges(part, &totalEdges);                                                       CHKERRQ(ierr);
  PetscPrintf(ctx->comm, "Elements: %d Nodes: %d Edges: %d\n", totalElements, totalNodes, totalEdges);
  CHKMEMQ;

  /* Replace old grid with refined grid */
  ierr = GridFinalizeBC(oldGrid);                                                                         CHKERRQ(ierr);
  ierr = GridDestroy(oldGrid);                                                                            CHKERRQ(ierr);
  ctx->grid = grid;
  PetscFunctionReturn(0);
}

/*------------------------------------------------- Grid Setup -------------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "StokesSetupGrid"
/*@
  StokesSetupGrid - This function sets all the functions,
  operators , and boundary conditions for the problem.
  It also sets the parameters associated with the fields.

  Collective on Grid

  Input Parameters:
+ grid - The grid
- ctx  - A StokesContext

  Options Database Keys:
. use_laplacian - Use the Laplacian in stead of the Rate-of-Strain tensor

  Level: intermediate

.seealso StokesCreateGrid()
@*/
int StokesSetupGrid(StokesContext ctx) {
  Grid       grid   = ctx->grid;
  int        vel    = 0;
  int        pres   = 1;
  double     nu     = ctx->physicalCtx.nu;
  double     invRho = 1.0/ctx->physicalCtx.fluidSG;
  PetscTruth opt;
  int        ierr;

  PetscFunctionBegin;
  /* Setup Problem */
  ierr = GridSetActiveField(grid, vel);                                                                   CHKERRQ(ierr);
  if (ctx->useML == PETSC_FALSE) {
    ierr = GridAddActiveField(grid, pres);                                                                CHKERRQ(ierr);
  }

  /* Setup Rhs */
  ierr = GridRegisterOperator(grid, vel, RateOfStrainTensor, &ctx->RofS);                                 CHKERRQ(ierr);
  ierr = PetscOptionsHasName(PETSC_NULL, "-use_laplacian", &opt);                                         CHKERRQ(ierr);
  if (opt == PETSC_TRUE)
    ctx->RofS = LAPLACIAN;
  ierr = GridSetRhsOperator(grid,   vel,  vel,  ctx->RofS,  -nu, PETSC_FALSE, PETSC_NULL);                CHKERRQ(ierr);
  if (ctx->useML == PETSC_FALSE) {
    ierr = GridAddRhsOperator(grid, pres, vel,  GRADIENT,  -invRho, PETSC_FALSE, PETSC_NULL);             CHKERRQ(ierr);
    ierr = GridAddRhsOperator(grid, vel,  pres, DIVERGENCE, invRho, PETSC_FALSE, PETSC_NULL);             CHKERRQ(ierr);
  }
  ierr = StokesSetupRhsFunction(grid, ctx);                                                               CHKERRQ(ierr);

  /* Setup Jacobian */
  ierr = GridSetMatOperator(grid,   vel,  vel,  ctx->RofS,   -nu, PETSC_FALSE, PETSC_NULL);               CHKERRQ(ierr);
  if (ctx->useML == PETSC_FALSE) {
    ierr = GridAddMatOperator(grid, pres, vel,  GRADIENT,   -invRho, PETSC_FALSE, PETSC_NULL);            CHKERRQ(ierr);
    ierr = GridAddMatOperator(grid, vel,  pres, DIVERGENCE,  invRho, PETSC_FALSE, PETSC_NULL);            CHKERRQ(ierr);
  }

  /* Setup Dirchlet boundary conditions */
  ierr = StokesSetupBC(grid, ctx);                                                                        CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "StokesSetupRhsFunction"
/*@ StokesSetupRhsFunction
  StokesSetupRhsFunction - This function chooses a forcing  function for the problem.

  Collective on Grid

  Input Parameters:
+ grid - The grid
- ctx  - A StokesContext

  Level: intermediate

  Options Database Keys:

.seealso StokesSetupGrid
@*/
int StokesSetupRhsFunction(Grid grid, StokesContext ctx) {
  int vel  = 0;
  int pres = 1;
  int ierr;

  PetscFunctionBegin;
  ierr = GridAddRhsFunction(grid, vel, VelocityRhsFunction, 1.0);                                         CHKERRQ(ierr);
  if (ctx->useML == PETSC_FALSE) {
    ierr = GridAddRhsFunction(grid, pres, PointFunctionZero, 1.0);                                        CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "StokesSetupBC"
/*@ StokesSetupBC
  StokesSetupBC - This function chooses boundary conditions for the problem.

  Collective on Grid

  Input Parameters:
+ grid - The grid
- ctx  - A StokesContext

  Level: intermediate

  Options Database Keys:
+ bc_reduce         - Explicitly reduce the system using boundary conditions
. bc_exact          - Use the exact solution for boundary conditions
. bc_pressure_none  - Do not impose boundary conditions on the pressure
. bc_pressure_full  - Impose boundary conditions over the entire boundary
. bc_pressure_exact - Use the exact solution for boundary conditions
. bc_pressure_x     - The x-coordinate of the pressure normalization point
. bc_pressure_y     - The y-coordinate of the pressure normalization point
- bc_pressure_z     - The z-coordinate of the pressure normalization point

.seealso StokesSetupGrid()
@*/
int StokesSetupBC(Grid grid, StokesContext ctx) {
  MeshGeometryContext *geomCtx = &ctx->geometryCtx;
  int                  vel     = 0;
  int                  pres    = 1;
  PetscTruth           reduceSystem;
  double               pX, pY, pZ;
  PetscTruth           opt;
  int                  ierr;

  PetscFunctionBegin;
  /* Setup reduction usng boundary conditions */
  ierr = PetscOptionsHasName(PETSC_NULL, "-bc_reduce", &reduceSystem);                                    CHKERRQ(ierr);

  ierr = PetscOptionsHasName(PETSC_NULL, "-bc_exact", &opt);                                              CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    ierr = GridSetBC(grid, OUTER_BD, vel, VelocitySolutionFunction, reduceSystem);                        CHKERRQ(ierr);
  } else {
    ierr = GridSetBC(grid, OUTER_BD, vel, PointFunctionZero, reduceSystem);                               CHKERRQ(ierr);
  }

  /* Pressure boundary conditions */
  ierr = PetscOptionsHasName(PETSC_NULL, "-bc_pressure_none", &opt);                                      CHKERRQ(ierr);
  if (opt == PETSC_FALSE) {
    ierr = PetscOptionsHasName(PETSC_NULL, "-bc_pressure_full", &opt);                                    CHKERRQ(ierr);
    if (opt == PETSC_TRUE) {
      ierr = GridAddBC(grid, OUTER_BD, pres, PressureSolutionFunction, reduceSystem);                     CHKERRQ(ierr);
    } else {
      pX   = geomCtx->start[0];
      pY   = geomCtx->start[1];
      pZ   = geomCtx->start[2];
      ierr = PetscOptionsGetReal(PETSC_NULL, "-bc_pressure_x", &pX, &opt);                                CHKERRQ(ierr);
      ierr = PetscOptionsGetReal(PETSC_NULL, "-bc_pressure_y", &pY, &opt);                                CHKERRQ(ierr);
      ierr = PetscOptionsGetReal(PETSC_NULL, "-bc_pressure_z", &pZ, &opt);                                CHKERRQ(ierr);
      ierr = PetscOptionsHasName(PETSC_NULL, "-bc_pressure_exact", &opt);                                 CHKERRQ(ierr);
      if (opt == PETSC_TRUE) {
        ierr = GridSetPointBC(grid, pX, pY, pZ, pres, PressureSolutionFunction, reduceSystem);            CHKERRQ(ierr);
      } else {
        ierr = GridSetPointBC(grid, pX, pY, pZ, pres, PointFunctionZero, reduceSystem);                   CHKERRQ(ierr);
      }
    }
  }
  ierr = GridSetBCContext(grid, ctx);                                                                     CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

/*------------------------------------------------- SLES Setup -------------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "StokesCreateStructures"
int StokesCreateStructures(StokesContext ctx) {
  VarOrdering pOrder;
  int         pres = 1;
  int         ierr;

  PetscFunctionBegin;
  /* Create the linear solver */
  ierr = SLESCreate(ctx->comm, &ctx->sles);                                                               CHKERRQ(ierr);
  ierr = SLESSetFromOptions(ctx->sles);                                                                   CHKERRQ(ierr);
  /* Create solution, rhs, projected rhs, and exact solution vectors */
  ierr = GVecCreate(ctx->grid, &ctx->u);                                                                  CHKERRQ(ierr);
  ierr = PetscObjectSetName((PetscObject) ctx->u, "Solution");                                            CHKERRQ(ierr);
  ierr = GVecDuplicate(ctx->u, &ctx->f);                                                                  CHKERRQ(ierr);
  ierr = PetscObjectSetName((PetscObject) ctx->f, "Rhs");                                                 CHKERRQ(ierr);
  ierr = GVecDuplicate(ctx->u, &ctx->origF);                                                              CHKERRQ(ierr);
  ierr = PetscObjectSetName((PetscObject) ctx->f, "OriginalRhs");                                         CHKERRQ(ierr);
  ierr = GVecDuplicate(ctx->u, &ctx->uExact);                                                             CHKERRQ(ierr);
  ierr = PetscObjectSetName((PetscObject) ctx->uExact, "ExactSolution");                                  CHKERRQ(ierr);
  if (ctx->useML == PETSC_TRUE) {
    FieldClassMap cm, reduceCM;
    VarOrdering   reduceOrder;
    Vec           reduceVec;

    /* Create the multiplier, and the exact multiplier */
    ierr = VarOrderingCreateGeneral(ctx->grid, 1, &pres, &pOrder);                                        CHKERRQ(ierr);
    ierr = GVecCreateRectangular(ctx->grid, pOrder, &ctx->p);                                             CHKERRQ(ierr);
    ierr = PetscObjectSetName((PetscObject) ctx->p, "Pressure");                                          CHKERRQ(ierr);
    ierr = GVecDuplicate(ctx->p, &ctx->pExact);                                                           CHKERRQ(ierr);
    ierr = PetscObjectSetName((PetscObject) ctx->pExact, "ExactPressure");                                CHKERRQ(ierr);

    ierr = VarOrderingGetClassMap(pOrder, &cm);                                                           CHKERRQ(ierr);
    ierr = FieldClassMapReduce(cm, ctx->grid, &reduceCM);                                                 CHKERRQ(ierr);
    ierr = VarOrderingCreateReduceGeneral(ctx->grid, cm, reduceCM, &reduceOrder);                         CHKERRQ(ierr);
    ierr = GVecCreateRectangularGhost(ctx->grid, reduceOrder, &reduceVec);                                CHKERRQ(ierr);
    ierr = GridCalcBCValues_Private(ctx->grid, reduceOrder, reduceVec, PETSC_FALSE, PETSC_NULL);          CHKERRQ(ierr);
    ierr = PetscObjectCompose((PetscObject) ctx->p,      "reductionOrder", (PetscObject) reduceOrder);    CHKERRQ(ierr);
    ierr = PetscObjectCompose((PetscObject) ctx->pExact, "reductionOrder", (PetscObject) reduceOrder);    CHKERRQ(ierr);
    ierr = PetscObjectCompose((PetscObject) ctx->p,      "reductionVec",   (PetscObject) reduceVec);      CHKERRQ(ierr);
    ierr = PetscObjectCompose((PetscObject) ctx->pExact, "reductionVec",   (PetscObject) reduceVec);      CHKERRQ(ierr);
    ierr = FieldClassMapDestroy(reduceCM);                                                                CHKERRQ(ierr);
    ierr = VarOrderingDestroy(pOrder);                                                                    CHKERRQ(ierr);
    ierr = VarOrderingDestroy(reduceOrder);                                                               CHKERRQ(ierr);
    ierr = VecDestroy(reduceVec);                                                                         CHKERRQ(ierr);
  }
  /* Create the system matrix */
  ierr = GMatCreate(ctx->grid, &ctx->A);                                                                  CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "StokesSetupKSP"
int StokesSetupKSP(KSP ksp, StokesContext ctx) {
  GVecErrorKSPMonitorCtx *monCtx = &ctx->monitorCtx;
  PetscViewer             v;
  PetscDraw               draw;
  PetscTruth              opt;
  int                     ierr;

  PetscFunctionBegin;
  /* Setup the multilevel preconditioner */
  if (ctx->useML == PETSC_TRUE) {
    ierr = KSPSetPreconditionerSide(ksp, PC_SYMMETRIC);                                                   CHKERRQ(ierr);
  }
  /* Setup convergence monitors */
  ierr = PetscOptionsHasName(PETSC_NULL, "-error_viewer", &opt);                                          CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    v    = PETSC_VIEWER_DRAW_(ctx->comm);
    ierr = PetscViewerSetFormat(v, PETSC_VIEWER_DRAW_LG);                                                 CHKERRQ(ierr);
    ierr = PetscViewerDrawGetDraw(v, 0, &draw);                                                           CHKERRQ(ierr);
    ierr = PetscDrawSetTitle(draw, "Error");                                                              CHKERRQ(ierr);
  } else {
    v    = PETSC_VIEWER_STDOUT_(ctx->comm);
  }
  monCtx->error_viewer      = v;
  monCtx->solution          = ctx->uExact;
  monCtx->norm_error_viewer = PETSC_VIEWER_STDOUT_(ctx->comm);
  ierr = KSPSetMonitor(ksp, GVecErrorKSPMonitor, monCtx, PETSC_NULL);                                     CHKERRQ(ierr);
  PetscFunctionReturn(0);
}


#undef  __FUNCT__
#define __FUNCT__ "StokesSetupPC"
int StokesSetupPC(PC pc, StokesContext ctx) {
  int        vel  = 0;
  int        pres = 1;
  PetscTruth opt;
  int        ierr;

  PetscFunctionBegin;
  /* Setup the multilevel preconditioner */
  if (ctx->useML == PETSC_TRUE) {
#ifndef PETSC_USE_DYANMIC_LIBRARIES
    ierr = GSolverInitializePackage(PETSC_NULL);                                                          CHKERRQ(ierr);
#endif
    ierr = PetscOptionsHasName(PETSC_NULL, "-bc_reduce", &opt);                                           CHKERRQ(ierr);
    if (opt == PETSC_FALSE) SETERRQ(PETSC_ERR_ARG_WRONGSTATE, "The -bc_reduce option is necessary for ML");

    ierr = PCSetType(pc, PCMULTILEVEL);                                                                   CHKERRQ(ierr);
    ierr = PCMultiLevelSetFields(pc, pres, vel);                                                          CHKERRQ(ierr);
    ierr = PCMultiLevelSetGradientOperator(pc, GRADIENT, DIVERGENCE, -1.0);                               CHKERRQ(ierr);
    ierr = PCSetVector(pc, ctx->f);                                                                       CHKERRQ(ierr);
  }
  if (ctx->useSchur == PETSC_TRUE) {
    ierr = PCSetType(pc, PCSCHUR);                                                                        CHKERRQ(ierr);
    ierr = PCSchurSetGradientOperator(pc, GRADIENT, DIVERGENCE);                                          CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "StokesSetupStructures"
int StokesSetupStructures(StokesContext ctx) {
  KSP          ksp;
  PC           pc;
  VarOrdering  pOrder;
  int          vel  = 0;
  int          pres = 1;
  MatStructure flag;
  PetscTruth   opt;
  int          ierr;

  PetscFunctionBegin;
  /* Setup the linear solver */
  ierr = SLESSetOperators(ctx->sles, ctx->A, ctx->A, SAME_NONZERO_PATTERN);                               CHKERRQ(ierr);
  ierr = SLESGetKSP(ctx->sles, &ksp);                                                                     CHKERRQ(ierr);
  ierr = StokesSetupKSP(ksp, ctx);                                                                        CHKERRQ(ierr);
  ierr = SLESGetPC(ctx->sles, &pc);                                                                       CHKERRQ(ierr);
  ierr = StokesSetupPC(pc, ctx);                                                                          CHKERRQ(ierr);
  /* Evaluate the rhs */
  ierr = GridEvaluateRhs(ctx->grid, PETSC_NULL, ctx->f, (PetscObject) ctx);                               CHKERRQ(ierr);
  ierr = VecCopy(ctx->f, ctx->origF);                                                                     CHKERRQ(ierr);
  /* Evaluate the exact solution */
  ierr = GVecEvaluateFunction(ctx->uExact, 1, &vel, VelocitySolutionFunction, 1.0, ctx);                  CHKERRQ(ierr);
  if (ctx->useML == PETSC_FALSE) {
    ierr = GVecEvaluateFunction(ctx->uExact, 1, &pres, PressureSolutionFunction, 1.0, ctx);               CHKERRQ(ierr);
  }
  /* Evaluate the exact multiplier */
  if (ctx->useML == PETSC_TRUE) {
    ierr = VarOrderingCreateGeneral(ctx->grid, 1, &pres, &pOrder);                                        CHKERRQ(ierr);
    ierr = GVecEvaluateFunctionRectangular(ctx->pExact, pOrder, PressureSolutionFunction, 1.0, ctx);      CHKERRQ(ierr);
    ierr = VarOrderingDestroy(pOrder);                                                                    CHKERRQ(ierr);
  }
  /* Evaluate the system matrix */
  flag = DIFFERENT_NONZERO_PATTERN;
  ierr = GridEvaluateSystemMatrix(ctx->grid, PETSC_NULL, &ctx->A, &ctx->A, &flag, (PetscObject) ctx);     CHKERRQ(ierr);
  ierr = MatCheckSymmetry(ctx->A);                                                                        CHKERRQ(ierr);
  /* Apply Dirchlet boundary conditions */
  ierr = GMatSetBoundary(ctx->A, 1.0, ctx);                                                               CHKERRQ(ierr);
  ierr = GVecSetBoundary(ctx->f, ctx);                                                                    CHKERRQ(ierr);
  /* View structures */
  ierr = PetscOptionsHasName(PETSC_NULL, "-mat_view", &opt);                                              CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    ierr = MatView(ctx->A, PETSC_VIEWER_STDOUT_(ctx->comm));                                              CHKERRQ(ierr);
    ierr = MatView(ctx->A, PETSC_VIEWER_DRAW_(ctx->comm));                                                CHKERRQ(ierr);
  }
  ierr = PetscOptionsHasName(PETSC_NULL, "-vec_view", &opt);                                              CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    ierr = VecView(ctx->f, PETSC_VIEWER_STDOUT_(ctx->comm));                                              CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "StokesDestroyStructures"
int StokesDestroyStructures(StokesContext ctx) {
  int ierr;

  PetscFunctionBegin;
  ierr = SLESDestroy(ctx->sles);                                                                          CHKERRQ(ierr);
  ierr = GVecDestroy(ctx->f);                                                                             CHKERRQ(ierr);
  ierr = GVecDestroy(ctx->origF);                                                                         CHKERRQ(ierr);
  ierr = GVecDestroy(ctx->u);                                                                             CHKERRQ(ierr);
  ierr = GVecDestroy(ctx->uExact);                                                                        CHKERRQ(ierr);
  if (ctx->useML == PETSC_TRUE) {
    ierr = GVecDestroy(ctx->p);                                                                           CHKERRQ(ierr);
    ierr = GVecDestroy(ctx->pExact);                                                                      CHKERRQ(ierr);
  }
  ierr = GMatDestroy(ctx->A);                                                                             CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

/*----------------------------------------------- Sanity Checks ------------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "MatCheckSymmetry"
int MatCheckSymmetry(Mat A) {
  Mat        trA;
  PetscTruth isSym;
  int        ierr;

  PetscFunctionBegin;
  ierr = MatTranspose(A, &trA);                                                                           CHKERRQ(ierr);
  ierr = MatEqual(A, trA, &isSym);                                                                        CHKERRQ(ierr);
  ierr = MatDestroy(trA);                                                                                 CHKERRQ(ierr);
  if (isSym == PETSC_FALSE) PetscFunctionReturn(1);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "StokesCheckSolution"
int StokesCheckSolution(StokesContext ctx, GVec u, GVec p, const char type[]) {
  GVec        r, projR;
  PC          pc;
  PetscScalar minusOne = -1.0;
  PetscScalar norm;
  int         ierr;

  PetscFunctionBegin;
  ierr = GVecDuplicate(u, &r);                                                                            CHKERRQ(ierr);
  ierr = GVecDuplicate(u, &projR);                                                                        CHKERRQ(ierr);
  ierr = SLESGetPC(ctx->sles, &pc);                                                                       CHKERRQ(ierr);

  /* Check the velocity solution */
  /* A u^* */
  ierr = MatMult(ctx->A, u, r);                                                                           CHKERRQ(ierr);
  /* f - A u^* */
  ierr = VecAYPX(&minusOne, ctx->origF, r);                                                               CHKERRQ(ierr);
  /* P^T_2 (f - A u^*) */
  if (ctx->useML == PETSC_TRUE) {
    ierr = PCApplySymmetricLeft(pc, r, projR);                                                            CHKERRQ(ierr);
    ierr = VecNorm(projR, NORM_2, &norm);                                                                 CHKERRQ(ierr);
    PetscPrintf(ctx->comm, "Residual of the %s velocity solution: %lf\n", type, norm);
  } else {
    ierr = VecNorm(projR, NORM_2, &norm);                                                                 CHKERRQ(ierr);
    PetscPrintf(ctx->comm, "Residual of the %s solution: %lf\n", type, norm);
  }

  /* Check the multiplier */
  if (ctx->useML == PETSC_TRUE) {
    /* P^T_2 B p^* */
    ierr = PCMultiLevelApplyGradient(pc, p, r);                                                           CHKERRQ(ierr);
    ierr = PCApplySymmetricLeft(pc, r, projR);                                                            CHKERRQ(ierr);
    ierr = VecNorm(projR, NORM_2, &norm);                                                                 CHKERRQ(ierr);
    PetscPrintf(ctx->comm, "Residual of the %s gradient contribution: %lf\n", type, norm);
    /* f - A u^* - B p^* */
    ierr = MatMult(ctx->A, u, projR);                                                                     CHKERRQ(ierr);
    ierr = VecAYPX(&minusOne, ctx->origF, projR);                                                         CHKERRQ(ierr);
    ierr = VecAXPY(&minusOne, r, projR);                                                                  CHKERRQ(ierr);
    ierr = VecNorm(projR, NORM_2, &norm);                                                                 CHKERRQ(ierr);
    PetscPrintf(ctx->comm, "Residual of the full %s solution: %lf\n", type, norm);
  }
 
  ierr = GVecDestroy(r);                                                                                  CHKERRQ(ierr);
  ierr = GVecDestroy(projR);                                                                              CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

/*----------------------------------------------- Problem Callbacks --------------------------------------------------*/

#undef  __FUNCT__
#define __FUNCT__ "VelocitySolutionFunction"
/*@
  VelocitySolutionFunction - This function is the velocity solution function for the problem.

  Not collective

  Input Parameters:
+ n      - The number of points
. comp   - The number of components
. x,y,z  - The points
. values - The output
- ctx    - A StokesContext

  Level: beginner

  Note:
  The solution is u = x^2 \hat x - 2 x y \hat y

.keywords velocity, solution
.seealso PressureSolutionFunction(), VelocityRhsFunction()
@*/
int VelocitySolutionFunction(int n, int comp, double *x, double *y, double *z, PetscScalar *values, void *ctx) {
  int i;

  PetscFunctionBegin;
  if (comp != 2) SETERRQ1(PETSC_ERR_ARG_WRONG, "Invalid number of components %d", comp);
  for(i = 0; i < n; i++) {
    values[i*2+0] = x[i]*x[i];
    values[i*2+1] = -2.0*x[i]*y[i];
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PressureSolutionFunction"
/*@
  PressureSolutionFunction - This function is the pressure solution function for the problem.

  Not collective

  Input Parameters:
+ n      - The number of points
. comp   - The number of components
. x,y,z  - The points
. values - The output
- ctx    - A StokesContext

  Level: beginner

  Note:
  The solution is p = x + y - 2, the integral of p is zero over the domain.

.keywords pressure, solution
.seealso VelocitySolutionFunction(), VelocityRhsFunction()
@*/
int PressureSolutionFunction(int n, int comp, double *x, double *y, double *z, PetscScalar *values, void *ctx) {
  int i;

  PetscFunctionBegin;
  if (comp != 1) SETERRQ1(PETSC_ERR_ARG_WRONG, "Invalid number of components %d", comp);
  for(i = 0; i < n; i++) {
      values[i] = x[i] + y[i] - 2.0;
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "VelocityRhsFunction"
/*@
  VelocityRhsFunction - This function is the forcing function for the problem.

  Not collective

  Input Parameters:
+ n      - The number of points
. comp   - The number of components
. x,y,z  - The points
. values - The output
- ctx    - A StokesContext

  Level: beginner

  Note:
  The rhs is -\nu \Delta u - \rho^{-1} \nabla p = -(2\nu + 1) \hat x - \hat y

.keywords velocity, rhs
.seealso VelocitySolutionFunction(), PressureSolutionFunction()
@*/
int VelocityRhsFunction(int n, int comp, double *x, double *y, double *z, PetscScalar *values, void *ctx) {
  StokesContext s      = (StokesContext) ctx;
  double        nu     = s->physicalCtx.nu;
  double        invRho = 1.0/s->physicalCtx.fluidSG;
  int           i;

  PetscFunctionBegin;
  if (comp != 2) SETERRQ1(PETSC_ERR_ARG_WRONG, "Invalid number of components %d", comp);
  for (i = 0; i < n; i++) {
    values[i*2+0] = -2.0*nu - invRho;
    values[i*2+1] = -invRho;
  }
  PetscFunctionReturn(0);
}

/*-------------------------------------------------- Operators -------------------------------------------------------*/
/* This defines the linear operator

     {1 \over 2} \nabla \cdot ( \nabla u + {\nabla u}^T )
*/
#undef  __FUNCT__
#define __FUNCT__ "RateOfStrainTensor"
int RateOfStrainTensor(Discretization disc, Discretization test, int rowSize, int colSize,
                       int globalRowStart, int globalColStart, int globalSize, double *coords,
                       PetscScalar alpha, PetscScalar *field, PetscScalar *array, void *ctx)
{
  int     comp;              /* The number of field components */
  int     funcs;             /* The number of shape functions */
  int     numQuadPoints;     /* Number of points used for Gaussian quadrature */
  double *quadWeights;       /* Weights in the standard element for Gaussian quadrature */
  double *quadShapeFuncDers; /* Shape function derivatives evaluated at quadrature points */
  double *quadTestFuncDers;  /* Test  function derivatives evaluated at quadrature points */
  double  dxxi;              /* \PartDer{x}{\xi}  */
  double  dxet;              /* \PartDer{x}{\eta} */
  double  dyxi;              /* \PartDer{y}{\xi}  */
  double  dyet;              /* \PartDer{y}{\eta} */
  double  dxix;              /* \PartDer{\xi}{x}  */
  double  detx;              /* \PartDer{\eta}{x} */
  double  dxiy;              /* \PartDer{\xi}{y}  */
  double  dety;              /* \PartDer{\eta}{y} */
  double  dphix;             /* \PartDer{\phi_j}{x} Shape */
  double  dphiy;             /* \PartDer{\phi_j}{y} Shape */
  double  dpsix;             /* \PartDer{\psi_i}{x} Test  */
  double  dpsiy;             /* \PartDer{\psi_i}{y} Test  */
  double  jac;               /* |J| for map to standard element */
  double  invjac;            /* |J^{-1}| for map from standard element */
  int     i, j, f, p;
  int     ierr;

  PetscFunctionBegin;
  ierr = DiscretizationGetNumComponents(disc, &comp);                                                     CHKERRQ(ierr);
  ierr = DiscretizationGetNumFunctions(disc, &funcs);                                                     CHKERRQ(ierr);
  if (comp != 2) SETERRQ(PETSC_ERR_ARG_WRONG, "Operator only valid with 2D vector field");

  ierr = DiscretizationGetNumQuadraturePoints(disc, &numQuadPoints);                                      CHKERRQ(ierr);
  ierr = DiscretizationGetQuadratureWeights(disc, &quadWeights);                                          CHKERRQ(ierr);
  ierr = DiscretizationGetQuadratureDerivatives(disc, &quadShapeFuncDers);                                CHKERRQ(ierr);
  ierr = DiscretizationGetQuadratureDerivatives(test, &quadTestFuncDers);                                 CHKERRQ(ierr);

  /* Bring out a factor of 1/2 */
  alpha *= 0.5;

  /* Calculate element matrix entries by Gaussian quadrature */
  for(p = 0; p < numQuadPoints; p++) {
    /* \PartDer{x}{\xi}(p)  = \sum^{funcs}_{f=1} x_f \PartDer{\phi^f(p)}{\xi}
       \PartDer{x}{\eta}(p) = \sum^{funcs}_{f=1} x_f \PartDer{\phi^f(p)}{\eta}
       \PartDer{y}{\xi}(p)  = \sum^{funcs}_{f=1} y_f \PartDer{\phi^f(p)}{\xi}
       \PartDer{y}{\eta}(p) = \sum^{funcs}_{f=1} y_f \PartDer{\phi^f(p)}{\eta} */
    dxxi = 0.0; dxet = 0.0;
    dyxi = 0.0; dyet = 0.0;
    for(f = 0; f < funcs; f++) {
      dxxi += coords[f*2]  *quadShapeFuncDers[p*funcs*2+f*2];
      dxet += coords[f*2]  *quadShapeFuncDers[p*funcs*2+f*2+1];
      dyxi += coords[f*2+1]*quadShapeFuncDers[p*funcs*2+f*2];
      dyet += coords[f*2+1]*quadShapeFuncDers[p*funcs*2+f*2+1];
    }
    jac  = fabs(dxxi*dyet - dxet*dyxi);
#ifdef PETSC_USE_BOPT_g
    if (jac < 1.0e-14) {
      PetscPrintf(PETSC_COMM_SELF, "p: %d x1: %lf y1: %lf x2: %lf y2: %lf x3: %lf y3: %lf\n",
                  p, coords[0], coords[1], coords[2], coords[3], coords[4], coords[5]);
      SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
    }
#endif
    /* These are the elements of the inverse matrix */
    invjac =  1.0/jac;
    dxix   =  dyet*invjac;
    dxiy   = -dxet*invjac;
    detx   = -dyxi*invjac;
    dety   =  dxxi*invjac;

    for(i = 0; i < funcs; i++) {
      for(j = 0; j < funcs; j++) {
        dpsix = quadTestFuncDers[p*funcs*2+i*2] *dxix + quadTestFuncDers[p*funcs*2+i*2+1] *detx;
        dphix = quadShapeFuncDers[p*funcs*2+j*2]*dxix + quadShapeFuncDers[p*funcs*2+j*2+1]*detx;
        dpsiy = quadTestFuncDers[p*funcs*2+i*2] *dxiy + quadTestFuncDers[p*funcs*2+i*2+1] *dety;
        dphiy = quadShapeFuncDers[p*funcs*2+j*2]*dxiy + quadShapeFuncDers[p*funcs*2+j*2+1]*dety;
        array[(i*comp+0+globalRowStart)*globalSize + j*comp+0+globalColStart] +=
          -alpha*(2.0*dpsix*dphix +     dpsiy*dphiy)*jac*quadWeights[p];
        array[(i*comp+0+globalRowStart)*globalSize + j*comp+1+globalColStart] +=
          -alpha*(    dpsiy*dphix                  )*jac*quadWeights[p];
        array[(i*comp+1+globalRowStart)*globalSize + j*comp+0+globalColStart] +=
          -alpha*(                      dpsix*dphiy)*jac*quadWeights[p];
        array[(i*comp+1+globalRowStart)*globalSize + j*comp+1+globalColStart] +=
          -alpha*(    dpsix*dphix + 2.0*dpsiy*dphiy)*jac*quadWeights[p];
      }
    }
  }
  PetscLogFlops((8*funcs + 8 + 38*funcs*funcs) * numQuadPoints);

  PetscFunctionReturn(0);
}

/*----------------------------------------------- Main Computation ---------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "StokesSolve"
int StokesSolve(StokesContext ctx, GVec f, GVec u, int *its) {
  PC  pc;
  int ierr;

  PetscFunctionBegin;
  /* Solve P^T_2 A P_2 (P^{-1} u)_2 = P^T_2 (f - A P_1 D^{-1} Z^T g) */
  ierr = SLESSolve(ctx->sles, f, u, its);                                                                 CHKERRQ(ierr);

  if (ctx->useML == PETSC_TRUE) {
    /* Calculate p = Z D^{-1} P^T_1 (f - A P_2 (P^{-1} u)_2 - A P_1 D^{-1} Z^T g) */
    ierr = SLESGetPC(ctx->sles, &pc);                                                                     CHKERRQ(ierr);
    ierr = PCMultiLevelGetMultiplier(pc, u, ctx->p);                                                      CHKERRQ(ierr);

    /* Recover u */
    ierr = PCMultiLevelBuildSolution(pc, u);                                                              CHKERRQ(ierr);

    /* Show pressure */
    ierr = GVecViewFromOptions(ctx->pExact, "Exact Pressure");                                            CHKERRQ(ierr);
    ierr = GVecViewFromOptions(ctx->p,      "Pressure");                                                  CHKERRQ(ierr);
  }

  /* Show solution */
  ierr = GVecViewFromOptions(ctx->uExact, "Exact Solution");                                              CHKERRQ(ierr);
  ierr = GVecViewFromOptions(ctx->u,      "Solution");                                                    CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "StokesComputeFlowField"
int StokesComputeFlowField(StokesContext ctx) {
  int its;  /* The iteration count for the linear solver */
  int loop;
  int ierr;

  /* Get command-line options */
  ierr = StokesContextSetup(ctx);                                                                         CHKERRQ(ierr);

  for(loop = 0; loop < ctx->numLoops; loop++) {
    if (loop >= ctx->refineStep) {
      ierr = StokesRefineGrid(ctx);                                                                       CHKERRQ(ierr);
    }

    /* Setup problem */
    ierr = StokesSetupGrid(ctx);                                                                          CHKERRQ(ierr);
    ierr = StokesCreateStructures(ctx);                                                                   CHKERRQ(ierr);
    ierr = StokesSetupStructures(ctx);                                                                    CHKERRQ(ierr);

    /* Check the exact solution */
    ierr = StokesCheckSolution(ctx, ctx->uExact, ctx->pExact, "exact");                                   CHKERRQ(ierr);

    /* Solve system */
    ierr = StokesSolve(ctx, ctx->f, ctx->u, &its);                                                        CHKERRQ(ierr);

    /* Check the computed solution */
    ierr = StokesCheckSolution(ctx, ctx->u, ctx->p, "computed");                                          CHKERRQ(ierr);

    /* Cleanup */
    ierr = StokesDestroyStructures(ctx);                                                                  CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}
