/* $Id: bilinear.h,v 1.3 2000/01/10 03:56:06 knepley Exp $ */
/*
     Include file for the bilinear component of PETSc
*/
#ifndef __BILINEAR_H
#define __BILINEAR_H

#include "petscmat.h"

typedef struct _p_Bilinear *Bilinear;

#define BILINEAR_DENSE_SEQ "dense_seq"
typedef char *BilinearType;

#define BILINEAR_SER_SEQDENSE_BINARY "seqdense_binary"
typedef char *BilinearSerializeType;

/* Logging support */
extern int BILINEAR_COOKIE;
extern int BILINEAR_Copy, BILINEAR_Convert, BILINEAR_SetValues, BILINEAR_AssemblyBegin, BILINEAR_AssemblyEnd;
extern int BILINEAR_ZeroEntries, BILINEAR_Mult, BILINEAR_FullMult, BILINEAR_Diamond, BILINEAR_LUFactor, BILINEAR_CholeskyFactor;

/* Option flags */
typedef enum {BILINEAR_SYMMETRIC, BILINEAR_NO_NEW_NONZERO_LOCATIONS, BILINEAR_YES_NEW_NONZERO_LOCATIONS,
              BILINEAR_NEW_NONZERO_LOCATION_ERROR, BILINEAR_NEW_NONZERO_ALLOCATION_ERROR,
              BILINEAR_IGNORE_OFF_PROC_ENTRIES} BilinearOption;

/* Query flags and structures */
typedef enum {INFO_LOCAL = 1, INFO_GLOBAL_MAX, INFO_GLOBAL_SUM} InfoType;
typedef struct {
  PetscLogDouble rows_global, cols_global, subcols_global; /* number of global rows and columns */
  PetscLogDouble rows_local, cols_local, subcols_local;    /* number of local rows and columns */
  PetscLogDouble nz_allocated, nz_used, nz_unneeded;       /* number of nonzeros */
  PetscLogDouble memory;                                   /* memory allocated */
  PetscLogDouble assemblies;                               /* number of matrix assemblies */
  PetscLogDouble mallocs;                                  /* number of mallocs during BilinearSetValues() */
} BilinearInfo;

typedef enum {BILINEAROP_SET_VALUES,
              BILINEAROP_MULT,
              BILINEAROP_LUFACTOR,
              BILINEAROP_CHOLESKYFACTOR,
              BILINEAROP_GETINFO,
              BILINEAROP_EQUAL,
              BILINEAROP_NORM,
              BILINEAROP_ASSEMBLY_BEGIN,
              BILINEAROP_ASSEMBLY_END,
              BILINEAROP_SET_OPTION,
              BILINEAROP_ZERO_ENTRIES,
              BILINEAROP_GET_SIZE,
              BILINEAROP_GET_LOCAL_SIZE,
              BILINEAROP_GET_OWNERSHIP_RANGE,
              BILINEAROP_GET_ARRAY,
              BILINEAROP_RESTORE_ARRAY,
              BILINEAROP_COPY,
              BILINEAROP_PRINT_HELP,
              BILINEAROP_DESTROY,
              BILINEAROP_VIEW
             } BilinearOperation;

EXTERN int BilinearInitializePackage(char *);

/* Generic Operations */
EXTERN int BilinearCreate(MPI_Comm, int, int, int, Bilinear *);
EXTERN int BilinearSetFromOptions(Bilinear);
EXTERN int BilinearView(Bilinear, PetscViewer);
EXTERN int BilinearCopy(Bilinear, Bilinear);
EXTERN int BilinearDuplicate(Bilinear, Bilinear *);
EXTERN int BilinearSerialize(MPI_Comm, Bilinear *, PetscViewer, PetscTruth);
EXTERN int BilinearDestroy(Bilinear);
EXTERN int BilinearViewFromOptions(Bilinear, char *);
EXTERN int BilinearSetOptionsPrefix(Bilinear, char *);
EXTERN int BilinearAppendOptionsPrefix(Bilinear, char *);
EXTERN int BilinearGetOptionsPrefix(Bilinear, char **);
EXTERN int BilinearPrintHelp(Bilinear);

/* Assembly Operations*/
#define BilinearSetValue(B,i,j,k,v,mode) \
{int _ierr,_row = i,_col = j,_subcol = k; PetscScalar _v = v; \
  _ierr = BilinearSetValues(B,1,&_row,1,&_col,1,&_subcol,&_v,mode);CHKERRQ(_ierr); \
}
EXTERN int BilinearSetValues(Bilinear, int, int *, int, int *, int, int *, PetscScalar *, InsertMode);
EXTERN int BilinearGetArray(Bilinear, PetscScalar **);
EXTERN int BilinearRestoreArray(Bilinear, PetscScalar **);
EXTERN int BilinearAssemblyBegin(Bilinear, MatAssemblyType);
EXTERN int BilinearAssemblyEnd(Bilinear, MatAssemblyType);
EXTERN int BilinearZeroEntries(Bilinear);

/* Evaluation Operations */
EXTERN int BilinearMult(Bilinear, Vec, Mat);
EXTERN int BilinearFullMult(Bilinear, Vec, Vec, Vec);
EXTERN int BilinearDiamond(Bilinear, Vec, Vec);

/* Factorization Operations */
EXTERN int BilinearLUFactor(Bilinear, IS, IS, IS, double);
EXTERN int BilinearCholeskyFactor(Bilinear, IS, double);

/* Bilinear Query Interface */
EXTERN int BilinearGetInfo(Bilinear, InfoType, BilinearInfo *);
EXTERN int BilinearGetSize(Bilinear, int *, int *, int *);
EXTERN int BilinearGetLocalSize(Bilinear, int *, int *, int *);
EXTERN int BilinearGetOwnershipRange(Bilinear, int *, int *);
EXTERN int BilinearEqual(Bilinear, Bilinear, PetscTruth*);
EXTERN int BilinearNorm(Bilinear, NormType, double *);
EXTERN int BilinearHasOperation(Bilinear, BilinearOperation, PetscTruth *);
EXTERN int BilinearSetOption(Bilinear, BilinearOption);
EXTERN int BilinearValid(Bilinear, PetscTruth *);

/* Dynamic creation and loading functions */
extern PetscFList BilinearList;
extern PetscTruth BilinearRegisterAllCalled;
EXTERN int BilinearSetType(Bilinear, BilinearType);
EXTERN int BilinearGetType(Bilinear, BilinearType *);
EXTERN int BilinearRegister(const char[],const char[],const char[],int(*)(Bilinear));
EXTERN int BilinearRegisterAll(const char []);
EXTERN int BilinearRegisterDestroy(void);
#if defined(PETSC_USE_DYNAMIC_LIBRARIES)
#define BilinearRegisterDynamic(a,b,c,d) BilinearRegister(a,b,c,0)
#else
#define BilinearRegisterDynamic(a,b,c,d) BilinearRegister(a,b,c,d)
#endif

extern PetscFList BilinearSerializeList;
extern PetscTruth BilinearSerializeRegisterAllCalled;
EXTERN int BilinearSetSerializeType(Bilinear, BilinearSerializeType);
EXTERN int BilinearGetSerializeType(Bilinear, BilinearSerializeType *);
EXTERN int BilinearSerializeRegister(const char [], const char [], const char [], int (*)(MPI_Comm, Bilinear *, PetscViewer, PetscTruth));
EXTERN int BilinearSerializeRegisterAll(const char []);
EXTERN int BilinearSerializeRegisterDestroy(void);
#if defined(PETSC_USE_DYNAMIC_LIBRARIES)
#define BilinearSerializeRegisterDynamic(a,b,c,d) BilinearSerializeRegister(a,b,c,0)
#else
#define BilinearSerializeRegisterDynamic(a,b,c,d) BilinearSerializeRegister(a,b,c,d)
#endif

#endif /* __BILINEAR_H */
