/* $Id: gsolver.h,v 1.6 2000/01/10 03:56:07 knepley Exp $ */

/*
  Solvers for grid-based problems
*/

#ifndef __GSOLVER_H
#define __GSOLVER_H

#include "grid.h"
#include "gvec.h"

#if defined(__PETSCSLES_H)
#include "petscsles.h"

EXTERN int GSolverInitializePackage(char *);

/* ----------------------------------------------------------------------------*/
/*
  Alternative monitors are provided for KSP solvers used with GVec and GMat objects.
*/
typedef struct {
  PetscViewer error_viewer;
  PetscViewer norm_error_viewer;
  GVec        solution;
  PetscTruth  isInner;
} GVecErrorKSPMonitorCtx;
 
extern int GVecSolutionKSPMonitor(KSP, int, PetscReal, void *);
extern int GVecResidualKSPMonitor(KSP, int, PetscReal, void *);
extern int GVecErrorKSPMonitor(KSP, int, PetscReal, void *);
extern int GVecRhsKSPMonitor(KSP, int, PetscReal, void *);

#define GVECPCMGMONITOR_SOLUTION 1
#define GVECPCMGMONITOR_RESIDUAL 2
#define GVECPCMGMONITOR_RHS      4
extern int GVecPCMGSetMonitor(PC, int);

/* This is a uniform interface for the solution of Schur complement systems */
typedef struct {
  GMat A;
  GMat B;
  SLES sles;
  GVec work, work2;
} UzawaContext;

extern int GMatCreateUzawa(GMat, GMat, SLES, GMat *);
extern int GMatDestroyUzawa(GMat);
extern int GMatMatMultUzawa(GMat, GVec, GVec);

#if defined(__PETSCSNES_H)
#include "petscsnes.h"

/* ----------------------------------------------------------------------------*/
/*
  The same alternative monitors are also provided for KSP solvers embedded in SNES.
*/
#define GSNES SNES

typedef struct {
  PetscViewer error_viewer;
  PetscViewer norm_error_viewer;
  int       (*solutionFunc)(Vec, void *);
  void       *ctx;
  GVec        solution;
} GSNESErrorMonitorCtx;

extern int GSNESCreate(Grid, void *, GSNES *);
extern int GSNESGetGrid(GSNES, Grid *);

extern int GSNESView(GSNES, PetscViewer);
extern int GSNESDuplicate(GSNES, GSNES *);
extern int GSNESDuplicateMonitors(GSNES, GSNES);
extern int GSNESReallocate(GSNES);
extern int GSNESDestroy(GSNES);

extern int GSNESEvaluateRhs(GSNES, GVec, GVec, PetscObject);
extern int GSNESEvaluateRhsFunction(GSNES, GVec, GVec, void *);
extern int GSNESEvaluateRhsOperator(GSNES, GVec, GVec, void *);
extern int GSNESEvaluateRhsLinearOperator(GSNES, GVec, GVec, void *);
extern int GSNESEvaluateRhsNonlinearOperator(GSNES, int, GVec[], GVec, void *);
extern int GSNESEvaluateJacobian(GSNES, GVec, GMat *, GMat *, MatStructure *, PetscObject);
extern int GSNESEvaluateJacobianMF(GSNES, GVec, GVec, GVec, void *);
extern int GSNESRhsBC(GSNES, Vec, void *);
extern int GSNESSolutionBC(GSNES, Vec, void *);
extern int GSNESUpdate(GSNES, int);

extern int GSNESDefaultComputeJacobianWithColoring(SNES, GVec, GMat *, GMat *, MatStructure *, void *);

extern int GSNESSolutionMonitor(SNES, int, PetscReal, void *);
extern int GSNESResidualMonitor(SNES, int, PetscReal, void *);
extern int GSNESErrorMonitor(SNES, int, PetscReal, void *);

#if defined(__PETSCTS_H)
#include "petscts.h"

/* ----------------------------------------------------------------------------*/
/*
  Alternative monitors are also provided.
*/
#define GTS TS

#define GTS_SER_BEULER_BINARY "gbeuler_binary"

/* The logging events */
extern int GTS_Reform, GTS_Reallocate;

typedef struct {
  PetscViewer error_viewer;
  PetscViewer norm_error_viewer;
  int       (*solutionFunc)(Vec, void *);
  void       *ctx;
  GVec        solution;
  double      norm_2;
  double      norm_max;
} GTSErrorMonitorCtx;

/* This is my idea for eliminating the proliferation of function for forming the
   Rhs and Jacobian (Matrix). We may in the future have more parameters than time,
   for instance a continuation parameter. Thus I want a system which can handle this
   with C calling sequences. I wrap the original user context in a package context,
   like GTSContext. The user must know that this wrapping happens when using a
   certain package, but his own context is preserved.
*/
struct _GTSContext {
  PETSCHEADER(int)
  double t;     /* Current time */
  double dt;    /* Current time step */
  Vec    sol;   /* Current solution */
  int    nwork; /* The number of work vectors */
  Vec   *work;  /* The work vectors */
};
typedef struct _GTSContext *GTSContext;


extern int GTSCreate(Grid, void *, GTS *);
extern int GTSGetGrid(GTS, Grid *);

extern int GTSView(GTS, PetscViewer);
extern int GTSSerialize(Grid, GTS *, PetscViewer, PetscTruth);
extern int GTSDuplicate(GTS, GTS *);
extern int GTSReform(GTS);
extern int GTSReallocate(GTS, GVec);
extern int GTSDestroy(GTS);

extern int GTSGetInitialTimeStep(GTS, double *);
extern int GTSGetTimeDependence(GTS, int, PetscTruth *);
extern int GTSSetTimeDependence(GTS, int, PetscTruth);
extern int GTSSetContext(GTS, void *);

extern int GTSEvaluateRhs(GTS, double, GVec, GVec, PetscObject);
extern int GTSEvaluateJacobian(GTS, double, GVec, GMat *, GMat *, MatStructure *, PetscObject);
extern int GTSEvaluateSystemMatrix(GTS, double, GMat *, GMat *, MatStructure *, PetscObject);
extern int GTSCalcBCValues(GTS);
extern int GTSRhsBC(GTS, Vec, void *);
extern int GTSSolutionBC(GTS, Vec, void *);

extern int GTSCreateContext(GTS, double, PetscObject, PetscObject *);
extern int GTSDestroyContext(GTS, PetscObject, PetscObject);
extern int GTSCreateConstraintContext(GTS);
extern int GTSDestroyConstraintContext(GTS);

extern int GTSPreStep(GTS);
extern int GTSUpdate(GTS, PetscReal, PetscReal *);
extern int GTSPostStep(GTS);

extern int GTSSolutionBCforGSNES(GSNES, GVec, void *);

extern int GTSSolutionMonitor(GTS, int, PetscReal, Vec, void *);
extern int GTSErrorMonitor(GTS, int, PetscReal, Vec, void *);
extern int GTSErrorMonitorforGSNES(GSNES, int, PetscReal, void *);

#endif /* __PETSCTS_H */
#endif /* __PETSCSNES_H */
#endif /* __PETSCSLES_H */
#endif /* __GSOLVER_H */
