/* $Id: mesh.h,v 1.16 2000/10/17 13:49:33 knepley Exp $ */

/*
    Routines for working with meshes
*/

#ifndef __MESH_H
#define __MESH_H

#include "petscdraw.h"
#include "petscvec.h"
#include "petscao.h"
#include "pointFunction.h"

/*
  A mesh object contains geometric information about a region, including
  possibly points, cells etc.
*/
typedef struct _Mesh *Mesh;

#define MESH_TRIANGULAR_1D "tri1d"
#define MESH_TRIANGULAR_2D "tri2d"
typedef char *MeshType;

#define MESH_SER_TRIANGULAR_1D_BINARY "tri_1d_binary"
#define MESH_SER_TRIANGULAR_2D_BINARY "tri_2d_binary"
typedef char *MeshSerializeType;

#define MESH_ORDER_TRIANGULAR_2D_RCM  "tri_2d_rcm"
#define MESH_ORDER_TRIANGULAR_2D_Test "tri_2d_test"
typedef char *MeshOrderingType;

/* The Partition definition */
#include "partition.h"

/* The structure for communicating 2D mesh boundary information */
typedef struct {
  int     numBd;       /* The number of closed boundaries */
  int     numVertices; /* The number of vertices */
  double *vertices;    /* The coordinates of the boundary vertices */
  int    *markers;     /* The markers of the boundary points */
  int     numSegments; /* The number of segments */
  int    *segments;    /* The boundary segment endpoints */
  int    *segMarkers;  /* The boundary segment markers */
  int     numHoles;    /* The number of holes in the mesh */
  double *holes;       /* The coordinates of the holes in the mesh */
} MeshBoundary2D;

/* Logging support */
extern int MESH_COOKIE;
extern int MESH_Reform, MESH_IsDistorted, MESH_Partition, MESH_SetupBoundary, MESH_MoveMesh, MESH_CalcNodeVelocities;
extern int MESH_CalcNodeAccelerations, MESH_CreateLocalCSR, MESH_CreateFullCSR, MESH_CreateDualCSR, MESH_LocatePoint;

EXTERN int MeshInitializePackage(char *);

/* Generic Operations */
extern int MeshCreate(MPI_Comm, Mesh *);
extern int MeshSetUp(Mesh);
extern int MeshSetFromOptions(Mesh);
extern int MeshView(Mesh, PetscViewer);
extern int MeshCopy(Mesh, Mesh);
extern int MeshDuplicate(Mesh, Mesh *);
extern int MeshSerialize(MPI_Comm, Mesh *, PetscViewer, PetscTruth);
extern int MeshDestroy(Mesh);
extern int MeshViewFromOptions(Mesh, char *);
extern int MeshSetUserContext(Mesh, void *);
extern int MeshGetUserContext(Mesh, void **);
extern int MeshSetOptionsPrefix(Mesh, char *);
extern int MeshAppendOptionsPrefix(Mesh, char *);
extern int MeshGetOptionsPrefix(Mesh, char **);
extern int MeshPrintHelp(Mesh);

/* Mesh-Specific Operations */
extern int MeshCheckBoundary(Mesh, MeshBoundary2D *);
extern int MeshSetBoundary(Mesh, MeshBoundary2D *);
extern int MeshSetReformBoundary(Mesh, MeshBoundary2D *);
extern int MeshPartition(Mesh);
extern int MeshCoarsen(Mesh, PointFunction, Mesh *);
extern int MeshRefine(Mesh, PointFunction, Mesh *);
extern int MeshResetNodes(Mesh, PetscTruth);
extern int MeshSaveMesh(Mesh);
extern int MeshRestoreMesh(Mesh);

/* Mesh Query Interface */
  /* mesh query functions */
extern int MeshSetDimension(Mesh, int);
extern int MeshGetDimension(Mesh, int *);
extern int MeshGetInfo(Mesh, int *, int *, int *, int *);
extern int MeshSetNumCorners(Mesh, int);
extern int MeshGetNumCorners(Mesh, int *);
extern int MeshSetBoundingBox(Mesh, PetscReal, PetscReal, PetscReal, PetscReal, PetscReal, PetscReal);
extern int MeshGetBoundingBox(Mesh, PetscReal *, PetscReal *, PetscReal *, PetscReal *, PetscReal *, PetscReal *);
extern int MeshSetLocalBoundingBox(Mesh, PetscReal, PetscReal, PetscReal, PetscReal, PetscReal, PetscReal);
extern int MeshGetLocalBoundingBox(Mesh, PetscReal *, PetscReal *, PetscReal *, PetscReal *, PetscReal *, PetscReal *);
extern int MeshUpdateBoundingBox(Mesh);
extern int MeshGetPartition(Mesh, Partition *);
extern int MeshSetMovement(Mesh, PetscTruth );
extern int MeshGetMovement(Mesh, PetscTruth *);
extern int MeshIsDistorted(Mesh, PetscTruth *);
extern int MeshGetMaxDegree(Mesh, int *);

  /* mesh boundary query functions */
extern int MeshGetNumBoundaries(Mesh, int *);
extern int MeshGetBoundarySize(Mesh, int, int *);
extern int MeshGetBoundaryIndex(Mesh, int, int *);
extern int MeshGetBoundaryStart(Mesh, int, PetscTruth, int *);
extern int MeshGetBoundaryNext(Mesh, int, PetscTruth, int *);
extern int MeshGetActiveBoundary(Mesh, int *);

  /* mesh node query functions */
extern int MeshGetNodeBoundary(Mesh, int, int *);
extern int MeshNodeIsVertex(Mesh, int, PetscTruth *);
extern int MeshGetNodeCoords(Mesh, int, double *, double *, double *);
extern int MeshSetNodeCoords(Mesh, int, double, double, double);
extern int MeshGetNodeCoordsSaved(Mesh, int, double *, double *, double *);
extern int MeshGetNearestNode(Mesh, double, double, double, PetscTruth, int *);
extern int MeshGetNearestBdNode(Mesh, double, double, double, int *);
extern int MeshGetNodeSupport(Mesh, int, int, int *, int **);
extern int MeshRestoreNodeSupport(Mesh, int, int, int *, int **);
extern int MeshGetNodeOrdering(Mesh, AO *);

  /* mesh element query functions */
extern int MeshGetElementNeighbor(Mesh, int, int, int *);
extern int MeshLocatePoint(Mesh, double, double, double, int *);

  /* mesh hole query functions */
extern int MeshSetHoleCoords(Mesh, int, Vec);

  /* mesh embedding query functions */
extern int MeshGetElementFromNode(Mesh, int, int *);
extern int MeshGetBdElementFromEdge(Mesh, int, int *);
extern int MeshGetNodeFromElement(Mesh, int, int, int *);
extern int MeshGetNodeFromEdge(Mesh, int, int, int *);
extern int MeshGetMidnodeFromEdge(Mesh, int, int *);

/* Mesh CSR Support Functions */
extern int MeshCreateLocalCSR(Mesh, int *, int *, int **, int **, double **, int, int *, PetscTruth);
extern int MeshDestroyLocalCSR(Mesh, int *, int *, double *);
extern int MeshCreateFullCSR(Mesh, PetscTruth, int *, int *, int **, int **);
extern int MeshDestroyFullCSR(Mesh, int *, int *);
extern int MeshCreateDualCSR(Mesh, int **, int **, int **, int);
extern int MeshDestroyDualCSR(Mesh, int *, int *, int *);

/* Mesh Drawing Support */
extern int MeshDrawLine(Mesh, PetscDraw, double, double, double, double, int);
extern int MeshDrawTriangle(Mesh, PetscDraw, double, double, double, double, double, double, int, int, int);
extern int MeshSetHighlightElement(Mesh, int);
extern int MeshGetHighlightElement(Mesh, int *);

/* Periodic Mesh Support */
extern int MeshIsPeriodic(Mesh, PetscTruth *);
extern int MeshSetPeriodicDimension(Mesh, int, PetscTruth);
extern int MeshIsPeriodicDimension(Mesh, int, PetscTruth *);
#if defined(MESH_PERIODIC_OPTIMIZED)

#define MeshPeriodicX(mesh, x) ((x < mesh->startX) ? x + mesh->sizeX : ((x > mesh->endX) ? x - mesh->sizeX : x))
#define MeshPeriodicY(mesh, y) ((y < mesh->startY) ? y + mesh->sizeY : ((y > mesh->endY) ? y - mesh->sizeY : y))
#define MeshPeriodicZ(mesh, z) ((z < mesh->startZ) ? z + mesh->sizeZ : ((z > mesh->endZ) ? z - mesh->sizeZ : z))
#define MeshPeriodicRelativeX(mesh, x, xO) ((x < xO - mesh->sizeX) ? x + mesh->sizeX : ((x > xO + mesh->sizeX) ? x - mesh->sizeX : x))
#define MeshPeriodicRelativeY(mesh, y, yO) ((y < yO - mesh->sizeY) ? y + mesh->sizeY : ((y > yO + mesh->sizeY) ? y - mesh->sizeY : y))
#define MeshPeriodicRelativeZ(mesh, z, zO) ((z < zO - mesh->sizeZ) ? z + mesh->sizeZ : ((z > zO + mesh->sizeZ) ? z - mesh->sizeZ : z))
#define MeshPeriodicDiffX(mesh, diff) ((fabs(diff) > mesh->sizeX) ? diff - copysign(1.0, diff)*mesh->sizeX : diff)
#define MeshPeriodicDiffY(mesh, diff) ((fabs(diff) > mesh->sizeY) ? diff - copysign(1.0, diff)*mesh->sizeY : diff)
#define MeshPeriodicDiffZ(mesh, diff) ((fabs(diff) > mesh->sizeZ) ? diff - copysign(1.0, diff)*mesh->sizeZ : diff)

#else

extern double MeshPeriodicX(Mesh mesh, double x);
extern double MeshPeriodicY(Mesh mesh, double y);
extern double MeshPeriodicZ(Mesh mesh, double z);
extern double MeshPeriodicRelativeX(Mesh mesh, double x, double xO);
extern double MeshPeriodicRelativeY(Mesh mesh, double y, double yO);
extern double MeshPeriodicRelativeZ(Mesh mesh, double z, double zO);
extern double MeshPeriodicDiffX(Mesh mesh, double diff);
extern double MeshPeriodicDiffY(Mesh mesh, double diff);
extern double MeshPeriodicDiffZ(Mesh mesh, double diff);

#endif /* MESH_PERIODIC_OPTIMIZED */

/* Mesh Boundary Support */
typedef struct
{
  int            dim;           /* The dimension of the simulation */
  double         size[3];       /* The size of the domain in each dimension */
  double         start[3];      /* The start of the domain in each dimension */
  PetscTruth     isPeriodic[3]; /* The flags for periodicity in each direction */
  double         refineDist;    /* The characteristic distance before wall size elements appear */
  double         maxArea;       /* The maximum area (volume) an element may have after refinement */
  double         areaMult;      /* The factor to multiply maxArea by at each refinement */
  PointFunction  areaFunc;      /* The function returning the maximum area (volume) of a refined element */
  void          *areaCtx;       /* The context for areaFunc */
} MeshGeometryContext;

typedef enum {OUTER_BD = 1, TOP_BD, BOTTOM_BD, LEFT_BD, RIGHT_BD} MeshBoundaryType;

extern int MeshBoundary1DCreateSimple(MPI_Comm, MeshGeometryContext *, MeshBoundary2D *);
extern int MeshBoundary1DDestroy(MPI_Comm, MeshBoundary2D *);

extern int MeshBoundary2DCreateSimple(MPI_Comm, MeshGeometryContext *, MeshBoundary2D *);
extern int MeshBoundary2DView(Mesh, MeshBoundary2D *, PetscViewer);
extern int MeshBoundary2DDestroy(MPI_Comm, MeshBoundary2D *);

extern int MeshBoundary2DSetBoundingBox(Mesh, MeshBoundary2D *);

/* Dynamic creation and loading functions */
extern PetscFList MeshList;
extern PetscTruth MeshRegisterAllCalled;
EXTERN int MeshSetType(Mesh, MeshType);
EXTERN int MeshGetType(Mesh, MeshType *);
EXTERN int MeshRegister(const char[],const char[],const char[],int(*)(Mesh));
EXTERN int MeshRegisterAll(const char []);
EXTERN int MeshRegisterDestroy(void);
#if defined(PETSC_USE_DYNAMIC_LIBRARIES)
#define MeshRegisterDynamic(a,b,c,d) MeshRegister(a,b,c,0)
#else
#define MeshRegisterDynamic(a,b,c,d) MeshRegister(a,b,c,d)
#endif

extern PetscFList MeshSerializeList;
extern PetscTruth MeshSerializeRegisterAllCalled;
EXTERN int MeshSetSerializeType(Mesh, MeshSerializeType);
EXTERN int MeshGetSerializeType(Mesh, MeshSerializeType *);
EXTERN int MeshSerializeRegister(const char [], const char [], const char [], int (*)(MPI_Comm, Mesh *, PetscViewer, PetscTruth));
EXTERN int MeshSerializeRegisterAll(const char []);
EXTERN int MeshSerializeRegisterDestroy(void);
#if defined(PETSC_USE_DYNAMIC_LIBRARIES)
#define MeshSerializeRegisterDynamic(a,b,c,d) MeshSerializeRegister(a,b,c,0)
#else
#define MeshSerializeRegisterDynamic(a,b,c,d) MeshSerializeRegister(a,b,c,d)
#endif

extern PetscFList MeshOrderingList;
extern PetscTruth MeshOrderingRegisterAllCalled;
EXTERN int MeshGetOrdering(Mesh, MeshOrderingType, AO *);
EXTERN int MeshOrderingRegister(const char [], const char [], const char [], int (*)(Mesh, MeshOrderingType, AO *));
EXTERN int MeshOrderingRegisterAll(const char []);
EXTERN int MeshOrderingRegisterDestroy(void);
EXTERN int MeshGetOrdering(Mesh, MeshOrderingType, AO *);
#if defined(PETSC_USE_DYNAMIC_LIBRARIES)
#define MeshOrderingRegisterDynamic(a,b,c,d) MeshOrderingRegister(a,b,c,0)
#else
#define MeshOrderingRegisterDynamic(a,b,c,d) MeshOrderingRegister(a,b,c,d)
#endif

/* 2D special functions */
EXTERN int MeshTriangular2DCalcAreas(Mesh, PetscTruth);
EXTERN int MeshTriangular2DCalcAspectRatios(Mesh, PetscTruth);

/* CSR special functions */
EXTERN int MeshCreateTriangular2DCSR(MPI_Comm, int, int, double *, int *, int *, int *, Mesh *);

#endif /* __MESH_H */
