/* $Id: stokes.h,v 1.1 2001/10/25 15:55:45 knepley Exp $ */

/* Main include file for the Stokes flow test problem */

#ifndef __STOKES_H
#define __STOKES_H

#include "petscts.h"
#include "gsolver.h"

extern int STOKES_COOKIE;
extern int STOKES_ComputeField;

/* This structure holds information about physical properties */
typedef struct {
  double nu;      /* The kinematic viscosity */
  double fluidSG; /* The specific gravity of the fluid */
} PhysicalContext;

/* This structure holds the problem geometry information */
typedef struct {
  int            dim;           /* The dimension of the simulation */
  double         size[3];       /* The size of the domain in each dimension */
  double         start[3];      /* The start of the domain in each dimension */
  PetscTruth     isPeriodic[3]; /* The flags for periodicity in each direction */
  double         refineDist;    /* The characteristic distance before wall size elements appear */
  double         maxArea;       /* The maximum area (volume) an element may have after refinement */
  double         areaMult;      /* The factor to multiply maxArea by at each refinement */
  PointFunction *areaFunc;      /* The function returning the maximum area (volume) of a refined element */
  void          *areaCtx;       /* The context for areaFunc */
  double         wallDensity;   /* The number of nodes per unit length on a channel boundary */
  double         wallSize;      /* The characteristic size of elements near the wall */
} GeometryContext;

struct _StokesContext {
  PETSCHEADER(int)
  Grid                   grid;        /* The problem grid */

  MeshBoundary2D         boundaryCtx; /* Describes the domain boundary */
  MeshGeometryContext    geometryCtx; /* Describes the domain geometry */
  PhysicalContext        physicalCtx; /* Describes the fluid properties */

  int                    RofS;        /* The rate-of-strain operator */
  PetscTruth             useML;       /* Flag for ML preconditioning */
  PetscTruth             useSchur;    /* Flag for Schur complement preconditioning */

  SLES                   sles;        /* The linear solver context */
  GMat                   A;           /* The velocity system matrix */
  GVec                   u;           /* The velocity solution vector */
  GVec                   p;           /* The pressure solution vector */
  GVec                   f;           /* The reduced rhs vector */
  GVec                   origF;       /* The original rhs vector */
  GVec                   uExact;      /* The exact velocity solution vector */
  GVec                   pExact;      /* The exact pressure solution vector */

  int                    dim;         /* The problem dimension */
  PetscTruth             linear;      /* Using linear elements (as opposed to quadratic) */
  int                    refineStep;  /* The step to start refining the mesh */
  int                    numLoops;    /* The number of systems to solve */

  GVecErrorKSPMonitorCtx monitorCtx;  /* Specialized monitor for grid vector linear solves */
};
typedef struct _StokesContext *StokesContext;

/* Creation functions */
EXTERN int StokesInitializePackage(char *);
EXTERN int StokesContextCreate(MPI_Comm, StokesContext *);
EXTERN int StokesContextSetup(StokesContext);
EXTERN int StokesContextCreateGrid(StokesContext);
EXTERN int StokesContextRefineGrid(StokesContext);
EXTERN int StokesCreateStructures(StokesContext);

/* Setup functions */
EXTERN int StokesSetupGrid(StokesContext);
EXTERN int StokesSetupRhsFunction(Grid, StokesContext);
EXTERN int StokesSetupBC(Grid, StokesContext);
EXTERN int StokesSetupKSP(KSP, StokesContext);
EXTERN int StokesSetupPC(PC, StokesContext);
EXTERN int StokesSetupStructures(StokesContext);

/* Cleanup functions */
EXTERN int StokesDestroyStructures(StokesContext);
EXTERN int StokesDestroy(StokesContext);
EXTERN int StokesContextDestroy(StokesContext);

/* Checking functions */
EXTERN int MatCheckSymmetry(Mat);
EXTERN int StokesCheckSolution(StokesContext, GVec, GVec, const char []);

/* Callback functions */
EXTERN int VelocitySolutionFunction(int, int, double *, double *, double *, PetscScalar *, void*);
EXTERN int PressureSolutionFunction(int, int, double *, double *, double *, PetscScalar *, void*);
EXTERN int VelocityRhsFunction(int, int, double *, double *, double *, PetscScalar *, void*);

/* Operator functions */
EXTERN int RateOfStrainTensor(Discretization, Discretization, int, int, int, int, int, double *,
                              PetscScalar, PetscScalar *, PetscScalar *, void *);

/* Solver functions */
EXTERN int StokesSolve(StokesContext, GVec, GVec, int *);
EXTERN int StokesComputeFlowField(StokesContext);

#endif /* __STOKES_H */
