/*
  Plee the Bear

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file controller_layout.cpp
 * \brief Implementation of the bear::engine::controller_layout class.
 * \author Julien Jorge
 */
#include "ptb/controller_layout.hpp"

#include <claw/logger.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the action associated to a keyboard key.
 * \param key The code of the key.
 * \param a The action done with this key.
 */
void ptb::controller_layout::set_key
( bear::input::key_code key, player_action::value_type a )
{
  remove_key(key);
  remove_action(a);
  m_keyboard[key] = a;
} // controller_layout::set_key()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the action associated to a keyboard key.
 * \param key The code of the key.
 */
ptb::player_action::value_type
ptb::controller_layout::operator()( bear::input::key_code key ) const
{
  keyboard_map::const_iterator it;

  it = m_keyboard.find(key);

  if ( it != m_keyboard.end() )
    return it->second;
  else
    return player_action::action_null;
} // controller_layout::operator()( key_code ) [const]

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the action associated to a key of a joystick.
 * \param joy The index of the joystick.
 * \param key The code of the key.
 * \param a The action done with this key.
 */
void ptb::controller_layout::set_joystick
( unsigned int joy, bear::input::joystick::joy_code key,
  player_action::value_type a )
{
  remove_joy(joy, key);
  remove_action(a);
  m_joystick[ bear::input::joystick_button(joy, key) ] = a;
} // controller_layout::set_joystick()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the action associated to a key of a joystick.
 * \param joy The index of the joystick.
 * \param key The code of the key.
 */
ptb::player_action::value_type
ptb::controller_layout::operator()
  ( unsigned int joy, bear::input::joystick::joy_code key ) const
{
  joystick_map::const_iterator it;

  it = m_joystick.find( bear::input::joystick_button(joy, key) );

  if ( it != m_joystick.end() )
    return it->second;
  else
    return player_action::action_null;
} // controller_layout::operator()( unsigned int, joy_code ) [const]

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the action associated to a mouse button.
 * \param button The code of the button.
 * \param key The code of the key.
 */
void ptb::controller_layout::set_mouse
( bear::input::mouse::mouse_code button, player_action::value_type a )
{
  remove_mouse(button);
  remove_action(a);
  m_mouse[button] = a;
} // controller_layout::operator()( mouse_code )

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the action associated to a mouse button.
 * \param button The code of the button.
 */
ptb::player_action::value_type
ptb::controller_layout::operator()
  ( bear::input::mouse::mouse_code button ) const
{
  mouse_map::const_iterator it;

  it = m_mouse.find(button);

  if ( it != m_mouse.end() )
    return it->second;
  else
    return player_action::action_null;
} // controller_layout::operator()( mouse_code ) [const]

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the keyboard key associated with an action.
 * \return bear::input::keyboard::kc_not_a_key if no key is associated with the
 *         action.
 */
bear::input::key_code
ptb::controller_layout::find_key( ptb::player_action::value_type a ) const
{
  bear::input::key_code result = bear::input::keyboard::kc_not_a_key;

  keyboard_map::const_iterator it;

  for ( it=m_keyboard.begin();
        (it!=m_keyboard.end())
          && (result == bear::input::keyboard::kc_not_a_key);
        ++it )
    if ( it->second == a )
      result = it->first;

  return result;
} // controller_layout::find_key()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the joystick key and index associated with an action.
 * \return result.button == bear::input::joystick::jc_invalid if no key is
 *         associated with the action.
 */
bear::input::joystick_button
ptb::controller_layout::find_joystick( ptb::player_action::value_type a ) const
{
  bear::input::joystick_button result( 0, bear::input::joystick::jc_invalid );

  joystick_map::const_iterator it;

  for ( it=m_joystick.begin();
        (it!=m_joystick.end())
          && (result.button == bear::input::joystick::jc_invalid);
        ++it )
    if ( it->second == a )
      result = it->first;

  return result;
} // controller_layout::find_joystick()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the mouse key and index associated with an action.
 * \return result == bear::input::mouse::mc_invalid if no button is associated
 *         with the action.
 */
bear::input::mouse::mouse_code
ptb::controller_layout::find_mouse( ptb::player_action::value_type a ) const
{
  bear::input::mouse::mouse_code result( bear::input::mouse::mc_invalid );

  mouse_map::const_iterator it;

  for ( it=m_mouse.begin();
        (it!=m_mouse.end()) && (result == bear::input::mouse::mc_invalid);
        ++it )
    if ( it->second == a )
      result = it->first;

  return result;
} // controller_layout::find_mouse()

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert the escaped action sequence of a string.
 * \param str (in/out) The string to convert.
 */
void ptb::controller_layout::escape_action_sequence
( const std::string& str, std::vector<std::string>& result ) const
{
  unsigned int ref = 0;
  unsigned int prev = 0;
  unsigned int current = 1;

  while ( current < str.size() )
    if ( str[prev] == '%' )
      {
        if ( str[current] == 'a' )
          {
            result.push_back( str.substr(ref, prev - ref) );
            current = append_action_string(result, str, current);
          }
        else
          result.push_back( str.substr(ref, current - ref + 1) );

        ref = current + 1;
        prev = ref;
        current = prev + 1;
      }
    else
      {
        ++prev;
        ++current;
      }

  if ( ref < str.size() )
    result.push_back( str.substr(ref) );
} // controller_layout::escape_action_sequence()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the layout from a stream.
 * \param f The stream from which we read.
 */
void ptb::controller_layout::load( std::istream& f )
{
  unsigned int n;
  unsigned int joy;
  bear::input::key_code key_code;
  bear::input::joystick::joy_code joy_code;
  bear::input::mouse::mouse_code mouse_code;
  int action;

  m_keyboard.clear();
  m_joystick.clear();
  m_mouse.clear();

  f >> n;

  for (unsigned int i=0; i!=n; ++i)
    if ( f >> key_code >> action )
      m_keyboard[key_code] = (ptb::player_action::value_type)action;

  f >> n;

  for (unsigned int i=0; i!=n; ++i)
    if ( f >> joy >> joy_code >> action )
      {
        if ( joy >= bear::input::joystick::number_of_joysticks() )
          claw::logger << claw::log_warning << "Invalid joystick index: "
                       << joy << std::endl;
        else
          m_joystick[ bear::input::joystick_button(joy, joy_code) ] =
            (ptb::player_action::value_type)action;
      }

  f >> n;

  for (unsigned int i=0; i!=n; ++i)
    if ( f >> mouse_code >> action )
      m_mouse[mouse_code] = (ptb::player_action::value_type)action;
} // controller_layout::load()

/*----------------------------------------------------------------------------*/
/**
 * \brief save the layout in a stream.
 * \param f The stream in which we write.
 */
void ptb::controller_layout::save( std::ostream& f ) const
{
  keyboard_map::const_iterator it_k;
  joystick_map::const_iterator it_j;
  mouse_map::const_iterator it_m;

  f << m_keyboard.size() << std::endl;

  for (it_k=m_keyboard.begin(); it_k!=m_keyboard.end(); ++it_k)
    f << it_k->first << " " << it_k->second << std::endl;

  f << m_joystick.size() << std::endl;

  for (it_j=m_joystick.begin(); it_j!=m_joystick.end(); ++it_j)
    f << it_j->first.joystick_index << " " << it_j->first.button << " "
      << it_j->second << std::endl;

  for (it_m=m_mouse.begin(); it_m!=m_mouse.end(); ++it_m)
    f << it_m->first << " " << it_m->second << std::endl;
} // controller_layout::save()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove an action from the controller.
 * \param key The keyboard key of the action to remove.
 */
void
 ptb::controller_layout::remove_key( bear::input::key_code key )
{
  m_keyboard.erase(key);
} // controller_layout::remove_key()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove an action from the controller.
 * \param joy The joystick index of the action to remove.
 * \param key The button of the action to remove.
 */
void ptb::controller_layout::remove_joy
( unsigned int joy, bear::input::joystick::joy_code key )
{
  m_joystick.erase( bear::input::joystick_button(joy, key) );
} // controller_layout::remove_joy()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove an action from the controller.
 * \param button The mouse button of the action to remove.
 */
void
ptb::controller_layout::remove_mouse( bear::input::mouse::mouse_code button )
{
  m_mouse.erase(button);
} // controller_layout::remove_mouse()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if no keys are configured.
 */
bool ptb::controller_layout::empty() const
{
  return m_joystick.empty() && m_keyboard.empty() && m_mouse.empty();
} // controller_layout::empty()

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert the "\\ai" escape sequence.
 * \param result (in/out) The string table to which we append the action.
 * \param str The string to convert.
 * \param current The position of the parameters of the 'a' in the \\a
 *        sequence.
 * \return The position of the last character read.
 */
unsigned int ptb::controller_layout::append_action_string
( std::vector<std::string>& result, const std::string& str,
  unsigned int current ) const
{
  std::string::size_type pos = str.find_first_of(';', current);
  bool ok = false;

  if ( pos != std::string::npos )
    {
      std::istringstream iss( str.substr(current+1, pos - current - 1) );
      unsigned int action;

      if ( iss >> action)
        if ( iss.rdbuf()->in_avail() == 0 )
          if ( append_action_string(result, action) )
            {
              current = pos;
              ok = true;
            }
    }

  if (!ok)
    result.push_back("%a");

  return current;
} // controller_layout::append_action_string()

/*----------------------------------------------------------------------------*/
/**
 * \brief Append to a string the string corresponding to the action of a given
 *        player.
 * \param str (in/out) The string table to which we append the action.
 * \param action The action from which we want the string.
 * \param player The player doing the action.
 * \return The position of the next character to read.
 */
bool ptb::controller_layout::append_action_string
( std::vector<std::string>& str, unsigned int action ) const
{
  bool result = true;

  bear::input::key_code key = find_key( action );
  bear::input::joystick_button joy = find_joystick( action );
  bear::input::mouse::mouse_code mouse = find_mouse( action );

  if ( key != bear::input::keyboard::kc_not_a_key )
    str.push_back( bear::input::keyboard::get_name_of(key) );
  else if ( joy.button != bear::input::joystick::jc_invalid )
    {
      std::ostringstream oss;
      oss << joy.joystick_index << ' ';

      str.push_back( "joystick " );
      str.push_back( oss.str() );
      str.push_back( bear::input::joystick::get_name_of(joy.button) );
    }
  else if ( mouse != bear::input::mouse::mc_invalid )
    str.push_back( bear::input::mouse::get_name_of(mouse) );
  else
    result = false;

  return result;
} // controller_layout::append_action_string()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove an action from all controllers.
 * \param a The action to remove.
 */
void ptb::controller_layout::remove_action( player_action::value_type a )
{
  bear::input::key_code key = find_key( a );

  while ( key != bear::input::keyboard::kc_not_a_key )
    {
      remove_key(key);
      key = find_key( a );
    }

  bear::input::joystick_button joy = find_joystick( a );

  while ( joy.button != bear::input::joystick::jc_invalid )
    {
      remove_joy(joy.joystick_index, joy.button);
      joy = find_joystick( a );
    }

  bear::input::mouse::mouse_code mouse = find_mouse( a );

  while ( mouse != bear::input::mouse::mc_invalid )
    {
      remove_mouse(mouse);
      find_mouse( a );
    }
} // controller_layout::remove_action()
