/*
  Plee the Bear

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file controller_layout.hpp
 * \brief This class makes the correspondence between the controllers keys and
 *        the actions of the player.
 * \author Julien Jorge
 */
#ifndef __PTB_CONTROLLER_LAYOUT_HPP__
#define __PTB_CONTROLLER_LAYOUT_HPP__

#include <map>

#include "ptb/player_action.hpp"
#include "input/keyboard.hpp"
#include "input/joystick_button.hpp"
#include "input/mouse.hpp"

namespace ptb
{
  /**
   * \brief This class makes the correspondence between the controllers keys
   *        and the actions of the player.
   * \author Julien Jorge
   */
  class controller_layout
  {
  private:
    /** \brief The type of the map to store the actions associated with the
        keyboard. */
    typedef std::map<bear::input::key_code,
                     player_action::value_type> keyboard_map;

    /** \brief The type of the map to store the actions associated with the
        joysticks. */
    typedef std::map<bear::input::joystick_button,
                     player_action::value_type> joystick_map;

    /** \brief The type of the map to store the actions associated with the
        mouse. */
    typedef std::map<bear::input::mouse::mouse_code,
                     player_action::value_type> mouse_map;

  public:
    void set_key( bear::input::key_code key, player_action::value_type a );
    player_action::value_type operator()( bear::input::key_code key ) const;

    void set_joystick
    ( unsigned int joy, bear::input::joystick::joy_code key,
      player_action::value_type a );
    player_action::value_type operator()
      ( unsigned int joy, bear::input::joystick::joy_code key ) const;

    void set_mouse
    ( bear::input::mouse::mouse_code button, player_action::value_type a );
    player_action::value_type
    operator()( bear::input::mouse::mouse_code button ) const;

    bear::input::key_code find_key( player_action::value_type a ) const;
    bear::input::joystick_button
    find_joystick( player_action::value_type a ) const;
    bear::input::mouse::mouse_code
    find_mouse( player_action::value_type a ) const;

    void escape_action_sequence
    ( const std::string& str, std::vector<std::string>& result ) const;

    void load( std::istream& f );
    void save( std::ostream& f ) const;

    void remove_key( bear::input::key_code key );
    void remove_joy( unsigned int joy, bear::input::joystick::joy_code key );
    void remove_mouse( bear::input::mouse::mouse_code button );

    bool empty() const;

  private:
    unsigned int append_action_string
    ( std::vector<std::string>& result, const std::string& str,
      unsigned int current ) const;
    bool append_action_string
    ( std::vector<std::string>& str, unsigned int action ) const;

    void remove_action( player_action::value_type a );

  private:
    /** \brief Actions associated with the keyboard. */
    keyboard_map m_keyboard;

    /** \brief Actions associated with the joysticks. */
    joystick_map m_joystick;

    /** \brief Actions associated with the mouse. */
    mouse_map m_mouse;

  }; // class controller_layout
} // namespace ptb

#endif // __PTB_CONTROLLER_LAYOUT_HPP__
