/*
  Plee the Bear

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file owl.cpp
 * \brief Implementation of the ptb::owl class.
 * \author Julien Jorge
 */
#include "ptb/item/owl.hpp"

#include "ptb/defines.hpp"
#include "ptb/level_variables.hpp"
#include "ptb/item/azelnut.hpp"
#include "ptb/item/plee/plee.hpp"

BASE_ITEM_EXPORT( owl, ptb )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::owl::owl()
  : m_action(action_idling), m_progress(NULL), m_first_player(NULL),
    m_second_player(NULL), m_has_azelnut(false)
{
  set_phantom(true);
} // owl::owl()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the media required by this class.
 */
void ptb::owl::pre_cache()
{
  get_level_globals().load_model("model/owl.cm");
} // owl::pre_cache()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do post creation actions.
 */
void ptb::owl::build()
{
  super::build();

  if ( m_other_player.empty() )
    m_other_player.push_back( gettext("Did you lose your friend?") );

  if ( m_already_have.empty() )
    m_already_have.push_back
      ( gettext("I already have my lucky azelnut. Go away!") );

  if ( m_happy.empty() )
    m_happy.push_back( gettext("So you found my lucky azelnut!") );

  if ( m_want_azelnut.empty() )
    m_want_azelnut.push_back
      ( gettext("I'd like to have my lucky azelnut back.") );

  if ( m_goodbye.empty() )
    m_goodbye.push_back( gettext("See you in an other world.") );

  if ( m_receive_azelnut.empty() )
    m_receive_azelnut.push_back( gettext("That's it!") );

  speaker_item::build( get_level_globals() );
  
  set_speaker_item(this);
  set_model_actor( get_level_globals().get_model("model/owl.cm") );

  start_model_action("idle");
} // owl::build()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the item.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::owl::progress( bear::universe::time_type elapsed_time )
{
  super::progress(elapsed_time);

  if ( !m_activated )
    m_action = action_idling;

  if ( has_finished_to_chat() )
    start_model_action("idle");

  if ( m_progress != NULL )
    (this->*m_progress)(elapsed_time);

  m_activated = false;
} // owl::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the owl is well initialized.
 */
bool ptb::owl::is_valid() const
{
  return super::is_valid() && !m_level_name.empty();
} // owl::is_valid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type <std::string>.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool ptb::owl::set_string_field
( const std::string& name, const std::string& value )
{
  bool result;

  if ( name == "owl.level" )
    {
      m_level_name = value;
      result = true;
    }
  else
    result = super::set_string_field( name, value );

  return result;
} // owl::set_string_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type list of <std::string>.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool ptb::owl::set_string_list_field
( const std::string& name, const std::vector<std::string>& value )
{
  bool result;

  if ( name == "owl.speech.want_an_azelnut" )
    {
      for(std::size_t i=0; i!=value.size(); ++i)
        if ( !value[i].empty() )
          m_want_azelnut.push_back( gettext(value[i].c_str()) );
      result = true;
    }
  else if ( name == "owl.speech.happy" )
    {
      for(std::size_t i=0; i!=value.size(); ++i)
        if ( !value[i].empty() )
          m_happy.push_back( gettext(value[i].c_str()) );
      result = true;
    }
  else if ( name == "owl.speech.other_player" )
    {
      for(std::size_t i=0; i!=value.size(); ++i)
        if ( !value[i].empty() )
          m_other_player.push_back( gettext(value[i].c_str()) );
      result = true;
    }
  else if ( name == "owl.speech.already_have" )
    {
      for(std::size_t i=0; i!=value.size(); ++i)
        if ( !value[i].empty() )
          m_already_have.push_back( gettext(value[i].c_str()) );
      result = true;
    }
  else if ( name == "owl.speech.bye" )
    {
      for(std::size_t i=0; i!=value.size(); ++i)
        if ( !value[i].empty() )
          m_goodbye.push_back( gettext(value[i].c_str()) );
      result = true;
    }
  else
    result = super::set_string_list_field( name, value );

  return result;
} // owl::set_string_list_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Activate on with only one player in a two players game.
 * \param p The player activating the owl.
 */
void ptb::owl::on_one_player( plee* p )
{
  m_activated = true;

  if ( m_has_azelnut )
    {
      if (m_action == action_idling)
        {
          say( m_already_have );
          start_model_action("speak");
          m_action = action_speaking;
        }
    }
  else if ( m_action == action_idling )
    {
      say( m_other_player );
      start_model_action("speak");
      m_action = action_wait_for_player;
    }
  else
    m_action = action_wait_for_player;
} // owl::on_one_player()

/*----------------------------------------------------------------------------*/
/**
 * \brief Activate with all players.
 * \param p1 The first player.
 * \param p2 The second player, in a two players game.
 */
void ptb::owl::on_all_players( plee* p1, plee* p2 )
{
  m_activated = true;

  if ( m_has_azelnut )
    {
      if (m_action == action_idling)
        {
          say( m_already_have );
          start_model_action("speak");
          m_action = action_speaking;
        }
    }
  else if ( m_action != action_start_level_scenario )
    {
      if ( level_variables::get_azelnut(get_level()) )
        {
          say( m_happy );
          start_model_action("speak");
          m_action = action_start_level_scenario;
          m_progress = &owl::progress_ask_azelnut;

          p1->set_marionette(true);
          if (p2!=NULL)
            p2->set_marionette(true);

          m_first_player = p1;
          m_second_player = p2;
        }
      else if ( m_action != action_speaking )
        {
          say( m_want_azelnut );
          start_model_action("speak");
          m_action = action_speaking;
        }
    }
} // owl::on_all_players()

/*----------------------------------------------------------------------------*/
/**
 * \brief Keep asking for the azelnut.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::owl::progress_ask_azelnut( bear::universe::time_type elapsed_time )
{
  bool ok(false);

  // move the player to face the owl
  if ( !get_rendering_attributes().is_mirrored() )
    {
      if ( m_first_player->get_left() < get_right() + 100)
        m_first_player->do_action(elapsed_time, player_action::move_right);
      else if (m_first_player->get_speed().x == 0)
        {
          ok = true;

          if ( !m_first_player->get_rendering_attributes().is_mirrored() )
            m_first_player->do_action(0.001, player_action::move_left);
        }
    }
  else if ( m_first_player->get_right() > get_left() - 100)
    m_first_player->do_action(elapsed_time, player_action::move_left);
  else if (m_first_player->get_speed().x == 0)
    {
      ok = true;

      if ( m_first_player->get_rendering_attributes().is_mirrored() )
        m_first_player->do_action(0.001, player_action::move_right);
    }

  if ( ok && has_finished_to_chat() )
    {
      start_model_action("idle");
      m_first_player->throw_azelnut();
      m_progress = NULL;
    }
} // owl::progress_ask_azelnut()

/*----------------------------------------------------------------------------*/
/**
 * \brief Wait until finised to talk.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::owl::progress_looking_azelnut
( bear::universe::time_type elapsed_time )
{
  if ( has_finished_to_chat() )
    {
      say( m_goodbye );
      start_model_action("bye");
      m_progress = &owl::progress_start_level;
    }
} // owl::progress_ask_azelnut()

/*----------------------------------------------------------------------------*/
/**
 * \brief Push the secret level, as soon as the owl stops talking.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::owl::progress_start_level
( bear::universe::time_type elapsed_time )
{
  if ( has_finished_to_chat() )
    {
      m_first_player->set_marionette(false);
      if (m_second_player != NULL)
        m_second_player->set_marionette(false);

      m_progress = NULL;
      start_model_action("idle");

      game_variables::set_next_level_name(m_level_name);
      bear::engine::game::get_instance().push_level( PTB_LOADING_LEVEL_NAME );
    }
} // owl::progress_start_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Collision with an other item.
 * \param that The other item of the collision.
 * \param info Some informations about the collision.
 */
void ptb::owl::collision
( bear::engine::base_item& that, bear::universe::collision_info& info )
{
  azelnut* a = dynamic_cast<azelnut*>(&that);

  if ( a != NULL )
    {
      say( m_receive_azelnut );
      start_model_action("look_azelnut");
      m_progress = &owl::progress_looking_azelnut;
      a->kill();
      m_has_azelnut = true;
    }
} // owl::collision()

/*----------------------------------------------------------------------------*/
/**
 * \brief Make the owl talking.
 * \param s The speeches among which we pick a sentence.
 */
void ptb::owl::say( const std::vector<std::string>& s )
{
  reset_speeches();
  add_speech( s[ (((double)rand() / RAND_MAX) * s.size()) ] );
  speak( get_level_globals() );
} // owl::say()
