//
// Context.cpp
//
// $Id: //poco/1.3/NetSSL_OpenSSL/src/Context.cpp#8 $
//
// Library: NetSSL_OpenSSL
// Package: SSLCore
// Module:  Context
//
// Copyright (c) 2006-2009, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#include "Poco/Net/Context.h"
#include "Poco/Net/SSLManager.h"
#include "Poco/Net/SSLException.h"
#include "Poco/Net/Utility.h"
#include "Poco/File.h"
#include "Poco/Path.h"
#include <openssl/bio.h>
#include <openssl/err.h>
#include <openssl/ssl.h>
#include <openssl/x509v3.h>


namespace Poco {
namespace Net {


Context::Context(
	Usage usage,
	const std::string& privateKeyFile, 
	const std::string& certificateFile,
	const std::string& caLocation, 
	VerificationMode verificationMode,
	int verificationDepth,
	bool loadDefaultCAs,
	const std::string& cypherList):
	_usage(usage),
	_mode(verificationMode),
	_pSSLContext(0)
{
	_pSSLContext = SSL_CTX_new(SSLv23_method());
	if (!_pSSLContext) 
	{
		unsigned long err = ERR_get_error();
		throw SSLException("Cannot create SSL_CTX object", ERR_error_string(err, 0));
	}
	SSL_CTX_set_default_passwd_cb(_pSSLContext, &SSLManager::privateKeyPasswdCallback);
	Utility::clearErrorStack();
	SSL_CTX_set_options(_pSSLContext, SSL_OP_ALL);
	
	int errCode = 0;
	if (!caLocation.empty())
	{
		Poco::File aFile(caLocation);
		if (aFile.isDirectory())
			errCode = SSL_CTX_load_verify_locations(_pSSLContext, 0, Poco::Path::transcode(caLocation).c_str());
		else
			errCode = SSL_CTX_load_verify_locations(_pSSLContext, Poco::Path::transcode(caLocation).c_str(), 0);
		if (errCode != 1)
		{
			std::string msg = Utility::getLastError();
			SSL_CTX_free(_pSSLContext);
			throw SSLContextException(std::string("Cannot load CA file/directory at ") + caLocation, msg);
		}
	}

	if (loadDefaultCAs)
	{
		errCode = SSL_CTX_set_default_verify_paths(_pSSLContext);
		if (errCode != 1)
		{
			std::string msg = Utility::getLastError();
			SSL_CTX_free(_pSSLContext);
			throw SSLContextException("Cannot load default CA certificates", msg);
		}
	}
	
	if (!privateKeyFile.empty())
	{
		errCode = SSL_CTX_use_PrivateKey_file(_pSSLContext, Poco::Path::transcode(privateKeyFile).c_str(), SSL_FILETYPE_PEM);
		if (errCode != 1)
		{
			std::string msg = Utility::getLastError();			
			SSL_CTX_free(_pSSLContext);
			throw SSLContextException(std::string("Error loading private key from file ") + privateKeyFile, msg);
		}
	}

	if (!certificateFile.empty())
	{
		errCode = SSL_CTX_use_certificate_chain_file(_pSSLContext, Poco::Path::transcode(certificateFile).c_str());
		if (errCode != 1)
		{
			std::string errMsg = Utility::getLastError();
			SSL_CTX_free(_pSSLContext);
			throw SSLContextException(std::string("Error loading certificate from file ") + privateKeyFile, errMsg);
		}
	}
	
	if (usage == SERVER_USE)
		SSL_CTX_set_verify(_pSSLContext, verificationMode, &SSLManager::verifyServerCallback);
	else
		SSL_CTX_set_verify(_pSSLContext, verificationMode, &SSLManager::verifyClientCallback);

	SSL_CTX_set_verify_depth(_pSSLContext, verificationDepth);
	SSL_CTX_set_mode(_pSSLContext, SSL_MODE_AUTO_RETRY);
	SSL_CTX_set_session_cache_mode(_pSSLContext, SSL_SESS_CACHE_OFF);
}


Context::~Context()
{
	SSL_CTX_free(_pSSLContext);
}


void Context::enableSessionCache(bool flag)
{
	SSL_CTX_set_session_cache_mode(_pSSLContext, flag ? SSL_SESS_CACHE_SERVER : SSL_SESS_CACHE_OFF);
}

	
bool Context::sessionCacheEnabled() const
{
	return SSL_CTX_get_session_cache_mode(_pSSLContext) != SSL_SESS_CACHE_OFF;
}


} } // namespace Poco::Net
