########################################################################
#
# File Name:            FtElements.py
#
# Documentation:        http://docs.4suite.org/4XSLT/FtElements.py.html
#
"""
Fourthought proprietary extension elements
WWW: http://4suite.org/4XSLT        e-mail: support@4suite.org

Copyright (c) 1999-2001 Fourthought Inc, USA.   All Rights Reserved.
See  http://4suite.org/COPYRIGHT  for license and copyright information
"""

import string
import xml.dom.ext
from xml.xslt import XsltElement, XsltException, Error
from xml.xslt import XSL_NAMESPACE
from xml.xslt import OutputParameters, TextWriter
from xml.xpath import Util, Conversions
from xml.xpath import FT_EXT_NAMESPACE, FT_OLD_EXT_NAMESPACE
from xml.xslt import ApplyImportsElement
from xml.xslt import ApplyTemplatesElement

class FtApplyTemplates(ApplyTemplatesElement.ApplyTemplatesElement):
    def setup(self):
        ApplyTemplatesElement.ApplyTemplatesElement.setup(self)

        #Overwrite the mode
        mode_attr = self.getAttributeNS('', 'mode')
        if mode_attr != '':
            self.__dict__['_mode'] = self.parseExpression(mode_attr)

    def _instantiateMode(self,context):
        rt = self._mode.evaluate(context)

        split_name = Util.ExpandQName(
            rt,
            namespaces=self._nss
            )
        return split_name


class WriteFileElement(XsltElement):
    def __init__(self, doc, uri=FT_EXT_NAMESPACE, localName='write-file',
                 prefix='ft', baseUri=''):
        XsltElement.__init__(self, doc, uri, localName, prefix, baseUri)

    def setup(self):
        self.__dict__['_nss'] = xml.dom.ext.GetAllNs(self)
        self.__dict__['_name'] = self.parseAVT(self.getAttributeNS('', 'name'))
        self.__dict__['_overwrite'] = self.parseAVT(self.getAttributeNS('', 'overwrite'))
        out = OutputParameters()
        for child in self.childNodes:
            if (child.namespaceURI, child.localName) == (FT_EXT_NAMESPACE, 'output'):
                out.parse(child, self._nss)
        self._outputParams = out
        return

    def instantiate(self, context, processor):
        origState = context.copy()
        context.processorNss = self._nss
        
        name = self._name.evaluate(context)
        overwrite = self._overwrite.evaluate(context)
        if overwrite == 'yes':
            f = open(name, 'w')
        else:
            f = open(name, 'a')
        processor.addHandler(self._outputParams, f)
        for child in self.childNodes:
            context = child.instantiate(context, processor)[0]
        processor.removeHandler()
        f.close()

        context.set(origState)
        return (context,)

    def __getinitargs__(self):
        return (None, self.namespaceURI, self.localName, self.prefix,
                self.baseUri)

    def __getstate__(self):
         base_state = XsltElement.__getstate__(self)
         new_state = (base_state, self._nss, self._name,self._overwrite,self._outputParams)
         return new_state

    def __setstate__(self, state):
        XsltElement.__setstate__(self, state[0])
        self._nss = state[1]
        self._name = state[2]
        self._overwrite = state[3]
        self._outputParams = state[4]
        return


class FtOutputElement(XsltElement):
    def __init__(self, doc, uri=FT_EXT_NAMESPACE, localName='output', prefix='ft', baseUri=''):
        XsltElement.__init__(self, doc, uri, localName, prefix, baseUri)

    def setup(self):
        return

    def __getinitargs__(self):
        return (None, self.namespaceURI, self.localName, self.prefix,
                self.baseUri)


class MessageOutputElement(XsltElement):
    def __init__(self, doc, uri=FT_EXT_NAMESPACE, localName='message-output', prefix='ft', baseUri=''):
        XsltElement.__init__(self, doc, uri, localName, prefix, baseUri)

    def setup(self):
        #FIXME: disable -> silent
        self.__dict__['_nss'] = xml.dom.ext.GetAllNs(self)
        if self.getAttributeNS('', 'file'):
            self.__dict__['_file'] = self.parseAVT(self.getAttributeNS('', 'file'))
        else:
            self.__dict__['_file'] = None
        if self.getAttributeNS('', 'disable'):
            self.__dict__['_disable'] = self.parseAVT(self.getAttributeNS('', 'disable'))
        else:
            self.__dict__['_disable'] = None
        if self.getAttributeNS('', 'overwrite'):
            self.__dict__['_overwrite'] = self.parseAVT(self.getAttributeNS('', 'overwrite'))
        else:
            self.__dict__['_overwrite'] = None
        return

    def instantiate(self, context, processor):
        if self._file:
            processor.setMessageFile(self._file.evaluate(context))
        if self._disabled:
            processor._messagesEnabled = self._disabled.evaluate(context) == yes
        if self._overwrite == 'yes':
            f = open(name, 'w')
        else:
            f = open(name, 'a')
        return (context,)

    def __getinitargs__(self):
        return (None, self.namespaceURI, self.localName, self.prefix,
                self.baseUri)

    def __getstate__(self):
         base_state = XsltElement.__getstate__(self)
         new_state = (base_state, self._nss, self._name,self._overwrite,self._outputParams)
         return new_state

    def __setstate__(self, state):
        XsltElement.__setstate__(self, state[0])
        self._nss = state[1]
        self._file = state[2]
        self._silent = state[3]
        return

class AssignElement(XsltElement):
    def __init__(self, doc, uri=FT_EXT_NAMESPACE, localName='assign', prefix='ft', baseUri=''):
        XsltElement.__init__(self, doc, uri, localName, prefix, baseUri)

    def setup(self):
        self._nss = xml.dom.ext.GetAllNs(self)
        name_attr = self.getAttributeNS('', 'name')
        self._name = Util.ExpandQName(name_attr, namespaces=self._nss)
        self._select = self.getAttributeNS('', 'select')
        if self._select:
            self._expr = self.parseExpression(self._select)
        else:
            self._expr = None
        return

    def instantiate(self, context, processor):
        origState = context.copy()
        context.setNamespaces(self._nss)
        context.currentInstruction = self

        if self._select:
            result = self._expr.evaluate(context)
        else:
            processor.pushResult()
            try:
                for child in self.childNodes:
                    context = child.instantiate(context, processor)[0]
            finally:
                result = processor.popResult()
            context.rtfs.append(result)

        context.set(origState)
        context.varBindings[self._name] = result
        context.stylesheet._primedContext.varBindings[self._name] = result
        return (context,)

    def __getinitargs__(self):
        return (None, self.namespaceURI, self.localName, self.prefix,
                self.baseUri)

    def __getstate__(self):
         base_state = XsltElement.__getstate__(self)
         new_state = (base_state, self._nss, self._name, self._select)
         return new_state

    def __setstate__(self, state):
        XsltElement.__setstate__(self, state[0])
        self._nss = state[1]
        self._select = state[2]
        return

class DumpKeysElement(XsltElement):
    def __init__(self, doc, uri=FT_EXT_NAMESPACE, localName='dump-keys', prefix='ft', baseUri=''):
        XsltElement.__init__(self, doc, uri, localName, prefix, baseUri)
        return
    
    def instantiate(self, context, processor):
        doc = context.node.ownerDocument or context.node
        msg = repr(processor.keys[doc])
        processor.xslMessage(msg)
        return (context,)


class ReplaceElement(XsltElement):
    def __init__(self, doc, uri=FT_EXT_NAMESPACE, localName='replace',
                 prefix='ft', baseUri=''):
        XsltElement.__init__(self, doc, uri, localName, prefix, baseUri)

    def setup(self):
        self._nss = xml.dom.ext.GetAllNs(self)
        self._string = self.getAttributeNS('', 'string')
        if self._string:
            self._string = self.parseExpression(self._string)
        else:
            self._string = None
        self._substring = self.getAttributeNS('', 'substring')
        self._substring = self.parseExpression(self._substring)
        return

    def instantiate(self, context, processor):
        origState = context.copy()
        context.processorNss = self._nss

        val = self._string and self._string.evaluate(context) or context.node
        text = Conversions.StringValue(val)

        substring = self._substring.evaluate(context)
        for s in string.splitfields(text, substring):
            processor.writers[-1].text(s)
            for child in self.childNodes:
                context = child.instantiate(context, processor)[0]

        context.set(origState)
        return (context,)

    def __getinitargs__(self):
        return (None, self.namespaceURI, self.localName, self.prefix,
                self.baseUri)

    def __getstate__(self):
         base_state = XsltElement.__getstate__(self)
         new_state = (base_state, self._nss, self._string, self._substring)
         return new_state

    def __setstate__(self, state):
        XsltElement.__setstate__(self, state[0])
        self._nss = state[1]
        self._string = state[2]
        self._substring = state[3]
        return


ExtElements = {
    (FT_EXT_NAMESPACE, 'apply-templates'): FtApplyTemplates,
    (FT_EXT_NAMESPACE, 'output'): FtOutputElement,
    (FT_EXT_NAMESPACE, 'write-file'): WriteFileElement,
    (FT_EXT_NAMESPACE, 'message-output'): MessageOutputElement,
    (FT_EXT_NAMESPACE, 'dump-keys'): DumpKeysElement,
    (FT_EXT_NAMESPACE, 'assign'): AssignElement,
    (FT_EXT_NAMESPACE, 'replace'): ReplaceElement,

    (FT_OLD_EXT_NAMESPACE, 'apply-templates'): FtApplyTemplates,
    (FT_OLD_EXT_NAMESPACE, 'output'): FtOutputElement,
    (FT_OLD_EXT_NAMESPACE, 'write-file'): WriteFileElement,
    (FT_OLD_EXT_NAMESPACE, 'message-output'): MessageOutputElement,
    (FT_OLD_EXT_NAMESPACE, 'dump-keys'): DumpKeysElement,
    (FT_OLD_EXT_NAMESPACE, 'assign'): AssignElement,
    }

