########################################################################
#
# File Name:            RtfWriter.py
#
# Documentation:        http://docs.4suite.org/4XSLT/RtfWriter.py.html
#

"""
A special, simple writer for capturing result-tree fragments
WWW: http://4suite.org/4XSLT        e-mail: support@4suite.org

Copyright (c) 1999-2001 Fourthought Inc., USA.   All Rights Reserved.
See  http://4suite.org/COPYRIGHT  for license and copyright information
"""

import string, os
import NullWriter
from xml.xslt import XSL_NAMESPACE
from xml.xpath import Util
from xml.dom.ext import SplitQName
from xml.dom import XMLNS_NAMESPACE, Node
from xml.xslt import OutputParameters

class RtfWriter(NullWriter.NullWriter):
    def __init__(self, outputParams=None):
        self._root = Document()
        self._curr_node = self._root
        self._outputParams = outputParams or OutputParameters()
        return

    def getResult(self):
        return self._root

    def startElement(self, name, namespace='', extraNss=None):
        extraNss = extraNss or {}
        prefix, localName = SplitQName(name)
        element = Element(name, namespace, prefix, localName)
        self._curr_node.appendChild(element)
        for (prefix, uri) in extraNss.items():
            if prefix:
                nodeName = 'xmlns:'+prefix
            else:
                nodeName = 'xmlns'
            attr = Attr(nodeName, XMLNS_NAMESPACE, prefix, 'xmlns', uri)
            element.setAttributeNodeNS(attr)
        self._curr_node = element
        return

    def endElement(self, name):
        parent = self._curr_node.parentNode
        parent.stringValue = parent.stringValue + self._curr_node.stringValue
        self._curr_node = parent
        return

    def attribute(self, name, value, namespace=''):
        if self._curr_node.attributes is not None:
            # Only add to elements
            prefix, localName = SplitQName(name)
            attr = Attr(name, namespace, prefix, localName, value)
            self._curr_node.setAttributeNodeNS(attr)
        return

    def text(self, data, escapeOutput=1, asis=0):
        self._curr_node.appendChild(Text(data))
        return

    def processingInstruction(self, target, data):
        self._curr_node.appendChild(ProcessingInstruction(target, data))
        return

    def comment(self, data):
        self._curr_node.appendChild(Comment(data))
        return

# The Result Tree DOM

class RTFNode(Node):
    def __init__(self, nodeType, nodeName, namespaceURI=None, prefix=None, localName=None):
        self.childNodes = []
        self.nodeName = nodeName
        self.nodeValue = None
        self.parentNode = None
        self.firstChild = None
        self.lastChild = None
        self.previousSibling = None
        self.nextSibling = None
        self.attributes = None
        self.ownerDocument = None
        self.namespaceURI = namespaceURI
        self.prefix = prefix
        self.localName = localName
        self.nodeType = nodeType
        return

    def appendChild(self, newChild):
        newChild.parentNode = self
        newChild.ownerDocument = self.ownerDocument or self
        if not self.firstChild:
            self.firstChild = newChild
        else:
            self.lastChild.nextSibling = newChild
            newChild.previousSibling = self.lastChild
        self.lastChild = newChild
        self.childNodes.append(newChild)
        self.stringValue = self.stringValue + newChild.stringValue
        return newChild

class Document(RTFNode):
    def __init__(self):
        RTFNode.__init__(self, Node.DOCUMENT_NODE, '#document')
        self.stringValue = ''

    def __repr__(self):
        return '<RTFDocument at %x>' % id(self)

class Element(RTFNode):
    def __init__(self, nodeName, namespaceURI, prefix, localName):
        RTFNode.__init__(self, Node.ELEMENT_NODE, nodeName, namespaceURI,
                         prefix,localName)
        self.attributes = {}
        self.stringValue = ''

    def setAttributeNodeNS(self, attr):
        attr.ownerDocument = self.ownerDocument
        attr.ownerElement = self
        if attr.localName == 'xmlns':
            self.attributes[(XMLNS_NAMESPACE, attr.prefix)] = attr
        else:
            self.attributes[(attr.namespaceURI, attr.localName)] = attr
        return
    
    def __repr__(self):
        return "<RTFElement at %x: %s with %d attributes and %d children>" % (
            id(self),
            self.nodeName,
            len(self.attributes),
            len(self.childNodes)
            )

class Attr(RTFNode):
    def __init__(self, nodeName, namespaceURI, prefix, localName, value):
        RTFNode.__init__(self, Node.ATTRIBUTE_NODE, nodeName, namespaceURI,
                      prefix,localName)
        self.ownerElement = None
        self.nodeValue = self.value = value
        self.name = nodeName
        self.stringValue = ''
        self.appendChild(Text(value))

    def __repr__(self):
         return "<RTFAttribute at %x: %s=%s>" % (
             id(self),
             self.name,
             repr(self.value)
             )

class Text(RTFNode):
    def __init__(self, data):
        RTFNode.__init__(self, Node.TEXT_NODE, '#text')
        self.stringValue = self.nodeValue = self.data = data

    def __repr__(self):
        # Trim to a managable size
        if len(self.data) > 20:
            data = self.data[:20] + '...'
        else:
            data = self.data
        return "<RTFText at %x: %s>" % (id(self), repr(data))

class ProcessingInstruction(RTFNode):
    def __init__(self, target, data):
        RTFNode.__init__(self, Node.PROCESSING_INSTRUCTION_NODE, target)
        self.target = target
        self.stringValue = self.nodeValue = self.data = data

    def __repr__(self):
        if len(self.data) > 20:
            data = self.data[:20] + '...'
        else:
            data = self.data
        return "<RTFProcessingInstruction at %x: %s %s>" % (
            id(self), self.target, data)

class Comment(RTFNode):
    def __init__(self, data):
        RTFNode.__init__(self, Node.COMMENT_NODE, '#comment')
        self.stringValue = self.nodeValue = self.data = data

    def __repr__(self):
        return "<RTFComment Node at %x: %s>" % (id(self), repr(self.data))

def ReleaseRtf(rtf):
    for child in rtf.childNodes:
        ReleaseRtf(child)
    rtf.childNodes = []
    if rtf.nodeType == Node.ELEMENT_NODE:
        for attr in rtf.attributes.values():
            ReleaseRtf(attr)
            attr.ownerElement = None
        rtf.attributes = {}
    rtf.parentNode = None
    rtf.firstChild = None
    rtf.lastChild = None
    rtf.previousSibling = None
    rtf.nextSibling = None
    rtf.ownerDocument = None
    return
