/***************************************************************************
                          qcupsprinterconfigdialog.cpp  -  description
                             -------------------
    begin                : Tue Jul 4 2000
    copyright            : (C) 2000 by Michael Goffioul
    email                : gofioul@emic.ucl.ac.be
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "qcupsprinterconfigdialog.h"

#include "cupshelper.h"
#include "qgeneralconfig.h"
#include "qtextconfig.h"
#include "qimageconfig.h"
#include "qhpgl2config.h"
#include "qppdconfigpage.h"
#include "ppd-util.h"

#include <qstring.h>
#include <qmessagebox.h>
#include <qfileinfo.h>

extern void showOptions(int n, cups_option_t *opts);

QCupsPrinterConfigDialog::QCupsPrinterConfigDialog(int, QWidget *parent, const char *name , bool useSave) : QTabDialog(parent,name,true)
{
	dest_ = 0;
	request_ = 0;
	ppd_ = 0;
	num_options_ = 0;
	options_ = 0;

	generalPage_ = new QGeneralConfig(this,"GeneralPage");
	addTab(generalPage_,tr("General"));

	imagePage_ = new QImageConfig(this,"ImagePage");
	addTab(imagePage_,tr("Image"));

	textPage_ = new QTextConfig(this,"TextPage");
	addTab(textPage_,tr("Text"));

	hpgl2Page_ = new QHPGL2Config(this,"HPGLPage");
	addTab(hpgl2Page_,tr("HP-GL/2"));

	advancedPage_ = new QPPDConfigPage(this,"PPDPage");
	addTab(advancedPage_,tr("Advanced"));

	connect(this,SIGNAL(selected(const QString&)),SLOT(tabSelected(const QString&)));

	currentPage_ = 0;

	setCancelButton(tr("Cancel"));
	setOkButton(tr("OK"));
	if (useSave) setDefaultButton(tr("Save"));

	resize(100,100);
}

QCupsPrinterConfigDialog::~QCupsPrinterConfigDialog()
{
	closePpd();
	closeRequest();
}

bool QCupsPrinterConfigDialog::initialize(cups_dest_t *dest)
{
	if (dest)
	{
		dest_ = dest;
		// copy options
		cupsFreeOptions(num_options_,options_);
		num_options_ = 0;
		options_ = 0;
		for (int i=0;i<dest->num_options;i++)
		{
			cups_option_t	*opt = dest->options+i;
			num_options_ = cupsAddOption(opt->name,opt->value,num_options_,&options_);
		}

		// set caption
		QString	str = tr("Configuration of %1").arg(dest->name);
		setCaption(str.data());

		// free some variables
		closeRequest();
		closePpd();

		// getting more informations from server about the printer
		request_ = CupsHelper::get()->newIppRequest();
		str.sprintf("ipp://%s:%d/printers/%s",CupsHelper::get()->host().data(),CupsHelper::get()->port(),dest_->name);
		ippAddString(request_,IPP_TAG_OPERATION,IPP_TAG_URI,"printer-uri",NULL,str.data());
		request_->request.op.operation_id = IPP_GET_PRINTER_ATTRIBUTES;
		str.sprintf("/printers/%s",dest_->name);
		request_ = CupsHelper::get()->processRequest(request_,str.data());
		if (!request_ || request_->state == IPP_ERROR)
		{
			closeRequest();
			return false;
		}

		// getting PPD file
		ppdFileName_ = cupsGetPPD(dest->name);
		if (!ppdFileName_.isEmpty())
		{
			// first check file size, if zero do nothing (workarounf for CUPS bug)
			QFileInfo	fi(ppdFileName_.data());
			if (fi.size() != 0) ppd_ = globalPpdOpenFile(ppdFileName_.data());
		}
		if (ppd_)
		{
			// mark all default options
			globalPpdMarkDefaults(ppd_);

			// construct PPD option tree
			globalCupsMarkOptions(ppd_,num_options_,options_);
			advancedPage_->initOptions(num_options_,options_,ppd_, request_);
		}
		else 	// disable "Advanced page"
			setTabEnabled(advancedPage_,false);

		return true;
	}
	else
		return false;
}

void QCupsPrinterConfigDialog::closePpd()
{
	if (ppd_)
	{
		globalPpdClose(ppd_);
		ppd_ = 0;
		unlink(ppdFileName_.data());
		ppdFileName_ = "";
	}
}

void QCupsPrinterConfigDialog::closeRequest()
{
	if (request_)
	{
		ippDelete(request_);
		request_ = 0;
	}
}

void QCupsPrinterConfigDialog::getCupsOptions(int& num_options, cups_option_t **options)
{
	// first get current tab's options and update all tabs (for synchronization)
	if (currentPage_) currentPage_->getCupsOptions(num_options_,&options_,true,request_);
	if (ppd_) globalCupsMarkOptions(ppd_,num_options_,options_);
	generalPage_->initOptions(num_options_,options_,ppd_, request_);
	imagePage_->initOptions(num_options_,options_,ppd_, request_);
	textPage_->initOptions(num_options_,options_,ppd_, request_);
	hpgl2Page_->initOptions(num_options_,options_,ppd_, request_);
	if (ppd_) advancedPage_->initOptions(num_options_,options_,ppd_, request_);

	// finally retrieve relevant options from each tab
	generalPage_->getCupsOptions(num_options,options,false,request_);
	imagePage_->getCupsOptions(num_options,options,false,request_);
	textPage_->getCupsOptions(num_options,options,false,request_);
	hpgl2Page_->getCupsOptions(num_options,options,false,request_);
	if (ppd_) advancedPage_->getCupsOptions(num_options,options,false,request_);
}

void QCupsPrinterConfigDialog::tabSelected(const QString& name)
{
	if (currentPage_) currentPage_->getCupsOptions(num_options_,&options_,true,request_);
	if (name == tr("General")) currentPage_ = generalPage_;
	else if (name == tr("Image")) currentPage_ = imagePage_;
	else if (name == tr("Text")) currentPage_ = textPage_;
	else if (name == tr("HP-GL/2")) currentPage_ = hpgl2Page_;
	else if (name == tr("Advanced")) currentPage_ = advancedPage_;
	else currentPage_ = 0;
	if (ppd_) globalCupsMarkOptions(ppd_,num_options_,options_);
	if (currentPage_) currentPage_->initOptions(num_options_,options_,ppd_, request_);
}

bool QCupsPrinterConfigDialog::hasConflict()
{
	bool	value(false);
	if (ppd_)
	{
		if (currentPage_) currentPage_->getCupsOptions(num_options_,&options_,true,request_);
		globalCupsMarkOptions(ppd_,num_options_,options_);
		value = (ppdConflicts(ppd_->ppd) > 0);
		if (value)
		{
			QString	msg = ppdConflictErrorMsg(ppd_->ppd);
			msg.append(tr("\n(Check the \"Advanced\" tab)"));
			QMessageBox::critical(this,tr("Configuration error"),msg.data(),QMessageBox::Ok|QMessageBox::Default,0);
		}
	}
	return value;
}


void QCupsPrinterConfigDialog::done(int result)
{
	// check first if conflict
	if (result == Accepted && hasConflict()) return;
	QDialog::done(result);
}
