/* provides an interface to the module tree
 * (this class is implemented as a singleton) */

#ifndef MODULE_TREE_H
#define MODULE_TREE_H

#include "dataset.h"
#include "base_node.h"
#include "path.h"

#include <string>
#include <vector>
using namespace std;

class Module;

class ModuleTreeListener
{
public:
        virtual ~ModuleTreeListener() {}
        
        virtual void return_message(const DataSet& msg, const Path& path) = 0;
};

class ModuleError
{
public:
        ModuleError(const string& d) : description(d) {}        
        string description;
};

class ModuleTree
{
public:
        ~ModuleTree();
        
        // tree access
        string get_type(const Path& target_module) throw (ModuleError);
        DataSet get_value(const Path& target_module, const string& varname) throw (ModuleError);
        void set_value(const Path& target_module, const string& varname,
                       const DataSet& value) throw (ModuleError);
        void set_value(const Path& target_module, const string& varname,
                       const string& value) throw (ModuleError);
        void create_child(const Path& target_module, const string& child_type) throw (ModuleError);
        void delete_child(const Path& target_module, int child_num) throw (ModuleError);
        int child_count(const Path& target_module) throw (ModuleError);
        void post_startup(const Path& target_module) throw (ModuleError);
        void pre_shutdown(const Path& target_module) throw (ModuleError);
        void overlay_defaults(const Path& target_module) throw (ModuleError);
        const DataSetMap *module_info(const Path& target_module) throw (ModuleError);
        void post_startup_recursive(const Path& path);
        void pre_shutdown_recursive(const Path& path);

        void init();
        void shutdown();
        
        void service();

        void delete_recursive(const Path& path);
        string issue_string_command(const string& cmdstring) throw (ModuleError);

        void register_listener(ModuleTreeListener *l);
        void unregister_listener(ModuleTreeListener *l);
        void return_message(const DataSet& msg, const Path& path);
private:
        void load(const string& config_file);
        void recursive_command(const Path& path, const string& command,
                               const string& parameter);
        string issue_command(const Path& target_module, const string& command,
                             const string& parameter) throw (ModuleError);

public:
        static ModuleTree* instance() {
                if (!c_instance) {
                        rpdbgmsg("Creating ModuleTree");
                        c_instance = new ModuleTree();
                }
                return c_instance;
        }

        static void destroy_instance() {
                rpdbgmsg("Destroying ModuleTree");
                if (c_instance)
                        delete c_instance;
                c_instance = 0;
        }
        
private:
        // class should be accessed via instance()
        ModuleTree() {}
        
        static ModuleTree* c_instance;

        vector<ModuleTreeListener *> retmsg_listeners;
        
private:
        BaseNode base_module;
};

#endif
