/*
 * GLX Hardware Device Driver for SiS 6326
 * Copyright (C) 2000 Jim Duchek
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Based on Mach64 driver: mach64dd.c
 *
 *    Jim Duchek <jim@linuxpimps.com>
 */

#include "glx_config.h"
#include "glx_symbols.h"
#include "pixmapstr.h"
#include "pixmap.h"
#include "sis6326glx.h"
#include "hw_buffer.h"


/***************************************
 * Mesa's Driver Functions
 ***************************************/


void sis6326DDExtensionsInit( GLcontext *ctx )
{
	/* disable the extensions we can't hardware accelerate.
	 * Note that when a client is running indirectly, libGL
	 * determines the extension string (none supported), not this.
	 */

	/* we can support paletted textures later if we really care */
	gl_extensions_disable( ctx, "GL_EXT_shared_texture_palette" );
	gl_extensions_disable( ctx, "GL_EXT_paletted_texture" );

	gl_extensions_disable( ctx, "GL_EXT_point_parameters" );
	gl_extensions_disable( ctx, "ARB_imaging" );
	gl_extensions_disable( ctx, "GL_EXT_blend_minmax" );
	gl_extensions_disable( ctx, "GL_EXT_blend_logic_op" );
	gl_extensions_disable( ctx, "GL_EXT_blend_subtract" );
	gl_extensions_disable( ctx, "GL_EXT_blend_color" );
	gl_extensions_disable( ctx, "GL_INGR_blend_func_separate" );
	gl_extensions_disable( ctx, "GL_EXT_texture3D" );
	gl_extensions_disable( ctx, "GL_MESA_window_pos" );
	gl_extensions_disable( ctx, "GL_MESA_resize_buffers" );
	gl_extensions_disable( ctx, "GL_SGIS_texture_edge_clamp" );
	gl_extensions_disable( ctx, "GL_MESA_window_pos" );

}


const GLubyte *sis6326DDGetString( GLcontext *ctx, GLenum name )
{
	switch ( name )
	{
	case GL_VENDOR:
		return "Utah GLX";
	case GL_RENDERER:
		return "GLX-SiS 6326";
	default:
		return 0;
	}
}


static GLint sis6326DDGetParameteri( const GLcontext *ctx, GLint param )
{
	switch ( param )
	{
	case DD_HAVE_HARDWARE_FOG:
		return 1;
	default:
		hwError( "sis6326GetParameteri(): unknown parameter!\n" );
		return 0; /* Should I really return 0? */
	}
}


/*
 * We allways have a single back buffer, but
 * if we are in "front buffer" mode, we will automatically
 * generate flush events
 */
static GLboolean sis6326DDSetBuffer( GLcontext *ctx, GLenum mode )
{
	switch ( mode ) {

	case GL_FRONT:
	case GL_FRONT_LEFT:
	case GL_FRONT_RIGHT:
	 	sis6326Ctx->frontBufferRendering = 1;
	        return GL_TRUE;

	case GL_BACK:
	case GL_BACK_LEFT:
	case GL_BACK_RIGHT:
	 	sis6326Ctx->frontBufferRendering = 0;
	        return GL_TRUE;

	default:
		return GL_FALSE;
	}
}

static void FlushFrontBufferRendering( void )
{
	/* if we haven't drawn anything since the last swapbuffers, no need to swap */
	if ( sis6326glx.c_triangles || sis6326glx.c_clears || sis6326glx.c_drawWaits ) {
		/* We need this extra entry point to the swapbuffers, because the normal
		   point has to flush vertex buffers to make sure everything is drawn,
		   but that recursively enters renderFinish */
		sis6326GLXSwapBuffersWithoutFlush( sis6326DB->xsBuf );
	}
}

static void sis6326DDRenderFinish( GLcontext *ctx )
{
	/* if we are front buffer rendering, fake a swap to screen to get everything
	   we have drawn visible. */
	if ( sis6326Ctx->frontBufferRendering ) {
		FlushFrontBufferRendering();
	}
}

static void sis6326DDFlush( GLcontext *ctx )
{
	/* with only two dma buffers, explicit flushing of a double buffered
	 * context is almost always a bad idea.  Because a swapbuffers does
	 * generate a flush, we are still compliant with the spec by ignoring
	 * it here.
	 */
	if ( sis6326Ctx->frontBufferRendering ) {
		FlushFrontBufferRendering();
		return;
	}
}

static void sis6326DDFinish( GLcontext *ctx )
{
	if ( sis6326Ctx->frontBufferRendering ) {
		glFlush();
		return;
	}
	/*
	 * Enforcing OpenGL finish semantics can cause
	 * a large performance overhead on applications that use it.
	 * Our two buffer dma system can never get more than a single
	 * frame behind the application, so the only case glFinish()
	 * is truly needed is for benchmarking.
	 */
	sis6326Finish();
}

/*
 * Null primitive functions for performance testing:
 */
static void sis6326_null_quad( GLcontext *ctx, GLuint v0, GLuint v1, GLuint v2, GLuint v3, GLuint pv ) { }
static void sis6326_null_triangle( GLcontext *ctx, GLuint v0, GLuint v1, GLuint v2, GLuint pv ) { }
static void sis6326_null_line( GLcontext *ctx, GLuint v0, GLuint v1, GLuint pv ) { }
static void sis6326_null_points( GLcontext *ctx, GLuint first, GLuint last ) { }

/*
 * sis6326DDUpdateStateFunctions
 * If the current state allows hardware rendering, set
 * ctx->Driver.TriangleFunc to our accelerated triangle function
 */
void sis6326DDUpdateStateFunctions( GLcontext *ctx )
{
	if ( !sis6326DB ) {
		return;
	}

	/* no acceleration if our back buffer isn't on card */
	if ( !sis6326DB->backBufferBlock ) {
		return;
	}

	/* no acceleration if our back buffer isn't on card */
	if ( ctx->Depth.Test && !sis6326DB->depthBufferBlock ) {
		return;
	}

	/* decide if we want a software fallback */
	if ( !sis6326glx.noFallback ) {
		/* no acceleration if any stencil operations are enabled */
		if ( ctx->Stencil.Enabled ) {
			hwMsg(0, "Crap, stencils!\n");
			return;
		}

		/* no acceleration if stipple is on */
		if ( ctx->Polygon.StippleFlag ) {
			return;
		}

		/* no acceleration if smoothing is on */
		if ( ctx->Polygon.SmoothFlag ) {
			return;
		}

		/* no acceleration if 3D texturing is on */
		if ( ctx->Texture.Enabled & (TEXTURE0_3D | TEXTURE1_3D) ) {
			return;
		}
	}

	/* use null primitive functions for performance testing */
	if ( sis6326glx.nullprims ) {
		ctx->Driver.QuadFunc = sis6326_null_quad;
		ctx->Driver.TriangleFunc = sis6326_null_triangle;
		ctx->Driver.LineFunc = sis6326_null_line;
		ctx->Driver.PointsFunc = sis6326_null_points;
		return;
	}

	/* use our triangle function */
	ctx->Driver.QuadFunc = sis6326Quad;
	ctx->Driver.TriangleFunc = sis6326Triangle;
	ctx->Driver.LineFunc = sis6326Line;
//	ctx->Driver.PointsFunc = sis6326Points;

	/* we do all of our actual state updates in BeginRender */
}

/*
 * Return the size of the current color buffer, checking to see if the
 * window has been resized.
 * There isn't an explicit "size has changed" entry point, so this serves
 * double duty.
 */
static void sis6326DDGetBufferSize( GLcontext *ctx, GLuint *width, GLuint *height )
{
	XSMesaContext		xsmesa;
	XSMesaBuffer		b;
	DrawablePtr		window;
	unsigned int		winwidth, winheight;

	xsmesa = (XSMesaContext) ctx->DriverCtx;
	b = xsmesa->xsm_buffer;
	window = b->frontbuffer;

	winwidth = window->width;
	winheight = window->height;
	*width = winwidth;
	*height = winheight;

	/* see if it has changed size */
	if ( winwidth != b->width || winheight != b->height ) {
	        b->width = winwidth;
	        b->height = winheight;

	        /* Allocate new back buffer :
		 * (also deallocate the old backimage, if any) */
	        b->backimage = sis6326GLXCreateImage( (WindowPtr)b->frontbuffer,
						     b->xsm_visual->pVisual->nplanes,
						     b->width,
						     b->height,
						     b->backimage );

		/* make it current */
		XSMesa = NULL;	/* force it to reset things */
		sis6326GLXBindBuffer( xsmesa, b );
	}
}

static void sis6326DDClearColor( GLcontext *ctx, GLubyte red, GLubyte green, GLubyte blue, GLubyte alpha )
{
	/* we just look in the context when we clear, but mesa
	 * requires this function to exist
	 */
}

static void sis6326DDAllocDepthBuffer( GLcontext *ctx )
{
	/* we always allocate the depth buffer with the color buffer */
}



void sis6326_setup_DD_pointers( GLcontext *ctx )
{
	xsmesa_setup_DD_pointers( ctx );

	ctx->Driver.GetBufferSize = sis6326DDGetBufferSize;
	ctx->Driver.AllocDepthBuffer = sis6326DDAllocDepthBuffer;
	ctx->Driver.SetBuffer = sis6326DDSetBuffer;
	ctx->Driver.GetString = sis6326DDGetString;
	ctx->Driver.UpdateState = sis6326DDUpdateState;
	ctx->Driver.RenderFinish = sis6326DDRenderFinish;
	ctx->Driver.Clear = sis6326Clear;
	ctx->Driver.GetParameteri = sis6326DDGetParameteri;
	ctx->Driver.Flush = sis6326DDFlush;
	ctx->Driver.Finish = sis6326DDFinish;
	ctx->Driver.UpdateState = sis6326DDUpdateStateFunctions;
	ctx->Driver.RenderStart = sis6326DDUpdateState;
	ctx->Driver.TexImage = sis6326TexImage;
	ctx->Driver.TexSubImage = sis6326TexSubImage;
	ctx->Driver.DeleteTexture = sis6326DeleteTexture;
	ctx->Driver.IsTextureResident = sis6326IsTextureResident;
	ctx->Driver.ClearColor = sis6326DDClearColor;


	ctx->Driver.TriangleCaps = 0;		/* no special handling yet */
	//  ( DD_TRI_LIGHT_TWOSIDE |
	//    DD_TRI_OFFSET ); /* FIXME: not correct! */

	SetDriverBufferFunctions( ctx, sis6326Finish, 
		sis6326DB->backBuffer,  sis6326DB->pitch, sis6326DB->height, CB_16BIT,
		sis6326DB->depthBuffer, sis6326DB->pitch, sis6326DB->height, DB_16BIT);
		
	sis6326DDUpdateState ( ctx );
}


/*
 * Local Variables:
 * mode: c
 * tab-width: 8
 * c-basic-offset: 8
 * End:
 */
