
/*
 * GLX Server Extension
 * Copyright (C) 1998, 1999 Terence Ripperda (ripperda@sgi.com)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * TERENCE RIPPERDA, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, 
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 * DEALINGS IN THE SOFTWARE.
 */

#include "glx_varray.h"
#include <GL/gl.h>
#include <stdio.h>
#include <stdlib.h>
#include "glxcommon.h"
#include "extnsionst.h"
#include "dixstruct.h"
#include "scrnintstr.h"
#include "glx_log.h"
#include "glxrender.h"
#include "glx_symbols.h"

#define GLX_LOG_FLOAT_ARRAY(comps, verts, stride, start)	\
   {								\
      int i, j;							\
      unsigned char *ptr = (unsigned char *) start;		\
      for (i = 0; i < verts; i++) {				\
         for (j = 0; j < comps; j++) {				\
            log_print(LOG_DATA, "%f ", *( ((float *) ptr) + j ) );	\
         }							\
         log_print(LOG_DATA, "\n");				\
         ptr += stride;						\
      }								\
   }

#define GLX_LOG_ARRAY_NEW(comps, type, off, verts, start)		\
   {									\
      log_print(LOG_DATA, "\tnumber components: %d\n", comps);		\
      log_print(LOG_DATA, "\tdata type: %e\n", type);			\
      log_print(LOG_DATA, "\toffset into array: %d\n", off);		\
      log_print(LOG_DATA, "\n\tstarting address of verts: %d\n", verts);\
      log_print(LOG_DATA, "\tstart of this array: %d\n", start);	\
   }

#define GLX_LOG_ARRAY(comps, type, off)	 				\
   {									\
      log_print(LOG_DATA, "\tnumber components: %d\n", comps);		\
      log_print(LOG_DATA, "\tdata type: %e\n", type);			\
      log_print(LOG_DATA, "\toffset into array: %d\n", off);		\
   }


int
get_length(GLenum array_type, struct array_info *records,
               int num_records, int num_vertices) {
   int comp_size = 0;
   int array_size = 0;
   int i = 0;

   for (i = 0; i < num_records; i++) {
      if ( ( (GLenum) (records[i].array_type) ) == array_type) {
         comp_size = records[i].j *
            GLX_data_size( (GLenum) records[i].data_type );
         comp_size = GLX_pad(comp_size);
         array_size = comp_size * num_vertices;
         break;
      }
   }

   return array_size;
}

int
GLX_varray_size(int num_verts, int num_arrays, void *arrays) {
   unsigned int size = 0;

   size  = get_length(GL_EDGE_FLAG_ARRAY, arrays, num_arrays, num_verts);
   size += get_length(GL_TEXTURE_COORD_ARRAY, arrays, num_arrays, num_verts);
   size += get_length(GL_COLOR_ARRAY, arrays, num_arrays, num_verts);
   size += get_length(GL_INDEX_ARRAY, arrays, num_arrays, num_verts);
   size += get_length(GL_NORMAL_ARRAY, arrays, num_arrays, num_verts);
   size += get_length(GL_VERTEX_ARRAY, arrays, num_arrays, num_verts);

   return (size / 4);
}

int
GLX_array_info_size(int num_arrays) {
   /** return number of arrays * number of elements in a single array **/
   return num_arrays * ( sizeof(struct array_info) / sizeof(int) );
}

/*
 * find a specific array record for a given array.
 * return the number of bytes the arrays elements require, based on the
 * array's type and number of components
 */

int
get_comp_size(GLenum array_type, struct array_info *records, 
               int num_records) {
   int comp_size = 0;
   int i = 0;

   for (i = 0; i < num_records; i++) {
      if ( ( (GLenum) (records[i].array_type) ) == array_type) {
         comp_size = records[i].j * 
            GLX_data_size( (GLenum) records[i].data_type );
         break;
      }
   }

   return comp_size;
}



GLenum
get_enum(GLenum array_type, struct array_info *records, 
               int num_records) {
   int i = 0;

   for (i = 0; i < num_records; i++) {
      if ( ( (GLenum) (records[i].array_type) ) == array_type) {
         return (GLenum) records[i].data_type;
      }
   }
   return GL_FALSE;
}


unsigned int
get_num_comps(GLenum array_type, struct array_info *records, 
               int num_records) {
   int i = 0;

   for (i = 0; i < num_records; i++) {
      if ( ( (GLenum) (records[i].array_type) ) == array_type) {
         return (unsigned int) records[i].j;
      }
   }
   return 0;
}

int
glx_parse_array_infos(struct glx_varray *tmp_varray, struct array_info *records,
                      int num_records) 
{
   int i;

   memset( tmp_varray, 0, sizeof(struct glx_varray) * GLX_MAX_ARRAY_INFOS);

   for (i = 0; i < num_records; i++) {
      switch (records[i].array_type) {
         case GL_VERTEX_ARRAY:
            tmp_varray[GLX_VERTEX].data_type = records[i].data_type;
            tmp_varray[GLX_VERTEX].num_comps = records[i].j;
            break;

         case GL_NORMAL_ARRAY:
            tmp_varray[GLX_NORMAL].data_type = records[i].data_type;
            tmp_varray[GLX_NORMAL].num_comps = records[i].j;
            break;

         case GL_TEXTURE_COORD_ARRAY:
            tmp_varray[GLX_TEXTURE].data_type = records[i].data_type;
            tmp_varray[GLX_TEXTURE].num_comps = records[i].j;
            break;

         case GL_COLOR_ARRAY:
            tmp_varray[GLX_COLOR].data_type = records[i].data_type;
            tmp_varray[GLX_COLOR].num_comps = records[i].j;
            break;

         case GL_INDEX_ARRAY:
            tmp_varray[GLX_INDEX].data_type = records[i].data_type;
            tmp_varray[GLX_INDEX].num_comps = records[i].j;
            break;

         case GL_EDGE_FLAG_ARRAY:
            tmp_varray[GLX_EDGEFLAG].data_type = records[i].data_type;
            tmp_varray[GLX_EDGEFLAG].num_comps = records[i].j;
            break;

         default:
            ErrorF("Wrong type when parsing Vertex Array!!\n");
      }
   }
   return 0;
}


/* 
 * since the beginning of each type of array in the packed and padded
 * interleaved array coming across the wire is offset cumulatively, use
 * a waterfall method to calculate the offsets.
 */

int
glx_varray_calc_offsets(struct glx_varray *arrays) {
   unsigned int i, j;
   unsigned int element_size = 0;

   for (i = 0; i < GLX_MAX_ARRAY_INFOS; i++) {
      if (arrays[i].num_comps != 0) {
         element_size = GLX_pad( arrays[i].num_comps *
             GLX_data_size(arrays[i].data_type) );
         for (j = i + 1; j < GLX_MAX_ARRAY_INFOS; j++) {
            arrays[j].offset += element_size;
         }
      }
   }
   return 0;
}


void
GLX_swap_varray(int num_comps, GLenum data_type, int num_vertices, 
                int glx_stride, void *start) {

   unsigned int data_size = GLX_data_size(data_type);
   unsigned char *array = (unsigned char *) start;
   register int _n, i;

   switch (data_size) {
      case 4:

            switch (num_comps) {
               case 3: 
                  for (i = 0; i < num_vertices; i++) {
                     swapl( ( ( (unsigned int *) array ) + 0 ), _n );
                     swapl( ( ( (unsigned int *) array ) + 1 ), _n );
                     swapl( ( ( (unsigned int *) array ) + 2 ), _n );
                     array += glx_stride;
                  }
                  break;
               case 4: 
                  for (i = 0; i < num_vertices; i++) {
                     swapl( ( (unsigned int *) array ) + 0, _n );
                     swapl( ( (unsigned int *) array ) + 1, _n );
                     swapl( ( (unsigned int *) array ) + 2, _n );
                     swapl( ( (unsigned int *) array ) + 3, _n );
                     array += glx_stride;
                  }
                  break;
               case 2: 
                  for (i = 0; i < num_vertices; i++) {
                     swapl( ( (unsigned int *) array ) + 0, _n );
                     swapl( ( (unsigned int *) array ) + 1, _n );
                     array += glx_stride;
                  }
                  break;
               case 1: 
                  for (i = 0; i < num_vertices; i++) {
                     swapl( ( (unsigned int *) array ) + 0, _n );
                     array += glx_stride;
                  }
                  break;
               default:
                  log_print(LOG_ALWAYS, 
                     "GLX VARRAYS: bad number of components to swap!\n");
                  break;
            }

         break;
      case 2:

            switch (num_comps) {
               case 3: 
                  for (i = 0; i < num_vertices; i++) {
                     swaps( ( (unsigned short *) array ) + 0, _n );
                     swaps( ( (unsigned short *) array ) + 1, _n );
                     swaps( ( (unsigned short *) array ) + 2, _n );
                     array += glx_stride;
                  }
                  break;
               case 4: 
                  for (i = 0; i < num_vertices; i++) {
                     swaps( ( (unsigned short *) array ) + 0, _n );
                     swaps( ( (unsigned short *) array ) + 1, _n );
                     swaps( ( (unsigned short *) array ) + 2, _n );
                     swaps( ( (unsigned short *) array ) + 3, _n );
                     array += glx_stride;
                  }
                  break;
               case 2: 
                  for (i = 0; i < num_vertices; i++) {
                     swaps( ( (unsigned short *) array ) + 0, _n );
                     swaps( ( (unsigned short *) array ) + 1, _n );
                     array += glx_stride;
                  }
                  break;
               case 1: 
                  for (i = 0; i < num_vertices; i++) {
                     swaps( ( (unsigned short *) array ) + 0, _n );
                     array += glx_stride;
                  }
                  break;
               default:
                  log_print(LOG_ALWAYS, 
                     "GLX VARRAYS: bad number of components to swap!\n");
                  break;
            }

         break;
      case 8:

            switch (num_comps) {
               case 3: 
                  for (i = 0; i < num_vertices; i++) {
                     swapd( ( (double *) array ) + 0, _n );
                     swapd( ( (double *) array ) + 1, _n );
                     swapd( ( (double *) array ) + 2, _n );
                     array += glx_stride;
                  }
                  break;
               case 4: 
                  for (i = 0; i < num_vertices; i++) {
                     swapd( ( (double *) array ) + 0, _n );
                     swapd( ( (double *) array ) + 1, _n );
                     swapd( ( (double *) array ) + 2, _n );
                     swapd( ( (double *) array ) + 3, _n );
                     array += glx_stride;
                  }
                  break;
               case 2: 
                  for (i = 0; i < num_vertices; i++) {
                     swapd( ( (double *) array ) + 0, _n );
                     swapd( ( (double *) array ) + 1, _n );
                     array += glx_stride;
                  }
                  break;
               case 1: 
                  for (i = 0; i < num_vertices; i++) {
                     swapd( ( (double *) array ) + 0, _n );
                     array += glx_stride;
                  }
                  break;
               default:
                  log_print(LOG_ALWAYS, 
                     "GLX VARRAYS: bad number of components to swap!\n");
                  break;
            }

         break;
      case 1:
         /** don't need to swap **/
         break;
      default:
         log_print(LOG_ALWAYS, 
            "GLX VARRAYS: Attempt to swap unrecognized data size array");
         break;
   }
}


int 
GLX_DrawArrays(int n, int m, GLenum mode, struct array_info *infos,
                 void *glx_vertices)
{
   unsigned int glx_stride;
   unsigned int i, j;
   unsigned char *start = NULL;
   unsigned int num_comps = 0;
   GLenum data_type;
   unsigned int offset = 0;
   struct glx_varray arrays[GLX_MAX_ARRAY_INFOS];
   unsigned int element_size = 0;
 
   if (glx_parse_array_infos(&arrays[0], infos, m)) return -1;

   /** calculate the stride from one element to the next **/
   glx_stride = 0;
   for (i = 0; i < GLX_MAX_ARRAY_INFOS; i++) {
      if (arrays[i].num_comps) {
         element_size = GLX_pad( arrays[i].num_comps *
             GLX_data_size(arrays[i].data_type) );

         /* calculate stride */
         glx_stride += element_size;

         /* calculate offset from start of element */
         for (j = i + 1; j < GLX_MAX_ARRAY_INFOS; j++) {
            arrays[j].offset += element_size;
         } 
      }
   }

   if (arrays[GLX_EDGEFLAG].num_comps) {
      num_comps = arrays[GLX_EDGEFLAG].num_comps;
      data_type = arrays[GLX_EDGEFLAG].data_type;
      offset = arrays[GLX_EDGEFLAG].offset; 
      start = (unsigned char *) glx_vertices + offset;
      glEdgeFlagPointer( glx_stride, glx_vertices);      
      glEnableClientState( GL_EDGE_FLAG_ARRAY );
   }
   else
   {
   	glDisableClientState(GL_EDGE_FLAG_ARRAY);
   }

   if (arrays[GLX_TEXTURE].num_comps) {
      num_comps = arrays[GLX_TEXTURE].num_comps;
      data_type = arrays[GLX_TEXTURE].data_type;
      offset = arrays[GLX_TEXTURE].offset; 
      start = (unsigned char *) glx_vertices + offset;
      glTexCoordPointer( num_comps, data_type, glx_stride, (void *) start);
      glEnableClientState( GL_TEXTURE_COORD_ARRAY );
   }
   else
   {
   	glDisableClientState (GL_TEXTURE_COORD_ARRAY);
   }
   

   if (arrays[GLX_COLOR].num_comps) {
      num_comps = arrays[GLX_COLOR].num_comps;
      data_type = arrays[GLX_COLOR].data_type;
      offset = arrays[GLX_COLOR].offset; 
      start = ( (unsigned char *) glx_vertices ) + offset;
      glColorPointer( num_comps, data_type, glx_stride, (void *) start);
      glEnableClientState( GL_COLOR_ARRAY );
   }
   else
   {
   	glDisableClientState (GL_COLOR_ARRAY);
   }
   if (arrays[GLX_INDEX].num_comps) {
      num_comps = arrays[GLX_INDEX].num_comps;
      data_type = arrays[GLX_INDEX].data_type;
      offset = arrays[GLX_INDEX].offset; 
      start = ( (unsigned char *) glx_vertices ) + offset;
      glIndexPointer( data_type, glx_stride, (void *) start);
      glEnableClientState( GL_INDEX_ARRAY );
   }
   else
   {
   	glDisableClientState (GL_INDEX_ARRAY);
   }

   if (arrays[GLX_NORMAL].num_comps) {
      log_print(LOG_TRACE, "Normals defined\n");
      num_comps = arrays[GLX_NORMAL].num_comps;
      data_type = arrays[GLX_NORMAL].data_type;
      offset = arrays[GLX_NORMAL].offset; 
      start = ( (unsigned char *) glx_vertices ) + offset;
      glNormalPointer( data_type, glx_stride, (void *) start);
      glEnableClientState( GL_NORMAL_ARRAY );
   }
   else
   {
   	glDisableClientState ( GL_NORMAL_ARRAY );
   }

   if (arrays[GLX_VERTEX].num_comps) {
      log_print(LOG_TRACE, "Vertices defined\n");
      num_comps = arrays[GLX_VERTEX].num_comps;
      data_type = arrays[GLX_VERTEX].data_type;
      offset = arrays[GLX_VERTEX].offset; 
      start = ( (unsigned char *) glx_vertices ) + offset;
      glVertexPointer( num_comps, data_type, glx_stride, (void *) start);
      glEnableClientState( GL_VERTEX_ARRAY );
   }
   else
   {
   	glDisableClientState ( GL_VERTEX_ARRAY );
   }
   
   glDrawArrays( mode, 0, n );

/*
   glDisableClientState(GL_VERTEX_ARRAY);
   glDisableClientState(GL_NORMAL_ARRAY);
   glDisableClientState(GL_COLOR_ARRAY);
   glDisableClientState(GL_INDEX_ARRAY);
   glDisableClientState(GL_TEXTURE_COORD_ARRAY);
   glDisableClientState(GL_EDGE_FLAG_ARRAY);
*/

   return 0;
}

int 
GLX_DrawArrays_swapped(int n, int m, GLenum mode, struct array_info *infos,
                 void *glx_vertices)
{
   unsigned int glx_stride;
   unsigned int i;
   unsigned char *start = NULL;
   unsigned int num_comps = 0;
   GLenum data_type;
   unsigned int offset = 0;

   struct glx_varray arrays[GLX_MAX_ARRAY_INFOS];
   memset( &arrays, 0, sizeof(struct glx_varray) * GLX_MAX_ARRAY_INFOS);

   if ( (infos == NULL) || (glx_vertices == NULL) )  return 1;
 
   log_print(LOG_TRACE, "swapped DrawArrays");

   /* get all enums */
   log_print(LOG_TRACE, "Getting enums");
   arrays[GLX_EDGEFLAG].data_type = get_enum(GL_EDGE_FLAG_ARRAY, infos, m);
   arrays[GLX_TEXTURE].data_type  = 
      get_enum(GL_TEXTURE_COORD_ARRAY, infos, m);
   arrays[GLX_COLOR].data_type    = get_enum(GL_COLOR_ARRAY, infos, m);
   arrays[GLX_INDEX].data_type    = get_enum(GL_INDEX_ARRAY, infos, m);
   arrays[GLX_NORMAL].data_type   = get_enum(GL_NORMAL_ARRAY, infos, m);
   arrays[GLX_VERTEX].data_type   = get_enum(GL_VERTEX_ARRAY, infos, m);

   /* get number of components for each array */
   log_print(LOG_TRACE, "Getting components");
   arrays[GLX_EDGEFLAG].num_comps = 
      get_num_comps(GL_EDGE_FLAG_ARRAY, infos, m);
   arrays[GLX_TEXTURE].num_comps  = 
      get_num_comps(GL_TEXTURE_COORD_ARRAY, infos, m);
   arrays[GLX_COLOR].num_comps    = get_num_comps(GL_COLOR_ARRAY, infos, m);
   arrays[GLX_INDEX].num_comps    = get_num_comps(GL_INDEX_ARRAY, infos, m);
   arrays[GLX_NORMAL].num_comps   = get_num_comps(GL_NORMAL_ARRAY, infos, m);
   arrays[GLX_VERTEX].num_comps   = get_num_comps(GL_VERTEX_ARRAY, infos, m);

   glx_varray_calc_offsets(arrays);

   /** calculate the stride from one element to the next **/
   glx_stride = 0;
   for (i = 0; i < GLX_MAX_ARRAY_INFOS; i++) {
      if (arrays[i].num_comps) {
         glx_stride += GLX_pad ( arrays[i].num_comps * 
            GLX_data_size( (GLenum) arrays[i].data_type ) );
      }
   }

   log_print(LOG_TRACE, "Vertex Array stride: %d\n", glx_stride);
   log_print(LOG_TRACE, "Num Vertices: %d\n", n);
   log_print(LOG_TRACE, "Enabled Arrays: %d\n", m);

   if (arrays[GLX_EDGEFLAG].num_comps) {
      log_print(LOG_TRACE, "Edge Flags defined\n");
      num_comps = arrays[GLX_EDGEFLAG].num_comps;
      data_type = arrays[GLX_EDGEFLAG].data_type;
      offset = arrays[GLX_EDGEFLAG].offset; 
      start = ( (unsigned char *) glx_vertices ) + offset;
      GLX_LOG_ARRAY(num_comps, data_type, offset);
      GLX_swap_varray(num_comps, data_type, n, glx_stride, start);
      glEdgeFlagPointer( glx_stride, start);      
      glEnableClientState( GL_EDGE_FLAG_ARRAY );
   }

   if (arrays[GLX_TEXTURE].num_comps) {
      log_print(LOG_TRACE, "Textures defined\n");
      num_comps = arrays[GLX_TEXTURE].num_comps;
      data_type = arrays[GLX_TEXTURE].data_type;
      offset = arrays[GLX_TEXTURE].offset; 
      start = ( (unsigned char *) glx_vertices ) + offset;
      GLX_LOG_ARRAY(num_comps, data_type, offset);
      GLX_swap_varray(num_comps, data_type, n, glx_stride, start);
      glTexCoordPointer( num_comps, data_type, glx_stride, (void *) start);
      glEnableClientState( GL_TEXTURE_COORD_ARRAY );
   }

   if (arrays[GLX_COLOR].num_comps) {
      log_print(LOG_TRACE, "Colors defined\n");
      num_comps = arrays[GLX_COLOR].num_comps;
      data_type = arrays[GLX_COLOR].data_type;
      offset = arrays[GLX_COLOR].offset; 
      start = ( (unsigned char *) glx_vertices ) + offset;
      GLX_LOG_ARRAY(num_comps, data_type, offset);
      GLX_swap_varray(num_comps, data_type, n, glx_stride, start);
      glColorPointer( num_comps, data_type, glx_stride, (void *) start);
      glEnableClientState( GL_COLOR_ARRAY );
   }

   if (arrays[GLX_INDEX].num_comps) {
      log_print(LOG_TRACE, "Indices defined\n");
      num_comps = arrays[GLX_INDEX].num_comps;
      data_type = arrays[GLX_INDEX].data_type;
      offset = arrays[GLX_INDEX].offset; 
      start = ( (unsigned char *) glx_vertices ) + offset;
      GLX_LOG_ARRAY(num_comps, data_type, offset);
      GLX_swap_varray(num_comps, data_type, n, glx_stride, start);
      glIndexPointer( data_type, glx_stride, (void *) start);
      glEnableClientState( GL_INDEX_ARRAY );
   }

   if (arrays[GLX_NORMAL].num_comps) {
      log_print(LOG_TRACE, "Normals defined\n");
      num_comps = arrays[GLX_NORMAL].num_comps;
      data_type = arrays[GLX_NORMAL].data_type;
      offset = arrays[GLX_NORMAL].offset; 
      start = ( (unsigned char *) glx_vertices ) + offset;
      GLX_swap_varray(num_comps, data_type, n, glx_stride, start);
      GLX_LOG_ARRAY_NEW(num_comps, data_type, offset, glx_vertices, start);
      GLX_LOG_FLOAT_ARRAY(num_comps, n, glx_stride, start);
      glNormalPointer( data_type, glx_stride, (void *) start);
      glEnableClientState( GL_NORMAL_ARRAY );
   }

   if (arrays[GLX_VERTEX].num_comps) {
      log_print(LOG_TRACE, "Vertices defined\n");
      num_comps = arrays[GLX_VERTEX].num_comps;
      data_type = arrays[GLX_VERTEX].data_type;
      offset = arrays[GLX_VERTEX].offset; 
      start = ( (unsigned char *) glx_vertices ) + offset;
      GLX_swap_varray(num_comps, data_type, n, glx_stride, start);
      GLX_LOG_ARRAY_NEW(num_comps, data_type, offset, glx_vertices, start);
      GLX_LOG_FLOAT_ARRAY(num_comps, n, glx_stride, start);
      glVertexPointer( num_comps, data_type, glx_stride, (void *) start);
      glEnableClientState( GL_VERTEX_ARRAY );
   }

   glDrawArrays( mode, 0, n );

   glDisableClientState(GL_VERTEX_ARRAY);
   glDisableClientState(GL_NORMAL_ARRAY);
   glDisableClientState(GL_COLOR_ARRAY);
   glDisableClientState(GL_INDEX_ARRAY);
   glDisableClientState(GL_TEXTURE_COORD_ARRAY);
   glDisableClientState(GL_EDGE_FLAG_ARRAY);

   return 0;
}


