/*
 *  $Id: graph-key-dialog.c 28778 2025-11-04 16:47:31Z yeti-dn $
 *  Copyright (C) 2003-2025 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <glib/gi18n-lib.h>
#include <gtk/gtk.h>

#include "libgwyddion/macros.h"
#include "libgwyui/gwyoptionmenus.h"
#include "libgwyui/graph.h"
#include "libgwyui/gwygraphmodel.h"
#include "libgwyui/utils.h"
#include "libgwyui/graph-key-dialog.h"

/* FIXME: Reverse dependence, change sanity.h to normal API and get rid of parts of it, as needed. */
#include "libgwyapp/sanity.h"

static void     dispose          (GObject *object);
static void     finalize         (GObject *object);
static gboolean deleted          (GtkWidget *widget,
                                  GdkEventAny *event);
static void     response         (GtkDialog *dialog,
                                  gint response_id);
static void     thickness_changed(GtkAdjustment *adj,
                                  GwyGraphKeyDialog *dialog);
static void     reverse_changed  (GtkToggleButton *button,
                                  GwyGraphKeyDialog *dialog);
static gboolean set_model        (GwyGraphKeyDialog *dialog,
                                  GwyGraphModel *model);
static void     model_notify     (GwyGraphKeyDialog *dialog,
                                  GParamSpec *param,
                                  GwyGraphModel *gmodel);

GtkDialogClass *parent_class = NULL;

G_DEFINE_TYPE(GwyGraphKeyDialog, _gwy_graph_key_dialog, GTK_TYPE_DIALOG)

static void
_gwy_graph_key_dialog_class_init(GwyGraphKeyDialogClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
    GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(klass);
    GtkDialogClass *dialog_class = GTK_DIALOG_CLASS(klass);

    parent_class = _gwy_graph_key_dialog_parent_class;

    gobject_class->dispose = dispose;
    gobject_class->finalize = finalize;

    widget_class->delete_event = deleted;

    dialog_class->response = response;
}

static void
finalize(GObject *object)
{
    G_OBJECT_CLASS(parent_class)->finalize(object);
}

static void
dispose(GObject *object)
{
    set_model(GWY_GRAPH_KEY_DIALOG(object), NULL);
    G_OBJECT_CLASS(parent_class)->dispose(object);
}

static void
_gwy_graph_key_dialog_init(GwyGraphKeyDialog *dialog)
{
    gtk_window_set_title(GTK_WINDOW(dialog), _("Label Properties"));
    gwy_add_button_to_dialog(GTK_DIALOG(dialog), GWY_STOCK_CLOSE, GWY_ICON_GTK_CLOSE, GTK_RESPONSE_CLOSE);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_CLOSE);

    GtkGrid *grid = GTK_GRID(gtk_grid_new());
    gtk_grid_set_row_spacing(grid, 2);
    gtk_grid_set_column_spacing(grid, 6);
    gtk_container_set_border_width(GTK_CONTAINER(grid), 4);

    GtkWidget *vbox = gtk_dialog_get_content_area(GTK_DIALOG(dialog));
    gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(grid), FALSE, TRUE, 0);

    gint row = 0;
    dialog->thickness = gtk_adjustment_new(1, 0, 6, 1, 5, 0);
    gwy_gtkgrid_attach_adjbar(grid, row++, _("_Frame thickness:"), NULL, G_OBJECT(dialog->thickness),
                              GWY_HSCALE_LINEAR);
    g_signal_connect(dialog->thickness, "value-changed", G_CALLBACK(thickness_changed), dialog);

    dialog->reversed = gtk_check_button_new_with_mnemonic(_("_Reversed layout"));
    g_signal_connect(dialog->reversed, "toggled", G_CALLBACK(reverse_changed), dialog);
    gtk_grid_attach(grid, dialog->reversed, 0, row++, 2, 1);

    gtk_widget_show_all(vbox);
}

static gboolean
deleted(GtkWidget *widget, G_GNUC_UNUSED GdkEventAny *event)
{
    gtk_widget_hide(widget);
    return TRUE;
}

static void
response(GtkDialog *dialog, gint response_id)
{
    if (response_id == GTK_RESPONSE_CLOSE)
        gtk_widget_hide(GTK_WIDGET(dialog));
}

GtkWidget*
_gwy_graph_key_dialog_new(void)
{
    return GTK_WIDGET(g_object_new(GWY_TYPE_GRAPH_KEY_DIALOG, NULL));
}

void
_gwy_graph_key_dialog_set_model(GwyGraphKeyDialog *dialog,
                                GwyGraphModel *model)
{
    g_return_if_fail(GWY_IS_GRAPH_KEY_DIALOG(dialog));
    if (set_model(dialog, model) && model)
        model_notify(dialog, NULL, dialog->graph_model);
}

static gboolean
set_model(GwyGraphKeyDialog *dialog, GwyGraphModel *model)
{
    return gwy_set_member_object(dialog, model, GWY_TYPE_GRAPH_MODEL, &dialog->graph_model,
                                 "notify", G_CALLBACK(model_notify), &dialog->notify_id, G_CONNECT_SWAPPED,
                                 NULL);
}

static void
thickness_changed(GtkAdjustment *adj, GwyGraphKeyDialog *dialog)
{
    if (!dialog->graph_model || dialog->updating)
        return;

    dialog->updating++;
    g_object_set(dialog->graph_model,
                 "label-frame-thickness", gwy_adjustment_get_int(adj),
                 NULL);
    dialog->updating--;
}

static void
reverse_changed(GtkToggleButton *button, GwyGraphKeyDialog *dialog)
{
    if (!dialog->graph_model || dialog->updating)
        return;

    dialog->updating++;
    g_object_set(dialog->graph_model,
                 "label-reverse", gtk_toggle_button_get_active(button),
                 NULL);
    dialog->updating--;
}

static void
model_notify(GwyGraphKeyDialog *dialog, GParamSpec *param, G_GNUC_UNUSED GwyGraphModel *gmodel)
{
    if (dialog->updating || !dialog->graph_model)
        return;

    dialog->updating++;
    if (!param || gwy_strequal(param->name, "label-reverse")) {
        gboolean reverse;
        g_object_get(dialog->graph_model, "label-reverse", &reverse, NULL);
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->reversed), reverse);
    }
    if (!param || gwy_strequal(param->name, "label-frame-thickness")) {
        gint thickness;
        g_object_get(dialog->graph_model, "label-frame-thickness", &thickness, NULL);
        gtk_adjustment_set_value(GTK_ADJUSTMENT(dialog->thickness), thickness);
    }
    dialog->updating--;
}

/* vim: set cin et columns=120 tw=118 ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
