////////////////////////////////////////////////////////////////////////
//
// Copyright (C) 2012-2026 The Octave Project Developers
//
// See the file COPYRIGHT.md in the top-level directory of this
// distribution or <https://octave.org/copyright/>.
//
// This file is part of Octave.
//
// Octave is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Octave is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Octave; see the file COPYING.  If not, see
// <https://www.gnu.org/licenses/>.
//
////////////////////////////////////////////////////////////////////////

#if defined (HAVE_CONFIG_H)
#  include "config.h"
#endif

#include <algorithm>
#include <iomanip>

#include "cdef-class.h"
#include "cdef-method.h"
#include "cdef-package.h"
#include "cdef-property.h"
#include "cdef-utils.h"
#include "defun.h"
#include "errwarn.h"
#include "interpreter-private.h"
#include "load-path.h"
#include "ls-oct-text.h"
#include "ov-classdef.h"
#include "ov-fcn-handle.h"
#include "ov-typeinfo.h"
#include "ov-usr-fcn.h"
#include "parse.h"
#include "pr-output.h"
#include "pt-eval.h"
#include "pt-misc.h"
#include "oct-lvalue.h"

std::vector<std::tuple<octave_map, uint32_t, bool>>
octave_classdef::saveobj (std::vector<bool>& is_new)
{
  octave::cdef_class cls = m_object.get_class ();
  octave::cdef_method meth = cls.find_method ("saveobj");

  // Vector with tuples consisting of a map with property values, the object id,
  // and an indicator whether the object has a custom return type
  std::vector<std::tuple<octave_map, uint32_t, bool>> m (numel ());

  octave::load_save_system& lss = octave::__get_load_save_system__ ();

  // The saveobj method needs to be called separately for each element of
  // N-D arrays.
  for (octave_idx_type n = 0; n < numel (); n++)
    {
      octave_value retval;

      octave_value_list ovl_idx;
      std::list<octave_value_list> idx_tmp;
      ovl_idx(0) = n+1;
      idx_tmp.push_back (ovl_idx);
      octave_value elem = (subsref ("(", idx_tmp, 1))(0);

      // Check if the element is already in the cache of objects that are
      // stored in this file.
      // Use the address of the cdef_object_rep as the key.
      const octave::cdef_object co = elem.classdef_object_value ()->get_object_ref ();
      bool is_new_elem;
      std::get<uint32_t> (m[n])
        = lss.get_mcos_object_cache_id (co.get_rep (), is_new_elem);
      is_new[n] = is_new_elem;

      if (is_new[n])
        {
          // Default behavior of 'loadobj' is to return this object's
          // map_value, including all protected, private and hidden properties.
          if (! meth.ok () || meth.is_static ())
            // Default behaviour of saving is triggered if saveobj is static
            std::get<octave_map> (m[n])
              = elem.classdef_object_value ()->map_value (false, true);
          else
            {
              retval = (meth.execute (elem.classdef_object_value ()->get_object_ref (),
                                      octave_value_list (), 1))(0);

              // default: no custom return type
              std::get<bool> (m[n]) = false;

              if (! retval.is_defined ())
                {
                  warning ("save: saveobj method does not return a value");
                  return m;
                }
              else if (! retval.is_classdef_object ()
                       || retval.class_name () != class_name ())
                {
                  // If retval is not an object of the matching class, we put
                  // the value in a map and set the 'custom_saveobj_ret_type'
                  // flag, which has to be encoded in the file metadata.
                  // It's the caller's responsibility to check the flag.
                  std::get<octave_map> (m[n]).setfield ("any", retval);
                  std::get<bool> (m[n]) = true;
                }

              else if (retval.is_classdef_object ())
                std::get<octave_map> (m[n])
                  = retval.classdef_object_value ()->map_value (false, true);
              else
                std::get<octave_map> (m[n]) = retval.map_value ();
            }
        }
    }

  return m;
}

void
octave_classdef::loadobj (std::vector<std::tuple<octave_map, uint32_t, bool>>& m,
                          dim_vector& dv)
{
  octave::cdef_object scalar_obj = m_object.copy ();
  octave::cdef_class cls = m_object.get_class ();
  octave::cdef_method meth = cls.find_method ("loadobj");

  octave::load_save_system& lss = octave::__get_load_save_system__ ();

  // The loadobj method needs to be called separately for each element of
  // N-D arrays.
  for (octave_idx_type n = 0; static_cast<size_t> (n) < m.size (); n++)
    {
      octave_value ovc;

      // Check if object is already loaded in cache
      uint32_t id = std::get<uint32_t> (m[n]);
      bool in_obj_cache = lss.is_mcos_object_cache_entry (id);
      if (in_obj_cache)
        ovc = lss.get_mcos_object_cache_entry (id);

      octave_map& prop_map = std::get<octave_map> (m[n]);

      if (! in_obj_cache || prop_map.nfields () > 0)
        {
          // Default behaviour of loading is triggered if loadobj is not static
          if (meth.ok () && meth.is_static () && prop_map.nfields () > 0)
            {
              octave_value ov;
              if (std::get<bool> (m[n]))
                {
                  octave_value any = prop_map.contents ("any").elem (0);
                  ov = (meth.execute (octave_value_list (any), 1))(0);
                }
              else
                {
                  // create object from saved properties
                  octave::cdef_object new_object;
                  if (in_obj_cache)
                    new_object = ovc.classdef_object_value ()->m_object;
                  else
                    new_object = scalar_obj.copy ();

                  bool props_changed = false;
                  string_vector fnames = prop_map.fieldnames ();
                  string_vector sv = map_keys ();
                  for (octave_idx_type i = 0; i < prop_map.nfields (); i++)
                    {
                      octave_idx_type j;
                      for (j = 0; j < sv.numel (); j++)
                        {
                          if (sv[j] == fnames(i))
                            {
                              new_object.set_property (0, sv[j], prop_map.contents (fnames(i)).xelem (0));
                              break;
                            }
                        }
                      if (j == sv.numel ())
                        {
                          // properties have been renamed or deleted
                          props_changed = true;
                          break;
                        }
                    }

                  if (props_changed)
                    // attempting to create the object failed
                    // call loadobj with struct
                    ov = (meth.execute (octave_value_list (prop_map), 1))(0);
                  else
                    {
                      if (! in_obj_cache)
                        ovc = octave::to_ov (new_object);

                      // pass object to loadobj
                      ov = (meth.execute (octave_value_list (ovc), 1))(0);
                    }
                }

              if (! ov.is_defined ())
                {
                  warning ("load: loadobj method does not return a value");
                  return;
                }

              // FIXME: A loadobj method can return any type. If the return
              //        type is not a classdef object of the correct class,
              //        then the loaded object must be replaced by whatever the
              //        return type and contents are.
              if (! ov.is_classdef_object ())
                {
                  std::string type = ov.type_name ();
                  warning ("load: loadobj method does not return correct type "
                           "'%s'. This is currently not supported.",
                           type.c_str ());
                  return;
                }
              else if (ov.class_name () != class_name ())
                {
                  std::string class_nm = ov.class_name ();
                  warning ("load: loadobj method does not return classdef object "
                           "of correct class '%s'. This is currently not supported.",
                           class_nm.c_str ());
                  return;
                }

              if (in_obj_cache)
                // Copy the results from the loadobj methods to the object in
                // the cache.
                ovc.classdef_object_value ()->m_object = ov.classdef_object_value ()->m_object;
              else
                ovc = ov;
            }
          else
            {
              if (std::get<bool> (m[n]))
                {
                  // If saveobj is overloaded by this classdef and it returned
                  // anything other than a classdef object of the correct
                  // class, then a variable named 'any' is meant to be passed
                  // to loadobj, but if loadobj is not overloaded, it should
                  // not fill in any property 'any' in the loaded object.
                  if (! prop_map.isfield ("any") || prop_map.numel () != 1)
                    {
                      warning ("load: expected scalar value for custom type when loading object");
                      return;
                    }

                  // FIXME: What should be done here?
                  octave_value any_val = (prop_map.getfield ("any"))(0);
                  std::string type = any_val.type_name ();
                  if (type != type_name ())
                    {
                      warning ("load: cannot restore value of object that was saved as a different type (%s)",
                               type.c_str ());
                      return;
                    }

                  std::string cls_nm = any_val.class_name ();
                  if (cls_nm != class_name ())
                    {
                      warning ("load: cannot restore value of object that was saved as a different class (%s)",
                               cls_nm.c_str ());
                      return;
                    }

                  // If the value in the "any" field has the correct type and
                  // class, we can load it like it were saved "normally".
                  // FIXME: Can this ever happen?
                  prop_map = any_val.classdef_object_value ()->map_value (false, true);
                }

              octave::cdef_object new_object;
              if (in_obj_cache)
                new_object = ovc.classdef_object_value ()->m_object;
              else
                new_object = scalar_obj.copy ();

              string_vector fnames = prop_map.fieldnames ();
              string_vector sv = map_keys ();
              for (octave_idx_type i = 0; i < prop_map.nfields (); i++)
                for (octave_idx_type j = 0; j < sv.numel (); j++)
                  if (sv[j] == fnames(i))
                    {
                      new_object.set_property (0, sv[j], prop_map.contents (fnames(i)).xelem (0));
                      break;
                    }

              if (! in_obj_cache)
                ovc = octave::to_ov (new_object);
            }

          lss.set_mcos_object_cache_entry (id, ovc);
        }

      if (m.size () == 1)
        m_object = ovc.classdef_object_value ()->m_object;
      else
        {
          octave_value_list ovl_idx;
          std::list<octave_value_list> idx_tmp;
          ovl_idx(0) = n+1;
          idx_tmp.push_back (ovl_idx);
          octave_value tmp = subsasgn ("(", idx_tmp, ovc);

          // FIXME: Is this assignment only needed for value classes?
          m_object = tmp.classdef_object_value ()->m_object;
        }
    }

  // Reshape to the correct dimensions.
  if (dv != dims ())
    {
      octave_value ov_reshaped = reshape (dv);
      m_object = ov_reshaped.classdef_object_value ()->m_object;
    }
}

bool
octave_classdef::save_ascii (std::ostream& os)
{
  os << "# classname: " << class_name () << "\n";

  const dim_vector dv = m_object.dims ();
  os << "# ndims: " << dv.ndims () << "\n";

  for (int i = 0; i < dv.ndims (); i++)
    os << ' ' << dv(i);
  os << "\n";

  std::vector<bool> is_new (numel ());
  std::vector<std::tuple<octave_map, uint32_t, bool>> m
    = saveobj (is_new);

  for (octave_idx_type n = 0; static_cast<size_t> (n) < m.size (); n++)
    {
      os << "# id: " << std::get<uint32_t> (m[n]) << "\n";

      // Check if this is a reference to an already existing object
      if (! is_new[n])
        {
          os << "# length: 0\n";
          continue;
        }

      octave_idx_type nf = std::get<octave_map> (m[n]).nfields ();

      os << "# length: " << nf << "\n";

      os << "# metadata: ";
      if (std::get<bool> (m[n]))
        os << "saveobj_defined";
      os << "\n";

      string_vector keys = std::get<octave_map> (m[n]).fieldnames ();

      for (octave_idx_type i = 0; i < nf; i++)
        {
          std::string key = keys(i);

          // querying values from an octave_map returns 'Cell' objects
          octave_value val = (std::get<octave_map> (m[n]).contents (key))(0);

          bool b = save_text_data (os, val, key, false, 0);

          if (! b)
            return ! os.fail ();
        }
    }

  return true;
}

bool
octave_classdef::load_ascii (std::istream& is)
{
  octave_idx_type len = 0;
  dim_vector dv (1, 1);

  if (extract_keyword (is, "ndims", len, true))
    {
      int mdims = std::max (static_cast<int> (len), 2);
      dv.resize (mdims);
      for (int i = 0; i < mdims; i++)
        is >> dv(i);
    }
  else
    error ("load: failed to extract keyword 'ndims' for classdef object");

  // vector for each element in the array containing:
  // * a map with the values of the class properties
  // * a unique identifier of the object in the file
  // * an indicator whether the object has a custom return type
  std::vector<std::tuple<octave_map, uint32_t, bool>> m (dv.numel ());

  for (octave_idx_type i = 0; i < dv.numel (); i++)
    {
      if (! extract_keyword (is, "id", std::get<uint32_t> (m[i]), true))
        error ("load: failed to extract keyword 'id' for classdef object");

      if (! extract_keyword (is, "length", len, true))
        error ("load: failed to extract keyword 'length' for classdef object");

      if (len < 0)
        error ("load: failed to extract number of properties for classdef object");

      if (len == 0)
        continue;

      std::string metadata;
      if (! extract_keyword (is, "metadata", metadata, true))
        error ("load: failed to extract keyword 'metadata' for classdef object");

      size_t pos = metadata.find ("saveobj_defined");
      std::get<bool> (m[i]) = (pos != std::string::npos);

      for (octave_idx_type j = 0; j < len; j++)
        {
          octave_value t2;
          bool dummy;

          std::string nm = read_text_data (is, "", dummy, t2, j, false);

          if (! is)
            break;

          // Set the field in the octave_map
          std::get<octave_map> (m[i]).setfield (nm, t2);
        }

      if (! is)
        error ("load: failed to load classdef object");
    }

  loadobj (m, dv);

  return true;
}

static bool
in_class_method (const octave::cdef_class& cls)
{
  octave::cdef_class ctx = octave::get_class_context ();

  return (ctx.ok () && octave::is_superclass (ctx, cls));
}

int octave_classdef::s_t_id (-1);

const std::string octave_classdef::s_t_name ("object");

void
octave_classdef::register_type (octave::type_info& ti)
{
  s_t_id = ti.register_type (octave_classdef::s_t_name, "<unknown>",
                             octave_value (new octave_classdef ()));
}

octave_value_list
octave_classdef::subsref (const std::string& type,
                          const std::list<octave_value_list>& idx,
                          int nargout)
{
  std::size_t skip = 0;
  octave_value_list retval;

  octave::cdef_class cls = m_object.get_class ();

  if (! in_class_method (cls) && ! called_from_builtin ())
    {
      octave::cdef_method meth = cls.find_method ("subsref");

      if (meth.ok ())
        {
          octave_value_list args;

          args(1) = make_idx_args (type, idx, "subsref");

          m_count++;
          args(0) = octave_value (this);

          if (nargout <= 0)
            {
              // If the last index type is not '()', the final value of nargout is
              // unknown. Try to get its value
              if (type.back () != '(')
                {
                  // See if method numArgumentsFromSubscript is defined
                  octave::cdef_method meth_nargout
                    = cls.find_method ("numArgumentsFromSubscript");

                  if (meth_nargout.ok ())
                    {
                      octave_value_list args_nargout (3);

                      args_nargout(0) = args(0);
                      args_nargout(1) = args(1);
                      // FIXME: Third argument should be one of the possible values of
                      //        the matlab.mixin.util.IndexingContext enumeration class.
                      args_nargout(2) = octave_value (Matrix ());
                      retval = meth_nargout.execute (args_nargout, 1, true,
                                                    "numArgumentsFromSubscript");

                      nargout = retval(0).strict_int_value
                        ("subsref: return value of 'numArgumentsFromSubscript' must be integer");
                    }
                  else
                    {
                      // Method numArgumentsFromSubscript undefined. Attempt to set up
                      // a proper value for nargout at least in the simple case where the
                      // cs-list-type expression - i.e., {} or ().x, is the leading one.

                      // Set up a proper nargout for the subsref call by calling numel.
                      octave_value_list tmp;
                      int nout;
                      if (type[0] != '.')
                        tmp = idx.front ();

                      nout = xnumel (tmp);
                      // Take nout as nargout for subsref, unless the index expression
                      // is a whole sentence starting with the form id.member and id is
                      // one element (in that case, nargout remains 0).
                      if (type[0] != '.' || nout != 1 || nargout < 0)
                        nargout = nout;
                    }
                }
              else if (nargout < 0)
                nargout = 1;
            }

          retval = meth.execute (args, nargout, true, "subsref");

          // Since we're handling subsref, if the list has more than one element
          // and the caller to subsref accepts more that one output, return
          // the elements as a comma-separated list so that we can pass it to the
          // evaluator
          if (retval.length () > 1 && (nargout < 0 || nargout > 1))
            {
              if (nargout <= 0 || nargout >= retval.length ())
                // Take the whole list
                retval = octave_value (retval);
              else
                // Take nargout elements of the list
                retval = octave_value (retval.slice (0, nargout));
            }

          return retval;
        }
    }

  // At this point, the default subsref mechanism must be used.

  retval = m_object.subsref (type, idx, nargout, skip, octave::cdef_class ());

  if (type.length () > skip && idx.size () > skip)
    retval = retval(0).next_subsref (nargout, type, idx, skip);

  return retval;
}

octave_value
octave_classdef::subsref (const std::string& type,
                          const std::list<octave_value_list>& idx,
                          bool auto_add)
{
  std::size_t skip = 0;
  octave_value_list retval;

  // This variant of subsref is used to create temporary values when doing
  // assignment with multi-level indexing.  AFAIK this is only used for internal
  // purpose (not sure we should even implement this).

  octave::cdef_class cls = m_object.get_class ();

  if (! in_class_method (cls))
    {
      octave::cdef_method meth = cls.find_method ("subsref");

      if (meth.ok ())
        {
          octave_value_list args;

          args(1) = make_idx_args (type, idx, "subsref");

          m_count++;
          args(0) = octave_value (this);

          retval = meth.execute (args, 1, true, "subsref");

          return retval.length () > 0 ? retval(0) : octave_value ();
        }
    }

  retval = m_object.subsref (type, idx, 1, skip,
                             octave::cdef_class (), auto_add);

  if (type.length () > skip && idx.size () > skip)
    retval = retval(0).next_subsref (1, type, idx, skip);

  return retval.length () > 0 ? retval(0) : octave_value ();
}

octave_value
octave_classdef::subsasgn (const std::string& type,
                           const std::list<octave_value_list>& idx,
                           const octave_value& rhs)
{
  octave_value retval;

  octave::cdef_class cls = m_object.get_class ();

  if (! in_class_method (cls) && ! called_from_builtin ())
    {
      octave::cdef_method meth = cls.find_method ("subsasgn");

      if (meth.ok ())
        {
          octave_value_list args;

          args(1) = make_idx_args (type, idx, "subsasgn");

          m_count++;
          args(0) = octave_value (this);
          args(2) = rhs;

          octave_value_list retlist;

          retlist = meth.execute (args, 1, true, "subsasgn");

          if (retlist.empty ())
            error ("overloaded method 'subsasgn' did not return any value");

          retval = retlist(0);
        }
    }

  if (! retval.is_defined ())
    retval = m_object.subsasgn (type, idx, rhs);

  return retval;
}

octave_value
octave_classdef::undef_subsasgn (const std::string& type,
                                 const std::list<octave_value_list>& idx,
                                 const octave_value& rhs)
{
  if (type.length () == 1 && type[0] == '(')
    {
      m_object = m_object.make_array ();

      return subsasgn (type, idx, rhs);
    }
  else
    return octave_base_value::undef_subsasgn (type, idx, rhs);

  return octave_value ();
}

Matrix
octave_classdef::size ()
{
  octave::cdef_class cls = m_object.get_class ();

  if (! in_class_method (cls) && ! called_from_builtin ())
    {
      octave::cdef_method meth = cls.find_method ("size");

      if (meth.ok ())
        {
          m_count++;
          octave_value_list args (1, octave_value (this));

          octave_value_list lv = meth.execute (args, 1, true, "size");
          if (lv.length () <= 0
              || ! lv(0).is_matrix_type () || ! lv(0).dims ().isvector ())
            error ("%s.size: invalid return value", class_name ().c_str ());

          return lv(0).matrix_value ();
        }
    }

  return octave_base_value::size ();
}

octave_idx_type
octave_classdef::xnumel (const octave_value_list& idx)
{
  octave_idx_type retval = -1;

  // FIXME: This method is only used in subsref and subsasgn operations, to find
  // out the number of elements in the cs-list corresponding to the subsref
  // output or the subsasgn lvalue.
  // This method currently calls the classdef's numel method to do its task, but
  // this is incompatible with Matlab. Matlab calls numArgumentsFromSubscript
  // for that purpose. We cannot call numArgumentsFromSubscript here because that
  // method needs all the information about the indices of the subsref/subsasgn
  // operation (possibly multiple levels of indexing of different types)

  octave::cdef_class cls = m_object.get_class ();

  if (! in_class_method (cls) && ! called_from_builtin ())
    {
      octave::cdef_method meth = cls.find_method ("numel");

      if (meth.ok ())
        {
          octave_value_list args (idx.length () + 1, octave_value ());

          m_count++;
          args(0) = octave_value (this);

          for (octave_idx_type i = 0; i < idx.length (); i++)
            args(i+1) = idx(i);

          // Temporarily set lvalue list of current statement to NULL, to avoid
          // using that list for the execution of the method "numel"
          octave::interpreter& interp = octave::__get_interpreter__ ();
          octave::tree_evaluator& tw = interp.get_evaluator();

          octave::unwind_action act ([&tw] (const std::list<octave::octave_lvalue> *lvl)
          {
            tw.set_lvalue_list (lvl);
          }, tw.lvalue_list ());
          tw.set_lvalue_list (nullptr);

          octave_value_list lv = meth.execute (args, 1, true, "numel");
          if (lv.length () != 1 || ! lv(0).is_scalar_type ())
            error ("@%s/numel: invalid return value", cls.get_name ().c_str ());

          retval = lv(0).idx_type_value (true);

          return retval;
        }
    }

  retval = octave_base_value::xnumel (idx);

  return retval;
}

octave_value
octave_classdef::reshape (const dim_vector& new_dims) const
{
  octave_value retval;

  octave::cdef_class cls = m_object.get_class ();

  if (! in_class_method (cls) && ! called_from_builtin ())
    {
      octave::cdef_method meth = cls.find_method ("reshape");

      if (meth.ok ())
        {
          octave_value_list args;

          args(0) = octave::to_ov (m_object.clone ());
          args(1) = new_dims.as_array ();

          octave_value_list retlist;

          retlist = meth.execute (args, 1, true, "reshape");

          if (retlist.empty ())
            error ("overloaded method 'reshape' did not return any value");

          retval = retlist(0);
        }
    }

  if (! retval.is_defined ())
    retval = m_object.reshape (new_dims);

  return retval;
}

void
octave_classdef::print (std::ostream& os, bool)
{
  print_raw (os);
}

void
octave_classdef::print_raw (std::ostream& os, bool) const
{
  octave::cdef_class cls = m_object.get_class ();

  if (cls.ok ())
    {
      bool is_array = m_object.is_array ();

      increment_indent_level ();

      indent (os);

      dim_vector dv = dims ();
      if (dv.ndims () > 4)
        os << dv.ndims () << "-D";
      else
        os << dims().str();

      os << " " << class_name () << " object";
      if (is_array)
        os << " array";
      os << " with properties:";
      newline (os);
      if (! Vcompact_format)
        newline (os);

      increment_indent_level ();

      std::map<octave::property_key, octave::cdef_property> property_map
        = cls.get_property_map ();

      std::size_t max_len = 0;
      for (const auto& pname_prop : property_map)
        {
          // FIXME: this loop duplicates a significant portion of the
          // loop below and the loop in Fproperties.

          const octave::cdef_property& prop = pname_prop.second;

          const std::string nm = prop.get_name ();

          octave_value acc = prop.get ("GetAccess");

          if (! acc.is_string () || acc.string_value () != "public")
            continue;

          octave_value hid = prop.get ("Hidden");

          if (hid.bool_value ())
            continue;

          std::size_t sz = nm.size ();

          if (sz > max_len)
            max_len = sz;
        }

      for (auto& pname_prop : property_map)
        {
          const octave::cdef_property& prop = pname_prop.second;

          const std::string nm = prop.get_name ();

          octave_value acc = prop.get ("GetAccess");

          if (! acc.is_string () || acc.string_value () != "public")
            continue;

          octave_value hid = prop.get ("Hidden");

          if (hid.bool_value ())
            continue;

          indent (os);

          if (is_array)
            os << "  " << nm;
          else
            {
              os << std::setw (max_len+2) << nm << ": ";

              octave_value val = prop.get_value (m_object, false);

              val.short_disp (os);
            }

          newline (os);
        }

      decrement_indent_level ();
      decrement_indent_level ();
    }
}

bool
octave_classdef::is_instance_of (const std::string& cls_name) const
{
  octave::cdef_class cls = octave::lookup_class (cls_name, false, false);

  if (cls.ok ())
    return is_superclass (cls, m_object.get_class ());

  return false;
}

octave_value
octave_classdef::superclass_ref (const std::string& meth,
                                 const std::string& cls)
{
  return octave_value (new octave_classdef_superclass_ref (meth, cls));
}

octave_value
octave_classdef::metaclass_query (const std::string& cls)
{
  return octave::to_ov (octave::lookup_class (cls));
}

bool
octave_classdef_meta::is_classdef_method (const std::string& cname) const
{
  bool retval = false;

  if (m_object.is_method ())
    {
      if (cname.empty ())
        retval = true;
      else
        {
          octave::cdef_method meth (m_object);

          return meth.is_defined_in_class (cname);
        }
    }

  return retval;
}

bool
octave_classdef_meta::is_classdef_constructor (const std::string& cname) const
{
  bool retval = false;

  if (m_object.is_class ())
    {
      if (cname.empty ())
        retval = true;
      else
        {
          octave::cdef_class cls (m_object);

          if (cls.get_name () == cname)
            retval = true;
        }
    }
  else if (m_object.is_method ())
    {
      octave::cdef_method meth (m_object);

      if (meth.is_constructor ())
        {
          std::string meth_name = meth.get_name ();

          // Only consider METH to be a constructor if the dispatch
          // class CNAME is the same as or derived from the class of
          // METH.

          if (cname == meth_name)
            retval = true;
          else
            {
              octave::cdef_class meth_cls = octave::lookup_class (meth_name, false, false);
              octave::cdef_class dispatch_cls = octave::lookup_class (cname, false, false);

              retval = octave::is_superclass (meth_cls, dispatch_cls);
            }
        }
    }

  return retval;
}

std::string
octave_classdef_meta::doc_string (const std::string& meth_name) const
{
  if (m_object.is_class ())
    {
      octave::cdef_class cls (m_object);

      if (meth_name.empty ())
        return cls.doc_string ();

      octave::cdef_method cdef_meth = cls.find_method (meth_name);

      if (cdef_meth.ok ())
        return cdef_meth.get_doc_string ();
    }

  return "";
}

std::string
octave_classdef_meta::file_name () const
{
  if (m_object.is_class ())
    {
      octave::cdef_class cls (m_object);

      return cls.file_name ();
    }

  return "";
}

octave_value_list
octave_classdef_superclass_ref::execute (octave::tree_evaluator& tw,
    int nargout,
    const octave_value_list& idx)
{
  octave_value_list retval;

  std::string meth_name;
  bool in_constructor;
  octave::cdef_class ctx;

  ctx = octave::get_class_context (meth_name, in_constructor);

  if (! ctx.ok ())
    error ("superclass calls can only occur in methods or constructors");

  std::string mname = m_method_name;
  std::string cname = m_class_name;

  // CLS is the superclass.  The lookup_class function handles
  // pkg.class names.

  octave::cdef_class cls = octave::lookup_class (cname);

  if (in_constructor)
    {
      if (! is_direct_superclass (cls, ctx))
        error ("'%s' is not a direct superclass of '%s'",
               cname.c_str (), ctx.get_name ().c_str ());

      if (! is_constructed_object (tw, mname))
        error ("cannot call superclass constructor with variable '%s'",
               mname.c_str ());

      octave_value sym = tw.varval (mname);

      cls.run_constructor (octave::to_cdef_ref (sym), idx);

      retval(0) = sym;
    }
  else
    {
      std::size_t pos = mname.find ('.');

      octave::cdef_object obj;

      if (pos != std::string::npos)
        {
          // We are looking at obj.meth.

          std::string oname = m_method_name.substr (0, pos);
          mname = mname.substr (pos + 1);

          octave_value tval = tw.varval (oname);

          // FIXME: Can we only call superclass methods on the current
          // object?  If so, and we are looking at something like
          //
          //   function meth (obj, ...)
          //     obj.meth@superclass (...)
          //
          // Do we need to verify that the object that was passed to
          // meth is the same as the object we find when looking up
          // obj in the expression obj.meth?  If so, what is the right
          // way to perform that check?

          if (tval.is_classdef_object ())
            {
              octave_classdef *cdobj = tval.classdef_object_value ();

              obj = cdobj->get_object ();
            }
        }

      if (mname != meth_name)
        error ("method name mismatch ('%s' != '%s')",
               mname.c_str (), meth_name.c_str ());

      if (! is_strict_superclass (cls, ctx))
        error ("'%s' is not a superclass of '%s'",
               cname.c_str (), ctx.get_name ().c_str ());

      // I see 2 possible implementations here:
      // 1) use cdef_object::subsref with a different class
      //    context; this avoids duplicating code, but
      //    assumes the object is always the first argument
      // 2) lookup the method manually and call
      //    cdef_method::execute; this duplicates part of
      //    logic in cdef_object::subsref, but avoid the
      //    assumption of 1)
      // Not being sure about the assumption of 1), I
      // go with option 2) for the time being.

      octave::cdef_method meth = cls.find_method (meth_name, false);

      if (! meth.ok ())
        error ("no method '%s' found in superclass '%s'",
               meth_name.c_str (), cname.c_str ());

      retval = (obj.ok ()
                ? meth.execute (obj, idx, nargout, true, meth_name)
                : meth.execute (idx, nargout, true, meth_name));
    }

  return retval;
}

bool
octave_classdef_superclass_ref::is_constructed_object (octave::tree_evaluator& tw,
    const std::string& nm)
{
  octave_function *of = tw.current_function ();

  if (of->is_classdef_constructor ())
    {
      octave_user_function *uf = of->user_function_value (true);

      if (uf)
        {
          octave::tree_parameter_list *ret_list = uf->return_list ();

          if (ret_list && ret_list->size () == 1)
            return (ret_list->front ()->name () == nm);
        }
    }

  return false;
}

OCTAVE_BEGIN_NAMESPACE(octave)

DEFUN (__meta_get_package__, args, ,
       doc: /* -*- texinfo -*-
@deftypefn {} {@var{pkg} =} __meta_get_package__ (@var{pkg_name})
Undocumented internal function.
@end deftypefn */)
{
  if (args.length () != 1)
    print_usage ();

  std::string cname = args(0).xstring_value ("PKG_NAME must be a string");

  return to_ov (lookup_package (cname));
}

DEFUN (metaclass, args, ,
       doc: /* -*- texinfo -*-
@deftypefn {} {@var{metaclass_obj} =} metaclass (obj)
Return the meta.class object corresponding to the class of @var{obj}.
@end deftypefn */)
{
  if (args.length () != 1)
    print_usage ();

  cdef_object obj = to_cdef (args(0));

  return to_ov (obj.get_class ());
}

// FIXME: What about dynamic properties if obj is a scalar, or the
// properties of the class of obj if obj is an array?  Probably there
// should be a function to do this job so that the DEFUN is just a
// simple wrapper.

DEFUN (properties, args, nargout,
       doc: /* -*- texinfo -*-
@deftypefn  {} {} properties (@var{obj})
@deftypefnx {} {} properties (@var{class_name})
@deftypefnx {} {@var{proplist} =} properties (@dots{})
Display or return the public properties for the classdef object @var{obj} or
the named class @var{class_name}.

If an output value is requested, return the list of property names in a cell
array.

Programming Note: Property names are returned if the @code{GetAccess} attribute
is public and if the @code{Hidden} attribute is false.
@seealso{methods}
@end deftypefn */)
{
  if (args.length () != 1)
    print_usage ();

  octave_value arg = args(0);

  std::string class_name;

  if (arg.isobject ())
    class_name = arg.class_name ();
  else if (arg.is_string ())
    class_name = arg.string_value ();
  else
    err_wrong_type_arg ("properties", arg);

  cdef_class cls;

  cls = lookup_class (class_name, false, true);

  if (! cls.ok ())
    error ("invalid class: %s", class_name.c_str ());

  std::map<octave::property_key, cdef_property> property_map =
    cls.get_property_map ();

  std::list<std::string> property_names;

  for (const auto& pname_prop : property_map)
    {
      // FIXME: this loop duplicates a significant portion of the loops
      // in octave_classdef::print_raw.
      const cdef_property& prop = pname_prop.second;

      octave_value acc = prop.get ("GetAccess");

      if (! acc.is_string () || acc.string_value () != "public")
        continue;

      octave_value hid = prop.get ("Hidden");

      if (hid.bool_value ())
        continue;

      property_names.push_back (pname_prop.second.get_name ());
    }

  if (nargout > 0)
    return octave_value (Cell (string_vector (property_names)));

  octave_stdout << "properties for class " << class_name << ":\n\n";

  for (const auto& nm : property_names)
    octave_stdout << "  " << nm << "\n";

  octave_stdout << std::endl;

  return octave_value ();
}

/*
%!assert (properties ("inputParser"),
%!        {"CaseSensitive"; "FunctionName"; "KeepUnmatched"; "PartialMatching";
%!         "StructExpand"; "Parameters"; "Results"; "Unmatched";
%!         "UsingDefaults"});
*/

// FIXME: Need to implement the -full option.

DEFMETHOD (__methods__, interp, args, ,
           doc: /* -*- texinfo -*-
@deftypefn  {} {[@var{mtds}, @var{found}] =} __methods__ (@var{obj})
@deftypefnx {} {[@var{mtds}, @var{found}] =} __methods__ ("classname")
Implement @code{methods} for Octave class objects and classnames.
@seealso{methods}
@end deftypefn */)
{
  // Input validation has already been done in methods.m.
  octave_value arg = args(0);

  std::string class_name;

  if (arg.isobject ())
    class_name = arg.class_name ();
  else if (arg.is_string ())
    class_name = arg.string_value ();
  else
    err_wrong_type_arg ("__methods__", arg);

  string_vector sv;
  bool found = false;

  cdef_class cls = lookup_class (class_name, false, true);

  if (cls.ok ())
    {
      // Find methods for classdef objects.
      std::map<std::string, cdef_method> method_map
        = cls.get_method_map (false, true);

      std::list<std::string> method_names;

      for (const auto& nm_mthd : method_map)
        {
          const cdef_method& method = nm_mthd.second;

          octave_value acc = method.get ("Access");

          if (! acc.is_string () || acc.string_value () != "public")
            continue;

          octave_value hid = method.get ("Hidden");

          if (hid.bool_value ())
            continue;

          method_names.push_back (nm_mthd.first);
        }

      sv = string_vector (method_names);
      found = true;
    }
  else
    {
      // Find methods for legacy @CLASS objects.
      load_path& lp = interp.get_load_path ();

      sv = string_vector (lp.methods (class_name));
      found = ! sv.empty ();
    }

  return ovl (Cell (sv), found);
}

/*
// BIST tests are in file methods.m
%!assert (1)
*/

OCTAVE_END_NAMESPACE(octave)
