/*
    beacondb.c - Beacon database 
    Copyright (C) 2024 Ladislav Vaiz <ok1zia@nagano.cz>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

*/

#include "header.h"

#include "beacondb.h"
#include "fifo.h"
#include "language2.h"
#include "main.h"
#include "map.h"
#include "subwin.h"
#include "tsdl.h"
#include "qsodb.h"

struct beacondb* gbeacondb;

struct beacondb *init_beacondb(){
    struct beacondb *beacondb = g_new0(struct beacondb, 1);
	beacondb->bands = g_ptr_array_new();

    return beacondb;
}

void free_beacon(struct beacon *beacon){
	if (!beacon) return;

	g_free(beacon->call);
	g_free(beacon->wwl);
	g_free(beacon->location);
	g_free(beacon->antenna);
	g_free(beacon->heading);
	g_free(beacon->status);
	g_free(beacon->last);

	g_free(beacon);
}

void free_beaconband(struct beaconband *beaconband){
	int i;
	for (i = 0; i < beaconband->beacons->len; i++) {
		struct beacon* b = (struct beacon*)g_ptr_array_index(beaconband->beacons, i);
		free_beacon(b);
	}
}

void free_beacondb(struct beacondb *beacondb){
	progress("Teminating beacon database"); //progress(VTEXT(T_FREE_NAMEDB));

	int i;
	for (i = 0; i < beacondb->bands->len; i++) {
		struct beaconband* b = (struct beaconband*)g_ptr_array_index(beacondb->bands, i);
		free_beaconband(b);
	}
	g_ptr_array_free(beacondb->bands, TRUE);
	g_free(beacondb);
}

struct beaconband* get_beacondb_band(struct beacondb* beacondb, char bandchar) {
	int i;
	bandchar = z_char_uc(bandchar);
	for (i = 0; i < beacondb->bands->len; i++) {
		struct beaconband* b = (struct beaconband*)g_ptr_array_index(beacondb->bands, i);
		if (b->bandchar == bandchar) return b;
	}
	return NULL;
}

int load_beacondb_from_mem(struct beacondb* beacondb, const char* file) {
	struct zjson* beacons = zjson_init(file);

	if (!beacons) {
		zjson_free(beacons);
		return -1;
	}

	while (1) {
		struct zjson* item = zjson_get_array(beacons, NULL);
		if (!item) break;
		
		char* call = zjson_get_str(item, "Beacon", NULL);
		char* wwl = zjson_get_str(item, "Locator", NULL);
		double freq = zjson_get_double(item, "Frequency", 0.0);

		struct config_band* cband = get_config_band_by_qrg(freq * 1000.0);

		if (call != NULL && wwl != NULL && cband != NULL) {
			struct beaconband* bband = get_beacondb_band(beacondb, cband->bandchar);
			if (!bband) {
				bband = g_new0(struct beaconband, 1);
				bband->bandchar = cband->bandchar;
				bband->beacons = g_ptr_array_new();
				g_ptr_array_add(beacondb->bands, bband);
			}

			struct beacon* beacon = g_new0(struct beacon, 1);
			beacon->call = g_strdup(call);
			beacon->wwl = g_strdup(wwl);
			beacon->freq = freq;
			beacon->location = zjson_get_str(item, "Location", NULL);
			beacon->antenna = zjson_get_str(item, "Antenna", NULL);
			beacon->heading = zjson_get_str(item, "Heading", NULL);
			beacon->power = zjson_get_double(item, "Power", 0.0);
			beacon->erp = zjson_get_double(item, "ERP", 0.0);
			beacon->status = zjson_get_str(item, "Status", NULL);
			beacon->last = zjson_get_str(item, "DateLastSpot", NULL);
			g_ptr_array_add(bband->beacons, beacon);
			beacon_compute_qrbqtf(beacon);
		}

		g_free(call);
		g_free(wwl);
	}
	zjson_free(beacons);

	return 0;
}

int load_beacondb_from_file(struct beacondb* beacondb, gchar* file) {
	/*dbg("load_beacondb_from_file(%s)\n", filebeacon);*/
	char *data = zfile_read_textfile(file);
	if (data == NULL) return -1;

	return load_beacondb_from_mem(beacondb, data);
}

void read_beacondb_files(struct beacondb* beacondb) {
	gchar* s;
	int ret = 0;

	progress("Loading beacon database"); //VTEXT(T_LOADE_BEACONDB));

	s = g_strconcat(tucnak_dir, "/tucnakbeacons", NULL);
	z_wokna(s);
	ret |= load_beacondb_from_file(beacondb, s);
	if (ret) {
		GString* gs = g_string_sized_new(300000);
		g_string_append(gs, txt_tucnakbeacons);
		g_string_append(gs, txt_tucnakbeacons1);
		g_string_append(gs, txt_tucnakbeacons2);
		g_string_append(gs, txt_tucnakbeacons3);
		load_beacondb_from_mem(beacondb, gs->str);
		g_string_free(gs, TRUE);
	}
	g_free(s);
}

void save_beacondb_file(const char* data) {
	progress("Saving beacon database"); //VTEXT(T_SAVE_BEACONDB));
	gchar* filename = g_strconcat(tucnak_dir, "/tucnakbeacons", NULL);
	z_wokna(filename);
	if (zfile_printfile(filename, "%s", data) < 1) {
		log_addf("Error saving %s", filename);
	}else{
		log_addf("Saved %s", filename);
	}
	g_free(filename);
	progress(NULL);
}

#ifdef Z_HAVE_SDL
void plot_beacon(struct subwin* sw, SDL_Surface* surface, struct beacon* beacon) {
	int kx, ky, px, py, color;
	SDL_Rect outline;

	outline.w = 9;
	outline.h = 22;
	kx = beacon->kx;
	ky = beacon->ky;
	km2px(sw, kx, ky, &px, &py);
	outline.x = px - 4;
	outline.y = py - 2;
	//z_rect2(surface, &outline, sdl->green);
	if (!z_overlapped_rect(&surface->clip_rect, &outline)) {
		return;
	}
	/*       dbg("kreslim qso\n"); */
	color = beacon == sw->minbeacon ? sdl->red : z_makecol(0, 64, 127);
	//z_cross(surface, px, py, color, sw->zoom);
	z_pip(surface, px, py, color, color, sw->zoom);

	int r = 16;
	if (sw->zoom < 10000) r = (r * sw->zoom) / 10000;
	//double rad = -M_PI/2;

	int vx = px;
	int vy = py;
	int dx = (int)round(0.2 * r);
	int vyy = (int)round(py + 2.0 + r);

	if (dx < 1) dx = 1;
	int i = 0;
	for (i = 0; i <= dx; i++) {
		int dy = (vyy-vy) * i / dx;
		z_line(surface, vx + i, vyy, vx + i, vy+dy, color);
		z_line(surface, vx - i, vyy, vx - i, vy+dy, color);
	}
	

}


void plot_beacons(struct subwin* sw, SDL_Surface * surface, struct band* band, SDL_Rect * area) {

	if (!band) {
		// all bands
		int j;
		for (j = 0; j < gbeacondb->bands->len; j++) {
			struct beaconband* bband = (struct beaconband*)g_ptr_array_index(gbeacondb->bands, j);
			int i;
			for (i = 0; i < bband->beacons->len; i++) {
				struct beacon* beacon = (struct beacon*)g_ptr_array_index(bband->beacons, i);
				plot_beacon(sw, surface, beacon);
			}
		}
		if (sw->minbeacon) plot_beacon(sw, surface, sw->minbeacon); // same loc different band
	}else{
		// one band
		struct beaconband* bband = get_beacondb_band(gbeacondb, band->bandchar);
		if (!bband) return;

		int i;
		for (i = 0; i < bband->beacons->len; i++) {
			struct beacon* beacon = (struct beacon*)g_ptr_array_index(bband->beacons, i);
			plot_beacon(sw, surface, beacon);
		}
	}

}
#endif

void beacon_compute_qrbqtf(struct beacon* beacon) {
	double qtf;

	if (!beacon) return;

	char* wwl = ctest ? ctest->pwwlo : cfg->pwwlo;
	qrbqtf(wwl, beacon->wwl, &beacon->qrb, &qtf, NULL, 2);
	beacon->qtf = (int)(qtf + 0.5);
	if (beacon->qrb < 0.1) {
		beacon->qrb = 0;
		beacon->qtf = 0;
	}
	beacon->qtfrad = qtf * M_PI / 180.0;
	beacon->kx = (int)(beacon->qrb * sin(beacon->qtfrad));
	beacon->ky = (int)(-beacon->qrb * cos(beacon->qtfrad));
	
}

void recalc_beaconband(struct beaconband* bband) {
	int i;
	for (i = 0; i < bband->beacons->len; i++) {
		struct beacon* beacon = (struct beacon*)g_ptr_array_index(bband->beacons, i);
		beacon_compute_qrbqtf(beacon);
	}
}

void beacondb_recalc_qrbqtf(struct beacondb *beacondb) {
	int i;
	for (i = 0; i < beacondb->bands->len; i++) {
		struct beaconband* b = (struct beaconband*)g_ptr_array_index(beacondb->bands, i);
		recalc_beaconband(b);
	}
}

#ifdef Z_HAVE_SDL
void plot_info_beacon(struct subwin* sw, SDL_Surface* surface, struct beacon* beacon) {
	//char s[10];
	int x, y;
	SDL_Rect rect;

	//dbg("plot_info_beacon('%s')\n", qso ? qso->callsign : "");
	rect.x = sw->info.x;  // same as plot_info_ac
	rect.y = sw->info.y;
	rect.w = sw->info.w;
	rect.h = sw->info.y + (11 * FONT_H) + 12;

	SDL_SetClipRect(surface, &rect);
	SDL_FillRect(surface, &rect, z_makecol(0, 0, 35));
	if (!beacon) return;

	int dy = 0;
	x = sw->info.x + 4;
	y = sw->info.y + 4;
	zsdl_printf(surface, x, y, sdl->green, 0, ZFONT_TRANSP, VTEXT(T_GCALL), q0(beacon->call)); y += FONT_H + dy;
	zsdl_printf(surface, x, y, sdl->green, 0, ZFONT_TRANSP, "QRG: %5.4f", beacon->freq); y += FONT_H + dy;
	zsdl_printf(surface, x, y, sdl->green, 0, ZFONT_TRANSP, VTEXT(T_GWWL), q0(beacon->wwl)); y += FONT_H + dy;
	zsdl_printf(surface, x, y, sdl->green, 0, ZFONT_TRANSP, "%s", q0(beacon->location)); y += FONT_H + dy;
	zsdl_printf(surface, x, y, sdl->green, 0, ZFONT_TRANSP, VTEXT(T_GQRB), (int)beacon->qrb); y += FONT_H + dy;
	zsdl_printf(surface, x, y, sdl->green, 0, ZFONT_TRANSP, VTEXT(T_GQTF), (int)beacon->qtf); y += FONT_H + dy;
	zsdl_printf(surface, x, y, sdl->green, 0, ZFONT_TRANSP, "Antenna: %s", q0(beacon->antenna)); y += FONT_H + dy;
	zsdl_printf(surface, x, y, sdl->green, 0, ZFONT_TRANSP, "Heading: %s", q0(beacon->heading)); y += FONT_H + dy;
	if (beacon->erp > 0.0) {
		zsdl_printf(surface, x, y, sdl->green, 0, ZFONT_TRANSP, "ERP: %4.3f", beacon->erp); y += FONT_H + dy;
	}else if (beacon->power > 0.0) {
		zsdl_printf(surface, x, y, sdl->green, 0, ZFONT_TRANSP, "Power: %4.3f", beacon->power); y += FONT_H + dy;
	}
	//zsdl_printf(surface, x, y, sdl->green, 0, ZFONT_TRANSP, "Status: %s", q0(beacon->status)); y += FONT_H + dy;
	zsdl_printf(surface, x, y, sdl->green, 0, ZFONT_TRANSP, "Spot: %s", q0(beacon->last)); y += FONT_H + dy;
	return;
}
#endif

void beaconspot_downloaded_callback(struct zhttp* http) {

	if (http->state == ZHTTPST_ERROR && http->errorstr) {
		log_addf("Error downloading from beaconspot.uk: %s", http->errorstr);
	}
	else {
		if (http_is_content_type(http, "application/json")) {
			char *data = g_new(char, http->response->len + 1);
			zbinbuf_getstr(http->response, http->dataofs, data, http->response->len + 1);
			struct zjson* json = zjson_init(data);
			struct zjson* beacons = zjson_get_object(json, "Beacons");
			if (!beacons) {
				log_addf("Error downloading from beaconspot.uk: Bad data format\n");
			}else {
				save_beacondb_file(beacons->gs->str);
				// log_addf already printed
				zjson_free(beacons);

				free_beacondb(gbeacondb);
				gbeacondb = init_beacondb();
				read_beacondb_files(gbeacondb);
				sw_set_gdirty(SWT_MAP);
			}
			zjson_free(json);
			g_free(data);
		}
		else {
			log_addf("Error downloading from beaconspot.uk: Bad Content-Type\n");
		}
	}
	zhttp_free(http);
}
\
void menu_import_beaconspot(void* arg) {
	struct zhttp *http = zhttp_init();
	zhttp_get(http, zsel, "https://www.beaconspot.uk/tucnakbeacons.php", beaconspot_downloaded_callback, http);
}

